/**************************************************************************//**
 * @file     NUC126.h
 * @version  V3.0
 * $Revision: 39 $
 * $Date: 16/10/25 4:25p $
 * @brief    NUC126 Series Peripheral Access Layer Header File
 *
 * @note
 *
 * @copyright SPDX-License-Identifier: Apache-2.0
 *
 * @copyright Copyright (C) 2016 Nuvoton Technology Corp. All rights reserved.
 ******************************************************************************/

/**
  \mainpage Introduction
  *
  *
  * This user manual describes the usage of NUC126 MCU device driver
  *
  * <b>Disclaimer</b>
  *
  * The Software is furnished "AS IS", without warranty as to performance or results, and
  * the entire risk as to performance or results is assumed by YOU. Nuvoton disclaims all
  * warranties, express, implied or otherwise, with regard to the Software, its use, or
  * operation, including without limitation any and all warranties of merchantability, fitness
  * for a particular purpose, and non-infringement of intellectual property rights.
  *
  * <b>Copyright Notice</b>
  *
  * Copyright (C) 2016~2024 Nuvoton Technology Corp. All rights reserved.
  */

#ifndef __NUC126_H__
#define __NUC126_H__


/*
 * ==========================================================================
 * ---------- Interrupt Number Definition -----------------------------------
 * ==========================================================================
 */

/**
 * @details  Interrupt Number Definition. The maximum of 32 Specific Interrupts are possible.
 */
typedef enum IRQn
{
    /******  Cortex-M0 Processor Exceptions Numbers ***************************************************/
    NonMaskableInt_IRQn       = -14,      /*!< 2 Non Maskable Interrupt                             */
    HardFault_IRQn            = -13,      /*!< 3 Cortex-M0 Hard Fault Interrupt                     */
    SVCall_IRQn               = -5,       /*!< 11 Cortex-M0 SV Call Interrupt                       */
    PendSV_IRQn               = -2,       /*!< 14 Cortex-M0 Pend SV Interrupt                       */
    SysTick_IRQn              = -1,       /*!< 15 Cortex-M0 System Tick Interrupt                   */

    /******  ARMIKMCU Swift specific Interrupt Numbers ************************************************/
    BOD_IRQn                  = 0,        /*!< Brown-Out Low Voltage Detected Interrupt             */
    WDT_IRQn                  = 1,        /*!< Watch Dog Timer Interrupt                            */
    EINT024_IRQn              = 2,        /*!< EINT0, EINT2 and EINT4 Interrupt                     */
    EINT135_IRQn              = 3,        /*!< EINT1, EINT3 and EINT5 Interrupt                     */
    GPAB_IRQn                 = 4,        /*!< GPIO_PA/PB Interrupt                                 */
    GPCDEF_IRQn               = 5,        /*!< GPIO_PC/PD/PE/PF Interrupt                           */
    PWM0_IRQn                 = 6,        /*!< PWM0 Interrupt                                       */
    PWM1_IRQn                 = 7,        /*!< PWM1 Interrupt                                       */
    TMR0_IRQn                 = 8,        /*!< TIMER0 Interrupt                                     */
    TMR1_IRQn                 = 9,        /*!< TIMER1 Interrupt                                     */
    TMR2_IRQn                 = 10,       /*!< TIMER2 Interrupt                                     */
    TMR3_IRQn                 = 11,       /*!< TIMER3 Interrupt                                     */
    UART02_IRQn               = 12,       /*!< UART0 and UART2 Interrupt                            */
    UART1_IRQn                = 13,       /*!< UART1 Interrupt                                      */
    SPI0_IRQn                 = 14,       /*!< SPI0 Interrupt                                       */
    SPI1_IRQn                 = 15,       /*!< SPI1 Interrupt                                       */
    I2C0_IRQn                 = 18,       /*!< I2C0 Interrupt                                       */
    I2C1_IRQn                 = 19,       /*!< I2C1 Interrupt                                       */
    USCI_IRQn                 = 22,       /*!< USCI0, USCI1 and USCI2 interrupt                     */
    USBD_IRQn                 = 23,       /*!< USB Device Interrupt                                 */
    SC01_IRQn                 = 24,       /*!< SC0 and SC1 Interrupt                                */
    ACMP01_IRQn               = 25,       /*!< ACMP0/1 Interrupt                                    */
    PDMA_IRQn                 = 26,       /*!< PDMA Interrupt                                       */
    PWRWU_IRQn                = 28,       /*!< Power Down Wake Up Interrupt                         */
    ADC_IRQn                  = 29,       /*!< ADC Interrupt                                        */
    CLKDIRC_IRQn              = 30,       /*!< Clock fail detect and IRC TRIM Interrupt             */
    RTC_IRQn                  = 31        /*!< Real Time Clock Interrupt                            */

} IRQn_Type;


/*
 * ==========================================================================
 * ----------- Processor and Core Peripheral Section ------------------------
 * ==========================================================================
 */

/* Configuration of the Cortex-M0 Processor and Core Peripherals */
#define __MPU_PRESENT           0       /*!< armikcmu does not provide a MPU present or not       */
#define __NVIC_PRIO_BITS        2       /*!< armikcmu Supports 2 Bits for the Priority Levels     */
#define __Vendor_SysTickConfig  0       /*!< Set to 1 if different SysTick Config is used         */
#define __FPU_PRESENT           0
#ifndef __SOFTFP__
# define __SOFTFP__             1
#endif


#include "core_cm0.h"                   /*!< Cortex-M0 processor and core peripherals             */
#include "system_NUC126.h"          /*!< NUC126 System                                    */


#if defined ( __CC_ARM   )
#pragma anon_unions
#endif


/**
 * Initialize the system clock
 *
 * @param  None
 * @return None
 *
 * @brief  Setup the microcontroller system
 *         Initialize the PLL and update the SystemFrequency variable
 */
extern void SystemInit(void);



/******************************************************************************/
/*                Device Specific Peripheral registers structures             */
/******************************************************************************/

/** @addtogroup REGISTER Control Register

  @{

*/


/*---------------------- Analog Comparator Controller -------------------------*/
/**
    @addtogroup ACMP Analog Comparator Controller(ACMP)
    Memory Mapped Structure for ACMP Controller
@{ */

typedef struct
{


    /**
     * @var ACMP_T::CTL
     * Offset: 0x00  Analog Comparator 0 Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ACMPEN    |Comparator Enable Bit
     * |        |          |0 = Comparator 0 Disabled.
     * |        |          |1 = Comparator 0 Enabled.
     * |[1]     |ACMPIE    |Comparator Interrupt Enable Bit
     * |        |          |0 = Comparator 0 interrupt Disabled.
     * |        |          |1 = Comparator 0 interrupt Enabled
     * |        |          |If WKEN (ACMP_CTL0[16]) is set to 1, the wake-up interrupt function will be enabled as well.
     * |[2]     |HYSEN     |Comparator Hysteresis Enable Bit
     * |        |          |0 = Comparator 0 hysteresis Disabled.
     * |        |          |1 = Comparator 0 hysteresis Enabled.
     * |[3]     |ACMPOINV  |Comparator Output Inverse
     * |        |          |0 = Comparator 0 output inverse Disabled.
     * |        |          |1 = Comparator 0 output inverse Enabled.
     * |[5:4]   |NEGSEL    |Comparator Negative Input Selection
     * |        |          |00 = ACMP0_N pin.
     * |        |          |01 = Internal comparator reference voltage (CRV).
     * |        |          |10 = Band-gap voltage.
     * |        |          |11 = ReservedGround.
     * |[7:6]   |POSSEL    |Comparator Positive Input Selection
     * |        |          |00 = Input from ACMP0_P0.
     * |        |          |01 = Input from ACMP0_P1.
     * |        |          |10 = Input from ACMP0_P2.
     * |        |          |11 = Input from ACMP0_P3.
     * |[9:8]   |INTPOL    |Interrupt Condition Polarity Selection
     * |        |          |ACMPIF0 will be set to 1 when comparator output edge condition is detected.
     * |        |          |00 = Rising edge or falling edge.
     * |        |          |01 = Rising edge.
     * |        |          |10 = Falling edge.
     * |        |          |11 = Reserved.
     * |[12]    |OUTSEL    |Comparator Output Selection
     * |        |          |0 = Comparator 0 output to ACMP0_O pin is unfiltered comparator output.
     * |        |          |1 = Comparator 0 output to ACMP0_O pin is from filter output.
     * |[15:13] |FILTSEL   |Comparator Output Filter Count Selection
     * |        |          |000 = Filter function is Disabled.
     * |        |          |001 = ACMP output is sampled 1 consecutive PCLK.
     * |        |          |010 = ACMP output is sampled 2 consecutive PCLKs.
     * |        |          |011 = ACMP output is sampled 4 consecutive PCLKs.
     * |        |          |100 = ACMP output is sampled 8 consecutive PCLKs.
     * |        |          |101 = ACMP output is sampled 16 consecutive PCLKs.
     * |        |          |110 = ACMP output is sampled 32 consecutive PCLKs.
     * |        |          |111 = ACMP output is sampled 64 consecutive PCLKs.
     * |[16]    |WKEN      |Power-down Wake-up Enable Bit
     * |        |          |0 = Wake-up function Disabled.
     * |        |          |1 = Wake-up function Enabled.
     * |[17]    |WLATEN    |Window Mode SelectionLatch Function Enable Bit
     * |        |          |0 = Window Latch Function Disabled.
     * |        |          |1 = Window Latch Function Enabled.
     * |[18]    |WCMPSEL   |Window Compare Mode Selection
     * |        |          |0 = Window Compare Mode Disabled.
     * |        |          |1 = Window Compare Mode is Selected.
     * @var ACMP_T::STATUS
     * Offset: 0x08  Analog Comparator Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ACMPIF0   |Comparator 0 Interrupt Flag
     * |        |          |This bit is set by hardware when the edge condition defined by INTPOL (ACMP_CTL0[9:8]) is detected on comparator 0 output
     * |        |          |This will generate an interrupt if ACMPIE (ACMP_CTL0[1]) is set to 1.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[1]     |ACMPIF1   |Comparator 1 Interrupt Flag
     * |        |          |This bit is set by hardware when the edge condition defined by INTPOL (ACMP_CTL1[9:8]) is detected on comparator 1 output
     * |        |          |This will cause an interrupt if ACMPIE (ACMP_CTL1[1]) is set to 1.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[4]     |ACMPO0    |Comparator 0 Output
     * |        |          |Synchronized to the PCLK to allow reading by software
     * |        |          |Cleared when the comparator 0 is disabled, i.e
     * |        |          |ACMPEN (ACMP_CTL0[0]) is cleared to 0.
     * |[5]     |ACMPO1    |Comparator 1 Output
     * |        |          |Synchronized to the PCLK to allow reading by software
     * |        |          |Cleared when the comparator 1 is disabled, i.e
     * |        |          |ACMPEN (ACMP_CTL1[0]) is cleared to 0.
     * |[8]     |WKIF0     |Comparator 0 Power-down Wake-up Interrupt Flag
     * |        |          |This bit will be set to 1 when ACMP0 wake-up interrupt event occurs.
     * |        |          |0 = No power-down wake-up occurred.
     * |        |          |1 = Power-down wake-up occurred.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[9]     |WKIF1     |Comparator 1 Power-down Wake-up Interrupt Flag
     * |        |          |This bit will be set to 1 when ACMP1 wake-up interrupt event occurs.
     * |        |          |0 = No power-down wake-up occurred.
     * |        |          |1 = Power-down wake-up occurred.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[12]    |ACMPS0    |Comparator 0 StatusResult
     * |        |          |Synchronized to the PCLK to allow reading by software
     * |        |          |Cleared when the comparator 0 is disabled, i.e
     * |        |          |ACMPEN (ACMP_CTL0[0]) is cleared to 0.Window Compare Output
     * |[13]    |ACMPS1    |Window Comparator 1 StatusResultOutput
     * |        |          |Synchronized to the PCLK to allow reading by software
     * |        |          |Cleared when the comparator 1 is disabled, i.e
     * |        |          |ACMPEN (ACMP_CTL1[0]) is cleared to 0.
     * |[16]    |ACMPWO    |Comparator Window Compare Output
     * |        |          |This bit shows the output status of window compare mode
     * |        |          |0 = The positvie input voltage is outside the window.
     * |        |          |1 = The positive input voltage is in the window.
     * @var ACMP_T::VREF
     * Offset: 0x0C  Analog Comparator Reference Voltage Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |CRVCTL    |Comparator Reference Voltage Setting
     * |        |          |CRV = CRV source voltage * (1/6+CRVCTL/24).
     * |[6]     |CRVSSEL   |CRV Source Voltage Selection
     * |        |          |0 = AVDD is selected as CRV voltage source.
     * |        |          |1 = The reference voltage defined by SYS_VREFCTL register is selected as CRV source voltage.
     */
    __IO uint32_t CTL[2];                /*!< [0x0000~0x0004] Analog Comparator 0/1 Control Register                    */
    __IO uint32_t STATUS;                /*!< [0x0008] Analog Comparator Status Register                                */
    __IO uint32_t VREF;                  /*!< [0x000c] Analog Comparator Reference Voltage Control Register             */

} ACMP_T;

/**
    @addtogroup ACMP_CONST ACMP Bit Field Definition
    Constant Definitions for ACMP Controller
@{ */

#define ACMP_CTL_ACMPEN_Pos             (0)                                                /*!< ACMP_T::CTL: ACMPEN Position          */
#define ACMP_CTL_ACMPEN_Msk             (0x1ul << ACMP_CTL_ACMPEN_Pos)                     /*!< ACMP_T::CTL: ACMPEN Mask              */

#define ACMP_CTL_ACMPIE_Pos             (1)                                                /*!< ACMP_T::CTL: ACMPIE Position          */
#define ACMP_CTL_ACMPIE_Msk             (0x1ul << ACMP_CTL_ACMPIE_Pos)                     /*!< ACMP_T::CTL: ACMPIE Mask              */

#define ACMP_CTL_HYSEN_Pos              (2)                                                /*!< ACMP_T::CTL: HYSEN Position           */
#define ACMP_CTL_HYSEN_Msk              (0x1ul << ACMP_CTL_HYSEN_Pos)                      /*!< ACMP_T::CTL: HYSEN Mask               */

#define ACMP_CTL_ACMPOINV_Pos           (3)                                                /*!< ACMP_T::CTL: ACMPOINV Position        */
#define ACMP_CTL_ACMPOINV_Msk           (0x1ul << ACMP_CTL_ACMPOINV_Pos)                   /*!< ACMP_T::CTL: ACMPOINV Mask            */

#define ACMP_CTL_NEGSEL_Pos             (4)                                                /*!< ACMP_T::CTL: NEGSEL Position          */
#define ACMP_CTL_NEGSEL_Msk             (0x3ul << ACMP_CTL_NEGSEL_Pos)                     /*!< ACMP_T::CTL: NEGSEL Mask              */

#define ACMP_CTL_POSSEL_Pos             (6)                                                /*!< ACMP_T::CTL: POSSEL Position          */
#define ACMP_CTL_POSSEL_Msk             (0x3ul << ACMP_CTL_POSSEL_Pos)                     /*!< ACMP_T::CTL: POSSEL Mask              */

#define ACMP_CTL_INTPOL_Pos             (8)                                                /*!< ACMP_T::CTL: INTPOL Position          */
#define ACMP_CTL_INTPOL_Msk             (0x3ul << ACMP_CTL_INTPOL_Pos)                     /*!< ACMP_T::CTL: INTPOL Mask              */

#define ACMP_CTL_OUTSEL_Pos             (12)                                               /*!< ACMP_T::CTL: OUTSEL Position          */
#define ACMP_CTL_OUTSEL_Msk             (0x1ul << ACMP_CTL_OUTSEL_Pos)                     /*!< ACMP_T::CTL: OUTSEL Mask              */

#define ACMP_CTL_FILTSEL_Pos            (13)                                               /*!< ACMP_T::CTL: FILTSEL Position         */
#define ACMP_CTL_FILTSEL_Msk            (0x7ul << ACMP_CTL_FILTSEL_Pos)                    /*!< ACMP_T::CTL: FILTSEL Mask             */

#define ACMP_CTL_WKEN_Pos               (16)                                               /*!< ACMP_T::CTL: WKEN Position            */
#define ACMP_CTL_WKEN_Msk               (0x1ul << ACMP_CTL_WKEN_Pos)                       /*!< ACMP_T::CTL: WKEN Mask                */

#define ACMP_CTL_WLATEN_Pos             (17)                                               /*!< ACMP_T::CTL: WLATEN Position          */
#define ACMP_CTL_WLATEN_Msk             (0x1ul << ACMP_CTL_WLATEN_Pos)                     /*!< ACMP_T::CTL: WLATEN Mask              */

#define ACMP_CTL_WCMPSEL_Pos            (18)                                               /*!< ACMP_T::CTL: WCMPSEL Position         */
#define ACMP_CTL_WCMPSEL_Msk            (0x1ul << ACMP_CTL_WCMPSEL_Pos)                    /*!< ACMP_T::CTL: WCMPSEL Mask             */

#define ACMP_STATUS_ACMPIF0_Pos          (0)                                               /*!< ACMP_T::STATUS: ACMPIF0 Position       */
#define ACMP_STATUS_ACMPIF0_Msk          (0x1ul << ACMP_STATUS_ACMPIF0_Pos)                /*!< ACMP_T::STATUS: ACMPIF0 Mask           */

#define ACMP_STATUS_ACMPIF1_Pos          (1)                                               /*!< ACMP_T::STATUS: ACMPIF1 Position       */
#define ACMP_STATUS_ACMPIF1_Msk          (0x1ul << ACMP_STATUS_ACMPIF1_Pos)                /*!< ACMP_T::STATUS: ACMPIF1 Mask           */

#define ACMP_STATUS_ACMPO0_Pos           (4)                                               /*!< ACMP_T::STATUS: ACMPO0 Position        */
#define ACMP_STATUS_ACMPO0_Msk           (0x1ul << ACMP_STATUS_ACMPO0_Pos)                 /*!< ACMP_T::STATUS: ACMPO0 Mask            */

#define ACMP_STATUS_ACMPO1_Pos           (5)                                               /*!< ACMP_T::STATUS: ACMPO1 Position        */
#define ACMP_STATUS_ACMPO1_Msk           (0x1ul << ACMP_STATUS_ACMPO1_Pos)                 /*!< ACMP_T::STATUS: ACMPO1 Mask            */

#define ACMP_STATUS_WKIF0_Pos            (8)                                               /*!< ACMP_T::STATUS: WKIF0 Position         */
#define ACMP_STATUS_WKIF0_Msk            (0x1ul << ACMP_STATUS_WKIF0_Pos)                  /*!< ACMP_T::STATUS: WKIF0 Mask             */

#define ACMP_STATUS_WKIF1_Pos            (9)                                               /*!< ACMP_T::STATUS: WKIF1 Position         */
#define ACMP_STATUS_WKIF1_Msk            (0x1ul << ACMP_STATUS_WKIF1_Pos)                  /*!< ACMP_T::STATUS: WKIF1 Mask             */

#define ACMP_STATUS_ACMPS0_Pos           (12)                                              /*!< ACMP_T::STATUS: ACMPS0 Position        */
#define ACMP_STATUS_ACMPS0_Msk           (0x1ul << ACMP_STATUS_ACMPS0_Pos)                 /*!< ACMP_T::STATUS: ACMPS0 Mask            */

#define ACMP_STATUS_ACMPS1_Pos           (13)                                              /*!< ACMP_T::STATUS: ACMPS1 Position        */
#define ACMP_STATUS_ACMPS1_Msk           (0x1ul << ACMP_STATUS_ACMPS1_Pos)                 /*!< ACMP_T::STATUS: ACMPS1 Mask            */

#define ACMP_STATUS_ACMPWO_Pos           (16)                                              /*!< ACMP_T::STATUS: ACMPWO Position        */
#define ACMP_STATUS_ACMPWO_Msk           (0x1ul << ACMP_STATUS_ACMPWO_Pos)                 /*!< ACMP_T::STATUS: ACMPWO Mask            */

#define ACMP_VREF_CRVCTL_Pos             (0)                                               /*!< ACMP_T::VREF: CRVCTL Position          */
#define ACMP_VREF_CRVCTL_Msk             (0xful << ACMP_VREF_CRVCTL_Pos)                   /*!< ACMP_T::VREF: CRVCTL Mask              */

#define ACMP_VREF_CRVSSEL_Pos            (6)                                               /*!< ACMP_T::VREF: CRVSSEL Position         */
#define ACMP_VREF_CRVSSEL_Msk            (0x1ul << ACMP_VREF_CRVSSEL_Pos)                  /*!< ACMP_T::VREF: CRVSSEL Mask             */

/**@}*/ /* ACMP_CONST */
/**@}*/ /* end of ACMP register group */


/*---------------------- Analog to Digital Converter -------------------------*/
/**
    @addtogroup ADC Analog to Digital Converter(ADC)
    Memory Mapped Structure for ADC Controller
@{ */

typedef struct
{


    /**
     * @var ADC_T::ADDR
     * Offset: 0x00~0x4C, 0x74~0x7C  ADC Data Register 0~19, 29~31
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RSLT      |A/D Conversion Result (Read Only)
     * |        |          |This field contains conversion result of ADC.
     * |[16]    |OVERRUN   |Overrun Flag (Read Only)
     * |        |          |If converted data in RSLT bits has not been read before new conversion result is loaded to this register, OVERRUN bit is set to 1.
     * |        |          |It is cleared by hardware after ADDR register is read.
     * |        |          |0 = Data in RSLT bits is not overwrote.
     * |        |          |1 = Data in RSLT bits is overwrote..
     * |[17]    |VALID     |Valid Flag (Read Only)
     * |        |          |This bit will be set to 1 when the conversion of the corresponding channel is completed.
     * |        |          |This bit will be cleared to 0 by hardware after ADDR register is read.
     * |        |          |0 = Data in RSLT bits is not valid.
     * |        |          |1 = Data in RSLT bits is valid.
     * @var ADC_T::ADCR
     * Offset: 0x80  ADC Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ADEN      |A/D Converter Enable
     * |        |          |0 = A/D converter Disabled.
     * |        |          |1 = A/D converter Enabled.
     * |        |          |Note: Before starting A/D conversion function, this bit should be set to 1.
     * |        |          |Clear it to 0 to disable A/D converter analog circuit to save power consumption.
     * |[1]     |ADIE      |A/D Interrupt Enable Control
     * |        |          |A/D conversion end interrupt request is generated if ADIE bit is set to 1.
     * |        |          |0 = A/D interrupt function Disabled.
     * |        |          |1 = A/D interrupt function Enabled.
     * |[3:2]   |ADMD      |A/D Converter Operation Mode Control
     * |        |          |00 = Single conversion.
     * |        |          |01 = Burst conversion.
     * |        |          |10 = Single-cycle Scan.
     * |        |          |11 = Continuous Scan.
     * |        |          |Note1: When changing the operation mode, software should clear ADST bit first.
     * |        |          |Note2: In Burst mode, the A/D result data is always at ADC Data Register 0.
     * |[5:4]   |TRGS      |Hardware Trigger Source
     * |        |          |00 = A/D conversion is started by external STADC pin.
     * |        |          |01 = Timer0 ~ Timer3 overflow pulse trigger.
     * |        |          |10 = Reserved.
     * |        |          |11 = A/D conversion is started by PWM trigger.
     * |        |          |Note: Software should clear TRGEN bit and ADST bit to 0 before changing TRGS bits.
     * |[7:6]   |TRGCOND   |External Trigger Condition
     * |        |          |These two bits decide external pin STADC trigger event is level or edge.
     * |        |          |The signal must be kept at stable state at least 8 PCLKs for level trigger and at least 4 PCLKs for edge trigger.
     * |        |          |00 = Low level.
     * |        |          |01 = High level.
     * |        |          |10 = Falling edge.
     * |        |          |11 = Rising edge.
     * |[8]     |TRGEN     |External Trigger Enable Control
     * |        |          |Enable or disable triggering of A/D conversion by external STADC pin, PWM trigger and Timer trigger.
     * |        |          |If external trigger is enabled, the ADST bit can be set to 1 by the selected hardware trigger source.
     * |        |          |0 = External trigger Disabled.
     * |        |          |1 = External trigger Enabled.
     * |        |          |Note: The ADC external trigger function is only supported in Single-cycle Scan mode.
     * |[9]     |PTEN      |PDMA Transfer Enable Bit
     * |        |          |When A/D conversion is completed, the converted data is loaded into ADDR0~19, ADDR29~ADDR31.
     * |        |          |Software can enable this bit to generate a PDMA data transfer request.
     * |        |          |0 = PDMA data transfer Disabled.
     * |        |          |1 = PDMA data transfer in ADDR0~19, ADDR29~ADDR31 Enabled.
     * |        |          |Note: When PTEN=1, software must set ADIE=0 to disable interrupt.
     * |[10]    |DIFFEN    |Differential Input Mode Control
     * |        |          |0 = Single-end analog input mode.
     * |        |          |1 = Differential analog input mode.
     * |        |          |Differential input voltage (Vdiff) = Vplus - Vminus,
     * |        |          |where Vplus is the analog input; Vminus is the inverted analog input.
     * |        |          |The Vplus of differential input paired channel x is from ADC0_CHy pin; Vminus is from ADC0_CHz pin, x=0,1..9, y=2*x, z=y+1.
     * |        |          |0 = Single-end analog input mode.
     * |        |          |1 = Differential analog input mode.
     * |        |          |Note: In Differential Input mode, only the even number of the two corresponding channels needs to be enabled in ADCHER register
     * |        |          |The conversion result will be placed to the corresponding data register of the enabled channel.
     * |[11]    |ADST      |A/D Conversion Start
     * |        |          |ADST bit can be set to 1 from four sources: software, external pin STADC, PWM trigger and Timer trigger.
     * |        |          |ADST bit will be cleared to 0 by hardware automatically at the ends of Single mode and Single-cycle Scan mode.
     * |        |          |In Continuous Scan mode and Burst mode, A/D conversion is continuously performed until software writes 0 to this bit or chip is reset.
     * |        |          |0 = Conversion stops and A/D converter enters idle state.
     * |        |          |1 = Conversion starts.
     * |[18:16] |SMPTSEL   |ADC Internal Sampling Time Selection
     * |        |          |Total ADC conversion cycle = sampling cycle + 12
     * |        |          |000 = 4 ADC clock for sampling; 16 ADC clock for complete conversion.
     * |        |          |001 = 5 ADC clock for sampling; 17 ADC clock for complete conversion.
     * |        |          |010 = 6 ADC clock for sampling; 18 ADC clock for complete conversion.
     * |        |          |011 = 7 ADC clock for sampling; 19 ADC clock for complete conversion.
     * |        |          |100 = 8 ADC clock for sampling; 20 ADC clock for complete conversion.
     * |        |          |101 = 9 ADC clock for sampling; 21 ADC clock for complete conversion.
     * |        |          |110 = 10 ADC clock for sampling; 22 ADC clock for complete conversion.
     * |        |          |111 = 11 ADC clock for sampling; 23 ADC clock for complete conversion.
     * |[31]    |DMOF      |Differential Input Mode Output Format
     * |        |          |If user enables differential input mode, the conversion result can be expressed with binary straight format (unsigned format) or 2's complement format (signed format).
     * |        |          |0 = A/D Conversion result will be filled in RSLT at ADDRx registers with unsigned format (straight binary format).
     * |        |          |1 = A/D Conversion result will be filled in RSLT at ADDRx registers with 2's complement format.
     * @var ADC_T::ADCHER
     * Offset: 0x84  ADC Channel Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |CHEN      |Analog Input Channel Enable Control
     * |        |          |Set ADCHENR[19:0] bits to enable the corresponding analog input channel 19 ~ 0
     * |        |          |If DIFFEN bit is set to 1, only the even number channel needs to be enabled.
     * |        |          |Besides, set ADCHENR[29] to ADCHENR[31] bits will enable internal channel for band-gap voltage, temperature sensor and battery power respectively.
     * |        |          |Other bits are reserved.
     * |        |          |0 = Channel Disabled.
     * |        |          |1 = Channel Enabled.
     * |        |          |Note 1 : If the internal channel for band-gap voltage (CHEN[29]) is active, the maximum sampling rate will be 300k SPS.
     * |        |          |Note 2 : If the internal channel for temperature sensor (CHEN[30]) is active, the maximum sampling rate will be 300k SPS.
     * @var ADC_T::ADCMPR
     * Offset: 0x88/0x8C  ADC Compare Register 0/1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CMPEN     |Compare Enable Control
     * |        |          |Set this bit to 1 to enable ADC controller to compare CMPD (ADCMPRx[27:16]) with specified channel conversion result when converted data is loaded into ADDR register.
     * |        |          |0 = Compare function Disabled.
     * |        |          |1 = Compare function Enabled.
     * |[1]     |CMPIE     |Compare Interrupt Enable Control
     * |        |          |If the compare function is enabled and the compare condition matches the setting of CMPCOND and CMPMATCNT, CMPFx bit will be asserted, in the meanwhile, if CMPIE bit is set to 1, a compare interrupt request is generated.
     * |        |          |0 = Compare function interrupt Disabled.
     * |        |          |1 = Compare function interrupt Enabled.
     * |[2]     |CMPCOND   |Compare Condition
     * |        |          |0 = Set the compare condition as that when a 12-bit A/D conversion result is less than the 12-bit CMPD (ADCMPRx[27:16])bits, the internal match counter will increase one.
     * |        |          |1 = Set the compare condition as that when a 12-bit A/D conversion result is greater than or equal to the 12-bit CMPD (ADCMPRx[27:16])bits, the internal match counter will increase one.
     * |        |          |Note: When the internal counter reaches to (CMPMATCNT +1), the CMPFx bit will be set.
     * |[7:3]   |CMPCH     |Compare Channel Selection
     * |        |          |00000 = Channel 0 conversion result is selected to be compared.
     * |        |          |00001 = Channel 1 conversion result is selected to be compared.
     * |        |          |00010 = Channel 2 conversion result is selected to be compared.
     * |        |          |00011 = Channel 3 conversion result is selected to be compared.
     * |        |          |00100 = Channel 4 conversion result is selected to be compared.
     * |        |          |00101 = Channel 5 conversion result is selected to be compared.
     * |        |          |00110 = Channel 6 conversion result is selected to be compared.
     * |        |          |00111 = Channel 7 conversion result is selected to be compared.
     * |        |          |01000 = Channel 8 conversion result is selected to be compared.
     * |        |          |01001 = Channel 9 conversion result is selected to be compared.
     * |        |          |01010 = Channel 10 conversion result is selected to be compared.
     * |        |          |01011 = Channel 11 conversion result is selected to be compared.
     * |        |          |01100 = Channel 12 conversion result is selected to be compared.
     * |        |          |01101 = Channel 13 conversion result is selected to be compared.
     * |        |          |01110 = Channel 14 conversion result is selected to be compared.
     * |        |          |01111 = Channel 15 conversion result is selected to be compared.
     * |        |          |10000 = Channel 16 conversion result is selected to be compared.
     * |        |          |10001 = Channel 17 conversion result is selected to be compared.
     * |        |          |10010 = Channel 18 conversion result is selected to be compared.
     * |        |          |10011 = Channel 19 conversion result is selected to be compared.
     * |        |          |11101 = Band-gap voltage conversion result is selected to be compared.
     * |        |          |11110 = Temperature sensor conversion result is selected to be compared.
     * |        |          |11111 = Battery power conversion result is selected to be compared.
     * |        |          |Others = Reserved.
     * |[11:8]  |CMPMATCNT |Compare Match Count
     * |        |          |When the specified A/D channel analog conversion result matches the compare condition defined by CMPCOND bit, the internal match counter will increase 1.
     * |        |          |When the internal counter reaches the value to (CMPMATCNT +1), the CMPFx bit will be set.
     * |[15]    |CMPWEN    |Compare Window Mode Enable Bit
     * |        |          |0 = Compare Window Mode Disabled.
     * |        |          |1 = Compare Window Mode Enabled.
     * |        |          |Note: This bit is only presented in ADCMPR0 register.
     * |[27:16] |CMPD      |Comparison Data
     * |        |          |The 12-bit data is used to compare with conversion result of specified channel.
     * |        |          |Note: CMPD bits should be filled in unsigned format (straight binary format).
     * @var ADC_T::ADSR0
     * Offset: 0x90  ADC Status Register0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ADF       |A/D Conversion End Flag
     * |        |          |A status flag that indicates the end of A/D conversion. Software can write 1 to clear this bit.
     * |        |          |ADF bit is set to 1 at the following three conditions:
     * |        |          |1. When A/D conversion ends in Single mode.
     * |        |          |2. When A/D conversion ends on all specified channels in Single-cycle Scan mode and Continuous Scan mode.
     * |        |          |3. When more than or equal to 8 samples in FIFO in Burst mode.
     * |[1]     |CMPF0     |Compare Flag 0
     * |        |          |When the A/D conversion result of the selected channel meets setting condition in ADCMPR0 register then this bit is set to 1.
     * |        |          |This bit is cleared by writing 1 to it.
     * |        |          |0 = Conversion result in ADDR does not meet ADCMPR0 setting.
     * |        |          |1 = Conversion result in ADDR meets ADCMPR0 setting.
     * |[2]     |CMPF1     |Compare Flag 1
     * |        |          |When the A/D conversion result of the selected channel meets setting condition in ADCMPR1 register then this bit is set to 1; it is cleared by writing 1 to it.
     * |        |          |0 = Conversion result in ADDR does not meet ADCMPR1 setting.
     * |        |          |1 = Conversion result in ADDR meets ADCMPR1 setting.
     * |[7]     |BUSY      |BUSY/IDLE (Read Only)
     * |        |          |This bit is a mirror of ADST bit in ADCR register.
     * |        |          |0 = A/D converter is in idle state.
     * |        |          |1 = A/D converter is busy at conversion.
     * |[8]     |VALIDF    |Data Valid Flag (Read Only)
     * |        |          |If any one of VALID (ADDRx[17]) is set, this flag will be set to 1.
     * |        |          |Note: When ADC is in burst mode and any conversion result is valid, this flag will be set to 1.
     * |[16]    |OVERRUNF  |Overrun Flag (Read Only)
     * |        |          |If any one of OVERRUN (ADDRx[16]) is set, this flag will be set to 1.
     * |        |          |Note: When ADC is in burst mode and the FIFO is overrun, this flag will be set to 1.
     * |[31:27] |CHANNEL   |Current Conversion Channel (Read Only)
     * |        |          |When BUSY=1, this filed reflects current conversion channel.
     * |        |          |When BUSY=0, it shows the number of the next converted channel.
     * @var ADC_T::ADSR1
     * Offset: 0x94  ADC Status Register1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |VALID     |Data Valid Flag (Read Only)
     * |        |          |VALID[31:29, 19:0] are the mirror of the VALID bits in ADDR31[17] ~ ADDR29[17], ADDR19[17]~ ADDR0[17].
     * |        |          |The other bits are reserved.
     * |        |          |Note: When ADC is in burst mode and any conversion result is valid, VALID[31:29, 19:0] will be set to 1.
     * @var ADC_T::ADSR2
     * Offset: 0x98  ADC Status Register2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |OVERRUN   |Overrun Flag (Read Only)
     * |        |          |OVERRUN[31:29, 19:0] are the mirror of the OVERRUN bit in ADDR31[16] ~ADDR29[16], ADDR19[16] ~ ADDR0[16].
     * |        |          |The other bits are reserved.
     * |        |          |Note: When ADC is in burst mode and the FIFO is overrun, OVERRUN[31:29, 19:0] will be set to 1.
     * @var ADC_T::ADTDCR
     * Offset: 0x9C  ADC Trigger Delay Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |PTDT      |PWM Trigger Delay Time
     * |        |          |Set this field will delay ADC start conversion time after PWM trigger.
     * |        |          |PWM trigger delay time is (4 * PTDT) * system clock
     * @var ADC_T::ADPDMA
     * Offset: 0x100  ADC PDMA Current Transfer Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[17:0]  |CURDAT    |ADC PDMA Current Transfer Data Register (Read Only)
     * |        |          |When PDMA transferring, read this register can monitor current PDMA transfer data.
     * |        |          |Current PDMA transfer data could be the content of ADDR0 ~ ADDR19 and ADDR29 ~ ADDR31 registers.
     */

    __I  uint32_t ADDR[32];              /*!< [0x0000 ~ 0x007c] ADC Data Register 31                                    */
    __IO uint32_t ADCR;                  /*!< [0x0080] ADC Control Register                                             */
    __IO uint32_t ADCHER;                /*!< [0x0084] ADC Channel Enable Register                                      */
    __IO uint32_t ADCMPR[2];             /*!< [0x0088 ~ 0x008C] ADC Compare Register 0 & 1                              */
    __IO uint32_t ADSR0;                 /*!< [0x0090] ADC Status Register0                                             */
    __I  uint32_t ADSR1;                 /*!< [0x0094] ADC Status Register1                                             */
    __I  uint32_t ADSR2;                 /*!< [0x0098] ADC Status Register2                                             */
    __IO uint32_t ADTDCR;                /*!< [0x009c] ADC Trigger Delay Control Register                               */
    __I  uint32_t RESERVE0[24];
    __I  uint32_t ADPDMA;                /*!< [0x0100] ADC PDMA Current Transfer Data Register                          */

} ADC_T;

/**
    @addtogroup ADC_CONST ADC Bit Field Definition
    Constant Definitions for ADC Controller
@{ */

#define ADC_ADDR_RSLT_Pos                (0)                                               /*!< ADC_T::ADDR: RSLT Position             */
#define ADC_ADDR_RSLT_Msk                (0xfffful << ADC_ADDR_RSLT_Pos)                   /*!< ADC_T::ADDR: RSLT Mask                 */

#define ADC_ADDR_OVERRUN_Pos             (16)                                              /*!< ADC_T::ADDR: OVERRUN Position          */
#define ADC_ADDR_OVERRUN_Msk             (0x1ul << ADC_ADDR_OVERRUN_Pos)                   /*!< ADC_T::ADDR: OVERRUN Mask              */

#define ADC_ADDR_VALID_Pos               (17)                                              /*!< ADC_T::ADDR: VALID Position            */
#define ADC_ADDR_VALID_Msk               (0x1ul << ADC_ADDR_VALID_Pos)                     /*!< ADC_T::ADDR: VALID Mask                */

#define ADC_ADCR_ADEN_Pos                (0)                                               /*!< ADC_T::ADCR: ADEN Position             */
#define ADC_ADCR_ADEN_Msk                (0x1ul << ADC_ADCR_ADEN_Pos)                      /*!< ADC_T::ADCR: ADEN Mask                 */

#define ADC_ADCR_ADIE_Pos                (1)                                               /*!< ADC_T::ADCR: ADIE Position             */
#define ADC_ADCR_ADIE_Msk                (0x1ul << ADC_ADCR_ADIE_Pos)                      /*!< ADC_T::ADCR: ADIE Mask                 */

#define ADC_ADCR_ADMD_Pos                (2)                                               /*!< ADC_T::ADCR: ADMD Position             */
#define ADC_ADCR_ADMD_Msk                (0x3ul << ADC_ADCR_ADMD_Pos)                      /*!< ADC_T::ADCR: ADMD Mask                 */

#define ADC_ADCR_TRGS_Pos                (4)                                               /*!< ADC_T::ADCR: TRGS Position             */
#define ADC_ADCR_TRGS_Msk                (0x3ul << ADC_ADCR_TRGS_Pos)                      /*!< ADC_T::ADCR: TRGS Mask                 */

#define ADC_ADCR_TRGCOND_Pos             (6)                                               /*!< ADC_T::ADCR: TRGCOND Position          */
#define ADC_ADCR_TRGCOND_Msk             (0x3ul << ADC_ADCR_TRGCOND_Pos)                   /*!< ADC_T::ADCR: TRGCOND Mask              */

#define ADC_ADCR_TRGEN_Pos               (8)                                               /*!< ADC_T::ADCR: TRGEN Position            */
#define ADC_ADCR_TRGEN_Msk               (0x1ul << ADC_ADCR_TRGEN_Pos)                     /*!< ADC_T::ADCR: TRGEN Mask                */

#define ADC_ADCR_PTEN_Pos                (9)                                               /*!< ADC_T::ADCR: PTEN Position             */
#define ADC_ADCR_PTEN_Msk                (0x1ul << ADC_ADCR_PTEN_Pos)                      /*!< ADC_T::ADCR: PTEN Mask                 */

#define ADC_ADCR_DIFFEN_Pos              (10)                                              /*!< ADC_T::ADCR: DIFFEN Position           */
#define ADC_ADCR_DIFFEN_Msk              (0x1ul << ADC_ADCR_DIFFEN_Pos)                    /*!< ADC_T::ADCR: DIFFEN Mask               */

#define ADC_ADCR_ADST_Pos                (11)                                              /*!< ADC_T::ADCR: ADST Position             */
#define ADC_ADCR_ADST_Msk                (0x1ul << ADC_ADCR_ADST_Pos)                      /*!< ADC_T::ADCR: ADST Mask                 */

#define ADC_ADCR_SMPTSEL_Pos             (16)                                              /*!< ADC_T::ADCR: SMPTSEL Position          */
#define ADC_ADCR_SMPTSEL_Msk             (0x7ul << ADC_ADCR_SMPTSEL_Pos)                   /*!< ADC_T::ADCR: SMPTSEL Mask              */

#define ADC_ADCR_DMOF_Pos                (31)                                              /*!< ADC_T::ADCR: DMOF Position             */
#define ADC_ADCR_DMOF_Msk                (0x1ul << ADC_ADCR_DMOF_Pos)                      /*!< ADC_T::ADCR: DMOF Mask                 */

#define ADC_ADCHER_CHEN_Pos              (0)                                               /*!< ADC_T::ADCHER: CHEN Position           */
#define ADC_ADCHER_CHEN_Msk              (0xfffffffful << ADC_ADCHER_CHEN_Pos)             /*!< ADC_T::ADCHER: CHEN Mask               */

#define ADC_ADCMPR_CMPEN_Pos             (0)                                               /*!< ADC_T::ADCMPR: CMPEN Position          */
#define ADC_ADCMPR_CMPEN_Msk             (0x1ul << ADC_ADCMPR_CMPEN_Pos)                   /*!< ADC_T::ADCMPR: CMPEN Mask              */

#define ADC_ADCMPR_CMPIE_Pos             (1)                                               /*!< ADC_T::ADCMPR: CMPIE Position          */
#define ADC_ADCMPR_CMPIE_Msk             (0x1ul << ADC_ADCMPR_CMPIE_Pos)                   /*!< ADC_T::ADCMPR: CMPIE Mask              */

#define ADC_ADCMPR_CMPCOND_Pos           (2)                                               /*!< ADC_T::ADCMPR: CMPCOND Position        */
#define ADC_ADCMPR_CMPCOND_Msk           (0x1ul << ADC_ADCMPR_CMPCOND_Pos)                 /*!< ADC_T::ADCMPR: CMPCOND Mask            */

#define ADC_ADCMPR_CMPCH_Pos             (3)                                               /*!< ADC_T::ADCMPR: CMPCH Position          */
#define ADC_ADCMPR_CMPCH_Msk             (0x1ful << ADC_ADCMPR_CMPCH_Pos)                  /*!< ADC_T::ADCMPR: CMPCH Mask              */

#define ADC_ADCMPR_CMPMATCNT_Pos         (8)                                               /*!< ADC_T::ADCMPR: CMPMATCNT Position      */
#define ADC_ADCMPR_CMPMATCNT_Msk         (0xful << ADC_ADCMPR_CMPMATCNT_Pos)               /*!< ADC_T::ADCMPR: CMPMATCNT Mask          */

#define ADC_ADCMPR_CMPWEN_Pos            (15)                                              /*!< ADC_T::ADCMPR: CMPWEN Position         */
#define ADC_ADCMPR_CMPWEN_Msk            (0x1ul << ADC_ADCMPR_CMPWEN_Pos)                  /*!< ADC_T::ADCMPR: CMPWEN Mask             */

#define ADC_ADCMPR_CMPD_Pos              (16)                                              /*!< ADC_T::ADCMPR: CMPD Position           */
#define ADC_ADCMPR_CMPD_Msk              (0xffful << ADC_ADCMPR_CMPD_Pos)                  /*!< ADC_T::ADCMPR: CMPD Mask               */

#define ADC_ADSR0_ADF_Pos                (0)                                               /*!< ADC_T::ADSR0: ADF Position             */
#define ADC_ADSR0_ADF_Msk                (0x1ul << ADC_ADSR0_ADF_Pos)                      /*!< ADC_T::ADSR0: ADF Mask                 */

#define ADC_ADSR0_CMPF0_Pos              (1)                                               /*!< ADC_T::ADSR0: CMPF0 Position           */
#define ADC_ADSR0_CMPF0_Msk              (0x1ul << ADC_ADSR0_CMPF0_Pos)                    /*!< ADC_T::ADSR0: CMPF0 Mask               */

#define ADC_ADSR0_CMPF1_Pos              (2)                                               /*!< ADC_T::ADSR0: CMPF1 Position           */
#define ADC_ADSR0_CMPF1_Msk              (0x1ul << ADC_ADSR0_CMPF1_Pos)                    /*!< ADC_T::ADSR0: CMPF1 Mask               */

#define ADC_ADSR0_BUSY_Pos               (7)                                               /*!< ADC_T::ADSR0: BUSY Position            */
#define ADC_ADSR0_BUSY_Msk               (0x1ul << ADC_ADSR0_BUSY_Pos)                     /*!< ADC_T::ADSR0: BUSY Mask                */

#define ADC_ADSR0_VALIDF_Pos             (8)                                               /*!< ADC_T::ADSR0: VALIDF Position          */
#define ADC_ADSR0_VALIDF_Msk             (0x1ul << ADC_ADSR0_VALIDF_Pos)                   /*!< ADC_T::ADSR0: VALIDF Mask              */

#define ADC_ADSR0_OVERRUNF_Pos           (16)                                              /*!< ADC_T::ADSR0: OVERRUNF Position        */
#define ADC_ADSR0_OVERRUNF_Msk           (0x1ul << ADC_ADSR0_OVERRUNF_Pos)                 /*!< ADC_T::ADSR0: OVERRUNF Mask            */

#define ADC_ADSR0_CHANNEL_Pos            (27)                                              /*!< ADC_T::ADSR0: CHANNEL Position         */
#define ADC_ADSR0_CHANNEL_Msk            (0x1ful << ADC_ADSR0_CHANNEL_Pos)                 /*!< ADC_T::ADSR0: CHANNEL Mask             */

#define ADC_ADSR1_VALID_Pos              (0)                                               /*!< ADC_T::ADSR1: VALID Position           */
#define ADC_ADSR1_VALID_Msk              (0xfffffffful << ADC_ADSR1_VALID_Pos)             /*!< ADC_T::ADSR1: VALID Mask               */

#define ADC_ADSR2_OVERRUN_Pos            (0)                                               /*!< ADC_T::ADSR2: OVERRUN Position         */
#define ADC_ADSR2_OVERRUN_Msk            (0xfffffffful << ADC_ADSR2_OVERRUN_Pos)           /*!< ADC_T::ADSR2: OVERRUN Mask             */

#define ADC_ADTDCR_PTDT_Pos              (0)                                               /*!< ADC_T::ADTDCR: PTDT Position           */
#define ADC_ADTDCR_PTDT_Msk              (0xfful << ADC_ADTDCR_PTDT_Pos)                   /*!< ADC_T::ADTDCR: PTDT Mask               */

#define ADC_ADPDMA_CURDAT_Pos            (0)                                               /*!< ADC_T::ADPDMA: CURDAT Position         */
#define ADC_ADPDMA_CURDAT_Msk            (0x3fffful << ADC_ADPDMA_CURDAT_Pos)              /*!< ADC_T::ADPDMA: CURDAT Mask             */

/**@}*/ /* ADC_CONST */
/**@}*/ /* end of ADC register group */


/*---------------------- System Clock Controller -------------------------*/
/**
    @addtogroup CLK System Clock Controller(CLK)
    Memory Mapped Structure for CLK Controller
@{ */

typedef struct
{


    /**
     * @var CLK_T::PWRCTL
     * Offset: 0x00  System Power-down Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |HXTEN     |HXT Enable Bit (Write Protect)
     * |        |          |The bit default value is set by flash controller user configuration register CONFIG0 [26:24].
     * |        |          |When the default clock source is from HXT, this bit is set to 1 automatically.
     * |        |          |0 = 4~24 MHz External High Speed Crystal (HXT) Disabled.
     * |        |          |1 = 4~24 MHz External High Speed Crystal (HXT) Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[1]     |LXTEN     |LXT Enable Bit (Write Protect)
     * |        |          |0 = 32.768 KHz External Low Speed Crystal (LXT) Disabled.
     * |        |          |1 = 32.768 KHz External Low Speed Crystal (LXT) Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[2]     |HIRCEN    |HIRC Enable Bit (Write Protect)
     * |        |          |0 = 22.1184 MHz internal high speed RC oscillator (HIRC) Disabled.
     * |        |          |1 = 22.1184 MHz internal high speed RC oscillator (HIRC) Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[3]     |LIRCEN    |LIRC Enable Bit (Write Protect)
     * |        |          |0 = 10 kHz internal low speed RC oscillator (LIRC) Disabled.
     * |        |          |1 = 10 kHz internal low speed RC oscillator (LIRC) Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[4]     |PDWKDLY   |Enable the Wake-up Delay Counter (Write Protect)
     * |        |          |When the chip wakes up from Power-down mode, the clock control will delay certain clock cycles to wait system clock stable.
     * |        |          |The delayed clock cycle is 4096 clock cycles when chip work at 4~24 MHz external high speed crystal oscillator (HXT), 256 clock cycles when chip work at 22.1184 MHz internal high speed RC oscillator (HIRC) and 512 clock cycles when chip work at 48 MHz internal high speed RC oscillator (HIRC48).
     * |        |          |0 = Clock cycles delay Disabled.
     * |        |          |1 = Clock cycles delay Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[5]     |PDWKIEN   |Power-down Mode Wake-up Interrupt Enable Bit (Write Protect)
     * |        |          |0 = Power-down mode wake-up interrupt Disabled.
     * |        |          |1 = Power-down mode wake-up interrupt Enabled.
     * |        |          |Note1: The interrupt will occur when both PDWKIF and PDWKIEN are high.
     * |        |          |Note2: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[6]     |PDWKIF    |Power-down Mode Wake-up Interrupt Status
     * |        |          |Set by u201CPower-down wake-up eventu201D, it indicates that resume from Power-down modeu201D
     * |        |          |The flag is set if the EINT0~5, GPIO, USBD, UART0~2, WDT, ACMP01, BOD, EBOD, RTC, TMR0~3, I2C0~1 or USCI0~2 wake-up occurred.
     * |        |          |Note1: Write 1 to clear the bit to 0.
     * |        |          |Note2: This bit works only if PDWKIEN (CLK_PWRCTL[5]) set to 1.
     * |[7]     |PDEN      |System Power-down Enable (Write Protect)
     * |        |          |When this bit is set to 1, Power-down mode is enabled and chip keeps active till the CPU sleep mode is also active and then the chip enters Power-down mode.
     * |        |          |When chip wakes up from Power-down mode, this bit is auto cleared.
     * |        |          |Users need to set this bit again for next Power-down.
     * |        |          |In Power-down mode, HXT, HIRC and the HIRC48 will be disabled in this mode, but LXT and LIRC are not controlled by Power-down mode.
     * |        |          |In Power-down mode, the PLL and system clock are disabled, and ignored the clock source selection.
     * |        |          |The clocks of peripheral are not controlled by Power-down mode, if the peripheral clock source is from LXT or LIRC.
     * |        |          |0 = Chip operating normally or chip in idle mode because of WFI command.
     * |        |          |1 = Chip waits CPU sleep command WFI and then enters Power-down mode.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[11:10] |HXTGAIN   |HXT Gain Control Bit (Write Protect)
     * |        |          |Gain control is used to enlarge the gain of crystal to make sure crystal work normally.
     * |        |          |If gain control is enabled, crystal will consume more power than gain control off.
     * |        |          |00 = HXT frequency is lower than from 8 MHz.
     * |        |          |01 = HXT frequency is from 8 MHz to 12 MHz.
     * |        |          |10 = HXT frequency is from 12 MHz to 16 MHz.
     * |        |          |11 = HXT frequency is higher than 16 MHz.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[12]    |HXTSELTYP |HXT Crystal Type Select Bit (Write Protect)
     * |        |          |This is a protected register. Please refer to open lock sequence to program it.
     * |        |          |0 = Select INV type.
     * |        |          |1 = Select GM type.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[13]    |HIRC48EN  |HIRC48 Enable Bit (Write Protect)
     * |        |          |0 = 48 MHz internal high speed RC oscillator (HIRC48) Disabled.
     * |        |          |1 = 48 MHz internal high speed RC oscillator (HIRC48) Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * @var CLK_T::AHBCLK
     * Offset: 0x04  AHB Devices Clock Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1]     |PDMACKEN  |PDMA Controller Clock Enable Bit
     * |        |          |0 = PDMA peripheral clock Disabled.
     * |        |          |1 = PDMA peripheral clock Enabled.
     * |[2]     |ISPCKEN   |Flash ISP Controller Clock Enable Bit
     * |        |          |0 = Flash ISP peripheral clock Disabled.
     * |        |          |1 = Flash ISP peripheral clock Enabled.
     * |[3]     |EBICKEN   |EBI Controller Clock Enable Bit
     * |        |          |0 = EBI peripheral clock Disabled.
     * |        |          |1 = EBI peripheral clock Enabled.
     * |[4]     |HDIVCKEN  |Hardware Divider Controller Clock Enable Bit
     * |        |          |0 = Hardware divider peripheral clock Disabled.
     * |        |          |1 = Hardware divider peripheral clock Enabled.
     * |[7]     |CRCCKEN   |CRC Generator Controller Clock Enable Bit
     * |        |          |0 = CRC peripheral clock Disabled.
     * |        |          |1 = CRC peripheral clock Enabled.
     * |[15]    |FMCIDLE   |Flash Memory Controller Clock Enable Bit in IDLE Mode
     * |        |          |0 = FMC peripheral clock Disabled when chip operating at IDLE mode.
     * |        |          |1 = FMC peripheral clock Enabled when chip operating at IDLE mode.
     * |[16]    |GPIOACKEN |General Purpose I/O PA Group Clock Enable Bit
     * |        |          |0 = GPIO PA group clock Disabled.
     * |        |          |1 = GPIO PA group clock Enabled.
     * |[17]    |GPIOBCKEN |General Purpose I/O PB Group Clock Enable Bit
     * |        |          |0 = GPIO PB group clock Disabled.
     * |        |          |1 = GPIO PB group clock Enabled.
     * |[18]    |GPIOCCKEN |General Purpose I/O PC Group Clock Enable Bit
     * |        |          |0 = GPIO PC group clock Disabled.
     * |        |          |1 = GPIO PC group clock Enabled.
     * |[19]    |GPIODCKEN |General Purpose I/O PD Group Clock Enable Bit
     * |        |          |0 = GPIO PD group clock Disabled.
     * |        |          |1 = GPIO PD group clock Enabled.
     * |[20]    |GPIOECKEN |General Purpose I/O PE Group Clock Enable Bit
     * |        |          |0 = GPIO PE group clock Disabled.
     * |        |          |1 = GPIO PE group clock Enabled.
     * |[21]    |GPIOFCKEN |General Purpose I/O PF Group Clock Enable Bit
     * |        |          |0 = GPIO PF group clock Disabled.
     * |        |          |1 = GPIO PF group clock Enabled.
     * @var CLK_T::APBCLK0
     * Offset: 0x08  APB Devices Clock Enable Control Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |WDTCKEN   |Watchdog Timer Clock Enable Bit (Write Protect)
     * |        |          |0 = Watchdog Timer Clock Disabled.
     * |        |          |1 = Watchdog Timer Clock Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[1]     |RTCCKEN   |Real-time-clock APB Interface Clock Enable Bit
     * |        |          |This bit is used to control the RTC APB clock only
     * |        |          |The RTC peripheral clock source is selected from RTCSEL(CLK_CLKSEL2[18]).
     * |        |          |It can be selected to external 32.768 kHz low speed crystal (LXT) or 10 kHz internal low speed RC oscillator (LIRC).
     * |        |          |0 = RTC Clock Disabled.
     * |        |          |1 = RTC Clock Enabled.
     * |[2]     |TMR0CKEN  |Timer0 Clock Enable Bit
     * |        |          |0 = Timer0 Clock Disabled.
     * |        |          |1 = Timer0 Clock Enabled.
     * |[3]     |TMR1CKEN  |Timer1 Clock Enable Bit
     * |        |          |0 = Timer1 Clock Disabled.
     * |        |          |1 = Timer1 Clock Enabled.
     * |[4]     |TMR2CKEN  |Timer2 Clock Enable Bit
     * |        |          |0 = Timer2 Clock Disabled.
     * |        |          |1 = Timer2 Clock Enabled.
     * |[5]     |TMR3CKEN  |Timer3 Clock Enable Bit
     * |        |          |0 = Timer3 Clock Disabled.
     * |        |          |1 = Timer3 Clock Enabled.
     * |[6]     |CLKOCKEN  |CLKO Clock Enable Bit
     * |        |          |0 = CLKO Clock Disabled.
     * |        |          |1 = CLKO Clock Enabled.
     * |[8]     |I2C0CKEN  |I2C0 Clock Enable Bit
     * |        |          |0 = I2C0 Clock Disabled.
     * |        |          |1 = I2C0 Clock Enabled.
     * |[9]     |I2C1CKEN  |I2C1 Clock Enable Bit
     * |        |          |0 = I2C1 Clock Disabled.
     * |        |          |1 = I2C1 Clock Enabled.
     * |[12]    |SPI0CKEN  |SPI0 Clock Enable Bit
     * |        |          |0 = SPI0 Clock Disabled.
     * |        |          |1 = SPI0 Clock Enabled.
     * |[13]    |SPI1CKEN  |SPI1 Clock Enable Bit
     * |        |          |0 = SPI1 Clock Disabled.
     * |        |          |1 = SPI1 Clock Enabled.
     * |[16]    |UART0CKEN |UART0 Clock Enable Bit
     * |        |          |0 = UART0 clock Disabled.
     * |        |          |1 = UART0 clock Enabled.
     * |[17]    |UART1CKEN |UART1 Clock Enable Bit
     * |        |          |0 = UART1 clock Disabled.
     * |        |          |1 = UART1 clock Enabled.
     * |[18]    |UART2CKEN |UART2 Clock Enable Bit
     * |        |          |0 = UART2 clock Disabled.
     * |        |          |1 = UART2 clock Enabled.
     * |[20]    |PWM0CKEN  |PWM0 Clock Enable Bit
     * |        |          |0 = PWM0 clock Disabled.
     * |        |          |1 = PWM0 clock Enabled.
     * |[21]    |PWM1CKEN  |PWM1 Clock Enable Bit
     * |        |          |0 = PWM1 clock Disabled.
     * |        |          |1 = PWM1 clock Enabled.
     * |[27]    |USBDCKEN  |USB Device Clock Enable Bit
     * |        |          |0 = USB Device clock Disabled.
     * |        |          |1 = USB Device clock Enabled.
     * |[28]    |ADCCKEN   |Analog-digital-converter (ADC) Clock Enable Bit
     * |        |          |0 = ADC clock Disabled.
     * |        |          |1 = ADC clock Enabled.
     * |[30]    |ACMP01CKEN|Analog Comparator 0/1 Clock Enable Bit
     * |        |          |0 = Analog Comparator 0/1 clock Disabled.
     * |        |          |1 = Analog Comparator 0/1 clock Enabled.
     * @var CLK_T::STATUS
     * Offset: 0x0C  Clock Status Monitor Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |HXTSTB    |HXT Clock Source Stable Flag (Read Only)
     * |        |          |0 = 4~24 MHz external high speed crystal oscillator (HXT) clock is not stable or disabled.
     * |        |          |1 = 4~24 MHz external high speed crystal oscillator (HXT) clock is stable and enabled.
     * |[1]     |LXTSTB    |LXT Clock Source Stable Flag (Read Only)
     * |        |          |0 = 32.768 kHz external low speed crystal oscillator (LXT) clock is not stable or disabled.
     * |        |          |1 = 32.768 kHz external low speed crystal oscillator (LXT) clock is stabled and enabled.
     * |[2]     |PLLSTB    |Internal PLL Clock Source Stable Flag (Read Only)
     * |        |          |0 = Internal PLL clock is not stable or disabled.
     * |        |          |1 = Internal PLL clock is stable and enabled.
     * |[3]     |LIRCSTB   |LIRC Clock Source Stable Flag (Read Only)
     * |        |          |0 = 10 kHz internal low speed RC oscillator (LIRC) clock is not stable or disabled.
     * |        |          |1 = 10 kHz internal low speed RC oscillator (LIRC) clock is stable and enabled.
     * |[4]     |HIRCSTB   |HIRC Clock Source Stable Flag (Read Only)
     * |        |          |0 = 22.1184 MHz internal high speed RC oscillator (HIRC) clock is not stable or disabled.
     * |        |          |1 = 22.1184 MHz internal high speed RC oscillator (HIRC) clock is stable and enabled.
     * |[5]     |HIRC48STB |HIRC48 Clock Source Stable Flag (Read Only)
     * |        |          |0 = 48 MHz internal high speed RC oscillator (HIRC48) clock is not stable or disabled.
     * |        |          |1 = 48 MHz internal high speed RC oscillator (HIRC48) clock is stable and enabled.
     * |[7]     |CLKSFAIL  |Clock Switching Fail Flag (Read Only)
     * |        |          |This bit is updated when software switches system clock source.
     * |        |          |If switch target clock is stable, this bit will be set to 0.
     * |        |          |If switch target clock is not stable, this bit will be set to 1.
     * |        |          |0 = Clock switching success.
     * |        |          |1 = Clock switching failure.
     * |        |          |Note: This bit is read only.
     * |        |          |After selected clock source is stable, hardware will switch system clock to selected clock automatically, and CLKSFAIL will be cleared automatically by hardware.
     * @var CLK_T::CLKSEL0
     * Offset: 0x10  Clock Source Select Control Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |HCLKSEL   |HCLK Clock Source Selection (Write Protect)
     * |        |          |Before clock switching, the related clock sources (both pre-select and new-select) must be turned on.
     * |        |          |The default value is reloaded from the value of CFOSC (CONFIG0[26:24]) in user configuration register of Flash controller by any reset.
     * |        |          |Therefore the default value is either 000b or 111b.
     * |        |          |000 = Clock source from HXT.
     * |        |          |001 = Clock source from LXT.
     * |        |          |010 = Clock source from PLL clock.
     * |        |          |011 = Clock source from LIRC.
     * |        |          |100 = Clock source from HIRC48.
     * |        |          |111 = Clock source from HIRC clock.
     * |        |          |Others = Reserved.
     * |        |          |Note: These bits are write protected. Refer to the SYS_REGLCTL register.
     * |[5:3]   |STCLKSEL  |Cortex-M0 SysTick Clock Source Selection (Write Protect)
     * |        |          |If SYST_CTRL[2]=0, SysTick uses listed clock source below.
     * |        |          |000 = Clock source from HXT.
     * |        |          |001 = Clock source from LXT.
     * |        |          |010 = Clock source from HXT/2.
     * |        |          |011 = Clock source from HCLK/2.
     * |        |          |111 = Clock source from HIRC/2.
     * |        |          |Note1: if SysTick clock source is not from HCLK (i.e. SYST_CTRL[2] = 0), SysTick clock source must less than or equal to HCLK/2.
     * |        |          |Note2: These bits are write protected. Refer to the SYS_REGLCTL register.
     * |[6]     |PCLK0SEL  |PCLK0 Clock Source Selection (Write Protect)
     * |        |          |0 = APB0 BUS clock source from HCLK.
     * |        |          |1 = APB0 BUS clock source from HCLK/2.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[7]     |PCLK1SEL  |PCLK1 Clock Source Selection (Write Protect)
     * |        |          |0 = APB1 BUS clock source from HCLK.
     * |        |          |1 = APB1 BUS clock source from HCLK/2.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * @var CLK_T::CLKSEL1
     * Offset: 0x14  Clock Source Select Control Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |WDTSEL    |Watchdog Timer Clock Source Selection (Write Protect)
     * |        |          |00 = Reserved.
     * |        |          |01 = Clock source from 32.768 kHz external low speed crystal oscillator (LXT) clock.
     * |        |          |10 = Clock source from HCLK/2048 clock.
     * |        |          |11 = Clock source from 10 kHz internal low speed RC oscillator (LIRC) clock.
     * |        |          |Note: This These bits is are write protected. Refer to the SYS_REGLCTL register.
     * |[3:2]   |ADCSEL    |ADC Clock Source Selection
     * |        |          |00 = Clock source from 4~24 MHz external high speed crystal oscillator (HXT) clock.
     * |        |          |01 = Clock source from PLL.
     * |        |          |10 = Clock source from PCLK0.
     * |        |          |11 = Clock source from 22.1184 MHz internal high speed RC oscillator (HIRC) clock.
     * |[10:8]  |TMR0SEL   |TIMER0 Clock Source Selection
     * |        |          |000 = Clock source from 4~24 MHz external high speed crystal oscillator (HXT) clock.
     * |        |          |001 = Clock source from 32.768 kHz external low speed crystal oscillator (LXT) clock.
     * |        |          |010 = Clock source from PCLK0.
     * |        |          |011 = Clock source from external clock T0 pin.
     * |        |          |101 = Clock source from 10 kHz internal low speed RC oscillator (LIRC) clock.
     * |        |          |111 = Clock source from 22.1184 MHz internal high speed RC oscillator (HIRC) clock.
     * |        |          |Others = Reserved.
     * |[14:12] |TMR1SEL   |TIMER1 Clock Source Selection
     * |        |          |000 = Clock source from 4~24 MHz external high speed crystal oscillator (HXT) clock.
     * |        |          |001 = Clock source from 32.768 kHz external low speed crystal oscillator (LXT) clock.
     * |        |          |010 = Clock source from PCLK0.
     * |        |          |011 = Clock source from external clock T1 pin.
     * |        |          |101 = Clock source from 10 kHz internal low speed RC oscillator (LIRC) clock.
     * |        |          |111 = Clock source from 22.1184 MHz internal high speed RC oscillator (HIRC) clock.
     * |        |          |Others = Reserved.
     * |[18:16] |TMR2SEL   |TIMER2 Clock Source Selection
     * |        |          |000 = Clock source from 4~24 MHz external high speed crystal oscillator (HXT) clock.
     * |        |          |001 = Clock source from 32.768 kHz external low speed crystal oscillator (LXT) clock.
     * |        |          |010 = Clock source from PCLK1.
     * |        |          |011 = Clock source from external clock T2 pin.
     * |        |          |101 = Clock source from 10 kHz internal low speed RC oscillator (LIRC) clock.
     * |        |          |111 = Clock source from 22.1184 MHz internal high speed RC oscillator (HIRC) clock.
     * |        |          |Others = Reserved.
     * |[22:20] |TMR3SEL   |TIMER3 Clock Source Selection
     * |        |          |000 = Clock source from 4~24 MHz external high speed crystal oscillator (HXT) clock.
     * |        |          |001 = Clock source from 32.768 kHz external low speed crystal oscillator (LXT) clock.
     * |        |          |010 = Clock source from PCLK1.
     * |        |          |011 = Clock source from external clock T3 pin.
     * |        |          |101 = Clock source from 10 kHz internal low speed RC oscillator (LIRC) clock.
     * |        |          |111 = Clock source from 22.1184 MHz internal high speed RC oscillator (HIRC) clock.
     * |        |          |Others = Reserved.
     * |[25:24] |UARTSEL   |UART Clock Source Selection
     * |        |          |00 = Clock source from 4~24 MHz external high speed crystal oscillator (HXT) clock.
     * |        |          |01 = Clock source from PLL clock.
     * |        |          |10 = Clock source from 32.768 kHz external low speed crystal oscillator (LXT) clock.
     * |        |          |11 = Clock source from 22.1184 MHz internal high speed RC oscillator (HIRC) clock.
     * |[28]    |PWM0SEL   |PWM0 Clock Source Selection
     * |        |          |The peripheral clock source of PWM0 is defined by PWM0SEL.
     * |        |          |0 = Clock source from PLL clock.
     * |        |          |1 = Clock source from PCLK0.
     * |[29]    |PWM1SEL   |PWM1 Clock Source Selection
     * |        |          |The peripheral clock source of PWM1 is defined by PWM1SEL.
     * |        |          |0 = Clock source from PLL clock.
     * |        |          |1 = Clock source from PCLK1.
     * @var CLK_T::CLKDIV0
     * Offset: 0x18  Clock Divider Number Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |HCLKDIV   |HCLK Clock Divide Number From HCLK Clock Source
     * |        |          |HCLK clock frequency = (HCLK clock source frequency) / (HCLKDIV + 1).
     * |[7:4]   |USBDIV    |USB Clock Divide Number From PLL Clock
     * |        |          |USB clock frequency = (PLL clock source frequency) / (USBDIV + 1).
     * |        |          |Note: If the HIRC48 is selected, it is delivery to USB clock directly.
     * |[11:8]  |UARTDIV   |UART Clock Divide Number From UART Clock Source
     * |        |          |UART clock frequency = (UART clock source frequency) / (UARTDIV + 1).
     * |[23:16] |ADCDIV    |ADC Clock Divide Number From ADC Clock Source
     * |        |          |ADC clock frequency = (ADC clock source frequency) / (ADCDIV + 1).
     * @var CLK_T::CLKSEL2
     * Offset: 0x1C  Clock Source Select Control Register 2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[4:2]   |CLKOSEL   |Clock Divider Clock Source Selection
     * |        |          |000 = Clock source from 4~24 MHz external high speed crystal oscillator (HXT) clock.
     * |        |          |001 = Clock source from 32.768 kHz external low speed crystal oscillator (LXT) clock.
     * |        |          |010 = Clock source from HCLK.
     * |        |          |011 = Clock source from 22.1184 MHz internal high speed RC oscillator (HIRC) clock.
     * |        |          |100 = Clock source from SOF (USB start of frame event).
     * |        |          |101 = Clock source from 48 MHz internal high speed RC oscillator (HIRC48) clock.
     * |        |          |Others = Reserved.
     * |[17:16] |WWDTSEL   |Window Watchdog Timer Clock Source Selection
     * |        |          |10 = Clock source from HCLK/2048 clock.
     * |        |          |11 = Clock source from 10 kHz internal low speed RC oscillator (LIRC).
     * |        |          |Others = Reserved.
     * |[18]    |RTCSEL    |RTC Clock Source Selection
     * |        |          |0 = Clock source from 32.768 kHz external low speed crystal oscillator (LXT).
     * |        |          |1 = Clock source from 10 kHz internal low speed RC oscillator (LIRC).
     * |[25:24] |SPI0SEL   |SPI0 Clock Source Selection
     * |        |          |00 = Clock source from 4~24 MHz external high speed crystal oscillator (HXT) clock.
     * |        |          |01 = Clock source from PLL clock.
     * |        |          |10 = Clock source from PCLK0.
     * |        |          |11 = Clock source from 48 MHz internal high speed RC oscillator (HIRC48) clock.
     * |[27:26] |SPI1SEL   |SPI1 Clock Source Selection
     * |        |          |00 = Clock source from 4~24 MHz external high speed crystal oscillator (HXT) clock.
     * |        |          |01 = Clock source from PLL clock.
     * |        |          |10 = Clock source from PCLK0.
     * |        |          |11 = Clock source from 48 MHz internal high speed RC oscillator (HIRC48) clock.
     * @var CLK_T::PLLCTL
     * Offset: 0x20  PLL Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[8:0]   |FBDIV     |PLL Feedback Divider Control
     * |        |          |Refer to the PLL formulas.
     * |[13:9]  |INDIV     |PLL Input Divider Control
     * |        |          |Refer to the PLL formulas.
     * |[15:14] |OUTDIV    |PLL Output Divider Control
     * |        |          |Refer to the PLL formulas.
     * |[16]    |PD        |Power-down Mode
     * |        |          |If set PDEN(CLK_PWRCTL[7]) bit to 1, the PLL will enter Power-down mode, too.
     * |        |          |0 = PLL is in normal mode.
     * |        |          |1 = PLL is in Power-down mode (default).
     * |[17]    |BP        |PLL Bypass Control
     * |        |          |0 = PLL is in normal mode (default).
     * |        |          |1 = PLL clock output is same as PLL input clock.
     * |[18]    |OE        |PLL OE (FOUT Enable) Control
     * |        |          |0 = PLL FOUT Enabled.
     * |        |          |1 = PLL FOUT is fixed low.
     * |[19]    |PLLSRC    |PLL Source Clock Selection
     * |        |          |0 = PLL source clock from external 4~24 MHz high-speed crystal (HXT).
     * |        |          |1 = PLL source clock from internal 22.1184 MHz high-speed oscillator (HIRC).
     * |[23]    |STBSEL    |PLL Stable Counter Selection
     * |        |          |0 = PLL stable time is 6144 PLL source clock (suitable for source clock is equal to or less than 12MHz).
     * |        |          |1 = PLL stable time is 12288 PLL source clock (suitable for source clock is larger than 12MHz).
     * @var CLK_T::CLKOCTL
     * Offset: 0x24  Clock Output Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |FREQSEL   |Clock Output Frequency Selection
     * |        |          |The formula of output frequency is Fout = Fin/2(N+1).
     * |        |          |Fin is the input clock frequency.
     * |        |          |Fout is the frequency of divider output clock.
     * |        |          |N is the 4-bit value of FREQSEL[3:0].
     * |[4]     |CLKOEN    |Clock Output Enable Bit
     * |        |          |0 = Clock Output function Disabled.
     * |        |          |1 = Clock Output function Enabled.
     * |[5]     |DIV1EN    |Clock Output Divide One Enable Bit
     * |        |          |0 = Clock Output will output clock with source frequency divided by FREQSEL.
     * |        |          |1 = Clock Output will output clock with source frequency.
     * |[6]     |CLK1HZEN  |Clock Output 1Hz Enable Bit
     * |        |          |0 = 1 Hz clock output for 32.768 kHz external low speed crystal oscillator (LXT) frequency compensation Disabled.
     * |        |          |1 = 1 Hz clock output for 32.768 kHz external low speed crystal oscillator (LXT) frequency compensation Enabled.
     * @var CLK_T::APBCLK1
     * Offset: 0x30  APB Devices Clock Enable Control Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SC0CKEN   |SC0 Clock Enable Bit
     * |        |          |0 = SC0 Clock Disabled.
     * |        |          |1 = SC0 Clock Enabled.
     * |[1]     |SC1CKEN   |SC1 Clock Enable Bit
     * |        |          |0 = SC1 clock Disabled.
     * |        |          |1 = SC1 clock Enabled.
     * |[8]     |USCI0CKEN |USCI0 Clock Enable Bit
     * |        |          |0 = USCI0 clock Disabled.
     * |        |          |1 = USCI0 clock Enabled.
     * |[9]     |USCI1CKEN |USCI1 Clock Enable Bit
     * |        |          |0 = USCI1 clock Disabled.
     * |        |          |1 = USCI1 clock Enabled.
     * |[10]    |USCI2CKEN |USCI2 Clock Enable Bit
     * |        |          |0 = USCI2 clock Disabled.
     * |        |          |1 = USCI2 clock Enabled.
     * @var CLK_T::CLKSEL3
     * Offset: 0x34  Clock Source Select Control Register 3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |SC0SEL    |SC0 Clock Source Selection
     * |        |          |00 = Clock source from 4~24 MHz external high speed crystal oscillator(HXT) clock.
     * |        |          |01 = Clock source from PLL clock.
     * |        |          |10 = Clock source from PCLK1.
     * |        |          |11 = Clock source from 22.1184 MHz internal high speed RC oscillator (HIRC) clock.
     * |[3:2]   |SC1SEL    |SC1 Clock Source Selection
     * |        |          |00 = Clock source from 4~24 MHz external high speed crystal oscillator (HXT) clock.
     * |        |          |01 = Clock source from PLL clock.
     * |        |          |10 = Clock source from PCLK1.
     * |        |          |11 = Clock source from 22.1184 MHz internal high speed RC oscillator (HIRC) clock.
     * |[8]     |USBDSEL   |USBD Clock Source Selection (Write Protect)
     * |        |          |0 = Clock source from 48MHz internal hight speed RC oscillator (HIRC48) clock.
     * |        |          |1 = Clock source from PLL clock.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * @var CLK_T::CLKDIV1
     * Offset: 0x38  Clock Divider Number Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |SC0DIV    |SC0 Clock Divide Number From SC0 Clock Source
     * |        |          |SC0 clock frequency = (SC0 clock source frequency ) / (SC0DIV + 1).
     * |[15:8]  |SC1DIV    |SC1 Clock Divide Number From SC1 Clock Source
     * |        |          |SC1 clock frequency = (SC1 clock source frequency ) / (SC1DIV + 1).
     * @var CLK_T::BODCLK
     * Offset: 0x40  Clock Source Select for BOD Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |VDETCKSEL |Clock Source Selection for Voltage Detector
     * |        |          |The Voltage Detector clock source for detecting external input voltage is defined by VDETCKSEL.
     * |        |          |0 = Clock source is from 10 kHz internal low speed RC oscillator (LIRC) clock.
     * |        |          |1 = Clock source is from 32.768 kHz external low speed crystal oscillator (LXT) clock.
     * |        |          |Note1: If LIRC is selected, LIRCEN (CLK_PWRCTL[3]) must be enabled.
     * |        |          |Note2: If LXT is selected, LXTEN (CLK_PWRCTL[1]) must be enabled.
     * @var CLK_T::CLKDCTL
     * Offset: 0x70  Clock Fail Detector Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[4]     |HXTFDEN   |HXT Clock Fail Detector Enable Bit
     * |        |          |0 = 4~24 MHz external high speed crystal oscillator (HXT) clock Fail detector Disabled.
     * |        |          |1 = 4~24 MHz external high speed crystal oscillator (HXT) clock Fail detector Enabled.
     * |[5]     |HXTFIEN   |HXT Clock Fail Interrupt Enable Bit
     * |        |          |0 = 4~24 MHz external high speed crystal oscillator (HXT) clock Fail interrupt Disabled.
     * |        |          |1 = 4~24 MHz external high speed crystal oscillator (HXT) clock Fail interrupt Enabled.
     * |[12]    |LXTFDEN   |LXT Clock Fail Detector Enable Bit
     * |        |          |0 = 32.768 kHz external low speed crystal oscillator (LXT) clock Fail detector Disabled.
     * |        |          |1 = 32.768 kHz external low speed crystal oscillator (LXT) clock Fail detector Enabled.
     * |[13]    |LXTFIEN   |LXT Clock Fail Interrupt Enable Bit
     * |        |          |0 = 32.768 kHz external low speed crystal oscillator (LXT) clock Fail interrupt Disabled.
     * |        |          |1 = 32.768 kHz external low speed crystal oscillator (LXT) clock Fail interrupt Enabled.
     * |[16]    |HXTFQDEN  |HXT Clock Frequency Monitor Enable Bit
     * |        |          |0 = 4~24 MHz external high speed crystal oscillator (HXT) clock frequency monitor Disabled.
     * |        |          |1 = 4~24 MHz external high speed crystal oscillator (HXT) clock frequency monitor Enabled.
     * |[17]    |HXTFQIEN  |HXT Clock Frequency Monitor Interrupt Enable Bit
     * |        |          |0 = 4~24 MHz external high speed crystal oscillator (HXT) clock frequency monitor fail interrupt Disabled.
     * |        |          |1 = 4~24 MHz external high speed crystal oscillator (HXT) clock frequency monitor fail interrupt Enabled.
     * @var CLK_T::CLKDSTS
     * Offset: 0x74  Clock Fail Detector Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |HXTFIF    |HXT Clock Fail Interrupt Flag (Write Protect)
     * |        |          |0 = 4~24 MHz external high speed crystal oscillator (HXT) clock normal.
     * |        |          |1 = 4~24 MHz external high speed crystal oscillator (HXT) clock stop.
     * |        |          |Note1: This bit can be cleared to 0 by software writing 1.
     * |        |          |Note2: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[1]     |LXTFIF    |LXT Clock Fail Interrupt Flag (Write Protect)
     * |        |          |0 = 32.768 kHz external low speed crystal oscillator (LXT) clock normal.
     * |        |          |1 = 32.768 kHz external low speed crystal oscillator (LXT) clock stop.
     * |        |          |Note1: This bit can be cleared to 0 by software writing 1.
     * |        |          |Note2: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[8]     |HXTFQIF   |HXT Clock Frequency Monitor Interrupt Flag (Write Protect)
     * |        |          |0 = 4~24 MHz external high speed crystal oscillator (HXT) clock normal.
     * |        |          |1 = 4~24 MHz external high speed crystal oscillator (HXT) clock frequency abnormal.
     * |        |          |Note1: This bit can be cleared to 0 by software writing 1.
     * |        |          |Note2: This bit is write protected. Refer to the SYS_REGLCTL register.
     * @var CLK_T::CDUPB
     * Offset: 0x78  Clock Frequency Detector Upper Boundary Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[9:0]   |UPERBD    |HXT Clock Frequency Detector Upper Boundary
     * |        |          |The bits define the high value of frequency monitor window.
     * |        |          |When HXT frequency monitor value higher than this register, the HXT frequency detect fail interrupt flag will set to 1.
     * @var CLK_T::CDLOWB
     * Offset: 0x7C  Clock Frequency Detector Low Boundary Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[9:0]   |LOWERBD   |HXT Clock Frequency Detector Low Boundary
     * |        |          |The bits define the low value of frequency monitor window.
     * |        |          |When HXT frequency monitor value lower than this register, the HXT frequency detect fail interrupt flag will set to 1.
     */


    __IO uint32_t PWRCTL;                /*!< [0x0000] System Power-down Control Register                               */
    __IO uint32_t AHBCLK;                /*!< [0x0004] AHB Devices Clock Enable Control Register                        */
    __IO uint32_t APBCLK0;               /*!< [0x0008] APB Devices Clock Enable Control Register 0                      */
    __I  uint32_t STATUS;                /*!< [0x000c] Clock Status Monitor Register                                    */
    __IO uint32_t CLKSEL0;               /*!< [0x0010] Clock Source Select Control Register 0                           */
    __IO uint32_t CLKSEL1;               /*!< [0x0014] Clock Source Select Control Register 1                           */
    __IO uint32_t CLKDIV0;               /*!< [0x0018] Clock Divider Number Register 0                                  */
    __IO uint32_t CLKSEL2;               /*!< [0x001c] Clock Source Select Control Register 2                           */
    __IO uint32_t PLLCTL;                /*!< [0x0020] PLL Control Register                                             */
    __IO uint32_t CLKOCTL;               /*!< [0x0024] Clock Output Control Register                                    */
    __I  uint32_t RESERVE0[2];
    __IO uint32_t APBCLK1;               /*!< [0x0030] APB Devices Clock Enable Control Register 1                      */
    __IO uint32_t CLKSEL3;               /*!< [0x0034] Clock Source Select Control Register 3                           */
    __IO uint32_t CLKDIV1;               /*!< [0x0038] Clock Divider Number Register 1                                  */
    __I  uint32_t RESERVE1[1];
    __IO uint32_t BODCLK;                /*!< [0x0040] Clock Source Select for BOD Control Register                     */
    __I  uint32_t RESERVE2[11];
    __IO uint32_t CLKDCTL;               /*!< [0x0070] Clock Fail Detector Control Register                             */
    __IO uint32_t CLKDSTS;               /*!< [0x0074] Clock Fail Detector Status Register                              */
    __IO uint32_t CDUPB;                 /*!< [0x0078] Clock Frequency Detector Upper Boundary Register                 */
    __IO uint32_t CDLOWB;                /*!< [0x007c] Clock Frequency Detector Low Boundary Register                   */


} CLK_T;

/**
    @addtogroup CLK_CONST CLK Bit Field Definition
    Constant Definitions for CLK Controller
@{ */

#define CLK_PWRCTL_HXTEN_Pos             (0)                                               /*!< CLK_T::PWRCTL: HXTEN Position          */
#define CLK_PWRCTL_HXTEN_Msk             (0x1ul << CLK_PWRCTL_HXTEN_Pos)                   /*!< CLK_T::PWRCTL: HXTEN Mask              */

#define CLK_PWRCTL_LXTEN_Pos             (1)                                               /*!< CLK_T::PWRCTL: LXTEN Position          */
#define CLK_PWRCTL_LXTEN_Msk             (0x1ul << CLK_PWRCTL_LXTEN_Pos)                   /*!< CLK_T::PWRCTL: LXTEN Mask              */

#define CLK_PWRCTL_HIRCEN_Pos            (2)                                               /*!< CLK_T::PWRCTL: HIRCEN Position         */
#define CLK_PWRCTL_HIRCEN_Msk            (0x1ul << CLK_PWRCTL_HIRCEN_Pos)                  /*!< CLK_T::PWRCTL: HIRCEN Mask             */

#define CLK_PWRCTL_LIRCEN_Pos            (3)                                               /*!< CLK_T::PWRCTL: LIRCEN Position         */
#define CLK_PWRCTL_LIRCEN_Msk            (0x1ul << CLK_PWRCTL_LIRCEN_Pos)                  /*!< CLK_T::PWRCTL: LIRCEN Mask             */

#define CLK_PWRCTL_PDWKDLY_Pos           (4)                                               /*!< CLK_T::PWRCTL: PDWKDLY Position        */
#define CLK_PWRCTL_PDWKDLY_Msk           (0x1ul << CLK_PWRCTL_PDWKDLY_Pos)                 /*!< CLK_T::PWRCTL: PDWKDLY Mask            */

#define CLK_PWRCTL_PDWKIEN_Pos           (5)                                               /*!< CLK_T::PWRCTL: PDWKIEN Position        */
#define CLK_PWRCTL_PDWKIEN_Msk           (0x1ul << CLK_PWRCTL_PDWKIEN_Pos)                 /*!< CLK_T::PWRCTL: PDWKIEN Mask            */

#define CLK_PWRCTL_PDWKIF_Pos            (6)                                               /*!< CLK_T::PWRCTL: PDWKIF Position         */
#define CLK_PWRCTL_PDWKIF_Msk            (0x1ul << CLK_PWRCTL_PDWKIF_Pos)                  /*!< CLK_T::PWRCTL: PDWKIF Mask             */

#define CLK_PWRCTL_PDEN_Pos              (7)                                               /*!< CLK_T::PWRCTL: PDEN Position           */
#define CLK_PWRCTL_PDEN_Msk              (0x1ul << CLK_PWRCTL_PDEN_Pos)                    /*!< CLK_T::PWRCTL: PDEN Mask               */

#define CLK_PWRCTL_HXTGAIN_Pos           (10)                                              /*!< CLK_T::PWRCTL: HXTGAIN Position        */
#define CLK_PWRCTL_HXTGAIN_Msk           (0x3ul << CLK_PWRCTL_HXTGAIN_Pos)                 /*!< CLK_T::PWRCTL: HXTGAIN Mask            */

#define CLK_PWRCTL_HXTSELTYP_Pos         (12)                                              /*!< CLK_T::PWRCTL: HXTSELTYP Position      */
#define CLK_PWRCTL_HXTSELTYP_Msk         (0x1ul << CLK_PWRCTL_HXTSELTYP_Pos)               /*!< CLK_T::PWRCTL: HXTSELTYP Mask          */

#define CLK_PWRCTL_HIRC48EN_Pos          (13)                                              /*!< CLK_T::PWRCTL: HIRC48EN Position       */
#define CLK_PWRCTL_HIRC48EN_Msk          (0x1ul << CLK_PWRCTL_HIRC48EN_Pos)                /*!< CLK_T::PWRCTL: HIRC48EN Mask           */

#define CLK_AHBCLK_PDMACKEN_Pos          (1)                                               /*!< CLK_T::AHBCLK: PDMACKEN Position       */
#define CLK_AHBCLK_PDMACKEN_Msk          (0x1ul << CLK_AHBCLK_PDMACKEN_Pos)                /*!< CLK_T::AHBCLK: PDMACKEN Mask           */

#define CLK_AHBCLK_ISPCKEN_Pos           (2)                                               /*!< CLK_T::AHBCLK: ISPCKEN Position        */
#define CLK_AHBCLK_ISPCKEN_Msk           (0x1ul << CLK_AHBCLK_ISPCKEN_Pos)                 /*!< CLK_T::AHBCLK: ISPCKEN Mask            */

#define CLK_AHBCLK_EBICKEN_Pos           (3)                                               /*!< CLK_T::AHBCLK: EBICKEN Position        */
#define CLK_AHBCLK_EBICKEN_Msk           (0x1ul << CLK_AHBCLK_EBICKEN_Pos)                 /*!< CLK_T::AHBCLK: EBICKEN Mask            */

#define CLK_AHBCLK_HDIVCKEN_Pos          (4)                                               /*!< CLK_T::AHBCLK: HDIVCKEN Position       */
#define CLK_AHBCLK_HDIVCKEN_Msk          (0x1ul << CLK_AHBCLK_HDIVCKEN_Pos)                /*!< CLK_T::AHBCLK: HDIVCKEN Mask           */

#define CLK_AHBCLK_CRCCKEN_Pos           (7)                                               /*!< CLK_T::AHBCLK: CRCCKEN Position        */
#define CLK_AHBCLK_CRCCKEN_Msk           (0x1ul << CLK_AHBCLK_CRCCKEN_Pos)                 /*!< CLK_T::AHBCLK: CRCCKEN Mask            */

#define CLK_AHBCLK_FMCIDLE_Pos           (15)                                              /*!< CLK_T::AHBCLK: FMCIDLE Position        */
#define CLK_AHBCLK_FMCIDLE_Msk           (0x1ul << CLK_AHBCLK_FMCIDLE_Pos)                 /*!< CLK_T::AHBCLK: FMCIDLE Mask            */

#define CLK_AHBCLK_GPIOACKEN_Pos         (16)                                              /*!< CLK_T::AHBCLK: GPIOACKEN Position      */
#define CLK_AHBCLK_GPIOACKEN_Msk         (0x1ul << CLK_AHBCLK_GPIOACKEN_Pos)               /*!< CLK_T::AHBCLK: GPIOACKEN Mask          */

#define CLK_AHBCLK_GPIOBCKEN_Pos         (17)                                              /*!< CLK_T::AHBCLK: GPIOBCKEN Position      */
#define CLK_AHBCLK_GPIOBCKEN_Msk         (0x1ul << CLK_AHBCLK_GPIOBCKEN_Pos)               /*!< CLK_T::AHBCLK: GPIOBCKEN Mask          */

#define CLK_AHBCLK_GPIOCCKEN_Pos         (18)                                              /*!< CLK_T::AHBCLK: GPIOCCKEN Position      */
#define CLK_AHBCLK_GPIOCCKEN_Msk         (0x1ul << CLK_AHBCLK_GPIOCCKEN_Pos)               /*!< CLK_T::AHBCLK: GPIOCCKEN Mask          */

#define CLK_AHBCLK_GPIODCKEN_Pos         (19)                                              /*!< CLK_T::AHBCLK: GPIODCKEN Position      */
#define CLK_AHBCLK_GPIODCKEN_Msk         (0x1ul << CLK_AHBCLK_GPIODCKEN_Pos)               /*!< CLK_T::AHBCLK: GPIODCKEN Mask          */

#define CLK_AHBCLK_GPIOECKEN_Pos         (20)                                              /*!< CLK_T::AHBCLK: GPIOECKEN Position      */
#define CLK_AHBCLK_GPIOECKEN_Msk         (0x1ul << CLK_AHBCLK_GPIOECKEN_Pos)               /*!< CLK_T::AHBCLK: GPIOECKEN Mask          */

#define CLK_AHBCLK_GPIOFCKEN_Pos         (21)                                              /*!< CLK_T::AHBCLK: GPIOFCKEN Position      */
#define CLK_AHBCLK_GPIOFCKEN_Msk         (0x1ul << CLK_AHBCLK_GPIOFCKEN_Pos)               /*!< CLK_T::AHBCLK: GPIOFCKEN Mask          */

#define CLK_APBCLK0_WDTCKEN_Pos          (0)                                               /*!< CLK_T::APBCLK0: WDTCKEN Position       */
#define CLK_APBCLK0_WDTCKEN_Msk          (0x1ul << CLK_APBCLK0_WDTCKEN_Pos)                /*!< CLK_T::APBCLK0: WDTCKEN Mask           */

#define CLK_APBCLK0_RTCCKEN_Pos          (1)                                               /*!< CLK_T::APBCLK0: RTCCKEN Position       */
#define CLK_APBCLK0_RTCCKEN_Msk          (0x1ul << CLK_APBCLK0_RTCCKEN_Pos)                /*!< CLK_T::APBCLK0: RTCCKEN Mask           */

#define CLK_APBCLK0_TMR0CKEN_Pos         (2)                                               /*!< CLK_T::APBCLK0: TMR0CKEN Position      */
#define CLK_APBCLK0_TMR0CKEN_Msk         (0x1ul << CLK_APBCLK0_TMR0CKEN_Pos)               /*!< CLK_T::APBCLK0: TMR0CKEN Mask          */

#define CLK_APBCLK0_TMR1CKEN_Pos         (3)                                               /*!< CLK_T::APBCLK0: TMR1CKEN Position      */
#define CLK_APBCLK0_TMR1CKEN_Msk         (0x1ul << CLK_APBCLK0_TMR1CKEN_Pos)               /*!< CLK_T::APBCLK0: TMR1CKEN Mask          */

#define CLK_APBCLK0_TMR2CKEN_Pos         (4)                                               /*!< CLK_T::APBCLK0: TMR2CKEN Position      */
#define CLK_APBCLK0_TMR2CKEN_Msk         (0x1ul << CLK_APBCLK0_TMR2CKEN_Pos)               /*!< CLK_T::APBCLK0: TMR2CKEN Mask          */

#define CLK_APBCLK0_TMR3CKEN_Pos         (5)                                               /*!< CLK_T::APBCLK0: TMR3CKEN Position      */
#define CLK_APBCLK0_TMR3CKEN_Msk         (0x1ul << CLK_APBCLK0_TMR3CKEN_Pos)               /*!< CLK_T::APBCLK0: TMR3CKEN Mask          */

#define CLK_APBCLK0_CLKOCKEN_Pos         (6)                                               /*!< CLK_T::APBCLK0: CLKOCKEN Position      */
#define CLK_APBCLK0_CLKOCKEN_Msk         (0x1ul << CLK_APBCLK0_CLKOCKEN_Pos)               /*!< CLK_T::APBCLK0: CLKOCKEN Mask          */

#define CLK_APBCLK0_I2C0CKEN_Pos         (8)                                               /*!< CLK_T::APBCLK0: I2C0CKEN Position      */
#define CLK_APBCLK0_I2C0CKEN_Msk         (0x1ul << CLK_APBCLK0_I2C0CKEN_Pos)               /*!< CLK_T::APBCLK0: I2C0CKEN Mask          */

#define CLK_APBCLK0_I2C1CKEN_Pos         (9)                                               /*!< CLK_T::APBCLK0: I2C1CKEN Position      */
#define CLK_APBCLK0_I2C1CKEN_Msk         (0x1ul << CLK_APBCLK0_I2C1CKEN_Pos)               /*!< CLK_T::APBCLK0: I2C1CKEN Mask          */

#define CLK_APBCLK0_SPI0CKEN_Pos         (12)                                              /*!< CLK_T::APBCLK0: SPI0CKEN Position      */
#define CLK_APBCLK0_SPI0CKEN_Msk         (0x1ul << CLK_APBCLK0_SPI0CKEN_Pos)               /*!< CLK_T::APBCLK0: SPI0CKEN Mask          */

#define CLK_APBCLK0_SPI1CKEN_Pos         (13)                                              /*!< CLK_T::APBCLK0: SPI1CKEN Position      */
#define CLK_APBCLK0_SPI1CKEN_Msk         (0x1ul << CLK_APBCLK0_SPI1CKEN_Pos)               /*!< CLK_T::APBCLK0: SPI1CKEN Mask          */

#define CLK_APBCLK0_UART0CKEN_Pos        (16)                                              /*!< CLK_T::APBCLK0: UART0CKEN Position     */
#define CLK_APBCLK0_UART0CKEN_Msk        (0x1ul << CLK_APBCLK0_UART0CKEN_Pos)              /*!< CLK_T::APBCLK0: UART0CKEN Mask         */

#define CLK_APBCLK0_UART1CKEN_Pos        (17)                                              /*!< CLK_T::APBCLK0: UART1CKEN Position     */
#define CLK_APBCLK0_UART1CKEN_Msk        (0x1ul << CLK_APBCLK0_UART1CKEN_Pos)              /*!< CLK_T::APBCLK0: UART1CKEN Mask         */

#define CLK_APBCLK0_UART2CKEN_Pos        (18)                                              /*!< CLK_T::APBCLK0: UART2CKEN Position     */
#define CLK_APBCLK0_UART2CKEN_Msk        (0x1ul << CLK_APBCLK0_UART2CKEN_Pos)              /*!< CLK_T::APBCLK0: UART2CKEN Mask         */

#define CLK_APBCLK0_PWM0CKEN_Pos         (20)                                              /*!< CLK_T::APBCLK0: PWM0CKEN Position      */
#define CLK_APBCLK0_PWM0CKEN_Msk         (0x1ul << CLK_APBCLK0_PWM0CKEN_Pos)               /*!< CLK_T::APBCLK0: PWM0CKEN Mask          */

#define CLK_APBCLK0_PWM1CKEN_Pos         (21)                                              /*!< CLK_T::APBCLK0: PWM1CKEN Position      */
#define CLK_APBCLK0_PWM1CKEN_Msk         (0x1ul << CLK_APBCLK0_PWM1CKEN_Pos)               /*!< CLK_T::APBCLK0: PWM1CKEN Mask          */

#define CLK_APBCLK0_USBDCKEN_Pos         (27)                                              /*!< CLK_T::APBCLK0: USBDCKEN Position      */
#define CLK_APBCLK0_USBDCKEN_Msk         (0x1ul << CLK_APBCLK0_USBDCKEN_Pos)               /*!< CLK_T::APBCLK0: USBDCKEN Mask          */

#define CLK_APBCLK0_ADCCKEN_Pos          (28)                                              /*!< CLK_T::APBCLK0: ADCCKEN Position       */
#define CLK_APBCLK0_ADCCKEN_Msk          (0x1ul << CLK_APBCLK0_ADCCKEN_Pos)                /*!< CLK_T::APBCLK0: ADCCKEN Mask           */

#define CLK_APBCLK0_ACMP01CKEN_Pos       (30)                                              /*!< CLK_T::APBCLK0: ACMP01CKEN Position    */
#define CLK_APBCLK0_ACMP01CKEN_Msk       (0x1ul << CLK_APBCLK0_ACMP01CKEN_Pos)             /*!< CLK_T::APBCLK0: ACMP01CKEN Mask        */

#define CLK_STATUS_HXTSTB_Pos            (0)                                               /*!< CLK_T::STATUS: HXTSTB Position         */
#define CLK_STATUS_HXTSTB_Msk            (0x1ul << CLK_STATUS_HXTSTB_Pos)                  /*!< CLK_T::STATUS: HXTSTB Mask             */

#define CLK_STATUS_LXTSTB_Pos            (1)                                               /*!< CLK_T::STATUS: LXTSTB Position         */
#define CLK_STATUS_LXTSTB_Msk            (0x1ul << CLK_STATUS_LXTSTB_Pos)                  /*!< CLK_T::STATUS: LXTSTB Mask             */

#define CLK_STATUS_PLLSTB_Pos            (2)                                               /*!< CLK_T::STATUS: PLLSTB Position         */
#define CLK_STATUS_PLLSTB_Msk            (0x1ul << CLK_STATUS_PLLSTB_Pos)                  /*!< CLK_T::STATUS: PLLSTB Mask             */

#define CLK_STATUS_LIRCSTB_Pos           (3)                                               /*!< CLK_T::STATUS: LIRCSTB Position        */
#define CLK_STATUS_LIRCSTB_Msk           (0x1ul << CLK_STATUS_LIRCSTB_Pos)                 /*!< CLK_T::STATUS: LIRCSTB Mask            */

#define CLK_STATUS_HIRCSTB_Pos           (4)                                               /*!< CLK_T::STATUS: HIRCSTB Position        */
#define CLK_STATUS_HIRCSTB_Msk           (0x1ul << CLK_STATUS_HIRCSTB_Pos)                 /*!< CLK_T::STATUS: HIRCSTB Mask            */

#define CLK_STATUS_HIRC48STB_Pos         (5)                                               /*!< CLK_T::STATUS: HIRC48STB Position      */
#define CLK_STATUS_HIRC48STB_Msk         (0x1ul << CLK_STATUS_HIRC48STB_Pos)               /*!< CLK_T::STATUS: HIRC48STB Mask          */

#define CLK_STATUS_CLKSFAIL_Pos          (7)                                               /*!< CLK_T::STATUS: CLKSFAIL Position       */
#define CLK_STATUS_CLKSFAIL_Msk          (0x1ul << CLK_STATUS_CLKSFAIL_Pos)                /*!< CLK_T::STATUS: CLKSFAIL Mask           */

#define CLK_CLKSEL0_HCLKSEL_Pos          (0)                                               /*!< CLK_T::CLKSEL0: HCLKSEL Position       */
#define CLK_CLKSEL0_HCLKSEL_Msk          (0x7ul << CLK_CLKSEL0_HCLKSEL_Pos)                /*!< CLK_T::CLKSEL0: HCLKSEL Mask           */

#define CLK_CLKSEL0_STCLKSEL_Pos         (3)                                               /*!< CLK_T::CLKSEL0: STCLKSEL Position      */
#define CLK_CLKSEL0_STCLKSEL_Msk         (0x7ul << CLK_CLKSEL0_STCLKSEL_Pos)               /*!< CLK_T::CLKSEL0: STCLKSEL Mask          */

#define CLK_CLKSEL0_PCLK0SEL_Pos         (6)                                               /*!< CLK_T::CLKSEL0: PCLK0SEL Position      */
#define CLK_CLKSEL0_PCLK0SEL_Msk         (0x1ul << CLK_CLKSEL0_PCLK0SEL_Pos)               /*!< CLK_T::CLKSEL0: PCLK0SEL Mask          */

#define CLK_CLKSEL0_PCLK1SEL_Pos         (7)                                               /*!< CLK_T::CLKSEL0: PCLK1SEL Position      */
#define CLK_CLKSEL0_PCLK1SEL_Msk         (0x1ul << CLK_CLKSEL0_PCLK1SEL_Pos)               /*!< CLK_T::CLKSEL0: PCLK1SEL Mask          */

#define CLK_CLKSEL1_WDTSEL_Pos           (0)                                               /*!< CLK_T::CLKSEL1: WDTSEL Position        */
#define CLK_CLKSEL1_WDTSEL_Msk           (0x3ul << CLK_CLKSEL1_WDTSEL_Pos)                 /*!< CLK_T::CLKSEL1: WDTSEL Mask            */

#define CLK_CLKSEL1_ADCSEL_Pos           (2)                                               /*!< CLK_T::CLKSEL1: ADCSEL Position        */
#define CLK_CLKSEL1_ADCSEL_Msk           (0x3ul << CLK_CLKSEL1_ADCSEL_Pos)                 /*!< CLK_T::CLKSEL1: ADCSEL Mask            */

#define CLK_CLKSEL1_TMR0SEL_Pos          (8)                                               /*!< CLK_T::CLKSEL1: TMR0SEL Position       */
#define CLK_CLKSEL1_TMR0SEL_Msk          (0x7ul << CLK_CLKSEL1_TMR0SEL_Pos)                /*!< CLK_T::CLKSEL1: TMR0SEL Mask           */

#define CLK_CLKSEL1_TMR1SEL_Pos          (12)                                              /*!< CLK_T::CLKSEL1: TMR1SEL Position       */
#define CLK_CLKSEL1_TMR1SEL_Msk          (0x7ul << CLK_CLKSEL1_TMR1SEL_Pos)                /*!< CLK_T::CLKSEL1: TMR1SEL Mask           */

#define CLK_CLKSEL1_TMR2SEL_Pos          (16)                                              /*!< CLK_T::CLKSEL1: TMR2SEL Position       */
#define CLK_CLKSEL1_TMR2SEL_Msk          (0x7ul << CLK_CLKSEL1_TMR2SEL_Pos)                /*!< CLK_T::CLKSEL1: TMR2SEL Mask           */

#define CLK_CLKSEL1_TMR3SEL_Pos          (20)                                              /*!< CLK_T::CLKSEL1: TMR3SEL Position       */
#define CLK_CLKSEL1_TMR3SEL_Msk          (0x7ul << CLK_CLKSEL1_TMR3SEL_Pos)                /*!< CLK_T::CLKSEL1: TMR3SEL Mask           */

#define CLK_CLKSEL1_UARTSEL_Pos          (24)                                              /*!< CLK_T::CLKSEL1: UARTSEL Position       */
#define CLK_CLKSEL1_UARTSEL_Msk          (0x3ul << CLK_CLKSEL1_UARTSEL_Pos)                /*!< CLK_T::CLKSEL1: UARTSEL Mask           */

#define CLK_CLKSEL1_PWM0SEL_Pos          (28)                                              /*!< CLK_T::CLKSEL1: PWM0SEL Position       */
#define CLK_CLKSEL1_PWM0SEL_Msk          (0x1ul << CLK_CLKSEL1_PWM0SEL_Pos)                /*!< CLK_T::CLKSEL1: PWM0SEL Mask           */

#define CLK_CLKSEL1_PWM1SEL_Pos          (29)                                              /*!< CLK_T::CLKSEL1: PWM1SEL Position       */
#define CLK_CLKSEL1_PWM1SEL_Msk          (0x1ul << CLK_CLKSEL1_PWM1SEL_Pos)                /*!< CLK_T::CLKSEL1: PWM1SEL Mask           */

#define CLK_CLKDIV0_HCLKDIV_Pos          (0)                                               /*!< CLK_T::CLKDIV0: HCLKDIV Position       */
#define CLK_CLKDIV0_HCLKDIV_Msk          (0xful << CLK_CLKDIV0_HCLKDIV_Pos)                /*!< CLK_T::CLKDIV0: HCLKDIV Mask           */

#define CLK_CLKDIV0_USBDIV_Pos           (4)                                               /*!< CLK_T::CLKDIV0: USBDIV Position        */
#define CLK_CLKDIV0_USBDIV_Msk           (0xful << CLK_CLKDIV0_USBDIV_Pos)                 /*!< CLK_T::CLKDIV0: USBDIV Mask            */

#define CLK_CLKDIV0_UARTDIV_Pos          (8)                                               /*!< CLK_T::CLKDIV0: UARTDIV Position       */
#define CLK_CLKDIV0_UARTDIV_Msk          (0xful << CLK_CLKDIV0_UARTDIV_Pos)                /*!< CLK_T::CLKDIV0: UARTDIV Mask           */

#define CLK_CLKDIV0_ADCDIV_Pos           (16)                                              /*!< CLK_T::CLKDIV0: ADCDIV Position        */
#define CLK_CLKDIV0_ADCDIV_Msk           (0xfful << CLK_CLKDIV0_ADCDIV_Pos)                /*!< CLK_T::CLKDIV0: ADCDIV Mask            */

#define CLK_CLKSEL2_CLKOSEL_Pos          (2)                                               /*!< CLK_T::CLKSEL2: CLKOSEL Position       */
#define CLK_CLKSEL2_CLKOSEL_Msk          (0x7ul << CLK_CLKSEL2_CLKOSEL_Pos)                /*!< CLK_T::CLKSEL2: CLKOSEL Mask           */

#define CLK_CLKSEL2_WWDTSEL_Pos          (16)                                              /*!< CLK_T::CLKSEL2: WWDTSEL Position       */
#define CLK_CLKSEL2_WWDTSEL_Msk          (0x3ul << CLK_CLKSEL2_WWDTSEL_Pos)                /*!< CLK_T::CLKSEL2: WWDTSEL Mask           */

#define CLK_CLKSEL2_RTCSEL_Pos           (18)                                              /*!< CLK_T::CLKSEL2: RTCSEL Position        */
#define CLK_CLKSEL2_RTCSEL_Msk           (0x1ul << CLK_CLKSEL2_RTCSEL_Pos)                 /*!< CLK_T::CLKSEL2: RTCSEL Mask            */

#define CLK_CLKSEL2_SPI0SEL_Pos          (24)                                              /*!< CLK_T::CLKSEL2: SPI0SEL Position       */
#define CLK_CLKSEL2_SPI0SEL_Msk          (0x3ul << CLK_CLKSEL2_SPI0SEL_Pos)                /*!< CLK_T::CLKSEL2: SPI0SEL Mask           */

#define CLK_CLKSEL2_SPI1SEL_Pos          (26)                                              /*!< CLK_T::CLKSEL2: SPI1SEL Position       */
#define CLK_CLKSEL2_SPI1SEL_Msk          (0x3ul << CLK_CLKSEL2_SPI1SEL_Pos)                /*!< CLK_T::CLKSEL2: SPI1SEL Mask           */

#define CLK_PLLCTL_FBDIV_Pos             (0)                                               /*!< CLK_T::PLLCTL: FBDIV Position          */
#define CLK_PLLCTL_FBDIV_Msk             (0x1fful << CLK_PLLCTL_FBDIV_Pos)                 /*!< CLK_T::PLLCTL: FBDIV Mask              */

#define CLK_PLLCTL_INDIV_Pos             (9)                                               /*!< CLK_T::PLLCTL: INDIV Position          */
#define CLK_PLLCTL_INDIV_Msk             (0x1ful << CLK_PLLCTL_INDIV_Pos)                  /*!< CLK_T::PLLCTL: INDIV Mask              */

#define CLK_PLLCTL_OUTDIV_Pos            (14)                                              /*!< CLK_T::PLLCTL: OUTDIV Position         */
#define CLK_PLLCTL_OUTDIV_Msk            (0x3ul << CLK_PLLCTL_OUTDIV_Pos)                  /*!< CLK_T::PLLCTL: OUTDIV Mask             */

#define CLK_PLLCTL_PD_Pos                (16)                                              /*!< CLK_T::PLLCTL: PD Position             */
#define CLK_PLLCTL_PD_Msk                (0x1ul << CLK_PLLCTL_PD_Pos)                      /*!< CLK_T::PLLCTL: PD Mask                 */

#define CLK_PLLCTL_BP_Pos                (17)                                              /*!< CLK_T::PLLCTL: BP Position             */
#define CLK_PLLCTL_BP_Msk                (0x1ul << CLK_PLLCTL_BP_Pos)                      /*!< CLK_T::PLLCTL: BP Mask                 */

#define CLK_PLLCTL_OE_Pos                (18)                                              /*!< CLK_T::PLLCTL: OE Position             */
#define CLK_PLLCTL_OE_Msk                (0x1ul << CLK_PLLCTL_OE_Pos)                      /*!< CLK_T::PLLCTL: OE Mask                 */

#define CLK_PLLCTL_PLLSRC_Pos            (19)                                              /*!< CLK_T::PLLCTL: PLLSRC Position         */
#define CLK_PLLCTL_PLLSRC_Msk            (0x1ul << CLK_PLLCTL_PLLSRC_Pos)                  /*!< CLK_T::PLLCTL: PLLSRC Mask             */

#define CLK_PLLCTL_STBSEL_Pos            (23)                                              /*!< CLK_T::PLLCTL: STBSEL Position         */
#define CLK_PLLCTL_STBSEL_Msk            (0x1ul << CLK_PLLCTL_STBSEL_Pos)                  /*!< CLK_T::PLLCTL: STBSEL Mask             */

#define CLK_CLKOCTL_FREQSEL_Pos          (0)                                               /*!< CLK_T::CLKOCTL: FREQSEL Position       */
#define CLK_CLKOCTL_FREQSEL_Msk          (0xful << CLK_CLKOCTL_FREQSEL_Pos)                /*!< CLK_T::CLKOCTL: FREQSEL Mask           */

#define CLK_CLKOCTL_CLKOEN_Pos           (4)                                               /*!< CLK_T::CLKOCTL: CLKOEN Position        */
#define CLK_CLKOCTL_CLKOEN_Msk           (0x1ul << CLK_CLKOCTL_CLKOEN_Pos)                 /*!< CLK_T::CLKOCTL: CLKOEN Mask            */

#define CLK_CLKOCTL_DIV1EN_Pos           (5)                                               /*!< CLK_T::CLKOCTL: DIV1EN Position        */
#define CLK_CLKOCTL_DIV1EN_Msk           (0x1ul << CLK_CLKOCTL_DIV1EN_Pos)                 /*!< CLK_T::CLKOCTL: DIV1EN Mask            */

#define CLK_CLKOCTL_CLK1HZEN_Pos         (6)                                               /*!< CLK_T::CLKOCTL: CLK1HZEN Position      */
#define CLK_CLKOCTL_CLK1HZEN_Msk         (0x1ul << CLK_CLKOCTL_CLK1HZEN_Pos)               /*!< CLK_T::CLKOCTL: CLK1HZEN Mask          */

#define CLK_APBCLK1_SC0CKEN_Pos          (0)                                               /*!< CLK_T::APBCLK1: SC0CKEN Position       */
#define CLK_APBCLK1_SC0CKEN_Msk          (0x1ul << CLK_APBCLK1_SC0CKEN_Pos)                /*!< CLK_T::APBCLK1: SC0CKEN Mask           */

#define CLK_APBCLK1_SC1CKEN_Pos          (1)                                               /*!< CLK_T::APBCLK1: SC1CKEN Position       */
#define CLK_APBCLK1_SC1CKEN_Msk          (0x1ul << CLK_APBCLK1_SC1CKEN_Pos)                /*!< CLK_T::APBCLK1: SC1CKEN Mask           */

#define CLK_APBCLK1_USCI0CKEN_Pos        (8)                                               /*!< CLK_T::APBCLK1: USCI0CKEN Position     */
#define CLK_APBCLK1_USCI0CKEN_Msk        (0x1ul << CLK_APBCLK1_USCI0CKEN_Pos)              /*!< CLK_T::APBCLK1: USCI0CKEN Mask         */

#define CLK_APBCLK1_USCI1CKEN_Pos        (9)                                               /*!< CLK_T::APBCLK1: USCI1CKEN Position     */
#define CLK_APBCLK1_USCI1CKEN_Msk        (0x1ul << CLK_APBCLK1_USCI1CKEN_Pos)              /*!< CLK_T::APBCLK1: USCI1CKEN Mask         */

#define CLK_APBCLK1_USCI2CKEN_Pos        (10)                                              /*!< CLK_T::APBCLK1: USCI2CKEN Position     */
#define CLK_APBCLK1_USCI2CKEN_Msk        (0x1ul << CLK_APBCLK1_USCI2CKEN_Pos)              /*!< CLK_T::APBCLK1: USCI2CKEN Mask         */

#define CLK_CLKSEL3_SC0SEL_Pos           (0)                                               /*!< CLK_T::CLKSEL3: SC0SEL Position        */
#define CLK_CLKSEL3_SC0SEL_Msk           (0x3ul << CLK_CLKSEL3_SC0SEL_Pos)                 /*!< CLK_T::CLKSEL3: SC0SEL Mask            */

#define CLK_CLKSEL3_SC1SEL_Pos           (2)                                               /*!< CLK_T::CLKSEL3: SC1SEL Position        */
#define CLK_CLKSEL3_SC1SEL_Msk           (0x3ul << CLK_CLKSEL3_SC1SEL_Pos)                 /*!< CLK_T::CLKSEL3: SC1SEL Mask            */

#define CLK_CLKSEL3_USBDSEL_Pos          (8)                                               /*!< CLK_T::CLKSEL3: USBDSEL Position       */
#define CLK_CLKSEL3_USBDSEL_Msk          (0x1ul << CLK_CLKSEL3_USBDSEL_Pos)                /*!< CLK_T::CLKSEL3: USBDSEL Mask           */

#define CLK_CLKDIV1_SC0DIV_Pos           (0)                                               /*!< CLK_T::CLKDIV1: SC0DIV Position        */
#define CLK_CLKDIV1_SC0DIV_Msk           (0xfful << CLK_CLKDIV1_SC0DIV_Pos)                /*!< CLK_T::CLKDIV1: SC0DIV Mask            */

#define CLK_CLKDIV1_SC1DIV_Pos           (8)                                               /*!< CLK_T::CLKDIV1: SC1DIV Position        */
#define CLK_CLKDIV1_SC1DIV_Msk           (0xfful << CLK_CLKDIV1_SC1DIV_Pos)                /*!< CLK_T::CLKDIV1: SC1DIV Mask            */

#define CLK_BODCLK_VDETCKSEL_Pos         (0)                                               /*!< CLK_T::BODCLK: VDETCKSEL Position      */
#define CLK_BODCLK_VDETCKSEL_Msk         (0x1ul << CLK_BODCLK_VDETCKSEL_Pos)               /*!< CLK_T::BODCLK: VDETCKSEL Mask          */

#define CLK_CLKDCTL_HXTFDEN_Pos          (4)                                               /*!< CLK_T::CLKDCTL: HXTFDEN Position       */
#define CLK_CLKDCTL_HXTFDEN_Msk          (0x1ul << CLK_CLKDCTL_HXTFDEN_Pos)                /*!< CLK_T::CLKDCTL: HXTFDEN Mask           */

#define CLK_CLKDCTL_HXTFIEN_Pos          (5)                                               /*!< CLK_T::CLKDCTL: HXTFIEN Position       */
#define CLK_CLKDCTL_HXTFIEN_Msk          (0x1ul << CLK_CLKDCTL_HXTFIEN_Pos)                /*!< CLK_T::CLKDCTL: HXTFIEN Mask           */

#define CLK_CLKDCTL_LXTFDEN_Pos          (12)                                              /*!< CLK_T::CLKDCTL: LXTFDEN Position       */
#define CLK_CLKDCTL_LXTFDEN_Msk          (0x1ul << CLK_CLKDCTL_LXTFDEN_Pos)                /*!< CLK_T::CLKDCTL: LXTFDEN Mask           */

#define CLK_CLKDCTL_LXTFIEN_Pos          (13)                                              /*!< CLK_T::CLKDCTL: LXTFIEN Position       */
#define CLK_CLKDCTL_LXTFIEN_Msk          (0x1ul << CLK_CLKDCTL_LXTFIEN_Pos)                /*!< CLK_T::CLKDCTL: LXTFIEN Mask           */

#define CLK_CLKDCTL_HXTFQDEN_Pos         (16)                                              /*!< CLK_T::CLKDCTL: HXTFQDEN Position      */
#define CLK_CLKDCTL_HXTFQDEN_Msk         (0x1ul << CLK_CLKDCTL_HXTFQDEN_Pos)               /*!< CLK_T::CLKDCTL: HXTFQDEN Mask          */

#define CLK_CLKDCTL_HXTFQIEN_Pos         (17)                                              /*!< CLK_T::CLKDCTL: HXTFQIEN Position      */
#define CLK_CLKDCTL_HXTFQIEN_Msk         (0x1ul << CLK_CLKDCTL_HXTFQIEN_Pos)               /*!< CLK_T::CLKDCTL: HXTFQIEN Mask          */

#define CLK_CLKDSTS_HXTFIF_Pos           (0)                                               /*!< CLK_T::CLKDSTS: HXTFIF Position        */
#define CLK_CLKDSTS_HXTFIF_Msk           (0x1ul << CLK_CLKDSTS_HXTFIF_Pos)                 /*!< CLK_T::CLKDSTS: HXTFIF Mask            */

#define CLK_CLKDSTS_LXTFIF_Pos           (1)                                               /*!< CLK_T::CLKDSTS: LXTFIF Position        */
#define CLK_CLKDSTS_LXTFIF_Msk           (0x1ul << CLK_CLKDSTS_LXTFIF_Pos)                 /*!< CLK_T::CLKDSTS: LXTFIF Mask            */

#define CLK_CLKDSTS_HXTFQIF_Pos          (8)                                               /*!< CLK_T::CLKDSTS: HXTFQIF Position       */
#define CLK_CLKDSTS_HXTFQIF_Msk          (0x1ul << CLK_CLKDSTS_HXTFQIF_Pos)                /*!< CLK_T::CLKDSTS: HXTFQIF Mask           */

#define CLK_CDUPB_UPERBD_Pos             (0)                                               /*!< CLK_T::CDUPB: UPERBD Position          */
#define CLK_CDUPB_UPERBD_Msk             (0x3fful << CLK_CDUPB_UPERBD_Pos)                 /*!< CLK_T::CDUPB: UPERBD Mask              */

#define CLK_CDLOWB_LOWERBD_Pos           (0)                                               /*!< CLK_T::CDLOWB: LOWERBD Position        */
#define CLK_CDLOWB_LOWERBD_Msk           (0x3fful << CLK_CDLOWB_LOWERBD_Pos)               /*!< CLK_T::CDLOWB: LOWERBD Mask            */

/**@}*/ /* CLK_CONST */
/**@}*/ /* end of CLK register group */



/*---------------------- Cyclic Redundancy Check Controller -------------------------*/
/**
    @addtogroup CRC Cyclic Redundancy Check Controller(CRC)
    Memory Mapped Structure for CRC Controller
@{ */

typedef struct
{


    /**
     * @var CRC_T::CTL
     * Offset: 0x00  CRC Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CRCEN     |CRC Channel Generator Enable Bit
     * |        |          |Set this bit 1 to enable CRC generator for CRC operation.
     * |        |          |0 = No effect.
     * |        |          |1 = CRC operation generator is activeEnabled.
     * |[1]     |CHKSINIT  |CChecksum Initialization
     * |        |          |Set this bit will auto reolad SEED (CRC_SEED [31:0]) to CHECKSUM (CRC_CHECKSUM[31:0]) as CRC operation initial value.RC Engine Reset
     * |        |          |0 = No effect.
     * |        |          |1 = InitialReolad SEED value to CHECKSUM as CRC operation initial value checksum value by auto reloadReset CRC_SEED register value to CRC_CHECKSUM register value.the internal CRC state machine
     * |        |          |The others contents of CRC_CTL register will not be cleared.
     * |        |          |Note1: This bit will be cleared automatically
     * |        |          |Note2: Setting this bit will reload the seed value from CRC_SEED register as checksum initial value.
     * |[24]    |DATREV    |Write Data Bit Order Reverse Enable Bit
     * |        |          |This bit is used to enable the bit order reverse function per byte for write data value DATA (CRC_DATA[31:0]) write data value in CRC_DAT register.
     * |        |          |0 = Bit order reversed for CRC DATACRC write data in Disabled.
     * |        |          |1 = Bit order reversed for CRC DATACRC write data in Enabled (per byte).
     * |        |          |Note: If the write data is 0xAABBCCDD, the bit order reverse for CRC write data in is 0x55DD33BB.
     * |[25]    |CHKSREV   |Checksum Bit Order Reverse Enable Bit
     * |        |          |This bit is used to enable the bit order reverse function for checksum result CHECKSUM (CRC_CHECKSUM[31:0]) write data value in CRC_CHECKSUM register.
     * |        |          |0 = Bit order reverse for CRC CHECKSUMCRC checksum Disabled.
     * |        |          |1 = Bit order reverse for CRC CHECKSUMCRC checksum Enabled.
     * |        |          |Note: If the checksum result is 0xDD7B0F2E, the bit order reverse result for CRC checksum is 0x74F0DEBB.
     * |[26]    |DATFMT    |Write Data 1u2019s Complement Enable Bit
     * |        |          |This bit is used to enable the 1u2019s complement function for write data value DATA (CRC_DATA[31:0]).in CRC_DAT register.
     * |        |          |0 = 1u2019s complement for CRC CRC writes data inDATA Disabled.
     * |        |          |1 = 1u2019s complement for CRC DATACRC writes data in Enabled.
     * |[27]    |CHKSFMT   |Checksum 1u2019s Complement Enable Bit
     * |        |          |This bit is used to enable the 1u2019s complement function for checksum result in CHECKSUM (CRC_CHECKSUM[31:0]) register.
     * |        |          |0 = 1u2019s complement for CRC CRC checksumCHECKSUM Disabled.
     * |        |          |1 = 1u2019s complement for CRC CHECKSUMCRC checksum Enabled.
     * |[29:28] |DATLEN    |CPU Write Data Length
     * |        |          |This field indicates the valid write data length of DATA (CRC_DAT[31:0]).
     * |        |          |00 = Data length is 8-bit mode.
     * |        |          |01 = Data length is 16-bit mode.
     * |        |          |1x = Data length is 32-bit mode.
     * |        |          |Note: When the write data length is 8-bit mode, the valid data in CRC_DAT register is only DATA[7:0] bits; if the write data length is 16-bit mode, the valid data in CRC_DAT register is only DATA[15:0]
     * |[31:30] |CRCMODE   |CRC Polynomial Mode
     * |        |          |This field indicates the CRC operation polynomial mode.
     * |        |          |00 = CRC-CCITT Polynomial mode.
     * |        |          |01 = CRC-8 Polynomial mode.
     * |        |          |10 = CRC-16 Polynomial mode.
     * |        |          |11 = CRC-32 Polynomial mode.
     * @var CRC_T::DAT
     * Offset: 0x04  CRC Write Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |DATA      |CRC Write Data Bits
     * |        |          |User can write data directly by CPU mode or use PDMA function to write data to this field to perform CRC operation.
     * |        |          |Note: When the write data length is 8-bit mode, the valid data in CRC_DAT register is only DATA[7:0] bits; if the write data length is 16-bit mode, the valid data in CRC_DAT register is only DATA[15:0].
     * @var CRC_T::SEED
     * Offset: 0x08  CRC Seed Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |SEED      |CRC Seed Value
     * |        |          |This field indicates the CRC seed value.
     * |        |          |Note1: This field SEED value will be reloaded to as checksum initial value CHECKSUM (CRC_CHECKSUM[31:0]) register) a after perform set CRC engine resetCHKSINIT (CRC_CTL[1]) to 1.
     * |        |          |Note2: The valid bits of CRC_SEED[31:0] is correlated to CRCMODE (CRC_CTL[31:30]).
     * @var CRC_T::CHECKSUM
     * Offset: 0x0C  CRC Checksum Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |CHECKSUM  |CRC Checksum Results
     * |        |          |This field indicates the CRC checksum result.
     * |        |          |Note: The valid bits of CRC_CHECKSUM[31:0] is correlated to CRCMODE (CRC_CTL[31:30]).
     */
    __IO uint32_t CTL;                   /*!< [0x0000] CRC Control Register                                             */
    __IO uint32_t DAT;                   /*!< [0x0004] CRC Write Data Register                                          */
    __IO uint32_t SEED;                  /*!< [0x0008] CRC Seed Register                                                */
    __I  uint32_t CHECKSUM;              /*!< [0x000c] CRC Checksum Register                                            */

} CRC_T;

/**
    @addtogroup CRC_CONST CRC Bit Field Definition
    Constant Definitions for CRC Controller
@{ */

#define CRC_CTL_CRCEN_Pos                (0)                                               /*!< CRC_T::CTL: CRCEN Position             */
#define CRC_CTL_CRCEN_Msk                (0x1ul << CRC_CTL_CRCEN_Pos)                      /*!< CRC_T::CTL: CRCEN Mask                 */

#define CRC_CTL_CHKSINIT_Pos             (1)                                               /*!< CRC_T::CTL: CHKSINIT Position          */
#define CRC_CTL_CHKSINIT_Msk             (0x1ul << CRC_CTL_CHKSINIT_Pos)                   /*!< CRC_T::CTL: CHKSINIT Mask              */

#define CRC_CTL_DATREV_Pos               (24)                                              /*!< CRC_T::CTL: DATREV Position            */
#define CRC_CTL_DATREV_Msk               (0x1ul << CRC_CTL_DATREV_Pos)                     /*!< CRC_T::CTL: DATREV Mask                */

#define CRC_CTL_CHKSREV_Pos              (25)                                              /*!< CRC_T::CTL: CHKSREV Position           */
#define CRC_CTL_CHKSREV_Msk              (0x1ul << CRC_CTL_CHKSREV_Pos)                    /*!< CRC_T::CTL: CHKSREV Mask               */

#define CRC_CTL_DATFMT_Pos               (26)                                              /*!< CRC_T::CTL: DATFMT Position            */
#define CRC_CTL_DATFMT_Msk               (0x1ul << CRC_CTL_DATFMT_Pos)                     /*!< CRC_T::CTL: DATFMT Mask                */

#define CRC_CTL_CHKSFMT_Pos              (27)                                              /*!< CRC_T::CTL: CHKSFMT Position           */
#define CRC_CTL_CHKSFMT_Msk              (0x1ul << CRC_CTL_CHKSFMT_Pos)                    /*!< CRC_T::CTL: CHKSFMT Mask               */

#define CRC_CTL_DATLEN_Pos               (28)                                              /*!< CRC_T::CTL: DATLEN Position            */
#define CRC_CTL_DATLEN_Msk               (0x3ul << CRC_CTL_DATLEN_Pos)                     /*!< CRC_T::CTL: DATLEN Mask                */

#define CRC_CTL_CRCMODE_Pos              (30)                                              /*!< CRC_T::CTL: CRCMODE Position           */
#define CRC_CTL_CRCMODE_Msk              (0x3ul << CRC_CTL_CRCMODE_Pos)                    /*!< CRC_T::CTL: CRCMODE Mask               */

#define CRC_DAT_DATA_Pos                 (0)                                               /*!< CRC_T::DAT: DATA Position              */
#define CRC_DAT_DATA_Msk                 (0xfffffffful << CRC_DAT_DATA_Pos)                /*!< CRC_T::DAT: DATA Mask                  */

#define CRC_SEED_SEED_Pos                (0)                                               /*!< CRC_T::SEED: SEED Position             */
#define CRC_SEED_SEED_Msk                (0xfffffffful << CRC_SEED_SEED_Pos)               /*!< CRC_T::SEED: SEED Mask                 */

#define CRC_CHECKSUM_CHECKSUM_Pos        (0)                                               /*!< CRC_T::CHECKSUM: CHECKSUM Position     */
#define CRC_CHECKSUM_CHECKSUM_Msk        (0xfffffffful << CRC_CHECKSUM_CHECKSUM_Pos)       /*!< CRC_T::CHECKSUM: CHECKSUM Mask         */

/**@}*/ /* CRC_CONST */
/**@}*/ /* end of CRC register group */


/*---------------------- External Bus Interface Controller -------------------------*/
/**
    @addtogroup EBI External Bus Interface Controller(EBI)
    Memory Mapped Structure for EBI Controller
@{ */

typedef struct
{


    /**
     * @var EBI_T::CTL0
     * Offset: 0x00  External Bus Interface Bank0 Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |EN        |EBI Enable Bit
     * |        |          |This bit is the functional enable bit for EBI.
     * |        |          |0 = EBI function Disabled.
     * |        |          |1 = EBI function Enabled.
     * |[1]     |DW16      |EBI Data Width 16-bit Select
     * |        |          |This bit defines if the EBI data width is 8-bit or 16-bit.
     * |        |          |0 = EBI data width is 8-bit.
     * |        |          |1 = EBI data width is 16-bit.
     * |[2]     |CSPOLINV  |Chip Select Pin Polar Inverse
     * |        |          |This bit defines the active level of EBI chip select pin (EBI_nCSx), x = 0 or 1..
     * |        |          |0 = Chip select pin (EBI_nCSx) is active low.
     * |        |          |1 = Chip select pin (EBI_nCSx) is active high.
     * |        |          |x = 0, 1
     * |[4]     |CACCESS   |Continuous Data Access Mode
     * |        |          |When con ttinuousenuous access mode enabled, the tASU, tALE and tLHD cycles are bypass for continuous data transfer request.
     * |        |          |0 = Continuous data access mode Disabled.
     * |        |          |1 = Continuous data access mode Enabled.
     * |[10:8]  |MCLKDIV   |External Output Clock Divider
     * |        |          |The frequency of EBI output clock (MCLK) is controlled by MCLKDIV as follow:
     * |        |          |000 = HCLK/1.
     * |        |          |001 = HCLK/2.
     * |        |          |010 = HCLK/4.
     * |        |          |011 = HCLK/8.
     * |        |          |100 = HCLK/16.
     * |        |          |101 = HCLK/32.
     * |        |          |110 = HCLK/64.
     * |        |          |111 = HCLK/128.
     * |[18:16] |TALE      |Extend Time Of of ALE
     * |        |          |The EBI_ALE high pulse period (tALE) to latch the address can be controlled by TALE.
     * |        |          |tALE = (TALE + 1)*EBI_MCLK.
     * |        |          |Note: This field only available in EBI_CTL0 register
     * |[24]    |WBUFEN    |EBI Write Buffer Enable Bit
     * |        |          |0 = EBI write buffer Disabled.
     * |        |          |1 = EBI write buffer Enabled.
     * |        |          |Note: This bit only available in EBI_CTL0 register
     * @var EBI_T::TCTL0
     * Offset: 0x04  External Bus Interface Bank0 Timing Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:3]   |TACC      |EBI Data Access Time
     * |        |          |TACC define data access time (tACC).
     * |        |          |tACC = (TACC + 1) * EBI_MCLK.
     * |[10:8]  |TAHD      |EBI Data Access Hold Time
     * |        |          |TAHD define data access hold time (tAHD).
     * |        |          |tAHD = (TAHD + 1) * EBI_MCLK.
     * |[15:12] |W2X       |Idle Cycle After Write
     * |        |          |This field defines the number of W2X idle cycle.
     * |        |          |When write action is finish, W2X idle cycle is inserted and EBI_nCSx return to idle state, x = 0 or 1.
     * |        |          |W2X idle cycle = (W2X * EBI_MCLK).
     * |        |          |When write action is finish, W2X idle cycle is inserted and EBI_nCSx return to idle state. (x = 0, 1)
     * |[22]    |RAHDOFF   |Access Hold Time Disable Control When Read
     * |        |          |0 = The Data Access Hold Time (tAHD) during EBI reading is Enabled.
     * |        |          |1 = The Data Access Hold Time (tAHD) during EBI reading is Disabled.
     * |[23]    |WAHDOFF   |Access Hold Time Disable Control When Write
     * |        |          |0 = The Data Access Hold Time (tAHD) during EBI writing is Enabled.
     * |        |          |1 = The Data Access Hold Time (tAHD) during EBI writing is Disabled.
     * |[27:24] |R2R       |Idle Cycle Between Read-to-read
     * |        |          |This field defines the number of R2R idle cycle.
     * |        |          |When read action is finish and next action is going to read, R2R idle cycle is inserted and EBI_nCSx return to idle state, x = 0 or 1.
     * |        |          |R2R idle cycle = (R2R * EBI_MCLK).
     * |        |          |When read action is finish and next action is going to read, R2R idle cycle is inserted and EBI_nCSx return to idle state
     * |        |          |(x = 0, 1)
     * @var EBI_T::CTL1
     * Offset: 0x10  External Bus Interface Bank1 Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |EN        |EBI Enable Bit
     * |        |          |This bit is the functional enable bit for EBI.
     * |        |          |0 = EBI function Disabled.
     * |        |          |1 = EBI function Enabled.
     * |[1]     |DW16      |EBI Data Width 16-bit Select
     * |        |          |This bit defines if the EBI data width is 8-bit or 16-bit.
     * |        |          |0 = EBI data width is 8-bit.
     * |        |          |1 = EBI data width is 16-bit.
     * |[2]     |CSPOLINV  |Chip Select Pin Polar Inverse
     * |        |          |This bit defines the active level of EBI chip select pin (EBI_nCSx), x = 0 or 1..
     * |        |          |0 = Chip select pin (EBI_nCSx) is active low.
     * |        |          |1 = Chip select pin (EBI_nCSx) is active high.
     * |        |          |x = 0, 1
     * |[4]     |CACCESS   |Continuous Data Access Mode
     * |        |          |When con ttinuousenuous access mode enabled, the tASU, tALE and tLHD cycles are bypass for continuous data transfer request.
     * |        |          |0 = Continuous data access mode Disabled.
     * |        |          |1 = Continuous data access mode Enabled.
     * |[10:8]  |MCLKDIV   |External Output Clock Divider
     * |        |          |The frequency of EBI output clock (MCLK) is controlled by MCLKDIV as follow:
     * |        |          |000 = HCLK/1.
     * |        |          |001 = HCLK/2.
     * |        |          |010 = HCLK/4.
     * |        |          |011 = HCLK/8.
     * |        |          |100 = HCLK/16.
     * |        |          |101 = HCLK/32.
     * |        |          |110 = HCLK/64.
     * |        |          |111 = HCLK/128.
     * |[18:16] |TALE      |Extend Time Of of ALE
     * |        |          |The EBI_ALE high pulse period (tALE) to latch the address can be controlled by TALE.
     * |        |          |tALE = (TALE + 1)*EBI_MCLK.
     * |        |          |Note: This field only available in EBI_CTL0 register
     * |[24]    |WBUFEN    |EBI Write Buffer Enable Bit
     * |        |          |0 = EBI write buffer Disabled.
     * |        |          |1 = EBI write buffer Enabled.
     * |        |          |Note: This bit only available in EBI_CTL0 register
     * @var EBI_T::TCTL1
     * Offset: 0x14  External Bus Interface Bank1 Timing Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:3]   |TACC      |EBI Data Access Time
     * |        |          |TACC define data access time (tACC).
     * |        |          |tACC = (TACC + 1) * EBI_MCLK.
     * |[10:8]  |TAHD      |EBI Data Access Hold Time
     * |        |          |TAHD define data access hold time (tAHD).
     * |        |          |tAHD = (TAHD + 1) * EBI_MCLK.
     * |[15:12] |W2X       |Idle Cycle After Write
     * |        |          |This field defines the number of W2X idle cycle.
     * |        |          |When write action is finish, W2X idle cycle is inserted and EBI_nCSx return to idle state, x = 0 or 1.
     * |        |          |W2X idle cycle = (W2X * EBI_MCLK).
     * |        |          |When write action is finish, W2X idle cycle is inserted and EBI_nCSx return to idle state. (x = 0, 1)
     * |[22]    |RAHDOFF   |Access Hold Time Disable Control When Read
     * |        |          |0 = The Data Access Hold Time (tAHD) during EBI reading is Enabled.
     * |        |          |1 = The Data Access Hold Time (tAHD) during EBI reading is Disabled.
     * |[23]    |WAHDOFF   |Access Hold Time Disable Control When Write
     * |        |          |0 = The Data Access Hold Time (tAHD) during EBI writing is Enabled.
     * |        |          |1 = The Data Access Hold Time (tAHD) during EBI writing is Disabled.
     * |[27:24] |R2R       |Idle Cycle Between Read-to-read
     * |        |          |This field defines the number of R2R idle cycle.
     * |        |          |When read action is finish and next action is going to read, R2R idle cycle is inserted and EBI_nCSx return to idle state, x = 0 or 1.
     * |        |          |R2R idle cycle = (R2R * EBI_MCLK).
     * |        |          |When read action is finish and next action is going to read, R2R idle cycle is inserted and EBI_nCSx return to idle state
     * |        |          |(x = 0, 1)
     */
    __IO uint32_t CTL0;                  /*!< [0x0000] External Bus Interface Bank0 Control Register                    */
    __IO uint32_t TCTL0;                 /*!< [0x0004] External Bus Interface Bank0 Timing Control Register             */
    __I  uint32_t RESERVE0[2];
    __IO uint32_t CTL1;                  /*!< [0x0010] External Bus Interface Bank1 Control Register                    */
    __IO uint32_t TCTL1;                 /*!< [0x0014] External Bus Interface Bank1 Timing Control Register             */

} EBI_T;

/**
    @addtogroup EBI_CONST EBI Bit Field Definition
    Constant Definitions for EBI Controller
@{ */

#define EBI_CTL_EN_Pos                   (0)                                               /*!< EBI_T::CTL: EN Position                  */
#define EBI_CTL_EN_Msk                   (0x1ul << EBI_CTL_EN_Pos)                         /*!< EBI_T::CTL: EN Mask                      */

#define EBI_CTL_DW16_Pos                 (1)                                               /*!< EBI_T::CTL: DW16 Position                */
#define EBI_CTL_DW16_Msk                 (0x1ul << EBI_CTL_DW16_Pos)                       /*!< EBI_T::CTL: DW16 Mask                    */

#define EBI_CTL_CSPOLINV_Pos             (2)                                               /*!< EBI_T::CTL: CSPOLINV Position            */
#define EBI_CTL_CSPOLINV_Msk             (0x1ul << EBI_CTL_CSPOLINV_Pos)                   /*!< EBI_T::CTL: CSPOLINV Mask                */

#define EBI_CTL_CACCESS_Pos              (4)                                               /*!< EBI_T::CTL: CACCESS Position             */
#define EBI_CTL_CACCESS_Msk              (0x1ul << EBI_CTL_CACCESS_Pos)                    /*!< EBI_T::CTL: CACCESS Mask                 */

#define EBI_CTL_MCLKDIV_Pos              (8)                                               /*!< EBI_T::CTL: MCLKDIV Position             */
#define EBI_CTL_MCLKDIV_Msk              (0x7ul << EBI_CTL_MCLKDIV_Pos)                    /*!< EBI_T::CTL: MCLKDIV Mask                 */

#define EBI_CTL_TALE_Pos                 (16)                                              /*!< EBI_T::CTL: TALE Position                */
#define EBI_CTL_TALE_Msk                 (0x7ul << EBI_CTL_TALE_Pos)                       /*!< EBI_T::CTL: TALE Mask                    */

#define EBI_CTL_WBUFEN_Pos               (24)                                              /*!< EBI_T::CTL: WBUFEN Position              */
#define EBI_CTL_WBUFEN_Msk               (0x1ul << EBI_CTL_WBUFEN_Pos)                     /*!< EBI_T::CTL: WBUFEN Mask                  */

#define EBI_TCTL_TACC_Pos                (3)                                               /*!< EBI_T::TCTL: TACC Position               */
#define EBI_TCTL_TACC_Msk                (0x1ful << EBI_TCTL_TACC_Pos)                     /*!< EBI_T::TCTL: TACC Mask                   */

#define EBI_TCTL_TAHD_Pos                (8)                                               /*!< EBI_T::TCTL: TAHD Position               */
#define EBI_TCTL_TAHD_Msk                (0x7ul << EBI_TCTL_TAHD_Pos)                      /*!< EBI_T::TCTL: TAHD Mask                   */

#define EBI_TCTL_W2X_Pos                 (12)                                              /*!< EBI_T::TCTL: W2X Position                */
#define EBI_TCTL_W2X_Msk                 (0xful << EBI_TCTL_W2X_Pos)                       /*!< EBI_T::TCTL: W2X Mask                    */

#define EBI_TCTL_RAHDOFF_Pos             (22)                                              /*!< EBI_T::TCTL: RAHDOFF Position            */
#define EBI_TCTL_RAHDOFF_Msk             (0x1ul << EBI_TCTL_RAHDOFF_Pos)                   /*!< EBI_T::TCTL: RAHDOFF Mask                */

#define EBI_TCTL_WAHDOFF_Pos             (23)                                              /*!< EBI_T::TCTL: WAHDOFF Position            */
#define EBI_TCTL_WAHDOFF_Msk             (0x1ul << EBI_TCTL_WAHDOFF_Pos)                   /*!< EBI_T::TCTL: WAHDOFF Mask                */

#define EBI_TCTL_R2R_Pos                 (24)                                              /*!< EBI_T::TCTL: R2R Position                */
#define EBI_TCTL_R2R_Msk                 (0xful << EBI_TCTL_R2R_Pos)                       /*!< EBI_T::TCTL: R2R Mask                    */

/**@}*/ /* EBI_CONST */
/**@}*/ /* end of EBI register group */


/*---------------------- Flash Memory Controller -------------------------*/
/**
    @addtogroup FMC Flash Memory Controller(FMC)
    Memory Mapped Structure for FMC Controller
@{ */

typedef struct
{


    /**
     * @var FMC_T::ISPCTL
     * Offset: 0x00  ISP Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ISPEN     |ISP Enable Bit (Write Protect)
     * |        |          |ISP function enable bit. Set this bit to enable ISP function.
     * |        |          |0 = ISP function Disabled.
     * |        |          |1 = ISP function Enabled.
     * |        |          |Note: This bit is write-protected. Refer to the SYS_REGLCTL register.
     * |[1]     |BS        |Boot Select (Write Protect)
     * |        |          |Set/clear this bit to select next booting from LDROM/APROM, respectively.
     * |        |          |This bit also functions as chip booting status flag, which can be used to check where chip booted from.
     * |        |          |This bit is initiated with the inversed value of CBS[1] (CONFIG0[7]) after any reset is happened except CPU reset (CPU is 1) or system reset (SYS) is happened.
     * |        |          |0 = Booting from APROM.
     * |        |          |1 = Booting from LDROM.
     * |        |          |Note: This bit is write-protected. Refer to the SYS_REGLCTL register.
     * |[2]     |SPUEN     |SPROM Update Enable Bit (Write Protect)
     * |        |          |0 = SPROM cannot be updated.
     * |        |          |1 = SPROM can be updated.
     * |        |          |Note: This bit is write-protected. Refer to the SYS_REGLCTL register.
     * |[3]     |APUEN     |APROM Update Enable Bit (Write Protect)
     * |        |          |0 = APROM cannot be updated when the chip runs in APROM.
     * |        |          |1 = APROM can be updated when the chip runs in APROM.
     * |        |          |Note: This bit is write-protected. Refer to the SYS_REGLCTL register.
     * |[4]     |CFGUEN    |CONFIG Update Enable Bit (Write Protect)
     * |        |          |0 = CONFIG cannot be updated.
     * |        |          |1 = CONFIG can be updated.
     * |        |          |Note: This bit is write-protected. Refer to the SYS_REGLCTL register.
     * |[5]     |LDUEN     |LDROM Update Enable Bit (Write Protect)
     * |        |          |LDROM update enable bit.
     * |        |          |0 = LDROM cannot be updated.
     * |        |          |1 = LDROM can be updated.
     * |        |          |Note: This bit is write-protected. Refer to the SYS_REGLCTL register.
     * |[6]     |ISPFF     |ISP Fail Flag (Write Protect)
     * |        |          |This bit is set by hardware when a triggered ISP meets any of the following conditions:
     * |        |          |This bit needs to be cleared by writing 1 to it.
     * |        |          |(1) APROM writes to itself if APUEN is set to 0.
     * |        |          |(2) LDROM writes to itself if LDUEN is set to 0.
     * |        |          |(3) CONFIG is erased/programmed if CFGUEN is set to 0.
     * |        |          |(4) SPROM is erased/programmed if SPUEN is set to 0.
     * |        |          |(5) SPROM is programmed at SPROM secured mode.
     * |        |          |(6) Page Erase command at LOCK mode with ICE connection.
     * |        |          |(7) Erase or Program command at brown-out detected.
     * |        |          |(8) Destination address is illegal, such as over an available range.
     * |        |          |(9) Invalid ISP commands.
     * |        |          |Note: This bit is write-protected. Refer to the SYS_REGLCTL register.
     * @var FMC_T::ISPADDR
     * Offset: 0x04  ISP Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ISPADDR   |ISP Address
     * |        |          |The NuMicrou00AEu00E4 NUC126 series is equipped with embedded flash.
     * |        |          |ISPADDR[1:0] must be kept 00 for ISP 32-bit operation.
     * |        |          |ISPADDR[2:0] must be kept 000 for ISP 64-bit operation.
     * |        |          |For Checksum Calculation command, this field is the flash starting address for checksum calculation, 512 bytes alignment is necessary for checksum calculation.
     * @var FMC_T::ISPDAT
     * Offset: 0x08  ISP Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ISPDAT    |ISP Data
     * |        |          |Write data to this register before ISP program operation.
     * |        |          |Read data from this register after ISP read operation.
     * |        |          |For Run Checksum Calculation command, ISPDAT is the memory size (byte) and 512 bytes alignment.
     * |        |          |For ISP Read Checksum command, ISPDAT is the checksum result.
     * |        |          |If ISPDAT = 0x0000_0000, it means that (1) the checksum calculation is in progress, (2) the memory range for checksum calculation is incorrect.
     * @var FMC_T::ISPCMD
     * Offset: 0x0C  ISP CMD Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[6:0]   |CMD       |ISP CMD
     * |        |          |ISP command table is shown below:
     * |        |          |0x00= FLASH Read.
     * |        |          |0x40= FLASH 64-bit Read.
     * |        |          |0x04= Read Unique ID.
     * |        |          |0x08= Read Flash All-One Result.
     * |        |          |0x0B= Read Company ID.
     * |        |          |0x0C= Read Device ID.
     * |        |          |0x0D= Read Checksum.
     * |        |          |0x21= FLASH 32-bit Program.
     * |        |          |0x22= FLASH Page Erase.
     * |        |          |0x26= FLASH Mass Erase.
     * |        |          |0x27= FLASH Multi-Word Program.
     * |        |          |0x28= Run Flash All-One Verification.
     * |        |          |0x2D= Run Checksum Calculation.
     * |        |          |0x2E= Vector Remap.
     * |        |          |0x61= FLASH 64-bit Program.
     * |        |          |The other commands are invalid.
     * @var FMC_T::ISPTRG
     * Offset: 0x10  ISP Trigger Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ISPGO     |ISP Start Trigger (Write Protect)
     * |        |          |Write 1 to start ISP operation and this bit will be cleared to 0 by hardware automatically when ISP operation is finished.
     * |        |          |0 = ISP operation is finished.
     * |        |          |1 = ISP is progressed.
     * |        |          |Note: This bit is write-protected. Refer to the SYS_REGLCTL register.
     * @var FMC_T::DFBA
     * Offset: 0x14  Data Flash Base Address
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |DFBA      |Data Flash Base Address
     * |        |          |This register indicates Data Flash start address. It is a read only register.
     * |        |          |The Data Flash is shared with APROM. the content of this register is loaded from CONFIG1.
     * |        |          |This register is valid when DFEN (CONFIG0[0]) =0 .
     * @var FMC_T::FTCTL
     * Offset: 0x18  Flash Access Time Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[6:4]   |FOM       |Frequency Optimization Mode (Write Protect)
     * |        |          |The NuMicro M0564X series support adjustable flash access timing to optimize the flash access cycles in different working frequency.
     * |        |          |0x1 = Frequency <= 24MHz.
     * |        |          |1x1 = Frequency <= 72MHz.
     * |        |          |Others = Frequency <= 48MHz.
     * |        |          |Note: This bit is write-protected. Refer to the SYS_REGLCTL register.
     * |[7]     |CACHEOFF  |Flash Cache Disable Control (Write Protect)
     * |        |          |0 = Flash Cache function Enabled (default).
     * |        |          |1 = Flash Cache [PT1]function Disabled.
     * |        |          |Note: This bit is write-protected. Refer to the SYS_REGLCTL register.
     * @var FMC_T::ISPSTS
     * Offset: 0x40  ISP Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ISPBUSY   |ISP Busy Flag (Read Only)
     * |        |          |Write 1 to start ISP operation and this bit will be cleared to 0 by hardware automatically when ISP operation is finished.
     * |        |          |This bit is the mirror of ISPGO(FMC_ISPTRG[0]).
     * |        |          |0 = ISP operation is finished.
     * |        |          |1 = ISP is progressed.
     * |[2:1]   |CBS       |Boot Selection of CONFIG (Read Only)
     * |        |          |This bit is initiated with the CBS (CONFIG0[7:6]) after any reset is happened except CPU reset (CPU is 1) or system reset (SYS) is happened.
     * |        |          |00 = LDROM with IAP mode.
     * |        |          |01 = LDROM without IAP mode.
     * |        |          |10 = APROM with IAP mode.
     * |        |          |11 = APROM without IAP mode.
     * |[6]     |ISPFF     |ISP Fail Flag (Write Protect)
     * |        |          |This bit is the mirror of ISPFF (FMC_ISPCTL[6]), it needs to be cleared by writing 1 to FMC_ISPCTL[6] or FMC_ISPSTS[6].
     * |        |          |This bit is set by hardware when a triggered ISP meets any of the following conditions:
     * |        |          |(1) APROM writes to itself if APUEN is set to 0.
     * |        |          |(2) LDROM writes to itself if LDUEN is set to 0.
     * |        |          |(3) CONFIG is erased/programmed if CFGUEN is set to 0.
     * |        |          |(4) SPROM is erased/programmed if SPUEN is set to 0.
     * |        |          |(5) SPROM is programmed at SPROM secured mode.
     * |        |          |(6) Page Erase command at LOCK mode with ICE connection.
     * |        |          |(7) Erase or Program command at brown-out detected.
     * |        |          |(8) Destination address is illegal, such as over an available range.
     * |        |          |(9) Invalid ISP commands.
     * |        |          |(10) system vector address is remapped to SPROM.
     * |        |          |Note: This bit is write-protected. Refer to the SYS_REGLCTL register.
     * |[7]     |ALLONE    |Flash All-one Verification Flag
     * |        |          |This bit is set by hardware if all of flash bits are 1, and clear if flash bits are not all 1 after "Run Flash All-One Verification" complete; this bit also can be clear by writing 1.
     * |        |          |0 = Flash bits are not all 1 after "Run Flash All-One Verification" complete.
     * |        |          |1 = All of flash bits are 1 after "Run Flash All-One Verification" complete.
     * |[29:9]  |VECMAP    |Vector Page Mapping Address (Read Only)
     * |        |          |All access to 0x0000_0000~0x0000_01FF is remapped to the flash memory or SRAM address {VECMAP[20:0], 9u2019h000} ~ {VECMAP[20:0], 9u2019h1FF}, except SPROM.
     * |        |          |VECMAP [20:19] = 00 system vector address is mapped to flash memory.
     * |        |          |VECMAP [20:19] = 10 system vector address is mapped to SRAM memory.
     * |        |          |VECMAP [18:12] should be 0.
     * |[31]    |SCODE     |Security Code Active Flag
     * |        |          |This bit is set by hardware when detecting SPROM secured code is active at flash initiation, or software writes 1 to this bit to make secured code active; this bit is clear by SPROM page erase operation.
     * |        |          |0 = Secured code is inactive.
     * |        |          |1 = Secured code is active.
     * @var FMC_T::MPDAT0
     * Offset: 0x80  ISP Data0 Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ISPDAT0   |ISP Data 0
     * |        |          |This register is the first 32-bit data for 32-bit/64-bit/multi-word programming, and it is also the mirror of FMC_ISPDAT, both registers keep the same data.
     * @var FMC_T::MPDAT1
     * Offset: 0x84  ISP Data1 Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ISPDAT1   |ISP Data 1
     * |        |          |This register is the second 32-bit data for 64-bit/multi-word programming.
     * @var FMC_T::MPDAT2
     * Offset: 0x88  ISP Data2 Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ISPDAT2   |ISP Data 2
     * |        |          |This register is the third 32-bit data for multi-word programming.
     * @var FMC_T::MPDAT3
     * Offset: 0x8C  ISP Data3 Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ISPDAT3   |ISP Data 3
     * |        |          |This register is the fourth 32-bit data for multi-word programming.
     * @var FMC_T::MPSTS
     * Offset: 0xC0  ISP Multi-Program Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |MPBUSY    |ISP Multi-word Program Busy Flag (Read Only)
     * |        |          |Write 1 to start ISP Multi-Word program operation and this bit will be cleared to 0 by hardware automatically when ISP Multi-Word program operation is finished.
     * |        |          |This bit is the mirror of ISPGO(FMC_ISPTRG[0]).
     * |        |          |0 = ISP Multi-Word program operation is finished.
     * |        |          |1 = ISP Multi-Word program operation is progressed.
     * |[1]     |PPGO      |ISP Multi-program Status (Read Only)
     * |        |          |0 = ISP multi-word program operation is not active.
     * |        |          |1 = ISP multi-word program operation is in progress.
     * |[2]     |ISPFF     |ISP Fail Flag (Read Only)
     * |        |          |This bit is the mirror of ISPFF (FMC_ISPCTL[6]), it needs to be cleared by writing 1 to FMC_ISPCTL[6] or FMC_ISPSTS[6].
     * |        |          |This bit is set by hardware when a triggered ISP meets any of the following conditions:
     * |        |          |(1) APROM writes to itself if APUEN is set to 0.
     * |        |          |(2) LDROM writes to itself if LDUEN is set to 0.
     * |        |          |(3) CONFIG is erased/programmed if CFGUEN is set to 0.
     * |        |          |(4) Page Erase command at LOCK mode with ICE connection.
     * |        |          |(5) Erase or Program command at brown-out detected.
     * |        |          |(6) Destination address is illegal, such as over an available range.
     * |        |          |(7) Invalid ISP commands.
     * |[4]     |D0        |ISP DATA 0 Flag (Read Only)
     * |        |          |This bit is set when FMC_MPDAT0 is written and auto-clear to 0 when the FMC_MPDAT0 data is programmed to flash complete.
     * |        |          |0 = FMC_MPDAT0 register is empty, or program to flash complete.
     * |        |          |1 = FMC_MPDAT0 register has been written, and not program to flash complete.
     * |[5]     |D1        |ISP DATA 1 Flag (Read Only)
     * |        |          |This bit is set when FMC_MPDAT1 is written and auto-clear to 0 when the FMC_MPDAT1 data is programmed to flash complete.
     * |        |          |0 = FMC_MPDAT1 register is empty, or program to flash complete.
     * |        |          |1 = FMC_MPDAT1 register has been written, and not program to flash complete.
     * |[6]     |D2        |ISP DATA 2 Flag (Read Only)
     * |        |          |This bit is set when FMC_MPDAT2 is written and auto-clear to 0 when the FMC_MPDAT2 data is programmed to flash complete.
     * |        |          |0 = FMC_MPDAT2 register is empty, or program to flash complete.
     * |        |          |1 = FMC_MPDAT2 register has been written, and not program to flash complete.
     * |[7]     |D3        |ISP DATA 3 Flag (Read Only)
     * |        |          |This bit is set when FMC_MPDAT3 is written and auto-clear to 0 when the FMC_MPDAT3 data is programmed to flash complete.
     * |        |          |0 = FMC_MPDAT3 register is empty, or program to flash complete.
     * |        |          |1 = FMC_MPDAT3 register has been written, and not program to flash complete.
     * @var FMC_T::MPADDR
     * Offset: 0xC4  ISP Multi-Program Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |MPADDR    |ISP Multi-word Program Address
     * |        |          |MPADDR is the address of ISP multi-word program operation when ISPGO flag is 1.
     * |        |          |MPADDR will keep the final ISP address when ISP multi-word program is complete.
     */

    __IO uint32_t ISPCTL;                /*!< [0x0000] ISP Control Register                                             */
    __IO uint32_t ISPADDR;               /*!< [0x0004] ISP Address Register                                             */
    __IO uint32_t ISPDAT;                /*!< [0x0008] ISP Data Register                                                */
    __IO uint32_t ISPCMD;                /*!< [0x000c] ISP CMD Register                                                 */
    __IO uint32_t ISPTRG;                /*!< [0x0010] ISP Trigger Control Register                                     */
    __I  uint32_t DFBA;                  /*!< [0x0014] Data Flash Base Address                                          */
    __IO uint32_t FTCTL;                 /*!< [0x0018] Flash Access Time Control Register                               */
    __I  uint32_t RESERVE0[9];
    __IO uint32_t ISPSTS;                /*!< [0x0040] ISP Status Register                                              */
    __I  uint32_t RESERVE1[15];
    __IO uint32_t MPDAT0;                /*!< [0x0080] ISP Data0 Register                                               */
    __IO uint32_t MPDAT1;                /*!< [0x0084] ISP Data1 Register                                               */
    __IO uint32_t MPDAT2;                /*!< [0x0088] ISP Data2 Register                                               */
    __IO uint32_t MPDAT3;                /*!< [0x008c] ISP Data3 Register                                               */
    __I  uint32_t RESERVE2[12];
    __I  uint32_t MPSTS;                 /*!< [0x00c0] ISP Multi-Program Status Register                                */
    __I  uint32_t MPADDR;                /*!< [0x00c4] ISP Multi-Program Address Register                               */

} FMC_T;

/**
    @addtogroup FMC_CONST FMC Bit Field Definition
    Constant Definitions for FMC Controller
@{ */

#define FMC_ISPCTL_ISPEN_Pos             (0)                                               /*!< FMC_T::ISPCTL: ISPEN Position          */
#define FMC_ISPCTL_ISPEN_Msk             (0x1ul << FMC_ISPCTL_ISPEN_Pos)                   /*!< FMC_T::ISPCTL: ISPEN Mask              */

#define FMC_ISPCTL_BS_Pos                (1)                                               /*!< FMC_T::ISPCTL: BS Position             */
#define FMC_ISPCTL_BS_Msk                (0x1ul << FMC_ISPCTL_BS_Pos)                      /*!< FMC_T::ISPCTL: BS Mask                 */

#define FMC_ISPCTL_SPUEN_Pos             (2)                                               /*!< FMC_T::ISPCTL: SPUEN Position          */
#define FMC_ISPCTL_SPUEN_Msk             (0x1ul << FMC_ISPCTL_SPUEN_Pos)                   /*!< FMC_T::ISPCTL: SPUEN Mask              */

#define FMC_ISPCTL_APUEN_Pos             (3)                                               /*!< FMC_T::ISPCTL: APUEN Position          */
#define FMC_ISPCTL_APUEN_Msk             (0x1ul << FMC_ISPCTL_APUEN_Pos)                   /*!< FMC_T::ISPCTL: APUEN Mask              */

#define FMC_ISPCTL_CFGUEN_Pos            (4)                                               /*!< FMC_T::ISPCTL: CFGUEN Position         */
#define FMC_ISPCTL_CFGUEN_Msk            (0x1ul << FMC_ISPCTL_CFGUEN_Pos)                  /*!< FMC_T::ISPCTL: CFGUEN Mask             */

#define FMC_ISPCTL_LDUEN_Pos             (5)                                               /*!< FMC_T::ISPCTL: LDUEN Position          */
#define FMC_ISPCTL_LDUEN_Msk             (0x1ul << FMC_ISPCTL_LDUEN_Pos)                   /*!< FMC_T::ISPCTL: LDUEN Mask              */

#define FMC_ISPCTL_ISPFF_Pos             (6)                                               /*!< FMC_T::ISPCTL: ISPFF Position          */
#define FMC_ISPCTL_ISPFF_Msk             (0x1ul << FMC_ISPCTL_ISPFF_Pos)                   /*!< FMC_T::ISPCTL: ISPFF Mask              */

#define FMC_ISPADDR_ISPADDR_Pos          (0)                                               /*!< FMC_T::ISPADDR: ISPADDR Position       */
#define FMC_ISPADDR_ISPADDR_Msk          (0xfffffffful << FMC_ISPADDR_ISPADDR_Pos)         /*!< FMC_T::ISPADDR: ISPADDR Mask           */

#define FMC_ISPDAT_ISPDAT_Pos            (0)                                               /*!< FMC_T::ISPDAT: ISPDAT Position         */
#define FMC_ISPDAT_ISPDAT_Msk            (0xfffffffful << FMC_ISPDAT_ISPDAT_Pos)           /*!< FMC_T::ISPDAT: ISPDAT Mask             */

#define FMC_ISPCMD_CMD_Pos               (0)                                               /*!< FMC_T::ISPCMD: CMD Position            */
#define FMC_ISPCMD_CMD_Msk               (0x7ful << FMC_ISPCMD_CMD_Pos)                    /*!< FMC_T::ISPCMD: CMD Mask                */

#define FMC_ISPTRG_ISPGO_Pos             (0)                                               /*!< FMC_T::ISPTRG: ISPGO Position          */
#define FMC_ISPTRG_ISPGO_Msk             (0x1ul << FMC_ISPTRG_ISPGO_Pos)                   /*!< FMC_T::ISPTRG: ISPGO Mask              */

#define FMC_DFBA_DFBA_Pos                (0)                                               /*!< FMC_T::DFBA: DFBA Position             */
#define FMC_DFBA_DFBA_Msk                (0xfffffffful << FMC_DFBA_DFBA_Pos)               /*!< FMC_T::DFBA: DFBA Mask                 */

#define FMC_FTCTL_FOM_Pos                (4)                                               /*!< FMC_T::FTCTL: FOM Position             */
#define FMC_FTCTL_FOM_Msk                (0x7ul << FMC_FTCTL_FOM_Pos)                      /*!< FMC_T::FTCTL: FOM Mask                 */

#define FMC_FTCTL_CACHEOFF_Pos           (7)                                               /*!< FMC_T::FTCTL: CACHEOFF Position        */
#define FMC_FTCTL_CACHEOFF_Msk           (0x1ul << FMC_FTCTL_CACHEOFF_Pos)                 /*!< FMC_T::FTCTL: CACHEOFF Mask            */

#define FMC_ISPSTS_ISPBUSY_Pos           (0)                                               /*!< FMC_T::ISPSTS: ISPBUSY Position        */
#define FMC_ISPSTS_ISPBUSY_Msk           (0x1ul << FMC_ISPSTS_ISPBUSY_Pos)                 /*!< FMC_T::ISPSTS: ISPBUSY Mask            */

#define FMC_ISPSTS_CBS_Pos               (1)                                               /*!< FMC_T::ISPSTS: CBS Position            */
#define FMC_ISPSTS_CBS_Msk               (0x3ul << FMC_ISPSTS_CBS_Pos)                     /*!< FMC_T::ISPSTS: CBS Mask                */

#define FMC_ISPSTS_ISPFF_Pos             (6)                                               /*!< FMC_T::ISPSTS: ISPFF Position          */
#define FMC_ISPSTS_ISPFF_Msk             (0x1ul << FMC_ISPSTS_ISPFF_Pos)                   /*!< FMC_T::ISPSTS: ISPFF Mask              */

#define FMC_ISPSTS_ALLONE_Pos            (7)                                               /*!< FMC_T::ISPSTS: ALLONE Position         */
#define FMC_ISPSTS_ALLONE_Msk            (0x1ul << FMC_ISPSTS_ALLONE_Pos)                  /*!< FMC_T::ISPSTS: ALLONE Mask             */

#define FMC_ISPSTS_VECMAP_Pos            (9)                                               /*!< FMC_T::ISPSTS: VECMAP Position         */
#define FMC_ISPSTS_VECMAP_Msk            (0x1ffffful << FMC_ISPSTS_VECMAP_Pos)             /*!< FMC_T::ISPSTS: VECMAP Mask             */

#define FMC_ISPSTS_SCODE_Pos             (31)                                              /*!< FMC_T::ISPSTS: SCODE Position          */
#define FMC_ISPSTS_SCODE_Msk             (0x1ul << FMC_ISPSTS_SCODE_Pos)                   /*!< FMC_T::ISPSTS: SCODE Mask              */

#define FMC_MPDAT0_ISPDAT0_Pos           (0)                                               /*!< FMC_T::MPDAT0: ISPDAT0 Position        */
#define FMC_MPDAT0_ISPDAT0_Msk           (0xfffffffful << FMC_MPDAT0_ISPDAT0_Pos)          /*!< FMC_T::MPDAT0: ISPDAT0 Mask            */

#define FMC_MPDAT1_ISPDAT1_Pos           (0)                                               /*!< FMC_T::MPDAT1: ISPDAT1 Position        */
#define FMC_MPDAT1_ISPDAT1_Msk           (0xfffffffful << FMC_MPDAT1_ISPDAT1_Pos)          /*!< FMC_T::MPDAT1: ISPDAT1 Mask            */

#define FMC_MPDAT2_ISPDAT2_Pos           (0)                                               /*!< FMC_T::MPDAT2: ISPDAT2 Position        */
#define FMC_MPDAT2_ISPDAT2_Msk           (0xfffffffful << FMC_MPDAT2_ISPDAT2_Pos)          /*!< FMC_T::MPDAT2: ISPDAT2 Mask            */

#define FMC_MPDAT3_ISPDAT3_Pos           (0)                                               /*!< FMC_T::MPDAT3: ISPDAT3 Position        */
#define FMC_MPDAT3_ISPDAT3_Msk           (0xfffffffful << FMC_MPDAT3_ISPDAT3_Pos)          /*!< FMC_T::MPDAT3: ISPDAT3 Mask            */

#define FMC_MPSTS_MPBUSY_Pos             (0)                                               /*!< FMC_T::MPSTS: MPBUSY Position          */
#define FMC_MPSTS_MPBUSY_Msk             (0x1ul << FMC_MPSTS_MPBUSY_Pos)                   /*!< FMC_T::MPSTS: MPBUSY Mask              */

#define FMC_MPSTS_PPGO_Pos               (1)                                               /*!< FMC_T::MPSTS: PPGO Position            */
#define FMC_MPSTS_PPGO_Msk               (0x1ul << FMC_MPSTS_PPGO_Pos)                     /*!< FMC_T::MPSTS: PPGO Mask                */

#define FMC_MPSTS_ISPFF_Pos              (2)                                               /*!< FMC_T::MPSTS: ISPFF Position           */
#define FMC_MPSTS_ISPFF_Msk              (0x1ul << FMC_MPSTS_ISPFF_Pos)                    /*!< FMC_T::MPSTS: ISPFF Mask               */

#define FMC_MPSTS_D0_Pos                 (4)                                               /*!< FMC_T::MPSTS: D0 Position              */
#define FMC_MPSTS_D0_Msk                 (0x1ul << FMC_MPSTS_D0_Pos)                       /*!< FMC_T::MPSTS: D0 Mask                  */

#define FMC_MPSTS_D1_Pos                 (5)                                               /*!< FMC_T::MPSTS: D1 Position              */
#define FMC_MPSTS_D1_Msk                 (0x1ul << FMC_MPSTS_D1_Pos)                       /*!< FMC_T::MPSTS: D1 Mask                  */

#define FMC_MPSTS_D2_Pos                 (6)                                               /*!< FMC_T::MPSTS: D2 Position              */
#define FMC_MPSTS_D2_Msk                 (0x1ul << FMC_MPSTS_D2_Pos)                       /*!< FMC_T::MPSTS: D2 Mask                  */

#define FMC_MPSTS_D3_Pos                 (7)                                               /*!< FMC_T::MPSTS: D3 Position              */
#define FMC_MPSTS_D3_Msk                 (0x1ul << FMC_MPSTS_D3_Pos)                       /*!< FMC_T::MPSTS: D3 Mask                  */

#define FMC_MPADDR_MPADDR_Pos            (0)                                               /*!< FMC_T::MPADDR: MPADDR Position         */
#define FMC_MPADDR_MPADDR_Msk            (0xfffffffful << FMC_MPADDR_MPADDR_Pos)           /*!< FMC_T::MPADDR: MPADDR Mask             */

/**@}*/ /* FMC_CONST */
/**@}*/ /* end of FMC register group */


/*---------------------- General Purpose Input/Output Controller -------------------------*/
/**
    @addtogroup GPIO General Purpose Input/Output Controller(GPIO)
    Memory Mapped Structure for GPIO Controller
@{ */

typedef struct
{


    /**
     * @var GPIO_T::MODE
     * Offset: 0x00/0x40/0x80/0xC0/0x100/0x140  PA-F I/O Mode Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2n+1:2n]|MODEn    |Port A-F I/O Pin[n] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Px.n is in Quasi-bidirectional mode.
     * |        |          |Note1: The initial value of this field is defined by CIOINI (CONFIG0 [10]).
     * |        |          |If CIOINI is set to 1, the default value is 0xFFFF_FFFF and all pins will be quasi-bidirectional mode after chip powered on.
     * |        |          |If CIOINI is set to 0, the default value is 0x0000_0000 and all pins will be input mode after chip powered on.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3: The PE.14/PE.15 pin is ignored.
     * @var GPIO_T::DINOFF
     * Offset: 0x04/0x44/0x84/0xC4/0x104/0x144  PA-F Digital Input Path Disable Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[n+16]  |DINOFFn   |Port A-F Pin[n] Digital Input Path Disable Control
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled.
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2: The PE.14/PE.15 pin is ignored.
     * @var GPIO_T::DOUT
     * Offset: 0x08/0x48/0x88/0xC8/0x108/0x148  PA-F Data Output Value
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[n]     |DOUTn     |Port A-F Pin[n] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output or Quasi-bidirectional mode.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2: The PE.14/PE.15 pin is ignored.
     * @var GPIO_T::DATMSK
     * Offset: 0x0C/0x4C/0x8C/0xCC/0x10C/0x14C  PA-F Data Output Write Mask
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[n]     |DATMSKn   |Port A-F Pin[n] Data Output Write Mask
     * |        |          |These bits are used to protect the corresponding DOUT (Px_DOUT[n]) bit.
     * |        |          |When the DATMSK (Px_DATMSK[n]) bit is set to 1, the corresponding DOUT (Px_DOUT[n]) bit is protected.
     * |        |          |If the write signal is masked, writing data to the protect bit is ignored.
     * |        |          |0 = Corresponding DOUT (Px_DOUT[n]) bit can be updated.
     * |        |          |1 = Corresponding DOUT (Px_DOUT[n]) bit protected.
     * |        |          |Note1: This function only protects the corresponding DOUT (Px_DOUT[n]) bit, and will not protect the corresponding PDIO (Pxn_PDIO[0]) bit.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3: The PE.14/PE.15 pin is ignored.
     * @var GPIO_T::PIN
     * Offset: 0x10/0x50/0x90/0xD0/0x110/0x150  PA-F Pin Value
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[n]     |PINn      |Port A-F Pin[n] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2: The PE.14/PE.15 pin is ignored.
     * @var GPIO_T::DBEN
     * Offset: 0x14/0x54/0x94/0xD4/0x114/0x154  PA-F De-Bounce Enable Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[n]     |DBEN0     |Port A-F Pin[n] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit.
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt.
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2: The PE.14/PE.15 pin is ignored.
     * @var GPIO_T::INTTYPE
     * Offset: 0x18/0x58/0x98/0xD8/0x118/0x158  PA-F Interrupt Trigger Type Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[n]     |TYPEn     |Port A-F Pin[n] Edge or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger.
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce.
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n]).
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2: The PE.14/PE.15 pin is ignored.
     * @var GPIO_T::INTEN
     * Offset: 0x1C/0x5C/0x9C/0xDC/0x11C/0x15C  PA-F Interrupt Enable Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[n]     |FLIENn    |Port A-F Pin[n] Falling Edge or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger(TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2: The PE.14/PE.15 pin is ignored.
     * |[n+16   |RHIENn    |Port A-F Pin[n] Rising Edge or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2: The PE.14/PE.15 pin is ignored.
     * @var GPIO_T::INTSRC
     * Offset: 0x20/0x60/0xA0/0xE0/0x120/0x160  PA-F Interrupt Source Flag
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[n]     |INTSRCn   |Port A-F Pin[n] Interrupt Source Flag
     * |        |          |Write Operation :
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation :
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2: The PE.14/PE.15 pin is ignored.
     * @var GPIO_T::SMTEN
     * Offset: 0x24/0x64/0xA4/0xE4/0x124/0x164  PA-F Input Schmitt Trigger Enable
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[n]     |SMTENn    |Port A-F Pin[n] Input Schmitt Trigger Enable Bit
     * |        |          |0 = Px.n input schmitt trigger function Disabled.
     * |        |          |1 = Px.n input schmitt trigger function Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2: The PE.14/PE.15 pin is ignored.
     * @var GPIO_T::SLEWCTL
     * Offset: 0x28/0x68/0xA8/0xE8/0x128/0x168  PA-F High Slew Rate Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[n]     |HSRENn    |Port A-F Pin[n] High Slew Rate Control
     * |        |          |0 = Px.n output with basic slew rate.
     * |        |          |1 = Px.n output with higher slew rate.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2: The PE.14/PE.15 pin is ignored.
     * @var GPIO_T::DRVCTL
     * Offset: 0x12C  PE High Drive Strength Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[n]     |HDRVENn   |Port E Pin[n] Driving Strength Control
     * |        |          |0 = Px.n output with basic driving strength.
     * |        |          |1 = Px.n output with high driving strength.
     * |        |          |Note:
     * |        |          |n=8,9..13 for port E.
     */


    __IO uint32_t MODE;                  /*!< [0x00/0x40/0x80/0xC0/0x100/0x140] PA-F I/O Mode Control                   */
    __IO uint32_t DINOFF;                /*!< [0x04/0x44/0x84/0xC4/0x104/0x144] PA-F Digital Input Path Disable Control */
    __IO uint32_t DOUT;                  /*!< [0x08/0x48/0x88/0xC8/0x108/0x148] PA-F Data Output Value                  */
    __IO uint32_t DATMSK;                /*!< [0x0C/0x4C/0x8C/0xCC/0x10C/0x14C] PA-F Data Output Write Mask             */
    __I  uint32_t PIN;                   /*!< [0x10/0x50/0x90/0xD0/0x110/0x150] PA-F Pin Value                          */
    __IO uint32_t DBEN;                  /*!< [0x14/0x54/0x94/0xD4/0x114/0x154] PA-F De-Bounce Enable Control           */
    __IO uint32_t INTTYPE;               /*!< [0x18/0x58/0x98/0xD8/0x118/0x158] PA-F Interrupt Trigger Type Control     */
    __IO uint32_t INTEN;                 /*!< [0x1C/0x5C/0x9C/0xDC/0x11C/0x15C] PA-F Interrupt Enable Control           */
    __IO uint32_t INTSRC;                /*!< [0x20/0x60/0xA0/0xE0/0x120/0x160] PA-F Interrupt Source Flag              */
    __IO uint32_t SMTEN;                 /*!< [0x24/0x64/0xA4/0xE4/0x124/0x164] PA-F Input Schmitt Trigger Enable       */
    __IO uint32_t SLEWCTL;               /*!< [0x28/0x68/0xA8/0xE8/0x128/0x168] PA-F High Slew Rate Control             */
    __IO uint32_t DRVCTL;                /*!< [0x012c] PE High Drive Strength Control                                   */


} GPIO_T;


typedef struct
{


    /**
     * @var GPIO_DBCTL_T::DBCTL
     * Offset: 0x180  Interrupt De-bounce Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |DBCLKSEL  |De-bounce Sampling Cycle Selection
     * |        |          |0000 = Sample interrupt input once per 1 clocks.
     * |        |          |0001 = Sample interrupt input once per 2 clocks.
     * |        |          |0010 = Sample interrupt input once per 4 clocks.
     * |        |          |0011 = Sample interrupt input once per 8 clocks.
     * |        |          |0100 = Sample interrupt input once per 16 clocks.
     * |        |          |0101 = Sample interrupt input once per 32 clocks.
     * |        |          |0110 = Sample interrupt input once per 64 clocks.
     * |        |          |0111 = Sample interrupt input once per 128 clocks.
     * |        |          |1000 = Sample interrupt input once per 256 clocks.
     * |        |          |1001 = Sample interrupt input once per 2*256 clocks.
     * |        |          |1010 = Sample interrupt input once per 4*256 clocks.
     * |        |          |1011 = Sample interrupt input once per 8*256 clocks.
     * |        |          |1100 = Sample interrupt input once per 16*256 clocks.
     * |        |          |1101 = Sample interrupt input once per 32*256 clocks.
     * |        |          |1110 = Sample interrupt input once per 64*256 clocks.
     * |        |          |1111 = Sample interrupt input once per 128*256 clocks.
     * |[4]     |DBCLKSRC  |De-bounce Counter Clock Source Selection
     * |        |          |0 = De-bounce counter clock source is the HCLK.
     * |        |          |1 = De-bounce counter clock source is the internal 10 kHz internal low speed oscillator.
     * |[5]     |ICLKON    |Interrupt Clock on Mode
     * |        |          |0 = Edge detection circuit is active only if I/O pin corresponding RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n]) bit is set to 1.
     * |        |          |1 = All I/O pins edge detection circuit is always active after reset.
     * |        |          |Note: It is recommended to disable this bit to save system power if no special application concern.
     */

    __IO uint32_t DBCTL;                 /*!< [0x0180] Interrupt De-bounce Control                                      */

} GPIO_DBCTL_T;



/**
    @addtogroup GPIO_CONST GPIO Bit Field Definition
    Constant Definitions for GPIO Controller
@{ */

#define GPIO_MODE_MODE0_Pos              (0)                                               /*!< GPIO_T::MODE: MODE0 Position           */
#define GPIO_MODE_MODE0_Msk              (0x3ul << GPIO_MODE_MODE0_Pos)                    /*!< GPIO_T::MODE: MODE0 Mask               */

#define GPIO_MODE_MODE1_Pos              (2)                                               /*!< GPIO_T::MODE: MODE1 Position           */
#define GPIO_MODE_MODE1_Msk              (0x3ul << GPIO_MODE_MODE1_Pos)                    /*!< GPIO_T::MODE: MODE1 Mask               */

#define GPIO_MODE_MODE2_Pos              (4)                                               /*!< GPIO_T::MODE: MODE2 Position           */
#define GPIO_MODE_MODE2_Msk              (0x3ul << GPIO_MODE_MODE2_Pos)                    /*!< GPIO_T::MODE: MODE2 Mask               */

#define GPIO_MODE_MODE3_Pos              (6)                                               /*!< GPIO_T::MODE: MODE3 Position           */
#define GPIO_MODE_MODE3_Msk              (0x3ul << GPIO_MODE_MODE3_Pos)                    /*!< GPIO_T::MODE: MODE3 Mask               */

#define GPIO_MODE_MODE4_Pos              (8)                                               /*!< GPIO_T::MODE: MODE4 Position           */
#define GPIO_MODE_MODE4_Msk              (0x3ul << GPIO_MODE_MODE4_Pos)                    /*!< GPIO_T::MODE: MODE4 Mask               */

#define GPIO_MODE_MODE5_Pos              (10)                                              /*!< GPIO_T::MODE: MODE5 Position           */
#define GPIO_MODE_MODE5_Msk              (0x3ul << GPIO_MODE_MODE5_Pos)                    /*!< GPIO_T::MODE: MODE5 Mask               */

#define GPIO_MODE_MODE6_Pos              (12)                                              /*!< GPIO_T::MODE: MODE6 Position           */
#define GPIO_MODE_MODE6_Msk              (0x3ul << GPIO_MODE_MODE6_Pos)                    /*!< GPIO_T::MODE: MODE6 Mask               */

#define GPIO_MODE_MODE7_Pos              (14)                                              /*!< GPIO_T::MODE: MODE7 Position           */
#define GPIO_MODE_MODE7_Msk              (0x3ul << GPIO_MODE_MODE7_Pos)                    /*!< GPIO_T::MODE: MODE7 Mask               */

#define GPIO_MODE_MODE8_Pos              (16)                                              /*!< GPIO_T::MODE: MODE8 Position           */
#define GPIO_MODE_MODE8_Msk              (0x3ul << GPIO_MODE_MODE8_Pos)                    /*!< GPIO_T::MODE: MODE8 Mask               */

#define GPIO_MODE_MODE9_Pos              (18)                                              /*!< GPIO_T::MODE: MODE9 Position           */
#define GPIO_MODE_MODE9_Msk              (0x3ul << GPIO_MODE_MODE9_Pos)                    /*!< GPIO_T::MODE: MODE9 Mask               */

#define GPIO_MODE_MODE10_Pos             (20)                                              /*!< GPIO_T::MODE: MODE10 Position          */
#define GPIO_MODE_MODE10_Msk             (0x3ul << GPIO_MODE_MODE10_Pos)                   /*!< GPIO_T::MODE: MODE10 Mask              */

#define GPIO_MODE_MODE11_Pos             (22)                                              /*!< GPIO_T::MODE: MODE11 Position          */
#define GPIO_MODE_MODE11_Msk             (0x3ul << GPIO_MODE_MODE11_Pos)                   /*!< GPIO_T::MODE: MODE11 Mask              */

#define GPIO_MODE_MODE12_Pos             (24)                                              /*!< GPIO_T::MODE: MODE12 Position          */
#define GPIO_MODE_MODE12_Msk             (0x3ul << GPIO_MODE_MODE12_Pos)                   /*!< GPIO_T::MODE: MODE12 Mask              */

#define GPIO_MODE_MODE13_Pos             (26)                                              /*!< GPIO_T::MODE: MODE13 Position          */
#define GPIO_MODE_MODE13_Msk             (0x3ul << GPIO_MODE_MODE13_Pos)                   /*!< GPIO_T::MODE: MODE13 Mask              */

#define GPIO_MODE_MODE14_Pos             (28)                                              /*!< GPIO_T::MODE: MODE14 Position          */
#define GPIO_MODE_MODE14_Msk             (0x3ul << GPIO_MODE_MODE14_Pos)                   /*!< GPIO_T::MODE: MODE14 Mask              */

#define GPIO_MODE_MODE15_Pos             (30)                                              /*!< GPIO_T::MODE: MODE15 Position          */
#define GPIO_MODE_MODE15_Msk             (0x3ul << GPIO_MODE_MODE15_Pos)                   /*!< GPIO_T::MODE: MODE15 Mask              */

#define GPIO_DINOFF_DINOFF0_Pos          (16)                                              /*!< GPIO_T::DINOFF: DINOFF0 Position       */
#define GPIO_DINOFF_DINOFF0_Msk          (0x1ul << GPIO_DINOFF_DINOFF0_Pos)                /*!< GPIO_T::DINOFF: DINOFF0 Mask           */

#define GPIO_DINOFF_DINOFF1_Pos          (17)                                              /*!< GPIO_T::DINOFF: DINOFF1 Position       */
#define GPIO_DINOFF_DINOFF1_Msk          (0x1ul << GPIO_DINOFF_DINOFF1_Pos)                /*!< GPIO_T::DINOFF: DINOFF1 Mask           */

#define GPIO_DINOFF_DINOFF2_Pos          (18)                                              /*!< GPIO_T::DINOFF: DINOFF2 Position       */
#define GPIO_DINOFF_DINOFF2_Msk          (0x1ul << GPIO_DINOFF_DINOFF2_Pos)                /*!< GPIO_T::DINOFF: DINOFF2 Mask           */

#define GPIO_DINOFF_DINOFF3_Pos          (19)                                              /*!< GPIO_T::DINOFF: DINOFF3 Position       */
#define GPIO_DINOFF_DINOFF3_Msk          (0x1ul << GPIO_DINOFF_DINOFF3_Pos)                /*!< GPIO_T::DINOFF: DINOFF3 Mask           */

#define GPIO_DINOFF_DINOFF4_Pos          (20)                                              /*!< GPIO_T::DINOFF: DINOFF4 Position       */
#define GPIO_DINOFF_DINOFF4_Msk          (0x1ul << GPIO_DINOFF_DINOFF4_Pos)                /*!< GPIO_T::DINOFF: DINOFF4 Mask           */

#define GPIO_DINOFF_DINOFF5_Pos          (21)                                              /*!< GPIO_T::DINOFF: DINOFF5 Position       */
#define GPIO_DINOFF_DINOFF5_Msk          (0x1ul << GPIO_DINOFF_DINOFF5_Pos)                /*!< GPIO_T::DINOFF: DINOFF5 Mask           */

#define GPIO_DINOFF_DINOFF6_Pos          (22)                                              /*!< GPIO_T::DINOFF: DINOFF6 Position       */
#define GPIO_DINOFF_DINOFF6_Msk          (0x1ul << GPIO_DINOFF_DINOFF6_Pos)                /*!< GPIO_T::DINOFF: DINOFF6 Mask           */

#define GPIO_DINOFF_DINOFF7_Pos          (23)                                              /*!< GPIO_T::DINOFF: DINOFF7 Position       */
#define GPIO_DINOFF_DINOFF7_Msk          (0x1ul << GPIO_DINOFF_DINOFF7_Pos)                /*!< GPIO_T::DINOFF: DINOFF7 Mask           */

#define GPIO_DINOFF_DINOFF8_Pos          (24)                                              /*!< GPIO_T::DINOFF: DINOFF8 Position       */
#define GPIO_DINOFF_DINOFF8_Msk          (0x1ul << GPIO_DINOFF_DINOFF8_Pos)                /*!< GPIO_T::DINOFF: DINOFF8 Mask           */

#define GPIO_DINOFF_DINOFF9_Pos          (25)                                              /*!< GPIO_T::DINOFF: DINOFF9 Position       */
#define GPIO_DINOFF_DINOFF9_Msk          (0x1ul << GPIO_DINOFF_DINOFF9_Pos)                /*!< GPIO_T::DINOFF: DINOFF9 Mask           */

#define GPIO_DINOFF_DINOFF10_Pos         (26)                                              /*!< GPIO_T::DINOFF: DINOFF10 Position      */
#define GPIO_DINOFF_DINOFF10_Msk         (0x1ul << GPIO_DINOFF_DINOFF10_Pos)               /*!< GPIO_T::DINOFF: DINOFF10 Mask          */

#define GPIO_DINOFF_DINOFF11_Pos         (27)                                              /*!< GPIO_T::DINOFF: DINOFF11 Position      */
#define GPIO_DINOFF_DINOFF11_Msk         (0x1ul << GPIO_DINOFF_DINOFF11_Pos)               /*!< GPIO_T::DINOFF: DINOFF11 Mask          */

#define GPIO_DINOFF_DINOFF12_Pos         (28)                                              /*!< GPIO_T::DINOFF: DINOFF12 Position      */
#define GPIO_DINOFF_DINOFF12_Msk         (0x1ul << GPIO_DINOFF_DINOFF12_Pos)               /*!< GPIO_T::DINOFF: DINOFF12 Mask          */

#define GPIO_DINOFF_DINOFF13_Pos         (29)                                              /*!< GPIO_T::DINOFF: DINOFF13 Position      */
#define GPIO_DINOFF_DINOFF13_Msk         (0x1ul << GPIO_DINOFF_DINOFF13_Pos)               /*!< GPIO_T::DINOFF: DINOFF13 Mask          */

#define GPIO_DINOFF_DINOFF14_Pos         (30)                                              /*!< GPIO_T::DINOFF: DINOFF14 Position      */
#define GPIO_DINOFF_DINOFF14_Msk         (0x1ul << GPIO_DINOFF_DINOFF14_Pos)               /*!< GPIO_T::DINOFF: DINOFF14 Mask          */

#define GPIO_DINOFF_DINOFF15_Pos         (31)                                              /*!< GPIO_T::DINOFF: DINOFF15 Position      */
#define GPIO_DINOFF_DINOFF15_Msk         (0x1ul << GPIO_DINOFF_DINOFF15_Pos)               /*!< GPIO_T::DINOFF: DINOFF15 Mask          */

#define GPIO_DOUT_DOUT0_Pos              (0)                                               /*!< GPIO_T::DOUT: DOUT0 Position           */
#define GPIO_DOUT_DOUT0_Msk              (0x1ul << GPIO_DOUT_DOUT0_Pos)                    /*!< GPIO_T::DOUT: DOUT0 Mask               */

#define GPIO_DOUT_DOUT1_Pos              (1)                                               /*!< GPIO_T::DOUT: DOUT1 Position           */
#define GPIO_DOUT_DOUT1_Msk              (0x1ul << GPIO_DOUT_DOUT1_Pos)                    /*!< GPIO_T::DOUT: DOUT1 Mask               */

#define GPIO_DOUT_DOUT2_Pos              (2)                                               /*!< GPIO_T::DOUT: DOUT2 Position           */
#define GPIO_DOUT_DOUT2_Msk              (0x1ul << GPIO_DOUT_DOUT2_Pos)                    /*!< GPIO_T::DOUT: DOUT2 Mask               */

#define GPIO_DOUT_DOUT3_Pos              (3)                                               /*!< GPIO_T::DOUT: DOUT3 Position           */
#define GPIO_DOUT_DOUT3_Msk              (0x1ul << GPIO_DOUT_DOUT3_Pos)                    /*!< GPIO_T::DOUT: DOUT3 Mask               */

#define GPIO_DOUT_DOUT4_Pos              (4)                                               /*!< GPIO_T::DOUT: DOUT4 Position           */
#define GPIO_DOUT_DOUT4_Msk              (0x1ul << GPIO_DOUT_DOUT4_Pos)                    /*!< GPIO_T::DOUT: DOUT4 Mask               */

#define GPIO_DOUT_DOUT5_Pos              (5)                                               /*!< GPIO_T::DOUT: DOUT5 Position           */
#define GPIO_DOUT_DOUT5_Msk              (0x1ul << GPIO_DOUT_DOUT5_Pos)                    /*!< GPIO_T::DOUT: DOUT5 Mask               */

#define GPIO_DOUT_DOUT6_Pos              (6)                                               /*!< GPIO_T::DOUT: DOUT6 Position           */
#define GPIO_DOUT_DOUT6_Msk              (0x1ul << GPIO_DOUT_DOUT6_Pos)                    /*!< GPIO_T::DOUT: DOUT6 Mask               */

#define GPIO_DOUT_DOUT7_Pos              (7)                                               /*!< GPIO_T::DOUT: DOUT7 Position           */
#define GPIO_DOUT_DOUT7_Msk              (0x1ul << GPIO_DOUT_DOUT7_Pos)                    /*!< GPIO_T::DOUT: DOUT7 Mask               */

#define GPIO_DOUT_DOUT8_Pos              (8)                                               /*!< GPIO_T::DOUT: DOUT8 Position           */
#define GPIO_DOUT_DOUT8_Msk              (0x1ul << GPIO_DOUT_DOUT8_Pos)                    /*!< GPIO_T::DOUT: DOUT8 Mask               */

#define GPIO_DOUT_DOUT9_Pos              (9)                                               /*!< GPIO_T::DOUT: DOUT9 Position           */
#define GPIO_DOUT_DOUT9_Msk              (0x1ul << GPIO_DOUT_DOUT9_Pos)                    /*!< GPIO_T::DOUT: DOUT9 Mask               */

#define GPIO_DOUT_DOUT10_Pos             (10)                                              /*!< GPIO_T::DOUT: DOUT10 Position          */
#define GPIO_DOUT_DOUT10_Msk             (0x1ul << GPIO_DOUT_DOUT10_Pos)                   /*!< GPIO_T::DOUT: DOUT10 Mask              */

#define GPIO_DOUT_DOUT11_Pos             (11)                                              /*!< GPIO_T::DOUT: DOUT11 Position          */
#define GPIO_DOUT_DOUT11_Msk             (0x1ul << GPIO_DOUT_DOUT11_Pos)                   /*!< GPIO_T::DOUT: DOUT11 Mask              */

#define GPIO_DOUT_DOUT12_Pos             (12)                                              /*!< GPIO_T::DOUT: DOUT12 Position          */
#define GPIO_DOUT_DOUT12_Msk             (0x1ul << GPIO_DOUT_DOUT12_Pos)                   /*!< GPIO_T::DOUT: DOUT12 Mask              */

#define GPIO_DOUT_DOUT13_Pos             (13)                                              /*!< GPIO_T::DOUT: DOUT13 Position          */
#define GPIO_DOUT_DOUT13_Msk             (0x1ul << GPIO_DOUT_DOUT13_Pos)                   /*!< GPIO_T::DOUT: DOUT13 Mask              */

#define GPIO_DOUT_DOUT14_Pos             (14)                                              /*!< GPIO_T::DOUT: DOUT14 Position          */
#define GPIO_DOUT_DOUT14_Msk             (0x1ul << GPIO_DOUT_DOUT14_Pos)                   /*!< GPIO_T::DOUT: DOUT14 Mask              */

#define GPIO_DOUT_DOUT15_Pos             (15)                                              /*!< GPIO_T::DOUT: DOUT15 Position          */
#define GPIO_DOUT_DOUT15_Msk             (0x1ul << GPIO_DOUT_DOUT15_Pos)                   /*!< GPIO_T::DOUT: DOUT15 Mask              */

#define GPIO_DATMSK_DATMSK0_Pos          (0)                                               /*!< GPIO_T::DATMSK: DATMSK0 Position       */
#define GPIO_DATMSK_DATMSK0_Msk          (0x1ul << GPIO_DATMSK_DATMSK0_Pos)                /*!< GPIO_T::DATMSK: DATMSK0 Mask           */

#define GPIO_DATMSK_DATMSK1_Pos          (1)                                               /*!< GPIO_T::DATMSK: DATMSK1 Position       */
#define GPIO_DATMSK_DATMSK1_Msk          (0x1ul << GPIO_DATMSK_DATMSK1_Pos)                /*!< GPIO_T::DATMSK: DATMSK1 Mask           */

#define GPIO_DATMSK_DATMSK2_Pos          (2)                                               /*!< GPIO_T::DATMSK: DATMSK2 Position       */
#define GPIO_DATMSK_DATMSK2_Msk          (0x1ul << GPIO_DATMSK_DATMSK2_Pos)                /*!< GPIO_T::DATMSK: DATMSK2 Mask           */

#define GPIO_DATMSK_DATMSK3_Pos          (3)                                               /*!< GPIO_T::DATMSK: DATMSK3 Position       */
#define GPIO_DATMSK_DATMSK3_Msk          (0x1ul << GPIO_DATMSK_DATMSK3_Pos)                /*!< GPIO_T::DATMSK: DATMSK3 Mask           */

#define GPIO_DATMSK_DATMSK4_Pos          (4)                                               /*!< GPIO_T::DATMSK: DATMSK4 Position       */
#define GPIO_DATMSK_DATMSK4_Msk          (0x1ul << GPIO_DATMSK_DATMSK4_Pos)                /*!< GPIO_T::DATMSK: DATMSK4 Mask           */

#define GPIO_DATMSK_DATMSK5_Pos          (5)                                               /*!< GPIO_T::DATMSK: DATMSK5 Position       */
#define GPIO_DATMSK_DATMSK5_Msk          (0x1ul << GPIO_DATMSK_DATMSK5_Pos)                /*!< GPIO_T::DATMSK: DATMSK5 Mask           */

#define GPIO_DATMSK_DATMSK6_Pos          (6)                                               /*!< GPIO_T::DATMSK: DATMSK6 Position       */
#define GPIO_DATMSK_DATMSK6_Msk          (0x1ul << GPIO_DATMSK_DATMSK6_Pos)                /*!< GPIO_T::DATMSK: DATMSK6 Mask           */

#define GPIO_DATMSK_DATMSK7_Pos          (7)                                               /*!< GPIO_T::DATMSK: DATMSK7 Position       */
#define GPIO_DATMSK_DATMSK7_Msk          (0x1ul << GPIO_DATMSK_DATMSK7_Pos)                /*!< GPIO_T::DATMSK: DATMSK7 Mask           */

#define GPIO_DATMSK_DATMSK8_Pos          (8)                                               /*!< GPIO_T::DATMSK: DATMSK8 Position       */
#define GPIO_DATMSK_DATMSK8_Msk          (0x1ul << GPIO_DATMSK_DATMSK8_Pos)                /*!< GPIO_T::DATMSK: DATMSK8 Mask           */

#define GPIO_DATMSK_DATMSK9_Pos          (9)                                               /*!< GPIO_T::DATMSK: DATMSK9 Position       */
#define GPIO_DATMSK_DATMSK9_Msk          (0x1ul << GPIO_DATMSK_DATMSK9_Pos)                /*!< GPIO_T::DATMSK: DATMSK9 Mask           */

#define GPIO_DATMSK_DATMSK10_Pos         (10)                                              /*!< GPIO_T::DATMSK: DATMSK10 Position      */
#define GPIO_DATMSK_DATMSK10_Msk         (0x1ul << GPIO_DATMSK_DATMSK10_Pos)               /*!< GPIO_T::DATMSK: DATMSK10 Mask          */

#define GPIO_DATMSK_DATMSK11_Pos         (11)                                              /*!< GPIO_T::DATMSK: DATMSK11 Position      */
#define GPIO_DATMSK_DATMSK11_Msk         (0x1ul << GPIO_DATMSK_DATMSK11_Pos)               /*!< GPIO_T::DATMSK: DATMSK11 Mask          */

#define GPIO_DATMSK_DATMSK12_Pos         (12)                                              /*!< GPIO_T::DATMSK: DATMSK12 Position      */
#define GPIO_DATMSK_DATMSK12_Msk         (0x1ul << GPIO_DATMSK_DATMSK12_Pos)               /*!< GPIO_T::DATMSK: DATMSK12 Mask          */

#define GPIO_DATMSK_DATMSK13_Pos         (13)                                              /*!< GPIO_T::DATMSK: DATMSK13 Position      */
#define GPIO_DATMSK_DATMSK13_Msk         (0x1ul << GPIO_DATMSK_DATMSK13_Pos)               /*!< GPIO_T::DATMSK: DATMSK13 Mask          */

#define GPIO_DATMSK_DATMSK14_Pos         (14)                                              /*!< GPIO_T::DATMSK: DATMSK14 Position      */
#define GPIO_DATMSK_DATMSK14_Msk         (0x1ul << GPIO_DATMSK_DATMSK14_Pos)               /*!< GPIO_T::DATMSK: DATMSK14 Mask          */

#define GPIO_DATMSK_DATMSK15_Pos         (15)                                              /*!< GPIO_T::DATMSK: DATMSK15 Position      */
#define GPIO_DATMSK_DATMSK15_Msk         (0x1ul << GPIO_DATMSK_DATMSK15_Pos)               /*!< GPIO_T::DATMSK: DATMSK15 Mask          */

#define GPIO_PIN_PIN0_Pos                (0)                                               /*!< GPIO_T::PIN: PIN0 Position             */
#define GPIO_PIN_PIN0_Msk                (0x1ul << GPIO_PIN_PIN0_Pos)                      /*!< GPIO_T::PIN: PIN0 Mask                 */

#define GPIO_PIN_PIN1_Pos                (1)                                               /*!< GPIO_T::PIN: PIN1 Position             */
#define GPIO_PIN_PIN1_Msk                (0x1ul << GPIO_PIN_PIN1_Pos)                      /*!< GPIO_T::PIN: PIN1 Mask                 */

#define GPIO_PIN_PIN2_Pos                (2)                                               /*!< GPIO_T::PIN: PIN2 Position             */
#define GPIO_PIN_PIN2_Msk                (0x1ul << GPIO_PIN_PIN2_Pos)                      /*!< GPIO_T::PIN: PIN2 Mask                 */

#define GPIO_PIN_PIN3_Pos                (3)                                               /*!< GPIO_T::PIN: PIN3 Position             */
#define GPIO_PIN_PIN3_Msk                (0x1ul << GPIO_PIN_PIN3_Pos)                      /*!< GPIO_T::PIN: PIN3 Mask                 */

#define GPIO_PIN_PIN4_Pos                (4)                                               /*!< GPIO_T::PIN: PIN4 Position             */
#define GPIO_PIN_PIN4_Msk                (0x1ul << GPIO_PIN_PIN4_Pos)                      /*!< GPIO_T::PIN: PIN4 Mask                 */

#define GPIO_PIN_PIN5_Pos                (5)                                               /*!< GPIO_T::PIN: PIN5 Position             */
#define GPIO_PIN_PIN5_Msk                (0x1ul << GPIO_PIN_PIN5_Pos)                      /*!< GPIO_T::PIN: PIN5 Mask                 */

#define GPIO_PIN_PIN6_Pos                (6)                                               /*!< GPIO_T::PIN: PIN6 Position             */
#define GPIO_PIN_PIN6_Msk                (0x1ul << GPIO_PIN_PIN6_Pos)                      /*!< GPIO_T::PIN: PIN6 Mask                 */

#define GPIO_PIN_PIN7_Pos                (7)                                               /*!< GPIO_T::PIN: PIN7 Position             */
#define GPIO_PIN_PIN7_Msk                (0x1ul << GPIO_PIN_PIN7_Pos)                      /*!< GPIO_T::PIN: PIN7 Mask                 */

#define GPIO_PIN_PIN8_Pos                (8)                                               /*!< GPIO_T::PIN: PIN8 Position             */
#define GPIO_PIN_PIN8_Msk                (0x1ul << GPIO_PIN_PIN8_Pos)                      /*!< GPIO_T::PIN: PIN8 Mask                 */

#define GPIO_PIN_PIN9_Pos                (9)                                               /*!< GPIO_T::PIN: PIN9 Position             */
#define GPIO_PIN_PIN9_Msk                (0x1ul << GPIO_PIN_PIN9_Pos)                      /*!< GPIO_T::PIN: PIN9 Mask                 */

#define GPIO_PIN_PIN10_Pos               (10)                                              /*!< GPIO_T::PIN: PIN10 Position            */
#define GPIO_PIN_PIN10_Msk               (0x1ul << GPIO_PIN_PIN10_Pos)                     /*!< GPIO_T::PIN: PIN10 Mask                */

#define GPIO_PIN_PIN11_Pos               (11)                                              /*!< GPIO_T::PIN: PIN11 Position            */
#define GPIO_PIN_PIN11_Msk               (0x1ul << GPIO_PIN_PIN11_Pos)                     /*!< GPIO_T::PIN: PIN11 Mask                */

#define GPIO_PIN_PIN12_Pos               (12)                                              /*!< GPIO_T::PIN: PIN12 Position            */
#define GPIO_PIN_PIN12_Msk               (0x1ul << GPIO_PIN_PIN12_Pos)                     /*!< GPIO_T::PIN: PIN12 Mask                */

#define GPIO_PIN_PIN13_Pos               (13)                                              /*!< GPIO_T::PIN: PIN13 Position            */
#define GPIO_PIN_PIN13_Msk               (0x1ul << GPIO_PIN_PIN13_Pos)                     /*!< GPIO_T::PIN: PIN13 Mask                */

#define GPIO_PIN_PIN14_Pos               (14)                                              /*!< GPIO_T::PIN: PIN14 Position            */
#define GPIO_PIN_PIN14_Msk               (0x1ul << GPIO_PIN_PIN14_Pos)                     /*!< GPIO_T::PIN: PIN14 Mask                */

#define GPIO_PIN_PIN15_Pos               (15)                                              /*!< GPIO_T::PIN: PIN15 Position            */
#define GPIO_PIN_PIN15_Msk               (0x1ul << GPIO_PIN_PIN15_Pos)                     /*!< GPIO_T::PIN: PIN15 Mask                */

#define GPIO_DBEN_DBEN0_Pos              (0)                                               /*!< GPIO_T::DBEN: DBEN0 Position           */
#define GPIO_DBEN_DBEN0_Msk              (0x1ul << GPIO_DBEN_DBEN0_Pos)                    /*!< GPIO_T::DBEN: DBEN0 Mask               */

#define GPIO_DBEN_DBEN1_Pos              (1)                                               /*!< GPIO_T::DBEN: DBEN1 Position           */
#define GPIO_DBEN_DBEN1_Msk              (0x1ul << GPIO_DBEN_DBEN1_Pos)                    /*!< GPIO_T::DBEN: DBEN1 Mask               */

#define GPIO_DBEN_DBEN2_Pos              (2)                                               /*!< GPIO_T::DBEN: DBEN2 Position           */
#define GPIO_DBEN_DBEN2_Msk              (0x1ul << GPIO_DBEN_DBEN2_Pos)                    /*!< GPIO_T::DBEN: DBEN2 Mask               */

#define GPIO_DBEN_DBEN3_Pos              (3)                                               /*!< GPIO_T::DBEN: DBEN3 Position           */
#define GPIO_DBEN_DBEN3_Msk              (0x1ul << GPIO_DBEN_DBEN3_Pos)                    /*!< GPIO_T::DBEN: DBEN3 Mask               */

#define GPIO_DBEN_DBEN4_Pos              (4)                                               /*!< GPIO_T::DBEN: DBEN4 Position           */
#define GPIO_DBEN_DBEN4_Msk              (0x1ul << GPIO_DBEN_DBEN4_Pos)                    /*!< GPIO_T::DBEN: DBEN4 Mask               */

#define GPIO_DBEN_DBEN5_Pos              (5)                                               /*!< GPIO_T::DBEN: DBEN5 Position           */
#define GPIO_DBEN_DBEN5_Msk              (0x1ul << GPIO_DBEN_DBEN5_Pos)                    /*!< GPIO_T::DBEN: DBEN5 Mask               */

#define GPIO_DBEN_DBEN6_Pos              (6)                                               /*!< GPIO_T::DBEN: DBEN6 Position           */
#define GPIO_DBEN_DBEN6_Msk              (0x1ul << GPIO_DBEN_DBEN6_Pos)                    /*!< GPIO_T::DBEN: DBEN6 Mask               */

#define GPIO_DBEN_DBEN7_Pos              (7)                                               /*!< GPIO_T::DBEN: DBEN7 Position           */
#define GPIO_DBEN_DBEN7_Msk              (0x1ul << GPIO_DBEN_DBEN7_Pos)                    /*!< GPIO_T::DBEN: DBEN7 Mask               */

#define GPIO_DBEN_DBEN8_Pos              (8)                                               /*!< GPIO_T::DBEN: DBEN8 Position           */
#define GPIO_DBEN_DBEN8_Msk              (0x1ul << GPIO_DBEN_DBEN8_Pos)                    /*!< GPIO_T::DBEN: DBEN8 Mask               */

#define GPIO_DBEN_DBEN9_Pos              (9)                                               /*!< GPIO_T::DBEN: DBEN9 Position           */
#define GPIO_DBEN_DBEN9_Msk              (0x1ul << GPIO_DBEN_DBEN9_Pos)                    /*!< GPIO_T::DBEN: DBEN9 Mask               */

#define GPIO_DBEN_DBEN10_Pos             (10)                                              /*!< GPIO_T::DBEN: DBEN10 Position          */
#define GPIO_DBEN_DBEN10_Msk             (0x1ul << GPIO_DBEN_DBEN10_Pos)                   /*!< GPIO_T::DBEN: DBEN10 Mask              */

#define GPIO_DBEN_DBEN11_Pos             (11)                                              /*!< GPIO_T::DBEN: DBEN11 Position          */
#define GPIO_DBEN_DBEN11_Msk             (0x1ul << GPIO_DBEN_DBEN11_Pos)                   /*!< GPIO_T::DBEN: DBEN11 Mask              */

#define GPIO_DBEN_DBEN12_Pos             (12)                                              /*!< GPIO_T::DBEN: DBEN12 Position          */
#define GPIO_DBEN_DBEN12_Msk             (0x1ul << GPIO_DBEN_DBEN12_Pos)                   /*!< GPIO_T::DBEN: DBEN12 Mask              */

#define GPIO_DBEN_DBEN13_Pos             (13)                                              /*!< GPIO_T::DBEN: DBEN13 Position          */
#define GPIO_DBEN_DBEN13_Msk             (0x1ul << GPIO_DBEN_DBEN13_Pos)                   /*!< GPIO_T::DBEN: DBEN13 Mask              */

#define GPIO_DBEN_DBEN14_Pos             (14)                                              /*!< GPIO_T::DBEN: DBEN14 Position          */
#define GPIO_DBEN_DBEN14_Msk             (0x1ul << GPIO_DBEN_DBEN14_Pos)                   /*!< GPIO_T::DBEN: DBEN14 Mask              */

#define GPIO_DBEN_DBEN15_Pos             (15)                                              /*!< GPIO_T::DBEN: DBEN15 Position          */
#define GPIO_DBEN_DBEN15_Msk             (0x1ul << GPIO_DBEN_DBEN15_Pos)                   /*!< GPIO_T::DBEN: DBEN15 Mask              */

#define GPIO_INTTYPE_TYPE0_Pos           (0)                                               /*!< GPIO_T::INTTYPE: TYPE0 Position        */
#define GPIO_INTTYPE_TYPE0_Msk           (0x1ul << GPIO_INTTYPE_TYPE0_Pos)                 /*!< GPIO_T::INTTYPE: TYPE0 Mask            */

#define GPIO_INTTYPE_TYPE1_Pos           (1)                                               /*!< GPIO_T::INTTYPE: TYPE1 Position        */
#define GPIO_INTTYPE_TYPE1_Msk           (0x1ul << GPIO_INTTYPE_TYPE1_Pos)                 /*!< GPIO_T::INTTYPE: TYPE1 Mask            */

#define GPIO_INTTYPE_TYPE2_Pos           (2)                                               /*!< GPIO_T::INTTYPE: TYPE2 Position        */
#define GPIO_INTTYPE_TYPE2_Msk           (0x1ul << GPIO_INTTYPE_TYPE2_Pos)                 /*!< GPIO_T::INTTYPE: TYPE2 Mask            */

#define GPIO_INTTYPE_TYPE3_Pos           (3)                                               /*!< GPIO_T::INTTYPE: TYPE3 Position        */
#define GPIO_INTTYPE_TYPE3_Msk           (0x1ul << GPIO_INTTYPE_TYPE3_Pos)                 /*!< GPIO_T::INTTYPE: TYPE3 Mask            */

#define GPIO_INTTYPE_TYPE4_Pos           (4)                                               /*!< GPIO_T::INTTYPE: TYPE4 Position        */
#define GPIO_INTTYPE_TYPE4_Msk           (0x1ul << GPIO_INTTYPE_TYPE4_Pos)                 /*!< GPIO_T::INTTYPE: TYPE4 Mask            */

#define GPIO_INTTYPE_TYPE5_Pos           (5)                                               /*!< GPIO_T::INTTYPE: TYPE5 Position        */
#define GPIO_INTTYPE_TYPE5_Msk           (0x1ul << GPIO_INTTYPE_TYPE5_Pos)                 /*!< GPIO_T::INTTYPE: TYPE5 Mask            */

#define GPIO_INTTYPE_TYPE6_Pos           (6)                                               /*!< GPIO_T::INTTYPE: TYPE6 Position        */
#define GPIO_INTTYPE_TYPE6_Msk           (0x1ul << GPIO_INTTYPE_TYPE6_Pos)                 /*!< GPIO_T::INTTYPE: TYPE6 Mask            */

#define GPIO_INTTYPE_TYPE7_Pos           (7)                                               /*!< GPIO_T::INTTYPE: TYPE7 Position        */
#define GPIO_INTTYPE_TYPE7_Msk           (0x1ul << GPIO_INTTYPE_TYPE7_Pos)                 /*!< GPIO_T::INTTYPE: TYPE7 Mask            */

#define GPIO_INTTYPE_TYPE8_Pos           (8)                                               /*!< GPIO_T::INTTYPE: TYPE8 Position        */
#define GPIO_INTTYPE_TYPE8_Msk           (0x1ul << GPIO_INTTYPE_TYPE8_Pos)                 /*!< GPIO_T::INTTYPE: TYPE8 Mask            */

#define GPIO_INTTYPE_TYPE9_Pos           (9)                                               /*!< GPIO_T::INTTYPE: TYPE9 Position        */
#define GPIO_INTTYPE_TYPE9_Msk           (0x1ul << GPIO_INTTYPE_TYPE9_Pos)                 /*!< GPIO_T::INTTYPE: TYPE9 Mask            */

#define GPIO_INTTYPE_TYPE10_Pos          (10)                                              /*!< GPIO_T::INTTYPE: TYPE10 Position       */
#define GPIO_INTTYPE_TYPE10_Msk          (0x1ul << GPIO_INTTYPE_TYPE10_Pos)                /*!< GPIO_T::INTTYPE: TYPE10 Mask           */

#define GPIO_INTTYPE_TYPE11_Pos          (11)                                              /*!< GPIO_T::INTTYPE: TYPE11 Position       */
#define GPIO_INTTYPE_TYPE11_Msk          (0x1ul << GPIO_INTTYPE_TYPE11_Pos)                /*!< GPIO_T::INTTYPE: TYPE11 Mask           */

#define GPIO_INTTYPE_TYPE12_Pos          (12)                                              /*!< GPIO_T::INTTYPE: TYPE12 Position       */
#define GPIO_INTTYPE_TYPE12_Msk          (0x1ul << GPIO_INTTYPE_TYPE12_Pos)                /*!< GPIO_T::INTTYPE: TYPE12 Mask           */

#define GPIO_INTTYPE_TYPE13_Pos          (13)                                              /*!< GPIO_T::INTTYPE: TYPE13 Position       */
#define GPIO_INTTYPE_TYPE13_Msk          (0x1ul << GPIO_INTTYPE_TYPE13_Pos)                /*!< GPIO_T::INTTYPE: TYPE13 Mask           */

#define GPIO_INTTYPE_TYPE14_Pos          (14)                                              /*!< GPIO_T::INTTYPE: TYPE14 Position       */
#define GPIO_INTTYPE_TYPE14_Msk          (0x1ul << GPIO_INTTYPE_TYPE14_Pos)                /*!< GPIO_T::INTTYPE: TYPE14 Mask           */

#define GPIO_INTTYPE_TYPE15_Pos          (15)                                              /*!< GPIO_T::INTTYPE: TYPE15 Position       */
#define GPIO_INTTYPE_TYPE15_Msk          (0x1ul << GPIO_INTTYPE_TYPE15_Pos)                /*!< GPIO_T::INTTYPE: TYPE15 Mask           */

#define GPIO_INTEN_FLIEN0_Pos            (0)                                               /*!< GPIO_T::INTEN: FLIEN0 Position         */
#define GPIO_INTEN_FLIEN0_Msk            (0x1ul << GPIO_INTEN_FLIEN0_Pos)                  /*!< GPIO_T::INTEN: FLIEN0 Mask             */

#define GPIO_INTEN_FLIEN1_Pos            (1)                                               /*!< GPIO_T::INTEN: FLIEN1 Position         */
#define GPIO_INTEN_FLIEN1_Msk            (0x1ul << GPIO_INTEN_FLIEN1_Pos)                  /*!< GPIO_T::INTEN: FLIEN1 Mask             */

#define GPIO_INTEN_FLIEN2_Pos            (2)                                               /*!< GPIO_T::INTEN: FLIEN2 Position         */
#define GPIO_INTEN_FLIEN2_Msk            (0x1ul << GPIO_INTEN_FLIEN2_Pos)                  /*!< GPIO_T::INTEN: FLIEN2 Mask             */

#define GPIO_INTEN_FLIEN3_Pos            (3)                                               /*!< GPIO_T::INTEN: FLIEN3 Position         */
#define GPIO_INTEN_FLIEN3_Msk            (0x1ul << GPIO_INTEN_FLIEN3_Pos)                  /*!< GPIO_T::INTEN: FLIEN3 Mask             */

#define GPIO_INTEN_FLIEN4_Pos            (4)                                               /*!< GPIO_T::INTEN: FLIEN4 Position         */
#define GPIO_INTEN_FLIEN4_Msk            (0x1ul << GPIO_INTEN_FLIEN4_Pos)                  /*!< GPIO_T::INTEN: FLIEN4 Mask             */

#define GPIO_INTEN_FLIEN5_Pos            (5)                                               /*!< GPIO_T::INTEN: FLIEN5 Position         */
#define GPIO_INTEN_FLIEN5_Msk            (0x1ul << GPIO_INTEN_FLIEN5_Pos)                  /*!< GPIO_T::INTEN: FLIEN5 Mask             */

#define GPIO_INTEN_FLIEN6_Pos            (6)                                               /*!< GPIO_T::INTEN: FLIEN6 Position         */
#define GPIO_INTEN_FLIEN6_Msk            (0x1ul << GPIO_INTEN_FLIEN6_Pos)                  /*!< GPIO_T::INTEN: FLIEN6 Mask             */

#define GPIO_INTEN_FLIEN7_Pos            (7)                                               /*!< GPIO_T::INTEN: FLIEN7 Position         */
#define GPIO_INTEN_FLIEN7_Msk            (0x1ul << GPIO_INTEN_FLIEN7_Pos)                  /*!< GPIO_T::INTEN: FLIEN7 Mask             */

#define GPIO_INTEN_FLIEN8_Pos            (8)                                               /*!< GPIO_T::INTEN: FLIEN8 Position         */
#define GPIO_INTEN_FLIEN8_Msk            (0x1ul << GPIO_INTEN_FLIEN8_Pos)                  /*!< GPIO_T::INTEN: FLIEN8 Mask             */

#define GPIO_INTEN_FLIEN9_Pos            (9)                                               /*!< GPIO_T::INTEN: FLIEN9 Position         */
#define GPIO_INTEN_FLIEN9_Msk            (0x1ul << GPIO_INTEN_FLIEN9_Pos)                  /*!< GPIO_T::INTEN: FLIEN9 Mask             */

#define GPIO_INTEN_FLIEN10_Pos           (10)                                              /*!< GPIO_T::INTEN: FLIEN10 Position        */
#define GPIO_INTEN_FLIEN10_Msk           (0x1ul << GPIO_INTEN_FLIEN10_Pos)                 /*!< GPIO_T::INTEN: FLIEN10 Mask            */

#define GPIO_INTEN_FLIEN11_Pos           (11)                                              /*!< GPIO_T::INTEN: FLIEN11 Position        */
#define GPIO_INTEN_FLIEN11_Msk           (0x1ul << GPIO_INTEN_FLIEN11_Pos)                 /*!< GPIO_T::INTEN: FLIEN11 Mask            */

#define GPIO_INTEN_FLIEN12_Pos           (12)                                              /*!< GPIO_T::INTEN: FLIEN12 Position        */
#define GPIO_INTEN_FLIEN12_Msk           (0x1ul << GPIO_INTEN_FLIEN12_Pos)                 /*!< GPIO_T::INTEN: FLIEN12 Mask            */

#define GPIO_INTEN_FLIEN13_Pos           (13)                                              /*!< GPIO_T::INTEN: FLIEN13 Position        */
#define GPIO_INTEN_FLIEN13_Msk           (0x1ul << GPIO_INTEN_FLIEN13_Pos)                 /*!< GPIO_T::INTEN: FLIEN13 Mask            */

#define GPIO_INTEN_FLIEN14_Pos           (14)                                              /*!< GPIO_T::INTEN: FLIEN14 Position        */
#define GPIO_INTEN_FLIEN14_Msk           (0x1ul << GPIO_INTEN_FLIEN14_Pos)                 /*!< GPIO_T::INTEN: FLIEN14 Mask            */

#define GPIO_INTEN_FLIEN15_Pos           (15)                                              /*!< GPIO_T::INTEN: FLIEN15 Position        */
#define GPIO_INTEN_FLIEN15_Msk           (0x1ul << GPIO_INTEN_FLIEN15_Pos)                 /*!< GPIO_T::INTEN: FLIEN15 Mask            */

#define GPIO_INTEN_RHIEN0_Pos            (16)                                              /*!< GPIO_T::INTEN: RHIEN0 Position         */
#define GPIO_INTEN_RHIEN0_Msk            (0x1ul << GPIO_INTEN_RHIEN0_Pos)                  /*!< GPIO_T::INTEN: RHIEN0 Mask             */

#define GPIO_INTEN_RHIEN1_Pos            (17)                                              /*!< GPIO_T::INTEN: RHIEN1 Position         */
#define GPIO_INTEN_RHIEN1_Msk            (0x1ul << GPIO_INTEN_RHIEN1_Pos)                  /*!< GPIO_T::INTEN: RHIEN1 Mask             */

#define GPIO_INTEN_RHIEN2_Pos            (18)                                              /*!< GPIO_T::INTEN: RHIEN2 Position         */
#define GPIO_INTEN_RHIEN2_Msk            (0x1ul << GPIO_INTEN_RHIEN2_Pos)                  /*!< GPIO_T::INTEN: RHIEN2 Mask             */

#define GPIO_INTEN_RHIEN3_Pos            (19)                                              /*!< GPIO_T::INTEN: RHIEN3 Position         */
#define GPIO_INTEN_RHIEN3_Msk            (0x1ul << GPIO_INTEN_RHIEN3_Pos)                  /*!< GPIO_T::INTEN: RHIEN3 Mask             */

#define GPIO_INTEN_RHIEN4_Pos            (20)                                              /*!< GPIO_T::INTEN: RHIEN4 Position         */
#define GPIO_INTEN_RHIEN4_Msk            (0x1ul << GPIO_INTEN_RHIEN4_Pos)                  /*!< GPIO_T::INTEN: RHIEN4 Mask             */

#define GPIO_INTEN_RHIEN5_Pos            (21)                                              /*!< GPIO_T::INTEN: RHIEN5 Position         */
#define GPIO_INTEN_RHIEN5_Msk            (0x1ul << GPIO_INTEN_RHIEN5_Pos)                  /*!< GPIO_T::INTEN: RHIEN5 Mask             */

#define GPIO_INTEN_RHIEN6_Pos            (22)                                              /*!< GPIO_T::INTEN: RHIEN6 Position         */
#define GPIO_INTEN_RHIEN6_Msk            (0x1ul << GPIO_INTEN_RHIEN6_Pos)                  /*!< GPIO_T::INTEN: RHIEN6 Mask             */

#define GPIO_INTEN_RHIEN7_Pos            (23)                                              /*!< GPIO_T::INTEN: RHIEN7 Position         */
#define GPIO_INTEN_RHIEN7_Msk            (0x1ul << GPIO_INTEN_RHIEN7_Pos)                  /*!< GPIO_T::INTEN: RHIEN7 Mask             */

#define GPIO_INTEN_RHIEN8_Pos            (24)                                              /*!< GPIO_T::INTEN: RHIEN8 Position         */
#define GPIO_INTEN_RHIEN8_Msk            (0x1ul << GPIO_INTEN_RHIEN8_Pos)                  /*!< GPIO_T::INTEN: RHIEN8 Mask             */

#define GPIO_INTEN_RHIEN9_Pos            (25)                                              /*!< GPIO_T::INTEN: RHIEN9 Position         */
#define GPIO_INTEN_RHIEN9_Msk            (0x1ul << GPIO_INTEN_RHIEN9_Pos)                  /*!< GPIO_T::INTEN: RHIEN9 Mask             */

#define GPIO_INTEN_RHIEN10_Pos           (26)                                              /*!< GPIO_T::INTEN: RHIEN10 Position        */
#define GPIO_INTEN_RHIEN10_Msk           (0x1ul << GPIO_INTEN_RHIEN10_Pos)                 /*!< GPIO_T::INTEN: RHIEN10 Mask            */

#define GPIO_INTEN_RHIEN11_Pos           (27)                                              /*!< GPIO_T::INTEN: RHIEN11 Position        */
#define GPIO_INTEN_RHIEN11_Msk           (0x1ul << GPIO_INTEN_RHIEN11_Pos)                 /*!< GPIO_T::INTEN: RHIEN11 Mask            */

#define GPIO_INTEN_RHIEN12_Pos           (28)                                              /*!< GPIO_T::INTEN: RHIEN12 Position        */
#define GPIO_INTEN_RHIEN12_Msk           (0x1ul << GPIO_INTEN_RHIEN12_Pos)                 /*!< GPIO_T::INTEN: RHIEN12 Mask            */

#define GPIO_INTEN_RHIEN13_Pos           (29)                                              /*!< GPIO_T::INTEN: RHIEN13 Position        */
#define GPIO_INTEN_RHIEN13_Msk           (0x1ul << GPIO_INTEN_RHIEN13_Pos)                 /*!< GPIO_T::INTEN: RHIEN13 Mask            */

#define GPIO_INTEN_RHIEN14_Pos           (30)                                              /*!< GPIO_T::INTEN: RHIEN14 Position        */
#define GPIO_INTEN_RHIEN14_Msk           (0x1ul << GPIO_INTEN_RHIEN14_Pos)                 /*!< GPIO_T::INTEN: RHIEN14 Mask            */

#define GPIO_INTEN_RHIEN15_Pos           (31)                                              /*!< GPIO_T::INTEN: RHIEN15 Position        */
#define GPIO_INTEN_RHIEN15_Msk           (0x1ul << GPIO_INTEN_RHIEN15_Pos)                 /*!< GPIO_T::INTEN: RHIEN15 Mask            */

#define GPIO_INTSRC_INTSRC0_Pos          (0)                                               /*!< GPIO_T::INTSRC: INTSRC0 Position       */
#define GPIO_INTSRC_INTSRC0_Msk          (0x1ul << GPIO_INTSRC_INTSRC0_Pos)                /*!< GPIO_T::INTSRC: INTSRC0 Mask           */

#define GPIO_INTSRC_INTSRC1_Pos          (1)                                               /*!< GPIO_T::INTSRC: INTSRC1 Position       */
#define GPIO_INTSRC_INTSRC1_Msk          (0x1ul << GPIO_INTSRC_INTSRC1_Pos)                /*!< GPIO_T::INTSRC: INTSRC1 Mask           */

#define GPIO_INTSRC_INTSRC2_Pos          (2)                                               /*!< GPIO_T::INTSRC: INTSRC2 Position       */
#define GPIO_INTSRC_INTSRC2_Msk          (0x1ul << GPIO_INTSRC_INTSRC2_Pos)                /*!< GPIO_T::INTSRC: INTSRC2 Mask           */

#define GPIO_INTSRC_INTSRC3_Pos          (3)                                               /*!< GPIO_T::INTSRC: INTSRC3 Position       */
#define GPIO_INTSRC_INTSRC3_Msk          (0x1ul << GPIO_INTSRC_INTSRC3_Pos)                /*!< GPIO_T::INTSRC: INTSRC3 Mask           */

#define GPIO_INTSRC_INTSRC4_Pos          (4)                                               /*!< GPIO_T::INTSRC: INTSRC4 Position       */
#define GPIO_INTSRC_INTSRC4_Msk          (0x1ul << GPIO_INTSRC_INTSRC4_Pos)                /*!< GPIO_T::INTSRC: INTSRC4 Mask           */

#define GPIO_INTSRC_INTSRC5_Pos          (5)                                               /*!< GPIO_T::INTSRC: INTSRC5 Position       */
#define GPIO_INTSRC_INTSRC5_Msk          (0x1ul << GPIO_INTSRC_INTSRC5_Pos)                /*!< GPIO_T::INTSRC: INTSRC5 Mask           */

#define GPIO_INTSRC_INTSRC6_Pos          (6)                                               /*!< GPIO_T::INTSRC: INTSRC6 Position       */
#define GPIO_INTSRC_INTSRC6_Msk          (0x1ul << GPIO_INTSRC_INTSRC6_Pos)                /*!< GPIO_T::INTSRC: INTSRC6 Mask           */

#define GPIO_INTSRC_INTSRC7_Pos          (7)                                               /*!< GPIO_T::INTSRC: INTSRC7 Position       */
#define GPIO_INTSRC_INTSRC7_Msk          (0x1ul << GPIO_INTSRC_INTSRC7_Pos)                /*!< GPIO_T::INTSRC: INTSRC7 Mask           */

#define GPIO_INTSRC_INTSRC8_Pos          (8)                                               /*!< GPIO_T::INTSRC: INTSRC8 Position       */
#define GPIO_INTSRC_INTSRC8_Msk          (0x1ul << GPIO_INTSRC_INTSRC8_Pos)                /*!< GPIO_T::INTSRC: INTSRC8 Mask           */

#define GPIO_INTSRC_INTSRC9_Pos          (9)                                               /*!< GPIO_T::INTSRC: INTSRC9 Position       */
#define GPIO_INTSRC_INTSRC9_Msk          (0x1ul << GPIO_INTSRC_INTSRC9_Pos)                /*!< GPIO_T::INTSRC: INTSRC9 Mask           */

#define GPIO_INTSRC_INTSRC10_Pos         (10)                                              /*!< GPIO_T::INTSRC: INTSRC10 Position      */
#define GPIO_INTSRC_INTSRC10_Msk         (0x1ul << GPIO_INTSRC_INTSRC10_Pos)               /*!< GPIO_T::INTSRC: INTSRC10 Mask          */

#define GPIO_INTSRC_INTSRC11_Pos         (11)                                              /*!< GPIO_T::INTSRC: INTSRC11 Position      */
#define GPIO_INTSRC_INTSRC11_Msk         (0x1ul << GPIO_INTSRC_INTSRC11_Pos)               /*!< GPIO_T::INTSRC: INTSRC11 Mask          */

#define GPIO_INTSRC_INTSRC12_Pos         (12)                                              /*!< GPIO_T::INTSRC: INTSRC12 Position      */
#define GPIO_INTSRC_INTSRC12_Msk         (0x1ul << GPIO_INTSRC_INTSRC12_Pos)               /*!< GPIO_T::INTSRC: INTSRC12 Mask          */

#define GPIO_INTSRC_INTSRC13_Pos         (13)                                              /*!< GPIO_T::INTSRC: INTSRC13 Position      */
#define GPIO_INTSRC_INTSRC13_Msk         (0x1ul << GPIO_INTSRC_INTSRC13_Pos)               /*!< GPIO_T::INTSRC: INTSRC13 Mask          */

#define GPIO_INTSRC_INTSRC14_Pos         (14)                                              /*!< GPIO_T::INTSRC: INTSRC14 Position      */
#define GPIO_INTSRC_INTSRC14_Msk         (0x1ul << GPIO_INTSRC_INTSRC14_Pos)               /*!< GPIO_T::INTSRC: INTSRC14 Mask          */

#define GPIO_INTSRC_INTSRC15_Pos         (15)                                              /*!< GPIO_T::INTSRC: INTSRC15 Position      */
#define GPIO_INTSRC_INTSRC15_Msk         (0x1ul << GPIO_INTSRC_INTSRC15_Pos)               /*!< GPIO_T::INTSRC: INTSRC15 Mask          */

#define GPIO_SMTEN_SMTEN0_Pos            (0)                                               /*!< GPIO_T::SMTEN: SMTEN0 Position         */
#define GPIO_SMTEN_SMTEN0_Msk            (0x1ul << GPIO_SMTEN_SMTEN0_Pos)                  /*!< GPIO_T::SMTEN: SMTEN0 Mask             */

#define GPIO_SMTEN_SMTEN1_Pos            (1)                                               /*!< GPIO_T::SMTEN: SMTEN1 Position         */
#define GPIO_SMTEN_SMTEN1_Msk            (0x1ul << GPIO_SMTEN_SMTEN1_Pos)                  /*!< GPIO_T::SMTEN: SMTEN1 Mask             */

#define GPIO_SMTEN_SMTEN2_Pos            (2)                                               /*!< GPIO_T::SMTEN: SMTEN2 Position         */
#define GPIO_SMTEN_SMTEN2_Msk            (0x1ul << GPIO_SMTEN_SMTEN2_Pos)                  /*!< GPIO_T::SMTEN: SMTEN2 Mask             */

#define GPIO_SMTEN_SMTEN3_Pos            (3)                                               /*!< GPIO_T::SMTEN: SMTEN3 Position         */
#define GPIO_SMTEN_SMTEN3_Msk            (0x1ul << GPIO_SMTEN_SMTEN3_Pos)                  /*!< GPIO_T::SMTEN: SMTEN3 Mask             */

#define GPIO_SMTEN_SMTEN4_Pos            (4)                                               /*!< GPIO_T::SMTEN: SMTEN4 Position         */
#define GPIO_SMTEN_SMTEN4_Msk            (0x1ul << GPIO_SMTEN_SMTEN4_Pos)                  /*!< GPIO_T::SMTEN: SMTEN4 Mask             */

#define GPIO_SMTEN_SMTEN5_Pos            (5)                                               /*!< GPIO_T::SMTEN: SMTEN5 Position         */
#define GPIO_SMTEN_SMTEN5_Msk            (0x1ul << GPIO_SMTEN_SMTEN5_Pos)                  /*!< GPIO_T::SMTEN: SMTEN5 Mask             */

#define GPIO_SMTEN_SMTEN6_Pos            (6)                                               /*!< GPIO_T::SMTEN: SMTEN6 Position         */
#define GPIO_SMTEN_SMTEN6_Msk            (0x1ul << GPIO_SMTEN_SMTEN6_Pos)                  /*!< GPIO_T::SMTEN: SMTEN6 Mask             */

#define GPIO_SMTEN_SMTEN7_Pos            (7)                                               /*!< GPIO_T::SMTEN: SMTEN7 Position         */
#define GPIO_SMTEN_SMTEN7_Msk            (0x1ul << GPIO_SMTEN_SMTEN7_Pos)                  /*!< GPIO_T::SMTEN: SMTEN7 Mask             */

#define GPIO_SMTEN_SMTEN8_Pos            (8)                                               /*!< GPIO_T::SMTEN: SMTEN8 Position         */
#define GPIO_SMTEN_SMTEN8_Msk            (0x1ul << GPIO_SMTEN_SMTEN8_Pos)                  /*!< GPIO_T::SMTEN: SMTEN8 Mask             */

#define GPIO_SMTEN_SMTEN9_Pos            (9)                                               /*!< GPIO_T::SMTEN: SMTEN9 Position         */
#define GPIO_SMTEN_SMTEN9_Msk            (0x1ul << GPIO_SMTEN_SMTEN9_Pos)                  /*!< GPIO_T::SMTEN: SMTEN9 Mask             */

#define GPIO_SMTEN_SMTEN10_Pos           (10)                                              /*!< GPIO_T::SMTEN: SMTEN10 Position        */
#define GPIO_SMTEN_SMTEN10_Msk           (0x1ul << GPIO_SMTEN_SMTEN10_Pos)                 /*!< GPIO_T::SMTEN: SMTEN10 Mask            */

#define GPIO_SMTEN_SMTEN11_Pos           (11)                                              /*!< GPIO_T::SMTEN: SMTEN11 Position        */
#define GPIO_SMTEN_SMTEN11_Msk           (0x1ul << GPIO_SMTEN_SMTEN11_Pos)                 /*!< GPIO_T::SMTEN: SMTEN11 Mask            */

#define GPIO_SMTEN_SMTEN12_Pos           (12)                                              /*!< GPIO_T::SMTEN: SMTEN12 Position        */
#define GPIO_SMTEN_SMTEN12_Msk           (0x1ul << GPIO_SMTEN_SMTEN12_Pos)                 /*!< GPIO_T::SMTEN: SMTEN12 Mask            */

#define GPIO_SMTEN_SMTEN13_Pos           (13)                                              /*!< GPIO_T::SMTEN: SMTEN13 Position        */
#define GPIO_SMTEN_SMTEN13_Msk           (0x1ul << GPIO_SMTEN_SMTEN13_Pos)                 /*!< GPIO_T::SMTEN: SMTEN13 Mask            */

#define GPIO_SMTEN_SMTEN14_Pos           (14)                                              /*!< GPIO_T::SMTEN: SMTEN14 Position        */
#define GPIO_SMTEN_SMTEN14_Msk           (0x1ul << GPIO_SMTEN_SMTEN14_Pos)                 /*!< GPIO_T::SMTEN: SMTEN14 Mask            */

#define GPIO_SMTEN_SMTEN15_Pos           (15)                                              /*!< GPIO_T::SMTEN: SMTEN15 Position        */
#define GPIO_SMTEN_SMTEN15_Msk           (0x1ul << GPIO_SMTEN_SMTEN15_Pos)                 /*!< GPIO_T::SMTEN: SMTEN15 Mask            */

#define GPIO_SLEWCTL_HSREN0_Pos          (0)                                               /*!< GPIO_T::SLEWCTL: HSREN0 Position       */
#define GPIO_SLEWCTL_HSREN0_Msk          (0x1ul << GPIO_SLEWCTL_HSREN0_Pos)                /*!< GPIO_T::SLEWCTL: HSREN0 Mask           */

#define GPIO_SLEWCTL_HSREN1_Pos          (1)                                               /*!< GPIO_T::SLEWCTL: HSREN1 Position       */
#define GPIO_SLEWCTL_HSREN1_Msk          (0x1ul << GPIO_SLEWCTL_HSREN1_Pos)                /*!< GPIO_T::SLEWCTL: HSREN1 Mask           */

#define GPIO_SLEWCTL_HSREN2_Pos          (2)                                               /*!< GPIO_T::SLEWCTL: HSREN2 Position       */
#define GPIO_SLEWCTL_HSREN2_Msk          (0x1ul << GPIO_SLEWCTL_HSREN2_Pos)                /*!< GPIO_T::SLEWCTL: HSREN2 Mask           */

#define GPIO_SLEWCTL_HSREN3_Pos          (3)                                               /*!< GPIO_T::SLEWCTL: HSREN3 Position       */
#define GPIO_SLEWCTL_HSREN3_Msk          (0x1ul << GPIO_SLEWCTL_HSREN3_Pos)                /*!< GPIO_T::SLEWCTL: HSREN3 Mask           */

#define GPIO_SLEWCTL_HSREN4_Pos          (4)                                               /*!< GPIO_T::SLEWCTL: HSREN4 Position       */
#define GPIO_SLEWCTL_HSREN4_Msk          (0x1ul << GPIO_SLEWCTL_HSREN4_Pos)                /*!< GPIO_T::SLEWCTL: HSREN4 Mask           */

#define GPIO_SLEWCTL_HSREN5_Pos          (5)                                               /*!< GPIO_T::SLEWCTL: HSREN5 Position       */
#define GPIO_SLEWCTL_HSREN5_Msk          (0x1ul << GPIO_SLEWCTL_HSREN5_Pos)                /*!< GPIO_T::SLEWCTL: HSREN5 Mask           */

#define GPIO_SLEWCTL_HSREN6_Pos          (6)                                               /*!< GPIO_T::SLEWCTL: HSREN6 Position       */
#define GPIO_SLEWCTL_HSREN6_Msk          (0x1ul << GPIO_SLEWCTL_HSREN6_Pos)                /*!< GPIO_T::SLEWCTL: HSREN6 Mask           */

#define GPIO_SLEWCTL_HSREN7_Pos          (7)                                               /*!< GPIO_T::SLEWCTL: HSREN7 Position       */
#define GPIO_SLEWCTL_HSREN7_Msk          (0x1ul << GPIO_SLEWCTL_HSREN7_Pos)                /*!< GPIO_T::SLEWCTL: HSREN7 Mask           */

#define GPIO_SLEWCTL_HSREN8_Pos          (8)                                               /*!< GPIO_T::SLEWCTL: HSREN8 Position       */
#define GPIO_SLEWCTL_HSREN8_Msk          (0x1ul << GPIO_SLEWCTL_HSREN8_Pos)                /*!< GPIO_T::SLEWCTL: HSREN8 Mask           */

#define GPIO_SLEWCTL_HSREN9_Pos          (9)                                               /*!< GPIO_T::SLEWCTL: HSREN9 Position       */
#define GPIO_SLEWCTL_HSREN9_Msk          (0x1ul << GPIO_SLEWCTL_HSREN9_Pos)                /*!< GPIO_T::SLEWCTL: HSREN9 Mask           */

#define GPIO_SLEWCTL_HSREN10_Pos         (10)                                              /*!< GPIO_T::SLEWCTL: HSREN10 Position      */
#define GPIO_SLEWCTL_HSREN10_Msk         (0x1ul << GPIO_SLEWCTL_HSREN10_Pos)               /*!< GPIO_T::SLEWCTL: HSREN10 Mask          */

#define GPIO_SLEWCTL_HSREN11_Pos         (11)                                              /*!< GPIO_T::SLEWCTL: HSREN11 Position      */
#define GPIO_SLEWCTL_HSREN11_Msk         (0x1ul << GPIO_SLEWCTL_HSREN11_Pos)               /*!< GPIO_T::SLEWCTL: HSREN11 Mask          */

#define GPIO_SLEWCTL_HSREN12_Pos         (12)                                              /*!< GPIO_T::SLEWCTL: HSREN12 Position      */
#define GPIO_SLEWCTL_HSREN12_Msk         (0x1ul << GPIO_SLEWCTL_HSREN12_Pos)               /*!< GPIO_T::SLEWCTL: HSREN12 Mask          */

#define GPIO_SLEWCTL_HSREN13_Pos         (13)                                              /*!< GPIO_T::SLEWCTL: HSREN13 Position      */
#define GPIO_SLEWCTL_HSREN13_Msk         (0x1ul << GPIO_SLEWCTL_HSREN13_Pos)               /*!< GPIO_T::SLEWCTL: HSREN13 Mask          */

#define GPIO_SLEWCTL_HSREN14_Pos         (14)                                              /*!< GPIO_T::SLEWCTL: HSREN14 Position      */
#define GPIO_SLEWCTL_HSREN14_Msk         (0x1ul << GPIO_SLEWCTL_HSREN14_Pos)               /*!< GPIO_T::SLEWCTL: HSREN14 Mask          */

#define GPIO_SLEWCTL_HSREN15_Pos         (15)                                              /*!< GPIO_T::SLEWCTL: HSREN15 Position      */
#define GPIO_SLEWCTL_HSREN15_Msk         (0x1ul << GPIO_SLEWCTL_HSREN15_Pos)               /*!< GPIO_T::SLEWCTL: HSREN15 Mask          */

#define GPIO_DRVCTL_HDRVEN8_Pos          (8)                                               /*!< GPIO_T::DRVCTL: HDRVEN8 Position       */
#define GPIO_DRVCTL_HDRVEN8_Msk          (0x1ul << GPIO_DRVCTL_HDRVEN8_Pos)                /*!< GPIO_T::DRVCTL: HDRVEN8 Mask           */

#define GPIO_DRVCTL_HDRVEN9_Pos          (9)                                               /*!< GPIO_T::DRVCTL: HDRVEN9 Position       */
#define GPIO_DRVCTL_HDRVEN9_Msk          (0x1ul << GPIO_DRVCTL_HDRVEN9_Pos)                /*!< GPIO_T::DRVCTL: HDRVEN9 Mask           */

#define GPIO_DRVCTL_HDRVEN10_Pos         (10)                                              /*!< GPIO_T::DRVCTL: HDRVEN10 Position      */
#define GPIO_DRVCTL_HDRVEN10_Msk         (0x1ul << GPIO_DRVCTL_HDRVEN10_Pos)               /*!< GPIO_T::DRVCTL: HDRVEN10 Mask          */

#define GPIO_DRVCTL_HDRVEN11_Pos         (11)                                              /*!< GPIO_T::DRVCTL: HDRVEN11 Position      */
#define GPIO_DRVCTL_HDRVEN11_Msk         (0x1ul << GPIO_DRVCTL_HDRVEN11_Pos)               /*!< GPIO_T::DRVCTL: HDRVEN11 Mask          */

#define GPIO_DRVCTL_HDRVEN12_Pos         (12)                                              /*!< GPIO_T::DRVCTL: HDRVEN12 Position      */
#define GPIO_DRVCTL_HDRVEN12_Msk         (0x1ul << GPIO_DRVCTL_HDRVEN12_Pos)               /*!< GPIO_T::DRVCTL: HDRVEN12 Mask          */

#define GPIO_DRVCTL_HDRVEN13_Pos         (13)                                              /*!< GPIO_T::DRVCTL: HDRVEN13 Position      */
#define GPIO_DRVCTL_HDRVEN13_Msk         (0x1ul << GPIO_DRVCTL_HDRVEN13_Pos)               /*!< GPIO_T::DRVCTL: HDRVEN13 Mask          */

#define GPIO_DBCTL_DBCLKSEL_Pos          (0)                                               /*!< GPIO_DBCTL_T::DBCTL: DBCLKSEL Position */
#define GPIO_DBCTL_DBCLKSEL_Msk          (0xful << GPIO_DBCTL_DBCLKSEL_Pos)                /*!< GPIO_DBCTL_T::DBCTL: DBCLKSEL Mask     */

#define GPIO_DBCTL_DBCLKSRC_Pos          (4)                                               /*!< GPIO_DBCTL_T::DBCTL: DBCLKSRC Position */
#define GPIO_DBCTL_DBCLKSRC_Msk          (0x1ul << GPIO_DBCTL_DBCLKSRC_Pos)                /*!< GPIO_DBCTL_T::DBCTL: DBCLKSRC Mask     */

#define GPIO_DBCTL_ICLKON_Pos            (5)                                               /*!< GPIO_DBCTL_T::DBCTL: ICLKON Position   */
#define GPIO_DBCTL_ICLKON_Msk            (0x1ul << GPIO_DBCTL_ICLKON_Pos)                  /*!< GPIO_DBCTL_T::DBCTL: ICLKON Mask       */


/**@}*/ /* GPIO_CONST */
/**@}*/ /* end of GPIO register group */

/*---------------------- Hardware Divider --------------------------------*/
/**
    @addtogroup HDIV Hardware Divider(HDIV)
    Memory Mapped Structure for HDIV Controller
@{ */

typedef struct
{


    /**
     * @var HDIV_T::DIVIDEND
     * Offset: 0x00  Dividend Source Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |DIVIDEND  |Dividend Source
     * |        |          |This register is given the dividend of divider before calculation starting.
     * @var HDIV_T::DIVISOR
     * Offset: 0x04  Divisor Source Resister
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |DIVISOR   |Divisor Source
     * |        |          |This register is given the divisor of divider before calculation starts.
     * |        |          |Note: When this register is written, hardware divider will start calculate.
     * @var HDIV_T::DIVQUO
     * Offset: 0x08  Quotient Result Resister
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |QUOTIENT  |Quotient Result
     * |        |          |This register holds the quotient result of divider after calculation complete.
     * @var HDIV_T::DIVREM
     * Offset: 0x0C  Remainder Result Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |REMAINDER |Remainder Result
     * |        |          |The remainder of hardware divider is 16-bit sign integer (REMAINDER[15:0]), which holds the remainder result of divider after calculation complete.
     * |        |          |The remainder of hardware divider with sign extension (REMAINDER[31:16]) to 32-bit integer.
     * |        |          |This register holds the remainder result of divider after calculation complete.
     * @var HDIV_T::DIVSTS
     * Offset: 0x10  Divider Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |FINISH    |Division Finish Flag
     * |        |          |0 = Under Calculation.
     * |        |          |1 = Calculation finished.
     * |        |          |The flag will become low when the divider is in calculation.
     * |        |          |The flag will go back to high once the calculation finished.
     * |[1]     |DIV0      |Divisor Zero Warning
     * |        |          |0 = The divisor is not 0.
     * |        |          |1 = The divisor is 0.
     * |        |          |Note: The DIV0 flag is used to indicate divide-by-zero situation and updated whenever DIVISOR is written
     * |        |          |This register is read only.
     */
    __IO uint32_t DIVIDEND;              /*!< [0x0000] Dividend Source Register                                         */
    __IO uint32_t DIVISOR;               /*!< [0x0004] Divisor Source Resister                                          */
    __IO uint32_t DIVQUO;                /*!< [0x0008] Quotient Result Resister                                         */
    __IO uint32_t DIVREM;                /*!< [0x000c] Remainder Result Register                                        */
    __I  uint32_t DIVSTS;                /*!< [0x0010] Divider Status Register                                          */

} HDIV_T;

/**
    @addtogroup HDIV_CONST HDIV Bit Field Definition
    Constant Definitions for HDIV Controller
@{ */

#define HDIV_DIVIDEND_DIVIDEND_Pos       (0)                                               /*!< HDIV_T::DIVIDEND: DIVIDEND Position    */
#define HDIV_DIVIDEND_DIVIDEND_Msk       (0xfffffffful << HDIV_DIVIDEND_DIVIDEND_Pos)      /*!< HDIV_T::DIVIDEND: DIVIDEND Mask        */

#define HDIV_DIVISOR_DIVISOR_Pos         (0)                                               /*!< HDIV_T::DIVISOR: DIVISOR Position      */
#define HDIV_DIVISOR_DIVISOR_Msk         (0xfffful << HDIV_DIVISOR_DIVISOR_Pos)            /*!< HDIV_T::DIVISOR: DIVISOR Mask          */

#define HDIV_DIVQUO_QUOTIENT_Pos         (0)                                               /*!< HDIV_T::DIVQUO: QUOTIENT Position      */
#define HDIV_DIVQUO_QUOTIENT_Msk         (0xfffffffful << HDIV_DIVQUO_QUOTIENT_Pos)        /*!< HDIV_T::DIVQUO: QUOTIENT Mask          */

#define HDIV_DIVREM_REMAINDER_Pos        (0)                                               /*!< HDIV_T::DIVREM: REMAINDER Position     */
#define HDIV_DIVREM_REMAINDER_Msk        (0xfffffffful << HDIV_DIVREM_REMAINDER_Pos)       /*!< HDIV_T::DIVREM: REMAINDER Mask         */

#define HDIV_DIVSTS_FINISH_Pos           (0)                                               /*!< HDIV_T::DIVSTS: FINISH Position        */
#define HDIV_DIVSTS_FINISH_Msk           (0x1ul << HDIV_DIVSTS_FINISH_Pos)                 /*!< HDIV_T::DIVSTS: FINISH Mask            */

#define HDIV_DIVSTS_DIV0_Pos             (1)                                               /*!< HDIV_T::DIVSTS: DIV0 Position          */
#define HDIV_DIVSTS_DIV0_Msk             (0x1ul << HDIV_DIVSTS_DIV0_Pos)                   /*!< HDIV_T::DIVSTS: DIV0 Mask              */

/**@}*/ /* HDIV_CONST */
/**@}*/ /* end of HDIV register group */


/*---------------------- Inter-IC Bus Controller -------------------------*/
/**
    @addtogroup I2C Inter-IC Bus Controller(I2C)
    Memory Mapped Structure for I2C Controller
@{ */

typedef struct
{


    /**
     * @var I2C_T::CTL
     * Offset: 0x00  I2C Control Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2]     |AA        |Assert Acknowledge Control
     * |        |          |When AA=1 prior to address or data is received, an acknowledged (low level to SDA) will be returned during the acknowledge clock pulse on the SCL line when 1.) A slave is acknowledging the address sent from master, 2.) The receiver devices are acknowledging the data sent by transmitter,
     * |        |          |When AA=0 prior to address or data received, a Not acknowledged (high level to SDA) will be returned during the acknowledge clock pulse on the SCL line.
     * |[3]     |SI        |I2C Interrupt Flag
     * |        |          |When a new I2C state is present in the I2C_STATUS register, the SI flag is set by hardware.
     * |        |          |If bit INTEN (I2C_CTL [7]) is set, the I2C interrupt is requested.
     * |        |          |SI must be cleared by software.Clear SI by writing 1 to this bit.
     * |        |          |For ACKMEN is set in slave read mode, the SI flag is set in 8th clock period for user to confirm the acknowledge bit and 9th clock period for user to read the data in the data buffer.
     * |[4]     |STO       |I2C STOP Control
     * |        |          |In Master mode, setting STO to transmit a STOP condition to bus then I2C controller will check the bus condition if a STOP condition is detected.
     * |        |          |This bit will be cleared by hardware automatically.
     * |[5]     |STA       |I2C START Control
     * |        |          |Setting STA to logic 1 to enter Master mode, the I2C hardware sends a START or repeat START condition to bus when the bus is free.
     * |[6]     |I2CEN     |I2C Controller Enable Bit
     * |        |          |Set to enable I2C serial function controller.When I2CEN=1 the I2C serial function enable.
     * |        |          |The multi-function pin function must set to SDA, and SCL of I2C function first.
     * |        |          |0 = I2C serial function Disabled.
     * |        |          |1 = I2C serial function Enabled.
     * |[7]     |INTEN     |Enable Interrupt
     * |        |          |0 = I2C interrupt Disabled.
     * |        |          |1 = I2C interrupt Enabled.
     * @var I2C_T::ADDR0
     * Offset: 0x04  I2C Slave Address Register0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |GC        |General Call Function
     * |        |          |0 = General Call Function Disabled.
     * |        |          |1 = General Call Function Enabled.
     * |[7:1]   |ADDR      |I2C Address
     * |        |          |The content of this register is irrelevant when I2C is in Master mode.
     * |        |          |In the slave mode, the seven most significant bits must be loaded with the chip's own address.
     * |        |          |The I2C hardware will react if either of the address is matched.
     * @var I2C_T::DAT
     * Offset: 0x08  I2C Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |DAT       |I2C Data
     * |        |          |Bit [7:0] is located with the 8-bit transferred/received data of I2C serial port.
     * @var I2C_T::STATUS
     * Offset: 0x0C  I2C Status Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |STATUS    |I2C Status
     * |        |          |There are 28 possible status codes. When the content of I2C_STATUS is F8H, no serial interrupt is requested.
     * |        |          |Others I2C_STATUS values correspond to defined I2C states.
     * |        |          |When each of these states is entered, a status interrupt is requested (SI = 1).
     * |        |          |A valid status code is present in I2C_STATUS one cycle after SI is set by hardware and is still present one cycle after SI has been reset by software.
     * |        |          |In addition, states 00H stands for a Bus Error. A Bus Error occurs when a START or STOP condition is present at an illegal position in the formation frame.
     * |        |          |Example of illegal position are during the serial transfer of an address byte, a data byte or an acknowledge bit.
     * @var I2C_T::CLKDIV
     * Offset: 0x10  I2C Clock Divided Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |DIVIDER   |I2C Clock Divided
     * |        |          |Indicates the I2C clock rate: Data Baud Rate of I2C = (system clock) / (4x (I2C_CLKDIV+1)).
     * |        |          |Note: The minimum value of I2C_CLKDIV is 4.
     * @var I2C_T::TOCTL
     * Offset: 0x14  I2C Time-out Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TOIF      |Time-out Flag
     * |        |          |This bit is set by hardware when I2C time-out happened and it can interrupt CPU if I2C interrupt enable bit (INTEN) is set to 1.
     * |        |          |Note: Software can write 1 to clear this bit.
     * |[1]     |TOCDIV4   |Time-out Counter Input Clock Divided by 4
     * |        |          |When Enabled, The time-out period is extend 4 times.
     * |        |          |0 = Time-out period is extend 4 times Disabled.
     * |        |          |1 = Time-out period is extend 4 times Enabled.
     * |[2]     |TOCEN     |Time-out Counter Enable Bit
     * |        |          |When Enabled, the 14-bit time-out counter will start counting when SI is clear.
     * |        |          |Setting flag SI to '1' will reset counter and re-start up counting after SI is cleared.
     * |        |          |0 = Time-out counter Disabled.
     * |        |          |1 = Time-out counter Enabled.
     * @var I2C_T::ADDR1
     * Offset: 0x18  I2C Slave Address Register1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |GC        |General Call Function
     * |        |          |0 = General Call Function Disabled.
     * |        |          |1 = General Call Function Enabled.
     * |[7:1]   |ADDR      |I2C Address
     * |        |          |The content of this register is irrelevant when I2C is in Master mode.
     * |        |          |In the slave mode, the seven most significant bits must be loaded with the chip's own address.
     * |        |          |The I2C hardware will react if either of the address is matched.
     * @var I2C_T::ADDR2
     * Offset: 0x1C  I2C Slave Address Register2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |GC        |General Call Function
     * |        |          |0 = General Call Function Disabled.
     * |        |          |1 = General Call Function Enabled.
     * |[7:1]   |ADDR      |I2C Address
     * |        |          |The content of this register is irrelevant when I2C is in Master mode.
     * |        |          |In the slave mode, the seven most significant bits must be loaded with the chip's own address.
     * |        |          |The I2C hardware will react if either of the address is matched.
     * @var I2C_T::ADDR3
     * Offset: 0x20  I2C Slave Address Register3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |GC        |General Call Function
     * |        |          |0 = General Call Function Disabled.
     * |        |          |1 = General Call Function Enabled.
     * |[7:1]   |ADDR      |I2C Address
     * |        |          |The content of this register is irrelevant when I2C is in Master mode.
     * |        |          |In the slave mode, the seven most significant bits must be loaded with the chip's own address.
     * |        |          |The I2C hardware will react if either of the address is matched.
     * @var I2C_T::ADDRMSK0
     * Offset: 0x24  I2C Slave Address Mask Register0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:1]   |ADDRMSK   |I2C Address Mask
     * |        |          |0 = Mask Disabled (the received corresponding register bit should be exact the same as address register.).
     * |        |          |1 = Mask Enabled (the received corresponding address bit is don't care.).
     * |        |          |I2C bus controllers support multiple address recognition with four address mask register.
     * |        |          |When the bit in the address mask register is set to one, it means the received corresponding address bit is don't-care.
     * |        |          |If the bit is set to zero, that means the received corresponding register bit should be exact the same as address register.
     * @var I2C_T::ADDRMSK1
     * Offset: 0x28  I2C Slave Address Mask Register1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:1]   |ADDRMSK   |I2C Address Mask
     * |        |          |0 = Mask Disabled (the received corresponding register bit should be exact the same as address register.).
     * |        |          |1 = Mask Enabled (the received corresponding address bit is don't care.).
     * |        |          |I2C bus controllers support multiple address recognition with four address mask register.
     * |        |          |When the bit in the address mask register is set to one, it means the received corresponding address bit is don't-care.
     * |        |          |If the bit is set to zero, that means the received corresponding register bit should be exact the same as address register.
     * @var I2C_T::ADDRMSK2
     * Offset: 0x2C  I2C Slave Address Mask Register2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:1]   |ADDRMSK   |I2C Address Mask
     * |        |          |0 = Mask Disabled (the received corresponding register bit should be exact the same as address register.).
     * |        |          |1 = Mask Enabled (the received corresponding address bit is don't care.).
     * |        |          |I2C bus controllers support multiple address recognition with four address mask register.
     * |        |          |When the bit in the address mask register is set to one, it means the received corresponding address bit is don't-care.
     * |        |          |If the bit is set to zero, that means the received corresponding register bit should be exact the same as address register.
     * @var I2C_T::ADDRMSK3
     * Offset: 0x30  I2C Slave Address Mask Register3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:1]   |ADDRMSK   |I2C Address Mask
     * |        |          |0 = Mask Disabled (the received corresponding register bit should be exact the same as address register.).
     * |        |          |1 = Mask Enabled (the received corresponding address bit is don't care.).
     * |        |          |I2C bus controllers support multiple address recognition with four address mask register.
     * |        |          |When the bit in the address mask register is set to one, it means the received corresponding address bit is don't-care.
     * |        |          |If the bit is set to zero, that means the received corresponding register bit should be exact the same as address register.
     * @var I2C_T::WKCTL
     * Offset: 0x3C  I2C Wake-up Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |WKEN      |I2C Wake-up Enable Bit
     * |        |          |0 = I2C wake-up function Disabled.
     * |        |          |1= I2C wake-up function Enabled.
     * |[7]     |NHDBUSEN  |I2C No Hold BUS Enable Bit
     * |        |          |0 = I2C don't hold bus after wake-up disable.
     * |        |          |1 = I2C don't hold bus after wake-up enable.
     * |        |          |Note: I2C controller could response when WKIF event is not clear, it may cause error data transmitted or received.
     * |        |          |If data transmitted or received when WKIF event is not clear, user must reset I2C controller and execute the original operation again.
     * @var I2C_T::WKSTS
     * Offset: 0x40  I2C Wake-up Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |WKIF      |I2C Wake-up Flag
     * |        |          |When chip is woken up from Power-down mode by I2C, this bit is set to 1.
     * |        |          |Software can write 1 to clear this bit.
     * |[1]     |WKAKDONE  |Wakeup Address Frame Acknowledge Bit Done
     * |        |          |0 = The ACK bit cycle of address match frame isn't done.
     * |        |          |1 = The ACK bit cycle of address match frame is done in power-down.
     * |        |          |Note: This bit can't release WKIF. Software can write 1 to clear this bit.
     * |[2]     |WRSTSWK   |Read/Write Status Bit in Address Wakeup Frame
     * |        |          |0 = Write command be record on the address match wakeup frame.
     * |        |          |1 = Read command be record on the address match wakeup frame.
     * |        |          |Note: This bit will be cleared when software can write 1 to WKAKDONE bit.
     * @var I2C_T::CTL1
     * Offset: 0x44  I2C Control Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TXPDMAEN  |PDMA Transmit Channel Available
     * |        |          |0 = Transmit PDMA function disable.
     * |        |          |1 = Transmit PDMA function enable.
     * |[1]     |RXPDMAEN  |PDMA Receive Channel Available
     * |        |          |0 = Receive PDMA function disable.
     * |        |          |1 = Receive PDMA function enable.
     * |[2]     |PDMARST   |PDMA Reset
     * |        |          |0 = No effect.
     * |        |          |1 = Reset the PDMA control logic. This bit will be cleared to 0 automatically.
     * |[3]     |OVRIEN    |I2C over Run Interrupt Control Bit
     * |        |          |Setting OVRIEN to logic 1 will send a interrupt to system when the TWOBUFEN bit is enabled and there is over run event in received buffer.
     * |[4]     |UDRIEN    |I2C Under Run Interrupt Control Bit
     * |        |          |Setting UDRIEN to logic 1 will send a interrupt to system when the TWOBUFEN bit is enabled and there is under run event happened in transmitted buffer.
     * |[5]     |TWOBUFEN  |Two-level Buffer Enable Bit
     * |        |          |0 = Two-level buffer Disabled.
     * |        |          |1 = Two-level buffer Enabled.
     * |        |          |Set to enable the two-level buffer for I2C transmitted or received buffer.
     * |        |          |It is used to improve the performance of the I2C bus.
     * |        |          |If this bit is set = 1, the control bit of STA for repeat start or STO bit should be set after the current SI is clear.
     * |        |          |For example: if there are 4 data shall be transmitted and then stop it.
     * |        |          |The STO bit shall be set after the 3rd data's SI event being clear.
     * |        |          |In this time, the 4th data can be transmitted and the I2C stop after the 4th data transmission done.
     * |[6]     |TWOBUFRST |Two-level Buffer Reset
     * |        |          |0 = No effect.
     * |        |          |1 = Reset the related counters, two-level buffer state machine, and the content of data buffer.
     * |[7]     |NSTRETCH  |No Stretch on the I2C Bus
     * |        |          |0 = The I2C SCL bus is stretched by hardware if the SI is not cleared in master mode.
     * |        |          |1 = The I2C SCL bus is not stretched by hardware if the SI is not cleared in master mode.
     * |[8]     |PDMASTR   |PDMA Stretch Bit
     * |        |          |0 = I2C send STOP automatically after PDMA transfer done. (only master TX)
     * |        |          |1 = I2C SCL bus is stretched by hardware after PDMA transfer done if the SI is not cleared. (only master TX)
     * @var I2C_T::STATUS1
     * Offset: 0x48  I2C Status Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[4]     |FULL      |Two-level Buffer Full
     * |        |          |This bit indicates two-level buffer TX or RX full or not when the TWOBUFEN = 1.
     * |        |          |This bit is set when POINTER is equal to 2.
     * |[5]     |EMPTY     |Two-level Buffer Empty
     * |        |          |This bit indicates two-level buffer TX or RX empty or not when the TWOBUFEN = 1.
     * |        |          |This bit is set when POINTER is equal to 0.
     * |[6]     |OVR       |I2C over Run Status Bit
     * |        |          |This bit indicates the received two-level buffer TX or RX is over run when the TWOBUFEN = 1.
     * |[7]     |UDR       |I2C Under Run Status Bit
     * |        |          |This bit indicates the transmitted two-level buffer TX or RX is under run when the TWOBUFEN = 1.
     * |[8]     |ONBUSY    |on Bus Busy
     * |        |          |Indicates that a communication is in progress on the bus. It is set by hardware when a START condition is detected.
     * |        |          |It is cleared by hardware when a STOP condition is detected.
     * |        |          |0 = The bus is IDLE (both SCLK and SDA High).
     * |        |          |1 = The bus is busy.
     * @var I2C_T::TMCTL
     * Offset: 0x4C  I2C Timing Configure Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |STCTL     |Setup Time Configure Control Register
     * |        |          |This field is used to generate a delay timing between SDA falling edge and SCL rising edge in transmission mode.
     * |        |          |The delay setup time is numbers of peripheral clock = STCTL x PCLK.
     * |        |          |Note: Setup time setting should not make SCL output less than three PCLKs.
     * |[11:6]  |HTCTL     |Hold Time Configure Control Register
     * |        |          |This field is used to generate the delay timing between SCL falling edge and SDA rising edge in transmission mode.
     * |        |          |The delay hold time is numbers of peripheral clock = HTCTL x PCLK.
     */

    __IO uint32_t CTL;                   /*!< [0x0000] I2C Control Register                                             */
    __IO uint32_t ADDR0;                 /*!< [0x0004] I2C Slave Address Register 0                                     */
    __IO uint32_t DAT;                   /*!< [0x0008] I2C Data Register                                                */
    __I  uint32_t STATUS;                /*!< [0x000c] I2C Status Register 0                                            */
    __IO uint32_t CLKDIV;                /*!< [0x0010] I2C Clock Divided Register                                       */
    __IO uint32_t TOCTL;                 /*!< [0x0014] I2C Time-out Control Register                                    */
    __IO uint32_t ADDR1;                 /*!< [0x0018] I2C Slave Address Register1                                      */
    __IO uint32_t ADDR2;                 /*!< [0x001c] I2C Slave Address Register2                                      */
    __IO uint32_t ADDR3;                 /*!< [0x0020] I2C Slave Address Register3                                      */
    __IO uint32_t ADDRMSK0;              /*!< [0x0024] I2C Slave Address Mask Register0                                 */
    __IO uint32_t ADDRMSK1;              /*!< [0x0028] I2C Slave Address Mask Register1                                 */
    __IO uint32_t ADDRMSK2;              /*!< [0x002c] I2C Slave Address Mask Register2                                 */
    __IO uint32_t ADDRMSK3;              /*!< [0x0030] I2C Slave Address Mask Register3                                 */
    __I  uint32_t RESERVE0[2];
    __IO uint32_t WKCTL;                 /*!< [0x003c] I2C Wake-up Control Register                                     */
    __IO uint32_t WKSTS;                 /*!< [0x0040] I2C Wake-up Status Register                                      */
    __IO uint32_t CTL1;                  /*!< [0x0044] I2C Control Register 1                                           */
    __I  uint32_t STATUS1;               /*!< [0x0048] I2C Status Register 1                                            */
    __IO uint32_t TMCTL;                 /*!< [0x004c] I2C Timing Configure Control Register                            */

} I2C_T;

/**
    @addtogroup I2C_CONST I2C Bit Field Definition
    Constant Definitions for I2C Controller
@{ */

#define I2C_CTL_AA_Pos                   (2)                                               /*!< I2C_T::CTL: AA Position                */
#define I2C_CTL_AA_Msk                   (0x1ul << I2C_CTL_AA_Pos)                         /*!< I2C_T::CTL: AA Mask                    */

#define I2C_CTL_SI_Pos                   (3)                                               /*!< I2C_T::CTL: SI Position                */
#define I2C_CTL_SI_Msk                   (0x1ul << I2C_CTL_SI_Pos)                         /*!< I2C_T::CTL: SI Mask                    */

#define I2C_CTL_STO_Pos                  (4)                                               /*!< I2C_T::CTL: STO Position               */
#define I2C_CTL_STO_Msk                  (0x1ul << I2C_CTL_STO_Pos)                        /*!< I2C_T::CTL: STO Mask                   */

#define I2C_CTL_STA_Pos                  (5)                                               /*!< I2C_T::CTL: STA Position               */
#define I2C_CTL_STA_Msk                  (0x1ul << I2C_CTL_STA_Pos)                        /*!< I2C_T::CTL: STA Mask                   */

#define I2C_CTL_I2CEN_Pos                (6)                                               /*!< I2C_T::CTL: I2CEN Position             */
#define I2C_CTL_I2CEN_Msk                (0x1ul << I2C_CTL_I2CEN_Pos)                      /*!< I2C_T::CTL: I2CEN Mask                 */

#define I2C_CTL_INTEN_Pos                (7)                                               /*!< I2C_T::CTL: INTEN Position             */
#define I2C_CTL_INTEN_Msk                (0x1ul << I2C_CTL_INTEN_Pos)                      /*!< I2C_T::CTL: INTEN Mask                 */

#define I2C_ADDR0_GC_Pos                 (0)                                               /*!< I2C_T::ADDR0: GC Position              */
#define I2C_ADDR0_GC_Msk                 (0x1ul << I2C_ADDR0_GC_Pos)                       /*!< I2C_T::ADDR0: GC Mask                  */

#define I2C_ADDR0_ADDR_Pos               (1)                                               /*!< I2C_T::ADDR0: ADDR Position            */
#define I2C_ADDR0_ADDR_Msk               (0x7ful << I2C_ADDR0_ADDR_Pos)                    /*!< I2C_T::ADDR0: ADDR Mask                */

#define I2C_DAT_DAT_Pos                  (0)                                               /*!< I2C_T::DAT: DAT Position               */
#define I2C_DAT_DAT_Msk                  (0xfful << I2C_DAT_DAT_Pos)                       /*!< I2C_T::DAT: DAT Mask                   */

#define I2C_STATUS_STATUS_Pos            (0)                                               /*!< I2C_T::STATUS: STATUS Position         */
#define I2C_STATUS_STATUS_Msk            (0xfful << I2C_STATUS_STATUS_Pos)                 /*!< I2C_T::STATUS: STATUS Mask             */

#define I2C_CLKDIV_DIVIDER_Pos           (0)                                               /*!< I2C_T::CLKDIV: DIVIDER Position        */
#define I2C_CLKDIV_DIVIDER_Msk           (0xfful << I2C_CLKDIV_DIVIDER_Pos)                /*!< I2C_T::CLKDIV: DIVIDER Mask            */

#define I2C_TOCTL_TOIF_Pos               (0)                                               /*!< I2C_T::TOCTL: TOIF Position            */
#define I2C_TOCTL_TOIF_Msk               (0x1ul << I2C_TOCTL_TOIF_Pos)                     /*!< I2C_T::TOCTL: TOIF Mask                */

#define I2C_TOCTL_TOCDIV4_Pos            (1)                                               /*!< I2C_T::TOCTL: TOCDIV4 Position         */
#define I2C_TOCTL_TOCDIV4_Msk            (0x1ul << I2C_TOCTL_TOCDIV4_Pos)                  /*!< I2C_T::TOCTL: TOCDIV4 Mask             */

#define I2C_TOCTL_TOCEN_Pos              (2)                                               /*!< I2C_T::TOCTL: TOCEN Position           */
#define I2C_TOCTL_TOCEN_Msk              (0x1ul << I2C_TOCTL_TOCEN_Pos)                    /*!< I2C_T::TOCTL: TOCEN Mask               */

#define I2C_ADDR1_GC_Pos                 (0)                                               /*!< I2C_T::ADDR1: GC Position              */
#define I2C_ADDR1_GC_Msk                 (0x1ul << I2C_ADDR1_GC_Pos)                       /*!< I2C_T::ADDR1: GC Mask                  */

#define I2C_ADDR1_ADDR_Pos               (1)                                               /*!< I2C_T::ADDR1: ADDR Position            */
#define I2C_ADDR1_ADDR_Msk               (0x7ful << I2C_ADDR1_ADDR_Pos)                    /*!< I2C_T::ADDR1: ADDR Mask                */

#define I2C_ADDR2_GC_Pos                 (0)                                               /*!< I2C_T::ADDR2: GC Position              */
#define I2C_ADDR2_GC_Msk                 (0x1ul << I2C_ADDR2_GC_Pos)                       /*!< I2C_T::ADDR2: GC Mask                  */

#define I2C_ADDR2_ADDR_Pos               (1)                                               /*!< I2C_T::ADDR2: ADDR Position            */
#define I2C_ADDR2_ADDR_Msk               (0x7ful << I2C_ADDR2_ADDR_Pos)                    /*!< I2C_T::ADDR2: ADDR Mask                */

#define I2C_ADDR3_GC_Pos                 (0)                                               /*!< I2C_T::ADDR3: GC Position              */
#define I2C_ADDR3_GC_Msk                 (0x1ul << I2C_ADDR3_GC_Pos)                       /*!< I2C_T::ADDR3: GC Mask                  */

#define I2C_ADDR3_ADDR_Pos               (1)                                               /*!< I2C_T::ADDR3: ADDR Position            */
#define I2C_ADDR3_ADDR_Msk               (0x7ful << I2C_ADDR3_ADDR_Pos)                    /*!< I2C_T::ADDR3: ADDR Mask                */

#define I2C_ADDRMSK0_ADDRMSK_Pos         (1)                                               /*!< I2C_T::ADDRMSK0: ADDRMSK Position      */
#define I2C_ADDRMSK0_ADDRMSK_Msk         (0x7ful << I2C_ADDRMSK0_ADDRMSK_Pos)              /*!< I2C_T::ADDRMSK0: ADDRMSK Mask          */

#define I2C_ADDRMSK1_ADDRMSK_Pos         (1)                                               /*!< I2C_T::ADDRMSK1: ADDRMSK Position      */
#define I2C_ADDRMSK1_ADDRMSK_Msk         (0x7ful << I2C_ADDRMSK1_ADDRMSK_Pos)              /*!< I2C_T::ADDRMSK1: ADDRMSK Mask          */

#define I2C_ADDRMSK2_ADDRMSK_Pos         (1)                                               /*!< I2C_T::ADDRMSK2: ADDRMSK Position      */
#define I2C_ADDRMSK2_ADDRMSK_Msk         (0x7ful << I2C_ADDRMSK2_ADDRMSK_Pos)              /*!< I2C_T::ADDRMSK2: ADDRMSK Mask          */

#define I2C_ADDRMSK3_ADDRMSK_Pos         (1)                                               /*!< I2C_T::ADDRMSK3: ADDRMSK Position      */
#define I2C_ADDRMSK3_ADDRMSK_Msk         (0x7ful << I2C_ADDRMSK3_ADDRMSK_Pos)              /*!< I2C_T::ADDRMSK3: ADDRMSK Mask          */

#define I2C_WKCTL_WKEN_Pos               (0)                                               /*!< I2C_T::WKCTL: WKEN Position            */
#define I2C_WKCTL_WKEN_Msk               (0x1ul << I2C_WKCTL_WKEN_Pos)                     /*!< I2C_T::WKCTL: WKEN Mask                */

#define I2C_WKCTL_NHDBUSEN_Pos           (7)                                               /*!< I2C_T::WKCTL: NHDBUSEN Position        */
#define I2C_WKCTL_NHDBUSEN_Msk           (0x1ul << I2C_WKCTL_NHDBUSEN_Pos)                 /*!< I2C_T::WKCTL: NHDBUSEN Mask            */

#define I2C_WKSTS_WKIF_Pos               (0)                                               /*!< I2C_T::WKSTS: WKIF Position            */
#define I2C_WKSTS_WKIF_Msk               (0x1ul << I2C_WKSTS_WKIF_Pos)                     /*!< I2C_T::WKSTS: WKIF Mask                */

#define I2C_WKSTS_WKAKDONE_Pos           (1)                                               /*!< I2C_T::WKSTS: WKAKDONE Position        */
#define I2C_WKSTS_WKAKDONE_Msk           (0x1ul << I2C_WKSTS_WKAKDONE_Pos)                 /*!< I2C_T::WKSTS: WKAKDONE Mask            */

#define I2C_WKSTS_WRSTSWK_Pos            (2)                                               /*!< I2C_T::WKSTS: WRSTSWK Position         */
#define I2C_WKSTS_WRSTSWK_Msk            (0x1ul << I2C_WKSTS_WRSTSWK_Pos)                  /*!< I2C_T::WKSTS: WRSTSWK Mask             */

#define I2C_CTL1_TXPDMAEN_Pos            (0)                                               /*!< I2C_T::CTL1: TXPDMAEN Position         */
#define I2C_CTL1_TXPDMAEN_Msk            (0x1ul << I2C_CTL1_TXPDMAEN_Pos)                  /*!< I2C_T::CTL1: TXPDMAEN Mask             */

#define I2C_CTL1_RXPDMAEN_Pos            (1)                                               /*!< I2C_T::CTL1: RXPDMAEN Position         */
#define I2C_CTL1_RXPDMAEN_Msk            (0x1ul << I2C_CTL1_RXPDMAEN_Pos)                  /*!< I2C_T::CTL1: RXPDMAEN Mask             */

#define I2C_CTL1_PDMARST_Pos             (2)                                               /*!< I2C_T::CTL1: PDMARST Position          */
#define I2C_CTL1_PDMARST_Msk             (0x1ul << I2C_CTL1_PDMARST_Pos)                   /*!< I2C_T::CTL1: PDMARST Mask              */

#define I2C_CTL1_OVRIEN_Pos              (3)                                               /*!< I2C_T::CTL1: OVRIEN Position           */
#define I2C_CTL1_OVRIEN_Msk              (0x1ul << I2C_CTL1_OVRIEN_Pos)                    /*!< I2C_T::CTL1: OVRIEN Mask               */

#define I2C_CTL1_UDRIEN_Pos              (4)                                               /*!< I2C_T::CTL1: UDRIEN Position           */
#define I2C_CTL1_UDRIEN_Msk              (0x1ul << I2C_CTL1_UDRIEN_Pos)                    /*!< I2C_T::CTL1: UDRIEN Mask               */

#define I2C_CTL1_TWOBUFEN_Pos            (5)                                               /*!< I2C_T::CTL1: TWOBUFEN Position         */
#define I2C_CTL1_TWOBUFEN_Msk            (0x1ul << I2C_CTL1_TWOBUFEN_Pos)                  /*!< I2C_T::CTL1: TWOBUFEN Mask             */

#define I2C_CTL1_TWOBUFRST_Pos           (6)                                               /*!< I2C_T::CTL1: TWOBUFRST Position        */
#define I2C_CTL1_TWOBUFRST_Msk           (0x1ul << I2C_CTL1_TWOBUFRST_Pos)                 /*!< I2C_T::CTL1: TWOBUFRST Mask            */

#define I2C_CTL1_NSTRETCH_Pos            (7)                                               /*!< I2C_T::CTL1: NSTRETCH Position         */
#define I2C_CTL1_NSTRETCH_Msk            (0x1ul << I2C_CTL1_NSTRETCH_Pos)                  /*!< I2C_T::CTL1: NSTRETCH Mask             */

#define I2C_CTL1_PDMASTR_Pos             (8)                                               /*!< I2C_T::CTL1: PDMASTR Position          */
#define I2C_CTL1_PDMASTR_Msk             (0x1ul << I2C_CTL1_PDMASTR_Pos)                   /*!< I2C_T::CTL1: PDMASTR Mask              */

#define I2C_STATUS1_FULL_Pos             (4)                                               /*!< I2C_T::STATUS1: FULL Position          */
#define I2C_STATUS1_FULL_Msk             (0x1ul << I2C_STATUS1_FULL_Pos)                   /*!< I2C_T::STATUS1: FULL Mask              */

#define I2C_STATUS1_EMPTY_Pos            (5)                                               /*!< I2C_T::STATUS1: EMPTY Position         */
#define I2C_STATUS1_EMPTY_Msk            (0x1ul << I2C_STATUS1_EMPTY_Pos)                  /*!< I2C_T::STATUS1: EMPTY Mask             */

#define I2C_STATUS1_OVR_Pos              (6)                                               /*!< I2C_T::STATUS1: OVR Position           */
#define I2C_STATUS1_OVR_Msk              (0x1ul << I2C_STATUS1_OVR_Pos)                    /*!< I2C_T::STATUS1: OVR Mask               */

#define I2C_STATUS1_UDR_Pos              (7)                                               /*!< I2C_T::STATUS1: UDR Position           */
#define I2C_STATUS1_UDR_Msk              (0x1ul << I2C_STATUS1_UDR_Pos)                    /*!< I2C_T::STATUS1: UDR Mask               */

#define I2C_STATUS1_ONBUSY_Pos           (8)                                               /*!< I2C_T::STATUS1: ONBUSY Position        */
#define I2C_STATUS1_ONBUSY_Msk           (0x1ul << I2C_STATUS1_ONBUSY_Pos)                 /*!< I2C_T::STATUS1: ONBUSY Mask            */

#define I2C_TMCTL_STCTL_Pos              (0)                                               /*!< I2C_T::TMCTL: STCTL Position           */
#define I2C_TMCTL_STCTL_Msk              (0x3ful << I2C_TMCTL_STCTL_Pos)                   /*!< I2C_T::TMCTL: STCTL Mask               */

#define I2C_TMCTL_HTCTL_Pos              (6)                                               /*!< I2C_T::TMCTL: HTCTL Position           */
#define I2C_TMCTL_HTCTL_Msk              (0x3ful << I2C_TMCTL_HTCTL_Pos)                   /*!< I2C_T::TMCTL: HTCTL Mask               */

/**@}*/ /* I2C_CONST */
/**@}*/ /* end of I2C register group */


/*---------------------- Peripheral Direct Memory Access Controller -------------------------*/
/**
    @addtogroup PDMA Peripheral Direct Memory Access Controller(PDMA)
    Memory Mapped Structure for PDMA Controller
@{ */



typedef struct
{
    /**
     * @var DSCT_T::CTL
     * Offset: 0x00/0x10/0x20/0x30/0x40  Descriptor Table Control Register of PDMA Channel 0~4
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |OPMODE    |PDMA Operation Mode Selection
     * |        |          |00 = Idle state: Channel is stopped or this table is complete, when PDMA finish channel table task, OPMODE will be cleared to idle state automatically.
     * |        |          |01 = Basic mode: The descriptor table only has one task
     * |        |          |When this task is finished, the TDIF(PDMA_INTSTS[1]) will be asserted.
     * |        |          |10 = Scatter-Gather mode: When operating in this mode, user must give the first descriptor table address in PDMA_DSCT_FIRST register; PDMA controller will ignore this task, then load the next task to execute.
     * |        |          |11 = Reserved.
     * |        |          |Note: Before filling transfer task in the Descriptor Table, user must check if the descriptor table is complete.
     * |[2]     |TXTYPE    |Transfer Type
     * |        |          |0 = Burst transfer type.
     * |        |          |1 = Single transfer type.
     * |[6:4]   |BURSIZE   |Burst Size
     * |        |          |This field is used for peripheral to determine the burst size or used for determine the re-arbitration size.
     * |        |          |000 = 128 Transfers.
     * |        |          |001 = 64 Transfers.
     * |        |          |010 = 32 Transfers.
     * |        |          |011 = 16 Transfers.
     * |        |          |100 = 8 Transfers.
     * |        |          |101 = 4 Transfers.
     * |        |          |110 = 2 Transfers.
     * |        |          |111 = 1 Transfers.
     * |        |          |Note: This field is only useful in burst transfer type.
     * |[7]     |TBINTDIS  |Table Interrupt Disable Bit
     * |        |          |This field can be used to decide whether to enable table interrupt or not
     * |        |          |If the TBINTDIS bit is enabled when PDMA controller finishes transfer task, it will not generates transfer done interrupt.
     * |        |          |0 = Table interrupt Enabled.
     * |        |          |1 = Table interrupt Disabled.
     * |        |          |Note: If this bit set to 1, the TEMPTYF will not be set.
     * |[9:8]   |SAINC     |Source Address Increment
     * |        |          |This Field Is Used To Set The Source Address Increment Size.
     * |        |          |11 = No Increment (Fixed Address).
     * |        |          |Others = Increment And Size Is Depended On TXWIDTH Selection.
     * |[11:10] |DAINC     |Destination Address Increment
     * |        |          |This field is used to set the destination address increment size.
     * |        |          |11 = No increment (fixed address).
     * |        |          |Others = Increment and size is depended on TXWIDTH selection.
     * |[13:12] |TXWIDTH   |Transfer Width Selection
     * |        |          |This field is used for transfer width.
     * |        |          |00 = One byte (8 bit) is transferred for every operation.
     * |        |          |01 = One half-word (16 bit) is transferred for every operation.
     * |        |          |10 = One word (32-bit) is transferred for every operation.
     * |        |          |11 = Reserved.
     * |        |          |Note: The PDMA transfer source address (PDMA_DSCT_SA) and PDMA transfer destination address (PDMA_DSCT_DA) should be alignment under the TXWIDTH selection
     * |        |          |For example, if source address is 0x2000_0202, but TXWIDTH is word transfer, the source address is not word alignment
     * |        |          |The source address is aligned when TXWIDTH is byte or half-word transfer.
     * |[29:16] |TXCNT     |Transfer Count
     * |        |          |The TXCNT represents the required number of PDMA transfer, the real transfer count is (TXCNT + 1); The maximum transfer count is 16384 , every transfer may be byte, half-word or word that is dependent on TXWIDTH field.
     * |        |          |Note: When PDMA finish each transfer data, this field will be decrease immediately.
     * @var DSCT_T::SA
     * Offset: 0x04/0x14/0x24/0x34/0x44  Source Address Register of PDMA Channel 0~4
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |SA        |PDMA Transfer Source Address Register
     * |        |          |This field indicates a 32-bit source address of PDMA controller.
     * |        |          |Note: The PDMA transfer source address should be aligned with the TXWIDTH(PDMA_DSCTn_CTL[13:12], n=0,1..4) selection.
     * @var DSCT_T::DA
     * Offset: 0x08/0x18/0x28/0x38/0x48  Destination Address Register of PDMA Channel 0~4
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |DA        |PDMA Transfer Destination Address Register
     * |        |          |This field indicates a 32-bit destination address of PDMA controller.
     * |        |          |Note: The PDMA transfer destination address should be aligned with the TXWIDTH(PDMA_DSCTn_CTL[13:12], n=0,1..4) selection.
     * @var DSCT_T::FIRST
     * Offset: 0x0C/0x1C/0x2C/0x3C/0x4C  First Scatter-Gather Descriptor Table Offset of PDMA Channel 0~4
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FIRST     |PDMA First Descriptor Table Offset
     * |        |          |This field indicates the offset of the first descriptor table address in system memory.
     * |        |          |Write Operation:
     * |        |          |If the system memory based address is 0x2000_0000 (PDMA_SCATBA), and the first descriptor table is start from 0x2000_0100, then this field must fill in 0x0100.
     * |        |          |Read Operation:
     * |        |          |When operating in scatter-gather mode, the last two bits FIRST[1:0] will become reserved.
     * |        |          |Note1: The first descriptor table address must be word boundary.
     * |        |          |Note2: Before filled transfer task in the descriptor table, user must check if the descriptor table is complete.
     * |[31:16] |NEXT      |PDMA Next Descriptor Table Offset
     * |        |          |This field indicates the offset of next descriptor table address in system memory.
     * |        |          |Note: write operation is useless in this field.
     */

    __IO uint32_t CTL;             /*!< [0x00/0x10/0x20/0x30/0x40] Descriptor Table Control Register of PDMA Channel 0~4              */
    __IO uint32_t SA;              /*!< [0x04/0x14/0x24/0x34/0x44] Source Address Register of PDMA Channel 0~4                        */
    __IO uint32_t DA;              /*!< [0x08/0x18/0x28/0x38/0x48] Destination Address Register of PDMA Channel 0~4                   */
    union
    {
        __IO uint32_t FIRST;       /*!< [0x0C/0x1C/0x2C/0x3C/0x4C] First Scatter-Gather Descriptor Table Offset of PDMA Channel 0~4   */
        __IO uint32_t NEXT;        /*!< Next Scatter-Gather Descriptor Table Offset                                                   */
    };

} DSCT_T;

typedef struct
{


    /**
     * @var PDMA_T::CURSCAT
     * Offset: 0x50/0x54/0x58/0x5C/0x60  Current Scatter-Gather Descriptor Table Address of PDMA Channel 0~4
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |CURADDR   |PDMA Current Description Address Register (Read Only)
     * |        |          |This field indicates a 32-bit current external description address of PDMA controller.
     * |        |          |Note: This field is read only and only used for Scatter-Gather mode to indicate the current external description address.
     * @var PDMA_T::CHCTL
     * Offset: 0x400  PDMA Channel Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CHEN0     |PDMA Channel N Enable Bit
     * |        |          |Set this bit to 1 to enable PDMAn operation. Channel cannot be active if it is not set as enabled.
     * |        |          |0 = PDMA channel [n] Disabled.
     * |        |          |1 = PDMA channel [n] Enabled.
     * |        |          |Note: Set PDMA_PAUSE or PDMA_RESET register will also clear this bit.
     * |[1]     |CHEN1     |PDMA Channel N Enable Bit
     * |        |          |Set this bit to 1 to enable PDMAn operation. Channel cannot be active if it is not set as enabled.
     * |        |          |0 = PDMA channel [n] Disabled.
     * |        |          |1 = PDMA channel [n] Enabled.
     * |        |          |Note: Set PDMA_PAUSE or PDMA_RESET register will also clear this bit.
     * |[2]     |CHEN2     |PDMA Channel N Enable Bit
     * |        |          |Set this bit to 1 to enable PDMAn operation. Channel cannot be active if it is not set as enabled.
     * |        |          |0 = PDMA channel [n] Disabled.
     * |        |          |1 = PDMA channel [n] Enabled.
     * |        |          |Note: Set PDMA_PAUSE or PDMA_RESET register will also clear this bit.
     * |[3]     |CHEN3     |PDMA Channel N Enable Bit
     * |        |          |Set this bit to 1 to enable PDMAn operation. Channel cannot be active if it is not set as enabled.
     * |        |          |0 = PDMA channel [n] Disabled.
     * |        |          |1 = PDMA channel [n] Enabled.
     * |        |          |Note: Set PDMA_PAUSE or PDMA_RESET register will also clear this bit.
     * |[4]     |CHEN4     |PDMA Channel N Enable Bit
     * |        |          |Set this bit to 1 to enable PDMAn operation. Channel cannot be active if it is not set as enabled.
     * |        |          |0 = PDMA channel [n] Disabled.
     * |        |          |1 = PDMA channel [n] Enabled.
     * |        |          |Note: Set PDMA_PAUSE or PDMA_RESET register will also clear this bit.
     * @var PDMA_T::PAUSE
     * Offset: 0x404  PDMA Transfer Pause Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |PAUSE0    |PDMA Channel N Transfer Pause Control Register (Write Only)
     * |        |          |User can set PAUSEn bit field to pause the PDMA transfer
     * |        |          |When user sets PAUSEn bit, the PDMA controller will pause the on-going transfer, then clear the channel enable bit CHEN(PDMA_CHCTL [n], n=0,1..4) and clear request active flag
     * |        |          |If re-enable the paused channel agian, the remaining transfers will be processed.
     * |        |          |0 = No effect.
     * |        |          |1 = Pause PDMA channel n transfer.
     * |[1]     |PAUSE1    |PDMA Channel N Transfer Pause Control Register (Write Only)
     * |        |          |User can set PAUSEn bit field to pause the PDMA transfer
     * |        |          |When user sets PAUSEn bit, the PDMA controller will pause the on-going transfer, then clear the channel enable bit CHEN(PDMA_CHCTL [n], n=0,1..4) and clear request active flag
     * |        |          |If re-enable the paused channel agian, the remaining transfers will be processed.
     * |        |          |0 = No effect.
     * |        |          |1 = Pause PDMA channel n transfer.
     * |[2]     |PAUSE2    |PDMA Channel N Transfer Pause Control Register (Write Only)
     * |        |          |User can set PAUSEn bit field to pause the PDMA transfer
     * |        |          |When user sets PAUSEn bit, the PDMA controller will pause the on-going transfer, then clear the channel enable bit CHEN(PDMA_CHCTL [n], n=0,1..4) and clear request active flag
     * |        |          |If re-enable the paused channel agian, the remaining transfers will be processed.
     * |        |          |0 = No effect.
     * |        |          |1 = Pause PDMA channel n transfer.
     * |[3]     |PAUSE3    |PDMA Channel N Transfer Pause Control Register (Write Only)
     * |        |          |User can set PAUSEn bit field to pause the PDMA transfer
     * |        |          |When user sets PAUSEn bit, the PDMA controller will pause the on-going transfer, then clear the channel enable bit CHEN(PDMA_CHCTL [n], n=0,1..4) and clear request active flag
     * |        |          |If re-enable the paused channel agian, the remaining transfers will be processed.
     * |        |          |0 = No effect.
     * |        |          |1 = Pause PDMA channel n transfer.
     * |[4]     |PAUSE4    |PDMA Channel N Transfer Pause Control Register (Write Only)
     * |        |          |User can set PAUSEn bit field to pause the PDMA transfer
     * |        |          |When user sets PAUSEn bit, the PDMA controller will pause the on-going transfer, then clear the channel enable bit CHEN(PDMA_CHCTL [n], n=0,1..4) and clear request active flag
     * |        |          |If re-enable the paused channel agian, the remaining transfers will be processed.
     * |        |          |0 = No effect.
     * |        |          |1 = Pause PDMA channel n transfer.
     * @var PDMA_T::SWREQ
     * Offset: 0x408  PDMA Software Request Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SWREQ0    |PDMA Channel N Software Request Register (Write Only)
     * |        |          |Set this bit to 1 to generate a software request to PDMA [n].
     * |        |          |0 = No effect.
     * |        |          |1 = Generate a software request.
     * |        |          |Note1: User can read PDMA_TRGSTS register to know which channel is on active
     * |        |          |Active flag may be triggered by software request or peripheral request.
     * |        |          |Note2: If user does not enable corresponding PDMA channel, the software request will be ignored.
     * |[1]     |SWREQ1    |PDMA Channel N Software Request Register (Write Only)
     * |        |          |Set this bit to 1 to generate a software request to PDMA [n].
     * |        |          |0 = No effect.
     * |        |          |1 = Generate a software request.
     * |        |          |Note1: User can read PDMA_TRGSTS register to know which channel is on active
     * |        |          |Active flag may be triggered by software request or peripheral request.
     * |        |          |Note2: If user does not enable corresponding PDMA channel, the software request will be ignored.
     * |[2]     |SWREQ2    |PDMA Channel N Software Request Register (Write Only)
     * |        |          |Set this bit to 1 to generate a software request to PDMA [n].
     * |        |          |0 = No effect.
     * |        |          |1 = Generate a software request.
     * |        |          |Note1: User can read PDMA_TRGSTS register to know which channel is on active
     * |        |          |Active flag may be triggered by software request or peripheral request.
     * |        |          |Note2: If user does not enable corresponding PDMA channel, the software request will be ignored.
     * |[3]     |SWREQ3    |PDMA Channel N Software Request Register (Write Only)
     * |        |          |Set this bit to 1 to generate a software request to PDMA [n].
     * |        |          |0 = No effect.
     * |        |          |1 = Generate a software request.
     * |        |          |Note1: User can read PDMA_TRGSTS register to know which channel is on active
     * |        |          |Active flag may be triggered by software request or peripheral request.
     * |        |          |Note2: If user does not enable corresponding PDMA channel, the software request will be ignored.
     * |[4]     |SWREQ4    |PDMA Channel N Software Request Register (Write Only)
     * |        |          |Set this bit to 1 to generate a software request to PDMA [n].
     * |        |          |0 = No effect.
     * |        |          |1 = Generate a software request.
     * |        |          |Note1: User can read PDMA_TRGSTS register to know which channel is on active
     * |        |          |Active flag may be triggered by software request or peripheral request.
     * |        |          |Note2: If user does not enable corresponding PDMA channel, the software request will be ignored.
     * @var PDMA_T::TRGSTS
     * Offset: 0x40C  PDMA Channel Request Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |REQSTS0   |PDMA Channel N Request Status (Read Only)
     * |        |          |This flag indicates whether channel[n] have a request or not, no matter request from software or peripheral
     * |        |          |When PDMA controller finishes channel transfer, this bit will be cleared automatically.
     * |        |          |0 = PDMA Channel n has no request.
     * |        |          |1 = PDMA Channel n has a request.
     * |        |          |Note: If user pauses or resets each PDMA transfer by setting PDMA_PAUSE or PDMA_RESET register respectively, this bit will be cleared automatically after finishing current transfer.
     * |[1]     |REQSTS1   |PDMA Channel N Request Status (Read Only)
     * |        |          |This flag indicates whether channel[n] have a request or not, no matter request from software or peripheral
     * |        |          |When PDMA controller finishes channel transfer, this bit will be cleared automatically.
     * |        |          |0 = PDMA Channel n has no request.
     * |        |          |1 = PDMA Channel n has a request.
     * |        |          |Note: If user pauses or resets each PDMA transfer by setting PDMA_PAUSE or PDMA_RESET register respectively, this bit will be cleared automatically after finishing current transfer.
     * |[2]     |REQSTS2   |PDMA Channel N Request Status (Read Only)
     * |        |          |This flag indicates whether channel[n] have a request or not, no matter request from software or peripheral
     * |        |          |When PDMA controller finishes channel transfer, this bit will be cleared automatically.
     * |        |          |0 = PDMA Channel n has no request.
     * |        |          |1 = PDMA Channel n has a request.
     * |        |          |Note: If user pauses or resets each PDMA transfer by setting PDMA_PAUSE or PDMA_RESET register respectively, this bit will be cleared automatically after finishing current transfer.
     * |[3]     |REQSTS3   |PDMA Channel N Request Status (Read Only)
     * |        |          |This flag indicates whether channel[n] have a request or not, no matter request from software or peripheral
     * |        |          |When PDMA controller finishes channel transfer, this bit will be cleared automatically.
     * |        |          |0 = PDMA Channel n has no request.
     * |        |          |1 = PDMA Channel n has a request.
     * |        |          |Note: If user pauses or resets each PDMA transfer by setting PDMA_PAUSE or PDMA_RESET register respectively, this bit will be cleared automatically after finishing current transfer.
     * |[4]     |REQSTS4   |PDMA Channel N Request Status (Read Only)
     * |        |          |This flag indicates whether channel[n] have a request or not, no matter request from software or peripheral
     * |        |          |When PDMA controller finishes channel transfer, this bit will be cleared automatically.
     * |        |          |0 = PDMA Channel n has no request.
     * |        |          |1 = PDMA Channel n has a request.
     * |        |          |Note: If user pauses or resets each PDMA transfer by setting PDMA_PAUSE or PDMA_RESET register respectively, this bit will be cleared automatically after finishing current transfer.
     * @var PDMA_T::PRISET
     * Offset: 0x410  PDMA Fixed Priority Setting Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |FPRISET0  |PDMA Channel N Fixed Priority Setting Register
     * |        |          |Set this bit to 1 to enable fixed priority level
     * |        |          |The fixed priority channel has higher priority than round-robin priority channel
     * |        |          |If multiple channels are set as the same priority, the higher number of channels have higher priority.
     * |        |          |Write Operation:
     * |        |          |0 = No effect.
     * |        |          |1 = Set PDMA channel [n] to fixed priority channel.
     * |        |          |Read Operation:
     * |        |          |0 = Corresponding PDMA channel is round-robin priority.
     * |        |          |1 = Corresponding PDMA channel is fixed priority.
     * |        |          |Note: This field only set to fixed priority, clear fixed priority use PDMA_PRICLR register.
     * |[1]     |FPRISET1  |PDMA Channel N Fixed Priority Setting Register
     * |        |          |Set this bit to 1 to enable fixed priority level
     * |        |          |The fixed priority channel has higher priority than round-robin priority channel
     * |        |          |If multiple channels are set as the same priority, the higher number of channels have higher priority.
     * |        |          |Write Operation:
     * |        |          |0 = No effect.
     * |        |          |1 = Set PDMA channel [n] to fixed priority channel.
     * |        |          |Read Operation:
     * |        |          |0 = Corresponding PDMA channel is round-robin priority.
     * |        |          |1 = Corresponding PDMA channel is fixed priority.
     * |        |          |Note: This field only set to fixed priority, clear fixed priority use PDMA_PRICLR register.
     * |[2]     |FPRISET2  |PDMA Channel N Fixed Priority Setting Register
     * |        |          |Set this bit to 1 to enable fixed priority level
     * |        |          |The fixed priority channel has higher priority than round-robin priority channel
     * |        |          |If multiple channels are set as the same priority, the higher number of channels have higher priority.
     * |        |          |Write Operation:
     * |        |          |0 = No effect.
     * |        |          |1 = Set PDMA channel [n] to fixed priority channel.
     * |        |          |Read Operation:
     * |        |          |0 = Corresponding PDMA channel is round-robin priority.
     * |        |          |1 = Corresponding PDMA channel is fixed priority.
     * |        |          |Note: This field only set to fixed priority, clear fixed priority use PDMA_PRICLR register.
     * |[3]     |FPRISET3  |PDMA Channel N Fixed Priority Setting Register
     * |        |          |Set this bit to 1 to enable fixed priority level
     * |        |          |The fixed priority channel has higher priority than round-robin priority channel
     * |        |          |If multiple channels are set as the same priority, the higher number of channels have higher priority.
     * |        |          |Write Operation:
     * |        |          |0 = No effect.
     * |        |          |1 = Set PDMA channel [n] to fixed priority channel.
     * |        |          |Read Operation:
     * |        |          |0 = Corresponding PDMA channel is round-robin priority.
     * |        |          |1 = Corresponding PDMA channel is fixed priority.
     * |        |          |Note: This field only set to fixed priority, clear fixed priority use PDMA_PRICLR register.
     * |[4]     |FPRISET4  |PDMA Channel N Fixed Priority Setting Register
     * |        |          |Set this bit to 1 to enable fixed priority level
     * |        |          |The fixed priority channel has higher priority than round-robin priority channel
     * |        |          |If multiple channels are set as the same priority, the higher number of channels have higher priority.
     * |        |          |Write Operation:
     * |        |          |0 = No effect.
     * |        |          |1 = Set PDMA channel [n] to fixed priority channel.
     * |        |          |Read Operation:
     * |        |          |0 = Corresponding PDMA channel is round-robin priority.
     * |        |          |1 = Corresponding PDMA channel is fixed priority.
     * |        |          |Note: This field only set to fixed priority, clear fixed priority use PDMA_PRICLR register.
     * @var PDMA_T::PRICLR
     * Offset: 0x414  PDMA Fixed Priority Clear Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |FPRICLR0  |PDMA Channel N Fixed Priority Clear Register (Write Only)
     * |        |          |Set this bit to 1 to clear fixed priority level.
     * |        |          |0 = No effect.
     * |        |          |1 = Clear PDMA channel [n] fixed priority setting.
     * |        |          |Note: User can read PDMA_PRISET register to know the channel priority.
     * |[1]     |FPRICLR1  |PDMA Channel N Fixed Priority Clear Register (Write Only)
     * |        |          |Set this bit to 1 to clear fixed priority level.
     * |        |          |0 = No effect.
     * |        |          |1 = Clear PDMA channel [n] fixed priority setting.
     * |        |          |Note: User can read PDMA_PRISET register to know the channel priority.
     * |[2]     |FPRICLR2  |PDMA Channel N Fixed Priority Clear Register (Write Only)
     * |        |          |Set this bit to 1 to clear fixed priority level.
     * |        |          |0 = No effect.
     * |        |          |1 = Clear PDMA channel [n] fixed priority setting.
     * |        |          |Note: User can read PDMA_PRISET register to know the channel priority.
     * |[3]     |FPRICLR3  |PDMA Channel N Fixed Priority Clear Register (Write Only)
     * |        |          |Set this bit to 1 to clear fixed priority level.
     * |        |          |0 = No effect.
     * |        |          |1 = Clear PDMA channel [n] fixed priority setting.
     * |        |          |Note: User can read PDMA_PRISET register to know the channel priority.
     * |[4]     |FPRICLR4  |PDMA Channel N Fixed Priority Clear Register (Write Only)
     * |        |          |Set this bit to 1 to clear fixed priority level.
     * |        |          |0 = No effect.
     * |        |          |1 = Clear PDMA channel [n] fixed priority setting.
     * |        |          |Note: User can read PDMA_PRISET register to know the channel priority.
     * @var PDMA_T::INTEN
     * Offset: 0x418  PDMA Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |INTEN0    |PDMA Channel N Interrupt Enable Register
     * |        |          |This field is used for enabling PDMA channel[n] interrupt.
     * |        |          |0 = PDMA channel n interrupt Disabled.
     * |        |          |1 = PDMA channel n interrupt Enabled.
     * |[1]     |INTEN1    |PDMA Channel N Interrupt Enable Register
     * |        |          |This field is used for enabling PDMA channel[n] interrupt.
     * |        |          |0 = PDMA channel n interrupt Disabled.
     * |        |          |1 = PDMA channel n interrupt Enabled.
     * |[2]     |INTEN2    |PDMA Channel N Interrupt Enable Register
     * |        |          |This field is used for enabling PDMA channel[n] interrupt.
     * |        |          |0 = PDMA channel n interrupt Disabled.
     * |        |          |1 = PDMA channel n interrupt Enabled.
     * |[3]     |INTEN3    |PDMA Channel N Interrupt Enable Register
     * |        |          |This field is used for enabling PDMA channel[n] interrupt.
     * |        |          |0 = PDMA channel n interrupt Disabled.
     * |        |          |1 = PDMA channel n interrupt Enabled.
      * |[4]     |INTEN4    |PDMA Channel N Interrupt Enable Register
     * |        |          |This field is used for enabling PDMA channel[n] interrupt.
     * |        |          |0 = PDMA channel n interrupt Disabled.
     * |        |          |1 = PDMA channel n interrupt Enabled.
     * @var PDMA_T::INTSTS
     * Offset: 0x41C  PDMA Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ABTIF     |PDMA Read/Write Target Abort Interrupt Flag (Read Only)
     * |        |          |This bit indicates that PDMA has target abort error; Software can read PDMA_ABTSTS register to find which channel has target abort error.
     * |        |          |0 = No AHB bus ERROR response received.
     * |        |          |1 = AHB bus ERROR response received.
     * |[1]     |TDIF      |Transfer Done Interrupt Flag (Read Only)
     * |        |          |This bit indicates that PDMA controller has finished transmission; User can read PDMA_TDSTS register to indicate which channel finished transfer.
     * |        |          |0 = Not finished yet.
     * |        |          |1 = PDMA channel has finished transmission.
     * |[2]     |TEIF      |Table Empty Interrupt Flag (Read Only)
     * |        |          |This bit indicates PDMA channel scatter-gather table is empty
     * |        |          |User can read PDMA_SCATSTS register to indicate which channel scatter-gather table is empty.
     * |        |          |0 = PDMA channel scatter-gather table is not empty.
     * |        |          |1 = PDMA channel scatter-gather table is empty.
     * |[8]     |REQTOF0   |PDMA Channel N Request Time-out Flag for Each Channel [N]
     * |        |          |This flag indicates that PDMA controller has waited peripheral request for a period defined by PDMA_TOC0, user can write 1 to clear these bits.
     * |        |          |0 = No request time-out.
     * |        |          |1 = Peripheral request time-out.
     * |[9]     |REQTOF1   |PDMA Channel N Request Time-out Flag for Each Channel [N]
     * |        |          |This flag indicates that PDMA controller has waited peripheral request for a period defined by PDMA_TOC10, user can write 1 to clear these bits.
     * |        |          |0 = No request time-out.
     * |        |          |1 = Peripheral request time-out.
     * @var PDMA_T::ABTSTS
     * Offset: 0x420  PDMA Channel Read/Write Target Abort Flag Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ABTIF0    |PDMA Channel 0 Read/Write Target Abort Interrupt Status Flag
     * |        |          |This bit indicates which PDMA controller has target abort error; User can write 1 to clear these bits.
     * |        |          |0 = No AHB bus ERROR response received when channel n transfer.
     * |        |          |1 = AHB bus ERROR response received when channel n transfer.
     * |[1]     |ABTIF1    |PDMA Channel 1 Read/Write Target Abort Interrupt Status Flag
     * |        |          |This bit indicates which PDMA controller has target abort error; User can write 1 to clear these bits.
     * |        |          |0 = No AHB bus ERROR response received when channel n transfer.
     * |        |          |1 = AHB bus ERROR response received when channel n transfer.
     * |[2]     |ABTIF2    |PDMA Channel 2 Read/Write Target Abort Interrupt Status Flag
     * |        |          |This bit indicates which PDMA controller has target abort error; User can write 1 to clear these bits.
     * |        |          |0 = No AHB bus ERROR response received when channel n transfer.
     * |        |          |1 = AHB bus ERROR response received when channel n transfer.
     * |[3]     |ABTIF3    |PDMA Channel 3 Read/Write Target Abort Interrupt Status Flag
     * |        |          |This bit indicates which PDMA controller has target abort error; User can write 1 to clear these bits.
     * |        |          |0 = No AHB bus ERROR response received when channel n transfer.
     * |        |          |1 = AHB bus ERROR response received when channel n transfer.
     * |[4]     |ABTIF4    |PDMA Channel 4 Read/Write Target Abort Interrupt Status Flag
     * |        |          |This bit indicates which PDMA controller has target abort error; User can write 1 to clear these bits.
     * |        |          |0 = No AHB bus ERROR response received when channel n transfer.
     * |        |          |1 = AHB bus ERROR response received when channel n transfer.
     * @var PDMA_T::TDSTS
     * Offset: 0x424  PDMA Channel Transfer Done Flag Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TDIF0     |PDMA Channel 0 Transfer Done Flag Register
     * |        |          |This bit indicates whether PDMA controller channel transfer has been finished or not, user can write 1 to clear these bits.
     * |        |          |0 = PDMA channel transfer has not finished.
     * |        |          |1 = PDMA channel has finished transmission.
     * |[1]     |TDIF1     |PDMA Channel 1 Transfer Done Flag Register
     * |        |          |This bit indicates whether PDMA controller channel transfer has been finished or not, user can write 1 to clear these bits.
     * |        |          |0 = PDMA channel transfer has not finished.
     * |        |          |1 = PDMA channel has finished transmission.
     * |[2]     |TDIF2     |PDMA Channel 2 Transfer Done Flag Register
     * |        |          |This bit indicates whether PDMA controller channel transfer has been finished or not, user can write 1 to clear these bits.
     * |        |          |0 = PDMA channel transfer has not finished.
     * |        |          |1 = PDMA channel has finished transmission.
     * |[3]     |TDIF3     |PDMA Channel 3 Transfer Done Flag Register
     * |        |          |This bit indicates whether PDMA controller channel transfer has been finished or not, user can write 1 to clear these bits.
     * |        |          |0 = PDMA channel transfer has not finished.
     * |        |          |1 = PDMA channel has finished transmission.
     * |[4]     |TDIF4     |PDMA Channel 4 Transfer Done Flag Register
     * |        |          |This bit indicates whether PDMA controller channel transfer has been finished or not, user can write 1 to clear these bits.
     * |        |          |0 = PDMA channel transfer has not finished.
     * |        |          |1 = PDMA channel has finished transmission.
     * @var PDMA_T::SCATSTS
     * Offset: 0x428  PDMA Scatter-Gather Table Empty Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TEMPTYF0  |Table Empty Flag Register
     * |        |          |T This bit indicates which PDMA channel table is empty when channel have a request , no matter request from software or peripheral, but operation mode of channel descriptor table is idle state, or channel has finished current transfer and next table operation mode is idle state for PDMA Scatter-Gather mode
     * |        |          |User can write 1 to clear these bits.
     * |        |          |0 = PDMA channel scatter-gather table is not empty.
     * |        |          |1 = PDMA channel scatter-gather table is empty and PDMA SWREQ has be set.
     * |[1]     |TEMPTYF1  |Table Empty Flag Register
     * |        |          |T This bit indicates which PDMA channel table is empty when channel have a request , no matter request from software or peripheral, but operation mode of channel descriptor table is idle state, or channel has finished current transfer and next table operation mode is idle state for PDMA Scatter-Gather mode
     * |        |          |User can write 1 to clear these bits.
     * |        |          |0 = PDMA channel scatter-gather table is not empty.
     * |        |          |1 = PDMA channel scatter-gather table is empty and PDMA SWREQ has be set.
     * |[2]     |TEMPTYF2  |Table Empty Flag Register
     * |        |          |T This bit indicates which PDMA channel table is empty when channel have a request , no matter request from software or peripheral, but operation mode of channel descriptor table is idle state, or channel has finished current transfer and next table operation mode is idle state for PDMA Scatter-Gather mode
     * |        |          |User can write 1 to clear these bits.
     * |        |          |0 = PDMA channel scatter-gather table is not empty.
     * |        |          |1 = PDMA channel scatter-gather table is empty and PDMA SWREQ has be set.
     * |[3]     |TEMPTYF3  |Table Empty Flag Register
     * |        |          |T This bit indicates which PDMA channel table is empty when channel have a request , no matter request from software or peripheral, but operation mode of channel descriptor table is idle state, or channel has finished current transfer and next table operation mode is idle state for PDMA Scatter-Gather mode
     * |        |          |User can write 1 to clear these bits.
     * |        |          |0 = PDMA channel scatter-gather table is not empty.
     * |        |          |1 = PDMA channel scatter-gather table is empty and PDMA SWREQ has be set.
     * |[4]     |TEMPTYF4  |Table Empty Flag Register
     * |        |          |T This bit indicates which PDMA channel table is empty when channel have a request , no matter request from software or peripheral, but operation mode of channel descriptor table is idle state, or channel has finished current transfer and next table operation mode is idle state for PDMA Scatter-Gather mode
     * |        |          |User can write 1 to clear these bits.
     * |        |          |0 = PDMA channel scatter-gather table is not empty.
     * |        |          |1 = PDMA channel scatter-gather table is empty and PDMA SWREQ has be set.
     * @var PDMA_T::TACTSTS
     * Offset: 0x42C  PDMA Transfer Active Flag Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TXACTF0   |PDMA Channel 0 Transfer on Active Flag Register (Read Only)
     * |        |          |This bit indicates which PDMA channel is in active.
     * |        |          |0 = PDMA channel is not finished.
     * |        |          |1 = PDMA channel is active.
     * |[1]     |TXACTF1   |PDMA Channel 1 Transfer on Active Flag Register (Read Only)
     * |        |          |This bit indicates which PDMA channel is in active.
     * |        |          |0 = PDMA channel is not finished.
     * |        |          |1 = PDMA channel is active.
     * |[2]     |TXACTF2   |PDMA Channel 2 Transfer on Active Flag Register (Read Only)
     * |        |          |This bit indicates which PDMA channel is in active.
     * |        |          |0 = PDMA channel is not finished.
     * |        |          |1 = PDMA channel is active.
     * |[3]     |TXACTF3   |PDMA Channel 3 Transfer on Active Flag Register (Read Only)
     * |        |          |This bit indicates which PDMA channel is in active.
     * |        |          |0 = PDMA channel is not finished.
     * |        |          |1 = PDMA channel is active.
     * |[4]     |TXACTF4   |PDMA Channel 4 Transfer on Active Flag Register (Read Only)
     * |        |          |This bit indicates which PDMA channel is in active.
     * |        |          |0 = PDMA channel is not finished.
     * |        |          |1 = PDMA channel is active.
     * @var PDMA_T::TOUTPSC
     * Offset: 0x430  PDMA Time-out Prescaler Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |TOUTPSC0  |PDMA Channel 0 Time-out Clock Source Prescaler Bits
     * |        |          |000 = PDMA channel 0 time-out clock source is HCLK/2^8.
     * |        |          |001 = PDMA channel 0 time-out clock source is HCLK/2^9.
     * |        |          |010 = PDMA channel 0 time-out clock source is HCLK/2^10.
     * |        |          |011 = PDMA channel 0 time-out clock source is HCLK/2^11.
     * |        |          |100 = PDMA channel 0 time-out clock source is HCLK/2^12.
     * |        |          |101 = PDMA channel 0 time-out clock source is HCLK/2^13.
     * |        |          |110 = PDMA channel 0 time-out clock source is HCLK/2^14.
     * |        |          |111 = PDMA channel 0 time-out clock source is HCLK/2^15.
     * |[6:4]   |TOUTPSC1  |PDMA Channel 1 Time-out Clock Source Prescaler Bits
     * |        |          |000 = PDMA channel 1 time-out clock source is HCLK/2^8.
     * |        |          |001 = PDMA channel 1 time-out clock source is HCLK/2^9.
     * |        |          |010 = PDMA channel 1 time-out clock source is HCLK/2^10.
     * |        |          |011 = PDMA channel 1 time-out clock source is HCLK/2^11.
     * |        |          |100 = PDMA channel 1 time-out clock source is HCLK/2^12.
     * |        |          |101 = PDMA channel 1 time-out clock source is HCLK/2^13.
     * |        |          |110 = PDMA channel 1 time-out clock source is HCLK/2^14.
     * |        |          |111 = PDMA channel 1 time-out clock source is HCLK/2^15.
     * @var PDMA_T::TOUTEN
     * Offset: 0x434  PDMA Time-out Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TOUTEN0   |PDMA Channel 0 Time-out Enable Bit
     * |        |          |0 = PDMA Channel 0 time-out function Disable.
     * |        |          |1 = PDMA Channel 0 time-out function Enable.
     * |[1]     |TOUTEN1   |PDMA Channel 1 Time-out Enable Bit
     * |        |          |0 = PDMA Channel 1 time-out function Disable.
     * |        |          |1 = PDMA Channel 1 time-out function Enable.
     * @var PDMA_T::TOUTIEN
     * Offset: 0x438  PDMA Time-out Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TOUTIEN0  |PDMA Channel 0 Time-out Interrupt Enable Bit
     * |        |          |0 = PDMA Channel 0 time-out interrupt Disable.
     * |        |          |1 = PDMA Channel 0 time-out interrupt Enable.
     * |[1]     |TOUTIEN1  |PDMA Channel 1 Time-out Interrupt Enable Bit
     * |        |          |0 = PDMA Channel 1 time-out interrupt Disable.
     * |        |          |1 = PDMA Channel 1 time-out interrupt Enable.
     * @var PDMA_T::SCATBA
     * Offset: 0x43C  PDMA Scatter-Gather Descriptor Table Base Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:16] |SCATBA    |PDMA Scatter-gather Descriptor Table Address Register
     * |        |          |In Scatter-Gather mode, this is the base address for calculating the next link - list address
     * |        |          |The next link address equation is
     * |        |          |Next Link Address = PDMA_SCATBA + PDMA_DSCT_FIRST.
     * |        |          |Note: Only useful in Scatter-Gather mode.
     * @var PDMA_T::TOC0_1
     * Offset: 0x440  PDMA Channel 0 and Channel 1 Time-out Counter Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |TOC0      |Time-out Counter for Channel 0
     * |        |          |This controls the period of time-out function for channel 0
     * |        |          |The calculation unit is based on TOUTPSC0 (PDMA_TOUTPSC[2:0]) clock.
     * |        |          |Time-out period = (Period of time-out clock) * (16-bit TOCn),n = 0,1.
     * |[31:16] |TOC1      |Time-out Counter for Channel 1
     * |        |          |This controls the period of time-out function for channel 1
     * |        |          |The calculation unit is based on TOUTPSC1 (PDMA_TOUTPSC[5:3]) clock
     * |        |          |The example of time-out period can refer TOC0 bit description.
     * @var PDMA_T::RESET
     * Offset: 0x460  PDMA Channel Reset Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RESET0    |PDMA Channel 0 Reset Control Register
     * |        |          |User can set this bit field to reset the PDMA channel
     * |        |          |When user sets RESETn bit, the PDMA controller will finish the on-going transfer then clear the channel enable bit CHEN(PDMA_CHCTL [0]) and clear request active flag
     * |        |          |If re-enable channel after channel reset, PDMA will re-load the channel description table to execute PDMA task.
     * |        |          |0 = No effect.
     * |        |          |1 = Reset PDMA channel 0.
     * |        |          |Note: This bit will be cleared automatically after finishing reset process.
     * |[1]     |RESET1    |PDMA Channel 1 Reset Control Register
     * |        |          |User can set this bit field to reset the PDMA channel
     * |        |          |When user sets RESETn bit, the PDMA controller will finish the on-going transfer then clear the channel enable bit CHEN(PDMA_CHCTL [1]) and clear request active flag
     * |        |          |If re-enable channel after channel reset, PDMA will re-load the channel description table to execute PDMA task.
     * |        |          |0 = No effect.
     * |        |          |1 = Reset PDMA channel 1.
     * |        |          |Note: This bit will be cleared automatically after finishing reset process.
     * |[2]     |RESET2    |PDMA Channel 2 Reset Control Register
     * |        |          |User can set this bit field to reset the PDMA channel
     * |        |          |When user sets RESETn bit, the PDMA controller will finish the on-going transfer then clear the channel enable bit CHEN(PDMA_CHCTL [2]) and clear request active flag
     * |        |          |If re-enable channel after channel reset, PDMA will re-load the channel description table to execute PDMA task.
     * |        |          |0 = No effect.
     * |        |          |1 = Reset PDMA channel 2.
     * |        |          |Note: This bit will be cleared automatically after finishing reset process.
     * |[3]     |RESET3    |PDMA Channel 3 Reset Control Register
     * |        |          |User can set this bit field to reset the PDMA channel
     * |        |          |When user sets RESETn bit, the PDMA controller will finish the on-going transfer then clear the channel enable bit CHEN(PDMA_CHCTL [3]) and clear request active flag
     * |        |          |If re-enable channel after channel reset, PDMA will re-load the channel description table to execute PDMA task.
     * |        |          |0 = No effect.
     * |        |          |1 = Reset PDMA channel 3.
     * |        |          |Note: This bit will be cleared automatically after finishing reset process.
     * |[4]     |RESET4    |PDMA Channel 4 Reset Control Register
     * |        |          |User can set this bit field to reset the PDMA channel
     * |        |          |When user sets RESETn bit, the PDMA controller will finish the on-going transfer then clear the channel enable bit CHEN(PDMA_CHCTL [4]) and clear request active flag
     * |        |          |If re-enable channel after channel reset, PDMA will re-load the channel description table to execute PDMA task.
     * |        |          |0 = No effect.
     * |        |          |1 = Reset PDMA channel 4.
     * |        |          |Note: This bit will be cleared automatically after finishing reset process.
     * @var PDMA_T::REQSEL0_3
     * Offset: 0x480  PDMA Channel 0 to Channel 3 Request Source Select Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |REQSRC0   |Channel 0 Request Source Selection
     * |        |          |This filed defines which peripheral is connected to PDMA channel 0
     * |        |          |User can configure the peripheral by setting REQSRC0.
     * |        |          |0 = Disable PDMA.
     * |        |          |1 = Reserved.
     * |        |          |2 = Channel connects to USB_TX.
     * |        |          |3 = Channel connects to USB_RX.
     * |        |          |4 = Channel connects to UART0_TX.
     * |        |          |5 = Channel connects to UART0_RX.
     * |        |          |6 = Channel connects to UART1_TX.
     * |        |          |7 = Channel connects to UART1_RX.
     * |        |          |8 = Channel connects to UART2_TX.
     * |        |          |9 = Channel connects to UART2_RX.
     * |        |          |16 = Channel connects to SPI0_TX.
     * |        |          |17 = Channel connects to SPI0_RX.
     * |        |          |18 = Channel connects to SPI1_TX.
     * |        |          |19 = Channel connects to SPI1_RX.
     * |        |          |20 = Channel connects to ADC_RX.
     * |        |          |21 = Channel connects to PWM0_P1_RX.
     * |        |          |22 = Channel connects to PWM0_P2_RX.
     * |        |          |23 = Channel connects to PWM0_P3_RX.
     * |        |          |24 = Channel connects to PWM1_P1_RX.
     * |        |          |25 = Channel connects to PWM1_P2_RX.
     * |        |          |26 = Channel connects to PWM1_P3_RX.
     * |        |          |27 = Reserved.
     * |        |          |28 = Channel connects to I2C0_TX.
     * |        |          |29 = Channel connects to I2C0_RX.
     * |        |          |30 = Channel connects to I2C1_TX.
     * |        |          |31 = Channel connects to I2C1_RX.
     * |        |          |32 = Channel connects to TMR0.
     * |        |          |33 = Channel connects to TMR1.
     * |        |          |34 = Channel connects to TMR2.
     * |        |          |35 = Channel connects to TMR3.
     * |        |          |Others = Reserved.
     * |        |          |Note 1: A request source cannot assign to two channels at the same time.
     * |        |          |Note 2: This field is useless when transfer between memory and memory.
     * |[13:8]  |REQSRC1   |Channel 1 Request Source Selection
     * |        |          |This filed defines which peripheral is connected to PDMA channel 1
     * |        |          |User can configure the peripheral setting by REQSRC1.
     * |        |          |Note: The channel configuration is the same as REQSRC0 field
     * |        |          |Please refer to the explanation of REQSRC0.
     * |[21:16] |REQSRC2   |Channel 2 Request Source Selection
     * |        |          |This filed defines which peripheral is connected to PDMA channel 2
     * |        |          |User can configure the peripheral setting by REQSRC2.
     * |        |          |Note: The channel configuration is the same as REQSRC0 field
     * |        |          |Please refer to the explanation of REQSRC0.
     * |[29:24] |REQSRC3   |Channel 3 Request Source Selection
     * |        |          |This filed defines which peripheral is connected to PDMA channel 3
     * |        |          |User can configure the peripheral setting by REQSRC3.
     * |        |          |Note: The channel configuration is the same as REQSRC0 field
     * |        |          |Please refer to the explanation of REQSRC0.
     * @var PDMA_T::REQSEL4
     * Offset: 0x484  PDMA Channel 4 Request Source Select Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |REQSRC4   |Channel 4 Request Source Selection
     * |        |          |This filed defines which peripheral is connected to PDMA channel 4
     * |        |          |User can configure the peripheral setting by REQSRC4.
     * |        |          |Note: The channel configuration is the same as REQSRC0 field
     * |        |          |Please refer to the explanation of REQSRC0.
     */

    DSCT_T        DSCT[5];               /*!< [0x0000 ~ 0x004C] DMA Embedded Description Table 0~4                      */
    __I  uint32_t CURSCAT[5];            /*!< [0x0050~0x0060] Current Scatter-Gather Descriptor Table Address of PDMA Channel 0~4 */
    __I  uint32_t RESERVE0[231];
    __IO uint32_t CHCTL;                 /*!< [0x0400] PDMA Channel Control Register                                    */
    __O  uint32_t PAUSE;                 /*!< [0x0404] PDMA Transfer Pause Control Register                             */
    __O  uint32_t SWREQ;                 /*!< [0x0408] PDMA Software Request Register                                   */
    __I  uint32_t TRGSTS;                /*!< [0x040c] PDMA Channel Request Status Register                             */
    __IO uint32_t PRISET;                /*!< [0x0410] PDMA Fixed Priority Setting Register                             */
    __O  uint32_t PRICLR;                /*!< [0x0414] PDMA Fixed Priority Clear Register                               */
    __IO uint32_t INTEN;                 /*!< [0x0418] PDMA Interrupt Enable Register                                   */
    __IO uint32_t INTSTS;                /*!< [0x041c] PDMA Interrupt Status Register                                   */
    __IO uint32_t ABTSTS;                /*!< [0x0420] PDMA Channel Read/Write Target Abort Flag Register               */
    __IO uint32_t TDSTS;                 /*!< [0x0424] PDMA Channel Transfer Done Flag Register                         */
    __IO uint32_t SCATSTS;               /*!< [0x0428] PDMA Scatter-Gather Table Empty Status Register                  */
    __I  uint32_t TACTSTS;               /*!< [0x042c] PDMA Transfer Active Flag Register                               */
    __IO uint32_t TOUTPSC;               /*!< [0x0430] PDMA Time-out Prescaler Register                                 */
    __IO uint32_t TOUTEN;                /*!< [0x0434] PDMA Time-out Enable Register                                    */
    __IO uint32_t TOUTIEN;               /*!< [0x0438] PDMA Time-out Interrupt Enable Register                          */
    __IO uint32_t SCATBA;                /*!< [0x043c] PDMA Scatter-Gather Descriptor Table Base Address Register       */
    __IO uint32_t TOC0_1;                /*!< [0x0440] PDMA Channel 0 and Channel 1 Time-out Counter Register           */
    __I  uint32_t RESERVE1[7];
    __IO uint32_t RESET;                 /*!< [0x0460] PDMA Channel Reset Control Register                              */
    __I  uint32_t RESERVE2[7];
    __IO uint32_t REQSEL0_3;             /*!< [0x0480] PDMA Channel 0 to Channel 3 Request Source Select Register       */
    __IO uint32_t REQSEL4;               /*!< [0x0484] PDMA Channel 4 Request Source Select Register                    */

} PDMA_T;




/**
    @addtogroup PDMA_CONST PDMA Bit Field Definition
    Constant Definitions for PDMA Controller
@{ */

#define PDMA_DSCT_CTL_OPMODE_Pos         (0)                                               /*!< PDMA_T::DSCT_CTL: OPMODE Position     */
#define PDMA_DSCT_CTL_OPMODE_Msk         (0x3ul << PDMA_DSCT_CTL_OPMODE_Pos)               /*!< PDMA_T::DSCT_CTL: OPMODE Mask         */

#define PDMA_DSCT_CTL_TXTYPE_Pos         (2)                                               /*!< PDMA_T::DSCT_CTL: TXTYPE Position     */
#define PDMA_DSCT_CTL_TXTYPE_Msk         (0x1ul << PDMA_DSCT_CTL_TXTYPE_Pos)               /*!< PDMA_T::DSCT_CTL: TXTYPE Mask         */

#define PDMA_DSCT_CTL_BURSIZE_Pos        (4)                                               /*!< PDMA_T::DSCT_CTL: BURSIZE Position    */
#define PDMA_DSCT_CTL_BURSIZE_Msk        (0x7ul << PDMA_DSCT_CTL_BURSIZE_Pos)              /*!< PDMA_T::DSCT_CTL: BURSIZE Mask        */

#define PDMA_DSCT_CTL_TBINTDIS_Pos       (7)                                               /*!< PDMA_T::DSCT_CTL: TBINTDIS Position   */
#define PDMA_DSCT_CTL_TBINTDIS_Msk       (0x1ul << PDMA_DSCT_CTL_TBINTDIS_Pos)             /*!< PDMA_T::DSCT_CTL: TBINTDIS Mask       */

#define PDMA_DSCT_CTL_SAINC_Pos          (8)                                               /*!< PDMA_T::DSCT_CTL: SAINC Position      */
#define PDMA_DSCT_CTL_SAINC_Msk          (0x3ul << PDMA_DSCT_CTL_SAINC_Pos)                /*!< PDMA_T::DSCT_CTL: SAINC Mask          */

#define PDMA_DSCT_CTL_DAINC_Pos          (10)                                              /*!< PDMA_T::DSCT_CTL: DAINC Position      */
#define PDMA_DSCT_CTL_DAINC_Msk          (0x3ul << PDMA_DSCT_CTL_DAINC_Pos)                /*!< PDMA_T::DSCT_CTL: DAINC Mask          */

#define PDMA_DSCT_CTL_TXWIDTH_Pos        (12)                                              /*!< PDMA_T::DSCT_CTL: TXWIDTH Position    */
#define PDMA_DSCT_CTL_TXWIDTH_Msk        (0x3ul << PDMA_DSCT_CTL_TXWIDTH_Pos)              /*!< PDMA_T::DSCT_CTL: TXWIDTH Mask        */

#define PDMA_DSCT_CTL_TXCNT_Pos          (16)                                              /*!< PDMA_T::DSCT_CTL: TXCNT Position      */
#define PDMA_DSCT_CTL_TXCNT_Msk          (0x3ffful << PDMA_DSCT_CTL_TXCNT_Pos)             /*!< PDMA_T::DSCT_CTL: TXCNT Mask          */

#define PDMA_DSCT_SA_SA_Pos              (0)                                               /*!< PDMA_T::DSCT_SA: SA Position          */
#define PDMA_DSCT_SA_SA_Msk              (0xfffffffful << PDMA_DSCT_SA_SA_Pos)             /*!< PDMA_T::DSCT_SA: SA Mask              */

#define PDMA_DSCT_DA_DA_Pos              (0)                                               /*!< PDMA_T::DSCT_DA: DA Position          */
#define PDMA_DSCT_DA_DA_Msk              (0xfffffffful << PDMA_DSCT_DA_DA_Pos)             /*!< PDMA_T::DSCT_DA: DA Mask              */

#define PDMA_DSCT_FIRST_FIRST_Pos        (0)                                               /*!< PDMA_T::DSCT_FIRST: FIRST Position    */
#define PDMA_DSCT_FIRST_FIRST_Msk        (0xfffful << PDMA_DSCT_FIRST_FIRST_Pos)           /*!< PDMA_T::DSCT_FIRST: FIRST Mask        */

#define PDMA_DSCT_FIRST_NEXT_Pos         (16)                                              /*!< PDMA_T::DSCT_FIRST: NEXT Position     */
#define PDMA_DSCT_FIRST_NEXT_Msk         (0xfffful << PDMA_DSCT_FIRST_NEXT_Pos)            /*!< PDMA_T::DSCT_FIRST: NEXT Mask         */

#define PDMA_CURSCAT_CURADDR_Pos         (0)                                               /*!< PDMA_T::CURSCAT: CURADDR Position     */
#define PDMA_CURSCAT_CURADDR_Msk         (0xfffffffful << PDMA_CURSCAT_CURADDR_Pos)        /*!< PDMA_T::CURSCAT: CURADDR Mask         */

#define PDMA_CHCTL_CHEN0_Pos             (0)                                               /*!< PDMA_T::CHCTL: CHEN0 Position          */
#define PDMA_CHCTL_CHEN0_Msk             (0x1ul << PDMA_CHCTL_CHEN0_Pos)                   /*!< PDMA_T::CHCTL: CHEN0 Mask              */

#define PDMA_CHCTL_CHEN1_Pos             (1)                                               /*!< PDMA_T::CHCTL: CHEN1 Position          */
#define PDMA_CHCTL_CHEN1_Msk             (0x1ul << PDMA_CHCTL_CHEN1_Pos)                   /*!< PDMA_T::CHCTL: CHEN1 Mask              */

#define PDMA_CHCTL_CHEN2_Pos             (2)                                               /*!< PDMA_T::CHCTL: CHEN2 Position          */
#define PDMA_CHCTL_CHEN2_Msk             (0x1ul << PDMA_CHCTL_CHEN2_Pos)                   /*!< PDMA_T::CHCTL: CHEN2 Mask              */

#define PDMA_CHCTL_CHEN3_Pos             (3)                                               /*!< PDMA_T::CHCTL: CHEN3 Position          */
#define PDMA_CHCTL_CHEN3_Msk             (0x1ul << PDMA_CHCTL_CHEN3_Pos)                   /*!< PDMA_T::CHCTL: CHEN3 Mask              */

#define PDMA_CHCTL_CHEN4_Pos             (4)                                               /*!< PDMA_T::CHCTL: CHEN4 Position          */
#define PDMA_CHCTL_CHEN4_Msk             (0x1ul << PDMA_CHCTL_CHEN4_Pos)                   /*!< PDMA_T::CHCTL: CHEN4 Mask              */

#define PDMA_PAUSE_PAUSE0_Pos            (0)                                               /*!< PDMA_T::PAUSE: PAUSE0 Position         */
#define PDMA_PAUSE_PAUSE0_Msk            (0x1ul << PDMA_PAUSE_PAUSE0_Pos)                  /*!< PDMA_T::PAUSE: PAUSE0 Mask             */

#define PDMA_PAUSE_PAUSE1_Pos            (1)                                               /*!< PDMA_T::PAUSE: PAUSE1 Position         */
#define PDMA_PAUSE_PAUSE1_Msk            (0x1ul << PDMA_PAUSE_PAUSE1_Pos)                  /*!< PDMA_T::PAUSE: PAUSE1 Mask             */

#define PDMA_PAUSE_PAUSE2_Pos            (2)                                               /*!< PDMA_T::PAUSE: PAUSE2 Position         */
#define PDMA_PAUSE_PAUSE2_Msk            (0x1ul << PDMA_PAUSE_PAUSE2_Pos)                  /*!< PDMA_T::PAUSE: PAUSE2 Mask             */

#define PDMA_PAUSE_PAUSE3_Pos            (3)                                               /*!< PDMA_T::PAUSE: PAUSE3 Position         */
#define PDMA_PAUSE_PAUSE3_Msk            (0x1ul << PDMA_PAUSE_PAUSE3_Pos)                  /*!< PDMA_T::PAUSE: PAUSE3 Mask             */

#define PDMA_PAUSE_PAUSE4_Pos            (4)                                               /*!< PDMA_T::PAUSE: PAUSE4 Position         */
#define PDMA_PAUSE_PAUSE4_Msk            (0x1ul << PDMA_PAUSE_PAUSE4_Pos)                  /*!< PDMA_T::PAUSE: PAUSE4 Mask             */

#define PDMA_SWREQ_SWREQ0_Pos            (0)                                               /*!< PDMA_T::SWREQ: SWREQ0 Position         */
#define PDMA_SWREQ_SWREQ0_Msk            (0x1ul << PDMA_SWREQ_SWREQ0_Pos)                  /*!< PDMA_T::SWREQ: SWREQ0 Mask             */

#define PDMA_SWREQ_SWREQ1_Pos            (1)                                               /*!< PDMA_T::SWREQ: SWREQ1 Position         */
#define PDMA_SWREQ_SWREQ1_Msk            (0x1ul << PDMA_SWREQ_SWREQ1_Pos)                  /*!< PDMA_T::SWREQ: SWREQ1 Mask             */

#define PDMA_SWREQ_SWREQ2_Pos            (2)                                               /*!< PDMA_T::SWREQ: SWREQ2 Position         */
#define PDMA_SWREQ_SWREQ2_Msk            (0x1ul << PDMA_SWREQ_SWREQ2_Pos)                  /*!< PDMA_T::SWREQ: SWREQ2 Mask             */

#define PDMA_SWREQ_SWREQ3_Pos            (3)                                               /*!< PDMA_T::SWREQ: SWREQ3 Position         */
#define PDMA_SWREQ_SWREQ3_Msk            (0x1ul << PDMA_SWREQ_SWREQ3_Pos)                  /*!< PDMA_T::SWREQ: SWREQ3 Mask             */

#define PDMA_SWREQ_SWREQ4_Pos            (4)                                               /*!< PDMA_T::SWREQ: SWREQ4 Position         */
#define PDMA_SWREQ_SWREQ4_Msk            (0x1ul << PDMA_SWREQ_SWREQ4_Pos)                  /*!< PDMA_T::SWREQ: SWREQ4 Mask             */

#define PDMA_TRGSTS_REQSTS0_Pos          (0)                                               /*!< PDMA_T::TRGSTS: REQSTS0 Position       */
#define PDMA_TRGSTS_REQSTS0_Msk          (0x1ul << PDMA_TRGSTS_REQSTS0_Pos)                /*!< PDMA_T::TRGSTS: REQSTS0 Mask           */

#define PDMA_TRGSTS_REQSTS1_Pos          (1)                                               /*!< PDMA_T::TRGSTS: REQSTS1 Position       */
#define PDMA_TRGSTS_REQSTS1_Msk          (0x1ul << PDMA_TRGSTS_REQSTS1_Pos)                /*!< PDMA_T::TRGSTS: REQSTS1 Mask           */

#define PDMA_TRGSTS_REQSTS2_Pos          (2)                                               /*!< PDMA_T::TRGSTS: REQSTS2 Position       */
#define PDMA_TRGSTS_REQSTS2_Msk          (0x1ul << PDMA_TRGSTS_REQSTS2_Pos)                /*!< PDMA_T::TRGSTS: REQSTS2 Mask           */

#define PDMA_TRGSTS_REQSTS3_Pos          (3)                                               /*!< PDMA_T::TRGSTS: REQSTS3 Position       */
#define PDMA_TRGSTS_REQSTS3_Msk          (0x1ul << PDMA_TRGSTS_REQSTS3_Pos)                /*!< PDMA_T::TRGSTS: REQSTS3 Mask           */

#define PDMA_TRGSTS_REQSTS4_Pos          (4)                                               /*!< PDMA_T::TRGSTS: REQSTS4 Position       */
#define PDMA_TRGSTS_REQSTS4_Msk          (0x1ul << PDMA_TRGSTS_REQSTS4_Pos)                /*!< PDMA_T::TRGSTS: REQSTS4 Mask           */

#define PDMA_PRISET_FPRISET0_Pos         (0)                                               /*!< PDMA_T::PRISET: FPRISET0 Position      */
#define PDMA_PRISET_FPRISET0_Msk         (0x1ul << PDMA_PRISET_FPRISET0_Pos)               /*!< PDMA_T::PRISET: FPRISET0 Mask          */

#define PDMA_PRISET_FPRISET1_Pos         (1)                                               /*!< PDMA_T::PRISET: FPRISET1 Position      */
#define PDMA_PRISET_FPRISET1_Msk         (0x1ul << PDMA_PRISET_FPRISET1_Pos)               /*!< PDMA_T::PRISET: FPRISET1 Mask          */

#define PDMA_PRISET_FPRISET2_Pos         (2)                                               /*!< PDMA_T::PRISET: FPRISET2 Position      */
#define PDMA_PRISET_FPRISET2_Msk         (0x1ul << PDMA_PRISET_FPRISET2_Pos)               /*!< PDMA_T::PRISET: FPRISET2 Mask          */

#define PDMA_PRISET_FPRISET3_Pos         (3)                                               /*!< PDMA_T::PRISET: FPRISET3 Position      */
#define PDMA_PRISET_FPRISET3_Msk         (0x1ul << PDMA_PRISET_FPRISET3_Pos)               /*!< PDMA_T::PRISET: FPRISET3 Mask          */

#define PDMA_PRISET_FPRISET4_Pos         (4)                                               /*!< PDMA_T::PRISET: FPRISET4 Position      */
#define PDMA_PRISET_FPRISET4_Msk         (0x1ul << PDMA_PRISET_FPRISET4_Pos)               /*!< PDMA_T::PRISET: FPRISET4 Mask          */

#define PDMA_PRICLR_FPRICLR0_Pos         (0)                                               /*!< PDMA_T::PRICLR: FPRICLR0 Position      */
#define PDMA_PRICLR_FPRICLR0_Msk         (0x1ul << PDMA_PRICLR_FPRICLR0_Pos)               /*!< PDMA_T::PRICLR: FPRICLR0 Mask          */

#define PDMA_PRICLR_FPRICLR1_Pos         (1)                                               /*!< PDMA_T::PRICLR: FPRICLR1 Position      */
#define PDMA_PRICLR_FPRICLR1_Msk         (0x1ul << PDMA_PRICLR_FPRICLR1_Pos)               /*!< PDMA_T::PRICLR: FPRICLR1 Mask          */

#define PDMA_PRICLR_FPRICLR2_Pos         (2)                                               /*!< PDMA_T::PRICLR: FPRICLR2 Position      */
#define PDMA_PRICLR_FPRICLR2_Msk         (0x1ul << PDMA_PRICLR_FPRICLR2_Pos)               /*!< PDMA_T::PRICLR: FPRICLR2 Mask          */

#define PDMA_PRICLR_FPRICLR3_Pos         (3)                                               /*!< PDMA_T::PRICLR: FPRICLR3 Position      */
#define PDMA_PRICLR_FPRICLR3_Msk         (0x1ul << PDMA_PRICLR_FPRICLR3_Pos)               /*!< PDMA_T::PRICLR: FPRICLR3 Mask          */

#define PDMA_PRICLR_FPRICLR4_Pos         (4)                                               /*!< PDMA_T::PRICLR: FPRICLR4 Position      */
#define PDMA_PRICLR_FPRICLR4_Msk         (0x1ul << PDMA_PRICLR_FPRICLR4_Pos)               /*!< PDMA_T::PRICLR: FPRICLR4 Mask          */

#define PDMA_INTEN_INTEN0_Pos            (0)                                               /*!< PDMA_T::INTEN: INTEN0 Position         */
#define PDMA_INTEN_INTEN0_Msk            (0x1ul << PDMA_INTEN_INTEN0_Pos)                  /*!< PDMA_T::INTEN: INTEN0 Mask             */

#define PDMA_INTEN_INTEN1_Pos            (1)                                               /*!< PDMA_T::INTEN: INTEN1 Position         */
#define PDMA_INTEN_INTEN1_Msk            (0x1ul << PDMA_INTEN_INTEN1_Pos)                  /*!< PDMA_T::INTEN: INTEN1 Mask             */

#define PDMA_INTEN_INTEN2_Pos            (2)                                               /*!< PDMA_T::INTEN: INTEN2 Position         */
#define PDMA_INTEN_INTEN2_Msk            (0x1ul << PDMA_INTEN_INTEN2_Pos)                  /*!< PDMA_T::INTEN: INTEN2 Mask             */

#define PDMA_INTEN_INTEN3_Pos            (3)                                               /*!< PDMA_T::INTEN: INTEN3 Position         */
#define PDMA_INTEN_INTEN3_Msk            (0x1ul << PDMA_INTEN_INTEN3_Pos)                  /*!< PDMA_T::INTEN: INTEN3 Mask             */

#define PDMA_INTEN_INTEN4_Pos            (4)                                               /*!< PDMA_T::INTEN: INTEN4 Position         */
#define PDMA_INTEN_INTEN4_Msk            (0x1ul << PDMA_INTEN_INTEN4_Pos)                  /*!< PDMA_T::INTEN: INTEN4 Mask             */

#define PDMA_INTSTS_ABTIF_Pos            (0)                                               /*!< PDMA_T::INTSTS: ABTIF Position         */
#define PDMA_INTSTS_ABTIF_Msk            (0x1ul << PDMA_INTSTS_ABTIF_Pos)                  /*!< PDMA_T::INTSTS: ABTIF Mask             */

#define PDMA_INTSTS_TDIF_Pos             (1)                                               /*!< PDMA_T::INTSTS: TDIF Position          */
#define PDMA_INTSTS_TDIF_Msk             (0x1ul << PDMA_INTSTS_TDIF_Pos)                   /*!< PDMA_T::INTSTS: TDIF Mask              */

#define PDMA_INTSTS_TEIF_Pos             (2)                                               /*!< PDMA_T::INTSTS: TEIF Position          */
#define PDMA_INTSTS_TEIF_Msk             (0x1ul << PDMA_INTSTS_TEIF_Pos)                   /*!< PDMA_T::INTSTS: TEIF Mask              */

#define PDMA_INTSTS_REQTOF0_Pos          (8)                                               /*!< PDMA_T::INTSTS: REQTOF0 Position       */
#define PDMA_INTSTS_REQTOF0_Msk          (0x1ul << PDMA_INTSTS_REQTOF0_Pos)                /*!< PDMA_T::INTSTS: REQTOF0 Mask           */

#define PDMA_INTSTS_REQTOF1_Pos          (9)                                               /*!< PDMA_T::INTSTS: REQTOF1 Position       */
#define PDMA_INTSTS_REQTOF1_Msk          (0x1ul << PDMA_INTSTS_REQTOF1_Pos)                /*!< PDMA_T::INTSTS: REQTOF1 Mask           */

#define PDMA_ABTSTS_ABTIF0_Pos           (0)                                               /*!< PDMA_T::ABTSTS: ABTIF0 Position        */
#define PDMA_ABTSTS_ABTIF0_Msk           (0x1ul << PDMA_ABTSTS_ABTIF0_Pos)                 /*!< PDMA_T::ABTSTS: ABTIF0 Mask            */

#define PDMA_ABTSTS_ABTIF1_Pos           (1)                                               /*!< PDMA_T::ABTSTS: ABTIF1 Position        */
#define PDMA_ABTSTS_ABTIF1_Msk           (0x1ul << PDMA_ABTSTS_ABTIF1_Pos)                 /*!< PDMA_T::ABTSTS: ABTIF1 Mask            */

#define PDMA_ABTSTS_ABTIF2_Pos           (2)                                               /*!< PDMA_T::ABTSTS: ABTIF2 Position        */
#define PDMA_ABTSTS_ABTIF2_Msk           (0x1ul << PDMA_ABTSTS_ABTIF2_Pos)                 /*!< PDMA_T::ABTSTS: ABTIF2 Mask            */

#define PDMA_ABTSTS_ABTIF3_Pos           (3)                                               /*!< PDMA_T::ABTSTS: ABTIF3 Position        */
#define PDMA_ABTSTS_ABTIF3_Msk           (0x1ul << PDMA_ABTSTS_ABTIF3_Pos)                 /*!< PDMA_T::ABTSTS: ABTIF3 Mask            */

#define PDMA_ABTSTS_ABTIF4_Pos           (4)                                               /*!< PDMA_T::ABTSTS: ABTIF4 Position        */
#define PDMA_ABTSTS_ABTIF4_Msk           (0x1ul << PDMA_ABTSTS_ABTIF4_Pos)                 /*!< PDMA_T::ABTSTS: ABTIF4 Mask            */

#define PDMA_TDSTS_TDIF0_Pos             (0)                                               /*!< PDMA_T::TDSTS: TDIF0 Position          */
#define PDMA_TDSTS_TDIF0_Msk             (0x1ul << PDMA_TDSTS_TDIF0_Pos)                   /*!< PDMA_T::TDSTS: TDIF0 Mask              */

#define PDMA_TDSTS_TDIF1_Pos             (1)                                               /*!< PDMA_T::TDSTS: TDIF1 Position          */
#define PDMA_TDSTS_TDIF1_Msk             (0x1ul << PDMA_TDSTS_TDIF1_Pos)                   /*!< PDMA_T::TDSTS: TDIF1 Mask              */

#define PDMA_TDSTS_TDIF2_Pos             (2)                                               /*!< PDMA_T::TDSTS: TDIF2 Position          */
#define PDMA_TDSTS_TDIF2_Msk             (0x1ul << PDMA_TDSTS_TDIF2_Pos)                   /*!< PDMA_T::TDSTS: TDIF2 Mask              */

#define PDMA_TDSTS_TDIF3_Pos             (3)                                               /*!< PDMA_T::TDSTS: TDIF3 Position          */
#define PDMA_TDSTS_TDIF3_Msk             (0x1ul << PDMA_TDSTS_TDIF3_Pos)                   /*!< PDMA_T::TDSTS: TDIF3 Mask              */

#define PDMA_TDSTS_TDIF4_Pos             (4)                                               /*!< PDMA_T::TDSTS: TDIF4 Position          */
#define PDMA_TDSTS_TDIF4_Msk             (0x1ul << PDMA_TDSTS_TDIF4_Pos)                   /*!< PDMA_T::TDSTS: TDIF4 Mask              */

#define PDMA_SCATSTS_TEMPTYF0_Pos        (0)                                               /*!< PDMA_T::SCATSTS: TEMPTYF0 Position     */
#define PDMA_SCATSTS_TEMPTYF0_Msk        (0x1ul << PDMA_SCATSTS_TEMPTYF0_Pos)              /*!< PDMA_T::SCATSTS: TEMPTYF0 Mask         */

#define PDMA_SCATSTS_TEMPTYF1_Pos        (1)                                               /*!< PDMA_T::SCATSTS: TEMPTYF1 Position     */
#define PDMA_SCATSTS_TEMPTYF1_Msk        (0x1ul << PDMA_SCATSTS_TEMPTYF1_Pos)              /*!< PDMA_T::SCATSTS: TEMPTYF1 Mask         */

#define PDMA_SCATSTS_TEMPTYF2_Pos        (2)                                               /*!< PDMA_T::SCATSTS: TEMPTYF2 Position     */
#define PDMA_SCATSTS_TEMPTYF2_Msk        (0x1ul << PDMA_SCATSTS_TEMPTYF2_Pos)              /*!< PDMA_T::SCATSTS: TEMPTYF2 Mask         */

#define PDMA_SCATSTS_TEMPTYF3_Pos        (3)                                               /*!< PDMA_T::SCATSTS: TEMPTYF3 Position     */
#define PDMA_SCATSTS_TEMPTYF3_Msk        (0x1ul << PDMA_SCATSTS_TEMPTYF3_Pos)              /*!< PDMA_T::SCATSTS: TEMPTYF3 Mask         */

#define PDMA_SCATSTS_TEMPTYF4_Pos        (4)                                               /*!< PDMA_T::SCATSTS: TEMPTYF4 Position     */
#define PDMA_SCATSTS_TEMPTYF4_Msk        (0x1ul << PDMA_SCATSTS_TEMPTYF4_Pos)              /*!< PDMA_T::SCATSTS: TEMPTYF4 Mask         */

#define PDMA_TACTSTS_TXACTF0_Pos         (0)                                               /*!< PDMA_T::TACTSTS: TXACTF0 Position      */
#define PDMA_TACTSTS_TXACTF0_Msk         (0x1ul << PDMA_TACTSTS_TXACTF0_Pos)               /*!< PDMA_T::TACTSTS: TXACTF0 Mask          */

#define PDMA_TACTSTS_TXACTF1_Pos         (1)                                               /*!< PDMA_T::TACTSTS: TXACTF1 Position      */
#define PDMA_TACTSTS_TXACTF1_Msk         (0x1ul << PDMA_TACTSTS_TXACTF1_Pos)               /*!< PDMA_T::TACTSTS: TXACTF1 Mask          */

#define PDMA_TACTSTS_TXACTF2_Pos         (2)                                               /*!< PDMA_T::TACTSTS: TXACTF2 Position      */
#define PDMA_TACTSTS_TXACTF2_Msk         (0x1ul << PDMA_TACTSTS_TXACTF2_Pos)               /*!< PDMA_T::TACTSTS: TXACTF2 Mask          */

#define PDMA_TACTSTS_TXACTF3_Pos         (3)                                               /*!< PDMA_T::TACTSTS: TXACTF3 Position      */
#define PDMA_TACTSTS_TXACTF3_Msk         (0x1ul << PDMA_TACTSTS_TXACTF3_Pos)               /*!< PDMA_T::TACTSTS: TXACTF3 Mask          */

#define PDMA_TACTSTS_TXACTF4_Pos         (4)                                               /*!< PDMA_T::TACTSTS: TXACTF4 Position      */
#define PDMA_TACTSTS_TXACTF4_Msk         (0x1ul << PDMA_TACTSTS_TXACTF4_Pos)               /*!< PDMA_T::TACTSTS: TXACTF4 Mask          */

#define PDMA_TOUTPSC_TOUTPSC0_Pos        (0)                                               /*!< PDMA_T::TOUTPSC: TOUTPSC0 Position     */
#define PDMA_TOUTPSC_TOUTPSC0_Msk        (0x7ul << PDMA_TOUTPSC_TOUTPSC0_Pos)              /*!< PDMA_T::TOUTPSC: TOUTPSC0 Mask         */

#define PDMA_TOUTPSC_TOUTPSC1_Pos        (4)                                               /*!< PDMA_T::TOUTPSC: TOUTPSC1 Position     */
#define PDMA_TOUTPSC_TOUTPSC1_Msk        (0x7ul << PDMA_TOUTPSC_TOUTPSC1_Pos)              /*!< PDMA_T::TOUTPSC: TOUTPSC1 Mask         */

#define PDMA_TOUTEN_TOUTEN0_Pos          (0)                                               /*!< PDMA_T::TOUTEN: TOUTEN0 Position       */
#define PDMA_TOUTEN_TOUTEN0_Msk          (0x1ul << PDMA_TOUTEN_TOUTEN0_Pos)                /*!< PDMA_T::TOUTEN: TOUTEN0 Mask           */

#define PDMA_TOUTEN_TOUTEN1_Pos          (1)                                               /*!< PDMA_T::TOUTEN: TOUTEN1 Position       */
#define PDMA_TOUTEN_TOUTEN1_Msk          (0x1ul << PDMA_TOUTEN_TOUTEN1_Pos)                /*!< PDMA_T::TOUTEN: TOUTEN1 Mask           */

#define PDMA_TOUTIEN_TOUTIEN0_Pos        (0)                                               /*!< PDMA_T::TOUTIEN: TOUTIEN0 Position     */
#define PDMA_TOUTIEN_TOUTIEN0_Msk        (0x1ul << PDMA_TOUTIEN_TOUTIEN0_Pos)              /*!< PDMA_T::TOUTIEN: TOUTIEN0 Mask         */

#define PDMA_TOUTIEN_TOUTIEN1_Pos        (1)                                               /*!< PDMA_T::TOUTIEN: TOUTIEN1 Position     */
#define PDMA_TOUTIEN_TOUTIEN1_Msk        (0x1ul << PDMA_TOUTIEN_TOUTIEN1_Pos)              /*!< PDMA_T::TOUTIEN: TOUTIEN1 Mask         */

#define PDMA_SCATBA_SCATBA_Pos           (16)                                              /*!< PDMA_T::SCATBA: SCATBA Position        */
#define PDMA_SCATBA_SCATBA_Msk           (0xfffful << PDMA_SCATBA_SCATBA_Pos)              /*!< PDMA_T::SCATBA: SCATBA Mask            */

#define PDMA_TOC0_1_TOC0_Pos             (0)                                               /*!< PDMA_T::TOC0_1: TOC0 Position          */
#define PDMA_TOC0_1_TOC0_Msk             (0xfffful << PDMA_TOC0_1_TOC0_Pos)                /*!< PDMA_T::TOC0_1: TOC0 Mask              */

#define PDMA_TOC0_1_TOC1_Pos             (16)                                              /*!< PDMA_T::TOC0_1: TOC1 Position          */
#define PDMA_TOC0_1_TOC1_Msk             (0xfffful << PDMA_TOC0_1_TOC1_Pos)                /*!< PDMA_T::TOC0_1: TOC1 Mask              */

#define PDMA_RESET_RESET0_Pos            (0)                                               /*!< PDMA_T::RESET: RESET0 Position         */
#define PDMA_RESET_RESET0_Msk            (0x1ul << PDMA_RESET_RESET0_Pos)                  /*!< PDMA_T::RESET: RESET0 Mask             */

#define PDMA_RESET_RESET1_Pos            (1)                                               /*!< PDMA_T::RESET: RESET1 Position         */
#define PDMA_RESET_RESET1_Msk            (0x1ul << PDMA_RESET_RESET1_Pos)                  /*!< PDMA_T::RESET: RESET1 Mask             */

#define PDMA_RESET_RESET2_Pos            (2)                                               /*!< PDMA_T::RESET: RESET2 Position         */
#define PDMA_RESET_RESET2_Msk            (0x1ul << PDMA_RESET_RESET2_Pos)                  /*!< PDMA_T::RESET: RESET2 Mask             */

#define PDMA_RESET_RESET3_Pos            (3)                                               /*!< PDMA_T::RESET: RESET3 Position         */
#define PDMA_RESET_RESET3_Msk            (0x1ul << PDMA_RESET_RESET3_Pos)                  /*!< PDMA_T::RESET: RESET3 Mask             */

#define PDMA_RESET_RESET4_Pos            (4)                                               /*!< PDMA_T::RESET: RESET4 Position         */
#define PDMA_RESET_RESET4_Msk            (0x1ul << PDMA_RESET_RESET4_Pos)                  /*!< PDMA_T::RESET: RESET4 Mask             */

#define PDMA_REQSEL0_3_REQSRC0_Pos       (0)                                               /*!< PDMA_T::REQSEL0_3: REQSRC0 Position    */
#define PDMA_REQSEL0_3_REQSRC0_Msk       (0x3ful << PDMA_REQSEL0_3_REQSRC0_Pos)            /*!< PDMA_T::REQSEL0_3: REQSRC0 Mask        */

#define PDMA_REQSEL0_3_REQSRC1_Pos       (8)                                               /*!< PDMA_T::REQSEL0_3: REQSRC1 Position    */
#define PDMA_REQSEL0_3_REQSRC1_Msk       (0x3ful << PDMA_REQSEL0_3_REQSRC1_Pos)            /*!< PDMA_T::REQSEL0_3: REQSRC1 Mask        */

#define PDMA_REQSEL0_3_REQSRC2_Pos       (16)                                              /*!< PDMA_T::REQSEL0_3: REQSRC2 Position    */
#define PDMA_REQSEL0_3_REQSRC2_Msk       (0x3ful << PDMA_REQSEL0_3_REQSRC2_Pos)            /*!< PDMA_T::REQSEL0_3: REQSRC2 Mask        */

#define PDMA_REQSEL0_3_REQSRC3_Pos       (24)                                              /*!< PDMA_T::REQSEL0_3: REQSRC3 Position    */
#define PDMA_REQSEL0_3_REQSRC3_Msk       (0x3ful << PDMA_REQSEL0_3_REQSRC3_Pos)            /*!< PDMA_T::REQSEL0_3: REQSRC3 Mask        */

#define PDMA_REQSEL4_REQSRC4_Pos         (0)                                               /*!< PDMA_T::REQSEL4: REQSRC4 Position      */
#define PDMA_REQSEL4_REQSRC4_Msk         (0x3ful << PDMA_REQSEL4_REQSRC4_Pos)              /*!< PDMA_T::REQSEL4: REQSRC4 Mask          */

/**@}*/ /* PDMA_CONST */
/**@}*/ /* end of PDMA register group */


/*---------------------- Pulse Width Modulation Controller -------------------------*/
/**
    @addtogroup PWM Pulse Width Modulation Controller(PWM)
    Memory Mapped Structure for PWM Controller
@{ */

typedef struct
{


    /**
     * @var PWM_T::CTL0
     * Offset: 0x00  PWM Control Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CTRLD0    |Center Re-load
     * |        |          |Bit 0 controls the PWM channel 0.
     * |        |          |In up-down counter type, PERIOD0 register will load to PBUF0 register at the end point of each period.
     * |        |          |CMPDAT0 register will load to CMPBUF0 register at the center point of a period.
     * |[1]     |CTRLD1    |Center Re-load
     * |        |          |Bit 1 controls the PWM channel 1.
     * |        |          |In up-down counter type, PERIOD1 register will load to PBUF1 register at the end point of each period.
     * |        |          |CMPDAT1 register will load to CMPBUF1 register at the center point of a period.
     * |[2]     |CTRLD2    |Center Re-load
     * |        |          |Bit 2 controls the PWM channel 2.
     * |        |          |In up-down counter type, PERIOD2 register will load to PBUF2 register at the end point of each period.
     * |        |          |CMPDAT2 register will load to CMPBUF2 register at the center point of a period.
     * |[3]     |CTRLD3    |Center Re-load
     * |        |          |Bit 3 controls the PWM channel 3.
     * |        |          |In up-down counter type, PERIOD3 register will load to PBUF3 register at the end point of each period.
     * |        |          |CMPDAT3 register will load to CMPBUF3 register at the center point of a period.
     * |[4]     |CTRLD4    |Center Re-load
     * |        |          |Bit 4 controls the PWM channel 4.
     * |        |          |In up-down counter type, PERIOD4 register will load to PBUF4 register at the end point of each period.
     * |        |          |CMPDAT4 register will load to CMPBUF4 register at the center point of a period.
     * |[5]     |CTRLD5    |Center Re-load
     * |        |          |Bit 5 controls the PWM channel 5.
     * |        |          |In up-down counter type, PERIOD5 register will load to PBUF5 register at the end point of each period.
     * |        |          |CMPDAT5 register will load to CMPBUF5 register at the center point of a period.
     * |[8]     |WINLDEN0  |Window Load Enable Bits
     * |        |          |Bit 0 controls the PWM channel 0.
     * |        |          |0 = PERIOD0 register will load to PBUF0 register at the end point of each period.
     * |        |          |CMPDAT0 register will load to CMPBUF0 register at the end point or center point of each period by setting CTRLD0 bit.
     * |        |          |1 = PERIOD0 register will load to PBUF0 and CMPDAT0 registers will load to CMPBUF0 register at the end point of each period when valid reload window is set.
     * |        |          |The valid reload window is set by software write 1 to PWM_LOAD register, and cleared by hardware after load success.
     * |[9]     |WINLDEN1  |Window Load Enable Bits
     * |        |          |Bit 1 controls the PWM channel 1.
     * |        |          |0 = PERIOD1 register will load to PBUF1 register at the end point of each period.
     * |        |          |CMPDAT1 register will load to CMPBUF1 register at the end point or center point of each period by setting CTRLD1 bit.
     * |        |          |1 = PERIOD1 register will load to PBUF1 and CMPDAT1 registers will load to CMPBUF1 register at the end point of each period when valid reload window is set.
     * |        |          |The valid reload window is set by software write 1 to PWM_LOAD register, and cleared by hardware after load success.
     * |[10]    |WINLDEN2  |Window Load Enable Bits
     * |        |          |Bit 2 controls the PWM channel 2.
     * |        |          |0 = PERIOD2 register will load to PBUF2 register at the end point of each period.
     * |        |          |CMPDAT2 register will load to CMPBUF2 register at the end point or center point of each period by setting CTRLD2 bit.
     * |        |          |1 = PERIOD2 register will load to PBUF2 and CMPDAT2 registers will load to CMPBUF2 register at the end point of each period when valid reload window is set.
     * |        |          |The valid reload window is set by software write 1 to PWM_LOAD register, and cleared by hardware after load success.
     * |[11]    |WINLDEN3  |Window Load Enable Bits
     * |        |          |Bit 3 controls the PWM channel 3.
     * |        |          |0 = PERIOD3 register will load to PBUF3 register at the end point of each period.
     * |        |          |CMPDAT3 register will load to CMPBUF3 register at the end point or center point of each period by setting CTRLD3 bit.
     * |        |          |1 = PERIOD3 register will load to PBUF3 and CMPDAT3 registers will load to CMPBUF3 register at the end point of each period when valid reload window is set.
     * |        |          |The valid reload window is set by software write 1 to PWM_LOAD register, and cleared by hardware after load success.
     * |[12]    |WINLDEN4  |Window Load Enable Bits
     * |        |          |Bit 4 controls the PWM channel 4.
     * |        |          |0 = PERIOD4 register will load to PBUF4 register at the end point of each period.
     * |        |          |CMPDAT4 register will load to CMPBUF4 register at the end point or center point of each period by setting CTRLD4 bit.
     * |        |          |1 = PERIOD4 register will load to PBUF4 and CMPDAT4 registers will load to CMPBUF4 register at the end point of each period when valid reload window is set.
     * |        |          |The valid reload window is set by software write 1 to PWM_LOAD register, and cleared by hardware after load success.
     * |[13]    |WINLDEN5  |Window Load Enable Bits
     * |        |          |Bit 5 controls the PWM channel 5.
     * |        |          |0 = PERIOD5 register will load to PBUF5 register at the end point of each period.
     * |        |          |CMPDAT5 register will load to CMPBUF5 register at the end point or center point of each period by setting CTRLD5 bit.
     * |        |          |1 = PERIOD5 register will load to PBUF5 and CMPDAT5 registers will load to CMPBUF5 register at the end point of each period when valid reload window is set.
     * |        |          |The valid reload window is set by software write 1 to PWM_LOAD register, and cleared by hardware after load success.
     * |[16]    |IMMLDEN0  |Immediately Load Enable Bits
     * |        |          |Bit 0 controls the PWM channel 0.
     * |        |          |0 = PERIOD0 register will load to PBUF0 register at the end point of each period.
     * |        |          |CMPDAT0 register will load to CMPBUF0 register at the end point or center point of each period by setting CTRLD0 bit.
     * |        |          |1 = PERIOD0/CMPDAT0 registers will load to PBUF0 and CMPBUF0 register immediately when software update PERIOD0/CMPDAT0 register.
     * |        |          |Note: If IMMLDEN0 bit is enabled, WINLDEN0 bit and CTRLD0 bits will be invalid.
     * |[17]    |IMMLDEN1  |Immediately Load Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PERIODn register will load to PBUFn register at the end point of each period.
     * |        |          |CMPDAT n register will load to CMPBUFn register at the end point or center point of each period by setting CTRLDn bit.
     * |        |          |1 = PERIODn/CMPDATn registers will load to PBUFn and CMPBUFn register immediately when software update PERIODn/CMPDATn register.
     * |        |          |Note: If IMMLDENn bit is enabled, WINLDENn bit and CTRLDn bits will be invalid.
     * |[18]    |IMMLDEN2  |Immediately Load Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PERIODn register will load to PBUFn register at the end point of each period.
     * |        |          |CMPDAT n register will load to CMPBUFn register at the end point or center point of each period by setting CTRLDn bit.
     * |        |          |1 = PERIODn/CMPDATn registers will load to PBUFn and CMPBUFn register immediately when software update PERIODn/CMPDATn register.
     * |        |          |Note: If IMMLDENn bit is enabled, WINLDENn bit and CTRLDn bits will be invalid.
     * |[19]    |IMMLDEN3  |Immediately Load Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PERIODn register will load to PBUFn register at the end point of each period.
     * |        |          |CMPDAT n register will load to CMPBUFn register at the end point or center point of each period by setting CTRLDn bit.
     * |        |          |1 = PERIODn/CMPDATn registers will load to PBUFn and CMPBUFn register immediately when software update PERIODn/CMPDATn register.
     * |        |          |Note: If IMMLDENn bit is enabled, WINLDENn bit and CTRLDn bits will be invalid.
     * |[20]    |IMMLDEN4  |Immediately Load Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PERIODn register will load to PBUFn register at the end point of each period.
     * |        |          |CMPDAT n register will load to CMPBUFn register at the end point or center point of each period by setting CTRLDn bit.
     * |        |          |1 = PERIODn/CMPDATn registers will load to PBUFn and CMPBUFn register immediately when software update PERIODn/CMPDATn register.
     * |        |          |Note: If IMMLDENn bit is enabled, WINLDENn bit and CTRLDn bits will be invalid.
     * |[21]    |IMMLDEN5  |Immediately Load Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PERIODn register will load to PBUFn register at the end point of each period.
     * |        |          |CMPDAT n register will load to CMPBUFn register at the end point or center point of each period by setting CTRLDn bit.
     * |        |          |1 = PERIODn/CMPDATn registers will load to PBUFn and CMPBUFn register immediately when software update PERIODn/CMPDATn register.
     * |        |          |Note: If IMMLDENn bit is enabled, WINLDENn bit and CTRLDn bits will be invalid.
     * |[24]    |GROUPEN   |Group Function Enable Bit
     * |        |          |0 = The output waveform of each PWM channel are independent.
     * |        |          |1 = Unify the PWMx_CH2 and PWMx_CH4 to output the same waveform as PWMx_CH0 and unify the PWMx_CH3 and PWMx_CH5 to output the same waveform as PWMx_CH1.
     * |[30]    |DBGHALT   |ICE Debug Mode Counter Halt (Write Protect)
     * |        |          |If counter halt is enabled, PWM all counters will keep current value until exit ICE debug mode.
     * |        |          |0 = ICE debug mode counter halt disable.
     * |        |          |1 = ICE debug mode counter halt enable.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[31]    |DBGTRIOFF |ICE Debug Mode Acknowledge Disable (Write Protect)
     * |        |          |0 = ICE debug mode acknowledgement effects PWM output.
     * |        |          |PWM pin will be forced as tri-state while ICE debug mode acknowledged.
     * |        |          |1 = ICE debug mode acknowledgement disabled.
     * |        |          |PWM pin will keep output no matter ICE debug mode acknowledged or not.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * @var PWM_T::CTL1
     * Offset: 0x04  PWM Control Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |CNTTYPE0  |PWM Counter Behavior Type
     * |        |          |Each bit n controls corresponding PWM channel n.
     * |        |          |00 = Up counter type (supports in capture mode).
     * |        |          |01 = Down count type (supports in capture mode).
     * |        |          |10 = Up-down counter type.
     * |        |          |11 = Reserved.
     * |[3:2]   |CNTTYPE1  |PWM Counter Behavior Type
     * |        |          |Each bit n controls corresponding PWM channel n.
     * |        |          |00 = Up counter type (supports in capture mode).
     * |        |          |01 = Down count type (supports in capture mode).
     * |        |          |10 = Up-down counter type.
     * |        |          |11 = Reserved.
     * |[5:4]   |CNTTYPE2  |PWM Counter Behavior Type
     * |        |          |Each bit n controls corresponding PWM channel n.
     * |        |          |00 = Up counter type (supports in capture mode).
     * |        |          |01 = Down count type (supports in capture mode).
     * |        |          |10 = Up-down counter type.
     * |        |          |11 = Reserved.
     * |[7:6]   |CNTTYPE3  |PWM Counter Behavior Type
     * |        |          |Each bit n controls corresponding PWM channel n.
     * |        |          |00 = Up counter type (supports in capture mode).
     * |        |          |01 = Down count type (supports in capture mode).
     * |        |          |10 = Up-down counter type.
     * |        |          |11 = Reserved.
     * |[9:8]   |CNTTYPE4  |PWM Counter Behavior Type
     * |        |          |Each bit n controls corresponding PWM channel n.
     * |        |          |00 = Up counter type (supports in capture mode).
     * |        |          |01 = Down count type (supports in capture mode).
     * |        |          |10 = Up-down counter type.
     * |        |          |11 = Reserved.
     * |[11:10] |CNTTYPE5  |PWM Counter Behavior Type
     * |        |          |Each bit n controls corresponding PWM channel n.
     * |        |          |00 = Up counter type (supports in capture mode).
     * |        |          |01 = Down count type (supports in capture mode).
     * |        |          |10 = Up-down counter type.
     * |        |          |11 = Reserved.
     * |[16]    |CNTMODE0  |PWM Counter Mode
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Auto-reload mode.
     * |        |          |1 = One-shot mode.
     * |[17]    |CNTMODE1  |PWM Counter Mode
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Auto-reload mode.
     * |        |          |1 = One-shot mode.
     * |[18]    |CNTMODE2  |PWM Counter Mode
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Auto-reload mode.
     * |        |          |1 = One-shot mode.
     * |[19]    |CNTMODE3  |PWM Counter Mode
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Auto-reload mode.
     * |        |          |1 = One-shot mode.
     * |[20]    |CNTMODE4  |PWM Counter Mode
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Auto-reload mode.
     * |        |          |1 = One-shot mode.
     * |[21]    |CNTMODE5  |PWM Counter Mode
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Auto-reload mode.
     * |        |          |1 = One-shot mode.
     * |[24]    |OUTMODE0  |PWM Output Mode
     * |        |          |Each bit n controls the output mode of corresponding PWM channel n.
     * |        |          |0 = PWM independent mode.
     * |        |          |1 = PWM complementary mode.
     * |        |          |Note: When operating in group function, these bits must all set to the same mode.
     * |[25]    |OUTMODE2  |PWM Output Mode
     * |        |          |Each bit n controls the output mode of corresponding PWM channel n.
     * |        |          |0 = PWM independent mode.
     * |        |          |1 = PWM complementary mode.
     * |        |          |Note: When operating in group function, these bits must all set to the same mode.
     * |[26]    |OUTMODE4  |PWM Output Mode
     * |        |          |Each bit n controls the output mode of corresponding PWM channel n.
     * |        |          |0 = PWM independent mode.
     * |        |          |1 = PWM complementary mode.
     * |        |          |Note: When operating in group function, these bits must all set to the same mode.
     * @var PWM_T::SYNC
     * Offset: 0x08  PWM Synchronization Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |PHSEN0    |SYNC Phase Enable Bits
     * |        |          |Each bit n denotes controls corresponding PWM channel 0,2,4 and m denotes channel 1,3,5.
     * |        |          |0 = PWM counter disable to load value of PHS(PWM_PHSn_m[15:0]) bits.
     * |        |          |1 = PWM counter enable to load value of PHS(PWM_PHSn_m[15:0]) bits.
     * |[1]     |PHSEN2    |SYNC Phase Enable Bits
     * |        |          |Each bit n denotes controls corresponding PWM channel 0,2,4 and m denotes channel 1,3,5.
     * |        |          |0 = PWM counter disable to load value of PHS(PWM_PHSn_m[15:0]) bits.
     * |        |          |1 = PWM counter enable to load value of PHS(PWM_PHSn_m[15:0]) bits.
     * |[2]     |PHSEN4    |SYNC Phase Enable Bits
     * |        |          |Each bit n denotes controls corresponding PWM channel 0,2,4 and m denotes channel 1,3,5.
     * |        |          |0 = PWM counter disable to load value of PHS(PWM_PHSn_m[15:0]) bits.
     * |        |          |1 = PWM counter enable to load value of PHS(PWM_PHSn_m[15:0]) bits.
     * |[9:8]   |SINSRC0   |PWM0_SYNC_IN Source Selection
     * |        |          |Each bit n controls corresponding PWM channel n.
     * |        |          |00 = Synchronize source from SYNC_IN or SWSYNC.
     * |        |          |01 = Counter equal to 0.
     * |        |          |10 = Counter equal to PWM_CMPDATm, m denotes 1, 3, 5.
     * |        |          |11 = SYNC_OUT signal will not be generated.
     * |[11:10] |SINSRC2   |PWM0_SYNC_IN Source Selection
     * |        |          |Each bit n controls corresponding PWM channel n.
     * |        |          |00 = Synchronize source from SYNC_IN or SWSYNC.
     * |        |          |01 = Counter equal to 0.
     * |        |          |10 = Counter equal to PWM_CMPDATm, m denotes 1, 3, 5.
     * |        |          |11 = SYNC_OUT signal will not be generated.
     * |[13:12] |SINSRC4   |PWM0_SYNC_IN Source Selection
     * |        |          |Each bit n controls corresponding PWM channel n.
     * |        |          |00 = Synchronize source from SYNC_IN or SWSYNC.
     * |        |          |01 = Counter equal to 0.
     * |        |          |10 = Counter equal to PWM_CMPDATm, m denotes 1, 3, 5.
     * |        |          |11 = SYNC_OUT signal will not be generated.
     * |[16]    |SNFLTEN   |PWM0_SYNC_IN Noise Filter Enable Bit
     * |        |          |0 = Noise filter of input PWM0_SYNC_IN pin is Disabled.
     * |        |          |1 = Noise filter of input PWM0_SYNC_IN pin is Enabled.
     * |[19:17] |SFLTCSEL  |SYNC Edge Detector Filter Clock Selection
     * |        |          |000 = Filter clock = HCLK.
     * |        |          |001 = Filter clock = HCLK/2.
     * |        |          |010 = Filter clock = HCLK/4.
     * |        |          |011 = Filter clock = HCLK/8.
     * |        |          |100 = Filter clock = HCLK/16.
     * |        |          |101 = Filter clock = HCLK/32.
     * |        |          |110 = Filter clock = HCLK/64.
     * |        |          |111 = Filter clock = HCLK/128.
     * |[22:20] |SFLTCNT   |SYNC Edge Detector Filter Count
     * |        |          |The register bits control the counter number of edge detector.
     * |[23]    |SINPINV   |SYNC Input Pin Inverse
     * |        |          |0 = The state of PWM0_SYNC_IN pin is passed to the negative edge detector.
     * |        |          |1 = The inversed state of PWM0_SYNC_IN pin is passed to the negative edge detector.
     * |[24]    |PHSDIR0   |PWM Phase Direction Control
     * |        |          |Each bit n controls corresponding PWM channel n.
     * |        |          |0 = Control PWM counter count decrement after synchronizing.
     * |        |          |1 = Control PWM counter count increment after synchronizing.
     * |[25]    |PHSDIR2   |PWM Phase Direction Control
     * |        |          |Each bit n controls corresponding PWM channel n.
     * |        |          |0 = Control PWM counter count decrement after synchronizing.
     * |        |          |1 = Control PWM counter count increment after synchronizing.
     * |[26]    |PHSDIR4   |PWM Phase Direction Control
     * |        |          |Each bit n controls corresponding PWM channel n.
     * |        |          |0 = Control PWM counter count decrement after synchronizing.
     * |        |          |1 = Control PWM counter count increment after synchronizing.
     * @var PWM_T::SWSYNC
     * Offset: 0x0C  PWM Software Control Synchronization Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SWSYNC0   |Software SYNC Function
     * |        |          |Each bit n controls corresponding PWM channel n.
     * |        |          |When SINSRCn (PWM_SYNC[13:8]) is selected to 0, SYNC_OUT source is come from SYNC_IN or this bit.
     * |[1]     |SWSYNC2   |Software SYNC Function
     * |        |          |Each bit n controls corresponding PWM channel n.
     * |        |          |When SINSRCn (PWM_SYNC[13:8]) is selected to 0, SYNC_OUT source is come from SYNC_IN or this bit.
     * |[2]     |SWSYNC4   |Software SYNC Function
     * |        |          |Each bit n controls corresponding PWM channel n.
     * |        |          |When SINSRCn (PWM_SYNC[13:8]) is selected to 0, SYNC_OUT source is come from SYNC_IN or this bit.
     * @var PWM_T::CLKSRC
     * Offset: 0x10  PWM Clock Source Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |ECLKSRC0  |PWMx_CH0/1 External Clock Source Select
     * |        |          |000 = PWMx_CLK, x denotes 0 or 1.
     * |        |          |001 = TIMER0 time-out event.
     * |        |          |010 = TIMER1 time-out event.
     * |        |          |011 = TIMER2 time-out event.
     * |        |          |100 = TIMER3 time-out event.
     * |        |          |Others = Reserved.
     * |[10:8]  |ECLKSRC2  |PWMx_CH2/3 External Clock Source Select
     * |        |          |000 = PWMx_CLK, x denotes 0 or 1.
     * |        |          |001 = TIMER0 time-out event.
     * |        |          |010 = TIMER1 time-out event.
     * |        |          |011 = TIMER2 time-out event.
     * |        |          |100 = TIMER3 time-out event.
     * |        |          |Others = Reserved.
     * |[18:16] |ECLKSRC4  |PWMx_CH4/5 External Clock Source Select
     * |        |          |000 = PWMx_CLK, x denotes 0 or 1.
     * |        |          |001 = TIMER0 time-out event.
     * |        |          |010 = TIMER1 time-out event.
     * |        |          |011 = TIMER2 time-out event.
     * |        |          |100 = TIMER3 time-out event.
     * |        |          |Others = Reserved.
     * @var PWM_T::CLKPSC0_1
     * Offset: 0x14  PWM Clock Pre-scale Register 0/1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |CLKPSC    |PWM Counter Clock Pre-scale
     * |        |          |The clock of PWM counter is decided by clock prescaler.
     * |        |          |Each PWM pair share one PWM counter clock prescaler.
     * |        |          |The clock of PWM counter is divided by (CLKPSC+ 1).
     * @var PWM_T::CLKPSC2_3
     * Offset: 0x18  PWM Clock Pre-scale Register 2/3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |CLKPSC    |PWM Counter Clock Pre-scale
     * |        |          |The clock of PWM counter is decided by clock prescaler.
     * |        |          |Each PWM pair share one PWM counter clock prescaler.
     * |        |          |The clock of PWM counter is divided by (CLKPSC+ 1).
     * @var PWM_T::CLKPSC4_5
     * Offset: 0x1C  PWM Clock Pre-scale Register 4/5
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |CLKPSC    |PWM Counter Clock Pre-scale
     * |        |          |The clock of PWM counter is decided by clock prescaler.
     * |        |          |Each PWM pair share one PWM counter clock prescaler.
     * |        |          |The clock of PWM counter is divided by (CLKPSC+ 1).
     * @var PWM_T::CNTEN
     * Offset: 0x20  PWM Counter Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CNTEN0    |PWM Counter Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM Counter and clock prescaler Stop Running.
     * |        |          |1 = PWM Counter and clock prescaler Start Running.
     * |[1]     |CNTEN1    |PWM Counter Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM Counter and clock prescaler Stop Running.
     * |        |          |1 = PWM Counter and clock prescaler Start Running.
     * |[2]     |CNTEN2    |PWM Counter Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM Counter and clock prescaler Stop Running.
     * |        |          |1 = PWM Counter and clock prescaler Start Running.
     * |[3]     |CNTEN3    |PWM Counter Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM Counter and clock prescaler Stop Running.
     * |        |          |1 = PWM Counter and clock prescaler Start Running.
     * |[4]     |CNTEN4    |PWM Counter Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM Counter and clock prescaler Stop Running.
     * |        |          |1 = PWM Counter and clock prescaler Start Running.
     * |[5]     |CNTEN5    |PWM Counter Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM Counter and clock prescaler Stop Running.
     * |        |          |1 = PWM Counter and clock prescaler Start Running.
     * @var PWM_T::CNTCLR
     * Offset: 0x24  PWM Clear Counter Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CNTCLR0   |Clear PWM Counter Control Bit
     * |        |          |It is automatically cleared by hardware. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No effect.
     * |        |          |1 = Clear 16-bit PWM counter to 0x0000(CNT(PWM_CNTn[15:0])).
     * |[1]     |CNTCLR1   |Clear PWM Counter Control Bit
     * |        |          |It is automatically cleared by hardware. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No effect.
     * |        |          |1 = Clear 16-bit PWM counter to 0x0000(CNT(PWM_CNTn[15:0])).
     * |[2]     |CNTCLR2   |Clear PWM Counter Control Bit
     * |        |          |It is automatically cleared by hardware. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No effect.
     * |        |          |1 = Clear 16-bit PWM counter to 0x0000(CNT(PWM_CNTn[15:0])).
     * |[3]     |CNTCLR3   |Clear PWM Counter Control Bit
     * |        |          |It is automatically cleared by hardware. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No effect.
     * |        |          |1 = Clear 16-bit PWM counter to 0x0000(CNT(PWM_CNTn[15:0])).
     * |[4]     |CNTCLR4   |Clear PWM Counter Control Bit
     * |        |          |It is automatically cleared by hardware. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No effect.
     * |        |          |1 = Clear 16-bit PWM counter to 0x0000(CNT(PWM_CNTn[15:0])).
     * |[5]     |CNTCLR5   |Clear PWM Counter Control Bit
     * |        |          |It is automatically cleared by hardware. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No effect.
     * |        |          |1 = Clear 16-bit PWM counter to 0x0000(CNT(PWM_CNTn[15:0])).
     * @var PWM_T::LOAD
     * Offset: 0x28  PWM Load Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |LOAD0     |Re-load PWM Comparator Register (CMPDAT) Control Bit
     * |        |          |This bit is software write, and hardware clear when current PWM period end
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Write Operation:
     * |        |          |0 = No effect.
     * |        |          |1 = Set load window of window loading mode.
     * |        |          |Read Operation:
     * |        |          |0 = No load window is set.
     * |        |          |1 = Load window is set.
     * |        |          |Note: This bit only use in window loading mode, WINLDENn(PWM_CTL0[13:8]) = 1.
     * |[1]     |LOAD1     |Re-load PWM Comparator Register (CMPDAT) Control Bit
     * |        |          |This bit is software write, and hardware clear when current PWM period end
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Write Operation:
     * |        |          |0 = No effect.
     * |        |          |1 = Set load window of window loading mode.
     * |        |          |Read Operation:
     * |        |          |0 = No load window is set.
     * |        |          |1 = Load window is set.
     * |        |          |Note: This bit only use in window loading mode, WINLDENn(PWM_CTL0[13:8]) = 1.
     * |[2]     |LOAD2     |Re-load PWM Comparator Register (CMPDAT) Control Bit
     * |        |          |This bit is software write, and hardware clear when current PWM period end
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Write Operation:
     * |        |          |0 = No effect.
     * |        |          |1 = Set load window of window loading mode.
     * |        |          |Read Operation:
     * |        |          |0 = No load window is set.
     * |        |          |1 = Load window is set.
     * |        |          |Note: This bit only use in window loading mode, WINLDENn(PWM_CTL0[13:8]) = 1.
     * |[3]     |LOAD3     |Re-load PWM Comparator Register (CMPDAT) Control Bit
     * |        |          |This bit is software write, and hardware clear when current PWM period end
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Write Operation:
     * |        |          |0 = No effect.
     * |        |          |1 = Set load window of window loading mode.
     * |        |          |Read Operation:
     * |        |          |0 = No load window is set.
     * |        |          |1 = Load window is set.
     * |        |          |Note: This bit only use in window loading mode, WINLDENn(PWM_CTL0[13:8]) = 1.
     * |[4]     |LOAD4     |Re-load PWM Comparator Register (CMPDAT) Control Bit
     * |        |          |This bit is software write, and hardware clear when current PWM period end
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Write Operation:
     * |        |          |0 = No effect.
     * |        |          |1 = Set load window of window loading mode.
     * |        |          |Read Operation:
     * |        |          |0 = No load window is set.
     * |        |          |1 = Load window is set.
     * |        |          |Note: This bit only use in window loading mode, WINLDENn(PWM_CTL0[13:8]) = 1.
     * |[5]     |LOAD5     |Re-load PWM Comparator Register (CMPDAT) Control Bit
     * |        |          |This bit is software write, and hardware clear when current PWM period end
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Write Operation:
     * |        |          |0 = No effect.
     * |        |          |1 = Set load window of window loading mode.
     * |        |          |Read Operation:
     * |        |          |0 = No load window is set.
     * |        |          |1 = Load window is set.
     * |        |          |Note: This bit only use in window loading mode, WINLDENn(PWM_CTL0[13:8]) = 1.
     * @var PWM_T::PERIOD
     * Offset: 0x30/0x34/0x38/0x3C/0x40/0x44  PWM Period Register 0~5
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |PERIOD    |PWM Period Register
     * |        |          |Up-Count mode:
     * |        |          |In this mode, PWM counter counts from 0 to PERIOD, and restarts from 0.
     * |        |          |PWM period time = (PERIOD+1) * (CLKPSC+1) * PWMx_CLK.
     * |        |          |Down-Count mode:
     * |        |          |In this mode, PWM counter counts from PERIOD to 0, and restarts from PERIOD.
     * |        |          |PWM period time = (PERIOD+1) * (CLKPSC+1) * PWMx_CLK.
     * |        |          |Up-Down-Count mode:
     * |        |          |In this mode, PWM counter counts from 0 to PERIOD, then decrements to 0 and repeats again.
     * |        |          |PWM period time = (2 * PERIOD) * (CLKPSC+1) * PWMx_CLK.
     * @var PWM_T::CMPDAT
     * Offset: 0x50/0x54/0x58/0x5C/0x60/0x64  PWM Comparator Register 0~5
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CMP       |PWM Comparator Register
     * |        |          |CMP bits use to compare with CNTR(PWM_CNTn[15:0]) bits to generate PWM waveform, interrupt and trigger ADC.
     * |        |          |In independent mode, CMPDAT0~5 registers denote as 6 independent PWMx_CH0~5 compared point.
     * |        |          |In complementary mode, CMPDAT0, 2, 4 registers denote as first compared point, and CMPDAT1, 3, 5 register denote as second compared point for the corresponding 3 complementary pairs PWMx_CH0 and PWMx_CH1, PWMx_CH2 and PWMx_CH3, PWMx_CH4 and PWMx_CH5.
     * @var PWM_T::DTCTL0_1
     * Offset: 0x70  PWM Dead-Time Control Register 0/1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |DTCNT     |Dead-time Counter (Write Protect)
     * |        |          |The dead-time can be calculated from the following formula:
     * |        |          |Dead-time = (DTCNT+1) * PWMx_CLK period, if DTCKSEL bit is 0.
     * |        |          |Dead-time = (DTCNT+1) * (CLKPSC (PWM_CLKPSCn [11:0])+1)*PWMx_CLK period, if DTCKSEL bit is 1.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[16]    |DTEN      |Enable Dead-time Insertion for PWM Pair (PWMx_CH0, PWMx_CH1) (PWMx_CH2, PWMx_CH3) (PWMx_CH4, PWMx_CH5) (Write Protect)
     * |        |          |Dead-time insertion is only active when this pair of complementary PWM is enabled.
     * |        |          |If dead- time insertion is inactive, the outputs of pin pair are complementary without any delay.
     * |        |          |0 = Dead-time insertion Disabled on the pin pair.
     * |        |          |1 = Dead-time insertion Enabled on the pin pair.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[24]    |DTCKSEL   |Dead-time Clock Select (Write Protect)
     * |        |          |0 = Dead-time clock source from PWMx_CLK without counter clock prescale.
     * |        |          |1 = Dead-time clock source from prescaler output with counter clock prescale.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * @var PWM_T::DTCTL2_3
     * Offset: 0x74  PWM Dead-Time Control Register 2/3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |DTCNT     |Dead-time Counter (Write Protect)
     * |        |          |The dead-time can be calculated from the following formula:
     * |        |          |Dead-time = (DTCNT+1) * PWMx_CLK period, if DTCKSEL bit is 0.
     * |        |          |Dead-time = (DTCNT+1) * (CLKPSC (PWM_CLKPSCn [11:0])+1)*PWMx_CLK period, if DTCKSEL bit is 1.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[16]    |DTEN      |Enable Dead-time Insertion for PWM Pair (PWMx_CH0, PWMx_CH1) (PWMx_CH2, PWMx_CH3) (PWMx_CH4, PWMx_CH5) (Write Protect)
     * |        |          |Dead-time insertion is only active when this pair of complementary PWM is enabled.
     * |        |          |If dead- time insertion is inactive, the outputs of pin pair are complementary without any delay.
     * |        |          |0 = Dead-time insertion Disabled on the pin pair.
     * |        |          |1 = Dead-time insertion Enabled on the pin pair.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[24]    |DTCKSEL   |Dead-time Clock Select (Write Protect)
     * |        |          |0 = Dead-time clock source from PWMx_CLK without counter clock prescale.
     * |        |          |1 = Dead-time clock source from prescaler output with counter clock prescale.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * @var PWM_T::DTCTL4_5
     * Offset: 0x78  PWM Dead-Time Control Register 4/5
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |DTCNT     |Dead-time Counter (Write Protect)
     * |        |          |The dead-time can be calculated from the following formula:
     * |        |          |Dead-time = (DTCNT+1) * PWMx_CLK period, if DTCKSEL bit is 0.
     * |        |          |Dead-time = (DTCNT+1) * (CLKPSC (PWM_CLKPSCn [11:0])+1)*PWMx_CLK period, if DTCKSEL bit is 1.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[16]    |DTEN      |Enable Dead-time Insertion for PWM Pair (PWMx_CH0, PWMx_CH1) (PWMx_CH2, PWMx_CH3) (PWMx_CH4, PWMx_CH5) (Write Protect)
     * |        |          |Dead-time insertion is only active when this pair of complementary PWM is enabled.
     * |        |          |If dead- time insertion is inactive, the outputs of pin pair are complementary without any delay.
     * |        |          |0 = Dead-time insertion Disabled on the pin pair.
     * |        |          |1 = Dead-time insertion Enabled on the pin pair.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[24]    |DTCKSEL   |Dead-time Clock Select (Write Protect)
     * |        |          |0 = Dead-time clock source from PWMx_CLK without counter clock prescale.
     * |        |          |1 = Dead-time clock source from prescaler output with counter clock prescale.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * @var PWM_T::PHS0_1
     * Offset: 0x80  PWM Counter Phase Register 0/1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |PHS       |PWM Synchronous Start Phase Bits
     * |        |          |PHS bits determines the PWM synchronous start phase value. These bits only use in synchronous function.
     * @var PWM_T::PHS2_3
     * Offset: 0x84  PWM Counter Phase Register 2/3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |PHS       |PWM Synchronous Start Phase Bits
     * |        |          |PHS bits determines the PWM synchronous start phase value. These bits only use in synchronous function.
     * @var PWM_T::PHS4_5
     * Offset: 0x88  PWM Counter Phase Register 4/5
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |PHS       |PWM Synchronous Start Phase Bits
     * |        |          |PHS bits determines the PWM synchronous start phase value. These bits only use in synchronous function.
     * @var PWM_T::CNT
     * Offset: 0x90/0x94/0x98/0x9C/0xA0/0xA4  PWM Counter Register 0~5
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CNT       |PWM Counter Data Bits (Read Only)
     * |        |          |User can monitor CNT to know the current value in 16-bit period counter.
     * |[16]    |DIRF      |PWM Direction Indicator Flag (Read Only)
     * |        |          |0 = Counter is Down count.
     * |        |          |1 = Counter is UP count.
     * @var PWM_T::WGCTL0
     * Offset: 0xB0  PWM Generation Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |ZPCTL0    |PWM Zero Point Control
     * |        |          |PWM can control output level on zero point event. Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM zero point output Low.
     * |        |          |10 = PWM zero point output High.
     * |        |          |11 = PWM zero point output Toggle.
     * |[3:2]   |ZPCTL1    |PWM Zero Point Control
     * |        |          |PWM can control output level on zero point event. Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM zero point output Low.
     * |        |          |10 = PWM zero point output High.
     * |        |          |11 = PWM zero point output Toggle.
     * |[5:4]   |ZPCTL2    |PWM Zero Point Control
     * |        |          |PWM can control output level on zero point event. Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM zero point output Low.
     * |        |          |10 = PWM zero point output High.
     * |        |          |11 = PWM zero point output Toggle.
     * |[7:6]   |ZPCTL3    |PWM Zero Point Control
     * |        |          |PWM can control output level on zero point event. Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM zero point output Low.
     * |        |          |10 = PWM zero point output High.
     * |        |          |11 = PWM zero point output Toggle.
     * |[9:8]   |ZPCTL4    |PWM Zero Point Control
     * |        |          |PWM can control output level on zero point event. Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM zero point output Low.
     * |        |          |10 = PWM zero point output High.
     * |        |          |11 = PWM zero point output Toggle.
     * |[11:10] |ZPCTL5    |PWM Zero Point Control
     * |        |          |PWM can control output level on zero point event. Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM zero point output Low.
     * |        |          |10 = PWM zero point output High.
     * |        |          |11 = PWM zero point output Toggle.
     * |[17:16] |PRDPCTL0  |PWM Period (Center) Point Control
     * |        |          |PWM can control output level on period(center) point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM period (center) point output Low.
     * |        |          |10 = PWM period (center) point output High.
     * |        |          |11 = PWM period (center) point output Toggle.
     * |        |          |Note: This bit is center point control when PWM counter operating in up-down counter type.
     * |[19:18] |PRDPCTL1  |PWM Period (Center) Point Control
     * |        |          |PWM can control output level on period(center) point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM period (center) point output Low.
     * |        |          |10 = PWM period (center) point output High.
     * |        |          |11 = PWM period (center) point output Toggle.
     * |        |          |Note: This bit is center point control when PWM counter operating in up-down counter type.
     * |[21:20] |PRDPCTL2  |PWM Period (Center) Point Control
     * |        |          |PWM can control output level on period(center) point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM period (center) point output Low.
     * |        |          |10 = PWM period (center) point output High.
     * |        |          |11 = PWM period (center) point output Toggle.
     * |        |          |Note: This bit is center point control when PWM counter operating in up-down counter type.
     * |[23:22] |PRDPCTL3  |PWM Period (Center) Point Control
     * |        |          |PWM can control output level on period(center) point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM period (center) point output Low.
     * |        |          |10 = PWM period (center) point output High.
     * |        |          |11 = PWM period (center) point output Toggle.
     * |        |          |Note: This bit is center point control when PWM counter operating in up-down counter type.
     * |[25:24] |PRDPCTL4  |PWM Period (Center) Point Control
     * |        |          |PWM can control output level on period(center) point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM period (center) point output Low.
     * |        |          |10 = PWM period (center) point output High.
     * |        |          |11 = PWM period (center) point output Toggle.
     * |        |          |Note: This bit is center point control when PWM counter operating in up-down counter type.
     * |[27:26] |PRDPCTL5  |PWM Period (Center) Point Control
     * |        |          |PWM can control output level on period(center) point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM period (center) point output Low.
     * |        |          |10 = PWM period (center) point output High.
     * |        |          |11 = PWM period (center) point output Toggle.
     * |        |          |Note: This bit is center point control when PWM counter operating in up-down counter type.
     * @var PWM_T::WGCTL1
     * Offset: 0xB4  PWM Generation Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |CMPUCTL0  |PWM Compare Up Point Control
     * |        |          |PWM can control output level on compare up point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM compare up point output Low.
     * |        |          |10 = PWM compare up point output High.
     * |        |          |11 = PWM compare up point output Toggle.
     * |        |          |Note: In complementary mode, CMPUCTL1, 3, 5 use as another CMPUCTL for channel 0, 2, 4.
     * |[3:2]   |CMPUCTL1  |PWM Compare Up Point Control
     * |        |          |PWM can control output level on compare up point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM compare up point output Low.
     * |        |          |10 = PWM compare up point output High.
     * |        |          |11 = PWM compare up point output Toggle.
     * |        |          |Note: In complementary mode, CMPUCTL1, 3, 5 use as another CMPUCTL for channel 0, 2, 4.
     * |[5:4]   |CMPUCTL2  |PWM Compare Up Point Control
     * |        |          |PWM can control output level on compare up point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM compare up point output Low.
     * |        |          |10 = PWM compare up point output High.
     * |        |          |11 = PWM compare up point output Toggle.
     * |        |          |Note: In complementary mode, CMPUCTL1, 3, 5 use as another CMPUCTL for channel 0, 2, 4.
     * |[7:6]   |CMPUCTL3  |PWM Compare Up Point Control
     * |        |          |PWM can control output level on compare up point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM compare up point output Low.
     * |        |          |10 = PWM compare up point output High.
     * |        |          |11 = PWM compare up point output Toggle.
     * |        |          |Note: In complementary mode, CMPUCTL1, 3, 5 use as another CMPUCTL for channel 0, 2, 4.
     * |[9:8]   |CMPUCTL4  |PWM Compare Up Point Control
     * |        |          |PWM can control output level on compare up point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM compare up point output Low.
     * |        |          |10 = PWM compare up point output High.
     * |        |          |11 = PWM compare up point output Toggle.
     * |        |          |Note: In complementary mode, CMPUCTL1, 3, 5 use as another CMPUCTL for channel 0, 2, 4.
     * |[11:10] |CMPUCTL5  |PWM Compare Up Point Control
     * |        |          |PWM can control output level on compare up point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM compare up point output Low.
     * |        |          |10 = PWM compare up point output High.
     * |        |          |11 = PWM compare up point output Toggle.
     * |        |          |Note: In complementary mode, CMPUCTL1, 3, 5 use as another CMPUCTL for channel 0, 2, 4.
     * |[17:16] |CMPDCTL0  |PWM Compare Down Point Control
     * |        |          |PWM can control output level on compare down point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM compare down point output Low.
     * |        |          |10 = PWM compare down point output High.
     * |        |          |11 = PWM compare down point output Toggle.
     * |        |          |Note: In complementary mode, CMPDCTL1, 3, 5 use as another CMPDCTL for channel 0, 2, 4.
     * |[19:18] |CMPDCTL1  |PWM Compare Down Point Control
     * |        |          |PWM can control output level on compare down point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM compare down point output Low.
     * |        |          |10 = PWM compare down point output High.
     * |        |          |11 = PWM compare down point output Toggle.
     * |        |          |Note: In complementary mode, CMPDCTL1, 3, 5 use as another CMPDCTL for channel 0, 2, 4.
     * |[21:20] |CMPDCTL2  |PWM Compare Down Point Control
     * |        |          |PWM can control output level on compare down point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM compare down point output Low.
     * |        |          |10 = PWM compare down point output High.
     * |        |          |11 = PWM compare down point output Toggle.
     * |        |          |Note: In complementary mode, CMPDCTL1, 3, 5 use as another CMPDCTL for channel 0, 2, 4.
     * |[23:22] |CMPDCTL3  |PWM Compare Down Point Control
     * |        |          |PWM can control output level on compare down point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM compare down point output Low.
     * |        |          |10 = PWM compare down point output High.
     * |        |          |11 = PWM compare down point output Toggle.
     * |        |          |Note: In complementary mode, CMPDCTL1, 3, 5 use as another CMPDCTL for channel 0, 2, 4.
     * |[25:24] |CMPDCTL4  |PWM Compare Down Point Control
     * |        |          |PWM can control output level on compare down point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM compare down point output Low.
     * |        |          |10 = PWM compare down point output High.
     * |        |          |11 = PWM compare down point output Toggle.
     * |        |          |Note: In complementary mode, CMPDCTL1, 3, 5 use as another CMPDCTL for channel 0, 2, 4.
     * |[27:26] |CMPDCTL5  |PWM Compare Down Point Control
     * |        |          |PWM can control output level on compare down point event
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM compare down point output Low.
     * |        |          |10 = PWM compare down point output High.
     * |        |          |11 = PWM compare down point output Toggle.
     * |        |          |Note: In complementary mode, CMPDCTL1, 3, 5 use as another CMPDCTL for channel 0, 2, 4.
     * @var PWM_T::MSKEN
     * Offset: 0xB8  PWM Mask Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |MSKEN0    |PWM Mask Enable Bits
     * |        |          |The PWM output signal will be masked when this bit is enabled.
     * |        |          |The corresponding PWM channel n will output MSKDATn (PWM_MSK[5:0]) data.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM output signal is non-masked.
     * |        |          |1 = PWM output signal is masked and output MSKDATn data.
     * |[1]     |MSKEN1    |PWM Mask Enable Bits
     * |        |          |The PWM output signal will be masked when this bit is enabled.
     * |        |          |The corresponding PWM channel n will output MSKDATn (PWM_MSK[5:0]) data.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM output signal is non-masked.
     * |        |          |1 = PWM output signal is masked and output MSKDATn data.
     * |[2]     |MSKEN2    |PWM Mask Enable Bits
     * |        |          |The PWM output signal will be masked when this bit is enabled.
     * |        |          |The corresponding PWM channel n will output MSKDATn (PWM_MSK[5:0]) data.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM output signal is non-masked.
     * |        |          |1 = PWM output signal is masked and output MSKDATn data.
     * |[3]     |MSKEN3    |PWM Mask Enable Bits
     * |        |          |The PWM output signal will be masked when this bit is enabled.
     * |        |          |The corresponding PWM channel n will output MSKDATn (PWM_MSK[5:0]) data.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM output signal is non-masked.
     * |        |          |1 = PWM output signal is masked and output MSKDATn data.
     * |[4]     |MSKEN4    |PWM Mask Enable Bits
     * |        |          |The PWM output signal will be masked when this bit is enabled.
     * |        |          |The corresponding PWM channel n will output MSKDATn (PWM_MSK[5:0]) data.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM output signal is non-masked.
     * |        |          |1 = PWM output signal is masked and output MSKDATn data.
     * |[5]     |MSKEN5    |PWM Mask Enable Bits
     * |        |          |The PWM output signal will be masked when this bit is enabled.
     * |        |          |The corresponding PWM channel n will output MSKDATn (PWM_MSK[5:0]) data.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM output signal is non-masked.
     * |        |          |1 = PWM output signal is masked and output MSKDATn data.
     * @var PWM_T::MSK
     * Offset: 0xBC  PWM Mask Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |MSKDAT0   |PWM Mask Data Bit
     * |        |          |This data bit control the state of PWMx_CHn output pin, if corresponding mask function is enabled.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Output logic low to PWMx_CHn.
     * |        |          |1 = Output logic high to PWMx_CHn.
     * |[1]     |MSKDAT1   |PWM Mask Data Bit
     * |        |          |This data bit control the state of PWMx_CHn output pin, if corresponding mask function is enabled.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Output logic low to PWMx_CHn.
     * |        |          |1 = Output logic high to PWMx_CHn.
     * |[2]     |MSKDAT2   |PWM Mask Data Bit
     * |        |          |This data bit control the state of PWMx_CHn output pin, if corresponding mask function is enabled.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Output logic low to PWMx_CHn.
     * |        |          |1 = Output logic high to PWMx_CHn.
     * |[3]     |MSKDAT3   |PWM Mask Data Bit
     * |        |          |This data bit control the state of PWMx_CHn output pin, if corresponding mask function is enabled.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Output logic low to PWMx_CHn.
     * |        |          |1 = Output logic high to PWMx_CHn.
     * |[4]     |MSKDAT4   |PWM Mask Data Bit
     * |        |          |This data bit control the state of PWMx_CHn output pin, if corresponding mask function is enabled.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Output logic low to PWMx_CHn.
     * |        |          |1 = Output logic high to PWMx_CHn.
     * |[5]     |MSKDAT5   |PWM Mask Data Bit
     * |        |          |This data bit control the state of PWMx_CHn output pin, if corresponding mask function is enabled.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Output logic low to PWMx_CHn.
     * |        |          |1 = Output logic high to PWMx_CHn.
     * @var PWM_T::BNF
     * Offset: 0xC0  PWM Brake Noise Filter Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |BRK0NFEN  |PWM Brake 0 Noise Filter Enable Bit
     * |        |          |0 = Noise filter of PWM Brake 0 Disabled.
     * |        |          |1 = Noise filter of PWM Brake 0 Enabled.
     * |[3:1]   |BRK0NFSEL |Brake 0 Edge Detector Filter Clock Selection
     * |        |          |000 = Filter clock = HCLK.
     * |        |          |001 = Filter clock = HCLK/2.
     * |        |          |010 = Filter clock = HCLK/4.
     * |        |          |011 = Filter clock = HCLK/8.
     * |        |          |100 = Filter clock = HCLK/16.
     * |        |          |101 = Filter clock = HCLK/32.
     * |        |          |110 = Filter clock = HCLK/64.
     * |        |          |111 = Filter clock = HCLK/128.
     * |[6:4]   |BRK0FCNT  |Brake 0 Edge Detector Filter Count
     * |        |          |The register bits control the Brake0 filter counter to count from 0 to BRK0FCNT.
     * |[7]     |BRK0PINV  |Brake 0 Pin Inverse
     * |        |          |0 = Brake pin event will be detected if PWM0_BRAKEx pin status transfer from low to high in edge-detect, or pin status is high in level-detect.
     * |        |          |1 = Brake pin event will be detected if PWM0_BRAKEx pin status transfer from high to low in edge-detect, or pin status is low in level-detect.
     * |[8]     |BRK1NFEN  |PWM Brake 1 Noise Filter Enable Bit
     * |        |          |0 = Noise filter of PWM Brake 1 Disabled.
     * |        |          |1 = Noise filter of PWM Brake 1 Enabled.
     * |[11:9]  |BRK1NFSEL |Brake 1 Edge Detector Filter Clock Selection
     * |        |          |000 = Filter clock = HCLK.
     * |        |          |001 = Filter clock = HCLK/2.
     * |        |          |010 = Filter clock = HCLK/4.
     * |        |          |011 = Filter clock = HCLK/8.
     * |        |          |100 = Filter clock = HCLK/16.
     * |        |          |101 = Filter clock = HCLK/32.
     * |        |          |110 = Filter clock = HCLK/64.
     * |        |          |111 = Filter clock = HCLK/128.
     * |[14:12] |BRK1FCNT  |Brake 1 Edge Detector Filter Count
     * |        |          |The register bits control the Brake1 filter counter to count from 0 to BRK1FCNT.
     * |[15]    |BRK1PINV  |Brake 1 Pin Inverse
     * |        |          |0 = Brake pin event will be detected if PWM1_BRAKEx pin status transfer from low to high in edge-detect, or pin status is high in level-detect.
     * |        |          |1 = Brake pin event will be detected if PWM1_BRAKEx pin status transfer from high to low in edge-detect, or pin status is low in level-detect.
     * |[16]    |BK0SRC    |Brake 0 Pin Source Select
     * |        |          |For PWM0 setting:
     * |        |          |0 = Brake 0 pin source come from PWM0_BRAKE0.
     * |        |          |1 = Brake 0 pin source come from PWM1_BRAKE0.
     * |        |          |For PWM1 setting:
     * |        |          |0 = Brake 0 pin source come from PWM1_BRAKE0.
     * |        |          |1 = Brake 0 pin source come from PWM0_BRAKE0.
     * |[24]    |BK1SRC    |Brake 1 Pin Source Select
     * |        |          |For PWM0 setting:
     * |        |          |0 = Brake 1 pin source come from PWM0_BRAKE1.
     * |        |          |1 = Brake 1 pin source come from PWM1_BRAKE1.
     * |        |          |For PWM1 setting:
     * |        |          |0 = Brake 1 pin source come from PWM1_BRAKE1.
     * |        |          |1 = Brake 1 pin source come from PWM0_BRAKE1.
     * @var PWM_T::FAILBRK
     * Offset: 0xC4  PWM System Fail Brake Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CSSBRKEN  |Clock Security System Detection Trigger PWM Brake Function Enable Bit
     * |        |          |0 = Brake Function triggered by clock fail detection Disabled.
     * |        |          |1 = Brake Function triggered by clock fail detection Enabled.
     * |[1]     |BODBRKEN  |Brown-out Detection Trigger PWM Brake Function Enable Bit
     * |        |          |0 = Brake Function triggered by BOD event Disabled.
     * |        |          |1 = Brake Function triggered by BOD event Enabled.
     * |[3]     |CORBRKEN  |Core Lockup Detection Trigger PWM Brake Function Enable Bit
     * |        |          |0 = Brake Function triggered by Core lockup event Disabled.
     * |        |          |1 = Brake Function triggered by Core lockup event Enabled.
     * @var PWM_T::BRKCTL0_1
     * Offset: 0xC8  PWM Brake Edge Detect Control Register 0/1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CPO0EBEN  |Enable ACMP0_O Digital Output As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = ACMP0_O as edge-detect brake source Disabled.
     * |        |          |1 = ACMP0_O as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[1]     |CPO1EBEN  |Enable ACMP1_O Digital Output As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = ACMP1_O as edge-detect brake source Disabled.
     * |        |          |1 = ACMP1_O as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[4]     |BRKP0EEN  |Enable PWMx_BRAKE0 Pin As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE0 pin as edge-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE0 pin as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[5]     |BRKP1EEN  |Enable PWMx_BRAKE1 Pin As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE1 pin as edge-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE1 pin as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[7]     |SYSEBEN   |Enable System Fail As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = System Fail condition as edge-detect brake source Disabled.
     * |        |          |1 = System Fail condition as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[8]     |CPO0LBEN  |Enable ACMP0_O Digital Output As Level-detect Brake Source (Write Protect)
     * |        |          |0 = ACMP0_O as level-detect brake source Disabled.
     * |        |          |1 = ACMP0_O as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[9]     |CPO1LBEN  |Enable ACMP1_O Digital Output As Level-detect Brake Source (Write Protect)
     * |        |          |0 = ACMP1_O as level-detect brake source Disabled.
     * |        |          |1 = ACMP1_O as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[12]    |BRKP0LEN  |Enable BKP0 Pin As Level-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE0 pin as level-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE0 pin as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[13]    |BRKP1LEN  |Enable BKP1 Pin As Level-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE1 pin as level-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE1 pin as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[15]    |SYSLBEN   |Enable System Fail As Level-detect Brake Source (Write Protect)
     * |        |          |0 = System Fail condition as level-detect brake source Disabled.
     * |        |          |1 = System Fail condition as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[17:16] |BRKAEVEN  |PWM Brake Action Select for Even Channel (Write Protect)
     * |        |          |00 = PWMx brake event will not affect even channels output.
     * |        |          |01 = PWM even channel output tri-state when PWMx brake event happened.
     * |        |          |10 = PWM even channel output low level when PWMx brake event happened.
     * |        |          |11 = PWM even channel output high level when PWMx brake event happened.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[19:18] |BRKAODD   |PWM Brake Action Select for Odd Channel (Write Protect)
     * |        |          |00 = PWMx brake event will not affect odd channels output.
     * |        |          |01 = PWM odd channel output tri-state when PWMx brake event happened.
     * |        |          |10 = PWM odd channel output low level when PWMx brake event happened.
     * |        |          |11 = PWM odd channel output high level when PWMx brake event happened.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[20]    |ADCEBEN   |Enable ADC Result Monitor (ADCRM) As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = ADCRM as edge-detect brake source Disabled.
     * |        |          |1 = ADCRM as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[28]    |ADCLBEN   |Enable ADC Result Monitor (ADCRM) As Level-detect Brake Source (Write Protect)
     * |        |          |0 = ADCRM as level-detect brake source Disabled.
     * |        |          |1 = ADCRM as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * @var PWM_T::BRKCTL2_3
     * Offset: 0xCC  PWM Brake Edge Detect Control Register 2/3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CPO0EBEN  |Enable ACMP0_O Digital Output As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = ACMP0_O as edge-detect brake source Disabled.
     * |        |          |1 = ACMP0_O as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[1]     |CPO1EBEN  |Enable ACMP1_O Digital Output As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = ACMP1_O as edge-detect brake source Disabled.
     * |        |          |1 = ACMP1_O as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[4]     |BRKP0EEN  |Enable PWMx_BRAKE0 Pin As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE0 pin as edge-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE0 pin as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[5]     |BRKP1EEN  |Enable PWMx_BRAKE1 Pin As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE1 pin as edge-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE1 pin as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[7]     |SYSEBEN   |Enable System Fail As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = System Fail condition as edge-detect brake source Disabled.
     * |        |          |1 = System Fail condition as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[8]     |CPO0LBEN  |Enable ACMP0_O Digital Output As Level-detect Brake Source (Write Protect)
     * |        |          |0 = ACMP0_O as level-detect brake source Disabled.
     * |        |          |1 = ACMP0_O as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[9]     |CPO1LBEN  |Enable ACMP1_O Digital Output As Level-detect Brake Source (Write Protect)
     * |        |          |0 = ACMP1_O as level-detect brake source Disabled.
     * |        |          |1 = ACMP1_O as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[12]    |BRKP0LEN  |Enable BKP0 Pin As Level-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE0 pin as level-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE0 pin as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[13]    |BRKP1LEN  |Enable BKP1 Pin As Level-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE1 pin as level-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE1 pin as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[15]    |SYSLBEN   |Enable System Fail As Level-detect Brake Source (Write Protect)
     * |        |          |0 = System Fail condition as level-detect brake source Disabled.
     * |        |          |1 = System Fail condition as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[17:16] |BRKAEVEN  |PWM Brake Action Select for Even Channel (Write Protect)
     * |        |          |00 = PWMx brake event will not affect even channels output.
     * |        |          |01 = PWM even channel output tri-state when PWMx brake event happened.
     * |        |          |10 = PWM even channel output low level when PWMx brake event happened.
     * |        |          |11 = PWM even channel output high level when PWMx brake event happened.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[19:18] |BRKAODD   |PWM Brake Action Select for Odd Channel (Write Protect)
     * |        |          |00 = PWMx brake event will not affect odd channels output.
     * |        |          |01 = PWM odd channel output tri-state when PWMx brake event happened.
     * |        |          |10 = PWM odd channel output low level when PWMx brake event happened.
     * |        |          |11 = PWM odd channel output high level when PWMx brake event happened.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[20]    |ADCEBEN   |Enable ADC Result Monitor (ADCRM) As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = ADCRM as edge-detect brake source Disabled.
     * |        |          |1 = ADCRM as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[28]    |ADCLBEN   |Enable ADC Result Monitor (ADCRM) As Level-detect Brake Source (Write Protect)
     * |        |          |0 = ADCRM as level-detect brake source Disabled.
     * |        |          |1 = ADCRM as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * @var PWM_T::BRKCTL4_5
     * Offset: 0xD0  PWM Brake Edge Detect Control Register 4/5
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CPO0EBEN  |Enable ACMP0_O Digital Output As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = ACMP0_O as edge-detect brake source Disabled.
     * |        |          |1 = ACMP0_O as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[1]     |CPO1EBEN  |Enable ACMP1_O Digital Output As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = ACMP1_O as edge-detect brake source Disabled.
     * |        |          |1 = ACMP1_O as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[4]     |BRKP0EEN  |Enable PWMx_BRAKE0 Pin As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE0 pin as edge-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE0 pin as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[5]     |BRKP1EEN  |Enable PWMx_BRAKE1 Pin As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE1 pin as edge-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE1 pin as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[7]     |SYSEBEN   |Enable System Fail As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = System Fail condition as edge-detect brake source Disabled.
     * |        |          |1 = System Fail condition as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[8]     |CPO0LBEN  |Enable ACMP0_O Digital Output As Level-detect Brake Source (Write Protect)
     * |        |          |0 = ACMP0_O as level-detect brake source Disabled.
     * |        |          |1 = ACMP0_O as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[9]     |CPO1LBEN  |Enable ACMP1_O Digital Output As Level-detect Brake Source (Write Protect)
     * |        |          |0 = ACMP1_O as level-detect brake source Disabled.
     * |        |          |1 = ACMP1_O as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[12]    |BRKP0LEN  |Enable BKP0 Pin As Level-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE0 pin as level-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE0 pin as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[13]    |BRKP1LEN  |Enable BKP1 Pin As Level-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE1 pin as level-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE1 pin as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[15]    |SYSLBEN   |Enable System Fail As Level-detect Brake Source (Write Protect)
     * |        |          |0 = System Fail condition as level-detect brake source Disabled.
     * |        |          |1 = System Fail condition as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[17:16] |BRKAEVEN  |PWM Brake Action Select for Even Channel (Write Protect)
     * |        |          |00 = PWMx brake event will not affect even channels output.
     * |        |          |01 = PWM even channel output tri-state when PWMx brake event happened.
     * |        |          |10 = PWM even channel output low level when PWMx brake event happened.
     * |        |          |11 = PWM even channel output high level when PWMx brake event happened.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[19:18] |BRKAODD   |PWM Brake Action Select for Odd Channel (Write Protect)
     * |        |          |00 = PWMx brake event will not affect odd channels output.
     * |        |          |01 = PWM odd channel output tri-state when PWMx brake event happened.
     * |        |          |10 = PWM odd channel output low level when PWMx brake event happened.
     * |        |          |11 = PWM odd channel output high level when PWMx brake event happened.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[20]    |ADCEBEN   |Enable ADC Result Monitor (ADCRM) As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = ADCRM as edge-detect brake source Disabled.
     * |        |          |1 = ADCRM as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[28]    |ADCLBEN   |Enable ADC Result Monitor (ADCRM) As Level-detect Brake Source (Write Protect)
     * |        |          |0 = ADCRM as level-detect brake source Disabled.
     * |        |          |1 = ADCRM as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * @var PWM_T::POLCTL
     * Offset: 0xD4  PWM Pin Polar Inverse Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |PINV0     |PWM PIN Polar Inverse Control
     * |        |          |The register controls polarity state of PWMx_CHn output pin
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWMx_CHn output pin polar inverse Disabled.
     * |        |          |1 = PWMx_CHn output pin polar inverse Enabled.
     * |[1]     |PINV1     |PWM PIN Polar Inverse Control
     * |        |          |The register controls polarity state of PWMx_CHn output pin
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWMx_CHn output pin polar inverse Disabled.
     * |        |          |1 = PWMx_CHn output pin polar inverse Enabled.
     * |[2]     |PINV2     |PWM PIN Polar Inverse Control
     * |        |          |The register controls polarity state of PWMx_CHn output pin
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWMx_CHn output pin polar inverse Disabled.
     * |        |          |1 = PWMx_CHn output pin polar inverse Enabled.
     * |[3]     |PINV3     |PWM PIN Polar Inverse Control
     * |        |          |The register controls polarity state of PWMx_CHn output pin
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWMx_CHn output pin polar inverse Disabled.
     * |        |          |1 = PWMx_CHn output pin polar inverse Enabled.
     * |[4]     |PINV4     |PWM PIN Polar Inverse Control
     * |        |          |The register controls polarity state of PWMx_CHn output pin
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWMx_CHn output pin polar inverse Disabled.
     * |        |          |1 = PWMx_CHn output pin polar inverse Enabled.
     * |[5]     |PINV5     |PWM PIN Polar Inverse Control
     * |        |          |The register controls polarity state of PWMx_CHn output pin
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWMx_CHn output pin polar inverse Disabled.
     * |        |          |1 = PWMx_CHn output pin polar inverse Enabled.
     * @var PWM_T::POEN
     * Offset: 0xD8  PWM Output Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |POEN0     |PWMx_CHn Pin Output Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWMx_CHn pin at tri-state.
     * |        |          |1 = PWMx_CHn pin in output mode.
     * |[1]     |POEN1     |PWMx_CHn Pin Output Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWMx_CHn pin at tri-state.
     * |        |          |1 = PWMx_CHn pin in output mode.
     * |[2]     |POEN2     |PWMx_CHn Pin Output Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWMx_CHn pin at tri-state.
     * |        |          |1 = PWMx_CHn pin in output mode.
     * |[3]     |POEN3     |PWMx_CHn Pin Output Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWMx_CHn pin at tri-state.
     * |        |          |1 = PWMx_CHn pin in output mode.
     * |[4]     |POEN4     |PWMx_CHn Pin Output Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWMx_CHn pin at tri-state.
     * |        |          |1 = PWMx_CHn pin in output mode.
     * |[5]     |POEN5     |PWMx_CHn Pin Output Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWMx_CHn pin at tri-state.
     * |        |          |1 = PWMx_CHn pin in output mode.
     * @var PWM_T::SWBRK
     * Offset: 0xDC  PWM Software Brake Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |BRKETRG0  |PWM Edge Brake Software Trigger (Write Only) (Write Protect)
     * |        |          |Write 1 to this bit will trigger Edge brake, and set BRKEIFn bits to 1 in PWM_INTSTS1 register.
     * |        |          |Each bit n controls the corresponding PWM pair n.
     * |        |          |Note: This register is write protected. Refer toREGWRPROT register.
     * |[1]     |BRKETRG2  |PWM Edge Brake Software Trigger (Write Only) (Write Protect)
     * |        |          |Write 1 to this bit will trigger Edge brake, and set BRKEIFn bits to 1 in PWM_INTSTS1 register.
     * |        |          |Each bit n controls the corresponding PWM pair n.
     * |        |          |Note: This register is write protected. Refer toREGWRPROT register.
     * |[2]     |BRKETRG4  |PWM Edge Brake Software Trigger (Write Only) (Write Protect)
     * |        |          |Write 1 to this bit will trigger Edge brake, and set BRKEIFn bits to 1 in PWM_INTSTS1 register.
     * |        |          |Each bit n controls the corresponding PWM pair n.
     * |        |          |Note: This register is write protected. Refer toREGWRPROT register.
     * |[8]     |BRKLTRG0  |PWM Level Brake Software Trigger (Write Only) (Write Protect)
     * |        |          |Write 1 to this bit will trigger level brake, and set BRKLIFn bits to 1 in PWM_INTSTS1 register.
     * |        |          |Each bit n controls the corresponding PWM pair n.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[9]     |BRKLTRG2  |PWM Level Brake Software Trigger (Write Only) (Write Protect)
     * |        |          |Write 1 to this bit will trigger level brake, and set BRKLIFn bits to 1 in PWM_INTSTS1 register.
     * |        |          |Each bit n controls the corresponding PWM pair n.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[10]    |BRKLTRG4  |PWM Level Brake Software Trigger (Write Only) (Write Protect)
     * |        |          |Write 1 to this bit will trigger level brake, and set BRKLIFn bits to 1 in PWM_INTSTS1 register.
     * |        |          |Each bit n controls the corresponding PWM pair n.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * @var PWM_T::INTEN0
     * Offset: 0xE0  PWM Interrupt Enable Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ZIEN0     |PWM Zero Point Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Zero point interrupt Disabled.
     * |        |          |1 = Zero point interrupt Enabled.
     * |        |          |Note: Odd channels will read always 0 at complementary mode.
     * |[1]     |ZIEN1     |PWM Zero Point Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Zero point interrupt Disabled.
     * |        |          |1 = Zero point interrupt Enabled.
     * |        |          |Note: Odd channels will read always 0 at complementary mode.
     * |[2]     |ZIEN2     |PWM Zero Point Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Zero point interrupt Disabled.
     * |        |          |1 = Zero point interrupt Enabled.
     * |        |          |Note: Odd channels will read always 0 at complementary mode.
     * |[3]     |ZIEN3     |PWM Zero Point Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Zero point interrupt Disabled.
     * |        |          |1 = Zero point interrupt Enabled.
     * |        |          |Note: Odd channels will read always 0 at complementary mode.
     * |[4]     |ZIEN4     |PWM Zero Point Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Zero point interrupt Disabled.
     * |        |          |1 = Zero point interrupt Enabled.
     * |        |          |Note: Odd channels will read always 0 at complementary mode.
     * |[5]     |ZIEN5     |PWM Zero Point Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Zero point interrupt Disabled.
     * |        |          |1 = Zero point interrupt Enabled.
     * |        |          |Note: Odd channels will read always 0 at complementary mode.
     * |[7]     |IFAIEN0_1 |PWM Channel 0/1 Interrupt Flag Accumulator Interrupt Enable Bit
     * |        |          |0 = Interrupt Flag accumulator interrupt Disabled.
     * |        |          |1 = Interrupt Flag accumulator interrupt Enabled.
     * |[8]     |PIEN0     |PWM Period Point Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Period point interrupt Disabled.
     * |        |          |1 = Period point interrupt Enabled.
     * |        |          |Note1: When up-down counter type period point means center point.
     * |        |          |Note2: Odd channels will read always 0 at complementary mode.
     * |[9]     |PIEN1     |PWM Period Point Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Period point interrupt Disabled.
     * |        |          |1 = Period point interrupt Enabled.
     * |        |          |Note1: When up-down counter type period point means center point.
     * |        |          |Note2: Odd channels will read always 0 at complementary mode.
     * |[10]    |PIEN2     |PWM Period Point Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Period point interrupt Disabled.
     * |        |          |1 = Period point interrupt Enabled.
     * |        |          |Note1: When up-down counter type period point means center point.
     * |        |          |Note2: Odd channels will read always 0 at complementary mode.
     * |[11]    |PIEN3     |PWM Period Point Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Period point interrupt Disabled.
     * |        |          |1 = Period point interrupt Enabled.
     * |        |          |Note1: When up-down counter type period point means center point.
     * |        |          |Note2: Odd channels will read always 0 at complementary mode.
     * |[12]    |PIEN4     |PWM Period Point Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Period point interrupt Disabled.
     * |        |          |1 = Period point interrupt Enabled.
     * |        |          |Note1: When up-down counter type period point means center point.
     * |        |          |Note2: Odd channels will read always 0 at complementary mode.
     * |[13]    |PIEN5     |PWM Period Point Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Period point interrupt Disabled.
     * |        |          |1 = Period point interrupt Enabled.
     * |        |          |Note1: When up-down counter type period point means center point.
     * |        |          |Note2: Odd channels will read always 0 at complementary mode.
     * |[15]    |IFAIEN2_3 |PWM Channel 2/3 Interrupt Flag Accumulator Interrupt Enable Bit
     * |        |          |0 = Interrupt Flag accumulator interrupt Disabled.
     * |        |          |1 = Interrupt Flag accumulator interrupt Enabled.
     * |[16]    |CMPUIEN0  |PWM Compare Up Count Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Compare up count interrupt Disabled.
     * |        |          |1 = Compare up count interrupt Enabled.
     * |        |          |Note: In complementary mode, CMPUIEN1, 3, 5 use as another CMPUIEN for channel 0, 2, 4.
     * |[17]    |CMPUIEN1  |PWM Compare Up Count Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Compare up count interrupt Disabled.
     * |        |          |1 = Compare up count interrupt Enabled.
     * |        |          |Note: In complementary mode, CMPUIEN1, 3, 5 use as another CMPUIEN for channel 0, 2, 4.
     * |[18]    |CMPUIEN2  |PWM Compare Up Count Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Compare up count interrupt Disabled.
     * |        |          |1 = Compare up count interrupt Enabled.
     * |        |          |Note: In complementary mode, CMPUIEN1, 3, 5 use as another CMPUIEN for channel 0, 2, 4.
     * |[19]    |CMPUIEN3  |PWM Compare Up Count Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Compare up count interrupt Disabled.
     * |        |          |1 = Compare up count interrupt Enabled.
     * |        |          |Note: In complementary mode, CMPUIEN1, 3, 5 use as another CMPUIEN for channel 0, 2, 4.
     * |[20]    |CMPUIEN4  |PWM Compare Up Count Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Compare up count interrupt Disabled.
     * |        |          |1 = Compare up count interrupt Enabled.
     * |        |          |Note: In complementary mode, CMPUIEN1, 3, 5 use as another CMPUIEN for channel 0, 2, 4.
     * |[21]    |CMPUIEN5  |PWM Compare Up Count Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Compare up count interrupt Disabled.
     * |        |          |1 = Compare up count interrupt Enabled.
     * |        |          |Note: In complementary mode, CMPUIEN1, 3, 5 use as another CMPUIEN for channel 0, 2, 4.
     * |[23]    |IFAIEN4_5 |PWM Channel 4/5 Interrupt Flag Accumulator Interrupt Enable Bit
     * |        |          |0 = Interrupt Flag accumulator interrupt Disabled.
     * |        |          |1 = Interrupt Flag accumulator interrupt Enabled.
     * |[24]    |CMPDIEN0  |PWM Compare Down Count Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Compare down count interrupt Disabled.
     * |        |          |1 = Compare down count interrupt Enabled.
     * |        |          |Note: In complementary mode, CMPDIEN1, 3, 5 use as another CMPDIEN for channel 0, 2, 4.
     * |[25]    |CMPDIEN1  |PWM Compare Down Count Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Compare down count interrupt Disabled.
     * |        |          |1 = Compare down count interrupt Enabled.
     * |        |          |Note: In complementary mode, CMPDIEN1, 3, 5 use as another CMPDIEN for channel 0, 2, 4.
     * |[26]    |CMPDIEN2  |PWM Compare Down Count Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Compare down count interrupt Disabled.
     * |        |          |1 = Compare down count interrupt Enabled.
     * |        |          |Note: In complementary mode, CMPDIEN1, 3, 5 use as another CMPDIEN for channel 0, 2, 4.
     * |[27]    |CMPDIEN3  |PWM Compare Down Count Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Compare down count interrupt Disabled.
     * |        |          |1 = Compare down count interrupt Enabled.
     * |        |          |Note: In complementary mode, CMPDIEN1, 3, 5 use as another CMPDIEN for channel 0, 2, 4.
     * |[28]    |CMPDIEN4  |PWM Compare Down Count Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Compare down count interrupt Disabled.
     * |        |          |1 = Compare down count interrupt Enabled.
     * |        |          |Note: In complementary mode, CMPDIEN1, 3, 5 use as another CMPDIEN for channel 0, 2, 4.
     * |[29]    |CMPDIEN5  |PWM Compare Down Count Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Compare down count interrupt Disabled.
     * |        |          |1 = Compare down count interrupt Enabled.
     * |        |          |Note: In complementary mode, CMPDIEN1, 3, 5 use as another CMPDIEN for channel 0, 2, 4.
     * @var PWM_T::INTEN1
     * Offset: 0xE4  PWM Interrupt Enable Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |BRKEIEN0_1|PWM Edge-detect Brake Interrupt Enable for Channel0/1 (Write Protect)
     * |        |          |0 = Edge-detect Brake interrupt for channel0/1 Disabled.
     * |        |          |1 = Edge-detect Brake interrupt for channel0/1 Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[1]     |BRKEIEN2_3|PWM Edge-detect Brake Interrupt Enable for Channel2/3 (Write Protect)
     * |        |          |0 = Edge-detect Brake interrupt for channel2/3 Disabled.
     * |        |          |1 = Edge-detect Brake interrupt for channel2/3 Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[2]     |BRKEIEN4_5|PWM Edge-detect Brake Interrupt Enable for Channel4/5 (Write Protect)
     * |        |          |0 = Edge-detect Brake interrupt for channel4/5 Disabled.
     * |        |          |1 = Edge-detect Brake interrupt for channel4/5 Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[8]     |BRKLIEN0_1|PWM Level-detect Brake Interrupt Enable for Channel0/1 (Write Protect)
     * |        |          |0 = Level-detect Brake interrupt for channel0/1 Disabled.
     * |        |          |1 = Level-detect Brake interrupt for channel0/1 Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[9]     |BRKLIEN2_3|PWM Level-detect Brake Interrupt Enable for Channel2/3 (Write Protect)
     * |        |          |0 = Level-detect Brake interrupt for channel2/3 Disabled.
     * |        |          |1 = Level-detect Brake interrupt for channel2/3 Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * |[10]    |BRKLIEN4_5|PWM Level-detect Brake Interrupt Enable for Channel4/5 (Write Protect)
     * |        |          |0 = Level-detect Brake interrupt for channel4/5 Disabled.
     * |        |          |1 = Level-detect Brake interrupt for channel4/5 Enabled.
     * |        |          |Note: This register is write protected. Refer toSYS_REGLCTL register.
     * @var PWM_T::INTSTS0
     * Offset: 0xE8  PWM Interrupt Flag Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ZIF0      |PWM Zero Point Interrupt Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |This bit is set by hardware when PWM counter reaches zero, software can write 1 to clear this bit to zero.
     * |[1]     |ZIF1      |PWM Zero Point Interrupt Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |This bit is set by hardware when PWM counter reaches zero, software can write 1 to clear this bit to zero.
     * |[2]     |ZIF2      |PWM Zero Point Interrupt Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |This bit is set by hardware when PWM counter reaches zero, software can write 1 to clear this bit to zero.
     * |[3]     |ZIF3      |PWM Zero Point Interrupt Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |This bit is set by hardware when PWM counter reaches zero, software can write 1 to clear this bit to zero.
     * |[4]     |ZIF4      |PWM Zero Point Interrupt Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |This bit is set by hardware when PWM counter reaches zero, software can write 1 to clear this bit to zero.
     * |[5]     |ZIF5      |PWM Zero Point Interrupt Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |This bit is set by hardware when PWM counter reaches zero, software can write 1 to clear this bit to zero.
     * |[7]     |IFAIF0_1  |PWM Channel 0/1 Interrupt Flag Accumulator Interrupt Flag
     * |        |          |Flag is set by hardware when condition match IFSEL0_1 bits in PWM_IFA register, software can clear this bit by writing 1 to it.
     * |[8]     |PIF0      |PWM Period Point Interrupt Flag
     * |        |          |This bit is set by hardware when PWM counter reaches PERIOD(PWM_PERIODn[15:0]), software can write 1 to clear this bit to zero.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |[9]     |PIF1      |PWM Period Point Interrupt Flag
     * |        |          |This bit is set by hardware when PWM counter reaches PERIOD(PWM_PERIODn[15:0]), software can write 1 to clear this bit to zero.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |[10]    |PIF2      |PWM Period Point Interrupt Flag
     * |        |          |This bit is set by hardware when PWM counter reaches PERIOD(PWM_PERIODn[15:0]), software can write 1 to clear this bit to zero.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |[11]    |PIF3      |PWM Period Point Interrupt Flag
     * |        |          |This bit is set by hardware when PWM counter reaches PERIOD(PWM_PERIODn[15:0]), software can write 1 to clear this bit to zero.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |[12]    |PIF4      |PWM Period Point Interrupt Flag
     * |        |          |This bit is set by hardware when PWM counter reaches PERIOD(PWM_PERIODn[15:0]), software can write 1 to clear this bit to zero.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |[13]    |PIF5      |PWM Period Point Interrupt Flag
     * |        |          |This bit is set by hardware when PWM counter reaches PERIOD(PWM_PERIODn[15:0]), software can write 1 to clear this bit to zero.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |[15]    |IFAIF2_3  |PWM Channel 2/3 Interrupt Flag Accumulator Interrupt Flag
     * |        |          |Flag is set by hardware when condition match IFSEL2_3 bits in PWM_IFA register, software can clear this bit by writing 1 to it.
     * |[16]    |CMPUIF0   |PWM Compare Up Count Interrupt Flag
     * |        |          |Flag is set by hardware when PWM counter up count and reaches CMP(PWM_CMPDATn[15:0]), software can clear this bit by writing 1 to it.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note1: If CMPDAT equal to PERIOD, this flag is not working in up counter type selection.
     * |        |          |Note2: In complementary mode, CMPUIF1, 3, 5 use as another CMPUIF for channel 0, 2, 4.
     * |[17]    |CMPUIF1   |PWM Compare Up Count Interrupt Flag
     * |        |          |Flag is set by hardware when PWM counter up count and reaches CMP(PWM_CMPDATn[15:0]), software can clear this bit by writing 1 to it.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note1: If CMPDAT equal to PERIOD, this flag is not working in up counter type selection.
     * |        |          |Note2: In complementary mode, CMPUIF1, 3, 5 use as another CMPUIF for channel 0, 2, 4.
     * |[18]    |CMPUIF2   |PWM Compare Up Count Interrupt Flag
     * |        |          |Flag is set by hardware when PWM counter up count and reaches CMP(PWM_CMPDATn[15:0]), software can clear this bit by writing 1 to it.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note1: If CMPDAT equal to PERIOD, this flag is not working in up counter type selection.
     * |        |          |Note2: In complementary mode, CMPUIF1, 3, 5 use as another CMPUIF for channel 0, 2, 4.
     * |[19]    |CMPUIF3   |PWM Compare Up Count Interrupt Flag
     * |        |          |Flag is set by hardware when PWM counter up count and reaches CMP(PWM_CMPDATn[15:0]), software can clear this bit by writing 1 to it.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note1: If CMPDAT equal to PERIOD, this flag is not working in up counter type selection.
     * |        |          |Note2: In complementary mode, CMPUIF1, 3, 5 use as another CMPUIF for channel 0, 2, 4.
     * |[20]    |CMPUIF4   |PWM Compare Up Count Interrupt Flag
     * |        |          |Flag is set by hardware when PWM counter up count and reaches CMP(PWM_CMPDATn[15:0]), software can clear this bit by writing 1 to it.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note1: If CMPDAT equal to PERIOD, this flag is not working in up counter type selection.
     * |        |          |Note2: In complementary mode, CMPUIF1, 3, 5 use as another CMPUIF for channel 0, 2, 4.
     * |[21]    |CMPUIF5   |PWM Compare Up Count Interrupt Flag
     * |        |          |Flag is set by hardware when PWM counter up count and reaches CMP(PWM_CMPDATn[15:0]), software can clear this bit by writing 1 to it.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note1: If CMPDAT equal to PERIOD, this flag is not working in up counter type selection.
     * |        |          |Note2: In complementary mode, CMPUIF1, 3, 5 use as another CMPUIF for channel 0, 2, 4.
     * |[23]    |IFAIF4_5  |PWM_Channel 4/5 Interrupt Flag Accumulator Interrupt Flag
     * |        |          |Flag is set by hardware when condition match IFSEL4_5 bits in PWM_IFA register, software can clear this bit by writing 1 to it.
     * |[24]    |CMPDIF0   |PWM Compare Down Count Interrupt Flag
     * |        |          |Flag is set by hardware when PWM counter down count and reaches CMP(PWM_CMPDATn[15:0]), software can clear this bit by writing 1 to it.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note1: If CMP equal to PERIOD, this flag is not working in down counter type selection.
     * |        |          |Note2: In complementary mode, CMPDIF1, 3, 5 use as another CMPDIF for channel 0, 2, 4.
     * |[25]    |CMPDIF1   |PWM Compare Down Count Interrupt Flag
     * |        |          |Flag is set by hardware when PWM counter down count and reaches CMP(PWM_CMPDATn[15:0]), software can clear this bit by writing 1 to it.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note1: If CMP equal to PERIOD, this flag is not working in down counter type selection.
     * |        |          |Note2: In complementary mode, CMPDIF1, 3, 5 use as another CMPDIF for channel 0, 2, 4.
     * |[26]    |CMPDIF2   |PWM Compare Down Count Interrupt Flag
     * |        |          |Flag is set by hardware when PWM counter down count and reaches CMP(PWM_CMPDATn[15:0]), software can clear this bit by writing 1 to it.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note1: If CMP equal to PERIOD, this flag is not working in down counter type selection.
     * |        |          |Note2: In complementary mode, CMPDIF1, 3, 5 use as another CMPDIF for channel 0, 2, 4.
     * |[27]    |CMPDIF3   |PWM Compare Down Count Interrupt Flag
     * |        |          |Flag is set by hardware when PWM counter down count and reaches CMP(PWM_CMPDATn[15:0]), software can clear this bit by writing 1 to it.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note1: If CMP equal to PERIOD, this flag is not working in down counter type selection.
     * |        |          |Note2: In complementary mode, CMPDIF1, 3, 5 use as another CMPDIF for channel 0, 2, 4.
     * |[28]    |CMPDIF4   |PWM Compare Down Count Interrupt Flag
     * |        |          |Flag is set by hardware when PWM counter down count and reaches CMP(PWM_CMPDATn[15:0]), software can clear this bit by writing 1 to it.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note1: If CMP equal to PERIOD, this flag is not working in down counter type selection.
     * |        |          |Note2: In complementary mode, CMPDIF1, 3, 5 use as another CMPDIF for channel 0, 2, 4.
     * |[29]    |CMPDIF5   |PWM Compare Down Count Interrupt Flag
     * |        |          |Flag is set by hardware when PWM counter down count and reaches CMP(PWM_CMPDATn[15:0]), software can clear this bit by writing 1 to it.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note1: If CMP equal to PERIOD, this flag is not working in down counter type selection.
     * |        |          |Note2: In complementary mode, CMPDIF1, 3, 5 use as another CMPDIF for channel 0, 2, 4.
     * @var PWM_T::INTSTS1
     * Offset: 0xEC  PWM Interrupt Flag Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |BRKEIF0   |PWM Channel 0 Edge-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM channel 0 edge-detect brake event do not happened.
     * |        |          |1 = When PWM channel 0 edge-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[1]     |BRKEIF1   |PWM Channel 1 Edge-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM channel 1 edge-detect brake event do not happened.
     * |        |          |1 = When PWM channel 1 edge-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[2]     |BRKEIF2   |PWM Channel 2 Edge-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM channel 2 edge-detect brake event do not happened.
     * |        |          |1 = When PWM channel 2 edge-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[3]     |BRKEIF3   |PWM Channel 3 Edge-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM channel 3 edge-detect brake event do not happened.
     * |        |          |1 = When PWM channel 3 edge-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[4]     |BRKEIF4   |PWM Channel 4 Edge-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM channel 4 edge-detect brake event do not happened.
     * |        |          |1 = When PWM channel 4 edge-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[5]     |BRKEIF5   |PWM Channel 5 Edge-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM channel 5 edge-detect brake event do not happened.
     * |        |          |1 = When PWM channel 5 edge-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[8]     |BRKLIF0   |PWM Channel 0 Level-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM channel 0 level-detect brake event do not happened.
     * |        |          |1 = When PWM channel 0 level-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[9]     |BRKLIF1   |PWM Channel 1 Level-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM channel 1 level-detect brake event do not happened.
     * |        |          |1 = When PWM channel 1 level-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[10]    |BRKLIF2   |PWM Channel 2 Level-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM channel 2 level-detect brake event do not happened.
     * |        |          |1 = When PWM channel 2 level-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[11]    |BRKLIF3   |PWM Channel 3 Level-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM channel 3 level-detect brake event do not happened.
     * |        |          |1 = When PWM channel 3 level-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[12]    |BRKLIF4   |PWM Channel 4 Level-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM channel 4 level-detect brake event do not happened.
     * |        |          |1 = When PWM channel 4 level-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[13]    |BRKLIF5   |PWM Channel 5 Level-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM channel 5 level-detect brake event do not happened.
     * |        |          |1 = When PWM channel 5 level-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[16]    |BRKESTS0  |PWM Channel 0 Edge-detect Brake Status
     * |        |          |0 = PWM channel 0 edge-detect brake state is released.
     * |        |          |1 = When PWM channel 0 edge-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM channel 0 at brake state, writing 1 to clear.
     * |[17]    |BRKESTS1  |PWM Channel 1 Edge-detect Brake Status
     * |        |          |0 = PWM channel 1 edge-detect brake state is released.
     * |        |          |1 = When PWM channel 1 edge-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM channel 1 at brake state, writing 1 to clear.
     * |[18]    |BRKESTS2  |PWM Channel 2 Edge-detect Brake Status
     * |        |          |0 = PWM channel 2 edge-detect brake state is released.
     * |        |          |1 = When PWM channel 2 edge-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM channel 2 at brake state, writing 1 to clear.
     * |[19]    |BRKESTS3  |PWM Channel n5 Edge-detect Brake Status
     * |        |          |0 = PWM channel 3 edge-detect brake state is released.
     * |        |          |1 = When PWM channel 3 edge-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM channel 3 at brake state, writing 1 to clear.
     * |[20]    |BRKESTS4  |PWM Channel 4 Edge-detect Brake Status
     * |        |          |0 = PWM channel 4 edge-detect brake state is released.
     * |        |          |1 = When PWM channel 4 edge-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM channel 4 at brake state, writing 1 to clear.
     * |[21]    |BRKESTS5  |PWM Channel 5 Edge-detect Brake Status
     * |        |          |0 = PWM channel 5 edge-detect brake state is released.
     * |        |          |1 = When PWM channel 5 edge-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM channel 5 at brake state, writing 1 to clear.
     * |[24]    |BRKLSTS0  |PWM Channel 0 Level-detect Brake Status (Read Only)
     * |        |          |0 = PWM channel 0 level-detect brake state is released.
     * |        |          |1 = When PWM channel 0 level-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM channel 0 at brake state.
     * |        |          |Note: This bit is read only and auto cleared by hardware.
     * |        |          |When enabled brake source return to high level, PWM will release brake state until current PWM period finished.
     * |        |          |The PWM waveform will start output from next full PWM period.
     * |[25]    |BRKLSTS1  |PWM Channel 1 Level-detect Brake Status (Read Only)
     * |        |          |0 = PWM channel 1 level-detect brake state is released.
     * |        |          |1 = When PWM channel 1 level-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM channel 1 at brake state.
     * |        |          |Note: This bit is read only and auto cleared by hardware.
     * |        |          |When enabled brake source return to high level, PWM will release brake state until current PWM period finished.
     * |        |          |The PWM waveform will start output from next full PWM period.
     * |[26]    |BRKLSTS2  |PWM Channel 2 Level-detect Brake Status (Read Only)
     * |        |          |0 = PWM channel 2 level-detect brake state is released.
     * |        |          |1 = When PWM channel 2 level-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM channel 2 at brake state.
     * |        |          |Note: This bit is read only and auto cleared by hardware.
     * |        |          |When enabled brake source return to high level, PWM will release brake state until current PWM period finished.
     * |        |          |The PWM waveform will start output from next full PWM period.
     * |[27]    |BRKLSTS3  |PWM Channel 3 Level-detect Brake Status (Read Only)
     * |        |          |0 = PWM channel 3 level-detect brake state is released.
     * |        |          |1 = When PWM channel 3 level-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM channel 3 at brake state.
     * |        |          |Note: This bit is read only and auto cleared by hardware.
     * |        |          |When enabled brake source return to high level, PWM will release brake state until current PWM period finished.
     * |        |          |The PWM waveform will start output from next full PWM period.
     * |[28]    |BRKLSTS4  |PWM Channel 4 Level-detect Brake Status (Read Only)
     * |        |          |0 = PWM channel 4 level-detect brake state is released.
     * |        |          |1 = When PWM channel 4 level-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM channel 4 at brake state.
     * |        |          |Note: This bit is read only and auto cleared by hardware.
     * |        |          |When enabled brake source return to high level, PWM will release brake state until current PWM period finished.
     * |        |          |The PWM waveform will start output from next full PWM period.
     * |[29]    |BRKLSTS5  |PWM Channel 5 Level-detect Brake Status (Read Only)
     * |        |          |0 = PWM channel 5 level-detect brake state is released.
     * |        |          |1 = When PWM channel 5 level-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM channel 5 at brake state.
     * |        |          |Note: This bit is read only and auto cleared by hardware.
     * |        |          |When enabled brake source return to high level, PWM will release brake state until current PWM period finished.
     * |        |          |The PWM waveform will start output from next full PWM period.
     * @var PWM_T::IFA
     * Offset: 0xF0  PWM Interrupt Flag Accumulator Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |IFCNT0_1  |PWM Channel 0/1 Interrupt Flag Counter
     * |        |          |The register sets the count number which defines how many times of PWM Channel 0/1 period occurs to set IFAIF0_1 bit to request the PWM period interrupt.
     * |        |          |PWM flag will be set in every IFCNT0_1 [3:0] times of PWM period.
     * |[6:4]   |IFSEL0_1  |PWM Channel 0/1 Interrupt Flag Accumulator Source Select
     * |        |          |000 = CNT equal to Zero in channel 0.
     * |        |          |001 = CNT equal to PERIOD in channel 0.
     * |        |          |010 = CNT equal to CMPU in channel 0.
     * |        |          |011 = CNT equal to CMPD in channel 0.
     * |        |          |100 = CNT equal to Zero in channel 1.
     * |        |          |101 = CNT equal to PERIOD in channel 1.
     * |        |          |110 = CNT equal to CMPU in channel 1.
     * |        |          |111 = CNT equal to CMPD in channel 1.
     * |[7]     |IFAEN0_1  |PWM Channel 0/1 Interrupt Flag Accumulator Enable Bit
     * |        |          |0 = PWM Channel 0/1 interrupt flag accumulator disable.
     * |        |          |1 = PWM Channel 0/1 interrupt flag accumulator enable.
     * |[11:8]  |IFCNT2_3  |PWM Channel 2/3 Interrupt Flag Counter
     * |        |          |The register sets the count number which defines how many times of PWM Channel 2/3 period occurs to set IFAIF2_3 bit to request the PWM period interrupt.
     * |        |          |PWM flag will be set in every IFCNT2_3[3:0] times of PWM period.
     * |[14:12] |IFSEL2_3  |PWM Channel 2/3 Interrupt Flag Accumulator Source Select
     * |        |          |000 = CNT equal to Zero in channel 2.
     * |        |          |001 = CNT equal to PERIOD in channel 2.
     * |        |          |010 = CNT equal to CMPU in channel 2.
     * |        |          |011 = CNT equal to CMPD in channel 2.
     * |        |          |100 = CNT equal to Zero in channel 3.
     * |        |          |101 = CNT equal to PERIOD in channel 3.
     * |        |          |110 = CNT equal to CMPU in channel 3.
     * |        |          |111 = CNT equal to CMPD in channel 3.
     * |[15]    |IFAEN2_3  |PWM Channel 2/3 Interrupt Flag Accumulator Enable Bit
     * |        |          |0 = PWM Channel 2/3 interrupt flag accumulator disable.
     * |        |          |1 = PWM Channel 2/3 interrupt flag accumulator enable.
     * |[19:16] |IFCNT4_5  |PWM Channel 4/5 Interrupt Flag Counter
     * |        |          |The register sets the count number which defines how many times of PWM Channel 4/5 period occurs to set IFAIF4_5 bit to request the PWM period interrupt.
     * |        |          |PWM flag will be set in every IFCNT4_5[3:0] times of PWM period.
     * |[22:20] |IFSEL4_5  |PWM Channel 4/5 Interrupt Flag Accumulator Source Select
     * |        |          |000 = CNT equal to Zero in channel 4.
     * |        |          |001 = CNT equal to PERIOD in channel 4.
     * |        |          |010 = CNT equal to CMPU in channel 4.
     * |        |          |011 = CNT equal to CMPD in channel 4.
     * |        |          |100 = CNT equal to Zero in channel 5.
     * |        |          |101 = CNT equal to PERIOD in channel 5.
     * |        |          |110 = CNT equal to CMPU in channel 5.
     * |        |          |111 = CNT equal to CMPD in channel 5.
     * |[23]    |IFAEN4_5  |PWM Channel 4/5 Interrupt Flag Accumulator Enable Bit
     * |        |          |0 = PWM Channel 4/5 interrupt flag accumulator disable.
     * |        |          |1 = PWM Channel 4/5 interrupt flag accumulator enable.
     * @var PWM_T::ADCTS0
     * Offset: 0xF8  PWM Trigger ADC Source Select Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |TRGSEL0   |PWM_CH0 Trigger ADC Source Select
     * |        |          |0000 = PWM_CH0 zero point.
     * |        |          |0001 = PWM_CH0 period point.
     * |        |          |0010 = PWM_CH0 zero or period point.
     * |        |          |0011 = PWM_CH0 up-count compared point.
     * |        |          |0100 = PWM_CH0 down-count compared point.
     * |        |          |0101 = PWM_CH1 zero point.
     * |        |          |0110 = PWM_CH1 period point.
     * |        |          |0111 = PWM_CH1 zero or period point.
     * |        |          |1000 = PWM_CH1 up-count compared point.
     * |        |          |1001 = PWM_CH1 down-count compared point.
     * |        |          |1010 = PWM_CH0 up-count free trigger compared point.
     * |        |          |1011 = PWM_CH0 down-count free trigger compared point.
     * |        |          |1100 = PWM_CH2 up-count free trigger compared point.
     * |        |          |1101 = PWM_CH2 down-count free trigger compared point.
     * |        |          |1110 = PWM_CH4 up-count free trigger compared point.
     * |        |          |1111 = PWM_CH4 down-count free trigger compared point.
     * |[7]     |TRGEN0    |PWM_CH0 Trigger ADC enable bit
     * |[11:8]  |TRGSEL1   |PWM_CH1 Trigger ADC Source Select
     * |        |          |0000 = PWM_CH0 zero point.
     * |        |          |0001 = PWM_CH0 period point.
     * |        |          |0010 = PWM_CH0 zero or period point.
     * |        |          |0011 = PWM_CH0 up-count compared point.
     * |        |          |0100 = PWM_CH0 down-count compared point.
     * |        |          |0101 = PWM_CH1 zero point.
     * |        |          |0110 = PWM_CH1 period point.
     * |        |          |0111 = PWM_CH1 zero or period point.
     * |        |          |1000 = PWM_CH1 up-count compared point.
     * |        |          |1001 = PWM_CH1 down-count compared point.
     * |        |          |1010 = PWM_CH0 up-count free trigger compared point.
     * |        |          |1011 = PWM_CH0 down-count free trigger compared point.
     * |        |          |1100 = PWM_CH2 up-count free trigger compared point.
     * |        |          |1101 = PWM_CH2 down-count free trigger compared point.
     * |        |          |1110 = PWM_CH4 up-count free trigger compared point.
     * |        |          |1111 = PWM_CH4 down-count free trigger compared point.
     * |[15]    |TRGEN1    |PWM_CH1 Trigger ADC enable bit
     * |[19:16] |TRGSEL2   |PWM_CH2 Trigger ADC Source Select
     * |        |          |0000 = PWM_CH2 zero point.
     * |        |          |0001 = PWM_CH2 period point.
     * |        |          |0010 = PWM_CH2 zero or period point.
     * |        |          |0011 = PWM_CH2 up-count compared point.
     * |        |          |0100 = PWM_CH2 down-count compared point.
     * |        |          |0101 = PWM_CH3 zero point.
     * |        |          |0110 = PWM_CH3 period point.
     * |        |          |0111 = PWM_CH3 zero or period point.
     * |        |          |1000 = PWM_CH3 up-count compared point.
     * |        |          |1001 = PWM_CH3 down-count compared point.
     * |        |          |1010 = PWM_CH0 up-count free trigger compared point.
     * |        |          |1011 = PWM_CH0 down-count free trigger compared point.
     * |        |          |1100 = PWM_CH2 up-count free trigger compared point.
     * |        |          |1101 = PWM_CH2 down-count free trigger compared point.
     * |        |          |1110 = PWM_CH4 up-count free trigger compared point.
     * |        |          |1111 = PWM_CH4 down-count free trigger compared point.
     * |[23]    |TRGEN2    |PWM_CH2 Trigger ADC enable bit
     * |[27:24] |TRGSEL3   |PWM_CH3 Trigger ADC Source Select
     * |        |          |0000 = PWM_CH2 zero point.
     * |        |          |0001 = PWM_CH2 period point.
     * |        |          |0010 = PWM_CH2 zero or period point.
     * |        |          |0011 = PWM_CH2 up-count compared point.
     * |        |          |0100 = PWM_CH2 down-count compared point.
     * |        |          |0101 = PWM_CH3 zero point.
     * |        |          |0110 = PWM_CH3 period point.
     * |        |          |0111 = PWM_CH3 zero or period point.
     * |        |          |1000 = PWM_CH3 up-count compared point.
     * |        |          |1001 = PWM_CH3 down-count compared point.
     * |        |          |1010 = PWM_CH0 up-count free trigger compared point.
     * |        |          |1011 = PWM_CH0 down-count free trigger compared point.
     * |        |          |1100 = PWM_CH2 up-count free trigger compared point.
     * |        |          |1101 = PWM_CH2 down-count free trigger compared point.
     * |        |          |1110 = PWM_CH4 up-count free trigger compared point.
     * |        |          |1111 = PWM_CH4 down-count free trigger compared point.
     * |[31]    |TRGEN3    |PWM_CH3 Trigger ADC enable bit
     * @var PWM_T::ADCTS1
     * Offset: 0xFC  PWM Trigger ADC Source Select Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |TRGSEL4   |PWM_CH4 Trigger ADC Source Select
     * |        |          |0000 = PWM_CH4 zero point.
     * |        |          |0001 = PWM_CH4 period point.
     * |        |          |0010 = PWM_CH4 zero or period point.
     * |        |          |0011 = PWM_CH4 up-count compared point.
     * |        |          |0100 = PWM_CH4 down-count compared point.
     * |        |          |0101 = PWM_CH5 zero point.
     * |        |          |0110 = PWM_CH5 period point.
     * |        |          |0111 = PWM_CH5 zero or period point.
     * |        |          |1000 = PWM_CH5 up-count compared point.
     * |        |          |1001 = PWM_CH5 down-count compared point.
     * |        |          |1010 = PWM_CH0 up-count free trigger compared point.
     * |        |          |1011 = PWM_CH0 down-count free trigger compared point.
     * |        |          |1100 = PWM_CH2 up-count free trigger compared point.
     * |        |          |1101 = PWM_CH2 down-count free trigger compared point.
     * |        |          |1110 = PWM_CH4 up-count free trigger compared point.
     * |        |          |1111 = PWM_CH4 down-count free trigger compared point.
     * |[7]     |TRGEN4    |PWM_CH4 Trigger ADC enable bit
     * |[11:8]  |TRGSEL5   |PWM_CH5 Trigger ADC Source Select
     * |        |          |0000 = PWM_CH4 zero point.
     * |        |          |0001 = PWM_CH4 period point.
     * |        |          |0010 = PWM_CH4 zero or period point.
     * |        |          |0011 = PWM_CH4 up-count compared point.
     * |        |          |0100 = PWM_CH4 down-count compared point.
     * |        |          |0101 = PWM_CH5 zero point.
     * |        |          |0110 = PWM_CH5 period point.
     * |        |          |0111 = PWM_CH5 zero or period point.
     * |        |          |1000 = PWM_CH5 up-count compared point.
     * |        |          |1001 = PWM_CH5 down-count compared point.
     * |        |          |1010 = PWM_CH0 up-count free trigger compared point.
     * |        |          |1011 = PWM_CH0 down-count free trigger compared point.
     * |        |          |1100 = PWM_CH2 up-count free trigger compared point.
     * |        |          |1101 = PWM_CH2 down-count free trigger compared point.
     * |        |          |1110 = PWM_CH4 up-count free trigger compared point.
     * |        |          |1111 = PWM_CH4 down-count free trigger compared point.
     * |[15]    |TRGEN5    |PWM_CH5 Trigger ADC enable bit
     * @var PWM_T::FTCMPDAT0_1
     * Offset: 0x100  PWM Free Trigger Compare Register 0/1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FTCMP     |PWM Free Trigger Compare Register
     * |        |          |FTCMP use to compare with even CNT(PWM_CNTm[15:0], m=0,2,4) to trigger ADC
     * |        |          |FTCMPDAT0, 2, 4 corresponding complementary pairs PWMx_CH0 and PWMx_CH1, PWMx_CH2 and PWMx_CH3, PWMx_CH4 and PWMx_CH5.
     * @var PWM_T::FTCMPDAT2_3
     * Offset: 0x104  PWM Free Trigger Compare Register 2/3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FTCMP     |PWM Free Trigger Compare Register
     * |        |          |FTCMP use to compare with even CNT(PWM_CNTm[15:0], m=0,2,4) to trigger ADC
     * |        |          |FTCMPDAT0, 2, 4 corresponding complementary pairs PWMx_CH0 and PWMx_CH1, PWMx_CH2 and PWMx_CH3, PWMx_CH4 and PWMx_CH5.
     * @var PWM_T::FTCMPDAT4_5
     * Offset: 0x108  PWM Free Trigger Compare Register 4/5
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FTCMP     |PWM Free Trigger Compare Register
     * |        |          |FTCMP use to compare with even CNT(PWM_CNTm[15:0], m=0,2,4) to trigger ADC
     * |        |          |FTCMPDAT0, 2, 4 corresponding complementary pairs PWMx_CH0 and PWMx_CH1, PWMx_CH2 and PWMx_CH3, PWMx_CH4 and PWMx_CH5.
     * @var PWM_T::SSCTL
     * Offset: 0x110  PWM Synchronous Start Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SSEN0     |PWM Synchronous Start Function Enable Bits
     * |        |          |When synchronous start function is enabled, the PWM counter enable register (PWM_CNTEN) can be enabled by writing PWM synchronous start trigger bit (CNTSEN).
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM synchronous start function Disabled.
     * |        |          |1 = PWM synchronous start function Enabled.
     * |[1]     |SSEN1     |PWM Synchronous Start Function Enable Bits
     * |        |          |When synchronous start function is enabled, the PWM counter enable register (PWM_CNTEN) can be enabled by writing PWM synchronous start trigger bit (CNTSEN).
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM synchronous start function Disabled.
     * |        |          |1 = PWM synchronous start function Enabled.
     * |[2]     |SSEN2     |PWM Synchronous Start Function Enable Bits
     * |        |          |When synchronous start function is enabled, the PWM counter enable register (PWM_CNTEN) can be enabled by writing PWM synchronous start trigger bit (CNTSEN).
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM synchronous start function Disabled.
     * |        |          |1 = PWM synchronous start function Enabled.
     * |[3]     |SSEN3     |PWM Synchronous Start Function Enable Bits
     * |        |          |When synchronous start function is enabled, the PWM counter enable register (PWM_CNTEN) can be enabled by writing PWM synchronous start trigger bit (CNTSEN).
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM synchronous start function Disabled.
     * |        |          |1 = PWM synchronous start function Enabled.
     * |[4]     |SSEN4     |PWM Synchronous Start Function Enable Bits
     * |        |          |When synchronous start function is enabled, the PWM counter enable register (PWM_CNTEN) can be enabled by writing PWM synchronous start trigger bit (CNTSEN).
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM synchronous start function Disabled.
     * |        |          |1 = PWM synchronous start function Enabled.
     * |[5]     |SSEN5     |PWM Synchronous Start Function Enable Bits
     * |        |          |When synchronous start function is enabled, the PWM counter enable register (PWM_CNTEN) can be enabled by writing PWM synchronous start trigger bit (CNTSEN).
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM synchronous start function Disabled.
     * |        |          |1 = PWM synchronous start function Enabled.
     * |[8]     |SSRC      |PWM Synchronous Start Source Select Bit
     * |        |          |0 = Synchronous start source come from PWM0.
     * |        |          |1 = Synchronous start source come from PWM1.
     * @var PWM_T::SSTRG
     * Offset: 0x114  PWM Synchronous Start Trigger Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CNTSEN    |PWM Counter Synchronous Start Enable (Write Only)
     * |        |          |PMW counter synchronous enable function is used to make selected PWM channels (PWMx_CHn) start counting at the same time.
     * |        |          |Writing this bit to 1 will also set the counter enable bit (CNTENn, n denotes channel 0 to 5) if correlated PWM channel counter synchronous start function is enabled.
     * @var PWM_T::LEBCTL
     * Offset: 0x118  PWM Leading Edge Blanking Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |LEBEN     |PWM Leading Edge Blanking Enable Bit
     * |        |          |0 = PWM Leading Edge Blanking Disabled.
     * |        |          |1 = PWM Leading Edge Blanking Enabled.
     * |[8]     |SRCEN0    |PWM Leading Edge Blanking Source From PWMx_CH0 Enable Bit
     * |        |          |0 = PWM Leading Edge Blanking Source from PWMx_CH0 Disabled.
     * |        |          |1 = PWM Leading Edge Blanking Source from PWMx_CH0 Enabled.
     * |[9]     |SRCEN2    |PWM Leading Edge Blanking Source From PWMx_CH2 Enable Bit
     * |        |          |0 = PWM Leading Edge Blanking Source from PWMx_CH2 Disabled.
     * |        |          |1 = PWM Leading Edge Blanking Source from PWMx_CH2 Enabled.
     * |[10]    |SRCEN4    |PWM Leading Edge Blanking Source From PWMx_CH4 Enable Bit
     * |        |          |0 = PWM Leading Edge Blanking Source from PWMx_CH4 Disabled.
     * |        |          |1 = PWM Leading Edge Blanking Source from PWMx_CH4 Enabled.
     * |[17:16] |TRGTYPE   |PWM Leading Edge Blanking Trigger Type
     * |        |          |00 = When detect leading edge blanking source rising edge, blanking counter start counting.
     * |        |          |01 = When detect leading edge blanking source falling edge, blanking counter start counting.
     * |        |          |10 = When detect leading edge blanking source rising or falling edge, blanking counter start counting.
     * |        |          |11 = Reserved.
     * @var PWM_T::LEBCNT
     * Offset: 0x11C  PWM Leading Edge Blanking Counter Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[8:0]   |LEBCNT    |PWM Leading Edge Blanking Counter
     * |        |          |This counter value decides leading edge blanking window size.
     * |        |          |Blanking window size = LEBCNT+1, and LEB counter clock base is ECLK.
     * @var PWM_T::STATUS
     * Offset: 0x120  PWM Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CNTMAXF0  |Time-base Counter Equal to 0xFFFF Latched Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = indicates the time-base counter(PWM_CNTn[15:0]) never reached its maximum value 0xFFFF.
     * |        |          |1 = indicates the time-base counter(PWM_CNTn[15:0]) reached its maximum value, software can write 1 to clear this bit.
     * |[1]     |CNTMAXF1  |Time-base Counter Equal to 0xFFFF Latched Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = indicates the time-base counter(PWM_CNTn[15:0]) never reached its maximum value 0xFFFF.
     * |        |          |1 = indicates the time-base counter(PWM_CNTn[15:0]) reached its maximum value, software can write 1 to clear this bit.
     * |[2]     |CNTMAXF2  |Time-base Counter Equal to 0xFFFF Latched Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = indicates the time-base counter(PWM_CNTn[15:0]) never reached its maximum value 0xFFFF.
     * |        |          |1 = indicates the time-base counter(PWM_CNTn[15:0]) reached its maximum value, software can write 1 to clear this bit.
     * |[3]     |CNTMAXF3  |Time-base Counter Equal to 0xFFFF Latched Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = indicates the time-base counter(PWM_CNTn[15:0]) never reached its maximum value 0xFFFF.
     * |        |          |1 = indicates the time-base counter(PWM_CNTn[15:0]) reached its maximum value, software can write 1 to clear this bit.
     * |[4]     |CNTMAXF4  |Time-base Counter Equal to 0xFFFF Latched Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = indicates the time-base counter(PWM_CNTn[15:0]) never reached its maximum value 0xFFFF.
     * |        |          |1 = indicates the time-base counter(PWM_CNTn[15:0]) reached its maximum value, software can write 1 to clear this bit.
     * |[5]     |CNTMAXF5  |Time-base Counter Equal to 0xFFFF Latched Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = indicates the time-base counter(PWM_CNTn[15:0]) never reached its maximum value 0xFFFF.
     * |        |          |1 = indicates the time-base counter(PWM_CNTn[15:0]) reached its maximum value, software can write 1 to clear this bit.
     * |[8]     |SYNCINF0  |Input Synchronization Latched Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Indicates no SYNC_IN event has occurred.
     * |        |          |1 = Indicates an SYNC_IN event has occurred, software can write 1 to clear this bit.
     * |[9]     |SYNCINF2  |Input Synchronization Latched Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Indicates no SYNC_IN event has occurred.
     * |        |          |1 = Indicates an SYNC_IN event has occurred, software can write 1 to clear this bit.
     * |[10]    |SYNCINF4  |Input Synchronization Latched Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Indicates no SYNC_IN event has occurred.
     * |        |          |1 = Indicates an SYNC_IN event has occurred, software can write 1 to clear this bit.
     * |[16]    |ADCTRGF0  |ADC Start of Conversion Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Indicates no ADC start of conversion trigger event has occurred.
     * |        |          |1 = Indicates an ADC start of conversion trigger event has occurred, software can write 1 to clear this bit.
     * |[17]    |ADCTRGF1  |ADC Start of Conversion Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Indicates no ADC start of conversion trigger event has occurred.
     * |        |          |1 = Indicates an ADC start of conversion trigger event has occurred, software can write 1 to clear this bit.
     * |[18]    |ADCTRGF2  |ADC Start of Conversion Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Indicates no ADC start of conversion trigger event has occurred.
     * |        |          |1 = Indicates an ADC start of conversion trigger event has occurred, software can write 1 to clear this bit.
     * |[19]    |ADCTRGF3  |ADC Start of Conversion Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Indicates no ADC start of conversion trigger event has occurred.
     * |        |          |1 = Indicates an ADC start of conversion trigger event has occurred, software can write 1 to clear this bit.
     * |[20]    |ADCTRGF4  |ADC Start of Conversion Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Indicates no ADC start of conversion trigger event has occurred.
     * |        |          |1 = Indicates an ADC start of conversion trigger event has occurred, software can write 1 to clear this bit.
     * |[21]    |ADCTRGF5  |ADC Start of Conversion Flag
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Indicates no ADC start of conversion trigger event has occurred.
     * |        |          |1 = Indicates an ADC start of conversion trigger event has occurred, software can write 1 to clear this bit.
     * @var PWM_T::CAPINEN
     * Offset: 0x200  PWM Capture Input Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CAPINEN0  |Capture Input Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM Channel capture input path Disabled.
     * |        |          |The input of PWM channel capture function is always regarded as 0.
     * |        |          |1 = PWM Channel capture input path Enabled.
     * |        |          |The input of PWM channel capture function comes from correlative multifunction pin.
     * |[1]     |CAPINEN1  |Capture Input Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM Channel capture input path Disabled.
     * |        |          |The input of PWM channel capture function is always regarded as 0.
     * |        |          |1 = PWM Channel capture input path Enabled.
     * |        |          |The input of PWM channel capture function comes from correlative multifunction pin.
     * |[2]     |CAPINEN2  |Capture Input Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM Channel capture input path Disabled.
     * |        |          |The input of PWM channel capture function is always regarded as 0.
     * |        |          |1 = PWM Channel capture input path Enabled.
     * |        |          |The input of PWM channel capture function comes from correlative multifunction pin.
     * |[3]     |CAPINEN3  |Capture Input Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM Channel capture input path Disabled.
     * |        |          |The input of PWM channel capture function is always regarded as 0.
     * |        |          |1 = PWM Channel capture input path Enabled.
     * |        |          |The input of PWM channel capture function comes from correlative multifunction pin.
     * |[4]     |CAPINEN4  |Capture Input Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM Channel capture input path Disabled.
     * |        |          |The input of PWM channel capture function is always regarded as 0.
     * |        |          |1 = PWM Channel capture input path Enabled.
     * |        |          |The input of PWM channel capture function comes from correlative multifunction pin.
     * |[5]     |CAPINEN5  |Capture Input Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = PWM Channel capture input path Disabled.
     * |        |          |The input of PWM channel capture function is always regarded as 0.
     * |        |          |1 = PWM Channel capture input path Enabled.
     * |        |          |The input of PWM channel capture function comes from correlative multifunction pin.
     * @var PWM_T::CAPCTL
     * Offset: 0x204  PWM Capture Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CAPEN0    |Capture Function Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture function Disabled. RCAPDAT/FCAPDAT registers will not be updated.
     * |        |          |1 = Capture function Enabled
     * |        |          |Capture latched the PWM counter value when detected rising or falling edge of input signal and saved to RCAPDAT (Rising latch) and FCAPDAT (Falling latch) registers.
     * |[1]     |CAPEN1    |Capture Function Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture function Disabled. RCAPDAT/FCAPDAT registers will not be updated.
     * |        |          |1 = Capture function Enabled
     * |        |          |Capture latched the PWM counter value when detected rising or falling edge of input signal and saved to RCAPDAT (Rising latch) and FCAPDAT (Falling latch) registers.
     * |[2]     |CAPEN2    |Capture Function Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture function Disabled. RCAPDAT/FCAPDAT registers will not be updated.
     * |        |          |1 = Capture function Enabled
     * |        |          |Capture latched the PWM counter value when detected rising or falling edge of input signal and saved to RCAPDAT (Rising latch) and FCAPDAT (Falling latch) registers.
     * |[3]     |CAPEN3    |Capture Function Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture function Disabled. RCAPDAT/FCAPDAT registers will not be updated.
     * |        |          |1 = Capture function Enabled
     * |        |          |Capture latched the PWM counter value when detected rising or falling edge of input signal and saved to RCAPDAT (Rising latch) and FCAPDAT (Falling latch) registers.
     * |[4]     |CAPEN4    |Capture Function Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture function Disabled. RCAPDAT/FCAPDAT registers will not be updated.
     * |        |          |1 = Capture function Enabled
     * |        |          |Capture latched the PWM counter value when detected rising or falling edge of input signal and saved to RCAPDAT (Rising latch) and FCAPDAT (Falling latch) registers.
     * |[5]     |CAPEN5    |Capture Function Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture function Disabled. RCAPDAT/FCAPDAT registers will not be updated.
     * |        |          |1 = Capture function Enabled
     * |        |          |Capture latched the PWM counter value when detected rising or falling edge of input signal and saved to RCAPDAT (Rising latch) and FCAPDAT (Falling latch) registers.
     * |[8]     |CAPINV0   |Capture Inverter Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture source inverter Disabled.
     * |        |          |1 = Capture source inverter Enabled. Reverse the input signal from GPIO.
     * |[9]     |CAPINV1   |Capture Inverter Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture source inverter Disabled.
     * |        |          |1 = Capture source inverter Enabled. Reverse the input signal from GPIO.
     * |[10]    |CAPINV2   |Capture Inverter Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture source inverter Disabled.
     * |        |          |1 = Capture source inverter Enabled. Reverse the input signal from GPIO.
     * |[11]    |CAPINV3   |Capture Inverter Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture source inverter Disabled.
     * |        |          |1 = Capture source inverter Enabled. Reverse the input signal from GPIO.
     * |[12]    |CAPINV4   |Capture Inverter Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture source inverter Disabled.
     * |        |          |1 = Capture source inverter Enabled. Reverse the input signal from GPIO.
     * |[13]    |CAPINV5   |Capture Inverter Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture source inverter Disabled.
     * |        |          |1 = Capture source inverter Enabled. Reverse the input signal from GPIO.
     * |[16]    |RCRLDEN0  |Rising Capture Reload Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Rising capture reload counter Disabled.
     * |        |          |1 = Rising capture reload counter Enabled.
     * |[17]    |RCRLDEN1  |Rising Capture Reload Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Rising capture reload counter Disabled.
     * |        |          |1 = Rising capture reload counter Enabled.
     * |[18]    |RCRLDEN2  |Rising Capture Reload Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Rising capture reload counter Disabled.
     * |        |          |1 = Rising capture reload counter Enabled.
     * |[19]    |RCRLDEN3  |Rising Capture Reload Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Rising capture reload counter Disabled.
     * |        |          |1 = Rising capture reload counter Enabled.
     * |[20]    |RCRLDEN4  |Rising Capture Reload Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Rising capture reload counter Disabled.
     * |        |          |1 = Rising capture reload counter Enabled.
     * |[21]    |RCRLDEN5  |Rising Capture Reload Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Rising capture reload counter Disabled.
     * |        |          |1 = Rising capture reload counter Enabled.
     * |[24]    |FCRLDEN0  |Falling Capture Reload Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Falling capture reload counter Disabled.
     * |        |          |1 = Falling capture reload counter Enabled.
     * |[25]    |FCRLDEN1  |Falling Capture Reload Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Falling capture reload counter Disabled.
     * |        |          |1 = Falling capture reload counter Enabled.
     * |[26]    |FCRLDEN2  |Falling Capture Reload Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Falling capture reload counter Disabled.
     * |        |          |1 = Falling capture reload counter Enabled.
     * |[27]    |FCRLDEN3  |Falling Capture Reload Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Falling capture reload counter Disabled.
     * |        |          |1 = Falling capture reload counter Enabled.
     * |[28]    |FCRLDEN4  |Falling Capture Reload Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Falling capture reload counter Disabled.
     * |        |          |1 = Falling capture reload counter Enabled.
     * |[29]    |FCRLDEN5  |Falling Capture Reload Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Falling capture reload counter Disabled.
     * |        |          |1 = Falling capture reload counter Enabled.
     * @var PWM_T::CAPSTS
     * Offset: 0x208  PWM Capture Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CRLIFOV0  |Capture Rising Latch Interrupt Flag Overrun Status (Read Only)
     * |        |          |This flag indicates if rising latch happened when the corresponding CRLIFn(PWM_CAPIF[5:0]) bit is 1.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note: This bit will be cleared automatically when user clear corresponding CRLIFn bit.
     * |[1]     |CRLIFOV1  |Capture Rising Latch Interrupt Flag Overrun Status (Read Only)
     * |        |          |This flag indicates if rising latch happened when the corresponding CRLIFn(PWM_CAPIF[5:0]) bit is 1.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note: This bit will be cleared automatically when user clear corresponding CRLIFn bit.
     * |[2]     |CRLIFOV2  |Capture Rising Latch Interrupt Flag Overrun Status (Read Only)
     * |        |          |This flag indicates if rising latch happened when the corresponding CRLIFn(PWM_CAPIF[5:0]) bit is 1.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note: This bit will be cleared automatically when user clear corresponding CRLIFn bit.
     * |[3]     |CRLIFOV3  |Capture Rising Latch Interrupt Flag Overrun Status (Read Only)
     * |        |          |This flag indicates if rising latch happened when the corresponding CRLIFn(PWM_CAPIF[5:0]) bit is 1.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note: This bit will be cleared automatically when user clear corresponding CRLIFn bit.
     * |[4]     |CRLIFOV4  |Capture Rising Latch Interrupt Flag Overrun Status (Read Only)
     * |        |          |This flag indicates if rising latch happened when the corresponding CRLIFn(PWM_CAPIF[5:0]) bit is 1.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note: This bit will be cleared automatically when user clear corresponding CRLIFn bit.
     * |[5]     |CRLIFOV5  |Capture Rising Latch Interrupt Flag Overrun Status (Read Only)
     * |        |          |This flag indicates if rising latch happened when the corresponding CRLIFn(PWM_CAPIF[5:0]) bit is 1.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note: This bit will be cleared automatically when user clear corresponding CRLIFn bit.
     * |[8]     |CFLIFOV0  |Capture Falling Latch Interrupt Flag Overrun Status (Read Only)
     * |        |          |This flag indicates if falling latch happened when the corresponding CFLIFn(PWM_CAPIF[13:8]) bit is 1.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note: This bit will be cleared automatically when user clear corresponding CFLIFn bit.
     * |[9]     |CFLIFOV1  |Capture Falling Latch Interrupt Flag Overrun Status (Read Only)
     * |        |          |This flag indicates if falling latch happened when the corresponding CFLIFn(PWM_CAPIF[13:8]) bit is 1.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note: This bit will be cleared automatically when user clear corresponding CFLIFn bit.
     * |[10]    |CFLIFOV2  |Capture Falling Latch Interrupt Flag Overrun Status (Read Only)
     * |        |          |This flag indicates if falling latch happened when the corresponding CFLIFn(PWM_CAPIF[13:8]) bit is 1.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note: This bit will be cleared automatically when user clear corresponding CFLIFn bit.
     * |[11]    |CFLIFOV3  |Capture Falling Latch Interrupt Flag Overrun Status (Read Only)
     * |        |          |This flag indicates if falling latch happened when the corresponding CFLIFn(PWM_CAPIF[13:8]) bit is 1.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note: This bit will be cleared automatically when user clear corresponding CFLIFn bit.
     * |[12]    |CFLIFOV4  |Capture Falling Latch Interrupt Flag Overrun Status (Read Only)
     * |        |          |This flag indicates if falling latch happened when the corresponding CFLIFn(PWM_CAPIF[13:8]) bit is 1.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note: This bit will be cleared automatically when user clear corresponding CFLIFn bit.
     * |[13]    |CFLIFOV5  |Capture Falling Latch Interrupt Flag Overrun Status (Read Only)
     * |        |          |This flag indicates if falling latch happened when the corresponding CFLIFn(PWM_CAPIF[13:8]) bit is 1.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |Note: This bit will be cleared automatically when user clear corresponding CFLIFn bit.
     * @var PWM_T::RCAPDAT0
     * Offset: 0x20C  PWM Rising Capture Data Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RCAPDAT   |PWM Rising Capture Data Register (Read Only)
     * |        |          |When rising capture condition happened, the PWM counter value will be saved in this register.
     * @var PWM_T::FCAPDAT0
     * Offset: 0x210  PWM Falling Capture Data Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FCAPDAT   |PWM Falling Capture Data Register (Read Only)
     * |        |          |When falling capture condition happened, the PWM counter value will be saved in this register.
     * @var PWM_T::RCAPDAT1
     * Offset: 0x214  PWM Rising Capture Data Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RCAPDAT   |PWM Rising Capture Data Register (Read Only)
     * |        |          |When rising capture condition happened, the PWM counter value will be saved in this register.
     * @var PWM_T::FCAPDAT1
     * Offset: 0x218  PWM Falling Capture Data Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FCAPDAT   |PWM Falling Capture Data Register (Read Only)
     * |        |          |When falling capture condition happened, the PWM counter value will be saved in this register.
     * @var PWM_T::RCAPDAT2
     * Offset: 0x21C  PWM Rising Capture Data Register 2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RCAPDAT   |PWM Rising Capture Data Register (Read Only)
     * |        |          |When rising capture condition happened, the PWM counter value will be saved in this register.
     * @var PWM_T::FCAPDAT2
     * Offset: 0x220  PWM Falling Capture Data Register 2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FCAPDAT   |PWM Falling Capture Data Register (Read Only)
     * |        |          |When falling capture condition happened, the PWM counter value will be saved in this register.
     * @var PWM_T::RCAPDAT3
     * Offset: 0x224  PWM Rising Capture Data Register 3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RCAPDAT   |PWM Rising Capture Data Register (Read Only)
     * |        |          |When rising capture condition happened, the PWM counter value will be saved in this register.
     * @var PWM_T::FCAPDAT3
     * Offset: 0x228  PWM Falling Capture Data Register 3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FCAPDAT   |PWM Falling Capture Data Register (Read Only)
     * |        |          |When falling capture condition happened, the PWM counter value will be saved in this register.
     * @var PWM_T::RCAPDAT4
     * Offset: 0x22C  PWM Rising Capture Data Register 4
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RCAPDAT   |PWM Rising Capture Data Register (Read Only)
     * |        |          |When rising capture condition happened, the PWM counter value will be saved in this register.
     * @var PWM_T::FCAPDAT4
     * Offset: 0x230  PWM Falling Capture Data Register 4
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FCAPDAT   |PWM Falling Capture Data Register (Read Only)
     * |        |          |When falling capture condition happened, the PWM counter value will be saved in this register.
     * @var PWM_T::RCAPDAT5
     * Offset: 0x234  PWM Rising Capture Data Register 5
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RCAPDAT   |PWM Rising Capture Data Register (Read Only)
     * |        |          |When rising capture condition happened, the PWM counter value will be saved in this register.
     * @var PWM_T::FCAPDAT5
     * Offset: 0x238  PWM Falling Capture Data Register 5
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FCAPDAT   |PWM Falling Capture Data Register (Read Only)
     * |        |          |When falling capture condition happened, the PWM counter value will be saved in this register.
     * @var PWM_T::PDMACTL
     * Offset: 0x23C  PWM PDMA Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CHEN0_1   |Channel 0/1 PDMA Enable
     * |        |          |0 = Channel 0/1 PDMA function Disabled.
     * |        |          |1 = Channel 0/1 PDMA function Enabled for the channel 0/1 captured data and transfer to memory.
     * |[2:1]   |CAPMOD0_1 |Select PWM_RCAPDAT0/1 or PWM_FCAPDAT0/1 to Do PDMA Transfer
     * |        |          |00 = Reserved.
     * |        |          |01 = PWM_RCAPDAT0/1 register.
     * |        |          |10 = PWM_FCAPDAT0/1 register.
     * |        |          |11 = Both PWM_RCAPDAT0/1 and PWM_FCAPDAT0/1 registers.
     * |[3]     |CAPORD0_1 |Capture Channel 0/1 Rising/Falling Order
     * |        |          |Set this bit to determine whether the PWM_RCAPDAT0/1 or PWM_FCAPDAT0/1 register is the first captured data transferred to memory through PDMA when CAPMOD0_1 bits are set to = 0x3.
     * |        |          |0 = PWM_FCAPDAT0/1 register is the first captured data to memory.
     * |        |          |1 = PWM_RCAPDAT0/1 register is the first captured data to memory.
     * |[4]     |CHSEL0_1  |Select Channel 0/1 to Do PDMA Transfer
     * |        |          |0 = Channel0.
     * |        |          |1 = Channel1.
     * |[8]     |CHEN2_3   |Channel 2/3 PDMA Enable
     * |        |          |0 = Channel 2/3 PDMA function Disabled.
     * |        |          |1 = Channel 2/3 PDMA function Enabled for the channel 2/3 captured data and transfer to memory.
     * |[10:9]  |CAPMOD2_3 |Select PWM_RCAPDAT2/3 or PWM_FCAODAT2/3 to Do PDMA Transfer
     * |        |          |00 = Reserved.
     * |        |          |01 = PWM_RCAPDAT2/3 register.
     * |        |          |10 = PWM_FCAPDAT2/3 register.
     * |        |          |11 = Both PWM_RCAPDAT2/3 and PWM_FCAPDAT2/3 registers.
     * |[11]    |CAPORD2_3 |Capture Channel 2/3 Rising/Falling Order
     * |        |          |Set this bit to determine whether the PWM_RCAPDAT2/3 or PWM_FCAPDAT2/3 register is the first captured data transferred to memory through PDMA when CAPMOD2_3 bits are set to =0x3.
     * |        |          |0 = PWM_FCAPDAT2/3 register is the first captured data to memory.
     * |        |          |1 = PWM_RCAPDAT2/3 register is the first captured data to memory.
     * |[12]    |CHSEL2_3  |Select Channel 2/3 to Do PDMA Transfer
     * |        |          |0 = Channel2.
     * |        |          |1 = Channel3.
     * |[16]    |CHEN4_5   |Channel 4/5 PDMA Enable
     * |        |          |0 = Channel 4/5 PDMA function Disabled.
     * |        |          |1 = Channel 4/5 PDMA function Enabled for the channel 4/5 captured data and transfer to memory.
     * |[18:17] |CAPMOD4_5 |Select PWM_RCAPDAT4/5 or PWM_FCAPDAT4/5 to Do PDMA Transfer
     * |        |          |00 = Reserved.
     * |        |          |01 = PWM_RCAPDAT4/5 register.
     * |        |          |10 = PWM_FCAPDAT4/5 register.
     * |        |          |11 = Both PWM_RCAPDAT4/5 and PWM_FCAPDAT4/5 registers.
     * |[19]    |CAPORD4_5 |Capture Channel 4/5 Rising/Falling Order
     * |        |          |Set this bit to determine whether the PWM_RCAPDAT4/5 or PWM_FCAPDAT4/5 register is the first captured data transferred to memory through PDMA when CAPMOD4_5 bits =are set to 0x3.
     * |        |          |0 = PWM_FCAPDAT4/5 register is the first captured data to memory.
     * |        |          |1 = PWM_RCAPDAT4/5 register is the first captured data to memory.
     * |[20]    |CHSEL4_5  |Select Channel 4/5 to Do PDMA Transfer
     * |        |          |0 = Channel4.
     * |        |          |1 = Channel5.
     * @var PWM_T::PDMACAP0_1
     * Offset: 0x240  PWM Capture Channel 0/1 PDMA Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CAPBUF    |PWM Capture PDMA Register (Read Only)
     * |        |          |This register is use as a buffer to transfer PWM capture rising or falling data to memory by PDMA.
     * @var PWM_T::PDMACAP2_3
     * Offset: 0x244  PWM Capture Channel 2/3 PDMA Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CAPBUF    |PWM Capture PDMA Register (Read Only)
     * |        |          |This register is use as a buffer to transfer PWM capture rising or falling data to memory by PDMA.
     * @var PWM_T::PDMACAP4_5
     * Offset: 0x248  PWM Capture Channel 4/5 PDMA Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CAPBUF    |PWM Capture PDMA Register (Read Only)
     * |        |          |This register is use as a buffer to transfer PWM capture rising or falling data to memory by PDMA.
     * @var PWM_T::CAPIEN
     * Offset: 0x250  PWM Capture Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CAPRIEN0  |PWM Capture Rising Latch Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture rising edge latch interrupt Disabled.
     * |        |          |1 = Capture rising edge latch interrupt Enabled.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CAPRIENn bit must be disabled.
     * |[1]     |CAPRIEN1  |PWM Capture Rising Latch Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture rising edge latch interrupt Disabled.
     * |        |          |1 = Capture rising edge latch interrupt Enabled.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CAPRIENn bit must be disabled.
     * |[2]     |CAPRIEN2  |PWM Capture Rising Latch Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture rising edge latch interrupt Disabled.
     * |        |          |1 = Capture rising edge latch interrupt Enabled.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CAPRIENn bit must be disabled.
     * |[3]     |CAPRIEN3  |PWM Capture Rising Latch Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture rising edge latch interrupt Disabled.
     * |        |          |1 = Capture rising edge latch interrupt Enabled.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CAPRIENn bit must be disabled.
     * |[4]     |CAPRIEN4  |PWM Capture Rising Latch Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture rising edge latch interrupt Disabled.
     * |        |          |1 = Capture rising edge latch interrupt Enabled.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CAPRIENn bit must be disabled.
     * |[5]     |CAPRIEN5  |PWM Capture Rising Latch Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture rising edge latch interrupt Disabled.
     * |        |          |1 = Capture rising edge latch interrupt Enabled.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CAPRIENn bit must be disabled.
     * |[8]     |CAPFIEN0  |PWM Capture Falling Latch Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture falling edge latch interrupt Disabled.
     * |        |          |1 = Capture falling edge latch interrupt Enabled.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CAPFIENn bit must be disabled.
     * |[9]     |CAPFIEN1  |PWM Capture Falling Latch Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture falling edge latch interrupt Disabled.
     * |        |          |1 = Capture falling edge latch interrupt Enabled.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CAPFIENn bit must be disabled.
     * |[10]    |CAPFIEN2  |PWM Capture Falling Latch Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture falling edge latch interrupt Disabled.
     * |        |          |1 = Capture falling edge latch interrupt Enabled.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CAPFIENn bit must be disabled.
     * |[11]    |CAPFIEN3  |PWM Capture Falling Latch Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture falling edge latch interrupt Disabled.
     * |        |          |1 = Capture falling edge latch interrupt Enabled.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CAPFIENn bit must be disabled.
     * |[12]    |CAPFIEN4  |PWM Capture Falling Latch Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture falling edge latch interrupt Disabled.
     * |        |          |1 = Capture falling edge latch interrupt Enabled.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CAPFIENn bit must be disabled.
     * |[13]    |CAPFIEN5  |PWM Capture Falling Latch Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = Capture falling edge latch interrupt Disabled.
     * |        |          |1 = Capture falling edge latch interrupt Enabled.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CAPFIENn bit must be disabled.
     * @var PWM_T::CAPIF
     * Offset: 0x254  PWM Capture Interrupt Flag Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CRLIF0    |PWM Capture Rising Latch Interrupt Flag
     * |        |          |This bit is writing 1 to clear. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No capture rising latch condition happened.
     * |        |          |1 = Capture rising latch condition happened, this flag will be set to high.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CRLIFn bit will cleared by hardware after PDMA transfer data.
     * |[1]     |CRLIF1    |PWM Capture Rising Latch Interrupt Flag
     * |        |          |This bit is writing 1 to clear. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No capture rising latch condition happened.
     * |        |          |1 = Capture rising latch condition happened, this flag will be set to high.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CRLIFn bit will cleared by hardware after PDMA transfer data.
     * |[2]     |CRLIF2    |PWM Capture Rising Latch Interrupt Flag
     * |        |          |This bit is writing 1 to clear. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No capture rising latch condition happened.
     * |        |          |1 = Capture rising latch condition happened, this flag will be set to high.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CRLIFn bit will cleared by hardware after PDMA transfer data.
     * |[3]     |CRLIF3    |PWM Capture Rising Latch Interrupt Flag
     * |        |          |This bit is writing 1 to clear. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No capture rising latch condition happened.
     * |        |          |1 = Capture rising latch condition happened, this flag will be set to high.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CRLIFn bit will cleared by hardware after PDMA transfer data.
     * |[4]     |CRLIF4    |PWM Capture Rising Latch Interrupt Flag
     * |        |          |This bit is writing 1 to clear. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No capture rising latch condition happened.
     * |        |          |1 = Capture rising latch condition happened, this flag will be set to high.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CRLIFn bit will cleared by hardware after PDMA transfer data.
     * |[5]     |CRLIF5    |PWM Capture Rising Latch Interrupt Flag
     * |        |          |This bit is writing 1 to clear. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No capture rising latch condition happened.
     * |        |          |1 = Capture rising latch condition happened, this flag will be set to high.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CRLIFn bit will cleared by hardware after PDMA transfer data.
     * |[8]     |CFLIF0    |PWM Capture Falling Latch Interrupt Flag
     * |        |          |This bit is writing 1 to clear. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No capture falling latch condition happened.
     * |        |          |1 = Capture falling latch condition happened, this flag will be set to high.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CFLIFn bit will cleared by hardware after PDMA transfer data.
     * |[9]     |CFLIF1    |PWM Capture Falling Latch Interrupt Flag
     * |        |          |This bit is writing 1 to clear. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No capture falling latch condition happened.
     * |        |          |1 = Capture falling latch condition happened, this flag will be set to high.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CFLIFn bit will cleared by hardware after PDMA transfer data.
     * |[10]    |CFLIF2    |PWM Capture Falling Latch Interrupt Flag
     * |        |          |This bit is writing 1 to clear. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No capture falling latch condition happened.
     * |        |          |1 = Capture falling latch condition happened, this flag will be set to high.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CFLIFn bit will cleared by hardware after PDMA transfer data.
     * |[11]    |CFLIF3    |PWM Capture Falling Latch Interrupt Flag
     * |        |          |This bit is writing 1 to clear. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No capture falling latch condition happened.
     * |        |          |1 = Capture falling latch condition happened, this flag will be set to high.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CFLIFn bit will cleared by hardware after PDMA transfer data.
     * |[12]    |CFLIF4    |PWM Capture Falling Latch Interrupt Flag
     * |        |          |This bit is writing 1 to clear. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No capture falling latch condition happened.
     * |        |          |1 = Capture falling latch condition happened, this flag will be set to high.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CFLIFn bit will cleared by hardware after PDMA transfer data.
     * |[13]    |CFLIF5    |PWM Capture Falling Latch Interrupt Flag
     * |        |          |This bit is writing 1 to clear. Each bit n controls the corresponding PWM channel n.
     * |        |          |0 = No capture falling latch condition happened.
     * |        |          |1 = Capture falling latch condition happened, this flag will be set to high.
     * |        |          |Note: When Capture with PDMA operating, corresponding channel CFLIFn bit will cleared by hardware after PDMA transfer data.
     * @var PWM_T::PBUF
     * Offset: 0x304/0x308/0x30C/0x310/0x314/0x318  PWM PERIOD0/PERIOD1/PERIOD2/PERIOD3/PERIOD4/PERIOD5 Buffer
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |PBUF      |PWM Period Register Buffer (Read Only)
     * |        |          |Used as PERIOD active register.
     * @var PWM_T::CMPBUF
     * Offset: 0x31C/0x320/0x324/0x328/0x32C/0x330  PWM CMPDAT0/CMPDAT1/CMPDAT2/CMPDAT3/CMPDAT4/CMPDAT5 Buffer
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CMPBUF    |PWM Comparator Register Buffer (Read Only)
     * |        |          |Used as CMPDAT active register.
     * @var PWM_T::CPSCBUF0_1
     * Offset: 0x334  PWM CLKPSC0_1 Buffer
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |CPSCBUF   |PWM Counter Clock Pre-scale Buffer
     * |        |          |Use as PWM counter clock pre-scare active register.
     * @var PWM_T::CPSCBUF2_3
     * Offset: 0x338  PWM CLKPSC2_3 Buffer
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |CPSCBUF   |PWM Counter Clock Pre-scale Buffer
     * |        |          |Use as PWM counter clock pre-scare active register.
     * @var PWM_T::CPSCBUF4_5
     * Offset: 0x33C  PWM CLKPSC4_5 Buffer
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |CPSCBUF   |PWM Counter Clock Pre-scale Buffer
     * |        |          |Use as PWM counter clock pre-scare active register.
     * @var PWM_T::FTCBUF0_1
     * Offset: 0x340  PWM FTCMPDAT0_1 Buffer
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FTCMPBUF  |PWM FTCMPDAT Buffer (Read Only)
     * |        |          |Used as FTCMPDAT active register.
     * @var PWM_T::FTCBUF2_3
     * Offset: 0x344  PWM FTCMPDAT2_3 Buffer
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FTCMPBUF  |PWM FTCMPDAT Buffer (Read Only)
     * |        |          |Used as FTCMPDAT active register.
     * @var PWM_T::FTCBUF4_5
     * Offset: 0x348  PWM FTCMPDAT4_5 Buffer
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FTCMPBUF  |PWM FTCMPDAT Buffer (Read Only)
     * |        |          |Used as FTCMPDAT active register.
     * @var PWM_T::FTCI
     * Offset: 0x34C  PWM FTCMPDAT Indicator Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |FTCMU0    |PWM FTCMPDAT Up Indicator
     * |        |          |Indicator will be set to high when FTCMPDATn(PWM_CTCMPDATn[15:0]) bits equal to PERIOD(PWM_PERIODn[15:0]) bitsn and DIRF(PWM_CNTn[16])= bit is 1, software can write 1 to clear this bit.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |[1]     |FTCMU2    |PWM FTCMPDAT Up Indicator
     * |        |          |Indicator will be set to high when FTCMPDATn(PWM_CTCMPDATn[15:0]) bits equal to PERIOD(PWM_PERIODn[15:0]) bitsn and DIRF(PWM_CNTn[16])= bit is 1, software can write 1 to clear this bit.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |[2]     |FTCMU4    |PWM FTCMPDAT Up Indicator
     * |        |          |Indicator will be set to high when FTCMPDATn(PWM_CTCMPDATn[15:0]) bits equal to PERIOD(PWM_PERIODn[15:0]) bitsn and DIRF(PWM_CNTn[16])= bit is 1, software can write 1 to clear this bit.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |[8]     |FTCMD0    |PWM FTCMPDAT Down Indicator
     * |        |          |Indicator will be set to high when FTCMPDATn(PWM_FTCMPDATn[15:0]) bits equal to PERIODn(PWM_PERIODn[15:0]) bits and DIRF(PWM_CNTn[16]) bit= is 0, software can write 1 to clear this bit.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |[9]     |FTCMD2    |PWM FTCMPDAT Down Indicator
     * |        |          |Indicator will be set to high when FTCMPDATn(PWM_FTCMPDATn[15:0]) bits equal to PERIODn(PWM_PERIODn[15:0]) bits and DIRF(PWM_CNTn[16]) bit= is 0, software can write 1 to clear this bit.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     * |[10]    |FTCMD4    |PWM FTCMPDAT Down Indicator
     * |        |          |Indicator will be set to high when FTCMPDATn(PWM_FTCMPDATn[15:0]) bits equal to PERIODn(PWM_PERIODn[15:0]) bits and DIRF(PWM_CNTn[16]) bit= is 0, software can write 1 to clear this bit.
     * |        |          |Each bit n controls the corresponding PWM channel n.
     */

    __IO uint32_t CTL0;                  /*!< [0x0000] PWM Control Register 0                                           */
    __IO uint32_t CTL1;                  /*!< [0x0004] PWM Control Register 1                                           */
    __IO uint32_t SYNC;                  /*!< [0x0008] PWM Synchronization Register                                     */
    __IO uint32_t SWSYNC;                /*!< [0x000c] PWM Software Control Synchronization Register                    */
    __IO uint32_t CLKSRC;                /*!< [0x0010] PWM Clock Source Register                                        */
    __IO uint32_t CLKPSC0_1;             /*!< [0x0014] PWM Clock Pre-scale Register 0/1                                 */
    __IO uint32_t CLKPSC2_3;             /*!< [0x0018] PWM Clock Pre-scale Register 2/3                                 */
    __IO uint32_t CLKPSC4_5;             /*!< [0x001c] PWM Clock Pre-scale Register 4/5                                 */
    __IO uint32_t CNTEN;                 /*!< [0x0020] PWM Counter Enable Register                                      */
    __IO uint32_t CNTCLR;                /*!< [0x0024] PWM Clear Counter Register                                       */
    __IO uint32_t LOAD;                  /*!< [0x0028] PWM Load Register                                                */
    __I  uint32_t RESERVE0[1];
    __IO uint32_t PERIOD[6];             /*!< [0x0030~0x0044]  PWM Period Register 0~5                                  */
    __I  uint32_t RESERVE1[2];
    __IO uint32_t CMPDAT[6];             /*!< [0x0050~0x0064]  PWM Comparator Register 0~5                              */
    __I  uint32_t RESERVE2[2];
    __IO uint32_t DTCTL0_1;              /*!< [0x0070] PWM Dead-Time Control Register 0/1                               */
    __IO uint32_t DTCTL2_3;              /*!< [0x0074] PWM Dead-Time Control Register 2/3                               */
    __IO uint32_t DTCTL4_5;              /*!< [0x0078] PWM Dead-Time Control Register 4/5                               */
    __I  uint32_t RESERVE3[1];
    __IO uint32_t PHS0_1;                /*!< [0x0080] PWM Counter Phase Register 0/1                                   */
    __IO uint32_t PHS2_3;                /*!< [0x0084] PWM Counter Phase Register 2/3                                   */
    __IO uint32_t PHS4_5;                /*!< [0x0088] PWM Counter Phase Register 4/5                                   */
    __I  uint32_t RESERVE4[1];
    __I  uint32_t CNT[6];                /*!< [0x0090~0x00A4  PWM Counter Register 0~5                                  */
    __I  uint32_t RESERVE5[2];
    __IO uint32_t WGCTL0;                /*!< [0x00b0] PWM Generation Register 0                                        */
    __IO uint32_t WGCTL1;                /*!< [0x00b4] PWM Generation Register 1                                        */
    __IO uint32_t MSKEN;                 /*!< [0x00b8] PWM Mask Enable Register                                         */
    __IO uint32_t MSK;                   /*!< [0x00bc] PWM Mask Data Register                                           */
    __IO uint32_t BNF;                   /*!< [0x00c0] PWM Brake Noise Filter Register                                  */
    __IO uint32_t FAILBRK;               /*!< [0x00c4] PWM System Fail Brake Control Register                           */
    __IO uint32_t BRKCTL0_1;             /*!< [0x00c8] PWM Brake Edge Detect Control Register 0/1                       */
    __IO uint32_t BRKCTL2_3;             /*!< [0x00cc] PWM Brake Edge Detect Control Register 2/3                       */
    __IO uint32_t BRKCTL4_5;             /*!< [0x00d0] PWM Brake Edge Detect Control Register 4/5                       */
    __IO uint32_t POLCTL;                /*!< [0x00d4] PWM Pin Polar Inverse Register                                   */
    __IO uint32_t POEN;                  /*!< [0x00d8] PWM Output Enable Register                                       */
    __O  uint32_t SWBRK;                 /*!< [0x00dc] PWM Software Brake Control Register                              */
    __IO uint32_t INTEN0;                /*!< [0x00e0] PWM Interrupt Enable Register 0                                  */
    __IO uint32_t INTEN1;                /*!< [0x00e4] PWM Interrupt Enable Register 1                                  */
    __IO uint32_t INTSTS0;               /*!< [0x00e8] PWM Interrupt Flag Register 0                                    */
    __IO uint32_t INTSTS1;               /*!< [0x00ec] PWM Interrupt Flag Register 1                                    */
    __IO uint32_t IFA;                   /*!< [0x00f0] PWM Interrupt Flag Accumulator Register                          */
    __I  uint32_t RESERVE6[1];
    __IO uint32_t ADCTS0;                /*!< [0x00f8] PWM Trigger ADC Source Select Register 0                         */
    __IO uint32_t ADCTS1;                /*!< [0x00fc] PWM Trigger ADC Source Select Register 1                         */
    __IO uint32_t FTCMPDAT0_1;           /*!< [0x0100] PWM Free Trigger Compare Register 0/1                            */
    __IO uint32_t FTCMPDAT2_3;           /*!< [0x0104] PWM Free Trigger Compare Register 2/3                            */
    __IO uint32_t FTCMPDAT4_5;           /*!< [0x0108] PWM Free Trigger Compare Register 4/5                            */
    __I  uint32_t RESERVE7[1];
    __IO uint32_t SSCTL;                 /*!< [0x0110] PWM Synchronous Start Control Register                           */
    __O  uint32_t SSTRG;                 /*!< [0x0114] PWM Synchronous Start Trigger Register                           */
    __IO uint32_t LEBCTL;                /*!< [0x0118] PWM Leading Edge Blanking Control Register                       */
    __IO uint32_t LEBCNT;                /*!< [0x011c] PWM Leading Edge Blanking Counter Register                       */
    __IO uint32_t STATUS;                /*!< [0x0120] PWM Status Register                                              */
    __I  uint32_t RESERVE8[55];
    __IO uint32_t CAPINEN;               /*!< [0x0200] PWM Capture Input Enable Register                                */
    __IO uint32_t CAPCTL;                /*!< [0x0204] PWM Capture Control Register                                     */
    __I  uint32_t CAPSTS;                /*!< [0x0208] PWM Capture Status Register                                      */
    __I  uint32_t RCAPDAT0;              /*!< [0x020c] PWM Rising Capture Data Register 0                               */
    __I  uint32_t FCAPDAT0;              /*!< [0x0210] PWM Falling Capture Data Register 0                              */
    __I  uint32_t RCAPDAT1;              /*!< [0x0214] PWM Rising Capture Data Register 1                               */
    __I  uint32_t FCAPDAT1;              /*!< [0x0218] PWM Falling Capture Data Register 1                              */
    __I  uint32_t RCAPDAT2;              /*!< [0x021c] PWM Rising Capture Data Register 2                               */
    __I  uint32_t FCAPDAT2;              /*!< [0x0220] PWM Falling Capture Data Register 2                              */
    __I  uint32_t RCAPDAT3;              /*!< [0x0224] PWM Rising Capture Data Register 3                               */
    __I  uint32_t FCAPDAT3;              /*!< [0x0228] PWM Falling Capture Data Register 3                              */
    __I  uint32_t RCAPDAT4;              /*!< [0x022c] PWM Rising Capture Data Register 4                               */
    __I  uint32_t FCAPDAT4;              /*!< [0x0230] PWM Falling Capture Data Register 4                              */
    __I  uint32_t RCAPDAT5;              /*!< [0x0234] PWM Rising Capture Data Register 5                               */
    __I  uint32_t FCAPDAT5;              /*!< [0x0238] PWM Falling Capture Data Register 5                              */
    __IO uint32_t PDMACTL;               /*!< [0x023c] PWM PDMA Control Register                                        */
    __I  uint32_t PDMACAP0_1;            /*!< [0x0240] PWM Capture Channel 0/1 PDMA Register                            */
    __I  uint32_t PDMACAP2_3;            /*!< [0x0244] PWM Capture Channel 2/3 PDMA Register                            */
    __I  uint32_t PDMACAP4_5;            /*!< [0x0248] PWM Capture Channel 4/5 PDMA Register                            */
    __I  uint32_t RESERVE9[1];
    __IO uint32_t CAPIEN;                /*!< [0x0250] PWM Capture Interrupt Enable Register                            */
    __IO uint32_t CAPIF;                 /*!< [0x0254] PWM Capture Interrupt Flag Register                              */
    __I  uint32_t RESERVE10[43];
    __I  uint32_t PBUF[6];               /*!< [0x0304~0x0318  PWM PERIOD0~5 Buffer                                      */
    __I  uint32_t CMPBUF[6];             /*!< [0x031C~0x0330  PWM CMPDAT0~5 Buffer                                      */
    __I  uint32_t CPSCBUF0_1;            /*!< [0x0334] PWM CLKPSC0_1 Buffer                                             */
    __I  uint32_t CPSCBUF2_3;            /*!< [0x0338] PWM CLKPSC2_3 Buffer                                             */
    __I  uint32_t CPSCBUF4_5;            /*!< [0x033c] PWM CLKPSC4_5 Buffer                                             */
    __I  uint32_t FTCBUF0_1;             /*!< [0x0340] PWM FTCMPDAT0_1 Buffer                                           */
    __I  uint32_t FTCBUF2_3;             /*!< [0x0344] PWM FTCMPDAT2_3 Buffer                                           */
    __I  uint32_t FTCBUF4_5;             /*!< [0x0348] PWM FTCMPDAT4_5 Buffer                                           */
    __IO uint32_t FTCI;                  /*!< [0x034c] PWM FTCMPDAT Indicator Register                                  */


} PWM_T;

/**
    @addtogroup PWM_CONST PWM Bit Field Definition
    Constant Definitions for PWM Controller
@{ */

#define PWM_CTL0_CTRLD0_Pos              (0)                                               /*!< PWM_T::CTL0: CTRLD0 Position           */
#define PWM_CTL0_CTRLD0_Msk              (0x1ul << PWM_CTL0_CTRLD0_Pos)                    /*!< PWM_T::CTL0: CTRLD0 Mask               */

#define PWM_CTL0_CTRLD1_Pos              (1)                                               /*!< PWM_T::CTL0: CTRLD1 Position           */
#define PWM_CTL0_CTRLD1_Msk              (0x1ul << PWM_CTL0_CTRLD1_Pos)                    /*!< PWM_T::CTL0: CTRLD1 Mask               */

#define PWM_CTL0_CTRLD2_Pos              (2)                                               /*!< PWM_T::CTL0: CTRLD2 Position           */
#define PWM_CTL0_CTRLD2_Msk              (0x1ul << PWM_CTL0_CTRLD2_Pos)                    /*!< PWM_T::CTL0: CTRLD2 Mask               */

#define PWM_CTL0_CTRLD3_Pos              (3)                                               /*!< PWM_T::CTL0: CTRLD3 Position           */
#define PWM_CTL0_CTRLD3_Msk              (0x1ul << PWM_CTL0_CTRLD3_Pos)                    /*!< PWM_T::CTL0: CTRLD3 Mask               */

#define PWM_CTL0_CTRLD4_Pos              (4)                                               /*!< PWM_T::CTL0: CTRLD4 Position           */
#define PWM_CTL0_CTRLD4_Msk              (0x1ul << PWM_CTL0_CTRLD4_Pos)                    /*!< PWM_T::CTL0: CTRLD4 Mask               */

#define PWM_CTL0_CTRLD5_Pos              (5)                                               /*!< PWM_T::CTL0: CTRLD5 Position           */
#define PWM_CTL0_CTRLD5_Msk              (0x1ul << PWM_CTL0_CTRLD5_Pos)                    /*!< PWM_T::CTL0: CTRLD5 Mask               */

#define PWM_CTL0_WINLDEN0_Pos            (8)                                               /*!< PWM_T::CTL0: WINLDEN0 Position         */
#define PWM_CTL0_WINLDEN0_Msk            (0x1ul << PWM_CTL0_WINLDEN0_Pos)                  /*!< PWM_T::CTL0: WINLDEN0 Mask             */

#define PWM_CTL0_WINLDEN1_Pos            (9)                                               /*!< PWM_T::CTL0: WINLDEN1 Position         */
#define PWM_CTL0_WINLDEN1_Msk            (0x1ul << PWM_CTL0_WINLDEN1_Pos)                  /*!< PWM_T::CTL0: WINLDEN1 Mask             */

#define PWM_CTL0_WINLDEN2_Pos            (10)                                              /*!< PWM_T::CTL0: WINLDEN2 Position         */
#define PWM_CTL0_WINLDEN2_Msk            (0x1ul << PWM_CTL0_WINLDEN2_Pos)                  /*!< PWM_T::CTL0: WINLDEN2 Mask             */

#define PWM_CTL0_WINLDEN3_Pos            (11)                                              /*!< PWM_T::CTL0: WINLDEN3 Position         */
#define PWM_CTL0_WINLDEN3_Msk            (0x1ul << PWM_CTL0_WINLDEN3_Pos)                  /*!< PWM_T::CTL0: WINLDEN3 Mask             */

#define PWM_CTL0_WINLDEN4_Pos            (12)                                              /*!< PWM_T::CTL0: WINLDEN4 Position         */
#define PWM_CTL0_WINLDEN4_Msk            (0x1ul << PWM_CTL0_WINLDEN4_Pos)                  /*!< PWM_T::CTL0: WINLDEN4 Mask             */

#define PWM_CTL0_WINLDEN5_Pos            (13)                                              /*!< PWM_T::CTL0: WINLDEN5 Position         */
#define PWM_CTL0_WINLDEN5_Msk            (0x1ul << PWM_CTL0_WINLDEN5_Pos)                  /*!< PWM_T::CTL0: WINLDEN5 Mask             */

#define PWM_CTL0_IMMLDEN0_Pos            (16)                                              /*!< PWM_T::CTL0: IMMLDEN0 Position         */
#define PWM_CTL0_IMMLDEN0_Msk            (0x1ul << PWM_CTL0_IMMLDEN0_Pos)                  /*!< PWM_T::CTL0: IMMLDEN0 Mask             */

#define PWM_CTL0_IMMLDEN1_Pos            (17)                                              /*!< PWM_T::CTL0: IMMLDEN1 Position         */
#define PWM_CTL0_IMMLDEN1_Msk            (0x1ul << PWM_CTL0_IMMLDEN1_Pos)                  /*!< PWM_T::CTL0: IMMLDEN1 Mask             */

#define PWM_CTL0_IMMLDEN2_Pos            (18)                                              /*!< PWM_T::CTL0: IMMLDEN2 Position         */
#define PWM_CTL0_IMMLDEN2_Msk            (0x1ul << PWM_CTL0_IMMLDEN2_Pos)                  /*!< PWM_T::CTL0: IMMLDEN2 Mask             */

#define PWM_CTL0_IMMLDEN3_Pos            (19)                                              /*!< PWM_T::CTL0: IMMLDEN3 Position         */
#define PWM_CTL0_IMMLDEN3_Msk            (0x1ul << PWM_CTL0_IMMLDEN3_Pos)                  /*!< PWM_T::CTL0: IMMLDEN3 Mask             */

#define PWM_CTL0_IMMLDEN4_Pos            (20)                                              /*!< PWM_T::CTL0: IMMLDEN4 Position         */
#define PWM_CTL0_IMMLDEN4_Msk            (0x1ul << PWM_CTL0_IMMLDEN4_Pos)                  /*!< PWM_T::CTL0: IMMLDEN4 Mask             */

#define PWM_CTL0_IMMLDEN5_Pos            (21)                                              /*!< PWM_T::CTL0: IMMLDEN5 Position         */
#define PWM_CTL0_IMMLDEN5_Msk            (0x1ul << PWM_CTL0_IMMLDEN5_Pos)                  /*!< PWM_T::CTL0: IMMLDEN5 Mask             */

#define PWM_CTL0_GROUPEN_Pos             (24)                                              /*!< PWM_T::CTL0: GROUPEN Position          */
#define PWM_CTL0_GROUPEN_Msk             (0x1ul << PWM_CTL0_GROUPEN_Pos)                   /*!< PWM_T::CTL0: GROUPEN Mask              */

#define PWM_CTL0_DBGHALT_Pos             (30)                                              /*!< PWM_T::CTL0: DBGHALT Position          */
#define PWM_CTL0_DBGHALT_Msk             (0x1ul << PWM_CTL0_DBGHALT_Pos)                   /*!< PWM_T::CTL0: DBGHALT Mask              */

#define PWM_CTL0_DBGTRIOFF_Pos           (31)                                              /*!< PWM_T::CTL0: DBGTRIOFF Position        */
#define PWM_CTL0_DBGTRIOFF_Msk           (0x1ul << PWM_CTL0_DBGTRIOFF_Pos)                 /*!< PWM_T::CTL0: DBGTRIOFF Mask            */

#define PWM_CTL1_CNTTYPE0_Pos            (0)                                               /*!< PWM_T::CTL1: CNTTYPE0 Position         */
#define PWM_CTL1_CNTTYPE0_Msk            (0x3ul << PWM_CTL1_CNTTYPE0_Pos)                  /*!< PWM_T::CTL1: CNTTYPE0 Mask             */

#define PWM_CTL1_CNTTYPE1_Pos            (2)                                               /*!< PWM_T::CTL1: CNTTYPE1 Position         */
#define PWM_CTL1_CNTTYPE1_Msk            (0x3ul << PWM_CTL1_CNTTYPE1_Pos)                  /*!< PWM_T::CTL1: CNTTYPE1 Mask             */

#define PWM_CTL1_CNTTYPE2_Pos            (4)                                               /*!< PWM_T::CTL1: CNTTYPE2 Position         */
#define PWM_CTL1_CNTTYPE2_Msk            (0x3ul << PWM_CTL1_CNTTYPE2_Pos)                  /*!< PWM_T::CTL1: CNTTYPE2 Mask             */

#define PWM_CTL1_CNTTYPE3_Pos            (6)                                               /*!< PWM_T::CTL1: CNTTYPE3 Position         */
#define PWM_CTL1_CNTTYPE3_Msk            (0x3ul << PWM_CTL1_CNTTYPE3_Pos)                  /*!< PWM_T::CTL1: CNTTYPE3 Mask             */

#define PWM_CTL1_CNTTYPE4_Pos            (8)                                               /*!< PWM_T::CTL1: CNTTYPE4 Position         */
#define PWM_CTL1_CNTTYPE4_Msk            (0x3ul << PWM_CTL1_CNTTYPE4_Pos)                  /*!< PWM_T::CTL1: CNTTYPE4 Mask             */

#define PWM_CTL1_CNTTYPE5_Pos            (10)                                              /*!< PWM_T::CTL1: CNTTYPE5 Position         */
#define PWM_CTL1_CNTTYPE5_Msk            (0x3ul << PWM_CTL1_CNTTYPE5_Pos)                  /*!< PWM_T::CTL1: CNTTYPE5 Mask             */

#define PWM_CTL1_CNTMODE0_Pos            (16)                                              /*!< PWM_T::CTL1: CNTMODE0 Position         */
#define PWM_CTL1_CNTMODE0_Msk            (0x1ul << PWM_CTL1_CNTMODE0_Pos)                  /*!< PWM_T::CTL1: CNTMODE0 Mask             */

#define PWM_CTL1_CNTMODE1_Pos            (17)                                              /*!< PWM_T::CTL1: CNTMODE1 Position         */
#define PWM_CTL1_CNTMODE1_Msk            (0x1ul << PWM_CTL1_CNTMODE1_Pos)                  /*!< PWM_T::CTL1: CNTMODE1 Mask             */

#define PWM_CTL1_CNTMODE2_Pos            (18)                                              /*!< PWM_T::CTL1: CNTMODE2 Position         */
#define PWM_CTL1_CNTMODE2_Msk            (0x1ul << PWM_CTL1_CNTMODE2_Pos)                  /*!< PWM_T::CTL1: CNTMODE2 Mask             */

#define PWM_CTL1_CNTMODE3_Pos            (19)                                              /*!< PWM_T::CTL1: CNTMODE3 Position         */
#define PWM_CTL1_CNTMODE3_Msk            (0x1ul << PWM_CTL1_CNTMODE3_Pos)                  /*!< PWM_T::CTL1: CNTMODE3 Mask             */

#define PWM_CTL1_CNTMODE4_Pos            (20)                                              /*!< PWM_T::CTL1: CNTMODE4 Position         */
#define PWM_CTL1_CNTMODE4_Msk            (0x1ul << PWM_CTL1_CNTMODE4_Pos)                  /*!< PWM_T::CTL1: CNTMODE4 Mask             */

#define PWM_CTL1_CNTMODE5_Pos            (21)                                              /*!< PWM_T::CTL1: CNTMODE5 Position         */
#define PWM_CTL1_CNTMODE5_Msk            (0x1ul << PWM_CTL1_CNTMODE5_Pos)                  /*!< PWM_T::CTL1: CNTMODE5 Mask             */

#define PWM_CTL1_OUTMODE0_Pos            (24)                                              /*!< PWM_T::CTL1: OUTMODE0 Position         */
#define PWM_CTL1_OUTMODE0_Msk            (0x1ul << PWM_CTL1_OUTMODE0_Pos)                  /*!< PWM_T::CTL1: OUTMODE0 Mask             */

#define PWM_CTL1_OUTMODE2_Pos            (25)                                              /*!< PWM_T::CTL1: OUTMODE2 Position         */
#define PWM_CTL1_OUTMODE2_Msk            (0x1ul << PWM_CTL1_OUTMODE2_Pos)                  /*!< PWM_T::CTL1: OUTMODE2 Mask             */

#define PWM_CTL1_OUTMODE4_Pos            (26)                                              /*!< PWM_T::CTL1: OUTMODE4 Position         */
#define PWM_CTL1_OUTMODE4_Msk            (0x1ul << PWM_CTL1_OUTMODE4_Pos)                  /*!< PWM_T::CTL1: OUTMODE4 Mask             */

#define PWM_SYNC_PHSEN0_Pos              (0)                                               /*!< PWM_T::SYNC: PHSEN0 Position           */
#define PWM_SYNC_PHSEN0_Msk              (0x1ul << PWM_SYNC_PHSEN0_Pos)                    /*!< PWM_T::SYNC: PHSEN0 Mask               */

#define PWM_SYNC_PHSEN2_Pos              (1)                                               /*!< PWM_T::SYNC: PHSEN2 Position           */
#define PWM_SYNC_PHSEN2_Msk              (0x1ul << PWM_SYNC_PHSEN2_Pos)                    /*!< PWM_T::SYNC: PHSEN2 Mask               */

#define PWM_SYNC_PHSEN4_Pos              (2)                                               /*!< PWM_T::SYNC: PHSEN4 Position           */
#define PWM_SYNC_PHSEN4_Msk              (0x1ul << PWM_SYNC_PHSEN4_Pos)                    /*!< PWM_T::SYNC: PHSEN4 Mask               */

#define PWM_SYNC_SINSRC0_Pos             (8)                                               /*!< PWM_T::SYNC: SINSRC0 Position          */
#define PWM_SYNC_SINSRC0_Msk             (0x3ul << PWM_SYNC_SINSRC0_Pos)                   /*!< PWM_T::SYNC: SINSRC0 Mask              */

#define PWM_SYNC_SINSRC2_Pos             (10)                                              /*!< PWM_T::SYNC: SINSRC2 Position          */
#define PWM_SYNC_SINSRC2_Msk             (0x3ul << PWM_SYNC_SINSRC2_Pos)                   /*!< PWM_T::SYNC: SINSRC2 Mask              */

#define PWM_SYNC_SINSRC4_Pos             (12)                                              /*!< PWM_T::SYNC: SINSRC4 Position          */
#define PWM_SYNC_SINSRC4_Msk             (0x3ul << PWM_SYNC_SINSRC4_Pos)                   /*!< PWM_T::SYNC: SINSRC4 Mask              */

#define PWM_SYNC_SNFLTEN_Pos             (16)                                              /*!< PWM_T::SYNC: SNFLTEN Position          */
#define PWM_SYNC_SNFLTEN_Msk             (0x1ul << PWM_SYNC_SNFLTEN_Pos)                   /*!< PWM_T::SYNC: SNFLTEN Mask              */

#define PWM_SYNC_SFLTCSEL_Pos            (17)                                              /*!< PWM_T::SYNC: SFLTCSEL Position         */
#define PWM_SYNC_SFLTCSEL_Msk            (0x7ul << PWM_SYNC_SFLTCSEL_Pos)                  /*!< PWM_T::SYNC: SFLTCSEL Mask             */

#define PWM_SYNC_SFLTCNT_Pos             (20)                                              /*!< PWM_T::SYNC: SFLTCNT Position          */
#define PWM_SYNC_SFLTCNT_Msk             (0x7ul << PWM_SYNC_SFLTCNT_Pos)                   /*!< PWM_T::SYNC: SFLTCNT Mask              */

#define PWM_SYNC_SINPINV_Pos             (23)                                              /*!< PWM_T::SYNC: SINPINV Position          */
#define PWM_SYNC_SINPINV_Msk             (0x1ul << PWM_SYNC_SINPINV_Pos)                   /*!< PWM_T::SYNC: SINPINV Mask              */

#define PWM_SYNC_PHSDIR0_Pos             (24)                                              /*!< PWM_T::SYNC: PHSDIR0 Position          */
#define PWM_SYNC_PHSDIR0_Msk             (0x1ul << PWM_SYNC_PHSDIR0_Pos)                   /*!< PWM_T::SYNC: PHSDIR0 Mask              */

#define PWM_SYNC_PHSDIR2_Pos             (25)                                              /*!< PWM_T::SYNC: PHSDIR2 Position          */
#define PWM_SYNC_PHSDIR2_Msk             (0x1ul << PWM_SYNC_PHSDIR2_Pos)                   /*!< PWM_T::SYNC: PHSDIR2 Mask              */

#define PWM_SYNC_PHSDIR4_Pos             (26)                                              /*!< PWM_T::SYNC: PHSDIR4 Position          */
#define PWM_SYNC_PHSDIR4_Msk             (0x1ul << PWM_SYNC_PHSDIR4_Pos)                   /*!< PWM_T::SYNC: PHSDIR4 Mask              */

#define PWM_SWSYNC_SWSYNC0_Pos           (0)                                               /*!< PWM_T::SWSYNC: SWSYNC0 Position        */
#define PWM_SWSYNC_SWSYNC0_Msk           (0x1ul << PWM_SWSYNC_SWSYNC0_Pos)                 /*!< PWM_T::SWSYNC: SWSYNC0 Mask            */

#define PWM_SWSYNC_SWSYNC2_Pos           (1)                                               /*!< PWM_T::SWSYNC: SWSYNC2 Position        */
#define PWM_SWSYNC_SWSYNC2_Msk           (0x1ul << PWM_SWSYNC_SWSYNC2_Pos)                 /*!< PWM_T::SWSYNC: SWSYNC2 Mask            */

#define PWM_SWSYNC_SWSYNC4_Pos           (2)                                               /*!< PWM_T::SWSYNC: SWSYNC4 Position        */
#define PWM_SWSYNC_SWSYNC4_Msk           (0x1ul << PWM_SWSYNC_SWSYNC4_Pos)                 /*!< PWM_T::SWSYNC: SWSYNC4 Mask            */

#define PWM_CLKSRC_ECLKSRC0_Pos          (0)                                               /*!< PWM_T::CLKSRC: ECLKSRC0 Position       */
#define PWM_CLKSRC_ECLKSRC0_Msk          (0x7ul << PWM_CLKSRC_ECLKSRC0_Pos)                /*!< PWM_T::CLKSRC: ECLKSRC0 Mask           */

#define PWM_CLKSRC_ECLKSRC2_Pos          (8)                                               /*!< PWM_T::CLKSRC: ECLKSRC2 Position       */
#define PWM_CLKSRC_ECLKSRC2_Msk          (0x7ul << PWM_CLKSRC_ECLKSRC2_Pos)                /*!< PWM_T::CLKSRC: ECLKSRC2 Mask           */

#define PWM_CLKSRC_ECLKSRC4_Pos          (16)                                              /*!< PWM_T::CLKSRC: ECLKSRC4 Position       */
#define PWM_CLKSRC_ECLKSRC4_Msk          (0x7ul << PWM_CLKSRC_ECLKSRC4_Pos)                /*!< PWM_T::CLKSRC: ECLKSRC4 Mask           */

#define PWM_CLKPSC0_1_CLKPSC_Pos         (0)                                               /*!< PWM_T::CLKPSC0_1: CLKPSC Position      */
#define PWM_CLKPSC0_1_CLKPSC_Msk         (0xffful << PWM_CLKPSC0_1_CLKPSC_Pos)             /*!< PWM_T::CLKPSC0_1: CLKPSC Mask          */

#define PWM_CLKPSC2_3_CLKPSC_Pos         (0)                                               /*!< PWM_T::CLKPSC2_3: CLKPSC Position      */
#define PWM_CLKPSC2_3_CLKPSC_Msk         (0xffful << PWM_CLKPSC2_3_CLKPSC_Pos)             /*!< PWM_T::CLKPSC2_3: CLKPSC Mask          */

#define PWM_CLKPSC4_5_CLKPSC_Pos         (0)                                               /*!< PWM_T::CLKPSC4_5: CLKPSC Position      */
#define PWM_CLKPSC4_5_CLKPSC_Msk         (0xffful << PWM_CLKPSC4_5_CLKPSC_Pos)             /*!< PWM_T::CLKPSC4_5: CLKPSC Mask          */

#define PWM_CNTEN_CNTEN0_Pos             (0)                                               /*!< PWM_T::CNTEN: CNTEN0 Position          */
#define PWM_CNTEN_CNTEN0_Msk             (0x1ul << PWM_CNTEN_CNTEN0_Pos)                   /*!< PWM_T::CNTEN: CNTEN0 Mask              */

#define PWM_CNTEN_CNTEN1_Pos             (1)                                               /*!< PWM_T::CNTEN: CNTEN1 Position          */
#define PWM_CNTEN_CNTEN1_Msk             (0x1ul << PWM_CNTEN_CNTEN1_Pos)                   /*!< PWM_T::CNTEN: CNTEN1 Mask              */

#define PWM_CNTEN_CNTEN2_Pos             (2)                                               /*!< PWM_T::CNTEN: CNTEN2 Position          */
#define PWM_CNTEN_CNTEN2_Msk             (0x1ul << PWM_CNTEN_CNTEN2_Pos)                   /*!< PWM_T::CNTEN: CNTEN2 Mask              */

#define PWM_CNTEN_CNTEN3_Pos             (3)                                               /*!< PWM_T::CNTEN: CNTEN3 Position          */
#define PWM_CNTEN_CNTEN3_Msk             (0x1ul << PWM_CNTEN_CNTEN3_Pos)                   /*!< PWM_T::CNTEN: CNTEN3 Mask              */

#define PWM_CNTEN_CNTEN4_Pos             (4)                                               /*!< PWM_T::CNTEN: CNTEN4 Position          */
#define PWM_CNTEN_CNTEN4_Msk             (0x1ul << PWM_CNTEN_CNTEN4_Pos)                   /*!< PWM_T::CNTEN: CNTEN4 Mask              */

#define PWM_CNTEN_CNTEN5_Pos             (5)                                               /*!< PWM_T::CNTEN: CNTEN5 Position          */
#define PWM_CNTEN_CNTEN5_Msk             (0x1ul << PWM_CNTEN_CNTEN5_Pos)                   /*!< PWM_T::CNTEN: CNTEN5 Mask              */

#define PWM_CNTCLR_CNTCLR0_Pos           (0)                                               /*!< PWM_T::CNTCLR: CNTCLR0 Position        */
#define PWM_CNTCLR_CNTCLR0_Msk           (0x1ul << PWM_CNTCLR_CNTCLR0_Pos)                 /*!< PWM_T::CNTCLR: CNTCLR0 Mask            */

#define PWM_CNTCLR_CNTCLR1_Pos           (1)                                               /*!< PWM_T::CNTCLR: CNTCLR1 Position        */
#define PWM_CNTCLR_CNTCLR1_Msk           (0x1ul << PWM_CNTCLR_CNTCLR1_Pos)                 /*!< PWM_T::CNTCLR: CNTCLR1 Mask            */

#define PWM_CNTCLR_CNTCLR2_Pos           (2)                                               /*!< PWM_T::CNTCLR: CNTCLR2 Position        */
#define PWM_CNTCLR_CNTCLR2_Msk           (0x1ul << PWM_CNTCLR_CNTCLR2_Pos)                 /*!< PWM_T::CNTCLR: CNTCLR2 Mask            */

#define PWM_CNTCLR_CNTCLR3_Pos           (3)                                               /*!< PWM_T::CNTCLR: CNTCLR3 Position        */
#define PWM_CNTCLR_CNTCLR3_Msk           (0x1ul << PWM_CNTCLR_CNTCLR3_Pos)                 /*!< PWM_T::CNTCLR: CNTCLR3 Mask            */

#define PWM_CNTCLR_CNTCLR4_Pos           (4)                                               /*!< PWM_T::CNTCLR: CNTCLR4 Position        */
#define PWM_CNTCLR_CNTCLR4_Msk           (0x1ul << PWM_CNTCLR_CNTCLR4_Pos)                 /*!< PWM_T::CNTCLR: CNTCLR4 Mask            */

#define PWM_CNTCLR_CNTCLR5_Pos           (5)                                               /*!< PWM_T::CNTCLR: CNTCLR5 Position        */
#define PWM_CNTCLR_CNTCLR5_Msk           (0x1ul << PWM_CNTCLR_CNTCLR5_Pos)                 /*!< PWM_T::CNTCLR: CNTCLR5 Mask            */

#define PWM_LOAD_LOAD0_Pos               (0)                                               /*!< PWM_T::LOAD: LOAD0 Position            */
#define PWM_LOAD_LOAD0_Msk               (0x1ul << PWM_LOAD_LOAD0_Pos)                     /*!< PWM_T::LOAD: LOAD0 Mask                */

#define PWM_LOAD_LOAD1_Pos               (1)                                               /*!< PWM_T::LOAD: LOAD1 Position            */
#define PWM_LOAD_LOAD1_Msk               (0x1ul << PWM_LOAD_LOAD1_Pos)                     /*!< PWM_T::LOAD: LOAD1 Mask                */

#define PWM_LOAD_LOAD2_Pos               (2)                                               /*!< PWM_T::LOAD: LOAD2 Position            */
#define PWM_LOAD_LOAD2_Msk               (0x1ul << PWM_LOAD_LOAD2_Pos)                     /*!< PWM_T::LOAD: LOAD2 Mask                */

#define PWM_LOAD_LOAD3_Pos               (3)                                               /*!< PWM_T::LOAD: LOAD3 Position            */
#define PWM_LOAD_LOAD3_Msk               (0x1ul << PWM_LOAD_LOAD3_Pos)                     /*!< PWM_T::LOAD: LOAD3 Mask                */

#define PWM_LOAD_LOAD4_Pos               (4)                                               /*!< PWM_T::LOAD: LOAD4 Position            */
#define PWM_LOAD_LOAD4_Msk               (0x1ul << PWM_LOAD_LOAD4_Pos)                     /*!< PWM_T::LOAD: LOAD4 Mask                */

#define PWM_LOAD_LOAD5_Pos               (5)                                               /*!< PWM_T::LOAD: LOAD5 Position            */
#define PWM_LOAD_LOAD5_Msk               (0x1ul << PWM_LOAD_LOAD5_Pos)                     /*!< PWM_T::LOAD: LOAD5 Mask                */

#define PWM_PERIOD_PERIOD_Pos            (0)                                               /*!< PWM_T::PERIOD: PERIOD Position         */
#define PWM_PERIOD_PERIOD_Msk            (0xfffful << PWM_PERIOD_PERIOD_Pos)               /*!< PWM_T::PERIOD: PERIOD Mask             */

#define PWM_CMPDAT_CMP_Pos               (0)                                               /*!< PWM_T::CMPDAT: CMP Position            */
#define PWM_CMPDAT_CMP_Msk               (0xfffful << PWM_CMPDAT_CMP_Pos)                  /*!< PWM_T::CMPDAT: CMP Mask                */

#define PWM_DTCTL0_1_DTCNT_Pos           (0)                                               /*!< PWM_T::DTCTL0_1: DTCNT Position        */
#define PWM_DTCTL0_1_DTCNT_Msk           (0xffful << PWM_DTCTL0_1_DTCNT_Pos)               /*!< PWM_T::DTCTL0_1: DTCNT Mask            */

#define PWM_DTCTL0_1_DTEN_Pos            (16)                                              /*!< PWM_T::DTCTL0_1: DTEN Position         */
#define PWM_DTCTL0_1_DTEN_Msk            (0x1ul << PWM_DTCTL0_1_DTEN_Pos)                  /*!< PWM_T::DTCTL0_1: DTEN Mask             */

#define PWM_DTCTL0_1_DTCKSEL_Pos         (24)                                              /*!< PWM_T::DTCTL0_1: DTCKSEL Position      */
#define PWM_DTCTL0_1_DTCKSEL_Msk         (0x1ul << PWM_DTCTL0_1_DTCKSEL_Pos)               /*!< PWM_T::DTCTL0_1: DTCKSEL Mask          */

#define PWM_DTCTL2_3_DTCNT_Pos           (0)                                               /*!< PWM_T::DTCTL2_3: DTCNT Position        */
#define PWM_DTCTL2_3_DTCNT_Msk           (0xffful << PWM_DTCTL2_3_DTCNT_Pos)               /*!< PWM_T::DTCTL2_3: DTCNT Mask            */

#define PWM_DTCTL2_3_DTEN_Pos            (16)                                              /*!< PWM_T::DTCTL2_3: DTEN Position         */
#define PWM_DTCTL2_3_DTEN_Msk            (0x1ul << PWM_DTCTL2_3_DTEN_Pos)                  /*!< PWM_T::DTCTL2_3: DTEN Mask             */

#define PWM_DTCTL2_3_DTCKSEL_Pos         (24)                                              /*!< PWM_T::DTCTL2_3: DTCKSEL Position      */
#define PWM_DTCTL2_3_DTCKSEL_Msk         (0x1ul << PWM_DTCTL2_3_DTCKSEL_Pos)               /*!< PWM_T::DTCTL2_3: DTCKSEL Mask          */

#define PWM_DTCTL4_5_DTCNT_Pos           (0)                                               /*!< PWM_T::DTCTL4_5: DTCNT Position        */
#define PWM_DTCTL4_5_DTCNT_Msk           (0xffful << PWM_DTCTL4_5_DTCNT_Pos)               /*!< PWM_T::DTCTL4_5: DTCNT Mask            */

#define PWM_DTCTL4_5_DTEN_Pos            (16)                                              /*!< PWM_T::DTCTL4_5: DTEN Position         */
#define PWM_DTCTL4_5_DTEN_Msk            (0x1ul << PWM_DTCTL4_5_DTEN_Pos)                  /*!< PWM_T::DTCTL4_5: DTEN Mask             */

#define PWM_DTCTL4_5_DTCKSEL_Pos         (24)                                              /*!< PWM_T::DTCTL4_5: DTCKSEL Position      */
#define PWM_DTCTL4_5_DTCKSEL_Msk         (0x1ul << PWM_DTCTL4_5_DTCKSEL_Pos)               /*!< PWM_T::DTCTL4_5: DTCKSEL Mask          */

#define PWM_PHS0_1_PHS_Pos               (0)                                               /*!< PWM_T::PHS0_1: PHS Position            */
#define PWM_PHS0_1_PHS_Msk               (0xfffful << PWM_PHS0_1_PHS_Pos)                  /*!< PWM_T::PHS0_1: PHS Mask                */

#define PWM_PHS2_3_PHS_Pos               (0)                                               /*!< PWM_T::PHS2_3: PHS Position            */
#define PWM_PHS2_3_PHS_Msk               (0xfffful << PWM_PHS2_3_PHS_Pos)                  /*!< PWM_T::PHS2_3: PHS Mask                */

#define PWM_PHS4_5_PHS_Pos               (0)                                               /*!< PWM_T::PHS4_5: PHS Position            */
#define PWM_PHS4_5_PHS_Msk               (0xfffful << PWM_PHS4_5_PHS_Pos)                  /*!< PWM_T::PHS4_5: PHS Mask                */

#define PWM_CNT_CNT_Pos                  (0)                                               /*!< PWM_T::CNT: CNT Position               */
#define PWM_CNT_CNT_Msk                  (0xfffful << PWM_CNT_CNT_Pos)                     /*!< PWM_T::CNT: CNT Mask                   */

#define PWM_CNT_DIRF_Pos                 (16)                                              /*!< PWM_T::CNT: DIRF Position              */
#define PWM_CNT_DIRF_Msk                 (0x1ul << PWM_CNT_DIRF_Pos)                       /*!< PWM_T::CNT: DIRF Mask                  */

#define PWM_WGCTL0_ZPCTL0_Pos            (0)                                               /*!< PWM_T::WGCTL0: ZPCTL0 Position         */
#define PWM_WGCTL0_ZPCTL0_Msk            (0x3ul << PWM_WGCTL0_ZPCTL0_Pos)                  /*!< PWM_T::WGCTL0: ZPCTL0 Mask             */

#define PWM_WGCTL0_ZPCTL1_Pos            (2)                                               /*!< PWM_T::WGCTL0: ZPCTL1 Position         */
#define PWM_WGCTL0_ZPCTL1_Msk            (0x3ul << PWM_WGCTL0_ZPCTL1_Pos)                  /*!< PWM_T::WGCTL0: ZPCTL1 Mask             */

#define PWM_WGCTL0_ZPCTL2_Pos            (4)                                               /*!< PWM_T::WGCTL0: ZPCTL2 Position         */
#define PWM_WGCTL0_ZPCTL2_Msk            (0x3ul << PWM_WGCTL0_ZPCTL2_Pos)                  /*!< PWM_T::WGCTL0: ZPCTL2 Mask             */

#define PWM_WGCTL0_ZPCTL3_Pos            (6)                                               /*!< PWM_T::WGCTL0: ZPCTL3 Position         */
#define PWM_WGCTL0_ZPCTL3_Msk            (0x3ul << PWM_WGCTL0_ZPCTL3_Pos)                  /*!< PWM_T::WGCTL0: ZPCTL3 Mask             */

#define PWM_WGCTL0_ZPCTL4_Pos            (8)                                               /*!< PWM_T::WGCTL0: ZPCTL4 Position         */
#define PWM_WGCTL0_ZPCTL4_Msk            (0x3ul << PWM_WGCTL0_ZPCTL4_Pos)                  /*!< PWM_T::WGCTL0: ZPCTL4 Mask             */

#define PWM_WGCTL0_ZPCTL5_Pos            (10)                                              /*!< PWM_T::WGCTL0: ZPCTL5 Position         */
#define PWM_WGCTL0_ZPCTL5_Msk            (0x3ul << PWM_WGCTL0_ZPCTL5_Pos)                  /*!< PWM_T::WGCTL0: ZPCTL5 Mask             */

#define PWM_WGCTL0_PRDPCTL0_Pos          (16)                                              /*!< PWM_T::WGCTL0: PRDPCTL0 Position       */
#define PWM_WGCTL0_PRDPCTL0_Msk          (0x3ul << PWM_WGCTL0_PRDPCTL0_Pos)                /*!< PWM_T::WGCTL0: PRDPCTL0 Mask           */

#define PWM_WGCTL0_PRDPCTL1_Pos          (18)                                              /*!< PWM_T::WGCTL0: PRDPCTL1 Position       */
#define PWM_WGCTL0_PRDPCTL1_Msk          (0x3ul << PWM_WGCTL0_PRDPCTL1_Pos)                /*!< PWM_T::WGCTL0: PRDPCTL1 Mask           */

#define PWM_WGCTL0_PRDPCTL2_Pos          (20)                                              /*!< PWM_T::WGCTL0: PRDPCTL2 Position       */
#define PWM_WGCTL0_PRDPCTL2_Msk          (0x3ul << PWM_WGCTL0_PRDPCTL2_Pos)                /*!< PWM_T::WGCTL0: PRDPCTL2 Mask           */

#define PWM_WGCTL0_PRDPCTL3_Pos          (22)                                              /*!< PWM_T::WGCTL0: PRDPCTL3 Position       */
#define PWM_WGCTL0_PRDPCTL3_Msk          (0x3ul << PWM_WGCTL0_PRDPCTL3_Pos)                /*!< PWM_T::WGCTL0: PRDPCTL3 Mask           */

#define PWM_WGCTL0_PRDPCTL4_Pos          (24)                                              /*!< PWM_T::WGCTL0: PRDPCTL4 Position       */
#define PWM_WGCTL0_PRDPCTL4_Msk          (0x3ul << PWM_WGCTL0_PRDPCTL4_Pos)                /*!< PWM_T::WGCTL0: PRDPCTL4 Mask           */

#define PWM_WGCTL0_PRDPCTL5_Pos          (26)                                              /*!< PWM_T::WGCTL0: PRDPCTL5 Position       */
#define PWM_WGCTL0_PRDPCTL5_Msk          (0x3ul << PWM_WGCTL0_PRDPCTL5_Pos)                /*!< PWM_T::WGCTL0: PRDPCTL5 Mask           */

#define PWM_WGCTL1_CMPUCTL0_Pos          (0)                                               /*!< PWM_T::WGCTL1: CMPUCTL0 Position       */
#define PWM_WGCTL1_CMPUCTL0_Msk          (0x3ul << PWM_WGCTL1_CMPUCTL0_Pos)                /*!< PWM_T::WGCTL1: CMPUCTL0 Mask           */

#define PWM_WGCTL1_CMPUCTL1_Pos          (2)                                               /*!< PWM_T::WGCTL1: CMPUCTL1 Position       */
#define PWM_WGCTL1_CMPUCTL1_Msk          (0x3ul << PWM_WGCTL1_CMPUCTL1_Pos)                /*!< PWM_T::WGCTL1: CMPUCTL1 Mask           */

#define PWM_WGCTL1_CMPUCTL2_Pos          (4)                                               /*!< PWM_T::WGCTL1: CMPUCTL2 Position       */
#define PWM_WGCTL1_CMPUCTL2_Msk          (0x3ul << PWM_WGCTL1_CMPUCTL2_Pos)                /*!< PWM_T::WGCTL1: CMPUCTL2 Mask           */

#define PWM_WGCTL1_CMPUCTL3_Pos          (6)                                               /*!< PWM_T::WGCTL1: CMPUCTL3 Position       */
#define PWM_WGCTL1_CMPUCTL3_Msk          (0x3ul << PWM_WGCTL1_CMPUCTL3_Pos)                /*!< PWM_T::WGCTL1: CMPUCTL3 Mask           */

#define PWM_WGCTL1_CMPUCTL4_Pos          (8)                                               /*!< PWM_T::WGCTL1: CMPUCTL4 Position       */
#define PWM_WGCTL1_CMPUCTL4_Msk          (0x3ul << PWM_WGCTL1_CMPUCTL4_Pos)                /*!< PWM_T::WGCTL1: CMPUCTL4 Mask           */

#define PWM_WGCTL1_CMPUCTL5_Pos          (10)                                              /*!< PWM_T::WGCTL1: CMPUCTL5 Position       */
#define PWM_WGCTL1_CMPUCTL5_Msk          (0x3ul << PWM_WGCTL1_CMPUCTL5_Pos)                /*!< PWM_T::WGCTL1: CMPUCTL5 Mask           */

#define PWM_WGCTL1_CMPDCTL0_Pos          (16)                                              /*!< PWM_T::WGCTL1: CMPDCTL0 Position       */
#define PWM_WGCTL1_CMPDCTL0_Msk          (0x3ul << PWM_WGCTL1_CMPDCTL0_Pos)                /*!< PWM_T::WGCTL1: CMPDCTL0 Mask           */

#define PWM_WGCTL1_CMPDCTL1_Pos          (18)                                              /*!< PWM_T::WGCTL1: CMPDCTL1 Position       */
#define PWM_WGCTL1_CMPDCTL1_Msk          (0x3ul << PWM_WGCTL1_CMPDCTL1_Pos)                /*!< PWM_T::WGCTL1: CMPDCTL1 Mask           */

#define PWM_WGCTL1_CMPDCTL2_Pos          (20)                                              /*!< PWM_T::WGCTL1: CMPDCTL2 Position       */
#define PWM_WGCTL1_CMPDCTL2_Msk          (0x3ul << PWM_WGCTL1_CMPDCTL2_Pos)                /*!< PWM_T::WGCTL1: CMPDCTL2 Mask           */

#define PWM_WGCTL1_CMPDCTL3_Pos          (22)                                              /*!< PWM_T::WGCTL1: CMPDCTL3 Position       */
#define PWM_WGCTL1_CMPDCTL3_Msk          (0x3ul << PWM_WGCTL1_CMPDCTL3_Pos)                /*!< PWM_T::WGCTL1: CMPDCTL3 Mask           */

#define PWM_WGCTL1_CMPDCTL4_Pos          (24)                                              /*!< PWM_T::WGCTL1: CMPDCTL4 Position       */
#define PWM_WGCTL1_CMPDCTL4_Msk          (0x3ul << PWM_WGCTL1_CMPDCTL4_Pos)                /*!< PWM_T::WGCTL1: CMPDCTL4 Mask           */

#define PWM_WGCTL1_CMPDCTL5_Pos          (26)                                              /*!< PWM_T::WGCTL1: CMPDCTL5 Position       */
#define PWM_WGCTL1_CMPDCTL5_Msk          (0x3ul << PWM_WGCTL1_CMPDCTL5_Pos)                /*!< PWM_T::WGCTL1: CMPDCTL5 Mask           */

#define PWM_MSKEN_MSKEN0_Pos             (0)                                               /*!< PWM_T::MSKEN: MSKEN0 Position          */
#define PWM_MSKEN_MSKEN0_Msk             (0x1ul << PWM_MSKEN_MSKEN0_Pos)                   /*!< PWM_T::MSKEN: MSKEN0 Mask              */

#define PWM_MSKEN_MSKEN1_Pos             (1)                                               /*!< PWM_T::MSKEN: MSKEN1 Position          */
#define PWM_MSKEN_MSKEN1_Msk             (0x1ul << PWM_MSKEN_MSKEN1_Pos)                   /*!< PWM_T::MSKEN: MSKEN1 Mask              */

#define PWM_MSKEN_MSKEN2_Pos             (2)                                               /*!< PWM_T::MSKEN: MSKEN2 Position          */
#define PWM_MSKEN_MSKEN2_Msk             (0x1ul << PWM_MSKEN_MSKEN2_Pos)                   /*!< PWM_T::MSKEN: MSKEN2 Mask              */

#define PWM_MSKEN_MSKEN3_Pos             (3)                                               /*!< PWM_T::MSKEN: MSKEN3 Position          */
#define PWM_MSKEN_MSKEN3_Msk             (0x1ul << PWM_MSKEN_MSKEN3_Pos)                   /*!< PWM_T::MSKEN: MSKEN3 Mask              */

#define PWM_MSKEN_MSKEN4_Pos             (4)                                               /*!< PWM_T::MSKEN: MSKEN4 Position          */
#define PWM_MSKEN_MSKEN4_Msk             (0x1ul << PWM_MSKEN_MSKEN4_Pos)                   /*!< PWM_T::MSKEN: MSKEN4 Mask              */

#define PWM_MSKEN_MSKEN5_Pos             (5)                                               /*!< PWM_T::MSKEN: MSKEN5 Position          */
#define PWM_MSKEN_MSKEN5_Msk             (0x1ul << PWM_MSKEN_MSKEN5_Pos)                   /*!< PWM_T::MSKEN: MSKEN5 Mask              */

#define PWM_MSK_MSKDAT0_Pos              (0)                                               /*!< PWM_T::MSK: MSKDAT0 Position           */
#define PWM_MSK_MSKDAT0_Msk              (0x1ul << PWM_MSK_MSKDAT0_Pos)                    /*!< PWM_T::MSK: MSKDAT0 Mask               */

#define PWM_MSK_MSKDAT1_Pos              (1)                                               /*!< PWM_T::MSK: MSKDAT1 Position           */
#define PWM_MSK_MSKDAT1_Msk              (0x1ul << PWM_MSK_MSKDAT1_Pos)                    /*!< PWM_T::MSK: MSKDAT1 Mask               */

#define PWM_MSK_MSKDAT2_Pos              (2)                                               /*!< PWM_T::MSK: MSKDAT2 Position           */
#define PWM_MSK_MSKDAT2_Msk              (0x1ul << PWM_MSK_MSKDAT2_Pos)                    /*!< PWM_T::MSK: MSKDAT2 Mask               */

#define PWM_MSK_MSKDAT3_Pos              (3)                                               /*!< PWM_T::MSK: MSKDAT3 Position           */
#define PWM_MSK_MSKDAT3_Msk              (0x1ul << PWM_MSK_MSKDAT3_Pos)                    /*!< PWM_T::MSK: MSKDAT3 Mask               */

#define PWM_MSK_MSKDAT4_Pos              (4)                                               /*!< PWM_T::MSK: MSKDAT4 Position           */
#define PWM_MSK_MSKDAT4_Msk              (0x1ul << PWM_MSK_MSKDAT4_Pos)                    /*!< PWM_T::MSK: MSKDAT4 Mask               */

#define PWM_MSK_MSKDAT5_Pos              (5)                                               /*!< PWM_T::MSK: MSKDAT5 Position           */
#define PWM_MSK_MSKDAT5_Msk              (0x1ul << PWM_MSK_MSKDAT5_Pos)                    /*!< PWM_T::MSK: MSKDAT5 Mask               */

#define PWM_BNF_BRK0NFEN_Pos             (0)                                               /*!< PWM_T::BNF: BRK0NFEN Position          */
#define PWM_BNF_BRK0NFEN_Msk             (0x1ul << PWM_BNF_BRK0NFEN_Pos)                   /*!< PWM_T::BNF: BRK0NFEN Mask              */

#define PWM_BNF_BRK0NFSEL_Pos            (1)                                               /*!< PWM_T::BNF: BRK0NFSEL Position         */
#define PWM_BNF_BRK0NFSEL_Msk            (0x7ul << PWM_BNF_BRK0NFSEL_Pos)                  /*!< PWM_T::BNF: BRK0NFSEL Mask             */

#define PWM_BNF_BRK0FCNT_Pos             (4)                                               /*!< PWM_T::BNF: BRK0FCNT Position          */
#define PWM_BNF_BRK0FCNT_Msk             (0x7ul << PWM_BNF_BRK0FCNT_Pos)                   /*!< PWM_T::BNF: BRK0FCNT Mask              */

#define PWM_BNF_BRK0PINV_Pos             (7)                                               /*!< PWM_T::BNF: BRK0PINV Position          */
#define PWM_BNF_BRK0PINV_Msk             (0x1ul << PWM_BNF_BRK0PINV_Pos)                   /*!< PWM_T::BNF: BRK0PINV Mask              */

#define PWM_BNF_BRK1NFEN_Pos             (8)                                               /*!< PWM_T::BNF: BRK1NFEN Position          */
#define PWM_BNF_BRK1NFEN_Msk             (0x1ul << PWM_BNF_BRK1NFEN_Pos)                   /*!< PWM_T::BNF: BRK1NFEN Mask              */

#define PWM_BNF_BRK1NFSEL_Pos            (9)                                               /*!< PWM_T::BNF: BRK1NFSEL Position         */
#define PWM_BNF_BRK1NFSEL_Msk            (0x7ul << PWM_BNF_BRK1NFSEL_Pos)                  /*!< PWM_T::BNF: BRK1NFSEL Mask             */

#define PWM_BNF_BRK1FCNT_Pos             (12)                                              /*!< PWM_T::BNF: BRK1FCNT Position          */
#define PWM_BNF_BRK1FCNT_Msk             (0x7ul << PWM_BNF_BRK1FCNT_Pos)                   /*!< PWM_T::BNF: BRK1FCNT Mask              */

#define PWM_BNF_BRK1PINV_Pos             (15)                                              /*!< PWM_T::BNF: BRK1PINV Position          */
#define PWM_BNF_BRK1PINV_Msk             (0x1ul << PWM_BNF_BRK1PINV_Pos)                   /*!< PWM_T::BNF: BRK1PINV Mask              */

#define PWM_BNF_BK0SRC_Pos               (16)                                              /*!< PWM_T::BNF: BK0SRC Position            */
#define PWM_BNF_BK0SRC_Msk               (0x1ul << PWM_BNF_BK0SRC_Pos)                     /*!< PWM_T::BNF: BK0SRC Mask                */

#define PWM_BNF_BK1SRC_Pos               (24)                                              /*!< PWM_T::BNF: BK1SRC Position            */
#define PWM_BNF_BK1SRC_Msk               (0x1ul << PWM_BNF_BK1SRC_Pos)                     /*!< PWM_T::BNF: BK1SRC Mask                */

#define PWM_FAILBRK_CSSBRKEN_Pos         (0)                                               /*!< PWM_T::FAILBRK: CSSBRKEN Position      */
#define PWM_FAILBRK_CSSBRKEN_Msk         (0x1ul << PWM_FAILBRK_CSSBRKEN_Pos)               /*!< PWM_T::FAILBRK: CSSBRKEN Mask          */

#define PWM_FAILBRK_BODBRKEN_Pos         (1)                                               /*!< PWM_T::FAILBRK: BODBRKEN Position      */
#define PWM_FAILBRK_BODBRKEN_Msk         (0x1ul << PWM_FAILBRK_BODBRKEN_Pos)               /*!< PWM_T::FAILBRK: BODBRKEN Mask          */

#define PWM_FAILBRK_CORBRKEN_Pos         (3)                                               /*!< PWM_T::FAILBRK: CORBRKEN Position      */
#define PWM_FAILBRK_CORBRKEN_Msk         (0x1ul << PWM_FAILBRK_CORBRKEN_Pos)               /*!< PWM_T::FAILBRK: CORBRKEN Mask          */

#define PWM_BRKCTL0_1_CPO0EBEN_Pos       (0)                                               /*!< PWM_T::BRKCTL0_1: CPO0EBEN Position    */
#define PWM_BRKCTL0_1_CPO0EBEN_Msk       (0x1ul << PWM_BRKCTL0_1_CPO0EBEN_Pos)             /*!< PWM_T::BRKCTL0_1: CPO0EBEN Mask        */

#define PWM_BRKCTL0_1_CPO1EBEN_Pos       (1)                                               /*!< PWM_T::BRKCTL0_1: CPO1EBEN Position    */
#define PWM_BRKCTL0_1_CPO1EBEN_Msk       (0x1ul << PWM_BRKCTL0_1_CPO1EBEN_Pos)             /*!< PWM_T::BRKCTL0_1: CPO1EBEN Mask        */

#define PWM_BRKCTL0_1_BRKP0EEN_Pos       (4)                                               /*!< PWM_T::BRKCTL0_1: BRKP0EEN Position    */
#define PWM_BRKCTL0_1_BRKP0EEN_Msk       (0x1ul << PWM_BRKCTL0_1_BRKP0EEN_Pos)             /*!< PWM_T::BRKCTL0_1: BRKP0EEN Mask        */

#define PWM_BRKCTL0_1_BRKP1EEN_Pos       (5)                                               /*!< PWM_T::BRKCTL0_1: BRKP1EEN Position    */
#define PWM_BRKCTL0_1_BRKP1EEN_Msk       (0x1ul << PWM_BRKCTL0_1_BRKP1EEN_Pos)             /*!< PWM_T::BRKCTL0_1: BRKP1EEN Mask        */

#define PWM_BRKCTL0_1_SYSEBEN_Pos        (7)                                               /*!< PWM_T::BRKCTL0_1: SYSEBEN Position     */
#define PWM_BRKCTL0_1_SYSEBEN_Msk        (0x1ul << PWM_BRKCTL0_1_SYSEBEN_Pos)              /*!< PWM_T::BRKCTL0_1: SYSEBEN Mask         */

#define PWM_BRKCTL0_1_CPO0LBEN_Pos       (8)                                               /*!< PWM_T::BRKCTL0_1: CPO0LBEN Position    */
#define PWM_BRKCTL0_1_CPO0LBEN_Msk       (0x1ul << PWM_BRKCTL0_1_CPO0LBEN_Pos)             /*!< PWM_T::BRKCTL0_1: CPO0LBEN Mask        */

#define PWM_BRKCTL0_1_CPO1LBEN_Pos       (9)                                               /*!< PWM_T::BRKCTL0_1: CPO1LBEN Position    */
#define PWM_BRKCTL0_1_CPO1LBEN_Msk       (0x1ul << PWM_BRKCTL0_1_CPO1LBEN_Pos)             /*!< PWM_T::BRKCTL0_1: CPO1LBEN Mask        */

#define PWM_BRKCTL0_1_BRKP0LEN_Pos       (12)                                              /*!< PWM_T::BRKCTL0_1: BRKP0LEN Position    */
#define PWM_BRKCTL0_1_BRKP0LEN_Msk       (0x1ul << PWM_BRKCTL0_1_BRKP0LEN_Pos)             /*!< PWM_T::BRKCTL0_1: BRKP0LEN Mask        */

#define PWM_BRKCTL0_1_BRKP1LEN_Pos       (13)                                              /*!< PWM_T::BRKCTL0_1: BRKP1LEN Position    */
#define PWM_BRKCTL0_1_BRKP1LEN_Msk       (0x1ul << PWM_BRKCTL0_1_BRKP1LEN_Pos)             /*!< PWM_T::BRKCTL0_1: BRKP1LEN Mask        */

#define PWM_BRKCTL0_1_SYSLBEN_Pos        (15)                                              /*!< PWM_T::BRKCTL0_1: SYSLBEN Position     */
#define PWM_BRKCTL0_1_SYSLBEN_Msk        (0x1ul << PWM_BRKCTL0_1_SYSLBEN_Pos)              /*!< PWM_T::BRKCTL0_1: SYSLBEN Mask         */

#define PWM_BRKCTL0_1_BRKAEVEN_Pos       (16)                                              /*!< PWM_T::BRKCTL0_1: BRKAEVEN Position    */
#define PWM_BRKCTL0_1_BRKAEVEN_Msk       (0x3ul << PWM_BRKCTL0_1_BRKAEVEN_Pos)             /*!< PWM_T::BRKCTL0_1: BRKAEVEN Mask        */

#define PWM_BRKCTL0_1_BRKAODD_Pos        (18)                                              /*!< PWM_T::BRKCTL0_1: BRKAODD Position     */
#define PWM_BRKCTL0_1_BRKAODD_Msk        (0x3ul << PWM_BRKCTL0_1_BRKAODD_Pos)              /*!< PWM_T::BRKCTL0_1: BRKAODD Mask         */

#define PWM_BRKCTL0_1_ADCEBEN_Pos        (20)                                              /*!< PWM_T::BRKCTL0_1: ADCEBEN Position     */
#define PWM_BRKCTL0_1_ADCEBEN_Msk        (0x1ul << PWM_BRKCTL0_1_ADCEBEN_Pos)              /*!< PWM_T::BRKCTL0_1: ADCEBEN Mask         */

#define PWM_BRKCTL0_1_ADCLBEN_Pos        (28)                                              /*!< PWM_T::BRKCTL0_1: ADCLBEN Position     */
#define PWM_BRKCTL0_1_ADCLBEN_Msk        (0x1ul << PWM_BRKCTL0_1_ADCLBEN_Pos)              /*!< PWM_T::BRKCTL0_1: ADCLBEN Mask         */

#define PWM_BRKCTL2_3_CPO0EBEN_Pos       (0)                                               /*!< PWM_T::BRKCTL2_3: CPO0EBEN Position    */
#define PWM_BRKCTL2_3_CPO0EBEN_Msk       (0x1ul << PWM_BRKCTL2_3_CPO0EBEN_Pos)             /*!< PWM_T::BRKCTL2_3: CPO0EBEN Mask        */

#define PWM_BRKCTL2_3_CPO1EBEN_Pos       (1)                                               /*!< PWM_T::BRKCTL2_3: CPO1EBEN Position    */
#define PWM_BRKCTL2_3_CPO1EBEN_Msk       (0x1ul << PWM_BRKCTL2_3_CPO1EBEN_Pos)             /*!< PWM_T::BRKCTL2_3: CPO1EBEN Mask        */

#define PWM_BRKCTL2_3_BRKP0EEN_Pos       (4)                                               /*!< PWM_T::BRKCTL2_3: BRKP0EEN Position    */
#define PWM_BRKCTL2_3_BRKP0EEN_Msk       (0x1ul << PWM_BRKCTL2_3_BRKP0EEN_Pos)             /*!< PWM_T::BRKCTL2_3: BRKP0EEN Mask        */

#define PWM_BRKCTL2_3_BRKP1EEN_Pos       (5)                                               /*!< PWM_T::BRKCTL2_3: BRKP1EEN Position    */
#define PWM_BRKCTL2_3_BRKP1EEN_Msk       (0x1ul << PWM_BRKCTL2_3_BRKP1EEN_Pos)             /*!< PWM_T::BRKCTL2_3: BRKP1EEN Mask        */

#define PWM_BRKCTL2_3_SYSEBEN_Pos        (7)                                               /*!< PWM_T::BRKCTL2_3: SYSEBEN Position     */
#define PWM_BRKCTL2_3_SYSEBEN_Msk        (0x1ul << PWM_BRKCTL2_3_SYSEBEN_Pos)              /*!< PWM_T::BRKCTL2_3: SYSEBEN Mask         */

#define PWM_BRKCTL2_3_CPO0LBEN_Pos       (8)                                               /*!< PWM_T::BRKCTL2_3: CPO0LBEN Position    */
#define PWM_BRKCTL2_3_CPO0LBEN_Msk       (0x1ul << PWM_BRKCTL2_3_CPO0LBEN_Pos)             /*!< PWM_T::BRKCTL2_3: CPO0LBEN Mask        */

#define PWM_BRKCTL2_3_CPO1LBEN_Pos       (9)                                               /*!< PWM_T::BRKCTL2_3: CPO1LBEN Position    */
#define PWM_BRKCTL2_3_CPO1LBEN_Msk       (0x1ul << PWM_BRKCTL2_3_CPO1LBEN_Pos)             /*!< PWM_T::BRKCTL2_3: CPO1LBEN Mask        */

#define PWM_BRKCTL2_3_BRKP0LEN_Pos       (12)                                              /*!< PWM_T::BRKCTL2_3: BRKP0LEN Position    */
#define PWM_BRKCTL2_3_BRKP0LEN_Msk       (0x1ul << PWM_BRKCTL2_3_BRKP0LEN_Pos)             /*!< PWM_T::BRKCTL2_3: BRKP0LEN Mask        */

#define PWM_BRKCTL2_3_BRKP1LEN_Pos       (13)                                              /*!< PWM_T::BRKCTL2_3: BRKP1LEN Position    */
#define PWM_BRKCTL2_3_BRKP1LEN_Msk       (0x1ul << PWM_BRKCTL2_3_BRKP1LEN_Pos)             /*!< PWM_T::BRKCTL2_3: BRKP1LEN Mask        */

#define PWM_BRKCTL2_3_SYSLBEN_Pos        (15)                                              /*!< PWM_T::BRKCTL2_3: SYSLBEN Position     */
#define PWM_BRKCTL2_3_SYSLBEN_Msk        (0x1ul << PWM_BRKCTL2_3_SYSLBEN_Pos)              /*!< PWM_T::BRKCTL2_3: SYSLBEN Mask         */

#define PWM_BRKCTL2_3_BRKAEVEN_Pos       (16)                                              /*!< PWM_T::BRKCTL2_3: BRKAEVEN Position    */
#define PWM_BRKCTL2_3_BRKAEVEN_Msk       (0x3ul << PWM_BRKCTL2_3_BRKAEVEN_Pos)             /*!< PWM_T::BRKCTL2_3: BRKAEVEN Mask        */

#define PWM_BRKCTL2_3_BRKAODD_Pos        (18)                                              /*!< PWM_T::BRKCTL2_3: BRKAODD Position     */
#define PWM_BRKCTL2_3_BRKAODD_Msk        (0x3ul << PWM_BRKCTL2_3_BRKAODD_Pos)              /*!< PWM_T::BRKCTL2_3: BRKAODD Mask         */

#define PWM_BRKCTL2_3_ADCEBEN_Pos        (20)                                              /*!< PWM_T::BRKCTL2_3: ADCEBEN Position     */
#define PWM_BRKCTL2_3_ADCEBEN_Msk        (0x1ul << PWM_BRKCTL2_3_ADCEBEN_Pos)              /*!< PWM_T::BRKCTL2_3: ADCEBEN Mask         */

#define PWM_BRKCTL2_3_ADCLBEN_Pos        (28)                                              /*!< PWM_T::BRKCTL2_3: ADCLBEN Position     */
#define PWM_BRKCTL2_3_ADCLBEN_Msk        (0x1ul << PWM_BRKCTL2_3_ADCLBEN_Pos)              /*!< PWM_T::BRKCTL2_3: ADCLBEN Mask         */

#define PWM_BRKCTL4_5_CPO0EBEN_Pos       (0)                                               /*!< PWM_T::BRKCTL4_5: CPO0EBEN Position    */
#define PWM_BRKCTL4_5_CPO0EBEN_Msk       (0x1ul << PWM_BRKCTL4_5_CPO0EBEN_Pos)             /*!< PWM_T::BRKCTL4_5: CPO0EBEN Mask        */

#define PWM_BRKCTL4_5_CPO1EBEN_Pos       (1)                                               /*!< PWM_T::BRKCTL4_5: CPO1EBEN Position    */
#define PWM_BRKCTL4_5_CPO1EBEN_Msk       (0x1ul << PWM_BRKCTL4_5_CPO1EBEN_Pos)             /*!< PWM_T::BRKCTL4_5: CPO1EBEN Mask        */

#define PWM_BRKCTL4_5_BRKP0EEN_Pos       (4)                                               /*!< PWM_T::BRKCTL4_5: BRKP0EEN Position    */
#define PWM_BRKCTL4_5_BRKP0EEN_Msk       (0x1ul << PWM_BRKCTL4_5_BRKP0EEN_Pos)             /*!< PWM_T::BRKCTL4_5: BRKP0EEN Mask        */

#define PWM_BRKCTL4_5_BRKP1EEN_Pos       (5)                                               /*!< PWM_T::BRKCTL4_5: BRKP1EEN Position    */
#define PWM_BRKCTL4_5_BRKP1EEN_Msk       (0x1ul << PWM_BRKCTL4_5_BRKP1EEN_Pos)             /*!< PWM_T::BRKCTL4_5: BRKP1EEN Mask        */

#define PWM_BRKCTL4_5_SYSEBEN_Pos        (7)                                               /*!< PWM_T::BRKCTL4_5: SYSEBEN Position     */
#define PWM_BRKCTL4_5_SYSEBEN_Msk        (0x1ul << PWM_BRKCTL4_5_SYSEBEN_Pos)              /*!< PWM_T::BRKCTL4_5: SYSEBEN Mask         */

#define PWM_BRKCTL4_5_CPO0LBEN_Pos       (8)                                               /*!< PWM_T::BRKCTL4_5: CPO0LBEN Position    */
#define PWM_BRKCTL4_5_CPO0LBEN_Msk       (0x1ul << PWM_BRKCTL4_5_CPO0LBEN_Pos)             /*!< PWM_T::BRKCTL4_5: CPO0LBEN Mask        */

#define PWM_BRKCTL4_5_CPO1LBEN_Pos       (9)                                               /*!< PWM_T::BRKCTL4_5: CPO1LBEN Position    */
#define PWM_BRKCTL4_5_CPO1LBEN_Msk       (0x1ul << PWM_BRKCTL4_5_CPO1LBEN_Pos)             /*!< PWM_T::BRKCTL4_5: CPO1LBEN Mask        */

#define PWM_BRKCTL4_5_BRKP0LEN_Pos       (12)                                              /*!< PWM_T::BRKCTL4_5: BRKP0LEN Position    */
#define PWM_BRKCTL4_5_BRKP0LEN_Msk       (0x1ul << PWM_BRKCTL4_5_BRKP0LEN_Pos)             /*!< PWM_T::BRKCTL4_5: BRKP0LEN Mask        */

#define PWM_BRKCTL4_5_BRKP1LEN_Pos       (13)                                              /*!< PWM_T::BRKCTL4_5: BRKP1LEN Position    */
#define PWM_BRKCTL4_5_BRKP1LEN_Msk       (0x1ul << PWM_BRKCTL4_5_BRKP1LEN_Pos)             /*!< PWM_T::BRKCTL4_5: BRKP1LEN Mask        */

#define PWM_BRKCTL4_5_SYSLBEN_Pos        (15)                                              /*!< PWM_T::BRKCTL4_5: SYSLBEN Position     */
#define PWM_BRKCTL4_5_SYSLBEN_Msk        (0x1ul << PWM_BRKCTL4_5_SYSLBEN_Pos)              /*!< PWM_T::BRKCTL4_5: SYSLBEN Mask         */

#define PWM_BRKCTL4_5_BRKAEVEN_Pos       (16)                                              /*!< PWM_T::BRKCTL4_5: BRKAEVEN Position    */
#define PWM_BRKCTL4_5_BRKAEVEN_Msk       (0x3ul << PWM_BRKCTL4_5_BRKAEVEN_Pos)             /*!< PWM_T::BRKCTL4_5: BRKAEVEN Mask        */

#define PWM_BRKCTL4_5_BRKAODD_Pos        (18)                                              /*!< PWM_T::BRKCTL4_5: BRKAODD Position     */
#define PWM_BRKCTL4_5_BRKAODD_Msk        (0x3ul << PWM_BRKCTL4_5_BRKAODD_Pos)              /*!< PWM_T::BRKCTL4_5: BRKAODD Mask         */

#define PWM_BRKCTL4_5_ADCEBEN_Pos        (20)                                              /*!< PWM_T::BRKCTL4_5: ADCEBEN Position     */
#define PWM_BRKCTL4_5_ADCEBEN_Msk        (0x1ul << PWM_BRKCTL4_5_ADCEBEN_Pos)              /*!< PWM_T::BRKCTL4_5: ADCEBEN Mask         */

#define PWM_BRKCTL4_5_ADCLBEN_Pos        (28)                                              /*!< PWM_T::BRKCTL4_5: ADCLBEN Position     */
#define PWM_BRKCTL4_5_ADCLBEN_Msk        (0x1ul << PWM_BRKCTL4_5_ADCLBEN_Pos)              /*!< PWM_T::BRKCTL4_5: ADCLBEN Mask         */

#define PWM_POLCTL_PINV0_Pos             (0)                                               /*!< PWM_T::POLCTL: PINV0 Position          */
#define PWM_POLCTL_PINV0_Msk             (0x1ul << PWM_POLCTL_PINV0_Pos)                   /*!< PWM_T::POLCTL: PINV0 Mask              */

#define PWM_POLCTL_PINV1_Pos             (1)                                               /*!< PWM_T::POLCTL: PINV1 Position          */
#define PWM_POLCTL_PINV1_Msk             (0x1ul << PWM_POLCTL_PINV1_Pos)                   /*!< PWM_T::POLCTL: PINV1 Mask              */

#define PWM_POLCTL_PINV2_Pos             (2)                                               /*!< PWM_T::POLCTL: PINV2 Position          */
#define PWM_POLCTL_PINV2_Msk             (0x1ul << PWM_POLCTL_PINV2_Pos)                   /*!< PWM_T::POLCTL: PINV2 Mask              */

#define PWM_POLCTL_PINV3_Pos             (3)                                               /*!< PWM_T::POLCTL: PINV3 Position          */
#define PWM_POLCTL_PINV3_Msk             (0x1ul << PWM_POLCTL_PINV3_Pos)                   /*!< PWM_T::POLCTL: PINV3 Mask              */

#define PWM_POLCTL_PINV4_Pos             (4)                                               /*!< PWM_T::POLCTL: PINV4 Position          */
#define PWM_POLCTL_PINV4_Msk             (0x1ul << PWM_POLCTL_PINV4_Pos)                   /*!< PWM_T::POLCTL: PINV4 Mask              */

#define PWM_POLCTL_PINV5_Pos             (5)                                               /*!< PWM_T::POLCTL: PINV5 Position          */
#define PWM_POLCTL_PINV5_Msk             (0x1ul << PWM_POLCTL_PINV5_Pos)                   /*!< PWM_T::POLCTL: PINV5 Mask              */

#define PWM_POEN_POEN0_Pos               (0)                                               /*!< PWM_T::POEN: POEN0 Position            */
#define PWM_POEN_POEN0_Msk               (0x1ul << PWM_POEN_POEN0_Pos)                     /*!< PWM_T::POEN: POEN0 Mask                */

#define PWM_POEN_POEN1_Pos               (1)                                               /*!< PWM_T::POEN: POEN1 Position            */
#define PWM_POEN_POEN1_Msk               (0x1ul << PWM_POEN_POEN1_Pos)                     /*!< PWM_T::POEN: POEN1 Mask                */

#define PWM_POEN_POEN2_Pos               (2)                                               /*!< PWM_T::POEN: POEN2 Position            */
#define PWM_POEN_POEN2_Msk               (0x1ul << PWM_POEN_POEN2_Pos)                     /*!< PWM_T::POEN: POEN2 Mask                */

#define PWM_POEN_POEN3_Pos               (3)                                               /*!< PWM_T::POEN: POEN3 Position            */
#define PWM_POEN_POEN3_Msk               (0x1ul << PWM_POEN_POEN3_Pos)                     /*!< PWM_T::POEN: POEN3 Mask                */

#define PWM_POEN_POEN4_Pos               (4)                                               /*!< PWM_T::POEN: POEN4 Position            */
#define PWM_POEN_POEN4_Msk               (0x1ul << PWM_POEN_POEN4_Pos)                     /*!< PWM_T::POEN: POEN4 Mask                */

#define PWM_POEN_POEN5_Pos               (5)                                               /*!< PWM_T::POEN: POEN5 Position            */
#define PWM_POEN_POEN5_Msk               (0x1ul << PWM_POEN_POEN5_Pos)                     /*!< PWM_T::POEN: POEN5 Mask                */

#define PWM_SWBRK_BRKETRG0_Pos           (0)                                               /*!< PWM_T::SWBRK: BRKETRG0 Position        */
#define PWM_SWBRK_BRKETRG0_Msk           (0x1ul << PWM_SWBRK_BRKETRG0_Pos)                 /*!< PWM_T::SWBRK: BRKETRG0 Mask            */

#define PWM_SWBRK_BRKETRG2_Pos           (1)                                               /*!< PWM_T::SWBRK: BRKETRG2 Position        */
#define PWM_SWBRK_BRKETRG2_Msk           (0x1ul << PWM_SWBRK_BRKETRG2_Pos)                 /*!< PWM_T::SWBRK: BRKETRG2 Mask            */

#define PWM_SWBRK_BRKETRG4_Pos           (2)                                               /*!< PWM_T::SWBRK: BRKETRG4 Position        */
#define PWM_SWBRK_BRKETRG4_Msk           (0x1ul << PWM_SWBRK_BRKETRG4_Pos)                 /*!< PWM_T::SWBRK: BRKETRG4 Mask            */

#define PWM_SWBRK_BRKLTRG0_Pos           (8)                                               /*!< PWM_T::SWBRK: BRKLTRG0 Position        */
#define PWM_SWBRK_BRKLTRG0_Msk           (0x1ul << PWM_SWBRK_BRKLTRG0_Pos)                 /*!< PWM_T::SWBRK: BRKLTRG0 Mask            */

#define PWM_SWBRK_BRKLTRG2_Pos           (9)                                               /*!< PWM_T::SWBRK: BRKLTRG2 Position        */
#define PWM_SWBRK_BRKLTRG2_Msk           (0x1ul << PWM_SWBRK_BRKLTRG2_Pos)                 /*!< PWM_T::SWBRK: BRKLTRG2 Mask            */

#define PWM_SWBRK_BRKLTRG4_Pos           (10)                                              /*!< PWM_T::SWBRK: BRKLTRG4 Position        */
#define PWM_SWBRK_BRKLTRG4_Msk           (0x1ul << PWM_SWBRK_BRKLTRG4_Pos)                 /*!< PWM_T::SWBRK: BRKLTRG4 Mask            */

#define PWM_INTEN0_ZIEN0_Pos             (0)                                               /*!< PWM_T::INTEN0: ZIEN0 Position          */
#define PWM_INTEN0_ZIEN0_Msk             (0x1ul << PWM_INTEN0_ZIEN0_Pos)                   /*!< PWM_T::INTEN0: ZIEN0 Mask              */

#define PWM_INTEN0_ZIEN1_Pos             (1)                                               /*!< PWM_T::INTEN0: ZIEN1 Position          */
#define PWM_INTEN0_ZIEN1_Msk             (0x1ul << PWM_INTEN0_ZIEN1_Pos)                   /*!< PWM_T::INTEN0: ZIEN1 Mask              */

#define PWM_INTEN0_ZIEN2_Pos             (2)                                               /*!< PWM_T::INTEN0: ZIEN2 Position          */
#define PWM_INTEN0_ZIEN2_Msk             (0x1ul << PWM_INTEN0_ZIEN2_Pos)                   /*!< PWM_T::INTEN0: ZIEN2 Mask              */

#define PWM_INTEN0_ZIEN3_Pos             (3)                                               /*!< PWM_T::INTEN0: ZIEN3 Position          */
#define PWM_INTEN0_ZIEN3_Msk             (0x1ul << PWM_INTEN0_ZIEN3_Pos)                   /*!< PWM_T::INTEN0: ZIEN3 Mask              */

#define PWM_INTEN0_ZIEN4_Pos             (4)                                               /*!< PWM_T::INTEN0: ZIEN4 Position          */
#define PWM_INTEN0_ZIEN4_Msk             (0x1ul << PWM_INTEN0_ZIEN4_Pos)                   /*!< PWM_T::INTEN0: ZIEN4 Mask              */

#define PWM_INTEN0_ZIEN5_Pos             (5)                                               /*!< PWM_T::INTEN0: ZIEN5 Position          */
#define PWM_INTEN0_ZIEN5_Msk             (0x1ul << PWM_INTEN0_ZIEN5_Pos)                   /*!< PWM_T::INTEN0: ZIEN5 Mask              */

#define PWM_INTEN0_IFAIEN0_1_Pos         (7)                                               /*!< PWM_T::INTEN0: IFAIEN0_1 Position      */
#define PWM_INTEN0_IFAIEN0_1_Msk         (0x1ul << PWM_INTEN0_IFAIEN0_1_Pos)               /*!< PWM_T::INTEN0: IFAIEN0_1 Mask          */

#define PWM_INTEN0_PIEN0_Pos             (8)                                               /*!< PWM_T::INTEN0: PIEN0 Position          */
#define PWM_INTEN0_PIEN0_Msk             (0x1ul << PWM_INTEN0_PIEN0_Pos)                   /*!< PWM_T::INTEN0: PIEN0 Mask              */

#define PWM_INTEN0_PIEN1_Pos             (9)                                               /*!< PWM_T::INTEN0: PIEN1 Position          */
#define PWM_INTEN0_PIEN1_Msk             (0x1ul << PWM_INTEN0_PIEN1_Pos)                   /*!< PWM_T::INTEN0: PIEN1 Mask              */

#define PWM_INTEN0_PIEN2_Pos             (10)                                              /*!< PWM_T::INTEN0: PIEN2 Position          */
#define PWM_INTEN0_PIEN2_Msk             (0x1ul << PWM_INTEN0_PIEN2_Pos)                   /*!< PWM_T::INTEN0: PIEN2 Mask              */

#define PWM_INTEN0_PIEN3_Pos             (11)                                              /*!< PWM_T::INTEN0: PIEN3 Position          */
#define PWM_INTEN0_PIEN3_Msk             (0x1ul << PWM_INTEN0_PIEN3_Pos)                   /*!< PWM_T::INTEN0: PIEN3 Mask              */

#define PWM_INTEN0_PIEN4_Pos             (12)                                              /*!< PWM_T::INTEN0: PIEN4 Position          */
#define PWM_INTEN0_PIEN4_Msk             (0x1ul << PWM_INTEN0_PIEN4_Pos)                   /*!< PWM_T::INTEN0: PIEN4 Mask              */

#define PWM_INTEN0_PIEN5_Pos             (13)                                              /*!< PWM_T::INTEN0: PIEN5 Position          */
#define PWM_INTEN0_PIEN5_Msk             (0x1ul << PWM_INTEN0_PIEN5_Pos)                   /*!< PWM_T::INTEN0: PIEN5 Mask              */

#define PWM_INTEN0_IFAIEN2_3_Pos         (15)                                              /*!< PWM_T::INTEN0: IFAIEN2_3 Position      */
#define PWM_INTEN0_IFAIEN2_3_Msk         (0x1ul << PWM_INTEN0_IFAIEN2_3_Pos)               /*!< PWM_T::INTEN0: IFAIEN2_3 Mask          */

#define PWM_INTEN0_CMPUIEN0_Pos          (16)                                              /*!< PWM_T::INTEN0: CMPUIEN0 Position       */
#define PWM_INTEN0_CMPUIEN0_Msk          (0x1ul << PWM_INTEN0_CMPUIEN0_Pos)                /*!< PWM_T::INTEN0: CMPUIEN0 Mask           */

#define PWM_INTEN0_CMPUIEN1_Pos          (17)                                              /*!< PWM_T::INTEN0: CMPUIEN1 Position       */
#define PWM_INTEN0_CMPUIEN1_Msk          (0x1ul << PWM_INTEN0_CMPUIEN1_Pos)                /*!< PWM_T::INTEN0: CMPUIEN1 Mask           */

#define PWM_INTEN0_CMPUIEN2_Pos          (18)                                              /*!< PWM_T::INTEN0: CMPUIEN2 Position       */
#define PWM_INTEN0_CMPUIEN2_Msk          (0x1ul << PWM_INTEN0_CMPUIEN2_Pos)                /*!< PWM_T::INTEN0: CMPUIEN2 Mask           */

#define PWM_INTEN0_CMPUIEN3_Pos          (19)                                              /*!< PWM_T::INTEN0: CMPUIEN3 Position       */
#define PWM_INTEN0_CMPUIEN3_Msk          (0x1ul << PWM_INTEN0_CMPUIEN3_Pos)                /*!< PWM_T::INTEN0: CMPUIEN3 Mask           */

#define PWM_INTEN0_CMPUIEN4_Pos          (20)                                              /*!< PWM_T::INTEN0: CMPUIEN4 Position       */
#define PWM_INTEN0_CMPUIEN4_Msk          (0x1ul << PWM_INTEN0_CMPUIEN4_Pos)                /*!< PWM_T::INTEN0: CMPUIEN4 Mask           */

#define PWM_INTEN0_CMPUIEN5_Pos          (21)                                              /*!< PWM_T::INTEN0: CMPUIEN5 Position       */
#define PWM_INTEN0_CMPUIEN5_Msk          (0x1ul << PWM_INTEN0_CMPUIEN5_Pos)                /*!< PWM_T::INTEN0: CMPUIEN5 Mask           */

#define PWM_INTEN0_IFAIEN4_5_Pos         (23)                                              /*!< PWM_T::INTEN0: IFAIEN4_5 Position      */
#define PWM_INTEN0_IFAIEN4_5_Msk         (0x1ul << PWM_INTEN0_IFAIEN4_5_Pos)               /*!< PWM_T::INTEN0: IFAIEN4_5 Mask          */

#define PWM_INTEN0_CMPDIEN0_Pos          (24)                                              /*!< PWM_T::INTEN0: CMPDIEN0 Position       */
#define PWM_INTEN0_CMPDIEN0_Msk          (0x1ul << PWM_INTEN0_CMPDIEN0_Pos)                /*!< PWM_T::INTEN0: CMPDIEN0 Mask           */

#define PWM_INTEN0_CMPDIEN1_Pos          (25)                                              /*!< PWM_T::INTEN0: CMPDIEN1 Position       */
#define PWM_INTEN0_CMPDIEN1_Msk          (0x1ul << PWM_INTEN0_CMPDIEN1_Pos)                /*!< PWM_T::INTEN0: CMPDIEN1 Mask           */

#define PWM_INTEN0_CMPDIEN2_Pos          (26)                                              /*!< PWM_T::INTEN0: CMPDIEN2 Position       */
#define PWM_INTEN0_CMPDIEN2_Msk          (0x1ul << PWM_INTEN0_CMPDIEN2_Pos)                /*!< PWM_T::INTEN0: CMPDIEN2 Mask           */

#define PWM_INTEN0_CMPDIEN3_Pos          (27)                                              /*!< PWM_T::INTEN0: CMPDIEN3 Position       */
#define PWM_INTEN0_CMPDIEN3_Msk          (0x1ul << PWM_INTEN0_CMPDIEN3_Pos)                /*!< PWM_T::INTEN0: CMPDIEN3 Mask           */

#define PWM_INTEN0_CMPDIEN4_Pos          (28)                                              /*!< PWM_T::INTEN0: CMPDIEN4 Position       */
#define PWM_INTEN0_CMPDIEN4_Msk          (0x1ul << PWM_INTEN0_CMPDIEN4_Pos)                /*!< PWM_T::INTEN0: CMPDIEN4 Mask           */

#define PWM_INTEN0_CMPDIEN5_Pos          (29)                                              /*!< PWM_T::INTEN0: CMPDIEN5 Position       */
#define PWM_INTEN0_CMPDIEN5_Msk          (0x1ul << PWM_INTEN0_CMPDIEN5_Pos)                /*!< PWM_T::INTEN0: CMPDIEN5 Mask           */

#define PWM_INTEN1_BRKEIEN0_1_Pos        (0)                                               /*!< PWM_T::INTEN1: BRKEIEN0_1 Position     */
#define PWM_INTEN1_BRKEIEN0_1_Msk        (0x1ul << PWM_INTEN1_BRKEIEN0_1_Pos)              /*!< PWM_T::INTEN1: BRKEIEN0_1 Mask         */

#define PWM_INTEN1_BRKEIEN2_3_Pos        (1)                                               /*!< PWM_T::INTEN1: BRKEIEN2_3 Position     */
#define PWM_INTEN1_BRKEIEN2_3_Msk        (0x1ul << PWM_INTEN1_BRKEIEN2_3_Pos)              /*!< PWM_T::INTEN1: BRKEIEN2_3 Mask         */

#define PWM_INTEN1_BRKEIEN4_5_Pos        (2)                                               /*!< PWM_T::INTEN1: BRKEIEN4_5 Position     */
#define PWM_INTEN1_BRKEIEN4_5_Msk        (0x1ul << PWM_INTEN1_BRKEIEN4_5_Pos)              /*!< PWM_T::INTEN1: BRKEIEN4_5 Mask         */

#define PWM_INTEN1_BRKLIEN0_1_Pos        (8)                                               /*!< PWM_T::INTEN1: BRKLIEN0_1 Position     */
#define PWM_INTEN1_BRKLIEN0_1_Msk        (0x1ul << PWM_INTEN1_BRKLIEN0_1_Pos)              /*!< PWM_T::INTEN1: BRKLIEN0_1 Mask         */

#define PWM_INTEN1_BRKLIEN2_3_Pos        (9)                                               /*!< PWM_T::INTEN1: BRKLIEN2_3 Position     */
#define PWM_INTEN1_BRKLIEN2_3_Msk        (0x1ul << PWM_INTEN1_BRKLIEN2_3_Pos)              /*!< PWM_T::INTEN1: BRKLIEN2_3 Mask         */

#define PWM_INTEN1_BRKLIEN4_5_Pos        (10)                                              /*!< PWM_T::INTEN1: BRKLIEN4_5 Position     */
#define PWM_INTEN1_BRKLIEN4_5_Msk        (0x1ul << PWM_INTEN1_BRKLIEN4_5_Pos)              /*!< PWM_T::INTEN1: BRKLIEN4_5 Mask         */

#define PWM_INTSTS0_ZIF0_Pos             (0)                                               /*!< PWM_T::INTSTS0: ZIF0 Position          */
#define PWM_INTSTS0_ZIF0_Msk             (0x1ul << PWM_INTSTS0_ZIF0_Pos)                   /*!< PWM_T::INTSTS0: ZIF0 Mask              */

#define PWM_INTSTS0_ZIF1_Pos             (1)                                               /*!< PWM_T::INTSTS0: ZIF1 Position          */
#define PWM_INTSTS0_ZIF1_Msk             (0x1ul << PWM_INTSTS0_ZIF1_Pos)                   /*!< PWM_T::INTSTS0: ZIF1 Mask              */

#define PWM_INTSTS0_ZIF2_Pos             (2)                                               /*!< PWM_T::INTSTS0: ZIF2 Position          */
#define PWM_INTSTS0_ZIF2_Msk             (0x1ul << PWM_INTSTS0_ZIF2_Pos)                   /*!< PWM_T::INTSTS0: ZIF2 Mask              */

#define PWM_INTSTS0_ZIF3_Pos             (3)                                               /*!< PWM_T::INTSTS0: ZIF3 Position          */
#define PWM_INTSTS0_ZIF3_Msk             (0x1ul << PWM_INTSTS0_ZIF3_Pos)                   /*!< PWM_T::INTSTS0: ZIF3 Mask              */

#define PWM_INTSTS0_ZIF4_Pos             (4)                                               /*!< PWM_T::INTSTS0: ZIF4 Position          */
#define PWM_INTSTS0_ZIF4_Msk             (0x1ul << PWM_INTSTS0_ZIF4_Pos)                   /*!< PWM_T::INTSTS0: ZIF4 Mask              */

#define PWM_INTSTS0_ZIF5_Pos             (5)                                               /*!< PWM_T::INTSTS0: ZIF5 Position          */
#define PWM_INTSTS0_ZIF5_Msk             (0x1ul << PWM_INTSTS0_ZIF5_Pos)                   /*!< PWM_T::INTSTS0: ZIF5 Mask              */

#define PWM_INTSTS0_IFAIF0_1_Pos         (7)                                               /*!< PWM_T::INTSTS0: IFAIF0_1 Position      */
#define PWM_INTSTS0_IFAIF0_1_Msk         (0x1ul << PWM_INTSTS0_IFAIF0_1_Pos)               /*!< PWM_T::INTSTS0: IFAIF0_1 Mask          */

#define PWM_INTSTS0_PIF0_Pos             (8)                                               /*!< PWM_T::INTSTS0: PIF0 Position          */
#define PWM_INTSTS0_PIF0_Msk             (0x1ul << PWM_INTSTS0_PIF0_Pos)                   /*!< PWM_T::INTSTS0: PIF0 Mask              */

#define PWM_INTSTS0_PIF1_Pos             (9)                                               /*!< PWM_T::INTSTS0: PIF1 Position          */
#define PWM_INTSTS0_PIF1_Msk             (0x1ul << PWM_INTSTS0_PIF1_Pos)                   /*!< PWM_T::INTSTS0: PIF1 Mask              */

#define PWM_INTSTS0_PIF2_Pos             (10)                                              /*!< PWM_T::INTSTS0: PIF2 Position          */
#define PWM_INTSTS0_PIF2_Msk             (0x1ul << PWM_INTSTS0_PIF2_Pos)                   /*!< PWM_T::INTSTS0: PIF2 Mask              */

#define PWM_INTSTS0_PIF3_Pos             (11)                                              /*!< PWM_T::INTSTS0: PIF3 Position          */
#define PWM_INTSTS0_PIF3_Msk             (0x1ul << PWM_INTSTS0_PIF3_Pos)                   /*!< PWM_T::INTSTS0: PIF3 Mask              */

#define PWM_INTSTS0_PIF4_Pos             (12)                                              /*!< PWM_T::INTSTS0: PIF4 Position          */
#define PWM_INTSTS0_PIF4_Msk             (0x1ul << PWM_INTSTS0_PIF4_Pos)                   /*!< PWM_T::INTSTS0: PIF4 Mask              */

#define PWM_INTSTS0_PIF5_Pos             (13)                                              /*!< PWM_T::INTSTS0: PIF5 Position          */
#define PWM_INTSTS0_PIF5_Msk             (0x1ul << PWM_INTSTS0_PIF5_Pos)                   /*!< PWM_T::INTSTS0: PIF5 Mask              */

#define PWM_INTSTS0_IFAIF2_3_Pos         (15)                                              /*!< PWM_T::INTSTS0: IFAIF2_3 Position      */
#define PWM_INTSTS0_IFAIF2_3_Msk         (0x1ul << PWM_INTSTS0_IFAIF2_3_Pos)               /*!< PWM_T::INTSTS0: IFAIF2_3 Mask          */

#define PWM_INTSTS0_CMPUIF0_Pos          (16)                                              /*!< PWM_T::INTSTS0: CMPUIF0 Position       */
#define PWM_INTSTS0_CMPUIF0_Msk          (0x1ul << PWM_INTSTS0_CMPUIF0_Pos)                /*!< PWM_T::INTSTS0: CMPUIF0 Mask           */

#define PWM_INTSTS0_CMPUIF1_Pos          (17)                                              /*!< PWM_T::INTSTS0: CMPUIF1 Position       */
#define PWM_INTSTS0_CMPUIF1_Msk          (0x1ul << PWM_INTSTS0_CMPUIF1_Pos)                /*!< PWM_T::INTSTS0: CMPUIF1 Mask           */

#define PWM_INTSTS0_CMPUIF2_Pos          (18)                                              /*!< PWM_T::INTSTS0: CMPUIF2 Position       */
#define PWM_INTSTS0_CMPUIF2_Msk          (0x1ul << PWM_INTSTS0_CMPUIF2_Pos)                /*!< PWM_T::INTSTS0: CMPUIF2 Mask           */

#define PWM_INTSTS0_CMPUIF3_Pos          (19)                                              /*!< PWM_T::INTSTS0: CMPUIF3 Position       */
#define PWM_INTSTS0_CMPUIF3_Msk          (0x1ul << PWM_INTSTS0_CMPUIF3_Pos)                /*!< PWM_T::INTSTS0: CMPUIF3 Mask           */

#define PWM_INTSTS0_CMPUIF4_Pos          (20)                                              /*!< PWM_T::INTSTS0: CMPUIF4 Position       */
#define PWM_INTSTS0_CMPUIF4_Msk          (0x1ul << PWM_INTSTS0_CMPUIF4_Pos)                /*!< PWM_T::INTSTS0: CMPUIF4 Mask           */

#define PWM_INTSTS0_CMPUIF5_Pos          (21)                                              /*!< PWM_T::INTSTS0: CMPUIF5 Position       */
#define PWM_INTSTS0_CMPUIF5_Msk          (0x1ul << PWM_INTSTS0_CMPUIF5_Pos)                /*!< PWM_T::INTSTS0: CMPUIF5 Mask           */

#define PWM_INTSTS0_IFAIF4_5_Pos         (23)                                              /*!< PWM_T::INTSTS0: IFAIF4_5 Position      */
#define PWM_INTSTS0_IFAIF4_5_Msk         (0x1ul << PWM_INTSTS0_IFAIF4_5_Pos)               /*!< PWM_T::INTSTS0: IFAIF4_5 Mask          */

#define PWM_INTSTS0_CMPDIF0_Pos          (24)                                              /*!< PWM_T::INTSTS0: CMPDIF0 Position       */
#define PWM_INTSTS0_CMPDIF0_Msk          (0x1ul << PWM_INTSTS0_CMPDIF0_Pos)                /*!< PWM_T::INTSTS0: CMPDIF0 Mask           */

#define PWM_INTSTS0_CMPDIF1_Pos          (25)                                              /*!< PWM_T::INTSTS0: CMPDIF1 Position       */
#define PWM_INTSTS0_CMPDIF1_Msk          (0x1ul << PWM_INTSTS0_CMPDIF1_Pos)                /*!< PWM_T::INTSTS0: CMPDIF1 Mask           */

#define PWM_INTSTS0_CMPDIF2_Pos          (26)                                              /*!< PWM_T::INTSTS0: CMPDIF2 Position       */
#define PWM_INTSTS0_CMPDIF2_Msk          (0x1ul << PWM_INTSTS0_CMPDIF2_Pos)                /*!< PWM_T::INTSTS0: CMPDIF2 Mask           */

#define PWM_INTSTS0_CMPDIF3_Pos          (27)                                              /*!< PWM_T::INTSTS0: CMPDIF3 Position       */
#define PWM_INTSTS0_CMPDIF3_Msk          (0x1ul << PWM_INTSTS0_CMPDIF3_Pos)                /*!< PWM_T::INTSTS0: CMPDIF3 Mask           */

#define PWM_INTSTS0_CMPDIF4_Pos          (28)                                              /*!< PWM_T::INTSTS0: CMPDIF4 Position       */
#define PWM_INTSTS0_CMPDIF4_Msk          (0x1ul << PWM_INTSTS0_CMPDIF4_Pos)                /*!< PWM_T::INTSTS0: CMPDIF4 Mask           */

#define PWM_INTSTS0_CMPDIF5_Pos          (29)                                              /*!< PWM_T::INTSTS0: CMPDIF5 Position       */
#define PWM_INTSTS0_CMPDIF5_Msk          (0x1ul << PWM_INTSTS0_CMPDIF5_Pos)                /*!< PWM_T::INTSTS0: CMPDIF5 Mask           */

#define PWM_INTSTS1_BRKEIF0_Pos          (0)                                               /*!< PWM_T::INTSTS1: BRKEIF0 Position       */
#define PWM_INTSTS1_BRKEIF0_Msk          (0x1ul << PWM_INTSTS1_BRKEIF0_Pos)                /*!< PWM_T::INTSTS1: BRKEIF0 Mask           */

#define PWM_INTSTS1_BRKEIF1_Pos          (1)                                               /*!< PWM_T::INTSTS1: BRKEIF1 Position       */
#define PWM_INTSTS1_BRKEIF1_Msk          (0x1ul << PWM_INTSTS1_BRKEIF1_Pos)                /*!< PWM_T::INTSTS1: BRKEIF1 Mask           */

#define PWM_INTSTS1_BRKEIF2_Pos          (2)                                               /*!< PWM_T::INTSTS1: BRKEIF2 Position       */
#define PWM_INTSTS1_BRKEIF2_Msk          (0x1ul << PWM_INTSTS1_BRKEIF2_Pos)                /*!< PWM_T::INTSTS1: BRKEIF2 Mask           */

#define PWM_INTSTS1_BRKEIF3_Pos          (3)                                               /*!< PWM_T::INTSTS1: BRKEIF3 Position       */
#define PWM_INTSTS1_BRKEIF3_Msk          (0x1ul << PWM_INTSTS1_BRKEIF3_Pos)                /*!< PWM_T::INTSTS1: BRKEIF3 Mask           */

#define PWM_INTSTS1_BRKEIF4_Pos          (4)                                               /*!< PWM_T::INTSTS1: BRKEIF4 Position       */
#define PWM_INTSTS1_BRKEIF4_Msk          (0x1ul << PWM_INTSTS1_BRKEIF4_Pos)                /*!< PWM_T::INTSTS1: BRKEIF4 Mask           */

#define PWM_INTSTS1_BRKEIF5_Pos          (5)                                               /*!< PWM_T::INTSTS1: BRKEIF5 Position       */
#define PWM_INTSTS1_BRKEIF5_Msk          (0x1ul << PWM_INTSTS1_BRKEIF5_Pos)                /*!< PWM_T::INTSTS1: BRKEIF5 Mask           */

#define PWM_INTSTS1_BRKLIFn_Pos          (8)                                               /*!< PWM_T::INTSTS1: BRKLIFn Position       */
#define PWM_INTSTS1_BRKLIFn_Msk          (0x1ul << PWM_INTSTS1_BRKLIFn_Pos)                /*!< PWM_T::INTSTS1: BRKLIFn Mask           */

#define PWM_INTSTS1_BRKESTS0_Pos         (16)                                              /*!< PWM_T::INTSTS1: BRKESTS0 Position      */
#define PWM_INTSTS1_BRKESTS0_Msk         (0x1ul << PWM_INTSTS1_BRKESTS0_Pos)               /*!< PWM_T::INTSTS1: BRKESTS0 Mask          */

#define PWM_INTSTS1_BRKESTS1_Pos         (17)                                              /*!< PWM_T::INTSTS1: BRKESTS1 Position      */
#define PWM_INTSTS1_BRKESTS1_Msk         (0x1ul << PWM_INTSTS1_BRKESTS1_Pos)               /*!< PWM_T::INTSTS1: BRKESTS1 Mask          */

#define PWM_INTSTS1_BRKESTS2_Pos         (18)                                              /*!< PWM_T::INTSTS1: BRKESTS2 Position      */
#define PWM_INTSTS1_BRKESTS2_Msk         (0x1ul << PWM_INTSTS1_BRKESTS2_Pos)               /*!< PWM_T::INTSTS1: BRKESTS2 Mask          */

#define PWM_INTSTS1_BRKESTS3_Pos         (19)                                              /*!< PWM_T::INTSTS1: BRKESTS3 Position      */
#define PWM_INTSTS1_BRKESTS3_Msk         (0x1ul << PWM_INTSTS1_BRKESTS3_Pos)               /*!< PWM_T::INTSTS1: BRKESTS3 Mask          */

#define PWM_INTSTS1_BRKESTS4_Pos         (20)                                              /*!< PWM_T::INTSTS1: BRKESTS4 Position      */
#define PWM_INTSTS1_BRKESTS4_Msk         (0x1ul << PWM_INTSTS1_BRKESTS4_Pos)               /*!< PWM_T::INTSTS1: BRKESTS4 Mask          */

#define PWM_INTSTS1_BRKESTS5_Pos         (21)                                              /*!< PWM_T::INTSTS1: BRKESTS5 Position      */
#define PWM_INTSTS1_BRKESTS5_Msk         (0x1ul << PWM_INTSTS1_BRKESTS5_Pos)               /*!< PWM_T::INTSTS1: BRKESTS5 Mask          */

#define PWM_INTSTS1_BRKLSTS0_Pos         (24)                                              /*!< PWM_T::INTSTS1: BRKLSTS0 Position      */
#define PWM_INTSTS1_BRKLSTS0_Msk         (0x1ul << PWM_INTSTS1_BRKLSTS0_Pos)               /*!< PWM_T::INTSTS1: BRKLSTS0 Mask          */

#define PWM_INTSTS1_BRKLSTS1_Pos         (25)                                              /*!< PWM_T::INTSTS1: BRKLSTS1 Position      */
#define PWM_INTSTS1_BRKLSTS1_Msk         (0x1ul << PWM_INTSTS1_BRKLSTS1_Pos)               /*!< PWM_T::INTSTS1: BRKLSTS1 Mask          */

#define PWM_INTSTS1_BRKLSTS2_Pos         (26)                                              /*!< PWM_T::INTSTS1: BRKLSTS2 Position      */
#define PWM_INTSTS1_BRKLSTS2_Msk         (0x1ul << PWM_INTSTS1_BRKLSTS2_Pos)               /*!< PWM_T::INTSTS1: BRKLSTS2 Mask          */

#define PWM_INTSTS1_BRKLSTS3_Pos         (27)                                              /*!< PWM_T::INTSTS1: BRKLSTS3 Position      */
#define PWM_INTSTS1_BRKLSTS3_Msk         (0x1ul << PWM_INTSTS1_BRKLSTS3_Pos)               /*!< PWM_T::INTSTS1: BRKLSTS3 Mask          */

#define PWM_INTSTS1_BRKLSTS4_Pos         (28)                                              /*!< PWM_T::INTSTS1: BRKLSTS4 Position      */
#define PWM_INTSTS1_BRKLSTS4_Msk         (0x1ul << PWM_INTSTS1_BRKLSTS4_Pos)               /*!< PWM_T::INTSTS1: BRKLSTS4 Mask          */

#define PWM_INTSTS1_BRKLSTS5_Pos         (29)                                              /*!< PWM_T::INTSTS1: BRKLSTS5 Position      */
#define PWM_INTSTS1_BRKLSTS5_Msk         (0x1ul << PWM_INTSTS1_BRKLSTS5_Pos)               /*!< PWM_T::INTSTS1: BRKLSTS5 Mask          */

#define PWM_IFA_IFCNT0_1_Pos             (0)                                               /*!< PWM_T::IFA: IFCNT0_1 Position          */
#define PWM_IFA_IFCNT0_1_Msk             (0xful << PWM_IFA_IFCNT0_1_Pos)                   /*!< PWM_T::IFA: IFCNT0_1 Mask              */

#define PWM_IFA_IFSEL0_1_Pos             (4)                                               /*!< PWM_T::IFA: IFSEL0_1 Position          */
#define PWM_IFA_IFSEL0_1_Msk             (0x7ul << PWM_IFA_IFSEL0_1_Pos)                   /*!< PWM_T::IFA: IFSEL0_1 Mask              */

#define PWM_IFA_IFAEN0_1_Pos             (7)                                               /*!< PWM_T::IFA: IFAEN0_1 Position          */
#define PWM_IFA_IFAEN0_1_Msk             (0x1ul << PWM_IFA_IFAEN0_1_Pos)                   /*!< PWM_T::IFA: IFAEN0_1 Mask              */

#define PWM_IFA_IFCNT2_3_Pos             (8)                                               /*!< PWM_T::IFA: IFCNT2_3 Position          */
#define PWM_IFA_IFCNT2_3_Msk             (0xful << PWM_IFA_IFCNT2_3_Pos)                   /*!< PWM_T::IFA: IFCNT2_3 Mask              */

#define PWM_IFA_IFSEL2_3_Pos             (12)                                              /*!< PWM_T::IFA: IFSEL2_3 Position          */
#define PWM_IFA_IFSEL2_3_Msk             (0x7ul << PWM_IFA_IFSEL2_3_Pos)                   /*!< PWM_T::IFA: IFSEL2_3 Mask              */

#define PWM_IFA_IFAEN2_3_Pos             (15)                                              /*!< PWM_T::IFA: IFAEN2_3 Position          */
#define PWM_IFA_IFAEN2_3_Msk             (0x1ul << PWM_IFA_IFAEN2_3_Pos)                   /*!< PWM_T::IFA: IFAEN2_3 Mask              */

#define PWM_IFA_IFCNT4_5_Pos             (16)                                              /*!< PWM_T::IFA: IFCNT4_5 Position          */
#define PWM_IFA_IFCNT4_5_Msk             (0xful << PWM_IFA_IFCNT4_5_Pos)                   /*!< PWM_T::IFA: IFCNT4_5 Mask              */

#define PWM_IFA_IFSEL4_5_Pos             (20)                                              /*!< PWM_T::IFA: IFSEL4_5 Position          */
#define PWM_IFA_IFSEL4_5_Msk             (0x7ul << PWM_IFA_IFSEL4_5_Pos)                   /*!< PWM_T::IFA: IFSEL4_5 Mask              */

#define PWM_IFA_IFAEN4_5_Pos             (23)                                              /*!< PWM_T::IFA: IFAEN4_5 Position          */
#define PWM_IFA_IFAEN4_5_Msk             (0x1ul << PWM_IFA_IFAEN4_5_Pos)                   /*!< PWM_T::IFA: IFAEN4_5 Mask              */

#define PWM_ADCTS0_TRGSEL0_Pos           (0)                                               /*!< PWM_T::ADCTS0: TRGSEL0 Position        */
#define PWM_ADCTS0_TRGSEL0_Msk           (0xful << PWM_ADCTS0_TRGSEL0_Pos)                 /*!< PWM_T::ADCTS0: TRGSEL0 Mask            */

#define PWM_ADCTS0_TRGEN0_Pos            (7)                                               /*!< PWM_T::ADCTS0: TRGEN0 Position         */
#define PWM_ADCTS0_TRGEN0_Msk            (0x1ul << PWM_ADCTS0_TRGEN0_Pos)                  /*!< PWM_T::ADCTS0: TRGEN0 Mask             */

#define PWM_ADCTS0_TRGSEL1_Pos           (8)                                               /*!< PWM_T::ADCTS0: TRGSEL1 Position        */
#define PWM_ADCTS0_TRGSEL1_Msk           (0xful << PWM_ADCTS0_TRGSEL1_Pos)                 /*!< PWM_T::ADCTS0: TRGSEL1 Mask            */

#define PWM_ADCTS0_TRGEN1_Pos            (15)                                              /*!< PWM_T::ADCTS0: TRGEN1 Position         */
#define PWM_ADCTS0_TRGEN1_Msk            (0x1ul << PWM_ADCTS0_TRGEN1_Pos)                  /*!< PWM_T::ADCTS0: TRGEN1 Mask             */

#define PWM_ADCTS0_TRGSEL2_Pos           (16)                                              /*!< PWM_T::ADCTS0: TRGSEL2 Position        */
#define PWM_ADCTS0_TRGSEL2_Msk           (0xful << PWM_ADCTS0_TRGSEL2_Pos)                 /*!< PWM_T::ADCTS0: TRGSEL2 Mask            */

#define PWM_ADCTS0_TRGEN2_Pos            (23)                                              /*!< PWM_T::ADCTS0: TRGEN2 Position         */
#define PWM_ADCTS0_TRGEN2_Msk            (0x1ul << PWM_ADCTS0_TRGEN2_Pos)                  /*!< PWM_T::ADCTS0: TRGEN2 Mask             */

#define PWM_ADCTS0_TRGSEL3_Pos           (24)                                              /*!< PWM_T::ADCTS0: TRGSEL3 Position        */
#define PWM_ADCTS0_TRGSEL3_Msk           (0xful << PWM_ADCTS0_TRGSEL3_Pos)                 /*!< PWM_T::ADCTS0: TRGSEL3 Mask            */

#define PWM_ADCTS0_TRGEN3_Pos            (31)                                              /*!< PWM_T::ADCTS0: TRGEN3 Position         */
#define PWM_ADCTS0_TRGEN3_Msk            (0x1ul << PWM_ADCTS0_TRGEN3_Pos)                  /*!< PWM_T::ADCTS0: TRGEN3 Mask             */

#define PWM_ADCTS1_TRGSEL4_Pos           (0)                                               /*!< PWM_T::ADCTS1: TRGSEL4 Position        */
#define PWM_ADCTS1_TRGSEL4_Msk           (0xful << PWM_ADCTS1_TRGSEL4_Pos)                 /*!< PWM_T::ADCTS1: TRGSEL4 Mask            */

#define PWM_ADCTS1_TRGEN4_Pos            (7)                                               /*!< PWM_T::ADCTS1: TRGEN4 Position         */
#define PWM_ADCTS1_TRGEN4_Msk            (0x1ul << PWM_ADCTS1_TRGEN4_Pos)                  /*!< PWM_T::ADCTS1: TRGEN4 Mask             */

#define PWM_ADCTS1_TRGSEL5_Pos           (8)                                               /*!< PWM_T::ADCTS1: TRGSEL5 Position        */
#define PWM_ADCTS1_TRGSEL5_Msk           (0xful << PWM_ADCTS1_TRGSEL5_Pos)                 /*!< PWM_T::ADCTS1: TRGSEL5 Mask            */

#define PWM_ADCTS1_TRGEN5_Pos            (15)                                              /*!< PWM_T::ADCTS1: TRGEN5 Position         */
#define PWM_ADCTS1_TRGEN5_Msk            (0x1ul << PWM_ADCTS1_TRGEN5_Pos)                  /*!< PWM_T::ADCTS1: TRGEN5 Mask             */

#define PWM_FTCMPDAT0_1_FTCMP_Pos        (0)                                               /*!< PWM_T::FTCMPDAT0_1: FTCMP Position     */
#define PWM_FTCMPDAT0_1_FTCMP_Msk        (0xfffful << PWM_FTCMPDAT0_1_FTCMP_Pos)           /*!< PWM_T::FTCMPDAT0_1: FTCMP Mask         */

#define PWM_FTCMPDAT2_3_FTCMP_Pos        (0)                                               /*!< PWM_T::FTCMPDAT2_3: FTCMP Position     */
#define PWM_FTCMPDAT2_3_FTCMP_Msk        (0xfffful << PWM_FTCMPDAT2_3_FTCMP_Pos)           /*!< PWM_T::FTCMPDAT2_3: FTCMP Mask         */

#define PWM_FTCMPDAT4_5_FTCMP_Pos        (0)                                               /*!< PWM_T::FTCMPDAT4_5: FTCMP Position     */
#define PWM_FTCMPDAT4_5_FTCMP_Msk        (0xfffful << PWM_FTCMPDAT4_5_FTCMP_Pos)           /*!< PWM_T::FTCMPDAT4_5: FTCMP Mask         */

#define PWM_SSCTL_SSEN0_Pos              (0)                                               /*!< PWM_T::SSCTL: SSEN0 Position           */
#define PWM_SSCTL_SSEN0_Msk              (0x1ul << PWM_SSCTL_SSEN0_Pos)                    /*!< PWM_T::SSCTL: SSEN0 Mask               */

#define PWM_SSCTL_SSEN1_Pos              (1)                                               /*!< PWM_T::SSCTL: SSEN1 Position           */
#define PWM_SSCTL_SSEN1_Msk              (0x1ul << PWM_SSCTL_SSEN1_Pos)                    /*!< PWM_T::SSCTL: SSEN1 Mask               */

#define PWM_SSCTL_SSEN2_Pos              (2)                                               /*!< PWM_T::SSCTL: SSEN2 Position           */
#define PWM_SSCTL_SSEN2_Msk              (0x1ul << PWM_SSCTL_SSEN2_Pos)                    /*!< PWM_T::SSCTL: SSEN2 Mask               */

#define PWM_SSCTL_SSEN3_Pos              (3)                                               /*!< PWM_T::SSCTL: SSEN3 Position           */
#define PWM_SSCTL_SSEN3_Msk              (0x1ul << PWM_SSCTL_SSEN3_Pos)                    /*!< PWM_T::SSCTL: SSEN3 Mask               */

#define PWM_SSCTL_SSEN4_Pos              (4)                                               /*!< PWM_T::SSCTL: SSEN4 Position           */
#define PWM_SSCTL_SSEN4_Msk              (0x1ul << PWM_SSCTL_SSEN4_Pos)                    /*!< PWM_T::SSCTL: SSEN4 Mask               */

#define PWM_SSCTL_SSEN5_Pos              (5)                                               /*!< PWM_T::SSCTL: SSEN5 Position           */
#define PWM_SSCTL_SSEN5_Msk              (0x1ul << PWM_SSCTL_SSEN5_Pos)                    /*!< PWM_T::SSCTL: SSEN5 Mask               */

#define PWM_SSCTL_SSRC_Pos               (8)                                               /*!< PWM_T::SSCTL: SSRC Position            */
#define PWM_SSCTL_SSRC_Msk               (0x1ul << PWM_SSCTL_SSRC_Pos)                     /*!< PWM_T::SSCTL: SSRC Mask                */

#define PWM_SSTRG_CNTSEN_Pos             (0)                                               /*!< PWM_T::SSTRG: CNTSEN Position          */
#define PWM_SSTRG_CNTSEN_Msk             (0x1ul << PWM_SSTRG_CNTSEN_Pos)                   /*!< PWM_T::SSTRG: CNTSEN Mask              */

#define PWM_LEBCTL_LEBEN_Pos             (0)                                               /*!< PWM_T::LEBCTL: LEBEN Position          */
#define PWM_LEBCTL_LEBEN_Msk             (0x1ul << PWM_LEBCTL_LEBEN_Pos)                   /*!< PWM_T::LEBCTL: LEBEN Mask              */

#define PWM_LEBCTL_SRCEN0_Pos            (8)                                               /*!< PWM_T::LEBCTL: SRCEN0 Position         */
#define PWM_LEBCTL_SRCEN0_Msk            (0x1ul << PWM_LEBCTL_SRCEN0_Pos)                  /*!< PWM_T::LEBCTL: SRCEN0 Mask             */

#define PWM_LEBCTL_SRCEN2_Pos            (9)                                               /*!< PWM_T::LEBCTL: SRCEN2 Position         */
#define PWM_LEBCTL_SRCEN2_Msk            (0x1ul << PWM_LEBCTL_SRCEN2_Pos)                  /*!< PWM_T::LEBCTL: SRCEN2 Mask             */

#define PWM_LEBCTL_SRCEN4_Pos            (10)                                              /*!< PWM_T::LEBCTL: SRCEN4 Position         */
#define PWM_LEBCTL_SRCEN4_Msk            (0x1ul << PWM_LEBCTL_SRCEN4_Pos)                  /*!< PWM_T::LEBCTL: SRCEN4 Mask             */

#define PWM_LEBCTL_TRGTYPE_Pos           (16)                                              /*!< PWM_T::LEBCTL: TRGTYPE Position        */
#define PWM_LEBCTL_TRGTYPE_Msk           (0x3ul << PWM_LEBCTL_TRGTYPE_Pos)                 /*!< PWM_T::LEBCTL: TRGTYPE Mask            */

#define PWM_LEBCNT_LEBCNT_Pos            (0)                                               /*!< PWM_T::LEBCNT: LEBCNT Position         */
#define PWM_LEBCNT_LEBCNT_Msk            (0x1fful << PWM_LEBCNT_LEBCNT_Pos)                /*!< PWM_T::LEBCNT: LEBCNT Mask             */

#define PWM_STATUS_CNTMAXF0_Pos          (0)                                               /*!< PWM_T::STATUS: CNTMAXF0 Position       */
#define PWM_STATUS_CNTMAXF0_Msk          (0x1ul << PWM_STATUS_CNTMAXF0_Pos)                /*!< PWM_T::STATUS: CNTMAXF0 Mask           */

#define PWM_STATUS_CNTMAXF1_Pos          (1)                                               /*!< PWM_T::STATUS: CNTMAXF1 Position       */
#define PWM_STATUS_CNTMAXF1_Msk          (0x1ul << PWM_STATUS_CNTMAXF1_Pos)                /*!< PWM_T::STATUS: CNTMAXF1 Mask           */

#define PWM_STATUS_CNTMAXF2_Pos          (2)                                               /*!< PWM_T::STATUS: CNTMAXF2 Position       */
#define PWM_STATUS_CNTMAXF2_Msk          (0x1ul << PWM_STATUS_CNTMAXF2_Pos)                /*!< PWM_T::STATUS: CNTMAXF2 Mask           */

#define PWM_STATUS_CNTMAXF3_Pos          (3)                                               /*!< PWM_T::STATUS: CNTMAXF3 Position       */
#define PWM_STATUS_CNTMAXF3_Msk          (0x1ul << PWM_STATUS_CNTMAXF3_Pos)                /*!< PWM_T::STATUS: CNTMAXF3 Mask           */

#define PWM_STATUS_CNTMAXF4_Pos          (4)                                               /*!< PWM_T::STATUS: CNTMAXF4 Position       */
#define PWM_STATUS_CNTMAXF4_Msk          (0x1ul << PWM_STATUS_CNTMAXF4_Pos)                /*!< PWM_T::STATUS: CNTMAXF4 Mask           */

#define PWM_STATUS_CNTMAXF5_Pos          (5)                                               /*!< PWM_T::STATUS: CNTMAXF5 Position       */
#define PWM_STATUS_CNTMAXF5_Msk          (0x1ul << PWM_STATUS_CNTMAXF5_Pos)                /*!< PWM_T::STATUS: CNTMAXF5 Mask           */

#define PWM_STATUS_SYNCINF0_Pos          (8)                                               /*!< PWM_T::STATUS: SYNCINF0 Position       */
#define PWM_STATUS_SYNCINF0_Msk          (0x1ul << PWM_STATUS_SYNCINF0_Pos)                /*!< PWM_T::STATUS: SYNCINF0 Mask           */

#define PWM_STATUS_SYNCINF2_Pos          (9)                                               /*!< PWM_T::STATUS: SYNCINF2 Position       */
#define PWM_STATUS_SYNCINF2_Msk          (0x1ul << PWM_STATUS_SYNCINF2_Pos)                /*!< PWM_T::STATUS: SYNCINF2 Mask           */

#define PWM_STATUS_SYNCINF4_Pos          (10)                                              /*!< PWM_T::STATUS: SYNCINF4 Position       */
#define PWM_STATUS_SYNCINF4_Msk          (0x1ul << PWM_STATUS_SYNCINF4_Pos)                /*!< PWM_T::STATUS: SYNCINF4 Mask           */

#define PWM_STATUS_ADCTRGF0_Pos          (16)                                              /*!< PWM_T::STATUS: ADCTRGF0 Position       */
#define PWM_STATUS_ADCTRGF0_Msk          (0x1ul << PWM_STATUS_ADCTRGF0_Pos)                /*!< PWM_T::STATUS: ADCTRGF0 Mask           */

#define PWM_STATUS_ADCTRGF1_Pos          (17)                                              /*!< PWM_T::STATUS: ADCTRGF1 Position       */
#define PWM_STATUS_ADCTRGF1_Msk          (0x1ul << PWM_STATUS_ADCTRGF1_Pos)                /*!< PWM_T::STATUS: ADCTRGF1 Mask           */

#define PWM_STATUS_ADCTRGF2_Pos          (18)                                              /*!< PWM_T::STATUS: ADCTRGF2 Position       */
#define PWM_STATUS_ADCTRGF2_Msk          (0x1ul << PWM_STATUS_ADCTRGF2_Pos)                /*!< PWM_T::STATUS: ADCTRGF2 Mask           */

#define PWM_STATUS_ADCTRGF3_Pos          (19)                                              /*!< PWM_T::STATUS: ADCTRGF3 Position       */
#define PWM_STATUS_ADCTRGF3_Msk          (0x1ul << PWM_STATUS_ADCTRGF3_Pos)                /*!< PWM_T::STATUS: ADCTRGF3 Mask           */

#define PWM_STATUS_ADCTRGF4_Pos          (20)                                              /*!< PWM_T::STATUS: ADCTRGF4 Position       */
#define PWM_STATUS_ADCTRGF4_Msk          (0x1ul << PWM_STATUS_ADCTRGF4_Pos)                /*!< PWM_T::STATUS: ADCTRGF4 Mask           */

#define PWM_STATUS_ADCTRGF5_Pos          (21)                                              /*!< PWM_T::STATUS: ADCTRGF5 Position       */
#define PWM_STATUS_ADCTRGF5_Msk          (0x1ul << PWM_STATUS_ADCTRGF5_Pos)                /*!< PWM_T::STATUS: ADCTRGF5 Mask           */

#define PWM_CAPINEN_CAPINEN0_Pos         (0)                                               /*!< PWM_T::CAPINEN: CAPINEN0 Position      */
#define PWM_CAPINEN_CAPINEN0_Msk         (0x1ul << PWM_CAPINEN_CAPINEN0_Pos)               /*!< PWM_T::CAPINEN: CAPINEN0 Mask          */

#define PWM_CAPINEN_CAPINEN1_Pos         (1)                                               /*!< PWM_T::CAPINEN: CAPINEN1 Position      */
#define PWM_CAPINEN_CAPINEN1_Msk         (0x1ul << PWM_CAPINEN_CAPINEN1_Pos)               /*!< PWM_T::CAPINEN: CAPINEN1 Mask          */

#define PWM_CAPINEN_CAPINEN2_Pos         (2)                                               /*!< PWM_T::CAPINEN: CAPINEN2 Position      */
#define PWM_CAPINEN_CAPINEN2_Msk         (0x1ul << PWM_CAPINEN_CAPINEN2_Pos)               /*!< PWM_T::CAPINEN: CAPINEN2 Mask          */

#define PWM_CAPINEN_CAPINEN3_Pos         (3)                                               /*!< PWM_T::CAPINEN: CAPINEN3 Position      */
#define PWM_CAPINEN_CAPINEN3_Msk         (0x1ul << PWM_CAPINEN_CAPINEN3_Pos)               /*!< PWM_T::CAPINEN: CAPINEN3 Mask          */

#define PWM_CAPINEN_CAPINEN4_Pos         (4)                                               /*!< PWM_T::CAPINEN: CAPINEN4 Position      */
#define PWM_CAPINEN_CAPINEN4_Msk         (0x1ul << PWM_CAPINEN_CAPINEN4_Pos)               /*!< PWM_T::CAPINEN: CAPINEN4 Mask          */

#define PWM_CAPINEN_CAPINEN5_Pos         (5)                                               /*!< PWM_T::CAPINEN: CAPINEN5 Position      */
#define PWM_CAPINEN_CAPINEN5_Msk         (0x1ul << PWM_CAPINEN_CAPINEN5_Pos)               /*!< PWM_T::CAPINEN: CAPINEN5 Mask          */

#define PWM_CAPCTL_CAPEN0_Pos            (0)                                               /*!< PWM_T::CAPCTL: CAPEN0 Position         */
#define PWM_CAPCTL_CAPEN0_Msk            (0x1ul << PWM_CAPCTL_CAPEN0_Pos)                  /*!< PWM_T::CAPCTL: CAPEN0 Mask             */

#define PWM_CAPCTL_CAPEN1_Pos            (1)                                               /*!< PWM_T::CAPCTL: CAPEN1 Position         */
#define PWM_CAPCTL_CAPEN1_Msk            (0x1ul << PWM_CAPCTL_CAPEN1_Pos)                  /*!< PWM_T::CAPCTL: CAPEN1 Mask             */

#define PWM_CAPCTL_CAPEN2_Pos            (2)                                               /*!< PWM_T::CAPCTL: CAPEN2 Position         */
#define PWM_CAPCTL_CAPEN2_Msk            (0x1ul << PWM_CAPCTL_CAPEN2_Pos)                  /*!< PWM_T::CAPCTL: CAPEN2 Mask             */

#define PWM_CAPCTL_CAPEN3_Pos            (3)                                               /*!< PWM_T::CAPCTL: CAPEN3 Position         */
#define PWM_CAPCTL_CAPEN3_Msk            (0x1ul << PWM_CAPCTL_CAPEN3_Pos)                  /*!< PWM_T::CAPCTL: CAPEN3 Mask             */

#define PWM_CAPCTL_CAPEN4_Pos            (4)                                               /*!< PWM_T::CAPCTL: CAPEN4 Position         */
#define PWM_CAPCTL_CAPEN4_Msk            (0x1ul << PWM_CAPCTL_CAPEN4_Pos)                  /*!< PWM_T::CAPCTL: CAPEN4 Mask             */

#define PWM_CAPCTL_CAPEN5_Pos            (5)                                               /*!< PWM_T::CAPCTL: CAPEN5 Position         */
#define PWM_CAPCTL_CAPEN5_Msk            (0x1ul << PWM_CAPCTL_CAPEN5_Pos)                  /*!< PWM_T::CAPCTL: CAPEN5 Mask             */

#define PWM_CAPCTL_CAPINV0_Pos           (8)                                               /*!< PWM_T::CAPCTL: CAPINV0 Position        */
#define PWM_CAPCTL_CAPINV0_Msk           (0x1ul << PWM_CAPCTL_CAPINV0_Pos)                 /*!< PWM_T::CAPCTL: CAPINV0 Mask            */

#define PWM_CAPCTL_CAPINV1_Pos           (9)                                               /*!< PWM_T::CAPCTL: CAPINV1 Position        */
#define PWM_CAPCTL_CAPINV1_Msk           (0x1ul << PWM_CAPCTL_CAPINV1_Pos)                 /*!< PWM_T::CAPCTL: CAPINV1 Mask            */

#define PWM_CAPCTL_CAPINV2_Pos           (10)                                              /*!< PWM_T::CAPCTL: CAPINV2 Position        */
#define PWM_CAPCTL_CAPINV2_Msk           (0x1ul << PWM_CAPCTL_CAPINV2_Pos)                 /*!< PWM_T::CAPCTL: CAPINV2 Mask            */

#define PWM_CAPCTL_CAPINV3_Pos           (11)                                              /*!< PWM_T::CAPCTL: CAPINV3 Position        */
#define PWM_CAPCTL_CAPINV3_Msk           (0x1ul << PWM_CAPCTL_CAPINV3_Pos)                 /*!< PWM_T::CAPCTL: CAPINV3 Mask            */

#define PWM_CAPCTL_CAPINV4_Pos           (12)                                              /*!< PWM_T::CAPCTL: CAPINV4 Position        */
#define PWM_CAPCTL_CAPINV4_Msk           (0x1ul << PWM_CAPCTL_CAPINV4_Pos)                 /*!< PWM_T::CAPCTL: CAPINV4 Mask            */

#define PWM_CAPCTL_CAPINV5_Pos           (13)                                              /*!< PWM_T::CAPCTL: CAPINV5 Position        */
#define PWM_CAPCTL_CAPINV5_Msk           (0x1ul << PWM_CAPCTL_CAPINV5_Pos)                 /*!< PWM_T::CAPCTL: CAPINV5 Mask            */

#define PWM_CAPCTL_RCRLDEN0_Pos          (16)                                              /*!< PWM_T::CAPCTL: RCRLDEN0 Position       */
#define PWM_CAPCTL_RCRLDEN0_Msk          (0x1ul << PWM_CAPCTL_RCRLDEN0_Pos)                /*!< PWM_T::CAPCTL: RCRLDEN0 Mask           */

#define PWM_CAPCTL_RCRLDEN1_Pos          (17)                                              /*!< PWM_T::CAPCTL: RCRLDEN1 Position       */
#define PWM_CAPCTL_RCRLDEN1_Msk          (0x1ul << PWM_CAPCTL_RCRLDEN1_Pos)                /*!< PWM_T::CAPCTL: RCRLDEN1 Mask           */

#define PWM_CAPCTL_RCRLDEN2_Pos          (18)                                              /*!< PWM_T::CAPCTL: RCRLDEN2 Position       */
#define PWM_CAPCTL_RCRLDEN2_Msk          (0x1ul << PWM_CAPCTL_RCRLDEN2_Pos)                /*!< PWM_T::CAPCTL: RCRLDEN2 Mask           */

#define PWM_CAPCTL_RCRLDEN3_Pos          (19)                                              /*!< PWM_T::CAPCTL: RCRLDEN3 Position       */
#define PWM_CAPCTL_RCRLDEN3_Msk          (0x1ul << PWM_CAPCTL_RCRLDEN3_Pos)                /*!< PWM_T::CAPCTL: RCRLDEN3 Mask           */

#define PWM_CAPCTL_RCRLDEN4_Pos          (20)                                              /*!< PWM_T::CAPCTL: RCRLDEN4 Position       */
#define PWM_CAPCTL_RCRLDEN4_Msk          (0x1ul << PWM_CAPCTL_RCRLDEN4_Pos)                /*!< PWM_T::CAPCTL: RCRLDEN4 Mask           */

#define PWM_CAPCTL_RCRLDEN5_Pos          (21)                                              /*!< PWM_T::CAPCTL: RCRLDEN5 Position       */
#define PWM_CAPCTL_RCRLDEN5_Msk          (0x1ul << PWM_CAPCTL_RCRLDEN5_Pos)                /*!< PWM_T::CAPCTL: RCRLDEN5 Mask           */

#define PWM_CAPCTL_FCRLDEN0_Pos          (24)                                              /*!< PWM_T::CAPCTL: FCRLDEN0 Position       */
#define PWM_CAPCTL_FCRLDEN0_Msk          (0x1ul << PWM_CAPCTL_FCRLDEN0_Pos)                /*!< PWM_T::CAPCTL: FCRLDEN0 Mask           */

#define PWM_CAPCTL_FCRLDEN1_Pos          (25)                                              /*!< PWM_T::CAPCTL: FCRLDEN1 Position       */
#define PWM_CAPCTL_FCRLDEN1_Msk          (0x1ul << PWM_CAPCTL_FCRLDEN1_Pos)                /*!< PWM_T::CAPCTL: FCRLDEN1 Mask           */

#define PWM_CAPCTL_FCRLDEN2_Pos          (26)                                              /*!< PWM_T::CAPCTL: FCRLDEN2 Position       */
#define PWM_CAPCTL_FCRLDEN2_Msk          (0x1ul << PWM_CAPCTL_FCRLDEN2_Pos)                /*!< PWM_T::CAPCTL: FCRLDEN2 Mask           */

#define PWM_CAPCTL_FCRLDEN3_Pos          (27)                                              /*!< PWM_T::CAPCTL: FCRLDEN3 Position       */
#define PWM_CAPCTL_FCRLDEN3_Msk          (0x1ul << PWM_CAPCTL_FCRLDEN3_Pos)                /*!< PWM_T::CAPCTL: FCRLDEN3 Mask           */

#define PWM_CAPCTL_FCRLDEN4_Pos          (28)                                              /*!< PWM_T::CAPCTL: FCRLDEN4 Position       */
#define PWM_CAPCTL_FCRLDEN4_Msk          (0x1ul << PWM_CAPCTL_FCRLDEN4_Pos)                /*!< PWM_T::CAPCTL: FCRLDEN4 Mask           */

#define PWM_CAPCTL_FCRLDEN5_Pos          (29)                                              /*!< PWM_T::CAPCTL: FCRLDEN5 Position       */
#define PWM_CAPCTL_FCRLDEN5_Msk          (0x1ul << PWM_CAPCTL_FCRLDEN5_Pos)                /*!< PWM_T::CAPCTL: FCRLDEN5 Mask           */

#define PWM_CAPSTS_CRLIFOV0_Pos          (0)                                               /*!< PWM_T::CAPSTS: CRLIFOV0 Position       */
#define PWM_CAPSTS_CRLIFOV0_Msk          (0x1ul << PWM_CAPSTS_CRLIFOV0_Pos)                /*!< PWM_T::CAPSTS: CRLIFOV0 Mask           */

#define PWM_CAPSTS_CRLIFOV1_Pos          (1)                                               /*!< PWM_T::CAPSTS: CRLIFOV1 Position       */
#define PWM_CAPSTS_CRLIFOV1_Msk          (0x1ul << PWM_CAPSTS_CRLIFOV1_Pos)                /*!< PWM_T::CAPSTS: CRLIFOV1 Mask           */

#define PWM_CAPSTS_CRLIFOV2_Pos          (2)                                               /*!< PWM_T::CAPSTS: CRLIFOV2 Position       */
#define PWM_CAPSTS_CRLIFOV2_Msk          (0x1ul << PWM_CAPSTS_CRLIFOV2_Pos)                /*!< PWM_T::CAPSTS: CRLIFOV2 Mask           */

#define PWM_CAPSTS_CRLIFOV3_Pos          (3)                                               /*!< PWM_T::CAPSTS: CRLIFOV3 Position       */
#define PWM_CAPSTS_CRLIFOV3_Msk          (0x1ul << PWM_CAPSTS_CRLIFOV3_Pos)                /*!< PWM_T::CAPSTS: CRLIFOV3 Mask           */

#define PWM_CAPSTS_CRLIFOV4_Pos          (4)                                               /*!< PWM_T::CAPSTS: CRLIFOV4 Position       */
#define PWM_CAPSTS_CRLIFOV4_Msk          (0x1ul << PWM_CAPSTS_CRLIFOV4_Pos)                /*!< PWM_T::CAPSTS: CRLIFOV4 Mask           */

#define PWM_CAPSTS_CRLIFOV5_Pos          (5)                                               /*!< PWM_T::CAPSTS: CRLIFOV5 Position       */
#define PWM_CAPSTS_CRLIFOV5_Msk          (0x1ul << PWM_CAPSTS_CRLIFOV5_Pos)                /*!< PWM_T::CAPSTS: CRLIFOV5 Mask           */

#define PWM_CAPSTS_CFLIFOV0_Pos          (8)                                               /*!< PWM_T::CAPSTS: CFLIFOV0 Position       */
#define PWM_CAPSTS_CFLIFOV0_Msk          (0x1ul << PWM_CAPSTS_CFLIFOV0_Pos)                /*!< PWM_T::CAPSTS: CFLIFOV0 Mask           */

#define PWM_CAPSTS_CFLIFOV1_Pos          (9)                                               /*!< PWM_T::CAPSTS: CFLIFOV1 Position       */
#define PWM_CAPSTS_CFLIFOV1_Msk          (0x1ul << PWM_CAPSTS_CFLIFOV1_Pos)                /*!< PWM_T::CAPSTS: CFLIFOV1 Mask           */

#define PWM_CAPSTS_CFLIFOV2_Pos          (10)                                              /*!< PWM_T::CAPSTS: CFLIFOV2 Position       */
#define PWM_CAPSTS_CFLIFOV2_Msk          (0x1ul << PWM_CAPSTS_CFLIFOV2_Pos)                /*!< PWM_T::CAPSTS: CFLIFOV2 Mask           */

#define PWM_CAPSTS_CFLIFOV3_Pos          (11)                                              /*!< PWM_T::CAPSTS: CFLIFOV3 Position       */
#define PWM_CAPSTS_CFLIFOV3_Msk          (0x1ul << PWM_CAPSTS_CFLIFOV3_Pos)                /*!< PWM_T::CAPSTS: CFLIFOV3 Mask           */

#define PWM_CAPSTS_CFLIFOV4_Pos          (12)                                              /*!< PWM_T::CAPSTS: CFLIFOV4 Position       */
#define PWM_CAPSTS_CFLIFOV4_Msk          (0x1ul << PWM_CAPSTS_CFLIFOV4_Pos)                /*!< PWM_T::CAPSTS: CFLIFOV4 Mask           */

#define PWM_CAPSTS_CFLIFOV5_Pos          (13)                                              /*!< PWM_T::CAPSTS: CFLIFOV5 Position       */
#define PWM_CAPSTS_CFLIFOV5_Msk          (0x1ul << PWM_CAPSTS_CFLIFOV5_Pos)                /*!< PWM_T::CAPSTS: CFLIFOV5 Mask           */

#define PWM_RCAPDAT0_RCAPDAT_Pos         (0)                                               /*!< PWM_T::RCAPDAT0: RCAPDAT Position      */
#define PWM_RCAPDAT0_RCAPDAT_Msk         (0xfffful << PWM_RCAPDAT0_RCAPDAT_Pos)            /*!< PWM_T::RCAPDAT0: RCAPDAT Mask          */

#define PWM_FCAPDAT0_FCAPDAT_Pos         (0)                                               /*!< PWM_T::FCAPDAT0: FCAPDAT Position      */
#define PWM_FCAPDAT0_FCAPDAT_Msk         (0xfffful << PWM_FCAPDAT0_FCAPDAT_Pos)            /*!< PWM_T::FCAPDAT0: FCAPDAT Mask          */

#define PWM_RCAPDAT1_RCAPDAT_Pos         (0)                                               /*!< PWM_T::RCAPDAT1: RCAPDAT Position      */
#define PWM_RCAPDAT1_RCAPDAT_Msk         (0xfffful << PWM_RCAPDAT1_RCAPDAT_Pos)            /*!< PWM_T::RCAPDAT1: RCAPDAT Mask          */

#define PWM_FCAPDAT1_FCAPDAT_Pos         (0)                                               /*!< PWM_T::FCAPDAT1: FCAPDAT Position      */
#define PWM_FCAPDAT1_FCAPDAT_Msk         (0xfffful << PWM_FCAPDAT1_FCAPDAT_Pos)            /*!< PWM_T::FCAPDAT1: FCAPDAT Mask          */

#define PWM_RCAPDAT2_RCAPDAT_Pos         (0)                                               /*!< PWM_T::RCAPDAT2: RCAPDAT Position      */
#define PWM_RCAPDAT2_RCAPDAT_Msk         (0xfffful << PWM_RCAPDAT2_RCAPDAT_Pos)            /*!< PWM_T::RCAPDAT2: RCAPDAT Mask          */

#define PWM_FCAPDAT2_FCAPDAT_Pos         (0)                                               /*!< PWM_T::FCAPDAT2: FCAPDAT Position      */
#define PWM_FCAPDAT2_FCAPDAT_Msk         (0xfffful << PWM_FCAPDAT2_FCAPDAT_Pos)            /*!< PWM_T::FCAPDAT2: FCAPDAT Mask          */

#define PWM_RCAPDAT3_RCAPDAT_Pos         (0)                                               /*!< PWM_T::RCAPDAT3: RCAPDAT Position      */
#define PWM_RCAPDAT3_RCAPDAT_Msk         (0xfffful << PWM_RCAPDAT3_RCAPDAT_Pos)            /*!< PWM_T::RCAPDAT3: RCAPDAT Mask          */

#define PWM_FCAPDAT3_FCAPDAT_Pos         (0)                                               /*!< PWM_T::FCAPDAT3: FCAPDAT Position      */
#define PWM_FCAPDAT3_FCAPDAT_Msk         (0xfffful << PWM_FCAPDAT3_FCAPDAT_Pos)            /*!< PWM_T::FCAPDAT3: FCAPDAT Mask          */

#define PWM_RCAPDAT4_RCAPDAT_Pos         (0)                                               /*!< PWM_T::RCAPDAT4: RCAPDAT Position      */
#define PWM_RCAPDAT4_RCAPDAT_Msk         (0xfffful << PWM_RCAPDAT4_RCAPDAT_Pos)            /*!< PWM_T::RCAPDAT4: RCAPDAT Mask          */

#define PWM_FCAPDAT4_FCAPDAT_Pos         (0)                                               /*!< PWM_T::FCAPDAT4: FCAPDAT Position      */
#define PWM_FCAPDAT4_FCAPDAT_Msk         (0xfffful << PWM_FCAPDAT4_FCAPDAT_Pos)            /*!< PWM_T::FCAPDAT4: FCAPDAT Mask          */

#define PWM_RCAPDAT5_RCAPDAT_Pos         (0)                                               /*!< PWM_T::RCAPDAT5: RCAPDAT Position      */
#define PWM_RCAPDAT5_RCAPDAT_Msk         (0xfffful << PWM_RCAPDAT5_RCAPDAT_Pos)            /*!< PWM_T::RCAPDAT5: RCAPDAT Mask          */

#define PWM_FCAPDAT5_FCAPDAT_Pos         (0)                                               /*!< PWM_T::FCAPDAT5: FCAPDAT Position      */
#define PWM_FCAPDAT5_FCAPDAT_Msk         (0xfffful << PWM_FCAPDAT5_FCAPDAT_Pos)            /*!< PWM_T::FCAPDAT5: FCAPDAT Mask          */

#define PWM_PDMACTL_CHEN0_1_Pos          (0)                                               /*!< PWM_T::PDMACTL: CHEN0_1 Position       */
#define PWM_PDMACTL_CHEN0_1_Msk          (0x1ul << PWM_PDMACTL_CHEN0_1_Pos)                /*!< PWM_T::PDMACTL: CHEN0_1 Mask           */

#define PWM_PDMACTL_CAPMOD0_1_Pos        (1)                                               /*!< PWM_T::PDMACTL: CAPMOD0_1 Position     */
#define PWM_PDMACTL_CAPMOD0_1_Msk        (0x3ul << PWM_PDMACTL_CAPMOD0_1_Pos)              /*!< PWM_T::PDMACTL: CAPMOD0_1 Mask         */

#define PWM_PDMACTL_CAPORD0_1_Pos        (3)                                               /*!< PWM_T::PDMACTL: CAPORD0_1 Position     */
#define PWM_PDMACTL_CAPORD0_1_Msk        (0x1ul << PWM_PDMACTL_CAPORD0_1_Pos)              /*!< PWM_T::PDMACTL: CAPORD0_1 Mask         */

#define PWM_PDMACTL_CHSEL0_1_Pos         (4)                                               /*!< PWM_T::PDMACTL: CHSEL0_1 Position      */
#define PWM_PDMACTL_CHSEL0_1_Msk         (0x1ul << PWM_PDMACTL_CHSEL0_1_Pos)               /*!< PWM_T::PDMACTL: CHSEL0_1 Mask          */

#define PWM_PDMACTL_CHEN2_3_Pos          (8)                                               /*!< PWM_T::PDMACTL: CHEN2_3 Position       */
#define PWM_PDMACTL_CHEN2_3_Msk          (0x1ul << PWM_PDMACTL_CHEN2_3_Pos)                /*!< PWM_T::PDMACTL: CHEN2_3 Mask           */

#define PWM_PDMACTL_CAPMOD2_3_Pos        (9)                                               /*!< PWM_T::PDMACTL: CAPMOD2_3 Position     */
#define PWM_PDMACTL_CAPMOD2_3_Msk        (0x3ul << PWM_PDMACTL_CAPMOD2_3_Pos)              /*!< PWM_T::PDMACTL: CAPMOD2_3 Mask         */

#define PWM_PDMACTL_CAPORD2_3_Pos        (11)                                              /*!< PWM_T::PDMACTL: CAPORD2_3 Position     */
#define PWM_PDMACTL_CAPORD2_3_Msk        (0x1ul << PWM_PDMACTL_CAPORD2_3_Pos)              /*!< PWM_T::PDMACTL: CAPORD2_3 Mask         */

#define PWM_PDMACTL_CHSEL2_3_Pos         (12)                                              /*!< PWM_T::PDMACTL: CHSEL2_3 Position      */
#define PWM_PDMACTL_CHSEL2_3_Msk         (0x1ul << PWM_PDMACTL_CHSEL2_3_Pos)               /*!< PWM_T::PDMACTL: CHSEL2_3 Mask          */

#define PWM_PDMACTL_CHEN4_5_Pos          (16)                                              /*!< PWM_T::PDMACTL: CHEN4_5 Position       */
#define PWM_PDMACTL_CHEN4_5_Msk          (0x1ul << PWM_PDMACTL_CHEN4_5_Pos)                /*!< PWM_T::PDMACTL: CHEN4_5 Mask           */

#define PWM_PDMACTL_CAPMOD4_5_Pos        (17)                                              /*!< PWM_T::PDMACTL: CAPMOD4_5 Position     */
#define PWM_PDMACTL_CAPMOD4_5_Msk        (0x3ul << PWM_PDMACTL_CAPMOD4_5_Pos)              /*!< PWM_T::PDMACTL: CAPMOD4_5 Mask         */

#define PWM_PDMACTL_CAPORD4_5_Pos        (19)                                              /*!< PWM_T::PDMACTL: CAPORD4_5 Position     */
#define PWM_PDMACTL_CAPORD4_5_Msk        (0x1ul << PWM_PDMACTL_CAPORD4_5_Pos)              /*!< PWM_T::PDMACTL: CAPORD4_5 Mask         */

#define PWM_PDMACTL_CHSEL4_5_Pos         (20)                                              /*!< PWM_T::PDMACTL: CHSEL4_5 Position      */
#define PWM_PDMACTL_CHSEL4_5_Msk         (0x1ul << PWM_PDMACTL_CHSEL4_5_Pos)               /*!< PWM_T::PDMACTL: CHSEL4_5 Mask          */

#define PWM_PDMACAP0_1_CAPBUF_Pos        (0)                                               /*!< PWM_T::PDMACAP0_1: CAPBUF Position     */
#define PWM_PDMACAP0_1_CAPBUF_Msk        (0xfffful << PWM_PDMACAP0_1_CAPBUF_Pos)           /*!< PWM_T::PDMACAP0_1: CAPBUF Mask         */

#define PWM_PDMACAP2_3_CAPBUF_Pos        (0)                                               /*!< PWM_T::PDMACAP2_3: CAPBUF Position     */
#define PWM_PDMACAP2_3_CAPBUF_Msk        (0xfffful << PWM_PDMACAP2_3_CAPBUF_Pos)           /*!< PWM_T::PDMACAP2_3: CAPBUF Mask         */

#define PWM_PDMACAP4_5_CAPBUF_Pos        (0)                                               /*!< PWM_T::PDMACAP4_5: CAPBUF Position     */
#define PWM_PDMACAP4_5_CAPBUF_Msk        (0xfffful << PWM_PDMACAP4_5_CAPBUF_Pos)           /*!< PWM_T::PDMACAP4_5: CAPBUF Mask         */

#define PWM_CAPIEN_CAPRIEN0_Pos          (0)                                               /*!< PWM_T::CAPIEN: CAPRIEN0 Position       */
#define PWM_CAPIEN_CAPRIEN0_Msk          (0x1ul << PWM_CAPIEN_CAPRIEN0_Pos)                /*!< PWM_T::CAPIEN: CAPRIEN0 Mask           */

#define PWM_CAPIEN_CAPRIEN1_Pos          (1)                                               /*!< PWM_T::CAPIEN: CAPRIEN1 Position       */
#define PWM_CAPIEN_CAPRIEN1_Msk          (0x1ul << PWM_CAPIEN_CAPRIEN1_Pos)                /*!< PWM_T::CAPIEN: CAPRIEN1 Mask           */

#define PWM_CAPIEN_CAPRIEN2_Pos          (2)                                               /*!< PWM_T::CAPIEN: CAPRIEN2 Position       */
#define PWM_CAPIEN_CAPRIEN2_Msk          (0x1ul << PWM_CAPIEN_CAPRIEN2_Pos)                /*!< PWM_T::CAPIEN: CAPRIEN2 Mask           */

#define PWM_CAPIEN_CAPRIEN3_Pos          (3)                                               /*!< PWM_T::CAPIEN: CAPRIEN3 Position       */
#define PWM_CAPIEN_CAPRIEN3_Msk          (0x1ul << PWM_CAPIEN_CAPRIEN3_Pos)                /*!< PWM_T::CAPIEN: CAPRIEN3 Mask           */

#define PWM_CAPIEN_CAPRIEN4_Pos          (4)                                               /*!< PWM_T::CAPIEN: CAPRIEN4 Position       */
#define PWM_CAPIEN_CAPRIEN4_Msk          (0x1ul << PWM_CAPIEN_CAPRIEN4_Pos)                /*!< PWM_T::CAPIEN: CAPRIEN4 Mask           */

#define PWM_CAPIEN_CAPRIEN5_Pos          (5)                                               /*!< PWM_T::CAPIEN: CAPRIEN5 Position       */
#define PWM_CAPIEN_CAPRIEN5_Msk          (0x1ul << PWM_CAPIEN_CAPRIEN5_Pos)                /*!< PWM_T::CAPIEN: CAPRIEN5 Mask           */

#define PWM_CAPIEN_CAPFIEN0_Pos          (8)                                               /*!< PWM_T::CAPIEN: CAPFIEN0 Position       */
#define PWM_CAPIEN_CAPFIEN0_Msk          (0x1ul << PWM_CAPIEN_CAPFIEN0_Pos)                /*!< PWM_T::CAPIEN: CAPFIEN0 Mask           */

#define PWM_CAPIEN_CAPFIEN1_Pos          (9)                                               /*!< PWM_T::CAPIEN: CAPFIEN1 Position       */
#define PWM_CAPIEN_CAPFIEN1_Msk          (0x1ul << PWM_CAPIEN_CAPFIEN1_Pos)                /*!< PWM_T::CAPIEN: CAPFIEN1 Mask           */

#define PWM_CAPIEN_CAPFIEN2_Pos          (10)                                              /*!< PWM_T::CAPIEN: CAPFIEN2 Position       */
#define PWM_CAPIEN_CAPFIEN2_Msk          (0x1ul << PWM_CAPIEN_CAPFIEN2_Pos)                /*!< PWM_T::CAPIEN: CAPFIEN2 Mask           */

#define PWM_CAPIEN_CAPFIEN3_Pos          (11)                                              /*!< PWM_T::CAPIEN: CAPFIEN3 Position       */
#define PWM_CAPIEN_CAPFIEN3_Msk          (0x1ul << PWM_CAPIEN_CAPFIEN3_Pos)                /*!< PWM_T::CAPIEN: CAPFIEN3 Mask           */

#define PWM_CAPIEN_CAPFIEN4_Pos          (12)                                              /*!< PWM_T::CAPIEN: CAPFIEN4 Position       */
#define PWM_CAPIEN_CAPFIEN4_Msk          (0x1ul << PWM_CAPIEN_CAPFIEN4_Pos)                /*!< PWM_T::CAPIEN: CAPFIEN4 Mask           */

#define PWM_CAPIEN_CAPFIEN5_Pos          (13)                                              /*!< PWM_T::CAPIEN: CAPFIEN5 Position       */
#define PWM_CAPIEN_CAPFIEN5_Msk          (0x1ul << PWM_CAPIEN_CAPFIEN5_Pos)                /*!< PWM_T::CAPIEN: CAPFIEN5 Mask           */

#define PWM_CAPIF_CRLIF0_Pos             (0)                                               /*!< PWM_T::CAPIF: CRLIF0 Position          */
#define PWM_CAPIF_CRLIF0_Msk             (0x1ul << PWM_CAPIF_CRLIF0_Pos)                   /*!< PWM_T::CAPIF: CRLIF0 Mask              */

#define PWM_CAPIF_CRLIF1_Pos             (1)                                               /*!< PWM_T::CAPIF: CRLIF1 Position          */
#define PWM_CAPIF_CRLIF1_Msk             (0x1ul << PWM_CAPIF_CRLIF1_Pos)                   /*!< PWM_T::CAPIF: CRLIF1 Mask              */

#define PWM_CAPIF_CRLIF2_Pos             (2)                                               /*!< PWM_T::CAPIF: CRLIF2 Position          */
#define PWM_CAPIF_CRLIF2_Msk             (0x1ul << PWM_CAPIF_CRLIF2_Pos)                   /*!< PWM_T::CAPIF: CRLIF2 Mask              */

#define PWM_CAPIF_CRLIF3_Pos             (3)                                               /*!< PWM_T::CAPIF: CRLIF3 Position          */
#define PWM_CAPIF_CRLIF3_Msk             (0x1ul << PWM_CAPIF_CRLIF3_Pos)                   /*!< PWM_T::CAPIF: CRLIF3 Mask              */

#define PWM_CAPIF_CRLIF4_Pos             (4)                                               /*!< PWM_T::CAPIF: CRLIF4 Position          */
#define PWM_CAPIF_CRLIF4_Msk             (0x1ul << PWM_CAPIF_CRLIF4_Pos)                   /*!< PWM_T::CAPIF: CRLIF4 Mask              */

#define PWM_CAPIF_CRLIF5_Pos             (5)                                               /*!< PWM_T::CAPIF: CRLIF5 Position          */
#define PWM_CAPIF_CRLIF5_Msk             (0x1ul << PWM_CAPIF_CRLIF5_Pos)                   /*!< PWM_T::CAPIF: CRLIF5 Mask              */

#define PWM_CAPIF_CFLIF0_Pos             (8)                                               /*!< PWM_T::CAPIF: CFLIF0 Position          */
#define PWM_CAPIF_CFLIF0_Msk             (0x1ul << PWM_CAPIF_CFLIF0_Pos)                   /*!< PWM_T::CAPIF: CFLIF0 Mask              */

#define PWM_CAPIF_CFLIF1_Pos             (9)                                               /*!< PWM_T::CAPIF: CFLIF1 Position          */
#define PWM_CAPIF_CFLIF1_Msk             (0x1ul << PWM_CAPIF_CFLIF1_Pos)                   /*!< PWM_T::CAPIF: CFLIF1 Mask              */

#define PWM_CAPIF_CFLIF2_Pos             (10)                                              /*!< PWM_T::CAPIF: CFLIF2 Position          */
#define PWM_CAPIF_CFLIF2_Msk             (0x1ul << PWM_CAPIF_CFLIF2_Pos)                   /*!< PWM_T::CAPIF: CFLIF2 Mask              */

#define PWM_CAPIF_CFLIF3_Pos             (11)                                              /*!< PWM_T::CAPIF: CFLIF3 Position          */
#define PWM_CAPIF_CFLIF3_Msk             (0x1ul << PWM_CAPIF_CFLIF3_Pos)                   /*!< PWM_T::CAPIF: CFLIF3 Mask              */

#define PWM_CAPIF_CFLIF4_Pos             (12)                                              /*!< PWM_T::CAPIF: CFLIF4 Position          */
#define PWM_CAPIF_CFLIF4_Msk             (0x1ul << PWM_CAPIF_CFLIF4_Pos)                   /*!< PWM_T::CAPIF: CFLIF4 Mask              */

#define PWM_CAPIF_CFLIF5_Pos             (13)                                              /*!< PWM_T::CAPIF: CFLIF5 Position          */
#define PWM_CAPIF_CFLIF5_Msk             (0x1ul << PWM_CAPIF_CFLIF5_Pos)                   /*!< PWM_T::CAPIF: CFLIF5 Mask              */

#define PWM_PBUF_PBUF_Pos                (0)                                               /*!< PWM_T::PBUF: PBUF Position             */
#define PWM_PBUF_PBUF_Msk                (0xfffful << PWM_PBUF_PBUF_Pos)                   /*!< PWM_T::PBUF: PBUF Mask                 */

#define PWM_CMPBUF_CMPBUF_Pos            (0)                                               /*!< PWM_T::CMPBUF: CMPBUF Position         */
#define PWM_CMPBUF_CMPBUF_Msk            (0xfffful << PWM_CMPBUF_CMPBUF_Pos)               /*!< PWM_T::CMPBUF: CMPBUF Mask             */

#define PWM_CPSCBUF0_1_CPSCBUF_Pos       (0)                                               /*!< PWM_T::CPSCBUF0_1: CPSCBUF Position    */
#define PWM_CPSCBUF0_1_CPSCBUF_Msk       (0xffful << PWM_CPSCBUF0_1_CPSCBUF_Pos)           /*!< PWM_T::CPSCBUF0_1: CPSCBUF Mask        */

#define PWM_CPSCBUF2_3_CPSCBUF_Pos       (0)                                               /*!< PWM_T::CPSCBUF2_3: CPSCBUF Position    */
#define PWM_CPSCBUF2_3_CPSCBUF_Msk       (0xffful << PWM_CPSCBUF2_3_CPSCBUF_Pos)           /*!< PWM_T::CPSCBUF2_3: CPSCBUF Mask        */

#define PWM_CPSCBUF4_5_CPSCBUF_Pos       (0)                                               /*!< PWM_T::CPSCBUF4_5: CPSCBUF Position    */
#define PWM_CPSCBUF4_5_CPSCBUF_Msk       (0xffful << PWM_CPSCBUF4_5_CPSCBUF_Pos)           /*!< PWM_T::CPSCBUF4_5: CPSCBUF Mask        */

#define PWM_FTCBUF0_1_FTCMPBUF_Pos       (0)                                               /*!< PWM_T::FTCBUF0_1: FTCMPBUF Position    */
#define PWM_FTCBUF0_1_FTCMPBUF_Msk       (0xfffful << PWM_FTCBUF0_1_FTCMPBUF_Pos)          /*!< PWM_T::FTCBUF0_1: FTCMPBUF Mask        */

#define PWM_FTCBUF2_3_FTCMPBUF_Pos       (0)                                               /*!< PWM_T::FTCBUF2_3: FTCMPBUF Position    */
#define PWM_FTCBUF2_3_FTCMPBUF_Msk       (0xfffful << PWM_FTCBUF2_3_FTCMPBUF_Pos)          /*!< PWM_T::FTCBUF2_3: FTCMPBUF Mask        */

#define PWM_FTCBUF4_5_FTCMPBUF_Pos       (0)                                               /*!< PWM_T::FTCBUF4_5: FTCMPBUF Position    */
#define PWM_FTCBUF4_5_FTCMPBUF_Msk       (0xfffful << PWM_FTCBUF4_5_FTCMPBUF_Pos)          /*!< PWM_T::FTCBUF4_5: FTCMPBUF Mask        */

#define PWM_FTCI_FTCMU0_Pos              (0)                                               /*!< PWM_T::FTCI: FTCMU0 Position           */
#define PWM_FTCI_FTCMU0_Msk              (0x1ul << PWM_FTCI_FTCMU0_Pos)                    /*!< PWM_T::FTCI: FTCMU0 Mask               */

#define PWM_FTCI_FTCMU2_Pos              (1)                                               /*!< PWM_T::FTCI: FTCMU2 Position           */
#define PWM_FTCI_FTCMU2_Msk              (0x1ul << PWM_FTCI_FTCMU2_Pos)                    /*!< PWM_T::FTCI: FTCMU2 Mask               */

#define PWM_FTCI_FTCMU4_Pos              (2)                                               /*!< PWM_T::FTCI: FTCMU4 Position           */
#define PWM_FTCI_FTCMU4_Msk              (0x1ul << PWM_FTCI_FTCMU4_Pos)                    /*!< PWM_T::FTCI: FTCMU4 Mask               */

#define PWM_FTCI_FTCMD0_Pos              (8)                                               /*!< PWM_T::FTCI: FTCMD0 Position           */
#define PWM_FTCI_FTCMD0_Msk              (0x1ul << PWM_FTCI_FTCMD0_Pos)                    /*!< PWM_T::FTCI: FTCMD0 Mask               */

#define PWM_FTCI_FTCMD2_Pos              (9)                                               /*!< PWM_T::FTCI: FTCMD2 Position           */
#define PWM_FTCI_FTCMD2_Msk              (0x1ul << PWM_FTCI_FTCMD2_Pos)                    /*!< PWM_T::FTCI: FTCMD2 Mask               */

#define PWM_FTCI_FTCMD4_Pos              (10)                                              /*!< PWM_T::FTCI: FTCMD4 Position           */
#define PWM_FTCI_FTCMD4_Msk              (0x1ul << PWM_FTCI_FTCMD4_Pos)                    /*!< PWM_T::FTCI: FTCMD4 Mask               */

/**@}*/ /* PWM_CONST */
/**@}*/ /* end of PWM register group */


/*---------------------- Real Time Clock Controller -------------------------*/
/**
    @addtogroup RTC Real Time Clock Controller(RTC)
    Memory Mapped Structure for RTC Controller
@{ */

typedef struct
{


    /**
     * @var RTC_T::INIT
     * Offset: 0x00  RTC Initiation Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |INIT_ACTIVE|RTC Active Status (Read Only)
     * |        |          |0 = RTC is at reset state.
     * |        |          |1 = RTC is at normal active state.
     * |[31:1]  |INIT      |RTC Initiation
     * |        |          |When RTC block is first powered on, RTC is at reset state
     * |        |          |User has to write a special number (0xA5EB13570x a5eb1357) to INIT to make RTC leaving reset state
     * |        |          |Once the INIT is written as 0xa5eb13570xA5EB1357, the RTC will be in un-resetat normal active state permanently.
     * |        |          |The INIT[31:1] is a write-only field and read value will be always 0.
     * @var RTC_T::RWEN
     * Offset: 0x04  RTC Access Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RWEN      |RTC Register Access Enable Password (Write Only)
     * |        |          |Writing 0xA965 to this field register will enable RTC register access period and keep 1024 RTC clocks.
     * |        |          |Note: Writing others vaule will clear RWENF and disable RTC register access function immediately.
     * |[16]    |RWENF     |RTC Register Access Enable Flag (Read Only)
     * |        |          |0 = RTC register read/write Disabled.
     * |        |          |1 = RTC register read/write Enabled.
     * |        |          |This bit will be set after RWEN RTC_RWEN[15:0] register is load a 0xA965, and be cleared automatically after 1024 RTC clocks expired.
     * |        |          |Note: RWENF will be mask to u201C0u201D during RTCBUSY is= 1, and first turn on RTCCKEN (CLK_APBCLK[1]) also.
     * |[24]    |RTCBUSY   |RTC Write Busy Flag
     * |        |          |This bit indicates RTC registers are writable or not.
     * |        |          |0: RTC register write canu2019t writeaccess enable .
     * |        |          |1: RTC register are writablewrite access disable , RTC under Busy Status..
     * |        |          |Note: RTCBUSY falg will be set when execute write RTC register command exceed 6 times within By Exceed RTC IP Prcessing Write Counter Capacity ( 6 counts Per 1 1120 PCLK cycles.).
     * @var RTC_T::FREQADJ
     * Offset: 0x08  RTC Frequency Compensation Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[21:0]  |FREQADJ   |Frequency Compensation RegisterValue
     * |        |          |User must to get actual clock freqency of LXT, LXT frequency.
     * |        |          |When CLK_CLKSEL2[18] = 0.
     * |        |          |FCR = 0x200000 * (32768 / LXT freqency period).
     * |        |          |LXT Note: This formula is suitable only when RTCSEL (CLK_CLKSEL2[18]) is 0, RTC clock source is from LXT.period: the clock period (Hz) of LXT.
     * |        |          |uLIRC period: the clock period (Hz) of LIRC.
     * |        |          |Note : LIRC can measure by softwaremethod.
     * @var RTC_T::TIME
     * Offset: 0x0C  RTC Time Loading Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |SEC       |1-Sec Time Digit (0~9)
     * |[6:4]   |TENSEC    |10-Sec Time Digit (0~5)
     * |[11:8]  |MIN       |1-Min Time Digit (0~9)
     * |[14:12] |TENMIN    |10-Min Time Digit (0~5)
     * |[19:16] |HR        |1-Hour Time Digit (0~9)
     * |[21:20] |TENHR     |10-Hour Time Digit (0~2)
     * |        |          |Note: When RTC runs as 12-hour time scale mode, RTC_TIME[21] (the high bit of TENHR[1:0]) means AM/PM indication, RTC_TIME[21] is 0 means AM hour and RTC_TIME[21] is 1 means PM hour (If RTC_TIME[21] is 1, it indicates PM time message.).
     * @var RTC_T::CAL
     * Offset: 0x10  RTC Calendar Loading Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |DAY       |1-Day Calendar Digit (0~9)
     * |[5:4]   |TENDAY    |10-Day Calendar Digit (0~3)
     * |[11:8]  |MON       |1-Month Calendar Digit (0~9)
     * |[12]    |TENMON    |10-Month Calendar Digit (0~1)
     * |[19:16] |YEAR      |1-Year Calendar Digit (0~9)
     * |[23:20] |TENYEAR   |10-Year Calendar Digit (0~9)
     * @var RTC_T::CLKFMT
     * Offset: 0x14  RTC Time Scale Selection Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |24HEN     |24-hour / 12-hour Time Scale Selection
     * |        |          |Indicates that RTC_TIME and RTC_TALM register are in 24-hour time scale or 12-hour time scale
     * |        |          |0 = 12-hour time scale with AM and PM indication selected.
     * |        |          |1 = 24-hour time scale selected.
     * @var RTC_T::WEEKDAY
     * Offset: 0x18  RTC Day of the Week Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |WEEKDAY   |Day of the Week Register
     * |        |          |000 = Sunday.
     * |        |          |001 = Monday.
     * |        |          |010 = Tuesday.
     * |        |          |011 = Wednesday.
     * |        |          |100 = Thursday.
     * |        |          |101 = Friday.
     * |        |          |110 = Saturday.
     * |        |          |111 = Reserved.
     * |        |          |Note: RTC will not check WEEKDAY setting with RTC_CAL is reasonable or not.
     * @var RTC_T::TALM
     * Offset: 0x1C  RTC Time Alarm Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |SEC       |1-Sec Time Digit of Alarm Setting (0~9)
     * |[6:4]   |TENSEC    |10-Sec Time Digit of Alarm Setting (0~5)
     * |[11:8]  |MIN       |1-Min Time Digit of Alarm Setting (0~9)
     * |[14:12] |TENMIN    |10-Min Time Digit of Alarm Setting (0~5)
     * |[19:16] |HR        |1-Hour Time Digit of Alarm Setting (0~9)
     * |[21:20] |TENHR     |10-Hour Time Digit of Alarm Setting (0~2) When RTC runs as 12-hour time scale mode, RTC_TIME[21] (the high bit of TENHR[1:0]) means AM/PM indication (If RTC_TIME[21] is 1, it indicates PM time message.)
     * @var RTC_T::CALM
     * Offset: 0x20  RTC Calendar Alarm Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |DAY       |1-Day Calendar Digit of Alarm Setting (0~9)
     * |[5:4]   |TENDAY    |10-Day Calendar Digit of Alarm Setting (0~3)
     * |[11:8]  |MON       |1-Month Calendar Digit of Alarm Setting (0~9)
     * |[12]    |TENMON    |10-Month Calendar Digit of Alarm Setting (0~1)
     * |[19:16] |YEAR      |1-Year Calendar Digit of Alarm Setting (0~9)
     * |[23:20] |TENYEAR   |10-Year Calendar Digit of Alarm Setting (0~9)
     * @var RTC_T::LEAPYEAR
     * Offset: 0x24  RTC Leap Year Indicaton Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |LEAPYEAR  |Leap Year Indication Register (Read Only)
     * |        |          |0 = This year is not a leap year.
     * |        |          |1 = This year is leap year.
     * @var RTC_T::INTEN
     * Offset: 0x28  RTC Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ALMIEN    |Alarm Interrupt Enable Bit
     * |        |          |0 = RTC aAlarm interrupt Disabled.
     * |        |          |1 = RTC aAlarm interrupt Enabled.
     * |[1]     |TICKIEN   |Time Tick Interrupt Enable Bit
     * |        |          |0 = RTC tTime tTick interrupt Disabled.
     * |        |          |1 = RTC tTime tTick interrupt Enabled.
     * @var RTC_T::INTSTS
     * Offset: 0x2C  RTC Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ALMIF     |RTC Alarm Interrupt Flag
     * |        |          |When RTC current RTC counter in time counters RTC_TIME and RTC_CAL are matched the alarm setting time registers RTC alarm settings in RTC_TALM and RTC_CALM, ALMIFthis bit will be set to 1 and an alarm interrupt signal will be generated if RTC Alarm Interrupt enabled ALMIEN (RTC_INTEN[0]) is enabledset to 1.
     * |        |          |Chip will also be waken up if RTC Alarm when alarm interrupt signal occurred Interrupt if chip is running at Power-down modeis enabled when chip is at Power-down mode.
     * |        |          |0 = Alarm condition is not matched.
     * |        |          |1 = Alarm condition is matched.
     * |        |          |Note: Write Writing 1 to clear this bit.
     * |[1]     |TICKIF    |RTC Time Tick Interrupt Flag
     * |        |          |When RTC time tick event happened, this bit TICKIF will be set to 1 and an time tick interrupt signal will be generated if RTC Tick Interrupt enabled TICKIEN (RTC_INTEN[1]) is enabledset to 1
     * |        |          |Chip will also be waken up when time tick interrupt signal occurred if RTC Tick Interrupt is enabled and this bit is set to 1 ifwhen chip is running at Power-down mode.
     * |        |          |0 = Tick condition does not occur.
     * |        |          |1 = Tick condition occurred.
     * |        |          |Note: Writing Write 1 to clear to clear this bit.
     * @var RTC_T::TICK
     * Offset: 0x30  RTC Time Tick Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |TICK      |Time Tick Register
     * |        |          |These bits are used to select RTC time tick period for pPeriodic tTime tTick iInterrupt request.
     * |        |          |000 = Time tick is 1 second.
     * |        |          |001 = Time tick is 1/2 second.
     * |        |          |010 = Time tick is 1/4 second.
     * |        |          |011 = Time tick is 1/8 second.
     * |        |          |100 = Time tick is 1/16 second.
     * |        |          |101 = Time tick is 1/32 second.
     * |        |          |110 = Time tick is 1/64 second.
     * |        |          |111 = Time tick is 1/128 second.
     * |        |          |Note: This register can be read back after the RTC register access enable bit RWENF (RTC_RWEN[16]) is active.
     * @var RTC_T::TAMSK
     * Offset: 0x34  RTC Time Alarm Mask Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |MSEC      |Mask 1-Sec Time Digit of aAlarm sSetting (0~9)
     * |[1]     |MTENSEC   |Mask 10-Sec Time Digit of aAlarm sSetting (0~5)
     * |[2]     |MMIN      |Mask 1-Min Time Digit of aAlarm sSetting (0~9)
     * |[3]     |MTENMIN   |Mask 10-Min Time Digit of aAlarm sSetting (0~5)
     * |[4]     |MHR       |Mask 1-Hour Time Digit of aAlarm sSetting (0~9)
     * |        |          |Note: MHR function is only for 24-hour time scale mode.
     * |[5]     |MTENHR    |Mask 10-Hour Time Digit of aAlarm sSetting (0~2)
     * |        |          |Note: MTENHR function is only for 24-hour time scale mode.
     * @var RTC_T::CAMSK
     * Offset: 0x38  RTC Calendar Alarm Mask Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |MDAY      |Mask 1-Day Calendar Digit of Alarm alarm Setting setting (0~9)
     * |[1]     |MTENDAY   |Mask 10-Day Calendar Digit of Alarm alarm Setting setting (0~3)
     * |[2]     |MMON      |Mask 1-Month Calendar Digit of Alarm alarm Setting setting (0~9)
     * |[3]     |MTENMON   |Mask 10-Month Calendar Digit of A alarm Setting setting (0~1)
     * |[4]     |MYEAR     |Mask 1-Year Calendar Digit of Alarm alarm Setting setting (0~9)
     * |[5]     |MTENYEAR  |Mask 10-Year Calendar Digit of A alarm Setting setting (0~9)
     * @var RTC_T::LXTCTL
     * Offset: 0x100  RTC 32 kHz Oscillator Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:1]   |GAIN      |Oscillator Gain Option
     * |        |          |User can select oscillator gain according to crystal external loading and operating temperature range
     * |        |          |The larger gain value corresponding to stronger driving capability and higher power consumption.
     * |        |          |000 = L0 mode.
     * |        |          |001 = L1 mode.
     * |        |          |010 = L2 mode.
     * |        |          |011 = L3 mode.
     * |        |          |100 = L4 mode.
     * |        |          |101 = L5 mode.
     * |        |          |110 = L6 mode.
     * |        |          |111 = L7 mode (Default).
     * @var RTC_T::LXTOCTL
     * Offset: 0x104  RTC X32KO Pin Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |OPMODE    |GPF0 IO Pin Operation Mode
     * |        |          |00 = X32KO (P5.0) is in Iinput only mode, without pull-up resistor.
     * |        |          |01 = X32KO (P5.0) is in output pPush- pull output mode.
     * |        |          |10 = X32KO (P5.0) is in Oopen- drain output mode.
     * |        |          |11 = X32KO (P5.0) is in Quasi-bidirectional modeinput only mode with internal pull up.
     * |[2]     |DOUT      |IO Pin Output Data
     * |        |          |0 = X32KO (P5.0) will driver output low in output mode.
     * |        |          |1 = X32KO (P5.0) will driver high in output mode output high.
     * |[3]     |CTLSEL    |IO Pin State Backup Selection
     * |        |          |When low speed 32 kHz oscillator (LXT) is disabled, X32KO (P5.0) pin can be used as GPIO P5.0 function
     * |        |          |User can program CTLSEL bit to decide X32KO (P5.0) I/O function is controlled by system power domain GPIO module or VBAT power domain RTC_LXTOCTL control register.
     * |        |          |0 = X32KO (P5.0) pin I/O function is controlled by GPIO module
     * |        |          |Hardware auto becomes CTLSEL =1 when system power is turned off..
     * |        |          |1 = X32KO (P5.0) pin I/O function is controlled by OPMODE and DOUT in RTC_LXTOCT at VBAT power domain, X32KO (P5.0) pin function and I/O status are controlled by OPMODE[1:0] and DOUT after CTLSEL it set to 1.
     * |        |          |Note: CTLSEL (this bit) will automatically be set to 1 automatically by hardware to 1 when system power is turned off andoff RTC is at normal active state, ACTIVE (RTC_INIT[0]) is 1 and RTC Active Status = 1.
     * @var RTC_T::LXTICTL
     * Offset: 0x108  RTC X32KI Pin Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |OPMODE    |IO Pin Operation Mode
     * |        |          |00 = X32KI (P5.1) is in Input mode without pull-up resistor.
     * |        |          |01 = X32KI (P5.1) is in Push-pull output mode.
     * |        |          |10 = X32KI (P5.1) is in Open-drain output mode.
     * |        |          |11 = X32KI (P5.1) is in Quasi-bidirectional mode.00 = X32KI (P5.1) is input only mode, without pull-up resistor.
     * |        |          |01 = X32KI (P5.1) is output push pull mode.
     * |        |          |10 = X32KI (P5.1) is open drain mode.
     * |        |          |11 = X32KI (P5.1) is input only mode with internal pull up.
     * |[2]     |DOUT      |IO Pin Output Data
     * |        |          |0 = X32KI (P5.1) will driver low in output mode.
     * |        |          |1 = X32KI (P5.1) will driver high in output mode.0 = X32KI (P5.1) output low.
     * |        |          |1 = X32KI (P5.1) output high.
     * |[3]     |CTLSEL    |IO Pin State Backup Selection
     * |        |          |When low speed 32 kHz oscillator (LXT) is disabled, X32KO pin can be used as GPIO P5.1 function
     * |        |          |User can program CTLSEL to decide X32KI (P5.1) I/O function is controlled by system power domain GPIO module or VBAT power domain RTC_LXTICTL register.
     * |        |          |0 = X32KI (P5.1) pin I/O function is controlled by GPIO module.
     * |        |          |1 = X32KI (P5.1) pin I/O function is controlled by OPMODE and DOUT in RTC_LXTICTL at VBAT power domain.
     * |        |          |Note: CTLSEL will be set to 1 automatically by hardware when system power is turned off and RTC is at normal active state, ACTIVE (RTC_INIT[0]) is 1.When low speed 32 kHz oscillator is disabled, X32KI (P5.1) pin can be used as GPIO function
     * |        |          |User can program CTLSEL bit to decide X32KI (P5.1) I/O function is controlled by system power domain GPIO module or VBAT power domain RTC_LXTICTL control register.
     * |        |          |0 = X32KI (P5.1) pin I/O function is controlled by GPIO module
     * |        |          |Hardware auto becomes CTLSEL =1 when system power is turned off.
     * |        |          |1 = X32KI (P5.1) pin I/O function is controlled by VBAT power domain, X32KI (P5.1) pin function and I/O status are controlled by OPMODE[1:0] and DOUT after CTLSEL it set to 1.
     * |        |          |Note:CTLSEL (this bit) will automatically be set by hardware to 1 when system power is off and RTC Active Status = 1.
     * @var RTC_T::P52CTL
     * Offset: 0x10C  RTC P52 Pin Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |OPMODE    |IO Pin Operation Mode
     * |        |          |00 = P5.2 is in Input mode without pull-up resistor.
     * |        |          |01 = P5.2 is in Push-pull output mode.
     * |        |          |10 = P5.2 is in Open-drain output mode.
     * |        |          |11 = P5.2 is in Quasi-bidirectional mode.00 = GPIO(P5.2) is input only mode, without pull-up resistor.
     * |        |          |01 = GPIO(P5.2) is output push pull mode.
     * |        |          |10 = GPIO(P5.2) is open drain mode.
     * |        |          |11 = GPIO(P5.2) is input only mode with internal pull up.
     * |[2]     |DOUT      |IO Pin Output Data
     * |        |          |0 = GPIO P5.2 will driver low in output mode.
     * |        |          |1 = GPIO P5.2 will driver high in output mode.0 = GPIO(P5.2) output low.
     * |        |          |1 = GPIO (P5.2) output high.
     * |[3]     |CTLSEL    |IO Pin State Backup Selection
     * |        |          |User can program CTLSEL to decide GPIO P5.2 I/O function is controlled by system power domain GPIO module or VBAT power domain RTC_LXTICTL register.
     * |        |          |0 = GPIO P5.2 pin I/O function is controlled by GPIO module.
     * |        |          |1 = GPIO P5.2 pin I/O function is controlled by OPMODE and DOUT in RTC_P52CTL at VBAT power domain.
     * |        |          |Note: CTLSEL will be set to 1 automatically by hardware when system power is turned off and RTC is at normal active state, ACTIVE (RTC_INIT[0]) is 1.User can program CTLSEL bit to decide GPIO (P5.2) I/O function is controlled by system power domain GPIO module or VBAT power domain RTC_GPIOCTL control register.
     * |        |          |0 = GPIO (P5.2) pin I/O function is controlled by GPIO module
     * |        |          |Hardware auto becomes CTLSEL =1 when system power is turned off.
     * |        |          |1 = GPIO (P5.2) pin I/O function is controlled by VBAT power domain, GPIO (P5.2) pin function and I/O status are controlled by OPMODE[1:0] and DOUT after CTLSEL it set to 1.
     * |        |          |Note:CTLSEL (this bit) will automatically be set by hardware to 1 when system power is off and RTC Active Status = 1.
     * @var RTC_T::DSTCTL
     * Offset: 0x110  RTC Daylight Saving Time Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ADDHR     |Add 1 Hour
     * |        |          |0 = No effect.
     * |        |          |1 = Indicates RTC hour digit has been addedAdds 1 one hour for summer time change.
     * |[1]     |SUBHR     |Subtract 1 Hour
     * |        |          |0 = No effect.
     * |        |          |1 = Indicates RTC hour digit has been sSubtractsed one1 hour for winter time change.
     * |        |          |Note: Writing this bit has no effect when RTC_TIME[21:16] is 0x00 in 24-hour time scale mode and RTC_TIME[21:16] is 0x12 in 12-hour time scale mode.
     * |[2]     |DSBAK     |Daylight Saving Back
     * |        |          |0= Daylight Saving Time function isChange is not performed.
     * |        |          |1= Daylight Saving Time function Change is performed.
     */

    __IO uint32_t INIT;                  /*!< [0x0000] RTC Initiation Register                                          */
    __IO uint32_t RWEN;                  /*!< [0x0004] RTC Access Enable Register                                       */
    __IO uint32_t FREQADJ;               /*!< [0x0008] RTC Frequency Compensation Register                              */
    __IO uint32_t TIME;                  /*!< [0x000c] RTC Time Loading Register                                        */
    __IO uint32_t CAL;                   /*!< [0x0010] RTC Calendar Loading Register                                    */
    __IO uint32_t CLKFMT;                /*!< [0x0014] RTC Time Scale Selection Register                                */
    __IO uint32_t WEEKDAY;               /*!< [0x0018] RTC Day of the Week Register                                     */
    __IO uint32_t TALM;                  /*!< [0x001c] RTC Time Alarm Register                                          */
    __IO uint32_t CALM;                  /*!< [0x0020] RTC Calendar Alarm Register                                      */
    __I  uint32_t LEAPYEAR;              /*!< [0x0024] RTC Leap Year Indicaton Register                                 */
    __IO uint32_t INTEN;                 /*!< [0x0028] RTC Interrupt Enable Register                                    */
    __IO uint32_t INTSTS;                /*!< [0x002c] RTC Interrupt Status Register                                    */
    __IO uint32_t TICK;                  /*!< [0x0030] RTC Time Tick Register                                           */
    __IO uint32_t TAMSK;                 /*!< [0x0034] RTC Time Alarm Mask Register                                     */
    __IO uint32_t CAMSK;                 /*!< [0x0038] RTC Calendar Alarm Mask Register                                 */
    __I  uint32_t RESERVE0[49];
    __IO uint32_t LXTCTL;                /*!< [0x0100] RTC 32 kHz Oscillator Control Register                           */
    __IO uint32_t LXTOCTL;               /*!< [0x0104] RTC X32KO Pin Control Register                                   */
    __IO uint32_t LXTICTL;               /*!< [0x0108] RTC X32KI Pin Control Register                                   */
    __IO uint32_t P52CTL;                /*!< [0x010c] RTC P52 Pin Control Register                                     */
    __IO uint32_t DSTCTL;                /*!< [0x0110] RTC Daylight Saving Time Control Register                        */

} RTC_T;

/**
    @addtogroup RTC_CONST RTC Bit Field Definition
    Constant Definitions for RTC Controller
@{ */

#define RTC_INIT_ACTIVE_Pos              (0)                                               /*!< RTC_T::INIT: ACTIVE Position           */
#define RTC_INIT_ACTIVE_Msk              (0x1ul << RTC_INIT_ACTIVE_Pos)                    /*!< RTC_T::INIT: ACTIVE Mask               */

#define RTC_INIT_INIT_Pos                (1)                                               /*!< RTC_T::INIT: INIT Position             */
#define RTC_INIT_INIT_Msk                (0x7ffffffful << RTC_INIT_INIT_Pos)               /*!< RTC_T::INIT: INIT Mask                 */

#define RTC_RWEN_RWEN_Pos                (0)                                               /*!< RTC_T::RWEN: RWEN Position             */
#define RTC_RWEN_RWEN_Msk                (0xfffful << RTC_RWEN_RWEN_Pos)                   /*!< RTC_T::RWEN: RWEN Mask                 */

#define RTC_RWEN_RWENF_Pos               (16)                                              /*!< RTC_T::RWEN: RWENF Position            */
#define RTC_RWEN_RWENF_Msk               (0x1ul << RTC_RWEN_RWENF_Pos)                     /*!< RTC_T::RWEN: RWENF Mask                */

#define RTC_RWEN_RTCBUSY_Pos             (24)                                              /*!< RTC_T::RWEN: RTCBUSY Position          */
#define RTC_RWEN_RTCBUSY_Msk             (0x1ul << RTC_RWEN_RTCBUSY_Pos)                   /*!< RTC_T::RWEN: RTCBUSY Mask              */

#define RTC_FREQADJ_FREQADJ_Pos          (0)                                               /*!< RTC_T::FREQADJ: FREQADJ Position       */
#define RTC_FREQADJ_FREQADJ_Msk          (0x3ffffful << RTC_FREQADJ_FREQADJ_Pos)           /*!< RTC_T::FREQADJ: FREQADJ Mask           */

#define RTC_TIME_SEC_Pos                 (0)                                               /*!< RTC_T::TIME: SEC Position              */
#define RTC_TIME_SEC_Msk                 (0xful << RTC_TIME_SEC_Pos)                       /*!< RTC_T::TIME: SEC Mask                  */

#define RTC_TIME_TENSEC_Pos              (4)                                               /*!< RTC_T::TIME: TENSEC Position           */
#define RTC_TIME_TENSEC_Msk              (0x7ul << RTC_TIME_TENSEC_Pos)                    /*!< RTC_T::TIME: TENSEC Mask               */

#define RTC_TIME_MIN_Pos                 (8)                                               /*!< RTC_T::TIME: MIN Position              */
#define RTC_TIME_MIN_Msk                 (0xful << RTC_TIME_MIN_Pos)                       /*!< RTC_T::TIME: MIN Mask                  */

#define RTC_TIME_TENMIN_Pos              (12)                                              /*!< RTC_T::TIME: TENMIN Position           */
#define RTC_TIME_TENMIN_Msk              (0x7ul << RTC_TIME_TENMIN_Pos)                    /*!< RTC_T::TIME: TENMIN Mask               */

#define RTC_TIME_HR_Pos                  (16)                                              /*!< RTC_T::TIME: HR Position               */
#define RTC_TIME_HR_Msk                  (0xful << RTC_TIME_HR_Pos)                        /*!< RTC_T::TIME: HR Mask                   */

#define RTC_TIME_TENHR_Pos               (20)                                              /*!< RTC_T::TIME: TENHR Position            */
#define RTC_TIME_TENHR_Msk               (0x3ul << RTC_TIME_TENHR_Pos)                     /*!< RTC_T::TIME: TENHR Mask                */

#define RTC_CAL_DAY_Pos                  (0)                                               /*!< RTC_T::CAL: DAY Position               */
#define RTC_CAL_DAY_Msk                  (0xful << RTC_CAL_DAY_Pos)                        /*!< RTC_T::CAL: DAY Mask                   */

#define RTC_CAL_TENDAY_Pos               (4)                                               /*!< RTC_T::CAL: TENDAY Position            */
#define RTC_CAL_TENDAY_Msk               (0x3ul << RTC_CAL_TENDAY_Pos)                     /*!< RTC_T::CAL: TENDAY Mask                */

#define RTC_CAL_MON_Pos                  (8)                                               /*!< RTC_T::CAL: MON Position               */
#define RTC_CAL_MON_Msk                  (0xful << RTC_CAL_MON_Pos)                        /*!< RTC_T::CAL: MON Mask                   */

#define RTC_CAL_TENMON_Pos               (12)                                              /*!< RTC_T::CAL: TENMON Position            */
#define RTC_CAL_TENMON_Msk               (0x1ul << RTC_CAL_TENMON_Pos)                     /*!< RTC_T::CAL: TENMON Mask                */

#define RTC_CAL_YEAR_Pos                 (16)                                              /*!< RTC_T::CAL: YEAR Position              */
#define RTC_CAL_YEAR_Msk                 (0xful << RTC_CAL_YEAR_Pos)                       /*!< RTC_T::CAL: YEAR Mask                  */

#define RTC_CAL_TENYEAR_Pos              (20)                                              /*!< RTC_T::CAL: TENYEAR Position           */
#define RTC_CAL_TENYEAR_Msk              (0xful << RTC_CAL_TENYEAR_Pos)                    /*!< RTC_T::CAL: TENYEAR Mask               */

#define RTC_CLKFMT_24HEN_Pos             (0)                                               /*!< RTC_T::CLKFMT: 24HEN Position          */
#define RTC_CLKFMT_24HEN_Msk             (0x1ul << RTC_CLKFMT_24HEN_Pos)                   /*!< RTC_T::CLKFMT: 24HEN Mask              */

#define RTC_WEEKDAY_WEEKDAY_Pos          (0)                                               /*!< RTC_T::WEEKDAY: WEEKDAY Position       */
#define RTC_WEEKDAY_WEEKDAY_Msk          (0x7ul << RTC_WEEKDAY_WEEKDAY_Pos)                /*!< RTC_T::WEEKDAY: WEEKDAY Mask           */

#define RTC_TALM_SEC_Pos                 (0)                                               /*!< RTC_T::TALM: SEC Position              */
#define RTC_TALM_SEC_Msk                 (0xful << RTC_TALM_SEC_Pos)                       /*!< RTC_T::TALM: SEC Mask                  */

#define RTC_TALM_TENSEC_Pos              (4)                                               /*!< RTC_T::TALM: TENSEC Position           */
#define RTC_TALM_TENSEC_Msk              (0x7ul << RTC_TALM_TENSEC_Pos)                    /*!< RTC_T::TALM: TENSEC Mask               */

#define RTC_TALM_MIN_Pos                 (8)                                               /*!< RTC_T::TALM: MIN Position              */
#define RTC_TALM_MIN_Msk                 (0xful << RTC_TALM_MIN_Pos)                       /*!< RTC_T::TALM: MIN Mask                  */

#define RTC_TALM_TENMIN_Pos              (12)                                              /*!< RTC_T::TALM: TENMIN Position           */
#define RTC_TALM_TENMIN_Msk              (0x7ul << RTC_TALM_TENMIN_Pos)                    /*!< RTC_T::TALM: TENMIN Mask               */

#define RTC_TALM_HR_Pos                  (16)                                              /*!< RTC_T::TALM: HR Position               */
#define RTC_TALM_HR_Msk                  (0xful << RTC_TALM_HR_Pos)                        /*!< RTC_T::TALM: HR Mask                   */

#define RTC_TALM_TENHR_Pos               (20)                                              /*!< RTC_T::TALM: TENHR Position            */
#define RTC_TALM_TENHR_Msk               (0x3ul << RTC_TALM_TENHR_Pos)                     /*!< RTC_T::TALM: TENHR Mask                */

#define RTC_CALM_DAY_Pos                 (0)                                               /*!< RTC_T::CALM: DAY Position              */
#define RTC_CALM_DAY_Msk                 (0xful << RTC_CALM_DAY_Pos)                       /*!< RTC_T::CALM: DAY Mask                  */

#define RTC_CALM_TENDAY_Pos              (4)                                               /*!< RTC_T::CALM: TENDAY Position           */
#define RTC_CALM_TENDAY_Msk              (0x3ul << RTC_CALM_TENDAY_Pos)                    /*!< RTC_T::CALM: TENDAY Mask               */

#define RTC_CALM_MON_Pos                 (8)                                               /*!< RTC_T::CALM: MON Position              */
#define RTC_CALM_MON_Msk                 (0xful << RTC_CALM_MON_Pos)                       /*!< RTC_T::CALM: MON Mask                  */

#define RTC_CALM_TENMON_Pos              (12)                                              /*!< RTC_T::CALM: TENMON Position           */
#define RTC_CALM_TENMON_Msk              (0x1ul << RTC_CALM_TENMON_Pos)                    /*!< RTC_T::CALM: TENMON Mask               */

#define RTC_CALM_YEAR_Pos                (16)                                              /*!< RTC_T::CALM: YEAR Position             */
#define RTC_CALM_YEAR_Msk                (0xful << RTC_CALM_YEAR_Pos)                      /*!< RTC_T::CALM: YEAR Mask                 */

#define RTC_CALM_TENYEAR_Pos             (20)                                              /*!< RTC_T::CALM: TENYEAR Position          */
#define RTC_CALM_TENYEAR_Msk             (0xful << RTC_CALM_TENYEAR_Pos)                   /*!< RTC_T::CALM: TENYEAR Mask              */

#define RTC_LEAPYEAR_LEAPYEAR_Pos        (0)                                               /*!< RTC_T::LEAPYEAR: LEAPYEAR Position     */
#define RTC_LEAPYEAR_LEAPYEAR_Msk        (0x1ul << RTC_LEAPYEAR_LEAPYEAR_Pos)              /*!< RTC_T::LEAPYEAR: LEAPYEAR Mask         */

#define RTC_INTEN_ALMIEN_Pos             (0)                                               /*!< RTC_T::INTEN: ALMIEN Position          */
#define RTC_INTEN_ALMIEN_Msk             (0x1ul << RTC_INTEN_ALMIEN_Pos)                   /*!< RTC_T::INTEN: ALMIEN Mask              */

#define RTC_INTEN_TICKIEN_Pos            (1)                                               /*!< RTC_T::INTEN: TICKIEN Position         */
#define RTC_INTEN_TICKIEN_Msk            (0x1ul << RTC_INTEN_TICKIEN_Pos)                  /*!< RTC_T::INTEN: TICKIEN Mask             */

#define RTC_INTSTS_ALMIF_Pos             (0)                                               /*!< RTC_T::INTSTS: ALMIF Position          */
#define RTC_INTSTS_ALMIF_Msk             (0x1ul << RTC_INTSTS_ALMIF_Pos)                   /*!< RTC_T::INTSTS: ALMIF Mask              */

#define RTC_INTSTS_TICKIF_Pos            (1)                                               /*!< RTC_T::INTSTS: TICKIF Position         */
#define RTC_INTSTS_TICKIF_Msk            (0x1ul << RTC_INTSTS_TICKIF_Pos)                  /*!< RTC_T::INTSTS: TICKIF Mask             */

#define RTC_TICK_TICK_Pos                (0)                                               /*!< RTC_T::TICK: TICK Position             */
#define RTC_TICK_TICK_Msk                (0x7ul << RTC_TICK_TICK_Pos)                      /*!< RTC_T::TICK: TICK Mask                 */

#define RTC_TAMSK_MSEC_Pos               (0)                                               /*!< RTC_T::TAMSK: MSEC Position            */
#define RTC_TAMSK_MSEC_Msk               (0x1ul << RTC_TAMSK_MSEC_Pos)                     /*!< RTC_T::TAMSK: MSEC Mask                */

#define RTC_TAMSK_MTENSEC_Pos            (1)                                               /*!< RTC_T::TAMSK: MTENSEC Position         */
#define RTC_TAMSK_MTENSEC_Msk            (0x1ul << RTC_TAMSK_MTENSEC_Pos)                  /*!< RTC_T::TAMSK: MTENSEC Mask             */

#define RTC_TAMSK_MMIN_Pos               (2)                                               /*!< RTC_T::TAMSK: MMIN Position            */
#define RTC_TAMSK_MMIN_Msk               (0x1ul << RTC_TAMSK_MMIN_Pos)                     /*!< RTC_T::TAMSK: MMIN Mask                */

#define RTC_TAMSK_MTENMIN_Pos            (3)                                               /*!< RTC_T::TAMSK: MTENMIN Position         */
#define RTC_TAMSK_MTENMIN_Msk            (0x1ul << RTC_TAMSK_MTENMIN_Pos)                  /*!< RTC_T::TAMSK: MTENMIN Mask             */

#define RTC_TAMSK_MHR_Pos                (4)                                               /*!< RTC_T::TAMSK: MHR Position             */
#define RTC_TAMSK_MHR_Msk                (0x1ul << RTC_TAMSK_MHR_Pos)                      /*!< RTC_T::TAMSK: MHR Mask                 */

#define RTC_TAMSK_MTENHR_Pos             (5)                                               /*!< RTC_T::TAMSK: MTENHR Position          */
#define RTC_TAMSK_MTENHR_Msk             (0x1ul << RTC_TAMSK_MTENHR_Pos)                   /*!< RTC_T::TAMSK: MTENHR Mask              */

#define RTC_CAMSK_MDAY_Pos               (0)                                               /*!< RTC_T::CAMSK: MDAY Position            */
#define RTC_CAMSK_MDAY_Msk               (0x1ul << RTC_CAMSK_MDAY_Pos)                     /*!< RTC_T::CAMSK: MDAY Mask                */

#define RTC_CAMSK_MTENDAY_Pos            (1)                                               /*!< RTC_T::CAMSK: MTENDAY Position         */
#define RTC_CAMSK_MTENDAY_Msk            (0x1ul << RTC_CAMSK_MTENDAY_Pos)                  /*!< RTC_T::CAMSK: MTENDAY Mask             */

#define RTC_CAMSK_MMON_Pos               (2)                                               /*!< RTC_T::CAMSK: MMON Position            */
#define RTC_CAMSK_MMON_Msk               (0x1ul << RTC_CAMSK_MMON_Pos)                     /*!< RTC_T::CAMSK: MMON Mask                */

#define RTC_CAMSK_MTENMON_Pos            (3)                                               /*!< RTC_T::CAMSK: MTENMON Position         */
#define RTC_CAMSK_MTENMON_Msk            (0x1ul << RTC_CAMSK_MTENMON_Pos)                  /*!< RTC_T::CAMSK: MTENMON Mask             */

#define RTC_CAMSK_MYEAR_Pos              (4)                                               /*!< RTC_T::CAMSK: MYEAR Position           */
#define RTC_CAMSK_MYEAR_Msk              (0x1ul << RTC_CAMSK_MYEAR_Pos)                    /*!< RTC_T::CAMSK: MYEAR Mask               */

#define RTC_CAMSK_MTENYEAR_Pos           (5)                                               /*!< RTC_T::CAMSK: MTENYEAR Position        */
#define RTC_CAMSK_MTENYEAR_Msk           (0x1ul << RTC_CAMSK_MTENYEAR_Pos)                 /*!< RTC_T::CAMSK: MTENYEAR Mask            */

#define RTC_LXTCTL_GAIN_Pos              (1)                                               /*!< RTC_T::LXTCTL: GAIN Position           */
#define RTC_LXTCTL_GAIN_Msk              (0x7ul << RTC_LXTCTL_GAIN_Pos)                    /*!< RTC_T::LXTCTL: GAIN Mask               */

#define RTC_LXTOCTL_OPMODE_Pos           (0)                                               /*!< RTC_T::LXTOCTL: OPMODE Position        */
#define RTC_LXTOCTL_OPMODE_Msk           (0x3ul << RTC_LXTOCTL_OPMODE_Pos)                 /*!< RTC_T::LXTOCTL: OPMODE Mask            */

#define RTC_LXTOCTL_DOUT_Pos             (2)                                               /*!< RTC_T::LXTOCTL: DOUT Position          */
#define RTC_LXTOCTL_DOUT_Msk             (0x1ul << RTC_LXTOCTL_DOUT_Pos)                   /*!< RTC_T::LXTOCTL: DOUT Mask              */

#define RTC_LXTOCTL_CTLSEL_Pos           (3)                                               /*!< RTC_T::LXTOCTL: CTLSEL Position        */
#define RTC_LXTOCTL_CTLSEL_Msk           (0x1ul << RTC_LXTOCTL_CTLSEL_Pos)                 /*!< RTC_T::LXTOCTL: CTLSEL Mask            */

#define RTC_LXTICTL_OPMODE_Pos           (0)                                               /*!< RTC_T::LXTICTL: OPMODE Position        */
#define RTC_LXTICTL_OPMODE_Msk           (0x3ul << RTC_LXTICTL_OPMODE_Pos)                 /*!< RTC_T::LXTICTL: OPMODE Mask            */

#define RTC_LXTICTL_DOUT_Pos             (2)                                               /*!< RTC_T::LXTICTL: DOUT Position          */
#define RTC_LXTICTL_DOUT_Msk             (0x1ul << RTC_LXTICTL_DOUT_Pos)                   /*!< RTC_T::LXTICTL: DOUT Mask              */

#define RTC_LXTICTL_CTLSEL_Pos           (3)                                               /*!< RTC_T::LXTICTL: CTLSEL Position        */
#define RTC_LXTICTL_CTLSEL_Msk           (0x1ul << RTC_LXTICTL_CTLSEL_Pos)                 /*!< RTC_T::LXTICTL: CTLSEL Mask            */

#define RTC_P52CTL_OPMODE_Pos            (0)                                               /*!< RTC_T::P52CTL: OPMODE Position         */
#define RTC_P52CTL_OPMODE_Msk            (0x3ul << RTC_P52CTL_OPMODE_Pos)                  /*!< RTC_T::P52CTL: OPMODE Mask             */

#define RTC_P52CTL_DOUT_Pos              (2)                                               /*!< RTC_T::P52CTL: DOUT Position           */
#define RTC_P52CTL_DOUT_Msk              (0x1ul << RTC_P52CTL_DOUT_Pos)                    /*!< RTC_T::P52CTL: DOUT Mask               */

#define RTC_P52CTL_CTLSEL_Pos            (3)                                               /*!< RTC_T::P52CTL: CTLSEL Position         */
#define RTC_P52CTL_CTLSEL_Msk            (0x1ul << RTC_P52CTL_CTLSEL_Pos)                  /*!< RTC_T::P52CTL: CTLSEL Mask             */

#define RTC_DSTCTL_ADDHR_Pos             (0)                                               /*!< RTC_T::DSTCTL: ADDHR Position          */
#define RTC_DSTCTL_ADDHR_Msk             (0x1ul << RTC_DSTCTL_ADDHR_Pos)                   /*!< RTC_T::DSTCTL: ADDHR Mask              */

#define RTC_DSTCTL_SUBHR_Pos             (1)                                               /*!< RTC_T::DSTCTL: SUBHR Position          */
#define RTC_DSTCTL_SUBHR_Msk             (0x1ul << RTC_DSTCTL_SUBHR_Pos)                   /*!< RTC_T::DSTCTL: SUBHR Mask              */

#define RTC_DSTCTL_DSBAK_Pos             (2)                                               /*!< RTC_T::DSTCTL: DSBAK Position          */
#define RTC_DSTCTL_DSBAK_Msk             (0x1ul << RTC_DSTCTL_DSBAK_Pos)                   /*!< RTC_T::DSTCTL: DSBAK Mask              */

/**@}*/ /* RTC_CONST */
/**@}*/ /* end of RTC register group */


/*---------------------- Smart Card Host Interface Controller -------------------------*/
/**
    @addtogroup SC Smart Card Host Interface Controller(SC)
    Memory Mapped Structure for SC Controller
@{ */

typedef struct
{


    /**
     * @var SC_T::DAT
     * Offset: 0x00  SC Receive/Transmit Holding Buffer Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |DAT       |Receive/Transmit Holding Buffer
     * |        |          |Write Operation:
     * |        |          |By writing data to DAT, the SC will send out an 8-bit data.
     * |        |          |Note: If SCEN (SC_SCn_CTL[0]) is not enabled, DAT cannot be programmed.
     * |        |          |Read Operation:
     * |        |          |By reading DAT, the SC will return an 8-bit received data.[WH1]
     * |        |          |Note: If SCEN (SC_CTL[0]) is not enabled, DAT cannot be programmed.
     * |        |          |[WH1]Tag59
     * @var SC_T::CTL
     * Offset: 0x04  SC Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SCEN      |SC Controller Enable Bit
     * |        |          |Set this bit to 1 to enable SC operation function. If this bit is cleared,
     * |        |          |0 = SC will force all transition to IDLE state.
     * |        |          |1 = SC controller is enabled and all function can work correctly.
     * |        |          |Note1: SCEN must be set to 1 before filling in other SC registers, or smart card will not work properly.[WH1]
     * |        |          |[WH1]Tag66
     * |[1]     |RXOFF     |RX Transition Disable Control Bit
     * |        |          |This bit is used for disable Rx receive transition function.
     * |        |          |0 = The receiver Enabled.
     * |        |          |1 = The receiver Disabled.
     * |        |          |Note1: If AUTOCEN (SC_SCn_CTL[3]) is enabled, this field is ignored.[WH1]
     * |        |          |[WH1]Tag65
     * |[2]     |TXOFF     |TX Transition Disable Control Bit
     * |        |          |This bit is used for disable Tx transmit ftransition function.
     * |        |          |0 = The transceiver Enabled.
     * |        |          |1 = The transceiver Disabled.[WH1]
     * |        |          |[WH1]Tag64
     * |[3]     |AUTOCEN   |Auto Convention Enable Bit
     * |        |          |This bit is used for enable auto convention function.
     * |        |          |0 = Auto-convention Disabled.
     * |        |          |1 = Auto-convention Enabled.
     * |        |          |If user enables auto convention function, the setting step must be done before Answer to Reset (ATR) state and the first data must be 0x3B or 0x3F
     * |        |          |After hardware received first data and stored it at buffer, hardware will decided the convention and change the CONSEL (SC_SCn_CTL[5:4]) bits automatically when received first data is 0x3B or 0x3F
     * |        |          |If received first byte is 0x3B, TS is direct convention, CONSEL (SC_SCn_CTL[5:4]) will be set to 00 automatically, otherwise the TS is inverse convention, and CONSEL (SC_SCn_CTL[5:4]) will be set to 11.
     * |        |          |If the first data is not 0x3B or 0x3F, hardware will set ACERRIF (SC_SCn_INTSTS[10]) and generate an interrupt signal to inform CPU when ACERRIEN (SC_SCn_INTEN[10]) is enabled.[WH1]
     * |        |          |[WH1]Tag10, 33, 34, 35
     * |[5:4]   |CONSEL    |Convention Selection
     * |        |          |00 = Direct convention.
     * |        |          |01 = Reserved.
     * |        |          |10 = Reserved.
     * |        |          |11 = Inverse convention.
     * |        |          |Note: If AUTOCEN (SC_SCn_CTL[3]) is enabled, this field is ignored.[WH1]
     * |        |          |[WH1]Tag10, 33, 34, 35
     * |[7:6]   |RXTRGLV   |Rx Buffer Trigger Level
     * |        |          |When the number of bytes in the receiving buffer equals the RXTRGLV, the RDAIF (SC_INTSTS[0]) will be set
     * |        |          |If RDAIEN (SC_SCn_INTEN[0]) is enabled, an interrupt signal will be generated to inform CPU.
     * |        |          |00 = Rx Buffer Trigger Level with 01 bytes.
     * |        |          |01 = Rx Buffer Trigger Level with 02 bytes.
     * |        |          |10 = Rx Buffer Trigger Level with 03 bytes.
     * |        |          |11 = Reserved. [WH1]
     * |        |          |[WH1]Tag7
     * |[12:8]  |BGT       |Block Guard Time (BGT)
     * |        |          |Block guard time means the minimum interval between the leading edges of two consecutive characters between different transfer directions
     * |        |          |This field indicates the counter for the bit length of block guard time
     * |        |          |According to ISO7816-3, in T = 0 mode, user must fill 15 (real block guard time = 16.5) to this field; in T = 1 mode, user must fill 21 (real block guard time = 22.5) to it.
     * |        |          |Note: The real block guard time is BGT + 1.[WH1]
     * |        |          |[WH1]Tag8
     * |[14:13] |TMRSEL    |Timer Channel Selection
     * |        |          |00 = All internal timer function Disabled.[WH1]
     * |        |          |.
     * |        |          |11 = Internal 24- bit Timer0timer and two 8- bit Timer0 and Timer1 are etimers Enabled
     * |        |          |User can configure them by setting SC_SCn_TMRCTL0[23:0], SC_SCn_TMRCTL1[7:0] and SC_SCn_TMRCTL2[7:0].
     * |        |          |Other configurations are reserved[WH2]
     * |        |          |[WH1]Tag9
     * |        |          |[WH2]Tag9
     * |[15]    |NSB       |Stop Bit Length
     * |        |          |This field indicates the length of stop bit.
     * |        |          |0 = The stop bit length is 2 ETU.
     * |        |          |1 = The stop bit length is 1 ETU.
     * |        |          |Note1: The default stop bit length is 2. SC and UART adopts NSB to program the stop bit length.
     * |        |          |Note2: In UART mode, RX can receive the data sequence in 1 stop bit or 2 stop bits with NSB is set to 0.[WH1]
     * |        |          |[WH1]Tag22
     * |[18:16] |RXRTY     |RX Error Retry Count Number
     * |        |          |This field indicates the maximum number of receiver retries that are allowed when parity error has occurred
     * |        |          |Note1: The real retry number is RXRTY + 1, so 8 is the maximum retry number.
     * |        |          |Note2: This field cannot be changed when RXRTYEN enabled
     * |        |          |The change flow is to disable RXRTYEN first and then fill in new retry value.[WH1]
     * |        |          |[WH1]Tag11
     * |[19]    |RXRTYEN   |RX Error Retry Enable Bit
     * |        |          |This bit enables receiver retry function when parity error has occurred.
     * |        |          |0 = RX error retry function Disabled.
     * |        |          |1 = RX error retry function Enabled.
     * |        |          |Note: User must fill in the RXRTY value before enabling this bit.
     * |[22:20] |TXRTY     |TX Error Retry Count Number
     * |        |          |This field indicates the maximum number of transmitter retries that are allowed when parity error has occurred.
     * |        |          |Note1: The real retry number is TXRTY + 1, so 8 is the maximum retry number.
     * |        |          |Note2: This field cannot be changed when TXRTYEN enabled
     * |        |          |The change flow is to disable TXRTYEN first and then fill in new retry value.[WH1]
     * |        |          |[WH1]Tag11
     * |[23]    |TXRTYEN   |TX Error Retry Enable Bit
     * |        |          |This bit enables transmitter retry function when parity error has occurred.
     * |        |          |0 = TX error retry function Disabled.
     * |        |          |1 = TX error retry function Enabled.
     * |[25:24] |CDDBSEL   |Card Detect De-bounce Selection
     * |        |          |This field indicates the card detect de-bounce selection.
     * |        |          |00 = De-bounce sample card insert once per 384 (128 * 3) SC modue clocks and de-bounce sample card removal once per 128 SC modue clocks.[WH1]
     * |        |          |Other configurations are reserved.[WH2]
     * |        |          |[WH1]Tag63
     * |        |          |[WH2]Tag63
     * |[26]    |CDLV      |Card Detect Level Selection
     * |        |          |0 = When hardware detects the card detect pin (SC_SCn_CD) from high to low, it indicates a card is detected.
     * |        |          |1 = When hardware detects the card detect pin (SC_SCn_CD) from low to high, it indicates a card is detected.
     * |        |          |Note: User must select card detect level before Smart Card controller enabled. [WH1]
     * |        |          |[WH1]Tag2
     * |[30]    |SYNC      |SYNC Flag Indicator (Read Only)
     * |        |          |Due to synchronization, user should check this bit before writing a new value to RXRTY and TXRTY fields.
     * |        |          |0 = Synchronizing is completion, user can write new data to RXRTY and TXRTY.
     * |        |          |1 = Last value is synchronizing. [WH1]
     * |        |          |[WH1]Tag62
     * @var SC_T::ALTCTL
     * Offset: 0x08  SC Alternate Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TXRST     |TX Software Reset
     * |        |          |When TXRST is set, all the bytes in the transmit buffer and TX Tx internal state machine will be cleared.
     * |        |          |0 = No effect.
     * |        |          |1 = Reset the TX Tx internal state machine and pointers.
     * |        |          |Note: This bit will be auto cleared after reset is complete.[WH1]
     * |        |          |[WH1]Tag71
     * |[1]     |RXRST     |Rx RX Software Reset
     * |        |          |When RXRST is set, all the bytes in the receive buffer and Rx internal state machine will be cleared.
     * |        |          |0 = No effect.
     * |        |          |1 = Reset the Rx internal state machine and pointers.
     * |        |          |Note: This bit will be auto cleared after reset is complete.[WH1]
     * |        |          |[WH1]Tag70
     * |[2]     |DACTEN    |Deactivation Sequence Generator Enable Bit
     * |        |          |This bit enables SC controller to initiate the card by deactivation sequence.
     * |        |          |0 = No effect.
     * |        |          |1 = Deactivation sequence generator Enabled.
     * |        |          |Note1: When the deactivation sequence completed, this bit will be cleared automatically and the INITIF (SC_SCn_INTSTS[8]) will be set to 1.
     * |        |          |Note2: This field will be cleared by setT TXRST (SC_SCn_ALTCTL[0]) and or RXRST (SC_SCn_ALTCTL[1])
     * |        |          |Thus, do not fill in this bit DACTEN, TXRST and or RXRST at the same time.
     * |        |          |Note3: If SCEN (SC_SCn_CTL[0]) is not enabled, this filed cannot be programmed.[WH1]
     * |        |          |[WH1]Tag14, 65
     * |[3]     |ACTEN     |Activation Sequence Generator Enable Bit
     * |        |          |This bit enables SC controller to initiate the card by activation sequence.
     * |        |          |0 = No effect.
     * |        |          |1 = Activation sequence generator Enabled.
     * |        |          |Note1: When the activation sequence completed, this bit will be cleared automatically and the INITIF (SC_SCn_INTSTS[8]) will be set to 1.
     * |        |          |Note2: This field will be cleared by set TXRST (SC_SCn_ALTCTL[0]) and or RXRST (SC_SCn_ALTCTL[1])
     * |        |          |Thus, do not fill in this bit ACTEN, TXRST and or RXRST at the same time.
     * |        |          |Note3: If SCEN (SC_SCn_CTL[0]) is not enabled, this filed cannot be programmed.[WH1]
     * |        |          |[WH1]Tag12, 65
     * |[4]     |WARSTEN   |Warm Reset Sequence Generator Enable Bit
     * |        |          |This bit enables SC controller to initiate the card by warm reset sequence.
     * |        |          |0 = No effect.
     * |        |          |1 = Warm reset sequence generator Enabled.
     * |        |          |Note1: When the warm reset sequence completed, this bit will be cleared automatically and the INITIF (SC_SCn_INTSTS[8]) will be set to 1.
     * |        |          |Note2: This field will be cleared by set TXRST (SC_SCn_ALTCTL[0]) and or RXRST (SC_SCn_ALTCTL[1])
     * |        |          |Thus, do not fill in this bit WARSTEN, TXRST and or RXRST at the same time.
     * |        |          |Note3: If SCEN (SC_SCn_CTL[0]) is not enabled, this filed cannot be programmed.[WH1]
     * |        |          |[WH1]Tag13, 65
     * |[5]     |CNTEN0    |Internal Timer0 Start Enable Bit
     * |        |          |This bit enables Timer 0 to start counting
     * |        |          |User can fill 0 to stop count it and set 1 to reload and start count
     * |        |          |The counter unit is ETU base.
     * |        |          |0 = Stops counting.
     * |        |          |1 = Start counting.
     * |        |          |Note1: This field is used for internal 24- bit timer when TMRSEL (SC_SCn_CTL[14:13]) is 11 only
     * |        |          |Do not fill CNTEN0 when TMRSEL (SC_CTL[14:13]) is not equale to 11.
     * |        |          |Note2: If the operation mode is not in auto-reload mode (SC_SCn_TMRCTL0[26] = 0), this bit will be auto-cleared by hardware.
     * |        |          |Note3: If SCEN (SC_SCn_CTL[0]) is not enabled, this filed cannot be programmed.[WH1]
     * |        |          |[WH1]Tag37
     * |[6]     |CNTEN1    |Internal Timer1 Start Enable Bit
     * |        |          |This bit enables Timer 1 to start counting
     * |        |          |User can fill 0 to stop count it and set 1 to reload and start count
     * |        |          |The counter unit is ETU base.
     * |        |          |0 = Stops counting.
     * |        |          |1 = Start counting.
     * |        |          |Note1: This field is used for internal 8- bit timer when TMRSEL (SC_SCn_CTL[14:13]) is 11 only
     * |        |          |Do not fill CNTEN1 when TMRSEL (SC_SCn_CTL[14:13]) is not equale to 11.
     * |        |          |Note2: If the operation mode is not in auto-reload mode (SC_SCn_TMRCTL1[26] = 0), this bit will be auto-cleared by hardware.
     * |        |          |Note3: If SCEN (SC_SCn_CTL[0]) is not enabled, this filed cannot be programmed.
     * |[7]     |CNTEN2    |Internal Timer2 Start Enable Bit
     * |        |          |This bit enables Timer 2 to start counting
     * |        |          |User can fill 0 to stop it count and set 1 to reload and start count
     * |        |          |The counter unit is ETU base.
     * |        |          |0 = Stops counting.
     * |        |          |1 = Start counting.
     * |        |          |Note1: This field is used for internal 8- bit timer when TMRSEL (SC_SCn_CTL[14:13]) is 11 only
     * |        |          |Do not fill in CNTEN2 when TMRSEL (SC_SCn_CTL[14:13]) is not equale to 11.
     * |        |          |Note2: If the operation mode is not in auto-reload mode (SC_SCn_TMRCTL2[26] = = 0), this bit will be auto-cleared by hardware.
     * |        |          |Note3: If SCEN (SC_SCn_CTL[0]) is not enabled, this filed cannot be programmed.
     * |[9:8]   |INITSEL   |Initial Timing Selection
     * |        |          |This fields indicates the initial timing of hardware activation, warm-reset or deactivation.
     * |        |          |The unit of initial timing is SC module clock.
     * |        |          |Activation: refer to SC Activation Sequence in Figure 1.1-4 Figure 1.1-4.
     * |        |          |Warm-reset: refer to Warm-Reset Sequence in Figure 1.1-5 Figure 1.1-5.
     * |        |          |Deactivation: refer to Deactivation Sequence in Figure 1.1-6 Figure 1.1-6.
     * |        |          |Note: When set activation and warm reset in Timer0 operation mode 0011, it may have deviation at most 128 SC module clock cycles.[WH1]
     * |        |          |[WH1]Tag12~15
     * |[11]    |ADACEN    |Auto Deactivation When Card Removal
     * |        |          |This bit is usde for enable hardware auto deactivation when smart card is removed.
     * |        |          |0 = Auto deactivation Disabled.
     * |        |          |1 = Auto deactivation Enabled.
     * |        |          |Note: When the card is removed, hardware will stop any process and then do deactivation sequence if this bit is set
     * |        |          |If auto deactivation process completes, hardware will set INITIF (SC_SCn_INTSTS[8]) also.[WH1]
     * |        |          |[WH1]Tag15
     * |[12]    |RXBGTEN   |Receiver Block Guard Time Function Enable Bit
     * |        |          |This bit enables the receiver block guard time function.
     * |        |          |0 = Receiver block guard time function Disabled.
     * |        |          |1 = Receiver block guard time function Enabled.[WH1]
     * |        |          |[WH1]Tag69
     * |[13]    |ACTSTS0   |Internal Timer0 Active Status (Read Only)
     * |        |          |This bit indicates the timer counter status of timer0.
     * |        |          |0 = Timer0 is not active.
     * |        |          |1 = Timer0 is active.
     * |        |          |Note: Timer0 is active does not always mean timer0 is counting the CNT (SC_SCn_TMRCTL0[23:0]).[WH1]
     * |        |          |[WH1]Tag68
     * |[14]    |ACTSTS1   |Internal Timer1 Active Status (Read Only)
     * |        |          |This bit indicates the timer counter status of timer1.
     * |        |          |0 = Timer1 is not active.
     * |        |          |1 = Timer1 is active.
     * |        |          |Note: Timer1 is active does not always mean timer1 is counting the CNT (SC_SCn_TMRCTL1[7:0]).
     * |[15]    |ACTSTS2   |Internal Timer2 Active Status (Read Only)
     * |        |          |This bit indicates the timer counter status of timer2.
     * |        |          |0 = Timer2 is not active.
     * |        |          |1 = Timer2 is active.
     * |        |          |Note: Timer2 is active does not always mean timer2 is counting the CNT (SC_SCn_TMRCTL2[7:0]).
     * |[31]    |SYNC      |SYNC Flag Indicator (Read Only)
     * |        |          |Due to synchronization, user should check this bit when writing a new value to SC_SCn_ALTCTL register.
     * |        |          |0 = Synchronizing is completion, user can write new data to SC_SCn_ALTCTL register.
     * |        |          |1 = Last value is synchronizing. [WH1]
     * |        |          |[WH1]Tag67
     * @var SC_T::EGT
     * Offset: 0x0C  SC Extra Guard Time Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |EGT       |Extra Guard Time
     * |        |          |This field indicates the extra guard time value.
     * |        |          |Note: The extra guard time unit is ETU base.[WH1]
     * |        |          |[WH1]Tag53
     * @var SC_T::RXTOUT
     * Offset: 0x10  SC Receive Buffer Time-out Counter Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[8:0]   |RFTM      |SC Receiver FIFO Time-out Counter
     * |        |          |The time-out down counter resets and starts counting whenever the RX Rx buffer received a new data
     * |        |          |Once the counter decrease to 1 and no new data is received or CPU does not read data by reading SC_SCn_DAT (SC_DAT[7:0]), a receiver time-out flag RBXTOIF (SC_SCn_INTSTS[9]) will be set, and hardware will generate an interrupt signal to inform CPU when RXBTOIEN (SC_SCn_INTEN[9]) is enabled.
     * |        |          |Note1: The counter unit is ETU based and the interval of time-out is (RFTM + 0.5) ETU time.
     * |        |          |Note2: Filling in all 0 to this field indicates willto disable this function.[WH1]
     * |        |          |[WH1]Tag72
     * @var SC_T::ETUCTL
     * Offset: 0x14  SC Element Time Unit Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |ETURDIV   |ETU Rate Divider
     * |        |          |The field is used for define ETU time unit.clock rate divider.
     * |        |          |The real ETU time unit is (ETURDIV + 1) * SC clock time.[WH1]
     * |        |          |Note: User can configure this field, but this field must be greater than 0x004.[WH2]
     * |        |          |[WH1]Tag6
     * |        |          |[WH2]Tag74
     * @var SC_T::INTEN
     * Offset: 0x18  SC Interrupt Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RDAIEN    |Receive Data Reach Interrupt Enable Bit
     * |        |          |This field is used to enable received data bytes in Rx buffer reaching trigger level RXTRGLV (SC_SCn_CTL[7:6]) interrupt.
     * |        |          |0 = Received data bytes in Rx buffer reach trigger level interrupt Disabled.
     * |        |          |1 = Received data bytes in Rx buffer reach trigger level interrupt Enabled.[WH1]
     * |        |          |[WH1]Tag7
     * |[1]     |TXEIEN    |Transmit Buffer Empty Interrupt Enable Bit
     * |        |          |This field is used to enable transmit buffer empty interrupt.
     * |        |          |0 = Transmit buffer empty interrupt Disabled.
     * |        |          |1 = Transmit buffer empty interrupt Enabled.[WH1]
     * |        |          |[WH1]Tag64, 71
     * |[2]     |TERRIEN   |Transfer Error Interrupt Enable Bit
     * |        |          |This field is used to enable transfer error interrupt
     * |        |          |The transfer error status states is at SC_SCn_STATUS register which includes receiver break error BEF (SC_SCn_STATUS[6]), frame error FEF (SC_SCn_STATUS[5]), parity error PEF (SC_SCn_STATUS[4]), receive buffer overflow error RXOV (SC_SCn_STATUS[0]), transmit buffer overflow error TXOV (SC_SCn_STATUS[8]), receiver retry over limit error RXOVERR (SC_SCn_STATUS[22]) and or transmitter retry over limit error TXOVERR (SC_SCn_STATUS[30]).
     * |        |          |0 = Transfer error interrupt Disabled.
     * |        |          |1 = Transfer error interrupt Enabled.[WH1]
     * |        |          |[WH1]Tag11, 36
     * |[3]     |TMR0IEN   |Timer0 Interrupt Enable Bit
     * |        |          |This field is used to enable Timer0 interrupt function.
     * |        |          |0 = Timer0 interrupt Disabled.
     * |        |          |1 = Timer0 interrupt Enabled.[WH1]
     * |        |          |[WH1]Tag76
     * |[4]     |TMR1IEN   |Timer1 Interrupt Enable Bit
     * |        |          |This field is used to enable the Timer1 interrupt function.
     * |        |          |0 = Timer1 interrupt Disabled.
     * |        |          |1 = Timer1 interrupt Enabled.
     * |[5]     |TMR2IEN   |Timer2 Interrupt Enable Bit
     * |        |          |This field is used to enable Timer2 interrupt function.
     * |        |          |0 = Timer2 interrupt Disabled.
     * |        |          |1 = Timer2 interrupt Enabled.
     * |[6]     |BGTIEN    |Block Guard Time Interrupt Enable Bit
     * |        |          |This field is used to enable block guard time interrupt in recevive direction.
     * |        |          |0 = Block guard time interrupt Disabled.
     * |        |          |1 = Block guard time interrupt Enabled.
     * |        |          |Note: This bit is valid only for recvive receive direction block guard time.[WH1]
     * |        |          |[WH1]Tag8, 50~53
     * |[7]     |CDIEN     |Card Detect Interrupt Enable Bit
     * |        |          |This field is used to enable card detect interrupt
     * |        |          |The card detect status is CDPINSTS (SC_SCn_STATUS[13]).
     * |        |          |0 = Card detect interrupt Disabled.
     * |        |          |1 = Card detect interrupt Enabled.[WH1]
     * |        |          |Note: Either cared insert or card remove event will generate crad detect event.
     * |        |          |[WH1]Tag75
     * |[8]     |INITIEN   |Initial End Interrupt Enable Bit
     * |        |          |This field is used to enable activation (ACTEN (SC_SCn_ALTCTL[3] = 1)), deactivation (DACTEN (SC_SCn_ALTCTL[2] = 1)) and warm reset (WARSTEN (SC_SCn_ALTCTL [4])) sequence complete interrupt.
     * |        |          |0 = Initial end interrupt Disabled.
     * |        |          |1 = Initial end interrupt Enabled.[WH1]
     * |        |          |[WH1]Tag12~15
     * |[9]     |RXTOIEN   |Receiver Buffer Time-out Interrupt Enable Bit
     * |        |          |This field is used to enable receiver buffer time-out interrupt.
     * |        |          |0 = Receiver buffer time-out interrupt Disabled.
     * |        |          |1 = Receiver buffer time-out interrupt Enabled.[WH1]
     * |        |          |[WH1]Tag72
     * |[10]    |ACERRIEN  |Auto Convention Error Interrupt Enable Bit
     * |        |          |This field is used to enable auto-convention error interrupt.
     * |        |          |0 = Auto-convention error interrupt Disabled.
     * |        |          |1 = Auto-convention error interrupt Enabled.[WH1]
     * |        |          |[WH1]Tag10
     * @var SC_T::INTSTS
     * Offset: 0x1C  SC Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RDAIF     |Receive Data Reach Interrupt Status Flag (Read Only)
     * |        |          |This field is used for received data bytes in Rx buffer reaching trigger level RXTRGLV (SC_SCn_CTL[7:6]) interrupt status flag.
     * |        |          |0 = Number of receive buffer is less than RXTRGLV setting.
     * |        |          |1 = Number of receive buffer data equals the RXTRGLV setting.
     * |        |          |Note: This bit is read only
     * |        |          |If user reads data from DAT (SC_SCn_DAT[7:0]) and remains receiver buffer data byte number is less than RXTRGLV, this bit will be cleared automatically.[WH1]
     * |        |          |[WH1]Tag7
     * |[1]     |TXEIF     |Transmit Buffer Empty Interrupt Status Flag (Read Only)
     * |        |          |This field is used for transmit buffer empty interrupt status flag.
     * |        |          |0 = Transmit buffer is not empty.
     * |        |          |1 = Transmit buffer is empty.
     * |        |          |Note: This bit is read only
     * |        |          |If user wants to clear this bit, user must write data to DAT (SC_SCn_DAT[7:0]) and then this bit will be cleared automatically.[WH1]
     * |        |          |[WH1]Tag64, 71
     * |[2]     |TERRIF    |Transfer Error Interrupt Status Flag
     * |        |          |This field is used for indicate transfer error interrupt status flag
     * |        |          |The transfer error states status is at SC_SCn_STATUS register which includes receiver break error BEF (SC_SCn_STATUS[6]), frame error FEF (SC_SCn_STATUS[5]), parity error PEF (SC_SCn_STATUS[4]), and receive buffer overflow error RXOV (SC_SCn_STATUS[0]), transmit buffer overflow error TXOV (SC_SCn_STATUS[8]), receiver retry over limit error RXOVERR (SC_SCn_STATUS[22]) or transmitter retry over limit error TXOVERR (SC_SCn_STATUS[30]).
     * |        |          |0 = Transfer error interrupt did not occur.
     * |        |          |1 = Transfer error interrupt occurred.
     * |        |          |Note1: This field is the status flag of BEF, FEF, PEF, RXOV, TXOV, RXOVERR or TXOVERR.
     * |        |          |Note2: This bit can be cleared by writing 1 to it.[WH1]
     * |        |          |[WH1]Tag11, 36
     * |[3]     |TMR0IF    |Timer0 Interrupt Status Flag
     * |        |          |This field is used for Timer0 interrupt status flag.
     * |        |          |0 = Timer0 interrupt did not occur.
     * |        |          |1 = Timer0 interrupt occurred.
     * |        |          |Note: This bit can be cleared by writing 1 to it.[WH1]
     * |        |          |[WH1]Tag76
     * |[4]     |TMR1IF    |Timer1 Interrupt Status Flag
     * |        |          |This field is used for Timer1 interrupt status flag.
     * |        |          |0 = Timer1 interrupt did not occur.
     * |        |          |1 = Timer1 interrupt occurred.
     * |        |          |Note: This bit can be cleared by writing 1 to it.
     * |[5]     |TMR2IF    |Timer2 Interrupt Status Flag
     * |        |          |This field is used for Timer2 interrupt status flag.
     * |        |          |0 = Timer2 interrupt did not occur.
     * |        |          |1 = Timer2 interrupt occurred.
     * |        |          |Note: This bit can be cleared by writing 1 to it.
     * |[6]     |BGTIF     |Block   Guard Time Interrupt Status Flag
     * |        |          |This field is   used for indicate block guard time interrupt status flag in recvive receive direction.
     * |        |          |0 = Block   guard time interrupt did not occur.
     * |        |          |1 = Block   guard time interrupt occurred.
     * |        |          |Note1: This bit is valid only when RXBGTEN (SC_SCn_ALTCTL[12])   is enabled.
     * |        |          |Note2: This bit can be cleared by writing 1 to   it.
     * |[7]     |CDIF      |Card Detect Interrupt Status Flag (Read Only)
     * |        |          |This field is used for card detect interrupt status flag
     * |        |          |The actual card detect status is in CINSERT (SC_SCn_STATUS[12]) and CREMOVE (SC_SCn_STATUS[11]).
     * |        |          |0 = Card detect event did not occur.
     * |        |          |1 = Card detect event occurred.
     * |        |          |Note1: This bit is read only, and will be cleared after user must to clear CINSERT or CREMOVE [WH1]status has been clearedto clear it.
     * |        |          |Note2: Either cared insert or card remove event will generate crad detect event.
     * |        |          |[WH1]Tag75
     * |[8]     |INITIF    |Initial End Interrupt Status Flag
     * |        |          |This field is used for activation (ACTEN (SC_SCn_ALTCTL[3])), deactivation (DACTEN (SC_SCn_ALTCTL[2])) and warm reset (WARSTEN (SC_SCn_ALTCTL[4])) sequence interrupt status flag.
     * |        |          |0 = Initial sequence is not complete.
     * |        |          |1 = Initial sequence is completed.
     * |        |          |Note: This bit can be cleared by writing 1 to it.[WH1]
     * |        |          |[WH1]Tag12~15
     * |[9]     |RXTOIF    |Receive Buffer Time-out Interrupt Status Flag (Read Only)
     * |        |          |This field is used for indicate receive buffer time-out interrupt status flag.
     * |        |          |0 = Receive buffer time-out interrupt did not occur.
     * |        |          |1 = Receive buffer time-out interrupt occurred.
     * |        |          |Note: This bit is read only, user must read all receive buffer remaining data by reading SC_SCn_DAT (SC_DAT[7:0])register to clear it.[WH1]
     * |        |          |[WH1]Tag72
     * |[10]    |ACERRIF   |Auto Convention Error Interrupt Status Flag
     * |        |          |This field indicates auto convention sequence error.
     * |        |          |0 = Received TS at ATR state is 0x3B or 0x3F.
     * |        |          |1 = Received TS at ATR state is neither 0x3B nor 0x3F.
     * |        |          |Note: This bit can be cleared by writing 1 to it.[WH1]
     * |        |          |[WH1]Tag10
     * @var SC_T::STATUS
     * Offset: 0x20  SC Transfer Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RXOV      |Receive Overflow Error Status Flag
     * |        |          |This bit is set when Rx buffer overflow.
     * |        |          |0 = Rx buffer is not overflow.
     * |        |          |1 = Rx buffer is overflow when the number of received bytes is greater than Rx buffer size (4 bytes).
     * |        |          |Note: This bit can be cleared by writing 1 to it.[WH1]
     * |        |          |[WH1]Tag5
     * |[1]     |RXEMPTY   |Receive Buffer Empty Status Flag (Read Only)
     * |        |          |This bit indicates Rx buffer is empty or not.
     * |        |          |0 = Rx buffer is not empty.
     * |        |          |1 = Rx buffer is empty, it means the last byte of in Rx buffer has been read from DAT (SC_SCn_DAT[7:0]) by CPU.
     * |[2]     |RXFULL    |Receive Buffer Full Status Flag (Read Only)
     * |        |          |This bit indicates Rx buffer is full or not.
     * |        |          |0 = Rx buffer count is less than 4.
     * |        |          |1 = Rx buffer count equals to 4.
     * |[4]     |PEF       |Receiver Parity Error Status Flag
     * |        |          |This bit is set to logic 1 whenever the received character does not have a valid u201Cparity bitu201D.
     * |        |          |0 = Receiver parity error flag did not occur.
     * |        |          |1 = Receiver parity error flag occurred.
     * |        |          |Note1: This bit can be cleared by writing 1 to it.
     * |        |          |Note2: If user CPU sets receiver retries function by setting RXRTYEN (SC_SCn_CTL[19]), hardware will not set this flag.[WH1]
     * |        |          |[WH1]Tag80
     * |[5]     |FEF       |Receiver Frame Error Status Flag
     * |        |          |This bit is set to logic 1 whenever the received character does not have a valid u201Cstop bitu201D (that is, the stop bit following the last data bit or parity bit is detected as logic 0).
     * |        |          |0 = Receiver frame error flag did not occur.
     * |        |          |1 = Receiver frame error flag occurred.
     * |        |          |Note1: This bit can be cleared by writing 1 to it.
     * |        |          |Note2: If user CPU sets receiver retries function by setting RXRTYEN (SC_SCn_CTL[19]), hardware will not set this flag.[WH1]
     * |        |          |[WH1]Tag79
     * |[6]     |BEF       |Receiver Break Error Status Flag
     * |        |          |This bit is set to logic 1 whenever the received data input (Rx) held in the u201Cspacing stateu201D (logic 0) is longer than a full word transmission time (that is, the total time of u201Cstart bitu201D + u201Cdata bitsu201D + u201Cparity bitu201D + u201Cstop bitsu201D).
     * |        |          |0 = Receiver break error flag did not occur.
     * |        |          |1 = Receiver break error flag occurred.
     * |        |          |Note1: This bit can be cleared by writing 1 to it.
     * |        |          |Note2: If CPU user sets receiver retries function by setting RXRTYEN (SC_SCn_CTL[19]), hardware will not set this flag.[WH1]
     * |        |          |[WH1]Tag78
     * |[8]     |TXOV      |Transmit Overflow Error Interrupt Status Flag
     * |        |          |This bit is set when Tx buffer overflow.
     * |        |          |0 = Tx buffer is not overflow.
     * |        |          |1 = Tx buffer is overflow, it means when Tx buffer is full and an additional write operation to DAT (SC_SCn_DAT[7:0]) when Tx buffer is already full.
     * |        |          |Note: This bit can be cleared by writing 1 to it.[WH1]
     * |        |          |[WH1]Tag5
     * |[9]     |TXEMPTY   |Transmit Buffer Empty Status Flag (Read Only)
     * |        |          |This bit indicates TX buffer is empty or not.
     * |        |          |0 = Tx buffer is not empty.
     * |        |          |1 = Tx buffer is empty, it means the last byte of Tx buffer has been transferred to Transmitter Shift Register.
     * |        |          |Note: This bit will be cleared when writing data into DAT (SC_SCn_DAT[7:0]).
     * |[10]    |TXFULL    |Transmit Buffer Full Status Flag (Read Only)
     * |        |          |This bit indicates Tx buffer is full or not.
     * |        |          |0 = Tx buffer count is less than 4.
     * |        |          |1 = Tx buffer count equals to 4.
     * |[11]    |CREMOVE   |Card Removal Status of SC_SCn_CD Pin
     * |        |          |This bit is set whenever card has been removal.
     * |        |          |0 = No effect.
     * |        |          |1 = Card removed.
     * |        |          |Note1: This bit can be cleared by writing u201C1u201D to it.
     * |        |          |Note2: Card detect function will start after SCEN (SC_SCn_CTL[0]) is set.[WH1]
     * |        |          |[WH1]Tag2
     * |[12]    |CINSERT   |Card Insert Status of SC_SCn_CD Pin
     * |        |          |This bit is set whenever card has been inserted.
     * |        |          |0 = No effect.
     * |        |          |1 = Card insert.
     * |        |          |Note1: This bit can be cleared by writing u201C1u201D to it.
     * |        |          |Note2: The card detect function will start after SCEN (SC_SCn_CTL[0]) setis set.
     * |[13]    |CDPINSTS  |Card Detect Pin Status (Read Only)
     * |        |          |This bit is the pin status of SC_SCn_CD.
     * |        |          |0 = The SC_SCn_CD pin state at low.
     * |        |          |1 = The SC_SCn_CD pin state at high.
     * |[18:16] |RXPOINT   |Receive Buffer Pointer Status (Read Only)
     * |        |          |This field indicates the Rx buffer pointer status
     * |        |          |When SC controller receives one byte from external device, RXPOINT increases one
     * |        |          |When one byte of in Rx buffer is read by CPUreading DAT (SC_DAT[7:0]), RXPOINT decreases one.[WH1]
     * |        |          |[WH1]Tag5
     * |[21]    |RXRTYERR  |Receiver Retry ErrorRetry Error
     * |        |          |This bit is used for indicate receiver error retry error retry and set by hardware.
     * |        |          |0 = No Rx retry transfer.
     * |        |          |1 = Rx has any error and retries transfer.
     * |        |          |Note1: This bit can be cleared by writing 1 to it.
     * |        |          |Note2: This bit is a flag and cannot generate any interrupt signal to CPU.
     * |        |          |Note3: If CPU user enables receiver retries function by setting RXRTYEN (SC_SCn_CTL[19]), the PEF (SC_STATUS[4]) bit hardware will not set this flag.[WH1]
     * |        |          |[WH1]Tag11
     * |[22]    |RXOVERR   |Receiver Oover Retry Error
     * |        |          |This bit is used for indicate receiver retry counts over than retry number limitation.
     * |        |          |0 = Receiver retries counts is not over than RXRTY (SC_SCn_CTL[18:16]) + 1.
     * |        |          |1 = Receiver retries counts over than RXRTY (SC_SCn_CTL[18:16]) + 1.
     * |        |          |Note1: This bit can be cleared by writing 1 to it.
     * |        |          |Note2: If CPU user enables receiver retries function by setting RXRTYEN (SC_SCn_CTL[19]), the PEF (SC_STATUS[4]) bit will not set hardware will not set this flag.
     * |[23]    |RXACT     |Receiver in Active Status Flag (Read Only)
     * |        |          |This bit indicates Rx transfer status.
     * |        |          |0 = This bit is cleared automatically when Rx transfer is finished.
     * |        |          |1 = This bit is set by hardware when Rx transfer is in active.
     * |        |          |Note: This bit is read only.[WH1]
     * |        |          |[WH1]Tag78
     * |[26:24] |TXPOINT   |Transmit Buffer Pointer Status (Read Only)
     * |        |          |This field indicates the Tx buffer pointer status
     * |        |          |When CPU writes data into DAT (SC_SCn_DAT[7:0]), TXPOINT increases one
     * |        |          |When one byte of Tx buffer is transferred to Transmitter Shift Registertransmitter shift register, TXPOINT decreases one.[WH1]
     * |        |          |[WH1]Tag5
     * |[29]    |TXRTYERR  |Transmitter Retry ErrorRetry Error
     * |        |          |This bit is used for indicate transmitter error retry and set by hardware..
     * |        |          |0 = No Tx retry transfer.
     * |        |          |1 = Tx has any error and retries transfer.
     * |        |          |Note1: This bit can be cleared by writing 1 to it.
     * |        |          |Note2: This bit is a flag and cannot generate any interrupt signal to CPU.[WH1]
     * |        |          |[WH1]Tag11
     * |[30]    |TXOVERR   |Transmitter Oover Retry Error
     * |        |          |This bit is used for transmitter retry counts over than retry number limitation.
     * |        |          |0 = Transmitter retries counts is not over than TXRTY (SC_SCn_CTL[22:20]) + 1.
     * |        |          |1 = Transmitter retries counts over than TXRTY (SC_SCn_CTL[22:20]) + 1.
     * |        |          |Note: This bit can be cleared by writing 1 to it.
     * |[31]    |TXACT     |Transmit in Active Status Flag (Read Only)
     * |        |          |This bit indicates Tx transmit status.
     * |        |          |0 = This bit is cleared automatically when Tx transfer is finished or the last byte transmission has completed.
     * |        |          |1 = Transmit is active or the STOP bit of last byte has not been transmitted when Tx transfer is in active.and this bit is set by hardware when Tx transfer is in active and the STOP bit of the last byte has not been transmitted.
     * |        |          |Note: This bit is read only.[WH1]
     * |        |          |[WH1]Tag77
     * @var SC_T::PINCTL
     * Offset: 0x24  SC Pin Control State Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |PWREN     |SC_SCn_PWR Pin Signal
     * |        |          |User can set PWRINV (SC_SCn_PINCTL[11]) and PWREN (SC_SCn_PINCTL[0]) to decide SC_SCn_PWR pin is in high or low level.
     * |        |          |Write this field bit can to drive SC_SCn_PWR pin
     * |        |          |Refer PWRINV (SC_SCn_PINCTL[11]) description for programming SC_SCn_PWR pin voltage level.
     * |        |          |Read this field bit to get SC_SCn_PWR signal status.
     * |        |          |0 = SC_SCn_PWR signal status is low.
     * |        |          |1 = SC_SCn_PWR signal status is high.
     * |        |          |Note: When operating at activation, warm reset or deactivation mode, this bit will be changed automatically
     * |        |          |Thus, do not fill in this field when operating in these modes.[WH1]
     * |        |          |[WH1]Tag84
     * |[1]     |SCRST     |SC_SCn_RST Pin Signal
     * |        |          |This bit is the signal status of SC_SCn_RST but user can drive SC_SCn_RST pin to high or low by setting control this bit.
     * |        |          |Write this bit can field to drive SC_SCn_RST pin.
     * |        |          |0 = Drive SC_SCn_RST pin to low.
     * |        |          |1 = Drive SC_SCn_RST pin to high.
     * |        |          |Read this bit field to get SC_SCn_RST signal status.
     * |        |          |0 = SC_SCn_RST signal status is low.
     * |        |          |1 = SC_SCn_RST signal status is high.
     * |        |          |Note: When operating at activation, warm reset or deactivation mode, this bit will be changed automatically
     * |        |          |Thus, do not fill in this field when operating in these modes.[WH1]
     * |        |          |[WH1]Tag84
     * |[6]     |CLKKEEP   |SC Clock Enable Bit
     * |        |          |0 = SC clock generation Disabled.
     * |        |          |1 = SC clock always keeps free running.
     * |        |          |Note: When operating in activation, warm reset or deactivation mode, this bit will be changed automatically
     * |        |          |Thus, do not fill in this field when operating in these modes.[WH1]
     * |        |          |[WH1]Tag66
     * |[9]     |SCDATA    |SC_SCn_DATA Pin Signal
     * |        |          |This bit is the signal status of SC_SCn_DATA but user can also drive SC_SCn_DATA pin to high or low by control this setting this bit.
     * |        |          |Write this bit can drive SC_RST pin.
     * |        |          |0 = Drive SC_SCn_DATA pin to low.
     * |        |          |1 = Drive SC_SCn_DATA pin to high.
     * |        |          |Read this bit field to get SC_SCn_DATA signal status.
     * |        |          |0 = SC_SCn_DATA signal status is low.
     * |        |          |1 = SC_SCn_DATA signal status is high.
     * |        |          |Note: When SC is at activation, warm reset or deactivation mode, this bit will be changed automatically
     * |        |          |Thus, do not fill in this field when SC is in these modes.[WH1]
     * |        |          |[WH1]Tag85
     * |[11]    |PWRINV    |SC_SCn_PWR Pin Inverse
     * |        |          |This bit is used for inverse the SC_SCn_PWR pin.
     * |        |          |There are four kinds of combination for SC_SCn_PWR pin setting by PWRINV (SC_SCn_PINCTL[11]) and PWREN (SC_SCn_PINCTL[0])
     * |        |          |A
     * |        |          |PWRINV (SC_SCn_PINCTL[11]) is bit 1 and PWREN (SC_SCn_PINCTL[0]) is bit 0 and all conditions as below list.,
     * |        |          |00 = SC_SCn_PWR pin is 0.
     * |        |          |01 = SC_SCn_PWR pin is 1.
     * |        |          |10 = SC_SCn_PWR pin is 1.
     * |        |          |11 = SC_SCn_PWR pin is 0.
     * |        |          |Note: User must select PWRINV (SC_SCn_PINCTL[11]) before smart card is enabled by SCEN (SC_SCn_CTL[0]).[WH1]
     * |        |          |[WH1]Tag84
     * |[16]    |DATSTS    |SC_SCn_DATA Pin Status (Read Only)
     * |        |          |This bit is the pin status of SC_SCn_DATA.
     * |        |          |0 = The SC_SCn_DATA pin status is low.
     * |        |          |1 = The SC_SCn_DATA pin status is high.
     * |[17]    |PWRSTS    |SCn_PWR Pin Status (Read Only)
     * |        |          |This bit is the pin status of SC_SCn_PWR.
     * |        |          |0 = SC_SCn_PWR pin to low.
     * |        |          |1 = SC_SCn_PWR pin to high.[WH1]
     * |        |          |[WH1]Tag84
     * |[18]    |RSTSTS    |SC_SCn_RST Pin Status (Read Only)
     * |        |          |This bit is the pin status of SC_SCn_RST.
     * |        |          |0 = SC_SCn_RST pin is low.
     * |        |          |1 = SC_SCn_RST pin is high. [WH1]
     * |        |          |[WH1]Tag83
     * |[30]    |SYNC      |SYNC Flag Indicator (Read Only)
     * |        |          |Due to synchronization, user should check this bit when writing a new value to SC_SCn_PINCTL register.
     * |        |          |0 = Synchronizing is completion, user can write new data to SC_SCn_PINCTL register.
     * |        |          |1 = Last value is synchronizing. [WH1]
     * |        |          |[WH1]Tag82
     * @var SC_T::TMRCTL0
     * Offset: 0x28  SC Timer0 Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[23:0]  |CNT       |Timer 0 Counter Value
     * |        |          |This field indicates the internal Timer0 counter values.
     * |        |          |Note: Unit of Timer01 counter is ETU base.[WH1]
     * |        |          |[WH1]Tag9, 37~49
     * |[27:24] |OPMODE    |Timer 0 Operation Mode Selection
     * |        |          |This field indicates the internal 824-bit Timer0 operation selection.
     * |        |          |Refer to Table 1.1-3 Table 1.1-3 for programming Timer0.[WH1]
     * |        |          |[WH1]Tag9, 37~49
     * |[31]    |SYNC      |SYNC Flag Indicator (Read Only)
     * |        |          |Due to synchronization, softwareuser should check this bit when writing a new value to SCn_TMRCTL0 register.
     * |        |          |0 = Synchronizing is completion, user can write new data to SCn_TMRCTL0 register.
     * |        |          |1 = Last value is synchronizing. [WH1]
     * |        |          |[WH1]Tag88
     * @var SC_T::TMRCTL1
     * Offset: 0x2C  SC Timer1 Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |CNT       |Timer 1 Counter Value
     * |        |          |This field indicates the internal Timer1 counter values.
     * |        |          |Note: Unit of Timer1 counter is ETU base.[WH1]
     * |        |          |[WH1]Tag9, 37~49
     * |[27:24] |OPMODE    |Timer 1 Operation Mode Selection
     * |        |          |This field indicates the internal 8-bit Timer1 operation selection.
     * |        |          |Refer to Table 1.1-3 Table 1.1-3 for programming Timer1.[WH1]
     * |        |          |[WH1]Tag9, 37~49
     * |[31]    |SYNC      |SYNC Flag Indicator (Read Only)
     * |        |          |Due to synchronization, softwareuser should check this bit when writing a new value to SCn_TMRCTL1 register.
     * |        |          |0 = Synchronizing is completion, user can write new data to SCn_TMRCTL1 register.
     * |        |          |1 = Last value is synchronizing. [WH1]
     * |        |          |[WH1]Tag88
     * @var SC_T::TMRCTL2
     * Offset: 0x30  SC Timer2 Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |CNT       |Timer 2 Counter Value
     * |        |          |This field indicates the internal Timer2 counter values.
     * |        |          |Note: Unit of Timer2 counter is ETU base.[WH1]
     * |        |          |[WH1]Tag9, 37~49
     * |[27:24] |OPMODE    |Timer 2 Operation Mode Selection
     * |        |          |This field indicates the internal 8-bit Timer2 operation selection
     * |        |          |Refer to Table 1.1-3 Table 1.1-3 for programming Timer2.[WH1]
     * |        |          |[WH1]Tag9, 37~49
     * |[31]    |SYNC      |SYNC Flag Indicator (Read Only)
     * |        |          |Due to synchronization, user should check this bit when writing a new value to SCn_TMRCTL2 register.
     * |        |          |0 = Synchronizing is completion, user can write new data to SCn_TMRCTL2 register.
     * |        |          |1 = Last value is synchronizing. [WH1]
     * |        |          |[WH1]Tag88
     * @var SC_T::UARTCTL
     * Offset: 0x34  SC UART Mode Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |UARTEN    |UART Mode Enable Bit
     * |        |          |Sets this bit to enable UART mode function.
     * |        |          |0 = Smart Card mode.
     * |        |          |1 = UART mode.
     * |        |          |Note1: When operating in UART mode, user must set CONSEL (SC_SCn_CTL[5:4]) = 00 and AUTOCEN (SC_SCn_CTL[3]) = 0.
     * |        |          |Note2: When operating in Smart Card mode, user must set UARTEN (SC_SCn_UARTCTL[0]) = 0.
     * |        |          |Note3: When UART mode is enabled, hardware will generate a reset SC event to reset FIFO and internal state machine.[WH1]
     * |        |          |[WH1]Tag94
     * |[5:4]   |WLS       |Word Length Selection
     * |        |          |This field is used for select uart UART data transfer length.
     * |        |          |00 = Word length is 8 bits.
     * |        |          |01 = Word length is 7 bits.
     * |        |          |10 = Word length is 6 bits.
     * |        |          |11 = Word length is 5 bits.
     * |        |          |Note: In smart card mode, this WLS field must be u201800u2019.[WH1]
     * |        |          |[WH1]Tag93
     * |[6]     |PBOFF     |Parity Bit Disable Control
     * |        |          |Sets tThis bit is used for disable parity check function.
     * |        |          |0 = Parity bit is generated or checked between the u201Clast data word bitu201D and u201Cstop bitu201D of the serial data.
     * |        |          |1 = Parity bit is not generated (transmitting data) or checked (receiving data) during transfer.
     * |        |          |Note: In smart card mode, this field bit must be u20180u2019 (default setting is with parity bit)[WH1].
     * |        |          |[WH1]Tag21
     * |[7]     |OPE       |Odd Parity Enable Bit
     * |        |          |This is used for odd/even parity selection.
     * |        |          |0 = Even number of logic 1u2019s are transmitted or check the data word and parity bits in receiving mode.
     * |        |          |1 = Odd number of logic 1u2019s are transmitted or check the data word and parity bits in receiving mode.
     * |        |          |Note: This bit has effect only when PBOFF bit is u20180u2019.
     * @var SC_T::TMRDAT0
     * Offset: 0x38  SC Timer0 Current Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[23:0]  |CNT0      |Timer0 Current Data Value (Read Only)
     * |        |          |This field indicates the current counter values of Timer0.[WH1]
     * |        |          |[WH1]Tag37
     * @var SC_T::TMRDAT12
     * Offset: 0x3C  SC Timer1/2 Current Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |CNT1      |Timer1 Current Data Value (Read Only)
     * |        |          |This field indicates the current counter values of Timer1.[WH1]
     * |        |          |[WH1]Tag37
     * |[15:8]  |CNT2      |Timer2 Current Data Value (Read Only)
     * |        |          |This field indicates the current counter values of Timer2.
     * @var SC_T::ACTCTL
     * Offset: 0x4C  SC Activation Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[4:0]   |T1EXT     |T1 Extend Time of Hardware Activation
     * |        |          |This field provide the configurable cycles to extend the activation time T1 period.
     * |        |          |Please refer to SC activation sequence in Figure 1.1-4.
     * |        |          |The cycle scaling factor is 2048 and .
     * |        |          |Extend cycles = (filled valueT1EXT * 2048) cycles.
     * |        |          |Refer to SC activation sequence in Figure 1.1-4 SC Activation Sequence.
     * |        |          |For example:,
     * |        |          |If SCLK is= 4MHz, each clock cycle cyiscle = 0.25us.,.
     * |        |          |Filled 20 to this T1EXT, thenfield
     * |        |          |Extend time == 20 * 2048 * 0.25us = 10.24 ms.
     * |        |          |Note: Setting 0 to this field conforms to the protocol ISO/IEC 7816-3[WH1].
     * |        |          |[WH1]Tag98
     */

    __IO uint32_t DAT;                   /*!< [0x0000] SC Receive/Transmit Holding Buffer Register                      */
    __IO uint32_t CTL;                   /*!< [0x0004] SC Control Register                                              */
    __IO uint32_t ALTCTL;                /*!< [0x0008] SC Alternate Control Register                                    */
    __IO uint32_t EGT;                   /*!< [0x000c] SC Extra Guard Time Register                                     */
    __IO uint32_t RXTOUT;                /*!< [0x0010] SC Receive Buffer Time-out Counter Register                      */
    __IO uint32_t ETUCTL;                /*!< [0x0014] SC Element Time Unit Control Register                            */
    __IO uint32_t INTEN;                 /*!< [0x0018] SC Interrupt Enable Control Register                             */
    __IO uint32_t INTSTS;                /*!< [0x001c] SC Interrupt Status Register                                     */
    __IO uint32_t STATUS;                /*!< [0x0020] SC Transfer Status Register                                      */
    __IO uint32_t PINCTL;                /*!< [0x0024] SC Pin Control State Register                                    */
    __IO uint32_t TMRCTL0;               /*!< [0x0028] SC Timer0 Control Register                                       */
    __IO uint32_t TMRCTL1;               /*!< [0x002c] SC Timer1 Control Register                                       */
    __IO uint32_t TMRCTL2;               /*!< [0x0030] SC Timer2 Control Register                                       */
    __IO uint32_t UARTCTL;               /*!< [0x0034] SC UART Mode Control Register                                    */
    __I  uint32_t TMRDAT0;               /*!< [0x0038] SC Timer0 Current Data Register                                  */
    __I  uint32_t TMRDAT12;              /*!< [0x003c] SC Timer1/2 Current Data Register                                */
    __I  uint32_t RESERVE0[3];
    __IO uint32_t ACTCTL;                /*!< [0x004c] SC Activation Control Register                                   */

////    __IO uint32_t DAT;           /* Offset: 0x00  SC Receiving/Transmit Holding Buffer Register.                     */
////    __IO uint32_t CTL;           /* Offset: 0x04  SC Control Register.                                               */
////    __IO uint32_t ALTCTL;        /* Offset: 0x08  SC Alternate Control Register.                                     */
////    __IO uint32_t EGT;           /* Offset: 0x0C  SC Extend Guard Time Register.                                     */
////    __IO uint32_t RXTOUT;        /* Offset: 0x10  SC Receive buffer Time-out Register.                               */
////    __IO uint32_t ETUCTL;        /* Offset: 0x14  SC ETU Control Register.                                           */
////    __IO uint32_t INTEN;         /* Offset: 0x18  SC Interrupt Enable Control Register.                              */
////    __IO uint32_t INTSTS;        /* Offset: 0x1C  SC Interrupt Status Register.                                      */
////    __IO uint32_t STATUS;        /* Offset: 0x20  SC Status Register.                                                */
////    __IO uint32_t PINCTL;        /* Offset: 0x24  SC Pin Control State Register.                                     */
////    __IO uint32_t TMRCTL0;       /* Offset: 0x28  SC Internal Timer Control Register 0.                              */
////    __IO uint32_t TMRCTL1;       /* Offset: 0x2C  SC Internal Timer Control Register 1.                              */
////    __IO uint32_t TMRCTL2;       /* Offset: 0x30  SC Internal Timer Control Register 2.                              */
////    __IO uint32_t UARTCTL;       /* Offset: 0x34  SC UART Mode Control Register.                                     */
////    __I  uint32_t TMRDAT0;       /* Offset: 0x38  SC Timer0 Current Data Register.                                   */
////    __I  uint32_t TMRDAT1_2;     /* Offset: 0x3C  SC Timer1/2 Current Data Register.                                 */
//////    __I  uint32_t RESERVE[3];
////        /* Not released in TRM */
////        __IO  uint32_t  PETCTL; /*!< Offset: 0x0040  SC Parity Error Timing Setting Register                     */
////        __IO  uint32_t  BGTEXT; /*!< Offset: 0x0044  SC Block Guard Timing Extent  Register                  */
////        __IO  uint32_t  EGTADJ; /*!< Offset: 0x0048  SC Block Guard Timing Extent  Register                 */
////    __IO uint32_t ACTCTL;        /* Offset: 0x4C  SC Activation Control Register.                                    */
////

} SC_T;

/**
    @addtogroup SC_CONST SC Bit Field Definition
    Constant Definitions for SC Controller
@{ */

#define SC_DAT_DAT_Pos                   (0)                                               /*!< SC_T::DAT: DAT Position                */
#define SC_DAT_DAT_Msk                   (0xfful << SC_DAT_DAT_Pos)                        /*!< SC_T::DAT: DAT Mask                    */

#define SC_CTL_SCEN_Pos                  (0)                                               /*!< SC_T::CTL: SCEN Position               */
#define SC_CTL_SCEN_Msk                  (0x1ul << SC_CTL_SCEN_Pos)                        /*!< SC_T::CTL: SCEN Mask                   */

#define SC_CTL_RXOFF_Pos                 (1)                                               /*!< SC_T::CTL: RXOFF Position              */
#define SC_CTL_RXOFF_Msk                 (0x1ul << SC_CTL_RXOFF_Pos)                       /*!< SC_T::CTL: RXOFF Mask                  */

#define SC_CTL_TXOFF_Pos                 (2)                                               /*!< SC_T::CTL: TXOFF Position              */
#define SC_CTL_TXOFF_Msk                 (0x1ul << SC_CTL_TXOFF_Pos)                       /*!< SC_T::CTL: TXOFF Mask                  */

#define SC_CTL_AUTOCEN_Pos               (3)                                               /*!< SC_T::CTL: AUTOCEN Position            */
#define SC_CTL_AUTOCEN_Msk               (0x1ul << SC_CTL_AUTOCEN_Pos)                     /*!< SC_T::CTL: AUTOCEN Mask                */

#define SC_CTL_CONSEL_Pos                (4)                                               /*!< SC_T::CTL: CONSEL Position             */
#define SC_CTL_CONSEL_Msk                (0x3ul << SC_CTL_CONSEL_Pos)                      /*!< SC_T::CTL: CONSEL Mask                 */

#define SC_CTL_RXTRGLV_Pos               (6)                                               /*!< SC_T::CTL: RXTRGLV Position            */
#define SC_CTL_RXTRGLV_Msk               (0x3ul << SC_CTL_RXTRGLV_Pos)                     /*!< SC_T::CTL: RXTRGLV Mask                */

#define SC_CTL_BGT_Pos                   (8)                                               /*!< SC_T::CTL: BGT Position                */
#define SC_CTL_BGT_Msk                   (0x1ful << SC_CTL_BGT_Pos)                        /*!< SC_T::CTL: BGT Mask                    */

#define SC_CTL_TMRSEL_Pos                (13)                                              /*!< SC_T::CTL: TMRSEL Position             */
#define SC_CTL_TMRSEL_Msk                (0x3ul << SC_CTL_TMRSEL_Pos)                      /*!< SC_T::CTL: TMRSEL Mask                 */

#define SC_CTL_NSB_Pos                   (15)                                              /*!< SC_T::CTL: NSB Position                */
#define SC_CTL_NSB_Msk                   (0x1ul << SC_CTL_NSB_Pos)                         /*!< SC_T::CTL: NSB Mask                    */

#define SC_CTL_RXRTY_Pos                 (16)                                              /*!< SC_T::CTL: RXRTY Position              */
#define SC_CTL_RXRTY_Msk                 (0x7ul << SC_CTL_RXRTY_Pos)                       /*!< SC_T::CTL: RXRTY Mask                  */

#define SC_CTL_RXRTYEN_Pos               (19)                                              /*!< SC_T::CTL: RXRTYEN Position            */
#define SC_CTL_RXRTYEN_Msk               (0x1ul << SC_CTL_RXRTYEN_Pos)                     /*!< SC_T::CTL: RXRTYEN Mask                */

#define SC_CTL_TXRTY_Pos                 (20)                                              /*!< SC_T::CTL: TXRTY Position              */
#define SC_CTL_TXRTY_Msk                 (0x7ul << SC_CTL_TXRTY_Pos)                       /*!< SC_T::CTL: TXRTY Mask                  */

#define SC_CTL_TXRTYEN_Pos               (23)                                              /*!< SC_T::CTL: TXRTYEN Position            */
#define SC_CTL_TXRTYEN_Msk               (0x1ul << SC_CTL_TXRTYEN_Pos)                     /*!< SC_T::CTL: TXRTYEN Mask                */

#define SC_CTL_CDDBSEL_Pos               (24)                                              /*!< SC_T::CTL: CDDBSEL Position            */
#define SC_CTL_CDDBSEL_Msk               (0x3ul << SC_CTL_CDDBSEL_Pos)                     /*!< SC_T::CTL: CDDBSEL Mask                */

#define SC_CTL_CDLV_Pos                  (26)                                              /*!< SC_T::CTL: CDLV Position               */
#define SC_CTL_CDLV_Msk                  (0x1ul << SC_CTL_CDLV_Pos)                        /*!< SC_T::CTL: CDLV Mask                   */

#define SC_CTL_SYNC_Pos                  (30)                                              /*!< SC_T::CTL: SYNC Position               */
#define SC_CTL_SYNC_Msk                  (0x1ul << SC_CTL_SYNC_Pos)                        /*!< SC_T::CTL: SYNC Mask                   */

#define SC_ALTCTL_TXRST_Pos              (0)                                               /*!< SC_T::ALTCTL: TXRST Position           */
#define SC_ALTCTL_TXRST_Msk              (0x1ul << SC_ALTCTL_TXRST_Pos)                    /*!< SC_T::ALTCTL: TXRST Mask               */

#define SC_ALTCTL_RXRST_Pos              (1)                                               /*!< SC_T::ALTCTL: RXRST Position           */
#define SC_ALTCTL_RXRST_Msk              (0x1ul << SC_ALTCTL_RXRST_Pos)                    /*!< SC_T::ALTCTL: RXRST Mask               */

#define SC_ALTCTL_DACTEN_Pos             (2)                                               /*!< SC_T::ALTCTL: DACTEN Position          */
#define SC_ALTCTL_DACTEN_Msk             (0x1ul << SC_ALTCTL_DACTEN_Pos)                   /*!< SC_T::ALTCTL: DACTEN Mask              */

#define SC_ALTCTL_ACTEN_Pos              (3)                                               /*!< SC_T::ALTCTL: ACTEN Position           */
#define SC_ALTCTL_ACTEN_Msk              (0x1ul << SC_ALTCTL_ACTEN_Pos)                    /*!< SC_T::ALTCTL: ACTEN Mask               */

#define SC_ALTCTL_WARSTEN_Pos            (4)                                               /*!< SC_T::ALTCTL: WARSTEN Position         */
#define SC_ALTCTL_WARSTEN_Msk            (0x1ul << SC_ALTCTL_WARSTEN_Pos)                  /*!< SC_T::ALTCTL: WARSTEN Mask             */

#define SC_ALTCTL_CNTEN0_Pos             (5)                                               /*!< SC_T::ALTCTL: CNTEN0 Position          */
#define SC_ALTCTL_CNTEN0_Msk             (0x1ul << SC_ALTCTL_CNTEN0_Pos)                   /*!< SC_T::ALTCTL: CNTEN0 Mask              */

#define SC_ALTCTL_CNTEN1_Pos             (6)                                               /*!< SC_T::ALTCTL: CNTEN1 Position          */
#define SC_ALTCTL_CNTEN1_Msk             (0x1ul << SC_ALTCTL_CNTEN1_Pos)                   /*!< SC_T::ALTCTL: CNTEN1 Mask              */

#define SC_ALTCTL_CNTEN2_Pos             (7)                                               /*!< SC_T::ALTCTL: CNTEN2 Position          */
#define SC_ALTCTL_CNTEN2_Msk             (0x1ul << SC_ALTCTL_CNTEN2_Pos)                   /*!< SC_T::ALTCTL: CNTEN2 Mask              */

#define SC_ALTCTL_INITSEL_Pos            (8)                                               /*!< SC_T::ALTCTL: INITSEL Position         */
#define SC_ALTCTL_INITSEL_Msk            (0x3ul << SC_ALTCTL_INITSEL_Pos)                  /*!< SC_T::ALTCTL: INITSEL Mask             */

#define SC_ALTCTL_ADACEN_Pos             (11)                                              /*!< SC_T::ALTCTL: ADACEN Position          */
#define SC_ALTCTL_ADACEN_Msk             (0x1ul << SC_ALTCTL_ADACEN_Pos)                   /*!< SC_T::ALTCTL: ADACEN Mask              */

#define SC_ALTCTL_RXBGTEN_Pos            (12)                                              /*!< SC_T::ALTCTL: RXBGTEN Position         */
#define SC_ALTCTL_RXBGTEN_Msk            (0x1ul << SC_ALTCTL_RXBGTEN_Pos)                  /*!< SC_T::ALTCTL: RXBGTEN Mask             */

#define SC_ALTCTL_ACTSTS0_Pos            (13)                                              /*!< SC_T::ALTCTL: ACTSTS0 Position         */
#define SC_ALTCTL_ACTSTS0_Msk            (0x1ul << SC_ALTCTL_ACTSTS0_Pos)                  /*!< SC_T::ALTCTL: ACTSTS0 Mask             */

#define SC_ALTCTL_ACTSTS1_Pos            (14)                                              /*!< SC_T::ALTCTL: ACTSTS1 Position         */
#define SC_ALTCTL_ACTSTS1_Msk            (0x1ul << SC_ALTCTL_ACTSTS1_Pos)                  /*!< SC_T::ALTCTL: ACTSTS1 Mask             */

#define SC_ALTCTL_ACTSTS2_Pos            (15)                                              /*!< SC_T::ALTCTL: ACTSTS2 Position         */
#define SC_ALTCTL_ACTSTS2_Msk            (0x1ul << SC_ALTCTL_ACTSTS2_Pos)                  /*!< SC_T::ALTCTL: ACTSTS2 Mask             */

#define SC_ALTCTL_SYNC_Pos               (31)                                              /*!< SC_T::ALTCTL: SYNC Position            */
#define SC_ALTCTL_SYNC_Msk               (0x1ul << SC_ALTCTL_SYNC_Pos)                     /*!< SC_T::ALTCTL: SYNC Mask                */

#define SC_EGT_EGT_Pos                   (0)                                               /*!< SC_T::EGT: EGT Position                */
#define SC_EGT_EGT_Msk                   (0xfful << SC_EGT_EGT_Pos)                        /*!< SC_T::EGT: EGT Mask                    */

#define SC_RXTOUT_RFTM_Pos               (0)                                               /*!< SC_T::RXTOUT: RFTM Position            */
#define SC_RXTOUT_RFTM_Msk               (0x1fful << SC_RXTOUT_RFTM_Pos)                   /*!< SC_T::RXTOUT: RFTM Mask                */

#define SC_ETUCTL_ETURDIV_Pos            (0)                                               /*!< SC_T::ETUCTL: ETURDIV Position         */
#define SC_ETUCTL_ETURDIV_Msk            (0xffful << SC_ETUCTL_ETURDIV_Pos)                /*!< SC_T::ETUCTL: ETURDIV Mask             */

#define SC_INTEN_RDAIEN_Pos              (0)                                               /*!< SC_T::INTEN: RDAIEN Position           */
#define SC_INTEN_RDAIEN_Msk              (0x1ul << SC_INTEN_RDAIEN_Pos)                    /*!< SC_T::INTEN: RDAIEN Mask               */

#define SC_INTEN_TXEIEN_Pos              (1)                                               /*!< SC_T::INTEN: TXEIEN Position           */
#define SC_INTEN_TXEIEN_Msk              (0x1ul << SC_INTEN_TXEIEN_Pos)                    /*!< SC_T::INTEN: TXEIEN Mask               */

#define SC_INTEN_TERRIEN_Pos             (2)                                               /*!< SC_T::INTEN: TERRIEN Position          */
#define SC_INTEN_TERRIEN_Msk             (0x1ul << SC_INTEN_TERRIEN_Pos)                   /*!< SC_T::INTEN: TERRIEN Mask              */

#define SC_INTEN_TMR0IEN_Pos             (3)                                               /*!< SC_T::INTEN: TMR0IEN Position          */
#define SC_INTEN_TMR0IEN_Msk             (0x1ul << SC_INTEN_TMR0IEN_Pos)                   /*!< SC_T::INTEN: TMR0IEN Mask              */

#define SC_INTEN_TMR1IEN_Pos             (4)                                               /*!< SC_T::INTEN: TMR1IEN Position          */
#define SC_INTEN_TMR1IEN_Msk             (0x1ul << SC_INTEN_TMR1IEN_Pos)                   /*!< SC_T::INTEN: TMR1IEN Mask              */

#define SC_INTEN_TMR2IEN_Pos             (5)                                               /*!< SC_T::INTEN: TMR2IEN Position          */
#define SC_INTEN_TMR2IEN_Msk             (0x1ul << SC_INTEN_TMR2IEN_Pos)                   /*!< SC_T::INTEN: TMR2IEN Mask              */

#define SC_INTEN_BGTIEN_Pos              (6)                                               /*!< SC_T::INTEN: BGTIEN Position           */
#define SC_INTEN_BGTIEN_Msk              (0x1ul << SC_INTEN_BGTIEN_Pos)                    /*!< SC_T::INTEN: BGTIEN Mask               */

#define SC_INTEN_CDIEN_Pos               (7)                                               /*!< SC_T::INTEN: CDIEN Position            */
#define SC_INTEN_CDIEN_Msk               (0x1ul << SC_INTEN_CDIEN_Pos)                     /*!< SC_T::INTEN: CDIEN Mask                */

#define SC_INTEN_INITIEN_Pos             (8)                                               /*!< SC_T::INTEN: INITIEN Position          */
#define SC_INTEN_INITIEN_Msk             (0x1ul << SC_INTEN_INITIEN_Pos)                   /*!< SC_T::INTEN: INITIEN Mask              */

#define SC_INTEN_RXTOIEN_Pos             (9)                                               /*!< SC_T::INTEN: RXTOIEN Position          */
#define SC_INTEN_RXTOIEN_Msk             (0x1ul << SC_INTEN_RXTOIEN_Pos)                   /*!< SC_T::INTEN: RXTOIEN Mask              */

#define SC_INTEN_ACERRIEN_Pos            (10)                                              /*!< SC_T::INTEN: ACERRIEN Position         */
#define SC_INTEN_ACERRIEN_Msk            (0x1ul << SC_INTEN_ACERRIEN_Pos)                  /*!< SC_T::INTEN: ACERRIEN Mask             */

#define SC_INTSTS_RDAIF_Pos              (0)                                               /*!< SC_T::INTSTS: RDAIF Position           */
#define SC_INTSTS_RDAIF_Msk              (0x1ul << SC_INTSTS_RDAIF_Pos)                    /*!< SC_T::INTSTS: RDAIF Mask               */

#define SC_INTSTS_TXEIF_Pos              (1)                                               /*!< SC_T::INTSTS: TXEIF Position           */
#define SC_INTSTS_TXEIF_Msk              (0x1ul << SC_INTSTS_TXEIF_Pos)                    /*!< SC_T::INTSTS: TXEIF Mask               */

#define SC_INTSTS_TERRIF_Pos             (2)                                               /*!< SC_T::INTSTS: TERRIF Position          */
#define SC_INTSTS_TERRIF_Msk             (0x1ul << SC_INTSTS_TERRIF_Pos)                   /*!< SC_T::INTSTS: TERRIF Mask              */

#define SC_INTSTS_TMR0IF_Pos             (3)                                               /*!< SC_T::INTSTS: TMR0IF Position          */
#define SC_INTSTS_TMR0IF_Msk             (0x1ul << SC_INTSTS_TMR0IF_Pos)                   /*!< SC_T::INTSTS: TMR0IF Mask              */

#define SC_INTSTS_TMR1IF_Pos             (4)                                               /*!< SC_T::INTSTS: TMR1IF Position          */
#define SC_INTSTS_TMR1IF_Msk             (0x1ul << SC_INTSTS_TMR1IF_Pos)                   /*!< SC_T::INTSTS: TMR1IF Mask              */

#define SC_INTSTS_TMR2IF_Pos             (5)                                               /*!< SC_T::INTSTS: TMR2IF Position          */
#define SC_INTSTS_TMR2IF_Msk             (0x1ul << SC_INTSTS_TMR2IF_Pos)                   /*!< SC_T::INTSTS: TMR2IF Mask              */

#define SC_INTSTS_BGTIF_Pos              (6)                                               /*!< SC_T::INTSTS: BGTIF Position           */
#define SC_INTSTS_BGTIF_Msk              (0x1ul << SC_INTSTS_BGTIF_Pos)                    /*!< SC_T::INTSTS: BGTIF Mask               */

#define SC_INTSTS_CDIF_Pos               (7)                                               /*!< SC_T::INTSTS: CDIF Position            */
#define SC_INTSTS_CDIF_Msk               (0x1ul << SC_INTSTS_CDIF_Pos)                     /*!< SC_T::INTSTS: CDIF Mask                */

#define SC_INTSTS_INITIF_Pos             (8)                                               /*!< SC_T::INTSTS: INITIF Position          */
#define SC_INTSTS_INITIF_Msk             (0x1ul << SC_INTSTS_INITIF_Pos)                   /*!< SC_T::INTSTS: INITIF Mask              */

#define SC_INTSTS_RXTOIF_Pos             (9)                                               /*!< SC_T::INTSTS: RXTOIF Position          */
#define SC_INTSTS_RXTOIF_Msk             (0x1ul << SC_INTSTS_RXTOIF_Pos)                   /*!< SC_T::INTSTS: RXTOIF Mask              */

#define SC_INTSTS_ACERRIF_Pos            (10)                                              /*!< SC_T::INTSTS: ACERRIF Position         */
#define SC_INTSTS_ACERRIF_Msk            (0x1ul << SC_INTSTS_ACERRIF_Pos)                  /*!< SC_T::INTSTS: ACERRIF Mask             */

#define SC_STATUS_RXOV_Pos               (0)                                               /*!< SC_T::STATUS: RXOV Position            */
#define SC_STATUS_RXOV_Msk               (0x1ul << SC_STATUS_RXOV_Pos)                     /*!< SC_T::STATUS: RXOV Mask                */

#define SC_STATUS_RXEMPTY_Pos            (1)                                               /*!< SC_T::STATUS: RXEMPTY Position         */
#define SC_STATUS_RXEMPTY_Msk            (0x1ul << SC_STATUS_RXEMPTY_Pos)                  /*!< SC_T::STATUS: RXEMPTY Mask             */

#define SC_STATUS_RXFULL_Pos             (2)                                               /*!< SC_T::STATUS: RXFULL Position          */
#define SC_STATUS_RXFULL_Msk             (0x1ul << SC_STATUS_RXFULL_Pos)                   /*!< SC_T::STATUS: RXFULL Mask              */

#define SC_STATUS_PEF_Pos                (4)                                               /*!< SC_T::STATUS: PEF Position             */
#define SC_STATUS_PEF_Msk                (0x1ul << SC_STATUS_PEF_Pos)                      /*!< SC_T::STATUS: PEF Mask                 */

#define SC_STATUS_FEF_Pos                (5)                                               /*!< SC_T::STATUS: FEF Position             */
#define SC_STATUS_FEF_Msk                (0x1ul << SC_STATUS_FEF_Pos)                      /*!< SC_T::STATUS: FEF Mask                 */

#define SC_STATUS_BEF_Pos                (6)                                               /*!< SC_T::STATUS: BEF Position             */
#define SC_STATUS_BEF_Msk                (0x1ul << SC_STATUS_BEF_Pos)                      /*!< SC_T::STATUS: BEF Mask                 */

#define SC_STATUS_TXOV_Pos               (8)                                               /*!< SC_T::STATUS: TXOV Position            */
#define SC_STATUS_TXOV_Msk               (0x1ul << SC_STATUS_TXOV_Pos)                     /*!< SC_T::STATUS: TXOV Mask                */

#define SC_STATUS_TXEMPTY_Pos            (9)                                               /*!< SC_T::STATUS: TXEMPTY Position         */
#define SC_STATUS_TXEMPTY_Msk            (0x1ul << SC_STATUS_TXEMPTY_Pos)                  /*!< SC_T::STATUS: TXEMPTY Mask             */

#define SC_STATUS_TXFULL_Pos             (10)                                              /*!< SC_T::STATUS: TXFULL Position          */
#define SC_STATUS_TXFULL_Msk             (0x1ul << SC_STATUS_TXFULL_Pos)                   /*!< SC_T::STATUS: TXFULL Mask              */

#define SC_STATUS_CREMOVE_Pos            (11)                                              /*!< SC_T::STATUS: CREMOVE Position         */
#define SC_STATUS_CREMOVE_Msk            (0x1ul << SC_STATUS_CREMOVE_Pos)                  /*!< SC_T::STATUS: CREMOVE Mask             */

#define SC_STATUS_CINSERT_Pos            (12)                                              /*!< SC_T::STATUS: CINSERT Position         */
#define SC_STATUS_CINSERT_Msk            (0x1ul << SC_STATUS_CINSERT_Pos)                  /*!< SC_T::STATUS: CINSERT Mask             */

#define SC_STATUS_CDPINSTS_Pos           (13)                                              /*!< SC_T::STATUS: CDPINSTS Position        */
#define SC_STATUS_CDPINSTS_Msk           (0x1ul << SC_STATUS_CDPINSTS_Pos)                 /*!< SC_T::STATUS: CDPINSTS Mask            */

#define SC_STATUS_RXPOINT_Pos            (16)                                              /*!< SC_T::STATUS: RXPOINT Position         */
#define SC_STATUS_RXPOINT_Msk            (0x7ul << SC_STATUS_RXPOINT_Pos)                  /*!< SC_T::STATUS: RXPOINT Mask             */

#define SC_STATUS_RXRTYERR_Pos           (21)                                              /*!< SC_T::STATUS: RXRTYERR Position        */
#define SC_STATUS_RXRTYERR_Msk           (0x1ul << SC_STATUS_RXRTYERR_Pos)                 /*!< SC_T::STATUS: RXRTYERR Mask            */

#define SC_STATUS_RXOVERR_Pos            (22)                                              /*!< SC_T::STATUS: RXOVERR Position         */
#define SC_STATUS_RXOVERR_Msk            (0x1ul << SC_STATUS_RXOVERR_Pos)                  /*!< SC_T::STATUS: RXOVERR Mask             */

#define SC_STATUS_RXACT_Pos              (23)                                              /*!< SC_T::STATUS: RXACT Position           */
#define SC_STATUS_RXACT_Msk              (0x1ul << SC_STATUS_RXACT_Pos)                    /*!< SC_T::STATUS: RXACT Mask               */

#define SC_STATUS_TXPOINT_Pos            (24)                                              /*!< SC_T::STATUS: TXPOINT Position         */
#define SC_STATUS_TXPOINT_Msk            (0x7ul << SC_STATUS_TXPOINT_Pos)                  /*!< SC_T::STATUS: TXPOINT Mask             */

#define SC_STATUS_TXRTYERR_Pos           (29)                                              /*!< SC_T::STATUS: TXRTYERR Position        */
#define SC_STATUS_TXRTYERR_Msk           (0x1ul << SC_STATUS_TXRTYERR_Pos)                 /*!< SC_T::STATUS: TXRTYERR Mask            */

#define SC_STATUS_TXOVERR_Pos            (30)                                              /*!< SC_T::STATUS: TXOVERR Position         */
#define SC_STATUS_TXOVERR_Msk            (0x1ul << SC_STATUS_TXOVERR_Pos)                  /*!< SC_T::STATUS: TXOVERR Mask             */

#define SC_STATUS_TXACT_Pos              (31)                                              /*!< SC_T::STATUS: TXACT Position           */
#define SC_STATUS_TXACT_Msk              (0x1ul << SC_STATUS_TXACT_Pos)                    /*!< SC_T::STATUS: TXACT Mask               */

#define SC_PINCTL_PWREN_Pos              (0)                                               /*!< SC_T::PINCTL: PWREN Position           */
#define SC_PINCTL_PWREN_Msk              (0x1ul << SC_PINCTL_PWREN_Pos)                    /*!< SC_T::PINCTL: PWREN Mask               */

#define SC_PINCTL_SCRST_Pos              (1)                                               /*!< SC_T::PINCTL: SCRST Position           */
#define SC_PINCTL_SCRST_Msk              (0x1ul << SC_PINCTL_SCRST_Pos)                    /*!< SC_T::PINCTL: SCRST Mask               */

#define SC_PINCTL_CLKKEEP_Pos            (6)                                               /*!< SC_T::PINCTL: CLKKEEP Position         */
#define SC_PINCTL_CLKKEEP_Msk            (0x1ul << SC_PINCTL_CLKKEEP_Pos)                  /*!< SC_T::PINCTL: CLKKEEP Mask             */

#define SC_PINCTL_SCDATA_Pos             (9)                                               /*!< SC_T::PINCTL: SCDATA Position          */
#define SC_PINCTL_SCDATA_Msk             (0x1ul << SC_PINCTL_SCDATA_Pos)                   /*!< SC_T::PINCTL: SCDATA Mask              */

#define SC_PINCTL_PWRINV_Pos             (11)                                              /*!< SC_T::PINCTL: PWRINV Position          */
#define SC_PINCTL_PWRINV_Msk             (0x1ul << SC_PINCTL_PWRINV_Pos)                   /*!< SC_T::PINCTL: PWRINV Mask              */

#define SC_PINCTL_DATSTS_Pos             (16)                                              /*!< SC_T::PINCTL: DATSTS Position          */
#define SC_PINCTL_DATSTS_Msk             (0x1ul << SC_PINCTL_DATSTS_Pos)                   /*!< SC_T::PINCTL: DATSTS Mask              */

#define SC_PINCTL_PWRSTS_Pos             (17)                                              /*!< SC_T::PINCTL: PWRSTS Position          */
#define SC_PINCTL_PWRSTS_Msk             (0x1ul << SC_PINCTL_PWRSTS_Pos)                   /*!< SC_T::PINCTL: PWRSTS Mask              */

#define SC_PINCTL_RSTSTS_Pos             (18)                                              /*!< SC_T::PINCTL: RSTSTS Position          */
#define SC_PINCTL_RSTSTS_Msk             (0x1ul << SC_PINCTL_RSTSTS_Pos)                   /*!< SC_T::PINCTL: RSTSTS Mask              */

#define SC_PINCTL_SYNC_Pos               (30)                                              /*!< SC_T::PINCTL: SYNC Position            */
#define SC_PINCTL_SYNC_Msk               (0x1ul << SC_PINCTL_SYNC_Pos)                     /*!< SC_T::PINCTL: SYNC Mask                */

#define SC_TMRCTL0_CNT_Pos               (0)                                               /*!< SC_T::TMRCTL0: CNT Position            */
#define SC_TMRCTL0_CNT_Msk               (0xfffffful << SC_TMRCTL0_CNT_Pos)                /*!< SC_T::TMRCTL0: CNT Mask                */

#define SC_TMRCTL0_OPMODE_Pos            (24)                                              /*!< SC_T::TMRCTL0: OPMODE Position         */
#define SC_TMRCTL0_OPMODE_Msk            (0xful << SC_TMRCTL0_OPMODE_Pos)                  /*!< SC_T::TMRCTL0: OPMODE Mask             */

#define SC_TMRCTL0_SYNC_Pos              (31)                                              /*!< SC_T::TMRCTL0: SYNC Position           */
#define SC_TMRCTL0_SYNC_Msk              (0x1ul << SC_TMRCTL0_SYNC_Pos)                    /*!< SC_T::TMRCTL0: SYNC Mask               */

#define SC_TMRCTL1_CNT_Pos               (0)                                               /*!< SC_T::TMRCTL1: CNT Position            */
#define SC_TMRCTL1_CNT_Msk               (0xfful << SC_TMRCTL1_CNT_Pos)                    /*!< SC_T::TMRCTL1: CNT Mask                */

#define SC_TMRCTL1_OPMODE_Pos            (24)                                              /*!< SC_T::TMRCTL1: OPMODE Position         */
#define SC_TMRCTL1_OPMODE_Msk            (0xful << SC_TMRCTL1_OPMODE_Pos)                  /*!< SC_T::TMRCTL1: OPMODE Mask             */

#define SC_TMRCTL1_SYNC_Pos              (31)                                              /*!< SC_T::TMRCTL1: SYNC Position           */
#define SC_TMRCTL1_SYNC_Msk              (0x1ul << SC_TMRCTL1_SYNC_Pos)                    /*!< SC_T::TMRCTL1: SYNC Mask               */

#define SC_TMRCTL2_CNT_Pos               (0)                                               /*!< SC_T::TMRCTL2: CNT Position            */
#define SC_TMRCTL2_CNT_Msk               (0xfful << SC_TMRCTL2_CNT_Pos)                    /*!< SC_T::TMRCTL2: CNT Mask                */

#define SC_TMRCTL2_OPMODE_Pos            (24)                                              /*!< SC_T::TMRCTL2: OPMODE Position         */
#define SC_TMRCTL2_OPMODE_Msk            (0xful << SC_TMRCTL2_OPMODE_Pos)                  /*!< SC_T::TMRCTL2: OPMODE Mask             */

#define SC_TMRCTL2_SYNC_Pos              (31)                                              /*!< SC_T::TMRCTL2: SYNC Position           */
#define SC_TMRCTL2_SYNC_Msk              (0x1ul << SC_TMRCTL2_SYNC_Pos)                    /*!< SC_T::TMRCTL2: SYNC Mask               */

#define SC_UARTCTL_UARTEN_Pos            (0)                                               /*!< SC_T::UARTCTL: UARTEN Position         */
#define SC_UARTCTL_UARTEN_Msk            (0x1ul << SC_UARTCTL_UARTEN_Pos)                  /*!< SC_T::UARTCTL: UARTEN Mask             */

#define SC_UARTCTL_WLS_Pos               (4)                                               /*!< SC_T::UARTCTL: WLS Position            */
#define SC_UARTCTL_WLS_Msk               (0x3ul << SC_UARTCTL_WLS_Pos)                     /*!< SC_T::UARTCTL: WLS Mask                */

#define SC_UARTCTL_PBOFF_Pos             (6)                                               /*!< SC_T::UARTCTL: PBOFF Position          */
#define SC_UARTCTL_PBOFF_Msk             (0x1ul << SC_UARTCTL_PBOFF_Pos)                   /*!< SC_T::UARTCTL: PBOFF Mask              */

#define SC_UARTCTL_OPE_Pos               (7)                                               /*!< SC_T::UARTCTL: OPE Position            */
#define SC_UARTCTL_OPE_Msk               (0x1ul << SC_UARTCTL_OPE_Pos)                     /*!< SC_T::UARTCTL: OPE Mask                */

#define SC_TMRDAT0_CNT0_Pos              (0)                                               /*!< SC_T::TMRDAT0: CNT0 Position           */
#define SC_TMRDAT0_CNT0_Msk              (0xfffffful << SC_TMRDAT0_CNT0_Pos)               /*!< SC_T::TMRDAT0: CNT0 Mask               */

#define SC_TMRDAT12_CNT1_Pos             (0)                                               /*!< SC_T::TMRDAT12: CNT1 Position          */
#define SC_TMRDAT12_CNT1_Msk             (0xfful << SC_TMRDAT12_CNT1_Pos)                  /*!< SC_T::TMRDAT12: CNT1 Mask              */

#define SC_TMRDAT12_CNT2_Pos             (8)                                               /*!< SC_T::TMRDAT12: CNT2 Position          */
#define SC_TMRDAT12_CNT2_Msk             (0xfful << SC_TMRDAT12_CNT2_Pos)                  /*!< SC_T::TMRDAT12: CNT2 Mask              */

#define SC_ACTCTL_T1EXT_Pos              (0)                                               /*!< SC_T::ACTCTL: T1EXT Position           */
#define SC_ACTCTL_T1EXT_Msk              (0x1ful << SC_ACTCTL_T1EXT_Pos)                   /*!< SC_T::ACTCTL: T1EXT Mask               */

/**@}*/ /* SC_CONST */
/**@}*/ /* end of SC register group */


/*---------------------- Serial Peripheral Interface Controller -------------------------*/
/**
    @addtogroup SPI Serial Peripheral Interface Controller(SPI)
    Memory Mapped Structure for SPI Controller
@{ */

typedef struct
{


    /**
     * @var SPI_T::CTL
     * Offset: 0x00  SPI Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SPIEN     |SPI Transfer Control Enable Bit
     * |        |          |In Master mode, the transfer will start when there is data in the FIFO buffer after this bit is set to 1
     * |        |          |In Slave mode, this device is ready to receive data when this bit is set to 1.
     * |        |          |0 = Transfer control Disabled.
     * |        |          |1 = Transfer control Enabled.
     * |        |          |Note: Before changing the configurations of SPIx_CTL, SPIx_CLKDIV, SPIx_SSCTL and SPIx_FIFOCTL registers, user shall clear the SPIEN (SPIx_CTL[0]) and confirm the SPIENSTS (SPIx_STATUS[15]) is 0.
     * |[1]     |RXNEG     |Receive on Negative Edge
     * |        |          |0 = Received data input signal is latched on the rising edge of SPI bus clock.
     * |        |          |1 = Received data input signal is latched on the falling edge of SPI bus clock.
     * |[2]     |TXNEG     |Transmit on Negative Edge
     * |        |          |0 = Transmitted data output signal is changed on the rising edge of SPI bus clock.
     * |        |          |1 = Transmitted data output signal is changed on the falling edge of SPI bus clock.
     * |[3]     |CLKPOL    |Clock Polarity
     * |        |          |0 = SPI bus clock is idle low.
     * |        |          |1 = SPI bus clock is idle high.
     * |[7:4]   |SUSPITV   |Suspend Interval (Master Only)
     * |        |          |The four bits provide configurable suspend interval between two successive transmit/receive transaction in a transfer
     * |        |          |The definition of the suspend interval is the interval between the last clock edge of the preceding transaction word and the first clock edge of the following transaction word
     * |        |          |The default value is 0x3
     * |        |          |The period of the suspend interval is obtained according to the following equation.
     * |        |          |(SUSPITV[3:0] + 0.5) * period of SPICLK clock cycle
     * |        |          |Example:
     * |        |          |SUSPITV = 0x0 u2026. 0.5 SPICLK clock cycle.
     * |        |          |SUSPITV = 0x1 u2026. 1.5 SPICLK clock cycle.
     * |        |          |u2026u2026
     * |        |          |SUSPITV = 0xE u2026. 14.5 SPICLK clock cycle.
     * |        |          |SUSPITV = 0xF u2026. 15.5 SPICLK clock cycle.
     * |[12:8]  |DWIDTH    |Data Width
     * |        |          |This field specifies how many bits can be transmitted / received in one transaction
     * |        |          |The minimum bit length is 8 bits and can up to 32 bits.
     * |        |          |DWIDTH = 0x08 u2026. 8 bits.
     * |        |          |DWIDTH = 0x09 u2026. 9 bits.
     * |        |          |u2026u2026
     * |        |          |DWIDTH = 0x1F u2026. 31 bits.
     * |        |          |DWIDTH = 0x00 u2026. 32 bits.
     * |[13]    |LSB       |Send LSB First
     * |        |          |0 = The MSB, which bit of transmit/receive register depends on the setting of DWIDTH, is transmitted/received first.
     * |        |          |1 = The LSB, bit 0 of the SPI TX register, is sent first to the SPI data output pin, and the first bit received from the SPI data input pin will be put in the LSB position of the RX register (bit 0 of SPI_RX).
     * |[14]    |HALFDPX   |SPI Half-duplex duplex TRANSMISSION Transfer Enable Bit
     * |        |          |This bit is used to select full-duplex or half-duplex for SPI transmissiontransfer
     * |        |          |The bit field DATDIR (SPIx_CTL[20]) can be used to set the data direction while in half-duplex transmissiontransfer.
     * |        |          |0 = SPI operates in full-duplex transmissiontransfer.
     * |        |          |1 = SPI operates in half-duplex transmissiontransfer.
     * |[15]    |RXONLY    |Receive-only FUNCTION Mode Enable Bit (Master Only)
     * |        |          |This bit field is only available in Master mode
     * |        |          |In receive-only mode, SPI Master will generate SPI bus clock continuously for receiving data bit from SPI slave device and assert the BUSY status
     * |        |          |If both AUTOSS (SPI_SSCTL[3]) and RXONLY are enabled, the output slave select signal will be activated.
     * |        |          |0 = Receive-only function mode Disabled.
     * |        |          |1 = Receive-only functionmode Enabled.
     * |        |          |Note: We suggest users switch to receive-only mode when BUSY (SPI_STATUS[0]) is low.
     * |[17]    |UNITIEN   |Unit Transfer Interrupt Enable Bit
     * |        |          |0 = SPI unit transfer interrupt Disabled.
     * |        |          |1 = SPI unit transfer interrupt Enabled.
     * |[18]    |SLAVE     |Slave Mode Control
     * |        |          |0 = Master mode.
     * |        |          |1 = Slave mode.
     * |[19]    |REORDER   |Byte Reorder Function Enable Bit
     * |        |          |0 = Byte Reorder function Disabled.
     * |        |          |1 = Byte Reorder function Enabled
     * |        |          |A byte suspend interval will be inserted among each byte
     * |        |          |The period of the byte suspend interval depends on the setting of SUSPITV.
     * |        |          |Note:
     * |        |          |Byte Reorder function is only available if DWIDTH is defined as 16, 24, and 32 bits.
     * |        |          |Byte Reorder function is not supported when the Quad or Dual I/O mode is enabled.
     * |[20]    |DATDIR    |Data Port Direction Control
     * |        |          |This bit is used to select the data input/output direction while in half-duplex transfer.ransmission.
     * |        |          |0 = SPI data is input direction.
     * |        |          |1 = SPI data is output direction.
     * @var SPI_T::CLKDIV
     * Offset: 0x04  SPI Clock Divider Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |DIVIDER   |Clock Divider
     * |        |          |The value in this field is the frequency divider for generating the peripheral clock, fspi_eclk, and the SPI bus clock of SPI Master
     * |        |          |The frequency is obtained according to the following equation.
     * |        |          |where
     * |        |          |is the peripheral clock source, which is defined in the clock control register, CLK_CLKSEL2.
     * |        |          |Note: Not supported in I2S mode.
     * @var SPI_T::SSCTL
     * Offset: 0x08  SPI Slave Select Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SS        |Slave Selection Control (Master Only)
     * |        |          |If AUTOSS bit is cleared to 0,
     * |        |          |0 = set the SPIx_SS line to inactive state.
     * |        |          |1 = set the SPIx_SS line to active state.
     * |        |          |If the AUTOSS bit is set to 1,
     * |        |          |0 = Keep the SPIx_SS line at inactive state.
     * |        |          |1 = SPIx_SS line will be automatically driven to active state for the duration of data transfer, and will be driven to inactive state for the rest of the time
     * |        |          |The active state of SPIx_SS is specified in SSACTPOL (SPIx_SSCTL[2]).
     * |[2]     |SSACTPOL  |Slave Selection Active Polarity
     * |        |          |This bit defines the active polarity of slave selection signal (SPIx_SS).
     * |        |          |0 = The slave selection signal SPIx_SS is active low.
     * |        |          |1 = The slave selection signal SPIx_SS is active high.
     * |[3]     |AUTOSS    |Automatic Slave Selection Function Enable Bit (Master Only)
     * |        |          |0 = Automatic slave selection function Disabled
     * |        |          |Slave selection signal will be asserted/de-asserted according to SS (SPIx_SSCTL[0]).
     * |        |          |1 = Automatic slave selection function Enabled.
     * |[8]     |SLVBEIEN  |Slave Mode Bit Count Error Interrupt Enable Bit
     * |        |          |0 = Slave mode bit count error interrupt Disabled.
     * |        |          |1 = Slave mode bit count error interrupt Enabled.
     * |[9]     |SLVURIEN  |Slave Mode TX Under Run Interrupt Enable Bit
     * |        |          |0 = Slave mode TX under run interrupt Disabled.
     * |        |          |1 = Slave mode TX under run interrupt Enabled.
     * |[12]    |SSACTIEN  |Slave Select Active Interrupt Enable Bit
     * |        |          |0 = Slave select active interrupt Disabled.
     * |        |          |1 = Slave select active interrupt Enabled.
     * |[13]    |SSINAIEN  |Slave Select Inactive Interrupt Enable Bit
     * |        |          |0 = Slave select inactive interrupt Disabled.
     * |        |          |1 = Slave select inactive interrupt Enabled.
     * @var SPI_T::PDMACTL
     * Offset: 0x0C  SPI PDMA Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TXPDMAEN  |Transmit PDMA Enable Bit
     * |        |          |0 = Transmit PDMA function Disabled.
     * |        |          |1 = Transmit PDMA function Enabled.
     * |        |          |Note: In SPI Master mode with full duplex transfer, if both TX and RX PDMA functions are enabled, RX PDMA function cannot be enabled prior to TX PDMA function
     * |        |          |User can enable TX PDMA function firstly or enable both functions simultaneously.
     * |[1]     |RXPDMAEN  |Receive PDMA Enable Bit
     * |        |          |0 = Receiver PDMA function Disabled.
     * |        |          |1 = Receiver PDMA function Enabled.
     * |[2]     |PDMARST   |PDMA Reset
     * |        |          |0 = No effect.
     * |        |          |1 = Reset the PDMA control logic of the SPI controller. This bit will be automatically cleared to 0.
     * @var SPI_T::FIFOCTL
     * Offset: 0x10  SPI FIFO Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RXRST     |Receive Reset
     * |        |          |0 = No effect.
     * |        |          |1 = Reset receive FIFO pointer and receive circuit
     * |        |          |The RXFULL bit will be cleared to 0 and the RXEMPTY bit will be set to 1
     * |        |          |This bit will be cleared to 0 by hardware about 3 system clock cycles + 2 peripheral clock cycles after it is set to 1
     * |        |          |User can read TXRXRST (SPIx_STATUS[23]) to check if reset is accomplished or not.
     * |        |          |Note: If there is slave receive time-out event, the RXRST will be set 1 when the SLVTORST (SPI_SSCTL[6]) is enabled.
     * |[1]     |TXRST     |Transmit Reset
     * |        |          |0 = No effect.
     * |        |          |1 = Reset transmit FIFO pointer and transmit circuit
     * |        |          |The TXFULL bit will be cleared to 0 and the TXEMPTY bit will be set to 1
     * |        |          |This bit will be cleared to 0 by hardware about 3 system clock cycles + 2 peripheral clock cycles after it is set to 1
     * |        |          |User can read TXRXRST (SPIx_STATUS[23]) to check if reset is accomplished or not.
     * |        |          |Note: If TX under-runflow event occurs in SPI Slave mode, this bit can be used to make SPI return to idle state.Note: If there is slave receive time-out event, the TXRST will be set to 1 when the SLVTORST (SPI_SSCTL[6]) is enabled.
     * |[2]     |RXTHIEN   |Receive FIFO Threshold Interrupt Enable Bit
     * |        |          |0 = RX FIFO threshold interrupt Disabled.
     * |        |          |1 = RX FIFO threshold interrupt Enabled.
     * |[3]     |TXTHIEN   |Transmit FIFO Threshold Interrupt Enable Bit
     * |        |          |0 = TX FIFO threshold interrupt Disabled.
     * |        |          |1 = TX FIFO threshold interrupt Enabled.
     * |[4]     |RXTOIEN   |Slave Receive Time-out Interrupt Enable Bit
     * |        |          |0 = Receive time-out interrupt Disabled.
     * |        |          |1 = Receive time-out interrupt Enabled.
     * |[5]     |RXOVIEN   |Receive FIFO Overrun Interrupt Enable Bit
     * |        |          |0 = Receive FIFO overrun interrupt Disabled.
     * |        |          |1 = Receive FIFO overrun interrupt Enabled.
     * |[6]     |TXUFPOL   |TX Underflow Data Polarity
     * |        |          |0 = The SPI data out is keep 0 if there is TX underflow event in Slave mode.
     * |        |          |1 = The SPI data out is keep 1 if there is TX underflow event in Slave mode.
     * |        |          |Note:
     * |        |          |1
     * |        |          |The TX underflow event occurs if there is not any data in TX FIFO when the slave selection signal is active.
     * |        |          |2. This bit should be set as 0 in I2S mode.
     * |        |          |3
     * |        |          |When TX underflow event occurs, SPIx_MISO pin state will be determined by this setting even though TX FIFO is not empty afterward
     * |        |          |Data stored in TX FIFO will be sent through SPIx_MISO pin in the next transfer frame.
     * |[7]     |TXUFIEN   |TX Underflow Interrupt Enable Bit
     * |        |          |0 = Slave TX underflow interrupt Disabled.
     * |        |          |1 = Slave TX underflow interrupt Enabled.
     * |[8]     |RXFBCLR   |Receive FIFO Buffer Clear
     * |        |          |0 = No effect.
     * |        |          |1 = Clear receive FIFO pointer
     * |        |          |The RXFULL bit will be cleared to 0 and the RXEMPTY bit will be set to 1
     * |        |          |This bit will be cleared to 0 by hardware about 1 system clock after it is set to 1.
     * |        |          |Note: The RX shift register will not be cleared.
     * |[9]     |TXFBCLR   |Transmit FIFO Buffer Clear
     * |        |          |0 = No effect.
     * |        |          |1 = Clear transmit FIFO pointer
     * |        |          |The TXFULL bit will be cleared to 0 and the TXEMPTY bit will be set to 1
     * |        |          |This bit will be cleared to 0 by hardware about 1 system clock after it is set to 1.
     * |        |          |Note: The TX shift register will not be cleared.
     * |[25:24] |RXTH      |Receive FIFO Threshold
     * |        |          |If the valid data count of the receive FIFO buffer is larger than the RXTH setting, the RXTHIF bit will be set to 1, else the RXTHIF bit will be cleared to 0
     * |[29:28] |TXTH      |Transmit FIFO Threshold
     * |        |          |If the valid data count of the transmit FIFO buffer is less than or equal to the TXTH setting, the TXTHIF bit will be set to 1, else the TXTHIF bit will be cleared to 0.
     * @var SPI_T::STATUS
     * Offset: 0x14  SPI Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |BUSY      |Busy Status (Read Only)
     * |        |          |0 = SPI controller is in idle state.
     * |        |          |1 = SPI controller is in busy state.
     * |        |          |The following listing are the bus busy conditions:
     * |        |          |a. SPIx_CTL[0] = 1 and the TXEMPTY = 0.
     * |        |          |b
     * |        |          |For SPI Master mode, SPIx_CTL[0] = 1 and the TXEMPTY = 1 but the current transaction is not finished yet.
     * |        |          |c. For SPI Master mode, SPIx_CTL[0] = 1 and RXONLY = 1.
     * |        |          |d
     * |        |          |For SPI Slave mode, the SPIx_CTL[0] = 1 and there is serial clock input into the SPI core logic when slave select is active.
     * |        |          |For SPI Slave mode, the SPIx_CTL[0] = 1 and the transmit buffer or transmit shift register is not empty even if the slave select is inactive.
     * |[1]     |UNITIF    |Unit Transfer Interrupt Flag
     * |        |          |0 = No transaction has been finished since this bit was cleared to 0.
     * |        |          |1 = SPI controller has finished one unit transfer.
     * |        |          |Note: This bit will be cleared by writing 1 to it.
     * |[2]     |SSACTIF   |Slave Select Active Interrupt Flag
     * |        |          |0 = Slave select active interrupt was cleared or not occurred.
     * |        |          |1 = Slave select active interrupt event occurred.
     * |        |          |Note: Only available in Slave mode. This bit will be cleared by writing 1 to it.
     * |[3]     |SSINAIF   |Slave Select Inactive Interrupt Flag
     * |        |          |0 = Slave select inactive interrupt was cleared or not occurred.
     * |        |          |1 = Slave select inactive interrupt event occurred.
     * |        |          |Note: Only available in Slave mode. This bit will be cleared by writing 1 to it.
     * |[4]     |SSLINE    |Slave Select Line Bus Status (Read Only)
     * |        |          |0 = The slave select line status is 0.
     * |        |          |1 = The slave select line status is 1.
     * |        |          |Note: This bit is only available in Slave mode
     * |        |          |If SSACTPOL (SPIx_SSCTL[2]) is set 0, and the SSLINE is 1, the SPI slave select is in inactive status.
     * |[6]     |SLVBEIF   |Slave Mode Bit Count Error Interrupt Flag
     * |        |          |In Slave mode, when the slave select line goes to inactive state, if bit counter is mismatch with DWIDTH, this interrupt flag will be set to 1.
     * |        |          |0 = No Slave mode bit count error event.
     * |        |          |1 = Slave mode bit count error event occurs.
     * |        |          |Note: If the slave select active but there is no any bus clock input, the SLVBCEEIF also active when the slave select goes to inactive state
     * |        |          |This bit will be cleared by writing 1 to it.
     * |[7]     |SLVURIF   |Slave Mode TX Under Run Interrupt Flag
     * |        |          |In Slave mode, if TX underflow event occurs and the slave select line goes to inactive state, this interrupt flag will be set to 1.
     * |        |          |0 = No Slave TX under run event.
     * |        |          |1 = Slave TX under run event occurs.
     * |        |          |Note: This bit will be cleared by writing 1 to it.
     * |[8]     |RXEMPTY   |Receive FIFO Buffer Empty Indicator (Read Only)
     * |        |          |0 = Receive FIFO buffer is not empty.
     * |        |          |1 = Receive FIFO buffer is empty.
     * |[9]     |RXFULL    |Receive FIFO Buffer Full Indicator (Read Only)
     * |        |          |0 = Receive FIFO buffer is not full.
     * |        |          |1 = Receive FIFO buffer is full.
     * |[10]    |RXTHIF    |Receive FIFO Threshold Interrupt Flag (Read Only)
     * |        |          |0 = The valid data count within the RXreceive FIFO buffer is smaller than or equal to the setting value of RXTH.
     * |        |          |1 = The valid data count within the receive FIFO buffer is larger than the setting value of RXTH.
     * |[11]    |RXOVIF    |Receive FIFO Overrun Interrupt Flag
     * |        |          |When the receive FIFO buffer is full, the follow-up data will be dropped and this bit will be set to 1.
     * |        |          |0 = No FIFO is over run.
     * |        |          |1 = Receive FIFO is over run.
     * |        |          |Note: This bit will be cleared by writing 1 to it.
     * |[12]    |RXTOIF    |Receive Time-out Interrupt Flag
     * |        |          |0 = No receive FIFO time-out event.
     * |        |          |1 = Receive FIFO buffer is not empty and no read operation on receive FIFO buffer over 64 SPI peripheral clock periods in Master mode or over 576 SPI peripheral clock periods in Slave mode
     * |        |          |When the received FIFO buffer is read by software, the time-out status will be cleared automatically.
     * |        |          |Note: This bit will be cleared by writing 1 to it.
     * |[15]    |SPIENSTS  |SPI Enable Status (Read Only)
     * |        |          |0 = The SPI controller is disabled.
     * |        |          |1 = The SPI controller is enabled.
     * |        |          |Note: The SPI peripheral clock is asynchronous with the system clock
     * |        |          |In order to make sure the SPI control logic is disabled, this bit indicates the real status of SPI controller.
     * |[16]    |TXEMPTY   |Transmit FIFO Buffer Empty Indicator (Read Only)
     * |        |          |0 = Transmit FIFO buffer is not empty.
     * |        |          |1 = Transmit FIFO buffer is empty.
     * |[17]    |TXFULL    |Transmit FIFO Buffer Full Indicator (Read Only)
     * |        |          |0 = Transmit FIFO buffer is not full.
     * |        |          |1 = Transmit FIFO buffer is full.
     * |[18]    |TXTHIF    |Transmit FIFO Threshold Interrupt Flag (Read Only)
     * |        |          |0 = The valid data count within the transmit FIFO buffer is larger than the setting value of TXTH.
     * |        |          |1 = The valid data count within the transmit FIFO buffer is less than or equal to the setting value of TXTH.
     * |[19]    |TXUFIF    |TX Underflow Interrupt Flag
     * |        |          |When the TX underflow event occurs, this bit will be set to 1, the state of data output pin depends on the setting of TXUFPOL.
     * |        |          |0 = No effect.
     * |        |          |1 = No data in Transmit FIFO and TX shift register when the slave selection signal is active.
     * |        |          |Note 1: This bit will be cleared by writing 1 to it.
     * |        |          |Note 2: If reset slaveu2019s transmission circuit when slave selection signal is active, this flag will be set to 1 after 2 peripheral clock cycles + 3 system clock cycles since the reset operation is done.
     * |[23]    |TXRXRST   |TX or RX Reset Status (Read Only)
     * |        |          |0 = The reset function of TXRST or RXRST is done.
     * |        |          |1 = Doing the reset function of TXRST or RXRST.
     * |        |          |Note: Both the reset operations of TXRST and RXRST need 3 system clock cycles + 2 peripheral clock cycles
     * |        |          |User can check the status of this bit to monitor the reset function is doing or done.
     * |[27:24] |RXCNT     |Receive FIFO Data Count (Read Only)
     * |        |          |This bit field indicates the valid data count of receive FIFO buffer.
     * |[31:28] |TXCNT     |Transmit FIFO Data Count (Read Only)
     * |        |          |This bit field indicates the valid data count of transmit FIFO buffer.
     * @var SPI_T::TX
     * Offset: 0x20  SPI Data Transmit Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |TX        |Data Transmit Register
     * |        |          |The data transmit registers pass through the transmitted data into the 4-level transmit FIFO buffers
     * |        |          |The number of valid bits depends on the setting of DWIDTH (SPIx_CTL[12:8]) in SPI mode or WDWIDTH (SPIx_I2SCTL[5:4]) in I2S mode.
     * |        |          |For exampleIn SPI mode, if DWIDTH is set to 0x08, the bits TX[7:0] will be transmitted
     * |        |          |If DWIDTH is set to 0x00 , the SPI controller will perform a 32-bit transfer.
     * |        |          |In I2S mode, if WDWIDTH (SPIx_I2SCTL[5:4]) is set to 0x2, the data width of audio channel is 24-bit and corresponding to TX[243:0]
     * |        |          |If WDWIDTH is set as 0x0, 0x1, or 0x3, all bits of this field are valid and referred to the data arrangement in I2S mode FIFO operation section
     * |        |          |Note: In Master mode, SPI controller will start to transfer the SPI bus clock after 1 APB clock and 6 peripheral clock cycles after user writes to this register.
     * @var SPI_T::RX
     * Offset: 0x30  SPI Data Receive Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |RX        |Data Receive Register
     * |        |          |There are 8-/4-level FIFO buffers in this controller
     * |        |          |The data receive register holds the data received from SPI data input pin
     * |        |          |If the RXEMPTY (SPIx_STATUS[8] or SPIx_I2SSTS[8]) is not set to 1, the receive FIFO buffers can be accessed through software by reading this register
     * |        |          |This is a read only register.
     * @var SPI_T::I2SCTL
     * Offset: 0x60  I2S Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |I2SEN     |I2S Controller Enable Bit
     * |        |          |0 = Disabled I2S mode.
     * |        |          |1 = Enabled I2S mode.
     * |        |          |Note:
     * |        |          |1. If enable this bit, I2Sx_BCLK will start to output in master Master mode.
     * |        |          |2
     * |        |          |Before changing the configurations of SPIx_I2SCTL, SPIx_I2SCLK, and SPIx_FIFOCTL registers, user shall clear the I2SEN (SPIx_I2SCTL[0]) and confirm the I2SENSTS (SPIx_I2SSTS[15]) is 0.
     * |[1]     |TXEN      |Transmit Enable Bit
     * |        |          |0 = Data transmit Disabled.
     * |        |          |1 = Data transmit Enabled.
     * |[2]     |RXEN      |Receive Enable Bit
     * |        |          |0 = Data receiving receive Disabled.
     * |        |          |1 = Data receiving receive Enabled.
     * |[3]     |MUTE      |Transmit Mute Enable Bit
     * |        |          |0 = Transmit data is shifted from buffer.
     * |        |          |1 = Transmit channel zero.
     * |[5:4]   |WDWIDTH   |Word Width
     * |        |          |00 = data size is 8-bit.
     * |        |          |01 = data size is 16-bit.
     * |        |          |10 = data size is 24-bit.
     * |        |          |11 = data size is 32-bit.
     * |[6]     |MONO      |Monaural Data
     * |        |          |0 = Data is stereo format.
     * |        |          |1 = Data is monaural format.
     * |[7]     |ORDER     |Stereo Data Order in FIFO
     * |        |          |0 = Left channel data at high byte.
     * |        |          |1 = Left channel data at low byte.
     * |[8]     |SLAVE     |Slave Mode
     * |        |          |I2S can operate as master or slave
     * |        |          |For Master mode, I2Sx_BCLK and I2Sx_LRCLK pins are output mode and send bit clock from NuMicrou00AEu00E4 NUC200 NUC126 series to Audio audio CODEC chip
     * |        |          |In Slave mode, I2Sx_BCLK and I2Sx_LRCLK pins are input mode and I2Sx_BCLK and I2Sx_LRCLK signals are received from outer Audio audio CODEC chip.
     * |        |          |0 = Master mode.
     * |        |          |1 = Slave mode.
     * |[15]    |MCLKEN    |Master Clock Enable Bit
     * |        |          |If MCLKEN is set to 1, I2S controller will generate master clock on SPIx_I2SMCLK pin for external audio devices.
     * |        |          |0 = Master clock Disabled.
     * |        |          |1 = Master clock Enabled.
     * |[16]    |RZCEN     |Right Channel Zero Cross Detection Enable Bit
     * |        |          |If this bit is set to 1, when right channel data sign bit change or next shift data bits are all 0 then RZCIF flag in SPIx_I2SSTS register is set to 1
     * |        |          |This function is only available in transmit operation.
     * |        |          |0 = Right channel zero cross detection Disabled.
     * |        |          |1 = Right channel zero cross detection Enabled.
     * |[17]    |LZCEN     |Left Channel Zero Cross Detection Enable Bit
     * |        |          |If this bit is set to 1, when left channel data sign bit changes or next shift data bits are all 0 then LZCIF flag in SPIx_I2SSTS register is set to 1
     * |        |          |This function is only available in transmit operation.
     * |        |          |0 = Left channel zero cross detection Disabled.
     * |        |          |1 = Left channel zero cross detection Enabled.
     * |[23]    |RXLCH     |Receive Left Channel Enable Bit
     * |        |          |When monaural format is selected (MONO = 1), I2S controller will receive right channel data if RXLCH is set to 0, and receive left channel data if RXLCH is set to 1.
     * |        |          |0 = Receive right channel data in Mono mode.
     * |        |          |1 = Receive left channel data in Mono mode.
     * |[24]    |RZCIEN    |Right Channel Zero- CCross Interrupt Enable Bit
     * |        |          |Interrupt occurs if this bit is set to 1 and right channel zero- cross event occurs.
     * |        |          |0 = Interrupt Disabled.
     * |        |          |1 = Interrupt Enabled.
     * |[25]    |LZCIEN    |Left Channel Zero- CCross Interrupt Enable Bit
     * |        |          |Interrupt occurs if this bit is set to 1 and left channel zero- cross event occurs.
     * |        |          |0 = Interrupt Disabled.
     * |        |          |1 = Interrupt Enabled.
     * |[29:28] |FORMAT    |Data Format Selection
     * |        |          |00 = I2S data format.
     * |        |          |01 = MSB justified data format.
     * |        |          |10 = PCM mode A.
     * |        |          |11 = PCM mode B.
     * @var SPI_T::I2SCLK
     * Offset: 0x64  I2S Clock Divider Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |MCLKDIV   |Master Clock Divider
     * |        |          |If MCLKEN is set to 1, I2S controller will generate master clock for external audio devices
     * |        |          |The frequency of master clock rate, fMCLKF_MCLK, is determined by the following expressions:.
     * |        |          |If MCLKDIV >= 1,. F_MCLK = F_I2SCLK/(2x(MCLKDIV)).
     * |        |          |If MCLKDIV = 0,. F_MCLK = F_I2SCLK.
     * |        |          |where
     * |        |          |is the frequency of I2S peripheral clock source, which is defined in the clock control register CLK_CLKSEL2
     * |        |          |F_I2SCLK is the frequency of I2S peripheral clock.
     * |        |          |In general, the master clock rate is 256 times sampling clock rate.
     * |[16:8]  |BCLKDIV   |Bit Clock Divider
     * |        |          |The I2S controller will generate bit clock in Master mode
     * |        |          |The bit clock rate frequency of bit clock , Ff_bclBCLKk, is determined by the following expression:.
     * |        |          |F_BCLK = F_I2SCLK /(2x(BCLKDIV + 1)) ,
     * |        |          |where
     * |        |          |F_I2SCLK is the frequency of I2S peripheral clock source, which is defined in the clock control register CLK_CLKSEL2.
     * |        |          |In I2S
     * |        |          |Slave mode, this field is used to define the frequency of peripheral clock and itu2019s determined by .
     * |        |          |The peripheral clock frequency in I2S Slave mode must be equal to or faster than 6 times of input bit clock.
     * @var SPI_T::I2SSTS
     * Offset: 0x68  I2S Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[4]     |RIGHT     |Right Channel (Read Only)
     * |        |          |This bit indicates the current transmit data is belong to which channel.
     * |        |          |0 = Left channel.
     * |        |          |1 = Right channel.
     * |[8]     |RXEMPTY   |Receive FIFO Buffer Empty Indicator (Read Only)
     * |        |          |0 = Receive FIFO buffer is not empty.
     * |        |          |1 = Receive FIFO buffer is empty.
     * |[9]     |RXFULL    |Receive FIFO Buffer Full Indicator (Read Only)
     * |        |          |0 = Receive FIFO buffer is not full.
     * |        |          |1 = Receive FIFO buffer is full.
     * |[10]    |RXTHIF    |Receive FIFO Threshold Interrupt Flag (Read Only)
     * |        |          |0 = The valid data count within the Rxreceive FIFO buffer is smaller than or equal to the setting value of RXTH.
     * |        |          |1 = The valid data count within the receive FIFO buffer is larger than the setting value of RXTH.
     * |        |          |Note: If RXTHIEN = 1 and RXTHIF = 1, the SPI/I2S controller will generate a SPI interrupt request.
     * |[11]    |RXOVIF    |Receive FIFO Overrun Interrupt Flag
     * |        |          |When the receive FIFO buffer is full, the follow-up data will be dropped and this bit will be set to 1.
     * |        |          |Note: This bit will be cleared by writing 1 to it.
     * |[12]    |RXTOIF    |Receive Time-out Interrupt Flag
     * |        |          |0 = No receive FIFO time-out event.
     * |        |          |1 = Receive FIFO buffer is not empty and no read operation on receive FIFO buffer over 64 SPI peripheral clock period in Master mode or over 576 SPI peripheral clock period in Slave mode
     * |        |          |When the received FIFO buffer is read by software, the time-out status will be cleared automatically.
     * |        |          |Note: This bit will be cleared by writing 1 to it.
     * |[15]    |I2SENSTS  |I2S Enable Status (Read Only)
     * |        |          |0 = The SPI/I2S control logic is disabled.
     * |        |          |1 = The SPI/I2S control logic is enabled.
     * |        |          |Note: The SPI peripheral clock is asynchronous with the system clock
     * |        |          |In order to make sure the SPI/I2S controller logic is disabled, this bit indicates the real status of SPI/I2S controller logic for user.
     * |[16]    |TXEMPTY   |Transmit FIFO Buffer Empty Indicator (Read Only)
     * |        |          |0 = Transmit FIFO buffer is not empty.
     * |        |          |1 = Transmit FIFO buffer is empty.
     * |[17]    |TXFULL    |Transmit FIFO Buffer Full Indicator (Read Only)
     * |        |          |0 = Transmit FIFO buffer is not full.
     * |        |          |1 = Transmit FIFO buffer is full.
     * |[18]    |TXTHIF    |Transmit FIFO Threshold Interrupt Flag (Read Only)
     * |        |          |0 = The valid data count within the transmit FIFO buffer is larger than the setting value of TXTH.
     * |        |          |1 = The valid data count within the transmit FIFO buffer is less than or equal to the setting value of TXTH.
     * |        |          |Note: If TXTHIEN = 1 and TXTHIF = 1, the SPI/I2S controller will generate a SPI interrupt request.
     * |[19]    |TXUFIF    |Transmit FIFO Underflow Interrupt Flag
     * |        |          |When the transmit FIFO buffer is empty and there is no datum written into the FIFO buffer, if there is more bus clock input, this bit will be set to 1.
     * |        |          |Note: This bit will be cleared by writing 1 to it.
     * |[20]    |RZCIF     |Right Channel Zero Cross Interrupt Flag
     * |        |          |0 = No zero cross event occurred on right channel.
     * |        |          |1 = Zero cross event occurred on right channel.
     * |[21]    |LZCIF     |Left Channel Zero Cross Interrupt Flag
     * |        |          |0 = No zero cross event occurred on left channel.
     * |        |          |1 = Zero cross event occurred on left channel.
     * |[23]    |TXRXRST   |TX or RX Reset Status (Read Only)
     * |        |          |0 = The reset function of TXRST or RXRST is done.
     * |        |          |1 = Doing the reset function of TXRST or RXRST.
     * |        |          |Note: Both the reset operations of TXRST and RXRST need 3 system clock cycles + 2 peripheral clock cycles
     * |        |          |User can check the status of this bit to monitor the reset function is doing or done.
     * |[26:24] |RXCNT     |Receive FIFO Data Count (Read Only)
     * |        |          |This bit field indicates the valid data count of receive FIFO buffer.
     * |[30:28] |TXCNT     |Transmit FIFO Data Count (Read Only)
     * |        |          |This bit field indicates the valid data count of transmit FIFO buffer.
     */

    __IO uint32_t CTL;                   /*!< [0x0000] SPI Control Register                                             */
    __IO uint32_t CLKDIV;                /*!< [0x0004] SPI Clock Divider Register                                       */
    __IO uint32_t SSCTL;                 /*!< [0x0008] SPI Slave Select Control Register                                */
    __IO uint32_t PDMACTL;               /*!< [0x000c] SPI PDMA Control Register                                        */
    __IO uint32_t FIFOCTL;               /*!< [0x0010] SPI FIFO Control Register                                        */
    __IO uint32_t STATUS;                /*!< [0x0014] SPI Status Register                                              */
    __I  uint32_t RESERVE0[2];
    __O  uint32_t TX;                    /*!< [0x0020] SPI Data Transmit Register                                       */
    __I  uint32_t RESERVE1[3];
    __I  uint32_t RX;                    /*!< [0x0030] SPI Data Receive Register                                        */
    __I  uint32_t RESERVE2[11];
    __IO uint32_t I2SCTL;                /*!< [0x0060] I2S Control Register                                             */
    __IO uint32_t I2SCLK;                /*!< [0x0064] I2S Clock Divider Control Register                               */
    __IO uint32_t I2SSTS;                /*!< [0x0068] I2S Status Register                                              */

} SPI_T;

/**
    @addtogroup SPI_CONST SPI Bit Field Definition
    Constant Definitions for SPI Controller
@{ */

#define SPI_CTL_SPIEN_Pos                (0)                                               /*!< SPI_T::CTL: SPIEN Position             */
#define SPI_CTL_SPIEN_Msk                (0x1ul << SPI_CTL_SPIEN_Pos)                      /*!< SPI_T::CTL: SPIEN Mask                 */

#define SPI_CTL_RXNEG_Pos                (1)                                               /*!< SPI_T::CTL: RXNEG Position             */
#define SPI_CTL_RXNEG_Msk                (0x1ul << SPI_CTL_RXNEG_Pos)                      /*!< SPI_T::CTL: RXNEG Mask                 */

#define SPI_CTL_TXNEG_Pos                (2)                                               /*!< SPI_T::CTL: TXNEG Position             */
#define SPI_CTL_TXNEG_Msk                (0x1ul << SPI_CTL_TXNEG_Pos)                      /*!< SPI_T::CTL: TXNEG Mask                 */

#define SPI_CTL_CLKPOL_Pos               (3)                                               /*!< SPI_T::CTL: CLKPOL Position            */
#define SPI_CTL_CLKPOL_Msk               (0x1ul << SPI_CTL_CLKPOL_Pos)                     /*!< SPI_T::CTL: CLKPOL Mask                */

#define SPI_CTL_SUSPITV_Pos              (4)                                               /*!< SPI_T::CTL: SUSPITV Position           */
#define SPI_CTL_SUSPITV_Msk              (0xful << SPI_CTL_SUSPITV_Pos)                    /*!< SPI_T::CTL: SUSPITV Mask               */

#define SPI_CTL_DWIDTH_Pos               (8)                                               /*!< SPI_T::CTL: DWIDTH Position            */
#define SPI_CTL_DWIDTH_Msk               (0x1ful << SPI_CTL_DWIDTH_Pos)                    /*!< SPI_T::CTL: DWIDTH Mask                */

#define SPI_CTL_LSB_Pos                  (13)                                              /*!< SPI_T::CTL: LSB Position               */
#define SPI_CTL_LSB_Msk                  (0x1ul << SPI_CTL_LSB_Pos)                        /*!< SPI_T::CTL: LSB Mask                   */

#define SPI_CTL_HALFDPX_Pos              (14)                                              /*!< SPI_T::CTL: HALFDPX Position           */
#define SPI_CTL_HALFDPX_Msk              (0x1ul << SPI_CTL_HALFDPX_Pos)                    /*!< SPI_T::CTL: HALFDPX Mask               */

#define SPI_CTL_RXONLY_Pos               (15)                                              /*!< SPI_T::CTL: RXONLY Position            */
#define SPI_CTL_RXONLY_Msk               (0x1ul << SPI_CTL_RXONLY_Pos)                     /*!< SPI_T::CTL: RXONLY Mask                */

#define SPI_CTL_UNITIEN_Pos              (17)                                              /*!< SPI_T::CTL: UNITIEN Position           */
#define SPI_CTL_UNITIEN_Msk              (0x1ul << SPI_CTL_UNITIEN_Pos)                    /*!< SPI_T::CTL: UNITIEN Mask               */

#define SPI_CTL_SLAVE_Pos                (18)                                              /*!< SPI_T::CTL: SLAVE Position             */
#define SPI_CTL_SLAVE_Msk                (0x1ul << SPI_CTL_SLAVE_Pos)                      /*!< SPI_T::CTL: SLAVE Mask                 */

#define SPI_CTL_REORDER_Pos              (19)                                              /*!< SPI_T::CTL: REORDER Position           */
#define SPI_CTL_REORDER_Msk              (0x1ul << SPI_CTL_REORDER_Pos)                    /*!< SPI_T::CTL: REORDER Mask               */

#define SPI_CTL_DATDIR_Pos               (20)                                              /*!< SPI_T::CTL: DATDIR Position            */
#define SPI_CTL_DATDIR_Msk               (0x1ul << SPI_CTL_DATDIR_Pos)                     /*!< SPI_T::CTL: DATDIR Mask                */

#define SPI_CLKDIV_DIVIDER_Pos           (0)                                               /*!< SPI_T::CLKDIV: DIVIDER Position        */
#define SPI_CLKDIV_DIVIDER_Msk           (0xfful << SPI_CLKDIV_DIVIDER_Pos)                /*!< SPI_T::CLKDIV: DIVIDER Mask            */

#define SPI_SSCTL_SS_Pos                 (0)                                               /*!< SPI_T::SSCTL: SS Position              */
#define SPI_SSCTL_SS_Msk                 (0x1ul << SPI_SSCTL_SS_Pos)                       /*!< SPI_T::SSCTL: SS Mask                  */

#define SPI_SSCTL_SSACTPOL_Pos           (2)                                               /*!< SPI_T::SSCTL: SSACTPOL Position        */
#define SPI_SSCTL_SSACTPOL_Msk           (0x1ul << SPI_SSCTL_SSACTPOL_Pos)                 /*!< SPI_T::SSCTL: SSACTPOL Mask            */

#define SPI_SSCTL_AUTOSS_Pos             (3)                                               /*!< SPI_T::SSCTL: AUTOSS Position          */
#define SPI_SSCTL_AUTOSS_Msk             (0x1ul << SPI_SSCTL_AUTOSS_Pos)                   /*!< SPI_T::SSCTL: AUTOSS Mask              */

#define SPI_SSCTL_SLVBEIEN_Pos           (8)                                               /*!< SPI_T::SSCTL: SLVBEIEN Position        */
#define SPI_SSCTL_SLVBEIEN_Msk           (0x1ul << SPI_SSCTL_SLVBEIEN_Pos)                 /*!< SPI_T::SSCTL: SLVBEIEN Mask            */

#define SPI_SSCTL_SLVURIEN_Pos           (9)                                               /*!< SPI_T::SSCTL: SLVURIEN Position        */
#define SPI_SSCTL_SLVURIEN_Msk           (0x1ul << SPI_SSCTL_SLVURIEN_Pos)                 /*!< SPI_T::SSCTL: SLVURIEN Mask            */

#define SPI_SSCTL_SSACTIEN_Pos           (12)                                              /*!< SPI_T::SSCTL: SSACTIEN Position        */
#define SPI_SSCTL_SSACTIEN_Msk           (0x1ul << SPI_SSCTL_SSACTIEN_Pos)                 /*!< SPI_T::SSCTL: SSACTIEN Mask            */

#define SPI_SSCTL_SSINAIEN_Pos           (13)                                              /*!< SPI_T::SSCTL: SSINAIEN Position        */
#define SPI_SSCTL_SSINAIEN_Msk           (0x1ul << SPI_SSCTL_SSINAIEN_Pos)                 /*!< SPI_T::SSCTL: SSINAIEN Mask            */

#define SPI_PDMACTL_TXPDMAEN_Pos         (0)                                               /*!< SPI_T::PDMACTL: TXPDMAEN Position      */
#define SPI_PDMACTL_TXPDMAEN_Msk         (0x1ul << SPI_PDMACTL_TXPDMAEN_Pos)               /*!< SPI_T::PDMACTL: TXPDMAEN Mask          */

#define SPI_PDMACTL_RXPDMAEN_Pos         (1)                                               /*!< SPI_T::PDMACTL: RXPDMAEN Position      */
#define SPI_PDMACTL_RXPDMAEN_Msk         (0x1ul << SPI_PDMACTL_RXPDMAEN_Pos)               /*!< SPI_T::PDMACTL: RXPDMAEN Mask          */

#define SPI_PDMACTL_PDMARST_Pos          (2)                                               /*!< SPI_T::PDMACTL: PDMARST Position       */
#define SPI_PDMACTL_PDMARST_Msk          (0x1ul << SPI_PDMACTL_PDMARST_Pos)                /*!< SPI_T::PDMACTL: PDMARST Mask           */

#define SPI_FIFOCTL_RXRST_Pos            (0)                                               /*!< SPI_T::FIFOCTL: RXRST Position         */
#define SPI_FIFOCTL_RXRST_Msk            (0x1ul << SPI_FIFOCTL_RXRST_Pos)                  /*!< SPI_T::FIFOCTL: RXRST Mask             */

#define SPI_FIFOCTL_TXRST_Pos            (1)                                               /*!< SPI_T::FIFOCTL: TXRST Position         */
#define SPI_FIFOCTL_TXRST_Msk            (0x1ul << SPI_FIFOCTL_TXRST_Pos)                  /*!< SPI_T::FIFOCTL: TXRST Mask             */

#define SPI_FIFOCTL_RXTHIEN_Pos          (2)                                               /*!< SPI_T::FIFOCTL: RXTHIEN Position       */
#define SPI_FIFOCTL_RXTHIEN_Msk          (0x1ul << SPI_FIFOCTL_RXTHIEN_Pos)                /*!< SPI_T::FIFOCTL: RXTHIEN Mask           */

#define SPI_FIFOCTL_TXTHIEN_Pos          (3)                                               /*!< SPI_T::FIFOCTL: TXTHIEN Position       */
#define SPI_FIFOCTL_TXTHIEN_Msk          (0x1ul << SPI_FIFOCTL_TXTHIEN_Pos)                /*!< SPI_T::FIFOCTL: TXTHIEN Mask           */

#define SPI_FIFOCTL_RXTOIEN_Pos          (4)                                               /*!< SPI_T::FIFOCTL: RXTOIEN Position       */
#define SPI_FIFOCTL_RXTOIEN_Msk          (0x1ul << SPI_FIFOCTL_RXTOIEN_Pos)                /*!< SPI_T::FIFOCTL: RXTOIEN Mask           */

#define SPI_FIFOCTL_RXOVIEN_Pos          (5)                                               /*!< SPI_T::FIFOCTL: RXOVIEN Position       */
#define SPI_FIFOCTL_RXOVIEN_Msk          (0x1ul << SPI_FIFOCTL_RXOVIEN_Pos)                /*!< SPI_T::FIFOCTL: RXOVIEN Mask           */

#define SPI_FIFOCTL_TXUFPOL_Pos          (6)                                               /*!< SPI_T::FIFOCTL: TXUFPOL Position       */
#define SPI_FIFOCTL_TXUFPOL_Msk          (0x1ul << SPI_FIFOCTL_TXUFPOL_Pos)                /*!< SPI_T::FIFOCTL: TXUFPOL Mask           */

#define SPI_FIFOCTL_TXUFIEN_Pos          (7)                                               /*!< SPI_T::FIFOCTL: TXUFIEN Position       */
#define SPI_FIFOCTL_TXUFIEN_Msk          (0x1ul << SPI_FIFOCTL_TXUFIEN_Pos)                /*!< SPI_T::FIFOCTL: TXUFIEN Mask           */

#define SPI_FIFOCTL_RXFBCLR_Pos          (8)                                               /*!< SPI_T::FIFOCTL: RXFBCLR Position       */
#define SPI_FIFOCTL_RXFBCLR_Msk          (0x1ul << SPI_FIFOCTL_RXFBCLR_Pos)                /*!< SPI_T::FIFOCTL: RXFBCLR Mask           */

#define SPI_FIFOCTL_TXFBCLR_Pos          (9)                                               /*!< SPI_T::FIFOCTL: TXFBCLR Position       */
#define SPI_FIFOCTL_TXFBCLR_Msk          (0x1ul << SPI_FIFOCTL_TXFBCLR_Pos)                /*!< SPI_T::FIFOCTL: TXFBCLR Mask           */

#define SPI_FIFOCTL_RXTH_Pos             (24)                                              /*!< SPI_T::FIFOCTL: RXTH Position          */
#define SPI_FIFOCTL_RXTH_Msk             (0x3ul << SPI_FIFOCTL_RXTH_Pos)                   /*!< SPI_T::FIFOCTL: RXTH Mask              */

#define SPI_FIFOCTL_TXTH_Pos             (28)                                              /*!< SPI_T::FIFOCTL: TXTH Position          */
#define SPI_FIFOCTL_TXTH_Msk             (0x3ul << SPI_FIFOCTL_TXTH_Pos)                   /*!< SPI_T::FIFOCTL: TXTH Mask              */

#define SPI_STATUS_BUSY_Pos              (0)                                               /*!< SPI_T::STATUS: BUSY Position           */
#define SPI_STATUS_BUSY_Msk              (0x1ul << SPI_STATUS_BUSY_Pos)                    /*!< SPI_T::STATUS: BUSY Mask               */

#define SPI_STATUS_UNITIF_Pos            (1)                                               /*!< SPI_T::STATUS: UNITIF Position         */
#define SPI_STATUS_UNITIF_Msk            (0x1ul << SPI_STATUS_UNITIF_Pos)                  /*!< SPI_T::STATUS: UNITIF Mask             */

#define SPI_STATUS_SSACTIF_Pos           (2)                                               /*!< SPI_T::STATUS: SSACTIF Position        */
#define SPI_STATUS_SSACTIF_Msk           (0x1ul << SPI_STATUS_SSACTIF_Pos)                 /*!< SPI_T::STATUS: SSACTIF Mask            */

#define SPI_STATUS_SSINAIF_Pos           (3)                                               /*!< SPI_T::STATUS: SSINAIF Position        */
#define SPI_STATUS_SSINAIF_Msk           (0x1ul << SPI_STATUS_SSINAIF_Pos)                 /*!< SPI_T::STATUS: SSINAIF Mask            */

#define SPI_STATUS_SSLINE_Pos            (4)                                               /*!< SPI_T::STATUS: SSLINE Position         */
#define SPI_STATUS_SSLINE_Msk            (0x1ul << SPI_STATUS_SSLINE_Pos)                  /*!< SPI_T::STATUS: SSLINE Mask             */

#define SPI_STATUS_SLVBEIF_Pos           (6)                                               /*!< SPI_T::STATUS: SLVBEIF Position        */
#define SPI_STATUS_SLVBEIF_Msk           (0x1ul << SPI_STATUS_SLVBEIF_Pos)                 /*!< SPI_T::STATUS: SLVBEIF Mask            */

#define SPI_STATUS_SLVURIF_Pos           (7)                                               /*!< SPI_T::STATUS: SLVURIF Position        */
#define SPI_STATUS_SLVURIF_Msk           (0x1ul << SPI_STATUS_SLVURIF_Pos)                 /*!< SPI_T::STATUS: SLVURIF Mask            */

#define SPI_STATUS_RXEMPTY_Pos           (8)                                               /*!< SPI_T::STATUS: RXEMPTY Position        */
#define SPI_STATUS_RXEMPTY_Msk           (0x1ul << SPI_STATUS_RXEMPTY_Pos)                 /*!< SPI_T::STATUS: RXEMPTY Mask            */

#define SPI_STATUS_RXFULL_Pos            (9)                                               /*!< SPI_T::STATUS: RXFULL Position         */
#define SPI_STATUS_RXFULL_Msk            (0x1ul << SPI_STATUS_RXFULL_Pos)                  /*!< SPI_T::STATUS: RXFULL Mask             */

#define SPI_STATUS_RXTHIF_Pos            (10)                                              /*!< SPI_T::STATUS: RXTHIF Position         */
#define SPI_STATUS_RXTHIF_Msk            (0x1ul << SPI_STATUS_RXTHIF_Pos)                  /*!< SPI_T::STATUS: RXTHIF Mask             */

#define SPI_STATUS_RXOVIF_Pos            (11)                                              /*!< SPI_T::STATUS: RXOVIF Position         */
#define SPI_STATUS_RXOVIF_Msk            (0x1ul << SPI_STATUS_RXOVIF_Pos)                  /*!< SPI_T::STATUS: RXOVIF Mask             */

#define SPI_STATUS_RXTOIF_Pos            (12)                                              /*!< SPI_T::STATUS: RXTOIF Position         */
#define SPI_STATUS_RXTOIF_Msk            (0x1ul << SPI_STATUS_RXTOIF_Pos)                  /*!< SPI_T::STATUS: RXTOIF Mask             */

#define SPI_STATUS_SPIENSTS_Pos          (15)                                              /*!< SPI_T::STATUS: SPIENSTS Position       */
#define SPI_STATUS_SPIENSTS_Msk          (0x1ul << SPI_STATUS_SPIENSTS_Pos)                /*!< SPI_T::STATUS: SPIENSTS Mask           */

#define SPI_STATUS_TXEMPTY_Pos           (16)                                              /*!< SPI_T::STATUS: TXEMPTY Position        */
#define SPI_STATUS_TXEMPTY_Msk           (0x1ul << SPI_STATUS_TXEMPTY_Pos)                 /*!< SPI_T::STATUS: TXEMPTY Mask            */

#define SPI_STATUS_TXFULL_Pos            (17)                                              /*!< SPI_T::STATUS: TXFULL Position         */
#define SPI_STATUS_TXFULL_Msk            (0x1ul << SPI_STATUS_TXFULL_Pos)                  /*!< SPI_T::STATUS: TXFULL Mask             */

#define SPI_STATUS_TXTHIF_Pos            (18)                                              /*!< SPI_T::STATUS: TXTHIF Position         */
#define SPI_STATUS_TXTHIF_Msk            (0x1ul << SPI_STATUS_TXTHIF_Pos)                  /*!< SPI_T::STATUS: TXTHIF Mask             */

#define SPI_STATUS_TXUFIF_Pos            (19)                                              /*!< SPI_T::STATUS: TXUFIF Position         */
#define SPI_STATUS_TXUFIF_Msk            (0x1ul << SPI_STATUS_TXUFIF_Pos)                  /*!< SPI_T::STATUS: TXUFIF Mask             */

#define SPI_STATUS_TXRXRST_Pos           (23)                                              /*!< SPI_T::STATUS: TXRXRST Position        */
#define SPI_STATUS_TXRXRST_Msk           (0x1ul << SPI_STATUS_TXRXRST_Pos)                 /*!< SPI_T::STATUS: TXRXRST Mask            */

#define SPI_STATUS_RXCNT_Pos             (24)                                              /*!< SPI_T::STATUS: RXCNT Position          */
#define SPI_STATUS_RXCNT_Msk             (0xful << SPI_STATUS_RXCNT_Pos)                   /*!< SPI_T::STATUS: RXCNT Mask              */

#define SPI_STATUS_TXCNT_Pos             (28)                                              /*!< SPI_T::STATUS: TXCNT Position          */
#define SPI_STATUS_TXCNT_Msk             (0xful << SPI_STATUS_TXCNT_Pos)                   /*!< SPI_T::STATUS: TXCNT Mask              */

#define SPI_TX_TX_Pos                    (0)                                               /*!< SPI_T::TX: TX Position                 */
#define SPI_TX_TX_Msk                    (0xfffffffful << SPI_TX_TX_Pos)                   /*!< SPI_T::TX: TX Mask                     */

#define SPI_RX_RX_Pos                    (0)                                               /*!< SPI_T::RX: RX Position                 */
#define SPI_RX_RX_Msk                    (0xfffffffful << SPI_RX_RX_Pos)                   /*!< SPI_T::RX: RX Mask                     */

#define SPI_I2SCTL_I2SEN_Pos             (0)                                               /*!< SPI_T::I2SCTL: I2SEN Position          */
#define SPI_I2SCTL_I2SEN_Msk             (0x1ul << SPI_I2SCTL_I2SEN_Pos)                   /*!< SPI_T::I2SCTL: I2SEN Mask              */

#define SPI_I2SCTL_TXEN_Pos              (1)                                               /*!< SPI_T::I2SCTL: TXEN Position           */
#define SPI_I2SCTL_TXEN_Msk              (0x1ul << SPI_I2SCTL_TXEN_Pos)                    /*!< SPI_T::I2SCTL: TXEN Mask               */

#define SPI_I2SCTL_RXEN_Pos              (2)                                               /*!< SPI_T::I2SCTL: RXEN Position           */
#define SPI_I2SCTL_RXEN_Msk              (0x1ul << SPI_I2SCTL_RXEN_Pos)                    /*!< SPI_T::I2SCTL: RXEN Mask               */

#define SPI_I2SCTL_MUTE_Pos              (3)                                               /*!< SPI_T::I2SCTL: MUTE Position           */
#define SPI_I2SCTL_MUTE_Msk              (0x1ul << SPI_I2SCTL_MUTE_Pos)                    /*!< SPI_T::I2SCTL: MUTE Mask               */

#define SPI_I2SCTL_WDWIDTH_Pos           (4)                                               /*!< SPI_T::I2SCTL: WDWIDTH Position        */
#define SPI_I2SCTL_WDWIDTH_Msk           (0x3ul << SPI_I2SCTL_WDWIDTH_Pos)                 /*!< SPI_T::I2SCTL: WDWIDTH Mask            */

#define SPI_I2SCTL_MONO_Pos              (6)                                               /*!< SPI_T::I2SCTL: MONO Position           */
#define SPI_I2SCTL_MONO_Msk              (0x1ul << SPI_I2SCTL_MONO_Pos)                    /*!< SPI_T::I2SCTL: MONO Mask               */

#define SPI_I2SCTL_ORDER_Pos             (7)                                               /*!< SPI_T::I2SCTL: ORDER Position          */
#define SPI_I2SCTL_ORDER_Msk             (0x1ul << SPI_I2SCTL_ORDER_Pos)                   /*!< SPI_T::I2SCTL: ORDER Mask              */

#define SPI_I2SCTL_SLAVE_Pos             (8)                                               /*!< SPI_T::I2SCTL: SLAVE Position          */
#define SPI_I2SCTL_SLAVE_Msk             (0x1ul << SPI_I2SCTL_SLAVE_Pos)                   /*!< SPI_T::I2SCTL: SLAVE Mask              */

#define SPI_I2SCTL_MCLKEN_Pos            (15)                                              /*!< SPI_T::I2SCTL: MCLKEN Position         */
#define SPI_I2SCTL_MCLKEN_Msk            (0x1ul << SPI_I2SCTL_MCLKEN_Pos)                  /*!< SPI_T::I2SCTL: MCLKEN Mask             */

#define SPI_I2SCTL_RZCEN_Pos             (16)                                              /*!< SPI_T::I2SCTL: RZCEN Position          */
#define SPI_I2SCTL_RZCEN_Msk             (0x1ul << SPI_I2SCTL_RZCEN_Pos)                   /*!< SPI_T::I2SCTL: RZCEN Mask              */

#define SPI_I2SCTL_LZCEN_Pos             (17)                                              /*!< SPI_T::I2SCTL: LZCEN Position          */
#define SPI_I2SCTL_LZCEN_Msk             (0x1ul << SPI_I2SCTL_LZCEN_Pos)                   /*!< SPI_T::I2SCTL: LZCEN Mask              */

#define SPI_I2SCTL_RXLCH_Pos             (23)                                              /*!< SPI_T::I2SCTL: RXLCH Position          */
#define SPI_I2SCTL_RXLCH_Msk             (0x1ul << SPI_I2SCTL_RXLCH_Pos)                   /*!< SPI_T::I2SCTL: RXLCH Mask              */

#define SPI_I2SCTL_RZCIEN_Pos            (24)                                              /*!< SPI_T::I2SCTL: RZCIEN Position         */
#define SPI_I2SCTL_RZCIEN_Msk            (0x1ul << SPI_I2SCTL_RZCIEN_Pos)                  /*!< SPI_T::I2SCTL: RZCIEN Mask             */

#define SPI_I2SCTL_LZCIEN_Pos            (25)                                              /*!< SPI_T::I2SCTL: LZCIEN Position         */
#define SPI_I2SCTL_LZCIEN_Msk            (0x1ul << SPI_I2SCTL_LZCIEN_Pos)                  /*!< SPI_T::I2SCTL: LZCIEN Mask             */

#define SPI_I2SCTL_FORMAT_Pos            (28)                                              /*!< SPI_T::I2SCTL: FORMAT Position         */
#define SPI_I2SCTL_FORMAT_Msk            (0x3ul << SPI_I2SCTL_FORMAT_Pos)                  /*!< SPI_T::I2SCTL: FORMAT Mask             */

#define SPI_I2SCLK_MCLKDIV_Pos           (0)                                               /*!< SPI_T::I2SCLK: MCLKDIV Position        */
#define SPI_I2SCLK_MCLKDIV_Msk           (0x3ful << SPI_I2SCLK_MCLKDIV_Pos)                /*!< SPI_T::I2SCLK: MCLKDIV Mask            */

#define SPI_I2SCLK_BCLKDIV_Pos           (8)                                               /*!< SPI_T::I2SCLK: BCLKDIV Position        */
#define SPI_I2SCLK_BCLKDIV_Msk           (0x1fful << SPI_I2SCLK_BCLKDIV_Pos)               /*!< SPI_T::I2SCLK: BCLKDIV Mask            */

#define SPI_I2SSTS_RIGHT_Pos             (4)                                               /*!< SPI_T::I2SSTS: RIGHT Position          */
#define SPI_I2SSTS_RIGHT_Msk             (0x1ul << SPI_I2SSTS_RIGHT_Pos)                   /*!< SPI_T::I2SSTS: RIGHT Mask              */

#define SPI_I2SSTS_RXEMPTY_Pos           (8)                                               /*!< SPI_T::I2SSTS: RXEMPTY Position        */
#define SPI_I2SSTS_RXEMPTY_Msk           (0x1ul << SPI_I2SSTS_RXEMPTY_Pos)                 /*!< SPI_T::I2SSTS: RXEMPTY Mask            */

#define SPI_I2SSTS_RXFULL_Pos            (9)                                               /*!< SPI_T::I2SSTS: RXFULL Position         */
#define SPI_I2SSTS_RXFULL_Msk            (0x1ul << SPI_I2SSTS_RXFULL_Pos)                  /*!< SPI_T::I2SSTS: RXFULL Mask             */

#define SPI_I2SSTS_RXTHIF_Pos            (10)                                              /*!< SPI_T::I2SSTS: RXTHIF Position         */
#define SPI_I2SSTS_RXTHIF_Msk            (0x1ul << SPI_I2SSTS_RXTHIF_Pos)                  /*!< SPI_T::I2SSTS: RXTHIF Mask             */

#define SPI_I2SSTS_RXOVIF_Pos            (11)                                              /*!< SPI_T::I2SSTS: RXOVIF Position         */
#define SPI_I2SSTS_RXOVIF_Msk            (0x1ul << SPI_I2SSTS_RXOVIF_Pos)                  /*!< SPI_T::I2SSTS: RXOVIF Mask             */

#define SPI_I2SSTS_RXTOIF_Pos            (12)                                              /*!< SPI_T::I2SSTS: RXTOIF Position         */
#define SPI_I2SSTS_RXTOIF_Msk            (0x1ul << SPI_I2SSTS_RXTOIF_Pos)                  /*!< SPI_T::I2SSTS: RXTOIF Mask             */

#define SPI_I2SSTS_I2SENSTS_Pos          (15)                                              /*!< SPI_T::I2SSTS: I2SENSTS Position       */
#define SPI_I2SSTS_I2SENSTS_Msk          (0x1ul << SPI_I2SSTS_I2SENSTS_Pos)                /*!< SPI_T::I2SSTS: I2SENSTS Mask           */

#define SPI_I2SSTS_TXEMPTY_Pos           (16)                                              /*!< SPI_T::I2SSTS: TXEMPTY Position        */
#define SPI_I2SSTS_TXEMPTY_Msk           (0x1ul << SPI_I2SSTS_TXEMPTY_Pos)                 /*!< SPI_T::I2SSTS: TXEMPTY Mask            */

#define SPI_I2SSTS_TXFULL_Pos            (17)                                              /*!< SPI_T::I2SSTS: TXFULL Position         */
#define SPI_I2SSTS_TXFULL_Msk            (0x1ul << SPI_I2SSTS_TXFULL_Pos)                  /*!< SPI_T::I2SSTS: TXFULL Mask             */

#define SPI_I2SSTS_TXTHIF_Pos            (18)                                              /*!< SPI_T::I2SSTS: TXTHIF Position         */
#define SPI_I2SSTS_TXTHIF_Msk            (0x1ul << SPI_I2SSTS_TXTHIF_Pos)                  /*!< SPI_T::I2SSTS: TXTHIF Mask             */

#define SPI_I2SSTS_TXUFIF_Pos            (19)                                              /*!< SPI_T::I2SSTS: TXUFIF Position         */
#define SPI_I2SSTS_TXUFIF_Msk            (0x1ul << SPI_I2SSTS_TXUFIF_Pos)                  /*!< SPI_T::I2SSTS: TXUFIF Mask             */

#define SPI_I2SSTS_RZCIF_Pos             (20)                                              /*!< SPI_T::I2SSTS: RZCIF Position          */
#define SPI_I2SSTS_RZCIF_Msk             (0x1ul << SPI_I2SSTS_RZCIF_Pos)                   /*!< SPI_T::I2SSTS: RZCIF Mask              */

#define SPI_I2SSTS_LZCIF_Pos             (21)                                              /*!< SPI_T::I2SSTS: LZCIF Position          */
#define SPI_I2SSTS_LZCIF_Msk             (0x1ul << SPI_I2SSTS_LZCIF_Pos)                   /*!< SPI_T::I2SSTS: LZCIF Mask              */

#define SPI_I2SSTS_TXRXRST_Pos           (23)                                              /*!< SPI_T::I2SSTS: TXRXRST Position        */
#define SPI_I2SSTS_TXRXRST_Msk           (0x1ul << SPI_I2SSTS_TXRXRST_Pos)                 /*!< SPI_T::I2SSTS: TXRXRST Mask            */

#define SPI_I2SSTS_RXCNT_Pos             (24)                                              /*!< SPI_T::I2SSTS: RXCNT Position          */
#define SPI_I2SSTS_RXCNT_Msk             (0x7ul << SPI_I2SSTS_RXCNT_Pos)                   /*!< SPI_T::I2SSTS: RXCNT Mask              */

#define SPI_I2SSTS_TXCNT_Pos             (28)                                              /*!< SPI_T::I2SSTS: TXCNT Position          */
#define SPI_I2SSTS_TXCNT_Msk             (0x7ul << SPI_I2SSTS_TXCNT_Pos)                   /*!< SPI_T::I2SSTS: TXCNT Mask              */

/**@}*/ /* SPI_CONST */
/**@}*/ /* end of SPI register group */


/*---------------------- System Manger Controller -------------------------*/
/**
    @addtogroup SYS System Manger Controller(SYS)
    Memory Mapped Structure for SYS Controller
@{ */

typedef struct
{


    /**
     * @var SYS_T::PDID
     * Offset: 0x00  Part Device Identification Number Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |PDID      |Part Device Identification Number (Read Only)
     * |        |          |This register reflects device part number code.
     * |        |          |Software can read this register to identify which device is used.
     * @var SYS_T::RSTSTS
     * Offset: 0x04  System Reset Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |PORF      |POR Reset Flag
     * |        |          |The POR reset flag is set by the Reset Signal from the Power-on Reset (POR) Controller or bit CHIPRST (SYS_IPRST0[0]) to indicate the previous reset source.
     * |        |          |0 = No reset from POR or CHIPRST.
     * |        |          |1 = Power-on Reset (POR) or CHIPRST had issued the reset signal to reset the system.
     * |        |          |Note: This bit can be cleared by software writing 1.
     * |[1]     |PINRF     |nRESET Pin Reset Flag
     * |        |          |The nRESET pin reset flag is set by the Reset Signal from the nRESET Pin to indicate the previous reset source.
     * |        |          |0 = No reset from nRESET pin.
     * |        |          |1 = Pin nRESET had issued the reset signal to reset the system.
     * |        |          |Note: This bit can be cleared by software writing 1.
     * |[2]     |WDTRF     |WDT Reset Flag
     * |        |          |The WDT reset flag is set by the Reset Signal from the Watchdog Timer or Window Watchdog Timer to indicate the previous reset source.
     * |        |          |0 = No reset from watchdog timer or window watchdog timer.
     * |        |          |1 = The watchdog timer or window watchdog timer had issued the reset signal to reset the system.
     * |        |          |Note1: This bit can be cleared by software writing 1.
     * |        |          |Note2: Watchdog Timer register RSTF(WDT_CTL[2]) bit is set if the system has been reset by WDT time-out reset.
     * |        |          |Window Watchdog Timer register WWDTRF(WWDT_STATUS[1]) bit is set if the system has been reset by WWDT time-out reset.
     * |[3]     |LVRF      |LVR Reset Flag
     * |        |          |The LVR reset flag is set by the Reset Signal from the Low Voltage Reset Controller to indicate the previous reset source.
     * |        |          |0 = No reset from LVR.
     * |        |          |1 = LVR controller had issued the reset signal to reset the system.
     * |        |          |Note: This bit can be cleared by software writing 1.
     * |[4]     |BODRF     |BOD Reset Flag
     * |        |          |The BOD reset flag is set by the Reset Signal from the Brown-out Detector to indicate the previous reset source.
     * |        |          |0 = No reset from BOD.
     * |        |          |1 = The BOD had issued the reset signal to reset the system.
     * |        |          |Note: This bit can be cleared by software writing 1.
     * |[5]     |MCURF     |MCU Reset Flag
     * |        |          |The MCU reset flag is set by the Reset Signal from the Cortex-M0 Core to indicate the previous reset source.
     * |        |          |0 = No reset from Cortex-M0.
     * |        |          |1 = The Cortex-M0 had issued the reset signal to reset the system by writing 1 to the bit SYSRESETREQ(AIRCR[2], Application Interrupt and Reset Control Register, address = 0xE000ED0C) in system control registers of Cortex-M0 core.
     * |        |          |Note: This bit can be cleared by software writing 1.
     * |[7]     |CPURF     |CPU Reset Flag
     * |        |          |The CPU reset flag is set by hardware if software writes CPURST (SYS_IPRST0[1]) 1 to reset Cortex-M0 Core and Flash Memory Controller (FMC).
     * |        |          |0 = No reset from CPU.
     * |        |          |1 = The Cortex-M0 Core and FMC are reset by software setting CPURST to 1.
     * |        |          |Note: This bit can be cleared by software writing 1.
     * |[8]     |CPULKRF   |CPU Lockup Reset Flag
     * |        |          |The CPU lockup reset flag is set by hardware if Cortex-M0 lockup happened.
     * |        |          |0 = No reset from CPU lockup happened.
     * |        |          |1 = The Cortex-M0 lockup happened and chip is reset.
     * |        |          |Note: This bit can be cleared by software writing 1.
     * @var SYS_T::IPRST0
     * Offset: 0x08  Peripheral  Reset Control Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CHIPRST   |Chip One-shot Reset (Write Protect)
     * |        |          |Setting this bit will reset the whole chip, including Processor core and all peripherals, and this bit will automatically return to 0 after the 2 clock cycles.
     * |        |          |The CHIPRST is same as the POR reset, all the chip controllers is reset and the chip setting from flash are also reload.
     * |        |          |0 = Chip normal operation.
     * |        |          |1 = Chip one-shot reset.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[1]     |CPURST    |Processor Core One-shot Reset (Write Protect)
     * |        |          |Setting this bit will only reset the processor core and Flash Memory Controller(FMC), and this bit will automatically return to 0 after the 2 clock cycles.
     * |        |          |0 = Processor core normal operation.
     * |        |          |1 = Processor core one-shot reset.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[2]     |PDMARST   |PDMA Controller Reset (Write Protect)
     * |        |          |Setting this bit to 1 will generate a reset signal to the PDMA.
     * |        |          |User needs to set this bit to 0 to release from reset state.
     * |        |          |0 = PDMA controller normal operation.
     * |        |          |1 = PDMA controller reset.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[3]     |EBIRST    |EBI Controller Reset (Write Protect)
     * |        |          |Set this bit to 1 will generate a reset signal to the EBI.
     * |        |          |User needs to set this bit to 0 to release from the reset state.
     * |        |          |0 = EBI controller normal operation.
     * |        |          |1 = EBI controller reset.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[4]     |HDIVRST   |HDIV Controller Reset (Write Protect)
     * |        |          |Set this bit to 1 will generate a reset signal to the HDIV controller.
     * |        |          |User needs to set this bit to 0 to release from the reset state.
     * |        |          |0 = HDIV controller normal operation.
     * |        |          |1 = HDIV controller reset.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[7]     |CRCRST    |CRC Calculation Controller Reset (Write Protect)
     * |        |          |Set this bit to 1 will generate a reset signal to the CRC calculation controller.
     * |        |          |User needs to set this bit to 0 to release from the reset state.
     * |        |          |0 = CRC calculation controller normal operation.
     * |        |          |1 = CRC calculation controller reset.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * @var SYS_T::IPRST1
     * Offset: 0x0C  Peripheral Reset Control Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1]     |GPIORST   |GPIO Controller Reset
     * |        |          |0 = GPIO controller normal operation.
     * |        |          |1 = GPIO controller reset.
     * |[2]     |TMR0RST   |Timer0 Controller Reset
     * |        |          |0 = Timer0 controller normal operation.
     * |        |          |1 = Timer0 controller reset.
     * |[3]     |TMR1RST   |Timer1 Controller Reset
     * |        |          |0 = Timer1 controller normal operation.
     * |        |          |1 = Timer1 controller reset.
     * |[4]     |TMR2RST   |Timer2 Controller Reset
     * |        |          |0 = Timer2 controller normal operation.
     * |        |          |1 = Timer2 controller reset.
     * |[5]     |TMR3RST   |Timer3 Controller Reset
     * |        |          |0 = Timer3 controller normal operation.
     * |        |          |1 = Timer3 controller reset.
     * |[8]     |I2C0RST   |I2C0 Controller Reset
     * |        |          |0 = I2C0 controller normal operation.
     * |        |          |1 = I2C0 controller reset.
     * |[9]     |I2C1RST   |I2C1 Controller Reset
     * |        |          |0 = I2C1 controller normal operation.
     * |        |          |1 = I2C1 controller reset.
     * |[12]    |SPI0RST   |SPI0 Controller Reset
     * |        |          |0 = SPI0 controller normal operation.
     * |        |          |1 = SPI0 controller reset.
     * |[13]    |SPI1RST   |SPI1 Controller Reset
     * |        |          |0 = SPI1 controller normal operation.
     * |        |          |1 = SPI1 controller reset.
     * |[16]    |UART0RST  |UART0 Controller Reset
     * |        |          |0 = UART0 controller normal operation.
     * |        |          |1 = UART0 controller reset.
     * |[17]    |UART1RST  |UART1 Controller Reset
     * |        |          |0 = UART1 controller normal operation.
     * |        |          |1 = UART1 controller reset.
     * |[18]    |UART2RST  |UART2 Controller Reset
     * |        |          |0 = UART2 controller normal operation.
     * |        |          |1 = UART2 controller reset.
     * |[20]    |PWM0RST   |PWM0 Controller Reset
     * |        |          |0 = PWM0 controller normal operation.
     * |        |          |1 = PWM0 controller reset.
     * |[21]    |PWM1RST   |PWM1 Controller Reset
     * |        |          |0 = PWM1 controller normal operation.
     * |        |          |1 = PWM1 controller reset.
     * |[22]    |ACMP01RST |ACMP01 Controller Reset
     * |        |          |0 = ACMP01 controller normal operation.
     * |        |          |1 = ACMP01 controller reset.
     * |[27]    |USBDRST   |USB Device Controller Reset
     * |        |          |0 = USB device controller normal operation.
     * |        |          |1 = USB device controller reset.
     * |[28]    |ADCRST    |ADC Controller Reset
     * |        |          |0 = ADC controller normal operation.
     * |        |          |1 = ADC controller reset.
     * @var SYS_T::IPRST2
     * Offset: 0x10  Peripheral Reset Control Register 2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SC0RST    |SC0 Controller Reset
     * |        |          |0 = SC0 controller normal operation.
     * |        |          |1 = SC0 controller reset.
     * |[1]     |SC1RST    |SC1 Controller Reset
     * |        |          |0 = SC1 controller normal operation.
     * |        |          |1 = SC1 controller reset.
     * |[8]     |USCI0RST  |USCI0 Controller Reset
     * |        |          |0 = USCI0 controller normal operation.
     * |        |          |1 = USCI0 controller reset.
     * |[9]     |USCI1RST  |USCI1 Controller Reset
     * |        |          |0 = USCI1 controller normal operation.
     * |        |          |1 = USCI1 controller reset.
     * |[10]    |USCI2RST  |USCI2 Controller Reset
     * |        |          |0 = USCI2 controller normal operation.
     * |        |          |1 = USCI2 controller reset.
     * @var SYS_T::BODCTL
     * Offset: 0x18  Brown-out Detector Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |BODEN     |Brown-out Detector Enable Bit (Write Protect)
     * |        |          |The default value is set by flash controller user configuration register CBODEN (CONFIG0 [23]).
     * |        |          |0 = Brown-out Detector function Disabled.
     * |        |          |1 = Brown-out Detector function Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[2:1]   |BODVL     |Brown-out Detector Threshold Voltage Selection (Write Protect)
     * |        |          |The default value is set by flash controller user configuration register CBOV (CONFIG0 [22:21]).
     * |        |          |00 = Brown-Out Detector threshold voltage is 2.2V.
     * |        |          |01 = Brown-Out Detector threshold voltage is 2.7V.
     * |        |          |10 = Brown-Out Detector threshold voltage is 3.7V.
     * |        |          |11 = Brown-Out Detector threshold voltage is 4.5V.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[3]     |BODRSTEN  |Brown-out Reset Enable Bit (Write Protect)
     * |        |          |The default value is set by flash controller user configuration register CBORST(CONFIG0[20]) bit.
     * |        |          |0 = Brown-out interrupt function Enabled.
     * |        |          |1 = Brown-out reset function Enabled.
     * |        |          |Note1:
     * |        |          |While the Brown-out Detector function is enabled (BODEN high) and BOD reset function is enabled (BODRSTEN high), BOD will assert a signal to reset chip when the detected voltage is lower than the threshold (BODOUT high).
     * |        |          |While the BOD function is enabled (BODEN high) and BOD interrupt function is enabled (BODRSTEN low), BOD will assert an interrupt if BODOUT is high
     * |        |          |BOD interrupt will keep till to the BODEN set to 0.
     * |        |          |BOD interrupt can be blocked by disabling the NVIC BOD interrupt or disabling BOD function (set BODEN low).
     * |        |          |Note2: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[4]     |BODIF     |Brown-out Detector Interrupt Flag
     * |        |          |0 = Brown-out Detector does not detect any voltage draft at VDD down through or up through the voltage of BODVL setting.
     * |        |          |1 = When Brown-out Detector detects the VDD is dropped down through the voltage of BODVL setting or the VDD is raised up through the voltage of BODVL setting, this bit is set to 1 and the brown-out interrupt is requested if brown-out interrupt is enabled.
     * |        |          |Note: This bit can be cleared by software writing 1.
     * |[5]     |BODLPM    |Brown-out Detector Low Power Mode (Write Protect)
     * |        |          |0 = BOD operate in normal mode (default).
     * |        |          |1 = BOD Low Power mode Enabled.
     * |        |          |Note1: The BOD consumes about 100uA in normal mode, the low power mode can reduce the current to about 1/10 but slow the BOD response.
     * |        |          |Note2: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[6]     |BODOUT    |Brown-out Detector Output Status
     * |        |          |0 = Brown-out Detector output status is 0.
     * |        |          |It means the detected voltage is higher than BODVL setting or BODEN is 0.
     * |        |          |1 = Brown-out Detector output status is 1.
     * |        |          |It means the detected voltage is lower than BODVL setting.
     * |        |          |If the BODEN is 0, BOD function disabled, this bit always responds 0.
     * |[7]     |LVREN     |Low Voltage Reset Enable Bit (Write Protect)
     * |        |          |The LVR function resets the chip when the input power voltage is lower than LVR circuit setting.
     * |        |          |LVR function is enabled by default.
     * |        |          |0 = Low Voltage Reset function Disabled.
     * |        |          |1 = Low Voltage Reset function Enabled.
     * |        |          |Note1: After enabling the bit, the LVR function will be active with 200us delay for LVR output stable (default).
     * |        |          |Note2: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[10:8]  |BODDGSEL  |Brown-out Detector Output De-glitch Time Select (Write Protect)
     * |        |          |000 = BOD output is sampled by RC10K clock.
     * |        |          |001 = 4 system clock (HCLK).
     * |        |          |010 = 8 system clock (HCLK).
     * |        |          |011 = 16 system clock (HCLK).
     * |        |          |100 = 32 system clock (HCLK).
     * |        |          |101 = 64 system clock (HCLK).
     * |        |          |110 = 128 system clock (HCLK).
     * |        |          |111 = 256 system clock (HCLK).
     * |        |          |Note: These bits are write protected. Refer to the SYS_REGLCTL register.
     * |[14:12] |LVRDGSEL  |LVR Output De-glitch Time Select (Write Protect)
     * |        |          |000 = Without de-glitch function.
     * |        |          |001 = 4 system clock (HCLK).
     * |        |          |010 = 8 system clock (HCLK).
     * |        |          |011 = 16 system clock (HCLK).
     * |        |          |100 = 32 system clock (HCLK).
     * |        |          |101 = 64 system clock (HCLK).
     * |        |          |110 = 128 system clock (HCLK).
     * |        |          |111 = 256 system clock (HCLK).
     * |        |          |Note: These bits are write protected. Refer to the SYS_REGLCTL register.
     * |[16]    |VDETEN    |Voltage Detector Enable Bit
     * |        |          |0 = VDET detect external input voltage function Disabled.
     * |        |          |1 = VDET detect external input voltage function Enabled.
     * |        |          |Note1: This function is still active in whole chip power-down mode.
     * |        |          |Note2: This function need use LIRC or LXT as VDET clock source, which is selected in VDETCKSEL (CLK_BODCLK[0]).
     * |        |          |Note3: The input pin for VDET detect voltage is selectabe by VDETPINSEL (SYS_BODCTL[17]).
     * |[17]    |VDETPINSEL|Voltage Detector External Input Voltage Pin Selection
     * |        |          |0 = The input voltage is from VDET_P0 (PB.0).
     * |        |          |1 = The input voltage is from VDET_P1 (PB.1).
     * |        |          |Note1: If VDET_P0 is selected, multi-function pin must be selected correctly in PB0MFP (SYS_GPB_MFPL[3:0]).
     * |        |          |Note2: If VDET_P1 is selected, multi-function pin must be selected correctly in PB1MFP (SYS_GPB_MFPL[7:4]).
     * |[18]    |VDETIEN   |Voltage Detector Interrupt Enable Bit
     * |        |          |0 = VDET interrupt Disabled.
     * |        |          |1 = VDET interrupt Enabled.
     * |[19]    |VDETIF    |Voltage Detector Interrupt Flag
     * |        |          |0 = VDET does not detect any voltage draft at external pin down through or up through the voltage of Bandgap.
     * |        |          |1 = When VDET detects the external pin is dropped down through the voltage of Bandgap or the external pin is raised up through the voltage of Bandgap, this bit is set to 1 and the brown-out interrupt is requested if brown-out interrupt is enabled.
     * |        |          |Note: This bit can be cleared by software writing 1.
     * |[24]    |VDETOUT   |Voltage Detector Output Status
     * |        |          |0 = VDET output status is 0.
     * |        |          |It means the detected voltage is higher than Bandgap or VDETEN is 0.
     * |        |          |1 = VDET output status is 1.
     * |        |          |It means the detected voltage is lower than Bandgap.
     * |        |          |If the VDETEN is 0, VDET function disabled, this bit always responds 0.
     * |[27:25] |VDETDGSEL |Voltage Detector Output De-glitch Time Select (Write Protect)
     * |        |          |000 = VDET output is sampled by VDET clock.
     * |        |          |001 = 16 system clock (HCLK).
     * |        |          |010 = 32 system clock (HCLK).
     * |        |          |011 = 64 system clock (HCLK).
     * |        |          |100 = 128 system clock (HCLK).
     * |        |          |101 = 256 system clock (HCLK).
     * |        |          |110 = 512 system clock (HCLK).
     * |        |          |111 = 1024 system clock (HCLK).
     * |        |          |Note: These bits are write protected. Refer to the SYS_REGLCTL register.
     * @var SYS_T::IVSCTL
     * Offset: 0x1C  Internal Voltage Source Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |VTEMPEN   |Temperature Sensor Enable Bit
     * |        |          |This bit is used to enable/disable temperature sensor function.
     * |        |          |0 = Temperature sensor function Disabled (default).
     * |        |          |1 = Temperature sensor function Enabled.
     * |        |          |Note: After this bit is set to 1, the value of temperature sensor output can be obtained from ADC conversion result.
     * |[1]     |VBATUGEN  |VBAT Unity Gain Buffer Enable Bit
     * |        |          |This bit is used to enable/disable VBAT unity gain buffer function.
     * |        |          |0 = VBAT unity gain buffer function Disabled (default).
     * |        |          |1 = VBAT unity gain buffer function Enabled.
     * |        |          |Note: After this bit is set to 1, the value of VBAT unity gain buffer output voltage can be obtained from ADC conversion result.
     * @var SYS_T::PORCTL
     * Offset: 0x24  Power-on Reset Controller Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |POROFF    |Power-on Reset Enable Bit (Write Protect)
     * |        |          |When powered on, the POR circuit generates a reset signal to reset the whole chip function, but noise on the power may cause the POR active again.
     * |        |          |User can disable internal POR circuit to avoid unpredictable noise to cause chip reset by writing 0x5AA5 to this field.
     * |        |          |The POR function will be active again when this field is set to another value or chip is reset by other reset source, including:
     * |        |          |nRESET, Watchdog, LVR reset, BOD reset, ICE reset command and the software-chip reset function.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * @var SYS_T::VREFCTL
     * Offset: 0x28  VREF Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[4:0]   |VREFCTL   |VREF Control Bits (Write Protect)
     * |        |          |00000 = From VREF pin.
     * |        |          |00011 = VREF is internal 2.56V.
     * |        |          |00111 = VREF is internal 2.048V.
     * |        |          |01011 = VREF is internal 3.072V.
     * |        |          |01111 = VREF is internal 4.096V.
     * |        |          |10000 = VREF is from AVDD.
     * |        |          |Others = Reserved.
     * |        |          |Note: These bit are write protected. Refer to the SYS_REGLCTL register.
     * @var SYS_T::GPA_MFPL
     * Offset: 0x30  GPIOA Low Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PA0MFP    |PA.0 Multi-function Pin Selection
     * |[7:4]   |PA1MFP    |PA.1 Multi-function Pin Selection
     * |[11:8]  |PA2MFP    |PA.2 Multi-function Pin Selection
     * |[15:12] |PA3MFP    |PA.3 Multi-function Pin Selection
     * |[19:16] |PA4MFP    |PA.4 Multi-function Pin Selection
     * |[23:20] |PA5MFP    |PA.5 Multi-function Pin Selection
     * |[27:24] |PA6MFP    |PA.6 Multi-function Pin Selection
     * |[31:28] |PA7MFP    |PA.7 Multi-function Pin Selection
     * @var SYS_T::GPA_MFPH
     * Offset: 0x34  GPIOA High Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PA8MFP    |PA.8 Multi-function Pin Selection
     * |[7:4]   |PA9MFP    |PA.9 Multi-function Pin Selection
     * |[11:8]  |PA10MFP   |PA.10 Multi-function Pin Selection
     * |[15:12] |PA11MFP   |PA.11 Multi-function Pin Selection
     * |[19:16] |PA12MFP   |PA.12 Multi-function Pin Selection
     * |[23:20] |PA13MFP   |PA.13 Multi-function Pin Selection
     * |[27:24] |PA14MFP   |PA.14 Multi-function Pin Selection
     * |[31:28] |PA15MFP   |PA.15 Multi-function Pin Selection
     * @var SYS_T::GPB_MFPL
     * Offset: 0x38  GPIOB Low Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PB0MFP    |PB.0 Multi-function Pin Selection
     * |[7:4]   |PB1MFP    |PB.1 Multi-function Pin Selection
     * |[11:8]  |PB2MFP    |PB.2 Multi-function Pin Selection
     * |[15:12] |PB3MFP    |PB.3 Multi-function Pin Selection
     * |[19:16] |PB4MFP    |PB.4 Multi-function Pin Selection
     * |[23:20] |PB5MFP    |PB.5 Multi-function Pin Selection
     * |[27:24] |PB6MFP    |PB.6 Multi-function Pin Selection
     * |[31:28] |PB7MFP    |PB.7 Multi-function Pin Selection
     * @var SYS_T::GPB_MFPH
     * Offset: 0x3C  GPIOB High Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PB8MFP    |PB.8 Multi-function Pin Selection
     * |[7:4]   |PB9MFP    |PB.9 Multi-function Pin Selection
     * |[11:8]  |PB10MFP   |PB.10 Multi-function Pin Selection
     * |[15:12] |PB11MFP   |PB.11 Multi-function Pin Selection
     * |[19:16] |PB12MFP   |PB.12 Multi-function Pin Selection
     * |[23:20] |PB13MFP   |PB.13 Multi-function Pin Selection
     * |[27:24] |PB14MFP   |PB.14 Multi-function Pin Selection
     * |[31:28] |PB15MFP   |PB.15 Multi-function Pin Selection
     * @var SYS_T::GPC_MFPL
     * Offset: 0x40  GPIOC Low Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PC0MFP    |PC.0 Multi-function Pin Selection
     * |[7:4]   |PC1MFP    |PC.1 Multi-function Pin Selection
     * |[11:8]  |PC2MFP    |PC.2 Multi-function Pin Selection
     * |[15:12] |PC3MFP    |PC.3 Multi-function Pin Selection
     * |[19:16] |PC4MFP    |PC.4 Multi-function Pin Selection
     * |[23:20] |PC5MFP    |PC.5 Multi-function Pin Selection
     * |[27:24] |PC6MFP    |PC.6 Multi-function Pin Selection
     * |[31:28] |PC7MFP    |PC.7 Multi-function Pin Selection
     * @var SYS_T::GPC_MFPH
     * Offset: 0x44  GPIOC High Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PC8MFP    |PC.8 Multi-function Pin Selection
     * |[7:4]   |PC9MFP    |PC.9 Multi-function Pin Selection
     * |[11:8]  |PC10MFP   |PC.10 Multi-function Pin Selection
     * |[15:12] |PC11MFP   |PC.11 Multi-function Pin Selection
     * |[19:16] |PC12MFP   |PC.12 Multi-function Pin Selection
     * |[23:20] |PC13MFP   |PC.13 Multi-function Pin Selection
     * |[27:24] |PC14MFP   |PC.14 Multi-function Pin Selection
     * |[31:28] |PC15MFP   |PC.15 Multi-function Pin Selection
     * @var SYS_T::GPD_MFPL
     * Offset: 0x48  GPIOD Low Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PD0MFP    |PD.0 Multi-function Pin Selection
     * |[7:4]   |PD1MFP    |PD.1 Multi-function Pin Selection
     * |[11:8]  |PD2MFP    |PD.2 Multi-function Pin Selection
     * |[15:12] |PD3MFP    |PD.3 Multi-function Pin Selection
     * |[19:16] |PD4MFP    |PD.4 Multi-function Pin Selection
     * |[23:20] |PD5MFP    |PD.5 Multi-function Pin Selection
     * |[27:24] |PD6MFP    |PD.6 Multi-function Pin Selection
     * |[31:28] |PD7MFP    |PD.7 Multi-function Pin Selection
     * @var SYS_T::GPD_MFPH
     * Offset: 0x4C  GPIOD High Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PD8MFP    |PD.8 Multi-function Pin Selection
     * |[7:4]   |PD9MFP    |PD.9 Multi-function Pin Selection
     * |[11:8]  |PD10MFP   |PD.10 Multi-function Pin Selection
     * |[15:12] |PD11MFP   |PD.11 Multi-function Pin Selection
     * |[19:16] |PD12MFP   |PD.12 Multi-function Pin Selection
     * |[23:20] |PD13MFP   |PD.13 Multi-function Pin Selection
     * |[27:24] |PD14MFP   |PD.14 Multi-function Pin Selection
     * |[31:28] |PD15MFP   |PD.15 Multi-function Pin Selection
     * @var SYS_T::GPE_MFPL
     * Offset: 0x50  GPIOE Low Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PE0MFP    |PE.0 Multi-function Pin Selection
     * |[7:4]   |PE1MFP    |PE.1 Multi-function Pin Selection
     * |[11:8]  |PE2MFP    |PE.2 Multi-function Pin Selection
     * |[15:12] |PE3MFP    |PE.3 Multi-function Pin Selection
     * |[19:16] |PE4MFP    |PE.4 Multi-function Pin Selection
     * |[23:20] |PE5MFP    |PE.5 Multi-function Pin Selection
     * |[27:24] |PE6MFP    |PE.6 Multi-function Pin Selection
     * |[31:28] |PE7MFP    |PE.7 Multi-function Pin Selection
     * @var SYS_T::GPE_MFPH
     * Offset: 0x54  GPIOE High Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PE8MFP    |PE.8 Multi-function Pin Selection
     * |[7:4]   |PE9MFP    |PE.9 Multi-function Pin Selection
     * |[11:8]  |PE10MFP   |PE.10 Multi-function Pin Selection
     * |[15:12] |PE11MFP   |PE.11 Multi-function Pin Selection
     * |[19:16] |PE12MFP   |PE.12 Multi-function Pin Selection
     * |[23:20] |PE13MFP   |PE.13 Multi-function Pin Selection
     * @var SYS_T::GPF_MFPL
     * Offset: 0x58  GPIOF Low Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PF0MFP    |PF.0 Multi-function Pin Selection
     * |[7:4]   |PF1MFP    |PF.1 Multi-function Pin Selection
     * |[11:8]  |PF2MFP    |PF.2 Multi-function Pin Selection
     * |[15:12] |PF3MFP    |PF.3 Multi-function Pin Selection
     * |        |          |The default value is set by flash controller user configuration register CFGXT1(CONFIG0[27]) bit.
     * |        |          |0 = PF.3 pin is configured as GPIO pins.
     * |        |          |1 = PF.3 pin is configured as external 4~24 MHz external high speed crystal oscillator (HXT) pins.
     * |[19:16] |PF4MFP    |PF.4 Multi-function Pin Selection
     * |        |          |The default value is set by flash controller user configuration register CFGXT1(CONFIG0[27]) bit.
     * |        |          |0 = PF.4 pin is configured as GPIO pins.
     * |        |          |1 = PF.4 pin is configured as external 4~24 MHz external high speed crystal oscillator (HXT) pins.
     * |[23:20] |PF5MFP    |PF.5 Multi-function Pin Selection
     * |[27:24] |PF6MFP    |PF.6 Multi-function Pin Selection
     * |[31:28] |PF7MFP    |PF.7 Multi-function Pin Selection
     * @var SYS_T::IRCTCTL0
     * Offset: 0x80  HIRC0 Trim Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |FREQSEL   |Trim Frequency Selection
     * |        |          |This field indicates the target frequency of internal high speed RC oscillator 0 (HIRC0) auto trim.
     * |        |          |During auto trim operation, if clock error detected with CESTOPEN(SYS_IRCTCTL0[8]) is set to 1 or trim retry limitation count reached, this field will be cleared to 00 automatically.
     * |        |          |00 = Disable HIRC0 auto trim function.
     * |        |          |01 = Enable HIRC0 auto trim function and trim HIRC to 22.1184 MHz.
     * |        |          |10 = Reserved.
     * |        |          |11 = Reserved.
     * |[5:4]   |LOOPSEL   |Trim Calculation Loop Selection
     * |        |          |This field defines that trim value calculation is based on how many clocks of reference clock (32.768 kHz, LXT).
     * |        |          |00 = Trim value calculation is based on average difference in 4 clocks of reference clock.
     * |        |          |01 = Trim value calculation is based on average difference in 8 clocks of reference clock.
     * |        |          |10 = Trim value calculation is based on average difference in 16 clocks of reference clock.
     * |        |          |11 = Trim value calculation is based on average difference in 32 clocks of reference clock.
     * |        |          |Note: For example, if LOOPSEL is set as 00, auto trim circuit will calculate trim value based on the average frequency difference in 4 clocks of reference clock.
     * |[7:6]   |RETRYCNT  |Trim Value Update Limitation Count
     * |        |          |This field defines that how many times the auto trim circuit will try to update the HIRC0 trim value before the frequency of HIRC0 locked.
     * |        |          |Once the HIRC0 locked, the internal trim value update counter will be reset.
     * |        |          |If the trim value update counter reached this limitation value and frequency of HIRC0 still does not lock, the auto trim operation will be disabled and FREQSEL(SYS_IRCTCTL0[1:0]) will be cleared to 00.
     * |        |          |00 = Trim retry count limitation is 64 loops.
     * |        |          |01 = Trim retry count limitation is 128 loops.
     * |        |          |10 = Trim retry count limitation is 256 loops.
     * |        |          |11 = Trim retry count limitation is 512 loops.
     * |[8]     |CESTOPEN  |Clock Error Stop Enable Bit
     * |        |          |0 = The trim operation is keep going if clock is inaccuracy.
     * |        |          |1 = The trim operation is stopped if clock is inaccuracy.
     * |[10]    |REFCKSEL  |Reference Clock Selection
     * |        |          |0 = HIRC trim reference clock is from LXT (32.768 kHz).
     * |        |          |1 = HIRC trim reference clock is from internal USB synchronous mode.
     * @var SYS_T::IRCTIEN
     * Offset: 0x84  HIRC Trim Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1]     |TFAILIEN  |HIRC0 Trim Failure Interrupt Enable Bit
     * |        |          |This bit controls if an interrupt will be triggered while HIRC0 trim value update limitation count reached and HIRC frequency still not locked on target frequency set by FREQSEL(SYS_IRCTCTL0[1:0]).
     * |        |          |If this bit is high and TFAILIF(SYS_IRCTSTS0[1]) is set during auto trim operation, an interrupt will be triggered to notify that HIRC0 trim value update limitation count was reached.
     * |        |          |0 = Disable TFAILIF(SYS_IRCTSTS0[1]) status to trigger an interrupt to CPU.
     * |        |          |1 = Enable TFAILIF(SYS_IRCTSTS0[1]) status to trigger an interrupt to CPU.
     * |[2]     |CLKEIEN   |HIRC0 Clock Error Interrupt Enable Bit
     * |        |          |This bit controls if CPU would get an interrupt while HIRC0 clock is inaccuracy during auto trim operation.
     * |        |          |If this bit is set to1, and CLKERRIF(SYS_IRCTSTS0[2]) is set during auto trim operation, an interrupt will be triggered to notify the clock frequency is inaccuracy.
     * |        |          |0 = Disable CLKERRIF(SYS_IRCTSTS0[2]) status to trigger an interrupt to CPU.
     * |        |          |1 = Enable CLKERRIF(SYS_IRCTSTS0[2]) status to trigger an interrupt to CPU.
     * |[9]     |TFAILIEN1 |HIRC1 Trim Failure Interrupt Enable Bit
     * |        |          |This bit controls if an interrupt will be triggered while HIRC1 trim value update limitation count reached and HIRC1 frequency still not locked on target frequency set by FREQSEL(SYS_IRCTCTL1[1:0]).
     * |        |          |If this bit is high and TFAILIF(SYS_IRCTSTS[1]) is set during auto trim operation, an interrupt will be triggered to notify that HIRC1 trim value update limitation count was reached.
     * |        |          |0 = Disable TFAILIF(SYS_IRCTSTS[1]) status to trigger an interrupt to CPU.
     * |        |          |1 = Enable TFAILIF(SYS_IRCTSTS[1]) status to trigger an interrupt to CPU.
     * |[10]    |CLKEIEN1  |HIRC1 Clock Error Interrupt Enable Bit
     * |        |          |This bit controls if CPU would get an interrupt while HIRC1 clock is inaccuracy during auto trim operation.
     * |        |          |If this bit is set to1, and CLKERRIF(SYS_IRCTSTS[2]) is set during auto trim operation, an interrupt will be triggered to notify the clock frequency is inaccuracy.
     * |        |          |0 = Disable CLKERRIF(SYS_IRCTSTS[2]) status to trigger an interrupt to CPU.
     * |        |          |1 = Enable CLKERRIF(SYS_IRCTSTS[2]) status to trigger an interrupt to CPU.
     * @var SYS_T::IRCTISTS
     * Offset: 0x88  HIRC Trim Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |FREQLOCK  |HIRC Frequency Lock Status
     * |        |          |This bit indicates the HIRC0 frequency is locked.
     * |        |          |This is a status bit and doesnu2019t trigger any interrupt.
     * |        |          |0 = The internal high-speed RC oscillator 0 frequency does not lock at 22.1184 MHz yet.
     * |        |          |1 = The internal high-speed RC oscillator 0 frequency locked at 22.1184 MHz.
     * |[1]     |TFAILIF   |Trim Failure Interrupt Status
     * |        |          |This bit indicates that HIRC0 trim value update limitation count reached and the HIRC0 clock frequency still does not be locked.
     * |        |          |Once this bit is set, the auto trim operation stopped and FREQSEL(SYS_iRCTCTL0[1:0]) will be cleared to 00 by hardware automatically.
     * |        |          |If this bit is set and TFAILIEN(SYS_IRCTIEN0[1]) is high, an interrupt will be triggered to notify that HIRC0 trim value update limitation count was reached.
     * |        |          |Write 1 to clear this to 0.
     * |        |          |0 = Trim value update limitation count does not reach.
     * |        |          |1 = Trim value update limitation count reached and HIRC frequency still not locked.
     * |[2]     |CLKERRIF  |Clock Error Interrupt Status
     * |        |          |When the frequency of 32.768 kHz external low speed crystal oscillator (LXT) or 22.1184 MHz internal high speed RC oscillator 0 (HIRC0) is shift larger to unreasonable value, this bit will be set and to be an indicate that clock frequency is inaccuracy.
     * |        |          |Once this bit is set to 1, the auto trim operation stopped and FREQSEL(SYS_IRCTCL0[1:0]) will be cleared to 00 by hardware automatically if CESTOPEN(SYS_IRCTCTL0[8]) is set to 1.
     * |        |          |If this bit is set and CLKEIEN(SYS_IRCTIEN0[2]) is high, an interrupt will be triggered to notify the clock frequency is inaccuracy.
     * |        |          |Write 1 to clear this to 0.
     * |        |          |0 = Clock frequency is accuracy.
     * |        |          |1 = Clock frequency is inaccuracy.
     * |[8]     |FREQLOCK1 |HIRC1 Frequency Lock Status
     * |        |          |This bit indicates the HIRC1 frequency is locked.
     * |        |          |This is a status bit and doesn not trigger any interrupt.
     * |        |          |0 = The internal high-speed RC oscillator 1 frequency does not lock at 48 MHz yet.
     * |        |          |1 = The internal high-speed RC oscillator 1 frequency locked at 48 MHz.
     * |[9]     |TFAILIF1  |HIRC1 Trim Failure Interrupt Status
     * |        |          |This bit indicates that HIRC1 trim value update limitation count reached and the HIRC1 clock frequency still doesn not be locked
     * |        |          |Once this bit is set, the auto trim operation stopped and FREQSEL(SYS_iRCTCTL1[1:0]) will be cleared to 00 by hardware automatically.
     * |        |          |If this bit is set and TFAILIEN(SYS_IRCTIEN1[1]) is high, an interrupt will be triggered to notify that HIRC1 trim value update limitation count was reached.
     * |        |          |Write 1 to clear this to 0.
     * |        |          |0 = HIRC1 trim value update limitation count does not reach.
     * |        |          |1 = HIRC1 trim value update limitation count reached and frequency still not locked.
     * |[10]    |CLKERRIF1 |HIRC1 Clock Error Interrupt Status
     * |        |          |When the frequency of 48 MHz internal high speed RC oscillator 1 (HIRC1) is shift larger to unreasonable value, this bit will be set and to be an indicate that clock frequency is inaccuracy.
     * |        |          |Once this bit is set to 1, the auto trim operation stopped and FREQSEL(SYS_IRCTCL1[1:0]) will be cleared to 00 by hardware automatically if CESTOPEN(SYS_IRCTCTL1[8]) is set to 1.
     * |        |          |If this bit is set and CLKEIEN(SYS_IRCTIEN1[2]) is high, an interrupt will be triggered to notify the clock frequency is inaccuracy.
     * |        |          |Write 1 to clear this to 0.
     * |        |          |0 = HIRC1 Clock frequency is accuracy.
     * |        |          |1 = HIRC1 Clock frequency is inaccuracy.
     * @var SYS_T::IRCTCTL1
     * Offset: 0x90  HIRC1 Trim Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |FREQSEL   |Trim Frequency Selection
     * |        |          |This field indicates the target frequency of internal high speed RC oscillator 1 (HIRC 1) auto trim.
     * |        |          |During auto trim operation, if clock error detected with CESTOPEN(SYS_IRCTCTL1[8]) is set to 1 or trim retry limitation count reached, this field will be cleared to 00 automatically.
     * |        |          |00 = Disable HIRC1 auto trim function.
     * |        |          |01 = Reserved.
     * |        |          |10 = Enable HIRC1 auto trim function and trim HIRC to 48 MHz.
     * |        |          |11 = Reserved.
     * |[5:4]   |LOOPSEL   |Trim Calculation Loop Selection
     * |        |          |This field defines that trim value calculation is based on how many clocks of reference clock.
     * |        |          |00 = Trim value calculation is based on average difference in 4 clocks of reference clock.
     * |        |          |01 = Trim value calculation is based on average difference in 8 clocks of reference clock.
     * |        |          |10 = Trim value calculation is based on average difference in 16 clocks of reference clock.
     * |        |          |11 = Trim value calculation is based on average difference in 32 clocks of reference clock.
     * |        |          |Note: For example, if LOOPSEL is set as 00, auto trim circuit will calculate trim value based on the average frequency difference in 4 clocks of reference clock.
     * |[7:6]   |RETRYCNT  |Trim Value Update Limitation Count
     * |        |          |This field defines that how many times the auto trim circuit will try to update the HIRC1 trim value before the frequency of HIRC1 locked.
     * |        |          |Once the HIRC1 locked, the internal trim value update counter will be reset.
     * |        |          |If the trim value update counter reached this limitation value and frequency of HIRC1 still does not lock, the auto trim operation will be disabled and FREQSEL(SYS_IRCTCTL1[1:0]) will be cleared to 00.
     * |        |          |00 = Trim retry count limitation is 64 loops.
     * |        |          |01 = Trim retry count limitation is 128 loops.
     * |        |          |10 = Trim retry count limitation is 256 loops.
     * |        |          |11 = Trim retry count limitation is 512 loops.
     * |[8]     |CESTOPEN  |Clock Error Stop Enable Bit
     * |        |          |0 = The trim operation is keep going if clock is inaccuracy.
     * |        |          |1 = The trim operation is stopped if clock is inaccuracy.
     * |[10]    |REFCKSEL  |Reference Clock Selection
     * |        |          |0 = HIRC trim reference clock is from LXT (32.768 kHz).
     * |        |          |1 = HIRC trim reference clock is from internal USB synchronous mode.
     * @var SYS_T::MODCTL
     * Offset: 0xC0  Modulation Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |MODEN     |Modulation Function Enable Bit
     * |        |          |This bit enables modulation funcion by modulating with PWM channel output and UART1_TXD.
     * |        |          |0 = Modulation Function Disabled.
     * |        |          |1 = Modulation Function Enabled.
     * |[1]     |MODH      |Modulation at Data High
     * |        |          |Select modulation pulse(PWM) at UART1_TXD high or low.
     * |        |          |0 = Modulation pulse at UART1_TXD low.
     * |        |          |1 = Modulation pulse at UART1_TXD high.
     * |[6:4]   |MODPWMSEL |PWM0 Channel Select for Modulation
     * |        |          |Select the PWM0 channel to modulate with the UART1_TXD.
     * |        |          |000 = PWM0 channel 0 modulate with UART1_TXD.
     * |        |          |001 = PWM0 channel 1 modulate with UART1_TXD.
     * |        |          |010 = PWM0 channel 2 modulate with UART1_TXD.
     * |        |          |011 = PWM0 channel 3 modulete with UART1_TXD.
     * |        |          |Others = Reserved.
     * |        |          |Note: This bis is valid while MODEN (SYS_MODCTL[0]) is set to 1.
     * @var SYS_T::SRAM_BISTCTL
     * Offset: 0xD0  System SRAM BIST Test Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SRBIST    |SRAM BIST Enable Bit (Write Protect)
     * |        |          |This bit enables BIST test for SRAM located in address 0x2000_0000~0x2000_4FFF
     * |        |          |0 = System SRAM BIST Disabled.
     * |        |          |1 = System SRAM BIST Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[2]     |CRBIST    |CACHE BIST Enable Bit (Write Protect)
     * |        |          |This bit enables BIST test for CACHE RAM.
     * |        |          |0 = System CACHE BIST Disabled.
     * |        |          |1 = System CACHE BIST Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[4]     |USBBIST   |USB BIST Enable Bit (Write Protect)
     * |        |          |This bit enables BIST test for USB RAM.
     * |        |          |0 = System USB BIST Disabled.
     * |        |          |1 = System USB BIST Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * @var SYS_T::SRAM_BISTSTS
     * Offset: 0xD4  System SRAM BIST Test Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SRBISTEF  |System SRAM BIST Fail Flag
     * |        |          |0 = System SRAM BIST test pass.
     * |        |          |1 = System SRAM BIST test fail.
     * |[2]     |CRBISTEF  |CACHE SRAM BIST Fail Flag
     * |        |          |0 = System CACHE RAM BIST test pass.
     * |        |          |1 = System CACHE RAM BIST test fail.
     * |[4]     |USBBEF    |USB SRAM BIST Fail Flag
     * |        |          |0 = USB SRAM BIST test pass.
     * |        |          |1 = USB SRAM BIST test fail.
     * |[16]    |SRBEND    |SRAM BIST Test Finish
     * |        |          |0 = System SRAM BIST active.
     * |        |          |1 = System SRAM BIST finish.
     * |[18]    |CRBEND    |CACHE SRAM BIST Test Finish
     * |        |          |0 = System CACHE RAM BIST is active.
     * |        |          |1 = System CACHE RAM BIST test finish.
     * |[20]    |USBBEND   |USB SRAM BIST Test Finish
     * |        |          |0 = USB SRAM BIST is active.
     * |        |          |1 = USB SRAM BIST test finish.
     * @var SYS_T::REGLCTL
     * Offset: 0x100  Register Lock Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |REGLCTL   |Register Lock Control Code
     * |        |          |Write operation:
     * |        |          |Some registers have write-protection function.
     * |        |          |Writing these registers have to disable the protected function by writing the sequence value "59h", "16h", "88h" to this field.
     * |        |          |After this sequence is completed, the REGLCTL bit will be set to 1 and write-protection registers can be normal write.
     * |        |          |Read operation:
     * |        |          |0 = Write-protection Enabled for writing protected registers. Any write to the protected register is ignored.
     * |        |          |1 = Write-protection Disabled for writing protected registers.
     * @var SYS_T::TSOFFSET
     * Offset: 0x114  Temperature Sensor Offset Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |VTEMP     |Temperature Sensor Offset Value
     * |        |          |This field reflects temperature sensor output voltage offset at 25 Celsius degree from flash.
     */

    __I  uint32_t PDID;                  /*!< [0x0000] Part Device Identification Number Register                       */
    __IO uint32_t RSTSTS;                /*!< [0x0004] System Reset Status Register                                     */
    __IO uint32_t IPRST0;                /*!< [0x0008] Peripheral  Reset Control Register 0                             */
    __IO uint32_t IPRST1;                /*!< [0x000c] Peripheral Reset Control Register 1                              */
    __IO uint32_t IPRST2;                /*!< [0x0010] Peripheral Reset Control Register 2                              */
    __I  uint32_t RESERVE0[1];
    __IO uint32_t BODCTL;                /*!< [0x0018] Brown-out Detector Control Register                              */
    __IO uint32_t IVSCTL;                /*!< [0x001c] Internal Voltage Source Control Register                         */
    __I  uint32_t RESERVE1[1];
    __IO uint32_t PORCTL;                /*!< [0x0024] Power-on Reset Controller Register                               */
    __IO uint32_t VREFCTL;               /*!< [0x0028] VREF Control Register                                            */
    __I  uint32_t RESERVE2[1];
    __IO uint32_t GPA_MFPL;              /*!< [0x0030] GPIOA Low Byte Multiple Function Control Register                */
    __IO uint32_t GPA_MFPH;              /*!< [0x0034] GPIOA High Byte Multiple Function Control Register               */
    __IO uint32_t GPB_MFPL;              /*!< [0x0038] GPIOB Low Byte Multiple Function Control Register                */
    __IO uint32_t GPB_MFPH;              /*!< [0x003c] GPIOB High Byte Multiple Function Control Register               */
    __IO uint32_t GPC_MFPL;              /*!< [0x0040] GPIOC Low Byte Multiple Function Control Register                */
    __IO uint32_t GPC_MFPH;              /*!< [0x0044] GPIOC High Byte Multiple Function Control Register               */
    __IO uint32_t GPD_MFPL;              /*!< [0x0048] GPIOD Low Byte Multiple Function Control Register                */
    __IO uint32_t GPD_MFPH;              /*!< [0x004c] GPIOD High Byte Multiple Function Control Register               */
    __IO uint32_t GPE_MFPL;              /*!< [0x0050] GPIOE Low Byte Multiple Function Control Register                */
    __IO uint32_t GPE_MFPH;              /*!< [0x0054] GPIOE High Byte Multiple Function Control Register               */
    __IO uint32_t GPF_MFPL;              /*!< [0x0058] GPIOF Low Byte Multiple Function Control Register                */
    __I  uint32_t RESERVE3[9];
    __IO uint32_t IRCTCTL0;              /*!< [0x0080] HIRC0 Trim Control Register                                      */
    __IO uint32_t IRCTIEN;               /*!< [0x0084] HIRC Trim Interrupt Enable Register                              */
    __IO uint32_t IRCTISTS;              /*!< [0x0088] HIRC Trim Interrupt Status Register                              */
    __I  uint32_t RESERVE4[1];
    __IO uint32_t IRCTCTL1;              /*!< [0x0090] HIRC1 Trim Control Register                                      */
    __I  uint32_t RESERVE5[11];
    __IO uint32_t MODCTL;                /*!< [0x00c0] Modulation Control Register                                      */
    __I  uint32_t RESERVE6[3];
    __IO uint32_t SRAM_BISTCTL;          /*!< [0x00d0] System SRAM BIST Test Control Register                           */
    __I  uint32_t SRAM_BISTSTS;          /*!< [0x00d4] System SRAM BIST Test Status Register                            */
    __I  uint32_t RESERVE7[10];
    __IO uint32_t REGLCTL;               /*!< [0x0100] Register Lock Control Register                                   */
    __I  uint32_t RESERVE8[4];
    __I  uint32_t TSOFFSET;              /*!< [0x0114] Temperature Sensor Offset Register                               */


} SYS_T;

/**
    @addtogroup SYS_CONST SYS Bit Field Definition
    Constant Definitions for SYS Controller
@{ */

#define SYS_PDID_PDID_Pos                (0)                                               /*!< SYS_T::PDID: PDID Position             */
#define SYS_PDID_PDID_Msk                (0xfffffffful << SYS_PDID_PDID_Pos)               /*!< SYS_T::PDID: PDID Mask                 */

#define SYS_RSTSTS_PORF_Pos              (0)                                               /*!< SYS_T::RSTSTS: PORF Position           */
#define SYS_RSTSTS_PORF_Msk              (0x1ul << SYS_RSTSTS_PORF_Pos)                    /*!< SYS_T::RSTSTS: PORF Mask               */

#define SYS_RSTSTS_PINRF_Pos             (1)                                               /*!< SYS_T::RSTSTS: PINRF Position          */
#define SYS_RSTSTS_PINRF_Msk             (0x1ul << SYS_RSTSTS_PINRF_Pos)                   /*!< SYS_T::RSTSTS: PINRF Mask              */

#define SYS_RSTSTS_WDTRF_Pos             (2)                                               /*!< SYS_T::RSTSTS: WDTRF Position          */
#define SYS_RSTSTS_WDTRF_Msk             (0x1ul << SYS_RSTSTS_WDTRF_Pos)                   /*!< SYS_T::RSTSTS: WDTRF Mask              */

#define SYS_RSTSTS_LVRF_Pos              (3)                                               /*!< SYS_T::RSTSTS: LVRF Position           */
#define SYS_RSTSTS_LVRF_Msk              (0x1ul << SYS_RSTSTS_LVRF_Pos)                    /*!< SYS_T::RSTSTS: LVRF Mask               */

#define SYS_RSTSTS_BODRF_Pos             (4)                                               /*!< SYS_T::RSTSTS: BODRF Position          */
#define SYS_RSTSTS_BODRF_Msk             (0x1ul << SYS_RSTSTS_BODRF_Pos)                   /*!< SYS_T::RSTSTS: BODRF Mask              */

#define SYS_RSTSTS_MCURF_Pos             (5)                                               /*!< SYS_T::RSTSTS: MCURF Position          */
#define SYS_RSTSTS_MCURF_Msk             (0x1ul << SYS_RSTSTS_MCURF_Pos)                   /*!< SYS_T::RSTSTS: MCURF Mask              */

#define SYS_RSTSTS_CPURF_Pos             (7)                                               /*!< SYS_T::RSTSTS: CPURF Position          */
#define SYS_RSTSTS_CPURF_Msk             (0x1ul << SYS_RSTSTS_CPURF_Pos)                   /*!< SYS_T::RSTSTS: CPURF Mask              */

#define SYS_RSTSTS_CPULKRF_Pos           (8)                                               /*!< SYS_T::RSTSTS: CPULKRF Position        */
#define SYS_RSTSTS_CPULKRF_Msk           (0x1ul << SYS_RSTSTS_CPULKRF_Pos)                 /*!< SYS_T::RSTSTS: CPULKRF Mask            */

#define SYS_IPRST0_CHIPRST_Pos           (0)                                               /*!< SYS_T::IPRST0: CHIPRST Position        */
#define SYS_IPRST0_CHIPRST_Msk           (0x1ul << SYS_IPRST0_CHIPRST_Pos)                 /*!< SYS_T::IPRST0: CHIPRST Mask            */

#define SYS_IPRST0_CPURST_Pos            (1)                                               /*!< SYS_T::IPRST0: CPURST Position         */
#define SYS_IPRST0_CPURST_Msk            (0x1ul << SYS_IPRST0_CPURST_Pos)                  /*!< SYS_T::IPRST0: CPURST Mask             */

#define SYS_IPRST0_PDMARST_Pos           (2)                                               /*!< SYS_T::IPRST0: PDMARST Position        */
#define SYS_IPRST0_PDMARST_Msk           (0x1ul << SYS_IPRST0_PDMARST_Pos)                 /*!< SYS_T::IPRST0: PDMARST Mask            */

#define SYS_IPRST0_EBIRST_Pos            (3)                                               /*!< SYS_T::IPRST0: EBIRST Position         */
#define SYS_IPRST0_EBIRST_Msk            (0x1ul << SYS_IPRST0_EBIRST_Pos)                  /*!< SYS_T::IPRST0: EBIRST Mask             */

#define SYS_IPRST0_HDIVRST_Pos           (4)                                               /*!< SYS_T::IPRST0: HDIVRST Position        */
#define SYS_IPRST0_HDIVRST_Msk           (0x1ul << SYS_IPRST0_HDIVRST_Pos)                 /*!< SYS_T::IPRST0: HDIVRST Mask            */

#define SYS_IPRST0_CRCRST_Pos            (7)                                               /*!< SYS_T::IPRST0: CRCRST Position         */
#define SYS_IPRST0_CRCRST_Msk            (0x1ul << SYS_IPRST0_CRCRST_Pos)                  /*!< SYS_T::IPRST0: CRCRST Mask             */

#define SYS_IPRST1_GPIORST_Pos           (1)                                               /*!< SYS_T::IPRST1: GPIORST Position        */
#define SYS_IPRST1_GPIORST_Msk           (0x1ul << SYS_IPRST1_GPIORST_Pos)                 /*!< SYS_T::IPRST1: GPIORST Mask            */

#define SYS_IPRST1_TMR0RST_Pos           (2)                                               /*!< SYS_T::IPRST1: TMR0RST Position        */
#define SYS_IPRST1_TMR0RST_Msk           (0x1ul << SYS_IPRST1_TMR0RST_Pos)                 /*!< SYS_T::IPRST1: TMR0RST Mask            */

#define SYS_IPRST1_TMR1RST_Pos           (3)                                               /*!< SYS_T::IPRST1: TMR1RST Position        */
#define SYS_IPRST1_TMR1RST_Msk           (0x1ul << SYS_IPRST1_TMR1RST_Pos)                 /*!< SYS_T::IPRST1: TMR1RST Mask            */

#define SYS_IPRST1_TMR2RST_Pos           (4)                                               /*!< SYS_T::IPRST1: TMR2RST Position        */
#define SYS_IPRST1_TMR2RST_Msk           (0x1ul << SYS_IPRST1_TMR2RST_Pos)                 /*!< SYS_T::IPRST1: TMR2RST Mask            */

#define SYS_IPRST1_TMR3RST_Pos           (5)                                               /*!< SYS_T::IPRST1: TMR3RST Position        */
#define SYS_IPRST1_TMR3RST_Msk           (0x1ul << SYS_IPRST1_TMR3RST_Pos)                 /*!< SYS_T::IPRST1: TMR3RST Mask            */

#define SYS_IPRST1_I2C0RST_Pos           (8)                                               /*!< SYS_T::IPRST1: I2C0RST Position        */
#define SYS_IPRST1_I2C0RST_Msk           (0x1ul << SYS_IPRST1_I2C0RST_Pos)                 /*!< SYS_T::IPRST1: I2C0RST Mask            */

#define SYS_IPRST1_I2C1RST_Pos           (9)                                               /*!< SYS_T::IPRST1: I2C1RST Position        */
#define SYS_IPRST1_I2C1RST_Msk           (0x1ul << SYS_IPRST1_I2C1RST_Pos)                 /*!< SYS_T::IPRST1: I2C1RST Mask            */

#define SYS_IPRST1_SPI0RST_Pos           (12)                                              /*!< SYS_T::IPRST1: SPI0RST Position        */
#define SYS_IPRST1_SPI0RST_Msk           (0x1ul << SYS_IPRST1_SPI0RST_Pos)                 /*!< SYS_T::IPRST1: SPI0RST Mask            */

#define SYS_IPRST1_SPI1RST_Pos           (13)                                              /*!< SYS_T::IPRST1: SPI1RST Position        */
#define SYS_IPRST1_SPI1RST_Msk           (0x1ul << SYS_IPRST1_SPI1RST_Pos)                 /*!< SYS_T::IPRST1: SPI1RST Mask            */

#define SYS_IPRST1_UART0RST_Pos          (16)                                              /*!< SYS_T::IPRST1: UART0RST Position       */
#define SYS_IPRST1_UART0RST_Msk          (0x1ul << SYS_IPRST1_UART0RST_Pos)                /*!< SYS_T::IPRST1: UART0RST Mask           */

#define SYS_IPRST1_UART1RST_Pos          (17)                                              /*!< SYS_T::IPRST1: UART1RST Position       */
#define SYS_IPRST1_UART1RST_Msk          (0x1ul << SYS_IPRST1_UART1RST_Pos)                /*!< SYS_T::IPRST1: UART1RST Mask           */

#define SYS_IPRST1_UART2RST_Pos          (18)                                              /*!< SYS_T::IPRST1: UART2RST Position       */
#define SYS_IPRST1_UART2RST_Msk          (0x1ul << SYS_IPRST1_UART2RST_Pos)                /*!< SYS_T::IPRST1: UART2RST Mask           */

#define SYS_IPRST1_PWM0RST_Pos           (20)                                              /*!< SYS_T::IPRST1: PWM0RST Position        */
#define SYS_IPRST1_PWM0RST_Msk           (0x1ul << SYS_IPRST1_PWM0RST_Pos)                 /*!< SYS_T::IPRST1: PWM0RST Mask            */

#define SYS_IPRST1_PWM1RST_Pos           (21)                                              /*!< SYS_T::IPRST1: PWM1RST Position        */
#define SYS_IPRST1_PWM1RST_Msk           (0x1ul << SYS_IPRST1_PWM1RST_Pos)                 /*!< SYS_T::IPRST1: PWM1RST Mask            */

#define SYS_IPRST1_ACMP01RST_Pos         (22)                                              /*!< SYS_T::IPRST1: ACMP01RST Position      */
#define SYS_IPRST1_ACMP01RST_Msk         (0x1ul << SYS_IPRST1_ACMP01RST_Pos)               /*!< SYS_T::IPRST1: ACMP01RST Mask          */

#define SYS_IPRST1_USBDRST_Pos           (27)                                              /*!< SYS_T::IPRST1: USBDRST Position        */
#define SYS_IPRST1_USBDRST_Msk           (0x1ul << SYS_IPRST1_USBDRST_Pos)                 /*!< SYS_T::IPRST1: USBDRST Mask            */

#define SYS_IPRST1_ADCRST_Pos            (28)                                              /*!< SYS_T::IPRST1: ADCRST Position         */
#define SYS_IPRST1_ADCRST_Msk            (0x1ul << SYS_IPRST1_ADCRST_Pos)                  /*!< SYS_T::IPRST1: ADCRST Mask             */

#define SYS_IPRST2_SC0RST_Pos            (0)                                               /*!< SYS_T::IPRST2: SC0RST Position         */
#define SYS_IPRST2_SC0RST_Msk            (0x1ul << SYS_IPRST2_SC0RST_Pos)                  /*!< SYS_T::IPRST2: SC0RST Mask             */

#define SYS_IPRST2_SC1RST_Pos            (1)                                               /*!< SYS_T::IPRST2: SC1RST Position         */
#define SYS_IPRST2_SC1RST_Msk            (0x1ul << SYS_IPRST2_SC1RST_Pos)                  /*!< SYS_T::IPRST2: SC1RST Mask             */

#define SYS_IPRST2_USCI0RST_Pos          (8)                                               /*!< SYS_T::IPRST2: USCI0RST Position       */
#define SYS_IPRST2_USCI0RST_Msk          (0x1ul << SYS_IPRST2_USCI0RST_Pos)                /*!< SYS_T::IPRST2: USCI0RST Mask           */

#define SYS_IPRST2_USCI1RST_Pos          (9)                                               /*!< SYS_T::IPRST2: USCI1RST Position       */
#define SYS_IPRST2_USCI1RST_Msk          (0x1ul << SYS_IPRST2_USCI1RST_Pos)                /*!< SYS_T::IPRST2: USCI1RST Mask           */

#define SYS_IPRST2_USCI2RST_Pos          (10)                                              /*!< SYS_T::IPRST2: USCI2RST Position       */
#define SYS_IPRST2_USCI2RST_Msk          (0x1ul << SYS_IPRST2_USCI2RST_Pos)                /*!< SYS_T::IPRST2: USCI2RST Mask           */

#define SYS_BODCTL_BODEN_Pos             (0)                                               /*!< SYS_T::BODCTL: BODEN Position          */
#define SYS_BODCTL_BODEN_Msk             (0x1ul << SYS_BODCTL_BODEN_Pos)                   /*!< SYS_T::BODCTL: BODEN Mask              */

#define SYS_BODCTL_BODVL_Pos             (1)                                               /*!< SYS_T::BODCTL: BODVL Position          */
#define SYS_BODCTL_BODVL_Msk             (0x3ul << SYS_BODCTL_BODVL_Pos)                   /*!< SYS_T::BODCTL: BODVL Mask              */

#define SYS_BODCTL_BODRSTEN_Pos          (3)                                               /*!< SYS_T::BODCTL: BODRSTEN Position       */
#define SYS_BODCTL_BODRSTEN_Msk          (0x1ul << SYS_BODCTL_BODRSTEN_Pos)                /*!< SYS_T::BODCTL: BODRSTEN Mask           */

#define SYS_BODCTL_BODIF_Pos             (4)                                               /*!< SYS_T::BODCTL: BODIF Position          */
#define SYS_BODCTL_BODIF_Msk             (0x1ul << SYS_BODCTL_BODIF_Pos)                   /*!< SYS_T::BODCTL: BODIF Mask              */

#define SYS_BODCTL_BODLPM_Pos            (5)                                               /*!< SYS_T::BODCTL: BODLPM Position         */
#define SYS_BODCTL_BODLPM_Msk            (0x1ul << SYS_BODCTL_BODLPM_Pos)                  /*!< SYS_T::BODCTL: BODLPM Mask             */

#define SYS_BODCTL_BODOUT_Pos            (6)                                               /*!< SYS_T::BODCTL: BODOUT Position         */
#define SYS_BODCTL_BODOUT_Msk            (0x1ul << SYS_BODCTL_BODOUT_Pos)                  /*!< SYS_T::BODCTL: BODOUT Mask             */

#define SYS_BODCTL_LVREN_Pos             (7)                                               /*!< SYS_T::BODCTL: LVREN Position          */
#define SYS_BODCTL_LVREN_Msk             (0x1ul << SYS_BODCTL_LVREN_Pos)                   /*!< SYS_T::BODCTL: LVREN Mask              */

#define SYS_BODCTL_BODDGSEL_Pos          (8)                                               /*!< SYS_T::BODCTL: BODDGSEL Position       */
#define SYS_BODCTL_BODDGSEL_Msk          (0x7ul << SYS_BODCTL_BODDGSEL_Pos)                /*!< SYS_T::BODCTL: BODDGSEL Mask           */

#define SYS_BODCTL_LVRDGSEL_Pos          (12)                                              /*!< SYS_T::BODCTL: LVRDGSEL Position       */
#define SYS_BODCTL_LVRDGSEL_Msk          (0x7ul << SYS_BODCTL_LVRDGSEL_Pos)                /*!< SYS_T::BODCTL: LVRDGSEL Mask           */

#define SYS_BODCTL_VDETEN_Pos            (16)                                              /*!< SYS_T::BODCTL: VDETEN Position         */
#define SYS_BODCTL_VDETEN_Msk            (0x1ul << SYS_BODCTL_VDETEN_Pos)                  /*!< SYS_T::BODCTL: VDETEN Mask             */

#define SYS_BODCTL_VDETPINSEL_Pos        (17)                                              /*!< SYS_T::BODCTL: VDETPINSEL Position     */
#define SYS_BODCTL_VDETPINSEL_Msk        (0x1ul << SYS_BODCTL_VDETPINSEL_Pos)              /*!< SYS_T::BODCTL: VDETPINSEL Mask         */

#define SYS_BODCTL_VDETIEN_Pos           (18)                                              /*!< SYS_T::BODCTL: VDETIEN Position        */
#define SYS_BODCTL_VDETIEN_Msk           (0x1ul << SYS_BODCTL_VDETIEN_Pos)                 /*!< SYS_T::BODCTL: VDETIEN Mask            */

#define SYS_BODCTL_VDETIF_Pos            (19)                                              /*!< SYS_T::BODCTL: VDETIF Position         */
#define SYS_BODCTL_VDETIF_Msk            (0x1ul << SYS_BODCTL_VDETIF_Pos)                  /*!< SYS_T::BODCTL: VDETIF Mask             */

#define SYS_BODCTL_VDETOUT_Pos           (24)                                              /*!< SYS_T::BODCTL: VDETOUT Position        */
#define SYS_BODCTL_VDETOUT_Msk           (0x1ul << SYS_BODCTL_VDETOUT_Pos)                 /*!< SYS_T::BODCTL: VDETOUT Mask            */

#define SYS_BODCTL_VDETDGSEL_Pos         (25)                                              /*!< SYS_T::BODCTL: VDETDGSEL Position      */
#define SYS_BODCTL_VDETDGSEL_Msk         (0x7ul << SYS_BODCTL_VDETDGSEL_Pos)               /*!< SYS_T::BODCTL: VDETDGSEL Mask          */

#define SYS_IVSCTL_VTEMPEN_Pos           (0)                                               /*!< SYS_T::IVSCTL: VTEMPEN Position        */
#define SYS_IVSCTL_VTEMPEN_Msk           (0x1ul << SYS_IVSCTL_VTEMPEN_Pos)                 /*!< SYS_T::IVSCTL: VTEMPEN Mask            */

#define SYS_IVSCTL_VBATUGEN_Pos          (1)                                               /*!< SYS_T::IVSCTL: VBATUGEN Position       */
#define SYS_IVSCTL_VBATUGEN_Msk          (0x1ul << SYS_IVSCTL_VBATUGEN_Pos)                /*!< SYS_T::IVSCTL: VBATUGEN Mask           */

#define SYS_PORCTL_POROFF_Pos            (0)                                               /*!< SYS_T::PORCTL: POROFF Position         */
#define SYS_PORCTL_POROFF_Msk            (0xfffful << SYS_PORCTL_POROFF_Pos)               /*!< SYS_T::PORCTL: POROFF Mask             */

#define SYS_VREFCTL_VREFCTL_Pos          (0)                                               /*!< SYS_T::VREFCTL: VREFCTL Position       */
#define SYS_VREFCTL_VREFCTL_Msk          (0x1ful << SYS_VREFCTL_VREFCTL_Pos)               /*!< SYS_T::VREFCTL: VREFCTL Mask           */

#define SYS_GPA_MFPL_PA0MFP_Pos          (0)                                               /*!< SYS_T::GPA_MFPL: PA0MFP Position       */
#define SYS_GPA_MFPL_PA0MFP_Msk          (0xful << SYS_GPA_MFPL_PA0MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA0MFP Mask           */

#define SYS_GPA_MFPL_PA1MFP_Pos          (4)                                               /*!< SYS_T::GPA_MFPL: PA1MFP Position       */
#define SYS_GPA_MFPL_PA1MFP_Msk          (0xful << SYS_GPA_MFPL_PA1MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA1MFP Mask           */

#define SYS_GPA_MFPL_PA2MFP_Pos          (8)                                               /*!< SYS_T::GPA_MFPL: PA2MFP Position       */
#define SYS_GPA_MFPL_PA2MFP_Msk          (0xful << SYS_GPA_MFPL_PA2MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA2MFP Mask           */

#define SYS_GPA_MFPL_PA3MFP_Pos          (12)                                              /*!< SYS_T::GPA_MFPL: PA3MFP Position       */
#define SYS_GPA_MFPL_PA3MFP_Msk          (0xful << SYS_GPA_MFPL_PA3MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA3MFP Mask           */

#define SYS_GPA_MFPL_PA4MFP_Pos          (16)                                              /*!< SYS_T::GPA_MFPL: PA4MFP Position       */
#define SYS_GPA_MFPL_PA4MFP_Msk          (0xful << SYS_GPA_MFPL_PA4MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA4MFP Mask           */

#define SYS_GPA_MFPL_PA5MFP_Pos          (20)                                              /*!< SYS_T::GPA_MFPL: PA5MFP Position       */
#define SYS_GPA_MFPL_PA5MFP_Msk          (0xful << SYS_GPA_MFPL_PA5MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA5MFP Mask           */

#define SYS_GPA_MFPL_PA6MFP_Pos          (24)                                              /*!< SYS_T::GPA_MFPL: PA6MFP Position       */
#define SYS_GPA_MFPL_PA6MFP_Msk          (0xful << SYS_GPA_MFPL_PA6MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA6MFP Mask           */

#define SYS_GPA_MFPL_PA7MFP_Pos          (28)                                              /*!< SYS_T::GPA_MFPL: PA7MFP Position       */
#define SYS_GPA_MFPL_PA7MFP_Msk          (0xful << SYS_GPA_MFPL_PA7MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA7MFP Mask           */

#define SYS_GPA_MFPH_PA8MFP_Pos          (0)                                               /*!< SYS_T::GPA_MFPH: PA8MFP Position       */
#define SYS_GPA_MFPH_PA8MFP_Msk          (0xful << SYS_GPA_MFPH_PA8MFP_Pos)                /*!< SYS_T::GPA_MFPH: PA8MFP Mask           */

#define SYS_GPA_MFPH_PA9MFP_Pos          (4)                                               /*!< SYS_T::GPA_MFPH: PA9MFP Position       */
#define SYS_GPA_MFPH_PA9MFP_Msk          (0xful << SYS_GPA_MFPH_PA9MFP_Pos)                /*!< SYS_T::GPA_MFPH: PA9MFP Mask           */

#define SYS_GPA_MFPH_PA10MFP_Pos         (8)                                               /*!< SYS_T::GPA_MFPH: PA10MFP Position      */
#define SYS_GPA_MFPH_PA10MFP_Msk         (0xful << SYS_GPA_MFPH_PA10MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA10MFP Mask          */

#define SYS_GPA_MFPH_PA11MFP_Pos         (12)                                              /*!< SYS_T::GPA_MFPH: PA11MFP Position      */
#define SYS_GPA_MFPH_PA11MFP_Msk         (0xful << SYS_GPA_MFPH_PA11MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA11MFP Mask          */

#define SYS_GPA_MFPH_PA12MFP_Pos         (16)                                              /*!< SYS_T::GPA_MFPH: PA12MFP Position      */
#define SYS_GPA_MFPH_PA12MFP_Msk         (0xful << SYS_GPA_MFPH_PA12MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA12MFP Mask          */

#define SYS_GPA_MFPH_PA13MFP_Pos         (20)                                              /*!< SYS_T::GPA_MFPH: PA13MFP Position      */
#define SYS_GPA_MFPH_PA13MFP_Msk         (0xful << SYS_GPA_MFPH_PA13MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA13MFP Mask          */

#define SYS_GPA_MFPH_PA14MFP_Pos         (24)                                              /*!< SYS_T::GPA_MFPH: PA14MFP Position      */
#define SYS_GPA_MFPH_PA14MFP_Msk         (0xful << SYS_GPA_MFPH_PA14MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA14MFP Mask          */

#define SYS_GPA_MFPH_PA15MFP_Pos         (28)                                              /*!< SYS_T::GPA_MFPH: PA15MFP Position      */
#define SYS_GPA_MFPH_PA15MFP_Msk         (0xful << SYS_GPA_MFPH_PA15MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA15MFP Mask          */

#define SYS_GPB_MFPL_PB0MFP_Pos          (0)                                               /*!< SYS_T::GPB_MFPL: PB0MFP Position       */
#define SYS_GPB_MFPL_PB0MFP_Msk          (0xful << SYS_GPB_MFPL_PB0MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB0MFP Mask           */

#define SYS_GPB_MFPL_PB1MFP_Pos          (4)                                               /*!< SYS_T::GPB_MFPL: PB1MFP Position       */
#define SYS_GPB_MFPL_PB1MFP_Msk          (0xful << SYS_GPB_MFPL_PB1MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB1MFP Mask           */

#define SYS_GPB_MFPL_PB2MFP_Pos          (8)                                               /*!< SYS_T::GPB_MFPL: PB2MFP Position       */
#define SYS_GPB_MFPL_PB2MFP_Msk          (0xful << SYS_GPB_MFPL_PB2MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB2MFP Mask           */

#define SYS_GPB_MFPL_PB3MFP_Pos          (12)                                              /*!< SYS_T::GPB_MFPL: PB3MFP Position       */
#define SYS_GPB_MFPL_PB3MFP_Msk          (0xful << SYS_GPB_MFPL_PB3MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB3MFP Mask           */

#define SYS_GPB_MFPL_PB4MFP_Pos          (16)                                              /*!< SYS_T::GPB_MFPL: PB4MFP Position       */
#define SYS_GPB_MFPL_PB4MFP_Msk          (0xful << SYS_GPB_MFPL_PB4MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB4MFP Mask           */

#define SYS_GPB_MFPL_PB5MFP_Pos          (20)                                              /*!< SYS_T::GPB_MFPL: PB5MFP Position       */
#define SYS_GPB_MFPL_PB5MFP_Msk          (0xful << SYS_GPB_MFPL_PB5MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB5MFP Mask           */

#define SYS_GPB_MFPL_PB6MFP_Pos          (24)                                              /*!< SYS_T::GPB_MFPL: PB6MFP Position       */
#define SYS_GPB_MFPL_PB6MFP_Msk          (0xful << SYS_GPB_MFPL_PB6MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB6MFP Mask           */

#define SYS_GPB_MFPL_PB7MFP_Pos          (28)                                              /*!< SYS_T::GPB_MFPL: PB7MFP Position       */
#define SYS_GPB_MFPL_PB7MFP_Msk          (0xful << SYS_GPB_MFPL_PB7MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB7MFP Mask           */

#define SYS_GPB_MFPH_PB8MFP_Pos          (0)                                               /*!< SYS_T::GPB_MFPH: PB8MFP Position       */
#define SYS_GPB_MFPH_PB8MFP_Msk          (0xful << SYS_GPB_MFPH_PB8MFP_Pos)                /*!< SYS_T::GPB_MFPH: PB8MFP Mask           */

#define SYS_GPB_MFPH_PB9MFP_Pos          (4)                                               /*!< SYS_T::GPB_MFPH: PB9MFP Position       */
#define SYS_GPB_MFPH_PB9MFP_Msk          (0xful << SYS_GPB_MFPH_PB9MFP_Pos)                /*!< SYS_T::GPB_MFPH: PB9MFP Mask           */

#define SYS_GPB_MFPH_PB10MFP_Pos         (8)                                               /*!< SYS_T::GPB_MFPH: PB10MFP Position      */
#define SYS_GPB_MFPH_PB10MFP_Msk         (0xful << SYS_GPB_MFPH_PB10MFP_Pos)               /*!< SYS_T::GPB_MFPH: PB10MFP Mask          */

#define SYS_GPB_MFPH_PB11MFP_Pos         (12)                                              /*!< SYS_T::GPB_MFPH: PB11MFP Position      */
#define SYS_GPB_MFPH_PB11MFP_Msk         (0xful << SYS_GPB_MFPH_PB11MFP_Pos)               /*!< SYS_T::GPB_MFPH: PB11MFP Mask          */

#define SYS_GPB_MFPH_PB12MFP_Pos         (16)                                              /*!< SYS_T::GPB_MFPH: PB12MFP Position      */
#define SYS_GPB_MFPH_PB12MFP_Msk         (0xful << SYS_GPB_MFPH_PB12MFP_Pos)               /*!< SYS_T::GPB_MFPH: PB12MFP Mask          */

#define SYS_GPB_MFPH_PB13MFP_Pos         (20)                                              /*!< SYS_T::GPB_MFPH: PB13MFP Position      */
#define SYS_GPB_MFPH_PB13MFP_Msk         (0xful << SYS_GPB_MFPH_PB13MFP_Pos)               /*!< SYS_T::GPB_MFPH: PB13MFP Mask          */

#define SYS_GPB_MFPH_PB14MFP_Pos         (24)                                              /*!< SYS_T::GPB_MFPH: PB14MFP Position      */
#define SYS_GPB_MFPH_PB14MFP_Msk         (0xful << SYS_GPB_MFPH_PB14MFP_Pos)               /*!< SYS_T::GPB_MFPH: PB14MFP Mask          */

#define SYS_GPB_MFPH_PB15MFP_Pos         (28)                                              /*!< SYS_T::GPB_MFPH: PB15MFP Position      */
#define SYS_GPB_MFPH_PB15MFP_Msk         (0xful << SYS_GPB_MFPH_PB15MFP_Pos)               /*!< SYS_T::GPB_MFPH: PB15MFP Mask          */

#define SYS_GPC_MFPL_PC0MFP_Pos          (0)                                               /*!< SYS_T::GPC_MFPL: PC0MFP Position       */
#define SYS_GPC_MFPL_PC0MFP_Msk          (0xful << SYS_GPC_MFPL_PC0MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC0MFP Mask           */

#define SYS_GPC_MFPL_PC1MFP_Pos          (4)                                               /*!< SYS_T::GPC_MFPL: PC1MFP Position       */
#define SYS_GPC_MFPL_PC1MFP_Msk          (0xful << SYS_GPC_MFPL_PC1MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC1MFP Mask           */

#define SYS_GPC_MFPL_PC2MFP_Pos          (8)                                               /*!< SYS_T::GPC_MFPL: PC2MFP Position       */
#define SYS_GPC_MFPL_PC2MFP_Msk          (0xful << SYS_GPC_MFPL_PC2MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC2MFP Mask           */

#define SYS_GPC_MFPL_PC3MFP_Pos          (12)                                              /*!< SYS_T::GPC_MFPL: PC3MFP Position       */
#define SYS_GPC_MFPL_PC3MFP_Msk          (0xful << SYS_GPC_MFPL_PC3MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC3MFP Mask           */

#define SYS_GPC_MFPL_PC4MFP_Pos          (16)                                              /*!< SYS_T::GPC_MFPL: PC4MFP Position       */
#define SYS_GPC_MFPL_PC4MFP_Msk          (0xful << SYS_GPC_MFPL_PC4MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC4MFP Mask           */

#define SYS_GPC_MFPL_PC5MFP_Pos          (20)                                              /*!< SYS_T::GPC_MFPL: PC5MFP Position       */
#define SYS_GPC_MFPL_PC5MFP_Msk          (0xful << SYS_GPC_MFPL_PC5MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC5MFP Mask           */

#define SYS_GPC_MFPL_PC6MFP_Pos          (24)                                              /*!< SYS_T::GPC_MFPL: PC6MFP Position       */
#define SYS_GPC_MFPL_PC6MFP_Msk          (0xful << SYS_GPC_MFPL_PC6MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC6MFP Mask           */

#define SYS_GPC_MFPL_PC7MFP_Pos          (28)                                              /*!< SYS_T::GPC_MFPL: PC7MFP Position       */
#define SYS_GPC_MFPL_PC7MFP_Msk          (0xful << SYS_GPC_MFPL_PC7MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC7MFP Mask           */

#define SYS_GPC_MFPH_PC8MFP_Pos          (0)                                               /*!< SYS_T::GPC_MFPH: PC8MFP Position       */
#define SYS_GPC_MFPH_PC8MFP_Msk          (0xful << SYS_GPC_MFPH_PC8MFP_Pos)                /*!< SYS_T::GPC_MFPH: PC8MFP Mask           */

#define SYS_GPC_MFPH_PC9MFP_Pos          (4)                                               /*!< SYS_T::GPC_MFPH: PC9MFP Position       */
#define SYS_GPC_MFPH_PC9MFP_Msk          (0xful << SYS_GPC_MFPH_PC9MFP_Pos)                /*!< SYS_T::GPC_MFPH: PC9MFP Mask           */

#define SYS_GPC_MFPH_PC10MFP_Pos         (8)                                               /*!< SYS_T::GPC_MFPH: PC10MFP Position      */
#define SYS_GPC_MFPH_PC10MFP_Msk         (0xful << SYS_GPC_MFPH_PC10MFP_Pos)               /*!< SYS_T::GPC_MFPH: PC10MFP Mask          */

#define SYS_GPC_MFPH_PC11MFP_Pos         (12)                                              /*!< SYS_T::GPC_MFPH: PC11MFP Position      */
#define SYS_GPC_MFPH_PC11MFP_Msk         (0xful << SYS_GPC_MFPH_PC11MFP_Pos)               /*!< SYS_T::GPC_MFPH: PC11MFP Mask          */

#define SYS_GPC_MFPH_PC12MFP_Pos         (16)                                              /*!< SYS_T::GPC_MFPH: PC12MFP Position      */
#define SYS_GPC_MFPH_PC12MFP_Msk         (0xful << SYS_GPC_MFPH_PC12MFP_Pos)               /*!< SYS_T::GPC_MFPH: PC12MFP Mask          */

#define SYS_GPC_MFPH_PC13MFP_Pos         (20)                                              /*!< SYS_T::GPC_MFPH: PC13MFP Position      */
#define SYS_GPC_MFPH_PC13MFP_Msk         (0xful << SYS_GPC_MFPH_PC13MFP_Pos)               /*!< SYS_T::GPC_MFPH: PC13MFP Mask          */

#define SYS_GPC_MFPH_PC14MFP_Pos         (24)                                              /*!< SYS_T::GPC_MFPH: PC14MFP Position      */
#define SYS_GPC_MFPH_PC14MFP_Msk         (0xful << SYS_GPC_MFPH_PC14MFP_Pos)               /*!< SYS_T::GPC_MFPH: PC14MFP Mask          */

#define SYS_GPC_MFPH_PC15MFP_Pos         (28)                                              /*!< SYS_T::GPC_MFPH: PC15MFP Position      */
#define SYS_GPC_MFPH_PC15MFP_Msk         (0xful << SYS_GPC_MFPH_PC15MFP_Pos)               /*!< SYS_T::GPC_MFPH: PC15MFP Mask          */

#define SYS_GPD_MFPL_PD0MFP_Pos          (0)                                               /*!< SYS_T::GPD_MFPL: PD0MFP Position       */
#define SYS_GPD_MFPL_PD0MFP_Msk          (0xful << SYS_GPD_MFPL_PD0MFP_Pos)                /*!< SYS_T::GPD_MFPL: PD0MFP Mask           */

#define SYS_GPD_MFPL_PD1MFP_Pos          (4)                                               /*!< SYS_T::GPD_MFPL: PD1MFP Position       */
#define SYS_GPD_MFPL_PD1MFP_Msk          (0xful << SYS_GPD_MFPL_PD1MFP_Pos)                /*!< SYS_T::GPD_MFPL: PD1MFP Mask           */

#define SYS_GPD_MFPL_PD2MFP_Pos          (8)                                               /*!< SYS_T::GPD_MFPL: PD2MFP Position       */
#define SYS_GPD_MFPL_PD2MFP_Msk          (0xful << SYS_GPD_MFPL_PD2MFP_Pos)                /*!< SYS_T::GPD_MFPL: PD2MFP Mask           */

#define SYS_GPD_MFPL_PD3MFP_Pos          (12)                                              /*!< SYS_T::GPD_MFPL: PD3MFP Position       */
#define SYS_GPD_MFPL_PD3MFP_Msk          (0xful << SYS_GPD_MFPL_PD3MFP_Pos)                /*!< SYS_T::GPD_MFPL: PD3MFP Mask           */

#define SYS_GPD_MFPL_PD4MFP_Pos          (16)                                              /*!< SYS_T::GPD_MFPL: PD4MFP Position       */
#define SYS_GPD_MFPL_PD4MFP_Msk          (0xful << SYS_GPD_MFPL_PD4MFP_Pos)                /*!< SYS_T::GPD_MFPL: PD4MFP Mask           */

#define SYS_GPD_MFPL_PD5MFP_Pos          (20)                                              /*!< SYS_T::GPD_MFPL: PD5MFP Position       */
#define SYS_GPD_MFPL_PD5MFP_Msk          (0xful << SYS_GPD_MFPL_PD5MFP_Pos)                /*!< SYS_T::GPD_MFPL: PD5MFP Mask           */

#define SYS_GPD_MFPL_PD6MFP_Pos          (24)                                              /*!< SYS_T::GPD_MFPL: PD6MFP Position       */
#define SYS_GPD_MFPL_PD6MFP_Msk          (0xful << SYS_GPD_MFPL_PD6MFP_Pos)                /*!< SYS_T::GPD_MFPL: PD6MFP Mask           */

#define SYS_GPD_MFPL_PD7MFP_Pos          (28)                                              /*!< SYS_T::GPD_MFPL: PD7MFP Position       */
#define SYS_GPD_MFPL_PD7MFP_Msk          (0xful << SYS_GPD_MFPL_PD7MFP_Pos)                /*!< SYS_T::GPD_MFPL: PD7MFP Mask           */

#define SYS_GPD_MFPH_PD8MFP_Pos          (0)                                               /*!< SYS_T::GPD_MFPH: PD8MFP Position       */
#define SYS_GPD_MFPH_PD8MFP_Msk          (0xful << SYS_GPD_MFPH_PD8MFP_Pos)                /*!< SYS_T::GPD_MFPH: PD8MFP Mask           */

#define SYS_GPD_MFPH_PD9MFP_Pos          (4)                                               /*!< SYS_T::GPD_MFPH: PD9MFP Position       */
#define SYS_GPD_MFPH_PD9MFP_Msk          (0xful << SYS_GPD_MFPH_PD9MFP_Pos)                /*!< SYS_T::GPD_MFPH: PD9MFP Mask           */

#define SYS_GPD_MFPH_PD10MFP_Pos         (8)                                               /*!< SYS_T::GPD_MFPH: PD10MFP Position      */
#define SYS_GPD_MFPH_PD10MFP_Msk         (0xful << SYS_GPD_MFPH_PD10MFP_Pos)               /*!< SYS_T::GPD_MFPH: PD10MFP Mask          */

#define SYS_GPD_MFPH_PD11MFP_Pos         (12)                                              /*!< SYS_T::GPD_MFPH: PD11MFP Position      */
#define SYS_GPD_MFPH_PD11MFP_Msk         (0xful << SYS_GPD_MFPH_PD11MFP_Pos)               /*!< SYS_T::GPD_MFPH: PD11MFP Mask          */

#define SYS_GPD_MFPH_PD12MFP_Pos         (16)                                              /*!< SYS_T::GPD_MFPH: PD12MFP Position      */
#define SYS_GPD_MFPH_PD12MFP_Msk         (0xful << SYS_GPD_MFPH_PD12MFP_Pos)               /*!< SYS_T::GPD_MFPH: PD12MFP Mask          */

#define SYS_GPD_MFPH_PD13MFP_Pos         (20)                                              /*!< SYS_T::GPD_MFPH: PD13MFP Position      */
#define SYS_GPD_MFPH_PD13MFP_Msk         (0xful << SYS_GPD_MFPH_PD13MFP_Pos)               /*!< SYS_T::GPD_MFPH: PD13MFP Mask          */

#define SYS_GPD_MFPH_PD14MFP_Pos         (24)                                              /*!< SYS_T::GPD_MFPH: PD14MFP Position      */
#define SYS_GPD_MFPH_PD14MFP_Msk         (0xful << SYS_GPD_MFPH_PD14MFP_Pos)               /*!< SYS_T::GPD_MFPH: PD14MFP Mask          */

#define SYS_GPD_MFPH_PD15MFP_Pos         (28)                                              /*!< SYS_T::GPD_MFPH: PD15MFP Position      */
#define SYS_GPD_MFPH_PD15MFP_Msk         (0xful << SYS_GPD_MFPH_PD15MFP_Pos)               /*!< SYS_T::GPD_MFPH: PD15MFP Mask          */

#define SYS_GPE_MFPL_PE0MFP_Pos          (0)                                               /*!< SYS_T::GPE_MFPL: PE0MFP Position       */
#define SYS_GPE_MFPL_PE0MFP_Msk          (0xful << SYS_GPE_MFPL_PE0MFP_Pos)                /*!< SYS_T::GPE_MFPL: PE0MFP Mask           */

#define SYS_GPE_MFPL_PE1MFP_Pos          (4)                                               /*!< SYS_T::GPE_MFPL: PE1MFP Position       */
#define SYS_GPE_MFPL_PE1MFP_Msk          (0xful << SYS_GPE_MFPL_PE1MFP_Pos)                /*!< SYS_T::GPE_MFPL: PE1MFP Mask           */

#define SYS_GPE_MFPL_PE2MFP_Pos          (8)                                               /*!< SYS_T::GPE_MFPL: PE2MFP Position       */
#define SYS_GPE_MFPL_PE2MFP_Msk          (0xful << SYS_GPE_MFPL_PE2MFP_Pos)                /*!< SYS_T::GPE_MFPL: PE2MFP Mask           */

#define SYS_GPE_MFPL_PE3MFP_Pos          (12)                                              /*!< SYS_T::GPE_MFPL: PE3MFP Position       */
#define SYS_GPE_MFPL_PE3MFP_Msk          (0xful << SYS_GPE_MFPL_PE3MFP_Pos)                /*!< SYS_T::GPE_MFPL: PE3MFP Mask           */

#define SYS_GPE_MFPL_PE4MFP_Pos          (16)                                              /*!< SYS_T::GPE_MFPL: PE4MFP Position       */
#define SYS_GPE_MFPL_PE4MFP_Msk          (0xful << SYS_GPE_MFPL_PE4MFP_Pos)                /*!< SYS_T::GPE_MFPL: PE4MFP Mask           */

#define SYS_GPE_MFPL_PE5MFP_Pos          (20)                                              /*!< SYS_T::GPE_MFPL: PE5MFP Position       */
#define SYS_GPE_MFPL_PE5MFP_Msk          (0xful << SYS_GPE_MFPL_PE5MFP_Pos)                /*!< SYS_T::GPE_MFPL: PE5MFP Mask           */

#define SYS_GPE_MFPL_PE6MFP_Pos          (24)                                              /*!< SYS_T::GPE_MFPL: PE6MFP Position       */
#define SYS_GPE_MFPL_PE6MFP_Msk          (0xful << SYS_GPE_MFPL_PE6MFP_Pos)                /*!< SYS_T::GPE_MFPL: PE6MFP Mask           */

#define SYS_GPE_MFPL_PE7MFP_Pos          (28)                                              /*!< SYS_T::GPE_MFPL: PE7MFP Position       */
#define SYS_GPE_MFPL_PE7MFP_Msk          (0xful << SYS_GPE_MFPL_PE7MFP_Pos)                /*!< SYS_T::GPE_MFPL: PE7MFP Mask           */

#define SYS_GPE_MFPH_PE8MFP_Pos          (0)                                               /*!< SYS_T::GPE_MFPH: PE8MFP Position       */
#define SYS_GPE_MFPH_PE8MFP_Msk          (0xful << SYS_GPE_MFPH_PE8MFP_Pos)                /*!< SYS_T::GPE_MFPH: PE8MFP Mask           */

#define SYS_GPE_MFPH_PE9MFP_Pos          (4)                                               /*!< SYS_T::GPE_MFPH: PE9MFP Position       */
#define SYS_GPE_MFPH_PE9MFP_Msk          (0xful << SYS_GPE_MFPH_PE9MFP_Pos)                /*!< SYS_T::GPE_MFPH: PE9MFP Mask           */

#define SYS_GPE_MFPH_PE10MFP_Pos         (8)                                               /*!< SYS_T::GPE_MFPH: PE10MFP Position      */
#define SYS_GPE_MFPH_PE10MFP_Msk         (0xful << SYS_GPE_MFPH_PE10MFP_Pos)               /*!< SYS_T::GPE_MFPH: PE10MFP Mask          */

#define SYS_GPE_MFPH_PE11MFP_Pos         (12)                                              /*!< SYS_T::GPE_MFPH: PE11MFP Position      */
#define SYS_GPE_MFPH_PE11MFP_Msk         (0xful << SYS_GPE_MFPH_PE11MFP_Pos)               /*!< SYS_T::GPE_MFPH: PE11MFP Mask          */

#define SYS_GPE_MFPH_PE12MFP_Pos         (16)                                              /*!< SYS_T::GPE_MFPH: PE12MFP Position      */
#define SYS_GPE_MFPH_PE12MFP_Msk         (0xful << SYS_GPE_MFPH_PE12MFP_Pos)               /*!< SYS_T::GPE_MFPH: PE12MFP Mask          */

#define SYS_GPE_MFPH_PE13MFP_Pos         (20)                                              /*!< SYS_T::GPE_MFPH: PE13MFP Position      */
#define SYS_GPE_MFPH_PE13MFP_Msk         (0xful << SYS_GPE_MFPH_PE13MFP_Pos)               /*!< SYS_T::GPE_MFPH: PE13MFP Mask          */

#define SYS_GPF_MFPL_PF0MFP_Pos          (0)                                               /*!< SYS_T::GPF_MFPL: PF0MFP Position       */
#define SYS_GPF_MFPL_PF0MFP_Msk          (0xful << SYS_GPF_MFPL_PF0MFP_Pos)                /*!< SYS_T::GPF_MFPL: PF0MFP Mask           */

#define SYS_GPF_MFPL_PF1MFP_Pos          (4)                                               /*!< SYS_T::GPF_MFPL: PF1MFP Position       */
#define SYS_GPF_MFPL_PF1MFP_Msk          (0xful << SYS_GPF_MFPL_PF1MFP_Pos)                /*!< SYS_T::GPF_MFPL: PF1MFP Mask           */

#define SYS_GPF_MFPL_PF2MFP_Pos          (8)                                               /*!< SYS_T::GPF_MFPL: PF2MFP Position       */
#define SYS_GPF_MFPL_PF2MFP_Msk          (0xful << SYS_GPF_MFPL_PF2MFP_Pos)                /*!< SYS_T::GPF_MFPL: PF2MFP Mask           */

#define SYS_GPF_MFPL_PF3MFP_Pos          (12)                                              /*!< SYS_T::GPF_MFPL: PF3MFP Position       */
#define SYS_GPF_MFPL_PF3MFP_Msk          (0xful << SYS_GPF_MFPL_PF3MFP_Pos)                /*!< SYS_T::GPF_MFPL: PF3MFP Mask           */

#define SYS_GPF_MFPL_PF4MFP_Pos          (16)                                              /*!< SYS_T::GPF_MFPL: PF4MFP Position       */
#define SYS_GPF_MFPL_PF4MFP_Msk          (0xful << SYS_GPF_MFPL_PF4MFP_Pos)                /*!< SYS_T::GPF_MFPL: PF4MFP Mask           */

#define SYS_GPF_MFPL_PF5MFP_Pos          (20)                                              /*!< SYS_T::GPF_MFPL: PF5MFP Position       */
#define SYS_GPF_MFPL_PF5MFP_Msk          (0xful << SYS_GPF_MFPL_PF5MFP_Pos)                /*!< SYS_T::GPF_MFPL: PF5MFP Mask           */

#define SYS_GPF_MFPL_PF6MFP_Pos          (24)                                              /*!< SYS_T::GPF_MFPL: PF6MFP Position       */
#define SYS_GPF_MFPL_PF6MFP_Msk          (0xful << SYS_GPF_MFPL_PF6MFP_Pos)                /*!< SYS_T::GPF_MFPL: PF6MFP Mask           */

#define SYS_GPF_MFPL_PF7MFP_Pos          (28)                                              /*!< SYS_T::GPF_MFPL: PF7MFP Position       */
#define SYS_GPF_MFPL_PF7MFP_Msk          (0xful << SYS_GPF_MFPL_PF7MFP_Pos)                /*!< SYS_T::GPF_MFPL: PF7MFP Mask           */

#define SYS_IRCTCTL0_FREQSEL_Pos         (0)                                               /*!< SYS_T::IRCTCTL0: FREQSEL Position      */
#define SYS_IRCTCTL0_FREQSEL_Msk         (0x3ul << SYS_IRCTCTL0_FREQSEL_Pos)               /*!< SYS_T::IRCTCTL0: FREQSEL Mask          */

#define SYS_IRCTCTL0_LOOPSEL_Pos         (4)                                               /*!< SYS_T::IRCTCTL0: LOOPSEL Position      */
#define SYS_IRCTCTL0_LOOPSEL_Msk         (0x3ul << SYS_IRCTCTL0_LOOPSEL_Pos)               /*!< SYS_T::IRCTCTL0: LOOPSEL Mask          */

#define SYS_IRCTCTL0_RETRYCNT_Pos        (6)                                               /*!< SYS_T::IRCTCTL0: RETRYCNT Position     */
#define SYS_IRCTCTL0_RETRYCNT_Msk        (0x3ul << SYS_IRCTCTL0_RETRYCNT_Pos)              /*!< SYS_T::IRCTCTL0: RETRYCNT Mask         */

#define SYS_IRCTCTL0_CESTOPEN_Pos        (8)                                               /*!< SYS_T::IRCTCTL0: CESTOPEN Position     */
#define SYS_IRCTCTL0_CESTOPEN_Msk        (0x1ul << SYS_IRCTCTL0_CESTOPEN_Pos)              /*!< SYS_T::IRCTCTL0: CESTOPEN Mask         */

#define SYS_IRCTCTL0_REFCKSEL_Pos        (10)                                              /*!< SYS_T::IRCTCTL0: REFCKSEL Position     */
#define SYS_IRCTCTL0_REFCKSEL_Msk        (0x1ul << SYS_IRCTCTL0_REFCKSEL_Pos)              /*!< SYS_T::IRCTCTL0: REFCKSEL Mask         */

#define SYS_IRCTIEN_TFAILIEN_Pos         (1)                                               /*!< SYS_T::IRCTIEN: TFAILIEN Position      */
#define SYS_IRCTIEN_TFAILIEN_Msk         (0x1ul << SYS_IRCTIEN_TFAILIEN_Pos)               /*!< SYS_T::IRCTIEN: TFAILIEN Mask          */

#define SYS_IRCTIEN_CLKEIEN_Pos          (2)                                               /*!< SYS_T::IRCTIEN: CLKEIEN Position       */
#define SYS_IRCTIEN_CLKEIEN_Msk          (0x1ul << SYS_IRCTIEN_CLKEIEN_Pos)                /*!< SYS_T::IRCTIEN: CLKEIEN Mask           */

#define SYS_IRCTIEN_TFAILIEN1_Pos        (9)                                               /*!< SYS_T::IRCTIEN: TFAILIEN1 Position     */
#define SYS_IRCTIEN_TFAILIEN1_Msk        (0x1ul << SYS_IRCTIEN_TFAILIEN1_Pos)              /*!< SYS_T::IRCTIEN: TFAILIEN1 Mask         */

#define SYS_IRCTIEN_CLKEIEN1_Pos         (10)                                              /*!< SYS_T::IRCTIEN: CLKEIEN1 Position      */
#define SYS_IRCTIEN_CLKEIEN1_Msk         (0x1ul << SYS_IRCTIEN_CLKEIEN1_Pos)               /*!< SYS_T::IRCTIEN: CLKEIEN1 Mask          */

#define SYS_IRCTISTS_FREQLOCK_Pos        (0)                                               /*!< SYS_T::IRCTISTS: FREQLOCK Position     */
#define SYS_IRCTISTS_FREQLOCK_Msk        (0x1ul << SYS_IRCTISTS_FREQLOCK_Pos)              /*!< SYS_T::IRCTISTS: FREQLOCK Mask         */

#define SYS_IRCTISTS_TFAILIF_Pos         (1)                                               /*!< SYS_T::IRCTISTS: TFAILIF Position      */
#define SYS_IRCTISTS_TFAILIF_Msk         (0x1ul << SYS_IRCTISTS_TFAILIF_Pos)               /*!< SYS_T::IRCTISTS: TFAILIF Mask          */

#define SYS_IRCTISTS_CLKERRIF_Pos        (2)                                               /*!< SYS_T::IRCTISTS: CLKERRIF Position     */
#define SYS_IRCTISTS_CLKERRIF_Msk        (0x1ul << SYS_IRCTISTS_CLKERRIF_Pos)              /*!< SYS_T::IRCTISTS: CLKERRIF Mask         */

#define SYS_IRCTISTS_FREQLOCK1_Pos       (8)                                               /*!< SYS_T::IRCTISTS: FREQLOCK1 Position    */
#define SYS_IRCTISTS_FREQLOCK1_Msk       (0x1ul << SYS_IRCTISTS_FREQLOCK1_Pos)             /*!< SYS_T::IRCTISTS: FREQLOCK1 Mask        */

#define SYS_IRCTISTS_TFAILIF1_Pos        (9)                                               /*!< SYS_T::IRCTISTS: TFAILIF1 Position     */
#define SYS_IRCTISTS_TFAILIF1_Msk        (0x1ul << SYS_IRCTISTS_TFAILIF1_Pos)              /*!< SYS_T::IRCTISTS: TFAILIF1 Mask         */

#define SYS_IRCTISTS_CLKERRIF1_Pos       (10)                                              /*!< SYS_T::IRCTISTS: CLKERRIF1 Position    */
#define SYS_IRCTISTS_CLKERRIF1_Msk       (0x1ul << SYS_IRCTISTS_CLKERRIF1_Pos)             /*!< SYS_T::IRCTISTS: CLKERRIF1 Mask        */

#define SYS_IRCTCTL1_FREQSEL_Pos         (0)                                               /*!< SYS_T::IRCTCTL1: FREQSEL Position      */
#define SYS_IRCTCTL1_FREQSEL_Msk         (0x3ul << SYS_IRCTCTL1_FREQSEL_Pos)               /*!< SYS_T::IRCTCTL1: FREQSEL Mask          */

#define SYS_IRCTCTL1_LOOPSEL_Pos         (4)                                               /*!< SYS_T::IRCTCTL1: LOOPSEL Position      */
#define SYS_IRCTCTL1_LOOPSEL_Msk         (0x3ul << SYS_IRCTCTL1_LOOPSEL_Pos)               /*!< SYS_T::IRCTCTL1: LOOPSEL Mask          */

#define SYS_IRCTCTL1_RETRYCNT_Pos        (6)                                               /*!< SYS_T::IRCTCTL1: RETRYCNT Position     */
#define SYS_IRCTCTL1_RETRYCNT_Msk        (0x3ul << SYS_IRCTCTL1_RETRYCNT_Pos)              /*!< SYS_T::IRCTCTL1: RETRYCNT Mask         */

#define SYS_IRCTCTL1_CESTOPEN_Pos        (8)                                               /*!< SYS_T::IRCTCTL1: CESTOPEN Position     */
#define SYS_IRCTCTL1_CESTOPEN_Msk        (0x1ul << SYS_IRCTCTL1_CESTOPEN_Pos)              /*!< SYS_T::IRCTCTL1: CESTOPEN Mask         */

#define SYS_IRCTCTL1_REFCKSEL_Pos        (10)                                              /*!< SYS_