/**************************************************************************//**
 * @file     gpio.c
 * @version  V3.00
 * $Revision: 1 $
 * $Date: 14/10/01 10:31a $
 * @brief    NUC029 series GPIO driver source file
 *
 * @note
 * @copyright SPDX-License-Identifier: Apache-2.0
 * @copyright Copyright (C) 2014 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include "NUC029xAN.h"


/** @addtogroup NUC029_Device_Driver NUC029 Device Driver
  @{
*/

/** @addtogroup NUC029_GPIO_Driver GPIO Driver
  @{
*/

/** @addtogroup NUC029_GPIO_EXPORTED_FUNCTIONS GPIO Exported Functions
  @{
*/

/**
 * @brief       Set GPIO operation mode
 *
 * @param[in]   port        GPIO port. It could be P0, P1, P2, P3 or P4.
 * @param[in]   u32PinMask  The single or multiple pins of specified GPIO port. It could be BIT0 ~ BIT7.
 * @param[in]   u32Mode     Operation mode. It could be \n
 *                          - \ref GPIO_PMD_INPUT
 *                          - \ref GPIO_PMD_OUTPUT
 *                          - \ref GPIO_PMD_OPEN_DRAIN
 *                          - \ref GPIO_PMD_QUASI
 *
 * @return      None
 *
 * @details     This function is used to set specified GPIO operation mode.
 */
void GPIO_SetMode(GPIO_T *port, uint32_t u32PinMask, uint32_t u32Mode)
{
    uint32_t i;

    for(i = 0; i < GPIO_PIN_MAX; i++)
    {
        if(u32PinMask & (1 << i))
        {
            port->PMD = (port->PMD & ~(0x3 << (i << 1))) | (u32Mode << (i << 1));
        }
    }
}

/**
 * @brief       Enable GPIO interrupt
 *
 * @param[in]   port            GPIO port. It could be P0, P1, P2, P3 or P4.
 * @param[in]   u32Pin          The pin of specified GPIO port. It could be 0 ~ 7.
 * @param[in]   u32IntAttribs   The interrupt attribute of specified GPIO pin. It could be \n
 *                              - \ref GPIO_INT_RISING
 *                              - \ref GPIO_INT_FALLING
 *                              - \ref GPIO_INT_BOTH_EDGE
 *                              - \ref GPIO_INT_HIGH
 *                              - \ref GPIO_INT_LOW
 *
 * @return      None
 *
 * @details     This function is used to enable specified GPIO pin interrupt.
 */
void GPIO_EnableInt(GPIO_T *port, uint32_t u32Pin, uint32_t u32IntAttribs)
{
    /* Configure interrupt mode of specified pin */
    port->IMD = (port->IMD & ~(1ul << u32Pin)) | (((u32IntAttribs >> 24) & 0xFFUL) << u32Pin);

    /* Enable interrupt function of specified pin */
    port->IEN = (port->IEN & ~(0x00010001ul << u32Pin)) | ((u32IntAttribs & 0xFFFFFFUL) << u32Pin);
}


/**
 * @brief       Disable GPIO interrupt
 *
 * @param[in]   port        GPIO port. It could be P0, P1, P2, P3 or P4.
 * @param[in]   u32Pin      The pin of specified GPIO port. It could be 0 ~ 7.
 *
 * @return      None
 *
 * @details     This function is used to disable specified GPIO pin interrupt.
 */
void GPIO_DisableInt(GPIO_T *port, uint32_t u32Pin)
{
    /* Configure interrupt mode of specified pin */
    port->IMD &= ~(1UL << u32Pin);

    /* Disable interrupt function of specified pin */
    port->IEN &= ~((0x00010001UL) << u32Pin);
}


/*@}*/ /* end of group NUC029_GPIO_EXPORTED_FUNCTIONS */

/*@}*/ /* end of group NUC029_GPIO_Driver */

/*@}*/ /* end of group NUC029_Device_Driver */

/*** (C) COPYRIGHT 2013 Nuvoton Technology Corp. ***/
