/**************************************************************************//**
 * @file     Nano103.h
 * @version  V1.00
 * $Revision: 45 $
 * $Date: 16/03/29 5:27p $
 * @brief    NANO103  peripheral access layer header file.
 *           This file contains all the peripheral register's definitions,
 *           bits definitions and memory mapping for NuMicro NANO103 MCU.
 *
 * @note
 * Copyright (C) 2018 Nuvoton Technology Corp. All rights reserved.
 *****************************************************************************/
/**
   \mainpage NuMicro NANO103 Driver Reference Guide
   *
   * <b>Introduction</b>
   *
   * This user manual describes the usage of NANO103 Series MCU device driver
   *
   * <b>Disclaimer</b>
   *
   * The Software is furnished "AS IS", without warranty as to performance or results, and
   * the entire risk as to performance or results is assumed by YOU. Nuvoton disclaims all
   * warranties, express, implied or otherwise, with regard to the Software, its use, or
   * operation, including without limitation any and all warranties of merchantability, fitness
   * for a particular purpose, and non-infringement of intellectual property rights.
   *
   * <b>Important Notice</b>
   *
   * Nuvoton Products are neither intended nor warranted for usage in systems or equipment,
   * any malfunction or failure of which may cause loss of human life, bodily injury or severe
   * property damage. Such applications are deemed, "Insecure Usage".
   *
   * Insecure usage includes, but is not limited to: equipment for surgical implementation,
   * atomic energy control instruments, airplane or spaceship instruments, the control or
   * operation of dynamic, brake or safety systems designed for vehicular use, traffic signal
   * instruments, all types of safety devices, and other applications intended to support or
   * sustain life.
   *
   * All Insecure Usage shall be made at customer's risk, and in the event that third parties
   * lay claims to Nuvoton as a result of customer's Insecure Usage, customer shall indemnify
   * the damages and liabilities thus incurred by Nuvoton.
   *
   * Please note that all data and specifications are subject to change without notice. All the
   * trademarks of products and companies mentioned in this datasheet belong to their respective
   * owners.
   *
   * <b>Copyright Notice</b>
   *
   * Copyright (C) 2015~2020 Nuvoton Technology Corp. All rights reserved.
   */
#ifndef __NANO103_H__
#define __NANO103_H__

#ifdef __cplusplus
extern "C" {
#endif

/** @addtogroup NANO103_Definitions NANO103 Definitions
  This file defines all structures and symbols for NANO103:
    - interrupt numbers
    - registers and bit fields
    - peripheral base address
    - peripheral ID
    - Peripheral definitions
  @{
*/

/******************************************************************************/
/*                Processor and Core Peripherals                              */
/******************************************************************************/
/** @addtogroup NANO103_CMSIS Device CMSIS Definitions
  Configuration of the Cortex-M0 Processor and Core Peripherals
  @{
*/

/**
 * @details  Interrupt Number Definition. The maximum of 32 Specific Interrupts are possible.
 */
typedef enum IRQn
{
    /******  Cortex-M0 Processor Exceptions Numbers *****************************************/

    NonMaskableInt_IRQn   = -14,    /*!< 2 Non Maskable Interrupt                           */
    HardFault_IRQn        = -13,    /*!< 3 Cortex-M0 Hard Fault Interrupt                   */
    SVCall_IRQn           = -5,     /*!< 11 Cortex-M0 SV Call Interrupt                     */
    PendSV_IRQn           = -2,     /*!< 14 Cortex-M0 Pend SV Interrupt                     */
    SysTick_IRQn          = -1,     /*!< 15 Cortex-M0 System Tick Interrupt                 */

    /******  NANO103 specific Interrupt Numbers ***********************************************/
    BOD_IRQn              = 0,      /*!< Brownout low voltage detected interrupt                   */
    WDT_IRQn              = 1,      /*!< Watch Dog Timer interrupt                                 */
    EINT0_IRQn            = 2,      /*!< External signal interrupt from PB.14 pin                  */
    EINT1_IRQn            = 3,      /*!< External signal interrupt from PB.15 pin                  */
    GPABC_IRQn            = 4,      /*!< External signal interrupt from PA[15:0]/PB[13:0]/PC[15:0] */
    GPDEF_IRQn            = 5,      /*!< External interrupt from PD[15:0]/PE[15:0]/PF[15:0]        */
    PWM0_IRQn             = 6,      /*!< PWM 0 interrupt                                           */
    TMR0_IRQn             = 8,      /*!< Timer 0 interrupt                                         */
    TMR1_IRQn             = 9,      /*!< Timer 1 interrupt                                         */
    TMR2_IRQn             = 10,     /*!< Timer 2 interrupt                                         */
    TMR3_IRQn             = 11,     /*!< Timer 3 interrupt                                         */
    UART0_IRQn            = 12,     /*!< UART0 interrupt                                           */
    UART1_IRQn            = 13,     /*!< UART1 interrupt                                           */
    SPI0_IRQn             = 14,     /*!< SPI0 interrupt                                            */
    SPI1_IRQn             = 15,     /*!< SPI1 interrupt                                            */
    SPI2_IRQn             = 16,     /*!< SPI2 interrupt                                            */
    HIRC_IRQn             = 17,     /*!< HIRC interrupt                                            */
    I2C0_IRQn             = 18,     /*!< I2C0 interrupt                                            */
    I2C1_IRQn             = 19,     /*!< I2C1 interrupt                                            */
    SC0_IRQn              = 21,     /*!< Smart Card 0 interrupt                                    */
    SC1_IRQn              = 22,     /*!< Smart Card 1 interrupt                                    */
    CKSD_IRQn             = 24,     /*!< Clock stop detect interrupt                               */
    PDMA_IRQn             = 26,     /*!< PDMA interrupt                                            */
    SPI3_IRQn             = 27,     /*!< SPI3 interrupt                                            */
    PDWU_IRQn             = 28,     /*!< Power Down Wake up interrupt                              */
    ADC_IRQn              = 29,     /*!< ADC interrupt                                             */
    ACMP_IRQn             = 30,     /*!< Analog Comparator interrupt                               */
    RTC_IRQn              = 31      /*!< Real time clock interrupt                                 */
} IRQn_Type;


/*
 * ==========================================================================
 * ----------- Processor and Core Peripheral Section ------------------------
 * ==========================================================================
 */

/* Configuration of the Cortex-M0 Processor and Core Peripherals */
#define __CM0_REV                0x0201    /*!< Core Revision r2p1                               */
#define __NVIC_PRIO_BITS         2         /*!< Number of Bits used for Priority Levels          */
#define __Vendor_SysTickConfig   0         /*!< Set to 1 if different SysTick Config is used     */
#define __MPU_PRESENT            0         /*!< MPU present or not                               */
#define __FPU_PRESENT            0         /*!< FPU present or not                               */

/*@}*/ /* end of group NANO103_CMSIS */


#include "core_cm0.h"                       /* Cortex-M0 processor and core peripherals           */
#include "system_Nano103.h"           /* NANO103 Series System include file             */
#include <stdint.h>

/******************************************************************************/
/*                Device Specific Peripheral registers structures             */
/******************************************************************************/
/** @addtogroup NANO103_Peripherals NANO103 Peripherals
  NANO103 Device Specific Peripheral registers structures
  @{
*/

#if defined ( __CC_ARM  )
#pragma anon_unions
#endif


/*---------------------- INT Controller -------------------------*/
/**
    @addtogroup INT Interrupt Controller (INTR)
    Memory Mapped Structure for INT Controller
@{ */

typedef struct
{


    /**
     * @var INT_T::IRQ0_SRC
     * Offset: 0x00  MCU IRQ0 (BOD_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ1_SRC
     * Offset: 0x04  MCU IRQ1 (WDT_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ2_SRC
     * Offset: 0x08  MCU IRQ2 (EINT0) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ3_SRC
     * Offset: 0x0C  MCU IRQ3 (EINT1) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ4_SRC
     * Offset: 0x10  MCU IRQ4 (GPABC_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ5_SRC
     * Offset: 0x14  MCU IRQ5 (GPDEF_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ6_SRC
     * Offset: 0x18  MCU IRQ6 (PWM0_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ7_SRC
     * Offset: 0x1C  Reserved
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ8_SRC
     * Offset: 0x20  MCU IRQ8 (TMR0_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ9_SRC
     * Offset: 0x24  MCU IRQ9 (TMR1_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ10_SRC
     * Offset: 0x28  MCU IRQ10 (TMR2_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ11_SRC
     * Offset: 0x2C  MCU IRQ11 (TMR3_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ12_SRC
     * Offset: 0x30  MCU IRQ12 (UART0_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ13_SRC
     * Offset: 0x34  MCU IRQ13 (UART1_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ14_SRC
     * Offset: 0x38  MCU IRQ14 (SPI0_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ15_SRC
     * Offset: 0x3C  MCU IRQ15 (SPI1_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ16_SRC
     * Offset: 0x40  MCU IRQ16 (SPI2_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ17_SRC
     * Offset: 0x44  MCU IRQ17 (IRC_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ18_SRC
     * Offset: 0x48  MCU IRQ18 (I2C0_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ19_SRC
     * Offset: 0x4C  MCU IRQ19 (I2C1_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ20_SRC
     * Offset: 0x50  Reserved
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ21_SRC
     * Offset: 0x54  MCU IRQ21 (SC0_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ22_SRC
     * Offset: 0x58  MCU IRQ22 (SC1_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ23_SRC
     * Offset: 0x5C  Reserved
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ24_SRC
     * Offset: 0x60  MCU IRQ24 (CKSD_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ25_SRC
     * Offset: 0x64  Reserved
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ26_SRC
     * Offset: 0x68  MCU IRQ26 (DMA_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ27_SRC
     * Offset: 0x6C  MCU IRQ27 (SPI3_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ28_SRC
     * Offset: 0x70  MCU IRQ28 (PDWU_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ29_SRC
     * Offset: 0x74  MCU IRQ29 (ADC_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ30_SRC
     * Offset: 0x78  MCU IRQ30 (ACMP_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::IRQ31_SRC
     * Offset: 0x7C  MCU IRQ31 (RTC_INT) interrupt source identify
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |INT_SRC   |Interrupt Source
     * |        |          |Define the interrupt sources for interrupt event.
     * @var INT_T::NMI_SEL
     * Offset: 0x80  NMI Source Interrupt Select Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[4:0]   |NMI_SEL   |NMI Interrupt to Cortex-M0 Can Be Selected From One of the Interrupt[31:0]
     * |        |          |The NMI_SEL bit[4:0] is used to select the NMI interrupt source
     * @var INT_T::MCU_IRQ
     * Offset: 0x84  MCU Interrupt Request Source Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |MCU_IRQ   |MCU IRQ Source Bits
     * |        |          |The MCU_IRQ collects all the interrupts from the peripherals and generates the synchronous interrupt to MCU Cortex-M0
     * |        |          |There are two modes to generate interrupt to Cortex-M0, the normal mode.
     * |        |          |The MCU_IRQ collects all interrupts from each peripheral and synchronizes them and then interrupts the Cortex-M0.
     * |        |          |When the MCU_IRQ[n] is "0", setting MCU_IRQ[n] "1" will generate an interrupt to Cortex-M0 NVIC[n].
     * |        |          |When the MCU_IRQ[n] is "1" (means an interrupt is asserted), setting the MCU_bit[n] will clear the interrupt
     * |        |          |Set MCU_IRQ[n] u201C0u201D: no any effect
     */
    __I  uint32_t IRQ0_SRC;              /*!< [0x0000] MCU IRQ0 (BOD_INT) interrupt source identify                     */
    __I  uint32_t IRQ1_SRC;              /*!< [0x0004] MCU IRQ1 (WDT_INT) interrupt source identify                     */
    __I  uint32_t IRQ2_SRC;              /*!< [0x0008] MCU IRQ2 (EINT0) interrupt source identify                       */
    __I  uint32_t IRQ3_SRC;              /*!< [0x000c] MCU IRQ3 (EINT1) interrupt source identify                       */
    __I  uint32_t IRQ4_SRC;              /*!< [0x0010] MCU IRQ4 (GPABC_INT) interrupt source identify                   */
    __I  uint32_t IRQ5_SRC;              /*!< [0x0014] MCU IRQ5 (GPDEF_INT) interrupt source identify                   */
    __I  uint32_t IRQ6_SRC;              /*!< [0x0018] MCU IRQ6 (PWM0_INT) interrupt source identify                    */
    __I  uint32_t IRQ7_SRC;              /*!< [0x001c] Reserved                                                         */
    __I  uint32_t IRQ8_SRC;              /*!< [0x0020] MCU IRQ8 (TMR0_INT) interrupt source identify                    */
    __I  uint32_t IRQ9_SRC;              /*!< [0x0024] MCU IRQ9 (TMR1_INT) interrupt source identify                    */
    __I  uint32_t IRQ10_SRC;             /*!< [0x0028] MCU IRQ10 (TMR2_INT) interrupt source identify                   */
    __I  uint32_t IRQ11_SRC;             /*!< [0x002c] MCU IRQ11 (TMR3_INT) interrupt source identify                   */
    __I  uint32_t IRQ12_SRC;             /*!< [0x0030] MCU IRQ12 (UART0_INT) interrupt source identify                  */
    __I  uint32_t IRQ13_SRC;             /*!< [0x0034] MCU IRQ13 (UART1_INT) interrupt source identify                  */
    __I  uint32_t IRQ14_SRC;             /*!< [0x0038] MCU IRQ14 (SPI0_INT) interrupt source identify                   */
    __I  uint32_t IRQ15_SRC;             /*!< [0x003c] MCU IRQ15 (SPI1_INT) interrupt source identify                   */
    __I  uint32_t IRQ16_SRC;             /*!< [0x0040] MCU IRQ16 (SPI2_INT) interrupt source identify                   */
    __I  uint32_t IRQ17_SRC;             /*!< [0x0044] MCU IRQ17 (IRC_INT) interrupt source identify                    */
    __I  uint32_t IRQ18_SRC;             /*!< [0x0048] MCU IRQ18 (I2C0_INT) interrupt source identify                   */
    __I  uint32_t IRQ19_SRC;             /*!< [0x004c] MCU IRQ19 (I2C1_INT) interrupt source identify                   */
    __I  uint32_t IRQ20_SRC;             /*!< [0x0050] Reserved                                                         */
    __I  uint32_t IRQ21_SRC;             /*!< [0x0054] MCU IRQ21 (SC0_INT) interrupt source identify                    */
    __I  uint32_t IRQ22_SRC;             /*!< [0x0058] MCU IRQ22 (SC1_INT) interrupt source identify                    */
    __I  uint32_t IRQ23_SRC;             /*!< [0x005c] Reserved                                                         */
    __I  uint32_t IRQ24_SRC;             /*!< [0x0060] MCU IRQ24 (CKSD_INT) interrupt source identify                   */
    __I  uint32_t IRQ25_SRC;             /*!< [0x0064] Reserved                                                         */
    __I  uint32_t IRQ26_SRC;             /*!< [0x0068] MCU IRQ26 (DMA_INT) interrupt source identify                    */
    __I  uint32_t IRQ27_SRC;             /*!< [0x006c] MCU IRQ27 (SPI3_INT) interrupt source identify                   */
    __I  uint32_t IRQ28_SRC;             /*!< [0x0070] MCU IRQ28 (PDWU_INT) interrupt source identify                   */
    __I  uint32_t IRQ29_SRC;             /*!< [0x0074] MCU IRQ29 (ADC_INT) interrupt source identify                    */
    __I  uint32_t IRQ30_SRC;             /*!< [0x0078] MCU IRQ30 (ACMP_INT) interrupt source identify                   */
    __I  uint32_t IRQ31_SRC;             /*!< [0x007c] MCU IRQ31 (RTC_INT) interrupt source identify                    */
    __IO uint32_t NMI_SEL;               /*!< [0x0080] NMI Source Interrupt Select Control Register                     */
    __IO uint32_t MCU_IRQ;               /*!< [0x0084] MCU Interrupt Request Source Register                            */


} INT_T;

/**
    @addtogroup INT_CONST INT Bit Field Definition
    Constant Definitions for INT Controller
@{ */

#define INT_IRQ0_SRC_INT_SRC_Pos              (0)                                               /*!< INT_T::IRQ0_SRC: INT_SRC Position           */
#define INT_IRQ0_SRC_INT_SRC_Msk              (0xful << INT_IRQ0_SRC_INT_SRC_Pos)               /*!< INT_T::IRQ0_SRC: INT_SRC Mask               */

#define INT_IRQ1_SRC_INT_SRC_Pos              (0)                                               /*!< INT_T::IRQ1_SRC: INT_SRC Position           */
#define INT_IRQ1_SRC_INT_SRC_Msk              (0xful << INT_IRQ1_SRC_INT_SRC_Pos)               /*!< INT_T::IRQ1_SRC: INT_SRC Mask               */

#define INT_IRQ2_SRC_INT_SRC_Pos              (0)                                               /*!< INT_T::IRQ2_SRC: INT_SRC Position           */
#define INT_IRQ2_SRC_INT_SRC_Msk              (0xful << INT_IRQ2_SRC_INT_SRC_Pos)               /*!< INT_T::IRQ2_SRC: INT_SRC Mask               */

#define INT_IRQ3_SRC_INT_SRC_Pos              (0)                                               /*!< INT_T::IRQ3_SRC: INT_SRC Position           */
#define INT_IRQ3_SRC_INT_SRC_Msk              (0xful << INT_IRQ3_SRC_INT_SRC_Pos)               /*!< INT_T::IRQ3_SRC: INT_SRC Mask               */

#define INT_IRQ4_SRC_INT_SRC_Pos              (0)                                               /*!< INT_T::IRQ4_SRC: INT_SRC Position           */
#define INT_IRQ4_SRC_INT_SRC_Msk              (0xful << INT_IRQ4_SRC_INT_SRC_Pos)               /*!< INT_T::IRQ4_SRC: INT_SRC Mask               */

#define INT_IRQ5_SRC_INT_SRC_Pos              (0)                                               /*!< INT_T::IRQ5_SRC: INT_SRC Position           */
#define INT_IRQ5_SRC_INT_SRC_Msk              (0xful << INT_IRQ5_SRC_INT_SRC_Pos)               /*!< INT_T::IRQ5_SRC: INT_SRC Mask               */

#define INT_IRQ6_SRC_INT_SRC_Pos              (0)                                               /*!< INT_T::IRQ6_SRC: INT_SRC Position           */
#define INT_IRQ6_SRC_INT_SRC_Msk              (0xful << INT_IRQ6_SRC6_INT_SRC_Pos)              /*!< INT_T::IRQ6_SRC: INT_SRC Mask               */

#define INT_IRQ7_SRC_INT_SRC_Pos              (0)                                               /*!< INT_T::IRQ7_SRC: INT_SRC Position           */
#define INT_IRQ7_SRC_INT_SRC_Msk              (0xful << INT_IRQ7_SRC_INT_SRC_Pos)               /*!< INT_T::IRQ7_SRC: INT_SRC Mask               */

#define INT_IRQ8_SRC_INT_SRC_Pos              (0)                                               /*!< INT_T::IRQ8_SRC: INT_SRC Position           */
#define INT_IRQ8_SRC_INT_SRC_Msk              (0xful << INT_IRQ8_SRC_INT_SRC_Pos)               /*!< INT_T::IRQ8_SRC: INT_SRC Mask               */

#define INT_IRQ9_SRC_INT_SRC_Pos              (0)                                               /*!< INT_T::IRQ9_SRC: INT_SRC Position           */
#define INT_IRQ9_SRC_INT_SRC_Msk              (0xful << INT_IRQ9_SRC_INT_SRC_Pos)               /*!< INT_T::IRQ9_SRC: INT_SRC Mask               */

#define INT_IRQ10_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ10_SRC: INT_SRC Position           */
#define INT_IRQ10_SRC_INT_SRC_Msk             (0xful << INT_IRQ10_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ10_SRC: INT_SRC Mask               */

#define INT_IRQ11_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ11_SRC: INT_SRC Position           */
#define INT_IRQ11_SRC_INT_SRC_Msk             (0xful << INT_IRQ11_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ11_SRC: INT_SRC Mask               */

#define INT_IRQ12_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ12_SRC: INT_SRC Position           */
#define INT_IRQ12_SRC_INT_SRC_Msk             (0xful << INT_IRQ12_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ12_SRC: INT_SRC Mask               */

#define INT_IRQ13_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ13_SRC: INT_SRC Position           */
#define INT_IRQ13_SRC_INT_SRC_Msk             (0xful << INT_IRQ13_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ13_SRC: INT_SRC Mask               */

#define INT_IRQ14_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ14_SRC: INT_SRC Position           */
#define INT_IRQ14_SRC_INT_SRC_Msk             (0xful << INT_IRQ14_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ14_SRC: INT_SRC Mask               */

#define INT_IRQ15_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ15_SRC: INT_SRC Position           */
#define INT_IRQ15_SRC_INT_SRC_Msk             (0xful << INT_IRQ15_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ15_SRC: INT_SRC Mask               */

#define INT_IRQ16_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ16_SRC: INT_SRC Position           */
#define INT_IRQ16_SRC_INT_SRC_Msk             (0xful << INT_IRQ16_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ16_SRC: INT_SRC Mask               */

#define INT_IRQ17_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ17_SRC: INT_SRC Position           */
#define INT_IRQ17_SRC_INT_SRC_Msk             (0xful << INT_IRQ17_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ17_SRC: INT_SRC Mask               */

#define INT_IRQ18_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ18_SRC: INT_SRC Position           */
#define INT_IRQ18_SRC_INT_SRC_Msk             (0xful << INT_IRQ18_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ18_SRC: INT_SRC Mask               */

#define INT_IRQ19_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ19_SRC: INT_SRC Position           */
#define INT_IRQ19_SRC_INT_SRC_Msk             (0xful << INT_IRQ19_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ19_SRC: INT_SRC Mask               */

#define INT_IRQ20_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ20_SRC: INT_SRC Position           */
#define INT_IRQ20_SRC_INT_SRC_Msk             (0xful << INT_IRQ20_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ20_SRC: INT_SRC Mask               */

#define INT_IRQ21_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ21_SRC: INT_SRC Position           */
#define INT_IRQ21_SRC_INT_SRC_Msk             (0xful << INT_IRQ21_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ21_SRC: INT_SRC Mask               */

#define INT_IRQ22_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ22_SRC: INT_SRC Position           */
#define INT_IRQ22_SRC_INT_SRC_Msk             (0xful << INT_IRQ22_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ22_SRC: INT_SRC Mask               */

#define INT_IRQ23_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ23_SRC: INT_SRC Position           */
#define INT_IRQ23_SRC_INT_SRC_Msk             (0xful << INT_IRQ23_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ23_SRC: INT_SRC Mask               */

#define INT_IRQ24_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ24_SRC: INT_SRC Position           */
#define INT_IRQ24_SRC_INT_SRC_Msk             (0xful << INT_IRQ24_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ24_SRC: INT_SRC Mask               */

#define INT_IRQ25_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ25_SRC: INT_SRC Position           */
#define INT_IRQ25_SRC_INT_SRC_Msk             (0xful << INT_IRQ25_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ25_SRC: INT_SRC Mask               */

#define INT_IRQ26_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ26_SRC: INT_SRC Position           */
#define INT_IRQ26_SRC_INT_SRC_Msk             (0xful << INT_IRQ26_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ26_SRC: INT_SRC Mask               */

#define INT_IRQ27_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ27_SRC: INT_SRC Position           */
#define INT_IRQ27_SRC_INT_SRC_Msk             (0xful << INT_IRQ27_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ27_SRC: INT_SRC Mask               */

#define INT_IRQ28_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ28_SRC: INT_SRC Position           */
#define INT_IRQ28_SRC_INT_SRC_Msk             (0xful << INT_IRQ28_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ28_SRC: INT_SRC Mask               */

#define INT_IRQ29_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ29_SRC: INT_SRC Position           */
#define INT_IRQ29_SRC_INT_SRC_Msk             (0xful << INT_IRQ29_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ29_SRC: INT_SRC Mask               */

#define INT_IRQ30_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ30_SRC: INT_SRC Position           */
#define INT_IRQ30_SRC_INT_SRC_Msk             (0xful << INT_IRQ30_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ30_SRC: INT_SRC Mask               */

#define INT_IRQ31_SRC_INT_SRC_Pos             (0)                                               /*!< INT_T::IRQ31_SRC: INT_SRC Position           */
#define INT_IRQ31_SRC_INT_SRC_Msk             (0xful << INT_IRQ31_SRC_INT_SRC_Pos)              /*!< INT_T::IRQ31_SRC: INT_SRC Mask               */

#define INT_NMI_SEL_NMI_SEL_Pos               (0)                                               /*!< INT_T::NMI_SEL: NMI_SEL Position           */
#define INT_NMI_SEL_NMI_SEL_Msk               (0x1ful << INT_NMI_SEL_NMI_SEL_Pos)               /*!< INT_T::NMI_SEL: NMI_SEL Mask               */

#define INT_MCU_IRQ_MCU_IRQ_Pos               (0)                                               /*!< INT_T::MCU_IRQ: MCU_IRQ Position           */
#define INT_MCU_IRQ_MCU_IRQ_Msk               (0xfffffffful << INT_MCU_IRQ_MCU_IRQ_Pos)         /*!< INT_T::MCU_IRQ: MCU_IRQ Mask               */

/**@}*/ /* INT_CONST */
/**@}*/ /* end of INT register group */


/*---------------------- System Manger Controller -------------------------*/
/**
    @addtogroup SYS System Manger Controller(SYS)
    Memory Mapped Structure for SYS Controller
@{ */

typedef struct
{


    /**
     * @var SYS_T::PDID
     * Offset: 0x00  Part Device Identification Number Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |PDID      |Part Device Identification Number (Read Only)
     * |        |          |This register reflects device part number code
     * |        |          |Software can read this register to identify which device is used.
     * @var SYS_T::RSTSTS
     * Offset: 0x04  System Reset Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |PORF      |POR Reset Flag
     * |        |          |The POR reset flag is set by the Reset Signal from the Power-on Reset (POR) Controller or bit CHIPRST (SYS_IPRST0[0]) to indicate the previous reset source.
     * |        |          |0 = No reset from POR or CHIPRST.
     * |        |          |1 = Power-on Reset (POR) or CHIPRST had issued the reset signal to reset the system.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[1]     |PINRF     |NRESET Pin Reset Flag
     * |        |          |The nRESET pin reset flag is set by the Reset Signal from the nRESET Pin to indicate the previous reset source.
     * |        |          |0 = No reset from nRESET pin.
     * |        |          |1 = Pin nRESET had issued the reset signal to reset the system.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[2]     |WDTRF     |WDT Reset Flag
     * |        |          |The WDT reset flag is set by the Reset Signal from the Watchdog Timer to indicate the previous reset source.
     * |        |          |0 = No reset from watchdog timer or window watchdog timer.
     * |        |          |1 = The watchdog timer had issued the reset signal to reset the system.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[3]     |LVRF      |LVR Reset Flag
     * |        |          |The LVR reset flag is set by the Reset Signal from the Low-Voltage Reset controller to indicate the previous reset source.
     * |        |          |0 = No reset from LVR.
     * |        |          |1 = The LVR had issued the reset signal to reset the system.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[4]     |BODRF     |BOD Reset Flag
     * |        |          |The BOD reset flag is set by the Reset Signal from the Brown-Out Detector to indicate the previous reset source.
     * |        |          |0 = No reset from BOD.
     * |        |          |1 = The BOD had issued the reset signal to reset the system.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[5]     |SYSRF     |System Reset Flag
     * |        |          |The system reset flag is set by the Reset Signal from the Cortex-M0 Core to indicate the previous reset source.
     * |        |          |0 = No reset from Cortex-M0.
     * |        |          |1 = The Cortex-M0 had issued the reset signal to reset the system by writing 1 to the bit SYSRESETREQ(AIRCR[2], Application Interrupt and Reset Control Register, address = 0xE000ED0C) in system control registers of Cortex-M0 core.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[7]     |CPURF     |CPU Reset Flag
     * |        |          |The CPU reset flag is set by hardware if software writes CPURST (SYS_IPRST0[1]) 1 to reset Cortex-M0 Core and Flash Memory Controller (FMC).
     * |        |          |0 = No reset from CPU.
     * |        |          |1 = The Cortex-M0 Core and FMC are reset by software setting CPURST to 1.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[8]     |LOCKRF    |Lockup Reset Flag
     * |        |          |0 = No reset from Cortex-M0.
     * |        |          |1 = The Cortex-M0 had issued the reset signal to reset the system by Cortex-M0 lockup event.
     * @var SYS_T::IPRST1
     * Offset: 0x08  Peripheral Reset Control Resister1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CHIPRST   |Chip One-shot Reset (Write Protect)
     * |        |          |Setting this bit will reset the whole chip, including Processor core and all peripherals, and this bit will automatically return to 0 after the 2 clock cycles.
     * |        |          |The CHIPRST is same as the POR reset, all the chip controllers is reset and the chip setting from flash are also reload.
     * |        |          |About the difference between CHIPRST and SYSRESETREQ(AIRCR[2]), please refer to section 6.2.2
     * |        |          |0 = Chip normal operation.
     * |        |          |1 = Chip one-shot reset.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[1]     |CPURST    |Processor Core One-shot Reset (Write Protect)
     * |        |          |Setting this bit will only reset the processor core and Flash Memory Controller(FMC), and this bit will automatically return to 0 after the 2 clock cycles.
     * |        |          |0 = Processor core normal operation.
     * |        |          |1 = Processor core one-shot reset.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[2]     |PDMARST   |PDMA Controller Reset (Write Protect)
     * |        |          |Setting this bit to 1 will generate a reset signal to the PDMA
     * |        |          |User needs to set this bit to 0 to release from reset state.
     * |        |          |0 = PDMA controller normal operation.
     * |        |          |1 = PDMA controller reset.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * @var SYS_T::IPRST2
     * Offset: 0x0C  Peripheral Reset Control Resister2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1]     |GPIORST   |GPIO Controller Reset
     * |        |          |0 = GPIO module normal operation.
     * |        |          |1 = GPIO module reset.
     * |[2]     |TMR0RST   |Timer0 Controller Reset
     * |        |          |0 = Timer0 module normal operation.
     * |        |          |1 = Timer0 module reset.
     * |[3]     |TMR1RST   |Timer1 Controller Reset
     * |        |          |0 = Timer1 module normal operation.
     * |        |          |1 = Timer1 module reset.
     * |[4]     |TMR2RST   |Timer2 Controller Reset
     * |        |          |0 = Timer2 module normal operation.
     * |        |          |1 = Timer2 module reset.
     * |[5]     |TMR3RST   |Timer3 Controller Reset
     * |        |          |0 = Timer3 module normal operation.
     * |        |          |1 = Timer3 module reset.
     * |[8]     |I2C0RST   |I2C0 Controller Reset
     * |        |          |0 = I2C0 module normal operation.
     * |        |          |1 = I2C0 module reset.
     * |[9]     |I2C1RST   |I2C1 Controller Reset
     * |        |          |0 = I2C1 module normal operation.
     * |        |          |1 = I2C1 module reset.
     * |[12]    |SPI0RST   |SPI0 Controller Reset
     * |        |          |0 = SPI0 module normal operation.
     * |        |          |1 = SPI0 module reset.
     * |[13]    |SPI1RST   |SPI1 Controller Reset
     * |        |          |0 = SPI1 module normal operation.
     * |        |          |1 = SPI1 module reset.
     * |[14]    |SPI2RST   |SPI2 Controller Reset
     * |        |          |0 = SPI2 module normal operation.
     * |        |          |1 = SPI2 module reset.
     * |[15]    |SPI3RST   |SPI3 Controller Reset
     * |        |          |0 = SPI3 module normal operation.
     * |        |          |1 = SPI3 module reset.
     * |[16]    |UART0RST  |UART0 Controller Reset
     * |        |          |0 = UART0 module normal operation.
     * |        |          |1 = UART0 module reset.
     * |[17]    |UART1RST  |UART1 Controller Reset
     * |        |          |0 = UART1 module normal operation.
     * |        |          |1 = UART1 module reset.
     * |[20]    |PWM0RST   |PWM0 Controller Reset
     * |        |          |0 = PWM0 module normal operation.
     * |        |          |1 = PWM0 module reset.
     * |[22]    |ACMP01RST |Comparator Controller Reset
     * |        |          |0 = Comparator module normal operation.
     * |        |          |1 = Comparator module reset.
     * |[28]    |ADCRST    |ADC Controller Reset
     * |        |          |0 = ADC module normal operation.
     * |        |          |1 = ADC module reset.
     * |[30]    |SC0RST    |SmartCard 0 Controller Reset
     * |        |          |0 = SmartCard module normal operation.
     * |        |          |1 = SmartCard module reset.
     * |[31]    |SC1RST    |SmartCard1 Controller Reset
     * |        |          |0 = SmartCard module normal operation.
     * |        |          |1 = SmartCard module reset.
     * @var SYS_T::MISCCTL
     * Offset: 0x0C  Miscellaneous Control Resister
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[6]     |POR33DIS  |POR 33 Disable
     * |        |          |0 = POR33 is enable in normal operation.
     * |        |          |1 = POR33 is disable in normal operation.
     * |[7]     |POR18DIS  |POR 18 Disable
     * |        |          |0 = POR18 is enable in normal operation.
     * |        |          |1 = POR18 is disable in normal operation.
     * @var SYS_T::TEMPCTL
     * Offset: 0x20  Temperature Sensor Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |VTEMPEN   |Temperature Sensor Enable Bit
     * |        |          |This bit is used to enable/disable temperature sensor function.
     * |        |          |0 = Temperature sensor function Disabled (default).
     * |        |          |1 = Temperature sensor function Enabled.
     * @var SYS_T::RCCFCTL
     * Offset: 0x28  RC Clock Filter Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |HIRC0FEN  |HIRC0 Clock Filter Enable Bit
     * |        |          |This bit is used to enable/disable HIRC0 clock filter function.
     * |        |          |0 = HIRC0 clock filter function Disabled.
     * |        |          |1 = HIRC0 clock filter function Enabled (default).
     * |[1]     |HIRC1FEN  |HIRC1 Clock Filter Enable Bit
     * |        |          |This bit is used to enable/disable HIRC1 clock filter function.
     * |        |          |0 = HIRC1 clock filter function Disabled.
     * |        |          |1 = HIRC1 clock filter function Enabled (default).
     * |[2]     |MRCFEN    |MRC Clock Filter Enable Bit
     * |        |          |This bit is used to enable/disable MRC clock filter function.
     * |        |          |0 = 4MHz MRC clock filter function Disabled.
     * |        |          |1 = 4MHz MRC clock filter function Enabled (default).
     * @var SYS_T::GPA_MFPL
     * Offset: 0x30  GPIOA Low Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PA0MFP    |PA.0 Multi-function Pin Selection
     * |        |          |0000 = GPIOA[0]
     * |        |          |0001 = ADC analog input0.
     * |        |          |0010 = Comparator1 P-end input.
     * |        |          |0011 = Timer0 capture input.
     * |        |          |0101 = PWM0 channel2 output/capture input.
     * |[7:4]   |PA1MFP    |PA.1 Multi-function Pin Selection
     * |        |          |0000 = GPIOA[1]
     * |        |          |0001 = ADC analog input1.
     * |        |          |0010 = Comparator1 N-end input.
     * |        |          |0110 = SPI0 2nd MISO (Master In, Slave Out) pin.
     * |[11:8]  |PA2MFP    |PA.2 Multi-function Pin Selection
     * |        |          |0000 = GPIOA[2]
     * |        |          |0001 = ADC analog input2.
     * |        |          |0101 = Data receiver input pin for UART1.
     * |[15:12] |PA3MFP    |PA.3 Multi-function Pin Selection
     * |        |          |0000 = GPIOA[3]
     * |        |          |0001 = ADC analog input3.
     * |        |          |0101 = Data transmitter output pin for UART1.
     * |        |          |0110 = SPI3 1st MOSI (Master Out, Slave In) pin.
     * |[19:16] |PA4MFP    |PA.4 Multi-function Pin Selection
     * |        |          |0000 = GPIOA[4]
     * |        |          |0001 = ADC analog input4.
     * |        |          |0101 = I2C0 data input/output pin.
     * |        |          |0110 = SPI3 1st MISO (Master In, Slave Out) pin.
     * |[23:20] |PA5MFP    |PA.5 Multi-function Pin Selection
     * |        |          |0000 = GPIOA[5]
     * |        |          |0001 = ADC analog input5.
     * |        |          |0101 = I2C0 clock pin.
     * |        |          |0110 = SPI3 serial clock pin.
     * |[27:24] |PA6MFP    |PA.6 Multi-function Pin Selection
     * |        |          |0000 = GPIOA[6]
     * |        |          |0001 = ADC analog input6.
     * |        |          |0010 = Comparator1 output.
     * |        |          |0011 = Timer3 capture input.
     * |        |          |0100 = Timer3 external counter input.
     * |        |          |0101 = PWM0 channel3 output/capture input.
     * |        |          |0111 = Timer3 toggle output.
     * @var SYS_T::GPA_MFPH
     * Offset: 0x34  GPIOA High Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PA8MFP    |PA.8 Multi-function Pin Selection
     * |        |          |0000 = GPIOA[8]
     * |        |          |0001 = I2C0 data input/output pin.
     * |        |          |0010 = Timer0 external counter input.
     * |        |          |0011 = SmartCard0 clock pin.
     * |        |          |0100 = SPI2 slave select pin.
     * |        |          |0101 = Timer0 toggle output.
     * |        |          |0110 = UART0 Clear to Send input pin.
     * |[7:4]   |PA9MFP    |PA.9 Multi-function Pin Selection
     * |        |          |0000 = GPIOA[9]
     * |        |          |0001 = I2C0 clock pin.
     * |        |          |0010 = Timer1 external counter input.
     * |        |          |0011 = SmartCard0 data pin.
     * |        |          |0100 = SPI0 serial clock pin.
     * |        |          |0101 = Timer1 toggle output.
     * |        |          |0110 = UART1 Request to Send output pin.
     * |        |          |0111 = Snooper pin.
     * |[11:8]  |PA10MFP   |PA.10 Multi-function Pin Selection
     * |        |          |0000 = GPIOA[10]
     * |        |          |0001 = I2C1 data input/output pin.
     * |        |          |0010 = Timer2 external counter input.
     * |        |          |0011 = SmartCard0 power pin.
     * |        |          |0100 = SPI2 1st MISO (Master In, Slave Out) pin.
     * |        |          |0101 = Timer2 toggle output.
     * |[15:12] |PA11MFP   |PA.11 Multi-function Pin Selection
     * |        |          |0000 = GPIOA[11]
     * |        |          |0001 = I2C1 clock pin.
     * |        |          |0010 = Timer3 external counter input.
     * |        |          |0011 = SmartCard0 reset pin.
     * |        |          |0100 = SPI2 1st MOSI (Master Out, Slave In) pin.
     * |        |          |0101 = Timer3 toggle output.
     * |[19:16] |PA12MFP   |PA.12 Multi-function Pin Selection
     * |        |          |0000 = GPIOA[12]
     * |        |          |0001 = PWM0 channel0 output/capture input.
     * |        |          |0011 = Timer0 capture input.
     * |        |          |0101 = I2C0 data input/output pin.
     * |[23:20] |PA13MFP   |PA.13 Multi-function Pin Selection
     * |        |          |0000 = GPIOA[13]
     * |        |          |0001 = PWM0 channel1 output/capture input.
     * |        |          |0011 = Timer1 capture input.
     * |        |          |0101 = I2C0 clock pin.
     * |[27:24] |PA14MFP   |PA.14 Multi-function Pin Selection
     * |        |          |0000 = GPIOA[14]
     * |        |          |0001 = PWM0 channel2 output/capture input.
     * |        |          |0010 = I2C1 data input/output pin.
     * |        |          |0011 = I2C1 data input/output pin.
     * |        |          |0101 = Timer2 external counter input.
     * |        |          |0110 = Data receiver input pin for UART0.
     * |        |          |0111 = Timer2 toggle output.
     * |[31:28] |PA15MFP   |PA.15 Multi-function Pin Selection
     * |        |          |0000 = GPIOA[15]
     * |        |          |0001 = PWM0 channel3 output/capture input.
     * |        |          |0010 = I2C1 clock pin.
     * |        |          |0011 = Timer1 capture input.
     * |        |          |0100 = SmartCard0 power pin.
     * |        |          |0110 = Data transmitter output pin for UART0.
     * |        |          |0111 = Timer3 toggle output.
     * @var SYS_T::GPB_MFPL
     * Offset: 0x38  GPIOB Low Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PB0MFP    |PB.0 Multi-function Pin Selection
     * |        |          |0000 = GPIOB[0]
     * |        |          |0001 = Data receiver input pin for UART0.
     * |        |          |0011 = SPI1 1st MOSI (Master Out, Slave In) pin.
     * |[7:4]   |PB1MFP    |PB.1 Multi-function Pin Selection
     * |        |          |0000 = GPIOB[1]
     * |        |          |0001 = Data transmitter output pin for UART0.
     * |        |          |0011 = SPI1 1st MISO (Master In, Slave Out) pin.
     * |[11:8]  |PB2MFP    |PB.2 Multi-function Pin Selection
     * |        |          |0000 = GPIOB[2]
     * |        |          |0001 = UART0 Request to Send output pin.
     * |        |          |0011 = SPI1 serial clock pin.
     * |        |          |0100 = Frequency Divider output pin.
     * |[15:12] |PB3MFP    |PB.3 Multi-function Pin Selection
     * |        |          |0000 = GPIOB[3]
     * |        |          |0001 = UART0 Clear to Send input pin.
     * |        |          |0011 = SPI1 slave select pin.
     * |        |          |0100 = SmartCard1 card detect pin.
     * |[19:16] |PB4MFP    |PB.4 Multi-function Pin Selection
     * |        |          |0000 = GPIOB[4]
     * |        |          |0001 = Data receiver input pin for UART1.
     * |        |          |0011 = SmartCard0 card detect pin.
     * |        |          |0100 = SPI2 slave select pin.
     * |        |          |0110 = RTC 1Hz output.
     * |[23:20] |PB5MFP    |PB.5 Multi-function Pin Selection
     * |        |          |0000 = GPIOB[5]
     * |        |          |0001 = Data transmitter output pin for UART1.
     * |        |          |0011 = SmartCard0 reset pin.
     * |        |          |0100 = SPI2 serial clock pin.
     * |[27:24] |PB6MFP    |PB.6 Multi-function Pin Selection
     * |        |          |0000 = GPIOB[6]
     * |        |          |0001 = UART1 Request to Send output pin.
     * |        |          |0100 = SPI2 1st MISO (Master In, Slave Out) pin.
     * |[31:28] |PB7MFP    |PB.7 Multi-function Pin Selection
     * |        |          |0000 = GPIOB[7]
     * |        |          |0001 = UART1 Clear to Send input pin.
     * @var SYS_T::GPB_MFPH
     * Offset: 0x3C  GPIOB High Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PB8MFP    |PB.8 Multi-function Pin Selection
     * |        |          |0000 = GPIOB[8]
     * |        |          |0001 = ADC external trigger input.
     * |        |          |0010 = Timer0 external counter input.
     * |        |          |0011 = External interrupt0 input pin.
     * |        |          |0100 = Timer0 toggle output.
     * |        |          |0111 = Snooper pin.
     * |[7:4]   |PB9MFP    |PB.9 Multi-function Pin Selection
     * |        |          |0000 = GPIOB[9]
     * |        |          |0001 = SPI1 slave select pin.
     * |        |          |0010 = Timer2 external counter input.
     * |        |          |0100 = Timer2 toggle output.
     * |        |          |0101 = External interrupt0 input pin.
     * |[11:8]  |PB10MFP   |PB.10 Multi-function Pin Selection
     * |        |          |0000 = GPIOB[10]
     * |        |          |0001 = SPI0 1st MOSI (Master Out, Slave In) pin.
     * |        |          |0100 = Timer2 toggle output.
     * |        |          |0101 = SPI0 slave select pin.
     * |[15:12] |PB11MFP   |PB.11 Multi-function Pin Selection
     * |        |          |0000 = GPIOB[11]
     * |        |          |0001 = PWM0 channel4 output/capture input.
     * |        |          |0010= Timer3 external counter input.
     * |        |          |0100 = Timer3 toggle output.
     * |        |          |0101 = SPI0 1st MISO (Master In, Slave Out) pin.
     * |[23:20] |PB13MFP   |PB.13 Multi-function Pin Selection
     * |        |          |0000 = GPIOB[13]
     * |        |          |0011 = SPI2 2nd MISO (Master In, Slave Out) pin.
     * |        |          |0111 = Snooper pin.
     * |[27:24] |PB14MFP   |PB.14 Multi-function Pin Selection
     * |        |          |0000 = GPIOB[14]
     * |        |          |0001 = External interrupt0 input pin.
     * |        |          |0011 = SPI2 2nd MOSI (Master Out, Slave In) pin.
     * |        |          |0100 = SPI2 slave select pin.
     * |[31:28] |PB15MFP   |PB.15 Multi-function Pin Selection
     * |        |          |0000 = GPIOB[15]
     * |        |          |0001 = External interrupt1 input pin.
     * |        |          |0011 = Snooper pin.
     * |        |          |0100 = SmartCard1 card detect pin.
     * @var SYS_T::GPC_MFPL
     * Offset: 0x40  GPIOC Low Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PC0MFP    |PC.0 Multi-function Pin Selection
     * |        |          |0000 = GPIOC[0]
     * |        |          |0001 = SPI0 slave select pin.
     * |        |          |0100 = SmartCard1 clock pin.
     * |        |          |0101 = PWM0 break1 input 1.
     * |[7:4]   |PC1MFP    |PC.1 Multi-function Pin Selection
     * |        |          |0000 = GPIOC[1]
     * |        |          |0001 = SPI0 serial clock pin.
     * |        |          |0100 = SmartCard1 data pin.
     * |        |          |0101 = PWM0 break1 input 0.
     * |[11:8]  |PC2MFP    |PC.2 Multi-function Pin Selection
     * |        |          |0000 = GPIOC[2]
     * |        |          |0001 = SPI0 1st MISO (Master In, Slave Out) pin.
     * |        |          |0100 = SmartCard1 power pin.
     * |        |          |0101 = PWM0 break0 input 1.
     * |[15:12] |PC3MFP    |PC.3 Multi-function Pin Selection
     * |        |          |0000 = GPIOC[3]
     * |        |          |0001 = SPI0 1st MOSI (Master Out, Slave In) pin.
     * |        |          |0100 = SmartCard1 reset pin.
     * |        |          |0101 = PWM0 break0 input 0.
     * |[27:24] |PC6MFP    |PC.6 Pin Function Selection
     * |        |          |0000 = GPIOC[6]
     * |        |          |0001 = Data receiver input pin for UART1.
     * |        |          |0011 = Timer0 capture input.
     * |        |          |0100 = SmartCard1 card detect pin.
     * |        |          |0101 = PWM0 channel0 output/capture input.
     * |[31:28] |PC7MFP    |PC.7 Multi-function Pin Selection
     * |        |          |0000 = GPIOC[7]
     * |        |          |0001 = Data transmitter output pin for UART1.
     * |        |          |0010 = ADC analog input7.
     * |        |          |0011 = Timer1 capture input.
     * |        |          |0101 = PWM0 channel1 output/capture input.
     * @var SYS_T::GPC_MFPH
     * Offset: 0x44  GPIOC High Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PC8MFP    |PC.8 Multi-function Pin Selection
     * |        |          |0000 = GPIOC[8]
     * |        |          |0001 = SPI1 slave select pin.
     * |        |          |0101 = I2C1 data input/output pin.
     * |[7:4]   |PC9MFP    |PC.9 Multi-function Pin Selection
     * |        |          |0000 = GPIOC[9]
     * |        |          |0001 = SPI1 serial clock pin.
     * |        |          |0101 = I2C1 clock pin.
     * |[11:8]  |PC10MFP   |PC.10 Multi-function Pin Selection
     * |        |          |0000 = GPIOC[10]
     * |        |          |0001 = SPI0 1st MISO (Master In, Slave Out) pin.
     * |        |          |0101 = Data receiver input pin for UART1.
     * |[15:12] |PC11MFP   |PC.11 Multi-function Pin Selection
     * |        |          |0000 = GPIOC[11]
     * |        |          |0001 = SPI1 1st MOSI (Master Out, Slave In) pin.
     * |        |          |0101 = Data transmitter output pin for UART1.
     * |[27:24] |PC14MFP   |PC.14 Multi-function Pin Selection
     * |        |          |0000 = GPIOC[14]
     * |        |          |0001 = UART0 Clear to Send input pin.
     * |[31:28] |PC15MFP   |PC.15 Multi-function Pin Selection
     * |        |          |0000 = GPIOC[15]
     * |        |          |0001 = UART1 Request to Send output pin.
     * |        |          |0011 = Timer0 capture input.
     * @var SYS_T::GPD_MFPL
     * Offset: 0x48  GPIOD Low Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[27:24] |PD6MFP    |PD.6 Multi-function Pin Selection
     * |        |          |0000 = GPIOD[6]
     * |        |          |0011 = SPI1 2nd MOSI (Master Out, Slave In) pin.
     * |        |          |0100 = SmartCard1 reset pin.
     * |[31:28] |PD7MFP    |PD.7 Multi-function Pin Selection
     * |        |          |0000 = GPIOD[7]
     * |        |          |0011 = SPI1 2nd MISO (Master In, Slave Out) pin.
     * |        |          |0100 = SmartCard1 power pin.
     * @var SYS_T::GPD_MFPH
     * Offset: 0x4C  GPIOD High Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[27:24] |PD14MFP   |PD.14 Multi-function Pin Selection
     * |        |          |0000 = GPIOD[14]
     * |        |          |0001 = SPI0 2nd MOSI (Master Out, Slave In) pin.
     * |[30:28] |PD15MFP   |PD.15 Multi-function Pin Selection
     * |        |          |0000 = GPIOD[15]
     * |        |          |0001 = SPI0 2nd MISO (Master In, Slave Out) pin.
     * |        |          |0100 = SmartCard1 clock pin.
     * @var SYS_T::GPE_MFPL
     * Offset: 0x50  GPIOE Low Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[23:20] |PE5MFP    |PE.5 Multi-function Pin Selection
     * |        |          |0000 = GPIOE[5]
     * |        |          |0001 = PWM0 channel5 output/capture input.
     * |        |          |0110 = RTC 1Hz output.
     * @var SYS_T::GPF_MFPL
     * Offset: 0x58  GPIOF Low Byte Multiple Function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PF0MFP    |PF.0 Multi-function Pin Selection
     * |        |          |0000 = GPIOF[1]
     * |        |          |0101 = External interrupt0 input pin.
     * |        |          |X111 = Serial wired debugger data pin
     * |[7:4]   |PF1MFP    |PF.1 Multi-function Pin Selection
     * |        |          |0000 = GPIOF[1]
     * |        |          |0100 = Frequency Divider output pin.
     * |        |          |0101 = External interrupt1 input pin.
     * |        |          |X111 = Serial wired debugger clock pin.
     * |[11:8]  |PF2MFP    |PF.2 Multi-function Pin Selection
     * |        |          |0000 = GPIOF[2]
     * |        |          |X111 = External 4~36 MHz (high speed) crystal output pin.
     * |[15:12] |PF3MFP    |PF.3 Multi-function Pin Selection
     * |        |          |0000 = GPIOF[3]
     * |        |          |X111 = External 4~36 MHz (high speed) crystal input pin.
     * |[27:24] |PF6MFP    |PF.6 Multi-function Pin Selection
     * |        |          |0000 = GPIOF[6]
     * |        |          |0001 = I2C1 data input/output pin.
     * |        |          |X111 = External 32.768 kHz crystal output pin(default).
     * |[31:28] |PF7MFP    |PF.7 Multi-function Pin Selection
     * |        |          |0000 = GPIOF[7]
     * |        |          |0001 = I2C1 clock pin.
     * |        |          |0011 = SmartCard0 card detect pin.
     * |        |          |X111 = External 32.768 kHz crystal input pin(default).
     * @var SYS_T::PORCTL
     * Offset: 0x60  Power-On-Reset Controller Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |POROFF    |Power-on Reset Enable Bit (Write Protect)
     * |        |          |When powered on, the POR circuit generates a reset signal to reset the whole chip function, but noise on the power may cause the POR active again
     * |        |          |User can disable internal POR circuit to avoid unpredictable noise to cause chip reset by writing 0x5AA5 to this field.
     * |        |          |The POR function will be active again when this field is set to another value or chip is reset by other reset source, including:
     * |        |          |nRESET, Watchdog, BOD reset, ICE reset command and the software-chip reset function.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * @var SYS_T::BODCTL
     * Offset: 0x64  Brown-out Detector Controller Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |BODEN     |Brown-out Detector Enable Bit (Write Protect)
     * |        |          |The default value is set by flash controller user configuration register CBODEN (CONFIG0 [])
     * |        |          |This Brown-out Detector only valid in Normal Mode.
     * |        |          |0 = Brown-out Detector function Disabled in Normal mode.
     * |        |          |1 = Brown-out Detector function Enabled in Normal mode.
     * |        |          |Note1: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |        |          |Note2: LIRC must be enabled before enable BOD.
     * |[2]     |BODIE     |BOD Interrupt Enable Control (Write Protect)
     * |        |          |0 = Interrupt does not issue when BOD occurs in Normal Mode.
     * |        |          |1 = Interrupt issues when BOD occurs in Normal Mode.
     * |        |          |Note1: While the BOD function is enabled (BODEN high) and BOD interrupt function is enabled (BODRSTEN low), BOD will assert an interrupt if BODOUT is high.
     * |        |          |Note2: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[3]     |BODREN    |Brown-out Reset Enable Bit (Write Protect)
     * |        |          |The default value is set by flash controller user configuration register CBOV(CONFIG0[]) bit .
     * |        |          |0 = Brown-out RESET function Disabled in Normal Mode.
     * |        |          |1 = Brown-out RESET function Enabled in Normal Mode.
     * |        |          |Note1: While the Brown-out Detector function is enabled (BODEN high) and BOD reset function is enabled (BODRSTEN high), BOD will assert a signal to reset chip when the detected voltage is lower than the threshold (BODOUT high).
     * |        |          |Note2: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[4]     |BODIF     |Brown-out Detector Interrupt Flag
     * |        |          |0 = Brown-out Detector does not detect any voltage drift at VDD down through or up through the target detected voltage after interrupt is enabled.
     * |        |          |1 = When Brown-out Detector detects the VDD is dropped down through the target detected voltage or the VDD is raised up through the target detected voltage, this bit is set to 1 and the brown-out interrupt is requested if brown-out interrupt is enabled.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[6]     |BODOUT    |Brown-out Detector Output Status
     * |        |          |0 = Brown-out Detector output status is 0.
     * |        |          |It means the detected voltage is higher than BODVL setting or BODEN is 0.
     * |        |          |1 = Brown-out Detector output status is 1.
     * |        |          |It means the detected voltage is lower than BODVL setting
     * |        |          |If the BODEN is 0, BOD function disabled , this bit always responds 0.
     * |        |          |Note: This bit is ready-only.
     * |[7]     |LVREN     |Low Voltage Reset Enable Bit (Write Protect)
     * |        |          |The LVR function resets the chip when the input power voltage is lower than LVR circuit setting
     * |        |          |LVR function is enabled by default.
     * |        |          |0 = Low Voltage Reset function Disabled.
     * |        |          |1 = Low Voltage Reset function Enabled.
     * |        |          |Note1: After enabling the bit, the LVR function will be active with 100us delay for LVR output stable (default).
     * |        |          |Note2: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |        |          |Note3: LIRC must be enabled before enable LVR.
     * |[8]     |LPBODEN   |Low Power Brown-out Detector Enable Bit (Write Protect)
     * |        |          |Low Power Brown-out Detector only valid in Power Down mode.
     * |        |          |0 = Low Power Brown-out Detector function Disabled in Power Down mode.
     * |        |          |1 = Low Power Brown-out Detector function Enabled in Power Down mode.
     * |        |          |Note1: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |        |          |Note2: LIRC must be enabled before enable BOD.
     * |[9]     |LPBODVL   |Low Power Brown-out Detector Threshold Voltage Selection (Write Protect)
     * |        |          |Low Power Brown-out Detector only valid in Power Down mode.
     * |        |          |0 = Low Power Brown-Out Detector threshold voltage is 2.0V in Power Down mode.
     * |        |          |1 = Low Power Brown-Out Detector threshold voltage is 2.5V in Power Down mode.
     * |        |          |Note1: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[10]    |LPBODIE   |Low Power BOD Interrupt Enable Control (Write Protect)
     * |        |          |Low Power Brown-out Detector only valid in Power Down mode.
     * |        |          |0 = Interrupt does not issue when LPBOD occurs in Power Down mode.
     * |        |          |1 = Interrupt issues when LPBOD occurs in Power Down mode
     * |        |          |Note1: While the LPBOD function is enabled (LPBODEN high) and LPBOD interrupt function is enabled (LPBODIE high), LPBOD will assert an interrupt if BODOUT is high
     * |        |          |Note2: This bit is write protected
     * |        |          |Refer to the SYS_REGLCTL register.
     * |[11]    |LPBODREN  |Low Power Brown-out Reset Enable Bit (Write Protect)
     * |        |          |Low Power Brown-out Detector only valid in Power Down mode.
     * |        |          |0 = Low power Brown-out Detector RESET function Disabled in Power Down mode.
     * |        |          |1 = Low Power Brown-out Detector RESET function Enabled in Power Down mode.
     * |        |          |Note1: While the Low power Brown-out Detector function is enabled (LPBODEN high) and LPBOD reset function is enabled (LPBODREN high), LPBOD will assert a signal to reset chip when the detected voltage is lower than the threshold (BODOUT high).
     * |        |          |Note2: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[15:12] |BODVL     |Brown-out Detector Threshold Voltage Selection (Write Protect)
     * |        |          |The default value is set by flash controller user configuration register CBOV (CONFIG0[]).
     * |        |          |0000 = Brown-Out Detector threshold voltage is 1.7V.
     * |        |          |0001 = Brown-Out Detector threshold voltage is 1.8V.
     * |        |          |0010 = Brown-Out Detector threshold voltage is 1.9V.
     * |        |          |0011 = Brown-Out Detector threshold voltage is 2.0V.
     * |        |          |0100 = Brown-Out Detector threshold voltage is 2.1V.
     * |        |          |0101 = Brown-Out Detector threshold voltage is 2.2V.
     * |        |          |0110 = Brown-Out Detector threshold voltage is 2.3V.
     * |        |          |0111 = Brown-Out Detector threshold voltage is 2.4V.
     * |        |          |1000 = Brown-Out Detector threshold voltage is 2.5V.
     * |        |          |1001 = Brown-Out Detector threshold voltage is 2.6V.
     * |        |          |1010 = Brown-Out Detector threshold voltage is 2.7V.
     * |        |          |1011 = Brown-Out Detector threshold voltage is 2.8V.
     * |        |          |1100 = Brown-Out Detector threshold voltage is 2.9V.
     * |        |          |1101 = Brown-Out Detector threshold voltage is 3.0V.
     * |        |          |1110 = Brown-Out Detector threshold voltage is 3.1V.
     * |        |          |1111 = Reserved.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[19:16] |LPBOD20TRIM|Low power BOD 2.0 TRIM Value (Write Protect)
     * |        |          |This value is used to control BOD20 detect voltage level in power-down mode, nominal 2.0 V
     * |        |          |Higher trim value, higher detection voltage.
     * |        |          |Note: These bits are write protected. Refer to the SYS_REGLCTL register.
     * |[23:20] |LPBOD25TRIM|Low power BOD 2.5 TRIM Value (Write Protect)
     * |        |          |This value is used to control LPBOD25 detect voltage level in power-down mode, nominal 2.5 V
     * |        |          |Higher trim value, higher detection voltage.
     * |        |          |Note: These bits are write protected. Refer to the SYS_REGLCTL register.
     * |[26:24] |BODDGSEL  |Brown-out Detector Output De-glitch Time Select (Write Protect)
     * |        |          |000 = BOD output is sampled by RC10K clock.
     * |        |          |001 = 4 system clock (HCLK).
     * |        |          |010 = 8 system clock (HCLK).
     * |        |          |011 = 16 system clock (HCLK).
     * |        |          |100 = 32 system clock (HCLK).
     * |        |          |101 = 64 system clock (HCLK).
     * |        |          |110 = 128 system clock (HCLK).
     * |        |          |111 = 256 system clock (HCLK).
     * |        |          |Note: These bits are write protected. Refer to the SYS_REGLCTL register.
     * |[30:28] |LVRDGSEL  |LVR Output De-glitch Time Select (Write Protect)
     * |        |          |000 = Without de-glitch function.
     * |        |          |001 = 4 system clock (HCLK).
     * |        |          |010 = 8 system clock (HCLK).
     * |        |          |011 = 16 system clock (HCLK).
     * |        |          |100 = 32 system clock (HCLK).
     * |        |          |101 = 64 system clock (HCLK).
     * |        |          |110 = 128 system clock (HCLK).
     * |        |          |111 = 256 system clock (HCLK).
     * |        |          |Note: These bits are write protected. Refer to the SYS_REGLCTL register.
     * @var SYS_T::IVREFCTL
     * Offset: 0x6C  Internal Voltage Reference Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |BGPEN     |Band-gap Enable Control (Write Protect)
     * |        |          |This is a protected register. Please refer to open lock sequence to program it.
     * |        |          |Band-gap is the reference voltage of internal reference voltage
     * |        |          |User must enable band-gap if want to enable internal 1.5, 1.8V or 2.5V reference voltage.
     * |        |          |0 = Disabled.
     * |        |          |1 = Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[1]     |REGEN     |Regulator Enable Control (Write Protect)
     * |        |          |Enable internal 1.5, 1.8V or 2.5V reference voltage.
     * |        |          |This is a protected register. Please refer to open lock sequence to program it.
     * |        |          |0 = Disabled.
     * |        |          |1 = Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[3:2]   |SEL25     |Regulator Output Voltage Selection (Write Protect)
     * |        |          |Select internal reference voltage level.
     * |        |          |00 = 1.5V.
     * |        |          |01 = 1.8V.
     * |        |          |10 = 2.5V.
     * |        |          |11 = 2.5V.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[4]     |EXTMODE   |Regulator External Mode (Write Protect)
     * |        |          |Users can output regulator output voltage in VREF pin if EXT_MODE is high.
     * |        |          |0 = No connection with external VREF pin.
     * |        |          |1 = Connect to external VREF pin
     * |        |          |Connect a 1uF to 10uF capacitor to AVSS will let internal voltage reference be more stable.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[11:8]  |VREFTRIM  |Internal Voltage Reference Trim (Write Protect)
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * @var SYS_T::LDOCTL
     * Offset: 0x70  LDO Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1]     |FASTWK    |Fast Wakeup Control Bit (Write Protect)
     * |        |          |0 = Fast Wakeup from Power-Down mode Disabled.
     * |        |          |1 = Fast Wakeup from Power-Down mode Enabled
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[3:2]   |LDOLVL    |LDO Output Voltage Select (Write Protect)
     * |        |          |00 = 1.2V.
     * |        |          |01 = 1.6V.
     * |        |          |10 = 1.8V.
     * |        |          |11 = 1.8V.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[4]     |LPRMEN    |Low-Power Run Mode Enable Bit (Write Protect)
     * |        |          |0 = Low-Power run mode Enabled.
     * |        |          |1 = Low-Power run mode Disabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[5]     |FMCLVEN   |Flash memory Low voltage Mode Enable Bit (Write Protect)
     * |        |          |0 = Flash memory low voltage(1.2V) mode Enabled.
     * |        |          |1 = Flash memory low voltage(1.2V) mode Disabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * @var SYS_T::BATDIVCTL
     * Offset: 0x74  Battery Voltage Divider Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |BATDIV2EN |Battery voltage divide 2 Enable Bit
     * |        |          |This bit is used to enable/disable battery voltage divider function.
     * |        |          |0 = Battery voltage divide 2 function Disabled (default).
     * |        |          |1 = Battery voltage divide 2 function Enabled.
     * @var SYS_T::WKSTS
     * Offset: 0x7C  System Wakeup Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ACMPWK    |ACMP Wake-up Flag (Read Only)
     * |        |          |This flag indicates that wakeup of device from Power-down mode was requested with ACMP wakeup event
     * |        |          |This flag is cleared when Power-down mode is entered.
     * |[1]     |I2C1WK    |I2C1 Wake-up Flag (Read Only)
     * |        |          |This flag indicates that wakeup of device from Power-down mode was requested with I2C1 wakeup event
     * |        |          |This flag is cleared when Power-down mode is entered.
     * |[2]     |I2C0WK    |I2C0 Wake-up Flag (Read Only)
     * |        |          |This flag indicates that wakeup of device from Power-down mode was requested with I2C0 wakeup event
     * |        |          |This flag is cleared when Power-down mode is entered.
     * |[3]     |TMR3WK    |TMR3 Wake-up Flag (Read Only)
     * |        |          |This flag indicates that wakeup of device from Power-down mode was requested withTMR3 wakeup event
     * |        |          |This flag is cleared when Power-down mode is entered.
     * |[4]     |TMR2WK    |TMR2 Wake-up Flag (Read Only)
     * |        |          |This flag indicates that wakeup of device from Power-down mode was requested withTMR2 wakeup event
     * |        |          |This flag is cleared when Power-down mode is entered.
     * |[5]     |TMR1WK    |TMR1 Wake-up Flag (Read Only)
     * |        |          |This flag indicates that wakeup of device from Power-down mode was requested withTMR1 wakeup event
     * |        |          |This flag is cleared when Power-down mode is entered.
     * |[6]     |TMR0WK    |TMR0 Wake-up Flag (Read Only)
     * |        |          |This flag indicates that wakeup of device from Power-down mode was requested withTMR0 wakeup event
     * |        |          |This flag is cleared when Power-down mode is entered.
     * |[7]     |WDTWK     |WDT Wake-up Flag (Read Only)
     * |        |          |This flag indicates that wakeup of device from Power-down mode was requested with WDT wakeup event
     * |        |          |This flag is cleared when Power-down mode is entered.
     * |[8]     |BODWK     |BOD Wake-up Flag (Read Only)
     * |        |          |This flag indicates that wakeup of device from Power-down mode was requested with BOD wakeup event
     * |        |          |This flag is cleared when Power-down mode is entered.
     * |[9]     |SPI3WK    |SPI3 Wake-up Flag (Read Only)
     * |        |          |This flag indicates that wakeup of device from Power-down mode was requested with SPI3 wakeup event
     * |        |          |This flag is cleared when Power-down mode is entered.
     * |[10]    |SPI2WK    |SPI2 Wake-up Flag (Read Only)
     * |        |          |This flag indicates that wakeup of device from Power-down mode was requested with SPI2 wakeup event
     * |        |          |This flag is cleared when Power-down mode is entered.
     * |[11]    |SPI1WK    |SPI1 Wake-up Flag (Read Only)
     * |        |          |This flag indicates that wakeup of device from Power-down mode was requested with SPI1 wakeup event
     * |        |          |This flag is cleared when Power-down mode is entered.
     * |[12]    |SPI0WK    |SPI0 Wake-up Flag (Read Only)
     * |        |          |This flag indicates that wakeup of device from Power-down mode was requested with SPI0 wakeup event
     * |        |          |This flag is cleared when Power-down mode is entered.
     * |[13]    |UART1WK   |UART1 Wake-up Flag (Read Only)
     * |        |          |This flag indicates that wakeup of device from Power-down mode was requested with UART1 wakeup event
     * |        |          |This flag is cleared when Power-down mode is entered.
     * |[14]    |UART0WK   |UART0 Wake-up Flag (Read Only)
     * |        |          |This flag indicates that wakeup of device from Power-down mode was requested with UART0 wakeup event
     * |        |          |This flag is cleared when Power-down mode is entered.
     * |[15]    |RTCWK     |RTC Wake-up Flag (Read Only)
     * |        |          |This flag indicates that wakeup of device from Power-down mode was requested with a RTC alarm or tick time happened
     * |        |          |This flag is cleared when Power-down mode is entered.
     * |[16]    |GPIOWK    |GPIO Wake-up Flag (Read Only)
     * |        |          |This flag indicates that wakeup of device from Power-down mode was requested with GPIO wakeup event
     * |        |          |This flag is cleared when Power-down mode is entered.
     * @var SYS_T::IRC0TCTL
     * Offset: 0x80  HIRC0 Trim Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |FREQSEL   |Trim Frequency Selection
     * |        |          |This field indicates the target frequency of 12 MHz internal high speed RC oscillator (HIRC0) auto trim.
     * |        |          |During auto trim operation, if clock error detected with CESTOPEN (SYS_IRC0TCTL[8]) is set to 1 or trim retry limitation count reached, this field will be cleared to 000 automatically.
     * |        |          |000 = Disable HIRC0 auto trim function.
     * |        |          |001 = Enable HIRC0 auto trim function and trim HIRC to 11.0592 MHz.
     * |        |          |010 = Enable HIRC0 auto trim function and trim HIRC to 12 MHz.
     * |        |          |011 = Enable HIRC0 auto trim function and trim HIRC to 12.288 MHz.
     * |        |          |100 = Enable HIRC0 auto trim function and trim HIRC to 16 MHz.
     * |        |          |Note: HIRC0 auto trim cannot work normally at power down mode
     * |        |          |These bits must be cleared before entering power down mode.
     * |[5:4]   |LOOPSEL   |Trim Calculation Loop Selection
     * |        |          |This field defines that trim value calculation is based on how many 32.768 kHz clock.
     * |        |          |00 = Trim value calculation is based on average difference in 4 32.768 kHz clock.
     * |        |          |01 = Trim value calculation is based on average difference in 8 32.768 kHz clock.
     * |        |          |10 = Trim value calculation is based on average difference in 16 32.768 kHz clock.
     * |        |          |11 = Trim value calculation is based on average difference in 32 32.768 kHz clock.
     * |        |          |Note: For example, if LOOPSEL is set as 00, auto trim circuit will calculate trim value based on the average frequency difference in 4 32.768 kHz clock.
     * |[7:6]   |RETRYCNT  |Trim Value Update Limitation Count
     * |        |          |This field defines that how many times the auto trim circuit will try to update the HIRC0 trim value before the frequency of HIRC0 locked.
     * |        |          |Once the HIRC0 locked, the internal trim value update counter will be reset.
     * |        |          |If the trim value update counter reached this limitation value and frequency of HIRC0 still doesn't lock, the auto trim operation will be disabled and FREQSEL (SYS_IRC0TCTL[1:0]) will be cleared to 00.
     * |        |          |00 = Trim retry count limitation is 64 loops.
     * |        |          |01 = Trim retry count limitation is 128 loops.
     * |        |          |10 = Trim retry count limitation is 256 loops.
     * |        |          |11 = Trim retry count limitation is 512 loops.
     * |[8]     |CESTOPEN  |Clock Error Stop Enable Bit
     * |        |          |This bit is used to control if stop the HIRC0 trim operation when 32.768 kHz clock error is detected.
     * |        |          |If set this bit high and 32.768 kHz clock error detected, the status CLKERRIF (SYS_IRC0TISTS[2]) would be set high and HIRC0 trim operation was stopped
     * |        |          |If this bit is low and 32.768 kHz clock error detected, the status CLKERRIF (SYS_IRC0TISTS[2]) would be set high and HIRC0 trim operation is continuously.
     * |        |          |0 = The trim operation is keep going if clock is inaccuracy.
     * |        |          |1 = The trim operation is stopped if clock is inaccuracy.
     * @var SYS_T::IRC0TIEN
     * Offset: 0x84  HIRC0 Trim Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1]     |TFAILIEN  |Trim Failure Interrupt Enable Bit
     * |        |          |This bit controls if an interrupt will be triggered while HIRC0 trim value update limitation count reached and HIRC0 frequency still not locked on target frequency set by FREQSEL (SYS_IRC0TCTL[1:0]).
     * |        |          |If this bit is high and TFAILIF (SYS_IRC0TSTS[1]) is set during auto trim operation, an interrupt will be triggered to notify that HIRC0 trim value update limitation count was reached.
     * |        |          |0 = Disable TFAILIF (SYS_IRC0TSTS[1]) status to trigger an interrupt to CPU.
     * |        |          |1 = Enable TFAILIF (SYS_IRC0TSTS[1]) status to trigger an interrupt to CPU.
     * |[2]     |CLKEIEN   |Clock Error Interrupt Enable Bit
     * |        |          |This bit controls if CPU would get an interrupt while clock is inaccuracy during auto trim operation.
     * |        |          |If this bit is set to1, and CLKERRIF (SYS_IRC0TSTS[2]) is set during auto trim operation, an interrupt will be triggered to notify the clock frequency is inaccuracy.
     * |        |          |0 = Disable CLKERRIF (SYS_IRC0TSTS[2]) status to trigger an interrupt to CPU.
     * |        |          |1 = Enable CLKERRIF (SYS_IRC0TSTS[2]) status to trigger an interrupt to CPU.
     * @var SYS_T::IRC0TISTS
     * Offset: 0x88  HIRC0 Trim Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |FREQLOCK  |HIRC0 Frequency Lock Status
     * |        |          |This bit indicates the HIRC0 frequency is locked.
     * |        |          |This is a status bit and doesn't trigger any interrupt.
     * |        |          |0 = The internal high-speed oscillator frequency doesn't lock at frequency set by FREQSEL (SYS_IRC0TCTL[2:0]).
     * |        |          |1 = The internal high-speed oscillator frequency locked at frequency set by FREQSEL (SYS_IRC0TCTL[2:0]).
     * |[1]     |TFAILIF   |Trim Failure Interrupt Status
     * |        |          |This bit indicates that HIRC0 trim value update limitation count reached and the HIRC0 clock frequency still doesn't be locked
     * |        |          |Once this bit is set, the auto trim operation stopped and FREQSEL (SYS_IRC0TCTL[1:0]) will be cleared to 00 by hardware automatically.
     * |        |          |If this bit is set and TFAILIEN (SYS_IRC0TIEN[1]) is high, an interrupt will be triggered to notify that HIRC0 trim value update limitation count was reached
     * |        |          |Write 1 to clear this to 0.
     * |        |          |0 = Trim value update limitation count does not reach.
     * |        |          |1 = Trim value update limitation count reached and HIRC frequency still not locked.
     * |[2]     |CLKERRIF  |Clock Error Interrupt Status
     * |        |          |When the frequency of 32.768 kHz external low speed crystal oscillator (LXT) or internal high speed RC oscillator (HIRC0) is shift larger to unreasonable value, this bit will be set and to be an indicate that clock frequency is inaccuracy
     * |        |          |Once this bit is set to 1, the auto trim operation stopped and FREQSEL (SYS_IRC0TCTL[1:0]) will be cleared to 00 by hardware automatically if CESTOPEN (SYS_IRC0TCTL[8]) is set to 1.
     * |        |          |If this bit is set and CLKEIEN (SYS_IRC0TIEN[2]) is high, an interrupt will be triggered to notify the clock frequency is inaccuracy
     * |        |          |Write 1 to clear this to 0.
     * |        |          |0 = Clock frequency is accuracy.
     * |        |          |1 = Clock frequency is inaccuracy.
     * @var SYS_T::IRC1TCTL
     * Offset: 0x90  HIRC1 Trim Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |FREQSEL   |Trim Frequency Selection
     * |        |          |This field indicates the target frequency of 36 MHz internal high speed RC oscillator (HIRC1) auto trim.
     * |        |          |During auto trim operation, if clock error detected with CESTOPEN (SYS_IRC1TCTL[8]) is set to 1 or trim retry limitation count reached, this field will be cleared to 00 automatically.
     * |        |          |00 = Disable HIRC1 auto trim function.
     * |        |          |01 = Reserved
     * |        |          |10 = Enable HIRC1 auto trim function and trim HIRC to 36 MHz.
     * |        |          |11 = Reserved.
     * |        |          |Note: HIRC1 auto trim cannot work normally at power down mode
     * |        |          |These bits must be cleared before entering power down mode.
     * |[5:4]   |LOOPSEL   |Trim Calculation Loop Selection
     * |        |          |This field defines that trim value calculation is based on how many 32.768 kHz clock.
     * |        |          |00 = Trim value calculation is based on average difference in 4 32.768 kHz clock.
     * |        |          |01 = Trim value calculation is based on average difference in 8 32.768 kHz clock.
     * |        |          |10 = Trim value calculation is based on average difference in 16 32.768 kHz clock.
     * |        |          |11 = Trim value calculation is based on average difference in 32 32.768 kHz clock.
     * |        |          |Note: For example, if LOOPSEL is set as 00, auto trim circuit will calculate trim value based on the average frequency difference in 4 32.768 kHz clock.
     * |[7:6]   |RETRYCNT  |Trim Value Update Limitation Count
     * |        |          |This field defines that how many times the auto trim circuit will try to update the HIRC1 trim value before the frequency of HIRC1 locked.
     * |        |          |Once the HIRC1 locked, the internal trim value update counter will be reset.
     * |        |          |If the trim value update counter reached this limitation value and frequency of HIRC1 still doesn't lock, the auto trim operation will be disabled and FREQSEL (SYS_IRC1TCTL[1:0]) will be cleared to 00.
     * |        |          |00 = Trim retry count limitation is 64 loops.
     * |        |          |01 = Trim retry count limitation is 128 loops.
     * |        |          |10 = Trim retry count limitation is 256 loops.
     * |        |          |11 = Trim retry count limitation is 512 loops.
     * |[8]     |CESTOPEN  |Clock Error Stop Enable Bit
     * |        |          |This bit is used to control if stop the HIRC1 trim operation when 32.768 kHz clock error is detected.
     * |        |          |If set this bit high and 32.768 kHz clock error detected, the status CLKERRIF (SYS_IRC1TISTS[2]) would be set high and HIRC1 trim operation was stopped
     * |        |          |If this bit is low and 32.768 kHz clock error detected, the status CLKERRIF (SYS_IRC1TISTS[2]) would be set high and HIRC1 trim operation is continuously.
     * |        |          |0 = The trim operation is keep going if clock is inaccuracy.
     * |        |          |1 = The trim operation is stopped if clock is inaccuracy.
     * @var SYS_T::IRC1TIEN
     * Offset: 0x94  HIRC1 Trim Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1]     |TFAILIEN  |Trim Failure Interrupt Enable Bit
     * |        |          |This bit controls if an interrupt will be triggered while HIRC1 trim value update limitation count reached and HIRC1 frequency still not locked on target frequency set by FREQSEL (SYS_IRC1TCTL[1:0]).
     * |        |          |If this bit is high and TFAILIF (SYS_IRC1TSTS[1]) is set during auto trim operation, an interrupt will be triggered to notify that HIRC1 trim value update limitation count was reached.
     * |        |          |0 = Disable TFAILIF (SYS_IRC1TSTS[1]) status to trigger an interrupt to CPU.
     * |        |          |1 = Enable TFAILIF (SYS_IRC1TSTS[1]) status to trigger an interrupt to CPU.
     * |[2]     |CLKEIEN   |Clock Error Interrupt Enable Bit
     * |        |          |This bit controls if CPU would get an interrupt while clock is inaccuracy during auto trim operation.
     * |        |          |If this bit is set to1, and CLKERRIF (SYS_IRC1TSTS[2]) is set during auto trim operation, an interrupt will be triggered to notify the clock frequency is inaccuracy.
     * |        |          |0 = Disable CLKERRIF (SYS_IRC1TSTS[2]) status to trigger an interrupt to CPU.
     * |        |          |1 = Enable CLKERRIF (SYS_IRC1TSTS[2]) status to trigger an interrupt to CPU.
     * @var SYS_T::IRC1TISTS
     * Offset: 0x98  HIRC1 Trim Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |FREQLOCK  |HIRC1 Frequency Lock Status
     * |        |          |This bit indicates the HIRC1 frequency is locked.
     * |        |          |This is a status bit and doesn't trigger any interrupt.
     * |        |          |0 = The internal high-speed oscillator frequency doesn't lock at 36 MHz yet.
     * |        |          |1 = The internal high-speed oscillator frequency locked at 36 MHz.
     * |[1]     |TFAILIF   |Trim Failure Interrupt Status
     * |        |          |This bit indicates that HIRC1 trim value update limitation count reached and the HIRC1 clock frequency still doesn't be locked
     * |        |          |Once this bit is set, the auto trim operation stopped and FREQSEL (SYS_IRC1TCTL[1:0]) will be cleared to 00 by hardware automatically.
     * |        |          |If this bit is set and TFAILIEN (SYS_IRC1TIEN[1]) is high, an interrupt will be triggered to notify that HIRC1 trim value update limitation count was reached
     * |        |          |Write 1 to clear this to 0.
     * |        |          |0 = Trim value update limitation count does not reach.
     * |        |          |1 = Trim value update limitation count reached and HIRC1 frequency still not locked.
     * |[2]     |CLKERRIF  |Clock Error Interrupt Status
     * |        |          |When the frequency of 32.768 kHz external low speed crystal oscillator (LXT) or 36 MHz internal high speed RC oscillator (HIRC1) is shift larger to unreasonable value, this bit will be set and to be an indicate that clock frequency is inaccuracy
     * |        |          |Once this bit is set to 1, the auto trim operation stopped and FREQSEL (SYS_IRC1TCTL[1:0]) will be cleared to 00 by hardware automatically if CESTOPEN (SYS_IRC1TCTL[8]) is set to 1.
     * |        |          |If this bit is set and CLKEIEN (SYS_IRC1TIEN[2]) is high, an interrupt will be triggered to notify the clock frequency is inaccuracy
     * |        |          |Write 1 to clear this to 0.
     * |        |          |0 = Clock frequency is accuracy.
     * |        |          |1 = Clock frequency is inaccuracy.
     * @var SYS_T::MIRCTCTL
     * Offset: 0xA0  MIRC Trim Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |FREQSEL   |Trim Frequency Selection
     * |        |          |This field indicates the target frequency of 4 MHz internal medium speed RC oscillator (MIRC) auto trim.
     * |        |          |During auto trim operation, if clock error detected with CESTOPEN (SYS_MIRCTCTL[8]) is set to 1 or trim retry limitation count reached, this field will be cleared to 00 automatically.
     * |        |          |00 = Disable MIRC auto trim function.
     * |        |          |01 = Reserved.
     * |        |          |10 = Enable MIRC auto trim function and trim HIRC to 4 MHz.
     * |        |          |11 = Reserved.
     * |        |          |Note: MIRC auto trim cannot work normally at power down mode
     * |        |          |These bits must be cleared before entering power down mode.
     * |[5:4]   |LOOPSEL   |Trim Calculation Loop Selection
     * |        |          |This field defines that trim value calculation is based on how many 32.768 kHz clock.
     * |        |          |00 = Trim value calculation is based on average difference in 4 32.768 kHz clock.
     * |        |          |01 = Trim value calculation is based on average difference in 8 32.768 kHz clock.
     * |        |          |10 = Trim value calculation is based on average difference in 16 32.768 kHz clock.
     * |        |          |11 = Trim value calculation is based on average difference in 32 32.768 kHz clock.
     * |        |          |Note: For example, if LOOPSEL is set as 00, auto trim circuit will calculate trim value based on the average frequency difference in 4 32.768 kHz clock.
     * |[7:6]   |RETRYCNT  |Trim Value Update Limitation Count
     * |        |          |This field defines that how many times the auto trim circuit will try to update the MIRC trim value before the frequency of MIRC locked.
     * |        |          |Once the MIRC locked, the internal trim value update counter will be reset.
     * |        |          |If the trim value update counter reached this limitation value and frequency of MIRC still doesn't lock, the auto trim operation will be disabled and FREQSEL (SYS_MIRCTCTL[1:0]) will be cleared to 00.
     * |        |          |00 = Trim retry count limitation is 64 loops.
     * |        |          |01 = Trim retry count limitation is 128 loops.
     * |        |          |10 = Trim retry count limitation is 256 loops.
     * |        |          |11 = Trim retry count limitation is 512 loops.
     * |[8]     |CESTOPEN  |Clock Error Stop Enable Bit
     * |        |          |This bit is used to control if stop the MIRC trim operation when 32.768 kHz clock error is detected.
     * |        |          |If set this bit high and 32.768 kHz clock error detected, the status CLKERRIF (SYS_MIRCTISTS[2]) would be set high and MIRC trim operation was stopped
     * |        |          |If this bit is low and 32.768 kHz clock error detected, the status CLKERRIF (SYS_MIRCTISTS[2]) would be set high and MIRC trim operation is continuously.
     * |        |          |0 = The trim operation is keep going if clock is inaccuracy.
     * |        |          |1 = The trim operation is stopped if clock is inaccuracy.
     * @var SYS_T::MIRCTIEN
     * Offset: 0xA4  MIRC Trim Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1]     |TFAILIEN  |Trim Failure Interrupt Enable Bit
     * |        |          |This bit controls if an interrupt will be triggered while MIRC trim value update limitation count reached and MIRC frequency still not locked on target frequency set by FREQSEL (SYS_MIRCTCTL[1:0]).
     * |        |          |If this bit is high and TFAILIF (SYS_MIRCTSTS[1]) is set during auto trim operation, an interrupt will be triggered to notify that MIRC trim value update limitation count was reached.
     * |        |          |0 = Disable TFAILIF (SYS_MIRCTSTS[1]) status to trigger an interrupt to CPU.
     * |        |          |1 = Enable TFAILIF (SYS_MIRCTSTS[1]) status to trigger an interrupt to CPU.
     * |[2]     |CLKEIEN   |Clock Error Interrupt Enable Bit
     * |        |          |This bit controls if CPU would get an interrupt while clock is inaccuracy during auto trim operation.
     * |        |          |If this bit is set to1, and CLKERRIF (SYS_MIRCTSTS[2]) is set during auto trim operation, an interrupt will be triggered to notify the clock frequency is inaccuracy.
     * |        |          |0 = Disable CLKERRIF (SYS_MIRCTSTS[2]) status to trigger an interrupt to CPU.
     * |        |          |1 = Enable CLKERRIF (SYS_MIRCTSTS[2]) status to trigger an interrupt to CPU.
     * @var SYS_T::MIRCTISTS
     * Offset: 0xA8  MIRC Trim Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |FREQLOCK  |MIRC Frequency Lock Status
     * |        |          |This bit indicates the MIRC frequency is locked.
     * |        |          |This is a status bit and doesn't trigger any interrupt.
     * |        |          |0 = The internal medium-speed oscillator frequency doesn't lock at 4 MHz yet.
     * |        |          |1 = The internal medium-speed oscillator frequency locked at 4 MHz.
     * |[1]     |TFAILIF   |Trim Failure Interrupt Status
     * |        |          |This bit indicates that MIRC trim value update limitation count reached and the MIRC clock frequency still doesn't be locked
     * |        |          |Once this bit is set, the auto trim operation stopped and FREQSEL (SYS_MIRCTCTL[1:0]) will be cleared to 00 by hardware automatically.
     * |        |          |If this bit is set and TFAILIEN (SYS_MIRCTIEN[1]) is high, an interrupt will be triggered to notify that MIRC trim value update limitation count was reached
     * |        |          |Write 1 to clear this to 0.
     * |        |          |0 = Trim value update limitation count does not reach.
     * |        |          |1 = Trim value update limitation count reached and MIRC frequency still not locked.
     * |[2]     |CLKERRIF  |Clock Error Interrupt Status
     * |        |          |When the frequency of 32.768 kHz external low speed crystal oscillator (LXT) or 4 MHz internal medium speed RC oscillator (MIRC) is shift larger to unreasonable value, this bit will be set and to be an indicate that clock frequency is inaccuracy
     * |        |          |Once this bit is set to 1, the auto trim operation stopped and FREQSEL (SYS_MIRCTCTL[1:0]) will be cleared to 00 by hardware automatically if CESTOPEN (SYS_MIRCTCTL[8]) is set to 1.
     * |        |          |If this bit is set and CLKEIEN (SYS_MIRCTIEN[2]) is high, an interrupt will be triggered to notify the clock frequency is inaccuracy
     * |        |          |Write 1 to clear this to 0.
     * |        |          |0 = Clock frequency is accuracy.
     * |        |          |1 = Clock frequency is inaccuracy.
     * @var SYS_T::REGLCTL
     * Offset: 0x100  Register Lock Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |REGLCTL   |Register Lock Control Code (Write Only)
     * |        |          |Some registers have write-protection function
     * |        |          |Writing these registers have to disable the protected function by writing the sequence value 59h, 16h, 88h to this field
     * |        |          |After this sequence is completed, the REGLCTL bit will be set to 1 and write-protection registers can be normal write.
     * |        |          |Register Lock Control Disable Index (Read Only)
     * |        |          |0 = Write-protection Enabled for writing protected registers
     * |        |          |Any write to the protected register is ignored.
     * |        |          |1 = Write-protection Disabled for writing protected registers.
     * @var SYS_T::RPDBCLK
     * Offset: 0x120  Reset Pin Debounce Clock Selection Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[6]     |RSTPDBCLK |Reset Pin Debounce Clock Selection Bit
     * |        |          |Before switch clock, both clock sources must be enabled.
     * |        |          |0 = MIRC  is selected as reset pin debounce clock.
     * |        |          |1 = HIRC0 is selected as reset pin debounce clock.(default)
     */
    __I  uint32_t PDID;                  /*!< [0x0000] Part Device Identification Number Register                       */
    __IO uint32_t RSTSTS;                /*!< [0x0004] System Reset Status Register                                     */
    __IO uint32_t IPRST1;                /*!< [0x0008] Peripheral Reset Control Resister1                               */
    __IO uint32_t IPRST2;                /*!< [0x000c] Peripheral Reset Control Resister2                               */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE[1];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t MISCCTL;               /*!< [0x0014] Miscellaneous Control Resister                                   */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE0[2];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t TEMPCTL;               /*!< [0x0020] Temperature Sensor Control Register                              */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE1[1];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t RCCFCTL;               /*!< [0x0028] RC Clock Filter Control Register                                 */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE2[1];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t GPA_MFPL;              /*!< [0x0030] GPIOA Low Byte Multiple Function Control Register                */
    __IO uint32_t GPA_MFPH;              /*!< [0x0034] GPIOA High Byte Multiple Function Control Register               */
    __IO uint32_t GPB_MFPL;              /*!< [0x0038] GPIOB Low Byte Multiple Function Control Register                */
    __IO uint32_t GPB_MFPH;              /*!< [0x003c] GPIOB High Byte Multiple Function Control Register               */
    __IO uint32_t GPC_MFPL;              /*!< [0x0040] GPIOC Low Byte Multiple Function Control Register                */
    __IO uint32_t GPC_MFPH;              /*!< [0x0044] GPIOC High Byte Multiple Function Control Register               */
    __IO uint32_t GPD_MFPL;              /*!< [0x0048] GPIOD Low Byte Multiple Function Control Register                */
    __IO uint32_t GPD_MFPH;              /*!< [0x004c] GPIOD High Byte Multiple Function Control Register               */
    __IO uint32_t GPE_MFPL;              /*!< [0x0050] GPIOE Low Byte Multiple Function Control Register                */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE3[1];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t GPF_MFPL;              /*!< [0x0058] GPIOF Low Byte Multiple Function Control Register                */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE4[1];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t PORCTL;                /*!< [0x0060] Power-On-Reset Controller Register                               */
    __IO uint32_t BODCTL;                /*!< [0x0064] Brown-out Detector Controller Register                           */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE5[1];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t IVREFCTL;              /*!< [0x006c] Internal Voltage Reference Control Register                      */
    __IO uint32_t LDOCTL;                /*!< [0x0070] LDO Control Register                                             */
    __IO uint32_t BATDIVCTL;             /*!< [0x0074] Battery Voltage Divider Control Register                         */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE6[1];
    /// @endcond //HIDDEN_SYMBOLS
    __I  uint32_t WKSTS;                 /*!< [0x007c] System Wakeup Status Register                                    */
    __IO uint32_t IRC0TCTL;              /*!< [0x0080] HIRC0 Trim Control Register                                      */
    __IO uint32_t IRC0TIEN;              /*!< [0x0084] HIRC0 Trim Interrupt Enable Register                             */
    __IO uint32_t IRC0TISTS;             /*!< [0x0088] HIRC0 Trim Interrupt Status Register                             */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE7[1];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t IRC1TCTL;              /*!< [0x0090] HIRC1 Trim Control Register                                      */
    __IO uint32_t IRC1TIEN;              /*!< [0x0094] HIRC1 Trim Interrupt Enable Register                             */
    __IO uint32_t IRC1TISTS;             /*!< [0x0098] HIRC1 Trim Interrupt Status Register                             */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE8[1];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t MIRCTCTL;              /*!< [0x00a0] MIRC Trim Control Register                                       */
    __IO uint32_t MIRCTIEN;              /*!< [0x00a4] MIRC Trim Interrupt Enable Register                              */
    __IO uint32_t MIRCTISTS;             /*!< [0x00a8] MIRC Trim Interrupt Status Register                              */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE9[21];
    /// @endcond //HIDDEN_SYMBOLS
    __O  uint32_t REGLCTL;               /*!< [0x0100] Register Lock Control Register                                   */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE10[7];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t RPDBCLK;               /*!< [0x0120] Reset Pin Debounce Clock Selection Register                      */

} SYS_T;

/**
    @addtogroup SYS_CONST SYS Bit Field Definition
    Constant Definitions for SYS Controller
@{ */

#define SYS_PDID_PDID_Pos                (0)                                               /*!< SYS_T::PDID: PDID Position             */
#define SYS_PDID_PDID_Msk                (0xfffffffful << SYS_PDID_PDID_Pos)               /*!< SYS_T::PDID: PDID Mask                 */

#define SYS_RSTSTS_PORF_Pos              (0)                                               /*!< SYS_T::RSTSTS: PORF Position           */
#define SYS_RSTSTS_PORF_Msk              (0x1ul << SYS_RSTSTS_PORF_Pos)                    /*!< SYS_T::RSTSTS: PORF Mask               */

#define SYS_RSTSTS_PINRF_Pos             (1)                                               /*!< SYS_T::RSTSTS: PINRF Position          */
#define SYS_RSTSTS_PINRF_Msk             (0x1ul << SYS_RSTSTS_PINRF_Pos)                   /*!< SYS_T::RSTSTS: PINRF Mask              */

#define SYS_RSTSTS_WDTRF_Pos             (2)                                               /*!< SYS_T::RSTSTS: WDTRF Position          */
#define SYS_RSTSTS_WDTRF_Msk             (0x1ul << SYS_RSTSTS_WDTRF_Pos)                   /*!< SYS_T::RSTSTS: WDTRF Mask              */

#define SYS_RSTSTS_LVRF_Pos              (3)                                               /*!< SYS_T::RSTSTS: LVRF Position           */
#define SYS_RSTSTS_LVRF_Msk              (0x1ul << SYS_RSTSTS_LVRF_Pos)                    /*!< SYS_T::RSTSTS: LVRF Mask               */

#define SYS_RSTSTS_BODRF_Pos             (4)                                               /*!< SYS_T::RSTSTS: BODRF Position          */
#define SYS_RSTSTS_BODRF_Msk             (0x1ul << SYS_RSTSTS_BODRF_Pos)                   /*!< SYS_T::RSTSTS: BODRF Mask              */

#define SYS_RSTSTS_SYSRF_Pos             (5)                                               /*!< SYS_T::RSTSTS: SYSRF Position          */
#define SYS_RSTSTS_SYSRF_Msk             (0x1ul << SYS_RSTSTS_SYSRF_Pos)                   /*!< SYS_T::RSTSTS: SYSRF Mask              */

#define SYS_RSTSTS_CPURF_Pos             (7)                                               /*!< SYS_T::RSTSTS: CPURF Position          */
#define SYS_RSTSTS_CPURF_Msk             (0x1ul << SYS_RSTSTS_CPURF_Pos)                   /*!< SYS_T::RSTSTS: CPURF Mask              */

#define SYS_RSTSTS_LOCKRF_Pos            (8)                                               /*!< SYS_T::RSTSTS: LOCKRF Position         */
#define SYS_RSTSTS_LOCKRF_Msk            (0x1ul << SYS_RSTSTS_LOCKRF_Pos)                  /*!< SYS_T::RSTSTS: LOCKRF Mask             */

#define SYS_IPRST1_CHIPRST_Pos           (0)                                               /*!< SYS_T::IPRST1: CHIPRST Position        */
#define SYS_IPRST1_CHIPRST_Msk           (0x1ul << SYS_IPRST1_CHIPRST_Pos)                 /*!< SYS_T::IPRST1: CHIPRST Mask            */

#define SYS_IPRST1_CPURST_Pos            (1)                                               /*!< SYS_T::IPRST1: CPURST Position         */
#define SYS_IPRST1_CPURST_Msk            (0x1ul << SYS_IPRST1_CPURST_Pos)                  /*!< SYS_T::IPRST1: CPURST Mask             */

#define SYS_IPRST1_PDMARST_Pos           (2)                                               /*!< SYS_T::IPRST1: PDMARST Position        */
#define SYS_IPRST1_PDMARST_Msk           (0x1ul << SYS_IPRST1_PDMARST_Pos)                 /*!< SYS_T::IPRST1: PDMARST Mask            */

#define SYS_IPRST2_GPIORST_Pos           (1)                                               /*!< SYS_T::IPRST2: GPIORST Position        */
#define SYS_IPRST2_GPIORST_Msk           (0x1ul << SYS_IPRST2_GPIORST_Pos)                 /*!< SYS_T::IPRST2: GPIORST Mask            */

#define SYS_IPRST2_TMR0RST_Pos           (2)                                               /*!< SYS_T::IPRST2: TMR0RST Position        */
#define SYS_IPRST2_TMR0RST_Msk           (0x1ul << SYS_IPRST2_TMR0RST_Pos)                 /*!< SYS_T::IPRST2: TMR0RST Mask            */

#define SYS_IPRST2_TMR1RST_Pos           (3)                                               /*!< SYS_T::IPRST2: TMR1RST Position        */
#define SYS_IPRST2_TMR1RST_Msk           (0x1ul << SYS_IPRST2_TMR1RST_Pos)                 /*!< SYS_T::IPRST2: TMR1RST Mask            */

#define SYS_IPRST2_TMR2RST_Pos           (4)                                               /*!< SYS_T::IPRST2: TMR2RST Position        */
#define SYS_IPRST2_TMR2RST_Msk           (0x1ul << SYS_IPRST2_TMR2RST_Pos)                 /*!< SYS_T::IPRST2: TMR2RST Mask            */

#define SYS_IPRST2_TMR3RST_Pos           (5)                                               /*!< SYS_T::IPRST2: TMR3RST Position        */
#define SYS_IPRST2_TMR3RST_Msk           (0x1ul << SYS_IPRST2_TMR3RST_Pos)                 /*!< SYS_T::IPRST2: TMR3RST Mask            */

#define SYS_IPRST2_I2C0RST_Pos           (8)                                               /*!< SYS_T::IPRST2: I2C0RST Position        */
#define SYS_IPRST2_I2C0RST_Msk           (0x1ul << SYS_IPRST2_I2C0RST_Pos)                 /*!< SYS_T::IPRST2: I2C0RST Mask            */

#define SYS_IPRST2_I2C1RST_Pos           (9)                                               /*!< SYS_T::IPRST2: I2C1RST Position        */
#define SYS_IPRST2_I2C1RST_Msk           (0x1ul << SYS_IPRST2_I2C1RST_Pos)                 /*!< SYS_T::IPRST2: I2C1RST Mask            */

#define SYS_IPRST2_SPI0RST_Pos           (12)                                              /*!< SYS_T::IPRST2: SPI0RST Position        */
#define SYS_IPRST2_SPI0RST_Msk           (0x1ul << SYS_IPRST2_SPI0RST_Pos)                 /*!< SYS_T::IPRST2: SPI0RST Mask            */

#define SYS_IPRST2_SPI1RST_Pos           (13)                                              /*!< SYS_T::IPRST2: SPI1RST Position        */
#define SYS_IPRST2_SPI1RST_Msk           (0x1ul << SYS_IPRST2_SPI1RST_Pos)                 /*!< SYS_T::IPRST2: SPI1RST Mask            */

#define SYS_IPRST2_SPI2RST_Pos           (14)                                              /*!< SYS_T::IPRST2: SPI2RST Position        */
#define SYS_IPRST2_SPI2RST_Msk           (0x1ul << SYS_IPRST2_SPI2RST_Pos)                 /*!< SYS_T::IPRST2: SPI2RST Mask            */

#define SYS_IPRST2_SPI3RST_Pos           (15)                                              /*!< SYS_T::IPRST2: SPI3RST Position        */
#define SYS_IPRST2_SPI3RST_Msk           (0x1ul << SYS_IPRST2_SPI3RST_Pos)                 /*!< SYS_T::IPRST2: SPI3RST Mask            */

#define SYS_IPRST2_UART0RST_Pos          (16)                                              /*!< SYS_T::IPRST2: UART0RST Position       */
#define SYS_IPRST2_UART0RST_Msk          (0x1ul << SYS_IPRST2_UART0RST_Pos)                /*!< SYS_T::IPRST2: UART0RST Mask           */

#define SYS_IPRST2_UART1RST_Pos          (17)                                              /*!< SYS_T::IPRST2: UART1RST Position       */
#define SYS_IPRST2_UART1RST_Msk          (0x1ul << SYS_IPRST2_UART1RST_Pos)                /*!< SYS_T::IPRST2: UART1RST Mask           */

#define SYS_IPRST2_PWM0RST_Pos           (20)                                              /*!< SYS_T::IPRST2: PWM0RST Position        */
#define SYS_IPRST2_PWM0RST_Msk           (0x1ul << SYS_IPRST2_PWM0RST_Pos)                 /*!< SYS_T::IPRST2: PWM0RST Mask            */

#define SYS_IPRST2_ACMP0RST_Pos          (22)                                              /*!< SYS_T::IPRST2: ACMP0RST Position       */
#define SYS_IPRST2_ACMP0RST_Msk          (0x1ul << SYS_IPRST2_ACMP0RST_Pos)                /*!< SYS_T::IPRST2: ACMP0RST Mask           */

#define SYS_IPRST2_ADCRST_Pos            (28)                                              /*!< SYS_T::IPRST2: ADCRST Position         */
#define SYS_IPRST2_ADCRST_Msk            (0x1ul << SYS_IPRST2_ADCRST_Pos)                  /*!< SYS_T::IPRST2: ADCRST Mask             */

#define SYS_IPRST2_SC0RST_Pos            (30)                                              /*!< SYS_T::IPRST2: SC0RST Position         */
#define SYS_IPRST2_SC0RST_Msk            (0x1ul << SYS_IPRST2_SC0RST_Pos)                  /*!< SYS_T::IPRST2: SC0RST Mask             */

#define SYS_IPRST2_SC1RST_Pos            (31)                                              /*!< SYS_T::IPRST2: SC1RST Position         */
#define SYS_IPRST2_SC1RST_Msk            (0x1ul << SYS_IPRST2_SC1RST_Pos)                  /*!< SYS_T::IPRST2: SC1RST Mask             */

#define SYS_MISCCTL_POR33DIS_Pos         (6)                                               /*!< SYS_T::MISCCTL: POR33DIS Position      */
#define SYS_MISCCTL_POR33DIS_Msk         (0x1ul << SYS_MISCCTL_POR33DIS_Pos)               /*!< SYS_T::MISCCTL: POR33DIS Mask          */

#define SYS_MISCCTL_POR18DIS_Pos         (7)                                               /*!< SYS_T::MISCCTL: POR18DIS Position      */
#define SYS_MISCCTL_POR18DIS_Msk         (0x1ul << SYS_MISCCTL_POR18DIS_Pos)               /*!< SYS_T::MISCCTL: POR18DIS Mask          */

#define SYS_TEMPCTL_VTEMPEN_Pos          (0)                                               /*!< SYS_T::TEMPCTL: VTEMPEN Position       */
#define SYS_TEMPCTL_VTEMPEN_Msk          (0x1ul << SYS_TEMPCTL_VTEMPEN_Pos)                /*!< SYS_T::TEMPCTL: VTEMPEN Mask           */

#define SYS_RCCFCTL_HIRC0FEN_Pos         (0)                                               /*!< SYS_T::RCCFCTL: HIRC0FEN Position      */
#define SYS_RCCFCTL_HIRC0FEN_Msk         (0x1ul << SYS_RCCFCTL_HIRC0FEN_Pos)               /*!< SYS_T::RCCFCTL: HIRC0FEN Mask          */

#define SYS_RCCFCTL_HIRC1FEN_Pos         (1)                                               /*!< SYS_T::RCCFCTL: HIRC1FEN Position      */
#define SYS_RCCFCTL_HIRC1FEN_Msk         (0x1ul << SYS_RCCFCTL_HIRC1FEN_Pos)               /*!< SYS_T::RCCFCTL: HIRC1FEN Mask          */

#define SYS_RCCFCTL_MRCFEN_Pos           (2)                                               /*!< SYS_T::RCCFCTL: MRCFEN Position        */
#define SYS_RCCFCTL_MRCFEN_Msk           (0x1ul << SYS_RCCFCTL_MRCFEN_Pos)                 /*!< SYS_T::RCCFCTL: MRCFEN Mask            */

#define SYS_GPA_MFPL_PA0MFP_Pos          (0)                                               /*!< SYS_T::GPA_MFPL: PA0MFP Position       */
#define SYS_GPA_MFPL_PA0MFP_Msk          (0xful << SYS_GPA_MFPL_PA0MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA0MFP Mask           */

#define SYS_GPA_MFPL_PA1MFP_Pos          (4)                                               /*!< SYS_T::GPA_MFPL: PA1MFP Position       */
#define SYS_GPA_MFPL_PA1MFP_Msk          (0xful << SYS_GPA_MFPL_PA1MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA1MFP Mask           */

#define SYS_GPA_MFPL_PA2MFP_Pos          (8)                                               /*!< SYS_T::GPA_MFPL: PA2MFP Position       */
#define SYS_GPA_MFPL_PA2MFP_Msk          (0xful << SYS_GPA_MFPL_PA2MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA2MFP Mask           */

#define SYS_GPA_MFPL_PA3MFP_Pos          (12)                                              /*!< SYS_T::GPA_MFPL: PA3MFP Position       */
#define SYS_GPA_MFPL_PA3MFP_Msk          (0xful << SYS_GPA_MFPL_PA3MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA3MFP Mask           */

#define SYS_GPA_MFPL_PA4MFP_Pos          (16)                                              /*!< SYS_T::GPA_MFPL: PA4MFP Position       */
#define SYS_GPA_MFPL_PA4MFP_Msk          (0xful << SYS_GPA_MFPL_PA4MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA4MFP Mask           */

#define SYS_GPA_MFPL_PA5MFP_Pos          (20)                                              /*!< SYS_T::GPA_MFPL: PA5MFP Position       */
#define SYS_GPA_MFPL_PA5MFP_Msk          (0xful << SYS_GPA_MFPL_PA5MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA5MFP Mask           */

#define SYS_GPA_MFPL_PA6MFP_Pos          (24)                                              /*!< SYS_T::GPA_MFPL: PA6MFP Position       */
#define SYS_GPA_MFPL_PA6MFP_Msk          (0xful << SYS_GPA_MFPL_PA6MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA6MFP Mask           */

#define SYS_GPA_MFPH_PA8MFP_Pos          (0)                                               /*!< SYS_T::GPA_MFPH: PA8MFP Position       */
#define SYS_GPA_MFPH_PA8MFP_Msk          (0xful << SYS_GPA_MFPH_PA8MFP_Pos)                /*!< SYS_T::GPA_MFPH: PA8MFP Mask           */

#define SYS_GPA_MFPH_PA9MFP_Pos          (4)                                               /*!< SYS_T::GPA_MFPH: PA9MFP Position       */
#define SYS_GPA_MFPH_PA9MFP_Msk          (0xful << SYS_GPA_MFPH_PA9MFP_Pos)                /*!< SYS_T::GPA_MFPH: PA9MFP Mask           */

#define SYS_GPA_MFPH_PA10MFP_Pos         (8)                                               /*!< SYS_T::GPA_MFPH: PA10MFP Position      */
#define SYS_GPA_MFPH_PA10MFP_Msk         (0xful << SYS_GPA_MFPH_PA10MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA10MFP Mask          */

#define SYS_GPA_MFPH_PA11MFP_Pos         (12)                                              /*!< SYS_T::GPA_MFPH: PA11MFP Position      */
#define SYS_GPA_MFPH_PA11MFP_Msk         (0xful << SYS_GPA_MFPH_PA11MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA11MFP Mask          */

#define SYS_GPA_MFPH_PA12MFP_Pos         (16)                                              /*!< SYS_T::GPA_MFPH: PA12MFP Position      */
#define SYS_GPA_MFPH_PA12MFP_Msk         (0xful << SYS_GPA_MFPH_PA12MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA12MFP Mask          */

#define SYS_GPA_MFPH_PA13MFP_Pos         (20)                                              /*!< SYS_T::GPA_MFPH: PA13MFP Position      */
#define SYS_GPA_MFPH_PA13MFP_Msk         (0xful << SYS_GPA_MFPH_PA13MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA13MFP Mask          */

#define SYS_GPA_MFPH_PA14MFP_Pos         (24)                                              /*!< SYS_T::GPA_MFPH: PA14MFP Position      */
#define SYS_GPA_MFPH_PA14MFP_Msk         (0xful << SYS_GPA_MFPH_PA14MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA14MFP Mask          */

#define SYS_GPA_MFPH_PA15MFP_Pos         (28)                                              /*!< SYS_T::GPA_MFPH: PA15MFP Position      */
#define SYS_GPA_MFPH_PA15MFP_Msk         (0xful << SYS_GPA_MFPH_PA15MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA15MFP Mask          */

#define SYS_GPB_MFPL_PB0MFP_Pos          (0)                                               /*!< SYS_T::GPB_MFPL: PB0MFP Position       */
#define SYS_GPB_MFPL_PB0MFP_Msk          (0xful << SYS_GPB_MFPL_PB0MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB0MFP Mask           */

#define SYS_GPB_MFPL_PB1MFP_Pos          (4)                                               /*!< SYS_T::GPB_MFPL: PB1MFP Position       */
#define SYS_GPB_MFPL_PB1MFP_Msk          (0xful << SYS_GPB_MFPL_PB1MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB1MFP Mask           */

#define SYS_GPB_MFPL_PB2MFP_Pos          (8)                                               /*!< SYS_T::GPB_MFPL: PB2MFP Position       */
#define SYS_GPB_MFPL_PB2MFP_Msk          (0xful << SYS_GPB_MFPL_PB2MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB2MFP Mask           */

#define SYS_GPB_MFPL_PB3MFP_Pos          (12)                                              /*!< SYS_T::GPB_MFPL: PB3MFP Position       */
#define SYS_GPB_MFPL_PB3MFP_Msk          (0xful << SYS_GPB_MFPL_PB3MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB3MFP Mask           */

#define SYS_GPB_MFPL_PB4MFP_Pos          (16)                                              /*!< SYS_T::GPB_MFPL: PB4MFP Position       */
#define SYS_GPB_MFPL_PB4MFP_Msk          (0xful << SYS_GPB_MFPL_PB4MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB4MFP Mask           */

#define SYS_GPB_MFPL_PB5MFP_Pos          (20)                                              /*!< SYS_T::GPB_MFPL: PB5MFP Position       */
#define SYS_GPB_MFPL_PB5MFP_Msk          (0xful << SYS_GPB_MFPL_PB5MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB5MFP Mask           */

#define SYS_GPB_MFPL_PB6MFP_Pos          (24)                                              /*!< SYS_T::GPB_MFPL: PB6MFP Position       */
#define SYS_GPB_MFPL_PB6MFP_Msk          (0xful << SYS_GPB_MFPL_PB6MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB6MFP Mask           */

#define SYS_GPB_MFPL_PB7MFP_Pos          (28)                                              /*!< SYS_T::GPB_MFPL: PB7MFP Position       */
#define SYS_GPB_MFPL_PB7MFP_Msk          (0xful << SYS_GPB_MFPL_PB7MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB7MFP Mask           */

#define SYS_GPB_MFPH_PB8MFP_Pos          (0)                                               /*!< SYS_T::GPB_MFPH: PB8MFP Position       */
#define SYS_GPB_MFPH_PB8MFP_Msk          (0xful << SYS_GPB_MFPH_PB8MFP_Pos)                /*!< SYS_T::GPB_MFPH: PB8MFP Mask           */

#define SYS_GPB_MFPH_PB9MFP_Pos          (4)                                               /*!< SYS_T::GPB_MFPH: PB9MFP Position       */
#define SYS_GPB_MFPH_PB9MFP_Msk          (0xful << SYS_GPB_MFPH_PB9MFP_Pos)                /*!< SYS_T::GPB_MFPH: PB9MFP Mask           */

#define SYS_GPB_MFPH_PB10MFP_Pos         (8)                                               /*!< SYS_T::GPB_MFPH: PB10MFP Position      */
#define SYS_GPB_MFPH_PB10MFP_Msk         (0xful << SYS_GPB_MFPH_PB10MFP_Pos)               /*!< SYS_T::GPB_MFPH: PB10MFP Mask          */

#define SYS_GPB_MFPH_PB11MFP_Pos         (12)                                              /*!< SYS_T::GPB_MFPH: PB11MFP Position      */
#define SYS_GPB_MFPH_PB11MFP_Msk         (0xful << SYS_GPB_MFPH_PB11MFP_Pos)               /*!< SYS_T::GPB_MFPH: PB11MFP Mask          */

#define SYS_GPB_MFPH_PB13MFP_Pos         (20)                                              /*!< SYS_T::GPB_MFPH: PB13MFP Position      */
#define SYS_GPB_MFPH_PB13MFP_Msk         (0xful << SYS_GPB_MFPH_PB13MFP_Pos)               /*!< SYS_T::GPB_MFPH: PB13MFP Mask          */

#define SYS_GPB_MFPH_PB14MFP_Pos         (24)                                              /*!< SYS_T::GPB_MFPH: PB14MFP Position      */
#define SYS_GPB_MFPH_PB14MFP_Msk         (0xful << SYS_GPB_MFPH_PB14MFP_Pos)               /*!< SYS_T::GPB_MFPH: PB14MFP Mask          */

#define SYS_GPB_MFPH_PB15MFP_Pos         (28)                                              /*!< SYS_T::GPB_MFPH: PB15MFP Position      */
#define SYS_GPB_MFPH_PB15MFP_Msk         (0xful << SYS_GPB_MFPH_PB15MFP_Pos)               /*!< SYS_T::GPB_MFPH: PB15MFP Mask          */

#define SYS_GPC_MFPL_PC0MFP_Pos          (0)                                               /*!< SYS_T::GPC_MFPL: PC0MFP Position       */
#define SYS_GPC_MFPL_PC0MFP_Msk          (0xful << SYS_GPC_MFPL_PC0MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC0MFP Mask           */

#define SYS_GPC_MFPL_PC1MFP_Pos          (4)                                               /*!< SYS_T::GPC_MFPL: PC1MFP Position       */
#define SYS_GPC_MFPL_PC1MFP_Msk          (0xful << SYS_GPC_MFPL_PC1MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC1MFP Mask           */

#define SYS_GPC_MFPL_PC2MFP_Pos          (8)                                               /*!< SYS_T::GPC_MFPL: PC2MFP Position       */
#define SYS_GPC_MFPL_PC2MFP_Msk          (0xful << SYS_GPC_MFPL_PC2MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC2MFP Mask           */

#define SYS_GPC_MFPL_PC3MFP_Pos          (12)                                              /*!< SYS_T::GPC_MFPL: PC3MFP Position       */
#define SYS_GPC_MFPL_PC3MFP_Msk          (0xful << SYS_GPC_MFPL_PC3MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC3MFP Mask           */

#define SYS_GPC_MFPL_PC6MFP_Pos          (24)                                              /*!< SYS_T::GPC_MFPL: PC6MFP Position       */
#define SYS_GPC_MFPL_PC6MFP_Msk          (0xful << SYS_GPC_MFPL_PC6MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC6MFP Mask           */

#define SYS_GPC_MFPL_PC7MFP_Pos          (28)                                              /*!< SYS_T::GPC_MFPL: PC7MFP Position       */
#define SYS_GPC_MFPL_PC7MFP_Msk          (0xful << SYS_GPC_MFPL_PC7MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC7MFP Mask           */

#define SYS_GPC_MFPH_PC8MFP_Pos          (0)                                               /*!< SYS_T::GPC_MFPH: PC8MFP Position       */
#define SYS_GPC_MFPH_PC8MFP_Msk          (0xful << SYS_GPC_MFPH_PC8MFP_Pos)                /*!< SYS_T::GPC_MFPH: PC8MFP Mask           */

#define SYS_GPC_MFPH_PC9MFP_Pos          (4)                                               /*!< SYS_T::GPC_MFPH: PC9MFP Position       */
#define SYS_GPC_MFPH_PC9MFP_Msk          (0xful << SYS_GPC_MFPH_PC9MFP_Pos)                /*!< SYS_T::GPC_MFPH: PC9MFP Mask           */

#define SYS_GPC_MFPH_PC10MFP_Pos         (8)                                               /*!< SYS_T::GPC_MFPH: PC10MFP Position      */
#define SYS_GPC_MFPH_PC10MFP_Msk         (0xful << SYS_GPC_MFPH_PC10MFP_Pos)               /*!< SYS_T::GPC_MFPH: PC10MFP Mask          */

#define SYS_GPC_MFPH_PC11MFP_Pos         (12)                                              /*!< SYS_T::GPC_MFPH: PC11MFP Position      */
#define SYS_GPC_MFPH_PC11MFP_Msk         (0xful << SYS_GPC_MFPH_PC11MFP_Pos)               /*!< SYS_T::GPC_MFPH: PC11MFP Mask          */

#define SYS_GPC_MFPH_PC14MFP_Pos         (24)                                              /*!< SYS_T::GPC_MFPH: PC14MFP Position      */
#define SYS_GPC_MFPH_PC14MFP_Msk         (0xful << SYS_GPC_MFPH_PC14MFP_Pos)               /*!< SYS_T::GPC_MFPH: PC14MFP Mask          */

#define SYS_GPC_MFPH_PC15MFP_Pos         (28)                                              /*!< SYS_T::GPC_MFPH: PC15MFP Position      */
#define SYS_GPC_MFPH_PC15MFP_Msk         (0xful << SYS_GPC_MFPH_PC15MFP_Pos)               /*!< SYS_T::GPC_MFPH: PC15MFP Mask          */

#define SYS_GPD_MFPL_PD6MFP_Pos          (24)                                              /*!< SYS_T::GPD_MFPL: PD6MFP Position       */
#define SYS_GPD_MFPL_PD6MFP_Msk          (0xful << SYS_GPD_MFPL_PD6MFP_Pos)                /*!< SYS_T::GPD_MFPL: PD6MFP Mask           */

#define SYS_GPD_MFPL_PD7MFP_Pos          (28)                                              /*!< SYS_T::GPD_MFPL: PD7MFP Position       */
#define SYS_GPD_MFPL_PD7MFP_Msk          (0xful << SYS_GPD_MFPL_PD7MFP_Pos)                /*!< SYS_T::GPD_MFPL: PD7MFP Mask           */

#define SYS_GPD_MFPH_PD14MFP_Pos         (24)                                              /*!< SYS_T::GPD_MFPH: PD14MFP Position      */
#define SYS_GPD_MFPH_PD14MFP_Msk         (0xful << SYS_GPD_MFPH_PD14MFP_Pos)               /*!< SYS_T::GPD_MFPH: PD14MFP Mask          */

#define SYS_GPD_MFPH_PD15MFP_Pos         (28)                                              /*!< SYS_T::GPD_MFPH: PD15MFP Position      */
#define SYS_GPD_MFPH_PD15MFP_Msk         (0x7ul << SYS_GPD_MFPH_PD15MFP_Pos)               /*!< SYS_T::GPD_MFPH: PD15MFP Mask          */

#define SYS_GPE_MFPL_PE5MFP_Pos          (20)                                              /*!< SYS_T::GPE_MFPL: PE5MFP Position       */
#define SYS_GPE_MFPL_PE5MFP_Msk          (0xful << SYS_GPE_MFPL_PE5MFP_Pos)                /*!< SYS_T::GPE_MFPL: PE5MFP Mask           */

#define SYS_GPF_MFPL_PF0MFP_Pos          (0)                                               /*!< SYS_T::GPF_MFPL: PF0MFP Position       */
#define SYS_GPF_MFPL_PF0MFP_Msk          (0xful << SYS_GPF_MFPL_PF0MFP_Pos)                /*!< SYS_T::GPF_MFPL: PF0MFP Mask           */

#define SYS_GPF_MFPL_PF1MFP_Pos          (4)                                               /*!< SYS_T::GPF_MFPL: PF1MFP Position       */
#define SYS_GPF_MFPL_PF1MFP_Msk          (0xful << SYS_GPF_MFPL_PF1MFP_Pos)                /*!< SYS_T::GPF_MFPL: PF1MFP Mask           */

#define SYS_GPF_MFPL_PF2MFP_Pos          (8)                                               /*!< SYS_T::GPF_MFPL: PF2MFP Position       */
#define SYS_GPF_MFPL_PF2MFP_Msk          (0xful << SYS_GPF_MFPL_PF2MFP_Pos)                /*!< SYS_T::GPF_MFPL: PF2MFP Mask           */

#define SYS_GPF_MFPL_PF3MFP_Pos          (12)                                              /*!< SYS_T::GPF_MFPL: PF3MFP Position       */
#define SYS_GPF_MFPL_PF3MFP_Msk          (0xful << SYS_GPF_MFPL_PF3MFP_Pos)                /*!< SYS_T::GPF_MFPL: PF3MFP Mask           */

#define SYS_GPF_MFPL_PF6MFP_Pos          (24)                                              /*!< SYS_T::GPF_MFPL: PF6MFP Position       */
#define SYS_GPF_MFPL_PF6MFP_Msk          (0xful << SYS_GPF_MFPL_PF6MFP_Pos)                /*!< SYS_T::GPF_MFPL: PF6MFP Mask           */

#define SYS_GPF_MFPL_PF7MFP_Pos          (28)                                              /*!< SYS_T::GPF_MFPL: PF7MFP Position       */
#define SYS_GPF_MFPL_PF7MFP_Msk          (0xful << SYS_GPF_MFPL_PF7MFP_Pos)                /*!< SYS_T::GPF_MFPL: PF7MFP Mask           */

#define SYS_PORCTL_POROFF_Pos            (0)                                               /*!< SYS_T::PORCTL: POROFF Position         */
#define SYS_PORCTL_POROFF_Msk            (0xfffful << SYS_PORCTL_POROFF_Pos)               /*!< SYS_T::PORCTL: POROFF Mask             */

#define SYS_BODCTL_BODEN_Pos             (0)                                               /*!< SYS_T::BODCTL: BODEN Position          */
#define SYS_BODCTL_BODEN_Msk             (0x1ul << SYS_BODCTL_BODEN_Pos)                   /*!< SYS_T::BODCTL: BODEN Mask              */

#define SYS_BODCTL_BODIE_Pos             (2)                                               /*!< SYS_T::BODCTL: BODIE Position          */
#define SYS_BODCTL_BODIE_Msk             (0x1ul << SYS_BODCTL_BODIE_Pos)                   /*!< SYS_T::BODCTL: BODIE Mask              */

#define SYS_BODCTL_BODREN_Pos            (3)                                               /*!< SYS_T::BODCTL: BODREN Position         */
#define SYS_BODCTL_BODREN_Msk            (0x1ul << SYS_BODCTL_BODREN_Pos)                  /*!< SYS_T::BODCTL: BODREN Mask             */

#define SYS_BODCTL_BODIF_Pos             (4)                                               /*!< SYS_T::BODCTL: BODIF Position          */
#define SYS_BODCTL_BODIF_Msk             (0x1ul << SYS_BODCTL_BODIF_Pos)                   /*!< SYS_T::BODCTL: BODIF Mask              */

#define SYS_BODCTL_BODOUT_Pos            (6)                                               /*!< SYS_T::BODCTL: BODOUT Position         */
#define SYS_BODCTL_BODOUT_Msk            (0x1ul << SYS_BODCTL_BODOUT_Pos)                  /*!< SYS_T::BODCTL: BODOUT Mask             */

#define SYS_BODCTL_LVREN_Pos             (7)                                               /*!< SYS_T::BODCTL: LVREN Position          */
#define SYS_BODCTL_LVREN_Msk             (0x1ul << SYS_BODCTL_LVREN_Pos)                   /*!< SYS_T::BODCTL: LVREN Mask              */

#define SYS_BODCTL_LPBODEN_Pos           (8)                                               /*!< SYS_T::BODCTL: LPBODEN Position        */
#define SYS_BODCTL_LPBODEN_Msk           (0x1ul << SYS_BODCTL_LPBODEN_Pos)                 /*!< SYS_T::BODCTL: LPBODEN Mask            */

#define SYS_BODCTL_LPBODVL_Pos           (9)                                               /*!< SYS_T::BODCTL: LPBODVL Position        */
#define SYS_BODCTL_LPBODVL_Msk           (0x1ul << SYS_BODCTL_LPBODVL_Pos)                 /*!< SYS_T::BODCTL: LPBODVL Mask            */

#define SYS_BODCTL_LPBODIE_Pos           (10)                                              /*!< SYS_T::BODCTL: LPBODIE Position        */
#define SYS_BODCTL_LPBODIE_Msk           (0x1ul << SYS_BODCTL_LPBODIE_Pos)                 /*!< SYS_T::BODCTL: LPBODIE Mask            */

#define SYS_BODCTL_LPBODREN_Pos          (11)                                              /*!< SYS_T::BODCTL: LPBODREN Position       */
#define SYS_BODCTL_LPBODREN_Msk          (0x1ul << SYS_BODCTL_LPBODREN_Pos)                /*!< SYS_T::BODCTL: LPBODREN Mask           */

#define SYS_BODCTL_BODVL_Pos             (12)                                              /*!< SYS_T::BODCTL: BODVL Position          */
#define SYS_BODCTL_BODVL_Msk             (0xful << SYS_BODCTL_BODVL_Pos)                   /*!< SYS_T::BODCTL: BODVL Mask              */

#define SYS_BODCTL_LPBOD20TRIM_Pos       (16)                                              /*!< SYS_T::BODCTL: LPBOD20TRIM Position    */
#define SYS_BODCTL_LPBOD20TRIM_Msk       (0xful << SYS_BODCTL_LPBOD20TRIM_Pos)             /*!< SYS_T::BODCTL: LPBOD20TRIM Mask        */

#define SYS_BODCTL_LPBOD25TRIM_Pos       (20)                                              /*!< SYS_T::BODCTL: LPBOD25TRIM Position    */
#define SYS_BODCTL_LPBOD25TRIM_Msk       (0xful << SYS_BODCTL_LPBOD25TRIM_Pos)             /*!< SYS_T::BODCTL: LPBOD25TRIM Mask        */

#define SYS_BODCTL_BODDGSEL_Pos          (24)                                              /*!< SYS_T::BODCTL: BODDGSEL Position       */
#define SYS_BODCTL_BODDGSEL_Msk          (0x7ul << SYS_BODCTL_BODDGSEL_Pos)                /*!< SYS_T::BODCTL: BODDGSEL Mask           */

#define SYS_BODCTL_LVRDGSEL_Pos          (28)                                              /*!< SYS_T::BODCTL: LVRDGSEL Position       */
#define SYS_BODCTL_LVRDGSEL_Msk          (0x7ul << SYS_BODCTL_LVRDGSEL_Pos)                /*!< SYS_T::BODCTL: LVRDGSEL Mask           */

#define SYS_IVREFCTL_BGPEN_Pos           (0)                                               /*!< SYS_T::IVREFCTL: BGPEN Position        */
#define SYS_IVREFCTL_BGPEN_Msk           (0x1ul << SYS_IVREFCTL_BGPEN_Pos)                 /*!< SYS_T::IVREFCTL: BGPEN Mask            */

#define SYS_IVREFCTL_REGEN_Pos           (1)                                               /*!< SYS_T::IVREFCTL: REGEN Position        */
#define SYS_IVREFCTL_REGEN_Msk           (0x1ul << SYS_IVREFCTL_REGEN_Pos)                 /*!< SYS_T::IVREFCTL: REGEN Mask            */

#define SYS_IVREFCTL_SEL25_Pos           (2)                                               /*!< SYS_T::IVREFCTL: SEL25 Position        */
#define SYS_IVREFCTL_SEL25_Msk           (0x3ul << SYS_IVREFCTL_SEL25_Pos)                 /*!< SYS_T::IVREFCTL: SEL25 Mask            */

#define SYS_IVREFCTL_EXTMODE_Pos         (4)                                               /*!< SYS_T::IVREFCTL: EXTMODE Position      */
#define SYS_IVREFCTL_EXTMODE_Msk         (0x1ul << SYS_IVREFCTL_EXTMODE_Pos)               /*!< SYS_T::IVREFCTL: EXTMODE Mask          */

#define SYS_IVREFCTL_VREFTRIM_Pos        (8)                                               /*!< SYS_T::IVREFCTL: VREFTRIM Position     */
#define SYS_IVREFCTL_VREFTRIM_Msk        (0xful << SYS_IVREFCTL_VREFTRIM_Pos)              /*!< SYS_T::IVREFCTL: VREFTRIM Mask         */

#define SYS_LDOCTL_FASTWK_Pos            (1)                                               /*!< SYS_T::LDOCTL: FASTWK Position         */
#define SYS_LDOCTL_FASTWK_Msk            (0x1ul << SYS_LDOCTL_FASTWK_Pos)                  /*!< SYS_T::LDOCTL: FASTWK Mask             */

#define SYS_LDOCTL_LDOLVL_Pos            (2)                                               /*!< SYS_T::LDOCTL: LDOLVL Position         */
#define SYS_LDOCTL_LDOLVL_Msk            (0x3ul << SYS_LDOCTL_LDOLVL_Pos)                  /*!< SYS_T::LDOCTL: LDOLVL Mask             */

#define SYS_LDOCTL_LPRMEN_Pos            (4)                                               /*!< SYS_T::LDOCTL: LPRMEN Position         */
#define SYS_LDOCTL_LPRMEN_Msk            (0x1ul << SYS_LDOCTL_LPRMEN_Pos)                  /*!< SYS_T::LDOCTL: LPRMEN Mask             */

#define SYS_LDOCTL_FMCLVEN_Pos           (5)                                               /*!< SYS_T::LDOCTL: FMCLVEN Position        */
#define SYS_LDOCTL_FMCLVEN_Msk           (0x1ul << SYS_LDOCTL_FMCLVEN_Pos)                 /*!< SYS_T::LDOCTL: FMCLVEN Mask            */

#define SYS_BATDIVCTL_BATDIV2EN_Pos      (0)                                               /*!< SYS_T::BATDIVCTL: BATDIV2EN Position   */
#define SYS_BATDIVCTL_BATDIV2EN_Msk      (0x1ul << SYS_BATDIVCTL_BATDIV2EN_Pos)            /*!< SYS_T::BATDIVCTL: BATDIV2EN Mask       */

#define SYS_WKSTS_ACMPWK_Pos             (0)                                               /*!< SYS_T::WKSTS: ACMPWK Position          */
#define SYS_WKSTS_ACMPWK_Msk             (0x1ul << SYS_WKSTS_ACMPWK_Pos)                   /*!< SYS_T::WKSTS: ACMPWK Mask              */

#define SYS_WKSTS_I2C1WK_Pos             (1)                                               /*!< SYS_T::WKSTS: I2C1WK Position          */
#define SYS_WKSTS_I2C1WK_Msk             (0x1ul << SYS_WKSTS_I2C1WK_Pos)                   /*!< SYS_T::WKSTS: I2C1WK Mask              */

#define SYS_WKSTS_I2C0WK_Pos             (2)                                               /*!< SYS_T::WKSTS: I2C0WK Position          */
#define SYS_WKSTS_I2C0WK_Msk             (0x1ul << SYS_WKSTS_I2C0WK_Pos)                   /*!< SYS_T::WKSTS: I2C0WK Mask              */

#define SYS_WKSTS_TMR3WK_Pos             (3)                                               /*!< SYS_T::WKSTS: TMR3WK Position          */
#define SYS_WKSTS_TMR3WK_Msk             (0x1ul << SYS_WKSTS_TMR3WK_Pos)                   /*!< SYS_T::WKSTS: TMR3WK Mask              */

#define SYS_WKSTS_TMR2WK_Pos             (4)                                               /*!< SYS_T::WKSTS: TMR2WK Position          */
#define SYS_WKSTS_TMR2WK_Msk             (0x1ul << SYS_WKSTS_TMR2WK_Pos)                   /*!< SYS_T::WKSTS: TMR2WK Mask              */

#define SYS_WKSTS_TMR1WK_Pos             (5)                                               /*!< SYS_T::WKSTS: TMR1WK Position          */
#define SYS_WKSTS_TMR1WK_Msk             (0x1ul << SYS_WKSTS_TMR1WK_Pos)                   /*!< SYS_T::WKSTS: TMR1WK Mask              */

#define SYS_WKSTS_TMR0WK_Pos             (6)                                               /*!< SYS_T::WKSTS: TMR0WK Position          */
#define SYS_WKSTS_TMR0WK_Msk             (0x1ul << SYS_WKSTS_TMR0WK_Pos)                   /*!< SYS_T::WKSTS: TMR0WK Mask              */

#define SYS_WKSTS_WDTWK_Pos              (7)                                               /*!< SYS_T::WKSTS: WDTWK Position           */
#define SYS_WKSTS_WDTWK_Msk              (0x1ul << SYS_WKSTS_WDTWK_Pos)                    /*!< SYS_T::WKSTS: WDTWK Mask               */

#define SYS_WKSTS_BODWK_Pos              (8)                                               /*!< SYS_T::WKSTS: BODWK Position           */
#define SYS_WKSTS_BODWK_Msk              (0x1ul << SYS_WKSTS_BODWK_Pos)                    /*!< SYS_T::WKSTS: BODWK Mask               */

#define SYS_WKSTS_SPI3WK_Pos             (9)                                               /*!< SYS_T::WKSTS: SPI3WK Position          */
#define SYS_WKSTS_SPI3WK_Msk             (0x1ul << SYS_WKSTS_SPI3WK_Pos)                   /*!< SYS_T::WKSTS: SPI3WK Mask              */

#define SYS_WKSTS_SPI2WK_Pos             (10)                                              /*!< SYS_T::WKSTS: SPI2WK Position          */
#define SYS_WKSTS_SPI2WK_Msk             (0x1ul << SYS_WKSTS_SPI2WK_Pos)                   /*!< SYS_T::WKSTS: SPI2WK Mask              */

#define SYS_WKSTS_SPI1WK_Pos             (11)                                              /*!< SYS_T::WKSTS: SPI1WK Position          */
#define SYS_WKSTS_SPI1WK_Msk             (0x1ul << SYS_WKSTS_SPI1WK_Pos)                   /*!< SYS_T::WKSTS: SPI1WK Mask              */

#define SYS_WKSTS_SPI0WK_Pos             (12)                                              /*!< SYS_T::WKSTS: SPI0WK Position          */
#define SYS_WKSTS_SPI0WK_Msk             (0x1ul << SYS_WKSTS_SPI0WK_Pos)                   /*!< SYS_T::WKSTS: SPI0WK Mask              */

#define SYS_WKSTS_UART1WK_Pos            (13)                                              /*!< SYS_T::WKSTS: UART1WK Position         */
#define SYS_WKSTS_UART1WK_Msk            (0x1ul << SYS_WKSTS_UART1WK_Pos)                  /*!< SYS_T::WKSTS: UART1WK Mask             */

#define SYS_WKSTS_UART0WK_Pos            (14)                                              /*!< SYS_T::WKSTS: UART0WK Position         */
#define SYS_WKSTS_UART0WK_Msk            (0x1ul << SYS_WKSTS_UART0WK_Pos)                  /*!< SYS_T::WKSTS: UART0WK Mask             */

#define SYS_WKSTS_RTCWK_Pos              (15)                                              /*!< SYS_T::WKSTS: RTCWK Position           */
#define SYS_WKSTS_RTCWK_Msk              (0x1ul << SYS_WKSTS_RTCWK_Pos)                    /*!< SYS_T::WKSTS: RTCWK Mask               */

#define SYS_WKSTS_GPIOWK_Pos             (16)                                              /*!< SYS_T::WKSTS: GPIOWK Position          */
#define SYS_WKSTS_GPIOWK_Msk             (0x1ul << SYS_WKSTS_GPIOWK_Pos)                   /*!< SYS_T::WKSTS: GPIOWK Mask              */

#define SYS_IRC0TCTL_FREQSEL_Pos         (0)                                               /*!< SYS_T::IRC0TCTL: FREQSEL Position      */
#define SYS_IRC0TCTL_FREQSEL_Msk         (0x7ul << SYS_IRC0TCTL_FREQSEL_Pos)               /*!< SYS_T::IRC0TCTL: FREQSEL Mask          */

#define SYS_IRC0TCTL_LOOPSEL_Pos         (4)                                               /*!< SYS_T::IRC0TCTL: LOOPSEL Position      */
#define SYS_IRC0TCTL_LOOPSEL_Msk         (0x3ul << SYS_IRC0TCTL_LOOPSEL_Pos)               /*!< SYS_T::IRC0TCTL: LOOPSEL Mask          */

#define SYS_IRC0TCTL_RETRYCNT_Pos        (6)                                               /*!< SYS_T::IRC0TCTL: RETRYCNT Position     */
#define SYS_IRC0TCTL_RETRYCNT_Msk        (0x3ul << SYS_IRC0TCTL_RETRYCNT_Pos)              /*!< SYS_T::IRC0TCTL: RETRYCNT Mask         */

#define SYS_IRC0TCTL_CESTOPEN_Pos        (8)                                               /*!< SYS_T::IRC0TCTL: CESTOPEN Position     */
#define SYS_IRC0TCTL_CESTOPEN_Msk        (0x1ul << SYS_IRC0TCTL_CESTOPEN_Pos)              /*!< SYS_T::IRC0TCTL: CESTOPEN Mask         */

#define SYS_IRC0TIEN_TFAILIEN_Pos        (1)                                               /*!< SYS_T::IRC0TIEN: TFAILIEN Position     */
#define SYS_IRC0TIEN_TFAILIEN_Msk        (0x1ul << SYS_IRC0TIEN_TFAILIEN_Pos)              /*!< SYS_T::IRC0TIEN: TFAILIEN Mask         */

#define SYS_IRC0TIEN_CLKEIEN_Pos         (2)                                               /*!< SYS_T::IRC0TIEN: CLKEIEN Position      */
#define SYS_IRC0TIEN_CLKEIEN_Msk         (0x1ul << SYS_IRC0TIEN_CLKEIEN_Pos)               /*!< SYS_T::IRC0TIEN: CLKEIEN Mask          */

#define SYS_IRC0TISTS_FREQLOCK_Pos       (0)                                               /*!< SYS_T::IRC0TISTS: FREQLOCK Position    */
#define SYS_IRC0TISTS_FREQLOCK_Msk       (0x1ul << SYS_IRC0TISTS_FREQLOCK_Pos)             /*!< SYS_T::IRC0TISTS: FREQLOCK Mask        */

#define SYS_IRC0TISTS_TFAILIF_Pos        (1)                                               /*!< SYS_T::IRC0TISTS: TFAILIF Position     */
#define SYS_IRC0TISTS_TFAILIF_Msk        (0x1ul << SYS_IRC0TISTS_TFAILIF_Pos)              /*!< SYS_T::IRC0TISTS: TFAILIF Mask         */

#define SYS_IRC0TISTS_CLKERRIF_Pos       (2)                                               /*!< SYS_T::IRC0TISTS: CLKERRIF Position    */
#define SYS_IRC0TISTS_CLKERRIF_Msk       (0x1ul << SYS_IRC0TISTS_CLKERRIF_Pos)             /*!< SYS_T::IRC0TISTS: CLKERRIF Mask        */

#define SYS_IRC1TCTL_FREQSEL_Pos         (0)                                               /*!< SYS_T::IRC1TCTL: FREQSEL Position      */
#define SYS_IRC1TCTL_FREQSEL_Msk         (0x3ul << SYS_IRC1TCTL_FREQSEL_Pos)               /*!< SYS_T::IRC1TCTL: FREQSEL Mask          */

#define SYS_IRC1TCTL_LOOPSEL_Pos         (4)                                               /*!< SYS_T::IRC1TCTL: LOOPSEL Position      */
#define SYS_IRC1TCTL_LOOPSEL_Msk         (0x3ul << SYS_IRC1TCTL_LOOPSEL_Pos)               /*!< SYS_T::IRC1TCTL: LOOPSEL Mask          */

#define SYS_IRC1TCTL_RETRYCNT_Pos        (6)                                               /*!< SYS_T::IRC1TCTL: RETRYCNT Position     */
#define SYS_IRC1TCTL_RETRYCNT_Msk        (0x3ul << SYS_IRC1TCTL_RETRYCNT_Pos)              /*!< SYS_T::IRC1TCTL: RETRYCNT Mask         */

#define SYS_IRC1TCTL_CESTOPEN_Pos        (8)                                               /*!< SYS_T::IRC1TCTL: CESTOPEN Position     */
#define SYS_IRC1TCTL_CESTOPEN_Msk        (0x1ul << SYS_IRC1TCTL_CESTOPEN_Pos)              /*!< SYS_T::IRC1TCTL: CESTOPEN Mask         */

#define SYS_IRC1TIEN_TFAILIEN_Pos        (1)                                               /*!< SYS_T::IRC1TIEN: TFAILIEN Position     */
#define SYS_IRC1TIEN_TFAILIEN_Msk        (0x1ul << SYS_IRC1TIEN_TFAILIEN_Pos)              /*!< SYS_T::IRC1TIEN: TFAILIEN Mask         */

#define SYS_IRC1TIEN_CLKEIEN_Pos         (2)                                               /*!< SYS_T::IRC1TIEN: CLKEIEN Position      */
#define SYS_IRC1TIEN_CLKEIEN_Msk         (0x1ul << SYS_IRC1TIEN_CLKEIEN_Pos)               /*!< SYS_T::IRC1TIEN: CLKEIEN Mask          */

#define SYS_IRC1TISTS_FREQLOCK_Pos       (0)                                               /*!< SYS_T::IRC1TISTS: FREQLOCK Position    */
#define SYS_IRC1TISTS_FREQLOCK_Msk       (0x1ul << SYS_IRC1TISTS_FREQLOCK_Pos)             /*!< SYS_T::IRC1TISTS: FREQLOCK Mask        */

#define SYS_IRC1TISTS_TFAILIF_Pos        (1)                                               /*!< SYS_T::IRC1TISTS: TFAILIF Position     */
#define SYS_IRC1TISTS_TFAILIF_Msk        (0x1ul << SYS_IRC1TISTS_TFAILIF_Pos)              /*!< SYS_T::IRC1TISTS: TFAILIF Mask         */

#define SYS_IRC1TISTS_CLKERRIF_Pos       (2)                                               /*!< SYS_T::IRC1TISTS: CLKERRIF Position    */
#define SYS_IRC1TISTS_CLKERRIF_Msk       (0x1ul << SYS_IRC1TISTS_CLKERRIF_Pos)             /*!< SYS_T::IRC1TISTS: CLKERRIF Mask        */

#define SYS_MIRCTCTL_FREQSEL_Pos         (0)                                               /*!< SYS_T::MIRCTCTL: FREQSEL Position      */
#define SYS_MIRCTCTL_FREQSEL_Msk         (0x3ul << SYS_MIRCTCTL_FREQSEL_Pos)               /*!< SYS_T::MIRCTCTL: FREQSEL Mask          */

#define SYS_MIRCTCTL_LOOPSEL_Pos         (4)                                               /*!< SYS_T::MIRCTCTL: LOOPSEL Position      */
#define SYS_MIRCTCTL_LOOPSEL_Msk         (0x3ul << SYS_MIRCTCTL_LOOPSEL_Pos)               /*!< SYS_T::MIRCTCTL: LOOPSEL Mask          */

#define SYS_MIRCTCTL_RETRYCNT_Pos        (6)                                               /*!< SYS_T::MIRCTCTL: RETRYCNT Position     */
#define SYS_MIRCTCTL_RETRYCNT_Msk        (0x3ul << SYS_MIRCTCTL_RETRYCNT_Pos)              /*!< SYS_T::MIRCTCTL: RETRYCNT Mask         */

#define SYS_MIRCTCTL_CESTOPEN_Pos        (8)                                               /*!< SYS_T::MIRCTCTL: CESTOPEN Position     */
#define SYS_MIRCTCTL_CESTOPEN_Msk        (0x1ul << SYS_MIRCTCTL_CESTOPEN_Pos)              /*!< SYS_T::MIRCTCTL: CESTOPEN Mask         */

#define SYS_MIRCTIEN_TFAILIEN_Pos        (1)                                               /*!< SYS_T::MIRCTIEN: TFAILIEN Position     */
#define SYS_MIRCTIEN_TFAILIEN_Msk        (0x1ul << SYS_MIRCTIEN_TFAILIEN_Pos)              /*!< SYS_T::MIRCTIEN: TFAILIEN Mask         */

#define SYS_MIRCTIEN_CLKEIEN_Pos         (2)                                               /*!< SYS_T::MIRCTIEN: CLKEIEN Position      */
#define SYS_MIRCTIEN_CLKEIEN_Msk         (0x1ul << SYS_MIRCTIEN_CLKEIEN_Pos)               /*!< SYS_T::MIRCTIEN: CLKEIEN Mask          */

#define SYS_MIRCTISTS_FREQLOCK_Pos       (0)                                               /*!< SYS_T::MIRCTISTS: FREQLOCK Position    */
#define SYS_MIRCTISTS_FREQLOCK_Msk       (0x1ul << SYS_MIRCTISTS_FREQLOCK_Pos)             /*!< SYS_T::MIRCTISTS: FREQLOCK Mask        */

#define SYS_MIRCTISTS_TFAILIF_Pos        (1)                                               /*!< SYS_T::MIRCTISTS: TFAILIF Position     */
#define SYS_MIRCTISTS_TFAILIF_Msk        (0x1ul << SYS_MIRCTISTS_TFAILIF_Pos)              /*!< SYS_T::MIRCTISTS: TFAILIF Mask         */

#define SYS_MIRCTISTS_CLKERRIF_Pos       (2)                                               /*!< SYS_T::MIRCTISTS: CLKERRIF Position    */
#define SYS_MIRCTISTS_CLKERRIF_Msk       (0x1ul << SYS_MIRCTISTS_CLKERRIF_Pos)             /*!< SYS_T::MIRCTISTS: CLKERRIF Mask        */

#define SYS_REGLCTL_REGLCTL_Pos          (0)                                               /*!< SYS_T::REGLCTL: REGLCTL Position       */
#define SYS_REGLCTL_REGLCTL_Msk          (0x1ul << SYS_REGLCTL_REGLCTL_Pos)                /*!< SYS_T::REGLCTL: REGLCTL Mask           */

#define SYS_RPDBCLK_RSTPDBCLK_Pos        (6)                                               /*!< SYS_T::RPDBCLK: RSTPDBCLK Position     */
#define SYS_RPDBCLK_RSTPDBCLK_Msk        (0x1ul << SYS_RPDBCLK_RSTPDBCLK_Pos)              /*!< SYS_T::RPDBCLK: RSTPDBCLK Mask         */

/**@}*/ /* SYS_CONST */
/**@}*/ /* end of SYS register group */


/*---------------------- System Clock Controller -------------------------*/
/**
    @addtogroup CLK System Clock Controller(CLK)
    Memory Mapped Structure for CLK Controller
@{ */

typedef struct
{


    /**
     * @var CLK_T::PWRCTL
     * Offset: 0x00  System Power-down Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |HXTEN     |HXT Enable Bit (Write Protect)
     * |        |          |0 = 4~32 MHz external high speed crystal (HXT) Disabled.
     * |        |          |1 = 4~32 MHz external high speed crystal (HXT) Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[1]     |LXTEN     |LXT Enable Bit (Write Protect)
     * |        |          |0 = 32.768 kHz external low speed crystal (LXT) Disabled.
     * |        |          |1 = 32.768 kHz external low speed crystal (LXT) Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[2]     |HIRC0EN   |HIRC0 Enable Bit (Write Protect)
     * |        |          |0 = 12~16 MHz internal high speed RC oscillator (HIRC0) Disabled.
     * |        |          |1 = 12~16 MHz internal high speed RC oscillator (HIRC0) Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[3]     |LIRCEN    |LIRC Enable Bit (Write Protect)
     * |        |          |0 = 10 kHz internal low speed RC oscillator (LIRC) Disabled.
     * |        |          |1 = 10 kHz internal low speed RC oscillator (LIRC) Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[4]     |PDWKDLY   |Enable the Wake-up Delay Counter (Write Protect)
     * |        |          |When the chip wakes up from Power-down mode, the clock control will delay 4096 clock cycles to wait system clock stable when chip works at 4~32 MHz external high speed crystal oscillator (HXT).
     * |        |          |0 = Clock cycles delay Disabled.
     * |        |          |1 = Clock cycles delay Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[5]     |PDWKIEN   |Power-Down Mode Wake-Up Interrupt Enable Bit (Write Protect)
     * |        |          |0 = Power-down mode wake-up interrupt Disabled.
     * |        |          |1 = Power-down mode wake-up interrupt Enabled.
     * |        |          |Note1: The interrupt (EINT0~1, GPIO,, UART0~1, WDT, ACMP01, BOD, RTC, TMR0~3, I2C0~1 or SPI0 ~3 )will occur when PDWKIEN are high.
     * |        |          |Note2: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[6]     |PDEN      |System Power-Down Enable (Write Protect)
     * |        |          |When this bit is set to 1, Power-down mode is enabled and chip Power-down behavior will depend on the PDWTCPU bit.
     * |        |          |(a) If the PDWTCPU is 0, then the chip enters Power-down mode immediately after the PDEN bit set
     * |        |          |(default)
     * |        |          |(b) if the PDWTCPU is 1, then the chip keeps active till the CPU sleep mode is also active and then the chip enters Power-down mode.
     * |        |          |When chip wakes up from Power-down mode, this bit is auto cleared
     * |        |          |Users need to set this bit again for next Power-down.
     * |        |          |In Power-down mode, HXT and the HIRC will be disabled in this mode, but LXT and LIRC are not controlled by Power-down mode.
     * |        |          |In Power-down mode, the PLL and system clock are disabled, and ignored the clock source selection
     * |        |          |The clocks of peripheral are not controlled by Power-down mode, if the peripheral clock source is from LXT or LIRC.
     * |        |          |0 = Chip operating normally or chip in idle mode because of WFI command.
     * |        |          |1 = Chip enters Power-down mode instant or wait CPU sleep command WFI.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[8]     |HXTSLTYP  |HXT Mode Selection (Write Protect)
     * |        |          |0 = High frequency crystal loop back path Disabled. It is used for external oscillator.
     * |        |          |1 = High frequency crystal loop back path Enabled. It is used for external crystal.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[12:10] |HXTGAIN   |HXT Gain Control Bit (Write Protect)
     * |        |          |Gain control is used to enlarge the gain of crystal to make sure crystal wok normally
     * |        |          |If gain control is enabled, crystal will consume more power than gain control off.
     * |        |          |000= HXT frequency is lower than from 4 MHz.
     * |        |          |001 = HXT frequency is from 4 MHz to 8 MHz.
     * |        |          |010 = HXT frequency is from 8 MHz to 12 MHz.
     * |        |          |011= HXT frequency is from 12 MHz to 16 MHz.
     * |        |          |100 = HXT frequency is from 16 MHz to 24 MHz.
     * |        |          |101 = HXT frequency is from 24 MHz to 32 MHz.
     * |        |          |110 = HXT frequency is from 32 MHz to 36 MHz.
     * |        |          |111 = HXT frequency is higher than 36 MHz.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[13]    |HIRC0FSEL |HIRC0 Output Frequency Select Bit
     * |        |          |0 = HIRC0 will output 12MHz clock.
     * |        |          |1 = HIRC0 will output 16MHz Clock.
     * |[14]    |HIRC0FSTOP|HIRC0 Stop Output When Frequency Changes (Write Protect)
     * |        |          |0 = HIRC0 will continue to output when HIRC frequency changes.
     * |        |          |1 = HIRC0 will suppress to output during first 16 clocks when HIRC frequency change.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[24]    |HIRC1EN   |HIRC1 Enable Bit (Write Protect)
     * |        |          |0 = 36 MHz internal high speed RC oscillator (HIRC1) Disabled.
     * |        |          |1 = 36 MHz internal high speed RC oscillator (HIRC1) Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[25]    |MIRCEN    |MIRC Enable Bit (Write Protect)
     * |        |          |0 = 4 MHz internal medium speed RC oscillator (MIRC) Disabled.
     * |        |          |1 = 4 MHz internal medium speed RC oscillator (MIRC) Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * @var CLK_T::AHBCLK
     * Offset: 0x04  AHB Devices Clock Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |GPIOCKEN  |GPIO Controller Clock Enable Control
     * |        |          |0 = GPIO peripheral clock Disabled.
     * |        |          |1 = GPIO peripheral clock Enabled.
     * |[1]     |PDMACKEN  |PDMA Controller Clock Enable Bit
     * |        |          |0 = PDMA peripheral clock Disabled.
     * |        |          |1 = PDMA peripheral clock Enabled.
     * |[2]     |ISPCKEN   |Flash ISP Controller Clock Enable Bit
     * |        |          |0 = Flash ISP peripheral clock Disabled.
     * |        |          |1 = Flash ISP peripheral clock Enabled.
     * |[4]     |SRAMCKEN  |SRAM Controller Clock Enable Control Bit
     * |        |          |0 = SRAM peripheral clock Disabled.
     * |        |          |1 = SRAM peripheral clock Enabled.
     * |[5]     |STCKEN    |System Tick Clock Enable Control Bit
     * |        |          |0 = System Tick Clock Disabled.
     * |        |          |1 = System Tick Clock Enabled.
     * @var CLK_T::APBCLK
     * Offset: 0x08  APB Devices Clock Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |WDTCKEN   |Watchdog Timer Clock Enable Control
     * |        |          |This is a protected register. Please refer to open lock sequence to program it.
     * |        |          |This bit is used to control the WDT APB clock only, The WDT engine Clock Source is from LIRC.
     * |        |          |0 = Watchdog Timer Clock Disabled.
     * |        |          |1 = Watchdog Timer Clock Enabled.
     * |[1]     |RTCCKEN   |Real-time-clock Clock Enable Control
     * |        |          |This bit is used to control the RTC APB clock only, The RTC engine Clock Source is from LXT.
     * |        |          |0 = Real-time-clock Clock Disabled.
     * |        |          |1 = Real-time-clock Clock Enabled.
     * |[2]     |TMR0CKEN  |Timer0 Clock Enable Control
     * |        |          |0 = Timer0 Clock Disabled.
     * |        |          |1 = Timer0 Clock Enabled.
     * |[3]     |TMR1CKEN  |Timer1 Clock Enable Control
     * |        |          |0 = Timer1 Clock Disabled.
     * |        |          |1 = Timer1 Clock Enabled.
     * |[4]     |TMR2CKEN  |Timer2 Clock Enable Control
     * |        |          |0 = Timer2 Clock Disabled.
     * |        |          |1 = Timer2 Clock Enabled.
     * |[5]     |TMR3CKEN  |Timer3 Clock Enable Control
     * |        |          |0 = Timer3 Clock Disabled.
     * |        |          |1 = Timer3 Clock Enabled.
     * |[6]     |CLKOCKEN  |ClocK Output Clock Enable Control
     * |        |          |0 = Clock Output Clock Disabled.
     * |        |          |1 = Clock Output Clock Enabled.
     * |[8]     |I2C0CKEN  |I2C0 Clock Enable Control
     * |        |          |0 = I2C0 Clock Disabled.
     * |        |          |1 = I2C0 Clock Enabled.
     * |[9]     |I2C1CKEN  |I2C1 Clock Enable Control
     * |        |          |0 = I2C1 Clock Disabled.
     * |        |          |1 = I2C1 Clock Enabled.
     * |[11]    |ACMP0CKEN |ACMP0 Clock Enable Control
     * |        |          |0 = ACMP0 Clock Disabled.
     * |        |          |1 = ACMP0 Clock Enabled.
     * |[12]    |SPI0CKEN  |SPI0 Clock Enable Control
     * |        |          |0 = SPI0 Clock Disabled.
     * |        |          |1 = SPI0 Clock Enabled.
     * |[13]    |SPI1CKEN  |SPI1 Clock Enable Control
     * |        |          |0 = SPI1 Clock Disabled.
     * |        |          |1 = SPI1 Clock Enabled.
     * |[14]    |SPI2CKEN  |SPI2 Clock Enable Control
     * |        |          |0 = SPI2 Clock Disabled.
     * |        |          |1 = SPI2 Clock Enabled.
     * |[15]    |SPI3CKEN  |SPI3 Clock Enable Control
     * |        |          |0 = SPI3 Clock Disabled.
     * |        |          |1 = SPI3 Clock Enabled.
     * |[16]    |UART0CKEN |UART0 Clock Enable Control
     * |        |          |0 = UART0 Clock Disabled.
     * |        |          |1 = UART0 Clock Enabled.
     * |[17]    |UART1CKEN |UART1 Clock Enable Control
     * |        |          |0 = UART1 Clock Disabled.
     * |        |          |1 = UART1 Clock Enabled.
     * |[20]    |PWM0CKEN  |PWM0 Clock Enable Control
     * |        |          |0 = PWM0 Clock Disabled.
     * |        |          |1 = PWM0 Clock Enabled.
     * |[28]    |ADCCKEN   |Analog-digital-converter (ADC) Clock Enable Control
     * |        |          |0 = ADC Clock Disabled.
     * |        |          |1 = ADC Clock Enabled.
     * |[30]    |SC0CKEN   |SmartCard 0 Clock Enable Control
     * |        |          |0 = SmartCard 0 Clock Disabled.
     * |        |          |1 = SmartCard 0 Clock Enabled.
     * |[31]    |SC1CKEN   |SmartCard 1 Clock Enable Control
     * |        |          |0 = SmartCard 1 Clock Disabled.
     * |        |          |1 = SmartCard 1 Clock Enabled.
     * @var CLK_T::STATUS
     * Offset: 0x0C  Clock status monitor Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |HXTSTB    |HXT Clock Source Stable Flag (Read Only)
     * |        |          |0 = 4~36 MHz external high speed crystal oscillator (HXT) clock is not stable or disabled.
     * |        |          |1 = 4~36 MHz external high speed crystal oscillator (HXT) clock is stable and enabled.
     * |[1]     |LXTSTB    |LXT Clock Source Stable Flag (Read Only)
     * |        |          |0 = 32.768 kHz external low speed crystal oscillator (LXT) clock is not stable or disabled.
     * |        |          |1 = 32.768 kHz external low speed crystal oscillator (LXT) clock is stabled and enabled.
     * |[2]     |PLLSTB    |Internal PLL Clock Source Stable Flag (Read Only)
     * |        |          |0 = Internal PLL clock is not stable or disabled.
     * |        |          |1 = Internal PLL clock is stable and enabled.
     * |[3]     |LIRCSTB   |LIRC Clock Source Stable Flag (Read Only)
     * |        |          |0 = 10 kHz internal low speed RC oscillator (LIRC) clock is not stable or disabled.
     * |        |          |1 = 10 kHz internal low speed RC oscillator (LIRC) clock is stable and enabled.
     * |[4]     |HIRC0STB  |HIRC0 Clock Source Stable Flag (Read Only)
     * |        |          |0 = 12~16 MHz internal high speed RC oscillator (HIRC0) clock is not stable or disabled.
     * |        |          |1 = 12~16 MHz internal high speed RC oscillator (HIRC0) clock is stable and enabled.
     * |[5]     |HIRC1STB  |HIRC Clock Source Stable Flag (Read Only)
     * |        |          |0 = 36 MHz internal high speed RC oscillator (HIRC1) clock is not stable or disabled.
     * |        |          |1 = 36 MHz internal high speed RC oscillator (HIRC1) clock is stable and enabled.
     * |[6]     |MIRCSTB   |MIRC Clock Source Stable Flag (Read Only)
     * |        |          |0 = 4 MHz internal medium speed RC oscillator (MIRC) clock is not stable or disabled.
     * |        |          |1 = 4 MHz internal medium speed RC oscillator (MIRC) clock is stable and enabled.
     * |[7]     |CLKSFAIL  |Clock Switching Fail Flag (Read Only)
     * |        |          |This bit is updated when software switches system clock source
     * |        |          |If switch target clock is stable, this bit will be set to 0
     * |        |          |If switch target clock is not stable, this bit will be set to 1.
     * |        |          |0 = Clock switching success.
     * |        |          |1 = Clock switching failure.
     * @var CLK_T::CLKSEL0
     * Offset: 0x10  Clock Source Select Control Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |HCLKSEL   |HCLK Clock Source Selection (Write Protect)
     * |        |          |Before clock switching, the related clock sources (both pre-select and new-select) must be turned on.
     * |        |          |000 = Clock source from HXT.
     * |        |          |001 = Clock source from LXT.
     * |        |          |010 = Clock source from PLL.
     * |        |          |011 = Clock source from LIRC.
     * |        |          |100= Clock source from HIRC1 or HIRC0 depend on HIRCSEL(CLK_CLKSEL0[3]) setting.
     * |        |          |Others = Clock source from MIRC.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[3]     |HIRCSEL   |HIRC Source Selection
     * |        |          |0 = Clock source from HIRC0 (12~16MHz).
     * |        |          |1 = Clock source from HIRC1 (36MHz).
     * |[4]     |ISPSEL    |ISP Clock Source Selection
     * |        |          |0 = Clock source from HIRC1 or HIRC0 depend on HIRCSEL(CLK_CLKSEL0[3]) setting.
     * |        |          |1 = Clock source from MIRC.
     * @var CLK_T::CLKSEL1
     * Offset: 0x14  Clock Source Select Control Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |UART0SEL  |UART0 Clock Source Selection
     * |        |          |000 = Clock source from 4~32 MHz external high speed crystal oscillator (HXT).
     * |        |          |001 = Clock source from 32.768 kHz external low speed crystal oscillator (LXT).
     * |        |          |010 = Clock source from PLL.
     * |        |          |011 = Clock source from 36 MHz internal high speed RC oscillator (HIRC1) or HIRC0 depend on HIRCSEL(CLK_CLKSEL0[3]) setting.
     * |        |          |Others = Clock source from 4 MHz internal medium speed RC oscillator (MIRC).
     * |[4]     |PWM0SEL   |PWM0 Clock Source Selection
     * |        |          |0 = Clock source from PLL.
     * |        |          |1 = Clock source from PCLK0.
     * |[10:8]  |TMR0SEL   |Timer0 Clock Source Selection
     * |        |          |000 = Clock source from 4~32 MHz external high speed crystal oscillator (HXT).
     * |        |          |001 = Clock source from 32.768 kHz external low speed crystal oscillator (LXT).
     * |        |          |010 = Clock source from 10 kHz internal low speed RC oscillator (LIRC).
     * |        |          |011 = Clock source from external clock pin.
     * |        |          |100 = Clock source from 36 MHz internal high speed RC oscillator (HIRC1) or HIRC0 depend on HIRCSEL(CLK_CLKSEL0[3]) setting.
     * |        |          |101 = Clock source from 4 MHz internal medium speed RC oscillator (MIRC).
     * |        |          |Others = Clock source from HCLK.
     * |[14:12] |TMR1SEL   |Timer1 Clock Source Selection
     * |        |          |000 = Clock source from 4~32 MHz external high speed crystal oscillator (HXT).
     * |        |          |001 = Clock source from 32.768 kHz external low speed crystal oscillator (LXT).
     * |        |          |010 = Clock source from 10 kHz internal low speed RC oscillator (LIRC).
     * |        |          |011 = Clock source from external clock pin.
     * |        |          |100 = Clock source from 36 MHz internal high speed RC oscillator (HIRC1) or HIRC0 depend on HIRCSEL(CLK_CLKSEL0[3]) setting.
     * |        |          |101 = Clock source from 4 MHz internal medium speed RC oscillator (MIRC).
     * |        |          |Others = Clock source from HCLK.
     * |[21:19] |ADCSEL    |ADC Clock Source Selection
     * |        |          |000 = Clock source from 4~32 MHz external high speed crystal oscillator (HXT).
     * |        |          |001 = Clock source from 32.768 kHz external low speed crystal oscillator (LXT).
     * |        |          |010 = Clock source from PLL.
     * |        |          |011 = Clock source from 36 MHz internal high speed RC oscillator (HIRC1) or HIRC0 depend on HIRCSEL(CLK_CLKSEL0[3]) setting.
     * |        |          |100 = Clock source from 4 MHz internal medium speed RC oscillator (MIRC).
     * |        |          |Others = Clock source from HCLK.
     * |[25:24] |SPI0SEL   |SPI0 Clock Source Selection
     * |        |          |00 = Clock source from PLL.
     * |        |          |01 = Clock source from HCLK.
     * |        |          |10 = Clock source from 4~32 MHz external high speed crystal oscillator (HXT).
     * |        |          |11 = Clock source from 36 MHz internal high speed RC oscillator (HIRC1) or HIRC0 depend on HIRCSEL(CLK_CLKSEL0[3]) setting.
     * |[27:26] |SPI2SEL   |SPI2 Clock Source Selection
     * |        |          |00 = Clock source from PLL.
     * |        |          |01 = Clock source from HCLK.
     * |        |          |10 = Clock source from 4~32 MHz external high speed crystal oscillator (HXT).
     * |        |          |11 = Clock source from 36 MHz internal high speed RC oscillator (HIRC1) or HIRC0 depend on HIRCSEL(CLK_CLKSEL0[3]) setting.
     * |[29:28] |WDTSEL    |WDT Clock Source Selection
     * |        |          |00 = reserved.
     * |        |          |01 = Clock source from LXT.
     * |        |          |10 = Clock source from HCLK/2048.
     * |        |          |11 = Clock source from LIRC
     * |[31:30] |WWDTSEL   |WDT Clock Source Selection
     * |        |          |00 = reserved.
     * |        |          |01 = reserved.
     * |        |          |10 = Clock source from HCLK/2048.
     * |        |          |11 = Clock source from LIRC
     * @var CLK_T::CLKSEL2
     * Offset: 0x18  Clock Source Select Control Register 2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |UART1SEL  |UART1 Clock Source Selection
     * |        |          |000 = Clock source from 4~32 MHz external high speed crystal oscillator (HXT).
     * |        |          |001 = Clock source from 32.768 kHz external low speed crystal oscillator (LXT).
     * |        |          |010 = Clock source from PLL.
     * |        |          |011 = Clock source from 36 MHz internal high speed RC oscillator (HIRC1) or HIRC0 depend on HIRCSEL(CLK_CLKSEL0[3]) setting.
     * |        |          |Others = Clock source from 4 MHz internal medium speed RC oscillator (MIRC).
     * |[6:4]   |CLKOSEL   |Clock Divider Clock Source Selection
     * |        |          |000 = Clock source from 4~32 MHz external high speed crystal oscillator (HXT).
     * |        |          |001 = Clock source from 32.768 kHz external low speed crystal oscillator (LXT).
     * |        |          |010 = Clock source from HCLK.
     * |        |          |011 = Clock source from 36 MHz internal high speed RC oscillator (HIRC1) or HIRC0 depend on HIRCSEL(CLK_CLKSEL0[3]) setting.
     * |        |          |Others = Clock source from 4 MHz internal medium speed RC oscillator (MIRC).
     * |[10:8]  |TMR2SEL   |Timer2 Clock Source Selection
     * |        |          |000 = Clock source from 4~32 MHz external high speed crystal oscillator (HXT).
     * |        |          |001 = Clock source from 32.768 kHz external low speed crystal oscillator (LXT).
     * |        |          |010 = Clock source from 10 kHz internal low speed RC oscillator (LIRC).
     * |        |          |011 = Clock source from external clock pin.
     * |        |          |100 = Clock source from 36 MHz internal high speed RC oscillator (HIRC1) or HIRC0 depend on HIRCSEL(CLK_CLKSEL0[3]) setting.
     * |        |          |101 = Clock source from 4 MHz internal medium speed RC oscillator (MIRC)
     * |        |          |Others = Clock source from HCLK.
     * |[14:12] |TMR3SEL   |Timer3 Clock Source Selection
     * |        |          |000 = Clock source from 4~32 MHz external high speed crystal oscillator (HXT).
     * |        |          |001 = Clock source from 32.768 kHz external low speed crystal oscillator (LXT).
     * |        |          |010 = Clock source from 10 kHz internal low speed RC oscillator (LIRC).
     * |        |          |011 = Clock source from external clock pin.
     * |        |          |100 = Clock source from 36 MHz internal high speed RC oscillator (HIRC1) or HIRC0 depend on HIRCSEL(CLK_CLKSEL0[3]) setting.
     * |        |          |101 = Clock source from 4 MHz internal medium speed RC oscillator (MIRC).
     * |        |          |Others = Clock source from HCLK.
     * |[18:16] |SC0SEL    |SC0 Clock Source Selection
     * |        |          |000 = Clock source from 4~32 MHz external high speed crystal oscillator (HXT).
     * |        |          |001 = Clock source from PLL.
     * |        |          |010 = Clock source from 36 MHz internal high speed RC oscillator (HIRC1) or HIRC0 depend on HIRCSEL(CLK_CLKSEL0[3]) setting.
     * |        |          |011 = Clock source from 4 MHz internal medium speed RC oscillator (MIRC).
     * |        |          |Others = Clock source from HCLK.
     * |[22:20] |SC1SEL    |SC1 Clock Source Selection
     * |        |          |000 = Clock source from 4~32 MHz external high speed crystal oscillator (HXT).
     * |        |          |001 = Clock source from PLL.
     * |        |          |010 = Clock source from 36 MHz internal high speed RC oscillator (HIRC1) or HIRC0 depend on HIRCSEL(CLK_CLKSEL0[3]) setting.
     * |        |          |011 = Clock source from 4 MHz internal medium speed RC oscillator (MIRC).
     * |        |          |Others = Clock source from HCLK.
     * |[25:24] |SPI1SEL   |SPI1 Clock Source Selection
     * |        |          |00 = Clock source from PLL.
     * |        |          |01 = Clock source from HCLK.
     * |        |          |10 = Clock source from 4~32 MHz external high speed crystal oscillator (HXT).
     * |        |          |11 = Clock source from 36 MHz internal high speed RC oscillator (HIRC1) or HIRC0 depend on HIRCSEL(CLK_CLKSEL0[3]) setting.
     * |[27:26] |SPI3SEL   |SPI3 Clock Source Selection
     * |        |          |00 = Clock source from PLL.
     * |        |          |01 = Clock source from HCLK.
     * |        |          |10 = Clock source from 4~32 MHz external high speed crystal oscillator (HXT).
     * |        |          |11 = Clock source from 36 MHz internal high speed RC oscillator (HIRC1) or HIRC0 depend on HIRCSEL(CLK_CLKSEL0[3]) setting.
     * @var CLK_T::CLKDIV0
     * Offset: 0x1C  Clock Divider Number Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |HCLKDIV   |HCLK Clock Divide Number From HCLK Clock Source
     * |        |          |HCLK clock frequency = (HCLK clock source frequency) / (HCLKDIV + 1).
     * |[11:8]  |UART0DIV  |UART0 Clock Divide Number From UART Clock Source
     * |        |          |The UART0 clock frequency = (UART0 Clock Source frequency) / (UART0DIV + 1).
     * |[15:12] |UART1DIV  |UART1 Clock Divide Number From UART Clock Source
     * |        |          |The UART1 clock frequency = (UART1 Clock Source frequency) / (UART1DIV + 1).
     * |[23:16] |ADCDIV    |ADC Clock Divide Number From ADC Clock Source
     * |        |          |ADC clock frequency = (ADC clock source frequency) / (ADCDIV + 1).
     * |[31:28] |SC0DIV    |SC0 Clock Divide Number From SC0 Clock Source
     * |        |          |SC0 clock frequency = (SC0 clock source frequency) / (SC0DIV + 1).
     * @var CLK_T::CLKDIV1
     * Offset: 0x20  Clock Divider Number Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |SC1DIV    |SC 1 Clock Divide Number From SC 1 Clock Source
     * |        |          |The SC 1 clock frequency = (SC 1 Clock Source frequency) / (SC1DIV + 1).
     * |[11:8]  |TMR0DIV   |Timer0 Clock Divide Number From Timer0 Clock Source
     * |        |          |The Timer0 clock frequency = (Timer0 Clock Source frequency) / (TMR0DIV + 1).
     * |[15:12] |TMR1DIV   |Timer1 Clock Divide Number From Timer1 Clock Source
     * |        |          |The Timer1 clock frequency = (Timer1 Clock Source frequency) / (TMR1DIV + 1).
     * |[19:16] |TMR2DIV   |Timer2 Clock Divide Number From Timer2 Clock Source
     * |        |          |The Timer2 clock frequency = (Timer2 Clock Source frequency) / (TMR2DIV + 1).
     * |[23:20] |TMR3DIV   |Timer3 Clock Divide Number From Timer3 Clock Source
     * |        |          |The Timer3 clock frequency = (Timer3 Clock Source frequency) / (TMR3DIV + 1).
     * @var CLK_T::PLLCTL
     * Offset: 0x24  PLL Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |PLLMLP    |PLL Multiple
     * |        |          |000000: Reserved
     * |        |          |000001: X1
     * |        |          |000010: X2
     * |        |          |000011: X3
     * |        |          |000100: X4
     * |        |          |...
     * |        |          |010000:X16
     * |        |          |...
     * |        |          |100000: X32
     * |        |          |100100: X36
     * |        |          |0thers: Reserved
     * |        |          |PLL output frequency: PLL input frequency * PLLMLP.
     * |        |          |PLL output frequency range: 16MHz ~ 36MHz
     * |[13:8]  |INDIV     |PLL Input Source Divider
     * |        |          |The PLL input clock frequency = (PLL Clock Source frequency) / (INDIV + 1).
     * |        |          |PLL input clock frequency range: 0.8MHz ~ 2MHz
     * |[15:14] |STBTSEL   |PLL Stable Time Selection
     * |        |          |00 = 100 cycle time of input clock source.
     * |        |          |01 = 120 cycle time of input clock source.
     * |        |          |10 = 180 cycle time of input clock source.
     * |        |          |11 = 240 cycle time of input clock source.
     * |[16]    |PD        |Power-down Mode
     * |        |          |If set the PDEN bit 1 in CLK_PWRCTL register, the PLL will enter Power-down mode too
     * |        |          |0 = PLL is in normal mode.
     * |        |          |1 = PLL is in power-down mode (default).
     * |[18:17] |PLLSRC    |PLL Source Clock Select
     * |        |          |00 = PLL source clock from HXT.
     * |        |          |01 = PLL source clock from HIRC0 or HIRC1.
     * |        |          |10 = PLL source clock from MIRC
     * |        |          |11 = reserved.
     * @var CLK_T::CLKOCTL
     * Offset: 0x28  Clock Output Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |FREQSEL   |Clock Output Frequency Selection
     * |        |          |The formula of output frequency is
     * |        |          |Fout = Fin/2(N+1).
     * |        |          |Fin is the input clock frequency.
     * |        |          |Fout is the frequency of divider output clock.
     * |        |          |N is the 4-bit value of FREQSEL[3:0].
     * |[4]     |CLKOEN    |Clock Output Enable Bit
     * |        |          |0 = Clock Output function Disabled.
     * |        |          |1 = Clock Output function Enabled.
     * |[5]     |DIV1EN    |Clock Output Divide One Enable Bit
     * |        |          |0 = Clock Output will output clock with source frequency divided by FREQSEL.
     * |        |          |1 = Clock Output will output clock with source frequency.
     * @var CLK_T::WKINTSTS
     * Offset: 0x30  Wake-up Interrupt Status
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |PDWKIF    |Wake-up Interrupt Status in Chip Power-down Mode
     * |        |          |This bit indicates that some event resumes chip from Power-down mode
     * |        |          |The status is set if external interrupts, UART, GPIO, RTC, USB, SPI, Timer, WDT, and BOD wake-up occurred.
     * |        |          |Write 1 to clear this bit.
     * @var CLK_T::APBDIV
     * Offset: 0x34  APB Clock Divider
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |APB0DIV   |APB0 Clock Divider
     * |        |          |APB0 PCLK0 can be divided from HCLK.
     * |        |          |000: PCLK0 = HCLK.
     * |        |          |001: PCLK0 = 1/2 HCLK.
     * |        |          |010: PCLK0 = 1/4 HCLK.
     * |        |          |011: PCLK0 = 1/8 HCLK.
     * |        |          |100: PCLK0 = 1/16 HCLK.
     * |        |          |Others: PCLK0 = HCLK.
     * |[6:4]   |APB1DIV   |APB1 Clock Divider
     * |        |          |APB1 PCLK1 can be divided from HCLK.
     * |        |          |000: PCLK1 = HCLK.
     * |        |          |001: PCLK1 = 1/2 HCLK.
     * |        |          |010: PCLK1 = 1/4 HCLK.
     * |        |          |011: PCLK1 = 1/8 HCLK.
     * |        |          |100: PCLK1 = 1/16 HCLK.
     * |        |          |Others: PCLK1 = HCLK.
     * @var CLK_T::CLKDCTL
     * Offset: 0x38  Clock Fail Detector Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |HXTFDEN   |HXT Clock Fail Detector Enable Bit
     * |        |          |0 = 4~32 MHz external high speed crystal oscillator (HXT) clock fail Detector Disabled.
     * |        |          |1 = 4~32 MHz external high speed crystal oscillator (HXT) clock fail Detector Enabled.
     * |[1]     |LXTFDEN   |LXT Clock Fail Detector Enable Bit
     * |        |          |0 = 32.768 kHz external low speed crystal oscillator (LXT) clock fail Detector Disabled.
     * |        |          |1 = 32.768 kHz external low speed crystal oscillator (LXT) clock fail Detector Enabled.
     * |[2]     |HXTFQDEN  |HXT Clock Frequency Monitor Enable Bit
     * |        |          |0 = 4~32 MHz external high speed crystal oscillator (HXT) clock frequency monitor Disabled.
     * |        |          |1 = 4~32 MHz external high speed crystal oscillator (HXT) clock frequency monitor Enabled.
     * @var CLK_T::CLKDIE
     * Offset: 0x3C  Clock Fail Detector Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |HXTFIEN   |HXT Clock Fail Interrupt Enable Bit
     * |        |          |0 = 4~32 MHz external high speed crystal oscillator (HXT) clock fail interrupt Disabled.
     * |        |          |1 = 4~32 MHz external high speed crystal oscillator (HXT) clock fail interrupt Enabled.
     * |[1]     |LXTFIEN   |LXT Clock Fail Interrupt Enable Bit
     * |        |          |0 = 32.768 kHz external low speed crystal oscillator (LXT) clock fail interrupt Disabled.
     * |        |          |1 = 32.768 kHz external low speed crystal oscillator (LXT) clock fail interrupt Enabled.
     * |[2]     |HXTFQIEN  |HXT Clock Frequency Monitor Interrupt Enable Bit
     * |        |          |0 = 4~32 MHz external high speed crystal oscillator (HXT) clock frequency monitor fail interrupt Disabled.
     * |        |          |1 = 4~32 MHz external high speed crystal oscillator (HXT) clock frequency monitor fail interrupt Enabled.
     * @var CLK_T::CLKDSTS
     * Offset: 0x40  Clock Fail Detector Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |HXTFIF    |HXT Clock Fail Interrupt Flag
     * |        |          |0 = 4~32 MHz external high speed crystal oscillator (HXT) clock is normal.
     * |        |          |1 = 4~32 MHz external high speed crystal oscillator (HXT) clock stops.
     * |        |          |Note: Write 1 to clear the bit to 0.
     * |[1]     |LXTFIF    |LXT Clock Fail Interrupt Flag
     * |        |          |0 = 32.768 kHz external low speed crystal oscillator (LXT) clock is normal.
     * |        |          |1 = 32.768 kHz external low speed crystal oscillator (LXT) stops.
     * |        |          |Note: Write 1 to clear the bit to 0.
     * |[2]     |HXTFQIF   |HXT Clock Frequency Monitor Interrupt Flag
     * |        |          |0 = 4~32 MHz external high speed crystal oscillator (HXT) clock is normal.
     * |        |          |1 = 4~32 MHz external high speed crystal oscillator (HXT) clock frequency is abnormal.
     * |        |          |Note: Write 1 to clear the bit to 0.
     * @var CLK_T::CDUPB
     * Offset: 0x44  Clock Frequency Detector Upper Boundary Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[10:0]  |UPERBD    |HXT Clock Frequency Detector Upper Boundary
     * |        |          |The bits define the high value of frequency monitor window.
     * |        |          |When HXT frequency monitor value higher than this register, the HXT frequency detect fail interrupt flag will set to 1.
     * @var CLK_T::CDLOWB
     * Offset: 0x48  Clock Frequency Detector Lower Boundary Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[10:0]  |LOWERBD   |HXT Clock Frequency Detector Lower Boundary
     * |        |          |The bits define the low value of frequency monitor window.
     * |        |          |When HXT frequency monitor value lower than this register, the HXT frequency detect fail interrupt flag will set to 1.
     */
    __IO uint32_t PWRCTL;                /*!< [0x0000] System Power-down Control Register                               */
    __IO uint32_t AHBCLK;                /*!< [0x0004] AHB Devices Clock Enable Control Register                        */
    __IO uint32_t APBCLK;                /*!< [0x0008] APB Devices Clock Enable Control Register                        */
    __I  uint32_t STATUS;                /*!< [0x000c] Clock status monitor Register                                    */
    __IO uint32_t CLKSEL0;               /*!< [0x0010] Clock Source Select Control Register 0                           */
    __IO uint32_t CLKSEL1;               /*!< [0x0014] Clock Source Select Control Register 1                           */
    __IO uint32_t CLKSEL2;               /*!< [0x0018] Clock Source Select Control Register 2                           */
    __IO uint32_t CLKDIV0;               /*!< [0x001c] Clock Divider Number Register 0                                  */
    __IO uint32_t CLKDIV1;               /*!< [0x0020] Clock Divider Number Register 1                                  */
    __IO uint32_t PLLCTL;                /*!< [0x0024] PLL Control Register                                             */
    __IO uint32_t CLKOCTL;               /*!< [0x0028] Clock Output Control Register                                    */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE0[1];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t WKINTSTS;              /*!< [0x0030] Wake-up Interrupt Status                                         */
    __IO uint32_t APBDIV;                /*!< [0x0034] APB Clock Divider                                                */
    __IO uint32_t CLKDCTL;               /*!< [0x0038] Clock Fail Detector Control Register                             */
    __IO uint32_t CLKDIE;                /*!< [0x003c] Clock Fail Detector Interrupt Enable Register                    */
    __IO uint32_t CLKDSTS;               /*!< [0x0040] Clock Fail Detector Status Register                              */
    __IO uint32_t CDUPB;                 /*!< [0x0044] Clock Frequency Detector Upper Boundary Register                 */
    __IO uint32_t CDLOWB;                /*!< [0x0048] Clock Frequency Detector Lower Boundary Register                 */
} CLK_T;

/**
    @addtogroup CLK_CONST CLK Bit Field Definition
    Constant Definitions for CLK Controller
@{ */

#define CLK_PWRCTL_HXTEN_Pos             (0)                                               /*!< CLK_T::PWRCTL: HXTEN Position          */
#define CLK_PWRCTL_HXTEN_Msk             (0x1ul << CLK_PWRCTL_HXTEN_Pos)                   /*!< CLK_T::PWRCTL: HXTEN Mask              */

#define CLK_PWRCTL_LXTEN_Pos             (1)                                               /*!< CLK_T::PWRCTL: LXTEN Position          */
#define CLK_PWRCTL_LXTEN_Msk             (0x1ul << CLK_PWRCTL_LXTEN_Pos)                   /*!< CLK_T::PWRCTL: LXTEN Mask              */

#define CLK_PWRCTL_HIRC0EN_Pos           (2)                                               /*!< CLK_T::PWRCTL: HIRC0EN Position        */
#define CLK_PWRCTL_HIRC0EN_Msk           (0x1ul << CLK_PWRCTL_HIRC0EN_Pos)                 /*!< CLK_T::PWRCTL: HIRC0EN Mask            */

#define CLK_PWRCTL_LIRCEN_Pos            (3)                                               /*!< CLK_T::PWRCTL: LIRCEN Position         */
#define CLK_PWRCTL_LIRCEN_Msk            (0x1ul << CLK_PWRCTL_LIRCEN_Pos)                  /*!< CLK_T::PWRCTL: LIRCEN Mask             */

#define CLK_PWRCTL_PDWKDLY_Pos           (4)                                               /*!< CLK_T::PWRCTL: PDWKDLY Position        */
#define CLK_PWRCTL_PDWKDLY_Msk           (0x1ul << CLK_PWRCTL_PDWKDLY_Pos)                 /*!< CLK_T::PWRCTL: PDWKDLY Mask            */

#define CLK_PWRCTL_PDWKIEN_Pos           (5)                                               /*!< CLK_T::PWRCTL: PDWKIEN Position        */
#define CLK_PWRCTL_PDWKIEN_Msk           (0x1ul << CLK_PWRCTL_PDWKIEN_Pos)                 /*!< CLK_T::PWRCTL: PDWKIEN Mask            */

#define CLK_PWRCTL_PDEN_Pos              (6)                                               /*!< CLK_T::PWRCTL: PDEN Position           */
#define CLK_PWRCTL_PDEN_Msk              (0x1ul << CLK_PWRCTL_PDEN_Pos)                    /*!< CLK_T::PWRCTL: PDEN Mask               */

#define CLK_PWRCTL_HXTSLTYP_Pos          (8)                                               /*!< CLK_T::PWRCTL: HXTSLTYP Position       */
#define CLK_PWRCTL_HXTSLTYP_Msk          (0x1ul << CLK_PWRCTL_HXTSLTYP_Pos)                /*!< CLK_T::PWRCTL: HXTSLTYP Mask           */

#define CLK_PWRCTL_HXTGAIN_Pos           (10)                                              /*!< CLK_T::PWRCTL: HXTGAIN Position        */
#define CLK_PWRCTL_HXTGAIN_Msk           (0x7ul << CLK_PWRCTL_HXTGAIN_Pos)                 /*!< CLK_T::PWRCTL: HXTGAIN Mask            */

#define CLK_PWRCTL_HIRC0FSEL_Pos         (13)                                              /*!< CLK_T::PWRCTL: HIRC0FSEL Position      */
#define CLK_PWRCTL_HIRC0FSEL_Msk         (0x1ul << CLK_PWRCTL_HIRC0FSEL_Pos)               /*!< CLK_T::PWRCTL: HIRC0FSEL Mask          */

#define CLK_PWRCTL_HIRC0FSTOP_Pos        (14)                                              /*!< CLK_T::PWRCTL: HIRC0FSTOP Position     */
#define CLK_PWRCTL_HIRC0FSTOP_Msk        (0x1ul << CLK_PWRCTL_HIRC0FSTOP_Pos)              /*!< CLK_T::PWRCTL: HIRC0FSTOP Mask         */

#define CLK_PWRCTL_HIRC1EN_Pos           (24)                                              /*!< CLK_T::PWRCTL: HIRC1EN Position        */
#define CLK_PWRCTL_HIRC1EN_Msk           (0x1ul << CLK_PWRCTL_HIRC1EN_Pos)                 /*!< CLK_T::PWRCTL: HIRC1EN Mask            */

#define CLK_PWRCTL_MIRCEN_Pos            (25)                                              /*!< CLK_T::PWRCTL: MIRCEN Position         */
#define CLK_PWRCTL_MIRCEN_Msk            (0x1ul << CLK_PWRCTL_MIRCEN_Pos)                  /*!< CLK_T::PWRCTL: MIRCEN Mask             */

#define CLK_AHBCLK_GPIOCKEN_Pos          (0)                                               /*!< CLK_T::AHBCLK: GPIOCKEN Position       */
#define CLK_AHBCLK_GPIOCKEN_Msk          (0x1ul << CLK_AHBCLK_GPIOCKEN_Pos)                /*!< CLK_T::AHBCLK: GPIOCKEN Mask           */

#define CLK_AHBCLK_PDMACKEN_Pos          (1)                                               /*!< CLK_T::AHBCLK: PDMACKEN Position       */
#define CLK_AHBCLK_PDMACKEN_Msk          (0x1ul << CLK_AHBCLK_PDMACKEN_Pos)                /*!< CLK_T::AHBCLK: PDMACKEN Mask           */

#define CLK_AHBCLK_ISPCKEN_Pos           (2)                                               /*!< CLK_T::AHBCLK: ISPCKEN Position        */
#define CLK_AHBCLK_ISPCKEN_Msk           (0x1ul << CLK_AHBCLK_ISPCKEN_Pos)                 /*!< CLK_T::AHBCLK: ISPCKEN Mask            */

#define CLK_AHBCLK_SRAMCKEN_Pos          (4)                                               /*!< CLK_T::AHBCLK: SRAMCKEN Position       */
#define CLK_AHBCLK_SRAMCKEN_Msk          (0x1ul << CLK_AHBCLK_SRAMCKEN_Pos)                /*!< CLK_T::AHBCLK: SRAMCKEN Mask           */

#define CLK_AHBCLK_STCKEN_Pos            (5)                                               /*!< CLK_T::AHBCLK: STCKEN Position         */
#define CLK_AHBCLK_STCKEN_Msk            (0x1ul << CLK_AHBCLK_STCKEN_Pos)                  /*!< CLK_T::AHBCLK: STCKEN Mask             */

#define CLK_APBCLK_WDTCKEN_Pos           (0)                                               /*!< CLK_T::APBCLK: WDTCKEN Position        */
#define CLK_APBCLK_WDTCKEN_Msk           (0x1ul << CLK_APBCLK_WDTCKEN_Pos)                 /*!< CLK_T::APBCLK: WDTCKEN Mask            */

#define CLK_APBCLK_RTCCKEN_Pos           (1)                                               /*!< CLK_T::APBCLK: RTCCKEN Position        */
#define CLK_APBCLK_RTCCKEN_Msk           (0x1ul << CLK_APBCLK_RTCCKEN_Pos)                 /*!< CLK_T::APBCLK: RTCCKEN Mask            */

#define CLK_APBCLK_TMR0CKEN_Pos          (2)                                               /*!< CLK_T::APBCLK: TMR0CKEN Position       */
#define CLK_APBCLK_TMR0CKEN_Msk          (0x1ul << CLK_APBCLK_TMR0CKEN_Pos)                /*!< CLK_T::APBCLK: TMR0CKEN Mask           */

#define CLK_APBCLK_TMR1CKEN_Pos          (3)                                               /*!< CLK_T::APBCLK: TMR1CKEN Position       */
#define CLK_APBCLK_TMR1CKEN_Msk          (0x1ul << CLK_APBCLK_TMR1CKEN_Pos)                /*!< CLK_T::APBCLK: TMR1CKEN Mask           */

#define CLK_APBCLK_TMR2CKEN_Pos          (4)                                               /*!< CLK_T::APBCLK: TMR2CKEN Position       */
#define CLK_APBCLK_TMR2CKEN_Msk          (0x1ul << CLK_APBCLK_TMR2CKEN_Pos)                /*!< CLK_T::APBCLK: TMR2CKEN Mask           */

#define CLK_APBCLK_TMR3CKEN_Pos          (5)                                               /*!< CLK_T::APBCLK: TMR3CKEN Position       */
#define CLK_APBCLK_TMR3CKEN_Msk          (0x1ul << CLK_APBCLK_TMR3CKEN_Pos)                /*!< CLK_T::APBCLK: TMR3CKEN Mask           */

#define CLK_APBCLK_CLKOCKEN_Pos          (6)                                               /*!< CLK_T::APBCLK: CLKOCKEN Position       */
#define CLK_APBCLK_CLKOCKEN_Msk          (0x1ul << CLK_APBCLK_CLKOCKEN_Pos)                /*!< CLK_T::APBCLK: CLKOCKEN Mask           */

#define CLK_APBCLK_I2C0CKEN_Pos          (8)                                               /*!< CLK_T::APBCLK: I2C0CKEN Position       */
#define CLK_APBCLK_I2C0CKEN_Msk          (0x1ul << CLK_APBCLK_I2C0CKEN_Pos)                /*!< CLK_T::APBCLK: I2C0CKEN Mask           */

#define CLK_APBCLK_I2C1CKEN_Pos          (9)                                               /*!< CLK_T::APBCLK: I2C1CKEN Position       */
#define CLK_APBCLK_I2C1CKEN_Msk          (0x1ul << CLK_APBCLK_I2C1CKEN_Pos)                /*!< CLK_T::APBCLK: I2C1CKEN Mask           */

#define CLK_APBCLK_ACMP0CKEN_Pos         (11)                                              /*!< CLK_T::APBCLK: ACMP0CKEN Position      */
#define CLK_APBCLK_ACMP0CKEN_Msk         (0x1ul << CLK_APBCLK_ACMP0CKEN_Pos)               /*!< CLK_T::APBCLK: ACMP0CKEN Mask          */

#define CLK_APBCLK_SPI0CKEN_Pos          (12)                                              /*!< CLK_T::APBCLK: SPI0CKEN Position       */
#define CLK_APBCLK_SPI0CKEN_Msk          (0x1ul << CLK_APBCLK_SPI0CKEN_Pos)                /*!< CLK_T::APBCLK: SPI0CKEN Mask           */

#define CLK_APBCLK_SPI1CKEN_Pos          (13)                                              /*!< CLK_T::APBCLK: SPI1CKEN Position       */
#define CLK_APBCLK_SPI1CKEN_Msk          (0x1ul << CLK_APBCLK_SPI1CKEN_Pos)                /*!< CLK_T::APBCLK: SPI1CKEN Mask           */

#define CLK_APBCLK_SPI2CKEN_Pos          (14)                                              /*!< CLK_T::APBCLK: SPI2CKEN Position       */
#define CLK_APBCLK_SPI2CKEN_Msk          (0x1ul << CLK_APBCLK_SPI2CKEN_Pos)                /*!< CLK_T::APBCLK: SPI2CKEN Mask           */

#define CLK_APBCLK_SPI3CKEN_Pos          (15)                                              /*!< CLK_T::APBCLK: SPI3CKEN Position       */
#define CLK_APBCLK_SPI3CKEN_Msk          (0x1ul << CLK_APBCLK_SPI3CKEN_Pos)                /*!< CLK_T::APBCLK: SPI3CKEN Mask           */

#define CLK_APBCLK_UART0CKEN_Pos         (16)                                              /*!< CLK_T::APBCLK: UART0CKEN Position      */
#define CLK_APBCLK_UART0CKEN_Msk         (0x1ul << CLK_APBCLK_UART0CKEN_Pos)               /*!< CLK_T::APBCLK: UART0CKEN Mask          */

#define CLK_APBCLK_UART1CKEN_Pos         (17)                                              /*!< CLK_T::APBCLK: UART1CKEN Position      */
#define CLK_APBCLK_UART1CKEN_Msk         (0x1ul << CLK_APBCLK_UART1CKEN_Pos)               /*!< CLK_T::APBCLK: UART1CKEN Mask          */

#define CLK_APBCLK_PWM0CKEN_Pos          (20)                                              /*!< CLK_T::APBCLK: PWM0CKEN Position       */
#define CLK_APBCLK_PWM0CKEN_Msk          (0x1ul << CLK_APBCLK_PWM0CKEN_Pos)                /*!< CLK_T::APBCLK: PWM0CKEN Mask           */

#define CLK_APBCLK_ADCCKEN_Pos           (28)                                              /*!< CLK_T::APBCLK: ADCCKEN Position        */
#define CLK_APBCLK_ADCCKEN_Msk           (0x1ul << CLK_APBCLK_ADCCKEN_Pos)                 /*!< CLK_T::APBCLK: ADCCKEN Mask            */

#define CLK_APBCLK_SC0CKEN_Pos           (30)                                              /*!< CLK_T::APBCLK: SC0CKEN Position        */
#define CLK_APBCLK_SC0CKEN_Msk           (0x1ul << CLK_APBCLK_SC0CKEN_Pos)                 /*!< CLK_T::APBCLK: SC0CKEN Mask            */

#define CLK_APBCLK_SC1CKEN_Pos           (31)                                              /*!< CLK_T::APBCLK: SC1CKEN Position        */
#define CLK_APBCLK_SC1CKEN_Msk           (0x1ul << CLK_APBCLK_SC1CKEN_Pos)                 /*!< CLK_T::APBCLK: SC1CKEN Mask            */

#define CLK_STATUS_HXTSTB_Pos            (0)                                               /*!< CLK_T::STATUS: HXTSTB Position         */
#define CLK_STATUS_HXTSTB_Msk            (0x1ul << CLK_STATUS_HXTSTB_Pos)                  /*!< CLK_T::STATUS: HXTSTB Mask             */

#define CLK_STATUS_LXTSTB_Pos            (1)                                               /*!< CLK_T::STATUS: LXTSTB Position         */
#define CLK_STATUS_LXTSTB_Msk            (0x1ul << CLK_STATUS_LXTSTB_Pos)                  /*!< CLK_T::STATUS: LXTSTB Mask             */

#define CLK_STATUS_PLLSTB_Pos            (2)                                               /*!< CLK_T::STATUS: PLLSTB Position         */
#define CLK_STATUS_PLLSTB_Msk            (0x1ul << CLK_STATUS_PLLSTB_Pos)                  /*!< CLK_T::STATUS: PLLSTB Mask             */

#define CLK_STATUS_LIRCSTB_Pos           (3)                                               /*!< CLK_T::STATUS: LIRCSTB Position        */
#define CLK_STATUS_LIRCSTB_Msk           (0x1ul << CLK_STATUS_LIRCSTB_Pos)                 /*!< CLK_T::STATUS: LIRCSTB Mask            */

#define CLK_STATUS_HIRC0STB_Pos          (4)                                               /*!< CLK_T::STATUS: HIRC0STB Position       */
#define CLK_STATUS_HIRC0STB_Msk          (0x1ul << CLK_STATUS_HIRC0STB_Pos)                /*!< CLK_T::STATUS: HIRC0STB Mask           */

#define CLK_STATUS_HIRC1STB_Pos          (5)                                               /*!< CLK_T::STATUS: HIRC1STB Position       */
#define CLK_STATUS_HIRC1STB_Msk          (0x1ul << CLK_STATUS_HIRC1STB_Pos)                /*!< CLK_T::STATUS: HIRC1STB Mask           */

#define CLK_STATUS_MIRCSTB_Pos           (6)                                               /*!< CLK_T::STATUS: MIRCSTB Position        */
#define CLK_STATUS_MIRCSTB_Msk           (0x1ul << CLK_STATUS_MIRCSTB_Pos)                 /*!< CLK_T::STATUS: MIRCSTB Mask            */

#define CLK_STATUS_CLKSFAIL_Pos          (7)                                               /*!< CLK_T::STATUS: CLKSFAIL Position       */
#define CLK_STATUS_CLKSFAIL_Msk          (0x1ul << CLK_STATUS_CLKSFAIL_Pos)                /*!< CLK_T::STATUS: CLKSFAIL Mask           */

#define CLK_CLKSEL0_HCLKSEL_Pos          (0)                                               /*!< CLK_T::CLKSEL0: HCLKSEL Position       */
#define CLK_CLKSEL0_HCLKSEL_Msk          (0x7ul << CLK_CLKSEL0_HCLKSEL_Pos)                /*!< CLK_T::CLKSEL0: HCLKSEL Mask           */

#define CLK_CLKSEL0_HIRCSEL_Pos          (3)                                               /*!< CLK_T::CLKSEL0: HIRCSEL Position       */
#define CLK_CLKSEL0_HIRCSEL_Msk          (0x1ul << CLK_CLKSEL0_HIRCSEL_Pos)                /*!< CLK_T::CLKSEL0: HIRCSEL Mask           */

#define CLK_CLKSEL0_ISPSEL_Pos           (4)                                               /*!< CLK_T::CLKSEL0: ISPSEL Position        */
#define CLK_CLKSEL0_ISPSEL_Msk           (0x1ul << CLK_CLKSEL0_ISPSEL_Pos)                 /*!< CLK_T::CLKSEL0: ISPSEL Mask            */

#define CLK_CLKSEL1_UART0SEL_Pos         (0)                                               /*!< CLK_T::CLKSEL1: UART0SEL Position      */
#define CLK_CLKSEL1_UART0SEL_Msk         (0x7ul << CLK_CLKSEL1_UART0SEL_Pos)               /*!< CLK_T::CLKSEL1: UART0SEL Mask          */

#define CLK_CLKSEL1_PWM0SEL_Pos          (4)                                               /*!< CLK_T::CLKSEL1: PWM0SEL Position       */
#define CLK_CLKSEL1_PWM0SEL_Msk          (0x1ul << CLK_CLKSEL1_PWM0SEL_Pos)                /*!< CLK_T::CLKSEL1: PWM0SEL Mask           */

#define CLK_CLKSEL1_TMR0SEL_Pos          (8)                                               /*!< CLK_T::CLKSEL1: TMR0SEL Position       */
#define CLK_CLKSEL1_TMR0SEL_Msk          (0x7ul << CLK_CLKSEL1_TMR0SEL_Pos)                /*!< CLK_T::CLKSEL1: TMR0SEL Mask           */

#define CLK_CLKSEL1_TMR1SEL_Pos          (12)                                              /*!< CLK_T::CLKSEL1: TMR1SEL Position       */
#define CLK_CLKSEL1_TMR1SEL_Msk          (0x7ul << CLK_CLKSEL1_TMR1SEL_Pos)                /*!< CLK_T::CLKSEL1: TMR1SEL Mask           */

#define CLK_CLKSEL1_ADCSEL_Pos           (19)                                              /*!< CLK_T::CLKSEL1: ADCSEL Position        */
#define CLK_CLKSEL1_ADCSEL_Msk           (0x7ul << CLK_CLKSEL1_ADCSEL_Pos)                 /*!< CLK_T::CLKSEL1: ADCSEL Mask            */

#define CLK_CLKSEL1_SPI0SEL_Pos          (24)                                              /*!< CLK_T::CLKSEL1: SPI0SEL Position       */
#define CLK_CLKSEL1_SPI0SEL_Msk          (0x3ul << CLK_CLKSEL1_SPI0SEL_Pos)                /*!< CLK_T::CLKSEL1: SPI0SEL Mask           */

#define CLK_CLKSEL1_SPI2SEL_Pos          (26)                                              /*!< CLK_T::CLKSEL1: SPI2SEL Position       */
#define CLK_CLKSEL1_SPI2SEL_Msk          (0x3ul << CLK_CLKSEL1_SPI2SEL_Pos)                /*!< CLK_T::CLKSEL1: SPI2SEL Mask           */

#define CLK_CLKSEL1_WDTSEL_Pos           (28)                                              /*!< CLK_T::CLKSEL1: WDTSEL Position        */
#define CLK_CLKSEL1_WDTSEL_Msk           (0x3ul << CLK_CLKSEL1_WDTSEL_Pos)                 /*!< CLK_T::CLKSEL1: WDTSEL Mask            */

#define CLK_CLKSEL1_WWDTSEL_Pos          (30)                                              /*!< CLK_T::CLKSEL1: WWDTSEL Position       */
#define CLK_CLKSEL1_WWDTSEL_Msk          (0x3ul << CLK_CLKSEL1_WWDTSEL_Pos)                /*!< CLK_T::CLKSEL1: WWDTSEL Mask           */

#define CLK_CLKSEL2_UART1SEL_Pos         (0)                                               /*!< CLK_T::CLKSEL2: UART1SEL Position      */
#define CLK_CLKSEL2_UART1SEL_Msk         (0x7ul << CLK_CLKSEL2_UART1SEL_Pos)               /*!< CLK_T::CLKSEL2: UART1SEL Mask          */

#define CLK_CLKSEL2_CLKOSEL_Pos          (4)                                               /*!< CLK_T::CLKSEL2: CLKOSEL Position       */
#define CLK_CLKSEL2_CLKOSEL_Msk          (0x7ul << CLK_CLKSEL2_CLKOSEL_Pos)                /*!< CLK_T::CLKSEL2: CLKOSEL Mask           */

#define CLK_CLKSEL2_TMR2SEL_Pos          (8)                                               /*!< CLK_T::CLKSEL2: TMR2SEL Position       */
#define CLK_CLKSEL2_TMR2SEL_Msk          (0x7ul << CLK_CLKSEL2_TMR2SEL_Pos)                /*!< CLK_T::CLKSEL2: TMR2SEL Mask           */

#define CLK_CLKSEL2_TMR3SEL_Pos          (12)                                              /*!< CLK_T::CLKSEL2: TMR3SEL Position       */
#define CLK_CLKSEL2_TMR3SEL_Msk          (0x7ul << CLK_CLKSEL2_TMR3SEL_Pos)                /*!< CLK_T::CLKSEL2: TMR3SEL Mask           */

#define CLK_CLKSEL2_SC0SEL_Pos           (16)                                              /*!< CLK_T::CLKSEL2: SC0SEL Position        */
#define CLK_CLKSEL2_SC0SEL_Msk           (0x7ul << CLK_CLKSEL2_SC0SEL_Pos)                 /*!< CLK_T::CLKSEL2: SC0SEL Mask            */

#define CLK_CLKSEL2_SC1SEL_Pos           (20)                                              /*!< CLK_T::CLKSEL2: SC1SEL Position        */
#define CLK_CLKSEL2_SC1SEL_Msk           (0x7ul << CLK_CLKSEL2_SC1SEL_Pos)                 /*!< CLK_T::CLKSEL2: SC1SEL Mask            */

#define CLK_CLKSEL2_SPI1SEL_Pos          (24)                                              /*!< CLK_T::CLKSEL2: SPI1SEL Position       */
#define CLK_CLKSEL2_SPI1SEL_Msk          (0x3ul << CLK_CLKSEL2_SPI1SEL_Pos)                /*!< CLK_T::CLKSEL2: SPI1SEL Mask           */

#define CLK_CLKSEL2_SPI3SEL_Pos          (26)                                              /*!< CLK_T::CLKSEL2: SPI3SEL Position       */
#define CLK_CLKSEL2_SPI3SEL_Msk          (0x3ul << CLK_CLKSEL2_SPI3SEL_Pos)                /*!< CLK_T::CLKSEL2: SPI3SEL Mask           */

#define CLK_CLKDIV0_HCLKDIV_Pos          (0)                                               /*!< CLK_T::CLKDIV0: HCLKDIV Position       */
#define CLK_CLKDIV0_HCLKDIV_Msk          (0xful << CLK_CLKDIV0_HCLKDIV_Pos)                /*!< CLK_T::CLKDIV0: HCLKDIV Mask           */

#define CLK_CLKDIV0_UART0DIV_Pos         (8)                                               /*!< CLK_T::CLKDIV0: UART0DIV Position      */
#define CLK_CLKDIV0_UART0DIV_Msk         (0xful << CLK_CLKDIV0_UART0DIV_Pos)               /*!< CLK_T::CLKDIV0: UART0DIV Mask          */

#define CLK_CLKDIV0_UART1DIV_Pos         (12)                                              /*!< CLK_T::CLKDIV0: UART1DIV Position      */
#define CLK_CLKDIV0_UART1DIV_Msk         (0xful << CLK_CLKDIV0_UART1DIV_Pos)               /*!< CLK_T::CLKDIV0: UART1DIV Mask          */

#define CLK_CLKDIV0_ADCDIV_Pos           (16)                                              /*!< CLK_T::CLKDIV0: ADCDIV Position        */
#define CLK_CLKDIV0_ADCDIV_Msk           (0xfful << CLK_CLKDIV0_ADCDIV_Pos)                /*!< CLK_T::CLKDIV0: ADCDIV Mask            */

#define CLK_CLKDIV0_SC0DIV_Pos           (28)                                              /*!< CLK_T::CLKDIV0: SC0DIV Position        */
#define CLK_CLKDIV0_SC0DIV_Msk           (0xful << CLK_CLKDIV0_SC0DIV_Pos)                 /*!< CLK_T::CLKDIV0: SC0DIV Mask            */

#define CLK_CLKDIV1_SC1DIV_Pos           (0)                                               /*!< CLK_T::CLKDIV1: SC1DIV Position        */
#define CLK_CLKDIV1_SC1DIV_Msk           (0xful << CLK_CLKDIV1_SC1DIV_Pos)                 /*!< CLK_T::CLKDIV1: SC1DIV Mask            */

#define CLK_CLKDIV1_TMR0DIV_Pos          (8)                                               /*!< CLK_T::CLKDIV1: TMR0DIV Position       */
#define CLK_CLKDIV1_TMR0DIV_Msk          (0xful << CLK_CLKDIV1_TMR0DIV_Pos)                /*!< CLK_T::CLKDIV1: TMR0DIV Mask           */

#define CLK_CLKDIV1_TMR1DIV_Pos          (12)                                              /*!< CLK_T::CLKDIV1: TMR1DIV Position       */
#define CLK_CLKDIV1_TMR1DIV_Msk          (0xful << CLK_CLKDIV1_TMR1DIV_Pos)                /*!< CLK_T::CLKDIV1: TMR1DIV Mask           */

#define CLK_CLKDIV1_TMR2DIV_Pos          (16)                                              /*!< CLK_T::CLKDIV1: TMR2DIV Position       */
#define CLK_CLKDIV1_TMR2DIV_Msk          (0xful << CLK_CLKDIV1_TMR2DIV_Pos)                /*!< CLK_T::CLKDIV1: TMR2DIV Mask           */

#define CLK_CLKDIV1_TMR3DIV_Pos          (20)                                              /*!< CLK_T::CLKDIV1: TMR3DIV Position       */
#define CLK_CLKDIV1_TMR3DIV_Msk          (0xful << CLK_CLKDIV1_TMR3DIV_Pos)                /*!< CLK_T::CLKDIV1: TMR3DIV Mask           */

#define CLK_PLLCTL_PLLMLP_Pos            (0)                                               /*!< CLK_T::PLLCTL: PLLMLP Position         */
#define CLK_PLLCTL_PLLMLP_Msk            (0x3ful << CLK_PLLCTL_PLLMLP_Pos)                 /*!< CLK_T::PLLCTL: PLLMLP Mask             */

#define CLK_PLLCTL_INDIV_Pos             (8)                                               /*!< CLK_T::PLLCTL: INDIV Position          */
#define CLK_PLLCTL_INDIV_Msk             (0x3ful << CLK_PLLCTL_INDIV_Pos)                  /*!< CLK_T::PLLCTL: INDIV Mask              */

#define CLK_PLLCTL_STBTSEL_Pos           (14)                                              /*!< CLK_T::PLLCTL: STBTSEL Position        */
#define CLK_PLLCTL_STBTSEL_Msk           (0x3ul << CLK_PLLCTL_STBTSEL_Pos)                 /*!< CLK_T::PLLCTL: STBTSEL Mask            */

#define CLK_PLLCTL_PD_Pos                (16)                                              /*!< CLK_T::PLLCTL: PD Position             */
#define CLK_PLLCTL_PD_Msk                (0x1ul << CLK_PLLCTL_PD_Pos)                      /*!< CLK_T::PLLCTL: PD Mask                 */

#define CLK_PLLCTL_PLLSRC_Pos            (17)                                              /*!< CLK_T::PLLCTL: PLLSRC Position         */
#define CLK_PLLCTL_PLLSRC_Msk            (0x3ul << CLK_PLLCTL_PLLSRC_Pos)                  /*!< CLK_T::PLLCTL: PLLSRC Mask             */

#define CLK_CLKOCTL_FREQSEL_Pos          (0)                                               /*!< CLK_T::CLKOCTL: FREQSEL Position       */
#define CLK_CLKOCTL_FREQSEL_Msk          (0xful << CLK_CLKOCTL_FREQSEL_Pos)                /*!< CLK_T::CLKOCTL: FREQSEL Mask           */

#define CLK_CLKOCTL_CLKOEN_Pos           (4)                                               /*!< CLK_T::CLKOCTL: CLKOEN Position        */
#define CLK_CLKOCTL_CLKOEN_Msk           (0x1ul << CLK_CLKOCTL_CLKOEN_Pos)                 /*!< CLK_T::CLKOCTL: CLKOEN Mask            */

#define CLK_CLKOCTL_DIV1EN_Pos           (5)                                               /*!< CLK_T::CLKOCTL: DIV1EN Position        */
#define CLK_CLKOCTL_DIV1EN_Msk           (0x1ul << CLK_CLKOCTL_DIV1EN_Pos)                 /*!< CLK_T::CLKOCTL: DIV1EN Mask            */

#define CLK_WKINTSTS_PDWKIF_Pos          (0)                                               /*!< CLK_T::WKINTSTS: PDWKIF Position       */
#define CLK_WKINTSTS_PDWKIF_Msk          (0x1ul << CLK_WKINTSTS_PDWKIF_Pos)                /*!< CLK_T::WKINTSTS: PDWKIF Mask           */

#define CLK_APBDIV_APB0DIV_Pos           (0)                                               /*!< CLK_T::APBDIV: APB0DIV Position        */
#define CLK_APBDIV_APB0DIV_Msk           (0x7ul << CLK_APBDIV_APB0DIV_Pos)                 /*!< CLK_T::APBDIV: APB0DIV Mask            */

#define CLK_APBDIV_APB1DIV_Pos           (4)                                               /*!< CLK_T::APBDIV: APB1DIV Position        */
#define CLK_APBDIV_APB1DIV_Msk           (0x7ul << CLK_APBDIV_APB1DIV_Pos)                 /*!< CLK_T::APBDIV: APB1DIV Mask            */

#define CLK_CLKDCTL_HXTFDEN_Pos          (0)                                               /*!< CLK_T::CLKDCTL: HXTFDEN Position       */
#define CLK_CLKDCTL_HXTFDEN_Msk          (0x1ul << CLK_CLKDCTL_HXTFDEN_Pos)                /*!< CLK_T::CLKDCTL: HXTFDEN Mask           */

#define CLK_CLKDCTL_LXTFDEN_Pos          (1)                                               /*!< CLK_T::CLKDCTL: LXTFDEN Position       */
#define CLK_CLKDCTL_LXTFDEN_Msk          (0x1ul << CLK_CLKDCTL_LXTFDEN_Pos)                /*!< CLK_T::CLKDCTL: LXTFDEN Mask           */

#define CLK_CLKDCTL_HXTFQDEN_Pos         (2)                                               /*!< CLK_T::CLKDCTL: HXTFQDEN Position      */
#define CLK_CLKDCTL_HXTFQDEN_Msk         (0x1ul << CLK_CLKDCTL_HXTFQDEN_Pos)               /*!< CLK_T::CLKDCTL: HXTFQDEN Mask          */

#define CLK_CLKDIE_HXTFIEN_Pos           (0)                                               /*!< CLK_T::CLKDIE: HXTFIEN Position        */
#define CLK_CLKDIE_HXTFIEN_Msk           (0x1ul << CLK_CLKDIE_HXTFIEN_Pos)                 /*!< CLK_T::CLKDIE: HXTFIEN Mask            */

#define CLK_CLKDIE_LXTFIEN_Pos           (1)                                               /*!< CLK_T::CLKDIE: LXTFIEN Position        */
#define CLK_CLKDIE_LXTFIEN_Msk           (0x1ul << CLK_CLKDIE_LXTFIEN_Pos)                 /*!< CLK_T::CLKDIE: LXTFIEN Mask            */

#define CLK_CLKDIE_HXTFQIEN_Pos          (2)                                               /*!< CLK_T::CLKDIE: HXTFQIEN Position       */
#define CLK_CLKDIE_HXTFQIEN_Msk          (0x1ul << CLK_CLKDIE_HXTFQIEN_Pos)                /*!< CLK_T::CLKDIE: HXTFQIEN Mask           */

#define CLK_CLKDSTS_HXTFIF_Pos           (0)                                               /*!< CLK_T::CLKDSTS: HXTFIF Position        */
#define CLK_CLKDSTS_HXTFIF_Msk           (0x1ul << CLK_CLKDSTS_HXTFIF_Pos)                 /*!< CLK_T::CLKDSTS: HXTFIF Mask            */

#define CLK_CLKDSTS_LXTFIF_Pos           (1)                                               /*!< CLK_T::CLKDSTS: LXTFIF Position        */
#define CLK_CLKDSTS_LXTFIF_Msk           (0x1ul << CLK_CLKDSTS_LXTFIF_Pos)                 /*!< CLK_T::CLKDSTS: LXTFIF Mask            */

#define CLK_CLKDSTS_HXTFQIF_Pos          (2)                                               /*!< CLK_T::CLKDSTS: HXTFQIF Position       */
#define CLK_CLKDSTS_HXTFQIF_Msk          (0x1ul << CLK_CLKDSTS_HXTFQIF_Pos)                /*!< CLK_T::CLKDSTS: HXTFQIF Mask           */

#define CLK_CDUPB_UPERBD_Pos             (0)                                               /*!< CLK_T::CDUPB: UPERBD Position          */
#define CLK_CDUPB_UPERBD_Msk             (0x7fful << CLK_CDUPB_UPERBD_Pos)                 /*!< CLK_T::CDUPB: UPERBD Mask              */

#define CLK_CDLOWB_LOWERBD_Pos           (0)                                               /*!< CLK_T::CDLOWB: LOWERBD Position        */
#define CLK_CDLOWB_LOWERBD_Msk           (0x7fful << CLK_CDLOWB_LOWERBD_Pos)               /*!< CLK_T::CDLOWB: LOWERBD Mask            */

/**@}*/ /* CLK_CONST */
/**@}*/ /* end of CLK register group */


/*---------------------- Flash Memory Controller -------------------------*/
/**
    @addtogroup FMC Flash Memory Controller(FMC)
    Memory Mapped Structure for FMC Controller
@{ */

typedef struct
{


    /**
     * @var FMC_T::ISPCTL
     * Offset: 0x00  ISP Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ISPEN     |ISP Enable Bit (Write Protect)
     * |        |          |ISP function enable bit. Set this bit to enable ISP function.
     * |        |          |0 = ISP function Disabled.
     * |        |          |1 = ISP function Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[1]     |BS        |Boot Select (Write Protect)
     * |        |          |Set/clear this bit to select next booting from LDROM/APROM, respectively
     * |        |          |This bit also functions as chip booting status flag, which can be used to check where chip booted from.
     * |        |          |This bit is initiated with the inversed value of CBS[1] (CONFIG0[7]) after any reset is happened except CPU reset (CPU is 1) or system reset (SYS) is happened.
     * |        |          |0 = Booting from APROM.
     * |        |          |1 = Booting from LDROM.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[3]     |APUEN     |APROM Update Enable Bit (Write Protect)
     * |        |          |0 = APROM cannot be updated when the chip runs in APROM booting without IAP mode.
     * |        |          |1 = APROM can be updated when the chip runs in APROM booting without IAP mode
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[4]     |CFGUEN    |CONFIG Update Enable Bit (Write Protect)
     * |        |          |0 = CONFIG cannot be updated.
     * |        |          |1 = CONFIG can be updated.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[5]     |LDUEN     |LDROM Update Enable Bit (Write Protect)
     * |        |          |LDROM update enable bit.
     * |        |          |0 = LDROM cannot be updated.
     * |        |          |1 = LDROM can be updated.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |        |          |In LDROM booting without IAP mode, LDROM cannot be updated even if LDUEN=1.
     * |[6]     |ISPFF     |ISP Fail Flag (Write Protect)
     * |        |          |This bit is set by hardware when a triggered ISP meets any of the following conditions:
     * |        |          |This bit needs to be cleared by writing 1 to it.
     * |        |          |(1) APROM writes to itself if APUEN is set to 0.
     * |        |          |(2) LDROM writes to itself if LDUEN is set to 0.
     * |        |          |(3) CONFIG is erased/programmed if CFGUEN is set to 0.
     * |        |          |(6) Page Erase command at LOCK mode with ICE connection
     * |        |          |(7) Erase or Program command at brown-out detected
     * |        |          |(8) Destination address is illegal, such as over an available range.
     * |        |          |(9) Invalid ISP commands
     * |        |          |(10) KPROM is erased/programmed if KEYLOCK is set to 1
     * |        |          |(11) APROM(not include Data Flash) is erased/programmed if KEYLOCK is set to 1
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * @var FMC_T::ISPADDR
     * Offset: 0x04  ISP Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ISPADDR   |ISP Address
     * |        |          |The Nano103 series is equipped with embedded flash
     * |        |          |ISPADDR [1:0] must be kept 00 for ISP 32-bit operation.
     * |        |          |For both CRC-32 Checksum Calculation and Flash All-One Verification commands, this field is the flash starting address for checksum calculation and 512 bytes address alignment is necessary.
     * @var FMC_T::ISPDAT
     * Offset: 0x08  ISP Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ISPDAT    |ISP Data
     * |        |          |Write data to this register before ISP program operation.
     * |        |          |Read data from this register after ISP read operation.
     * |        |          |For Run CRC-32 Checksum Calculation command, ISPDAT is the memory size (byte) and 512 bytes alignment
     * |        |          |For ISP Read CRC-32 Checksum command, ISPDAT is the checksum result
     * |        |          |If ISPDAT = 0x0000_0000, it means that (1) the checksum calculation is in progress, (2) the memory range for checksum calculation is incorrect.
     * @var FMC_T::ISPCMD
     * Offset: 0x0C  ISP CMD Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |CMD       |ISP CMD
     * |        |          |ISP command table is shown below:
     * |        |          |0x00 = FLASH 32-bit Read.
     * |        |          |0x04 = Read Unique ID.
     * |        |          |0x08 = Read All-One Verification Result.
     * |        |          |0x0B = Read Company ID.
     * |        |          |0x0C = Read Device ID.
     * |        |          |0x0D = Read CRC-32 Checksum.
     * |        |          |0x21 = FLASH 32-bit Program.
     * |        |          |0x22 = FLASH Page Erase.
     * |        |          |0x26 = FLASH Mass Erase.
     * |        |          |0x28 = Run All-One Verification.
     * |        |          |0x2D = Run CRC-32 Checksum Calculation.
     * |        |          |0x2E = Vector Remap.
     * |        |          |The other commands are invalid.
     * @var FMC_T::ISPTRG
     * Offset: 0x10  ISP Trigger Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ISPGO     |ISP Start Trigger (Write Protect)
     * |        |          |Write 1 to start ISP operation and this bit will be cleared to 0 by hardware automatically when ISP operation is finished.
     * |        |          |0 = ISP operation is finished.
     * |        |          |1 = ISP is progressed.
     * |        |          |Note:This bit is write protected. Refer to the SYS_REGLCTL register.
     * @var FMC_T::DFBA
     * Offset: 0x14  Data Flash Base Address
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |DFBA      |Data Flash Base Address
     * |        |          |This register indicates Data Flash start address. It is a read only register.
     * |        |          |The Data Flash is shared with APROM. the content of this register is loaded from CONFIG1.
     * |        |          |This register is valid when DFEN (CONFIG0[0]) =0 .
     * @var FMC_T::FTCTL
     * Offset: 0x18  Flash Access Time Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[6:4]   |FOM       |Frequency Optimization Mode (Write Protect)
     * |        |          |The Nano103 series supports adjustable flash access timing to optimize the flash access cycles in different working frequency.
     * |        |          |001 = Frequency <= 20MHz.
     * |        |          |100 = Frequency <= 36MHz. (default power-on setting)
     * |        |          |Others = Reserved
     * |        |          |Note:This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[7]     |CACHEOFF  |Flash Cache Disable Control (Write Protect)
     * |        |          |0 = Flash Cache function Enabled (default).
     * |        |          |1 = Flash Cache function Disabled.
     * |        |          |Note:This bit is write protected. Refer to the SYS_REGLCTL register.
     * @var FMC_T::ISPSTS
     * Offset: 0x40  ISP Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ISPBUSY   |ISP Busy Flag (Read Only)
     * |        |          |Write 1 to start ISP operation and this bit will be cleared to 0 by hardware automatically when ISP operation is finished.
     * |        |          |This bit is the mirror of ISPGO(FMC_ISPTRG[0]).
     * |        |          |0 = ISP operation is finished.
     * |        |          |1 = ISP is progressed.
     * |[2:1]   |CBS       |Boot Selection of CONFIG (Read Only)
     * |        |          |This bit is initiated with the CBS (CONFIG0 [7:6]) after any reset is happened except CPU reset (CPURF(SYS_RSTSTS[7]) is 1) or system reset (SYSRF(SYS_RSTSTS[5]) is happened.
     * |        |          |00 = LDROM with IAP mode.
     * |        |          |01 = LDROM without IAP mode.
     * |        |          |10 = APROM with IAP mode.
     * |        |          |11 = APROM without IAP mode.
     * |[5]     |PGFF      |Flash Program with Fast Verification Flag(Read Only)
     * |        |          |This bit is set if data is mismatched at ISP programming verification
     * |        |          |This bit is clear by performing ISP flash erase or ISP read CID operation
     * |        |          |0 = Flash Program is success.
     * |        |          |1 = Flash Program is fail. Program data is different with data in the flash memory.
     * |[6]     |ISPFF     |ISP Fail Flag (Write Protect)
     * |        |          |This bit is the mirror of ISPFF (FMC_ISPCTL[6]), it needs to be cleared by writing 1 to FMC_ISPCTL[6] or FMC_ISPSTS[6]
     * |        |          |This bit is set by hardware when a triggered ISP meets any of the following conditions:
     * |        |          |(1) APROM writes to itself if APUEN is set to 0.
     * |        |          |(2) LDROM writes to itself if LDUEN is set to 0.
     * |        |          |(3) CONFIG is erased/programmed if CFGUEN is set to 0.
     * |        |          |(4) Page Erase command at LOCK mode with ICE connection
     * |        |          |(5) Erase or Program command at brown-out detected
     * |        |          |(6) Destination address is illegal, such as over an available range.
     * |        |          |(7) Invalid ISP commands
     * |        |          |(8) KPROM is erased/programmed if KEYLOCK is set to 1
     * |        |          |(9) APROM(not include Data Flash) is erased/programmed if KEYLOCK is set to 1
     * |        |          |Note: This bit is write-protected. Refer to the SYS_REGLCTL register.
     * |[7]     |ALLONE    |Flash All-one Verification Flag
     * |        |          |This bit is set by hardware if all of flash bits are 1, and clear if flash bits are not all 1 after "Run Flash All-One Verification" complete; this bit also can be clear by writing 1.
     * |        |          |0 = All of flash bits are 1 after "Run Flash All-One Verification" complete.
     * |        |          |1 = Flash bits are not all 1 after "Run Flash All-One Verification" complete.
     * |[29:9]  |VECMAP    |Vector Page Mapping Address (Read Only)
     * |        |          |All access to 0x0000_0000~0x0000_01FF is remapped to the flash memory or SRAM address {VECMAP[20:0], 9'h000} ~ {VECMAP[20:0], 9'h1FF}
     * |        |          |VECMAP [20:19] = 00 system vector address is mapped to flash memory.
     * |        |          |VECMAP [20:19] = 10 system vector address is mapped to SRAM memory.
     * |        |          |VECMAP [18:12] should be 0.
     * @var FMC_T::KEY0
     * Offset: 0x50  KEY0 Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |KEY0      |KEY0 Data (Write Only)
     * |        |          |Write KEY0 data to this register before KEY Comparison operation.
     * @var FMC_T::KEY1
     * Offset: 0x54  KEY1 Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |KEY1      |KEY1 Data (Write Only)
     * |        |          |Write KEY1 data to this register before KEY Comparison operation.
     * @var FMC_T::KEY2
     * Offset: 0x58  KEY2 Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |KEY2      |KEY2 Data (Write Only)
     * |        |          |Write KEY2 data to this register before KEY Comparison operation.
     * @var FMC_T::KEYTRG
     * Offset: 0x5C  KEY Comparison Trigger Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |KEYGO     |KEY Comparison Start Trigger (Write Protection)
     * |        |          |Write 1 to start KEY comparison operation and this bit will be cleared to 0 by hardware automatically when KEY comparison operation is finished.
     * |        |          |This trigger operation is valid while FORBID (FMC_KEYSTS [3]) is 0.
     * |        |          |0 = KEY comparison operation is finished.
     * |        |          |1 = KEY comparison is progressed.
     * |        |          |Note:This bit is write-protected. Refer to the SYS_REGLCTL register.
     * |[1]     |TCEN      |Time-out Counting Enable Bit (Write Protection)
     * |        |          |0 = Time-out counting Disabled.
     * |        |          |1 = Time-out counting Enabled if key is matched after key comparison finish.
     * |        |          |10 minutes is at least for time-out, and average is about 20 minutes.
     * |        |          |Note:This bit is write-protected. Refer to the SYS_REGLCTL register.
     * @var FMC_T::KEYSTS
     * Offset: 0x60  KEY Comparison Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |KEYBUSY   |KEY Comparison Busy (Read Only)
     * |        |          |0 = KEY comparison is finished.
     * |        |          |1 = KEY comparison is busy.
     * |[1]     |KEYLOCK   |KEY LOCK Flag
     * |        |          |This bit is set to 1 if KEYMATCH (FMC_KEYSTS [2]) is 0 and cleared to 0 if KEYMATCH is 1 in Security Key protection
     * |        |          |This bit also can be set to 1 while
     * |        |          |l CPU write 1 to KEYLOCK(FMC_KEYSTS[1]) or
     * |        |          |l KEYFLAG(FMC_KEYSTS[4]) is 1 at power-on or reset or
     * |        |          |l KEYENROM is programmed a non-0xFF value or
     * |        |          |l Time-out event or
     * |        |          |l FORBID(FMC_KEYSTS[3]) is 1
     * |        |          |0 = KPROM and APROM (not include Data Flash) is not in write protection.
     * |        |          |1 = KPROM and APROM (not include Data Flash) is in write protection.
     * |        |          |CONFIG write protect is depended on CFGFLAG
     * |[2]     |KEYMATCH  |KEY Match Flag(Read Only)
     * |        |          |This bit is set to 1 after KEY comparison complete if the KEY0, KEY1 and KEY2 are matched with the 96-bit security keys in KPROM; and cleared to 0 if KEYs are unmatched
     * |        |          |This bit is also cleared to 0 while
     * |        |          |l CPU writing 1 to KEYLOCK(FMC_KEYSTS[1]) or
     * |        |          |l Time-out event or
     * |        |          |l KPROM is erased or
     * |        |          |l KEYENROM is programmed to a non-0xFF value.
     * |        |          |l Chip is in Power-down mode.
     * |        |          |0 = KEY0, KEY1, and KEY2 are unmatched with the KPROM setting.
     * |        |          |1 = KEY0, KEY1, and KEY2 are matched with the KPROM setting.
     * |[3]     |FORBID    |KEY Comparison Forbidden Flag(Read Only)
     * |        |          |This bit is set to 1 whenKECNT(FMC_KECNT[4:0])is more than KEMAX (FMC_KECNT[12:8]) orKPCNT (FMC_KPCNT [2:0])is more than KPMAX (FMC_KPCNT [10:8]).
     * |        |          |0 = KEY comparison is not forbidden.
     * |        |          |1 = KEY comparison is forbidden, KEYGO (FMC_KEYTRG [0]) cannot trigger.
     * |[4]     |KEYFLAG   |KEY Protection Enable Flag(Read Only)
     * |        |          |This bit is set while the KEYENROM [7:0] is not 0xFF at power-on or reset
     * |        |          |This bit is cleared to 0 by hardware while KPROM is erased
     * |        |          |This bit is set to 1 by hardware while KEYENROM is programmed to a non-0xFF value.
     * |        |          |0 = Security Key protection Disabled.
     * |        |          |1 = Security KeyprotectionEnabled.
     * |[5]     |CFGFLAG   |CONFIG Write-protection Enable Flag(Read Only)
     * |        |          |This bit is set while the KEYENROM [0] is 0 at power-on or reset
     * |        |          |This bit is cleared to 0 by hardware while KPROM is erased
     * |        |          |This bit is set to 1 by hardware while KEYENROM[0] is programmed to 0.
     * |        |          |0 = CONFIG write-protection Disabled.
     * |        |          |1 = CONFIG write-protection Enabled.
     * @var FMC_T::KECNT
     * Offset: 0x64  KEY-Unmatched Counting Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |KECNT     |Error Key Entry Counter at Each Power-on (Read Only)
     * |        |          |KECNT is increased when entry keys is wrong in Security Key protection.
     * |        |          |KECNT is cleared to 0 if key comparison is matched or system power-on.
     * |[13:8]  |KEMAX     |Maximum Number for Error Key Entry at Each Power-on (Read Only).
     * |        |          |KEMAX is the maximum error key entry number at each power-on.
     * |        |          |When KEMAXROM of KPROM is erased or programmed, KEMAX will also be updated.
     * |        |          |KEMAX is used to limit KECNT(FMC_KECNT[5:0]) maximum counting.
     * |        |          |The FORBID (FMC_KEYSTS [3]) will be set to 1 when KECNT is more than KEMAX.
     * @var FMC_T::KPCNT
     * Offset: 0x68  KEY-Unmatched Power-on Counting Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |KPCNT     |Power-on Counter for Error Key Entry(Read Only).
     * |        |          |KPCNT is the power-on counting for error key entry in Security Key protection.
     * |        |          |KPCNT is cleared to 0 if key comparison is matched.
     * |[11:8]  |KPMAX     |Power-on Maximum Number for Error Key Entry (Read Only).
     * |        |          |KPMAX is the power-on maximum number for error key entry.
     * |        |          |When KPMAXROM of KPROM is erased or programmed, KPMAX will also be updated.
     * |        |          |KPMAX is used to limit KPCNT (FMC_KPCNT [3:0]) maximum counting.
     * |        |          |The FORBID(FMC_KEYSTS[3]) will be set to 1 when KPCNT is more than KPMAX.
     */
    __IO uint32_t ISPCTL;                /*!< [0x0000] ISP Control Register                                             */
    __IO uint32_t ISPADDR;               /*!< [0x0004] ISP Address Register                                             */
    __IO uint32_t ISPDAT;                /*!< [0x0008] ISP Data Register                                                */
    __IO uint32_t ISPCMD;                /*!< [0x000c] ISP CMD Register                                                 */
    __IO uint32_t ISPTRG;                /*!< [0x0010] ISP Trigger Control Register                                     */
    __I  uint32_t DFBA;                  /*!< [0x0014] Data Flash Base Address                                          */
    __IO uint32_t FTCTL;                 /*!< [0x0018] Flash Access Time Control Register                               */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE0[9];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t ISPSTS;                /*!< [0x0040] ISP Status Register                                              */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE1[3];
    /// @endcond //HIDDEN_SYMBOLS
    __O  uint32_t KEY0;                  /*!< [0x0050] KEY0 Data Register                                               */
    __O  uint32_t KEY1;                  /*!< [0x0054] KEY1 Data Register                                               */
    __O  uint32_t KEY2;                  /*!< [0x0058] KEY2 Data Register                                               */
    __IO uint32_t KEYTRG;                /*!< [0x005c] KEY Comparison Trigger Control Register                          */
    __IO uint32_t KEYSTS;                /*!< [0x0060] KEY Comparison Status Register                                   */
    __I  uint32_t KECNT;                 /*!< [0x0064] KEY-Unmatched Counting Register                                  */
    __I  uint32_t KPCNT;                 /*!< [0x0068] KEY-Unmatched Power-on Counting Register                         */

} FMC_T;

/**
    @addtogroup FMC_CONST FMC Bit Field Definition
    Constant Definitions for FMC Controller
@{ */

#define FMC_ISPCTL_ISPEN_Pos             (0)                                               /*!< FMC_T::ISPCTL: ISPEN Position          */
#define FMC_ISPCTL_ISPEN_Msk             (0x1ul << FMC_ISPCTL_ISPEN_Pos)                   /*!< FMC_T::ISPCTL: ISPEN Mask              */

#define FMC_ISPCTL_BS_Pos                (1)                                               /*!< FMC_T::ISPCTL: BS Position             */
#define FMC_ISPCTL_BS_Msk                (0x1ul << FMC_ISPCTL_BS_Pos)                      /*!< FMC_T::ISPCTL: BS Mask                 */

#define FMC_ISPCTL_APUEN_Pos             (3)                                               /*!< FMC_T::ISPCTL: APUEN Position          */
#define FMC_ISPCTL_APUEN_Msk             (0x1ul << FMC_ISPCTL_APUEN_Pos)                   /*!< FMC_T::ISPCTL: APUEN Mask              */

#define FMC_ISPCTL_CFGUEN_Pos            (4)                                               /*!< FMC_T::ISPCTL: CFGUEN Position         */
#define FMC_ISPCTL_CFGUEN_Msk            (0x1ul << FMC_ISPCTL_CFGUEN_Pos)                  /*!< FMC_T::ISPCTL: CFGUEN Mask             */

#define FMC_ISPCTL_LDUEN_Pos             (5)                                               /*!< FMC_T::ISPCTL: LDUEN Position          */
#define FMC_ISPCTL_LDUEN_Msk             (0x1ul << FMC_ISPCTL_LDUEN_Pos)                   /*!< FMC_T::ISPCTL: LDUEN Mask              */

#define FMC_ISPCTL_ISPFF_Pos             (6)                                               /*!< FMC_T::ISPCTL: ISPFF Position          */
#define FMC_ISPCTL_ISPFF_Msk             (0x1ul << FMC_ISPCTL_ISPFF_Pos)                   /*!< FMC_T::ISPCTL: ISPFF Mask              */

#define FMC_ISPADDR_ISPADDR_Pos          (0)                                               /*!< FMC_T::ISPADDR: ISPADDR Position       */
#define FMC_ISPADDR_ISPADDR_Msk          (0xfffffffful << FMC_ISPADDR_ISPADDR_Pos)         /*!< FMC_T::ISPADDR: ISPADDR Mask           */

#define FMC_ISPDAT_ISPDAT_Pos            (0)                                               /*!< FMC_T::ISPDAT: ISPDAT Position         */
#define FMC_ISPDAT_ISPDAT_Msk            (0xfffffffful << FMC_ISPDAT_ISPDAT_Pos)           /*!< FMC_T::ISPDAT: ISPDAT Mask             */

#define FMC_ISPCMD_CMD_Pos               (0)                                               /*!< FMC_T::ISPCMD: CMD Position            */
#define FMC_ISPCMD_CMD_Msk               (0x3ful << FMC_ISPCMD_CMD_Pos)                    /*!< FMC_T::ISPCMD: CMD Mask                */

#define FMC_ISPTRG_ISPGO_Pos             (0)                                               /*!< FMC_T::ISPTRG: ISPGO Position          */
#define FMC_ISPTRG_ISPGO_Msk             (0x1ul << FMC_ISPTRG_ISPGO_Pos)                   /*!< FMC_T::ISPTRG: ISPGO Mask              */

#define FMC_DFBA_DFBA_Pos                (0)                                               /*!< FMC_T::DFBA: DFBA Position             */
#define FMC_DFBA_DFBA_Msk                (0xfffffffful << FMC_DFBA_DFBA_Pos)               /*!< FMC_T::DFBA: DFBA Mask                 */

#define FMC_FTCTL_FOM_Pos                (4)                                               /*!< FMC_T::FTCTL: FOM Position             */
#define FMC_FTCTL_FOM_Msk                (0x7ul << FMC_FTCTL_FOM_Pos)                      /*!< FMC_T::FTCTL: FOM Mask                 */

#define FMC_FTCTL_CACHEOFF_Pos           (7)                                               /*!< FMC_T::FTCTL: CACHEOFF Position        */
#define FMC_FTCTL_CACHEOFF_Msk           (0x1ul << FMC_FTCTL_CACHEOFF_Pos)                 /*!< FMC_T::FTCTL: CACHEOFF Mask            */

#define FMC_ISPSTS_ISPBUSY_Pos           (0)                                               /*!< FMC_T::ISPSTS: ISPBUSY Position        */
#define FMC_ISPSTS_ISPBUSY_Msk           (0x1ul << FMC_ISPSTS_ISPBUSY_Pos)                 /*!< FMC_T::ISPSTS: ISPBUSY Mask            */

#define FMC_ISPSTS_CBS_Pos               (1)                                               /*!< FMC_T::ISPSTS: CBS Position            */
#define FMC_ISPSTS_CBS_Msk               (0x3ul << FMC_ISPSTS_CBS_Pos)                     /*!< FMC_T::ISPSTS: CBS Mask                */

#define FMC_ISPSTS_PGFF_Pos              (5)                                               /*!< FMC_T::ISPSTS: PGFF Position           */
#define FMC_ISPSTS_PGFF_Msk              (0x1ul << FMC_ISPSTS_PGFF_Pos)                    /*!< FMC_T::ISPSTS: PGFF Mask               */

#define FMC_ISPSTS_ISPFF_Pos             (6)                                               /*!< FMC_T::ISPSTS: ISPFF Position          */
#define FMC_ISPSTS_ISPFF_Msk             (0x1ul << FMC_ISPSTS_ISPFF_Pos)                   /*!< FMC_T::ISPSTS: ISPFF Mask              */

#define FMC_ISPSTS_ALLONE_Pos            (7)                                               /*!< FMC_T::ISPSTS: ALLONE Position         */
#define FMC_ISPSTS_ALLONE_Msk            (0x1ul << FMC_ISPSTS_ALLONE_Pos)                  /*!< FMC_T::ISPSTS: ALLONE Mask             */

#define FMC_ISPSTS_VECMAP_Pos            (9)                                               /*!< FMC_T::ISPSTS: VECMAP Position         */
#define FMC_ISPSTS_VECMAP_Msk            (0x1ffffful << FMC_ISPSTS_VECMAP_Pos)             /*!< FMC_T::ISPSTS: VECMAP Mask             */

#define FMC_KEY0_KEY0_Pos                (0)                                               /*!< FMC_T::KEY0: KEY0 Position             */
#define FMC_KEY0_KEY0_Msk                (0xfffffffful << FMC_KEY0_KEY0_Pos)               /*!< FMC_T::KEY0: KEY0 Mask                 */

#define FMC_KEY1_KEY1_Pos                (0)                                               /*!< FMC_T::KEY1: KEY1 Position             */
#define FMC_KEY1_KEY1_Msk                (0xfffffffful << FMC_KEY1_KEY1_Pos)               /*!< FMC_T::KEY1: KEY1 Mask                 */

#define FMC_KEY2_KEY2_Pos                (0)                                               /*!< FMC_T::KEY2: KEY2 Position             */
#define FMC_KEY2_KEY2_Msk                (0xfffffffful << FMC_KEY2_KEY2_Pos)               /*!< FMC_T::KEY2: KEY2 Mask                 */

#define FMC_KEYTRG_KEYGO_Pos             (0)                                               /*!< FMC_T::KEYTRG: KEYGO Position          */
#define FMC_KEYTRG_KEYGO_Msk             (0x1ul << FMC_KEYTRG_KEYGO_Pos)                   /*!< FMC_T::KEYTRG: KEYGO Mask              */

#define FMC_KEYTRG_TCEN_Pos              (1)                                               /*!< FMC_T::KEYTRG: TCEN Position           */
#define FMC_KEYTRG_TCEN_Msk              (0x1ul << FMC_KEYTRG_TCEN_Pos)                    /*!< FMC_T::KEYTRG: TCEN Mask               */

#define FMC_KEYSTS_KEYBUSY_Pos           (0)                                               /*!< FMC_T::KEYSTS: KEYBUSY Position        */
#define FMC_KEYSTS_KEYBUSY_Msk           (0x1ul << FMC_KEYSTS_KEYBUSY_Pos)                 /*!< FMC_T::KEYSTS: KEYBUSY Mask            */

#define FMC_KEYSTS_KEYLOCK_Pos           (1)                                               /*!< FMC_T::KEYSTS: KEYLOCK Position        */
#define FMC_KEYSTS_KEYLOCK_Msk           (0x1ul << FMC_KEYSTS_KEYLOCK_Pos)                 /*!< FMC_T::KEYSTS: KEYLOCK Mask            */

#define FMC_KEYSTS_KEYMATCH_Pos          (2)                                               /*!< FMC_T::KEYSTS: KEYMATCH Position       */
#define FMC_KEYSTS_KEYMATCH_Msk          (0x1ul << FMC_KEYSTS_KEYMATCH_Pos)                /*!< FMC_T::KEYSTS: KEYMATCH Mask           */

#define FMC_KEYSTS_FORBID_Pos            (3)                                               /*!< FMC_T::KEYSTS: FORBID Position         */
#define FMC_KEYSTS_FORBID_Msk            (0x1ul << FMC_KEYSTS_FORBID_Pos)                  /*!< FMC_T::KEYSTS: FORBID Mask             */

#define FMC_KEYSTS_KEYFLAG_Pos           (4)                                               /*!< FMC_T::KEYSTS: KEYFLAG Position        */
#define FMC_KEYSTS_KEYFLAG_Msk           (0x1ul << FMC_KEYSTS_KEYFLAG_Pos)                 /*!< FMC_T::KEYSTS: KEYFLAG Mask            */

#define FMC_KEYSTS_CFGFLAG_Pos           (5)                                               /*!< FMC_T::KEYSTS: CFGFLAG Position        */
#define FMC_KEYSTS_CFGFLAG_Msk           (0x1ul << FMC_KEYSTS_CFGFLAG_Pos)                 /*!< FMC_T::KEYSTS: CFGFLAG Mask            */

#define FMC_KECNT_KECNT_Pos              (0)                                               /*!< FMC_T::KECNT: KECNT Position           */
#define FMC_KECNT_KECNT_Msk              (0x3ful << FMC_KECNT_KECNT_Pos)                   /*!< FMC_T::KECNT: KECNT Mask               */

#define FMC_KECNT_KEMAX_Pos              (8)                                               /*!< FMC_T::KECNT: KEMAX Position           */
#define FMC_KECNT_KEMAX_Msk              (0x3ful << FMC_KECNT_KEMAX_Pos)                   /*!< FMC_T::KECNT: KEMAX Mask               */

#define FMC_KPCNT_KPCNT_Pos              (0)                                               /*!< FMC_T::KPCNT: KPCNT Position           */
#define FMC_KPCNT_KPCNT_Msk              (0xful << FMC_KPCNT_KPCNT_Pos)                    /*!< FMC_T::KPCNT: KPCNT Mask               */

#define FMC_KPCNT_KPMAX_Pos              (8)                                               /*!< FMC_T::KPCNT: KPMAX Position           */
#define FMC_KPCNT_KPMAX_Msk              (0xful << FMC_KPCNT_KPMAX_Pos)                    /*!< FMC_T::KPCNT: KPMAX Mask               */

/**@}*/ /* FMC_CONST */
/**@}*/ /* end of FMC register group */


/*---------------------- General Purpose Input/Output Controller -------------------------*/
/**
    @addtogroup GPIO General Purpose Input/Output Controller(GPIO)
    Memory Mapped Structure for GPIO Controller
@{ */

typedef struct
{


    /**
     * @var GPIO_T::MODE
     * Offset: 0x00  Pn I/O Mode Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |MODE0     |Port A-f I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Reserved.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[3:2]   |MODE1     |Port A-f I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Reserved.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[5:4]   |MODE2     |Port A-f I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Reserved.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[7:6]   |MODE3     |Port A-f I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Reserved.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[9:8]   |MODE4     |Port A-f I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Reserved.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[11:10] |MODE5     |Port A-f I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Reserved.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[13:12] |MODE6     |Port A-f I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Reserved.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[15:14] |MODE7     |Port A-f I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Reserved.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[17:16] |MODE8     |Port A-f I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Reserved.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[19:18] |MODE9     |Port A-f I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Reserved.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[21:20] |MODE10    |Port A-f I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Reserved.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[23:22] |MODE11    |Port A-f I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Reserved.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[25:24] |MODE12    |Port A-f I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Reserved.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[27:26] |MODE13    |Port A-f I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Reserved.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[29:28] |MODE14    |Port A-f I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Reserved.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[31:30] |MODE15    |Port A-f I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Reserved.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * @var GPIO_T::DINOFF
     * Offset: 0x04  Pn Digital Input Path Disable Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[16]    |DINOFF0   |Port A-f Pin[N] Digital Input Path Disable Bit
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[17]    |DINOFF1   |Port A-f Pin[N] Digital Input Path Disable Bit
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[18]    |DINOFF2   |Port A-f Pin[N] Digital Input Path Disable Bit
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[19]    |DINOFF3   |Port A-f Pin[N] Digital Input Path Disable Bit
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[20]    |DINOFF4   |Port A-f Pin[N] Digital Input Path Disable Bit
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[21]    |DINOFF5   |Port A-f Pin[N] Digital Input Path Disable Bit
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[22]    |DINOFF6   |Port A-f Pin[N] Digital Input Path Disable Bit
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[23]    |DINOFF7   |Port A-f Pin[N] Digital Input Path Disable Bit
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[24]    |DINOFF8   |Port A-f Pin[N] Digital Input Path Disable Bit
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[25]    |DINOFF9   |Port A-f Pin[N] Digital Input Path Disable Bit
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[26]    |DINOFF10  |Port A-f Pin[N] Digital Input Path Disable Bit
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[27]    |DINOFF11  |Port A-f Pin[N] Digital Input Path Disable Bit
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[28]    |DINOFF12  |Port A-f Pin[N] Digital Input Path Disable Bit
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[29]    |DINOFF13  |Port A-f Pin[N] Digital Input Path Disable Bit
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[30]    |DINOFF14  |Port A-f Pin[N] Digital Input Path Disable Bit
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[31]    |DINOFF15  |Port A-f Pin[N] Digital Input Path Disable Bit
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * @var GPIO_T::DOUT
     * Offset: 0x08  Pn Data Output Value
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |DOUT0     |Port A-f Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[1]     |DOUT1     |Port A-f Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[2]     |DOUT2     |Port A-f Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[3]     |DOUT3     |Port A-f Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[4]     |DOUT4     |Port A-f Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[5]     |DOUT5     |Port A-f Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[6]     |DOUT6     |Port A-f Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[7]     |DOUT7     |Port A-f Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[8]     |DOUT8     |Port A-f Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[9]     |DOUT9     |Port A-f Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[10]    |DOUT10    |Port A-f Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[11]    |DOUT11    |Port A-f Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[12]    |DOUT12    |Port A-f Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[13]    |DOUT13    |Port A-f Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[14]    |DOUT14    |Port A-f Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[15]    |DOUT15    |Port A-f Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * @var GPIO_T::DATMSK
     * Offset: 0x0C  Pn Data Output Write Mask
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |DMASK0    |Port A-f Pin[N] Data Output Write Mask
     * |        |          |These bits are used to protect the corresponding DOUT (Px_DOUT[n]) bit
     * |        |          |When the DATMSK (Px_DATMSK[n]) bit is set to 1, the corresponding DOUT (Px_DOUT[n]) bit is protected
     * |        |          |If the write signal is masked, writing data to the protect bit is ignored.
     * |        |          |0 = Corresponding DOUT (Px_DOUT[n]) bit can be updated.
     * |        |          |1 = Corresponding DOUT (Px_DOUT[n]) bit protected.
     * |        |          |Note1: This function only protects the corresponding DOUT (Px_DOUT[n]) bit, and will not protect the corresponding PDIO (Pxn_PDIO[0]) bit.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[1]     |DMASK1    |Port A-f Pin[N] Data Output Write Mask
     * |        |          |These bits are used to protect the corresponding DOUT (Px_DOUT[n]) bit
     * |        |          |When the DATMSK (Px_DATMSK[n]) bit is set to 1, the corresponding DOUT (Px_DOUT[n]) bit is protected
     * |        |          |If the write signal is masked, writing data to the protect bit is ignored.
     * |        |          |0 = Corresponding DOUT (Px_DOUT[n]) bit can be updated.
     * |        |          |1 = Corresponding DOUT (Px_DOUT[n]) bit protected.
     * |        |          |Note1: This function only protects the corresponding DOUT (Px_DOUT[n]) bit, and will not protect the corresponding PDIO (Pxn_PDIO[0]) bit.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[2]     |DMASK2    |Port A-f Pin[N] Data Output Write Mask
     * |        |          |These bits are used to protect the corresponding DOUT (Px_DOUT[n]) bit
     * |        |          |When the DATMSK (Px_DATMSK[n]) bit is set to 1, the corresponding DOUT (Px_DOUT[n]) bit is protected
     * |        |          |If the write signal is masked, writing data to the protect bit is ignored.
     * |        |          |0 = Corresponding DOUT (Px_DOUT[n]) bit can be updated.
     * |        |          |1 = Corresponding DOUT (Px_DOUT[n]) bit protected.
     * |        |          |Note1: This function only protects the corresponding DOUT (Px_DOUT[n]) bit, and will not protect the corresponding PDIO (Pxn_PDIO[0]) bit.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[3]     |DMASK3    |Port A-f Pin[N] Data Output Write Mask
     * |        |          |These bits are used to protect the corresponding DOUT (Px_DOUT[n]) bit
     * |        |          |When the DATMSK (Px_DATMSK[n]) bit is set to 1, the corresponding DOUT (Px_DOUT[n]) bit is protected
     * |        |          |If the write signal is masked, writing data to the protect bit is ignored.
     * |        |          |0 = Corresponding DOUT (Px_DOUT[n]) bit can be updated.
     * |        |          |1 = Corresponding DOUT (Px_DOUT[n]) bit protected.
     * |        |          |Note1: This function only protects the corresponding DOUT (Px_DOUT[n]) bit, and will not protect the corresponding PDIO (Pxn_PDIO[0]) bit.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[4]     |DMASK4    |Port A-f Pin[N] Data Output Write Mask
     * |        |          |These bits are used to protect the corresponding DOUT (Px_DOUT[n]) bit
     * |        |          |When the DATMSK (Px_DATMSK[n]) bit is set to 1, the corresponding DOUT (Px_DOUT[n]) bit is protected
     * |        |          |If the write signal is masked, writing data to the protect bit is ignored.
     * |        |          |0 = Corresponding DOUT (Px_DOUT[n]) bit can be updated.
     * |        |          |1 = Corresponding DOUT (Px_DOUT[n]) bit protected.
     * |        |          |Note1: This function only protects the corresponding DOUT (Px_DOUT[n]) bit, and will not protect the corresponding PDIO (Pxn_PDIO[0]) bit.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[5]     |DMASK5    |Port A-f Pin[N] Data Output Write Mask
     * |        |          |These bits are used to protect the corresponding DOUT (Px_DOUT[n]) bit
     * |        |          |When the DATMSK (Px_DATMSK[n]) bit is set to 1, the corresponding DOUT (Px_DOUT[n]) bit is protected
     * |        |          |If the write signal is masked, writing data to the protect bit is ignored.
     * |        |          |0 = Corresponding DOUT (Px_DOUT[n]) bit can be updated.
     * |        |          |1 = Corresponding DOUT (Px_DOUT[n]) bit protected.
     * |        |          |Note1: This function only protects the corresponding DOUT (Px_DOUT[n]) bit, and will not protect the corresponding PDIO (Pxn_PDIO[0]) bit.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[6]     |DMASK6    |Port A-f Pin[N] Data Output Write Mask
     * |        |          |These bits are used to protect the corresponding DOUT (Px_DOUT[n]) bit
     * |        |          |When the DATMSK (Px_DATMSK[n]) bit is set to 1, the corresponding DOUT (Px_DOUT[n]) bit is protected
     * |        |          |If the write signal is masked, writing data to the protect bit is ignored.
     * |        |          |0 = Corresponding DOUT (Px_DOUT[n]) bit can be updated.
     * |        |          |1 = Corresponding DOUT (Px_DOUT[n]) bit protected.
     * |        |          |Note1: This function only protects the corresponding DOUT (Px_DOUT[n]) bit, and will not protect the corresponding PDIO (Pxn_PDIO[0]) bit.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[7]     |DMASK7    |Port A-f Pin[N] Data Output Write Mask
     * |        |          |These bits are used to protect the corresponding DOUT (Px_DOUT[n]) bit
     * |        |          |When the DATMSK (Px_DATMSK[n]) bit is set to 1, the corresponding DOUT (Px_DOUT[n]) bit is protected
     * |        |          |If the write signal is masked, writing data to the protect bit is ignored.
     * |        |          |0 = Corresponding DOUT (Px_DOUT[n]) bit can be updated.
     * |        |          |1 = Corresponding DOUT (Px_DOUT[n]) bit protected.
     * |        |          |Note1: This function only protects the corresponding DOUT (Px_DOUT[n]) bit, and will not protect the corresponding PDIO (Pxn_PDIO[0]) bit.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[8]     |DMASK8    |Port A-f Pin[N] Data Output Write Mask
     * |        |          |These bits are used to protect the corresponding DOUT (Px_DOUT[n]) bit
     * |        |          |When the DATMSK (Px_DATMSK[n]) bit is set to 1, the corresponding DOUT (Px_DOUT[n]) bit is protected
     * |        |          |If the write signal is masked, writing data to the protect bit is ignored.
     * |        |          |0 = Corresponding DOUT (Px_DOUT[n]) bit can be updated.
     * |        |          |1 = Corresponding DOUT (Px_DOUT[n]) bit protected.
     * |        |          |Note1: This function only protects the corresponding DOUT (Px_DOUT[n]) bit, and will not protect the corresponding PDIO (Pxn_PDIO[0]) bit.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[9]     |DMASK9    |Port A-f Pin[N] Data Output Write Mask
     * |        |          |These bits are used to protect the corresponding DOUT (Px_DOUT[n]) bit
     * |        |          |When the DATMSK (Px_DATMSK[n]) bit is set to 1, the corresponding DOUT (Px_DOUT[n]) bit is protected
     * |        |          |If the write signal is masked, writing data to the protect bit is ignored.
     * |        |          |0 = Corresponding DOUT (Px_DOUT[n]) bit can be updated.
     * |        |          |1 = Corresponding DOUT (Px_DOUT[n]) bit protected.
     * |        |          |Note1: This function only protects the corresponding DOUT (Px_DOUT[n]) bit, and will not protect the corresponding PDIO (Pxn_PDIO[0]) bit.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[10]    |DMASK10   |Port A-f Pin[N] Data Output Write Mask
     * |        |          |These bits are used to protect the corresponding DOUT (Px_DOUT[n]) bit
     * |        |          |When the DATMSK (Px_DATMSK[n]) bit is set to 1, the corresponding DOUT (Px_DOUT[n]) bit is protected
     * |        |          |If the write signal is masked, writing data to the protect bit is ignored.
     * |        |          |0 = Corresponding DOUT (Px_DOUT[n]) bit can be updated.
     * |        |          |1 = Corresponding DOUT (Px_DOUT[n]) bit protected.
     * |        |          |Note1: This function only protects the corresponding DOUT (Px_DOUT[n]) bit, and will not protect the corresponding PDIO (Pxn_PDIO[0]) bit.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[11]    |DMASK11   |Port A-f Pin[N] Data Output Write Mask
     * |        |          |These bits are used to protect the corresponding DOUT (Px_DOUT[n]) bit
     * |        |          |When the DATMSK (Px_DATMSK[n]) bit is set to 1, the corresponding DOUT (Px_DOUT[n]) bit is protected
     * |        |          |If the write signal is masked, writing data to the protect bit is ignored.
     * |        |          |0 = Corresponding DOUT (Px_DOUT[n]) bit can be updated.
     * |        |          |1 = Corresponding DOUT (Px_DOUT[n]) bit protected.
     * |        |          |Note1: This function only protects the corresponding DOUT (Px_DOUT[n]) bit, and will not protect the corresponding PDIO (Pxn_PDIO[0]) bit.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[12]    |DMASK12   |Port A-f Pin[N] Data Output Write Mask
     * |        |          |These bits are used to protect the corresponding DOUT (Px_DOUT[n]) bit
     * |        |          |When the DATMSK (Px_DATMSK[n]) bit is set to 1, the corresponding DOUT (Px_DOUT[n]) bit is protected
     * |        |          |If the write signal is masked, writing data to the protect bit is ignored.
     * |        |          |0 = Corresponding DOUT (Px_DOUT[n]) bit can be updated.
     * |        |          |1 = Corresponding DOUT (Px_DOUT[n]) bit protected.
     * |        |          |Note1: This function only protects the corresponding DOUT (Px_DOUT[n]) bit, and will not protect the corresponding PDIO (Pxn_PDIO[0]) bit.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[13]    |DMASK13   |Port A-f Pin[N] Data Output Write Mask
     * |        |          |These bits are used to protect the corresponding DOUT (Px_DOUT[n]) bit
     * |        |          |When the DATMSK (Px_DATMSK[n]) bit is set to 1, the corresponding DOUT (Px_DOUT[n]) bit is protected
     * |        |          |If the write signal is masked, writing data to the protect bit is ignored.
     * |        |          |0 = Corresponding DOUT (Px_DOUT[n]) bit can be updated.
     * |        |          |1 = Corresponding DOUT (Px_DOUT[n]) bit protected.
     * |        |          |Note1: This function only protects the corresponding DOUT (Px_DOUT[n]) bit, and will not protect the corresponding PDIO (Pxn_PDIO[0]) bit.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[14]    |DMASK14   |Port A-f Pin[N] Data Output Write Mask
     * |        |          |These bits are used to protect the corresponding DOUT (Px_DOUT[n]) bit
     * |        |          |When the DATMSK (Px_DATMSK[n]) bit is set to 1, the corresponding DOUT (Px_DOUT[n]) bit is protected
     * |        |          |If the write signal is masked, writing data to the protect bit is ignored.
     * |        |          |0 = Corresponding DOUT (Px_DOUT[n]) bit can be updated.
     * |        |          |1 = Corresponding DOUT (Px_DOUT[n]) bit protected.
     * |        |          |Note1: This function only protects the corresponding DOUT (Px_DOUT[n]) bit, and will not protect the corresponding PDIO (Pxn_PDIO[0]) bit.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[15]    |DMASK15   |Port A-f Pin[N] Data Output Write Mask
     * |        |          |These bits are used to protect the corresponding DOUT (Px_DOUT[n]) bit
     * |        |          |When the DATMSK (Px_DATMSK[n]) bit is set to 1, the corresponding DOUT (Px_DOUT[n]) bit is protected
     * |        |          |If the write signal is masked, writing data to the protect bit is ignored.
     * |        |          |0 = Corresponding DOUT (Px_DOUT[n]) bit can be updated.
     * |        |          |1 = Corresponding DOUT (Px_DOUT[n]) bit protected.
     * |        |          |Note1: This function only protects the corresponding DOUT (Px_DOUT[n]) bit, and will not protect the corresponding PDIO (Pxn_PDIO[0]) bit.
     * |        |          |Note2:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note3:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * @var GPIO_T::PIN
     * Offset: 0x10  Pn Pin Value
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |PIN0      |Port A-f Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[1]     |PIN1      |Port A-f Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[2]     |PIN2      |Port A-f Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[3]     |PIN3      |Port A-f Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[4]     |PIN4      |Port A-f Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[5]     |PIN5      |Port A-f Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[6]     |PIN6      |Port A-f Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[7]     |PIN7      |Port A-f Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[8]     |PIN8      |Port A-f Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[9]     |PIN9      |Port A-f Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[10]    |PIN10     |Port A-f Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[11]    |PIN11     |Port A-f Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[12]    |PIN12     |Port A-f Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[13]    |PIN13     |Port A-f Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[14]    |PIN14     |Port A-f Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[15]    |PIN15     |Port A-f Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * @var GPIO_T::DBEN
     * Offset: 0x14  Pn De-Bounce Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |DBEN0     |Port A-f Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[1]     |DBEN1     |Port A-f Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[2]     |DBEN2     |Port A-f Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[3]     |DBEN3     |Port A-f Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[4]     |DBEN4     |Port A-f Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[5]     |DBEN5     |Port A-f Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[6]     |DBEN6     |Port A-f Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[7]     |DBEN7     |Port A-f Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[8]     |DBEN8     |Port A-f Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[9]     |DBEN9     |Port A-f Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[10]    |DBEN10    |Port A-f Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[11]    |DBEN11    |Port A-f Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[12]    |DBEN12    |Port A-f Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[13]    |DBEN13    |Port A-f Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[14]    |DBEN14    |Port A-f Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[15]    |DBEN15    |Port A-f Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * @var GPIO_T::INTTYPE
     * Offset: 0x18  Pn Interrupt Trigger Type Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TYPE0     |Port A-f Pin[N] Edge or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n])
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[1]     |TYPE1     |Port A-f Pin[N] Edge or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n])
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[2]     |TYPE2     |Port A-f Pin[N] Edge or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n])
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[3]     |TYPE3     |Port A-f Pin[N] Edge or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n])
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[4]     |TYPE4     |Port A-f Pin[N] Edge or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n])
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[5]     |TYPE5     |Port A-f Pin[N] Edge or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n])
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[6]     |TYPE6     |Port A-f Pin[N] Edge or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n])
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[7]     |TYPE7     |Port A-f Pin[N] Edge or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n])
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[8]     |TYPE8     |Port A-f Pin[N] Edge or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n])
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[9]     |TYPE9     |Port A-f Pin[N] Edge or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n])
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[10]    |TYPE10    |Port A-f Pin[N] Edge or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n])
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[11]    |TYPE11    |Port A-f Pin[N] Edge or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n])
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[12]    |TYPE12    |Port A-f Pin[N] Edge or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n])
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[13]    |TYPE13    |Port A-f Pin[N] Edge or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n])
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[14]    |TYPE14    |Port A-f Pin[N] Edge or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n])
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[15]    |TYPE15    |Port A-f Pin[N] Edge or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n])
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignored.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * @var GPIO_T::INTEN
     * Offset: 0x1C  Pn Interrupt Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |FLIEN0    |Port A-f Pin[N] Falling Edge or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger(TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[1]     |FLIEN1    |Port A-f Pin[N] Falling Edge or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger(TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[2]     |FLIEN2    |Port A-f Pin[N] Falling Edge or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger(TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[3]     |FLIEN3    |Port A-f Pin[N] Falling Edge or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger(TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[4]     |FLIEN4    |Port A-f Pin[N] Falling Edge or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger(TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[5]     |FLIEN5    |Port A-f Pin[N] Falling Edge or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger(TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[6]     |FLIEN6    |Port A-f Pin[N] Falling Edge or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger(TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[7]     |FLIEN7    |Port A-f Pin[N] Falling Edge or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger(TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[8]     |FLIEN8    |Port A-f Pin[N] Falling Edge or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger(TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[9]     |FLIEN9    |Port A-f Pin[N] Falling Edge or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger(TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[10]    |FLIEN10   |Port A-f Pin[N] Falling Edge or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger(TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[11]    |FLIEN11   |Port A-f Pin[N] Falling Edge or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger(TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[12]    |FLIEN12   |Port A-f Pin[N] Falling Edge or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger(TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[13]    |FLIEN13   |Port A-f Pin[N] Falling Edge or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger(TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[14]    |FLIEN14   |Port A-f Pin[N] Falling Edge or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger(TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[15]    |FLIEN15   |Port A-f Pin[N] Falling Edge or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger(TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[16]    |RHIEN0    |Port A-f Pin[N] Rising Edge or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[17]    |RHIEN1    |Port A-f Pin[N] Rising Edge or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[18]    |RHIEN2    |Port A-f Pin[N] Rising Edge or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[19]    |RHIEN3    |Port A-f Pin[N] Rising Edge or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[20]    |RHIEN4    |Port A-f Pin[N] Rising Edge or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[21]    |RHIEN5    |Port A-f Pin[N] Rising Edge or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[22]    |RHIEN6    |Port A-f Pin[N] Rising Edge or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[23]    |RHIEN7    |Port A-f Pin[N] Rising Edge or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[24]    |RHIEN8    |Port A-f Pin[N] Rising Edge or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[25]    |RHIEN9    |Port A-f Pin[N] Rising Edge or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[26]    |RHIEN10   |Port A-f Pin[N] Rising Edge or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[27]    |RHIEN11   |Port A-f Pin[N] Rising Edge or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[28]    |RHIEN12   |Port A-f Pin[N] Rising Edge or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[29]    |RHIEN13   |Port A-f Pin[N] Rising Edge or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[30]    |RHIEN14   |Port A-f Pin[N] Rising Edge or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[31]    |RHIEN15   |Port A-f Pin[N] Rising Edge or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * @var GPIO_T::INTSRC
     * Offset: 0x20  Pn Interrupt Source Flag
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |INTSRC0   |Port A-f Pin[N] Interrupt Source Flag
     * |        |          |Write Operation :
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation :
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[1]     |INTSRC1   |Port A-f Pin[N] Interrupt Source Flag
     * |        |          |Write Operation :
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation :
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[2]     |INTSRC2   |Port A-f Pin[N] Interrupt Source Flag
     * |        |          |Write Operation :
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation :
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[3]     |INTSRC3   |Port A-f Pin[N] Interrupt Source Flag
     * |        |          |Write Operation :
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation :
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[4]     |INTSRC4   |Port A-f Pin[N] Interrupt Source Flag
     * |        |          |Write Operation :
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation :
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[5]     |INTSRC5   |Port A-f Pin[N] Interrupt Source Flag
     * |        |          |Write Operation :
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation :
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[6]     |INTSRC6   |Port A-f Pin[N] Interrupt Source Flag
     * |        |          |Write Operation :
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation :
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[7]     |INTSRC7   |Port A-f Pin[N] Interrupt Source Flag
     * |        |          |Write Operation :
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation :
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[8]     |INTSRC8   |Port A-f Pin[N] Interrupt Source Flag
     * |        |          |Write Operation :
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation :
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[9]     |INTSRC9   |Port A-f Pin[N] Interrupt Source Flag
     * |        |          |Write Operation :
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation :
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[10]    |INTSRC10  |Port A-f Pin[N] Interrupt Source Flag
     * |        |          |Write Operation :
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation :
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[11]    |INTSRC11  |Port A-f Pin[N] Interrupt Source Flag
     * |        |          |Write Operation :
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation :
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[12]    |INTSRC12  |Port A-f Pin[N] Interrupt Source Flag
     * |        |          |Write Operation :
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation :
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[13]    |INTSRC13  |Port A-f Pin[N] Interrupt Source Flag
     * |        |          |Write Operation :
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation :
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[14]    |INTSRC14  |Port A-f Pin[N] Interrupt Source Flag
     * |        |          |Write Operation :
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation :
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[15]    |INTSRC15  |Port A-f Pin[N] Interrupt Source Flag
     * |        |          |Write Operation :
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation :
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * @var GPIO_T::PUEN
     * Offset: 0x24  Pn Pull-Up Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |PUEN0     |Port A-f Pin[N] Pull-up Enable Bit
     * |        |          |Read :
     * |        |          |0 = Px.n internal pull-up resistor Disabled.
     * |        |          |1 = Px.n internal pull-up resistor Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[1]     |PUEN1     |Port A-f Pin[N] Pull-up Enable Bit
     * |        |          |Read :
     * |        |          |0 = Px.n internal pull-up resistor Disabled.
     * |        |          |1 = Px.n internal pull-up resistor Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[2]     |PUEN2     |Port A-f Pin[N] Pull-up Enable Bit
     * |        |          |Read :
     * |        |          |0 = Px.n internal pull-up resistor Disabled.
     * |        |          |1 = Px.n internal pull-up resistor Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[3]     |PUEN3     |Port A-f Pin[N] Pull-up Enable Bit
     * |        |          |Read :
     * |        |          |0 = Px.n internal pull-up resistor Disabled.
     * |        |          |1 = Px.n internal pull-up resistor Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[4]     |PUEN4     |Port A-f Pin[N] Pull-up Enable Bit
     * |        |          |Read :
     * |        |          |0 = Px.n internal pull-up resistor Disabled.
     * |        |          |1 = Px.n internal pull-up resistor Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[5]     |PUEN5     |Port A-f Pin[N] Pull-up Enable Bit
     * |        |          |Read :
     * |        |          |0 = Px.n internal pull-up resistor Disabled.
     * |        |          |1 = Px.n internal pull-up resistor Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[6]     |PUEN6     |Port A-f Pin[N] Pull-up Enable Bit
     * |        |          |Read :
     * |        |          |0 = Px.n internal pull-up resistor Disabled.
     * |        |          |1 = Px.n internal pull-up resistor Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[7]     |PUEN7     |Port A-f Pin[N] Pull-up Enable Bit
     * |        |          |Read :
     * |        |          |0 = Px.n internal pull-up resistor Disabled.
     * |        |          |1 = Px.n internal pull-up resistor Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[8]     |PUEN8     |Port A-f Pin[N] Pull-up Enable Bit
     * |        |          |Read :
     * |        |          |0 = Px.n internal pull-up resistor Disabled.
     * |        |          |1 = Px.n internal pull-up resistor Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[9]     |PUEN9     |Port A-f Pin[N] Pull-up Enable Bit
     * |        |          |Read :
     * |        |          |0 = Px.n internal pull-up resistor Disabled.
     * |        |          |1 = Px.n internal pull-up resistor Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[10]    |PUEN10    |Port A-f Pin[N] Pull-up Enable Bit
     * |        |          |Read :
     * |        |          |0 = Px.n internal pull-up resistor Disabled.
     * |        |          |1 = Px.n internal pull-up resistor Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[11]    |PUEN11    |Port A-f Pin[N] Pull-up Enable Bit
     * |        |          |Read :
     * |        |          |0 = Px.n internal pull-up resistor Disabled.
     * |        |          |1 = Px.n internal pull-up resistor Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[12]    |PUEN12    |Port A-f Pin[N] Pull-up Enable Bit
     * |        |          |Read :
     * |        |          |0 = Px.n internal pull-up resistor Disabled.
     * |        |          |1 = Px.n internal pull-up resistor Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[13]    |PUEN13    |Port A-f Pin[N] Pull-up Enable Bit
     * |        |          |Read :
     * |        |          |0 = Px.n internal pull-up resistor Disabled.
     * |        |          |1 = Px.n internal pull-up resistor Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[14]    |PUEN14    |Port A-f Pin[N] Pull-up Enable Bit
     * |        |          |Read :
     * |        |          |0 = Px.n internal pull-up resistor Disabled.
     * |        |          |1 = Px.n internal pull-up resistor Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[15]    |PUEN15    |Port A-f Pin[N] Pull-up Enable Bit
     * |        |          |Read :
     * |        |          |0 = Px.n internal pull-up resistor Disabled.
     * |        |          |1 = Px.n internal pull-up resistor Enabled.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * @var GPIO_T::INTSTS
     * Offset: 0x28  Pn Interrupt Status
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |FLISTS0   |Port A-f Pin[N] Falling Edge Interrupt Status
     * |        |          |If the interrupt is falling edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No falling edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an falling edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[1]     |FLISTS1   |Port A-f Pin[N] Falling Edge Interrupt Status
     * |        |          |If the interrupt is falling edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No falling edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an falling edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[2]     |FLISTS2   |Port A-f Pin[N] Falling Edge Interrupt Status
     * |        |          |If the interrupt is falling edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No falling edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an falling edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[3]     |FLISTS3   |Port A-f Pin[N] Falling Edge Interrupt Status
     * |        |          |If the interrupt is falling edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No falling edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an falling edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[4]     |FLISTS4   |Port A-f Pin[N] Falling Edge Interrupt Status
     * |        |          |If the interrupt is falling edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No falling edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an falling edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[5]     |FLISTS5   |Port A-f Pin[N] Falling Edge Interrupt Status
     * |        |          |If the interrupt is falling edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No falling edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an falling edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[6]     |FLISTS6   |Port A-f Pin[N] Falling Edge Interrupt Status
     * |        |          |If the interrupt is falling edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No falling edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an falling edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[7]     |FLISTS7   |Port A-f Pin[N] Falling Edge Interrupt Status
     * |        |          |If the interrupt is falling edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No falling edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an falling edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[8]     |FLISTS8   |Port A-f Pin[N] Falling Edge Interrupt Status
     * |        |          |If the interrupt is falling edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No falling edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an falling edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[9]     |FLISTS9   |Port A-f Pin[N] Falling Edge Interrupt Status
     * |        |          |If the interrupt is falling edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No falling edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an falling edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[10]    |FLISTS10  |Port A-f Pin[N] Falling Edge Interrupt Status
     * |        |          |If the interrupt is falling edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No falling edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an falling edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[11]    |FLISTS11  |Port A-f Pin[N] Falling Edge Interrupt Status
     * |        |          |If the interrupt is falling edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No falling edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an falling edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[12]    |FLISTS12  |Port A-f Pin[N] Falling Edge Interrupt Status
     * |        |          |If the interrupt is falling edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No falling edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an falling edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[13]    |FLISTS13  |Port A-f Pin[N] Falling Edge Interrupt Status
     * |        |          |If the interrupt is falling edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No falling edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an falling edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[14]    |FLISTS14  |Port A-f Pin[N] Falling Edge Interrupt Status
     * |        |          |If the interrupt is falling edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No falling edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an falling edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[15]    |FLISTS15  |Port A-f Pin[N] Falling Edge Interrupt Status
     * |        |          |If the interrupt is falling edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No falling edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an falling edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[16]    |RHISTS0   |Port A-f Pin[N] Rising Edge Interrupt Status
     * |        |          |If the interrupt is rising edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No rising edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an rising edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[17]    |RHISTS1   |Port A-f Pin[N] Rising Edge Interrupt Status
     * |        |          |If the interrupt is rising edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No rising edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an rising edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[18]    |RHISTS2   |Port A-f Pin[N] Rising Edge Interrupt Status
     * |        |          |If the interrupt is rising edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No rising edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an rising edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[19]    |RHISTS3   |Port A-f Pin[N] Rising Edge Interrupt Status
     * |        |          |If the interrupt is rising edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No rising edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an rising edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[20]    |RHISTS4   |Port A-f Pin[N] Rising Edge Interrupt Status
     * |        |          |If the interrupt is rising edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No rising edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an rising edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[21]    |RHISTS5   |Port A-f Pin[N] Rising Edge Interrupt Status
     * |        |          |If the interrupt is rising edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No rising edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an rising edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[22]    |RHISTS6   |Port A-f Pin[N] Rising Edge Interrupt Status
     * |        |          |If the interrupt is rising edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No rising edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an rising edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[23]    |RHISTS7   |Port A-f Pin[N] Rising Edge Interrupt Status
     * |        |          |If the interrupt is rising edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No rising edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an rising edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[24]    |RHISTS8   |Port A-f Pin[N] Rising Edge Interrupt Status
     * |        |          |If the interrupt is rising edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No rising edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an rising edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[25]    |RHISTS9   |Port A-f Pin[N] Rising Edge Interrupt Status
     * |        |          |If the interrupt is rising edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No rising edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an rising edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[26]    |RHISTS10  |Port A-f Pin[N] Rising Edge Interrupt Status
     * |        |          |If the interrupt is rising edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No rising edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an rising edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[27]    |RHISTS11  |Port A-f Pin[N] Rising Edge Interrupt Status
     * |        |          |If the interrupt is rising edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No rising edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an rising edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[28]    |RHISTS12  |Port A-f Pin[N] Rising Edge Interrupt Status
     * |        |          |If the interrupt is rising edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No rising edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an rising edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[29]    |RHISTS13  |Port A-f Pin[N] Rising Edge Interrupt Status
     * |        |          |If the interrupt is rising edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No rising edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an rising edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[30]    |RHISTS14  |Port A-f Pin[N] Rising Edge Interrupt Status
     * |        |          |If the interrupt is rising edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No rising edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an rising edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     * |[31]    |RHISTS15  |Port A-f Pin[N] Rising Edge Interrupt Status
     * |        |          |If the interrupt is rising edge trigger for each of the corresponding input Px.n pin, this bit will be set after interrupt occurred and automatically cleared by interrupt source flag (INTSRC (Px_INTSRC[n])).
     * |        |          |0 = No rising edge interrupt at Px.n.
     * |        |          |1 = Px.n generates an rising edge interrupt.
     * |        |          |Note1:
     * |        |          |Max. n=15 for port A/B/C/D/E.
     * |        |          |Max. n=7 for port F.
     * |        |          |Note2:
     * |        |          |The PA.7/PB.12/PC.4/PC.5/PC.12/PC.13/PD.0/PD.1/PD.2/PD.3/PD.4/PD.5/PD.8/PD.9/
     * |        |          |PD.10/PD.11/PD.12/PD.13/PE.0/PE.1/PE.2/PE.3/PE.4/PE.6/PE.7/PE.8/PE.9/PE.10/PE.11/
     * |        |          |PE.12/PE.13/PE.14/PE.15/PF.4/PF.5 pin is ignored.
     */
    __IO uint32_t MODE;               /*!< [0x0000] Pn I/O Mode Control                                              */
    __IO uint32_t DINOFF;             /*!< [0x0004] Pn Digital Input Path Disable Control                            */
    __IO uint32_t DOUT;               /*!< [0x0008] Pn Data Output Value                                             */
    __IO uint32_t DATMSK;             /*!< [0x000c] Pn Data Output Write Mask                                        */
    __I  uint32_t PIN;                /*!< [0x0010] Pn Pin Value                                                     */
    __IO uint32_t DBEN;               /*!< [0x0014] Pn De-Bounce Enable Control Register                             */
    __IO uint32_t INTTYPE;            /*!< [0x0018] Pn Interrupt Trigger Type Control                                */
    __IO uint32_t INTEN;              /*!< [0x001c] Pn Interrupt Enable Control Register                             */
    __IO uint32_t INTSRC;             /*!< [0x0020] Pn Interrupt Source Flag                                         */
    __IO uint32_t PUEN;               /*!< [0x0024] Pn Pull-Up Enable Control Register                               */
    __I  uint32_t INTSTS;             /*!< [0x0028] Pn Interrupt Status Register                                     */

} GPIO_T;

typedef struct
{

    /**
     * @var GP_DB_T::DBCTL
     * Offset: 0x180  Interrupt De-bounce Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |DBCLKSEL  |De-bounce Sampling Cycle Selection
     * |        |          |0000 = Sample interrupt input once per 1 clocks.
     * |        |          |0001 = Sample interrupt input once per 2 clocks.
     * |        |          |0010 = Sample interrupt input once per 4 clocks.
     * |        |          |0011 = Sample interrupt input once per 8 clocks.
     * |        |          |0100 = Sample interrupt input once per 16 clocks.
     * |        |          |0101 = Sample interrupt input once per 32 clocks.
     * |        |          |0110 = Sample interrupt input once per 64 clocks.
     * |        |          |0111 = Sample interrupt input once per 128 clocks.
     * |        |          |1000 = Sample interrupt input once per 256 clocks.
     * |        |          |1001 = Sample interrupt input once per 2*256 clocks.
     * |        |          |1010 = Sample interrupt input once per 4*256 clocks.
     * |        |          |1011 = Sample interrupt input once per 8*256 clocks.
     * |        |          |1100 = Sample interrupt input once per 16*256 clocks.
     * |        |          |1101 = Sample interrupt input once per 32*256 clocks.
     * |        |          |1110 = Sample interrupt input once per 64*256 clocks.
     * |        |          |1111 = Sample interrupt input once per 128*256 clocks.
     * |[4]     |DBCLKSRC  |De-bounce Counter Clock Source Selection
     * |        |          |0 = De-bounce counter clock source is the HCLK.
     * |        |          |1 = De-bounce counter clock source is the 10 kHz internal low speed RC oscillator (LIRC).
     * |[5]     |ICLKON    |Interrupt Clock on Mode
     * |        |          |0 = Edge detection circuit is active only if I/O pin corresponding RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n]) bit is set to 1.
     * |        |          |1 = All I/O pins edge detection circuit is always active after reset.
     * |        |          |Note: It is recommended to disable this bit to save system power if no special application concern.
     */
    __IO uint32_t DBCTL;            /*!< [0x0180] Interrupt De-bounce Control Register                             */
} GP_DB_T;

/**
    @addtogroup GPIO_CONST GPIO Bit Field Definition
    Constant Definitions for GPIO Controller
@{ */

#define GPIO_MODE_MODE0_Pos              (0)                                               /*!< GPIO_T::MODE: MODE0 Position           */
#define GPIO_MODE_MODE0_Msk              (0x3ul << GPIO_MODE_MODE0_Pos)                    /*!< GPIO_T::MODE: MODE0 Mask               */

#define GPIO_MODE_MODE1_Pos              (2)                                               /*!< GPIO_T::MODE: MODE1 Position           */
#define GPIO_MODE_MODE1_Msk              (0x3ul << GPIO_MODE_MODE1_Pos)                    /*!< GPIO_T::MODE: MODE1 Mask               */

#define GPIO_MODE_MODE2_Pos              (4)                                               /*!< GPIO_T::MODE: MODE2 Position           */
#define GPIO_MODE_MODE2_Msk              (0x3ul << GPIO_MODE_MODE2_Pos)                    /*!< GPIO_T::MODE: MODE2 Mask               */

#define GPIO_MODE_MODE3_Pos              (6)                                               /*!< GPIO_T::MODE: MODE3 Position           */
#define GPIO_MODE_MODE3_Msk              (0x3ul << GPIO_MODE_MODE3_Pos)                    /*!< GPIO_T::MODE: MODE3 Mask               */

#define GPIO_MODE_MODE4_Pos              (8)                                               /*!< GPIO_T::MODE: MODE4 Position           */
#define GPIO_MODE_MODE4_Msk              (0x3ul << GPIO_MODE_MODE4_Pos)                    /*!< GPIO_T::MODE: MODE4 Mask               */

#define GPIO_MODE_MODE5_Pos              (10)                                              /*!< GPIO_T::MODE: MODE5 Position           */
#define GPIO_MODE_MODE5_Msk              (0x3ul << GPIO_MODE_MODE5_Pos)                    /*!< GPIO_T::MODE: MODE5 Mask               */

#define GPIO_MODE_MODE6_Pos              (12)                                              /*!< GPIO_T::MODE: MODE6 Position           */
#define GPIO_MODE_MODE6_Msk              (0x3ul << GPIO_MODE_MODE6_Pos)                    /*!< GPIO_T::MODE: MODE6 Mask               */

#define GPIO_MODE_MODE7_Pos              (14)                                              /*!< GPIO_T::MODE: MODE7 Position           */
#define GPIO_MODE_MODE7_Msk              (0x3ul << GPIO_MODE_MODE7_Pos)                    /*!< GPIO_T::MODE: MODE7 Mask               */

#define GPIO_MODE_MODE8_Pos              (16)                                              /*!< GPIO_T::MODE: MODE8 Position           */
#define GPIO_MODE_MODE8_Msk              (0x3ul << GPIO_MODE_MODE8_Pos)                    /*!< GPIO_T::MODE: MODE8 Mask               */

#define GPIO_MODE_MODE9_Pos              (18)                                              /*!< GPIO_T::MODE: MODE9 Position           */
#define GPIO_MODE_MODE9_Msk              (0x3ul << GPIO_MODE_MODE9_Pos)                    /*!< GPIO_T::MODE: MODE9 Mask               */

#define GPIO_MODE_MODE10_Pos             (20)                                              /*!< GPIO_T::MODE: MODE10 Position          */
#define GPIO_MODE_MODE10_Msk             (0x3ul << GPIO_MODE_MODE10_Pos)                   /*!< GPIO_T::MODE: MODE10 Mask              */

#define GPIO_MODE_MODE11_Pos             (22)                                              /*!< GPIO_T::MODE: MODE11 Position          */
#define GPIO_MODE_MODE11_Msk             (0x3ul << GPIO_MODE_MODE11_Pos)                   /*!< GPIO_T::MODE: MODE11 Mask              */

#define GPIO_MODE_MODE12_Pos             (24)                                              /*!< GPIO_T::MODE: MODE12 Position          */
#define GPIO_MODE_MODE12_Msk             (0x3ul << GPIO_MODE_MODE12_Pos)                   /*!< GPIO_T::MODE: MODE12 Mask              */

#define GPIO_MODE_MODE13_Pos             (26)                                              /*!< GPIO_T::MODE: MODE13 Position          */
#define GPIO_MODE_MODE13_Msk             (0x3ul << GPIO_MODE_MODE13_Pos)                   /*!< GPIO_T::MODE: MODE13 Mask              */

#define GPIO_MODE_MODE14_Pos             (28)                                              /*!< GPIO_T::MODE: MODE14 Position          */
#define GPIO_MODE_MODE14_Msk             (0x3ul << GPIO_MODE_MODE14_Pos)                   /*!< GPIO_T::MODE: MODE14 Mask              */

#define GPIO_MODE_MODE15_Pos             (30)                                              /*!< GPIO_T::MODE: MODE15 Position          */
#define GPIO_MODE_MODE15_Msk             (0x3ul << GPIO_MODE_MODE15_Pos)                   /*!< GPIO_T::MODE: MODE15 Mask              */

#define GPIO_DINOFF_DINOFF0_Pos          (16)                                              /*!< GPIO_T::DINOFF: DINOFF0 Position       */
#define GPIO_DINOFF_DINOFF0_Msk          (0x1ul << GPIO_DINOFF_DINOFF0_Pos)                /*!< GPIO_T::DINOFF: DINOFF0 Mask           */

#define GPIO_DINOFF_DINOFF1_Pos          (17)                                              /*!< GPIO_T::DINOFF: DINOFF1 Position       */
#define GPIO_DINOFF_DINOFF1_Msk          (0x1ul << GPIO_DINOFF_DINOFF1_Pos)                /*!< GPIO_T::DINOFF: DINOFF1 Mask           */

#define GPIO_DINOFF_DINOFF2_Pos          (18)                                              /*!< GPIO_T::DINOFF: DINOFF2 Position       */
#define GPIO_DINOFF_DINOFF2_Msk          (0x1ul << GPIO_DINOFF_DINOFF2_Pos)                /*!< GPIO_T::DINOFF: DINOFF2 Mask           */

#define GPIO_DINOFF_DINOFF3_Pos          (19)                                              /*!< GPIO_T::DINOFF: DINOFF3 Position       */
#define GPIO_DINOFF_DINOFF3_Msk          (0x1ul << GPIO_DINOFF_DINOFF3_Pos)                /*!< GPIO_T::DINOFF: DINOFF3 Mask           */

#define GPIO_DINOFF_DINOFF4_Pos          (20)                                              /*!< GPIO_T::DINOFF: DINOFF4 Position       */
#define GPIO_DINOFF_DINOFF4_Msk          (0x1ul << GPIO_DINOFF_DINOFF4_Pos)                /*!< GPIO_T::DINOFF: DINOFF4 Mask           */

#define GPIO_DINOFF_DINOFF5_Pos          (21)                                              /*!< GPIO_T::DINOFF: DINOFF5 Position       */
#define GPIO_DINOFF_DINOFF5_Msk          (0x1ul << GPIO_DINOFF_DINOFF5_Pos)                /*!< GPIO_T::DINOFF: DINOFF5 Mask           */

#define GPIO_DINOFF_DINOFF6_Pos          (22)                                              /*!< GPIO_T::DINOFF: DINOFF6 Position       */
#define GPIO_DINOFF_DINOFF6_Msk          (0x1ul << GPIO_DINOFF_DINOFF6_Pos)                /*!< GPIO_T::DINOFF: DINOFF6 Mask           */

#define GPIO_DINOFF_DINOFF7_Pos          (23)                                              /*!< GPIO_T::DINOFF: DINOFF7 Position       */
#define GPIO_DINOFF_DINOFF7_Msk          (0x1ul << GPIO_DINOFF_DINOFF7_Pos)                /*!< GPIO_T::DINOFF: DINOFF7 Mask           */

#define GPIO_DINOFF_DINOFF8_Pos          (24)                                              /*!< GPIO_T::DINOFF: DINOFF8 Position       */
#define GPIO_DINOFF_DINOFF8_Msk          (0x1ul << GPIO_DINOFF_DINOFF8_Pos)                /*!< GPIO_T::DINOFF: DINOFF8 Mask           */

#define GPIO_DINOFF_DINOFF9_Pos          (25)                                              /*!< GPIO_T::DINOFF: DINOFF9 Position       */
#define GPIO_DINOFF_DINOFF9_Msk          (0x1ul << GPIO_DINOFF_DINOFF9_Pos)                /*!< GPIO_T::DINOFF: DINOFF9 Mask           */

#define GPIO_DINOFF_DINOFF10_Pos         (26)                                              /*!< GPIO_T::DINOFF: DINOFF10 Position      */
#define GPIO_DINOFF_DINOFF10_Msk         (0x1ul << GPIO_DINOFF_DINOFF10_Pos)               /*!< GPIO_T::DINOFF: DINOFF10 Mask          */

#define GPIO_DINOFF_DINOFF11_Pos         (27)                                              /*!< GPIO_T::DINOFF: DINOFF11 Position      */
#define GPIO_DINOFF_DINOFF11_Msk         (0x1ul << GPIO_DINOFF_DINOFF11_Pos)               /*!< GPIO_T::DINOFF: DINOFF11 Mask          */

#define GPIO_DINOFF_DINOFF12_Pos         (28)                                              /*!< GPIO_T::DINOFF: DINOFF12 Position      */
#define GPIO_DINOFF_DINOFF12_Msk         (0x1ul << GPIO_DINOFF_DINOFF12_Pos)               /*!< GPIO_T::DINOFF: DINOFF12 Mask          */

#define GPIO_DINOFF_DINOFF13_Pos         (29)                                              /*!< GPIO_T::DINOFF: DINOFF13 Position      */
#define GPIO_DINOFF_DINOFF13_Msk         (0x1ul << GPIO_DINOFF_DINOFF13_Pos)               /*!< GPIO_T::DINOFF: DINOFF13 Mask          */

#define GPIO_DINOFF_DINOFF14_Pos         (30)                                              /*!< GPIO_T::DINOFF: DINOFF14 Position      */
#define GPIO_DINOFF_DINOFF14_Msk         (0x1ul << GPIO_DINOFF_DINOFF14_Pos)               /*!< GPIO_T::DINOFF: DINOFF14 Mask          */

#define GPIO_DINOFF_DINOFF15_Pos         (31)                                              /*!< GPIO_T::DINOFF: DINOFF15 Position      */
#define GPIO_DINOFF_DINOFF15_Msk         (0x1ul << GPIO_DINOFF_DINOFF15_Pos)               /*!< GPIO_T::DINOFF: DINOFF15 Mask          */

#define GPIO_DOUT_DOUT0_Pos              (0)                                               /*!< GPIO_T::DOUT: DOUT0 Position           */
#define GPIO_DOUT_DOUT0_Msk              (0x1ul << GPIO_DOUT_DOUT0_Pos)                    /*!< GPIO_T::DOUT: DOUT0 Mask               */

#define GPIO_DOUT_DOUT1_Pos              (1)                                               /*!< GPIO_T::DOUT: DOUT1 Position           */
#define GPIO_DOUT_DOUT1_Msk              (0x1ul << GPIO_DOUT_DOUT1_Pos)                    /*!< GPIO_T::DOUT: DOUT1 Mask               */

#define GPIO_DOUT_DOUT2_Pos              (2)                                               /*!< GPIO_T::DOUT: DOUT2 Position           */
#define GPIO_DOUT_DOUT2_Msk              (0x1ul << GPIO_DOUT_DOUT2_Pos)                    /*!< GPIO_T::DOUT: DOUT2 Mask               */

#define GPIO_DOUT_DOUT3_Pos              (3)                                               /*!< GPIO_T::DOUT: DOUT3 Position           */
#define GPIO_DOUT_DOUT3_Msk              (0x1ul << GPIO_DOUT_DOUT3_Pos)                    /*!< GPIO_T::DOUT: DOUT3 Mask               */

#define GPIO_DOUT_DOUT4_Pos              (4)                                               /*!< GPIO_T::DOUT: DOUT4 Position           */
#define GPIO_DOUT_DOUT4_Msk              (0x1ul << GPIO_DOUT_DOUT4_Pos)                    /*!< GPIO_T::DOUT: DOUT4 Mask               */

#define GPIO_DOUT_DOUT5_Pos              (5)                                               /*!< GPIO_T::DOUT: DOUT5 Position           */
#define GPIO_DOUT_DOUT5_Msk              (0x1ul << GPIO_DOUT_DOUT5_Pos)                    /*!< GPIO_T::DOUT: DOUT5 Mask               */

#define GPIO_DOUT_DOUT6_Pos              (6)                                               /*!< GPIO_T::DOUT: DOUT6 Position           */
#define GPIO_DOUT_DOUT6_Msk              (0x1ul << GPIO_DOUT_DOUT6_Pos)                    /*!< GPIO_T::DOUT: DOUT6 Mask               */

#define GPIO_DOUT_DOUT7_Pos              (7)                                               /*!< GPIO_T::DOUT: DOUT7 Position           */
#define GPIO_DOUT_DOUT7_Msk              (0x1ul << GPIO_DOUT_DOUT7_Pos)                    /*!< GPIO_T::DOUT: DOUT7 Mask               */

#define GPIO_DOUT_DOUT8_Pos              (8)                                               /*!< GPIO_T::DOUT: DOUT8 Position           */
#define GPIO_DOUT_DOUT8_Msk              (0x1ul << GPIO_DOUT_DOUT8_Pos)                    /*!< GPIO_T::DOUT: DOUT8 Mask               */

#define GPIO_DOUT_DOUT9_Pos              (9)                                               /*!< GPIO_T::DOUT: DOUT9 Position           */
#define GPIO_DOUT_DOUT9_Msk              (0x1ul << GPIO_DOUT_DOUT9_Pos)                    /*!< GPIO_T::DOUT: DOUT9 Mask               */

#define GPIO_DOUT_DOUT10_Pos             (10)                                              /*!< GPIO_T::DOUT: DOUT10 Position          */
#define GPIO_DOUT_DOUT10_Msk             (0x1ul << GPIO_DOUT_DOUT10_Pos)                   /*!< GPIO_T::DOUT: DOUT10 Mask              */

#define GPIO_DOUT_DOUT11_Pos             (11)                                              /*!< GPIO_T::DOUT: DOUT11 Position          */
#define GPIO_DOUT_DOUT11_Msk             (0x1ul << GPIO_DOUT_DOUT11_Pos)                   /*!< GPIO_T::DOUT: DOUT11 Mask              */

#define GPIO_DOUT_DOUT12_Pos             (12)                                              /*!< GPIO_T::DOUT: DOUT12 Position          */
#define GPIO_DOUT_DOUT12_Msk             (0x1ul << GPIO_DOUT_DOUT12_Pos)                   /*!< GPIO_T::DOUT: DOUT12 Mask              */

#define GPIO_DOUT_DOUT13_Pos             (13)                                              /*!< GPIO_T::DOUT: DOUT13 Position          */
#define GPIO_DOUT_DOUT13_Msk             (0x1ul << GPIO_DOUT_DOUT13_Pos)                   /*!< GPIO_T::DOUT: DOUT13 Mask              */

#define GPIO_DOUT_DOUT14_Pos             (14)                                              /*!< GPIO_T::DOUT: DOUT14 Position          */
#define GPIO_DOUT_DOUT14_Msk             (0x1ul << GPIO_DOUT_DOUT14_Pos)                   /*!< GPIO_T::DOUT: DOUT14 Mask              */

#define GPIO_DOUT_DOUT15_Pos             (15)                                              /*!< GPIO_T::DOUT: DOUT15 Position          */
#define GPIO_DOUT_DOUT15_Msk             (0x1ul << GPIO_DOUT_DOUT15_Pos)                   /*!< GPIO_T::DOUT: DOUT15 Mask              */

#define GPIO_DATMSK_DMASK0_Pos           (0)                                               /*!< GPIO_T::DATMSK: DMASK0 Position        */
#define GPIO_DATMSK_DMASK0_Msk           (0x1ul << GPIO_DATMSK_DMASK0_Pos)                 /*!< GPIO_T::DATMSK: DMASK0 Mask            */

#define GPIO_DATMSK_DMASK1_Pos           (1)                                               /*!< GPIO_T::DATMSK: DMASK1 Position        */
#define GPIO_DATMSK_DMASK1_Msk           (0x1ul << GPIO_DATMSK_DMASK1_Pos)                 /*!< GPIO_T::DATMSK: DMASK1 Mask            */

#define GPIO_DATMSK_DMASK2_Pos           (2)                                               /*!< GPIO_T::DATMSK: DMASK2 Position        */
#define GPIO_DATMSK_DMASK2_Msk           (0x1ul << GPIO_DATMSK_DMASK2_Pos)                 /*!< GPIO_T::DATMSK: DMASK2 Mask            */

#define GPIO_DATMSK_DMASK3_Pos           (3)                                               /*!< GPIO_T::DATMSK: DMASK3 Position        */
#define GPIO_DATMSK_DMASK3_Msk           (0x1ul << GPIO_DATMSK_DMASK3_Pos)                 /*!< GPIO_T::DATMSK: DMASK3 Mask            */

#define GPIO_DATMSK_DMASK4_Pos           (4)                                               /*!< GPIO_T::DATMSK: DMASK4 Position        */
#define GPIO_DATMSK_DMASK4_Msk           (0x1ul << GPIO_DATMSK_DMASK4_Pos)                 /*!< GPIO_T::DATMSK: DMASK4 Mask            */

#define GPIO_DATMSK_DMASK5_Pos           (5)                                               /*!< GPIO_T::DATMSK: DMASK5 Position        */
#define GPIO_DATMSK_DMASK5_Msk           (0x1ul << GPIO_DATMSK_DMASK5_Pos)                 /*!< GPIO_T::DATMSK: DMASK5 Mask            */

#define GPIO_DATMSK_DMASK6_Pos           (6)                                               /*!< GPIO_T::DATMSK: DMASK6 Position        */
#define GPIO_DATMSK_DMASK6_Msk           (0x1ul << GPIO_DATMSK_DMASK6_Pos)                 /*!< GPIO_T::DATMSK: DMASK6 Mask            */

#define GPIO_DATMSK_DMASK7_Pos           (7)                                               /*!< GPIO_T::DATMSK: DMASK7 Position        */
#define GPIO_DATMSK_DMASK7_Msk           (0x1ul << GPIO_DATMSK_DMASK7_Pos)                 /*!< GPIO_T::DATMSK: DMASK7 Mask            */

#define GPIO_DATMSK_DMASK8_Pos           (8)                                               /*!< GPIO_T::DATMSK: DMASK8 Position        */
#define GPIO_DATMSK_DMASK8_Msk           (0x1ul << GPIO_DATMSK_DMASK8_Pos)                 /*!< GPIO_T::DATMSK: DMASK8 Mask            */

#define GPIO_DATMSK_DMASK9_Pos           (9)                                               /*!< GPIO_T::DATMSK: DMASK9 Position        */
#define GPIO_DATMSK_DMASK9_Msk           (0x1ul << GPIO_DATMSK_DMASK9_Pos)                 /*!< GPIO_T::DATMSK: DMASK9 Mask            */

#define GPIO_DATMSK_DMASK10_Pos          (10)                                              /*!< GPIO_T::DATMSK: DMASK10 Position       */
#define GPIO_DATMSK_DMASK10_Msk          (0x1ul << GPIO_DATMSK_DMASK10_Pos)                /*!< GPIO_T::DATMSK: DMASK10 Mask           */

#define GPIO_DATMSK_DMASK11_Pos          (11)                                              /*!< GPIO_T::DATMSK: DMASK11 Position       */
#define GPIO_DATMSK_DMASK11_Msk          (0x1ul << GPIO_DATMSK_DMASK11_Pos)                /*!< GPIO_T::DATMSK: DMASK11 Mask           */

#define GPIO_DATMSK_DMASK12_Pos          (12)                                              /*!< GPIO_T::DATMSK: DMASK12 Position       */
#define GPIO_DATMSK_DMASK12_Msk          (0x1ul << GPIO_DATMSK_DMASK12_Pos)                /*!< GPIO_T::DATMSK: DMASK12 Mask           */

#define GPIO_DATMSK_DMASK13_Pos          (13)                                              /*!< GPIO_T::DATMSK: DMASK13 Position       */
#define GPIO_DATMSK_DMASK13_Msk          (0x1ul << GPIO_DATMSK_DMASK13_Pos)                /*!< GPIO_T::DATMSK: DMASK13 Mask           */

#define GPIO_DATMSK_DMASK14_Pos          (14)                                              /*!< GPIO_T::DATMSK: DMASK14 Position       */
#define GPIO_DATMSK_DMASK14_Msk          (0x1ul << GPIO_DATMSK_DMASK14_Pos)                /*!< GPIO_T::DATMSK: DMASK14 Mask           */

#define GPIO_DATMSK_DMASK15_Pos          (15)                                              /*!< GPIO_T::DATMSK: DMASK15 Position       */
#define GPIO_DATMSK_DMASK15_Msk          (0x1ul << GPIO_DATMSK_DMASK15_Pos)                /*!< GPIO_T::DATMSK: DMASK15 Mask           */

#define GPIO_PIN_PIN0_Pos                (0)                                               /*!< GPIO_T::PIN: PIN0 Position             */
#define GPIO_PIN_PIN0_Msk                (0x1ul << GPIO_PIN_PIN0_Pos)                      /*!< GPIO_T::PIN: PIN0 Mask                 */

#define GPIO_PIN_PIN1_Pos                (1)                                               /*!< GPIO_T::PIN: PIN1 Position             */
#define GPIO_PIN_PIN1_Msk                (0x1ul << GPIO_PIN_PIN1_Pos)                      /*!< GPIO_T::PIN: PIN1 Mask                 */

#define GPIO_PIN_PIN2_Pos                (2)                                               /*!< GPIO_T::PIN: PIN2 Position             */
#define GPIO_PIN_PIN2_Msk                (0x1ul << GPIO_PIN_PIN2_Pos)                      /*!< GPIO_T::PIN: PIN2 Mask                 */

#define GPIO_PIN_PIN3_Pos                (3)                                               /*!< GPIO_T::PIN: PIN3 Position             */
#define GPIO_PIN_PIN3_Msk                (0x1ul << GPIO_PIN_PIN3_Pos)                      /*!< GPIO_T::PIN: PIN3 Mask                 */

#define GPIO_PIN_PIN4_Pos                (4)                                               /*!< GPIO_T::PIN: PIN4 Position             */
#define GPIO_PIN_PIN4_Msk                (0x1ul << GPIO_PIN_PIN4_Pos)                      /*!< GPIO_T::PIN: PIN4 Mask                 */

#define GPIO_PIN_PIN5_Pos                (5)                                               /*!< GPIO_T::PIN: PIN5 Position             */
#define GPIO_PIN_PIN5_Msk                (0x1ul << GPIO_PIN_PIN5_Pos)                      /*!< GPIO_T::PIN: PIN5 Mask                 */

#define GPIO_PIN_PIN6_Pos                (6)                                               /*!< GPIO_T::PIN: PIN6 Position             */
#define GPIO_PIN_PIN6_Msk                (0x1ul << GPIO_PIN_PIN6_Pos)                      /*!< GPIO_T::PIN: PIN6 Mask                 */

#define GPIO_PIN_PIN7_Pos                (7)                                               /*!< GPIO_T::PIN: PIN7 Position             */
#define GPIO_PIN_PIN7_Msk                (0x1ul << GPIO_PIN_PIN7_Pos)                      /*!< GPIO_T::PIN: PIN7 Mask                 */

#define GPIO_PIN_PIN8_Pos                (8)                                               /*!< GPIO_T::PIN: PIN8 Position             */
#define GPIO_PIN_PIN8_Msk                (0x1ul << GPIO_PIN_PIN8_Pos)                      /*!< GPIO_T::PIN: PIN8 Mask                 */

#define GPIO_PIN_PIN9_Pos                (9)                                               /*!< GPIO_T::PIN: PIN9 Position             */
#define GPIO_PIN_PIN9_Msk                (0x1ul << GPIO_PIN_PIN9_Pos)                      /*!< GPIO_T::PIN: PIN9 Mask                 */

#define GPIO_PIN_PIN10_Pos               (10)                                              /*!< GPIO_T::PIN: PIN10 Position            */
#define GPIO_PIN_PIN10_Msk               (0x1ul << GPIO_PIN_PIN10_Pos)                     /*!< GPIO_T::PIN: PIN10 Mask                */

#define GPIO_PIN_PIN11_Pos               (11)                                              /*!< GPIO_T::PIN: PIN11 Position            */
#define GPIO_PIN_PIN11_Msk               (0x1ul << GPIO_PIN_PIN11_Pos)                     /*!< GPIO_T::PIN: PIN11 Mask                */

#define GPIO_PIN_PIN12_Pos               (12)                                              /*!< GPIO_T::PIN: PIN12 Position            */
#define GPIO_PIN_PIN12_Msk               (0x1ul << GPIO_PIN_PIN12_Pos)                     /*!< GPIO_T::PIN: PIN12 Mask                */

#define GPIO_PIN_PIN13_Pos               (13)                                              /*!< GPIO_T::PIN: PIN13 Position            */
#define GPIO_PIN_PIN13_Msk               (0x1ul << GPIO_PIN_PIN13_Pos)                     /*!< GPIO_T::PIN: PIN13 Mask                */

#define GPIO_PIN_PIN14_Pos               (14)                                              /*!< GPIO_T::PIN: PIN14 Position            */
#define GPIO_PIN_PIN14_Msk               (0x1ul << GPIO_PIN_PIN14_Pos)                     /*!< GPIO_T::PIN: PIN14 Mask                */

#define GPIO_PIN_PIN15_Pos               (15)                                              /*!< GPIO_T::PIN: PIN15 Position            */
#define GPIO_PIN_PIN15_Msk               (0x1ul << GPIO_PIN_PIN15_Pos)                     /*!< GPIO_T::PIN: PIN15 Mask                */

#define GPIO_DBEN_DBEN0_Pos              (0)                                               /*!< GPIO_T::DBEN: DBEN0 Position           */
#define GPIO_DBEN_DBEN0_Msk              (0x1ul << GPIO_DBEN_DBEN0_Pos)                    /*!< GPIO_T::DBEN: DBEN0 Mask               */

#define GPIO_DBEN_DBEN1_Pos              (1)                                               /*!< GPIO_T::DBEN: DBEN1 Position           */
#define GPIO_DBEN_DBEN1_Msk              (0x1ul << GPIO_DBEN_DBEN1_Pos)                    /*!< GPIO_T::DBEN: DBEN1 Mask               */

#define GPIO_DBEN_DBEN2_Pos              (2)                                               /*!< GPIO_T::DBEN: DBEN2 Position           */
#define GPIO_DBEN_DBEN2_Msk              (0x1ul << GPIO_DBEN_DBEN2_Pos)                    /*!< GPIO_T::DBEN: DBEN2 Mask               */

#define GPIO_DBEN_DBEN3_Pos              (3)                                               /*!< GPIO_T::DBEN: DBEN3 Position           */
#define GPIO_DBEN_DBEN3_Msk              (0x1ul << GPIO_DBEN_DBEN3_Pos)                    /*!< GPIO_T::DBEN: DBEN3 Mask               */

#define GPIO_DBEN_DBEN4_Pos              (4)                                               /*!< GPIO_T::DBEN: DBEN4 Position           */
#define GPIO_DBEN_DBEN4_Msk              (0x1ul << GPIO_DBEN_DBEN4_Pos)                    /*!< GPIO_T::DBEN: DBEN4 Mask               */

#define GPIO_DBEN_DBEN5_Pos              (5)                                               /*!< GPIO_T::DBEN: DBEN5 Position           */
#define GPIO_DBEN_DBEN5_Msk              (0x1ul << GPIO_DBEN_DBEN5_Pos)                    /*!< GPIO_T::DBEN: DBEN5 Mask               */

#define GPIO_DBEN_DBEN6_Pos              (6)                                               /*!< GPIO_T::DBEN: DBEN6 Position           */
#define GPIO_DBEN_DBEN6_Msk              (0x1ul << GPIO_DBEN_DBEN6_Pos)                    /*!< GPIO_T::DBEN: DBEN6 Mask               */

#define GPIO_DBEN_DBEN7_Pos              (7)                                               /*!< GPIO_T::DBEN: DBEN7 Position           */
#define GPIO_DBEN_DBEN7_Msk              (0x1ul << GPIO_DBEN_DBEN7_Pos)                    /*!< GPIO_T::DBEN: DBEN7 Mask               */

#define GPIO_DBEN_DBEN8_Pos              (8)                                               /*!< GPIO_T::DBEN: DBEN8 Position           */
#define GPIO_DBEN_DBEN8_Msk              (0x1ul << GPIO_DBEN_DBEN8_Pos)                    /*!< GPIO_T::DBEN: DBEN8 Mask               */

#define GPIO_DBEN_DBEN9_Pos              (9)                                               /*!< GPIO_T::DBEN: DBEN9 Position           */
#define GPIO_DBEN_DBEN9_Msk              (0x1ul << GPIO_DBEN_DBEN9_Pos)                    /*!< GPIO_T::DBEN: DBEN9 Mask               */

#define GPIO_DBEN_DBEN10_Pos             (10)                                              /*!< GPIO_T::DBEN: DBEN10 Position          */
#define GPIO_DBEN_DBEN10_Msk             (0x1ul << GPIO_DBEN_DBEN10_Pos)                   /*!< GPIO_T::DBEN: DBEN10 Mask              */

#define GPIO_DBEN_DBEN11_Pos             (11)                                              /*!< GPIO_T::DBEN: DBEN11 Position          */
#define GPIO_DBEN_DBEN11_Msk             (0x1ul << GPIO_DBEN_DBEN11_Pos)                   /*!< GPIO_T::DBEN: DBEN11 Mask              */

#define GPIO_DBEN_DBEN12_Pos             (12)                                              /*!< GPIO_T::DBEN: DBEN12 Position          */
#define GPIO_DBEN_DBEN12_Msk             (0x1ul << GPIO_DBEN_DBEN12_Pos)                   /*!< GPIO_T::DBEN: DBEN12 Mask              */

#define GPIO_DBEN_DBEN13_Pos             (13)                                              /*!< GPIO_T::DBEN: DBEN13 Position          */
#define GPIO_DBEN_DBEN13_Msk             (0x1ul << GPIO_DBEN_DBEN13_Pos)                   /*!< GPIO_T::DBEN: DBEN13 Mask              */

#define GPIO_DBEN_DBEN14_Pos             (14)                                              /*!< GPIO_T::DBEN: DBEN14 Position          */
#define GPIO_DBEN_DBEN14_Msk             (0x1ul << GPIO_DBEN_DBEN14_Pos)                   /*!< GPIO_T::DBEN: DBEN14 Mask              */

#define GPIO_DBEN_DBEN15_Pos             (15)                                              /*!< GPIO_T::DBEN: DBEN15 Position          */
#define GPIO_DBEN_DBEN15_Msk             (0x1ul << GPIO_DBEN_DBEN15_Pos)                   /*!< GPIO_T::DBEN: DBEN15 Mask              */

#define GPIO_INTTYPE_TYPE0_Pos           (0)                                               /*!< GPIO_T::INTTYPE: TYPE0 Position        */
#define GPIO_INTTYPE_TYPE0_Msk           (0x1ul << GPIO_INTTYPE_TYPE0_Pos)                 /*!< GPIO_T::INTTYPE: TYPE0 Mask            */

#define GPIO_INTTYPE_TYPE1_Pos           (1)                                               /*!< GPIO_T::INTTYPE: TYPE1 Position        */
#define GPIO_INTTYPE_TYPE1_Msk           (0x1ul << GPIO_INTTYPE_TYPE1_Pos)                 /*!< GPIO_T::INTTYPE: TYPE1 Mask            */

#define GPIO_INTTYPE_TYPE2_Pos           (2)                                               /*!< GPIO_T::INTTYPE: TYPE2 Position        */
#define GPIO_INTTYPE_TYPE2_Msk           (0x1ul << GPIO_INTTYPE_TYPE2_Pos)                 /*!< GPIO_T::INTTYPE: TYPE2 Mask            */

#define GPIO_INTTYPE_TYPE3_Pos           (3)                                               /*!< GPIO_T::INTTYPE: TYPE3 Position        */
#define GPIO_INTTYPE_TYPE3_Msk           (0x1ul << GPIO_INTTYPE_TYPE3_Pos)                 /*!< GPIO_T::INTTYPE: TYPE3 Mask            */

#define GPIO_INTTYPE_TYPE4_Pos           (4)                                               /*!< GPIO_T::INTTYPE: TYPE4 Position        */
#define GPIO_INTTYPE_TYPE4_Msk           (0x1ul << GPIO_INTTYPE_TYPE4_Pos)                 /*!< GPIO_T::INTTYPE: TYPE4 Mask            */

#define GPIO_INTTYPE_TYPE5_Pos           (5)                                               /*!< GPIO_T::INTTYPE: TYPE5 Position        */
#define GPIO_INTTYPE_TYPE5_Msk           (0x1ul << GPIO_INTTYPE_TYPE5_Pos)                 /*!< GPIO_T::INTTYPE: TYPE5 Mask            */

#define GPIO_INTTYPE_TYPE6_Pos           (6)                                               /*!< GPIO_T::INTTYPE: TYPE6 Position        */
#define GPIO_INTTYPE_TYPE6_Msk           (0x1ul << GPIO_INTTYPE_TYPE6_Pos)                 /*!< GPIO_T::INTTYPE: TYPE6 Mask            */

#define GPIO_INTTYPE_TYPE7_Pos           (7)                                               /*!< GPIO_T::INTTYPE: TYPE7 Position        */
#define GPIO_INTTYPE_TYPE7_Msk           (0x1ul << GPIO_INTTYPE_TYPE7_Pos)                 /*!< GPIO_T::INTTYPE: TYPE7 Mask            */

#define GPIO_INTTYPE_TYPE8_Pos           (8)                                               /*!< GPIO_T::INTTYPE: TYPE8 Position        */
#define GPIO_INTTYPE_TYPE8_Msk           (0x1ul << GPIO_INTTYPE_TYPE8_Pos)                 /*!< GPIO_T::INTTYPE: TYPE8 Mask            */

#define GPIO_INTTYPE_TYPE9_Pos           (9)                                               /*!< GPIO_T::INTTYPE: TYPE9 Position        */
#define GPIO_INTTYPE_TYPE9_Msk           (0x1ul << GPIO_INTTYPE_TYPE9_Pos)                 /*!< GPIO_T::INTTYPE: TYPE9 Mask            */

#define GPIO_INTTYPE_TYPE10_Pos          (10)                                              /*!< GPIO_T::INTTYPE: TYPE10 Position       */
#define GPIO_INTTYPE_TYPE10_Msk          (0x1ul << GPIO_INTTYPE_TYPE10_Pos)                /*!< GPIO_T::INTTYPE: TYPE10 Mask           */

#define GPIO_INTTYPE_TYPE11_Pos          (11)                                              /*!< GPIO_T::INTTYPE: TYPE11 Position       */
#define GPIO_INTTYPE_TYPE11_Msk          (0x1ul << GPIO_INTTYPE_TYPE11_Pos)                /*!< GPIO_T::INTTYPE: TYPE11 Mask           */

#define GPIO_INTTYPE_TYPE12_Pos          (12)                                              /*!< GPIO_T::INTTYPE: TYPE12 Position       */
#define GPIO_INTTYPE_TYPE12_Msk          (0x1ul << GPIO_INTTYPE_TYPE12_Pos)                /*!< GPIO_T::INTTYPE: TYPE12 Mask           */

#define GPIO_INTTYPE_TYPE13_Pos          (13)                                              /*!< GPIO_T::INTTYPE: TYPE13 Position       */
#define GPIO_INTTYPE_TYPE13_Msk          (0x1ul << GPIO_INTTYPE_TYPE13_Pos)                /*!< GPIO_T::INTTYPE: TYPE13 Mask           */

#define GPIO_INTTYPE_TYPE14_Pos          (14)                                              /*!< GPIO_T::INTTYPE: TYPE14 Position       */
#define GPIO_INTTYPE_TYPE14_Msk          (0x1ul << GPIO_INTTYPE_TYPE14_Pos)                /*!< GPIO_T::INTTYPE: TYPE14 Mask           */

#define GPIO_INTTYPE_TYPE15_Pos          (15)                                              /*!< GPIO_T::INTTYPE: TYPE15 Position       */
#define GPIO_INTTYPE_TYPE15_Msk          (0x1ul << GPIO_INTTYPE_TYPE15_Pos)                /*!< GPIO_T::INTTYPE: TYPE15 Mask           */

#define GPIO_INTEN_FLIEN0_Pos            (0)                                               /*!< GPIO_T::INTEN: FLIEN0 Position         */
#define GPIO_INTEN_FLIEN0_Msk            (0x1ul << GPIO_INTEN_FLIEN0_Pos)                  /*!< GPIO_T::INTEN: FLIEN0 Mask             */

#define GPIO_INTEN_FLIEN1_Pos            (1)                                               /*!< GPIO_T::INTEN: FLIEN1 Position         */
#define GPIO_INTEN_FLIEN1_Msk            (0x1ul << GPIO_INTEN_FLIEN1_Pos)                  /*!< GPIO_T::INTEN: FLIEN1 Mask             */

#define GPIO_INTEN_FLIEN2_Pos            (2)                                               /*!< GPIO_T::INTEN: FLIEN2 Position         */
#define GPIO_INTEN_FLIEN2_Msk            (0x1ul << GPIO_INTEN_FLIEN2_Pos)                  /*!< GPIO_T::INTEN: FLIEN2 Mask             */

#define GPIO_INTEN_FLIEN3_Pos            (3)                                               /*!< GPIO_T::INTEN: FLIEN3 Position         */
#define GPIO_INTEN_FLIEN3_Msk            (0x1ul << GPIO_INTEN_FLIEN3_Pos)                  /*!< GPIO_T::INTEN: FLIEN3 Mask             */

#define GPIO_INTEN_FLIEN4_Pos            (4)                                               /*!< GPIO_T::INTEN: FLIEN4 Position         */
#define GPIO_INTEN_FLIEN4_Msk            (0x1ul << GPIO_INTEN_FLIEN4_Pos)                  /*!< GPIO_T::INTEN: FLIEN4 Mask             */

#define GPIO_INTEN_FLIEN5_Pos            (5)                                               /*!< GPIO_T::INTEN: FLIEN5 Position         */
#define GPIO_INTEN_FLIEN5_Msk            (0x1ul << GPIO_INTEN_FLIEN5_Pos)                  /*!< GPIO_T::INTEN: FLIEN5 Mask             */

#define GPIO_INTEN_FLIEN6_Pos            (6)                                               /*!< GPIO_T::INTEN: FLIEN6 Position         */
#define GPIO_INTEN_FLIEN6_Msk            (0x1ul << GPIO_INTEN_FLIEN6_Pos)                  /*!< GPIO_T::INTEN: FLIEN6 Mask             */

#define GPIO_INTEN_FLIEN7_Pos            (7)                                               /*!< GPIO_T::INTEN: FLIEN7 Position         */
#define GPIO_INTEN_FLIEN7_Msk            (0x1ul << GPIO_INTEN_FLIEN7_Pos)                  /*!< GPIO_T::INTEN: FLIEN7 Mask             */

#define GPIO_INTEN_FLIEN8_Pos            (8)                                               /*!< GPIO_T::INTEN: FLIEN8 Position         */
#define GPIO_INTEN_FLIEN8_Msk            (0x1ul << GPIO_INTEN_FLIEN8_Pos)                  /*!< GPIO_T::INTEN: FLIEN8 Mask             */

#define GPIO_INTEN_FLIEN9_Pos            (9)                                               /*!< GPIO_T::INTEN: FLIEN9 Position         */
#define GPIO_INTEN_FLIEN9_Msk            (0x1ul << GPIO_INTEN_FLIEN9_Pos)                  /*!< GPIO_T::INTEN: FLIEN9 Mask             */

#define GPIO_INTEN_FLIEN10_Pos           (10)                                              /*!< GPIO_T::INTEN: FLIEN10 Position        */
#define GPIO_INTEN_FLIEN10_Msk           (0x1ul << GPIO_INTEN_FLIEN10_Pos)                 /*!< GPIO_T::INTEN: FLIEN10 Mask            */

#define GPIO_INTEN_FLIEN11_Pos           (11)                                              /*!< GPIO_T::INTEN: FLIEN11 Position        */
#define GPIO_INTEN_FLIEN11_Msk           (0x1ul << GPIO_INTEN_FLIEN11_Pos)                 /*!< GPIO_T::INTEN: FLIEN11 Mask            */

#define GPIO_INTEN_FLIEN12_Pos           (12)                                              /*!< GPIO_T::INTEN: FLIEN12 Position        */
#define GPIO_INTEN_FLIEN12_Msk           (0x1ul << GPIO_INTEN_FLIEN12_Pos)                 /*!< GPIO_T::INTEN: FLIEN12 Mask            */

#define GPIO_INTEN_FLIEN13_Pos           (13)                                              /*!< GPIO_T::INTEN: FLIEN13 Position        */
#define GPIO_INTEN_FLIEN13_Msk           (0x1ul << GPIO_INTEN_FLIEN13_Pos)                 /*!< GPIO_T::INTEN: FLIEN13 Mask            */

#define GPIO_INTEN_FLIEN14_Pos           (14)                                              /*!< GPIO_T::INTEN: FLIEN14 Position        */
#define GPIO_INTEN_FLIEN14_Msk           (0x1ul << GPIO_INTEN_FLIEN14_Pos)                 /*!< GPIO_T::INTEN: FLIEN14 Mask            */

#define GPIO_INTEN_FLIEN15_Pos           (15)                                              /*!< GPIO_T::INTEN: FLIEN15 Position        */
#define GPIO_INTEN_FLIEN15_Msk           (0x1ul << GPIO_INTEN_FLIEN15_Pos)                 /*!< GPIO_T::INTEN: FLIEN15 Mask            */

#define GPIO_INTEN_RHIEN0_Pos            (16)                                              /*!< GPIO_T::INTEN: RHIEN0 Position         */
#define GPIO_INTEN_RHIEN0_Msk            (0x1ul << GPIO_INTEN_RHIEN0_Pos)                  /*!< GPIO_T::INTEN: RHIEN0 Mask             */

#define GPIO_INTEN_RHIEN1_Pos            (17)                                              /*!< GPIO_T::INTEN: RHIEN1 Position         */
#define GPIO_INTEN_RHIEN1_Msk            (0x1ul << GPIO_INTEN_RHIEN1_Pos)                  /*!< GPIO_T::INTEN: RHIEN1 Mask             */

#define GPIO_INTEN_RHIEN2_Pos            (18)                                              /*!< GPIO_T::INTEN: RHIEN2 Position         */
#define GPIO_INTEN_RHIEN2_Msk            (0x1ul << GPIO_INTEN_RHIEN2_Pos)                  /*!< GPIO_T::INTEN: RHIEN2 Mask             */

#define GPIO_INTEN_RHIEN3_Pos            (19)                                              /*!< GPIO_T::INTEN: RHIEN3 Position         */
#define GPIO_INTEN_RHIEN3_Msk            (0x1ul << GPIO_INTEN_RHIEN3_Pos)                  /*!< GPIO_T::INTEN: RHIEN3 Mask             */

#define GPIO_INTEN_RHIEN4_Pos            (20)                                              /*!< GPIO_T::INTEN: RHIEN4 Position         */
#define GPIO_INTEN_RHIEN4_Msk            (0x1ul << GPIO_INTEN_RHIEN4_Pos)                  /*!< GPIO_T::INTEN: RHIEN4 Mask             */

#define GPIO_INTEN_RHIEN5_Pos            (21)                                              /*!< GPIO_T::INTEN: RHIEN5 Position         */
#define GPIO_INTEN_RHIEN5_Msk            (0x1ul << GPIO_INTEN_RHIEN5_Pos)                  /*!< GPIO_T::INTEN: RHIEN5 Mask             */

#define GPIO_INTEN_RHIEN6_Pos            (22)                                              /*!< GPIO_T::INTEN: RHIEN6 Position         */
#define GPIO_INTEN_RHIEN6_Msk            (0x1ul << GPIO_INTEN_RHIEN6_Pos)                  /*!< GPIO_T::INTEN: RHIEN6 Mask             */

#define GPIO_INTEN_RHIEN7_Pos            (23)                                              /*!< GPIO_T::INTEN: RHIEN7 Position         */
#define GPIO_INTEN_RHIEN7_Msk            (0x1ul << GPIO_INTEN_RHIEN7_Pos)                  /*!< GPIO_T::INTEN: RHIEN7 Mask             */

#define GPIO_INTEN_RHIEN8_Pos            (24)                                              /*!< GPIO_T::INTEN: RHIEN8 Position         */
#define GPIO_INTEN_RHIEN8_Msk            (0x1ul << GPIO_INTEN_RHIEN8_Pos)                  /*!< GPIO_T::INTEN: RHIEN8 Mask             */

#define GPIO_INTEN_RHIEN9_Pos            (25)                                              /*!< GPIO_T::INTEN: RHIEN9 Position         */
#define GPIO_INTEN_RHIEN9_Msk            (0x1ul << GPIO_INTEN_RHIEN9_Pos)                  /*!< GPIO_T::INTEN: RHIEN9 Mask             */

#define GPIO_INTEN_RHIEN10_Pos           (26)                                              /*!< GPIO_T::INTEN: RHIEN10 Position        */
#define GPIO_INTEN_RHIEN10_Msk           (0x1ul << GPIO_INTEN_RHIEN10_Pos)                 /*!< GPIO_T::INTEN: RHIEN10 Mask            */

#define GPIO_INTEN_RHIEN11_Pos           (27)                                              /*!< GPIO_T::INTEN: RHIEN11 Position        */
#define GPIO_INTEN_RHIEN11_Msk           (0x1ul << GPIO_INTEN_RHIEN11_Pos)                 /*!< GPIO_T::INTEN: RHIEN11 Mask            */

#define GPIO_INTEN_RHIEN12_Pos           (28)                                              /*!< GPIO_T::INTEN: RHIEN12 Position        */
#define GPIO_INTEN_RHIEN12_Msk           (0x1ul << GPIO_INTEN_RHIEN12_Pos)                 /*!< GPIO_T::INTEN: RHIEN12 Mask            */

#define GPIO_INTEN_RHIEN13_Pos           (29)                                              /*!< GPIO_T::INTEN: RHIEN13 Position        */
#define GPIO_INTEN_RHIEN13_Msk           (0x1ul << GPIO_INTEN_RHIEN13_Pos)                 /*!< GPIO_T::INTEN: RHIEN13 Mask            */

#define GPIO_INTEN_RHIEN14_Pos           (30)                                              /*!< GPIO_T::INTEN: RHIEN14 Position        */
#define GPIO_INTEN_RHIEN14_Msk           (0x1ul << GPIO_INTEN_RHIEN14_Pos)                 /*!< GPIO_T::INTEN: RHIEN14 Mask            */

#define GPIO_INTEN_RHIEN15_Pos           (31)                                              /*!< GPIO_T::INTEN: RHIEN15 Position        */
#define GPIO_INTEN_RHIEN15_Msk           (0x1ul << GPIO_INTEN_RHIEN15_Pos)                 /*!< GPIO_T::INTEN: RHIEN15 Mask            */

#define GPIO_INTSRC_INTSRC0_Pos          (0)                                               /*!< GPIO_T::INTSRC: INTSRC0 Position       */
#define GPIO_INTSRC_INTSRC0_Msk          (0x1ul << GPIO_INTSRC_INTSRC0_Pos)                /*!< GPIO_T::INTSRC: INTSRC0 Mask           */

#define GPIO_INTSRC_INTSRC1_Pos          (1)                                               /*!< GPIO_T::INTSRC: INTSRC1 Position       */
#define GPIO_INTSRC_INTSRC1_Msk          (0x1ul << GPIO_INTSRC_INTSRC1_Pos)                /*!< GPIO_T::INTSRC: INTSRC1 Mask           */

#define GPIO_INTSRC_INTSRC2_Pos          (2)                                               /*!< GPIO_T::INTSRC: INTSRC2 Position       */
#define GPIO_INTSRC_INTSRC2_Msk          (0x1ul << GPIO_INTSRC_INTSRC2_Pos)                /*!< GPIO_T::INTSRC: INTSRC2 Mask           */

#define GPIO_INTSRC_INTSRC3_Pos          (3)                                               /*!< GPIO_T::INTSRC: INTSRC3 Position       */
#define GPIO_INTSRC_INTSRC3_Msk          (0x1ul << GPIO_INTSRC_INTSRC3_Pos)                /*!< GPIO_T::INTSRC: INTSRC3 Mask           */

#define GPIO_INTSRC_INTSRC4_Pos          (4)                                               /*!< GPIO_T::INTSRC: INTSRC4 Position       */
#define GPIO_INTSRC_INTSRC4_Msk          (0x1ul << GPIO_INTSRC_INTSRC4_Pos)                /*!< GPIO_T::INTSRC: INTSRC4 Mask           */

#define GPIO_INTSRC_INTSRC5_Pos          (5)                                               /*!< GPIO_T::INTSRC: INTSRC5 Position       */
#define GPIO_INTSRC_INTSRC5_Msk          (0x1ul << GPIO_INTSRC_INTSRC5_Pos)                /*!< GPIO_T::INTSRC: INTSRC5 Mask           */

#define GPIO_INTSRC_INTSRC6_Pos          (6)                                               /*!< GPIO_T::INTSRC: INTSRC6 Position       */
#define GPIO_INTSRC_INTSRC6_Msk          (0x1ul << GPIO_INTSRC_INTSRC6_Pos)                /*!< GPIO_T::INTSRC: INTSRC6 Mask           */

#define GPIO_INTSRC_INTSRC7_Pos          (7)                                               /*!< GPIO_T::INTSRC: INTSRC7 Position       */
#define GPIO_INTSRC_INTSRC7_Msk          (0x1ul << GPIO_INTSRC_INTSRC7_Pos)                /*!< GPIO_T::INTSRC: INTSRC7 Mask           */

#define GPIO_INTSRC_INTSRC8_Pos          (8)                                               /*!< GPIO_T::INTSRC: INTSRC8 Position       */
#define GPIO_INTSRC_INTSRC8_Msk          (0x1ul << GPIO_INTSRC_INTSRC8_Pos)                /*!< GPIO_T::INTSRC: INTSRC8 Mask           */

#define GPIO_INTSRC_INTSRC9_Pos          (9)                                               /*!< GPIO_T::INTSRC: INTSRC9 Position       */
#define GPIO_INTSRC_INTSRC9_Msk          (0x1ul << GPIO_INTSRC_INTSRC9_Pos)                /*!< GPIO_T::INTSRC: INTSRC9 Mask           */

#define GPIO_INTSRC_INTSRC10_Pos         (10)                                              /*!< GPIO_T::INTSRC: INTSRC10 Position      */
#define GPIO_INTSRC_INTSRC10_Msk         (0x1ul << GPIO_INTSRC_INTSRC10_Pos)               /*!< GPIO_T::INTSRC: INTSRC10 Mask          */

#define GPIO_INTSRC_INTSRC11_Pos         (11)                                              /*!< GPIO_T::INTSRC: INTSRC11 Position      */
#define GPIO_INTSRC_INTSRC11_Msk         (0x1ul << GPIO_INTSRC_INTSRC11_Pos)               /*!< GPIO_T::INTSRC: INTSRC11 Mask          */

#define GPIO_INTSRC_INTSRC12_Pos         (12)                                              /*!< GPIO_T::INTSRC: INTSRC12 Position      */
#define GPIO_INTSRC_INTSRC12_Msk         (0x1ul << GPIO_INTSRC_INTSRC12_Pos)               /*!< GPIO_T::INTSRC: INTSRC12 Mask          */

#define GPIO_INTSRC_INTSRC13_Pos         (13)                                              /*!< GPIO_T::INTSRC: INTSRC13 Position      */
#define GPIO_INTSRC_INTSRC13_Msk         (0x1ul << GPIO_INTSRC_INTSRC13_Pos)               /*!< GPIO_T::INTSRC: INTSRC13 Mask          */

#define GPIO_INTSRC_INTSRC14_Pos         (14)                                              /*!< GPIO_T::INTSRC: INTSRC14 Position      */
#define GPIO_INTSRC_INTSRC14_Msk         (0x1ul << GPIO_INTSRC_INTSRC14_Pos)               /*!< GPIO_T::INTSRC: INTSRC14 Mask          */

#define GPIO_INTSRC_INTSRC15_Pos         (15)                                              /*!< GPIO_T::INTSRC: INTSRC15 Position      */
#define GPIO_INTSRC_INTSRC15_Msk         (0x1ul << GPIO_INTSRC_INTSRC15_Pos)               /*!< GPIO_T::INTSRC: INTSRC15 Mask          */

#define GPIO_PUEN_PUEN0_Pos              (0)                                               /*!< GPIO_T::PUEN: PUEN0 Position           */
#define GPIO_PUEN_PUEN0_Msk              (0x1ul << GPIO_PUEN_PUEN0_Pos)                    /*!< GPIO_T::PUEN: PUEN0 Mask               */

#define GPIO_PUEN_PUEN1_Pos              (1)                                               /*!< GPIO_T::PUEN: PUEN1 Position           */
#define GPIO_PUEN_PUEN1_Msk              (0x1ul << GPIO_PUEN_PUEN1_Pos)                    /*!< GPIO_T::PUEN: PUEN1 Mask               */

#define GPIO_PUEN_PUEN2_Pos              (2)                                               /*!< GPIO_T::PUEN: PUEN2 Position           */
#define GPIO_PUEN_PUEN2_Msk              (0x1ul << GPIO_PUEN_PUEN2_Pos)                    /*!< GPIO_T::PUEN: PUEN2 Mask               */

#define GPIO_PUEN_PUEN3_Pos              (3)                                               /*!< GPIO_T::PUEN: PUEN3 Position           */
#define GPIO_PUEN_PUEN3_Msk              (0x1ul << GPIO_PUEN_PUEN3_Pos)                    /*!< GPIO_T::PUEN: PUEN3 Mask               */

#define GPIO_PUEN_PUEN4_Pos              (4)                                               /*!< GPIO_T::PUEN: PUEN4 Position           */
#define GPIO_PUEN_PUEN4_Msk              (0x1ul << GPIO_PUEN_PUEN4_Pos)                    /*!< GPIO_T::PUEN: PUEN4 Mask               */

#define GPIO_PUEN_PUEN5_Pos              (5)                                               /*!< GPIO_T::PUEN: PUEN5 Position           */
#define GPIO_PUEN_PUEN5_Msk              (0x1ul << GPIO_PUEN_PUEN5_Pos)                    /*!< GPIO_T::PUEN: PUEN5 Mask               */

#define GPIO_PUEN_PUEN6_Pos              (6)                                               /*!< GPIO_T::PUEN: PUEN6 Position           */
#define GPIO_PUEN_PUEN6_Msk              (0x1ul << GPIO_PUEN_PUEN6_Pos)                    /*!< GPIO_T::PUEN: PUEN6 Mask               */

#define GPIO_PUEN_PUEN7_Pos              (7)                                               /*!< GPIO_T::PUEN: PUEN7 Position           */
#define GPIO_PUEN_PUEN7_Msk              (0x1ul << GPIO_PUEN_PUEN7_Pos)                    /*!< GPIO_T::PUEN: PUEN7 Mask               */

#define GPIO_PUEN_PUEN8_Pos              (8)                                               /*!< GPIO_T::PUEN: PUEN8 Position           */
#define GPIO_PUEN_PUEN8_Msk              (0x1ul << GPIO_PUEN_PUEN8_Pos)                    /*!< GPIO_T::PUEN: PUEN8 Mask               */

#define GPIO_PUEN_PUEN9_Pos              (9)                                               /*!< GPIO_T::PUEN: PUEN9 Position           */
#define GPIO_PUEN_PUEN9_Msk              (0x1ul << GPIO_PUEN_PUEN9_Pos)                    /*!< GPIO_T::PUEN: PUEN9 Mask               */

#define GPIO_PUEN_PUEN10_Pos             (10)                                              /*!< GPIO_T::PUEN: PUEN10 Position          */
#define GPIO_PUEN_PUEN10_Msk             (0x1ul << GPIO_PUEN_PUEN10_Pos)                   /*!< GPIO_T::PUEN: PUEN10 Mask              */

#define GPIO_PUEN_PUEN11_Pos             (11)                                              /*!< GPIO_T::PUEN: PUEN11 Position          */
#define GPIO_PUEN_PUEN11_Msk             (0x1ul << GPIO_PUEN_PUEN11_Pos)                   /*!< GPIO_T::PUEN: PUEN11 Mask              */

#define GPIO_PUEN_PUEN12_Pos             (12)                                              /*!< GPIO_T::PUEN: PUEN12 Position          */
#define GPIO_PUEN_PUEN12_Msk             (0x1ul << GPIO_PUEN_PUEN12_Pos)                   /*!< GPIO_T::PUEN: PUEN12 Mask              */

#define GPIO_PUEN_PUEN13_Pos             (13)                                              /*!< GPIO_T::PUEN: PUEN13 Position          */
#define GPIO_PUEN_PUEN13_Msk             (0x1ul << GPIO_PUEN_PUEN13_Pos)                   /*!< GPIO_T::PUEN: PUEN13 Mask              */

#define GPIO_PUEN_PUEN14_Pos             (14)                                              /*!< GPIO_T::PUEN: PUEN14 Position          */
#define GPIO_PUEN_PUEN14_Msk             (0x1ul << GPIO_PUEN_PUEN14_Pos)                   /*!< GPIO_T::PUEN: PUEN14 Mask              */

#define GPIO_PUEN_PUEN15_Pos             (15)                                              /*!< GPIO_T::PUEN: PUEN15 Position          */
#define GPIO_PUEN_PUEN15_Msk             (0x1ul << GPIO_PUEN_PUEN15_Pos)                   /*!< GPIO_T::PUEN: PUEN15 Mask              */

#define GPIO_INTSTS_FLISTS0_Pos          (0)                                               /*!< GPIO_T::INTSTS: FLISTS0 Position       */
#define GPIO_INTSTS_FLISTS0_Msk          (0x1ul << GPIO_INTSTS_FLISTS0_Pos)                /*!< GPIO_T::INTSTS: FLISTS0 Mask           */

#define GPIO_INTSTS_FLISTS1_Pos          (1)                                               /*!< GPIO_T::INTSTS: FLISTS1 Position       */
#define GPIO_INTSTS_FLISTS1_Msk          (0x1ul << GPIO_INTSTS_FLISTS1_Pos)                /*!< GPIO_T::INTSTS: FLISTS1 Mask           */

#define GPIO_INTSTS_FLISTS2_Pos          (2)                                               /*!< GPIO_T::INTSTS: FLISTS2 Position       */
#define GPIO_INTSTS_FLISTS2_Msk          (0x1ul << GPIO_INTSTS_FLISTS2_Pos)                /*!< GPIO_T::INTSTS: FLISTS2 Mask           */

#define GPIO_INTSTS_FLISTS3_Pos          (3)                                               /*!< GPIO_T::INTSTS: FLISTS3 Position       */
#define GPIO_INTSTS_FLISTS3_Msk          (0x1ul << GPIO_INTSTS_FLISTS3_Pos)                /*!< GPIO_T::INTSTS: FLISTS3 Mask           */

#define GPIO_INTSTS_FLISTS4_Pos          (4)                                               /*!< GPIO_T::INTSTS: FLISTS4 Position       */
#define GPIO_INTSTS_FLISTS4_Msk          (0x1ul << GPIO_INTSTS_FLISTS4_Pos)                /*!< GPIO_T::INTSTS: FLISTS4 Mask           */

#define GPIO_INTSTS_FLISTS5_Pos          (5)                                               /*!< GPIO_T::INTSTS: FLISTS5 Position       */
#define GPIO_INTSTS_FLISTS5_Msk          (0x1ul << GPIO_INTSTS_FLISTS5_Pos)                /*!< GPIO_T::INTSTS: FLISTS5 Mask           */

#define GPIO_INTSTS_FLISTS6_Pos          (6)                                               /*!< GPIO_T::INTSTS: FLISTS6 Position       */
#define GPIO_INTSTS_FLISTS6_Msk          (0x1ul << GPIO_INTSTS_FLISTS6_Pos)                /*!< GPIO_T::INTSTS: FLISTS6 Mask           */

#define GPIO_INTSTS_FLISTS7_Pos          (7)                                               /*!< GPIO_T::INTSTS: FLISTS7 Position       */
#define GPIO_INTSTS_FLISTS7_Msk          (0x1ul << GPIO_INTSTS_FLISTS7_Pos)                /*!< GPIO_T::INTSTS: FLISTS7 Mask           */

#define GPIO_INTSTS_FLISTS8_Pos          (8)                                               /*!< GPIO_T::INTSTS: FLISTS8 Position       */
#define GPIO_INTSTS_FLISTS8_Msk          (0x1ul << GPIO_INTSTS_FLISTS8_Pos)                /*!< GPIO_T::INTSTS: FLISTS8 Mask           */

#define GPIO_INTSTS_FLISTS9_Pos          (9)                                               /*!< GPIO_T::INTSTS: FLISTS9 Position       */
#define GPIO_INTSTS_FLISTS9_Msk          (0x1ul << GPIO_INTSTS_FLISTS9_Pos)                /*!< GPIO_T::INTSTS: FLISTS9 Mask           */

#define GPIO_INTSTS_FLISTS10_Pos         (10)                                              /*!< GPIO_T::INTSTS: FLISTS10 Position      */
#define GPIO_INTSTS_FLISTS10_Msk         (0x1ul << GPIO_INTSTS_FLISTS10_Pos)               /*!< GPIO_T::INTSTS: FLISTS10 Mask          */

#define GPIO_INTSTS_FLISTS11_Pos         (11)                                              /*!< GPIO_T::INTSTS: FLISTS11 Position      */
#define GPIO_INTSTS_FLISTS11_Msk         (0x1ul << GPIO_INTSTS_FLISTS11_Pos)               /*!< GPIO_T::INTSTS: FLISTS11 Mask          */

#define GPIO_INTSTS_FLISTS12_Pos         (12)                                              /*!< GPIO_T::INTSTS: FLISTS12 Position      */
#define GPIO_INTSTS_FLISTS12_Msk         (0x1ul << GPIO_INTSTS_FLISTS12_Pos)               /*!< GPIO_T::INTSTS: FLISTS12 Mask          */

#define GPIO_INTSTS_FLISTS13_Pos         (13)                                              /*!< GPIO_T::INTSTS: FLISTS13 Position      */
#define GPIO_INTSTS_FLISTS13_Msk         (0x1ul << GPIO_INTSTS_FLISTS13_Pos)               /*!< GPIO_T::INTSTS: FLISTS13 Mask          */

#define GPIO_INTSTS_FLISTS14_Pos         (14)                                              /*!< GPIO_T::INTSTS: FLISTS14 Position      */
#define GPIO_INTSTS_FLISTS14_Msk         (0x1ul << GPIO_INTSTS_FLISTS14_Pos)               /*!< GPIO_T::INTSTS: FLISTS14 Mask          */

#define GPIO_INTSTS_FLISTS15_Pos         (15)                                              /*!< GPIO_T::INTSTS: FLISTS15 Position      */
#define GPIO_INTSTS_FLISTS15_Msk         (0x1ul << GPIO_INTSTS_FLISTS15_Pos)               /*!< GPIO_T::INTSTS: FLISTS15 Mask          */

#define GPIO_INTSTS_RHISTS0_Pos          (16)                                              /*!< GPIO_T::INTSTS: RHISTS0 Position       */
#define GPIO_INTSTS_RHISTS0_Msk          (0x1ul << GPIO_INTSTS_RHISTS0_Pos)                /*!< GPIO_T::INTSTS: RHISTS0 Mask           */

#define GPIO_INTSTS_RHISTS1_Pos          (17)                                              /*!< GPIO_T::INTSTS: RHISTS1 Position       */
#define GPIO_INTSTS_RHISTS1_Msk          (0x1ul << GPIO_INTSTS_RHISTS1_Pos)                /*!< GPIO_T::INTSTS: RHISTS1 Mask           */

#define GPIO_INTSTS_RHISTS2_Pos          (18)                                              /*!< GPIO_T::INTSTS: RHISTS2 Position       */
#define GPIO_INTSTS_RHISTS2_Msk          (0x1ul << GPIO_INTSTS_RHISTS2_Pos)                /*!< GPIO_T::INTSTS: RHISTS2 Mask           */

#define GPIO_INTSTS_RHISTS3_Pos          (19)                                              /*!< GPIO_T::INTSTS: RHISTS3 Position       */
#define GPIO_INTSTS_RHISTS3_Msk          (0x1ul << GPIO_INTSTS_RHISTS3_Pos)                /*!< GPIO_T::INTSTS: RHISTS3 Mask           */

#define GPIO_INTSTS_RHISTS4_Pos          (20)                                              /*!< GPIO_T::INTSTS: RHISTS4 Position       */
#define GPIO_INTSTS_RHISTS4_Msk          (0x1ul << GPIO_INTSTS_RHISTS4_Pos)                /*!< GPIO_T::INTSTS: RHISTS4 Mask           */

#define GPIO_INTSTS_RHISTS5_Pos          (21)                                              /*!< GPIO_T::INTSTS: RHISTS5 Position       */
#define GPIO_INTSTS_RHISTS5_Msk          (0x1ul << GPIO_INTSTS_RHISTS5_Pos)                /*!< GPIO_T::INTSTS: RHISTS5 Mask           */

#define GPIO_INTSTS_RHISTS6_Pos          (22)                                              /*!< GPIO_T::INTSTS: RHISTS6 Position       */
#define GPIO_INTSTS_RHISTS6_Msk          (0x1ul << GPIO_INTSTS_RHISTS6_Pos)                /*!< GPIO_T::INTSTS: RHISTS6 Mask           */

#define GPIO_INTSTS_RHISTS7_Pos          (23)                                              /*!< GPIO_T::INTSTS: RHISTS7 Position       */
#define GPIO_INTSTS_RHISTS7_Msk          (0x1ul << GPIO_INTSTS_RHISTS7_Pos)                /*!< GPIO_T::INTSTS: RHISTS7 Mask           */

#define GPIO_INTSTS_RHISTS8_Pos          (24)                                              /*!< GPIO_T::INTSTS: RHISTS8 Position       */
#define GPIO_INTSTS_RHISTS8_Msk          (0x1ul << GPIO_INTSTS_RHISTS8_Pos)                /*!< GPIO_T::INTSTS: RHISTS8 Mask           */

#define GPIO_INTSTS_RHISTS9_Pos          (25)                                              /*!< GPIO_T::INTSTS: RHISTS9 Position       */
#define GPIO_INTSTS_RHISTS9_Msk          (0x1ul << GPIO_INTSTS_RHISTS9_Pos)                /*!< GPIO_T::INTSTS: RHISTS9 Mask           */

#define GPIO_INTSTS_RHISTS10_Pos         (26)                                              /*!< GPIO_T::INTSTS: RHISTS10 Position      */
#define GPIO_INTSTS_RHISTS10_Msk         (0x1ul << GPIO_INTSTS_RHISTS10_Pos)               /*!< GPIO_T::INTSTS: RHISTS10 Mask          */

#define GPIO_INTSTS_RHISTS11_Pos         (27)                                              /*!< GPIO_T::INTSTS: RHISTS11 Position      */
#define GPIO_INTSTS_RHISTS11_Msk         (0x1ul << GPIO_INTSTS_RHISTS11_Pos)               /*!< GPIO_T::INTSTS: RHISTS11 Mask          */

#define GPIO_INTSTS_RHISTS12_Pos         (28)                                              /*!< GPIO_T::INTSTS: RHISTS12 Position      */
#define GPIO_INTSTS_RHISTS12_Msk         (0x1ul << GPIO_INTSTS_RHISTS12_Pos)               /*!< GPIO_T::INTSTS: RHISTS12 Mask          */

#define GPIO_INTSTS_RHISTS13_Pos         (29)                                              /*!< GPIO_T::INTSTS: RHISTS13 Position      */
#define GPIO_INTSTS_RHISTS13_Msk         (0x1ul << GPIO_INTSTS_RHISTS13_Pos)               /*!< GPIO_T::INTSTS: RHISTS13 Mask          */

#define GPIO_INTSTS_RHISTS14_Pos         (30)                                              /*!< GPIO_T::INTSTS: RHISTS14 Position      */
#define GPIO_INTSTS_RHISTS14_Msk         (0x1ul << GPIO_INTSTS_RHISTS14_Pos)               /*!< GPIO_T::INTSTS: RHISTS14 Mask          */

#define GPIO_INTSTS_RHISTS15_Pos         (31)                                              /*!< GPIO_T::INTSTS: RHISTS15 Position      */
#define GPIO_INTSTS_RHISTS15_Msk         (0x1ul << GPIO_INTSTS_RHISTS15_Pos)               /*!< GPIO_T::INTSTS: RHISTS15 Mask          */

#define GPIO_DBCTL_DBCLKSEL_Pos          (0)                                               /*!< GP_DB_T::DBCTL: DBCLKSEL Position       */
#define GPIO_DBCTL_DBCLKSEL_Msk          (0xful << GPIO_DBCTL_DBCLKSEL_Pos)                /*!< GP_DB_T::DBCTL: DBCLKSEL Mask           */

#define GPIO_DBCTL_DBCLKSRC_Pos          (4)                                               /*!< GP_DB_T::DBCTL: DBCLKSRC Position       */
#define GPIO_DBCTL_DBCLKSRC_Msk          (0x1ul << GPIO_DBCTL_DBCLKSRC_Pos)                /*!< GP_DB_T::DBCTL: DBCLKSRC Mask           */

#define GPIO_DBCTL_ICLKON_Pos            (5)                                               /*!< GP_DB_T::DBCTL: ICLKON Position         */
#define GPIO_DBCTL_ICLKON_Msk            (0x1ul << GPIO_DBCTL_ICLKON_Pos)                  /*!< GP_DB_T::DBCTL: ICLKON Mask             */


/**@}*/ /* GPIO_CONST */
/**@}*/ /* end of GPIO register group */

/*---------------------- Peripheral Direct Memory Access Controller -------------------------*/
/**
    @addtogroup DMA Direct Memory Access Controller(DMA)
    Memory Mapped Structure for DMA Controller
@{ */

typedef struct
{


    /**
     * @var PDMA_CH_T::CTLn
     * Offset: 0x00  PDMA channel n Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CHEN      |PDMA Channel Enable Bit
     * |        |          |Setting this bit to 1 enables PDMA operation
     * |        |          |If this bit is cleared, PDMA will ignore all PDMA request and force Bus Master into IDLE state.
     * |        |          |Note: SWRST (PDMA_CTLn[1], n= 1~4) will clear this bit.
     * |[1]     |SWRST     |Software Engine Reset
     * |        |          |0 = No effect.
     * |        |          |1 = Reset the internal state machine, pointers and internal buffer
     * |        |          |The contents of all control registers will not be cleared
     * |        |          |This bit will be automatically cleared after few clock cycles.
     * |[5:4]   |SASEL     |Transfer Source Address Direction Selection
     * |        |          |00 = Transfer Source address is incremented successively.
     * |        |          |01 = Reserved.
     * |        |          |10 = Transfer Source address is fixed (This feature can be used when data where transferred from a single source to multiple destinations).
     * |        |          |11 = Transfer Source address is wrap around (When the PDMA_CCNT is equal to 0, the PDMA_CSA and PDMA_CCNT registers will be updated by PDMA_SA and PDMA_CNT automatically
     * |        |          |PDMA will start another transfer without user trigger until CHEN disabled
     * |        |          |When the CHEN is disabled, the PDMA will complete the active transfer but the remained data which in the PDMA buffer will not transfer to destination address).
     * |[7:6]   |DASEL     |Transfer Destination Address Direction Selection
     * |        |          |00 = Transfer Destination address is incremented successively.
     * |        |          |01 = Reserved.
     * |        |          |10 = Transfer Destination address is fixed
     * |        |          |(This feature can be used when data transferred from multiple sources to a single destination).
     * |        |          |11 = Transfer Destination address is wrapped around (When the PDMA_CCNT is equal to 0, the PDMA_CDA and PDMA_CCNT registers will be updated by PDMA_DA and PDMA_CNT automatically
     * |        |          |PDMA will start another transfer without user trigger until CHEN disabled
     * |        |          |When the CHEN is disabled, the PDMA will complete the active transfer but the remained data which in the PDMA buffer will not transfer to destination address).
     * |[12]    |TOUTEN    |Time-out Enable Bit
     * |        |          |This bit will enable PDMA Time-out counter (PDMA_TOCn, n=1~4)
     * |        |          |While this counter counts to 0, the TOUTIF (PDMA_INTSTSn[6], n=1~4) will be set.
     * |        |          |0 = PDMA internal counter Disabled.
     * |        |          |1 = PDMA internal counter Enabled.
     * |[20:19] |TXWIDTH   |Transfer Width Selection
     * |        |          |This field is used for transfer width.
     * |        |          |00 = One word (32-bit) is transferred for every PDMA operation.
     * |        |          |01 = One byte (8-bit) is transferred for every PDMA operation.
     * |        |          |10 = One half-word (16-bit) is transferred for every PDMA operation.
     * |        |          |11 = Reserved.
     * |[23]    |TRIGEN    |Trigger Enable Bit
     * |        |          |0 = No effect.
     * |        |          |1 = PDMA data transfer Enabled.
     * |        |          |Note1: When PDMA transfer completed, this bit will be cleared automatically.
     * |        |          |Note2: If the bus error occurs, all PDMA transfer will be stopped
     * |        |          |User must reset all PDMA channels, and then trigger again.
     * @var PDMA_CH_T::SAn
     * Offset: 0x04  PDMA channel n Source Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |SA        |PDMA Transfer Source Address Bits
     * |        |          |This field indicates a 32-bit source address of PDMA.
     * |        |          |Note: The source address must be word alignment.
     * @var PDMA_CH_T::DAn
     * Offset: 0x08  PDMA channel n Destination Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |DA        |PDMA Transfer Destination Address Bits
     * |        |          |This field indicates a 32-bit destination address of PDMA.
     * |        |          |Note: The Destination address must be word alignment.
     * @var PDMA_CH_T::CNTn
     * Offset: 0x0C  PDMA channel n Transfer Count Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |TCNT      |PDMA Transfer Count Bits
     * |        |          |This field indicates a 16-bit transfer count number of PDMA.
     * |[31:16] |PCNTITH   |PDMA Periodic Count Interrupt Threshold
     * |        |          |This field indicates how many data transferred to generate periodic interrupt
     * |        |          |Note: write 0 to this field to disable this function.
     * @var PDMA_CH_T::CSAn
     * Offset: 0x14  PDMA channel n Current Source Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |CSA       |PDMA Current Source Address Bits (Read Only)
     * |        |          |This field indicates the source address where the PDMA transfer just occurred.
     * @var PDMA_CH_T::CDAn
     * Offset: 0x18  PDMA channel n Current Destination Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |CDA       |PDMA Current Destination Address Bits (Read Only)
     * |        |          |This field indicates the destination address where the PDMA transfer just occurred.
     * @var PDMA_CH_T::CCNTn
     * Offset: 0x1C  PDMA channel n Current Transfer Count Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CCNT      |PDMA Current Count Bits (Read Only)
     * |        |          |This field indicates the current remained transfer count of PDMA.
     * |        |          |Note: This field value will be cleared to 0 when user sets SWRST (PDMA_CTLn[1],n=1~4) to 1.
     * @var PDMA_CH_T::INTENn
     * Offset: 0x20  PDMA channel n Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TABTIEN   |PDMA Read/Write Target Abort Interrupt Enable Bit
     * |        |          |0 = Target abort interrupt Disabled during PDMA transfer.
     * |        |          |1 = Target abort interrupt Enabled during PDMA transfer.
     * |[1]     |TDIEN     |PDMA Transfer Done Interrupt Enable Bit
     * |        |          |0 = Interrupt Disabled when PDMA transfer is done.
     * |        |          |1 = Interrupt Enabled when PDMA transfer is done.
     * |[6]     |TOUTIEN   |Time-out Interrupt Enable Bit
     * |        |          |0 = Time-out interrupt Disabled.
     * |        |          |1 = Time-out interrupt Enabled.
     * |[8]     |PCNTIEN   |Periodic Count Interrupt Enable Bit
     * |        |          |This field indicates how many data transferred to generate interrupt periodically.
     * |        |          |0 = Periodic transfer count interrupt Disabled.
     * |        |          |1 = Periodic transfer count interrupt Enabled.
     * @var PDMA_CH_T::INTSTSn
     * Offset: 0x24  PDMA channel n Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TABTIF    |PDMA Read/Write Target Abort Interrupt Status Flag
     * |        |          |0 = No bus ERROR response received.
     * |        |          |1 = Bus ERROR response received.
     * |        |          |Note1: This bit is cleared by writing 1 to it.
     * |        |          |Note2: This bit indicates bus master received error response or not, if bus master received error response, it means that target abort is happened
     * |        |          |PDMA controller will stop transfer and respond this event to user then go to IDLE state
     * |        |          |When target abort occurred, user must reset PDMA controller, and then transfer those data again.
     * |[1]     |TDIF      |Transfer Done Interrupt Status Flag
     * |        |          |This bit indicates that PDMA has finished all transfer.
     * |        |          |0 = Not finished yet.
     * |        |          |1 = Done.
     * |        |          |Note: This bit is cleared by writing 1 to it.
     * |[6]     |TOUTIF    |Time-out Interrupt Status Flag
     * |        |          |This flag indicated that PDMA has waited peripheral request for a period defined by PDMA_TOC.
     * |        |          |0 = No time-out flag.
     * |        |          |1 = Time-out flag.
     * |        |          |Note: This bit is cleared by writing 1 to it.
     * |[8]     |PCNTIF    |Periodic Count Interrupt Status Flag
     * |        |          |This flag indicates PCNTITH (PDMA_CNTn[30:16], n=1~4) data has been transferred.
     * |        |          |Note: This bit is cleared by writing 1 to it.
     * @var PDMA_CH_T::TOCn
     * Offset: 0x28  PDMA channel n Time-out Counter Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |TOC       |PDMA Time-out Period Counter
     * |        |          |Each PDMA channel contains an internal counter
     * |        |          |This internal counter will reload and start counting when completing each peripheral request service
     * |        |          |The internal counter loads the value of TOC (PDAM_TOCn[15:0], n=1~4) and starts counting down when setting TOUTEN (PDMA_CTLn[12], n=1~4)
     * |        |          |PDMA will request interrupt when this internal counter reaches 0 and TOUTIEN (PDMA_INTENn[6], n=1~4) is 1
     * |[18:16] |TPSC      |PDMA Time-out Counter Clock Source Prescaler
     * |        |          |000 = PDMA time-out clock source is HCLK/28.
     * |        |          |001 = PDMA time-out clock source is HCLK/29.
     * |        |          |010 = PDMA time-out clock source is HCLK/210.
     * |        |          |011 = PDMA time-out clock source is HCLK/211.
     * |        |          |100 = PDMA time-out clock source is HCLK/212.
     * |        |          |101 = PDMA time-out clock source is HCLK/213.
     * |        |          |110 = PDMA time-out clock source is HCLK/214.
     * |        |          |111 = PDMA time-out clock source is HCLK/215.
     */
    __IO uint32_t CTLn;                  /*!< [0x0000] PDMA channel n Control Register                                  */
    __IO uint32_t SAn;                   /*!< [0x0004] PDMA channel n Source Address Register                           */
    __IO uint32_t DAn;                   /*!< [0x0008] PDMA channel n Destination Address Register                      */
    __IO uint32_t CNTn;                  /*!< [0x000c] PDMA channel n Transfer Count Register                           */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE0[1];
    /// @endcond //HIDDEN_SYMBOLS
    __I  uint32_t CSAn;                  /*!< [0x0014] PDMA channel n Current Source Address Register                   */
    __I  uint32_t CDAn;                  /*!< [0x0018] PDMA channel n Current Destination Address Register              */
    __I  uint32_t CCNTn;                 /*!< [0x001c] PDMA channel n Current Transfer Count Register                   */
    __IO uint32_t INTENn;                /*!< [0x0020] PDMA channel n Interrupt Enable Register                         */
    __IO uint32_t INTSTSn;               /*!< [0x0024] PDMA channel n Interrupt Status Register                         */
    __IO uint32_t TOCn;                  /*!< [0x0028] PDMA channel n Time-out Counter Register                         */

} PDMA_CH_T;



typedef struct
{


    /**
     * @var DMA_CRC_T::CTL
     * Offset: 0x00  CRC Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CRCEN     |CRC Channel Enable Bit
     * |        |          |0 = No effect.
     * |        |          |1 = CRC operation Enabled.
     * |        |          |When operating in CRC DMA mode (TRIGEN (CRC_CTL[23]) = 1), if user clears this bit, the PDMA operation will be continuous until all CRC DMA operation is done, and the TRIGEN bit will keep 1 until all CRC DMA operation done
     * |        |          |But in this case, the CRCTDIF (CRC_DMAINTSTS[1]) flag will be inactive, user can read CRC checksum result only if TRIGEN clears to 0
     * |        |          |When operating in CRC DMA mode (TRIGEN (CRC_CTL[23]) = 1), if user wants to stop the transfer immediately, user can write 1 to CRCRST (CRC_CTL[1]) bit to stop the transmission.
     * |[1]     |CRCRST    |CRC Engine Reset Bit
     * |        |          |0 = No effect.
     * |        |          |1 = Reset the internal CRC state machine and internal buffer
     * |        |          |The others contents of CRC_CTL register will not be cleared.
     * |        |          |Note1: This bit will be cleared automatically.
     * |        |          |Note2: When operating in CPU mode, setting this bit will reload the seed value from CRC_SEED register as checksum initial value.
     * |[23]    |TRIGEN    |Trigger Enable Bit
     * |        |          |This bit is used to trigger the CRC DMA transfer.
     * |        |          |0 = No effect.
     * |        |          |1 = CRC DMA data read or write transfer Enabled.
     * |        |          |Note1: If this bit asserts which indicates the CRC engine operation in CRC DMA mode, do not fill in any data in CRC_DAT register.
     * |        |          |Note2: When CRC DMA transfer completed, this bit will be cleared automatically.
     * |        |          |Note3: If the bus error occurs when CRC DMA transfer data, all CRC DMA transfer will be stopped
     * |        |          |User must reset all DMA channel before trigger DMA again.
     * |[24]    |DATREV    |Write Data Bit Order Reverse
     * |        |          |This bit is used to enable the bit order reverse function for writing data value in CRC_DTA register.
     * |        |          |0 = Bit order reverse for CRC data write in Disabled.
     * |        |          |1 = Bit order reverse for CRC data write in Enabled (per byte).
     * |        |          |Note: If the write data is 0xAABBCCDD, the bit order reverse for CRC data write in is 0x55DD33BB.
     * |[25]    |CHKSREV   |Checksum Bit Order Reverse
     * |        |          |This bit is used to enable the bit order reverse function for checksum result in CRC_CHECKSUM register.
     * |        |          |0 = Bit order reverse for CRC checksum Disabled.
     * |        |          |1 = Bit order reverse for CRC checksum Enabled.
     * |        |          |Note: If the checksum result is 0XDD7B0F2E, the bit order reverse for CRC checksum is 0x74F0DEBB
     * |[26]    |DATFMT    |Write Data 1's Complement
     * |        |          |This bit is used to enable the 1's complement function for write data value in CRC_DTA register.
     * |        |          |0 = 1's complement for CRC writes data in Disabled.
     * |        |          |1 = 1's complement for CRC writes data in Enabled.
     * |[27]    |CHKSFMT   |Checksum 1's Complement
     * |        |          |This bit is used to enable the 1's complement function for checksum result in CRC_CHECKSUM register.
     * |        |          |0 = 1's complement for CRC checksum Disabled.
     * |        |          |1 = 1's complement for CRC checksum Enabled.
     * |[29:28] |DATLEN    |CPU Write Data Length
     * |        |          |This field indicates the CPU write data length only when operating in CPU mode.
     * |        |          |00 = The write data length is 8-bit mode.
     * |        |          |01 = The write data length is 16-bit mode.
     * |        |          |10 = The write data length is 32-bit mode.
     * |        |          |11 = Reserved.
     * |        |          |Note1: This field is only valid when operating in CPU mode.
     * |        |          |Note2: When the write data length is 8-bit mode, the valid data in CRC_DAT register is only DATA [7:0] bits; if the write data length is 16-bit mode, the valid data in CRC_DAT register is only DATA [15:0].
     * |[31:30] |CRCMODE   |CRC Polynomial Mode
     * |        |          |This field indicates the CRC operation polynomial mode.
     * |        |          |00 = CRC-CCITT Polynomial Mode.
     * |        |          |01 = CRC-8 Polynomial Mode.
     * |        |          |10 = CRC-16 Polynomial Mode.
     * |        |          |11 = CRC-32 Polynomial Mode.
     * @var DMA_CRC_T::DMASA
     * Offset: 0x04  CRC DMA Source Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |SA        |CRC DMA Transfer Source Address Bits
     * |        |          |This field indicates a 32-bit source address of CRC DMA.
     * |        |          |Note: The source address must be word alignment.
     * @var DMA_CRC_T::DMABCNT
     * Offset: 0x0C  CRC DMA Transfer Byte Count Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |BCNT      |CRC DMA Transfer Byte Count
     * |        |          |This field indicates a 16-bit total transfer byte count number of CRC DMA.
     * @var DMA_CRC_T::DMACSA
     * Offset: 0x14  CRC DMA Current Source Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |CSA       |CRC DMA Current Source Address Bits (Read Only)
     * |        |          |This field indicates the current source address where the CRC DMA transfer just occurs.
     * @var DMA_CRC_T::DMACBCNT
     * Offset: 0x1C  CRC DMA Current Transfer Byte Count Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CBCNT     |CRC DMA Current Remained Byte Count (Read Only)
     * |        |          |This field indicates the current remained byte count of CRC DMA.
     * |        |          |Note: Setting the CRCRST (CRC_CTL[1]) bit to 1 will clear this register value.
     * @var DMA_CRC_T::DMAINTEN
     * Offset: 0x20  CRC DMA Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TABTIEN   |CRC DMA Read/Write Target Abort Interrupt Enable Bit
     * |        |          |Enable this bit will generate the CRC DMA Target Abort interrupt signal while TABTIF (CRC_DMAINTSTS[0]) bit is set to 1.
     * |        |          |0 = Target abort interrupt Disabled during CRC DMA transfer.
     * |        |          |1 = Target abort interrupt Enabled during CRC DMA transfer.
     * |[1]     |TDIEN     |CRC DMA Block Transfer Done Interrupt Enable Bit
     * |        |          |Enable this bit will generate the CRC DMA Transfer Done interrupt signal while TDIF (CRC_DMAINTSTS[1]) bit is set to 1.
     * |        |          |0 = Interrupt Disabled when CRC DMA transfer done.
     * |        |          |1 = Interrupt Enabled when CRC DMA transfer done.
     * @var DMA_CRC_T::DMAISTS
     * Offset: 0x24  CRC DMA Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TABTIF    |CRC DMA Read/Write Target Abort Interrupt Flag
     * |        |          |This bit indicates that CRC bus has error or not during CRC DMA transfer.
     * |        |          |0 = No bus error response received during CRC DMA transfer.
     * |        |          |1 = Bus error response received during CRC DMA transfer.
     * |        |          |Note1: This bit is cleared by writing 1 to it.
     * |        |          |Note2: This bit indicates bus master received error response or not
     * |        |          |If bus master received error response, it means that CRC transfer target abort is happened
     * |        |          |DMA will stop transfer and respond this event to user then CRC state machine goes to IDLE state
     * |        |          |When target abort occurred, user must reset DMA before transfer those data again.
     * |[1]     |TDIF      |CRC DMA Transfer Done Interrupt Flag
     * |        |          |This bit indicates that CRC DMA transfer has finished or not.
     * |        |          |0 = Not finished if TRIGEN (CRC_CTL[23]) has enabled.
     * |        |          |1 = CRC transfer done if TRIGEN (CRC_CTL[23]) has enabled.
     * |        |          |Note1: This bit is cleared by writing 1 to it.
     * |        |          |Note2: When CRC DMA transfer is done, TRIGEN (CRC_CTL[23]) will be cleared automatically.
     * @var DMA_CRC_T::DAT
     * Offset: 0x80  CRC Write Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |DATA      |CRC Write Data Bits
     * |        |          |When operating in CPU mode, user can write data to this field to perform CRC operation.
     * |        |          |When operating in DMA mode, this field indicates the DMA read data from memory and cannot be written by user.
     * |        |          |Note: When the write data length is 8-bit mode, the valid data in CRC_DAT register are only DATA[7:0] bits; if the write data length is 16-bit mode, the valid data in CRC_DAT register are only DATA[15:0] bits.
     * @var DMA_CRC_T::SEED
     * Offset: 0x84  CRC Seed Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |SEED      |CRC Seed Value
     * |        |          |This field indicates the CRC seed value.
     * @var DMA_CRC_T::CHECKSUM
     * Offset: 0x88  CRC Checksum Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |CHECKSUM  |CRC Checksum Results
     * |        |          |This field indicates the CRC checksum result
     */
    __IO uint32_t CTL;                   /*!< [0x0000] CRC Control Register                                             */
    __IO uint32_t DMASA;                 /*!< [0x0004] CRC DMA Source Address Register                                  */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE0[1];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t DMABCNT;               /*!< [0x000c] CRC DMA Transfer Byte Count Register                             */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE1[1];
    /// @endcond //HIDDEN_SYMBOLS
    __I  uint32_t DMACSA;                /*!< [0x0014] CRC DMA Current Source Address Register                          */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE2[1];
    /// @endcond //HIDDEN_SYMBOLS
    __I  uint32_t DMACBCNT;              /*!< [0x001c] CRC DMA Current Transfer Byte Count Register                     */
    __IO uint32_t DMAINTEN;              /*!< [0x0020] CRC DMA Interrupt Enable Register                                */
    __IO uint32_t DMAISTS;               /*!< [0x0024] CRC DMA Interrupt Status Register                                */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE3[22];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t DAT;                   /*!< [0x0080] CRC Write Data Register                                          */
    __IO uint32_t SEED;                  /*!< [0x0084] CRC Seed Register                                                */
    __I  uint32_t CHECKSUM;              /*!< [0x0088] CRC Checksum Register                                            */

} DMA_CRC_T;



typedef struct
{


    /**
     * @var DMA_GCR_T::GCTL
     * Offset: 0x00  PDMA Global Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[9]     |CKEN1     |PDMA Controller Channel 1 Clock Enable Bit
     * |        |          |0 = PDMA channel 1 clock Disabled.
     * |        |          |1 = PDMA channel 1 clock Enabled.
     * |[10]    |CKEN2     |PDMA Controller Channel 2 Clock Enable Bit
     * |        |          |0 = PDMA channel 2 clock Disabled.
     * |        |          |1 = PDMA channel 2 clock Enabled.
     * |[11]    |CKEN3     |PDMA Controller Channel 3 Clock Enable Bit
     * |        |          |0 = PDMA channel 3 clock Disabled.
     * |        |          |1 = PDMA channel 3 clock Enabled.
     * |[12]    |CKEN4     |PDMA Controller Channel 4 Clock Enable Bit
     * |        |          |0 = PDMA channel 4 clock Disabled.
     * |        |          |1 = PDMA channel 4 clock Enabled.
     * |[24]    |CKENCRC   |CRC Controller Clock Enable Bit
     * |        |          |0 = CRC channel clock Disabled.
     * |        |          |1 = CRC channel clock Enabled.
     * @var DMA_GCR_T::REQSEL0
     * Offset: 0x04  PDMA Request Source Select Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[12:8]  |REQSRC1   |Channel 1 Request Source Selection
     * |        |          |This filed defines which peripheral is connected to PDMA channel 1
     * |        |          |User can configure the peripheral by setting CH1_SEL.
     * |        |          |00000 = Connect to SPI0_TX.
     * |        |          |00001 = Connect to SPI1_TX.
     * |        |          |00010 = Connect to UART0_TX.
     * |        |          |00011 = Connect to UART1_TX.
     * |        |          |00100 = Reserved.
     * |        |          |00101 = Connect to SPI3_TX.
     * |        |          |00110 = Reserved.
     * |        |          |00111 = Reserved.
     * |        |          |01000 = Connect to SPI2_TX.
     * |        |          |01001 = Connect to TMR0.
     * |        |          |01010 = Connect to TMR1.
     * |        |          |01011 = Connect to TMR2.
     * |        |          |01100 = Connect to TMR3.
     * |        |          |10000 = Connect to SPI0_RX.
     * |        |          |10001 = Connect to SPI1_RX.
     * |        |          |10010 = Connect to UART0_RX.
     * |        |          |10011 = Connect to UART1_RX.
     * |        |          |10100 = Reserved.
     * |        |          |10101 = Connect to SPI3_RX.
     * |        |          |10110 = Connect to ADC.
     * |        |          |10111 = Reserved.
     * |        |          |11000 = Connect to SPI2_RX.
     * |        |          |11001 = Reserved.
     * |        |          |11010 = Reserved.
     * |        |          |11011 = Reserved.
     * |        |          |11100 = Reserved.
     * |        |          |Others = Disable to connected any peripheral.
     * |[20:16] |REQSRC2   |Channel 2 Request Source Selection
     * |        |          |This filed defines which peripheral is connected to PDMA channel 2
     * |        |          |User can configure the peripheral setting by REQSRC2.
     * |        |          |Note: The channel configuration is the same as REQSRC1 field
     * |        |          |Please refer to the explanation of REQSRC1.
     * |[28:24] |REQSRC3   |Channel 3 Request Source Selection
     * |        |          |This filed defines which peripheral is connected to PDMA channel 3
     * |        |          |User can configure the peripheral setting by REQSRC3.
     * |        |          |Note: The channel configuration is the same as REQSRC1 field
     * |        |          |Please refer to the explanation of REQSRC1.
     * @var DMA_GCR_T::REQSEL1
     * Offset: 0x08  PDMA Request Source Select Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[4:0]   |REQSRC4   |Channel 4 Request Source Selection
     * |        |          |This filed defines which peripheral is connected to PDMA channel 4
     * |        |          |User can configure the peripheral setting by REQSRC4.
     * |        |          |Note: The channel configuration is the same as REQSRC1 field
     * |        |          |Please refer to the explanation of REQSRC1.
     * @var DMA_GCR_T::GINTSTS
     * Offset: 0x0C  PDMA Global Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1]     |IF1       |PDMA Channel 1 Interrupt Status (Read Only)
     * |        |          |This bit indicates the interrupt status of PDMA channel 1.
     * |[2]     |IF2       |PDMA Channel 2 Interrupt Status Flag of (Read Only)
     * |        |          |This bit indicates the interrupt status of PDMA channel 2.
     * |[3]     |IF3       |PDMA Channel 3 Interrupt Status (Read Only)
     * |        |          |This bit indicates the interrupt status of PDMA channel 3.
     * |[4]     |IF4       |PDMA Channel 4 Interrupt Status Flag (Read Only)
     * |        |          |This bit indicates the interrupt status of PDMA channel 4.
     * |[16]    |IFCRC     |CRC Controller Interrupt Status Flag (Read Only)
     * |        |          |This bit indicates the interrupt status of CRC controller
     */
    __IO uint32_t GCTL;                  /*!< [0x0000] PDMA Global Control Register                                     */
    __IO uint32_t REQSEL0;               /*!< [0x0004] PDMA Request Source Select Register 0                            */
    __IO uint32_t REQSEL1;               /*!< [0x0008] PDMA Request Source Select Register 1                            */
    __I  uint32_t GINTSTS;               /*!< [0x000c] PDMA Global Interrupt Status Register                            */

} DMA_GCR_T;

/**
    @addtogroup PDMA_CH_CONST PDMA_CH Bit Field Definition
    Constant Definitions for PDMA_CH Controller
@{ */

#define PDMA_CH_CTLn_CHEN_Pos            (0)                                               /*!< PDMA_CH_T::CTLn: CHEN Position         */
#define PDMA_CH_CTLn_CHEN_Msk            (0x1ul << PDMA_CH_CTLn_CHEN_Pos)                  /*!< PDMA_CH_T::CTLn: CHEN Mask             */

#define PDMA_CH_CTLn_SWRST_Pos           (1)                                               /*!< PDMA_CH_T::CTLn: SWRST Position        */
#define PDMA_CH_CTLn_SWRST_Msk           (0x1ul << PDMA_CH_CTLn_SWRST_Pos)                 /*!< PDMA_CH_T::CTLn: SWRST Mask            */

#define PDMA_CH_CTLn_SASEL_Pos           (4)                                               /*!< PDMA_CH_T::CTLn: SASEL Position        */
#define PDMA_CH_CTLn_SASEL_Msk           (0x3ul << PDMA_CH_CTLn_SASEL_Pos)                 /*!< PDMA_CH_T::CTLn: SASEL Mask            */

#define PDMA_CH_CTLn_DASEL_Pos           (6)                                               /*!< PDMA_CH_T::CTLn: DASEL Position        */
#define PDMA_CH_CTLn_DASEL_Msk           (0x3ul << PDMA_CH_CTLn_DASEL_Pos)                 /*!< PDMA_CH_T::CTLn: DASEL Mask            */

#define PDMA_CH_CTLn_TOUTEN_Pos          (12)                                              /*!< PDMA_CH_T::CTLn: TOUTEN Position       */
#define PDMA_CH_CTLn_TOUTEN_Msk          (0x1ul << PDMA_CH_CTLn_TOUTEN_Pos)                /*!< PDMA_CH_T::CTLn: TOUTEN Mask           */

#define PDMA_CH_CTLn_TXWIDTH_Pos         (19)                                              /*!< PDMA_CH_T::CTLn: TXWIDTH Position      */
#define PDMA_CH_CTLn_TXWIDTH_Msk         (0x3ul << PDMA_CH_CTLn_TXWIDTH_Pos)               /*!< PDMA_CH_T::CTLn: TXWIDTH Mask          */

#define PDMA_CH_CTLn_TRIGEN_Pos          (23)                                              /*!< PDMA_CH_T::CTLn: TRIGEN Position       */
#define PDMA_CH_CTLn_TRIGEN_Msk          (0x1ul << PDMA_CH_CTLn_TRIGEN_Pos)                /*!< PDMA_CH_T::CTLn: TRIGEN Mask           */

#define PDMA_CH_SAn_SA_Pos               (0)                                               /*!< PDMA_CH_T::SAn: SA Position            */
#define PDMA_CH_SAn_SA_Msk               (0xfffffffful << PDMA_CH_SAn_SA_Pos)              /*!< PDMA_CH_T::SAn: SA Mask                */

#define PDMA_CH_DAn_DA_Pos               (0)                                               /*!< PDMA_CH_T::DAn: DA Position            */
#define PDMA_CH_DAn_DA_Msk               (0xfffffffful << PDMA_CH_DAn_DA_Pos)              /*!< PDMA_CH_T::DAn: DA Mask                */

#define PDMA_CH_CNTn_TCNT_Pos            (0)                                               /*!< PDMA_CH_T::CNTn: TCNT Position         */
#define PDMA_CH_CNTn_TCNT_Msk            (0xfffful << PDMA_CH_CNTn_TCNT_Pos)               /*!< PDMA_CH_T::CNTn: TCNT Mask             */

#define PDMA_CH_CNTn_PCNTITH_Pos         (16)                                              /*!< PDMA_CH_T::CNTn: PCNTITH Position      */
#define PDMA_CH_CNTn_PCNTITH_Msk         (0xfffful << PDMA_CH_CNTn_PCNTITH_Pos)            /*!< PDMA_CH_T::CNTn: PCNTITH Mask          */

#define PDMA_CH_CSAn_CSA_Pos             (0)                                               /*!< PDMA_CH_T::CSAn: CSA Position          */
#define PDMA_CH_CSAn_CSA_Msk             (0xfffffffful << PDMA_CH_CSAn_CSA_Pos)            /*!< PDMA_CH_T::CSAn: CSA Mask              */

#define PDMA_CH_CDAn_CDA_Pos             (0)                                               /*!< PDMA_CH_T::CDAn: CDA Position          */
#define PDMA_CH_CDAn_CDA_Msk             (0xfffffffful << PDMA_CH_CDAn_CDA_Pos)            /*!< PDMA_CH_T::CDAn: CDA Mask              */

#define PDMA_CH_CCNTn_CCNT_Pos           (0)                                               /*!< PDMA_CH_T::CCNTn: CCNT Position        */
#define PDMA_CH_CCNTn_CCNT_Msk           (0xfffful << PDMA_CH_CCNTn_CCNT_Pos)              /*!< PDMA_CH_T::CCNTn: CCNT Mask            */

#define PDMA_CH_INTENn_TABTIEN_Pos       (0)                                               /*!< PDMA_CH_T::INTENn: TABTIEN Position    */
#define PDMA_CH_INTENn_TABTIEN_Msk       (0x1ul << PDMA_CH_INTENn_TABTIEN_Pos)             /*!< PDMA_CH_T::INTENn: TABTIEN Mask        */

#define PDMA_CH_INTENn_TDIEN_Pos         (1)                                               /*!< PDMA_CH_T::INTENn: TDIEN Position      */
#define PDMA_CH_INTENn_TDIEN_Msk         (0x1ul << PDMA_CH_INTENn_TDIEN_Pos)               /*!< PDMA_CH_T::INTENn: TDIEN Mask          */

#define PDMA_CH_INTENn_TOUTIEN_Pos       (6)                                               /*!< PDMA_CH_T::INTENn: TOUTIEN Position    */
#define PDMA_CH_INTENn_TOUTIEN_Msk       (0x1ul << PDMA_CH_INTENn_TOUTIEN_Pos)             /*!< PDMA_CH_T::INTENn: TOUTIEN Mask        */

#define PDMA_CH_INTENn_PCNTIEN_Pos       (8)                                               /*!< PDMA_CH_T::INTENn: PCNTIEN Position    */
#define PDMA_CH_INTENn_PCNTIEN_Msk       (0x1ul << PDMA_CH_INTENn_PCNTIEN_Pos)             /*!< PDMA_CH_T::INTENn: PCNTIEN Mask        */

#define PDMA_CH_INTSTSn_TABTIF_Pos       (0)                                               /*!< PDMA_CH_T::INTSTSn: TABTIF Position    */
#define PDMA_CH_INTSTSn_TABTIF_Msk       (0x1ul << PDMA_CH_INTSTSn_TABTIF_Pos)             /*!< PDMA_CH_T::INTSTSn: TABTIF Mask        */

#define PDMA_CH_INTSTSn_TDIF_Pos         (1)                                               /*!< PDMA_CH_T::INTSTSn: TDIF Position      */
#define PDMA_CH_INTSTSn_TDIF_Msk         (0x1ul << PDMA_CH_INTSTSn_TDIF_Pos)               /*!< PDMA_CH_T::INTSTSn: TDIF Mask          */

#define PDMA_CH_INTSTSn_TOUTIF_Pos       (6)                                               /*!< PDMA_CH_T::INTSTSn: TOUTIF Position    */
#define PDMA_CH_INTSTSn_TOUTIF_Msk       (0x1ul << PDMA_CH_INTSTSn_TOUTIF_Pos)             /*!< PDMA_CH_T::INTSTSn: TOUTIF Mask        */

#define PDMA_CH_INTSTSn_PCNTIF_Pos       (8)                                               /*!< PDMA_CH_T::INTSTSn: PCNTIF Position    */
#define PDMA_CH_INTSTSn_PCNTIF_Msk       (0x1ul << PDMA_CH_INTSTSn_PCNTIF_Pos)             /*!< PDMA_CH_T::INTSTSn: PCNTIF Mask        */

#define PDMA_CH_TOCn_TOC_Pos             (0)                                               /*!< PDMA_CH_T::TOCn: TOC Position          */
#define PDMA_CH_TOCn_TOC_Msk             (0xfffful << PDMA_CH_TOCn_TOC_Pos)                /*!< PDMA_CH_T::TOCn: TOC Mask              */

#define PDMA_CH_TOCn_TPSC_Pos            (16)                                              /*!< PDMA_CH_T::TOCn: TPSC Position         */
#define PDMA_CH_TOCn_TPSC_Msk            (0x7ul << PDMA_CH_TOCn_TPSC_Pos)                  /*!< PDMA_CH_T::TOCn: TPSC Mask             */

/**@}*/ /* PDMA_CH_CONST */


/**
    @addtogroup CRC_CONST CRC Bit Field Definition
    Constant Definitions for CRC Controller
@{ */

#define DMA_CRC_CTL_CRCEN_Pos            (0)                                               /*!< DMA_CRC_T::CTL: CRCEN Position             */
#define DMA_CRC_CTL_CRCEN_Msk            (0x1ul << DMA_CRC_CTL_CRCEN_Pos)                  /*!< DMA_CRC_T::CTL: CRCEN Mask                 */

#define DMA_CRC_CTL_CRCRST_Pos           (1)                                               /*!< DMA_CRC_T::CTL: CRCRST Position            */
#define DMA_CRC_CTL_CRCRST_Msk           (0x1ul << DMA_CRC_CTL_CRCRST_Pos)                 /*!< DMA_CRC_T::CTL: CRCRST Mask                */

#define DMA_CRC_CTL_TRIGEN_Pos           (23)                                              /*!< DMA_CRC_T::CTL: TRIGEN Position            */
#define DMA_CRC_CTL_TRIGEN_Msk           (0x1ul << DMA_CRC_CTL_TRIGEN_Pos)                 /*!< DMA_CRC_T::CTL: TRIGEN Mask                */

#define DMA_CRC_CTL_DATREV_Pos           (24)                                              /*!< DMA_CRC_T::CTL: DATREV Position            */
#define DMA_CRC_CTL_DATREV_Msk           (0x1ul << DMA_CRC_CTL_DATREV_Pos)                 /*!< DMA_CRC_T::CTL: DATREV Mask                */

#define DMA_CRC_CTL_CHKSREV_Pos          (25)                                              /*!< DMA_CRC_T::CTL: CHKSREV Position           */
#define DMA_CRC_CTL_CHKSREV_Msk          (0x1ul << DMA_CRC_CTL_CHKSREV_Pos)                /*!< DMA_CRC_T::CTL: CHKSREV Mask               */

#define DMA_CRC_CTL_DATFMT_Pos           (26)                                              /*!< DMA_CRC_T::CTL: DATFMT Position            */
#define DMA_CRC_CTL_DATFMT_Msk           (0x1ul << DMA_CRC_CTL_DATFMT_Pos)                 /*!< DMA_CRC_T::CTL: DATFMT Mask                */

#define DMA_CRC_CTL_CHKSFMT_Pos          (27)                                              /*!< DMA_CRC_T::CTL: CHKSFMT Position           */
#define DMA_CRC_CTL_CHKSFMT_Msk          (0x1ul << DMA_CRC_CTL_CHKSFMT_Pos)                /*!< DMA_CRC_T::CTL: CHKSFMT Mask               */

#define DMA_CRC_CTL_DATLEN_Pos           (28)                                              /*!< DMA_CRC_T::CTL: DATLEN Position            */
#define DMA_CRC_CTL_DATLEN_Msk           (0x3ul << DMA_CRC_CTL_DATLEN_Pos)                 /*!< DMA_CRC_T::CTL: DATLEN Mask                */

#define DMA_CRC_CTL_CRCMODE_Pos          (30)                                              /*!< DMA_CRC_T::CTL: CRCMODE Position           */
#define DMA_CRC_CTL_CRCMODE_Msk          (0x3ul << DMA_CRC_CTL_CRCMODE_Pos)                /*!< DMA_CRC_T::CTL: CRCMODE Mask               */

#define DMA_CRC_DMASA_SA_Pos             (0)                                               /*!< DMA_CRC_T::DMASA: SA Position              */
#define DMA_CRC_DMASA_SA_Msk             (0xfffffffful << DMA_CRC_DMASA_SA_Pos)            /*!< DMA_CRC_T::DMASA: SA Mask                  */

#define DMA_CRC_DMABCNT_BCNT_Pos         (0)                                               /*!< DMA_CRC_T::DMABCNT: BCNT Position          */
#define DMA_CRC_DMABCNT_BCNT_Msk         (0xfffful << DMA_CRC_DMABCNT_BCNT_Pos)            /*!< DMA_CRC_T::DMABCNT: BCNT Mask              */

#define DMA_CRC_DMACSA_CSA_Pos           (0)                                               /*!< DMA_CRC_T::DMACSA: CSA Position            */
#define DMA_CRC_DMACSA_CSA_Msk           (0xfffffffful << DMA_CRC_DMACSA_CSA_Pos)          /*!< DMA_CRC_T::DMACSA: CSA Mask                */

#define DMA_CRC_DMACBCNT_CBCNT_Pos       (0)                                               /*!< DMA_CRC_T::DMACBCNT: CBCNT Position        */
#define DMA_CRC_DMACBCNT_CBCNT_Msk       (0xfffful << DMA_CRC_DMACBCNT_CBCNT_Pos)          /*!< DMA_CRC_T::DMACBCNT: CBCNT Mask            */

#define DMA_CRC_DMAINTEN_TABTIEN_Pos     (0)                                               /*!< DMA_CRC_T::DMAINTEN: TABTIEN Position      */
#define DMA_CRC_DMAINTEN_TABTIEN_Msk     (0x1ul << DMA_CRC_DMAINTEN_TABTIEN_Pos)           /*!< DMA_CRC_T::DMAINTEN: TABTIEN Mask          */

#define DMA_CRC_DMAINTEN_TDIEN_Pos       (1)                                               /*!< DMA_CRC_T::DMAINTEN: TDIEN Position        */
#define DMA_CRC_DMAINTEN_TDIEN_Msk       (0x1ul << DMA_CRC_DMAINTEN_TDIEN_Pos)             /*!< DMA_CRC_T::DMAINTEN: TDIEN Mask            */

#define DMA_CRC_DMAISTS_TABTIF_Pos       (0)                                               /*!< DMA_CRC_T::DMAISTS: TABTIF Position        */
#define DMA_CRC_DMAISTS_TABTIF_Msk       (0x1ul << DMA_CRC_DMAISTS_TABTIF_Pos)             /*!< DMA_CRC_T::DMAISTS: TABTIF Mask            */

#define DMA_CRC_DMAISTS_TDIF_Pos         (1)                                               /*!< DMA_CRC_T::DMAISTS: TDIF Position          */
#define DMA_CRC_DMAISTS_TDIF_Msk         (0x1ul << DMA_CRC_DMAISTS_TDIF_Pos)               /*!< DMA_CRC_T::DMAISTS: TDIF Mask              */

#define DMA_CRC_DAT_DATA_Pos             (0)                                               /*!< DMA_CRC_T::DAT: DATA Position              */
#define DMA_CRC_DAT_DATA_Msk             (0xfffffffful << DMA_CRC_DAT_DATA_Pos)            /*!< DMA_CRC_T::DAT: DATA Mask                  */

#define DMA_CRC_SEED_SEED_Pos            (0)                                               /*!< DMA_CRC_T::SEED: SEED Position             */
#define DMA_CRC_SEED_SEED_Msk            (0xfffffffful << DMA_CRC_SEED_SEED_Pos)           /*!< DMA_CRC_T::SEED: SEED Mask                 */

#define DMA_CRC_CHECKSUM_CHECKSUM_Pos    (0)                                               /*!< DMA_CRC_T::CHECKSUM: CHECKSUM Position     */
#define DMA_CRC_CHECKSUM_CHECKSUM_Msk    (0xfffffffful << DMA_CRC_CHECKSUM_CHECKSUM_Pos)   /*!< DMA_CRC_T::CHECKSUM: CHECKSUM Mask         */

/**@}*/ /* CRC_CONST */

/**
    @addtogroup PDMA_GCR_CONST PDMA_GCR Bit Field Definition
    Constant Definitions for PDMA_GCR Controller
@{ */

#define DMA_GCR_GCTL_CKEN1_Pos           (9)                                               /*!< DMA_GCR_T::GCTL: CKEN1 Position       */
#define DMA_GCR_GCTL_CKEN1_Msk           (0x1ul << DMA_GCR_GCTL_CKEN1_Pos)                 /*!< DMA_GCR_T::GCTL: CKEN1 Mask           */

#define DMA_GCR_GCTL_CKEN2_Pos           (10)                                              /*!< DMA_GCR_T::GCTL: CKEN2 Position       */
#define DMA_GCR_GCTL_CKEN2_Msk           (0x1ul << DMA_GCR_GCTL_CKEN2_Pos)                 /*!< DMA_GCR_T::GCTL: CKEN2 Mask           */

#define DMA_GCR_GCTL_CKEN3_Pos           (11)                                              /*!< DMA_GCR_T::GCTL: CKEN3 Position       */
#define DMA_GCR_GCTL_CKEN3_Msk           (0x1ul << DMA_GCR_GCTL_CKEN3_Pos)                 /*!< DMA_GCR_T::GCTL: CKEN3 Mask           */

#define DMA_GCR_GCTL_CKEN4_Pos           (12)                                              /*!< DMA_GCR_T::GCTL: CKEN4 Position       */
#define DMA_GCR_GCTL_CKEN4_Msk           (0x1ul << DMA_GCR_GCTL_CKEN4_Pos)                 /*!< DMA_GCR_T::GCTL: CKEN4 Mask           */

#define DMA_GCR_GCTL_CKENCRC_Pos         (24)                                              /*!< DMA_GCR_T::GCTL: CKENCRC Position     */
#define DMA_GCR_GCTL_CKENCRC_Msk         (0x1ul << DMA_GCR_GCTL_CKENCRC_Pos)               /*!< DMA_GCR_T::GCTL: CKENCRC Mask         */

#define DMA_GCR_REQSEL0_REQSRC1_Pos      (8)                                               /*!< DMA_GCR_T::REQSEL0: REQSRC1 Position  */
#define DMA_GCR_REQSEL0_REQSRC1_Msk      (0x1ful << DMA_GCR_REQSEL0_REQSRC1_Pos)           /*!< DMA_GCR_T::REQSEL0: REQSRC1 Mask      */

#define DMA_GCR_REQSEL0_REQSRC2_Pos      (16)                                              /*!< DMA_GCR_T::REQSEL0: REQSRC2 Position  */
#define DMA_GCR_REQSEL0_REQSRC2_Msk      (0x1ful << DMA_GCR_REQSEL0_REQSRC2_Pos)           /*!< DMA_GCR_T::REQSEL0: REQSRC2 Mask      */

#define DMA_GCR_REQSEL0_REQSRC3_Pos      (24)                                              /*!< DMA_GCR_T::REQSEL0: REQSRC3 Position  */
#define DMA_GCR_REQSEL0_REQSRC3_Msk      (0x1ful << DMA_GCR_REQSEL0_REQSRC3_Pos)           /*!< DMA_GCR_T::REQSEL0: REQSRC3 Mask      */

#define DMA_GCR_REQSEL1_REQSRC4_Pos      (0)                                               /*!< DMA_GCR_T::REQSEL1: REQSRC4 Position  */
#define DMA_GCR_REQSEL1_REQSRC4_Msk      (0x1ful << DMA_GCR_REQSEL1_REQSRC4_Pos)           /*!< DMA_GCR_T::REQSEL1: REQSRC4 Mask      */

#define DMA_GCR_GINTSTS_IF1_Pos          (1)                                               /*!< DMA_GCR_T::GINTSTS: IF1 Position      */
#define DMA_GCR_GINTSTS_IF1_Msk          (0x1ul << DMA_GCR_GINTSTS_IF1_Pos)                /*!< DMA_GCR_T::GINTSTS: IF1 Mask          */

#define DMA_GCR_GINTSTS_IF2_Pos          (2)                                               /*!< DMA_GCR_T::GINTSTS: IF2 Position      */
#define DMA_GCR_GINTSTS_IF2_Msk          (0x1ul << DMA_GCR_GINTSTS_IF2_Pos)                /*!< DMA_GCR_T::GINTSTS: IF2 Mask          */

#define DMA_GCR_GINTSTS_IF3_Pos          (3)                                               /*!< DMA_GCR_T::GINTSTS: IF3 Position      */
#define DMA_GCR_GINTSTS_IF3_Msk          (0x1ul << DMA_GCR_GINTSTS_IF3_Pos)                /*!< DMA_GCR_T::GINTSTS: IF3 Mask          */

#define DMA_GCR_GINTSTS_IF4_Pos          (4)                                               /*!< DMA_GCR_T::GINTSTS: IF4 Position      */
#define DMA_GCR_GINTSTS_IF4_Msk          (0x1ul << DMA_GCR_GINTSTS_IF4_Pos)                /*!< DMA_GCR_T::GINTSTS: IF4 Mask          */

#define DMA_GCR_GINTSTS_IFCRC_Pos        (16)                                              /*!< DMA_GCR_T::GINTSTS: IFCRC Position    */
#define DMA_GCR_GINTSTS_IFCRC_Msk        (0x1ul << DMA_GCR_GINTSTS_IFCRC_Pos)              /*!< DMA_GCR_T::GINTSTS: IFCRC Mask        */

/**@}*/ /* PDMA_GCR_CONST */
/**@}*/ /* end of DMA register group */


/*---------------------- Timer Controller -------------------------*/
/**
    @addtogroup TMR Timer Controller(TMR)
    Memory Mapped Structure for TMR Controller
@{ */

typedef struct
{


    /**
     * @var TIMER_T::CTL
     * Offset: 0x00  Timer Control and Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CNTEN     |Timer Counting Enable Bit
     * |        |          |0 = Stops/Suspends counting.
     * |        |          |1 = Starts counting.
     * |        |          |Note1: In stop status, set CNTEN to 1 enables 24-bit counter keeps up counting from the last stop counting value.
     * |        |          |Note2: This bit is auto-cleared by hardware in one-shot mode (OPMODE (TIMERx_CTL[5:4]) = 00) when the timer interrupt flag TIF (TIMERx_INTSTS[0]) is generated.
     * |        |          |Note3: Writing this bit 1 will not take any effect if RSTCNT (TIMERx_CTL[1]) is also set to 1 at the same time.
     * |[1]     |RSTCNT    |Timer Counter Reset Bit
     * |        |          |Setting this bit will reset the internal 8-bit prescale counter, 24-bit up counter value CNT (TIMERx_CNT[23:0]) and also force CNTEN (TIMERx_CTL[0]) to 0.
     * |        |          |0 = No effect.
     * |        |          |1 = Reset internal 8-bit prescale counter, 24-bit up counter value and CNTEN bit.
     * |        |          |Note: This bit will be auto cleared and takes at least 3 TIMERx_CLK clock cycles.
     * |[2]     |WKEN      |Wake-up Function Enable Bit
     * |        |          |If this bit is set to 1, while CNTIF (TIMERx_INTSTS[0]) or CAPIF (TIMERx_INTSTS[1]) is 1, the timer interrupt signal will generate a wake-up trigger event to CPU.
     * |        |          |0 = Wake-up function Disabled if timer interrupt signal generated.
     * |        |          |1 = Wake-up function Enabled if timer interrupt signal generated.
     * |[3]     |ICEDEBUG  |ICE Debug Mode Acknowledge Disable Bit
     * |        |          |0 = ICE debug mode acknowledgement affects TIMER counting.
     * |        |          |Timer counter will be held while CPU is held by ICE.
     * |        |          |1 = ICE debug mode acknowledgement Disabled.
     * |        |          |Timer counter will keep going no matter CPU is held by ICE or not.
     * |[5:4]   |OPMODE    |Timer Counting Mode Selection
     * |        |          |00 = The Timer controller is operated in One-shot mode.
     * |        |          |01 = The Timer controller is operated in Periodic mode.
     * |        |          |10 = The Timer controller is operated in Toggle-output mode.
     * |        |          |11 = The Timer controller is operated in Continuous Counting mode.
     * |[7]     |ACTSTS    |Timer Active Status Bit (Read Only)
     * |        |          |This bit indicates the 24-bit up counter status.
     * |        |          |0 = 24-bit up counter is not active.
     * |        |          |1 = 24-bit up counter is active.
     * |[8]     |TRGADC    |Trigger ADC Enable Bit
     * |        |          |If this bit is set to 1, timer time-out interrupt or capture interrupt can trigger ADC.
     * |        |          |0 = Timer interrupt trigger ADC Disabled.
     * |        |          |1 = Timer interrupt trigger ADC Enabled.
     * |        |          |Note: If TRGSSEL (TIMERx_CTL[11]) is set to 0, the time-out interrupt signal will trigger ADC.
     * |        |          |If TRGSSEL (TIMERx_CTL[11]) is set to 1, the capture interrupt signal will trigger ADC.
     * |[10]    |TRGPDMA   |Timer Trigger PDMA Enable Bit
     * |        |          |If this bit is set to 1, timer time-out interrupt or capture interrupt can trigger PDMA.
     * |        |          |0 = Timer interrupt trigger PWM Disabled.
     * |        |          |1 = Timer interrupt trigger PWM Enabled.
     * |        |          |Note: If TRGSSEL (TIMERx_CTL[11]) is set to 0, the time-out interrupt signal will trigger PWM.
     * |        |          |If TRGSSEL (TIMERx_CTL[11]) is set to 1, the capture interrupt signal will trigger PWM.
     * |[11]    |TRGSSEL   |Trigger Source Selection
     * |        |          |If this bit is set to 1, capture interrupt can trigger ADC, PDMA and PWM
     * |        |          |Otherwise, time-out interrupt can trigger ADC, PDMA and PWM.
     * |        |          |0 = Time-out interrupt is used to trigger ADC, PDMA and PWM.
     * |        |          |1 = Capture interrupt is used to trigger ADC, PDMA and PWM.
     * |[12]    |EXTCNTEN  |Event Counter Mode Enable Bit
     * |        |          |This bit is for external counting pin function enabled.
     * |        |          |0 = Event counter mode Disabled.
     * |        |          |1 = Event counter mode Enabled.
     * |        |          |Note: When timer is used as an event counter, this bit should be set to 1 and HCLK as timer clock source.
     * |[13]    |CNTPHASE  |Timer External Count Phase
     * |        |          |This bit indicates the detection phase of external counting pin Tx (x= 0~3).
     * |        |          |0 = A Falling edge of external counting pin will be counted.
     * |        |          |1 = A Rising edge of external counting pin will be counted.
     * |[14]    |CNTDBEN   |Timer Counter Pin De-bounce Enable Bit
     * |        |          |0 = Tx (x= 0~3) pin de-bounce Disabled.
     * |        |          |1 = Tx (x= 0~3) pin de-bounce Enabled.
     * |        |          |Note: If this bit is set to 1, the edge detection of Tx pin is detected with de-bounce circuit.
     * |[16]    |CAPEN     |Timer External Capture Pin Enable Bit
     * |        |          |This bit enables the Tx_EXT pin.
     * |        |          |0 = Tx_EXT (x= 0~3) pin Disabled.
     * |        |          |1 = Tx_EXT (x= 0~3) pin Enabled.
     * |        |          |Note1: For TIMERx_CTL, if INTRTGEN (TIMERx_CTL[24]) is set to 1, the CAPEN will be forced to low and the TC pin transition is ignored (where x = 0 or 2).
     * |        |          |Note2: For TIMERx+1_CTL, if INTRTGEN (TIMERx_CTL[24]) is set to 1, the CAPEN will be forced to high (where x = 0 or 2).
     * |[17]    |CAPFUNCS  |Capture Function Selection
     * |        |          |0 = External Capture Mode Enabled.
     * |        |          |1 = External Reset Mode Enabled.
     * |        |          |Note1: When CAPFUNCS is 0, transition on Tx_EXT (x= 0~3) pin is using to save the 24-bit timer counter value.
     * |        |          |Note2: When CAPFUNCS is 1, transition on Tx_EXT (x= 0~3) pin is using to reset the 24-bit timer counter value.
     * |        |          |Note3: For TIMERx+1_CTL (x = 0 or 2), if INTRTGEN (TIMERx_CTL[24]) is set to 1, the CAPFUNCS will be forced to low.
     * |[19:18] |CAPEDGE   |Timer External Capture Pin Edge Detection
     * |        |          |For timer counter reset function and free-counting mode of timer capture function, the configurations are:
     * |        |          |00 = A Falling edge on Tx_EXT (x= 0~3) pin will be detected.
     * |        |          |01 = A Rising edge on Tx_EXT (x= 0~3) pin will be detected.
     * |        |          |10 = Either Rising or Falling edge on Tx_EXT (x= 0~3) pin will be detected.
     * |        |          |11 = Either Rising or Falling edge on Tx_EXT (x= 0~3) pin will be detected.
     * |        |          |For trigger-counting mode of timer capture function, the configurations are:
     * |        |          |00 = 1st falling edge on TC pin triggers 24-bit timer to start counting, while 2nd falling edge triggers 24-bit timer to stop counting.
     * |        |          |01 = 1st rising edge on TC pin triggers 24-bit timer to start counting, while 2nd rising edge triggers 24-bit timer to stop counting.
     * |        |          |10 = Falling edge on TC pin triggers 24-bit timer to start counting, while rising edge triggers 24-bit timer to stop counting.
     * |        |          |11 = Rising edge on TC pin triggers 24-bit timer to start counting, while falling edge triggers 24-bit timer to stop counting.
     * |        |          |Note: For TIMERx+1_CTL, if INTRTGEN (TIMERx_CTL[24]) is set to 1, the CAPEDGE will be forced to 11 (where x = 0 or 2).
     * |[20]    |CAPCNTMD  |Timer Capture Counting Mode Selection
     * |        |          |This bit indicates the behavior of 24-bit up-counting timer while CAPEN (TIMERx_CTL[16]) is set to high.
     * |        |          |If this bit is 0, the free-counting mode, the behavior of 24-bit up-counting timer is defined by OPMODE (TIMERx_CTL[5:4]) field
     * |        |          |When CAPEN (TIMERx_CTL[16]) is set, CAPFUNCS (TIMERx_CTL[17]) is 0, and the transition of TC pin matches the CAPEDGE (TIMERx_CTL[19:18]) setting, the value of 24-bit up-counting timer will be saved into register TIMERx_CAP.
     * |        |          |If this bit is 1, Trigger-counting mode, 24-bit up-counting timer will be not counting and keep its value at 0
     * |        |          |When CAPEN (TIMERx_CTL[16]) is set, CAPFUNCS (TIMERx_CTL[17]) is 0, and once the transition of external pin matches the 1st transition of CAPEDGE (TIMERx_CTL[19:18]) setting, the 24-bit up-counting timer will start counting
     * |        |          |And then if the transition of external pin matches the 2nd transition of CAPEDGE (TIMERx_CTL[19:18]) setting, the 24-bit up-counting timer will stop counting
     * |        |          |And its value will be saved into register TIMERx_CAP.
     * |        |          |0 = Capture with free-counting timer mode.
     * |        |          |1 = Capture with trigger-counting timer mode.
     * |        |          |Note: For TIMERx+1_CTL, if INTRTGEN (TIMERx_CTL[24]) is set, the CAPCNTMD will be forced to high, the capture with Trigger-counting Timer mode (where x = 0 or 2).
     * |[22]    |CAPDBEN   |Timer External Capture Pin De-bounce Enable Bit
     * |        |          |0 = Tx_EXT (x= 0~3) pin de-bounce Disabled.
     * |        |          |1 = Tx_EXT (x= 0~3) pin de-bounce Enabled.
     * |        |          |Note1: If this bit is enabled, the edge detection of Tx_EXT pin is detected with de-bounce circuit.
     * |        |          |Note2: For Timer 1 and 3, when INTRTGEN (TIMERx_CTL[24]) is high, the capture signal is from internal of chip and the de-bounce circuit would not take effect no matter this bit is high or low.
     * |[23]    |CMPCTL    |Timer Compared Mode Selection
     * |        |          |0 = The behavior selection in one-shot, periodic or Toggle-output mode Disabled.
     * |        |          |When user updates CMPDAT (TIMERx_CMP) while timer is running in One-shot, Periodic or Toggle-output mode, CNT (TIMERx_CNT) will be reset to default value.
     * |        |          |1 = The behavior selection in one-shot, periodic or Toggle-output mode Enabled.
     * |        |          |When user updates CMPDAT (TIMERx_CMP) while timer is running in One-shot, Periodic or Toggle-output mode, the limitations as bellows list,
     * |        |          |If updated CMPDAT (TIMERx_CMP) value > CNT (TIMERx_CNT), CMPDAT (TIMERx_CMP) will be updated and CNT (TIMERx_CNT) keep running continually.
     * |        |          |If updated CMPDAT (TIMERx_CMP) value = CNT (TIMERx_CNT), timer time-out interrupt will be asserted immediately.
     * |        |          |If updated CMPDAT (TIMERx_CMP) value < CNT (TIMERx_CNT), CNT (TIMERx_CNT) will be reset to default value
     * |        |          |At the same time, prescale counter reloaded.
     * |[24]    |INTRTGEN  |Inter-timer Trigger Function Enable Bit
     * |        |          |If INTRTGEN is set to 1 TIMERx (x = 0 or 2), TIMERx and Timerx=1 are operating at inter-timer trigger mode.
     * |        |          |When Inter-timer Trigger function is enabled, TIMERx is operating at event counting mode to count the input event from Tx pin and TIMERx+1 is operating at external capture trigger-counting mode.
     * |        |          |0 = Inter-timer trigger function Disabled.
     * |        |          |1 = Inter-timer trigger function Enabled.
     * |        |          |Note: In TIMERx+1_CTL, this bit is always 0.
     * |[25]    |INTRTGMD  |Inter-timer Trigger Mode Selection
     * |        |          |This bit controls the TIMERx (x = 0 or 2) operating behavior when INTRTGEN (TIMERx_CTL[24]) is set to 1.
     * |        |          |If INTRTGMD is set to 0 and INTRTGEN (TIMERx_CTL[24]) is set to 1, the TIMERx is operating at event counting mode to count the all input events from Tx pin.
     * |        |          |If INTRTGMD and INTRTGEN (TIMERx_CTL[24]) are both set to 1, TIMERx is operating at event counting mode and the number of first incoming events (defined by EVNTDPCNT (TIMERx_ECTL[31:24])) are ignored.
     * |        |          |0 = TIMERx count the all input events from Tx pin.
     * |        |          |1 = TIMERx ignored the number of first incoming events based on EVNTDPCNT (TIMERx_ECTL[31:24]).
     * |        |          |Note: In TIMERx+1_CTL, this bit is always 0.
     * |[28]    |TRGPWM    |Trigger PWM Enable Bit
     * |        |          |If this bit is set to 1, timer time-out interrupt or capture interrupt can trigger PWM.
     * |        |          |0 = Timer interrupt trigger PWM Disabled.
     * |        |          |1 = Timer interrupt trigger PWM Enabled.
     * |        |          |Note: If TRGSSEL (TIMERx_CTL[11]) = 0, time-out interrupt signal will trigger PWM.
     * |        |          |If TRGSSEL (TIMERx_CTL[11]) = 1, capture interrupt signal will trigger PWM.
     * @var TIMER_T::PRECNT
     * Offset: 0x04  Timer Pre-Scale Counter Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |PSC       |Prescale Counter
     * |        |          |Timer input clock or event source is divided by (PSC+1) before it is fed to the timer up counter
     * |        |          |If this field is 0 (PSC = 0), then there is no scaling.
     * |        |          |Note: If the PSC value is changed, CNT (TIMERx_CNT) is reset to 0 and prescale counter is reloaded.
     * @var TIMER_T::CMP
     * Offset: 0x08  Timer Compare Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[23:0]  |CMPDAT    |Timer Compared Value
     * |        |          |CMPDAT is a 24-bit compared value register
     * |        |          |When the internal 24-bit up counter value is equal to CMPDAT value, the CNTIF (TIMERx_INTSTS[0] Timer Interrupt Flag) will be set to 1.
     * |        |          |Time-out period = (Period of timer clock input) * (8-bit PSC + 1) * (24-bit CMPDAT).
     * |        |          |Note1: Never write 0x0 or 0x1 in CMPDAT, or the core will run into unknown state.
     * |        |          |Note2: When the timer is operating in Continuous Counting mode (OPMODE (TIMERx_CTL[5:4] is 11), the 24-bit up counter will keep counting continuously even if user writes a new value into CMPDAT field.
     * |        |          |Note3: When the timer is not operating in Continuous Counting mode (OPMODE (TIMERx_CTL[5:4] is not 11), the 24-bit up counter will restart counting from 0 and use the newest CMPDAT value as the timer compared value when user writes a new value into the CMPDAT field
     * |        |          |In addition, the prescale counter will be reloaded.
     * @var TIMER_T::INTEN
     * Offset: 0x0C  Timer Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CNTIEN    |Timer Interrupt Enable Bit
     * |        |          |0 = Timer Interrupt Disabled.
     * |        |          |1 = Timer Interrupt Enabled.
     * |        |          |Note: If this bit is enabled, when the timer interrupt flag CNTIF (TIMERx_INTSTS[0]) is set to 1, the timer interrupt signal is generated and informed to CPU.
     * |[1]     |CAPIEN    |Timer External Capture Interrupt Enable Bit
     * |        |          |0 = Tx_EXT (x= 0~3) pin detection Interrupt Disabled.
     * |        |          |1 = Tx_EXT (x= 0~3) pin detection Interrupt Enabled.
     * |        |          |Note: CAPIEN is used to enable timer external interrupt
     * |        |          |If CAPIEN is enabled, the timer will rise an interrupt when CAPIF (TIMERx_INTSTS[1]) is 1.
     * @var TIMER_T::INTSTS
     * Offset: 0x10  Timer Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CNTIF     |Timer Interrupt Status
     * |        |          |This bit indicates the interrupt flag status of Timer while 24-bit timer up counter CNT (TIMERx_CNT[23:0]) value reaches to CMPDAT (TIMERx_CMP[23:0]) value.
     * |        |          |0 = No effect.
     * |        |          |1 = CNT (TIMERx_CNT[23:0]) value matches the CMPDAT (TIMERx_CMP[23:0]) value.
     * |        |          |Note: This bit is cleared by writing 1 to it.
     * |[1]     |CAPIF     |Timer External Capture Interrupt Flag
     * |        |          |This bit indicates the timer external capture interrupt flag status.
     * |        |          |0 = Tx_EXT (x= 0~3) pin interrupt did not occur.
     * |        |          |1 = Tx_EXT (x= 0~3) pin interrupt occurred.
     * |        |          |Note1: This bit is cleared by writing 1 to it.
     * |        |          |Note2: When CAPEN (TIMERx_CTL[16]) bit is set, CAPFUNCS (TIMERx_CTL[17]) bit is 0, and a transition on Tx_EXT (x= 0~3) pin matched the CAPEDGE (TIMERx_CTL[19:18]) setting, this bit will set to 1 by hardware.
     * |        |          |Note3: If a new incoming capture event detected before CPU clearing the CAPIF status, the Timer will keep register TIMERx_CAP unchanged and drop the new capture value.
     * |[4]     |TWKF      |Timer Wake-up Flag
     * |        |          |This bit indicates the interrupt wake-up flag status of timer.
     * |        |          |0 = Timer does not cause CPU wake-up.
     * |        |          |1 = CPU wake-up from Idle or Power-down mode if timer time-out interrupt signal generated.
     * |        |          |Note: This bit is cleared by writing 1 to it.
     * |[5]     |CAPDATOF  |Capture Data Overflow Flag
     * |        |          |This status is to indicate there is a new incoming capture event detected before CPU clearing the CAPIF (TIMERx_INTSTS[1]) status.
     * |        |          |If the above condition occurred, the Timer will keep register TIMERx_CAP unchanged and drop the new capture value.
     * |        |          |0 = New incoming capture event didnu2019t detect before CPU clearing CAPIF (TIMERx_INTSTS[1]) status.
     * |        |          |1 = New incoming capture event detected before CPU clearing CAPIF (TIMERx_INTSTS[1]) status.
     * |        |          |Note: This bit is cleared by writing 1 to it.
     * |[6]     |CAPFEDF   |Capture Falling Edge Detected Flag
     * |        |          |This flag indicates the edge detected on Tx_EXT pin is rising edge or falling edge.
     * |        |          |0 = Rising edge detected on Tx_EXT pin.
     * |        |          |1 = Falling edge detected on Tx_EXT pin.
     * |        |          |Note1: The timer updates this flag when it updates the Timer Capture Data (TMR_CAP[23:0]) value.
     * |        |          |Note2: When a new incoming capture event detected before CPU clearing the CAPIF (TIMERx_INTSTS[1]) status, Timer will keep this bit unchanged.
     * @var TIMER_T::CNT
     * Offset: 0x14  Timer Counter Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[23:0]  |CNT       |Timer Counter Data (Read)
     * |        |          |This field can reflect the internal 24-bit timer counter value or external event input counter value from Tx (x=0~3) pin.
     * |        |          |Counter Reset (Write)
     * |        |          |User can write any value to TIEMRx_CNT to reset internal 24-bit timer up-counter and 8-bit pre-scale counter
     * |        |          |This reset operation wouldn't affect any other timer control registers and circuit
     * |        |          |After reset completed, the 24-bit timer up-counter and 8-bit pre-scale counter restart the counting based on the TIMERx_CTL register setting.
     * |[31]    |RSTACT    |Reset Active
     * |        |          |This bit indicates if the counter reset operation active.
     * |        |          |When user write this register, timer starts to reset its internal 24-bit timer up-counter and 8-bit pre-scale counter to 0
     * |        |          |At the same time, timer set this flag to 1 to indicate the counter reset operation is in progress
     * |        |          |Once the counter reset operation done, timer clear this bit to 0 automatically.
     * |        |          |0 = Reset operation is done.
     * |        |          |1 = Reset operation triggered by writing TIMERx_CNT is in progress.
     * |        |          |Note: This bit is read only. Write operation wouldn't take any effect.
     * @var TIMER_T::CAP
     * Offset: 0x18  Timer Capture Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[23:0]  |CAPDAT    |Timer Capture Data Register
     * |        |          |When CAPEN (TIMERx_CTL[16]) bit is set, CAPFUNCS (TIMERx_CTL[17]) bit is 0, CAPCNTMD (TIMERx_CTL[20]) bit is 0, and the transition on Tx_EXT pin matched the CAPEDGE (TIMERx_CTL[19:18]) setting, CAPIF (TIMERx_INTSTS[1]) will set to 1 and the current timer counter value CNT (TIMERx_CNT[23:0]) will be auto-loaded into this CAPDAT field.
     * |        |          |When CAPEN (TIMERx_CTL[16]) bit is set, CAPFUNCS (TIMERx_CTL[17]) bit is 0, CAPCNTMD (TIMERx_CTL[20]) bit is 1, and the transition on Tx_EXT pin matched the 2nd transition of CAPEDGE (TIMERx_CTL[19:18]) setting, CAPIF (TIMERx_INTSTS[1]) will set to 1 and the current timer counter value CNT (TIMERx_CNT[23:0]) will be auto-loaded into this CAPDAT field.
     * |        |          |Note: When edge transition defined by CAPEDGE (TIMERx_CTL[19:18]) is detected on Tx_EXT (x = 0 ~ 3) before CPU clears the CAPIF (TIMERxISR[1]) status, the timer keeps this value unchanged and CAPDATOF (TIMERx_INTSTS[5]) is set to 1.
     * @var TIMER_T::ECTL
     * Offset: 0x20  Timer Extended Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:24] |EVNTDPCNT |Event Drop Count
     * |        |          |This field indicates timer how many events dropped after inter-timer trigger function enable.
     * |        |          |For example, if user configured EVNTDPCNT to 7, timer would drop 7 first incoming events and starts the inter-timer trigger operation when it get 8th event.
     * |        |          |Note: ECNTDPCNT only takes effect when INTRTGEN (TIMERx_CTL[24]) INTRTGMD (TIMERx_CTL[25]) are both set to 1.
     */
    __IO uint32_t CTL;                   /*!< [0x0000] Timer Control and Status Register                              */
    __IO uint32_t PRECNT;                /*!< [0x0004] Timer Pre-Scale Counter Register                               */
    __IO uint32_t CMP;                   /*!< [0x0008] Timer Compare Register                                         */
    __IO uint32_t INTEN;                 /*!< [0x000c] Timer Interrupt Enable Register                                */
    __IO uint32_t INTSTS;                /*!< [0x0010] Timer Interrupt Status Register                                */
    __IO uint32_t CNT;                   /*!< [0x0014] Timer Counter Data Register                                    */
    __I  uint32_t CAP;                   /*!< [0x0018] Timer Capture Data Register                                    */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE0[1];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t ECTL;                  /*!< [0x0020] Timer Extended Control Register                                */

} TIMER_T;

/**
    @addtogroup TMR_CONST TMR Bit Field Definition
    Constant Definitions for TMR Controller
@{ */
#define TIMER_CTL_CNTEN_Pos                (0)                                               /*!< TIMER_T::CTL CNTEN Position              */
#define TIMER_CTL_CNTEN_Msk                (0x1ul << TIMER_CTL_CNTEN_Pos)                    /*!< TIMER_T::CTL CNTEN Mask                  */

#define TIMER_CTL_RSTCNT_Pos               (1)                                               /*!< TIMER_T::CTL RSTCNT Position             */
#define TIMER_CTL_RSTCNT_Msk               (0x1ul << TIMER_CTL_RSTCNT_Pos)                   /*!< TIMER_T::CTL RSTCNT Mask                 */

#define TIMER_CTL_WKEN_Pos                 (2)                                               /*!< TIMER_T::CTL WKEN Position               */
#define TIMER_CTL_WKEN_Msk                 (0x1ul << TIMER_CTL_WKEN_Pos)                     /*!< TIMER_T::CTL WKEN Mask                   */

#define TIMER_CTL_ICEDEBUG_Pos             (3)                                               /*!< TIMER_T::CTL ICEDEBUG Position           */
#define TIMER_CTL_ICEDEBUG_Msk             (0x1ul << TIMER_CTL_ICEDEBUG_Pos)                 /*!< TIMER_T::CTL ICEDEBUG Mask               */

#define TIMER_CTL_OPMODE_Pos               (4)                                               /*!< TIMER_T::CTL OPMODE Position             */
#define TIMER_CTL_OPMODE_Msk               (0x3ul << TIMER_CTL_OPMODE_Pos)                   /*!< TIMER_T::CTL OPMODE Mask                 */

#define TIMER_CTL_ACTSTS_Pos               (7)                                               /*!< TIMER_T::CTL ACTSTS Position             */
#define TIMER_CTL_ACTSTS_Msk               (0x1ul << TIMER_CTL_ACTSTS_Pos)                   /*!< TIMER_T::CTL ACTSTS Mask                 */

#define TIMER_CTL_TRGADC_Pos               (8)                                               /*!< TIMER_T::CTL TRGADC Position             */
#define TIMER_CTL_TRGADC_Msk               (0x1ul << TIMER_CTL_TRGADC_Pos)                   /*!< TIMER_T::CTL TRGADC Mask                 */

#define TIMER_CTL_TRGPDMA_Pos              (10)                                              /*!< TIMER_T::CTL TRGPDMA Position            */
#define TIMER_CTL_TRGPDMA_Msk              (0x1ul << TIMER_CTL_TRGPDMA_Pos)                  /*!< TIMER_T::CTL TRGPDMA Mask                */

#define TIMER_CTL_TRGSSEL_Pos              (11)                                              /*!< TIMER_T::CTL TRGSSEL Position            */
#define TIMER_CTL_TRGSSEL_Msk              (0x1ul << TIMER_CTL_TRGSSEL_Pos)                  /*!< TIMER_T::CTL TRGSSEL Mask                */

#define TIMER_CTL_EXTCNTEN_Pos             (12)                                              /*!< TIMER_T::CTL EXTCNTEN Position           */
#define TIMER_CTL_EXTCNTEN_Msk             (0x1ul << TIMER_CTL_EXTCNTEN_Pos)                 /*!< TIMER_T::CTL EXTCNTEN Mask               */

#define TIMER_CTL_CNTPHASE_Pos             (13)                                              /*!< TIMER_T::CTL CNTPHASE Position           */
#define TIMER_CTL_CNTPHASE_Msk             (0x1ul << TIMER_CTL_CNTPHASE_Pos)                 /*!< TIMER_T::CTL CNTPHASE Mask               */

#define TIMER_CTL_CNTDBEN_Pos              (14)                                              /*!< TIMER_T::CTL CNTDBEN Position            */
#define TIMER_CTL_CNTDBEN_Msk              (0x1ul << TIMER_CTL_CNTDBEN_Pos)                  /*!< TIMER_T::CTL CNTDBEN Mask                */

#define TIMER_CTL_CAPEN_Pos                (16)                                              /*!< TIMER_T::CTL CAPEN Position              */
#define TIMER_CTL_CAPEN_Msk                (0x1ul << TIMER_CTL_CAPEN_Pos)                    /*!< TIMER_T::CTL CAPEN Mask                  */

#define TIMER_CTL_CAPFUNCS_Pos             (17)                                              /*!< TIMER_T::CTL CAPFUNCS Position           */
#define TIMER_CTL_CAPFUNCS_Msk             (0x1ul << TIMER_CTL_CAPFUNCS_Pos)                 /*!< TIMER_T::CTL CAPFUNCS Mask               */

#define TIMER_CTL_CAPEDGE_Pos              (18)                                              /*!< TIMER_T::CTL CAPEDGE Position            */
#define TIMER_CTL_CAPEDGE_Msk              (0x3ul << TIMER_CTL_CAPEDGE_Pos)                  /*!< TIMER_T::CTL CAPEDGE Mask                */

#define TIMER_CTL_CAPCNTMD_Pos             (20)                                              /*!< TIMER_T::CTL CAPCNTMD Position           */
#define TIMER_CTL_CAPCNTMD_Msk             (0x1ul << TIMER_CTL_CAPCNTMD_Pos)                 /*!< TIMER_T::CTL CAPCNTMD Mask               */

#define TIMER_CTL_CAPDBEN_Pos              (22)                                              /*!< TIMER_T::CTL CAPDBEN Position            */
#define TIMER_CTL_CAPDBEN_Msk              (0x1ul << TIMER_CTL_CAPDBEN_Pos)                  /*!< TIMER_T::CTL CAPDBEN Mask                */

#define TIMER_CTL_CMPCTL_Pos               (23)                                              /*!< TIMER_T::CTL CMPCTL Position             */
#define TIMER_CTL_CMPCTL_Msk               (0x1ul << TIMER_CTL_CMPCTL_Pos)                   /*!< TIMER_T::CTL CMPCTL Mask                 */

#define TIMER_CTL_INTRTGEN_Pos             (24)                                              /*!< TIMER_T::CTL INTRTGEN Position           */
#define TIMER_CTL_INTRTGEN_Msk             (0x1ul << TIMER_CTL_INTRTGEN_Pos)                 /*!< TIMER_T::CTL INTRTGEN Mask               */

#define TIMER_CTL_INTRTGMD_Pos             (25)                                              /*!< TIMER_T::CTL INTRTGMD Position           */
#define TIMER_CTL_INTRTGMD_Msk             (0x1ul << TIMER_CTL_INTRTGMD_Pos)                 /*!< TIMER_T::CTL INTRTGMD Mask               */

#define TIMER_CTL_TRGPWM_Pos               (28)                                              /*!< TIMER_T::CTL TRGPWM Position             */
#define TIMER_CTL_TRGPWM_Msk               (0x1ul << TIMER_CTL_TRGPWM_Pos)                   /*!< TIMER_T::CTL TRGPWM Mask                 */

#define TIMER_PRECNT_PSC_Pos               (0)                                               /*!< TIMER_T::PRECNT: PSC Position            */
#define TIMER_PRECNT_PSC_Msk               (0xfful << TIMER_PRECNT_PSC_Pos)                  /*!< TIMER_T::PRECNT: PSC Mask                */

#define TIMER_CMP_CMPDAT_Pos               (0)                                               /*!< TIMER_T::CMP: CMPDAT Position            */
#define TIMER_CMP_CMPDAT_Msk               (0xfffffful << TIMER_CMP_CMPDAT_P)                /*!< TIMER_T::CMP: CMPDAT Mask                */

#define TIMER_INTEN_CNTIEN_Pos             (0)                                               /*!< TIMER_T::INTEN: CNTIEN Position          */
#define TIMER_INTEN_CNTIEN_Msk             (0x1ul << TIMER_INTEN_CNTIEN_Pos)                 /*!< TIMER_T::INTEN: CNTIEN Mask              */

#define TIMER_INTEN_CAPIEN_Pos             (1)                                               /*!< TIMER_T::INTEN: CAPIEN Position          */
#define TIMER_INTEN_CAPIEN_Msk             (0x1ul << TIMER_INTEN_CAPIEN_Pos)                 /*!< TIMER_T::INTEN: CAPIEN Mask              */

#define TIMER_INTSTS_CNTIF_Pos             (0)                                               /*!< TIMER_T::INTSTS: CNTIF Position          */
#define TIMER_INTSTS_CNTIF_Msk             (0x1ul << TIMER_INTSTS_CNTIF_Pos)                 /*!< TIMER_T::INTSTS: CNTIF Mask              */

#define TIMER_INTSTS_CAPIF_Pos             (1)                                               /*!< TIMER_T::INTSTS: CAPIF Position          */
#define TIMER_INTSTS_CAPIF_Msk             (0x1ul << TIMER_INTSTS_CAPIF_Pos)                 /*!< TIMER_T::INTSTS: CAPIF Mask              */

#define TIMER_INTSTS_TWKF_Pos              (4)                                               /*!< TIMER_T::INTSTS: TWKF Position           */
#define TIMER_INTSTS_TWKF_Msk              (0x1ul << TIMER_INTSTS_TWKF_Pos)                  /*!< TIMER_T::INTSTS: TWKF Mask               */

#define TIMER_INTSTS_CAPDATOF_Pos          (5)                                               /*!< TIMER_T::INTSTS: CAPDATOF Position       */
#define TIMER_INTSTS_CAPDATOF_Msk          (0x1ul << TIMER_INTSTS_CAPDATOF_Pos)              /*!< TIMER_T::INTSTS: CAPDATOF Mask           */

#define TIMER_INTSTS_CAPFEDF_Pos           (6)                                               /*!< TIMER_T::INTSTS: CAPFEDF Position        */
#define TIMER_INTSTS_CAPFEDF_Msk           (0x1ul << TIMER_INTSTS_CAPFEDF_Pos)               /*!< TIMER_T::INTSTS: CAPFEDF Mask            */

#define TIMER_CNT_CNT_Pos                  (0)                                               /*!< TIMER_T::CNT: CNT Position               */
#define TIMER_CNT_CNT_Msk                  (0xfffffful << TIMER_CNT_CNT_Pos)                 /*!< TIMER_T::CNT: CNT Mask                   */

#define TIMER_CNT_RSTACT_Pos               (31)                                              /*!< TIMER_T::CNT: RSTACT Position            */
#define TIMER_CNT_RSTACT_Msk               (0x1ul << TIMER_CNT_RSTACT_Pos)                   /*!< TIMER_T::CNT: RSTACT Mask                */

#define TIMER_CAP_CAPDAT_Pos               (0)                                               /*!< TIMER_T::CAP: CAPDAT Position            */
#define TIMER_CAP_CAPDAT_Msk               (0xfffffful << TIMER_CAP_CAPDAT_Pos)              /*!< TIMER_T::CAP: CAPDAT Mask                */

#define TIMER_ECTL_EVNTDPCNT_Pos           (24)                                              /*!< TIMER_T::ECTL: EVNTDPCNT Position        */
#define TIMER_ECTL_EVNTDPCNT_Msk           (0xfful << TIMER_ECTL_EVNTDPCNT_Pos)              /*!< TIMER_T::ECTL: EVNTDPCNT Mask            */

/**@}*/ /* TMR_CONST */
/**@}*/ /* end of TMR register group */


/*---------------------- Pulse Width Modulation Controller -------------------------*/
/**
    @addtogroup PWM Pulse Width Modulation Controller(PWM)
    Memory Mapped Structure for PWM Controller
@{ */

typedef struct
{


    /**
     * @var PWM_T::CTL0
     * Offset: 0x00  PWM0 Control Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |CTRLDn    |Center Re-load
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |In up-down counter type, PERIOD will load to PBUF at the end point of each period
     * |        |          |CMPDAT will load to CMPBUF at the center point of a period
     * |[21:16] |IMMLDENn  |Immediately Load Enable Bits
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |0 = PERIOD will load to PBUF at the end point of each period
     * |        |          |CMPDAT will load to CMPBUF at the end point or center point of each period by setting CTRLD bit.
     * |        |          |1 = PERIOD/CMPDAT will load to PBUF and CMPBUF immediately when software update PERIOD/CMPDAT.
     * |        |          |Note: If IMMLDENn is enabled, WINLDENn and CTRLDn will be invalid.
     * |[30]    |DBGHALT   |ICE Debug Mode Counter Halt (Write Protect)
     * |        |          |If counter halt is enabled, PWM0 all counters will keep current value until exit ICE debug mode.
     * |        |          |0 = ICE debug mode counter halt Disabled.
     * |        |          |1 = ICE debug mode counter halt Enabled.
     * |        |          |Note: This bit is write protected. Refer to SYS_REGLCTL register.
     * |[31]    |DBGTRIOFF |ICE Debug Mode Acknowledge Disable Bit (Write Protect)
     * |        |          |0 = ICE debug mode acknowledgement effects PWM0 output.
     * |        |          |PWM0 pin will be forced as tri-state while ICE debug mode acknowledged.
     * |        |          |1 = ICE debug mode acknowledgement disabled.
     * |        |          |PWM0 pin will keep output no matter ICE debug mode acknowledged or not.
     * |        |          |Note: This bit is write protected. Refer to SYS_REGLCTL register.
     * @var PWM_T::CTL1
     * Offset: 0x04  PWM0 Control Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |CNTTYPE0  |PWM0 Counter Behavior Type 0
     * |        |          |Each bit n controls corresponding PWM0 channel n.
     * |        |          |00 = Up counter type (supports in capture mode).
     * |        |          |01 = Down count type (supports in capture mode).
     * |        |          |10 = Up-down counter type.
     * |        |          |11 = Reserved.
     * |[5:4]   |CNTTYPE2  |PWM0 Counter Behavior Type 2
     * |        |          |Each bit n controls corresponding PWM0 channel n.
     * |        |          |00 = Up counter type (supports in capture mode).
     * |        |          |01 = Down count type (supports in capture mode).
     * |        |          |10 = Up-down counter type.
     * |        |          |11 = Reserved.
     * |[9:8]   |CNTTYPE4  |PWM0 Counter Behavior Type 4
     * |        |          |Each bit n controls corresponding PWM0 channel n.
     * |        |          |00 = Up counter type (supports in capture mode).
     * |        |          |01 = Down count type (supports in capture mode).
     * |        |          |10 = Up-down counter type.
     * |        |          |11 = Reserved.
     * |[26:24] |PWMMODEn  |PWM0 Mode
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |0 = PWM0 independent mode.
     * |        |          |1 = PWM0 complementary mode.
     * |        |          |Note: When operating in group function, these bits must all set to the same mode.
     * @var PWM_T::CLKSRC
     * Offset: 0x10  PWM0 Clock Source Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |ECLKSRC0  |PWM0_CH01 External Clock Source Select
     * |        |          |000 = PWMx_CLK, x denotes 0 or 1.
     * |        |          |001 = TIMER0 overflow.
     * |        |          |010 = TIMER1 overflow.
     * |        |          |011 = TIMER2 overflow.
     * |        |          |100 = TIMER3 overflow.
     * |        |          |Others = Reserved.
     * |[10:8]  |ECLKSRC2  |PWM0_CH23 External Clock Source Select
     * |        |          |000 = PWMx_CLK, x denotes 0 or 1.
     * |        |          |001 = TIMER0 overflow.
     * |        |          |010 = TIMER1 overflow.
     * |        |          |011 = TIMER2 overflow.
     * |        |          |100 = TIMER3 overflow.
     * |        |          |Others = Reserved.
     * |[18:16] |ECLKSRC4  |PWM0_CH45 External Clock Source Select
     * |        |          |000 = PWMx_CLK, x denotes 0 or 1.
     * |        |          |001 = TIMER0 overflow.
     * |        |          |010 = TIMER1 overflow.
     * |        |          |011 = TIMER2 overflow.
     * |        |          |100 = TIMER3 overflow.
     * |        |          |Others = Reserved.
     * @var PWM_T::CLKPSC0_1
     * Offset: 0x14  PWM0 Clock Pre-Scale Register 0_1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |CLKPSC    |PWM0 Counter Clock Pre-scale
     * |        |          |The clock of PWM0 counter is decided by clock prescaler
     * |        |          |Each PWM0 pair share one PWM0 counter clock prescaler
     * |        |          |The clock of PWM0 counter is divided by (CLKPSC+ 1)
     * @var PWM_T::CLKPSC2_3
     * Offset: 0x18  PWM0 Clock Pre-Scale Register 2_3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |CLKPSC    |PWM0 Counter Clock Pre-scale
     * |        |          |The clock of PWM0 counter is decided by clock prescaler
     * |        |          |Each PWM0 pair share one PWM0 counter clock prescaler
     * |        |          |The clock of PWM0 counter is divided by (CLKPSC+ 1)
     * @var PWM_T::CLKPSC4_5
     * Offset: 0x1C  PWM0 Clock Pre-Scale Register 4_5
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |CLKPSC    |PWM0 Counter Clock Pre-scale
     * |        |          |The clock of PWM0 counter is decided by clock prescaler
     * |        |          |Each PWM0 pair share one PWM0 counter clock prescaler
     * |        |          |The clock of PWM0 counter is divided by (CLKPSC+ 1)
     * @var PWM_T::CNTEN
     * Offset: 0x20  PWM0 Counter Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CNTEN0    |PWM0 Counter Enable Bit 0
     * |        |          |0 = PWM0 Counter0_1 and clock prescaler0 Stop Running.
     * |        |          |1 = PWM0 Counter0_1 and clock prescaler0 Start Running.
     * |[2]     |CNTEN2    |PWM0 Counter Enable Bit 2
     * |        |          |0 = PWM0 Counter2_3 and clock prescaler2 Stop Running.
     * |        |          |1 = PWM0 Counter2_3 and clock prescaler2 Start Running.
     * |[4]     |CNTEN4    |PWM0 Counter Enable Bit 4
     * |        |          |0 = PWM0 Counter4_5 and clock prescaler4 Stop Running.
     * |        |          |1 = PWM0 Counter4_5 and clock prescaler4 Start Running.
     * @var PWM_T::CNTCLR
     * Offset: 0x24  PWM0 Clear Counter Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CNTCLR0   |Clear PWM0 Counter Control Bit 0
     * |        |          |It is automatically cleared by hardware.
     * |        |          |0 = No effect.
     * |        |          |1 = Clear 16-bit PWM0 counter to 0000H.
     * |[2]     |CNTCLR2   |Clear PWM0 Counter Control Bit 2
     * |        |          |It is automatically cleared by hardware.
     * |        |          |0 = No effect.
     * |        |          |1 = Clear 16-bit PWM0 counter to 0000H.
     * |[4]     |CNTCLR4   |Clear PWM0 Counter Control Bit 4
     * |        |          |It is automatically cleared by hardware.
     * |        |          |0 = No effect.
     * |        |          |1 = Clear 16-bit PWM0 counter to 0000H.
     * @var PWM_T::PERIOD
     * Offset: 0x30  PWM0 Period Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |PERIOD    |PWM0 Period Register
     * |        |          |Up-Count mode: In this mode, PWM0 counter counts from 0 to PERIOD, and restarts from 0.
     * |        |          |Down-Count mode: In this mode, PWM0 counter counts from PERIOD to 0, and restarts from PERIOD.
     * |        |          |PWM0 period time = (PERIOD+1) * PWM0_CLK period.
     * |        |          |Up-Down-Count mode: In this mode, PWM0 counter counts from 0 to PERIOD, then decrements to 0 and repeats again.
     * |        |          |PWM0 period time = 2 * PERIOD * PWM0_CLK period.
     * @var PWM_T::CMPDAT
     * Offset: 0x50  PWM0 Comparator Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CMPDAT    |PWM0 Comparator Register
     * |        |          |CMPDAT use to compare with CNTR to generate PWM0 waveform, interrupt and trigger ADC.
     * |        |          |In independent mode, CMPDAT0~5 denote as 6 independent PWM0_CH0~5 compared point.
     * |        |          |In complementary mode, CMPDAT0, 2, 4 denote as first compared point, and CMPDAT1, 3, 5 denote as second compared point for the corresponding 3 complementary pairs PWM0_CH0 and PWM0_CH1, PWM0_CH2 and PWM0_CH3, PWM0_CH4 and PWM0_CH5.
     * @var PWM_T::DTCTL0_1
     * Offset: 0x70  PWM0 Dead-Time Control Register 0_1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |DTCNT     |Dead-time Counter (Write Protect)
     * |        |          |The dead-time can be calculated from the following formula:
     * |        |          |Dead-time = (DTCNT[11:0]+1) * PWM0_CLK period.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[16]    |DTEN      |Enable Dead-time Insertion for PWM0 Pair (PWM0_CH0, PWM0_CH1) (PWM0_CH2, PWM0_CH3) (PWM0_CH4, PWM0_CH5) (Write Protect)
     * |        |          |Dead-time insertion is only active when this PWM0 pair complementary mode is enabled
     * |        |          |If dead- time insertion is inactive, the outputs of pin pair are complementary without any delay.
     * |        |          |0 = Dead-time insertion Disabled on the pin pair.
     * |        |          |1 = Dead-time insertion Enabled on the pin pair.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[24]    |DTCKSEL   |Dead-time Clock Select (Write Protect)
     * |        |          |0 = Dead-time clock source from PWM0_CLKn.
     * |        |          |1 = Dead-time clock source from prescaler output.
     * |        |          |Note: This register is write protected. Refer to REGWRPROT register.
     * @var PWM_T::DTCTL2_3
     * Offset: 0x74  PWM0 Dead-Time Control Register 2_3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |DTCNT     |Dead-time Counter (Write Protect)
     * |        |          |The dead-time can be calculated from the following formula:
     * |        |          |Dead-time = (DTCNT[11:0]+1) * PWM0_CLK period.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[16]    |DTEN      |Enable Dead-time Insertion for PWM0 Pair (PWM0_CH0, PWM0_CH1) (PWM0_CH2, PWM0_CH3) (PWM0_CH4, PWM0_CH5) (Write Protect)
     * |        |          |Dead-time insertion is only active when this PWM0 pair complementary mode is enabled
     * |        |          |If dead- time insertion is inactive, the outputs of pin pair are complementary without any delay.
     * |        |          |0 = Dead-time insertion Disabled on the pin pair.
     * |        |          |1 = Dead-time insertion Enabled on the pin pair.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[24]    |DTCKSEL   |Dead-time Clock Select (Write Protect)
     * |        |          |0 = Dead-time clock source from PWM0_CLKn.
     * |        |          |1 = Dead-time clock source from prescaler output.
     * |        |          |Note: This register is write protected. Refer to REGWRPROT register.
     * @var PWM_T::DTCTL4_5
     * Offset: 0x78  PWM0 Dead-Time Control Register 4_5
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |DTCNT     |Dead-time Counter (Write Protect)
     * |        |          |The dead-time can be calculated from the following formula:
     * |        |          |Dead-time = (DTCNT[11:0]+1) * PWM0_CLK period.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[16]    |DTEN      |Enable Dead-time Insertion for PWM0 Pair (PWM0_CH0, PWM0_CH1) (PWM0_CH2, PWM0_CH3) (PWM0_CH4, PWM0_CH5) (Write Protect)
     * |        |          |Dead-time insertion is only active when this PWM0 pair complementary mode is enabled
     * |        |          |If dead- time insertion is inactive, the outputs of pin pair are complementary without any delay.
     * |        |          |0 = Dead-time insertion Disabled on the pin pair.
     * |        |          |1 = Dead-time insertion Enabled on the pin pair.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[24]    |DTCKSEL   |Dead-time Clock Select (Write Protect)
     * |        |          |0 = Dead-time clock source from PWM0_CLKn.
     * |        |          |1 = Dead-time clock source from prescaler output.
     * |        |          |Note: This register is write protected. Refer to REGWRPROT register.
     * @var PWM_T::CNT
     * Offset: 0x90  PWM0 Counter Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CNT       |PWM0 Data Register (Read Only)
     * |        |          |User can monitor CNTR to know the current value in 16-bit period counter.
     * |[16]    |DIRF      |PWM0 Direction Indicator Flag (Read Only)
     * |        |          |0 = Counter is Down count.
     * |        |          |1 = Counter is UP count.
     * @var PWM_T::WGCTL0
     * Offset: 0xB0  PWM0 Waveform Generation Control Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |ZPCTLn    |PWM0 Zero Point Control
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM0 zero point output Low.
     * |        |          |10 = PWM0 zero point output High.
     * |        |          |11 = PWM0 zero point output Toggle.
     * |        |          |PWM0 can control output level when PWM0 counter count to zero.
     * |[27:16] |PRDPCTLn  |PWM0 Period (Center) Point Control
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM0 period (center) point output Low.
     * |        |          |10 = PWM0 period (center) point output High.
     * |        |          |11 = PWM0 period (center) point output Toggle.
     * |        |          |PWM0 can control output level when PWM0 counter count to (PERIODn+1).
     * |        |          |Note: This bit is center point control when PWM0 counter operating in up-down counter type.
     * @var PWM_T::WGCTL1
     * Offset: 0xB4  PWM0 Waveform Generation Control Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |CMPUCTLn  |PWM0 Compare Up Point Control
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM0 compare up point output Low.
     * |        |          |10 = PWM0 compare up point output High.
     * |        |          |11 = PWM0 compare up point output Toggle.
     * |        |          |PWM0 can control output level when PWM0 counter up count to CMPDAT.
     * |        |          |Note: In complementary mode, CMPUCTL1, 3, 5 use as another CMPUCTL for channel 0, 2, 4.
     * |[27:16] |CMPDCTLn  |PWM0 Compare Down Point Control
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |00 = Do nothing.
     * |        |          |01 = PWM0 compare down point output Low.
     * |        |          |10 = PWM0 compare down point output High.
     * |        |          |11 = PWM0 compare down point output Toggle.
     * |        |          |PWM0 can control output level when PWM0 counter down count to CMPDAT.
     * |        |          |Note: In complementary mode, CMPDCTL1, 3, 5 use as another CMPDCTL for channel 0, 2, 4.
     * @var PWM_T::MSKEN
     * Offset: 0xB8  PWM0 Mask Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |MSKENn    |PWM0 Mask Enable Bits
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |The PWM0 output signal will be masked when this bit is enabled
     * |        |          |The corresponding PWM0 channel n will output MSKDATn (PWM0_MSK[5:0]) data.
     * |        |          |0 = PWM0 output signal is non-masked.
     * |        |          |1 = PWM0 output signal is masked and output MSKDATn data.
     * @var PWM_T::MSK
     * Offset: 0xBC  PWM0 Mask Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |MSKDATn   |PWM0 Mask Data Bits
     * |        |          |This data bit control the state of PWM0_CHn output pin, if corresponding mask function is enabled.
     * |        |          |0 = Output logic low to PWM0_CHn.
     * |        |          |1 = Output logic high to PWM0_CHn.
     * @var PWM_T::BNF
     * Offset: 0xC0  PWM0 Brake Noise Filter Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |BRK0FEN   |PWM0 Brake 0 Noise Filter Enable Bit
     * |        |          |0 = Noise filter of PWM0 Brake 0 Disabled.
     * |        |          |1 = Noise filter of PWM0 Brake 0 Enabled.
     * |[3:1]   |BRK0FCS   |Brake 0 Edge Detector Filter Clock Selection
     * |        |          |000 = Filter clock = HCLK.
     * |        |          |001 = Filter clock = HCLK/2.
     * |        |          |010 = Filter clock = HCLK/4.
     * |        |          |011 = Filter clock = HCLK/8.
     * |        |          |100 = Filter clock = HCLK/16.
     * |        |          |101 = Filter clock = HCLK/32.
     * |        |          |110 = Filter clock = HCLK/64.
     * |        |          |111 = Filter clock = HCLK/128.
     * |[6:4]   |BRK0FCNT  |Brake 0 Edge Detector Filter Count
     * |        |          |The register bits control the Brake0 filter counter to count from 0 to BRK1FCNT.
     * |[7]     |BRK0PINV  |Brake 0 Pin Inverse
     * |        |          |0 = The state of pin PWMx_BRAKE0 is passed to the negative edge Detector.
     * |        |          |1 = The inversed state of pin PWMx_BRAKE10 is passed to the negative edge Detector.
     * |[8]     |BRK1FEN   |PWM0 Brake 1 Noise Filter Enable Bit
     * |        |          |0 = Noise filter of PWM0 Brake 1 Disabled.
     * |        |          |1 = Noise filter of PWM0 Brake 1 Enabled.
     * |[11:9]  |BRK1FCS   |Brake 1 Edge Detector Filter Clock Selection
     * |        |          |000 = Filter clock = HCLK.
     * |        |          |001 = Filter clock = HCLK/2.
     * |        |          |010 = Filter clock = HCLK/4.
     * |        |          |011 = Filter clock = HCLK/8.
     * |        |          |100 = Filter clock = HCLK/16.
     * |        |          |101 = Filter clock = HCLK/32.
     * |        |          |110 = Filter clock = HCLK/64.
     * |        |          |111 = Filter clock = HCLK/128.
     * |[14:12] |BRK1FCNT  |Brake 1 Edge Detector Filter Count
     * |        |          |The register bits control the Brake1 filter counter to count from 0 to BRK1FCNT.
     * |[15]    |BRK1PINV  |Brake 1 Pin Inverse
     * |        |          |0 = The state of pin PWMx_BRAKE1 is passed to the negative edge Detector.
     * |        |          |1 = The inversed state of pin PWMx_BRAKE1 is passed to the negative edge Detector.
     * |[16]    |BK0SRC    |Brake 0 Pin Source Select
     * |        |          |For PWM0 setting:
     * |        |          |0 = Brake 0 pin source come from PWM0_BRAKE0.
     * |        |          |1 = Brake 0 pin source come from PWM1_BRAKE0.
     * |        |          |For PWM1 setting:
     * |        |          |0 = Brake 0 pin source come from PWM1_BRAKE0.
     * |        |          |1 = Brake 0 pin source come from PWM0_BRAKE0.
     * |[24]    |BK1SRC    |Brake 1 Pin Source Select
     * |        |          |For PWM0 setting:
     * |        |          |0 = Brake 1 pin source come from PWM0_BRAKE1.
     * |        |          |1 = Brake 1 pin source come from PWM1_BRAKE1.
     * |        |          |For PWM1 setting:
     * |        |          |0 = Brake 1 pin source come from PWM1_BRAKE1.
     * |        |          |1 = Brake 1 pin source come from PWM0_BRAKE1.
     * @var PWM_T::FAILBRK
     * Offset: 0xC4  PWM0 System Fail Brake Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1]     |BODBRKEN  |Brown-out Detection Trigger PWM0 Brake Function 0 Enable Bit
     * |        |          |0 = Brake Function triggered by BOD Disabled.
     * |        |          |1 = Brake Function triggered by BOD Enabled.
     * |[3]     |CORBRKEN  |Core Lockup Detection Trigger PWM0 Brake Function 0 Enable Bit
     * |        |          |0 = Brake Function triggered by Core lockup detection Disabled.
     * |        |          |1 = Brake Function triggered by Core lockup detection Enabled.
     * @var PWM_T::BRKCTL0_1
     * Offset: 0xC8  PWM0 Brake Edge Detect Control Register 0_1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[4]     |BRKP0EEN  |Enable PWMx_BRAKE0 Pin As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = BKP0 pin as edge-detect brake source Disabled.
     * |        |          |1 = BKP0 pin as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[5]     |BRKP1EEN  |Enable PWMx_BRAKE1 Pin As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = BKP1 pin as edge-detect brake source Disabled.
     * |        |          |1 = BKP1 pin as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[7]     |SYSEEN    |Enable System Fail As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = System Fail condition as edge-detect brake source Disabled.
     * |        |          |1 = System Fail condition as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[12]    |BRKP0LEN  |Enable BKP0 Pin As Level-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE0 pin as level-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE0 pin as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[13]    |BRKP1LEN  |Enable BKP1 Pin As Level-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE1 pin as level-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE1 pin as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[15]    |SYSLEN    |Enable System Fail As Level-detect Brake Source (Write Protect)
     * |        |          |0 = System Fail condition as level-detect brake source Disabled.
     * |        |          |1 = System Fail condition as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[17:16] |BRKAEVEN  |PWM0 Brake Action Select for Even Channel (Write Protect)
     * |        |          |00 = PWM0 even channel brake function not affect channel output.
     * |        |          |01 = PWM0 even channel output tri-state when brake happened.
     * |        |          |10 = PWM0 even channel output low level when brake happened.
     * |        |          |11 = PWM0 even channel output high level when brake happened.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[19:18] |BRKAODD   |PWM0 Brake Action Select for Odd Channel (Write Protect)
     * |        |          |00 = PWM0 odd channel brake function not affect channel output.
     * |        |          |01 = PWM0 odd channel output tri-state when brake happened.
     * |        |          |10 = PWM0 odd channel output low level when brake happened.
     * |        |          |11 = PWM0 odd channel output high level when brake happened.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * @var PWM_T::BRKCTL2_3
     * Offset: 0xCC  PWM0 Brake Edge Detect Control Register 2_3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[4]     |BRKP0EEN  |Enable PWMx_BRAKE0 Pin As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = BKP0 pin as edge-detect brake source Disabled.
     * |        |          |1 = BKP0 pin as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[5]     |BRKP1EEN  |Enable PWMx_BRAKE1 Pin As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = BKP1 pin as edge-detect brake source Disabled.
     * |        |          |1 = BKP1 pin as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[7]     |SYSEEN    |Enable System Fail As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = System Fail condition as edge-detect brake source Disabled.
     * |        |          |1 = System Fail condition as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[12]    |BRKP0LEN  |Enable BKP0 Pin As Level-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE0 pin as level-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE0 pin as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[13]    |BRKP1LEN  |Enable BKP1 Pin As Level-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE1 pin as level-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE1 pin as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[15]    |SYSLEN    |Enable System Fail As Level-detect Brake Source (Write Protect)
     * |        |          |0 = System Fail condition as level-detect brake source Disabled.
     * |        |          |1 = System Fail condition as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[17:16] |BRKAEVEN  |PWM0 Brake Action Select for Even Channel (Write Protect)
     * |        |          |00 = PWM0 even channel brake function not affect channel output.
     * |        |          |01 = PWM0 even channel output tri-state when brake happened.
     * |        |          |10 = PWM0 even channel output low level when brake happened.
     * |        |          |11 = PWM0 even channel output high level when brake happened.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[19:18] |BRKAODD   |PWM0 Brake Action Select for Odd Channel (Write Protect)
     * |        |          |00 = PWM0 odd channel brake function not affect channel output.
     * |        |          |01 = PWM0 odd channel output tri-state when brake happened.
     * |        |          |10 = PWM0 odd channel output low level when brake happened.
     * |        |          |11 = PWM0 odd channel output high level when brake happened.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * @var PWM_T::BRKCTL4_5
     * Offset: 0xD0  PWM0 Brake Edge Detect Control Register 4_5
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[4]     |BRKP0EEN  |Enable PWMx_BRAKE0 Pin As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = BKP0 pin as edge-detect brake source Disabled.
     * |        |          |1 = BKP0 pin as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[5]     |BRKP1EEN  |Enable PWMx_BRAKE1 Pin As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = BKP1 pin as edge-detect brake source Disabled.
     * |        |          |1 = BKP1 pin as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[7]     |SYSEEN    |Enable System Fail As Edge-detect Brake Source (Write Protect)
     * |        |          |0 = System Fail condition as edge-detect brake source Disabled.
     * |        |          |1 = System Fail condition as edge-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[12]    |BRKP0LEN  |Enable BKP0 Pin As Level-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE0 pin as level-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE0 pin as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[13]    |BRKP1LEN  |Enable BKP1 Pin As Level-detect Brake Source (Write Protect)
     * |        |          |0 = PWMx_BRAKE1 pin as level-detect brake source Disabled.
     * |        |          |1 = PWMx_BRAKE1 pin as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[15]    |SYSLEN    |Enable System Fail As Level-detect Brake Source (Write Protect)
     * |        |          |0 = System Fail condition as level-detect brake source Disabled.
     * |        |          |1 = System Fail condition as level-detect brake source Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[17:16] |BRKAEVEN  |PWM0 Brake Action Select for Even Channel (Write Protect)
     * |        |          |00 = PWM0 even channel brake function not affect channel output.
     * |        |          |01 = PWM0 even channel output tri-state when brake happened.
     * |        |          |10 = PWM0 even channel output low level when brake happened.
     * |        |          |11 = PWM0 even channel output high level when brake happened.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[19:18] |BRKAODD   |PWM0 Brake Action Select for Odd Channel (Write Protect)
     * |        |          |00 = PWM0 odd channel brake function not affect channel output.
     * |        |          |01 = PWM0 odd channel output tri-state when brake happened.
     * |        |          |10 = PWM0 odd channel output low level when brake happened.
     * |        |          |11 = PWM0 odd channel output high level when brake happened.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * @var PWM_T::POLCTL
     * Offset: 0xD4  PWM0 Pin Polar Inverse Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |PINVn     |PWM0 PIN Polar Inverse Control
     * |        |          |The register controls polarity state of PWM0 output
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |0 = PWM0 output polar inverse Disabled.
     * |        |          |1 = PWM0 output polar inverse Enabled.
     * @var PWM_T::POEN
     * Offset: 0xD8  PWM0 Output Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |POENn     |PWM0 Pin Output Enable Bits
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |0 = PWM0 pin at tri-state.
     * |        |          |1 = PWM0 pin in output mode.
     * @var PWM_T::SWBRK
     * Offset: 0xDC  PWM0 Software Brake Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |BRKETRGn  |PWM0 Edge Brake Software Trigger (Write Only) (Write Protect)
     * |        |          |Each bit n controls the corresponding PWM0 pair n.
     * |        |          |Write 1 to this bit will trigger Edge brake, and set BRKEIFn to 1 in PWM0_INTSTS1 register.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[10:8]  |BRKLTRGn  |PWM0 Level Brake Software Trigger (Write Only) (Write Protect)
     * |        |          |Each bit n controls the corresponding PWM0 pair n.
     * |        |          |Write 1 to this bit will trigger level brake, and set BRKLIFn to 1 in PWM0_INTSTS1 register.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * @var PWM_T::INTEN0
     * Offset: 0xE0  PWM0 Interrupt Enable Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ZIEN0     |PWM0 Zero Point Interrupt Enable Bit 0
     * |        |          |0 = PWM0 counter0_1 zero point interrupt Disabled.
     * |        |          |1 = PWM0 counter0_1 zero point interrupt Enabled.
     * |        |          |Note: Odd channels will read always 0 at complementary mode.
     * |[2]     |ZIEN2     |PWM0 Zero Point Interrupt Enable Bit 2
     * |        |          |0 = PWM0 counter2_3 zero point interrupt Disabled.
     * |        |          |1 = PWM0 counter2_3 zero point interrupt Enabled.
     * |        |          |Note: Odd channels will read always 0 at complementary mode.
     * |[4]     |ZIEN4     |PWM0 Zero Point Interrupt Enable Bit 4
     * |        |          |0 = PWM0 counter4_5 zero point interrupt Disabled.
     * |        |          |1 = PWM0 counter4_5 zero point interrupt Enabled.
     * |        |          |Note: Odd channels will read always 0 at complementary mode.
     * |[8]     |PIEN0     |PWM0 Period Point Interrupt Enable Bit 0
     * |        |          |0 = PWM0 counter0_1 period point interrupt Disabled.
     * |        |          |1 = PWM0 counter0_1 period point interrupt Enabled.
     * |        |          |Note: When operating in up-down counter type, period point means center point.
     * |[10]    |PIEN2     |PWM0 Period Point Interrupt Enable Bit 2
     * |        |          |0 = PWM0 counter2_3 period point interrupt Disabled.
     * |        |          |1 = PWM0 counter2_3 period point interrupt Enabled.
     * |        |          |Note: When operating in up-down counter type, period point means center point.
     * |[12]    |PIEN4     |PWM0 Period Point Interrupt Enable Bit 4
     * |        |          |0 = PWM0 counter4_5 period point interrupt Disabled.
     * |        |          |1 = PWM0 counter4_5 period point interrupt Enabled.
     * |        |          |Note: When operating in up-down counter type, period point means center point.
     * |[21:16] |CMPUIENn  |PWM0 Compare Up Count Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |0 = Compare up count interrupt Disabled.
     * |        |          |1 = Compare up count interrupt Enabled.
     * |        |          |Note: In complementary mode, CMPUIEN1, 3, 5 use as another CMPUIEN for channel 0, 2, 4.
     * |[29:24] |CMPDIENn  |PWM0 Compare Down Count Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |0 = Compare down count interrupt Disabled.
     * |        |          |1 = Compare down count interrupt Enabled.
     * |        |          |Note: In complementary mode, CMPDIEN1, 3, 5 use as another CMPDIEN for channel 0, 2, 4.
     * @var PWM_T::INTEN1
     * Offset: 0xE4  PWM0 Interrupt Enable Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |BRKEIEN0_1|PWM0 Edge-detect Brake Interrupt Enable Bit for Channel0/1 (Write Protect)
     * |        |          |0 = Edge-detect Brake interrupt for channel0/1 Disabled.
     * |        |          |1 = Edge-detect Brake interrupt for channel0/1 Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[1]     |BRKEIEN2_3|PWM0 Edge-detect Brake Interrupt Enable Bit for Channel2/3 (Write Protect)
     * |        |          |0 = Edge-detect Brake interrupt for channel2/3 Disabled.
     * |        |          |1 = Edge-detect Brake interrupt for channel2/3 Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[2]     |BRKEIEN4_5|PWM0 Edge-detect Brake Interrupt Enable Bit for Channel4/5 (Write Protect)
     * |        |          |0 = Edge-detect Brake interrupt for channel4/5 Disabled.
     * |        |          |1 = Edge-detect Brake interrupt for channel4/5 Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[8]     |BRKLIEN0_1|PWM0 Level-detect Brake Interrupt Enable Bit for Channel0/1 (Write Protect)
     * |        |          |0 = Level-detect Brake interrupt for channel0/1 Disabled.
     * |        |          |1 = Level-detect Brake interrupt for channel0/1 Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[9]     |BRKLIEN2_3|PWM0 Level-detect Brake Interrupt Enable Bit for Channel2/3 (Write Protect)
     * |        |          |0 = Level-detect Brake interrupt for channel2/3 Disabled.
     * |        |          |1 = Level-detect Brake interrupt for channel2/3 Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * |[10]    |BRKLIEN4_5|PWM0 Level-detect Brake Interrupt Enable Bit for Channel4/5 (Write Protect)
     * |        |          |0 = Level-detect Brake interrupt for channel4/5 Disabled.
     * |        |          |1 = Level-detect Brake interrupt for channel4/5 Enabled.
     * |        |          |Note: This register is write protected. Refer to SYS_REGLCTL register.
     * @var PWM_T::INTSTS0
     * Offset: 0xE8  PWM0 Interrupt Flag Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ZIF0      |PWM0 Zero Point Interrupt Flag 0
     * |        |          |This bit is set by hardware when PWM0_CH0 counter reaches zero, software can write 1 to clear this bit to zero.
     * |[2]     |ZIF2      |PWM0 Zero Point Interrupt Flag 2
     * |        |          |This bit is set by hardware when PWM0_CH2 counter reaches zero, software can write 1 to clear this bit to zero.
     * |[4]     |ZIF4      |PWM0 Zero Point Interrupt Flag 4
     * |        |          |This bit is set by hardware when PWM0_CH4 counter reaches zero, software can write 1 to clear this bit to zero.
     * |[8]     |PIF0      |PWM0 Period Point Interrupt Flag 0
     * |        |          |This bit is set by hardware when PWM0_CH0 counter reaches PWM0_PERIOD0, software can write 1 to clear this bit to zero.
     * |[10]    |PIF2      |PWM0 Period Point Interrupt Flag 2
     * |        |          |This bit is set by hardware when PWM0_CH2 counter reaches PWM0_PERIOD2, software can write 1 to clear this bit to zero.
     * |[12]    |PIF4      |PWM0 Period Point Interrupt Flag 4
     * |        |          |This bit is set by hardware when PWM0_CH4 counter reaches PWM0_PERIOD4, software can write 1 to clear this bit to zero.
     * |[21:16] |CMPUIFn   |PWM0 Compare Up Count Interrupt Flag
     * |        |          |Flag is set by hardware when PWM0 counter up count and reaches PWM0_CMPDATn, software can clear this bit by writing 1 to it
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |Note1: If CMPDAT equal to PERIOD, this flag is not working in up counter type selection.
     * |        |          |Note2: In complementary mode, CMPUIF1, 3, 5 use as another CMPUIF for channel 0, 2, 4.
     * |[29:24] |CMPDIFn   |PWM0 Compare Down Count Interrupt Flag
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |Flag is set by hardware when PWM0 counter down count and reaches PWM0_CMPDATn, software can clear this bit by writing 1 to it.
     * |        |          |Note1: If CMPDAT equal to PERIOD, this flag is not working in down counter type selection.
     * |        |          |Note2: In complementary mode, CMPDIF1, 3, 5 use as another CMPDIF for channel 0, 2, 4.
     * @var PWM_T::INTSTS1
     * Offset: 0xEC  PWM0 Interrupt Flag Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |BRKEIF0   |PWM0 Channel0 Edge-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM0 channel0 edge-detect brake event do not happened.
     * |        |          |1 = When PWM0 channel0 edge-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This bit is write protected. Refer to SYS_REGLCTL register.
     * |[1]     |BRKEIF1   |PWM0 Channel1 Edge-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM0 channel1 edge-detect brake event do not happened.
     * |        |          |1 = When PWM0 channel1 edge-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This bit is write protected. Refer to SYS_REGLCTL register.
     * |[2]     |BRKEIF2   |PWM0 Channel2 Edge-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM0 channel2 edge-detect brake event do not happened.
     * |        |          |1 = When PWM0 channel2 edge-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This bit is write protected. Refer to SYS_REGLCTL register.
     * |[3]     |BRKEIF3   |PWM0 Channel3 Edge-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM0 channel3 edge-detect brake event do not happened.
     * |        |          |1 = When PWM0 channel3 edge-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This bit is write protected. Refer to SYS_REGLCTL register.
     * |[4]     |BRKEIF4   |PWM0 Channel4 Edge-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM0 channel4 edge-detect brake event do not happened.
     * |        |          |1 = When PWM0 channel4 edge-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This bit is write protected. Refer to SYS_REGLCTL register.
     * |[5]     |BRKEIF5   |PWM0 Channel5 Edge-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM0 channel5 edge-detect brake event do not happened.
     * |        |          |1 = When PWM0 channel5 edge-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This bit is write protected. Refer to SYS_REGLCTL register.
     * |[8]     |BRKLIF0   |PWM0 Channel0 Level-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM0 channel0 level-detect brake event do not happened.
     * |        |          |1 = When PWM0 channel0 level-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This bit is write protected. Refer to SYS_REGLCTL register.
     * |[9]     |BRKLIF1   |PWM0 Channel1 Level-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM0 channel1 level-detect brake event do not happened.
     * |        |          |1 = When PWM0 channel1 level-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This bit is write protected. Refer to SYS_REGLCTL register.
     * |[10]    |BRKLIF2   |PWM0 Channel2 Level-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM0 channel2 level-detect brake event do not happened.
     * |        |          |1 = When PWM0 channel2 level-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This bit is write protected. Refer to SYS_REGLCTL register.
     * |[11]    |BRKLIF3   |PWM0 Channel3 Level-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM0 channel3 level-detect brake event do not happened.
     * |        |          |1 = When PWM0 channel3 level-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This bit is write protected. Refer to SYS_REGLCTL register.
     * |[12]    |BRKLIF4   |PWM0 Channel4 Level-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM0 channel4 level-detect brake event do not happened.
     * |        |          |1 = When PWM0 channel4 level-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This bit is write protected. Refer to SYS_REGLCTL register.
     * |[13]    |BRKLIF5   |PWM0 Channel5 Level-detect Brake Interrupt Flag (Write Protect)
     * |        |          |0 = PWM0 channel5 level-detect brake event do not happened.
     * |        |          |1 = When PWM0 channel5 level-detect brake event happened, this bit is set to 1, writing 1 to clear.
     * |        |          |Note: This bit is write protected. Refer to SYS_REGLCTL register.
     * |[16]    |BRKESTS0  |PWM0 Channel0 Edge-detect Brake Status (Read Only)
     * |        |          |0 = PWM0 channel0 edge-detect brake state is released.
     * |        |          |1 = When PWM0 channel0 edge-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM0 channel0 at brake state, writing 1 to clear.
     * |[17]    |BRKESTS1  |PWM0 Channel1 Edge-detect Brake Status (Read Only)
     * |        |          |0 = PWM0 channel1 edge-detect brake state is released.
     * |        |          |1 = When PWM0 channel1 edge-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM0 channel1 at brake state, writing 1 to clear.
     * |[18]    |BRKESTS2  |PWM0 Channel2 Edge-detect Brake Status (Read Only)
     * |        |          |0 = PWM0 channel2 edge-detect brake state is released.
     * |        |          |1 = When PWM0 channel2 edge-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM0 channel2 at brake state, writing 1 to clear.
     * |[19]    |BRKESTS3  |PWM0 Channel3 Edge-detect Brake Status (Read Only)
     * |        |          |0 = PWM0 channel3 edge-detect brake state is released.
     * |        |          |1 = When PWM0 channel3 edge-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM0 channel3 at brake state, writing 1 to clear.
     * |[20]    |BRKESTS4  |PWM0 Channel4 Edge-detect Brake Status (Read Only)
     * |        |          |0 = PWM0 channel4 edge-detect brake state is released.
     * |        |          |1 = When PWM0 channel4 edge-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM0 channel4 at brake state, writing 1 to clear.
     * |[21]    |BRKESTS5  |PWM0 Channel5 Edge-detect Brake Status (Read Only)
     * |        |          |0 = PWM0 channel5 edge-detect brake state is released.
     * |        |          |1 = When PWM0 channel5 edge-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM0 channel5 at brake state, writing 1 to clear.
     * |[24]    |BRKLSTS0  |PWM0 Channel0 Level-detect Brake Status (Read Only)
     * |        |          |0 = PWM0 channel0 level-detect brake state is released.
     * |        |          |1 = When PWM0 channel0 level-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM0 channel0 at brake state.
     * |        |          |Note: This bit is read only and auto cleared by hardware
     * |        |          |When enabled brake source return to high level, PWM0 will release brake state until current PWM0 period finished
     * |        |          |The PWM0 waveform will start output from next full PWM0 period.
     * |[25]    |BRKLSTS1  |PWM0 Channel1 Level-detect Brake Status (Read Only)
     * |        |          |0 = PWM0 channel1 level-detect brake state is released.
     * |        |          |1 = When PWM0 channel1 level-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM0 channel1 at brake state.
     * |        |          |Note: This bit is read only and auto cleared by hardware
     * |        |          |When enabled brake source return to high level, PWM0 will release brake state until current PWM0 period finished
     * |        |          |The PWM0 waveform will start output from next full PWM0 period.
     * |[26]    |BRKLSTS2  |PWM0 Channel2 Level-detect Brake Status (Read Only)
     * |        |          |0 = PWM0 channel2 level-detect brake state is released.
     * |        |          |1 = When PWM0 channel2 level-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM0 channel2 at brake state.
     * |        |          |Note: This bit is read only and auto cleared by hardware
     * |        |          |When enabled brake source return to high level, PWM0 will release brake state until current PWM0 period finished
     * |        |          |The PWM0 waveform will start output from next full PWM0 period.
     * |[27]    |BRKLSTS3  |PWM0 Channel3 Level-detect Brake Status (Read Only)
     * |        |          |0 = PWM0 channel3 level-detect brake state is released.
     * |        |          |1 = When PWM0 channel3 level-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM0 channel3 at brake state.
     * |        |          |Note: This bit is read only and auto cleared by hardware
     * |        |          |When enabled brake source return to high level, PWM0 will release brake state until current PWM0 period finished
     * |        |          |The PWM0 waveform will start output from next full PWM0 period.
     * |[28]    |BRKLSTS4  |PWM0 Channel4 Level-detect Brake Status (Read Only)
     * |        |          |0 = PWM0 channel4 level-detect brake state is released.
     * |        |          |1 = When PWM0 channel4 level-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM0 channel4 at brake state.
     * |        |          |Note: This bit is read only and auto cleared by hardware
     * |        |          |When enabled brake source return to high level, PWM0 will release brake state until current PWM0 period finished
     * |        |          |The PWM0 waveform will start output from next full PWM0 period.
     * |[29]    |BRKLSTS5  |PWM0 Channel5 Level-detect Brake Status (Read Only)
     * |        |          |0 = PWM0 channel5 level-detect brake state is released.
     * |        |          |1 = When PWM0 channel5 level-detect brake detects a falling edge of any enabled brake source; this flag will be set to indicate the PWM0 channel5 at brake state.
     * |        |          |Note: This bit is read only and auto cleared by hardware
     * |        |          |When enabled brake source return to high level, PWM0 will release brake state until current PWM0 period finished
     * |        |          |The PWM0 waveform will start output from next full PWM0 period.
     * @var PWM_T::ADCTS0
     * Offset: 0xF8  PWM0 Trigger ADC Source Select Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |TRGSEL0   |PWM0_CH0 Trigger ADC Source Select
     * |        |          |0000 = PWM0_CH0 zero point.
     * |        |          |0001 = PWM0_CH0 period point.
     * |        |          |0010 = PWM0_CH0 zero or period point.
     * |        |          |0011 = PWM0_CH0 up-count CMPDAT point.
     * |        |          |0100 = PWM0_CH0 down-count CMPDAT point.
     * |        |          |0101 = Reserved.
     * |        |          |0110 = Reserved.
     * |        |          |0111 = Reserved.
     * |        |          |1000 = PWM0_CH1 up-count CMPDAT point.
     * |        |          |1001 = PWM0_CH1 down-count CMPDAT point.
     * |        |          |Others reserved
     * |[7]     |TRGEN0    |PWM0_CH0 Trigger EADC Enable Bit
     * |        |          |0 = PWM0_CH0 Trigger EADC Disabled.
     * |        |          |1 = PWM0_CH0 Trigger EADC Enabled.
     * |[11:8]  |TRGSEL1   |PWM0_CH1 Trigger ADC Source Select
     * |        |          |0000 = PWM0_CH0 zero point.
     * |        |          |0001 = PWM0_CH0 period point.
     * |        |          |0010 = PWM0_CH0 zero or period point.
     * |        |          |0011 = PWM0_CH0 up-count CMPDAT point.
     * |        |          |0100 = PWM0_CH0 down-count CMPDAT point.
     * |        |          |0101 = Reserved.
     * |        |          |0110 = Reserved.
     * |        |          |0111 = Reserved.
     * |        |          |1000 = PWM0_CH1 up-count CMPDAT point.
     * |        |          |1001 = PWM0_CH1 down-count CMPDAT point.
     * |        |          |Others reserved
     * |[15]    |TRGEN1    |PWM0_CH1 Trigger EADC Enable Bit
     * |        |          |0 = PWM0_CH1 Trigger EADC Disabled.
     * |        |          |1 = PWM0_CH1 Trigger EADC Enabled.
     * |[19:16] |TRGSEL2   |PWM0_CH2 Trigger ADC Source Select
     * |        |          |0000 = PWM0_CH2 zero point.
     * |        |          |0001 = PWM0_CH2 period point.
     * |        |          |0010 = PWM0_CH2 zero or period point.
     * |        |          |0011 = PWM0_CH2 up-count CMPDAT point.
     * |        |          |0100 = PWM0_CH2 down-count CMPDAT point.
     * |        |          |0101 = Reserved.
     * |        |          |0110 = Reserved.
     * |        |          |0111 = Reserved.
     * |        |          |1000 = PWM0_CH3 up-count CMPDAT point.
     * |        |          |1001 = PWM0_CH3 down-count CMPDAT point.
     * |        |          |Others reserved
     * |[23]    |TRGEN2    |PWM0_CH2 Trigger EADC Enable Bit
     * |        |          |0 = PWM0_CH2 Trigger EADC Disabled.
     * |        |          |1 = PWM0_CH2 Trigger EADC Enabled.
     * |[27:24] |TRGSEL3   |PWM0_CH3 Trigger ADC Source Select
     * |        |          |0000 = PWM0_CH2 zero point.
     * |        |          |0001 = PWM0_CH2 period point.
     * |        |          |0010 = PWM0_CH2 zero or period point.
     * |        |          |0011 = PWM0_CH2 up-count CMPDAT point.
     * |        |          |0100 = PWM0_CH2 down-count CMPDAT point.
     * |        |          |0101 = Reserved.
     * |        |          |0110 = Reserved.
     * |        |          |0111 = Reserved.
     * |        |          |1000 = PWM0_CH3 up-count CMPDAT point.
     * |        |          |1001 = PWM0_CH3 down-count CMPDAT point.
     * |        |          |Others reserved
     * |[31]    |TRGEN3    |PWM0_CH3 Trigger EADC Enable Bit
     * |        |          |0 = PWM0_CH3 Trigger EADC Disabled.
     * |        |          |1 = PWM0_CH3 Trigger EADC Enabled.
     * @var PWM_T::ADCTS1
     * Offset: 0xFC  PWM0 Trigger ADC Source Select Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |TRGSEL4   |PWM0_CH4 Trigger ADC Source Select
     * |        |          |0000 = PWM0_CH4 zero point.
     * |        |          |0001 = PWM0_CH4 period point.
     * |        |          |0010 = PWM0_CH4 zero or period point.
     * |        |          |0011 = PWM0_CH4 up-count CMPDAT point.
     * |        |          |0100 = PWM0_CH4 down-count CMPDAT point.
     * |        |          |0101 = Reserved.
     * |        |          |0110 = Reserved.
     * |        |          |0111 = Reserved.
     * |        |          |1000 = PWM0_CH5 up-count CMPDAT point.
     * |        |          |1001 = PWM0_CH5 down-count CMPDAT point.
     * |        |          |Others reserved
     * |[7]     |TRGEN4    |PWM0_CH4 Trigger EADC Enable Bit
     * |        |          |0 = PWM0_CH4 Trigger EADC Disabled.
     * |        |          |1 = PWM0_CH4 Trigger EADC Enabled.
     * |[11:8]  |TRGSEL5   |PWM0_CH5 Trigger ADC Source Select
     * |        |          |0000 = PWM0_CH4 zero point.
     * |        |          |0001 = PWM0_CH4 period point.
     * |        |          |0010 = PWM0_CH4 zero or period point.
     * |        |          |0011 = PWM0_CH4 up-count CMPDAT point.
     * |        |          |0100 = PWM0_CH4 down-count CMPDAT point.
     * |        |          |0101 = Reserved.
     * |        |          |0110 = Reserved.
     * |        |          |0111 = Reserved.
     * |        |          |1000 = PWM0_CH5 up-count CMPDAT point.
     * |        |          |1001 = PWM0_CH5 down-count CMPDAT point.
     * |        |          |Others reserved
     * |[15]    |TRGEN5    |PWM0_CH5 Trigger EADC Enable Bit
     * |        |          |0 = PWM0_CH5 Trigger EADC Disabled.
     * |        |          |1 = PWM0_CH5 Trigger EADC Enabled.
     * @var PWM_T::STATUS
     * Offset: 0x120  PWM0 Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CNTMAX0   |Time-base Counter 0 Equal to 0xFFFF Latched Status
     * |        |          |0 = indicates the time-base counter never reached its maximum value 0xFFFF.
     * |        |          |1 = indicates the time-base counter reached its maximum value, software can write 1 to clear this bit.
     * |[2]     |CNTMAX2   |Time-base Counter 2 Equal to 0xFFFF Latched Status
     * |        |          |0 = indicates the time-base counter never reached its maximum value 0xFFFF.
     * |        |          |1 = indicates the time-base counter reached its maximum value, software can write 1 to clear this bit.
     * |[4]     |CNTMAX4   |Time-base Counter 4 Equal to 0xFFFF Latched Status
     * |        |          |0 = indicates the time-base counter never reached its maximum value 0xFFFF.
     * |        |          |1 = indicates the time-base counter reached its maximum value, software can write 1 to clear this bit.
     * |[21:16] |ADCTRGn   |ADC Start of Conversion Status
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |0 = Indicates no ADC start of conversion trigger event has occurred.
     * |        |          |1 = Indicates an ADC start of conversion trigger event has occurred, software can write 1 to clear this bit.
     * @var PWM_T::CAPINEN
     * Offset: 0x200  PWM0 Capture Input Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |CAPINENn  |Capture Input Enable Bits
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |0 = PWM0 Channel capture input path Disabled
     * |        |          |The input of PWM0 channel capture function is always regarded as 0.
     * |        |          |1 = PWM0 Channel capture input path Enabled
     * |        |          |The input of PWM0 channel capture function comes from correlative multifunction pin.
     * @var PWM_T::CAPCTL
     * Offset: 0x204  PWM0 Capture Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |CAPENn    |Capture Function Enable Bits
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |0 = Capture function Disabled. RCAPDAT/FCAPDAT register will not be updated.
     * |        |          |1 = Capture function Enabled
     * |        |          |Capture latched the PWM0 counter value when detected rising or falling edge of input signal and saved to RCAPDAT (Rising latch) and FCAPDAT (Falling latch).
     * |[13:8]  |CAPINVn   |Capture Inverter Enable Bits
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |0 = Capture source inverter Disabled.
     * |        |          |1 = Capture source inverter Enabled. Reverse the input signal from GPIO.
     * |[21:16] |RCRLDENn  |Rising Capture Reload Enable Bits
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |0 = Rising capture reload counter Disabled.
     * |        |          |1 = Rising capture reload counter Enabled.
     * |[29:24] |FCRLDENn  |Falling Capture Reload Enable Bits
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |0 = Falling capture reload counter Disabled.
     * |        |          |1 = Falling capture reload counter Enabled.
     * @var PWM_T::CAPSTS
     * Offset: 0x208  PWM0 Capture Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |CRIFOVn   |Capture Rising Interrupt Flag Overrun Status (Read Only)
     * |        |          |This flag indicates if rising latch happened when the corresponding CAPRIF is 1
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |Note: This bit will be cleared automatically when user clear corresponding CAPRIF.
     * |[13:8]  |CFIFOVn   |Capture Falling Interrupt Flag Overrun Status (Read Only)
     * |        |          |This flag indicates if falling latch happened when the corresponding CAPFIF is 1
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |Note: This bit will be cleared automatically when user clear corresponding CAPFIF.
     * @var PWM_T::RCAPDAT0
     * Offset: 0x20C  PWM0 Rising Capture Data Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RCAPDAT   |PWM0 Rising Capture Data Register (Read Only)
     * |        |          |When rising capture condition happened, the PWM0 counter value will be saved in this register.
     * @var PWM_T::FCAPDAT0
     * Offset: 0x210  PWM0 Falling Capture Data Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FCAPDAT   |PWM0 Falling Capture Data Register (Read Only)
     * |        |          |When falling capture condition happened, the PWM0 counter value will be saved in this register.
     * @var PWM_T::RCAPDAT1
     * Offset: 0x214  PWM0 Rising Capture Data Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RCAPDAT   |PWM0 Rising Capture Data Register (Read Only)
     * |        |          |When rising capture condition happened, the PWM0 counter value will be saved in this register.
     * @var PWM_T::FCAPDAT1
     * Offset: 0x218  PWM0 Falling Capture Data Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FCAPDAT   |PWM0 Falling Capture Data Register (Read Only)
     * |        |          |When falling capture condition happened, the PWM0 counter value will be saved in this register.
     * @var PWM_T::RCAPDAT2
     * Offset: 0x21C  PWM0 Rising Capture Data Register 2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RCAPDAT   |PWM0 Rising Capture Data Register (Read Only)
     * |        |          |When rising capture condition happened, the PWM0 counter value will be saved in this register.
     * @var PWM_T::FCAPDAT2
     * Offset: 0x220  PWM0 Falling Capture Data Register 2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FCAPDAT   |PWM0 Falling Capture Data Register (Read Only)
     * |        |          |When falling capture condition happened, the PWM0 counter value will be saved in this register.
     * @var PWM_T::RCAPDAT3
     * Offset: 0x224  PWM0 Rising Capture Data Register 3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RCAPDAT   |PWM0 Rising Capture Data Register (Read Only)
     * |        |          |When rising capture condition happened, the PWM0 counter value will be saved in this register.
     * @var PWM_T::FCAPDAT3
     * Offset: 0x228  PWM0 Falling Capture Data Register 3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FCAPDAT   |PWM0 Falling Capture Data Register (Read Only)
     * |        |          |When falling capture condition happened, the PWM0 counter value will be saved in this register.
     * @var PWM_T::RCAPDAT4
     * Offset: 0x22C  PWM0 Rising Capture Data Register 4
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RCAPDAT   |PWM0 Rising Capture Data Register (Read Only)
     * |        |          |When rising capture condition happened, the PWM0 counter value will be saved in this register.
     * @var PWM_T::FCAPDAT4
     * Offset: 0x230  PWM0 Falling Capture Data Register 4
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FCAPDAT   |PWM0 Falling Capture Data Register (Read Only)
     * |        |          |When falling capture condition happened, the PWM0 counter value will be saved in this register.
     * @var PWM_T::RCAPDAT5
     * Offset: 0x234  PWM0 Rising Capture Data Register 5
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RCAPDAT   |PWM0 Rising Capture Data Register (Read Only)
     * |        |          |When rising capture condition happened, the PWM0 counter value will be saved in this register.
     * @var PWM_T::FCAPDAT5
     * Offset: 0x238  PWM0 Falling Capture Data Register 5
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FCAPDAT   |PWM0 Falling Capture Data Register (Read Only)
     * |        |          |When falling capture condition happened, the PWM0 counter value will be saved in this register.
     * @var PWM_T::CAPIEN
     * Offset: 0x250  PWM0 Capture Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |CAPRIENn  |PWM0 Capture Rising Latch Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |0 = Capture rising edge latch interrupt Disabled.
     * |        |          |1 = Capture rising edge latch interrupt Enabled.
     * |[13:8]  |CAPFIENn  |PWM0 Capture Falling Latch Interrupt Enable Bits
     * |        |          |Each bit n controls the corresponding PWM0 channel n.
     * |        |          |0 = Capture falling edge latch interrupt Disabled.
     * |        |          |1 = Capture falling edge latch interrupt Enabled.
     * @var PWM_T::CAPIF
     * Offset: 0x254  PWM0 Capture Interrupt Flag Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |CAPRIFn   |PWM0 Capture Rising Latch Interrupt Flag
     * |        |          |This bit is writing 1 to clear. Each bit n controls the corresponding PWM0 channel n.
     * |        |          |0 = No capture rising latch condition happened.
     * |        |          |1 = Capture rising latch condition happened, this flag will be set to high.
     * |[13:8]  |CAPFIFn   |PWM0 Capture Falling Latch Interrupt Flag
     * |        |          |This bit is writing 1 to clear. Each bit n controls the corresponding PWM0 channel n.
     * |        |          |0 = No capture falling latch condition happened.
     * |        |          |1 = Capture falling latch condition happened, this flag will be set to high.
     * @var PWM_T::SELFTEST
     * Offset: 0x300  PWM0 Self-test Mode Enable
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * @var PWM_T::PBUF0
     * Offset: 0x304  PWM0 PERIOD0 Buffer
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |PBUF      |PWM0 Period Register Buffer (Read Only)
     * |        |          |Used as PERIOD active register.
     * @var PWM_T::PBUF2
     * Offset: 0x30C  PWM0 PERIOD2 Buffer
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |PBUF      |PWM0 Period Register Buffer (Read Only)
     * |        |          |Used as PERIOD active register.
     * @var PWM_T::PBUF4
     * Offset: 0x314  PWM0 PERIOD4 Buffer
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |PBUF      |PWM0 Period Register Buffer (Read Only)
     * |        |          |Used as PERIOD active register.
     * @var PWM_T::CMPBUF0
     * Offset: 0x31C  PWM0 CMPDAT0 Buffer
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CMPBUF    |PWM0 Comparator Register Buffer (Read Only)
     * |        |          |Used as CMP active register.
     * @var PWM_T::CMPBUF1
     * Offset: 0x320  PWM0 CMPDAT1 Buffer
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CMPBUF    |PWM0 Comparator Register Buffer (Read Only)
     * |        |          |Used as CMP active register.
     * @var PWM_T::CMPBUF2
     * Offset: 0x324  PWM0 CMPDAT2 Buffer
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CMPBUF    |PWM0 Comparator Register Buffer (Read Only)
     * |        |          |Used as CMP active register.
     * @var PWM_T::CMPBUF3
     * Offset: 0x328  PWM0 CMPDAT3 Buffer
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CMPBUF    |PWM0 Comparator Register Buffer (Read Only)
     * |        |          |Used as CMP active register.
     * @var PWM_T::CMPBUF4
     * Offset: 0x32C  PWM0 CMPDAT4 Buffer
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CMPBUF    |PWM0 Comparator Register Buffer (Read Only)
     * |        |          |Used as CMP active register.
     * @var PWM_T::CMPBUF5
     * Offset: 0x330  PWM0 CMPDAT5 Buffer
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CMPBUF    |PWM0 Comparator Register Buffer (Read Only)
     * |        |          |Used as CMP active register.
     */
    __IO uint32_t CTL0;                  /*!< [0x0000] PWM0 Control Register 0                                          */
    __IO uint32_t CTL1;                  /*!< [0x0004] PWM0 Control Register 1                                          */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE0[2];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t CLKSRC;                /*!< [0x0010] PWM0 Clock Source Register                                       */
    __IO uint32_t CLKPSC0_1;             /*!< [0x0014] PWM0 Clock Pre-Scale Register 0_1                                */
    __IO uint32_t CLKPSC2_3;             /*!< [0x0018] PWM0 Clock Pre-Scale Register 2_3                                */
    __IO uint32_t CLKPSC4_5;             /*!< [0x001c] PWM0 Clock Pre-Scale Register 4_5                                */
    __IO uint32_t CNTEN;                 /*!< [0x0020] PWM0 Counter Enable Register                                     */
    __IO uint32_t CNTCLR;                /*!< [0x0024] PWM0 Clear Counter Register                                      */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE1[2];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t PERIOD[6];             /*!< [0x0030] PWM0 Period Register 0,2,4                                       */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE4[2];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t CMPDAT[6];             /*!< [0x0050] PWM0 Comparator Register 0,1,2,3,4,5                             */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE5[2];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t DTCTL0_1;              /*!< [0x0070] PWM0 Dead-Time Control Register 0_1                              */
    __IO uint32_t DTCTL2_3;              /*!< [0x0074] PWM0 Dead-Time Control Register 2_3                              */
    __IO uint32_t DTCTL4_5;              /*!< [0x0078] PWM0 Dead-Time Control Register 4_5                              */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE6[5];
    /// @endcond //HIDDEN_SYMBOLS
    __I  uint32_t CNT[6];                /*!< [0x0090] PWM0 Counter Register 0,2,4                                      */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE9[2];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t WGCTL0;                /*!< [0x00b0] PWM0 Waveform Generation Control Register 0                      */
    __IO uint32_t WGCTL1;                /*!< [0x00b4] PWM0 Waveform Generation Control Register 1                      */
    __IO uint32_t MSKEN;                 /*!< [0x00b8] PWM0 Mask Enable Register                                        */
    __IO uint32_t MSK;                   /*!< [0x00bc] PWM0 Mask Data Register                                          */
    __IO uint32_t BNF;                   /*!< [0x00c0] PWM0 Brake Noise Filter Register                                 */
    __IO uint32_t FAILBRK;               /*!< [0x00c4] PWM0 System Fail Brake Control Register                          */
    __IO uint32_t BRKCTL0_1;             /*!< [0x00c8] PWM0 Brake Edge Detect Control Register 0_1                      */
    __IO uint32_t BRKCTL2_3;             /*!< [0x00cc] PWM0 Brake Edge Detect Control Register 2_3                      */
    __IO uint32_t BRKCTL4_5;             /*!< [0x00d0] PWM0 Brake Edge Detect Control Register 4_5                      */
    __IO uint32_t POLCTL;                /*!< [0x00d4] PWM0 Pin Polar Inverse Register                                  */
    __IO uint32_t POEN;                  /*!< [0x00d8] PWM0 Output Enable Register                                      */
    __O  uint32_t SWBRK;                 /*!< [0x00dc] PWM0 Software Brake Control Register                             */
    __IO uint32_t INTEN0;                /*!< [0x00e0] PWM0 Interrupt Enable Register 0                                 */
    __IO uint32_t INTEN1;                /*!< [0x00e4] PWM0 Interrupt Enable Register 1                                 */
    __IO uint32_t INTSTS0;               /*!< [0x00e8] PWM0 Interrupt Flag Register 0                                   */
    __IO uint32_t INTSTS1;               /*!< [0x00ec] PWM0 Interrupt Flag Register 1                                   */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE10[2];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t ADCTS0;                /*!< [0x00f8] PWM0 Trigger ADC Source Select Register 0                        */
    __IO uint32_t ADCTS1;                /*!< [0x00fc] PWM0 Trigger ADC Source Select Register 1                        */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE11[8];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t STATUS;                /*!< [0x0120] PWM0 Status Register                                             */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE12[55];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t CAPINEN;               /*!< [0x0200] PWM0 Capture Input Enable Register                               */
    __IO uint32_t CAPCTL;                /*!< [0x0204] PWM0 Capture Control Register                                    */
    __I  uint32_t CAPSTS;                /*!< [0x0208] PWM0 Capture Status Register                                     */
    __I  uint32_t RCAPDAT0;              /*!< [0x020c] PWM0 Rising Capture Data Register 0                              */
    __I  uint32_t FCAPDAT0;              /*!< [0x0210] PWM0 Falling Capture Data Register 0                             */
    __I  uint32_t RCAPDAT1;              /*!< [0x0214] PWM0 Rising Capture Data Register 1                              */
    __I  uint32_t FCAPDAT1;              /*!< [0x0218] PWM0 Falling Capture Data Register 1                             */
    __I  uint32_t RCAPDAT2;              /*!< [0x021c] PWM0 Rising Capture Data Register 2                              */
    __I  uint32_t FCAPDAT2;              /*!< [0x0220] PWM0 Falling Capture Data Register 2                             */
    __I  uint32_t RCAPDAT3;              /*!< [0x0224] PWM0 Rising Capture Data Register 3                              */
    __I  uint32_t FCAPDAT3;              /*!< [0x0228] PWM0 Falling Capture Data Register 3                             */
    __I  uint32_t RCAPDAT4;              /*!< [0x022c] PWM0 Rising Capture Data Register 4                              */
    __I  uint32_t FCAPDAT4;              /*!< [0x0230] PWM0 Falling Capture Data Register 4                             */
    __I  uint32_t RCAPDAT5;              /*!< [0x0234] PWM0 Rising Capture Data Register 5                              */
    __I  uint32_t FCAPDAT5;              /*!< [0x0238] PWM0 Falling Capture Data Register 5                             */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE13[5];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t CAPIEN;                /*!< [0x0250] PWM0 Capture Interrupt Enable Register                           */
    __IO uint32_t CAPIF;                 /*!< [0x0254] PWM0 Capture Interrupt Flag Register                             */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE14[42];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t SELFTEST;              /*!< [0x0300] PWM0 Self-test Mode Enable                                       */
    __I  uint32_t PBUF0;                 /*!< [0x0304] PWM0 PERIOD0 Buffer                                              */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE15[1];
    /// @endcond //HIDDEN_SYMBOLS
    __I  uint32_t PBUF2;                 /*!< [0x030c] PWM0 PERIOD2 Buffer                                              */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE16[1];
    /// @endcond //HIDDEN_SYMBOLS
    __I  uint32_t PBUF4;                 /*!< [0x0314] PWM0 PERIOD4 Buffer                                              */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE17[1];
    /// @endcond //HIDDEN_SYMBOLS
    __I  uint32_t CMPBUF0;               /*!< [0x031c] PWM0 CMPDAT0 Buffer                                              */
    __I  uint32_t CMPBUF1;               /*!< [0x0320] PWM0 CMPDAT1 Buffer                                              */
    __I  uint32_t CMPBUF2;               /*!< [0x0324] PWM0 CMPDAT2 Buffer                                              */
    __I  uint32_t CMPBUF3;               /*!< [0x0328] PWM0 CMPDAT3 Buffer                                              */
    __I  uint32_t CMPBUF4;               /*!< [0x032c] PWM0 CMPDAT4 Buffer                                              */
    __I  uint32_t CMPBUF5;               /*!< [0x0330] PWM0 CMPDAT5 Buffer                                              */

} PWM_T;

/**
    @addtogroup PWM_CONST PWM Bit Field Definition
    Constant Definitions for PWM Controller
@{ */

#define PWM_CTL0_CTRLDn_Pos              (0)                                               /*!< PWM_T::CTL0: CTRLDn Position           */
#define PWM_CTL0_CTRLDn_Msk              (0x3ful << PWM_CTL0_CTRLDn_Pos)                   /*!< PWM_T::CTL0: CTRLDn Mask               */

#define PWM_CTL0_IMMLDENn_Pos            (16)                                              /*!< PWM_T::CTL0: IMMLDENn Position         */
#define PWM_CTL0_IMMLDENn_Msk            (0x3ful << PWM_CTL0_IMMLDENn_Pos)                 /*!< PWM_T::CTL0: IMMLDENn Mask             */

#define PWM_CTL0_DBGHALT_Pos             (30)                                              /*!< PWM_T::CTL0: DBGHALT Position          */
#define PWM_CTL0_DBGHALT_Msk             (0x1ul << PWM_CTL0_DBGHALT_Pos)                   /*!< PWM_T::CTL0: DBGHALT Mask              */

#define PWM_CTL0_DBGTRIOFF_Pos           (31)                                              /*!< PWM_T::CTL0: DBGTRIOFF Position        */
#define PWM_CTL0_DBGTRIOFF_Msk           (0x1ul << PWM_CTL0_DBGTRIOFF_Pos)                 /*!< PWM_T::CTL0: DBGTRIOFF Mask            */

#define PWM_CTL1_CNTTYPE0_Pos            (0)                                               /*!< PWM_T::CTL1: CNTTYPE0 Position         */
#define PWM_CTL1_CNTTYPE0_Msk            (0x3ul << PWM_CTL1_CNTTYPE0_Pos)                  /*!< PWM_T::CTL1: CNTTYPE0 Mask             */

#define PWM_CTL1_CNTTYPE2_Pos            (4)                                               /*!< PWM_T::CTL1: CNTTYPE2 Position         */
#define PWM_CTL1_CNTTYPE2_Msk            (0x3ul << PWM_CTL1_CNTTYPE2_Pos)                  /*!< PWM_T::CTL1: CNTTYPE2 Mask             */

#define PWM_CTL1_CNTTYPE4_Pos            (8)                                               /*!< PWM_T::CTL1: CNTTYPE4 Position         */
#define PWM_CTL1_CNTTYPE4_Msk            (0x3ul << PWM_CTL1_CNTTYPE4_Pos)                  /*!< PWM_T::CTL1: CNTTYPE4 Mask             */

#define PWM_CTL1_PWMMODEn_Pos            (24)                                              /*!< PWM_T::CTL1: PWMMODEn Position         */
#define PWM_CTL1_PWMMODEn_Msk            (0x7ul << PWM_CTL1_PWMMODEn_Pos)                  /*!< PWM_T::CTL1: PWMMODEn Mask             */

#define PWM_CLKSRC_ECLKSRC0_Pos          (0)                                               /*!< PWM_T::CLKSRC: ECLKSRC0 Position       */
#define PWM_CLKSRC_ECLKSRC0_Msk          (0x7ul << PWM_CLKSRC_ECLKSRC0_Pos)                /*!< PWM_T::CLKSRC: ECLKSRC0 Mask           */

#define PWM_CLKSRC_ECLKSRC2_Pos          (8)                                               /*!< PWM_T::CLKSRC: ECLKSRC2 Position       */
#define PWM_CLKSRC_ECLKSRC2_Msk          (0x7ul << PWM_CLKSRC_ECLKSRC2_Pos)                /*!< PWM_T::CLKSRC: ECLKSRC2 Mask           */

#define PWM_CLKSRC_ECLKSRC4_Pos          (16)                                              /*!< PWM_T::CLKSRC: ECLKSRC4 Position       */
#define PWM_CLKSRC_ECLKSRC4_Msk          (0x7ul << PWM_CLKSRC_ECLKSRC4_Pos)                /*!< PWM_T::CLKSRC: ECLKSRC4 Mask           */

#define PWM_CLKPSC0_1_CLKPSC_Pos         (0)                                               /*!< PWM_T::CLKPSC0_1: CLKPSC Position      */
#define PWM_CLKPSC0_1_CLKPSC_Msk         (0xffful << PWM_CLKPSC0_1_CLKPSC_Pos)             /*!< PWM_T::CLKPSC0_1: CLKPSC Mask          */

#define PWM_CLKPSC2_3_CLKPSC_Pos         (0)                                               /*!< PWM_T::CLKPSC2_3: CLKPSC Position      */
#define PWM_CLKPSC2_3_CLKPSC_Msk         (0xffful << PWM_CLKPSC2_3_CLKPSC_Pos)             /*!< PWM_T::CLKPSC2_3: CLKPSC Mask          */

#define PWM_CLKPSC4_5_CLKPSC_Pos         (0)                                               /*!< PWM_T::CLKPSC4_5: CLKPSC Position      */
#define PWM_CLKPSC4_5_CLKPSC_Msk         (0xffful << PWM_CLKPSC4_5_CLKPSC_Pos)             /*!< PWM_T::CLKPSC4_5: CLKPSC Mask          */

#define PWM_CNTEN_CNTEN0_Pos             (0)                                               /*!< PWM_T::CNTEN: CNTEN0 Position          */
#define PWM_CNTEN_CNTEN0_Msk             (0x1ul << PWM_CNTEN_CNTEN0_Pos)                   /*!< PWM_T::CNTEN: CNTEN0 Mask              */

#define PWM_CNTEN_CNTEN2_Pos             (2)                                               /*!< PWM_T::CNTEN: CNTEN2 Position          */
#define PWM_CNTEN_CNTEN2_Msk             (0x1ul << PWM_CNTEN_CNTEN2_Pos)                   /*!< PWM_T::CNTEN: CNTEN2 Mask              */

#define PWM_CNTEN_CNTEN4_Pos             (4)                                               /*!< PWM_T::CNTEN: CNTEN4 Position          */
#define PWM_CNTEN_CNTEN4_Msk             (0x1ul << PWM_CNTEN_CNTEN4_Pos)                   /*!< PWM_T::CNTEN: CNTEN4 Mask              */

#define PWM_CNTCLR_CNTCLR0_Pos           (0)                                               /*!< PWM_T::CNTCLR: CNTCLR0 Position        */
#define PWM_CNTCLR_CNTCLR0_Msk           (0x1ul << PWM_CNTCLR_CNTCLR0_Pos)                 /*!< PWM_T::CNTCLR: CNTCLR0 Mask            */

#define PWM_CNTCLR_CNTCLR2_Pos           (2)                                               /*!< PWM_T::CNTCLR: CNTCLR2 Position        */
#define PWM_CNTCLR_CNTCLR2_Msk           (0x1ul << PWM_CNTCLR_CNTCLR2_Pos)                 /*!< PWM_T::CNTCLR: CNTCLR2 Mask            */

#define PWM_CNTCLR_CNTCLR4_Pos           (4)                                               /*!< PWM_T::CNTCLR: CNTCLR4 Position        */
#define PWM_CNTCLR_CNTCLR4_Msk           (0x1ul << PWM_CNTCLR_CNTCLR4_Pos)                 /*!< PWM_T::CNTCLR: CNTCLR4 Mask            */

#define PWM_PERIOD0_PERIOD_Pos           (0)                                               /*!< PWM_T::PERIOD: PERIOD Position        */
#define PWM_PERIOD0_PERIOD_Msk           (0xfffful << PWM_PERIOD0_PERIOD_Pos)              /*!< PWM_T::PERIOD: PERIOD Mask            */

#define PWM_CMPDAT0_CMPDAT_Pos           (0)                                               /*!< PWM_T::CMPDAT: CMPDAT Position        */
#define PWM_CMPDAT0_CMPDAT_Msk           (0xfffful << PWM_CMPDAT0_CMPDAT_Pos)              /*!< PWM_T::CMPDAT: CMPDAT Mask            */

#define PWM_DTCTL0_1_DTCNT_Pos           (0)                                               /*!< PWM_T::DTCTL0_1: DTCNT Position        */
#define PWM_DTCTL0_1_DTCNT_Msk           (0xffful << PWM_DTCTL0_1_DTCNT_Pos)               /*!< PWM_T::DTCTL0_1: DTCNT Mask            */

#define PWM_DTCTL0_1_DTEN_Pos            (16)                                              /*!< PWM_T::DTCTL0_1: DTEN Position         */
#define PWM_DTCTL0_1_DTEN_Msk            (0x1ul << PWM_DTCTL0_1_DTEN_Pos)                  /*!< PWM_T::DTCTL0_1: DTEN Mask             */

#define PWM_DTCTL0_1_DTCKSEL_Pos         (24)                                              /*!< PWM_T::DTCTL0_1: DTCKSEL Position      */
#define PWM_DTCTL0_1_DTCKSEL_Msk         (0x1ul << PWM_DTCTL0_1_DTCKSEL_Pos)               /*!< PWM_T::DTCTL0_1: DTCKSEL Mask          */

#define PWM_DTCTL2_3_DTCNT_Pos           (0)                                               /*!< PWM_T::DTCTL2_3: DTCNT Position        */
#define PWM_DTCTL2_3_DTCNT_Msk           (0xffful << PWM_DTCTL2_3_DTCNT_Pos)               /*!< PWM_T::DTCTL2_3: DTCNT Mask            */

#define PWM_DTCTL2_3_DTEN_Pos            (16)                                              /*!< PWM_T::DTCTL2_3: DTEN Position         */
#define PWM_DTCTL2_3_DTEN_Msk            (0x1ul << PWM_DTCTL2_3_DTEN_Pos)                  /*!< PWM_T::DTCTL2_3: DTEN Mask             */

#define PWM_DTCTL2_3_DTCKSEL_Pos         (24)                                              /*!< PWM_T::DTCTL2_3: DTCKSEL Position      */
#define PWM_DTCTL2_3_DTCKSEL_Msk         (0x1ul << PWM_DTCTL2_3_DTCKSEL_Pos)               /*!< PWM_T::DTCTL2_3: DTCKSEL Mask          */

#define PWM_DTCTL4_5_DTCNT_Pos           (0)                                               /*!< PWM_T::DTCTL4_5: DTCNT Position        */
#define PWM_DTCTL4_5_DTCNT_Msk           (0xffful << PWM_DTCTL4_5_DTCNT_Pos)               /*!< PWM_T::DTCTL4_5: DTCNT Mask            */

#define PWM_DTCTL4_5_DTEN_Pos            (16)                                              /*!< PWM_T::DTCTL4_5: DTEN Position         */
#define PWM_DTCTL4_5_DTEN_Msk            (0x1ul << PWM_DTCTL4_5_DTEN_Pos)                  /*!< PWM_T::DTCTL4_5: DTEN Mask             */

#define PWM_DTCTL4_5_DTCKSEL_Pos         (24)                                              /*!< PWM_T::DTCTL4_5: DTCKSEL Position      */
#define PWM_DTCTL4_5_DTCKSEL_Msk         (0x1ul << PWM_DTCTL4_5_DTCKSEL_Pos)               /*!< PWM_T::DTCTL4_5: DTCKSEL Mask          */

#define PWM_CNT0_CNT_Pos                 (0)                                               /*!< PWM_T::CNT: CNT Position              */
#define PWM_CNT0_CNT_Msk                 (0xfffful << PWM_CNT0_CNT_Pos)                    /*!< PWM_T::CNT: CNT Mask                  */

#define PWM_CNT0_DIRF_Pos                (16)                                              /*!< PWM_T::CNT: DIRF Position             */
#define PWM_CNT0_DIRF_Msk                (0x1ul << PWM_CNT0_DIRF_Pos)                      /*!< PWM_T::CNT: DIRF Mask                 */

#define PWM_WGCTL0_ZPCTLn_Pos            (0)                                               /*!< PWM_T::WGCTL0: ZPCTLn Position         */
#define PWM_WGCTL0_ZPCTLn_Msk            (0xffful << PWM_WGCTL0_ZPCTLn_Pos)                /*!< PWM_T::WGCTL0: ZPCTLn Mask             */

#define PWM_WGCTL0_PRDPCTLn_Pos          (16)                                              /*!< PWM_T::WGCTL0: PRDPCTLn Position       */
#define PWM_WGCTL0_PRDPCTLn_Msk          (0xffful << PWM_WGCTL0_PRDPCTLn_Pos)              /*!< PWM_T::WGCTL0: PRDPCTLn Mask           */

#define PWM_WGCTL1_CMPUCTLn_Pos          (0)                                               /*!< PWM_T::WGCTL1: CMPUCTLn Position       */
#define PWM_WGCTL1_CMPUCTLn_Msk          (0xffful << PWM_WGCTL1_CMPUCTLn_Pos)              /*!< PWM_T::WGCTL1: CMPUCTLn Mask           */

#define PWM_WGCTL1_CMPDCTLn_Pos          (16)                                              /*!< PWM_T::WGCTL1: CMPDCTLn Position       */
#define PWM_WGCTL1_CMPDCTLn_Msk          (0xffful << PWM_WGCTL1_CMPDCTLn_Pos)              /*!< PWM_T::WGCTL1: CMPDCTLn Mask           */

#define PWM_MSKEN_MSKENn_Pos             (0)                                               /*!< PWM_T::MSKEN: MSKENn Position          */
#define PWM_MSKEN_MSKENn_Msk             (0x3ful << PWM_MSKEN_MSKENn_Pos)                  /*!< PWM_T::MSKEN: MSKENn Mask              */

#define PWM_MSK_MSKDATn_Pos              (0)                                               /*!< PWM_T::MSK: MSKDATn Position           */
#define PWM_MSK_MSKDATn_Msk              (0x3ful << PWM_MSK_MSKDATn_Pos)                   /*!< PWM_T::MSK: MSKDATn Mask               */

#define PWM_BNF_BRK0FEN_Pos              (0)                                               /*!< PWM_T::BNF: BRK0FEN Position           */
#define PWM_BNF_BRK0FEN_Msk              (0x1ul << PWM_BNF_BRK0FEN_Pos)                    /*!< PWM_T::BNF: BRK0FEN Mask               */

#define PWM_BNF_BRK0FCS_Pos              (1)                                               /*!< PWM_T::BNF: BRK0FCS Position           */
#define PWM_BNF_BRK0FCS_Msk              (0x7ul << PWM_BNF_BRK0FCS_Pos)                    /*!< PWM_T::BNF: BRK0FCS Mask               */

#define PWM_BNF_BRK0FCNT_Pos             (4)                                               /*!< PWM_T::BNF: BRK0FCNT Position          */
#define PWM_BNF_BRK0FCNT_Msk             (0x7ul << PWM_BNF_BRK0FCNT_Pos)                   /*!< PWM_T::BNF: BRK0FCNT Mask              */

#define PWM_BNF_BRK0PINV_Pos             (7)                                               /*!< PWM_T::BNF: BRK0PINV Position          */
#define PWM_BNF_BRK0PINV_Msk             (0x1ul << PWM_BNF_BRK0PINV_Pos)                   /*!< PWM_T::BNF: BRK0PINV Mask              */

#define PWM_BNF_BRK1FEN_Pos              (8)                                               /*!< PWM_T::BNF: BRK1FEN Position           */
#define PWM_BNF_BRK1FEN_Msk              (0x1ul << PWM_BNF_BRK1FEN_Pos)                    /*!< PWM_T::BNF: BRK1FEN Mask               */

#define PWM_BNF_BRK1FCS_Pos              (9)                                               /*!< PWM_T::BNF: BRK1FCS Position           */
#define PWM_BNF_BRK1FCS_Msk              (0x7ul << PWM_BNF_BRK1FCS_Pos)                    /*!< PWM_T::BNF: BRK1FCS Mask               */

#define PWM_BNF_BRK1FCNT_Pos             (12)                                              /*!< PWM_T::BNF: BRK1FCNT Position          */
#define PWM_BNF_BRK1FCNT_Msk             (0x7ul << PWM_BNF_BRK1FCNT_Pos)                   /*!< PWM_T::BNF: BRK1FCNT Mask              */

#define PWM_BNF_BRK1PINV_Pos             (15)                                              /*!< PWM_T::BNF: BRK1PINV Position          */
#define PWM_BNF_BRK1PINV_Msk             (0x1ul << PWM_BNF_BRK1PINV_Pos)                   /*!< PWM_T::BNF: BRK1PINV Mask              */

#define PWM_BNF_BK0SRC_Pos               (16)                                              /*!< PWM_T::BNF: BK0SRC Position            */
#define PWM_BNF_BK0SRC_Msk               (0x1ul << PWM_BNF_BK0SRC_Pos)                     /*!< PWM_T::BNF: BK0SRC Mask                */

#define PWM_BNF_BK1SRC_Pos               (24)                                              /*!< PWM_T::BNF: BK1SRC Position            */
#define PWM_BNF_BK1SRC_Msk               (0x1ul << PWM_BNF_BK1SRC_Pos)                     /*!< PWM_T::BNF: BK1SRC Mask                */

#define PWM_FAILBRK_BODBRKEN_Pos         (1)                                               /*!< PWM_T::FAILBRK: BODBRKEN Position      */
#define PWM_FAILBRK_BODBRKEN_Msk         (0x1ul << PWM_FAILBRK_BODBRKEN_Pos)               /*!< PWM_T::FAILBRK: BODBRKEN Mask          */

#define PWM_FAILBRK_CORBRKEN_Pos         (3)                                               /*!< PWM_T::FAILBRK: CORBRKEN Position      */
#define PWM_FAILBRK_CORBRKEN_Msk         (0x1ul << PWM_FAILBRK_CORBRKEN_Pos)               /*!< PWM_T::FAILBRK: CORBRKEN Mask          */

#define PWM_BRKCTL0_1_BRKP0EEN_Pos       (4)                                               /*!< PWM_T::BRKCTL0_1: BRKP0EEN Position    */
#define PWM_BRKCTL0_1_BRKP0EEN_Msk       (0x1ul << PWM_BRKCTL0_1_BRKP0EEN_Pos)             /*!< PWM_T::BRKCTL0_1: BRKP0EEN Mask        */

#define PWM_BRKCTL0_1_BRKP1EEN_Pos       (5)                                               /*!< PWM_T::BRKCTL0_1: BRKP1EEN Position    */
#define PWM_BRKCTL0_1_BRKP1EEN_Msk       (0x1ul << PWM_BRKCTL0_1_BRKP1EEN_Pos)             /*!< PWM_T::BRKCTL0_1: BRKP1EEN Mask        */

#define PWM_BRKCTL0_1_SYSEEN_Pos         (7)                                               /*!< PWM_T::BRKCTL0_1: SYSEEN Position      */
#define PWM_BRKCTL0_1_SYSEEN_Msk         (0x1ul << PWM_BRKCTL0_1_SYSEEN_Pos)               /*!< PWM_T::BRKCTL0_1: SYSEEN Mask          */

#define PWM_BRKCTL0_1_BRKP0LEN_Pos       (12)                                              /*!< PWM_T::BRKCTL0_1: BRKP0LEN Position    */
#define PWM_BRKCTL0_1_BRKP0LEN_Msk       (0x1ul << PWM_BRKCTL0_1_BRKP0LEN_Pos)             /*!< PWM_T::BRKCTL0_1: BRKP0LEN Mask        */

#define PWM_BRKCTL0_1_BRKP1LEN_Pos       (13)                                              /*!< PWM_T::BRKCTL0_1: BRKP1LEN Position    */
#define PWM_BRKCTL0_1_BRKP1LEN_Msk       (0x1ul << PWM_BRKCTL0_1_BRKP1LEN_Pos)             /*!< PWM_T::BRKCTL0_1: BRKP1LEN Mask        */

#define PWM_BRKCTL0_1_SYSLEN_Pos         (15)                                              /*!< PWM_T::BRKCTL0_1: SYSLEN Position      */
#define PWM_BRKCTL0_1_SYSLEN_Msk         (0x1ul << PWM_BRKCTL0_1_SYSLEN_Pos)               /*!< PWM_T::BRKCTL0_1: SYSLEN Mask          */

#define PWM_BRKCTL0_1_BRKAEVEN_Pos       (16)                                              /*!< PWM_T::BRKCTL0_1: BRKAEVEN Position    */
#define PWM_BRKCTL0_1_BRKAEVEN_Msk       (0x3ul << PWM_BRKCTL0_1_BRKAEVEN_Pos)             /*!< PWM_T::BRKCTL0_1: BRKAEVEN Mask        */

#define PWM_BRKCTL0_1_BRKAODD_Pos        (18)                                              /*!< PWM_T::BRKCTL0_1: BRKAODD Position     */
#define PWM_BRKCTL0_1_BRKAODD_Msk        (0x3ul << PWM_BRKCTL0_1_BRKAODD_Pos)              /*!< PWM_T::BRKCTL0_1: BRKAODD Mask         */

#define PWM_BRKCTL2_3_BRKP0EEN_Pos       (4)                                               /*!< PWM_T::BRKCTL2_3: BRKP0EEN Position    */
#define PWM_BRKCTL2_3_BRKP0EEN_Msk       (0x1ul << PWM_BRKCTL2_3_BRKP0EEN_Pos)             /*!< PWM_T::BRKCTL2_3: BRKP0EEN Mask        */

#define PWM_BRKCTL2_3_BRKP1EEN_Pos       (5)                                               /*!< PWM_T::BRKCTL2_3: BRKP1EEN Position    */
#define PWM_BRKCTL2_3_BRKP1EEN_Msk       (0x1ul << PWM_BRKCTL2_3_BRKP1EEN_Pos)             /*!< PWM_T::BRKCTL2_3: BRKP1EEN Mask        */

#define PWM_BRKCTL2_3_SYSEEN_Pos         (7)                                               /*!< PWM_T::BRKCTL2_3: SYSEEN Position      */
#define PWM_BRKCTL2_3_SYSEEN_Msk         (0x1ul << PWM_BRKCTL2_3_SYSEEN_Pos)               /*!< PWM_T::BRKCTL2_3: SYSEEN Mask          */

#define PWM_BRKCTL2_3_BRKP0LEN_Pos       (12)                                              /*!< PWM_T::BRKCTL2_3: BRKP0LEN Position    */
#define PWM_BRKCTL2_3_BRKP0LEN_Msk       (0x1ul << PWM_BRKCTL2_3_BRKP0LEN_Pos)             /*!< PWM_T::BRKCTL2_3: BRKP0LEN Mask        */

#define PWM_BRKCTL2_3_BRKP1LEN_Pos       (13)                                              /*!< PWM_T::BRKCTL2_3: BRKP1LEN Position    */
#define PWM_BRKCTL2_3_BRKP1LEN_Msk       (0x1ul << PWM_BRKCTL2_3_BRKP1LEN_Pos)             /*!< PWM_T::BRKCTL2_3: BRKP1LEN Mask        */

#define PWM_BRKCTL2_3_SYSLEN_Pos         (15)                                              /*!< PWM_T::BRKCTL2_3: SYSLEN Position      */
#define PWM_BRKCTL2_3_SYSLEN_Msk         (0x1ul << PWM_BRKCTL2_3_SYSLEN_Pos)               /*!< PWM_T::BRKCTL2_3: SYSLEN Mask          */

#define PWM_BRKCTL2_3_BRKAEVEN_Pos       (16)                                              /*!< PWM_T::BRKCTL2_3: BRKAEVEN Position    */
#define PWM_BRKCTL2_3_BRKAEVEN_Msk       (0x3ul << PWM_BRKCTL2_3_BRKAEVEN_Pos)             /*!< PWM_T::BRKCTL2_3: BRKAEVEN Mask        */

#define PWM_BRKCTL2_3_BRKAODD_Pos        (18)                                              /*!< PWM_T::BRKCTL2_3: BRKAODD Position     */
#define PWM_BRKCTL2_3_BRKAODD_Msk        (0x3ul << PWM_BRKCTL2_3_BRKAODD_Pos)              /*!< PWM_T::BRKCTL2_3: BRKAODD Mask         */

#define PWM_BRKCTL4_5_BRKP0EEN_Pos       (4)                                               /*!< PWM_T::BRKCTL4_5: BRKP0EEN Position    */
#define PWM_BRKCTL4_5_BRKP0EEN_Msk       (0x1ul << PWM_BRKCTL4_5_BRKP0EEN_Pos)             /*!< PWM_T::BRKCTL4_5: BRKP0EEN Mask        */

#define PWM_BRKCTL4_5_BRKP1EEN_Pos       (5)                                               /*!< PWM_T::BRKCTL4_5: BRKP1EEN Position    */
#define PWM_BRKCTL4_5_BRKP1EEN_Msk       (0x1ul << PWM_BRKCTL4_5_BRKP1EEN_Pos)             /*!< PWM_T::BRKCTL4_5: BRKP1EEN Mask        */

#define PWM_BRKCTL4_5_SYSEEN_Pos         (7)                                               /*!< PWM_T::BRKCTL4_5: SYSEEN Position      */
#define PWM_BRKCTL4_5_SYSEEN_Msk         (0x1ul << PWM_BRKCTL4_5_SYSEEN_Pos)               /*!< PWM_T::BRKCTL4_5: SYSEEN Mask          */

#define PWM_BRKCTL4_5_BRKP0LEN_Pos       (12)                                              /*!< PWM_T::BRKCTL4_5: BRKP0LEN Position    */
#define PWM_BRKCTL4_5_BRKP0LEN_Msk       (0x1ul << PWM_BRKCTL4_5_BRKP0LEN_Pos)             /*!< PWM_T::BRKCTL4_5: BRKP0LEN Mask        */

#define PWM_BRKCTL4_5_BRKP1LEN_Pos       (13)                                              /*!< PWM_T::BRKCTL4_5: BRKP1LEN Position    */
#define PWM_BRKCTL4_5_BRKP1LEN_Msk       (0x1ul << PWM_BRKCTL4_5_BRKP1LEN_Pos)             /*!< PWM_T::BRKCTL4_5: BRKP1LEN Mask        */

#define PWM_BRKCTL4_5_SYSLEN_Pos         (15)                                              /*!< PWM_T::BRKCTL4_5: SYSLEN Position      */
#define PWM_BRKCTL4_5_SYSLEN_Msk         (0x1ul << PWM_BRKCTL4_5_SYSLEN_Pos)               /*!< PWM_T::BRKCTL4_5: SYSLEN Mask          */

#define PWM_BRKCTL4_5_BRKAEVEN_Pos       (16)                                              /*!< PWM_T::BRKCTL4_5: BRKAEVEN Position    */
#define PWM_BRKCTL4_5_BRKAEVEN_Msk       (0x3ul << PWM_BRKCTL4_5_BRKAEVEN_Pos)             /*!< PWM_T::BRKCTL4_5: BRKAEVEN Mask        */

#define PWM_BRKCTL4_5_BRKAODD_Pos        (18)                                              /*!< PWM_T::BRKCTL4_5: BRKAODD Position     */
#define PWM_BRKCTL4_5_BRKAODD_Msk        (0x3ul << PWM_BRKCTL4_5_BRKAODD_Pos)              /*!< PWM_T::BRKCTL4_5: BRKAODD Mask         */

#define PWM_POLCTL_PINVn_Pos             (0)                                               /*!< PWM_T::POLCTL: PINVn Position          */
#define PWM_POLCTL_PINVn_Msk             (0x3ful << PWM_POLCTL_PINVn_Pos)                  /*!< PWM_T::POLCTL: PINVn Mask              */

#define PWM_POEN_POENn_Pos               (0)                                               /*!< PWM_T::POEN: POENn Position            */
#define PWM_POEN_POENn_Msk               (0x3ful << PWM_POEN_POENn_Pos)                    /*!< PWM_T::POEN: POENn Mask                */

#define PWM_SWBRK_BRKETRGn_Pos           (0)                                               /*!< PWM_T::SWBRK: BRKETRGn Position        */
#define PWM_SWBRK_BRKETRGn_Msk           (0x7ul << PWM_SWBRK_BRKETRGn_Pos)                 /*!< PWM_T::SWBRK: BRKETRGn Mask            */

#define PWM_SWBRK_BRKLTRGn_Pos           (8)                                               /*!< PWM_T::SWBRK: BRKLTRGn Position        */
#define PWM_SWBRK_BRKLTRGn_Msk           (0x7ul << PWM_SWBRK_BRKLTRGn_Pos)                 /*!< PWM_T::SWBRK: BRKLTRGn Mask            */

#define PWM_INTEN0_ZIEN0_Pos             (0)                                               /*!< PWM_T::INTEN0: ZIEN0 Position          */
#define PWM_INTEN0_ZIEN0_Msk             (0x1ul << PWM_INTEN0_ZIEN0_Pos)                   /*!< PWM_T::INTEN0: ZIEN0 Mask              */

#define PWM_INTEN0_ZIEN2_Pos             (2)                                               /*!< PWM_T::INTEN0: ZIEN2 Position          */
#define PWM_INTEN0_ZIEN2_Msk             (0x1ul << PWM_INTEN0_ZIEN2_Pos)                   /*!< PWM_T::INTEN0: ZIEN2 Mask              */

#define PWM_INTEN0_ZIEN4_Pos             (4)                                               /*!< PWM_T::INTEN0: ZIEN4 Position          */
#define PWM_INTEN0_ZIEN4_Msk             (0x1ul << PWM_INTEN0_ZIEN4_Pos)                   /*!< PWM_T::INTEN0: ZIEN4 Mask              */

#define PWM_INTEN0_PIEN0_Pos             (8)                                               /*!< PWM_T::INTEN0: PIEN0 Position          */
#define PWM_INTEN0_PIEN0_Msk             (0x1ul << PWM_INTEN0_PIEN0_Pos)                   /*!< PWM_T::INTEN0: PIEN0 Mask              */

#define PWM_INTEN0_PIEN2_Pos             (10)                                              /*!< PWM_T::INTEN0: PIEN2 Position          */
#define PWM_INTEN0_PIEN2_Msk             (0x1ul << PWM_INTEN0_PIEN2_Pos)                   /*!< PWM_T::INTEN0: PIEN2 Mask              */

#define PWM_INTEN0_PIEN4_Pos             (12)                                              /*!< PWM_T::INTEN0: PIEN4 Position          */
#define PWM_INTEN0_PIEN4_Msk             (0x1ul << PWM_INTEN0_PIEN4_Pos)                   /*!< PWM_T::INTEN0: PIEN4 Mask              */

#define PWM_INTEN0_CMPUIENn_Pos          (16)                                              /*!< PWM_T::INTEN0: CMPUIENn Position       */
#define PWM_INTEN0_CMPUIENn_Msk          (0x3ful << PWM_INTEN0_CMPUIENn_Pos)               /*!< PWM_T::INTEN0: CMPUIENn Mask           */

#define PWM_INTEN0_CMPDIENn_Pos          (24)                                              /*!< PWM_T::INTEN0: CMPDIENn Position       */
#define PWM_INTEN0_CMPDIENn_Msk          (0x3ful << PWM_INTEN0_CMPDIENn_Pos)               /*!< PWM_T::INTEN0: CMPDIENn Mask           */

#define PWM_INTEN1_BRKEIEN0_1_Pos        (0)                                               /*!< PWM_T::INTEN1: BRKEIEN0_1 Position     */
#define PWM_INTEN1_BRKEIEN0_1_Msk        (0x1ul << PWM_INTEN1_BRKEIEN0_1_Pos)              /*!< PWM_T::INTEN1: BRKEIEN0_1 Mask         */

#define PWM_INTEN1_BRKEIEN2_3_Pos        (1)                                               /*!< PWM_T::INTEN1: BRKEIEN2_3 Position     */
#define PWM_INTEN1_BRKEIEN2_3_Msk        (0x1ul << PWM_INTEN1_BRKEIEN2_3_Pos)              /*!< PWM_T::INTEN1: BRKEIEN2_3 Mask         */

#define PWM_INTEN1_BRKEIEN4_5_Pos        (2)                                               /*!< PWM_T::INTEN1: BRKEIEN4_5 Position     */
#define PWM_INTEN1_BRKEIEN4_5_Msk        (0x1ul << PWM_INTEN1_BRKEIEN4_5_Pos)              /*!< PWM_T::INTEN1: BRKEIEN4_5 Mask         */

#define PWM_INTEN1_BRKLIEN0_1_Pos        (8)                                               /*!< PWM_T::INTEN1: BRKLIEN0_1 Position     */
#define PWM_INTEN1_BRKLIEN0_1_Msk        (0x1ul << PWM_INTEN1_BRKLIEN0_1_Pos)              /*!< PWM_T::INTEN1: BRKLIEN0_1 Mask         */

#define PWM_INTEN1_BRKLIEN2_3_Pos        (9)                                               /*!< PWM_T::INTEN1: BRKLIEN2_3 Position     */
#define PWM_INTEN1_BRKLIEN2_3_Msk        (0x1ul << PWM_INTEN1_BRKLIEN2_3_Pos)              /*!< PWM_T::INTEN1: BRKLIEN2_3 Mask         */

#define PWM_INTEN1_BRKLIEN4_5_Pos        (10)                                              /*!< PWM_T::INTEN1: BRKLIEN4_5 Position     */
#define PWM_INTEN1_BRKLIEN4_5_Msk        (0x1ul << PWM_INTEN1_BRKLIEN4_5_Pos)              /*!< PWM_T::INTEN1: BRKLIEN4_5 Mask         */

#define PWM_INTSTS0_ZIF0_Pos             (0)                                               /*!< PWM_T::INTSTS0: ZIF0 Position          */
#define PWM_INTSTS0_ZIF0_Msk             (0x1ul << PWM_INTSTS0_ZIF0_Pos)                   /*!< PWM_T::INTSTS0: ZIF0 Mask              */

#define PWM_INTSTS0_ZIF2_Pos             (2)                                               /*!< PWM_T::INTSTS0: ZIF2 Position          */
#define PWM_INTSTS0_ZIF2_Msk             (0x1ul << PWM_INTSTS0_ZIF2_Pos)                   /*!< PWM_T::INTSTS0: ZIF2 Mask              */

#define PWM_INTSTS0_ZIF4_Pos             (4)                                               /*!< PWM_T::INTSTS0: ZIF4 Position          */
#define PWM_INTSTS0_ZIF4_Msk             (0x1ul << PWM_INTSTS0_ZIF4_Pos)                   /*!< PWM_T::INTSTS0: ZIF4 Mask              */

#define PWM_INTSTS0_PIF0_Pos             (8)                                               /*!< PWM_T::INTSTS0: PIF0 Position          */
#define PWM_INTSTS0_PIF0_Msk             (0x1ul << PWM_INTSTS0_PIF0_Pos)                   /*!< PWM_T::INTSTS0: PIF0 Mask              */

#define PWM_INTSTS0_PIF2_Pos             (10)                                              /*!< PWM_T::INTSTS0: PIF2 Position          */
#define PWM_INTSTS0_PIF2_Msk             (0x1ul << PWM_INTSTS0_PIF2_Pos)                   /*!< PWM_T::INTSTS0: PIF2 Mask              */

#define PWM_INTSTS0_PIF4_Pos             (12)                                              /*!< PWM_T::INTSTS0: PIF4 Position          */
#define PWM_INTSTS0_PIF4_Msk             (0x1ul << PWM_INTSTS0_PIF4_Pos)                   /*!< PWM_T::INTSTS0: PIF4 Mask              */

#define PWM_INTSTS0_CMPUIFn_Pos          (16)                                              /*!< PWM_T::INTSTS0: CMPUIFn Position       */
#define PWM_INTSTS0_CMPUIFn_Msk          (0x3ful << PWM_INTSTS0_CMPUIFn_Pos)               /*!< PWM_T::INTSTS0: CMPUIFn Mask           */

#define PWM_INTSTS0_CMPDIFn_Pos          (24)                                              /*!< PWM_T::INTSTS0: CMPDIFn Position       */
#define PWM_INTSTS0_CMPDIFn_Msk          (0x3ful << PWM_INTSTS0_CMPDIFn_Pos)               /*!< PWM_T::INTSTS0: CMPDIFn Mask           */

#define PWM_INTSTS1_BRKEIF0_Pos          (0)                                               /*!< PWM_T::INTSTS1: BRKEIF0 Position       */
#define PWM_INTSTS1_BRKEIF0_Msk          (0x1ul << PWM_INTSTS1_BRKEIF0_Pos)                /*!< PWM_T::INTSTS1: BRKEIF0 Mask           */

#define PWM_INTSTS1_BRKEIF1_Pos          (1)                                               /*!< PWM_T::INTSTS1: BRKEIF1 Position       */
#define PWM_INTSTS1_BRKEIF1_Msk          (0x1ul << PWM_INTSTS1_BRKEIF1_Pos)                /*!< PWM_T::INTSTS1: BRKEIF1 Mask           */

#define PWM_INTSTS1_BRKEIF2_Pos          (2)                                               /*!< PWM_T::INTSTS1: BRKEIF2 Position       */
#define PWM_INTSTS1_BRKEIF2_Msk          (0x1ul << PWM_INTSTS1_BRKEIF2_Pos)                /*!< PWM_T::INTSTS1: BRKEIF2 Mask           */

#define PWM_INTSTS1_BRKEIF3_Pos          (3)                                               /*!< PWM_T::INTSTS1: BRKEIF3 Position       */
#define PWM_INTSTS1_BRKEIF3_Msk          (0x1ul << PWM_INTSTS1_BRKEIF3_Pos)                /*!< PWM_T::INTSTS1: BRKEIF3 Mask           */

#define PWM_INTSTS1_BRKEIF4_Pos          (4)                                               /*!< PWM_T::INTSTS1: BRKEIF4 Position       */
#define PWM_INTSTS1_BRKEIF4_Msk          (0x1ul << PWM_INTSTS1_BRKEIF4_Pos)                /*!< PWM_T::INTSTS1: BRKEIF4 Mask           */

#define PWM_INTSTS1_BRKEIF5_Pos          (5)                                               /*!< PWM_T::INTSTS1: BRKEIF5 Position       */
#define PWM_INTSTS1_BRKEIF5_Msk          (0x1ul << PWM_INTSTS1_BRKEIF5_Pos)                /*!< PWM_T::INTSTS1: BRKEIF5 Mask           */

#define PWM_INTSTS1_BRKLIF0_Pos          (8)                                               /*!< PWM_T::INTSTS1: BRKLIF0 Position       */
#define PWM_INTSTS1_BRKLIF0_Msk          (0x1ul << PWM_INTSTS1_BRKLIF0_Pos)                /*!< PWM_T::INTSTS1: BRKLIF0 Mask           */

#define PWM_INTSTS1_BRKLIF1_Pos          (9)                                               /*!< PWM_T::INTSTS1: BRKLIF1 Position       */
#define PWM_INTSTS1_BRKLIF1_Msk          (0x1ul << PWM_INTSTS1_BRKLIF1_Pos)                /*!< PWM_T::INTSTS1: BRKLIF1 Mask           */

#define PWM_INTSTS1_BRKLIF2_Pos          (10)                                              /*!< PWM_T::INTSTS1: BRKLIF2 Position       */
#define PWM_INTSTS1_BRKLIF2_Msk          (0x1ul << PWM_INTSTS1_BRKLIF2_Pos)                /*!< PWM_T::INTSTS1: BRKLIF2 Mask           */

#define PWM_INTSTS1_BRKLIF3_Pos          (11)                                              /*!< PWM_T::INTSTS1: BRKLIF3 Position       */
#define PWM_INTSTS1_BRKLIF3_Msk          (0x1ul << PWM_INTSTS1_BRKLIF3_Pos)                /*!< PWM_T::INTSTS1: BRKLIF3 Mask           */

#define PWM_INTSTS1_BRKLIF4_Pos          (12)                                              /*!< PWM_T::INTSTS1: BRKLIF4 Position       */
#define PWM_INTSTS1_BRKLIF4_Msk          (0x1ul << PWM_INTSTS1_BRKLIF4_Pos)                /*!< PWM_T::INTSTS1: BRKLIF4 Mask           */

#define PWM_INTSTS1_BRKLIF5_Pos          (13)                                              /*!< PWM_T::INTSTS1: BRKLIF5 Position       */
#define PWM_INTSTS1_BRKLIF5_Msk          (0x1ul << PWM_INTSTS1_BRKLIF5_Pos)                /*!< PWM_T::INTSTS1: BRKLIF5 Mask           */

#define PWM_INTSTS1_BRKESTS0_Pos         (16)                                              /*!< PWM_T::INTSTS1: BRKESTS0 Position      */
#define PWM_INTSTS1_BRKESTS0_Msk         (0x1ul << PWM_INTSTS1_BRKESTS0_Pos)               /*!< PWM_T::INTSTS1: BRKESTS0 Mask          */

#define PWM_INTSTS1_BRKESTS1_Pos         (17)                                              /*!< PWM_T::INTSTS1: BRKESTS1 Position      */
#define PWM_INTSTS1_BRKESTS1_Msk         (0x1ul << PWM_INTSTS1_BRKESTS1_Pos)               /*!< PWM_T::INTSTS1: BRKESTS1 Mask          */

#define PWM_INTSTS1_BRKESTS2_Pos         (18)                                              /*!< PWM_T::INTSTS1: BRKESTS2 Position      */
#define PWM_INTSTS1_BRKESTS2_Msk         (0x1ul << PWM_INTSTS1_BRKESTS2_Pos)               /*!< PWM_T::INTSTS1: BRKESTS2 Mask          */

#define PWM_INTSTS1_BRKESTS3_Pos         (19)                                              /*!< PWM_T::INTSTS1: BRKESTS3 Position      */
#define PWM_INTSTS1_BRKESTS3_Msk         (0x1ul << PWM_INTSTS1_BRKESTS3_Pos)               /*!< PWM_T::INTSTS1: BRKESTS3 Mask          */

#define PWM_INTSTS1_BRKESTS4_Pos         (20)                                              /*!< PWM_T::INTSTS1: BRKESTS4 Position      */
#define PWM_INTSTS1_BRKESTS4_Msk         (0x1ul << PWM_INTSTS1_BRKESTS4_Pos)               /*!< PWM_T::INTSTS1: BRKESTS4 Mask          */

#define PWM_INTSTS1_BRKESTS5_Pos         (21)                                              /*!< PWM_T::INTSTS1: BRKESTS5 Position      */
#define PWM_INTSTS1_BRKESTS5_Msk         (0x1ul << PWM_INTSTS1_BRKESTS5_Pos)               /*!< PWM_T::INTSTS1: BRKESTS5 Mask          */

#define PWM_INTSTS1_BRKLSTS0_Pos         (24)                                              /*!< PWM_T::INTSTS1: BRKLSTS0 Position      */
#define PWM_INTSTS1_BRKLSTS0_Msk         (0x1ul << PWM_INTSTS1_BRKLSTS0_Pos)               /*!< PWM_T::INTSTS1: BRKLSTS0 Mask          */

#define PWM_INTSTS1_BRKLSTS1_Pos         (25)                                              /*!< PWM_T::INTSTS1: BRKLSTS1 Position      */
#define PWM_INTSTS1_BRKLSTS1_Msk         (0x1ul << PWM_INTSTS1_BRKLSTS1_Pos)               /*!< PWM_T::INTSTS1: BRKLSTS1 Mask          */

#define PWM_INTSTS1_BRKLSTS2_Pos         (26)                                              /*!< PWM_T::INTSTS1: BRKLSTS2 Position      */
#define PWM_INTSTS1_BRKLSTS2_Msk         (0x1ul << PWM_INTSTS1_BRKLSTS2_Pos)               /*!< PWM_T::INTSTS1: BRKLSTS2 Mask          */

#define PWM_INTSTS1_BRKLSTS3_Pos         (27)                                              /*!< PWM_T::INTSTS1: BRKLSTS3 Position      */
#define PWM_INTSTS1_BRKLSTS3_Msk         (0x1ul << PWM_INTSTS1_BRKLSTS3_Pos)               /*!< PWM_T::INTSTS1: BRKLSTS3 Mask          */

#define PWM_INTSTS1_BRKLSTS4_Pos         (28)                                              /*!< PWM_T::INTSTS1: BRKLSTS4 Position      */
#define PWM_INTSTS1_BRKLSTS4_Msk         (0x1ul << PWM_INTSTS1_BRKLSTS4_Pos)               /*!< PWM_T::INTSTS1: BRKLSTS4 Mask          */

#define PWM_INTSTS1_BRKLSTS5_Pos         (29)                                              /*!< PWM_T::INTSTS1: BRKLSTS5 Position      */
#define PWM_INTSTS1_BRKLSTS5_Msk         (0x1ul << PWM_INTSTS1_BRKLSTS5_Pos)               /*!< PWM_T::INTSTS1: BRKLSTS5 Mask          */

#define PWM_ADCTS0_TRGSEL0_Pos           (0)                                               /*!< PWM_T::ADCTS0: TRGSEL0 Position        */
#define PWM_ADCTS0_TRGSEL0_Msk           (0xful << PWM_ADCTS0_TRGSEL0_Pos)                 /*!< PWM_T::ADCTS0: TRGSEL0 Mask            */

#define PWM_ADCTS0_TRGEN0_Pos            (7)                                               /*!< PWM_T::ADCTS0: TRGEN0 Position         */
#define PWM_ADCTS0_TRGEN0_Msk            (0x1ul << PWM_ADCTS0_TRGEN0_Pos)                  /*!< PWM_T::ADCTS0: TRGEN0 Mask             */

#define PWM_ADCTS0_TRGSEL1_Pos           (8)                                               /*!< PWM_T::ADCTS0: TRGSEL1 Position        */
#define PWM_ADCTS0_TRGSEL1_Msk           (0xful << PWM_ADCTS0_TRGSEL1_Pos)                 /*!< PWM_T::ADCTS0: TRGSEL1 Mask            */

#define PWM_ADCTS0_TRGEN1_Pos            (15)                                              /*!< PWM_T::ADCTS0: TRGEN1 Position         */
#define PWM_ADCTS0_TRGEN1_Msk            (0x1ul << PWM_ADCTS0_TRGEN1_Pos)                  /*!< PWM_T::ADCTS0: TRGEN1 Mask             */

#define PWM_ADCTS0_TRGSEL2_Pos           (16)                                              /*!< PWM_T::ADCTS0: TRGSEL2 Position        */
#define PWM_ADCTS0_TRGSEL2_Msk           (0xful << PWM_ADCTS0_TRGSEL2_Pos)                 /*!< PWM_T::ADCTS0: TRGSEL2 Mask            */

#define PWM_ADCTS0_TRGEN2_Pos            (23)                                              /*!< PWM_T::ADCTS0: TRGEN2 Position         */
#define PWM_ADCTS0_TRGEN2_Msk            (0x1ul << PWM_ADCTS0_TRGEN2_Pos)                  /*!< PWM_T::ADCTS0: TRGEN2 Mask             */

#define PWM_ADCTS0_TRGSEL3_Pos           (24)                                              /*!< PWM_T::ADCTS0: TRGSEL3 Position        */
#define PWM_ADCTS0_TRGSEL3_Msk           (0xful << PWM_ADCTS0_TRGSEL3_Pos)                 /*!< PWM_T::ADCTS0: TRGSEL3 Mask            */

#define PWM_ADCTS0_TRGEN3_Pos            (31)                                              /*!< PWM_T::ADCTS0: TRGEN3 Position         */
#define PWM_ADCTS0_TRGEN3_Msk            (0x1ul << PWM_ADCTS0_TRGEN3_Pos)                  /*!< PWM_T::ADCTS0: TRGEN3 Mask             */

#define PWM_ADCTS1_TRGSEL4_Pos           (0)                                               /*!< PWM_T::ADCTS1: TRGSEL4 Position        */
#define PWM_ADCTS1_TRGSEL4_Msk           (0xful << PWM_ADCTS1_TRGSEL4_Pos)                 /*!< PWM_T::ADCTS1: TRGSEL4 Mask            */

#define PWM_ADCTS1_TRGEN4_Pos            (7)                                               /*!< PWM_T::ADCTS1: TRGEN4 Position         */
#define PWM_ADCTS1_TRGEN4_Msk            (0x1ul << PWM_ADCTS1_TRGEN4_Pos)                  /*!< PWM_T::ADCTS1: TRGEN4 Mask             */

#define PWM_ADCTS1_TRGSEL5_Pos           (8)                                               /*!< PWM_T::ADCTS1: TRGSEL5 Position        */
#define PWM_ADCTS1_TRGSEL5_Msk           (0xful << PWM_ADCTS1_TRGSEL5_Pos)                 /*!< PWM_T::ADCTS1: TRGSEL5 Mask            */

#define PWM_ADCTS1_TRGEN5_Pos            (15)                                              /*!< PWM_T::ADCTS1: TRGEN5 Position         */
#define PWM_ADCTS1_TRGEN5_Msk            (0x1ul << PWM_ADCTS1_TRGEN5_Pos)                  /*!< PWM_T::ADCTS1: TRGEN5 Mask             */

#define PWM_STATUS_CNTMAX0_Pos           (0)                                               /*!< PWM_T::STATUS: CNTMAX0 Position        */
#define PWM_STATUS_CNTMAX0_Msk           (0x1ul << PWM_STATUS_CNTMAX0_Pos)                 /*!< PWM_T::STATUS: CNTMAX0 Mask            */

#define PWM_STATUS_CNTMAX2_Pos           (2)                                               /*!< PWM_T::STATUS: CNTMAX2 Position        */
#define PWM_STATUS_CNTMAX2_Msk           (0x1ul << PWM_STATUS_CNTMAX2_Pos)                 /*!< PWM_T::STATUS: CNTMAX2 Mask            */

#define PWM_STATUS_CNTMAX4_Pos           (4)                                               /*!< PWM_T::STATUS: CNTMAX4 Position        */
#define PWM_STATUS_CNTMAX4_Msk           (0x1ul << PWM_STATUS_CNTMAX4_Pos)                 /*!< PWM_T::STATUS: CNTMAX4 Mask            */

#define PWM_STATUS_ADCTRGn_Pos           (16)                                              /*!< PWM_T::STATUS: ADCTRGn Position        */
#define PWM_STATUS_ADCTRGn_Msk           (0x3ful << PWM_STATUS_ADCTRGn_Pos)                /*!< PWM_T::STATUS: ADCTRGn Mask            */

#define PWM_CAPINEN_CAPINENn_Pos         (0)                                               /*!< PWM_T::CAPINEN: CAPINENn Position      */
#define PWM_CAPINEN_CAPINENn_Msk         (0x3ful << PWM_CAPINEN_CAPINENn_Pos)              /*!< PWM_T::CAPINEN: CAPINENn Mask          */

#define PWM_CAPCTL_CAPENn_Pos            (0)                                               /*!< PWM_T::CAPCTL: CAPENn Position         */
#define PWM_CAPCTL_CAPENn_Msk            (0x3ful << PWM_CAPCTL_CAPENn_Pos)                 /*!< PWM_T::CAPCTL: CAPENn Mask             */

#define PWM_CAPCTL_CAPINVn_Pos           (8)                                               /*!< PWM_T::CAPCTL: CAPINVn Position        */
#define PWM_CAPCTL_CAPINVn_Msk           (0x3ful << PWM_CAPCTL_CAPINVn_Pos)                /*!< PWM_T::CAPCTL: CAPINVn Mask            */

#define PWM_CAPCTL_RCRLDENn_Pos          (16)                                              /*!< PWM_T::CAPCTL: RCRLDENn Position       */
#define PWM_CAPCTL_RCRLDENn_Msk          (0x3ful << PWM_CAPCTL_RCRLDENn_Pos)               /*!< PWM_T::CAPCTL: RCRLDENn Mask           */

#define PWM_CAPCTL_FCRLDENn_Pos          (24)                                              /*!< PWM_T::CAPCTL: FCRLDENn Position       */
#define PWM_CAPCTL_FCRLDENn_Msk          (0x3ful << PWM_CAPCTL_FCRLDENn_Pos)               /*!< PWM_T::CAPCTL: FCRLDENn Mask           */

#define PWM_CAPSTS_CRIFOVn_Pos           (0)                                               /*!< PWM_T::CAPSTS: CRIFOVn Position        */
#define PWM_CAPSTS_CRIFOVn_Msk           (0x3ful << PWM_CAPSTS_CRIFOVn_Pos)                /*!< PWM_T::CAPSTS: CRIFOVn Mask            */

#define PWM_CAPSTS_CFIFOVn_Pos           (8)                                               /*!< PWM_T::CAPSTS: CFIFOVn Position        */
#define PWM_CAPSTS_CFIFOVn_Msk           (0x3ful << PWM_CAPSTS_CFIFOVn_Pos)                /*!< PWM_T::CAPSTS: CFIFOVn Mask            */

#define PWM_RCAPDAT0_RCAPDAT_Pos         (0)                                               /*!< PWM_T::RCAPDAT0: RCAPDAT Position      */
#define PWM_RCAPDAT0_RCAPDAT_Msk         (0xfffful << PWM_RCAPDAT0_RCAPDAT_Pos)            /*!< PWM_T::RCAPDAT0: RCAPDAT Mask          */

#define PWM_FCAPDAT0_FCAPDAT_Pos         (0)                                               /*!< PWM_T::FCAPDAT0: FCAPDAT Position      */
#define PWM_FCAPDAT0_FCAPDAT_Msk         (0xfffful << PWM_FCAPDAT0_FCAPDAT_Pos)            /*!< PWM_T::FCAPDAT0: FCAPDAT Mask          */

#define PWM_RCAPDAT1_RCAPDAT_Pos         (0)                                               /*!< PWM_T::RCAPDAT1: RCAPDAT Position      */
#define PWM_RCAPDAT1_RCAPDAT_Msk         (0xfffful << PWM_RCAPDAT1_RCAPDAT_Pos)            /*!< PWM_T::RCAPDAT1: RCAPDAT Mask          */

#define PWM_FCAPDAT1_FCAPDAT_Pos         (0)                                               /*!< PWM_T::FCAPDAT1: FCAPDAT Position      */
#define PWM_FCAPDAT1_FCAPDAT_Msk         (0xfffful << PWM_FCAPDAT1_FCAPDAT_Pos)            /*!< PWM_T::FCAPDAT1: FCAPDAT Mask          */

#define PWM_RCAPDAT2_RCAPDAT_Pos         (0)                                               /*!< PWM_T::RCAPDAT2: RCAPDAT Position      */
#define PWM_RCAPDAT2_RCAPDAT_Msk         (0xfffful << PWM_RCAPDAT2_RCAPDAT_Pos)            /*!< PWM_T::RCAPDAT2: RCAPDAT Mask          */

#define PWM_FCAPDAT2_FCAPDAT_Pos         (0)                                               /*!< PWM_T::FCAPDAT2: FCAPDAT Position      */
#define PWM_FCAPDAT2_FCAPDAT_Msk         (0xfffful << PWM_FCAPDAT2_FCAPDAT_Pos)            /*!< PWM_T::FCAPDAT2: FCAPDAT Mask          */

#define PWM_RCAPDAT3_RCAPDAT_Pos         (0)                                               /*!< PWM_T::RCAPDAT3: RCAPDAT Position      */
#define PWM_RCAPDAT3_RCAPDAT_Msk         (0xfffful << PWM_RCAPDAT3_RCAPDAT_Pos)            /*!< PWM_T::RCAPDAT3: RCAPDAT Mask          */

#define PWM_FCAPDAT3_FCAPDAT_Pos         (0)                                               /*!< PWM_T::FCAPDAT3: FCAPDAT Position      */
#define PWM_FCAPDAT3_FCAPDAT_Msk         (0xfffful << PWM_FCAPDAT3_FCAPDAT_Pos)            /*!< PWM_T::FCAPDAT3: FCAPDAT Mask          */

#define PWM_RCAPDAT4_RCAPDAT_Pos         (0)                                               /*!< PWM_T::RCAPDAT4: RCAPDAT Position      */
#define PWM_RCAPDAT4_RCAPDAT_Msk         (0xfffful << PWM_RCAPDAT4_RCAPDAT_Pos)            /*!< PWM_T::RCAPDAT4: RCAPDAT Mask          */

#define PWM_FCAPDAT4_FCAPDAT_Pos         (0)                                               /*!< PWM_T::FCAPDAT4: FCAPDAT Position      */
#define PWM_FCAPDAT4_FCAPDAT_Msk         (0xfffful << PWM_FCAPDAT4_FCAPDAT_Pos)            /*!< PWM_T::FCAPDAT4: FCAPDAT Mask          */

#define PWM_RCAPDAT5_RCAPDAT_Pos         (0)                                               /*!< PWM_T::RCAPDAT5: RCAPDAT Position      */
#define PWM_RCAPDAT5_RCAPDAT_Msk         (0xfffful << PWM_RCAPDAT5_RCAPDAT_Pos)            /*!< PWM_T::RCAPDAT5: RCAPDAT Mask          */

#define PWM_FCAPDAT5_FCAPDAT_Pos         (0)                                               /*!< PWM_T::FCAPDAT5: FCAPDAT Position      */
#define PWM_FCAPDAT5_FCAPDAT_Msk         (0xfffful << PWM_FCAPDAT5_FCAPDAT_Pos)            /*!< PWM_T::FCAPDAT5: FCAPDAT Mask          */

#define PWM_CAPIEN_CAPRIENn_Pos          (0)                                               /*!< PWM_T::CAPIEN: CAPRIENn Position       */
#define PWM_CAPIEN_CAPRIENn_Msk          (0x3ful << PWM_CAPIEN_CAPRIENn_Pos)               /*!< PWM_T::CAPIEN: CAPRIENn Mask           */

#define PWM_CAPIEN_CAPFIENn_Pos          (8)                                               /*!< PWM_T::CAPIEN: CAPFIENn Position       */
#define PWM_CAPIEN_CAPFIENn_Msk          (0x3ful << PWM_CAPIEN_CAPFIENn_Pos)               /*!< PWM_T::CAPIEN: CAPFIENn Mask           */

#define PWM_CAPIF_CAPRIFn_Pos            (0)                                               /*!< PWM_T::CAPIF: CAPRIFn Position         */
#define PWM_CAPIF_CAPRIFn_Msk            (0x3ful << PWM_CAPIF_CAPRIFn_Pos)                 /*!< PWM_T::CAPIF: CAPRIFn Mask             */

#define PWM_CAPIF_CAPFIFn_Pos            (8)                                               /*!< PWM_T::CAPIF: CAPFIFn Position         */
#define PWM_CAPIF_CAPFIFn_Msk            (0x3ful << PWM_CAPIF_CAPFIFn_Pos)                 /*!< PWM_T::CAPIF: CAPFIFn Mask             */

#define PWM_PBUF0_PBUF_Pos               (0)                                               /*!< PWM_T::PBUF0: PBUF Position            */
#define PWM_PBUF0_PBUF_Msk               (0xfffful << PWM_PBUF0_PBUF_Pos)                  /*!< PWM_T::PBUF0: PBUF Mask                */

#define PWM_PBUF2_PBUF_Pos               (0)                                               /*!< PWM_T::PBUF2: PBUF Position            */
#define PWM_PBUF2_PBUF_Msk               (0xfffful << PWM_PBUF2_PBUF_Pos)                  /*!< PWM_T::PBUF2: PBUF Mask                */

#define PWM_PBUF4_PBUF_Pos               (0)                                               /*!< PWM_T::PBUF4: PBUF Position            */
#define PWM_PBUF4_PBUF_Msk               (0xfffful << PWM_PBUF4_PBUF_Pos)                  /*!< PWM_T::PBUF4: PBUF Mask                */

#define PWM_CMPBUF0_CMPBUF_Pos           (0)                                               /*!< PWM_T::CMPBUF0: CMPBUF Position        */
#define PWM_CMPBUF0_CMPBUF_Msk           (0xfffful << PWM_CMPBUF0_CMPBUF_Pos)              /*!< PWM_T::CMPBUF0: CMPBUF Mask            */

#define PWM_CMPBUF1_CMPBUF_Pos           (0)                                               /*!< PWM_T::CMPBUF1: CMPBUF Position        */
#define PWM_CMPBUF1_CMPBUF_Msk           (0xfffful << PWM_CMPBUF1_CMPBUF_Pos)              /*!< PWM_T::CMPBUF1: CMPBUF Mask            */

#define PWM_CMPBUF2_CMPBUF_Pos           (0)                                               /*!< PWM_T::CMPBUF2: CMPBUF Position        */
#define PWM_CMPBUF2_CMPBUF_Msk           (0xfffful << PWM_CMPBUF2_CMPBUF_Pos)              /*!< PWM_T::CMPBUF2: CMPBUF Mask            */

#define PWM_CMPBUF3_CMPBUF_Pos           (0)                                               /*!< PWM_T::CMPBUF3: CMPBUF Position        */
#define PWM_CMPBUF3_CMPBUF_Msk           (0xfffful << PWM_CMPBUF3_CMPBUF_Pos)              /*!< PWM_T::CMPBUF3: CMPBUF Mask            */

#define PWM_CMPBUF4_CMPBUF_Pos           (0)                                               /*!< PWM_T::CMPBUF4: CMPBUF Position        */
#define PWM_CMPBUF4_CMPBUF_Msk           (0xfffful << PWM_CMPBUF4_CMPBUF_Pos)              /*!< PWM_T::CMPBUF4: CMPBUF Mask            */

#define PWM_CMPBUF5_CMPBUF_Pos           (0)                                               /*!< PWM_T::CMPBUF5: CMPBUF Position        */
#define PWM_CMPBUF5_CMPBUF_Msk           (0xfffful << PWM_CMPBUF5_CMPBUF_Pos)              /*!< PWM_T::CMPBUF5: CMPBUF Mask            */

/**@}*/ /* PWM_CONST */
/**@}*/ /* end of PWM register group */


/*---------------------- Watch Dog Timer Controller -------------------------*/
/**
    @addtogroup WDT Watch Dog Timer Controller(WDT)
    Memory Mapped Structure for WDT Controller
@{ */

typedef struct
{


    /**
     * @var WDT_T::CTL
     * Offset: 0x00  Watchdog Timer Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |WTR       |Clear Watchdog Timer (Write Protect)
     * |        |          |Please refer to open lock sequence to program it.
     * |        |          |Setting this bit will clear the Watchdog timer.
     * |        |          |0 = No effect.
     * |        |          |1 = Reset the contents of the Watchdog timer.
     * |        |          |Note: This bit will be auto cleared after 1 PCLK clock cycle.
     * |[1]     |WTRE      |Watchdog Timer Reset Function Enable Bit (Write Protect)
     * |        |          |Please refer to open lock sequence to program it.
     * |        |          |Setting this bit will enable the Watchdog timer reset function.
     * |        |          |0 = Watchdog timer reset function Disabled.
     * |        |          |1 = Watchdog timer reset function Enabled.
     * |[2]     |WTWKE     |Watchdog Timer Wake-up Function Enable Bit (Write Protect)
     * |        |          |Please refer to open lock sequence to program it.
     * |        |          |0 = Watchdog timer Wake-up CPU function Disabled.
     * |        |          |1 = Wake-up function Enabled so that Watchdog timer time-out can wake up CPU from Power-down mode.
     * |[3]     |WTE       |Watchdog Timer Enable Bit (Write Protect)
     * |        |          |Please refer to open lock sequence to program it.
     * |        |          |0 = Watchdog timer Disabled (this action will reset the internal counter).
     * |        |          |1 = Watchdog timer Enabled.
     * |[6:4]   |WTIS      |Watchdog Timer Interval Selection (Write Protect)
     * |        |          |Please refer to open lock sequence to program it.
     * |        |          |The three bits select the time-out interval for the Watchdog timer. This count is free running counter.
     * |        |          |Please refer to Table 6.11-1.
     * |[9:8]   |WTRDSEL   |Watchdog Timer Reset Delay Selection
     * |        |          |When watchdog timeout happened, software has a time named watchdog reset delay period to clear watchdog timer to prevent watchdog reset happened
     * |        |          |Software can select a suitable value of watchdog reset delay period for different watchdog timeout period.
     * |        |          |00 = Watchdog reset delay period is 1026 watchdog clock.
     * |        |          |01 = Watchdog reset delay period is 130 watchdog clock.
     * |        |          |10 = Watchdog reset delay period is 18 watchdog clock.
     * |        |          |11 = Watchdog reset delay period is 3 watchdog clock.
     * |        |          |Note: This bit will be reset if watchdog reset happened
     * |[31]    |DBGEN     |WDT Debug Mode Enable Control (Write Protect)
     * |        |          |0 = WDT stopped counting if system is in Debug mode.
     * |        |          |1 = WDT still counted even system is in Debug mode.
     * @var WDT_T::INTEN
     * Offset: 0x04  Watchdog Timer Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |WDT_IE    |Watchdog Timer Time-out Interrupt Enable Bit
     * |        |          |0 = Watchdog timer time-out interrupt Disabled.
     * |        |          |1 = Watchdog timer time-out interrupt Enabled.
     * @var WDT_T::STATUS
     * Offset: 0x08  Watchdog Timer Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |WDT_IS    |Watchdog Timer Time-out Interrupt Status
     * |        |          |If the Watchdog timer time-out interrupt is enabled, then the hardware will set this bit to indicate that the Watchdog timer time-out interrupt has occurred
     * |        |          |If the Watchdog timer time-out interrupt is not enabled, then this bit indicates that a time-out period has elapsed.
     * |        |          |0 = Watchdog timer time-out interrupt did not occur.
     * |        |          |1 = Watchdog timer time-out interrupt occurred.
     * |        |          |Note: This bit is read only, but can be cleared by writing 1 to it.
     * |[1]     |WDT_RST_IS|Watchdog Timer Reset Status
     * |        |          |When the Watchdog timer initiates a reset, the hardware will set this bit
     * |        |          |This flag can be read by software to determine the source of reset
     * |        |          |Software is responsible to clear it manually by writing 1 to it
     * |        |          |If WTRE is disabled, then the Watchdog timer has no effect on this bit.
     * |        |          |0 = Watchdog timer reset did not occur.
     * |        |          |1 = Watchdog timer reset occurred.
     * |        |          |Note: This bit is read only, but can be cleared by writing 1 to it.
     * |[2]     |WDT_WAKE_IS|Watchdog Timer Wake-up Status
     * |        |          |If Watchdog timer causes system to wake up from Power-down mode, this bit will be set to 1
     * |        |          |It must be cleared by software with a write 1 to this bit.
     * |        |          |0 = Watchdog timer does not cause system wake-up.
     * |        |          |1 = Wake system up from Power-down mode by Watchdog time-out.
     * |        |          |Note1: When system in Power-down mode and watchdog time-out, hardware will set WDT_WAKE_IS and WDT_IS.
     * |        |          |Note2: After one engine clock, this bit can be cleared by writing 1 to it
     */
    __IO uint32_t CTL;                   /*!< [0x0000] Watchdog Timer Control Register                                  */
    __IO uint32_t INTEN;                   /*!< [0x0004] Watchdog Timer Interrupt Enable Register                         */
    __IO uint32_t STATUS;                   /*!< [0x0008] Watchdog Timer Interrupt Status Register                         */

} WDT_T;

/**
    @addtogroup WDT_CONST WDT Bit Field Definition
    Constant Definitions for WDT Controller
@{ */

#define WDT_CTL_RSTCNT_Pos                  (0)                                               /*!< WDT_T::CTL: WTR Position               */
#define WDT_CTL_RSTCNT_Msk                  (0x1ul << WDT_CTL_RSTCNT_Pos)                        /*!< WDT_T::CTL: WTR Mask                   */

#define WDT_CTL_RSTEN_Pos                 (1)                                               /*!< WDT_T::CTL: WTRE Position              */
#define WDT_CTL_RSTEN_Msk                 (0x1ul << WDT_CTL_RSTEN_Pos)                       /*!< WDT_T::CTL: WTRE Mask                  */

#define WDT_CTL_WKEN_Pos                (2)                                               /*!< WDT_T::CTL: WTWKE Position             */
#define WDT_CTL_WKEN_Msk                (0x1ul << WDT_CTL_WKEN_Pos)                      /*!< WDT_T::CTL: WTWKE Mask                 */

#define WDT_CTL_WDTEN_Pos                  (3)                                               /*!< WDT_T::CTL: WTE Position               */
#define WDT_CTL_WDTEN_Msk                  (0x1ul << WDT_CTL_WDTEN_Pos)                        /*!< WDT_T::CTL: WTE Mask                   */

#define WDT_CTL_WTIS_Pos                 (4)                                               /*!< WDT_T::CTL: WTIS Position              */
#define WDT_CTL_WTIS_Msk                 (0x7ul << WDT_CTL_WTIS_Pos)                       /*!< WDT_T::CTL: WTIS Mask                  */

#define WDT_CTL_WTRDSEL_Pos              (8)                                               /*!< WDT_T::CTL: WTRDSEL Position           */
#define WDT_CTL_WTRDSEL_Msk              (0x3ul << WDT_CTL_WTRDSEL_Pos)                    /*!< WDT_T::CTL: WTRDSEL Mask               */

#define WDT_CTL_DBGEN_Pos                (31)                                              /*!< WDT_T::CTL: DBGEN Position             */
#define WDT_CTL_DBGEN_Msk                (0x1ul << WDT_CTL_DBGEN_Pos)                      /*!< WDT_T::CTL: DBGEN Mask                 */

#define WDT_INTEN_WDTIE_Pos               (0)                                               /*!< WDT_T::INTEN: WDT_IE Position            */
#define WDT_INTEN_WDTIE_Msk               (0x1ul << WDT_INTEN_WDTIE_Pos)                     /*!< WDT_T::INTEN: WDT_IE Mask                */

#define WDT_STATUS_WDTIF_Pos               (0)                                               /*!< WDT_T::STATUS: WDT_IS Position            */
#define WDT_STATUS_WDTIF_Msk               (0x1ul << WDT_STATUS_WDTIF_Pos)                     /*!< WDT_T::STATUS: WDT_IS Mask                */

#define WDT_STATUS_RSTF_Pos           (1)                                               /*!< WDT_T::STATUS: WDT_RST_IS Position        */
#define WDT_STATUS_RSTF_Msk           (0x1ul << WDT_STATUS_RSTF_Pos)                 /*!< WDT_T::STATUS: WDT_RST_IS Mask            */

#define WDT_STATUS_WKF_Pos          (2)                                               /*!< WDT_T::STATUS: WDT_WAKE_IS Position       */
#define WDT_STATUS_WKF_Msk          (0x1ul << WDT_STATUS_WKF_Pos)                /*!< WDT_T::STATUS: WDT_WAKE_IS Mask           */

/**@}*/ /* WDT_CONST */
/**@}*/ /* end of WDT register group */


/*---------------------- Window Watchdog Timer -------------------------*/
/**
    @addtogroup WWDT Window Watchdog Timer(WWDT)
    Memory Mapped Structure for WWDT Controller
@{ */

typedef struct
{


    /**
     * @var WWDT_T::RLDCNT
     * Offset: 0x00  Window Watchdog Timer Reload Counter Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |WWDT_RLD  |Window Watchdog Timer Reload Counter Register
     * |        |          |Writing 0x00005AA5 to this register will reload the Window Watchdog Timer counter value to 0x3F.
     * |        |          |Note: This register can only be written when WWDT counter value between 0 and WINCMP, otherwise WWDT will generate RESET signal.
     * @var WWDT_T::CTL
     * Offset: 0x04  Window Watchdog Timer Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |WWDTEN    |Window Watchdog Enable Bit
     * |        |          |Set this bit to enable Window Watchdog timer.
     * |        |          |0 = Window Watchdog timer function Disabled.
     * |        |          |1 = Window Watchdog timer function Enabled.
     * |[11:8]  |PERIODSEL |WWDT Pre-scale Period Select
     * |        |          |These three bits select the pre-scale for the WWDT counter period.
     * |        |          |Please refer toTable 6.12-1 WWDT Prescaler Value Selection.
     * |[21:16] |WINCMP    |WWDT Window Compare Bits
     * |        |          |Set this register to adjust the valid reload window.
     * |        |          |Note: WWDTRLD register can only be written when WWDT counter value between 0 and WINCMP, otherwise WWDT will generate RESET signal.
     * |[31]    |DBGEN     |WWDT Debug Enable Bit
     * |        |          |0 = WWDT stopped count if system is in Debug mode.
     * |        |          |1 = WWDT still counted even system is in Debug mode.
     * @var WWDT_T::INTEN
     * Offset: 0x08  Window Watchdog Timer Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |WWDTIE    |WWDT Interrupt Enable Bit
     * |        |          |Setting this bit will enable the Window Watchdog timer interrupt function.
     * |        |          |0 = Watchdog timer interrupt function Disabled.
     * |        |          |1 = Watchdog timer interrupt function Enabled.
     * @var WWDT_T::STATUS
     * Offset: 0x0C  Window Watchdog Timer Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |WWDTIF    |WWDT Compare Match Interrupt Flag
     * |        |          |When WWCMP matches the WWDT counter, this bit is set to 1
     * |        |          |This bit can be cleared by writing '1' to it.
     * |[1]     |WWDTRF    |WWDT Reset Flag
     * |        |          |When the WWDT counter down counts to 0 or writes WWDTRLD during WWDT counter larger than WINCMP, chip will be reset and this bit is set to 1
     * |        |          |This bit can be cleared by writing '1' to it.
     * @var WWDT_T::CNT
     * Offset: 0x10  Window Watchdog Timer Counter Value Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |WWDT_VAL  |WWDT Counter Value
     * |        |          |This register reflects the current counter value of window watchdog.
     */
    __O  uint32_t RLDCNT;                   /*!< [0x0000] Window Watchdog Timer Reload Counter Register                    */
    __IO uint32_t CTL;                    /*!< [0x0004] Window Watchdog Timer Control Register                           */
    __IO uint32_t INTEN;                   /*!< [0x0008] Window Watchdog Timer Interrupt Enable Register                  */
    __IO uint32_t STATUS;                   /*!< [0x000c] Window Watchdog Timer Status Register                            */
    __I  uint32_t CNT;                   /*!< [0x0010] Window Watchdog Timer Counter Value Register                     */

} WWDT_T;

/**
    @addtogroup WWDT_CONST WWDT Bit Field Definition
    Constant Definitions for WWDT Controller
@{ */

#define WWDT_RLDCNT_RLDCNT_Pos            (0)                                               /*!< WWDT_T::RLDCNT: WWDT_RLD Position         */
#define WWDT_RLDCNT_RLDCNT_Msk            (0xfffffffful << WWDT_RLDCNT_RLDCNT_Pos)           /*!< WWDT_T::RLDCNT: WWDT_RLD Mask             */

#define WWDT_CTL_WWDTEN_Pos               (0)                                               /*!< WWDT_T::CTL: WWDTEN Position            */
#define WWDT_CTL_WWDTEN_Msk               (0x1ul << WWDT_CTL_WWDTEN_Pos)                     /*!< WWDT_T::CTL: WWDTEN Mask                */

#define WWDT_CTL_PERIODSEL_Pos            (8)                                               /*!< WWDT_T::CTL: PERIODSEL Position         */
#define WWDT_CTL_PERIODSEL_Msk            (0xful << WWDT_CTL_PERIODSEL_Pos)                  /*!< WWDT_T::CTL: PERIODSEL Mask             */

#define WWDT_CTL_WINCMP_Pos               (16)                                              /*!< WWDT_T::CTL: WINCMP Position            */
#define WWDT_CTL_WINCMP_Msk               (0x3ful << WWDT_CTL_WINCMP_Pos)                    /*!< WWDT_T::CTL: WINCMP Mask                */

#define WWDT_CTL_DBGEN_Pos                (31)                                              /*!< WWDT_T::CTL: DBGEN Position             */
#define WWDT_CTL_DBGEN_Msk                (0x1ul << WWDT_CTL_DBGEN_Pos)                      /*!< WWDT_T::CTL: DBGEN Mask                 */

#define WWDT_INTEN_WWDTIE_Pos              (0)                                               /*!< WWDT_T::INTEN: WWDTIE Position           */
#define WWDT_INTEN_WWDTIE_Msk              (0x1ul << WWDT_INTEN_WWDTIE_Pos)                    /*!< WWDT_T::INTEN: WWDTIE Mask               */

#define WWDT_STATUS_WWDTIF_Pos              (0)                                               /*!< WWDT_T::STATUS: WWDTIF Position           */
#define WWDT_STATUS_WWDTIF_Msk              (0x1ul << WWDT_STATUS_WWDTIF_Pos)                    /*!< WWDT_T::STATUS: WWDTIF Mask               */

#define WWDT_STATUS_WWDTRF_Pos              (1)                                               /*!< WWDT_T::STATUS: WWDTRF Position           */
#define WWDT_STATUS_WWDTRF_Msk              (0x1ul << WWDT_STATUS_WWDTRF_Pos)                    /*!< WWDT_T::STATUS: WWDTRF Mask               */

#define WWDT_CNT_WWDT_CNTDAT_Pos            (0)                                               /*!< WWDT_T::CNT: WWDT_VAL Position         */
#define WWDT_CNT_WWDT_CNTDAT_Msk            (0x3ful << WWDT_CNT_WWDT_CNTDAT_Pos)                 /*!< WWDT_T::CNT: WWDT_VAL Mask             */

/**@}*/ /* WWDT_CONST */
/**@}*/ /* end of WWDT register group */


/*---------------------- Real Time Clock Controller -------------------------*/
/**
    @addtogroup RTC Real Time Clock Controller(RTC)
    Memory Mapped Structure for RTC Controller
@{ */

typedef struct
{


    /**
     * @var RTC_T::INIT
     * Offset: 0x00  RTC Initiation Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |INIT_ACTIVE|RTC Active Status (Read Only)
     * |        |          |0 = RTC is at reset state.
     * |        |          |1 = RTC is at normal active state.
     * |[31:1]  |INIT      |RTC Initiation
     * |        |          |When RTC block is powered on, RTC is at reset state
     * |        |          |User has to write a number (0x a5eb1357) to INIT to make RTC leaving reset state
     * |        |          |Once the INIT is written as 0xa5eb1357, the RTC will be in un-reset state permanently.
     * |        |          |The INIT is a write-only field and read value will be always 0.
     * @var RTC_T::RWEN
     * Offset: 0x04  RTC Access Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RWEN      |RTC Register Access Enable Password (Write Only)
     * |        |          |Writing 0xA965 to this register will enable RTC access and keep 1024 RTC clock.
     * |        |          |Writing other value will clear RWENF.
     * |[16]    |RWENF     |RTC Register Access Enable Flag (Read Only)
     * |        |          |0 = RTC register read/write Disabled.
     * |        |          |1 = RTC register read/write Enabled.
     * |        |          |This bit will be set after RTC_RWEN[15:0] register is load a 0xA965, and be cleared automatically after 1024 RTC clock .
     * |        |          |Note: RWENF will be mask to 0 during RTCBUSY = 1, and first turn on RTCCKEN (CLK_APBCLK[1]) also.
     * |[24]    |RTCBUSY   |RTC Write Busy Flag
     * |        |          |0: RTC write access enable
     * |        |          |1: RTC write access disable , RTC under Busy Status.
     * |        |          |Note: BUSY By Exceed RTC IP Processing Write Counter Capacity ( 6 counts Per 1120 PCLK cycles) .
     * @var RTC_T::FREQADJ
     * Offset: 0x08  RTC Frequency Compensation Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[21:0]  |FREQADJ   |Frequency Compensation Register
     * |        |          |FREQADJ = 32768 * 0x200000 / (LXT period).
     * |        |          |LXT period: the clock period (Hz) of LXT.
     * @var RTC_T::TIME
     * Offset: 0x0C  RTC Time Loading Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |SEC       |1-Sec Time Digit (0~9)
     * |[6:4]   |TENSEC    |10-Sec Time Digit (0~5)
     * |[11:8]  |MIN       |1-Min Time Digit (0~9)
     * |[14:12] |TENMIN    |10-Min Time Digit (0~5)
     * |[19:16] |HR        |1-Hour Time Digit (0~9)
     * |[21:20] |TENHR     |10-hour Time Digit (0~2)
     * |        |          |When RTC runs as 12-hour time scale mode, RTC_TIME[21] (the high bit of TENHR[1:0]) means AM/PM indication
     * |        |          |(If RTC_TIME[21] is 1, it indicates PM time message.) the high bit of TENHR (RTC_TIME[21]) means AM/PM indication.
     * @var RTC_T::CAL
     * Offset: 0x10  RTC Calendar Loading Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |DAY       |1-Day Calendar Digit (0~9)
     * |[5:4]   |TENDAY    |10-Day Calendar Digit (0~3)
     * |[11:8]  |MON       |1-Month Calendar Digit (0~9)
     * |[12]    |TENMON    |10-Month Calendar Digit (0~1)
     * |[19:16] |YEAR      |1-Year Calendar Digit (0~9)
     * |[23:20] |TENYEAR   |10-Year Calendar Digit (0~9)
     * @var RTC_T::CLKFMT
     * Offset: 0x14  RTC Time Scale Selection Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |24HEN     |24-hour / 12-hour Time Scale Selection
     * |        |          |Indicates that RTC_TIME and RTC_TALM are in 24-hour time scale or 12-hour time scale
     * |        |          |0 = 12-hour time scale with AM and PM indication selected.
     * |        |          |1 = 24-hour time scale selected.
     * @var RTC_T::WEEKDAY
     * Offset: 0x18  RTC Day of the Week Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |WEEKDAY   |Day of the Week Register
     * |        |          |000 = Sunday.
     * |        |          |001 = Monday.
     * |        |          |010 = Tuesday.
     * |        |          |011 = Wednesday.
     * |        |          |100 = Thursday.
     * |        |          |101 = Friday.
     * |        |          |110 = Saturday.
     * |        |          |111 = Reserved.
     * @var RTC_T::TALM
     * Offset: 0x1C  RTC Time Alarm Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |SEC       |1-Sec Time Digit of Alarm Setting (0~9)
     * |[6:4]   |TENSEC    |10-Sec Time Digit of Alarm Setting (0~5)
     * |[11:8]  |MIN       |1-Min Time Digit of Alarm Setting (0~9)
     * |[14:12] |TENMIN    |10-Min Time Digit of Alarm Setting (0~5)
     * |[19:16] |HR        |1-Hour Time Digit of Alarm Setting (0~9)
     * |[21:20] |TENHR     |10-hour Time Digit of Alarm Setting (0~2)
     * |        |          |When RTC runs as 12-hour time scale mode, RTC_TIME[21] (the high bit of TENHR[1:0]) means AM/PM indication
     * |        |          |(If RTC_TIME[21] is 1, it indicates PM time message.)the high bit of TENHR (RTC_TIME[21]) means AM/PM indication.
     * @var RTC_T::CALM
     * Offset: 0x20  RTC Calendar Alarm Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |DAY       |1-Day Calendar Digit of Alarm Setting (0~9)
     * |[5:4]   |TENDAY    |10-Day Calendar Digit of Alarm Setting (0~3)
     * |[11:8]  |MON       |1-Month Calendar Digit of Alarm Setting (0~9)
     * |[12]    |TENMON    |10-Month Calendar Digit of Alarm Setting (0~1)
     * |[19:16] |YEAR      |1-Year Calendar Digit of Alarm Setting (0~9)
     * |[23:20] |TENYEAR   |10-Year Calendar Digit of Alarm Setting (0~9)
     * @var RTC_T::LEAPYEAR
     * Offset: 0x24  RTC Leap Year Indicator Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |LEAPYEAR  |Leap Year Indication Register (Read Only)
     * |        |          |0 = This year is not a leap year.
     * |        |          |1 = This year is leap year.
     * @var RTC_T::INTEN
     * Offset: 0x28  RTC Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ALMIEN    |Alarm Interrupt Enable Bit
     * |        |          |0 = RTC Alarm interrupt Disabled.
     * |        |          |1 = RTC Alarm interrupt Enabled.
     * |[1]     |TICKIEN   |Time Tick Interrupt Enable Bit
     * |        |          |0 = RTC Time Tick interrupt Disabled.
     * |        |          |1 = RTC Time Tick interrupt Enabled.
     * |[2]     |SNPDIEN   |Snoop Detection Interrupt Enable Bit
     * |        |          |0 = Snoop detected interrupt Disabled.
     * |        |          |1 = Snoop detected interrupt Enabled.
     * @var RTC_T::INTSTS
     * Offset: 0x2C  RTC Interrupt Indicator Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ALMIF     |RTC Alarm Interrupt Flag
     * |        |          |When RTC time counters RTC_TIME and RTC_CAL match the alarm setting time registers RTC_TALM and RTC_CALM, this bit will be set to 1 and an interrupt will be generated if RTC Alarm Interrupt enabled ALMIEN (RTC_INTEN[0]) is set to 1
     * |        |          |Chip will be waken up if RTC Alarm Interrupt is enabled when chip is at Power-down mode.
     * |        |          |0 = Alarm condition is not matched.
     * |        |          |1 = Alarm condition is matched.
     * |        |          |Note: Write 1 to clear this bit.
     * |[1]     |TICKIF    |RTC Time Tick Interrupt Flag
     * |        |          |When RTC time tick happened, this bit will be set to 1 and an interrupt will be generated if RTC Tick Interrupt enabled TICKIEN (RTC_INTEN[1]) is set to 1
     * |        |          |Chip will also be waken up if RTC Tick Interrupt is enabled and this bit is set to 1 when chip is running at Power-down mode.
     * |        |          |0 = Tick condition does not occur.
     * |        |          |1 = Tick condition occur.
     * |        |          |Note: Write 1 to clear to clear this bit.
     * |[2]     |SNPDIF    |Snoop Detect Interrupt Flag
     * |        |          |When tamper pin transition event is detected, this bit is set to 1 and an interrupt is generated if Snoop Detection Interrupt enabled SNPDIEN (RTC_INTEN[2]) is set to1
     * |        |          |Chip will be waken up from Power-down mode if spare register snooper detect interrupt is enabled.
     * |        |          |0 = No snoop event is detected.
     * |        |          |1 = Snoop event is detected.
     * |        |          |Note: Write 1 to clear this bit.
     * @var RTC_T::TICK
     * Offset: 0x30  RTC Time Tick Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |TICK      |Time Tick Register
     * |        |          |These bits are used to select RTC time tick period for Periodic Time Tick Interrupt request.
     * |        |          |000 = Time tick is 1 second.
     * |        |          |001 = Time tick is 1/2 second.
     * |        |          |010 = Time tick is 1/4 second.
     * |        |          |011 = Time tick is 1/8 second.
     * |        |          |100 = Time tick is 1/16 second.
     * |        |          |101 = Time tick is 1/32 second.
     * |        |          |110 = Time tick is 1/64 second.
     * |        |          |111 = Time tick is 1/128 second.
     * |        |          |Note: This register can be read back after the RTC register access enable bit RWENF (RTC_RWEN[16]) is active.
     * @var RTC_T::TAMSK
     * Offset: 0x34  RTC Time Alarm Mask Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |MSEC      |Mask 1-Sec Time Digit of Alarm Setting (0~9)
     * |[1]     |MTENSEC   |Mask 10-Sec Time Digit of Alarm Setting (0~5)
     * |[2]     |MMIN      |Mask 1-Min Time Digit of Alarm Setting (0~9)
     * |[3]     |MTENMIN   |Mask 10-Min Time Digit of Alarm Setting (0~5)
     * |[4]     |MHR       |Mask 1-Hour Time Digit of Alarm Setting (0~9)
     * |[5]     |MTENHR    |Mask 10-Hour Time Digit of Alarm Setting (0~2)
     * @var RTC_T::CAMSK
     * Offset: 0x38  RTC Calendar Alarm Mask Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |MDAY      |Mask 1-Day Calendar Digit of Alarm Setting (0~9)
     * |[1]     |MTENDAY   |Mask 10-Day Calendar Digit of Alarm Setting (0~3)
     * |[2]     |MMON      |Mask 1-Month Calendar Digit of Alarm Setting (0~9)
     * |[3]     |MTENMON   |Mask 10-Month Calendar Digit of Alarm Setting (0~1)
     * |[4]     |MYEAR     |Mask 1-Year Calendar Digit of Alarm Setting (0~9)
     * |[5]     |MTENYEAR  |Mask 10-Year Calendar Digit of Alarm Setting (0~9)
     * @var RTC_T::SPRCTL
     * Offset: 0x3C  RTC Spare Functional Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SNPDEN    |Snoop Detection Enable Bit
     * |        |          |0 = TAMPER pin detection is Disabled.
     * |        |          |1 = TAMPER pin detection is Enabled.
     * |[1]     |SNPTYPE0  |Snoop Detection Level
     * |        |          |This bit controls TAMPER detect event is rising edge or falling edge.
     * |        |          |0 = Rising edge detection.
     * |        |          |1 = Falling edge detection.
     * |[2]     |SPRRWEN   |Spare Register Enable Bit
     * |        |          |0 = Spare register is Disabled.
     * |        |          |1 = Spare register is Enabled.
     * |        |          |Note: When spare register is disabled, RTC_SPR0 ~ RTC_SPR4 cannot be accessed
     * |        |          |Did not change the content of the spare register, but read data all 0.
     * |[5]     |SPRCSTS   |SPR Clear Flag
     * |        |          |This bit indicates if the RTC_SPR0 ~RTC_SPR4 content is cleared when specify snoop event is detected.
     * |        |          |0 = Spare register content is not cleared.
     * |        |          |1 = Spare register content is cleared.
     * |        |          |Writes 1 to clear this bit.
     * @var RTC_T::SPR
     * Offset: 0x40 ~ 0x50 RTC Spare Register 0 ~ 4
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |SPARE     |Spare Register
     * |        |          |This field is used to store back-up information defined by user.
     * |        |          |This field will be cleared by hardware automatically once a snooper pin event is detected.
     * |        |          |Before storing back-up information in to RTC_SPRx register, user should write 0xA965 to RTC_RWEN[15:0] to make sure register read/write enable bit REWNF (RTC_RWEN[16]) is enabled.
     * @var RTC_T::LXTCTL
     * Offset: 0x100  RTC 32.768 kHz Oscillator Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |LXT_TYPE  |LXT TYPE Selection
     * |        |          |0 = Crystal type ( Crystal connect to X32KI with X32KO).
     * |        |          |1 = Oscillator type ( LXT source from X32KI PIN , X32KO as GPIO).
     * @var RTC_T::LXTOCTL
     * Offset: 0x104  X32KO Pin Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |OPMODE    |GPF0 Operation Mode
     * |        |          |00 = X32KO (PF.6) is input only mode, without pull-up resistor.
     * |        |          |01 = X32KO (PF.6) is output push pull mode.
     * |        |          |10 = X32KO (PF.6) is open drain mode.
     * |        |          |11 = X32KO (PF.6) is input only mode with internal pull up.
     * |[2]     |DOUT      |IO Output Data
     * |        |          |0 = X32KO (PF.6) output low.
     * |        |          |1 = X32KO (PF.6) output high.
     * |[3]     |CTLSEL    |IO Pin State Backup Selection
     * |        |          |When low speed 32 kHz oscillator is disabled, X32KO (PF.6) pin can be used as GPIO function
     * |        |          |User can program CTLSEL bit to decide X32KO (PF.6) I/O function is controlled by system power domain GPIO module or VBAT power domain RTC_LXTOCTL control register.
     * |        |          |0 = X32KO (PF.6) pin I/O function is controlled by GPIO module
     * |        |          |It becomes floating when system power is turned off.
     * |        |          |1 = X32KO (PF.6) pin I/O function is controlled by VBAT power domain, X32KO (PF.6) pin function and I/O status are controlled by OPMODE[1:0] and DOUT after CTLSEL it set to 1
     * |        |          |I/O pin keeps the previous state after system power is turned off.
     * |        |          |Note:CTLSEL (this bit) will automatically be set by hardware to 1 when system power is off and RTC Active Status = 1.
     * @var RTC_T::LXTICTL
     * Offset: 0x108  X32KI Pin Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |OPMODE    |IO Operation Mode
     * |        |          |00 = X32KI (PF.7) is input only mode, without pull-up resistor.
     * |        |          |01 = X32KI (PF.7) is output push pull mode.
     * |        |          |10 = X32KI (PF.7) is open drain mode.
     * |        |          |11 = X32KI (PF.7) is input only mode with internal pull up.
     * |[2]     |DOUT      |IO Output Data
     * |        |          |0 = X32KI (PF.7) output low.
     * |        |          |1 = X32KI (PF.7) output high.
     * |[3]     |CTLSEL    |IO Pin State Backup Selection
     * |        |          |When low speed 32 kHz oscillator is disabled, X32KI (PF.7) pin can be used as GPIO function
     * |        |          |User can program CTLSEL bit to decide X32KI (PF.7) I/O function is controlled by system power domain GPIO module or VBAT power domain RTC_LXTICTL control register.
     * |        |          |0 = X32KI (PF.7) pin I/O function is controlled by GPIO module
     * |        |          |It becomes floating state when system power is turned off.
     * |        |          |1 = X32KI (PF.7) pin I/O function is controlled by VBAT power domain, X32KI (PF.7) pin function and I/O status are controlled by OPMODE[1:0] and DOUT after CTLSEL it set to 1
     * |        |          |I/O pin keeps the previous state after system power is turned off.
     * |        |          |Note:CTLSEL (this bit) will automatically be set by hardware to 1 when system power is off and RTC Active Status = 1.
     * @var RTC_T::TAMPCTL
     * Offset: 0x10C  TAMPER Pin Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |OPMODE    |IO Operation Mode
     * |        |          |00 = TAMPER (LQFP64:PB.13/LQFP48:PA.9/QFN32:PB.8) is input only mode, without pull-up resistor.
     * |        |          |01 = TAMPER (LQFP64:PB.13/LQFP48:PA.9/QFN32:PB.8) is output push pull mode.
     * |        |          |10 = TAMPER (LQFP64:PB.13/LQFP48:PA.9/QFN32:PB.8) is open drain mode.
     * |        |          |11 = TAMPER (LQFP64:PB.13/LQFP48:PA.9/QFN32:PB.8) is input only mode with internal pull up.
     * |[2]     |DOUT      |IO Output Data
     * |        |          |0 = TAMPER (LQFP64:PB.13/LQFP48:PA.9/QFN32:PB.8) output low.
     * |        |          |1 = TAMPER (LQFP64:PB.13/LQFP48:PA.9/QFN32:PB.8) output high.
     * |[3]     |CTLSEL    |IO Pin State Backup Selection
     * |        |          |When tamper function is disabled, TAMPER pin can be used as GPIO function
     * |        |          |User can program CTLSEL bit to decide (LQFP64:PB.13/LQFP48:PA.9/QFN32:PB.8) I/O function is controlled by system power domain GPIO module or VBAT power domain RTC_TAMPCTL control register.
     * |        |          |0 =TAMPER (LQFP64:PB.13/LQFP48:PA.9/QFN32:PB.8) I/O function is controlled by GPIO module
     * |        |          |It becomes floating state when system power is turned off.
     * |        |          |1 =TAMPER (LQFP64:PB.13/LQFP48:PA.9/QFN32:PB.8) I/O function is controlled by VBAT power domain
     * |        |          |LQFP64:PB.13/LQFP48:PA.9/QFN32:PB.8 function and I/O status are controlled by OPMODE[1:0] and DOUT after CTLSEL it set to 1
     * |        |          |I/O pin state keeps previous state after system power is turned off.
     * |        |          |Note:CTLSEL (this bit) will automatically be set by hardware to 1 when system power is off and RTC Active Status = 1.
     * @var RTC_T::MISCCTL
     * Offset: 0x1F0  Miscellaneous Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[13:12] |GAINSEL   |LXT Gain Selection
     * |        |          |00 = Gain 0.
     * |        |          |01 = Gain 1.
     * |        |          |10 = Gain 2.
     * |        |          |11 = Gain 3.
     */
    __IO uint32_t INIT;                  /*!< [0x0000] RTC Initiation Register                                          */
    __IO uint32_t RWEN;                  /*!< [0x0004] RTC Access Enable Register                                       */
    __IO uint32_t FREQADJ;               /*!< [0x0008] RTC Frequency Compensation Register                              */
    __IO uint32_t TIME;                  /*!< [0x000c] RTC Time Loading Register                                        */
    __IO uint32_t CAL;                   /*!< [0x0010] RTC Calendar Loading Register                                    */
    __IO uint32_t CLKFMT;                /*!< [0x0014] RTC Time Scale Selection Register                                */
    __IO uint32_t WEEKDAY;               /*!< [0x0018] RTC Day of the Week Register                                     */
    __IO uint32_t TALM;                  /*!< [0x001c] RTC Time Alarm Register                                          */
    __IO uint32_t CALM;                  /*!< [0x0020] RTC Calendar Alarm Register                                      */
    __I  uint32_t LEAPYEAR;              /*!< [0x0024] RTC Leap Year Indicator Register                                 */
    __IO uint32_t INTEN;                 /*!< [0x0028] RTC Interrupt Enable Register                                    */
    __IO uint32_t INTSTS;                /*!< [0x002c] RTC Interrupt Indicator Register                                 */
    __IO uint32_t TICK;                  /*!< [0x0030] RTC Time Tick Register                                           */
    __IO uint32_t TAMSK;                 /*!< [0x0034] RTC Time Alarm Mask Register                                     */
    __IO uint32_t CAMSK;                 /*!< [0x0038] RTC Calendar Alarm Mask Register                                 */
    __IO uint32_t SPRCTL;                /*!< [0x003c] RTC Spare Functional Control Register                            */
    __IO uint32_t SPR[5];                /*!< [0x0040] ~ [0x0050] RTC Spare Register 0 ~ 4                              */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE0[43];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t LXTCTL;                /*!< [0x0100] RTC 32.768 kHz Oscillator Control Register                       */
    __IO uint32_t LXTOCTL;               /*!< [0x0104] X32KO Pin Control Register                                       */
    __IO uint32_t LXTICTL;               /*!< [0x0108] X32KI Pin Control Register                                       */
    __IO uint32_t TAMPCTL;               /*!< [0x010c] TAMPER Pin Control Register                                      */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE1[56];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t MISCCTL;               /*!< [0x01F0] Miscellaneous Control Register                                   */
} RTC_T;

/**
    @addtogroup RTC_CONST RTC Bit Field Definition
    Constant Definitions for RTC Controller
@{ */

#define RTC_INIT_INIT_ACTIVE_Pos         (0)                                               /*!< RTC_T::INIT: INIT_ACTIVE Position      */
#define RTC_INIT_INIT_ACTIVE_Msk         (0x1ul << RTC_INIT_INIT_ACTIVE_Pos)               /*!< RTC_T::INIT: INIT_ACTIVE Mask          */

#define RTC_INIT_INIT_Pos                (1)                                               /*!< RTC_T::INIT: INIT Position             */
#define RTC_INIT_INIT_Msk                (0x7ffffffful << RTC_INIT_INIT_Pos)               /*!< RTC_T::INIT: INIT Mask                 */

#define RTC_RWEN_RWEN_Pos                (0)                                               /*!< RTC_T::RWEN: RWEN Position             */
#define RTC_RWEN_RWEN_Msk                (0xfffful << RTC_RWEN_RWEN_Pos)                   /*!< RTC_T::RWEN: RWEN Mask                 */

#define RTC_RWEN_RWENF_Pos               (16)                                              /*!< RTC_T::RWEN: RWENF Position            */
#define RTC_RWEN_RWENF_Msk               (0x1ul << RTC_RWEN_RWENF_Pos)                     /*!< RTC_T::RWEN: RWENF Mask                */

#define RTC_RWEN_RTCBUSY_Pos             (24)                                              /*!< RTC_T::RWEN: RTCBUSY Position          */
#define RTC_RWEN_RTCBUSY_Msk             (0x1ul << RTC_RWEN_RTCBUSY_Pos)                   /*!< RTC_T::RWEN: RTCBUSY Mask              */

#define RTC_FREQADJ_FREQADJ_Pos          (0)                                               /*!< RTC_T::FREQADJ: FREQADJ Position               */
#define RTC_FREQADJ_FREQADJ_Msk          (0x3ffffful << RTC_FCR_FCR_Pos)                   /*!< RTC_T::FREQADJ: FREQADJ Mask                   */

#define RTC_TIME_SEC_Pos                 (0)                                               /*!< RTC_T::TIME: SEC Position              */
#define RTC_TIME_SEC_Msk                 (0xful << RTC_TIME_SEC_Pos)                       /*!< RTC_T::TIME: SEC Mask                  */

#define RTC_TIME_TENSEC_Pos              (4)                                               /*!< RTC_T::TIME: TENSEC Position           */
#define RTC_TIME_TENSEC_Msk              (0x7ul << RTC_TIME_TENSEC_Pos)                    /*!< RTC_T::TIME: TENSEC Mask               */

#define RTC_TIME_MIN_Pos                 (8)                                               /*!< RTC_T::TIME: MIN Position              */
#define RTC_TIME_MIN_Msk                 (0xful << RTC_TIME_MIN_Pos)                       /*!< RTC_T::TIME: MIN Mask                  */

#define RTC_TIME_TENMIN_Pos              (12)                                              /*!< RTC_T::TIME: TENMIN Position           */
#define RTC_TIME_TENMIN_Msk              (0x7ul << RTC_TIME_TENMIN_Pos)                    /*!< RTC_T::TIME: TENMIN Mask               */

#define RTC_TIME_HR_Pos                  (16)                                              /*!< RTC_T::TIME: HR Position               */
#define RTC_TIME_HR_Msk                  (0xful << RTC_TIME_HR_Pos)                        /*!< RTC_T::TIME: HR Mask                   */

#define RTC_TIME_TENHR_Pos               (20)                                              /*!< RTC_T::TIME: TENHR Position            */
#define RTC_TIME_TENHR_Msk               (0x3ul << RTC_TIME_TENHR_Pos)                     /*!< RTC_T::TIME: TENHR Mask                */

#define RTC_CAL_DAY_Pos                  (0)                                               /*!< RTC_T::CAL: DAY Position               */
#define RTC_CAL_DAY_Msk                  (0xful << RTC_CAL_DAY_Pos)                        /*!< RTC_T::CAL: DAY Mask                   */

#define RTC_CAL_TENDAY_Pos               (4)                                               /*!< RTC_T::CAL: TENDAY Position            */
#define RTC_CAL_TENDAY_Msk               (0x3ul << RTC_CAL_TENDAY_Pos)                     /*!< RTC_T::CAL: TENDAY Mask                */

#define RTC_CAL_MON_Pos                  (8)                                               /*!< RTC_T::CAL: MON Position               */
#define RTC_CAL_MON_Msk                  (0xful << RTC_CAL_MON_Pos)                        /*!< RTC_T::CAL: MON Mask                   */

#define RTC_CAL_TENMON_Pos               (12)                                              /*!< RTC_T::CAL: TENMON Position            */
#define RTC_CAL_TENMON_Msk               (0x1ul << RTC_CAL_TENMON_Pos)                     /*!< RTC_T::CAL: TENMON Mask                */

#define RTC_CAL_YEAR_Pos                 (16)                                              /*!< RTC_T::CAL: YEAR Position              */
#define RTC_CAL_YEAR_Msk                 (0xful << RTC_CAL_YEAR_Pos)                       /*!< RTC_T::CAL: YEAR Mask                  */

#define RTC_CAL_TENYEAR_Pos              (20)                                              /*!< RTC_T::CAL: TENYEAR Position           */
#define RTC_CAL_TENYEAR_Msk              (0xful << RTC_CAL_TENYEAR_Pos)                    /*!< RTC_T::CAL: TENYEAR Mask               */

#define RTC_CLKFMT_24HEN_Pos             (0)                                               /*!< RTC_T::CLKFMT: 24HEN Position          */
#define RTC_CLKFMT_24HEN_Msk             (0x1ul << RTC_CLKFMT_24HEN_Pos)                   /*!< RTC_T::CLKFMT: 24HEN Mask              */

#define RTC_WEEKDAY_WEEKDAY_Pos          (0)                                               /*!< RTC_T::WEEKDAY: WEEKDAY Position       */
#define RTC_WEEKDAY_WEEKDAY_Msk          (0x7ul << RTC_WEEKDAY_WEEKDAY_Pos)                /*!< RTC_T::WEEKDAY: WEEKDAY Mask           */

#define RTC_TALM_SEC_Pos                 (0)                                               /*!< RTC_T::TALM: SEC Position              */
#define RTC_TALM_SEC_Msk                 (0xful << RTC_TALM_SEC_Pos)                       /*!< RTC_T::TALM: SEC Mask                  */

#define RTC_TALM_TENSEC_Pos              (4)                                               /*!< RTC_T::TALM: TENSEC Position           */
#define RTC_TALM_TENSEC_Msk              (0x7ul << RTC_TALM_TENSEC_Pos)                    /*!< RTC_T::TALM: TENSEC Mask               */

#define RTC_TALM_MIN_Pos                 (8)                                               /*!< RTC_T::TALM: MIN Position              */
#define RTC_TALM_MIN_Msk                 (0xful << RTC_TALM_MIN_Pos)                       /*!< RTC_T::TALM: MIN Mask                  */

#define RTC_TALM_TENMIN_Pos              (12)                                              /*!< RTC_T::TALM: TENMIN Position           */
#define RTC_TALM_TENMIN_Msk              (0x7ul << RTC_TALM_TENMIN_Pos)                    /*!< RTC_T::TALM: TENMIN Mask               */

#define RTC_TALM_HR_Pos                  (16)                                              /*!< RTC_T::TALM: HR Position               */
#define RTC_TALM_HR_Msk                  (0xful << RTC_TALM_HR_Pos)                        /*!< RTC_T::TALM: HR Mask                   */

#define RTC_TALM_TENHR_Pos               (20)                                              /*!< RTC_T::TALM: TENHR Position            */
#define RTC_TALM_TENHR_Msk               (0x3ul << RTC_TALM_TENHR_Pos)                     /*!< RTC_T::TALM: TENHR Mask                */

#define RTC_CALM_DAY_Pos                 (0)                                               /*!< RTC_T::CALM: DAY Position              */
#define RTC_CALM_DAY_Msk                 (0xful << RTC_CALM_DAY_Pos)                       /*!< RTC_T::CALM: DAY Mask                  */

#define RTC_CALM_TENDAY_Pos              (4)                                               /*!< RTC_T::CALM: TENDAY Position           */
#define RTC_CALM_TENDAY_Msk              (0x3ul << RTC_CALM_TENDAY_Pos)                    /*!< RTC_T::CALM: TENDAY Mask               */

#define RTC_CALM_MON_Pos                 (8)                                               /*!< RTC_T::CALM: MON Position              */
#define RTC_CALM_MON_Msk                 (0xful << RTC_CALM_MON_Pos)                       /*!< RTC_T::CALM: MON Mask                  */

#define RTC_CALM_TENMON_Pos              (12)                                              /*!< RTC_T::CALM: TENMON Position           */
#define RTC_CALM_TENMON_Msk              (0x1ul << RTC_CALM_TENMON_Pos)                    /*!< RTC_T::CALM: TENMON Mask               */

#define RTC_CALM_YEAR_Pos                (16)                                              /*!< RTC_T::CALM: YEAR Position             */
#define RTC_CALM_YEAR_Msk                (0xful << RTC_CALM_YEAR_Pos)                      /*!< RTC_T::CALM: YEAR Mask                 */

#define RTC_CALM_TENYEAR_Pos             (20)                                              /*!< RTC_T::CALM: TENYEAR Position          */
#define RTC_CALM_TENYEAR_Msk             (0xful << RTC_CALM_TENYEAR_Pos)                   /*!< RTC_T::CALM: TENYEAR Mask              */

#define RTC_LEAPYEAR_LEAPYEAR_Pos        (0)                                               /*!< RTC_T::LEAPYEAR: LEAPYEAR Position     */
#define RTC_LEAPYEAR_LEAPYEAR_Msk        (0x1ul << RTC_LEAPYEAR_LEAPYEAR_Pos)              /*!< RTC_T::LEAPYEAR: LEAPYEAR Mask         */

#define RTC_INTEN_ALMIEN_Pos             (0)                                               /*!< RTC_T::INTEN: ALMIEN Position          */
#define RTC_INTEN_ALMIEN_Msk             (0x1ul << RTC_INTEN_ALMIEN_Pos)                   /*!< RTC_T::INTEN: ALMIEN Mask              */

#define RTC_INTEN_TICKIEN_Pos            (1)                                               /*!< RTC_T::INTEN: TICKIEN Position         */
#define RTC_INTEN_TICKIEN_Msk            (0x1ul << RTC_INTEN_TICKIEN_Pos)                  /*!< RTC_T::INTEN: TICKIEN Mask             */

#define RTC_INTEN_SNPDIEN_Pos            (2)                                               /*!< RTC_T::INTEN: SNPDIEN Position         */
#define RTC_INTEN_SNPDIEN_Msk            (0x1ul << RTC_INTEN_SNPDIEN_Pos)                  /*!< RTC_T::INTEN: SNPDIEN Mask             */

#define RTC_INTSTS_ALMIF_Pos             (0)                                               /*!< RTC_T::INTSTS: ALMIF Position          */
#define RTC_INTSTS_ALMIF_Msk             (0x1ul << RTC_INTSTS_ALMIF_Pos)                   /*!< RTC_T::INTSTS: ALMIF Mask              */

#define RTC_INTSTS_TICKIF_Pos            (1)                                               /*!< RTC_T::INTSTS: TICKIF Position         */
#define RTC_INTSTS_TICKIF_Msk            (0x1ul << RTC_INTSTS_TICKIF_Pos)                  /*!< RTC_T::INTSTS: TICKIF Mask             */

#define RTC_INTSTS_SNPDIF_Pos            (2)                                               /*!< RTC_T::INTSTS: SNPDIF Position         */
#define RTC_INTSTS_SNPDIF_Msk            (0x1ul << RTC_INTSTS_SNPDIF_Pos)                  /*!< RTC_T::INTSTS: SNPDIF Mask             */

#define RTC_TICK_TICK_Pos                (0)                                               /*!< RTC_T::TICK: TICK Position             */
#define RTC_TICK_TICK_Msk                (0x7ul << RTC_TICK_TICK_Pos)                      /*!< RTC_T::TICK: TICK Mask                 */

#define RTC_TAMSK_MSEC_Pos               (0)                                               /*!< RTC_T::TAMSK: MSEC Position            */
#define RTC_TAMSK_MSEC_Msk               (0x1ul << RTC_TAMSK_MSEC_Pos)                     /*!< RTC_T::TAMSK: MSEC Mask                */

#define RTC_TAMSK_MTENSEC_Pos            (1)                                               /*!< RTC_T::TAMSK: MTENSEC Position         */
#define RTC_TAMSK_MTENSEC_Msk            (0x1ul << RTC_TAMSK_MTENSEC_Pos)                  /*!< RTC_T::TAMSK: MTENSEC Mask             */

#define RTC_TAMSK_MMIN_Pos               (2)                                               /*!< RTC_T::TAMSK: MMIN Position            */
#define RTC_TAMSK_MMIN_Msk               (0x1ul << RTC_TAMSK_MMIN_Pos)                     /*!< RTC_T::TAMSK: MMIN Mask                */

#define RTC_TAMSK_MTENMIN_Pos            (3)                                               /*!< RTC_T::TAMSK: MTENMIN Position         */
#define RTC_TAMSK_MTENMIN_Msk            (0x1ul << RTC_TAMSK_MTENMIN_Pos)                  /*!< RTC_T::TAMSK: MTENMIN Mask             */

#define RTC_TAMSK_MHR_Pos                (4)                                               /*!< RTC_T::TAMSK: MHR Position             */
#define RTC_TAMSK_MHR_Msk                (0x1ul << RTC_TAMSK_MHR_Pos)                      /*!< RTC_T::TAMSK: MHR Mask                 */

#define RTC_TAMSK_MTENHR_Pos             (5)                                               /*!< RTC_T::TAMSK: MTENHR Position          */
#define RTC_TAMSK_MTENHR_Msk             (0x1ul << RTC_TAMSK_MTENHR_Pos)                   /*!< RTC_T::TAMSK: MTENHR Mask              */

#define RTC_CAMSK_MDAY_Pos               (0)                                               /*!< RTC_T::CAMSK: MDAY Position            */
#define RTC_CAMSK_MDAY_Msk               (0x1ul << RTC_CAMSK_MDAY_Pos)                     /*!< RTC_T::CAMSK: MDAY Mask                */

#define RTC_CAMSK_MTENDAY_Pos            (1)                                               /*!< RTC_T::CAMSK: MTENDAY Position         */
#define RTC_CAMSK_MTENDAY_Msk            (0x1ul << RTC_CAMSK_MTENDAY_Pos)                  /*!< RTC_T::CAMSK: MTENDAY Mask             */

#define RTC_CAMSK_MMON_Pos               (2)                                               /*!< RTC_T::CAMSK: MMON Position            */
#define RTC_CAMSK_MMON_Msk               (0x1ul << RTC_CAMSK_MMON_Pos)                     /*!< RTC_T::CAMSK: MMON Mask                */

#define RTC_CAMSK_MTENMON_Pos            (3)                                               /*!< RTC_T::CAMSK: MTENMON Position         */
#define RTC_CAMSK_MTENMON_Msk            (0x1ul << RTC_CAMSK_MTENMON_Pos)                  /*!< RTC_T::CAMSK: MTENMON Mask             */

#define RTC_CAMSK_MYEAR_Pos              (4)                                               /*!< RTC_T::CAMSK: MYEAR Position           */
#define RTC_CAMSK_MYEAR_Msk              (0x1ul << RTC_CAMSK_MYEAR_Pos)                    /*!< RTC_T::CAMSK: MYEAR Mask               */

#define RTC_CAMSK_MTENYEAR_Pos           (5)                                               /*!< RTC_T::CAMSK: MTENYEAR Position        */
#define RTC_CAMSK_MTENYEAR_Msk           (0x1ul << RTC_CAMSK_MTENYEAR_Pos)                 /*!< RTC_T::CAMSK: MTENYEAR Mask            */

#define RTC_SPRCTL_SNPDEN_Pos            (0)                                               /*!< RTC_T::SPRCTL: SNPDEN Position         */
#define RTC_SPRCTL_SNPDEN_Msk            (0x1ul << RTC_SPRCTL_SNPDEN_Pos)                  /*!< RTC_T::SPRCTL: SNPDEN Mask             */

#define RTC_SPRCTL_SNPTYPE0_Pos          (1)                                               /*!< RTC_T::SPRCTL: SNPTYPE0 Position       */
#define RTC_SPRCTL_SNPTYPE0_Msk          (0x1ul << RTC_SPRCTL_SNPTYPE0_Pos)                /*!< RTC_T::SPRCTL: SNPTYPE0 Mask           */

#define RTC_SPRCTL_SPRRWEN_Pos           (2)                                               /*!< RTC_T::SPRCTL: SPRRWEN Position        */
#define RTC_SPRCTL_SPRRWEN_Msk           (0x1ul << RTC_SPRCTL_SPRRWEN_Pos)                 /*!< RTC_T::SPRCTL: SPRRWEN Mask            */

#define RTC_SPRCTL_SPRCSTS_Pos           (5)                                               /*!< RTC_T::SPRCTL: SPRCSTS Position        */
#define RTC_SPRCTL_SPRCSTS_Msk           (0x1ul << RTC_SPRCTL_SPRCSTS_Pos)                 /*!< RTC_T::SPRCTL: SPRCSTS Mask            */

#define RTC_SPR0_SPARE_Pos               (0)                                               /*!< RTC_T::SPR: SPARE0 Position            */
#define RTC_SPR0_SPARE_Msk               (0xfffffffful << RTC_SPR0_SPARE_Pos)              /*!< RTC_T::SPR: SPARE0 Mask                */

#define RTC_SPR1_SPARE_Pos               (0)                                               /*!< RTC_T::SPR: SPARE1 Position            */
#define RTC_SPR1_SPARE_Msk               (0xfffffffful << RTC_SPR1_SPARE_Pos)              /*!< RTC_T::SPR: SPARE1 Mask                */

#define RTC_SPR2_SPARE_Pos               (0)                                               /*!< RTC_T::SPR: SPARE2 Position            */
#define RTC_SPR2_SPARE_Msk               (0xfffffffful << RTC_SPR2_SPARE_Pos)              /*!< RTC_T::SPR: SPARE2 Mask                */

#define RTC_SPR3_SPARE_Pos               (0)                                               /*!< RTC_T::SPR: SPARE3 Position            */
#define RTC_SPR3_SPARE_Msk               (0xfffffffful << RTC_SPR3_SPARE_Pos)              /*!< RTC_T::SPR: SPARE3 Mask                */

#define RTC_SPR4_SPARE_Pos               (0)                                               /*!< RTC_T::SPR: SPARE4 Position            */
#define RTC_SPR4_SPARE_Msk               (0xfffffffful << RTC_SPR4_SPARE_Pos)              /*!< RTC_T::SPR: SPARE4 Mask                */

#define RTC_LXTCTL_LXT_TYPE_Pos          (0)                                               /*!< RTC_T::LXTCTL: LXT_TYPE Position       */
#define RTC_LXTCTL_LXT_TYPE_Msk          (0x1ul << RTC_LXTCTL_LXT_TYPE_Pos)                /*!< RTC_T::LXTCTL: LXT_TYPE Mask           */

#define RTC_LXTOCTL_OPMODE_Pos           (0)                                               /*!< RTC_T::LXTOCTL: OPMODE Position        */
#define RTC_LXTOCTL_OPMODE_Msk           (0x3ul << RTC_LXTOCTL_OPMODE_Pos)                 /*!< RTC_T::LXTOCTL: OPMODE Mask            */

#define RTC_LXTOCTL_DOUT_Pos             (2)                                               /*!< RTC_T::LXTOCTL: DOUT Position          */
#define RTC_LXTOCTL_DOUT_Msk             (0x1ul << RTC_LXTOCTL_DOUT_Pos)                   /*!< RTC_T::LXTOCTL: DOUT Mask              */

#define RTC_LXTOCTL_CTLSEL_Pos           (3)                                               /*!< RTC_T::LXTOCTL: CTLSEL Position        */
#define RTC_LXTOCTL_CTLSEL_Msk           (0x1ul << RTC_LXTOCTL_CTLSEL_Pos)                 /*!< RTC_T::LXTOCTL: CTLSEL Mask            */

#define RTC_LXTICTL_OPMODE_Pos           (0)                                               /*!< RTC_T::LXTICTL: OPMODE Position        */
#define RTC_LXTICTL_OPMODE_Msk           (0x3ul << RTC_LXTICTL_OPMODE_Pos)                 /*!< RTC_T::LXTICTL: OPMODE Mask            */

#define RTC_LXTICTL_DOUT_Pos             (2)                                               /*!< RTC_T::LXTICTL: DOUT Position          */
#define RTC_LXTICTL_DOUT_Msk             (0x1ul << RTC_LXTICTL_DOUT_Pos)                   /*!< RTC_T::LXTICTL: DOUT Mask              */

#define RTC_LXTICTL_CTLSEL_Pos           (3)                                               /*!< RTC_T::LXTICTL: CTLSEL Position        */
#define RTC_LXTICTL_CTLSEL_Msk           (0x1ul << RTC_LXTICTL_CTLSEL_Pos)                 /*!< RTC_T::LXTICTL: CTLSEL Mask            */

#define RTC_TAMPCTL_OPMODE_Pos           (0)                                               /*!< RTC_T::TAMPCTL: OPMODE Position        */
#define RTC_TAMPCTL_OPMODE_Msk           (0x3ul << RTC_TAMPCTL_OPMODE_Pos)                 /*!< RTC_T::TAMPCTL: OPMODE Mask            */

#define RTC_TAMPCTL_DOUT_Pos             (2)                                               /*!< RTC_T::TAMPCTL: DOUT Position          */
#define RTC_TAMPCTL_DOUT_Msk             (0x1ul << RTC_TAMPCTL_DOUT_Pos)                   /*!< RTC_T::TAMPCTL: DOUT Mask              */

#define RTC_TAMPCTL_CTLSEL_Pos           (3)                                               /*!< RTC_T::TAMPCTL: CTLSEL Position        */
#define RTC_TAMPCTL_CTLSEL_Msk           (0x1ul << RTC_TAMPCTL_CTLSEL_Pos)                 /*!< RTC_T::TAMPCTL: CTLSEL Mask            */

#define RTC_MISCCTL_GAINSEL_Pos          (12)                                              /*!< RTC_T::MISCCTL: GAINSEL Position       */
#define RTC_MISCCTL_GAINSEL_Msk          (0x3ul << RTC_MISCCTL_GAINSEL_Pos)                /*!< RTC_T::MISCCTL: GAINSEL Mask           */

/**@}*/ /* RTC_CONST */
/**@}*/ /* end of RTC register group */


/*---------------------- Universal Asynchronous Receiver/Transmitter Controller -------------------------*/
/**
    @addtogroup UART Universal Asynchronous Receiver/Transmitter Controller(UART)
    Memory Mapped Structure for UART Controller
@{ */

typedef struct
{


    /**
     * @var UART_T::DAT
     * Offset: 0x00  UART Receive/Transmit Buffer Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |DAT       |Receive /Transmit Buffer
     * |        |          |Write Operation:
     * |        |          |By writing one byte to this register, the data byte will be stored in transmitter FIFO. The UART Controller will send out the data stored in transmitter FIFO top location through the UART_DAT.
     * |        |          |Read Operation:
     * |        |          |By reading this register, the UART will return an 8-bit data received from receiving FIFO.
     * @var UART_T::CTRL
     * Offset: 0x04  UART Control Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RXRST     |RX Field Software Reset
     * |        |          |When RXRST (UART_CTL[0]) is set, all the byte in the receiver FIFO and RX internal state machine are cleared.
     * |        |          |0 = No effect.
     * |        |          |1 = Reset the RX internal state machine and pointers.
     * |        |          |Note: This bit will automatically clear at least 3 UART peripheral clock cycles.
     * |[1]     |TXRST     |TX Field Software Reset
     * |        |          |When TXRST (UART_CTL[1]) is set, all the byte in the transmit FIFO and TX internal state machine are cleared.
     * |        |          |0 = No effect.
     * |        |          |1 = Reset the TX internal state machine and pointers.
     * |        |          |Note: This bit will automatically clear at least 3 UART peripheral clock cycles
     * |[2]     |RXOFF     |Receiver Disable Bit
     * |        |          |0 = Receiver Enabled.
     * |        |          |1 = Receiver Disabled.
     * |        |          |Note1: In RS-485 NMM mode, user can set this bit to receive data before detecting address byte.
     * |        |          |Note2: In RS-485 AAD mode, this bit will be setting to 1 automatically.
     * |        |          |Note3: In RS-485 AUD mode and LIN break + sync +PID header mode, hardware will control data automatically, so don't fill any value to this bit.
     * |[3]     |TXOFF     |Transfer Disable Bit
     * |        |          |0 = Transfer Enabled.
     * |        |          |1 = Transfer Disabled.
     * |[4]     |ATORTSEN  |nRTS Auto-flow Control Enable Bit
     * |        |          |0 = nRTS auto-flow control Disabled.
     * |        |          |1 = nRTS auto-flow control Enabled.
     * |        |          |Note: When nRTS auto-flow is enabled, if the number of bytes in the RX FIFO equals the RTSTRGLV (UART_TLCTL[13:12]), the UART will de-assert nRTS signal.
     * |[5]     |ATOCTSEN  |nCTS Auto-flow Control Enable Bit
     * |        |          |0 = nCTS auto-flow control Disabled.
     * |        |          |1 = nCTS auto-flow control Enabled.
     * |        |          |Note: When nCTS auto-flow is enabled, the UART will send data to external device if nCTS input assert (UART will not send data to device until nCTS is asserted).
     * |[6]     |RXDMAEN   |RX DMA Enable Bit
     * |        |          |This bit can enable or disable RX DMA service.
     * |        |          |0 = RX DMA Disabled.
     * |        |          |1 = RX DMA Enabled.
     * |[7]     |TXDMAEN   |TX DMA Enable Bit
     * |        |          |This bit can enable or disable TX DMA service.
     * |        |          |0 = TX DMA Disabled.
     * |        |          |1 = TX DMA Enabled.
     * |[8]     |FTOEN     |Frame Time Out Enable Bit
     * |        |          |This bit is used to enable the timer counter even the FIFO is still empty.
     * |        |          |0 = Frame time out Disabled.
     * |        |          |1 = Frame time out Enabled.
     * |[12]    |ABRDEN    |Auto-baud Rate Detect Enable Bit
     * |        |          |0 = Auto-baud rate detect function Disabled.
     * |        |          |1 = Auto-baud rate detect function Enabled.
     * |        |          |Note: When the auto-baud rate detect operation finishes, hardware will clear this bit and the associated interrupt (ABRIF) will be generated (If ABRIEN (UART_IER [7]) be enabled).
     * |[14:13] |ABRDBITS  |Auto-baud Rate Detect Bit Length
     * |        |          |00 = 1-bit time from Start bit to the 1st rising edge. The input pattern shall be 0x01.
     * |        |          |01 = 2-bit time from Start bit to the 1st rising edge. The input pattern shall be 0x02.
     * |        |          |10 = 4-bit time from Start bit to the 1st rising edge. The input pattern shall be 0x08.
     * |        |          |11 = 8-bit time from Start bit to the 1st rising edge. The input pattern shall be 0x80.
     * |        |          |Note: The calculation of bit number includes the START bit.
     * @var UART_T::LINE
     * Offset: 0x08  UART Transfer Line Control Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |WLS       |Word Length Selection
     * |        |          |This field sets UART word length.
     * |        |          |00 = 5 bits.
     * |        |          |01 = 6 bits.
     * |        |          |10 = 7 bits.
     * |        |          |11 = 8 bits.
     * |[2]     |NSB       |Number of STOP Bit
     * |        |          |0 = One STOP bit is generated in the transmitted data.
     * |        |          |1 = When select 5-bit word length, 1.5 STOP bit is generated in the transmitted data
     * |        |          |When select 6-, 7- and 8-bit word length, 2 STOP bit is generated in the transmitted data.
     * |[3]     |PBE       |Parity Bit Enable Bit
     * |        |          |0 = No parity bit generated Disabled.
     * |        |          |1 = Parity bit generated Enabled.
     * |        |          |Note: Parity bit is generated on each outgoing character and is checked on each incoming data.
     * |[4]     |EPE       |Even Parity Enable Bit
     * |        |          |0 = Odd number of logic 1's is transmitted and checked in each word.
     * |        |          |1 = Even number of logic 1's is transmitted and checked in each word.
     * |        |          |Note: This bit has effect only when PBE (UART_TLCTL[3]) is set.
     * |[5]     |SPE       |Stick Parity Enable Bit
     * |        |          |0 = Stick parity Disabled.
     * |        |          |1 = Stick parity Enabled.
     * |        |          |Note: If PBE (UART_TLCTL[3]) and EPE (UART_TLCTL[4]) are logic 1, the parity bit is transmitted and checked as logic 0
     * |        |          |If PBE (UART_TLCTL[3]) is 1 and EPE (UART_TLCTL[4]) is 0 then the parity bit is transmitted and checked as 1.
     * |[6]     |BCB       |Break Control Bit
     * |        |          |0 = Break Control Disabled.
     * |        |          |1 = Break Control Enabled.
     * |        |          |Note: When this bit is set to logic 1, the serial data output (TX) is forced to the Spacing State (logic 0)
     * |        |          |This bit acts only on TX line and has no effect on the transmitter logic.
     * |[9:8]   |RFITL     |RX FIFO Interrupt Trigger Level
     * |        |          |When the number of bytes in the receive FIFO equals the RFITL, the RDAIF will be set (if RDAIEN (UART_IER [0]) enabled, and an interrupt will be generated).
     * |        |          |0000 = RX FIFO Interrupt Trigger Level is 1 byte.
     * |        |          |0001 = RX FIFO Interrupt Trigger Level is 4 bytes.
     * |        |          |0010 = RX FIFO Interrupt Trigger Level is 8 bytes.
     * |        |          |0011 = RX FIFO Interrupt Trigger Level is 14 bytes.
     * |        |          |Note: When operating in IrDA mode or RS-485 mode, the RFITL must be set to 0.
     * |[13:12] |RTSTRGLV  |nRTS Trigger Level for Auto-flow Control Use
     * |        |          |00 = nRTS Trigger Level is 1 byte.
     * |        |          |01 = nRTS Trigger Level is 4 bytes.
     * |        |          |10 = nRTS Trigger Level is 8 bytes.
     * |        |          |11 = nRTS Trigger Level is 14 bytes.
     * |        |          |Note: This field is used for auto nRTS flow control.
     * @var UART_T::INTEN
     * Offset: 0x0C  UART Interrupt Enable Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RDAIEN    |Receive Data Available Interrupt Enable Bit
     * |        |          |0 = Receive data available interrupt Disabled.
     * |        |          |1 = Receive data available interrupt Enabled.
     * |[1]     |THREIEN   |Transmit Holding Register Empty Interrupt Enable Bit
     * |        |          |0 = Transmit holding register empty interrupt Disabled.
     * |        |          |1 = Transmit holding register empty interrupt Enabled.
     * |[2]     |RLSIEN    |Receive Line Status Interrupt Enable Bit
     * |        |          |0 = Receive Line Status interrupt Disabled.
     * |        |          |1 = Receive Line Status interrupt Enabled.
     * |[3]     |MODEMIEN  |Modem Status Interrupt Enable Bit
     * |        |          |0 = Modem status interrupt Disabled.
     * |        |          |1 = Modem status interrupt Enabled.
     * |[4]     |RXTOIEN   |RX Time-out Interrupt Enable Bit
     * |        |          |0 = RX time-out interrupt Disabled.
     * |        |          |1 = RX time-out interrupt Enabled.
     * |[5]     |BUFERRIEN |Buffer Error Interrupt Enable Bit
     * |        |          |0 = Buffer error interrupt Disabled.
     * |        |          |1 = Buffer error interrupt Enabled.
     * |[6]     |WKUPIEN   |Wake-up Interrupt Enable Bit
     * |        |          |0 = Wake-up system function Disabled.
     * |        |          |1 = Wake-up system function Enabled, when the system is in Power-down mode, one of the wake-up event will wake-up system from Power-down mode.
     * |        |          |Note: Hardware will clear one of the wake-up status bits in UART_WKUPSTS when the wake-up operation finishes and system clock work stable
     * |[7]     |ABRIEN    |Auto-baud Rate Interrupt Enable Bit
     * |        |          |0 = Auto-baud rate interrupt Disabled.
     * |        |          |1 = Auto-baud rate interrupt Enabled.
     * |[8]     |LINIEN    |LIN Bus Interrupt Enable Bit
     * |        |          |0 = LIN bus interrupt Disabled.
     * |        |          |1 = LIN bus interrupt Enabled.
     * |        |          |Note: This bit is used for LIN function mode.
     * |[9]     |TXENDIEN  |Transmitter Empty Interrupt Enable Bit
     * |        |          |0 = Transmit Empty interrupt Disabled.
     * |        |          |1 = Transmit Empty interrupt Enabled.
     * |        |          |Note: If the bit is enabled, there is interrupt event when the TXENDF (UART_FSR[11]) is activated.
     * @var UART_T::INTSTS
     * Offset: 0x10  UART Interrupt Status Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RDAIF     |Receive Data Available Interrupt Flag (Read Only)
     * |        |          |When the number of bytes in the RX FIFO equals the RFITL then the RDAIF(UART_ISR[0]) will be set
     * |        |          |If RDAIEN (UART_IER [0]) is enabled, the RDA interrupt will be generated.
     * |        |          |0 = No RDA interrupt flag is generated.
     * |        |          |1 = RDA interrupt flag is generated.
     * |        |          |Note: This bit is read only and it will be cleared when the number of unread bytes of RX FIFO drops below the threshold level (RFITL(UART_TLCTL[9:8])
     * |[1]     |THREIF    |Transmit Holding Register Empty Interrupt Flag (Read Only)
     * |        |          |This bit is set when the last data of TX FIFO is transferred to Transmitter Shift Register
     * |        |          |If THREIEN (UART_IER[1]) is enabled, the THRE interrupt will be generated.
     * |        |          |0 = No THRE interrupt flag is generated.
     * |        |          |1 = THRE interrupt flag is generated.
     * |        |          |Note: This bit is read only and it will be cleared when writing data into UART_DAT (TX FIFO not empty)
     * |[2]     |RLSIF     |Receive Line Interrupt Flag (Read Only)
     * |        |          |This bit is set when the RX receive data have parity error, frame error or break error (at least one of 3 bits, BIF(UART_FSR[6]), FEF(UART_FSR[5]) and PEF(UART_FSR[4]), is set)
     * |        |          |If RLSIEN (UART_IER [2]) is enabled, the RLS interrupt will be generated.
     * |        |          |0 = No RLS interrupt flag is generated.
     * |        |          |1 = RLS interrupt flag is generated.
     * |        |          |Note1: In RS-485 function mode, this field is set include receiver detect and received address byte character (bit9 = '1') bit"
     * |        |          |At the same time, the bit of ADDRDETF (UART_TRSR[0]) is also set.
     * |        |          |Note2: This bit is read only and reset to 0 when all bits of BIF (UART_FSR[6]), FEF(UART_FSR[5]) and PEF(UART_FSR[4]) are cleared.
     * |        |          |Note3: In RS-485 function mode, this bit is read only and reset to 0 when all bits of BIF (UART_FSR[6]) , FEF(UART_FSR[5]) and PEF(UART_FSR[4]) and ADDRDETF (UART_TRSR[0]) are cleared.
     * |[3]     |MODEMIF   |MODEM Interrupt Flag (Read Only) Channel
     * |        |          |This bit is set when the nCTS pin has state change (CTSDETF (UART_MCSR[18]) = 1)
     * |        |          |If MODEMIEN (UART_IER [3]) is enabled, the Modem interrupt will be generated.
     * |        |          |0 = No Modem interrupt flag is generated.
     * |        |          |1 = Modem interrupt flag is generated.
     * |        |          |Note: This bit is read only and reset to 0 when bit CTSDETF is cleared by a write 1 on CTSDETF(UART_MCSR[18]).
     * |[4]     |RXTOIF    |Rime-out Interrupt Flag (Read Only)
     * |        |          |This bit is set when the RX FIFO is not empty and no activities occurred in the RX FIFO and the time-out counter equal to TOIC
     * |        |          |If RXTOIEN (UART_IER [4]) is enabled, the Tout interrupt will be generated.
     * |        |          |0 = No Time-out interrupt flag is generated.
     * |        |          |1 = Time-out interrupt flag is generated.
     * |        |          |Note: This bit is read only and user can read UART_DAT (RX is in active) to clear it
     * |[5]     |BUFERRIF  |Buffer Error Interrupt Flag (Read Only)
     * |        |          |This bit is set when the TX FIFO or RX FIFO overflows (TXOVIF (UART_FSR[8]) or RXOVIF (UART_FSR[0]) is set)
     * |        |          |When BUFERRIF (UART_ISR[5])is set, the transfer is not correct
     * |        |          |If BFERRIEN (UART_IER [5]) is enabled, the buffer error interrupt will be generated.
     * |        |          |0 = No buffer error interrupt flag is generated.
     * |        |          |1 = Buffer error interrupt flag is generated.
     * |        |          |Note: This bit is read only
     * |        |          |This bit is cleared if both of RXOVIF(UART_FSR[0]) and TXOVIF(UART_FSR[8]) are cleared to 0 by writing 1 to RXOVIF(UART_FSR[0]) and TXOVIF(UART_FSR[8]).
     * |[6]     |WKUPIF    |Wake-up Interrupt Flag (Read Only)
     * |        |          |This bit is set if chip wake-up from power-down state by one of UART controller wake-up event.
     * |        |          |0 = Chip stays in power-down state.
     * |        |          |1 = Chip wake-up from power-down state by one of UART controller wake-up event.
     * |        |          |Note1: If WKDATEN (UART_IER[6]) is enabled, the wake-up interrupt is generated.
     * |        |          |Note2: This bit is read only, but can be cleared by writing '1' to one of UART_WKUPSTS[4:0] (THRTOWKSTS or THRWKSTS or CTSWKSTS or DATWKSTS or ADRWKSTS).
     * |[7]     |ABRIF     |Auto-baud Rate Interrupt Status Flag (Read Only)
     * |        |          |This bit is set when auto-baud rate detection function finished or the auto-baud rate counter was overflow and if ABRIEN (UART_IER[7]) is set then the auto-baud rate interrupt will be generated.
     * |        |          |0 = No Auto-Baud Rate interrupt is generated.
     * |        |          |1 = Auto-Baud Rate interrupt is generated.
     * |        |          |Note1: This bit is read only, but can be cleared by writing 1 to ABRDTOIF (UART_TRSR[2]) or ABRDIF (UART_TRSR[1]).
     * |        |          |Note2: This bit is cleared when both the ABRDTOIF and ABRDIF are cleared.
     * |[8]     |LINIF     |LIN Interrupt Status Flag (Read Only)
     * |        |          |This bit is set when the LIN TX header transmitted, RX header received or the SIN does not equal SOUT and if LINIEN(UART_IER[8]) is set then the LIN interrupt will be generated.
     * |        |          |0 = No LIN interrupt is generated.
     * |        |          |1 = LIN interrupt is generated.
     * |        |          |Note1: This bit is read only, but can be cleared by it by writing 1 to BITEF (UART_TRSR[5]), LINTXIF (UART_TRSR[3]) or LINRXIF (UART_TRSR[4]).
     * |        |          |Note2: This bit is cleared when both the BITEF, LINTXIF and LINRXIF are cleared.
     * @var UART_T::TRSR
     * Offset: 0x14  UART Transfer Status Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ADDRDETF  |RS-485 Address Byte Detection Status Flag (Read Only)
     * |        |          |0 = Receiver detects a data that is not an address bit (bit 9 ='0').
     * |        |          |1 = Receiver detects a data that is an address bit (bit 9 ='1').
     * |        |          |Note1: This field is used for RS-485 function mode and ADDRDEN (UART_ALT_CSR[19]) is set to 1 to enable Address detection mode .
     * |        |          |Note2: This bit is read only, but can be cleared by writing '1' to it.
     * |[1]     |ABRDIF    |Auto-baud Rate Interrupt (Read Only)
     * |        |          |This bit is set to logic 1 when auto-baud rate detect function finished.
     * |        |          |0 = No Auto- Baud Rate interrupt is generated.
     * |        |          |1= Auto-Baud Rate interrupt is generated.
     * |        |          |Note: This bit is read only, but can be cleared by writing 1 to it.
     * |[2]     |ABRDTOIF  |Auto-baud Rate Time-out Interrupt (Read Only)
     * |        |          |0 = Auto-baud rate counter is underflow.
     * |        |          |1 = Auto-baud rate counter is overflow.
     * |        |          |Note1: This bit is set to logic 1 in Auto-baud Rate Detect mode and the baud rate counter is overflow.
     * |        |          |Note2: This bit is read only, but can be cleared by writing 1 to it.
     * |[3]     |LINTXIF   |LIN TX Interrupt Flag (Read Only)
     * |        |          |This bit is set to logic 1 when LIN transmitted header field
     * |        |          |The header may be break field or break field + sync field or break field + sync field + PID field, it can be choose by setting LINHSEL (UART_ALT_CSR[5:4]) register.
     * |        |          |0 = No LIN Transmit interrupt is generated.
     * |        |          |1 = LIN Transmit interrupt is generated.
     * |        |          |Note: This bit is read only, but can be cleared by writing 1 to it.
     * |[4]     |LINRXIF   |LIN RX Interrupt Flag (Read Only)
     * |        |          |This bit is set to logic 1 when received LIN header field
     * |        |          |The header may be break field or break field + sync field or break field + sync field + PID field, and it can be choose by setting LINHSEL (UART_ALT_CSR[5:4]) register.
     * |        |          |0 = No LIN Rx interrupt is generated.
     * |        |          |1 = LIN Rx interrupt is generated.
     * |        |          |Note: This bit is read only, but can be cleared by writing 1 to it.
     * |[5]     |BITEF     |Bit Error Detect Status Flag (Read Only)
     * |        |          |At TX transfer state, hardware will monitoring the bus state, if the input pin (SIN) state is not equal to the output pin (SOUT) state, BITEF will be set.
     * |        |          |When occur bit error, hardware will generate an interrupt to CPU (LININT).
     * |        |          |0 = No Bit error interrupt is generated.
     * |        |          |1 = Bit error interrupt is generated.
     * |        |          |Note1: This bit is read only, but it can be cleared by writing 1 to it.
     * |        |          |Note2: This bit is only valid when enabling the bit error detection function (BITERREN (UART_ALT_CSR[8]) = 1).
     * |[7]     |RXBUSY    |Receive Busy Status (Read Only)
     * |        |          |0 = The receiver machine stays in idle state.
     * |        |          |1 = The receiver machine stays in no Idle state.
     * |        |          |Note: The user can use this to check the busy status in receiver mode
     * |        |          |If the user want to enter the power down, this bit shall be confirm in Idle state and there is 2 UART clock latency for receiver pin.
     * |[8]     |SLVSYNCF  |LIN RX SYNC Error Flag (Read Only)
     * |        |          |This bit is set to logic 1 when LIN received incorrect SYNC field.
     * |        |          |User can choose the header by setting LINHSEL (UART_ALT_CSR[5:4]) register.
     * |        |          |0 = No LIN Rx sync error is generated.
     * |        |          |1 = LIN Rx sync error is generated.
     * |        |          |Note: This bit is read only, but can be cleared by writing 1 to LINRXIF.
     * @var UART_T::FIFOSTS
     * Offset: 0x18  UART FIFO Status Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RXOVIF    |RX Overflow Error Status Flag (Read Only)
     * |        |          |This bit is set when RX FIFO overflow
     * |        |          |If the number of bytes of received data is greater than RX_FIFO (UART_RBR) size, this bit will be set.
     * |        |          |0 = RX FIFO is not overflow.
     * |        |          |1 = RX FIFO is overflow.
     * |        |          |Note: This bit is read only, but can be cleared by writing 1 to it.
     * |[1]     |RXEMPTY   |Receiver FIFO Empty (Read Only)
     * |        |          |This bit initiate RX FIFO empty or not.
     * |        |          |0 = RX FIFO is not empty.
     * |        |          |1 = RX FIFO is empty.
     * |        |          |Note: When the last byte of RX FIFO has been read by CPU, hardware sets this bit high
     * |        |          |It will be cleared when UART receives any new data.
     * |[2]     |RXFULL    |Receiver FIFO Full (Read Only)
     * |        |          |This bit initiates RX FIFO full or not.
     * |        |          |0 = RX FIFO is not full.
     * |        |          |1 = RX FIFO is full.
     * |        |          |Note: This bit is set when the number of usage in RX FIFO Buffer is equal to 16, otherwise is cleared by hardware.
     * |[4]     |PEF       |Parity Error State Status Flag (Read Only)
     * |        |          |This bit is set to logic 1 whenever the received character does not have a valid parity bit.
     * |        |          |0 = No parity error is generated.
     * |        |          |1 = Parity error is generated.
     * |        |          |Note: This bit is read only, but can be cleared by writing '1' to it.
     * |[5]     |FEF       |Framing Error Flag (Read Only)
     * |        |          |This bit is set to logic 1 whenever the received character does not have a valid stop bit (that is, the stop bit following the last data bit or parity bit is detected as logic 0).
     * |        |          |0 = No framing error is generated.
     * |        |          |1 = Framing error is generated.
     * |        |          |Note: This bit is read only, but can be cleared by writing '1' to it.
     * |[6]     |BIF       |Break Interrupt Flag( Read Only)
     * |        |          |This bit is set to logic 1 whenever the received data input (RX) is held in the spacing state (logic 0) for longer than a full word transmission time (that is, the total time of start bit + data bits + parity + stop bits).
     * |        |          |0 = No Break interrupt is generated.
     * |        |          |1 = Break interrupt is generated.
     * |        |          |Note: This bit is read only, but can be cleared by writing '1' to it.
     * |[8]     |TXOVIF    |TX Overflow Error Interrupt Status Flag (Read Only)
     * |        |          |If TX FIFO (UART_DAT) is full, an additional write to UART_DAT will cause this bit to logic 1.
     * |        |          |0 = TX FIFO did not overflow.
     * |        |          |1 = TX FIFO overflowed.
     * |        |          |Note: This bit is read only, but can be cleared by writing 1 to it.
     * |[9]     |TXEMPTY   |Transmitter FIFO Empty (Read Only)
     * |        |          |This bit indicates TX FIFO empty or not.
     * |        |          |0 = TX FIFO is not empty.
     * |        |          |1 = TX FIFO is empty.
     * |        |          |Note: When the last byte of TX FIFO has been transferred to Transmitter Shift Register, hardware sets this bit high
     * |        |          |It will be cleared when writing data into DAT (TX FIFO not empty).
     * |[10]    |TXFULL    |Transmitter FIFO Full (Read Only)
     * |        |          |This bit indicates TX FIFO full or not.
     * |        |          |0 = TX FIFO is not full.
     * |        |          |1 = TX FIFO is full.
     * |        |          |Note: This bit is set when the number of usage in TX FIFO Buffer is equal to 16, otherwise is cleared by hardware.
     * |[11]    |TXENDF    |Transmitter Empty Flag (Read Only)
     * |        |          |This bit is set by hardware when TX FIFO (UART_DAT) is empty and the STOP bit of the last byte has been transmitted.
     * |        |          |0 = TX FIFO is not empty or the STOP bit of the last byte has been not transmitted.
     * |        |          |1 = TX FIFO is empty and the STOP bit of the last byte has been transmitted.
     * |        |          |Note: This bit is cleared automatically when TX FIFO is not empty or the last byte transmission has not completed.
     * |[20:16] |RXPTR     |RX FIFO Pointer (Read Only)
     * |        |          |This field indicates the RX FIFO Buffer Pointer
     * |        |          |When UART receives one byte from external device, RXPTR increases one
     * |        |          |When one byte of RX FIFO is read by CPU, RXPTR decreases one.
     * |        |          |The Maximum value shown in RXPTR is 15
     * |        |          |When the using level of RX FIFO Buffer equal to 16, the RXFULL bit is set to 1 and RXPTR will show 0
     * |        |          |As one byte of RX FIFO is read by CPU, the RXFULL bit is cleared to 0 and RXPTR will show 15
     * |[28:24] |TXPTR     |TX-fIFO Pointer (Read Only)
     * |        |          |This field indicates the TX FIFO Buffer Pointer
     * |        |          |When CPU writes one byte into UART_DAT, TXPTR increases one
     * |        |          |When one byte of TX FIFO is transferred to Transmitter Shift Register, TXPTR decreases one.
     * |        |          |The Maximum value shown in TXPTR is 15
     * |        |          |When the using level of TX FIFO Buffer equal to 16, the TXFULL bit is set to 1 and TXPTR will show 0
     * |        |          |As one byte of TX FIFO is transferred to Transmitter Shift Register, the TXFULL bit is cleared to 0 and TXPTR will show 15
     * @var UART_T::MODEM
     * Offset: 0x1C  UART Modem Control Status Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RTSACTLV  |nRTS Pin Active Level
     * |        |          |This bit defines the active level state of nRTS pin output.
     * |        |          |0 =n RTS pin output is high level active.
     * |        |          |1 = nRTS pin output is low level active. (Default)
     * |[1]     |RTSSTS    |nRTS Pin State (Read Only)
     * |        |          |This bit mirror from nRTS pin output of voltage logic status.
     * |        |          |0 = nRTS pin output is low level voltage logic state.
     * |        |          |1 = nRTS pin output is high level voltage logic state.
     * |[16]    |CTSACTLV  |nCTS Trigger Level
     * |        |          |This bit defines the active level state of nCTS pin input.
     * |        |          |0 = nCTS pin input is high level active.
     * |        |          |1 = nCTS pin input is low level active. (Default)
     * |[17]    |CTSSTS    |nCTS Pin Status (Read Only)
     * |        |          |This bit mirror from nCTS pin input of voltage logic status.
     * |        |          |0 = nCTS pin input is low level voltage logic state.
     * |        |          |1 = nCTS pin input is high level voltage logic state.
     * |        |          |Note: This bit echoes when UART Controller peripheral clock is enabled, and nCTS multi-function port is selected.
     * |[18]    |CTSDETF   |Detect nCTS State Change Flag (Read Only)
     * |        |          |This bit is set whenever nCTS input has change state, and it will generate Modem interrupt to CPU when MODEMIEN (UART_IER[3]).
     * |        |          |0 = nCTS input has not change state.
     * |        |          |1 = nCTS input has change state.
     * |        |          |Note: This bit is read only, but it can be cleared by writing 1 to it.
     * @var UART_T::TOUT
     * Offset: 0x20  UART Time-Out Control Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[8:0]   |TOIC      |Time-out Comparator
     * |        |          |The time-out counter resets and starts counting (the counting clock = baud rate) whenever the RX FIFO receives a new data word
     * |        |          |Once the content of time-out counter is equal to that of time-out interrupt comparator (TOIC (UART_TMCTL[8:0])), a receiver time-out interrupt (RXTOIF(UART_ISR[4])) is generated if RXTOIEN (UART_IER [4]) enabled
     * |        |          |A new incoming data word or RX FIFO empty will clear RXTOIF(UART_ISR[4])
     * |        |          |In order to avoid receiver time-out interrupt generation immediately during one character is being received, TOIC value should be set between 40 and 255
     * |        |          |So, for example, if TOIC is set with 40, the time-out interrupt is generated after four characters are not received when 1 stop bit and no parity check is set for UART transfer.
     * |        |          |Note1: Fill all 0 to this field indicates to disable this function.
     * |        |          |Note2: The real time-out value is TOIC + 1.
     * |        |          |Note3: The counting clock is baud rate clock.
     * |        |          |Note4: The UART data format is start bit + 8 data bits + parity bit + stop bit, although software can configure this field by any value but it is recommend to fill this field great than 0xA.
     * |[23:16] |DLY       |TX Delay Time Value
     * |        |          |This field is used to programming the transfer delay time between the last stop bit and next start bit
     * |        |          |The unit is bit time.
     * |        |          |Note1: Fill all 0 to this field indicates to disable this function.
     * |        |          |Note2: The real delay value is DLY.
     * |        |          |Note3: The counting clock is baud rate clock.
     * @var UART_T::BAUD
     * Offset: 0x24  UART Baud Rate Divisor Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |BRD       |Baud Rate Divider
     * |        |          |The field indicates the baud rate divider
     * |        |          |This filed is used in baud rate calculation
     * |        |          |The detail description is shown in UART Controller Baud Rate Generation.
     * |[31]    |DIV16EN   |Divider 16 Enable Control
     * |        |          |The BRD = Baud Rate Divider, and the baud rate equation is  Baud Rate = UART_CLK/ [M * (BRD + 1)]; The default value of M is 16.
     * |        |          |0 = The equation of baud rate is UART_CLK / [ (BRD+1)].
     * |        |          |1 = The equation of baud rate is UART_CLK / [16 * (BRD+1)].
     * |        |          |Note: In IrDA mode, this bit must clear to 0.
     * @var UART_T::IRDA
     * Offset: 0x30  UART IrDA Control Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1]     |TXEN      |IrDA Receiver/Transmitter Selection Enable Bit
     * |        |          |0 = IrDA Transmitter Disabled and Receiver Enabled. (Default)
     * |        |          |1 = IrDA Transmitter Enabled and Receiver Disabled.
     * |[5]     |TXINV     |IrDA Inverse Transmitting Output Signal
     * |        |          |0 = None inverse transmitting signal. (Default)
     * |        |          |1 = Inverse transmitting output signal.
     * |[6]     |RXINV     |IrDA Inverse Receive Input Signal
     * |        |          |0 = None inverse receiving input signal.
     * |        |          |1 = Inverse receiving input signal. (Default)
     * @var UART_T::ALTCTL
     * Offset: 0x34  UART Alternate Control State Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |BRKFL     |LIN TX Break Field Count
     * |        |          |The field contains 3-bit LIN TX break field count.
     * |        |          |Note: The break field length is BRKFL + 8.
     * |[5:4]   |LINHSEL   |LIN Header Selection
     * |        |          |00 = The LIN header includes break field.
     * |        |          |01 = The LIN header includes break field + sync field.
     * |        |          |10 = The LIN header includes break field + sync field + PID field.
     * |        |          |11 = Reserved.
     * |[6]     |LINRXEN   |LIN RX Enable Control
     * |        |          |When LIN RX mode enabled and received a break field or sync field or PID field (Select by LIN_Header_SEL), the controller will generator a interrupt to CPU (LININT)
     * |        |          |0 = LIN RX mode Disabled.
     * |        |          |1 = LIN RX mode Enabled.
     * |[7]     |LINTXEN   |LIN TX Header Trigger Enable Bit
     * |        |          |The LIN TX header can be break field or break and sync field or break, sync and frame ID field, it is depend on setting LINHSEL (UART_ALT_CSR[5:4]).
     * |        |          |0 = Send LIN TX header Disabled.
     * |        |          |1 = Send LIN TX header Enabled.
     * |        |          |Note1: This bit will be cleared automatically and generate a interrupt to CPU (LININT).
     * |        |          |Note2: When transmitter header field (it may be break or break + sync or break + sync + frame ID selected by LINHSEL (UART_ALT_CSR[5:4]) field) transfer operation finished, this bit will be cleared automatically.
     * |        |          |Note3: If user wants to receive transmit data, it recommended to enable LINRXEN bit.
     * |[8]     |BITERREN  |Bit Error Detect Enable Bit
     * |        |          |0 = Bit error detection Disabled.
     * |        |          |1 = Bit error detection Enabled.
     * |        |          |Note: In LIN function mode, when bit error occurs, the BITEF (UART_TRSR[5]) flag will be asserted
     * |        |          |If the LINIEN (UART_IER[8]) = 1, an interrupt will be generated .
     * |[16]    |RS485NMM  |RS-485 Normal Multi-drop Operation Mode (NMM)
     * |        |          |0 = RS-485 Normal Multi-drop Operation mode (NMM) Disabled.
     * |        |          |1 = RS-485 Normal Multi-drop Operation mode (NMM) Enabled.
     * |        |          |Note: It cannot be active with RS-485_AAD operation mode.
     * |[17]    |RS485AAD  |RS-485 Auto Address Detection Operation Mode (AAD)
     * |        |          |0 = RS-485 Auto Address Detection Operation mode (AAD) Disabled.
     * |        |          |1 = RS-485 Auto Address Detection Operation mode (AAD) Enabled.
     * |        |          |Note: It cannot be active with RS485NMM operation mode.
     * |[18]    |RS485AUD  |RS-485 Auto Direction Function (AUD)
     * |        |          |0 = RS-485 Auto Direction Operation function (AUD) Disabled.
     * |        |          |1 = RS-485 Auto Direction Operation function (AUD) Enabled.
     * |        |          |Note: It can be active with RS485AAD or RS485NMM operation mode.
     * |[19]    |ADDRDEN   |RS-485 Address Detection Enable Bit
     * |        |          |This bit is used to enable RS-485 Address Detection mode.
     * |        |          |0 = Address detection mode Disabled.
     * |        |          |1 = Address detection mode Enabled.
     * |        |          |Note: This bit is used for RS-485 any operation mode.
     * |[31:24] |ADRMPID   |Address / PID Match Value Register
     * |        |          |When in the RS-485 Function Mode, this field contains the RS-485 address match values.
     * |        |          |When in the LIN Function mode, this field contains the LIN protected identifier field, software fills ID0~ID5 (PID [5:0]), hardware will calculate P0 and P1.
     * |        |          |Note: This field is used for RS-485 auto address detection mode or used for LIN protected identifier field (PID).
     * @var UART_T::FUNCSEL
     * Offset: 0x38  UART Function Select Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |FUNCSEL   |Function Selection
     * |        |          |00 = UART function mode.
     * |        |          |01 = LIN function mode.
     * |        |          |10 = IrDA function mode.
     * |        |          |11 = RS-485 function mode.
     * @var UART_T::BRCOMPAT
     * Offset: 0x3C  UART Baud Rate Compensation Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[8:0]   |BRCOMPAT  |Baud Rate Compensation Patten
     * |        |          |These 9 bits are used to define the relative bit is compensated or not
     * |        |          |BRCOMPAT[7:0] is used to define the compensation of D[7:0] and BRCOMPAT{[8] is used to define the parity bit.
     * |[31]    |BRCOMPDEC |Baud Rate Compensation Decrease
     * |        |          |0 = Positive (increase one module clock) compensation for each compensated bit.
     * |        |          |1 = Negative (decrease one module clock) compensation for each compensated bit.
     * @var UART_T::WKUPEN
     * Offset: 0x40  UART Wake-up Enable Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |WKCTSEN   |nCTS Wake-up Enable Bit
     * |        |          |When the system is in power-down mode, an external nCTS change will wake-up system from power-down mode.
     * |        |          |0 = nCTS wake-up function Disabled.
     * |        |          |1 = nCTS wake-up function Enabled.
     * |[1]     |WKDATEN   |Incoming Data Wake-up Enable Bit
     * |        |          |0 = Incoming data wake-up function Disabled.
     * |        |          |1 = Incoming data wake-up function Enabled when the system is in power-down mode, incoming data will wake-up system from power-down mode.
     * |        |          |Note: Hardware will clear this bit when the incoming data wake-up operation finishes and system clock work stable
     * |[2]     |WKTHREN   |FIFO Threshold Reach Wake-up Enable Bit
     * |        |          |0 = Received FIFO threshold reach wake-up function Disabled.
     * |        |          |1 = Received FIFO threshold reach wake-up function Enabled when the system is in power-down mode.
     * |        |          |Note: It is suggest the function is enabled in UART mode and the UART clock is selected in 32K.
     * |[3]     |WKTHRTOEN |FIFO Threshold Reach Time Out Wake-up Enable Bit
     * |        |          |0 = Received FIFO threshold no reach and time out wake-up function Disabled.
     * |        |          |1 = Received FIFO threshold no reach and time out wake-up function Enabled when the system is in power-down mode.
     * |        |          |Note: It is suggest the function is enabled when the WKTHREN (UART_WKUPEN[2]) is set to 1.
     * |[4]     |WKADRMEN  |RS-485 Address Match Wake-up Enable Bit
     * |        |          |0 = RS-485 ADD mode address match wake-up function Disabled.
     * |        |          |1 = RS-485 AAD mode address match wake-up function Enabled when the system is in power-down mode.
     * @var UART_T::WKUPSTS
     * Offset: 0x44  UART Wake-up Status Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CTSWKSTS  |nCTS Wake-up Flag (Read Only)
     * |        |          |0 = Chip stays in power-down state.
     * |        |          |1 = Chip wake-up from power-down state by nCTS wake-up.
     * |        |          |Note1: If WKCTSEN (UART_ WKUPEN [0])is enabled, the wake-up function is generated.
     * |        |          |Note2: This bit is read only, but can be cleared by writing '1' to it.
     * |[1]     |DATWKSTS  |Data Wake-up Flag (Read Only)
     * |        |          |This bit is set if chip wake-up from power-down state by data wake-up.
     * |        |          |0 = Chip stays in power-down state.
     * |        |          |1 = Chip wake-up from power-down state by data wake-up.
     * |        |          |Note1: If WKDATEN (UART_ WKUPEN [1]) is enabled, the wake-up function is generated.
     * |        |          |Note2: This bit is read only, but can be cleared by writing '1' to it
     * |[2]     |THRWKSTS  |Threshold Wake-up Flag (Read Only)
     * |        |          |0 = Chip stays in power-down state.
     * |        |          |1 = Chip wake-up from power-down state by FIFO threshold wake-up.
     * |        |          |Note1: If WKTHREN (UART_ WKUPEN [2])is enabled, the wake-up function is generated.
     * |        |          |Note2: This bit is read only, but can be cleared by writing '1' to it.
     * |[3]     |THRTOWKSTS|Threshold Wake-up Time Out Flag (Read Only)
     * |        |          |0 = Chip stays in power-down state.
     * |        |          |1 = Chip wake-up from power-down state by FIFO threshold time out wake-up.
     * |        |          |Note1: If WKTHRTOEN (UART_ WKUPEN [3])is enabled, the wake-up function is generated.
     * |        |          |Note2: This bit is read only, but can be cleared by writing '1' to it.
     * |[4]     |ADRWKSTS  |RS-485 Address Byte Detection Wake-up Flag (Read Only)
     * |        |          |0 = Chip stays in power-down state.
     * |        |          |1 = Chip wake-up from power-down state by Receiver detects a data that is an address bit (bit 9 ='1').
     * |        |          |Note1: If WKADRMEN (UART_WKUPEN[4])is enabled, the wake-up function is generated.
     * |        |          |Note2: This field is used for RS-485 function mode and ADDRDEN (UART_ALT_CSR[19]) is set to 1 to enable Address detection mode .
     * |        |          |Note2: This bit is read only, but can be cleared by writing '1' to it.
     */
    __IO uint32_t DAT;                   /*!< [0x0000] UART Receive/Transmit Buffer Register                            */
    __IO uint32_t CTRL;                   /*!< [0x0004] UART Control Register.                                           */
    __IO uint32_t LINE;                 /*!< [0x0008] UART Transfer Line Control Register.                             */
    __IO uint32_t INTEN;                   /*!< [0x000c] UART Interrupt Enable Register.                                  */
    __IO uint32_t INTSTS;                   /*!< [0x0010] UART Interrupt Status Register.                                  */
    __IO uint32_t TRSR;                  /*!< [0x0014] UART Transfer Status Register.                                   */
    __IO uint32_t FIFOSTS;                   /*!< [0x0018] UART FIFO Status Register.                                       */
    __IO uint32_t MODEM;                  /*!< [0x001c] UART Modem Control Status Register.                              */
    __IO uint32_t TOUT;                 /*!< [0x0020] UART Time-Out Control Register.                                  */
    __IO uint32_t BAUD;                  /*!< [0x0024] UART Baud Rate Divisor Register.                                 */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE0[2];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t IRDA;                  /*!< [0x0030] UART IrDA Control Register.                                      */
    __IO uint32_t ALTCTL;               /*!< [0x0034] UART Alternate Control State Register.                           */
    __IO uint32_t FUNCSEL;               /*!< [0x0038] UART Function Select Register.                                   */
    __IO uint32_t BRCOMPAT;              /*!< [0x003c] UART Baud Rate Compensation Register.                            */
    __IO uint32_t WKUPEN;                /*!< [0x0040] UART Wake-up Enable Register.                                    */
    __IO uint32_t WKUPSTS;               /*!< [0x0044] UART Wake-up Status Register.                                    */

} UART_T;

/**
    @addtogroup UART_CONST UART Bit Field Definition
    Constant Definitions for UART Controller
@{ */

#define UART_DAT_DAT_Pos                 (0)                                               /*!< UART_T::DAT: DAT Position              */
#define UART_DAT_DAT_Msk                 (0xfful << UART_RBR_RBR_Pos)                      /*!< UART_T::DAT: DAT Mask                  */

#define UART_CTRL_RXRST_Pos               (0)                                              /*!< UART_T::CTRL: RXRST Position            */
#define UART_CTRL_RXRST_Msk               (0x1ul << UART_CTRL_RXRST_Pos)                   /*!< UART_T::CTRL: RXRST Mask                */

#define UART_CTRL_TXRST_Pos               (1)                                              /*!< UART_T::CTRL: TXRST Position            */
#define UART_CTRL_TXRST_Msk               (0x1ul << UART_CTRL_TXRST_Pos)                   /*!< UART_T::CTRL: TXRST Mask                */

#define UART_CTRL_RXOFF_Pos               (2)                                              /*!< UART_T::CTRL: RXOFF Position            */
#define UART_CTRL_RXOFF_Msk               (0x1ul << UART_CTRL_RXOFF_Pos)                   /*!< UART_T::CTRL: RXOFF Mask                */

#define UART_CTRL_TXOFF_Pos               (3)                                              /*!< UART_T::CTRL: TXOFF Position            */
#define UART_CTRL_TXOFF_Msk               (0x1ul << UART_CTRL_TXOFF_Pos)                   /*!< UART_T::CTRL: TXOFF Mask                */

#define UART_CTRL_ATORTSEN_Pos            (4)                                              /*!< UART_T::CTRL: ATORTSEN Position         */
#define UART_CTRL_ATORTSEN_Msk            (0x1ul << UART_CTRL_ATORTSEN_Pos)                /*!< UART_T::CTRL: ATORTSEN Mask             */

#define UART_CTRL_ATOCTSEN_Pos            (5)                                              /*!< UART_T::CTRL: ATOCTSEN Position         */
#define UART_CTRL_ATOCTSEN_Msk            (0x1ul << UART_CTRL_ATOCTSEN_Pos)                /*!< UART_T::CTRL: ATOCTSEN Mask             */

#define UART_CTRL_RXDMAEN_Pos             (6)                                              /*!< UART_T::CTRL: RXDMAEN Position          */
#define UART_CTRL_RXDMAEN_Msk             (0x1ul << UART_CTRL_RXDMAEN_Pos)                 /*!< UART_T::CTRL: RXDMAEN Mask              */

#define UART_CTRL_TXDMAEN_Pos             (7)                                              /*!< UART_T::CTRL: TXDMAEN Position          */
#define UART_CTRL_TXDMAEN_Msk             (0x1ul << UART_CTRL_TXDMAEN_Pos)                 /*!< UART_T::CTRL: TXDMAEN Mask              */

#define UART_CTRL_FTOEN_Pos               (8)                                              /*!< UART_T::CTRL: FTOEN Position            */
#define UART_CTRL_FTOEN_Msk               (0x1ul << UART_CTRL_FTOEN_Pos)                   /*!< UART_T::CTRL: FTOEN Mask                */

#define UART_CTRL_ABRDEN_Pos              (12)                                             /*!< UART_T::CTRL: ABRDEN Position           */
#define UART_CTRL_ABRDEN_Msk              (0x1ul << UART_CTRL_ABRDEN_Pos)                  /*!< UART_T::CTRL: ABRDEN Mask               */

#define UART_CTRL_ABRDBITS_Pos            (13)                                             /*!< UART_T::CTRL: ABRDBITS Position         */
#define UART_CTRL_ABRDBITS_Msk            (0x3ul << UART_CTRL_ABRDBITS_Pos)                /*!< UART_T::CTRL: ABRDBITS Mask             */

#define UART_LINE_WLS_Pos               (0)                                               /*!< UART_T::LINE: WLS Position            */
#define UART_LINE_WLS_Msk               (0x3ul << UART_LINE_WLS_Pos)                     /*!< UART_T::LINE: WLS Mask                */

#define UART_LINE_NSB_Pos               (2)                                               /*!< UART_T::LINE: NSB Position            */
#define UART_LINE_NSB_Msk               (0x1ul << UART_LINE_NSB_Pos)                     /*!< UART_T::LINE: NSB Mask                */

#define UART_LINE_PBE_Pos               (3)                                               /*!< UART_T::LINE: PBE Position            */
#define UART_LINE_PBE_Msk               (0x1ul << UART_LINE_PBE_Pos)                     /*!< UART_T::LINE: PBE Mask                */

#define UART_LINE_EPE_Pos               (4)                                               /*!< UART_T::LINE: EPE Position            */
#define UART_LINE_EPE_Msk               (0x1ul << UART_LINE_EPE_Pos)                     /*!< UART_T::LINE: EPE Mask                */

#define UART_LINE_SPE_Pos               (5)                                               /*!< UART_T::LINE: SPE Position            */
#define UART_LINE_SPE_Msk               (0x1ul << UART_LINE_SPE_Pos)                     /*!< UART_T::LINE: SPE Mask                */

#define UART_LINE_BCB_Pos               (6)                                               /*!< UART_T::LINE: BCB Position            */
#define UART_LINE_BCB_Msk               (0x1ul << UART_LINE_BCB_Pos)                     /*!< UART_T::LINE: BCB Mask                */

#define UART_LINE_RFITL_Pos             (8)                                               /*!< UART_T::LINE: RFITL Position          */
#define UART_LINE_RFITL_Msk             (0x3ul << UART_LINE_RFITL_Pos)                   /*!< UART_T::LINE: RFITL Mask              */

#define UART_LINE_RTSTRGLV_Pos          (12)                                              /*!< UART_T::LINE: RTSTRGLV Position       */
#define UART_LINE_RTSTRGLV_Msk          (0x3ul << UART_LINE_RTSTRGLV_Pos)                /*!< UART_T::LINE: RTSTRGLV Mask           */

#define UART_INTEN_RDAIEN_Pos              (0)                                             /*!< UART_T::INTEN: RDAIEN Position           */
#define UART_INTEN_RDAIEN_Msk              (0x1ul << UART_INTEN_RDAIEN_Pos)                /*!< UART_T::INTEN: RDAIEN Mask               */

#define UART_INTEN_THREIEN_Pos             (1)                                             /*!< UART_T::INTEN: THREIEN Position          */
#define UART_INTEN_THREIEN_Msk             (0x1ul << UART_INTEN_THREIEN_Pos)               /*!< UART_T::INTEN: THREIEN Mask              */

#define UART_INTEN_RLSIEN_Pos              (2)                                             /*!< UART_T::INTEN: RLSIEN Position           */
#define UART_INTEN_RLSIEN_Msk              (0x1ul << UART_INTEN_RLSIEN_Pos)                /*!< UART_T::INTEN: RLSIEN Mask               */

#define UART_INTEN_MODEMIEN_Pos            (3)                                             /*!< UART_T::INTEN: MODEMIEN Position         */
#define UART_INTEN_MODEMIEN_Msk            (0x1ul << UART_INTEN_MODEMIEN_Pos)              /*!< UART_T::INTEN: MODEMIEN Mask             */

#define UART_INTEN_RXTOIEN_Pos             (4)                                             /*!< UART_T::INTEN: RXTOIEN Position          */
#define UART_INTEN_RXTOIEN_Msk             (0x1ul << UART_INTEN_RXTOIEN_Pos)               /*!< UART_T::INTEN: RXTOIEN Mask              */

#define UART_INTEN_BUFERRIEN_Pos           (5)                                             /*!< UART_T::INTEN: BUFERRIEN Position        */
#define UART_INTEN_BUFERRIEN_Msk           (0x1ul << UART_INTEN_BUFERRIEN_Pos)             /*!< UART_T::INTEN: BUFERRIEN Mask            */

#define UART_INTEN_WKUPIEN_Pos             (6)                                             /*!< UART_T::INTEN: WKUPIEN Position          */
#define UART_INTEN_WKUPIEN_Msk             (0x1ul << UART_INTEN_WKUPIEN_Pos)               /*!< UART_T::INTEN: WKUPIEN Mask              */

#define UART_INTEN_ABRIEN_Pos              (7)                                             /*!< UART_T::INTEN: ABRIEN Position           */
#define UART_INTEN_ABRIEN_Msk              (0x1ul << UART_INTEN_ABRIEN_Pos)                /*!< UART_T::INTEN: ABRIEN Mask               */

#define UART_INTEN_LINIEN_Pos              (8)                                             /*!< UART_T::INTEN: LINIEN Position           */
#define UART_INTEN_LINIEN_Msk              (0x1ul << UART_INTEN_LINIEN_Pos)                /*!< UART_T::INTEN: LINIEN Mask               */

#define UART_INTEN_TXENDIEN_Pos            (9)                                             /*!< UART_T::INTEN: TXENDIEN Position         */
#define UART_INTEN_TXENDIEN_Msk            (0x1ul << UART_INTEN_TXENDIEN_Pos)              /*!< UART_T::INTEN: TXENDIEN Mask             */

#define UART_INTSTS_RDAIF_Pos               (0)                                            /*!< UART_T::INTSTS: RDAIF Position            */
#define UART_INTSTS_RDAIF_Msk               (0x1ul << UART_INTSTS_RDAIF_Pos)               /*!< UART_T::INTSTS: RDAIF Mask                */

#define UART_INTSTS_THREIF_Pos              (1)                                            /*!< UART_T::INTSTS: THREIF Position           */
#define UART_INTSTS_THREIF_Msk              (0x1ul << UART_INTSTS_THREIF_Pos)              /*!< UART_T::INTSTS: THREIF Mask               */

#define UART_INTSTS_RLSIF_Pos               (2)                                            /*!< UART_T::INTSTS: RLSIF Position            */
#define UART_INTSTS_RLSIF_Msk               (0x1ul << UART_INTSTS_RLSIF_Pos)               /*!< UART_T::INTSTS: RLSIF Mask                */

#define UART_INTSTS_MODEMIF_Pos             (3)                                            /*!< UART_T::INTSTS: MODEMIF Position          */
#define UART_INTSTS_MODEMIF_Msk             (0x1ul << UART_INTSTS_MODEMIF_Pos)             /*!< UART_T::INTSTS: MODEMIF Mask              */

#define UART_INTSTS_RXTOIF_Pos              (4)                                            /*!< UART_T::INTSTS: RXTOIF Position           */
#define UART_INTSTS_RXTOIF_Msk              (0x1ul << UART_INTSTS_RXTOIF_Pos)              /*!< UART_T::INTSTS: RXTOIF Mask               */

#define UART_INTSTS_BUFERRIF_Pos            (5)                                            /*!< UART_T::INTSTS: BUFERRIF Position         */
#define UART_INTSTS_BUFERRIF_Msk            (0x1ul << UART_INTSTS_BUFERRIF_Pos)            /*!< UART_T::INTSTS: BUFERRIF Mask             */

#define UART_INTSTS_WKUPIF_Pos              (6)                                            /*!< UART_T::INTSTS: WKUPIF Position           */
#define UART_INTSTS_WKUPIF_Msk              (0x1ul << UART_INTSTS_WKUPIF_Pos)              /*!< UART_T::INTSTS: WKUPIF Mask               */

#define UART_INTSTS_ABRIF_Pos               (7)                                            /*!< UART_T::INTSTS: ABRIF Position            */
#define UART_INTSTS_ABRIF_Msk               (0x1ul << UART_INTSTS_ABRIF_Pos)               /*!< UART_T::INTSTS: ABRIF Mask                */

#define UART_INTSTS_LINIF_Pos              (8)                                             /*!< UART_T::INTSTS: LINEIF Position           */
#define UART_INTSTS_LINIF_Msk              (0x1ul << UART_INTSTS_LINIF_Pos)                /*!< UART_T::INTSTS: LINEIF Mask               */

#define UART_TRSR_ADDRDETF_Pos           (0)                                               /*!< UART_T::TRSR: ADDRDETF Position        */
#define UART_TRSR_ADDRDETF_Msk           (0x1ul << UART_TRSR_ADDRDETF_Pos)                 /*!< UART_T::TRSR: ADDRDETF Mask            */

#define UART_TRSR_ABRDIF_Pos             (1)                                               /*!< UART_T::TRSR: ABRDIF Position          */
#define UART_TRSR_ABRDIF_Msk             (0x1ul << UART_TRSR_ABRDIF_Pos)                   /*!< UART_T::TRSR: ABRDIF Mask              */

#define UART_TRSR_ABRDTOIF_Pos           (2)                                               /*!< UART_T::TRSR: ABRDTOIF Position        */
#define UART_TRSR_ABRDTOIF_Msk           (0x1ul << UART_TRSR_ABRDTOIF_Pos)                 /*!< UART_T::TRSR: ABRDTOIF Mask            */

#define UART_TRSR_LINTXIF_Pos            (3)                                               /*!< UART_T::TRSR: LINTXIF Position         */
#define UART_TRSR_LINTXIF_Msk            (0x1ul << UART_TRSR_LINTXIF_Pos)                  /*!< UART_T::TRSR: LINTXIF Mask             */

#define UART_TRSR_LINRXIF_Pos            (4)                                               /*!< UART_T::TRSR: LINRXIF Position         */
#define UART_TRSR_LINRXIF_Msk            (0x1ul << UART_TRSR_LINRXIF_Pos)                  /*!< UART_T::TRSR: LINRXIF Mask             */

#define UART_TRSR_BITEF_Pos              (5)                                               /*!< UART_T::TRSR: BITEF Position           */
#define UART_TRSR_BITEF_Msk              (0x1ul << UART_TRSR_BITEF_Pos)                    /*!< UART_T::TRSR: BITEF Mask               */

#define UART_TRSR_RXBUSY_Pos             (7)                                               /*!< UART_T::TRSR: RXBUSY Position          */
#define UART_TRSR_RXBUSY_Msk             (0x1ul << UART_TRSR_RXBUSY_Pos)                   /*!< UART_T::TRSR: RXBUSY Mask              */

#define UART_TRSR_SLVSYNCF_Pos           (8)                                               /*!< UART_T::TRSR: SLVSYNCF Position        */
#define UART_TRSR_SLVSYNCF_Msk           (0x1ul << UART_TRSR_SLVSYNCF_Pos)                 /*!< UART_T::TRSR: SLVSYNCF Mask            */

#define UART_FIFOSTS_RXOVIF_Pos              (0)                                           /*!< UART_T::FIFOSTS: RXOVIF Position           */
#define UART_FIFOSTS_RXOVIF_Msk              (0x1ul << UART_FIFOSTS_RXOVIF_Pos)            /*!< UART_T::FIFOSTS: RXOVIF Mask               */

#define UART_FIFOSTS_RXEMPTY_Pos             (1)                                           /*!< UART_T::FIFOSTS: RXEMPTY Position          */
#define UART_FIFOSTS_RXEMPTY_Msk             (0x1ul << UART_FIFOSTS_RXEMPTY_Pos)           /*!< UART_T::FIFOSTS: RXEMPTY Mask              */

#define UART_FIFOSTS_RXFULL_Pos              (2)                                           /*!< UART_T::FIFOSTS: RXFULL Position           */
#define UART_FIFOSTS_RXFULL_Msk              (0x1ul << UART_FIFOSTS_RXFULL_Pos)            /*!< UART_T::FIFOSTS: RXFULL Mask               */

#define UART_FIFOSTS_PEF_Pos                 (4)                                           /*!< UART_T::FIFOSTS: PEF Position              */
#define UART_FIFOSTS_PEF_Msk                 (0x1ul << UART_FIFOSTS_PEF_Pos)               /*!< UART_T::FIFOSTS: PEF Mask                  */

#define UART_FIFOSTS_FEF_Pos                 (5)                                           /*!< UART_T::FIFOSTS: FEF Position              */
#define UART_FIFOSTS_FEF_Msk                 (0x1ul << UART_FIFOSTS_FEF_Pos)               /*!< UART_T::FIFOSTS: FEF Mask                  */

#define UART_FIFOSTS_BIF_Pos                 (6)                                           /*!< UART_T::FIFOSTS: BIF Position              */
#define UART_FIFOSTS_BIF_Msk                 (0x1ul << UART_FIFOSTS_BIF_Pos)               /*!< UART_T::FIFOSTS: BIF Mask                  */

#define UART_FIFOSTS_TXOVIF_Pos              (8)                                           /*!< UART_T::FIFOSTS: TXOVIF Position           */
#define UART_FIFOSTS_TXOVIF_Msk              (0x1ul << UART_FIFOSTS_TXOVIF_Pos)            /*!< UART_T::FIFOSTS: TXOVIF Mask               */

#define UART_FIFOSTS_TXEMPTY_Pos             (9)                                           /*!< UART_T::FIFOSTS: TXEMPTY Position          */
#define UART_FIFOSTS_TXEMPTY_Msk             (0x1ul << UART_FIFOSTS_TXEMPTY_Pos)           /*!< UART_T::FIFOSTS: TXEMPTY Mask              */

#define UART_FIFOSTS_TXFULL_Pos              (10)                                          /*!< UART_T::FIFOSTS: TXFULL Position           */
#define UART_FIFOSTS_TXFULL_Msk              (0x1ul << UART_FIFOSTS_TXFULL_Pos)            /*!< UART_T::FIFOSTS: TXFULL Mask               */

#define UART_FIFOSTS_TXENDF_Pos              (11)                                          /*!< UART_T::FIFOSTS: TXENDF Position           */
#define UART_FIFOSTS_TXENDF_Msk              (0x1ul << UART_FIFOSTS_TXENDF_Pos)            /*!< UART_T::FIFOSTS: TXENDF Mask               */

#define UART_FIFOSTS_RXPTR_Pos               (16)                                          /*!< UART_T::FIFOSTS: RXPTR Position            */
#define UART_FIFOSTS_RXPTR_Msk               (0x1ful << UART_FIFOSTS_RXPTR_Pos)            /*!< UART_T::FIFOSTS: RXPTR Mask                */

#define UART_FIFOSTS_TXPTR_Pos               (24)                                          /*!< UART_T::FIFOSTS: TXPTR Position            */
#define UART_FIFOSTS_TXPTR_Msk               (0x1ful << UART_FIFOSTS_TXPTR_Pos)            /*!< UART_T::FIFOSTS: TXPTR Mask                */

#define UART_MODEM_RTSACTLV_Pos           (0)                                              /*!< UART_T::MODEM: RTSACTLV Position        */
#define UART_MODEM_RTSACTLV_Msk           (0x1ul << UART_MODEM_RTSACTLV_Pos)               /*!< UART_T::MODEM: RTSACTLV Mask            */

#define UART_MODEM_RTSSTS_Pos             (1)                                              /*!< UART_T::MODEM: RTSSTS Position          */
#define UART_MODEM_RTSSTS_Msk             (0x1ul << UART_MODEM_RTSSTS_Pos)                 /*!< UART_T::MODEM: RTSSTS Mask              */

#define UART_MODEM_CTSACTLV_Pos           (16)                                             /*!< UART_T::MODEM: CTSACTLV Position        */
#define UART_MODEM_CTSACTLV_Msk           (0x1ul << UART_MODEM_CTSACTLV_Pos)               /*!< UART_T::MODEM: CTSACTLV Mask            */

#define UART_MODEM_CTSSTS_Pos             (17)                                             /*!< UART_T::MODEM: CTSSTS Position          */
#define UART_MODEM_CTSSTS_Msk             (0x1ul << UART_MODEM_CTSSTS_Pos)                 /*!< UART_T::MODEM: CTSSTS Mask              */

#define UART_MODEM_CTSDETF_Pos            (18)                                             /*!< UART_T::MODEM: CTSDETF Position         */
#define UART_MODEM_CTSDETF_Msk            (0x1ul << UART_MODEM_CTSDETF_Pos)                /*!< UART_T::MODEM: CTSDETF Mask             */

#define UART_TOUT_TOIC_Pos              (0)                                                /*!< UART_T::TOUT: TOIC Position           */
#define UART_TOUT_TOIC_Msk              (0x1fful << UART_TOUT_TOIC_Pos)                    /*!< UART_T::TOUT: TOIC Mask               */

#define UART_TOUT_DLY_Pos               (16)                                               /*!< UART_T::TOUT: DLY Position            */
#define UART_TOUT_DLY_Msk               (0xfful << UART_TOUT_DLY_Pos)                      /*!< UART_T::TOUT: DLY Mask                */

#define UART_BAUD_BRD_Pos                (0)                                               /*!< UART_T::BAUD: BRD Position             */
#define UART_BAUD_BRD_Msk                (0xfffful << UART_BAUD_BRD_Pos)                   /*!< UART_T::BAUD: BRD Mask                 */

#define UART_BAUD_DIV16EN_Pos            (31)                                              /*!< UART_T::BAUD: DIV16EN Position         */
#define UART_BAUD_DIV16EN_Msk            (0x1ul << UART_BAUD_DIV16EN_Pos)                  /*!< UART_T::BAUD: DIV16EN Mask             */

#define UART_IRDA_TXEN_Pos               (1)                                               /*!< UART_T::IRDA: TXEN Position            */
#define UART_IRDA_TXEN_Msk               (0x1ul << UART_IRDA_TXEN_Pos)                     /*!< UART_T::IRDA: TXEN Mask                */

#define UART_IRDA_TXINV_Pos              (5)                                               /*!< UART_T::IRDA: TXINV Position           */
#define UART_IRDA_TXINV_Msk              (0x1ul << UART_IRDA_TXINV_Pos)                    /*!< UART_T::IRDA: TXINV Mask               */

#define UART_IRDA_RXINV_Pos              (6)                                               /*!< UART_T::IRDA: RXINV Position           */
#define UART_IRDA_RXINV_Msk              (0x1ul << UART_IRDA_RXINV_Pos)                    /*!< UART_T::IRDA: RXINV Mask               */

#define UART_ALTCTL_BRKFL_Pos           (0)                                                /*!< UART_T::ALTCTL: BRKFL Position        */
#define UART_ALTCTL_BRKFL_Msk           (0x7ul << UART_ALTCTL_BRKFL_Pos)                   /*!< UART_T::ALTCTL: BRKFL Mask            */

#define UART_ALTCTL_LINHSEL_Pos         (4)                                                /*!< UART_T::ALTCTL: LINHSEL Position      */
#define UART_ALTCTL_LINHSEL_Msk         (0x3ul << UART_ALTCTL_LINHSEL_Pos)                 /*!< UART_T::ALTCTL: LINHSEL Mask          */

#define UART_ALTCTL_LINRXEN_Pos         (6)                                                /*!< UART_T::ALTCTL: LINRXEN Position      */
#define UART_ALTCTL_LINRXEN_Msk         (0x1ul << UART_ALTCTL_LINRXEN_Pos)                 /*!< UART_T::ALTCTL: LINRXEN Mask          */

#define UART_ALTCTL_LINTXEN_Pos         (7)                                                /*!< UART_T::ALTCTL: LINTXEN Position      */
#define UART_ALTCTL_LINTXEN_Msk         (0x1ul << UART_ALTCTL_LINTXEN_Pos)                 /*!< UART_T::ALTCTL: LINTXEN Mask          */

#define UART_ALTCTL_BITERREN_Pos        (8)                                                /*!< UART_T::ALTCTL: BITERREN Position     */
#define UART_ALTCTL_BITERREN_Msk        (0x1ul << UART_ALTCTL_BITERREN_Pos)                /*!< UART_T::ALTCTL: BITERREN Mask         */

#define UART_ALTCTL_RS485NMM_Pos        (16)                                               /*!< UART_T::ALTCTL: RS485NMM Position     */
#define UART_ALTCTL_RS485NMM_Msk        (0x1ul << UART_ALTCTL_RS485NMM_Pos)                /*!< UART_T::ALTCTL: RS485NMM Mask         */

#define UART_ALTCTL_RS485AAD_Pos        (17)                                               /*!< UART_T::ALTCTL: RS485AAD Position     */
#define UART_ALTCTL_RS485AAD_Msk        (0x1ul << UART_ALTCTL_RS485AAD_Pos)                /*!< UART_T::ALTCTL: RS485AAD Mask         */

#define UART_ALTCTL_RS485AUD_Pos        (18)                                               /*!< UART_T::ALTCTL: RS485AUD Position     */
#define UART_ALTCTL_RS485AUD_Msk        (0x1ul << UART_ALTCTL_RS485AUD_Pos)                /*!< UART_T::ALTCTL: RS485AUD Mask         */

#define UART_ALTCTL_ADDRDEN_Pos         (19)                                               /*!< UART_T::ALTCTL: ADDRDEN Position      */
#define UART_ALTCTL_ADDRDEN_Msk         (0x1ul << UART_ALTCTL_ADDRDEN_Pos)                 /*!< UART_T::ALTCTL: ADDRDEN Mask          */

#define UART_ALTCTL_ADRMPID_Pos         (24)                                               /*!< UART_T::ALTCTL: ADRMPID Position      */
#define UART_ALTCTL_ADRMPID_Msk         (0xfful << UART_ALTCTL_ADRMPID_Pos)                /*!< UART_T::ALTCTL: ADRMPID Mask          */

#define UART_FUNCSEL_FUNCSEL_Pos         (0)                                               /*!< UART_T::FUNCSEL: FUNCSEL Position      */
#define UART_FUNCSEL_FUNCSEL_Msk         (0x3ul << UART_FUNCSEL_FUNCSEL_Pos)               /*!< UART_T::FUNCSEL: FUNCSEL Mask          */

#define UART_BRCOMPAT_BRCOMPAT_Pos       (0)                                               /*!< UART_T::BRCOMPAT: BRCOMPAT Position    */
#define UART_BRCOMPAT_BRCOMPAT_Msk       (0x1fful << UART_BRCOMPAT_BRCOMPAT_Pos)           /*!< UART_T::BRCOMPAT: BRCOMPAT Mask        */

#define UART_BRCOMPAT_BRCOMPDEC_Pos      (31)                                              /*!< UART_T::BRCOMPAT: BRCOMPDEC Position   */
#define UART_BRCOMPAT_BRCOMPDEC_Msk      (0x1ul << UART_BRCOMPAT_BRCOMPDEC_Pos)            /*!< UART_T::BRCOMPAT: BRCOMPDEC Mask       */

#define UART_WKUPEN_WKCTSEN_Pos          (0)                                               /*!< UART_T::WKUPEN: WKCTSEN Position       */
#define UART_WKUPEN_WKCTSEN_Msk          (0x1ul << UART_WKUPEN_WKCTSEN_Pos)                /*!< UART_T::WKUPEN: WKCTSEN Mask           */

#define UART_WKUPEN_WKDATEN_Pos          (1)                                               /*!< UART_T::WKUPEN: WKDATEN Position       */
#define UART_WKUPEN_WKDATEN_Msk          (0x1ul << UART_WKUPEN_WKDATEN_Pos)                /*!< UART_T::WKUPEN: WKDATEN Mask           */

#define UART_WKUPEN_WKTHREN_Pos          (2)                                               /*!< UART_T::WKUPEN: WKTHREN Position       */
#define UART_WKUPEN_WKTHREN_Msk          (0x1ul << UART_WKUPEN_WKTHREN_Pos)                /*!< UART_T::WKUPEN: WKTHREN Mask           */

#define UART_WKUPEN_WKTHRTOEN_Pos        (3)                                               /*!< UART_T::WKUPEN: WKTHRTOEN Position     */
#define UART_WKUPEN_WKTHRTOEN_Msk        (0x1ul << UART_WKUPEN_WKTHRTOEN_Pos)              /*!< UART_T::WKUPEN: WKTHRTOEN Mask         */

#define UART_WKUPEN_WKADRMEN_Pos         (4)                                               /*!< UART_T::WKUPEN: WKADRMEN Position      */
#define UART_WKUPEN_WKADRMEN_Msk         (0x1ul << UART_WKUPEN_WKADRMEN_Pos)               /*!< UART_T::WKUPEN: WKADRMEN Mask          */

#define UART_WKUPSTS_CTSWKSTS_Pos        (0)                                               /*!< UART_T::WKUPSTS: CTSWKSTS Position     */
#define UART_WKUPSTS_CTSWKSTS_Msk        (0x1ul << UART_WKUPSTS_CTSWKSTS_Pos)              /*!< UART_T::WKUPSTS: CTSWKSTS Mask         */

#define UART_WKUPSTS_DATWKSTS_Pos        (1)                                               /*!< UART_T::WKUPSTS: DATWKSTS Position     */
#define UART_WKUPSTS_DATWKSTS_Msk        (0x1ul << UART_WKUPSTS_DATWKSTS_Pos)              /*!< UART_T::WKUPSTS: DATWKSTS Mask         */

#define UART_WKUPSTS_THRWKSTS_Pos        (2)                                               /*!< UART_T::WKUPSTS: THRWKSTS Position     */
#define UART_WKUPSTS_THRWKSTS_Msk        (0x1ul << UART_WKUPSTS_THRWKSTS_Pos)              /*!< UART_T::WKUPSTS: THRWKSTS Mask         */

#define UART_WKUPSTS_THRTOWKSTS_Pos      (3)                                               /*!< UART_T::WKUPSTS: THRTOWKSTS Position   */
#define UART_WKUPSTS_THRTOWKSTS_Msk      (0x1ul << UART_WKUPSTS_THRTOWKSTS_Pos)            /*!< UART_T::WKUPSTS: THRTOWKSTS Mask       */

#define UART_WKUPSTS_ADRWKSTS_Pos        (4)                                               /*!< UART_T::WKUPSTS: ADRWKSTS Position     */
#define UART_WKUPSTS_ADRWKSTS_Msk        (0x1ul << UART_WKUPSTS_ADRWKSTS_Pos)              /*!< UART_T::WKUPSTS: ADRWKSTS Mask         */

/**@}*/ /* UART_CONST */
/**@}*/ /* end of UART register group */


/*---------------------- Smart Card Host Interface Controller -------------------------*/
/**
    @addtogroup SC Smart Card Host Interface Controller(SC)
    Memory Mapped Structure for SC Controller
@{ */

typedef struct
{


    /**
     * @var SC_T::DAT
     * Offset: 0x00  SC Receive/Transmit Holding Buffer Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |DAT       |Receive/Transmit Holding Buffer
     * |        |          |Write Operation:
     * |        |          |By writing data to DAT, the SC will send out an 8-bit data.
     * |        |          |Note: If SCEN(SC_CTL[0]) is not enabled, DAT cannot be programmed.
     * |        |          |Read Operation:
     * |        |          |By reading DAT, the SC will return an 8-bit received data.
     * @var SC_T::CTL
     * Offset: 0x04  SC Control Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SCEN      |SC Engine Enable Bit
     * |        |          |Set this bit to 1 to enable SC operation
     * |        |          |If this bit is cleared, SC will force all transition to IDLE state
     * |        |          |Note1: SCEN must be set to 1 before filling in other registers, or smart card will not work properly.
     * |        |          |Note2: If SCEN is activated, all function can work correctly
     * |        |          |If SCEN is not activated, when CPU write data to SMC, only Flip-flop which works in PCLK domain will turn on for two PCLK cycle, Flip-flop working in SCLK domain will not be turn on.
     * |[1]     |RXOFF     |RX Transition Disable Bit
     * |        |          |0 = The receiver Enabled.
     * |        |          |1 = The receiver Disabled.
     * |        |          |Note1: If AUTOCEN (SC_CTL[3])is enabled, these fields must be ignored.
     * |        |          |Note2: After hardware activation and hardware warm reset are done, RXOFF is set to 0 automatically.
     * |[2]     |TXOFF     |TX Transition Disable Bit
     * |        |          |0 = The transceiver Enabled.
     * |        |          |1 = The transceiver Disabled.
     * |[3]     |AUTOCEN   |Auto Convention Enable Bit
     * |        |          |0 = Auto-convention Disabled.
     * |        |          |1 = Auto-convention Enabled
     * |        |          |When hardware receives TS in answer to reset state and the TS is direct convention, CONSEL(SC_CTL[5:4]) will be set to 00 automatically, otherwise if the TS is inverse convention, and CONSEL (SC_CTL[5:4]) will be set to 11.
     * |        |          |If software enables auto convention function, the setting step must be done before Answer to Reset state and the first data must be 0x3B or 0x3F
     * |        |          |After hardware received first data and stored it at buffer, hardware will decided the convention and change the CONSEL (SC_CTL[5:4]) bits automatically
     * |        |          |If the first data is not 0x3B or 0x3F, hardware will generate an interrupt INT_ACON_ERR (if ACERRIEN (SC_INTEN[10]) = 1 to CPU.
     * |[5:4]   |CONSEL    |Convention Selection
     * |        |          |00 = Direct convention.
     * |        |          |01 = Reserved.
     * |        |          |10 = Reserved.
     * |        |          |11 = Inverse convention.
     * |        |          |Note: If AUTOCEN(SC_CTL[3]) is enabled, this field is ignored.
     * |[7:6]   |RXTRGLV   |Rx Buffer Trigger Level
     * |        |          |When the number of bytes in the receiving buffer equals the RXTRGLV, the RDAIF will be set (if [RDAIEN](SC_INTEN[0])  is enabled, an interrupt will be generated).
     * |        |          |00 = INTR_RDA Trigger Level with 01 Bytes.
     * |        |          |01 = INTR_RDA Trigger Level with 02 Bytes.
     * |        |          |10 = INTR_RDA Trigger Level with 03 Bytes.
     * |        |          |11 = Reserved.
     * |[12:8]  |BGT       |Block Guard Time (BGT)
     * |        |          |Block guard time means the minimum bit length between the leading edges of two consecutive characters between different transfer directions
     * |        |          |This field indicates the counter for the bit length of block guard time
     * |        |          |According to ISO7816-3, in T = 0 mode, software must fill 15 (real block guard time = 16.5) to this field; in T = 1 mode, software must fill 21 (real block guard time = 22.5) to it.
     * |        |          |Note: The real block guard time is BGT + 1.
     * |[14:13] |TMRSEL    |Timer Selection
     * |        |          |00 = All internal timer function Disabled.
     * |        |          |11 = Internal 24 bit timer and two 8 bit timers Enabled
     * |        |          |Software can configure them by setting SC_TMRCTL0 [23:0], SC_TMRCTL1 [7:0] and SC_TMRCTL2 [7:0].
     * |        |          |Other configurations are reserved
     * |[15]    |NSB       |Stop Bit Length
     * |        |          |This field indicates the length of stop bit.
     * |        |          |0 = The stop bit length is 2 ETU.
     * |        |          |1= The stop bit length is 1 ETU.
     * |        |          |Note: The default stop bit length is 2. SMC and UART adopt NSB to program the stop bit length.
     * |[18:16] |RXRTY     |RX Error Retry Count Number
     * |        |          |This field indicates the maximum number of receiver retries that are allowed when parity error has occurred
     * |        |          |Note1: The real retry number is RX_ERETRY + 1, so 8 is the maximum retry number.
     * |        |          |Note2: This field cannot be changed when RX_ERETRY_EN enabled
     * |        |          |The change flow is to disable RX_ETRTRY_EN first and then fill in new retry value.
     * |[19]    |RXRTYEN   |RX Error Retry Enable Bit
     * |        |          |This bit enables receiver retry function when parity error has occurred.
     * |        |          |0 = RX error retry function Disabled.
     * |        |          |1 = RX error retry function Enabled.
     * |        |          |Note: Software must fill in the RXRTY value before enabling this bit.
     * |[22:20] |TXRTY     |TX Error Retry Count Number
     * |        |          |This field indicates the maximum number of transmitter retries that are allowed when parity error has occurred.
     * |        |          |Note1: The real retry number is TXRTY + 1, so 8 is the maximum retry number.
     * |        |          |Note2: This field cannot be changed when TXRTYEN enabled
     * |        |          |The change flow is to disable TXRTYEN first and then fill in new retry value.
     * |[23]    |TXRTYEN   |TX Error Retry Enable Bit
     * |        |          |This bit enables transmitter retry function when parity error has occurred.
     * |        |          |0 = TX error retry function Disabled.
     * |        |          |1 = TX error retry function Enabled.
     * |[25:24] |CDDBSEL   |Card Detect De-bounce Selection
     * |        |          |This field indicates the card detect de-bounce selection.
     * |        |          |00 = De-bounce sample card insert once per 384 (128 * 3) engine clocks and de-bounce sample card removal once per 128 engine clocks.
     * |        |          |Other configurations are reserved.
     * |[30]    |SYNC      |SYNC Flag Indicator(Read Only)
     * |        |          |Due to synchronization, software should check this bit before writing a new value to RXRTY and TXRTY.SYNC flag is
     * |        |          |0 = synchronizing is completion, user can write new data to RXRTY and TXRTY.
     * |        |          |1 = Last value is synchronizing.
     * @var SC_T::ALTCTL
     * Offset: 0x08  SC Alternate Control Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TXRST     |TX Software Reset
     * |        |          |When TXRST is set, all the bytes in the transmit buffer and TX internal state machine will be cleared.
     * |        |          |0 = No effect.
     * |        |          |1 = Reset the TX internal state machine and pointers.
     * |        |          |Note: This bit will be auto cleared after reset is complete.
     * |[1]     |RXRST     |Rx Software Reset
     * |        |          |When RXRST is set, all the bytes in the receiver buffer and Rx internal state machine will be cleared.
     * |        |          |0 = No effect.
     * |        |          |1 = Reset the Rx internal state machine and pointers.
     * |        |          |Note: This bit will be auto cleared after reset is complete.
     * |[2]     |DACTEN    |Deactivation Sequence Generator Enable Bit
     * |        |          |This bit enables SC controller to initiate the card by deactivation sequence
     * |        |          |0 = No effect.
     * |        |          |1 = Deactivation sequence generator Enabled.
     * |        |          |Note1: When the deactivation sequence completed, this bit will be cleared automatically and the INITIF(SC_INTSTS[8]) will be set to 1.
     * |        |          |Note2: This field will be cleared by TXRST (SC_ALTCTL[0]) and RXRST(SC_ALTCTL[1]).Thus,do not fill in this bit, TXRST, and RXRST at the same time.
     * |        |          |Note3: If SCEN (SC_CTL[0]) is not enabled, this filed cannot be programmed.
     * |[3]     |ACTEN     |Activation Sequence Generator Enable Bit
     * |        |          |This bit enables SC controller to initiate the card by activation sequence
     * |        |          |0 = No effect.
     * |        |          |1 = Activation sequence generator Enabled.
     * |        |          |Note1: When the activation sequence completed, this bit will be cleared automatically and the INITIF(SC_INTSTS[8]) will be set to 1, RXOFF(SC_CTL[2]) will be clear to 0.
     * |        |          |Note2: This field will be cleared by TXRST(SC_ALTCTL[0]) and RXRST(SC_ALTCTL[1]).Thus,do not fill in this bit, TXRST(SC_ALTCTL[0]), and RXRST(SC_ALTCTL[1]) at the same time.
     * |        |          |Note3: If SCEN(SC_CTL[0]) is not enabled, this filed cannot be programmed.
     * |[4]     |WARSTEN   |Warm Reset Sequence Generator Enable Bit
     * |        |          |This bit enables SC controller to initiate the card by warm reset sequence
     * |        |          |0 = No effect.
     * |        |          |1 = Warm reset sequence generator Enabled.
     * |        |          |Note1: When the warm reset sequence completed, this bit will be cleared automatically and the INITIF(SC_INTSTS[8]) will be set to 1, RXOFF(SC_CTL[2]) will be clear to 0.
     * |        |          |Note2: This field will be cleared by TXRST(SC_ALTCTL[0]) and RXRST(SC_ALTCTL[1]).Thus,do not fill in this bit, TXRST, and RXRST at the same time.
     * |        |          |Note3: If SCEN(SC_CTL[0]) is not enabled, this filed cannot be programmed.
     * |[5]     |CNTEN0    |Internal Timer0 Start Enable Bit
     * |        |          |This bit enables Timer 0 to start counting
     * |        |          |Software can fill 0 to stop it and set 1 to reload and count.
     * |        |          |0 = Stops counting.
     * |        |          |1 = Start counting.
     * |        |          |Note1: This field is used for internal 24 bit timer when TMRSEL (SC_CTL[14:13]) = 11,01,10.
     * |        |          |Note2: If the operation mode is not in auto-reload mode (SC_TMRCTL0[26] = 0), this bit will be auto-cleared by hardware.
     * |        |          |Note3: If SCEN(SC_CTL[0]) is not enabled, this filed cannot be programmed.
     * |[6]     |CNTEN1    |Internal Timer1 Start Enable Bit
     * |        |          |This bit enables Timer 1 to start counting
     * |        |          |Software can fill 0 to stop it and set 1 to reload and count.
     * |        |          |0 = Stops counting.
     * |        |          |1 = Start counting.
     * |        |          |Note1: This field is used for internal 8 bit timer when TMRSEL(SC_CTL[14:13]) = 11
     * |        |          |Do not fill in CNTEN1 when TMRSEL(SC_CTL[14:13]) = 00 or TMRSEL(SC_CTL[14:13]) = 01.
     * |        |          |Note2: If the operation mode is not in auto-reload mode (SC_TMRCTL1[26] = 0), this bit will be auto-cleared by hardware.
     * |        |          |Note3: If SCEN(SC_CTL[0]) is not enabled, this filed cannot be programmed.
     * |[7]     |CNTEN2    |Internal Timer2 Start Enable Bit
     * |        |          |This bit enables Timer 2 to start counting
     * |        |          |Software can fill 0 to stop it and set 1 to reload and count.
     * |        |          |0 = Stops counting.
     * |        |          |1 = Start counting.
     * |        |          |Note1: This field is used for internal 8 bit timer when TMRSEL(SC_CTL[14:13]) = 11
     * |        |          |Do not fill in CNTEN2 when TMRSEL(SC_CTL[14:13])= 00 or TMRSEL(SC_CTL[14:13]) = 01 or TMRSEL(SC_CTL[14:13]) = 10.
     * |        |          |Note2: If the operation mode is not in auto-reload mode (SC_TMRCTL2[26] = 0), this bit will be auto-cleared by hardware.
     * |        |          |Note3: If SCEN(SC_CTL[0]) is not enabled, this filed cannot be programmed.
     * |[9:8]   |INITSEL   |Initial Timing Selection
     * |        |          |This fields indicates the timing of hardware initial state (activation or warm-reset or deactivation).
     * |        |          |Unit: SC clock
     * |        |          |Activation: Refer to SC Activation Sequence in Figure 6.15-4 SC Activation Sequence.
     * |        |          |Warm-reset: Refer to Warm-Reset Sequence in Figure 6.15-5 SC Warm Reset Sequence
     * |        |          |Deactivation: Refer to Deactivation Sequence in Figure 6.15-6 SC Deactivation Sequence
     * |        |          |Note: When set Activation and Warm reset in mode 11, it may have deviation at most 128 cycles.
     * |[12]    |RXBGTEN   |Receiver Block Guard Time Function Enable Bit
     * |        |          |0 = Receiver block guard time function Disabled.
     * |        |          |1 = Receiver block guard time function Enabled.
     * |[13]    |ACTSTS0   |Internal Timer0 Active State (Read Only)
     * |        |          |This bit indicates the timer counter status of timer0.
     * |        |          |0 = Timer0 is not active.
     * |        |          |1 = Timer0 is active.
     * |        |          |Note: Timer0 is active does not always mean timer0 is counting the CNT(SC_TMRCTL0[23:0]).
     * |[14]    |ACTSTS1   |Internal Timer1 Active State (Read Only)
     * |        |          |This bit indicates the timer counter status of timer1.
     * |        |          |0 = Timer1 is not active.
     * |        |          |1 = Timer1 is active.
     * |        |          |Note: Timer1 is active does not always mean timer1 is counting the CNT(SC_TMRCTL1[7:0]).
     * |[15]    |ACTSTS2   |Internal Timer2 Active State (Read Only)
     * |        |          |This bit indicates the timer counter status of timer2.
     * |        |          |0 = Timer2 is not active.
     * |        |          |1 = Timer2 is active.
     * |        |          |Note: Timer2 is active does not always mean timer2 is counting the CNT(SC_TMRCTL2[7:0]).
     * |[16]    |OUTSEL    |Smartcard Data Pin Output Mode Selection
     * |        |          |Use this bit to select smartcard data pin output mode.
     * |        |          |0 = Quasi mode.
     * |        |          |1 = Open-drain mode.
     * @var SC_T::EGT
     * Offset: 0x0C  SC Extra Guard Time Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |EGT       |Extra Guard Time
     * |        |          |This field indicates the extra guard timer value.
     * |        |          |Note: The counter is ETU base .
     * @var SC_T::RXTOUT
     * Offset: 0x10  SC Receive buffer Time-out Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[8:0]   |RFTM      |SC Receiver FIFO Time-out
     * |        |          |The time-out counter resets and starts counting whenever the RX buffer received a new data word
     * |        |          |Once the counter decrease to 1 and no new data is received or CPU does not read data by reading SC_DAT buffer, a receiver time-out interrupt INT_RTMR will be generated(if RXTOIF(SC_INTEN[9]) = 1 ).
     * |        |          |Note1:The counter unit is ETU based and the interval of time-out is RFTM + 0.5.
     * |        |          |Note2: Filling in all 0 to this field indicates to disable this function.
     * @var SC_T::ETUCTL
     * Offset: 0x14  SC Element Time Unit Control Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |ETURDIV   |ETU Rate Divider
     * |        |          |The field indicates the clock rate divider.
     * |        |          |The real ETU is ETURDIV + 1.
     * |        |          |Note: Software can configure this field, but this field must be greater than 0x004.
     * @var SC_T::INTEN
     * Offset: 0x18  SC Interrupt Enable Control Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RDAIEN    |Receive Data Reach Interrupt Enable Bit
     * |        |          |This field is used to enable received data reaching trigger level RXTRGLV (SC_CTL[7:6]) interrupt.
     * |        |          |0 = Receive data reach trigger level interrupt Disabled.
     * |        |          |1 = Receive data reach trigger level interrupt Enabled.
     * |[1]     |TBEIEN    |Transmit Buffer Empty Interrupt Enable Bit
     * |        |          |This field is used to enable transmit buffer empty interrupt.
     * |        |          |0 = Transmit buffer empty interrupt Disabled.
     * |        |          |1 = Transmit buffer empty interrupt Enabled.
     * |[2]     |TERRIEN   |Transfer Error Interrupt Enable Bit
     * |        |          |This field is used to enable transfer error interrupt
     * |        |          |The transfer error states is at SC_STATUS register which includes receiver break error BEF(SC_STATUS[6]), frame error FEF(SC_STATUS[5]), parity error PEF(SC_STATUS[4]), receiver buffer overflow error RXOV(SC_STATUS[0]), transmit buffer overflow error TXOV(SC_STATUS[8]), receiver retry over limit error RXOVERR(SC_STATUS[22]) and transmitter retry over limit error TXOVERR (SC_STATUS[30]).
     * |        |          |0 = Transfer error interrupt Disabled.
     * |        |          |1 = Transfer error interrupt Enabled.
     * |[3]     |TMR0IEN   |Timer0 Interrupt Enable Bit
     * |        |          |This field is used to enable TMR0 interrupt.
     * |        |          |0 = Timer0 interrupt Disabled.
     * |        |          |1 = Timer0 interrupt Enabled.
     * |[4]     |TMR1IEN   |Timer1 Interrupt Enable Bit
     * |        |          |This field is used to enable the TMR1 interrupt.
     * |        |          |0 = Timer1 interrupt Disabled.
     * |        |          |1 = Timer1 interrupt Enabled.
     * |[5]     |TMR2IEN   |Timer2 Interrupt Enable Bit
     * |        |          |This field is used to enable TMR2 interrupt.
     * |        |          |0 = Timer2 interrupt Disabled.
     * |        |          |1 = Timer2 interrupt Enabled.
     * |[6]     |BGTIEN    |Block Guard Time Interrupt Enable Bit
     * |        |          |This field is used to enable block guard time interrupt.
     * |        |          |0 = Block guard time Disabled.
     * |        |          |1 = Block guard time Enabled.
     * |[7]     |CDIEN     |Card Detect Interrupt Enable Bit
     * |        |          |This field is used to enable card detect interrupt. The card detect status is CINSERT(SC_STATUS[12])
     * |        |          |0 = Card detect interrupt Disabled.
     * |        |          |1 = Card detect interrupt Enabled.
     * |[8]     |INITIEN   |Initial End Interrupt Enable Bit
     * |        |          |This field is used to enable activation (ACTEN(SC_ALTCTL[3] = 1)), deactivation ((DACTEN SC_ALTCTL[2]) = 1) and warm reset (WARSTEN (SC_ALTCTL [4])) sequence interrupt.
     * |        |          |0 = Initial end interrupt Disabled.
     * |        |          |1 = Initial end interrupt Enabled.
     * |[9]     |RXTOIEN   |Receiver Buffer Time-out Interrupt Enable Bit
     * |        |          |This field is used to enable receiver buffer time-out interrupt.
     * |        |          |0 = Receiver buffer time-out interrupt Disabled.
     * |        |          |1 = Receiver buffer time-out interrupt Enabled.
     * |[10]    |ACERRIEN  |Auto Convention Error Interrupt Enable Bit
     * |        |          |This field is used to enable auto-convention error interrupt.
     * |        |          |0 = Auto-convention error interrupt Disabled.
     * |        |          |1 = Auto-convention error interrupt Enabled.
     * @var SC_T::INTSTS
     * Offset: 0x1C  SC Interrupt Status Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RDAIF     |Receive Data Reach Interrupt Status Flag (Read Only)
     * |        |          |This field is used for received data reaching trigger level RXTRGLV (SC_CTL[7:6]) interrupt status flag.
     * |        |          |Note: This field is the status flag of received data reaching RXTRGLV (SC_CTL[7:6])
     * |        |          |If software reads data from SC_DAT and receiver buffer data byte number is less than RXTRGLV (SC_CTL[7:6]), this bit will be cleared automatically.
     * |[1]     |TBEIF     |Transmit Buffer Empty Interrupt Status Flag (Read Only)
     * |        |          |This field is used for transmit buffer empty interrupt status flag.
     * |        |          |Note: This field is the status flag of transmit buffer empty state
     * |        |          |If software wants to clear this bit, software must write data to DAT(SC_DAT[7:0]) buffer and then this bit will be cleared automatically.
     * |[2]     |TERRIF    |Transfer Error Interrupt Status Flag (Read Only)
     * |        |          |This field is used for transfer error interrupt status flag
     * |        |          |The transfer error states is at SC_STATUS register which includes receiver break error BEF(SC_STATUS[6]), frame error FEF(SC_STATUS[5], parity error PEF(SC_STATUS[4] and receiver buffer overflow error RXOV(SC_STATUS[0]), transmit buffer overflow error TXOV(SC_STATUS[8]), receiver retry over limit error RXOVERR(SC_STATUS[22] and transmitter retry over limit error TXOVERR(SC_STATUS[30]).
     * |        |          |Note: This field is the status flag of BEF(SC_STATUS[6]), FEF(SC_STATUS[5]), PEF(SC_STATUS[4]), RXOV(SC_STATUS[0]), TXOV(SC_STATUS[8]), RXOVERR(SC_STATUS[22]) or TXOVERR(SC_STATUS[30])
     * |        |          |So, if software wants to clear this bit, software must write 1 to each field.
     * |[3]     |TMR0IF    |Timer0 Interrupt Status Flag (Read Only)
     * |        |          |This field is used for TMR0 interrupt status flag.
     * |        |          |Note: This bit is read only, but it can be cleared by writing 1 to it.
     * |[4]     |TMR1IF    |Timer1 Interrupt Status Flag (Read Only)
     * |        |          |This field is used for TMR1 interrupt status flag.
     * |        |          |Note: This bit is read only, but it can be cleared by writing 1 to it.
     * |[5]     |TMR2IF    |Timer2 Interrupt Status Flag (Read Only)
     * |        |          |This field is used for TMR2 interrupt status flag.
     * |        |          |Note: This bit is read only, but it can be cleared by writing 1 to it.
     * |[6]     |BGTIF     |Block Guard Time Interrupt Status Flag (Read Only)
     * |        |          |This field is used for block guard time interrupt status flag.
     * |        |          |Note1: This bit is valid when RXBGTEN (SC_ALTCTL[12]) is enabled.
     * |        |          |Note2: This bit is read only, but it can be cleared by writing u201C1u201D to it.
     * |[7]     |CDIF      |Card Detect Interrupt Status Flag (Read Only)
     * |        |          |This field is used for card detect interrupt status flag
     * |        |          |The card detect status is CINSERT (SC_STATUS[12]) and CREMOVE(SC_STATUS[11]).
     * |        |          |Note: This field is the status flag of CINSERT(SC_STATUS[12]) or CREMOVE(SC_STATUS[11])]
     * |        |          |So if software wants to clear these bits, software must write 1 to these field.
     * |[8]     |INITIF    |Initial End Interrupt Status Flag (Read Only)
     * |        |          |This field is used for activation (ACTEN(SC_ALTCTL[3])), deactivation (DACTEN (SC_ALTCTL[2])) and warm reset (WARSTEN (SC_ALTCTL[4])) sequence interrupt status flag.
     * |        |          |Note: This bit is read only, but it can be cleared by writing 1 to it.
     * |[9]     |RXTOIF    |Receiver Buffer Time-out Interrupt Status Flag (Read Only)
     * |        |          |This field is used for receiver buffer time-out interrupt status flag.
     * |        |          |Note: This field is the status flag of receiver buffer time-out state
     * |        |          |If software wants to clear this bit, software must read all receiver buffer remaining data by reading SC_DAT buffer,
     * |[10]    |ACERRIF   |Auto Convention Error Interrupt Status Flag (Read Only)
     * |        |          |This field indicates auto convention sequence error
     * |        |          |If the received TS at ATR state is neither 0x3B nor 0x3F, this bit will be set.
     * |        |          |Note: This bit is read only, but it can be cleared by writing 1 to it.
     * @var SC_T::STATUS
     * Offset: 0x20  SC Transfer Status Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RXOV      |RX Overflow Error Status Flag (Read Only)
     * |        |          |This bit is set when RX buffer overflow.
     * |        |          |If the number of received bytes is greater than Rx Buffer size (4 bytes), this bit will be set.
     * |        |          |Note: This bit is read only, but it can be cleared by writing 1 to it.
     * |[1]     |RXEMPTY   |Receiver Buffer Empty Status Flag(Read Only)
     * |        |          |This bit indicates RX buffer empty or not.
     * |        |          |When the last byte of Rx buffer has been read by CPU, hardware sets this bit high
     * |        |          |It will be cleared when SC receives any new data.
     * |[2]     |RXFULL    |Receiver Buffer Full Status Flag (Read Only)
     * |        |          |This bit indicates RX buffer full or not.
     * |        |          |This bit is set when RX pointer is equal to 4, otherwise it is cleared by hardware.
     * |[4]     |PEF       |Receiver Parity Error Status Flag (Read Only)
     * |        |          |This bit is set to logic 1 whenever the received character does not have a valid parity bit.
     * |        |          |Note1: This bit is read only, but it can be cleared by writing 1 to it.
     * |        |          |Note2: If CPU sets receiver retries function by setting RXRTYEN(SC_CTL[19]) , hardware will not set this flag.
     * |[5]     |FEF       |Receiver Frame Error Status Flag (Read Only)
     * |        |          |This bit is set to logic 1 whenever the received character does not have a valid stop bit (that is, the stop bit following the last data bit or parity bit is detected as logic 0).
     * |        |          |Note1: This bit is read only, but it can be cleared by writing 1 to it.
     * |        |          |Note2: If CPU sets receiver retries function by setting RXRTYEN(SC_CTL[19]) , hardware will not set this flag.
     * |[6]     |BEF       |Receiver Break Error Status Flag (Read Only)
     * |        |          |This bit is set to logic 1 whenever the received data input (RX) held in the spacing state (logic 0) is longer than a full word transmission time (that is, the total time of start bit + data bits + parity + stop bits).
     * |        |          |Note1: This bit is read only, but it can be cleared by writing 1 to it.
     * |        |          |Note2: If CPU sets receiver retries function by setting RXRTYEN(SC_CTL[19]) , hardware will not set this flag.
     * |[8]     |TXOV      |TX Overflow Error Interrupt Status Flag (Read Only)
     * |        |          |If TX buffer is full, an additional write to DAT(SC_DAT[7:0]) will cause this bit be set to 1 by hardware.
     * |        |          |Note: This bit is read only, but it can be cleared by writing 1 to it.
     * |[9]     |TXEMPTY   |Transmit Buffer Empty Status Flag (Read Only)
     * |        |          |This bit indicates TX buffer empty or not.
     * |        |          |When the last byte of TX buffer has been transferred to Transmitter Shift Register, hardware sets this bit high
     * |        |          |It will be cleared when writing data into DAT(SC_DAT[7:0]) (TX buffer not empty).
     * |[10]    |TXFULL    |Transmit Buffer Full Status Flag (Read Only)
     * |        |          |This bit indicates TX buffer full or not.This bit is set when TX pointer is equal to 4, otherwise is cleared by hardware.
     * |[17:16] |RXPOINT   |Receiver Buffer Pointer Status Flag (Read Only)
     * |        |          |This field indicates the RX buffer pointer status flag
     * |        |          |When SC receives one byte from external device, RXPOINT(SC_STATUS[17:16]) increases one
     * |        |          |When one byte of RX buffer is read by CPU, RXPOINT(SC_STATUS[17:16]) decreases one.
     * |[21]    |RXRERR    |Receiver Retry Error (Read Only)
     * |        |          |This bit is set by hardware when RX has any error and retries transfer.
     * |        |          |Note1: This bit is read only, but it can be cleared by writing 1 to it.
     * |        |          |Note2 This bit is a flag and cannot generate any interrupt to CPU.
     * |        |          |Note3: If CPU enables receiver retry function by setting RXRTYEN (SC_CTL[19]) , the PEF(SC_STATUS[4]) flag will be ignored (hardware will not set PEF(SC_STATUS[4])).
     * |[22]    |RXOVERR   |Receiver over Retry Error (Read Only)
     * |        |          |This bit is set by hardware when RX transfer error retry over retry number limit.
     * |        |          |Note1: This bit is read only, but it can be cleared by writing 1 to it.
     * |        |          |Note2: If CPU enables receiver retries function by setting RXRTYEN (SC_CTL[19]), the PEF(SC_STATUS[4]) flag will be ignored (hardware will not set PEF(SC_STATUS[4])).
     * |[23]    |RXACT     |Receiver in Active Status Flag (Read Only)
     * |        |          |This bit is set by hardware when RX transfer is in active.
     * |        |          |This bit is cleared automatically when RX transfer is finished.
     * |[25:24] |TXPOINT   |Transmit Buffer Pointer Status Flag (Read Only)
     * |        |          |This field indicates the TX buffer pointer status flag
     * |        |          |When CPU writes data into SC_DAT, TXPOINT increases one
     * |        |          |When one byte of TX Buffer is transferred to transmitter shift register, TXPOINT decreases one.
     * |[29]    |TXRERR    |Transmitter Retry Error (Read Only)
     * |        |          |This bit is set by hardware when transmitter re-transmits.
     * |        |          |Note1: This bit is read only, but it can be cleared by writing 1 to it.
     * |        |          |Note2 This bit is a flag and cannot generate any interrupt to CPU.
     * |[30]    |TXOVERR   |Transmitter over Retry Error (Read Only)
     * |        |          |This bit is set by hardware when transmitter re-transmits over retry number limitation.
     * |        |          |Note: This bit is read only, but it can be cleared by writing 1 to it.
     * |[31]    |TXACT     |Transmit in Active Status Flag (Read Only)
     * |        |          |0 = This bit is cleared automatically when TX transfer is finished or the last byte transmission has completed.
     * |        |          |1 = This bit is set by hardware when TX transfer is in active and the STOP bit of the last byte has been transmitted.
     * @var SC_T::PINCTL
     * Offset: 0x24  SC Pin Control State Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |PWREN     |SC_PWREN Pin Signal
     * |        |          |Software can set PWREN (SC_PINCTL[0]) and PWRINV (SC_PINCTL[11])to decide SC_PWR pin is in high or low level.
     * |        |          |Write this field to drive SC_PWR pin
     * |        |          |Refer PWRINV (SC_PINCTL[11]) description for programming SC_PWR pin voltage level.
     * |        |          |Read this field to get SC_PWR pin status.
     * |        |          |0 = SC_PWR pin status is low.
     * |        |          |1 = SC_PWR pin status is high.
     * |        |          |Note: When operating at hardware activation, warm reset or deactivation mode, this bit will be changed automatically
     * |        |          |Thus,do not fill in this field when operating in these modes.
     * |[1]     |SCRST     |SC_RST Pin Signal
     * |        |          |This bit is the pin status of SC_RST but user can drive SC_RST pin to high or low by setting this bit.
     * |        |          |Write this field to drive SC_RST pin.
     * |        |          |0 = Drive SC_RST pin to low.
     * |        |          |1 = Drive SC_RST pin to high.
     * |        |          |Read this field to get SC_RST pin status.
     * |        |          |0 = SC_RST pin status is low.
     * |        |          |1 = SC_RST pin status is high.
     * |        |          |Note: When operating at activation, warm reset or deactivation mode, this bit will be changed automatically
     * |        |          |Thus,do not fill in this field when operating in these modes.
     * |[2]     |CREMOVE   |Card Detect Removal Status of SC_CD Pin (Read Only)
     * |        |          |This bit is set whenever card has been removal.
     * |        |          |0 = No effect.
     * |        |          |1 = Card removed.
     * |        |          |Note1: This bit is read only, but it can be cleared by writing u201C1u201D to it.
     * |        |          |Note2: Card detect engine will start after SCEN (SC_CTL[0])set.
     * |[3]     |CINSERT   |Card Detect Insert Status of SC_CD Pin (Read Only)
     * |        |          |This bit is set whenever card has been inserted.
     * |        |          |0 = No effect.
     * |        |          |1 = Card insert.
     * |        |          |Note1: This bit is read only, but it can be cleared by writing u201C1u201D to it.
     * |        |          |Note2: The card detect engine will start after SCEN (SC_CTL[0]) set.
     * |[4]     |CDPINSTS  |Card Detect Status of SC_CD Pin Status (Read Only)
     * |        |          |This bit is the pin status flag of SC_CD
     * |        |          |0 = The SC_CD pin state at low.
     * |        |          |1 = The SC_CD pin state at high.
     * |[6]     |CLKKEEP   |SC Clock Enable Bit
     * |        |          |0 = SC clock generation Disabled.
     * |        |          |1 = SC clock always keeps free running.
     * |        |          |Note: When operating in activation, warm reset or deactivation mode, this bit will be changed automatically
     * |        |          |Thus,do not fill in this field when operating in these modes.
     * |[7]     |ADACEN    |Auto Deactivation When Card Removal
     * |        |          |0 = Auto deactivation Disabled when hardware detected the card removal.
     * |        |          |1 = Auto deactivation Enabled when hardware detected the card removal.
     * |        |          |Note: When the card is removed, hardware will stop any process and then do deactivation sequence (if this bit is set)
     * |        |          |If this process completes, hardware will generate an interrupt INITIF to CPU.
     * |[9]     |SCDOUT    |SC Data Output Pin
     * |        |          |This bit is the pin status of SCDATOUT but user can drive SCDATOUT pin to high or low by setting this bit.
     * |        |          |0 = Drive SCDATOUT pin to low.
     * |        |          |1 = Drive SCDATOUT pin to high.
     * |        |          |Note: When SC is at activation, warm reset or deactivation mode, this bit will be changed automatically
     * |        |          |Thus,do not fill in this field when SC is in these modes.
     * |[10]    |CDLV      |Card Detect Level
     * |        |          |0 = When hardware detects the card detect pin (SC_CD) from high to low, it indicates a card is detected.
     * |        |          |1 = When hardware detects the card detect pin from low to high, it indicates a card is detected.
     * |        |          |Note: Software must select card detect level before activate Smart Card.
     * |[11]    |PWRINV    |SC_POW Pin Inverse
     * |        |          |This bit is used for inverse the SC_POW pin.
     * |        |          |There are four kinds of combination for SC_POW pin setting by PWRINV(SC_PINCTL[11]) and PWREN(SC_PINCTL[0])
     * |        |          |PWRINV (SC_PINCTL[11]) is bit 1 and PWREN(SC_PINCTL[0]) is bit 0 for SC_POW_Pin as high or low voltage selection.
     * |        |          |00 = SC_POW_ Pin is 0.
     * |        |          |01 = SC_POW _Pin is 1.
     * |        |          |10 = SC_POW _Pin is 1.
     * |        |          |11 = SC_POW_ Pin is 0.
     * |        |          |Note:Software must select PWRINV (SC_PINCTL[11]) before Smart Card is enabled by SCEN (SC_CTL[0]).
     * |[16]    |DATSTS    |SC Data Input Pin Status (Read Only)
     * |        |          |This bit is the pin status of SC_DAT
     * |        |          |0 = The SC_DAT pin is low.
     * |        |          |1 = The SC_DAT pin is high.
     * |[30]    |SYNC      |SYNC Flag Indicator(Read Only)
     * |        |          |Due to synchronization, software should check this bit when writing a new value to SC_PINCTL register.
     * |        |          |0 = Synchronizing is completion, user can write new data to SC_PINCTL register.
     * |        |          |1 = Last value is synchronizing.
     * @var SC_T::TMRCTL0
     * Offset: 0x28  SC Internal Timer 0 Control Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[23:0]  |CNT       |Timer 0 Counter Value (ETU Based)
     * |        |          |This field indicates the internal timer operation values.
     * |[27:24] |OPMODE    |Timer 0 Operation Mode Selection
     * |        |          |This field indicates the internal 24-bit timer operation selection.
     * |        |          |Refer toTable 6.15-3 Timer Operation Mode for programming Timer0.
     * |[31]    |SYNC      |SYNC Flag Indicator(Read Only)
     * |        |          |Due to synchronization, software should check this bit when writing a new value to the SC_TMRCTL0 register.
     * |        |          |0 = Synchronizing is completion, user can write new data to SC_TMRCTL0 register.
     * |        |          |1 = Last value is synchronizing.
     * @var SC_T::TMRCTL1
     * Offset: 0x2C  SC Internal Timer 1 Control Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |CNT       |Timer 1 Counter Value (ETU Based)
     * |        |          |This field indicates the internal timer operation values.
     * |[27:24] |OPMODE    |Timer 1 Operation Mode Selection
     * |        |          |This field indicates the internal 8-bit timer operation selection.
     * |        |          |Refer toTable 6.15-3 Timer Operation Mode for programming Timer1.
     * |[31]    |SYNC      |SYNC Flag Indicator(Read Only)
     * |        |          |Due to synchronization, software should check this bit when writing a new value to the SC_TMRCTL1 register.
     * |        |          |0 = Synchronizing is completion, user can write new data to SC_TMRCTL1 register.
     * |        |          |1 = Last value is synchronizing.
     * @var SC_T::TMRCTL2
     * Offset: 0x30  SC Internal Timer 2 Control Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |CNT       |Timer 2 Counter Value (ETU Based)
     * |        |          |This field indicates the internal timer operation values.
     * |[27:24] |OPMODE    |Timer 2 Operation Mode Selection
     * |        |          |This field indicates the internal 8-bit timer operation selection
     * |        |          |Refer to Table 6.15-3 Timer Operation Mode for programming Timer2.
     * |[31]    |SYNC      |SYNC Flag Indicator(Read Only)
     * |        |          |Due to synchronization, software should check this bit when writing a new value to SC_TMRCTL2 register.
     * |        |          |0 = Synchronizing is completion, user can write new data to SC_TMRCTL2 register.
     * |        |          |1 = Last value is synchronizing.
     * @var SC_T::UARTCTL
     * Offset: 0x34  SC UART Mode Control Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |UARTEN    |UART Mode Enable Bit
     * |        |          |0 = Smart Card mode.
     * |        |          |1 = UART mode.
     * |        |          |Note1: When operating in UART mode, user must set CONSEL (SC_CTL[5:4]) = 00 and AUTOCEN(SC_CTL[3]) = 0.
     * |        |          |Note2: When operating in Smart Card mode, user must set UARTEN(SC_UARTCTL [0]) = 00.
     * |        |          |Note3: When UART is enabled, hardware will generate a reset to reset FIFO and internal state machine.
     * |[5:4]   |WLS       |Word Length Selection
     * |        |          |00 = Word length is 8 bits.
     * |        |          |01 = Word length is 7 bits.
     * |        |          |10 = Word length is 6 bits.
     * |        |          |11 = Word length is 5 bits.
     * |        |          |Note: In smart card mode, this WLS must be '00'
     * |[6]     |PBOFF     |Parity Bit Disable Control
     * |        |          |0 = Parity bit is generated or checked between the last data word bit and stop bit of the serial data.
     * |        |          |1 = Parity bit is not generated (transmitting data) or checked (receiving data) during transfer.
     * |        |          |Note: In smart card mode, this field must be '0' (default setting is with parity bit)
     * |[7]     |OPE       |Odd Parity Enable Bit
     * |        |          |0 = Even number of logic 1's are transmitted or check the data word and parity bits in receiving mode.
     * |        |          |1 = Odd number of logic 1's are transmitted or check the data word and parity bits in receiving mode.
     * |        |          |Note: This bit has effect only when PBOFF bit is '0'.
     * @var SC_T::ACTCTL
     * Offset: 0x40  SC Activation Control Register.
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[4:0]   |T1EXT     |Configurable Cycles T1EXT in Hardware Activation
     * |        |          |This field provide the configurable cycles to extend the Activation time T1
     * |        |          |The cycle scaling factor is 2048.
     * |        |          |Extend cycles = (filled value * 2048) cycles.
     * |        |          |Refer to SC Activation Sequence in Figure 6.15-4 SC Activation Sequence.
     * |        |          |For example,
     * |        |          |SCLK = 4Mhz, each cycle = 0.25us,.
     * |        |          |Filled 20 to this field
     * |        |          |Extend time = 20*2048*0.25us = 10.24 ms.
     * |        |          |Note: setting 0 to this field conforms to the protocol ISO/IEC 7816-3
     */
    __IO uint32_t DAT;                   /*!< [0x0000] SC Receive/Transmit Holding Buffer Register.                    */
    __IO uint32_t CTL;                   /*!< [0x0004] SC Control Register.                                            */
    __IO uint32_t ALTCTL;                /*!< [0x0008] SC Alternate Control Register.                                  */
    __IO uint32_t EGT;                   /*!< [0x000c] SC Extra Guard Time Register.                                   */
    __IO uint32_t RXTOUT;                /*!< [0x0010] SC Receive buffer Time-out Register.                            */
    __IO uint32_t ETUCTL;                /*!< [0x0014] SC Element Time Unit Control Register.                          */
    __IO uint32_t INTEN;                 /*!< [0x0018] SC Interrupt Enable Control Register.                           */
    __IO uint32_t INTSTS;                /*!< [0x001c] SC Interrupt Status Register.                                   */
    __IO uint32_t STATUS;                /*!< [0x0020] SC Transfer Status Register.                                    */
    __IO uint32_t PINCTL;                /*!< [0x0024] SC Pin Control State Register.                                  */
    __IO uint32_t TMRCTL0;               /*!< [0x0028] SC Internal Timer 0 Control Register.                           */
    __IO uint32_t TMRCTL1;               /*!< [0x002c] SC Internal Timer 1 Control Register.                           */
    __IO uint32_t TMRCTL2;               /*!< [0x0030] SC Internal Timer 2 Control Register.                           */
    __IO uint32_t UARTCTL;               /*!< [0x0034] SC UART Mode Control Register.                                  */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE0[2];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t ACTCTL;                /*!< [0x0040] SC Activation Control Register.                                 */

} SC_T;

/**
    @addtogroup SC_CONST SC Bit Field Definition
    Constant Definitions for SC Controller
@{ */

#define SC_DAT_DAT_Pos                   (0)                                               /*!< SC_T::DAT: DAT Position                */
#define SC_DAT_DAT_Msk                   (0xfful << SC_DAT_DAT_Pos)                        /*!< SC_T::DAT: DAT Mask                    */

#define SC_CTL_SCEN_Pos                  (0)                                               /*!< SC_T::CTL: SCEN Position               */
#define SC_CTL_SCEN_Msk                  (0x1ul << SC_CTL_SCEN_Pos)                        /*!< SC_T::CTL: SCEN Mask                   */

#define SC_CTL_RXOFF_Pos                 (1)                                               /*!< SC_T::CTL: RXOFF Position              */
#define SC_CTL_RXOFF_Msk                 (0x1ul << SC_CTL_RXOFF_Pos)                       /*!< SC_T::CTL: RXOFF Mask                  */

#define SC_CTL_TXOFF_Pos                 (2)                                               /*!< SC_T::CTL: TXOFF Position              */
#define SC_CTL_TXOFF_Msk                 (0x1ul << SC_CTL_TXOFF_Pos)                       /*!< SC_T::CTL: TXOFF Mask                  */

#define SC_CTL_AUTOCEN_Pos               (3)                                               /*!< SC_T::CTL: AUTOCEN Position            */
#define SC_CTL_AUTOCEN_Msk               (0x1ul << SC_CTL_AUTOCEN_Pos)                     /*!< SC_T::CTL: AUTOCEN Mask                */

#define SC_CTL_CONSEL_Pos                (4)                                               /*!< SC_T::CTL: CONSEL Position             */
#define SC_CTL_CONSEL_Msk                (0x3ul << SC_CTL_CONSEL_Pos)                      /*!< SC_T::CTL: CONSEL Mask                 */

#define SC_CTL_RXTRGLV_Pos               (6)                                               /*!< SC_T::CTL: RXTRGLV Position            */
#define SC_CTL_RXTRGLV_Msk               (0x3ul << SC_CTL_RXTRGLV_Pos)                     /*!< SC_T::CTL: RXTRGLV Mask                */

#define SC_CTL_BGT_Pos                   (8)                                               /*!< SC_T::CTL: BGT Position                */
#define SC_CTL_BGT_Msk                   (0x1ful << SC_CTL_BGT_Pos)                        /*!< SC_T::CTL: BGT Mask                    */

#define SC_CTL_TMRSEL_Pos                (13)                                              /*!< SC_T::CTL: TMRSEL Position             */
#define SC_CTL_TMRSEL_Msk                (0x3ul << SC_CTL_TMRSEL_Pos)                      /*!< SC_T::CTL: TMRSEL Mask                 */

#define SC_CTL_NSB_Pos                   (15)                                              /*!< SC_T::CTL: NSB Position                */
#define SC_CTL_NSB_Msk                   (0x1ul << SC_CTL_NSB_Pos)                         /*!< SC_T::CTL: NSB Mask                    */

#define SC_CTL_RXRTY_Pos                 (16)                                              /*!< SC_T::CTL: RXRTY Position              */
#define SC_CTL_RXRTY_Msk                 (0x7ul << SC_CTL_RXRTY_Pos)                       /*!< SC_T::CTL: RXRTY Mask                  */

#define SC_CTL_RXRTYEN_Pos               (19)                                              /*!< SC_T::CTL: RXRTYEN Position            */
#define SC_CTL_RXRTYEN_Msk               (0x1ul << SC_CTL_RXRTYEN_Pos)                     /*!< SC_T::CTL: RXRTYEN Mask                */

#define SC_CTL_TXRTY_Pos                 (20)                                              /*!< SC_T::CTL: TXRTY Position              */
#define SC_CTL_TXRTY_Msk                 (0x7ul << SC_CTL_TXRTY_Pos)                       /*!< SC_T::CTL: TXRTY Mask                  */

#define SC_CTL_TXRTYEN_Pos               (23)                                              /*!< SC_T::CTL: TXRTYEN Position            */
#define SC_CTL_TXRTYEN_Msk               (0x1ul << SC_CTL_TXRTYEN_Pos)                     /*!< SC_T::CTL: TXRTYEN Mask                */

#define SC_CTL_CDDBSEL_Pos               (24)                                              /*!< SC_T::CTL: CDDBSEL Position            */
#define SC_CTL_CDDBSEL_Msk               (0x3ul << SC_CTL_CDDBSEL_Pos)                     /*!< SC_T::CTL: CDDBSEL Mask                */

#define SC_CTL_SYNC_Pos                  (30)                                              /*!< SC_T::CTL: SYNC Position               */
#define SC_CTL_SYNC_Msk                  (0x1ul << SC_CTL_SYNC_Pos)                        /*!< SC_T::CTL: SYNC Mask                   */

#define SC_ALTCTL_TXRST_Pos              (0)                                               /*!< SC_T::ALTCTL: TXRST Position           */
#define SC_ALTCTL_TXRST_Msk              (0x1ul << SC_ALTCTL_TXRST_Pos)                    /*!< SC_T::ALTCTL: TXRST Mask               */

#define SC_ALTCTL_RXRST_Pos              (1)                                               /*!< SC_T::ALTCTL: RXRST Position           */
#define SC_ALTCTL_RXRST_Msk              (0x1ul << SC_ALTCTL_RXRST_Pos)                    /*!< SC_T::ALTCTL: RXRST Mask               */

#define SC_ALTCTL_DACTEN_Pos             (2)                                               /*!< SC_T::ALTCTL: DACTEN Position          */
#define SC_ALTCTL_DACTEN_Msk             (0x1ul << SC_ALTCTL_DACTEN_Pos)                   /*!< SC_T::ALTCTL: DACTEN Mask              */

#define SC_ALTCTL_ACTEN_Pos              (3)                                               /*!< SC_T::ALTCTL: ACTEN Position           */
#define SC_ALTCTL_ACTEN_Msk              (0x1ul << SC_ALTCTL_ACTEN_Pos)                    /*!< SC_T::ALTCTL: ACTEN Mask               */

#define SC_ALTCTL_WARSTEN_Pos            (4)                                               /*!< SC_T::ALTCTL: WARSTEN Position         */
#define SC_ALTCTL_WARSTEN_Msk            (0x1ul << SC_ALTCTL_WARSTEN_Pos)                  /*!< SC_T::ALTCTL: WARSTEN Mask             */

#define SC_ALTCTL_CNTEN0_Pos             (5)                                               /*!< SC_T::ALTCTL: CNTEN0 Position          */
#define SC_ALTCTL_CNTEN0_Msk             (0x1ul << SC_ALTCTL_CNTEN0_Pos)                   /*!< SC_T::ALTCTL: CNTEN0 Mask              */

#define SC_ALTCTL_CNTEN1_Pos             (6)                                               /*!< SC_T::ALTCTL: CNTEN1 Position          */
#define SC_ALTCTL_CNTEN1_Msk             (0x1ul << SC_ALTCTL_CNTEN1_Pos)                   /*!< SC_T::ALTCTL: CNTEN1 Mask              */

#define SC_ALTCTL_CNTEN2_Pos             (7)                                               /*!< SC_T::ALTCTL: CNTEN2 Position          */
#define SC_ALTCTL_CNTEN2_Msk             (0x1ul << SC_ALTCTL_CNTEN2_Pos)                   /*!< SC_T::ALTCTL: CNTEN2 Mask              */

#define SC_ALTCTL_INITSEL_Pos            (8)                                               /*!< SC_T::ALTCTL: INITSEL Position         */
#define SC_ALTCTL_INITSEL_Msk            (0x3ul << SC_ALTCTL_INITSEL_Pos)                  /*!< SC_T::ALTCTL: INITSEL Mask             */

#define SC_ALTCTL_RXBGTEN_Pos            (12)                                              /*!< SC_T::ALTCTL: RXBGTEN Position         */
#define SC_ALTCTL_RXBGTEN_Msk            (0x1ul << SC_ALTCTL_RXBGTEN_Pos)                  /*!< SC_T::ALTCTL: RXBGTEN Mask             */

#define SC_ALTCTL_ACTSTS0_Pos            (13)                                              /*!< SC_T::ALTCTL: ACTSTS0 Position         */
#define SC_ALTCTL_ACTSTS0_Msk            (0x1ul << SC_ALTCTL_ACTSTS0_Pos)                  /*!< SC_T::ALTCTL: ACTSTS0 Mask             */

#define SC_ALTCTL_ACTSTS1_Pos            (14)                                              /*!< SC_T::ALTCTL: ACTSTS1 Position         */
#define SC_ALTCTL_ACTSTS1_Msk            (0x1ul << SC_ALTCTL_ACTSTS1_Pos)                  /*!< SC_T::ALTCTL: ACTSTS1 Mask             */

#define SC_ALTCTL_ACTSTS2_Pos            (15)                                              /*!< SC_T::ALTCTL: ACTSTS2 Position         */
#define SC_ALTCTL_ACTSTS2_Msk            (0x1ul << SC_ALTCTL_ACTSTS2_Pos)                  /*!< SC_T::ALTCTL: ACTSTS2 Mask             */

#define SC_ALTCTL_OUTSEL_Pos             (16)                                              /*!< SC_T::ALTCTL: OUTSEL Position          */
#define SC_ALTCTL_OUTSEL_Msk             (0x1ul << SC_ALTCTL_OUTSEL_Pos)                   /*!< SC_T::ALTCTL: OUTSEL Mask              */

#define SC_EGT_EGT_Pos                   (0)                                               /*!< SC_T::EGT: EGT Position                */
#define SC_EGT_EGT_Msk                   (0xfful << SC_EGT_EGT_Pos)                        /*!< SC_T::EGT: EGT Mask                    */

#define SC_RXTOUT_RFTM_Pos               (0)                                               /*!< SC_T::RXTOUT: RFTM Position            */
#define SC_RXTOUT_RFTM_Msk               (0x1fful << SC_RXTOUT_RFTM_Pos)                   /*!< SC_T::RXTOUT: RFTM Mask                */

#define SC_ETUCTL_ETURDIV_Pos            (0)                                               /*!< SC_T::ETUCTL: ETURDIV Position         */
#define SC_ETUCTL_ETURDIV_Msk            (0xffful << SC_ETUCTL_ETURDIV_Pos)                /*!< SC_T::ETUCTL: ETURDIV Mask             */

#define SC_INTEN_RDAIEN_Pos              (0)                                               /*!< SC_T::INTEN: RDAIEN Position           */
#define SC_INTEN_RDAIEN_Msk              (0x1ul << SC_INTEN_RDAIEN_Pos)                    /*!< SC_T::INTEN: RDAIEN Mask               */

#define SC_INTEN_TBEIEN_Pos              (1)                                               /*!< SC_T::INTEN: TBEIEN Position           */
#define SC_INTEN_TBEIEN_Msk              (0x1ul << SC_INTEN_TBEIEN_Pos)                    /*!< SC_T::INTEN: TBEIEN Mask               */

#define SC_INTEN_TERRIEN_Pos             (2)                                               /*!< SC_T::INTEN: TERRIEN Position          */
#define SC_INTEN_TERRIEN_Msk             (0x1ul << SC_INTEN_TERRIEN_Pos)                   /*!< SC_T::INTEN: TERRIEN Mask              */

#define SC_INTEN_TMR0IEN_Pos             (3)                                               /*!< SC_T::INTEN: TMR0IEN Position          */
#define SC_INTEN_TMR0IEN_Msk             (0x1ul << SC_INTEN_TMR0IEN_Pos)                   /*!< SC_T::INTEN: TMR0IEN Mask              */

#define SC_INTEN_TMR1IEN_Pos             (4)                                               /*!< SC_T::INTEN: TMR1IEN Position          */
#define SC_INTEN_TMR1IEN_Msk             (0x1ul << SC_INTEN_TMR1IEN_Pos)                   /*!< SC_T::INTEN: TMR1IEN Mask              */

#define SC_INTEN_TMR2IEN_Pos             (5)                                               /*!< SC_T::INTEN: TMR2IEN Position          */
#define SC_INTEN_TMR2IEN_Msk             (0x1ul << SC_INTEN_TMR2IEN_Pos)                   /*!< SC_T::INTEN: TMR2IEN Mask              */

#define SC_INTEN_BGTIEN_Pos              (6)                                               /*!< SC_T::INTEN: BGTIEN Position           */
#define SC_INTEN_BGTIEN_Msk              (0x1ul << SC_INTEN_BGTIEN_Pos)                    /*!< SC_T::INTEN: BGTIEN Mask               */

#define SC_INTEN_CDIEN_Pos               (7)                                               /*!< SC_T::INTEN: CDIEN Position            */
#define SC_INTEN_CDIEN_Msk               (0x1ul << SC_INTEN_CDIEN_Pos)                     /*!< SC_T::INTEN: CDIEN Mask                */

#define SC_INTEN_INITIEN_Pos             (8)                                               /*!< SC_T::INTEN: INITIEN Position          */
#define SC_INTEN_INITIEN_Msk             (0x1ul << SC_INTEN_INITIEN_Pos)                   /*!< SC_T::INTEN: INITIEN Mask              */

#define SC_INTEN_RXTOIEN_Pos             (9)                                               /*!< SC_T::INTEN: RXTOIEN Position          */
#define SC_INTEN_RXTOIEN_Msk             (0x1ul << SC_INTEN_RXTOIEN_Pos)                   /*!< SC_T::INTEN: RXTOIEN Mask              */

#define SC_INTEN_ACERRIEN_Pos            (10)                                              /*!< SC_T::INTEN: ACERRIEN Position         */
#define SC_INTEN_ACERRIEN_Msk            (0x1ul << SC_INTEN_ACERRIEN_Pos)                  /*!< SC_T::INTEN: ACERRIEN Mask             */

#define SC_INTSTS_RDAIF_Pos              (0)                                               /*!< SC_T::INTSTS: RDAIF Position           */
#define SC_INTSTS_RDAIF_Msk              (0x1ul << SC_INTSTS_RDAIF_Pos)                    /*!< SC_T::INTSTS: RDAIF Mask               */

#define SC_INTSTS_TBEIF_Pos              (1)                                               /*!< SC_T::INTSTS: TBEIF Position           */
#define SC_INTSTS_TBEIF_Msk              (0x1ul << SC_INTSTS_TBEIF_Pos)                    /*!< SC_T::INTSTS: TBEIF Mask               */

#define SC_INTSTS_TERRIF_Pos             (2)                                               /*!< SC_T::INTSTS: TERRIF Position          */
#define SC_INTSTS_TERRIF_Msk             (0x1ul << SC_INTSTS_TERRIF_Pos)                   /*!< SC_T::INTSTS: TERRIF Mask              */

#define SC_INTSTS_TMR0IF_Pos             (3)                                               /*!< SC_T::INTSTS: TMR0IF Position          */
#define SC_INTSTS_TMR0IF_Msk             (0x1ul << SC_INTSTS_TMR0IF_Pos)                   /*!< SC_T::INTSTS: TMR0IF Mask              */

#define SC_INTSTS_TMR1IF_Pos             (4)                                               /*!< SC_T::INTSTS: TMR1IF Position          */
#define SC_INTSTS_TMR1IF_Msk             (0x1ul << SC_INTSTS_TMR1IF_Pos)                   /*!< SC_T::INTSTS: TMR1IF Mask              */

#define SC_INTSTS_TMR2IF_Pos             (5)                                               /*!< SC_T::INTSTS: TMR2IF Position          */
#define SC_INTSTS_TMR2IF_Msk             (0x1ul << SC_INTSTS_TMR2IF_Pos)                   /*!< SC_T::INTSTS: TMR2IF Mask              */

#define SC_INTSTS_BGTIF_Pos              (6)                                               /*!< SC_T::INTSTS: BGTIF Position           */
#define SC_INTSTS_BGTIF_Msk              (0x1ul << SC_INTSTS_BGTIF_Pos)                    /*!< SC_T::INTSTS: BGTIF Mask               */

#define SC_INTSTS_CDIF_Pos               (7)                                               /*!< SC_T::INTSTS: CDIF Position            */
#define SC_INTSTS_CDIF_Msk               (0x1ul << SC_INTSTS_CDIF_Pos)                     /*!< SC_T::INTSTS: CDIF Mask                */

#define SC_INTSTS_INITIF_Pos             (8)                                               /*!< SC_T::INTSTS: INITIF Position          */
#define SC_INTSTS_INITIF_Msk             (0x1ul << SC_INTSTS_INITIF_Pos)                   /*!< SC_T::INTSTS: INITIF Mask              */

#define SC_INTSTS_RXTOIF_Pos             (9)                                               /*!< SC_T::INTSTS: RXTOIF Position          */
#define SC_INTSTS_RXTOIF_Msk             (0x1ul << SC_INTSTS_RXTOIF_Pos)                   /*!< SC_T::INTSTS: RXTOIF Mask              */

#define SC_INTSTS_ACERRIF_Pos            (10)                                              /*!< SC_T::INTSTS: ACERRIF Position         */
#define SC_INTSTS_ACERRIF_Msk            (0x1ul << SC_INTSTS_ACERRIF_Pos)                  /*!< SC_T::INTSTS: ACERRIF Mask             */

#define SC_STATUS_RXOV_Pos               (0)                                               /*!< SC_T::STATUS: RXOV Position            */
#define SC_STATUS_RXOV_Msk               (0x1ul << SC_STATUS_RXOV_Pos)                     /*!< SC_T::STATUS: RXOV Mask                */

#define SC_STATUS_RXEMPTY_Pos            (1)                                               /*!< SC_T::STATUS: RXEMPTY Position         */
#define SC_STATUS_RXEMPTY_Msk            (0x1ul << SC_STATUS_RXEMPTY_Pos)                  /*!< SC_T::STATUS: RXEMPTY Mask             */

#define SC_STATUS_RXFULL_Pos             (2)                                               /*!< SC_T::STATUS: RXFULL Position          */
#define SC_STATUS_RXFULL_Msk             (0x1ul << SC_STATUS_RXFULL_Pos)                   /*!< SC_T::STATUS: RXFULL Mask              */

#define SC_STATUS_PEF_Pos                (4)                                               /*!< SC_T::STATUS: PEF Position             */
#define SC_STATUS_PEF_Msk                (0x1ul << SC_STATUS_PEF_Pos)                      /*!< SC_T::STATUS: PEF Mask                 */

#define SC_STATUS_FEF_Pos                (5)                                               /*!< SC_T::STATUS: FEF Position             */
#define SC_STATUS_FEF_Msk                (0x1ul << SC_STATUS_FEF_Pos)                      /*!< SC_T::STATUS: FEF Mask                 */

#define SC_STATUS_BEF_Pos                (6)                                               /*!< SC_T::STATUS: BEF Position             */
#define SC_STATUS_BEF_Msk                (0x1ul << SC_STATUS_BEF_Pos)                      /*!< SC_T::STATUS: BEF Mask                 */

#define SC_STATUS_TXOV_Pos               (8)                                               /*!< SC_T::STATUS: TXOV Position            */
#define SC_STATUS_TXOV_Msk               (0x1ul << SC_STATUS_TXOV_Pos)                     /*!< SC_T::STATUS: TXOV Mask                */

#define SC_STATUS_TXEMPTY_Pos            (9)                                               /*!< SC_T::STATUS: TXEMPTY Position         */
#define SC_STATUS_TXEMPTY_Msk            (0x1ul << SC_STATUS_TXEMPTY_Pos)                  /*!< SC_T::STATUS: TXEMPTY Mask             */

#define SC_STATUS_TXFULL_Pos             (10)                                              /*!< SC_T::STATUS: TXFULL Position          */
#define SC_STATUS_TXFULL_Msk             (0x1ul << SC_STATUS_TXFULL_Pos)                   /*!< SC_T::STATUS: TXFULL Mask              */

#define SC_STATUS_RXPOINT_Pos            (16)                                              /*!< SC_T::STATUS: RXPOINT Position         */
#define SC_STATUS_RXPOINT_Msk            (0x3ul << SC_STATUS_RXPOINT_Pos)                  /*!< SC_T::STATUS: RXPOINT Mask             */

#define SC_STATUS_RXRERR_Pos             (21)                                              /*!< SC_T::STATUS: RXRERR Position          */
#define SC_STATUS_RXRERR_Msk             (0x1ul << SC_STATUS_RXRERR_Pos)                   /*!< SC_T::STATUS: RXRERR Mask              */

#define SC_STATUS_RXOVERR_Pos            (22)                                              /*!< SC_T::STATUS: RXOVERR Position         */
#define SC_STATUS_RXOVERR_Msk            (0x1ul << SC_STATUS_RXOVERR_Pos)                  /*!< SC_T::STATUS: RXOVERR Mask             */

#define SC_STATUS_RXACT_Pos              (23)                                              /*!< SC_T::STATUS: RXACT Position           */
#define SC_STATUS_RXACT_Msk              (0x1ul << SC_STATUS_RXACT_Pos)                    /*!< SC_T::STATUS: RXACT Mask               */

#define SC_STATUS_TXPOINT_Pos            (24)                                              /*!< SC_T::STATUS: TXPOINT Position         */
#define SC_STATUS_TXPOINT_Msk            (0x3ul << SC_STATUS_TXPOINT_Pos)                  /*!< SC_T::STATUS: TXPOINT Mask             */

#define SC_STATUS_TXRERR_Pos             (29)                                              /*!< SC_T::STATUS: TXRERR Position          */
#define SC_STATUS_TXRERR_Msk             (0x1ul << SC_STATUS_TXRERR_Pos)                   /*!< SC_T::STATUS: TXRERR Mask              */

#define SC_STATUS_TXOVERR_Pos            (30)                                              /*!< SC_T::STATUS: TXOVERR Position         */
#define SC_STATUS_TXOVERR_Msk            (0x1ul << SC_STATUS_TXOVERR_Pos)                  /*!< SC_T::STATUS: TXOVERR Mask             */

#define SC_STATUS_TXACT_Pos              (31)                                              /*!< SC_T::STATUS: TXACT Position           */
#define SC_STATUS_TXACT_Msk              (0x1ul << SC_STATUS_TXACT_Pos)                    /*!< SC_T::STATUS: TXACT Mask               */

#define SC_PINCTL_PWREN_Pos              (0)                                               /*!< SC_T::PINCTL: PWREN Position           */
#define SC_PINCTL_PWREN_Msk              (0x1ul << SC_PINCTL_PWREN_Pos)                    /*!< SC_T::PINCTL: PWREN Mask               */

#define SC_PINCTL_SCRST_Pos              (1)                                               /*!< SC_T::PINCTL: SCRST Position           */
#define SC_PINCTL_SCRST_Msk              (0x1ul << SC_PINCTL_SCRST_Pos)                    /*!< SC_T::PINCTL: SCRST Mask               */

#define SC_PINCTL_CREMOVE_Pos            (2)                                               /*!< SC_T::PINCTL: CREMOVE Position         */
#define SC_PINCTL_CREMOVE_Msk            (0x1ul << SC_PINCTL_CREMOVE_Pos)                  /*!< SC_T::PINCTL: CREMOVE Mask             */

#define SC_PINCTL_CINSERT_Pos            (3)                                               /*!< SC_T::PINCTL: CINSERT Position         */
#define SC_PINCTL_CINSERT_Msk            (0x1ul << SC_PINCTL_CINSERT_Pos)                  /*!< SC_T::PINCTL: CINSERT Mask             */

#define SC_PINCTL_CDPINSTS_Pos           (4)                                               /*!< SC_T::PINCTL: CDPINSTS Position        */
#define SC_PINCTL_CDPINSTS_Msk           (0x1ul << SC_PINCTL_CDPINSTS_Pos)                 /*!< SC_T::PINCTL: CDPINSTS Mask            */

#define SC_PINCTL_CLKKEEP_Pos            (6)                                               /*!< SC_T::PINCTL: CLKKEEP Position         */
#define SC_PINCTL_CLKKEEP_Msk            (0x1ul << SC_PINCTL_CLKKEEP_Pos)                  /*!< SC_T::PINCTL: CLKKEEP Mask             */

#define SC_PINCTL_ADACEN_Pos             (7)                                               /*!< SC_T::PINCTL: ADACEN Position          */
#define SC_PINCTL_ADACEN_Msk             (0x1ul << SC_PINCTL_ADACEN_Pos)                   /*!< SC_T::PINCTL: ADACEN Mask              */

#define SC_PINCTL_SCDOUT_Pos             (9)                                               /*!< SC_T::PINCTL: SCDOUT Position          */
#define SC_PINCTL_SCDOUT_Msk             (0x1ul << SC_PINCTL_SCDOUT_Pos)                   /*!< SC_T::PINCTL: SCDOUT Mask              */

#define SC_PINCTL_CDLV_Pos               (10)                                              /*!< SC_T::PINCTL: CDLV Position            */
#define SC_PINCTL_CDLV_Msk               (0x1ul << SC_PINCTL_CDLV_Pos)                     /*!< SC_T::PINCTL: CDLV Mask                */

#define SC_PINCTL_PWRINV_Pos             (11)                                              /*!< SC_T::PINCTL: PWRINV Position          */
#define SC_PINCTL_PWRINV_Msk             (0x1ul << SC_PINCTL_PWRINV_Pos)                   /*!< SC_T::PINCTL: PWRINV Mask              */

#define SC_PINCTL_DATSTS_Pos             (16)                                              /*!< SC_T::PINCTL: DATSTS Position          */
#define SC_PINCTL_DATSTS_Msk             (0x1ul << SC_PINCTL_DATSTS_Pos)                   /*!< SC_T::PINCTL: DATSTS Mask              */

#define SC_PINCTL_SYNC_Pos               (30)                                              /*!< SC_T::PINCTL: SYNC Position            */
#define SC_PINCTL_SYNC_Msk               (0x1ul << SC_PINCTL_SYNC_Pos)                     /*!< SC_T::PINCTL: SYNC Mask                */

#define SC_TMRCTL0_CNT_Pos               (0)                                               /*!< SC_T::TMRCTL0: CNT Position            */
#define SC_TMRCTL0_CNT_Msk               (0xfffffful << SC_TMRCTL0_CNT_Pos)                /*!< SC_T::TMRCTL0: CNT Mask                */

#define SC_TMRCTL0_OPMODE_Pos            (24)                                              /*!< SC_T::TMRCTL0: OPMODE Position         */
#define SC_TMRCTL0_OPMODE_Msk            (0xful << SC_TMRCTL0_OPMODE_Pos)                  /*!< SC_T::TMRCTL0: OPMODE Mask             */

#define SC_TMRCTL0_SYNC_Pos              (31)                                              /*!< SC_T::TMRCTL0: SYNC Position           */
#define SC_TMRCTL0_SYNC_Msk              (0x1ul << SC_TMRCTL0_SYNC_Pos)                    /*!< SC_T::TMRCTL0: SYNC Mask               */

#define SC_TMRCTL1_CNT_Pos               (0)                                               /*!< SC_T::TMRCTL1: CNT Position            */
#define SC_TMRCTL1_CNT_Msk               (0xfful << SC_TMRCTL1_CNT_Pos)                    /*!< SC_T::TMRCTL1: CNT Mask                */

#define SC_TMRCTL1_OPMODE_Pos            (24)                                              /*!< SC_T::TMRCTL1: OPMODE Position         */
#define SC_TMRCTL1_OPMODE_Msk            (0xful << SC_TMRCTL1_OPMODE_Pos)                  /*!< SC_T::TMRCTL1: OPMODE Mask             */

#define SC_TMRCTL1_SYNC_Pos              (31)                                              /*!< SC_T::TMRCTL1: SYNC Position           */
#define SC_TMRCTL1_SYNC_Msk              (0x1ul << SC_TMRCTL1_SYNC_Pos)                    /*!< SC_T::TMRCTL1: SYNC Mask               */

#define SC_TMRCTL2_CNT_Pos               (0)                                               /*!< SC_T::TMRCTL2: CNT Position            */
#define SC_TMRCTL2_CNT_Msk               (0xfful << SC_TMRCTL2_CNT_Pos)                    /*!< SC_T::TMRCTL2: CNT Mask                */

#define SC_TMRCTL2_OPMODE_Pos            (24)                                              /*!< SC_T::TMRCTL2: OPMODE Position         */
#define SC_TMRCTL2_OPMODE_Msk            (0xful << SC_TMRCTL2_OPMODE_Pos)                  /*!< SC_T::TMRCTL2: OPMODE Mask             */

#define SC_TMRCTL2_SYNC_Pos              (31)                                              /*!< SC_T::TMRCTL2: SYNC Position           */
#define SC_TMRCTL2_SYNC_Msk              (0x1ul << SC_TMRCTL2_SYNC_Pos)                    /*!< SC_T::TMRCTL2: SYNC Mask               */

#define SC_UARTCTL_UARTEN_Pos            (0)                                               /*!< SC_T::UARTCTL: UARTEN Position         */
#define SC_UARTCTL_UARTEN_Msk            (0x1ul << SC_UARTCTL_UARTEN_Pos)                  /*!< SC_T::UARTCTL: UARTEN Mask             */

#define SC_UARTCTL_WLS_Pos               (4)                                               /*!< SC_T::UARTCTL: WLS Position            */
#define SC_UARTCTL_WLS_Msk               (0x3ul << SC_UARTCTL_WLS_Pos)                     /*!< SC_T::UARTCTL: WLS Mask                */

#define SC_UARTCTL_PBOFF_Pos             (6)                                               /*!< SC_T::UARTCTL: PBOFF Position          */
#define SC_UARTCTL_PBOFF_Msk             (0x1ul << SC_UARTCTL_PBOFF_Pos)                   /*!< SC_T::UARTCTL: PBOFF Mask              */

#define SC_UARTCTL_OPE_Pos               (7)                                               /*!< SC_T::UARTCTL: OPE Position            */
#define SC_UARTCTL_OPE_Msk               (0x1ul << SC_UARTCTL_OPE_Pos)                     /*!< SC_T::UARTCTL: OPE Mask                */

#define SC_ACTCTL_T1EXT_Pos              (0)                                               /*!< SC_T::ACTCTL: T1EXT Position           */
#define SC_ACTCTL_T1EXT_Msk              (0x1ful << SC_ACTCTL_T1EXT_Pos)                   /*!< SC_T::ACTCTL: T1EXT Mask               */

/**@}*/ /* SC_CONST */
/**@}*/ /* end of SC register group */


/*---------------------- Inter-IC Bus Controller -------------------------*/
/**
    @addtogroup I2C Inter-IC Bus Controller(I2C)
    Memory Mapped Structure for I2C Controller
@{ */

typedef struct
{


    /**
     * @var I2C_T::CTL
     * Offset: 0x00  I2C Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |I2CEN     |I2C Function Enable Bit
     * |        |          |0 = I2C function Disabled.
     * |        |          |1 = I2C function Enabled.
     * |[1]     |AA        |Assert Acknowledge Control Bit
     * |        |          |When AA =1 prior to address or data is received, an acknowledged (low level to SDA) will be returned during the acknowledge clock pulse on the SCL line when
     * |        |          |1.)A slave is acknowledging the address sent from master,
     * |        |          |2.) The receiver devices are acknowledging the data sent by transmitter
     * |        |          |When AA=0 prior to address or data received, a Not acknowledged (high level to SDA) will be returned during the acknowledge clock pulse on the SCL line.
     * |[2]     |STO       |I2C STOP Control Bit
     * |        |          |In Master mode, setting STO to transmit a STOP condition to bus then I2C controller will check the bus condition if a STOP condition is detected
     * |        |          |This bit will be cleared by hardware automatically.
     * |[3]     |STA       |I2C START Command
     * |        |          |Setting STA to logic 1 to enter Master mode, the I2C hardware sends a START or repeat START condition to bus when the bus is free.
     * |[4]     |SI        |I2C Status
     * |        |          |When a new state is present in the I2C_STATUS register, if the INTEN bit is set, the I2C interrupt is requested
     * |        |          |It must write one by software to this bit after the INTSTS (I2C_INTSTS[0]) is set to 1 and the I2C protocol function will go ahead until the STOP is active or the I2CEN is disabled.
     * |        |          |0 = I2C's Status disabled and the I2C protocol function will go ahead.
     * |        |          |1 = I2C's Status active.
     * |        |          |Note: If software wants to skip clearing INTSTS (I2C_INTSTS[0]), it also can write 1 to SI bit and must set INTEN bit
     * |        |          |That INTSTS (I2C_INTSTS[0]) will be cleared when SI is cleared.
     * |[7]     |INTEN     |Interrupt Enable Bit
     * |        |          |0 = I2C interrupt Disabled.
     * |        |          |1 = I2C interrupt Enabled.
     * @var I2C_T::INTSTS
     * Offset: 0x04  I2C Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |INTSTS    |I2C STATUS's Interrupt Status
     * |        |          |When a new I2C state is present in the I2C_STATUS register, the INTSTS flag is set by hardware
     * |        |          |If bit INTEN (I2C_CTL [7]) is set, the I2C interrupt is requested.This bit must be cleared by software writing 1 .
     * |        |          |Note: If software wants to skip clearing INTSTS, it can also write 1 to SI (I2C_CTL [4]) bit and must set INTEN (I2C_CTL [7]) bit
     * |        |          |INISTS will be cleared when SI is cleared.
     * |[1]     |TOIF      |Time-out Status
     * |        |          |0 = No Time-out flag.
     * |        |          |1 = Time-out flag active and it is set by hardware. It can interrupt CPU when INTEN bit is set.
     * |        |          |Note: This bit can be cleared by writing 1 to it.
     * |[7]     |WKAKDONE  |Wake-up Address Frame Acknowledge Bit Done
     * |        |          |0 = The ACK bit cycle of address match frame is not done.
     * |        |          |1 = The ACK bit cycle of address match frame is done in power-down.
     * |        |          |Note: This bit can be cleared by writing 1 to it.
     * @var I2C_T::STATUS
     * Offset: 0x08  I2C Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |STATUS    |I2C Status Bits (Read Only)
     * |        |          |The three least significant bits are always 0
     * |        |          |The five most significant bits contain the status code
     * |        |          |There are 28 possible status codes
     * |        |          |When the content of I2C_STATUS is F8H, no serial interrupt is requested
     * |        |          |Other I2C_STATUS values correspond to defined I2C states
     * |        |          |When each of these states is entered, a status interrupt is requested (INTSTS = 1)
     * |        |          |A valid status code is present in I2C_STATUS one cycle after INTSTS is set by hardware and is still present one cycle after INTSTS has been reset by software
     * |        |          |In addition, states 00H stands for a Bus Error
     * |        |          |A Bus Error occurs when a START or STOP condition is present at an illegal position in the formation frame
     * |        |          |Example of illegal position are during the serial transfer of an address byte, a data byte or an acknowledge bit.
     * @var I2C_T::CLKDIV
     * Offset: 0x0C  I2C Clock Divided Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |DIVIDER   |I2C Clock Divided Bits
     * |        |          |Indicates the I2C clock rate bits: Data Baud Rate of I2C = PCLK /( 4 x ( I2C_CLKDIV + 1)).
     * |        |          |Note: The minimum value of I2C_CLKDIV is 4.
     * @var I2C_T::TOCTL
     * Offset: 0x10  I2C Time-out Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TOCEN     |Time-out Counter Enable Bit
     * |        |          |When this bit is set to enabled and clock be stretched, the 14 bits time-out counter will start counting.
     * |        |          |0 = Time-out counter Disabled.
     * |        |          |1 = Time-out counter Enabled.
     * |[1]     |TOCDIV4   |Time-out Counter Input Clock Divider by 4
     * |        |          |When enabled, the time-out period is extended 4 times.
     * |        |          |0 = Time-out counter input clock divider by 4 Disabled.
     * |        |          |1 = Time-out counter input clock divider by 4 Enabled.
     * @var I2C_T::DAT
     * Offset: 0x14  I2C Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |DAT       |I2C Data
     * |        |          |Bit [7:0] is located with the 8-bit transferred/received data of I2C serial port.
     * @var I2C_T::ADDR0
     * Offset: 0x18  I2C Slave Address Register0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |GC        |General Call Function Control
     * |        |          |0 = General Call Function Disabled.
     * |        |          |1 = General Call Function Enabled.
     * |        |          |Note: Refer to Address Register section for more detailed information.
     * |[7:1]   |ADDR      |I2C Salve Address Bits
     * |        |          |The content of this register is irrelevant when the device is in Master mode
     * |        |          |In the Slave mode, the seven most significant bits must be loaded with the device's own address
     * |        |          |The device will react if either of the address is matched.
     * @var I2C_T::ADDR1
     * Offset: 0x1C  I2C Slave Address Register1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |GC        |General Call Function Control
     * |        |          |0 = General Call Function Disabled.
     * |        |          |1 = General Call Function Enabled.
     * |        |          |Note: Refer to Address Register section for more detailed information.
     * |[7:1]   |ADDR      |I2C Salve Address Bits
     * |        |          |The content of this register is irrelevant when the device is in Master mode
     * |        |          |In the Slave mode, the seven most significant bits must be loaded with the device's own address
     * |        |          |The device will react if either of the address is matched.
     * @var I2C_T::ADDRMSK0
     * Offset: 0x28  I2C Slave Address Mask Register0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:1]   |ADDRMSK   |I2C Slave Address Mask Bits
     * |        |          |0 = Mask disable (the received corresponding register bit should be exact the same as address register).
     * |        |          |1 = Mask enable (the received corresponding address bit is don't care).
     * |        |          |I2C bus controllers support multiple address recognition with two address mask register
     * |        |          |When the bit in the address mask register is set to one, it means the received corresponding address bit is don't-care
     * |        |          |If the bit is set to zero, that means the received corresponding register bit should be exact the same as address register.
     * @var I2C_T::ADDRMSK1
     * Offset: 0x2C  I2C Slave Address Mask Register1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:1]   |ADDRMSK   |I2C Slave Address Mask Bits
     * |        |          |0 = Mask disable (the received corresponding register bit should be exact the same as address register).
     * |        |          |1 = Mask enable (the received corresponding address bit is don't care).
     * |        |          |I2C bus controllers support multiple address recognition with two address mask register
     * |        |          |When the bit in the address mask register is set to one, it means the received corresponding address bit is don't-care
     * |        |          |If the bit is set to zero, that means the received corresponding register bit should be exact the same as address register.
     * @var I2C_T::CTL2
     * Offset: 0x40  I2C Control Register 2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |WKUPEN    |I2C Wake-up Function Enable Bit
     * |        |          |0 = I2C wake-up function Disabled.
     * |        |          |1 = I2C wake-up function Enabled.
     * |[1]     |OVIEN     |I2C Overrun Interrupt Control Bit
     * |        |          |0 = Overrun event interrupt Disabled.
     * |        |          |1 = Send a interrupt to system when the TWOLVBUF bit is enabled and there is overrun event in received buffer.
     * |[2]     |URIEN     |I2C Under run Interrupt Control Bit
     * |        |          |0 = Under run event interrupt Disabled.
     * |        |          |1 = Send a interrupt to system when the TWOLVBUF bit is enabled and there is under-run event happened in transmitted buffer.
     * |[4]     |TWOLVBUF  |Two Level Buffer Enable Bit
     * |        |          |0 = Two level buffer Disabled.
     * |        |          |1 = Two level buffer Enabled.
     * |[5]     |NOSTRETCH |I2C BuS Stretch
     * |        |          |0 = The I2C SCL bus is stretched by hardware if the SI (I2C_CTL[4]) is not cleared.
     * |        |          |1 = The I2C SCL bus is not stretched by hardware if the SI is not cleared.
     * |[6]     |DATMODE   |Data Mode Enable Bit
     * |        |          |0 = Data mode Disabled.
     * |        |          |1 = Data mode Enabled.
     * |[7]     |MSDAT     |Master or Slave in Data Mode Enable Control
     * |        |          |0 = Master writes data to device.
     * |        |          |1 = Slave reads data from device.
     * @var I2C_T::STATUS2
     * Offset: 0x44  I2C Status Register 2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |WKIF      |Wake-up Interrupt Flag
     * |        |          |0 = Wake-up flag is inactive.
     * |        |          |1 = Wake-up flag is active.
     * |        |          |Note: This bit can be cleared by writing 1 to it.
     * |[1]     |OVIF      |I2C Overrun Status Bit
     * |        |          |0 = The received buffer is not overrun when the TWOLVBUF = 1.
     * |        |          |1 = The received buffer is overrun when the TWOLVBUF = 1.
     * |        |          |Note: This bit can be cleared by writing 1 to it.
     * |[2]     |URIF      |I2C Under run Status Bit
     * |        |          |0 = The transmitted buffer is not Under run when the TWOLVBUF = 1.
     * |        |          |1 = The transmitted buffer is Under run when the TWOLVBUF = 1.
     * |        |          |Note: This bit can be cleared by writing 1 to it.
     * |[3]     |WRSTSWK   |I2C Read/Write Status Bit in Address Wake-up Frame
     * |        |          |0 = Write command is recorded on the address match wake-up frame.
     * |        |          |1 = Read command is recorded on the address match wake-up frame.
     * |[4]     |FULL      |I2C Two Level Buffer Full
     * |        |          |0 = TX buffer no full when the TWOLVBUF = 1.
     * |        |          |1 = TX buffer full when the TWOLVBUF = 1.
     * |[5]     |EMPTY     |I2C Two Level Buffer Empty
     * |        |          |0 = RX buffer is not empty when the TWOLVBUF = 1.
     * |        |          |1 = RX buffer is empty when the TWOLVBUF = 1.
     * |[6]     |BUSFREE   |Bus Free Status
     * |        |          |The bus status in the controller.
     * |        |          |0 = I2C's 'Start' condition is detected on the bus.
     * |        |          |1 = Bus is free and released by 'STOP' condition or the controller is disabled.
     */
    __IO uint32_t CTL;                   /*!< [0x0000] I2C Control Register                                             */
    __IO uint32_t INTSTS;                /*!< [0x0004] I2C Interrupt Status Register                                    */
    __I  uint32_t STATUS;                /*!< [0x0008] I2C Status Register                                              */
    __IO uint32_t CLKDIV;                /*!< [0x000c] I2C Clock Divided Register                                       */
    __IO uint32_t TOCTL;                 /*!< [0x0010] I2C Time-out Control Register                                    */
    __IO uint32_t DAT;                   /*!< [0x0014] I2C Data Register                                                */
    __IO uint32_t ADDR0;                 /*!< [0x0018] I2C Slave Address Register0                                      */
    __IO uint32_t ADDR1;                 /*!< [0x001c] I2C Slave Address Register1                                      */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE0[2];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t ADDRMSK0;              /*!< [0x0028] I2C Slave Address Mask Register0                                 */
    __IO uint32_t ADDRMSK1;              /*!< [0x002c] I2C Slave Address Mask Register1                                 */
    /// @cond HIDDEN_SYMBOLS
    __I  uint32_t RESERVE1[4];
    /// @endcond //HIDDEN_SYMBOLS
    __IO uint32_t CTL2;                  /*!< [0x0040] I2C Control Register 2                                           */
    __IO uint32_t STATUS2;               /*!< [0x0044] I2C Status Register 2                                            */

} I2C_T;

/**
    @addtogroup I2C_CONST I2C Bit Field Definition
    Constant Definitions for I2C Controller
@{ */

#define I2C_CTL_I2CEN_Pos                (0)                                               /*!< I2C_T::CTL: I2CEN Position             */
#define I2C_CTL_I2CEN_Msk                (0x1ul << I2C_CTL_I2CEN_Pos)                      /*!< I2C_T::CTL: I2CEN Mask                 */

#define I2C_CTL_AA_Pos                   (1)                                               /*!< I2C_T::CTL: AA Position                */
#define I2C_CTL_AA_Msk                   (0x1ul << I2C_CTL_AA_Pos)                         /*!< I2C_T::CTL: AA Mask                    */

#define I2C_CTL_STO_Pos                  (2)                                               /*!< I2C_T::CTL: STO Position               */
#define I2C_CTL_STO_Msk                  (0x1ul << I2C_CTL_STO_Pos)                        /*!< I2C_T::CTL: STO Mask                   */

#define I2C_CTL_STA_Pos                  (3)                                               /*!< I2C_T::CTL: STA Position               */
#define I2C_CTL_STA_Msk                  (0x1ul << I2C_CTL_STA_Pos)                        /*!< I2C_T::CTL: STA Mask                   */

#define I2C_CTL_SI_Pos                   (4)                                               /*!< I2C_T::CTL: SI Position                */
#define I2C_CTL_SI_Msk                   (0x1ul << I2C_CTL_SI_Pos)                         /*!< I2C_T::CTL: SI Mask                    */

#define I2C_CTL_INTEN_Pos                (7)                                               /*!< I2C_T::CTL: INTEN Position             */
#define I2C_CTL_INTEN_Msk                (0x1ul << I2C_CTL_INTEN_Pos)                      /*!< I2C_T::CTL: INTEN Mask                 */

#define I2C_INTSTS_INTSTS_Pos            (0)                                               /*!< I2C_T::INTSTS: INTSTS Position         */
#define I2C_INTSTS_INTSTS_Msk            (0x1ul << I2C_INTSTS_INTSTS_Pos)                  /*!< I2C_T::INTSTS: INTSTS Mask             */

#define I2C_INTSTS_TOIF_Pos              (1)                                               /*!< I2C_T::INTSTS: TOIF Position           */
#define I2C_INTSTS_TOIF_Msk              (0x1ul << I2C_INTSTS_TOIF_Pos)                    /*!< I2C_T::INTSTS: TOIF Mask               */

#define I2C_INTSTS_WKAKDONE_Pos          (7)                                               /*!< I2C_T::INTSTS: WKAKDONE Position       */
#define I2C_INTSTS_WKAKDONE_Msk          (0x1ul << I2C_INTSTS_WKAKDONE_Pos)                /*!< I2C_T::INTSTS: WKAKDONE Mask           */

#define I2C_STATUS_STATUS_Pos            (0)                                               /*!< I2C_T::STATUS: STATUS Position         */
#define I2C_STATUS_STATUS_Msk            (0xfful << I2C_STATUS_STATUS_Pos)                 /*!< I2C_T::STATUS: STATUS Mask             */

#define I2C_CLKDIV_DIVIDER_Pos           (0)                                               /*!< I2C_T::CLKDIV: DIVIDER Position        */
#define I2C_CLKDIV_DIVIDER_Msk           (0xfful << I2C_CLKDIV_DIVIDER_Pos)                /*!< I2C_T::CLKDIV: DIVIDER Mask            */

#define I2C_TOCTL_TOCEN_Pos              (0)                                               /*!< I2C_T::TOCTL: TOCEN Position           */
#define I2C_TOCTL_TOCEN_Msk              (0x1ul << I2C_TOCTL_TOCEN_Pos)                    /*!< I2C_T::TOCTL: TOCEN Mask               */

#define I2C_TOCTL_TOCDIV4_Pos            (1)                                               /*!< I2C_T::TOCTL: TOCDIV4 Position         */
#define I2C_TOCTL_TOCDIV4_Msk            (0x1ul << I2C_TOCTL_TOCDIV4_Pos)                  /*!< I2C_T::TOCTL: TOCDIV4 Mask             */

#define I2C_DAT_DAT_Pos                  (0)                                               /*!< I2C_T::DAT: DAT Position               */
#define I2C_DAT_DAT_Msk                  (0xfful << I2C_DAT_DAT_Pos)                       /*!< I2C_T::DAT: DAT Mask                   */

#define I2C_ADDR0_GC_Pos                 (0)                                               /*!< I2C_T::ADDR0: GC Position              */
#define I2C_ADDR0_GC_Msk                 (0x1ul << I2C_ADDR0_GC_Pos)                       /*!< I2C_T::ADDR0: GC Mask                  */

#define I2C_ADDR0_ADDR_Pos               (1)                                               /*!< I2C_T::ADDR0: ADDR Position            */
#define I2C_ADDR0_ADDR_Msk               (0x7ful << I2C_ADDR0_ADDR_Pos)                    /*!< I2C_T::ADDR0: ADDR Mask                */

#define I2C_ADDR1_GC_Pos                 (0)                                               /*!< I2C_T::ADDR1: GC Position              */
#define I2C_ADDR1_GC_Msk                 (0x1ul << I2C_ADDR1_GC_Pos)                       /*!< I2C_T::ADDR1: GC Mask                  */

#define I2C_ADDR1_ADDR_Pos               (1)                                               /*!< I2C_T::