package com.nuvoton.slidetab;

import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothServerSocket;
import android.bluetooth.BluetoothSocket;
import android.content.Context;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.util.Log;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.UUID;
/**
 * This class does all the work for setting up and managing Bluetooth
 * connections with other devices. It has a thread that listens for
 * incoming connections, a thread for connecting with a device, and a
 * thread for performing data transmissions when connected.
 */
public class BluetoothService {
    // Debugging
    private static final String TAG = "BluetoothService";
    private static final boolean D = true;

    // Name for the SDP record when creating server socket
    private static final String NAME = "BluetoothChat";

    // Unique UUID for this application
    //private static final UUID MY_UUID = UUID.fromString("fa87c0d0-afac-11de-8a39-0800200c9a66");
    private static final UUID MY_UUID = UUID.fromString("00001101-0000-1000-8000-00805F9B34FB");

    // Member fields
    private final BluetoothAdapter mAdapter;
    private final Handler mHandler;
    private AcceptThread mAcceptThread;
    private ConnectThread mConnectThread;
    private ConnectedThread mConnectedThread;
    private int mState;
    private int mType;
    private boolean mTypechangeflag=false;

    // Constants that indicate the current connection state
    public static final int STATE_NONE = 0;       // we're doing nothing
    public static final int STATE_LISTEN = 1;     // now listening for incoming connections
    public static final int STATE_CONNECTING = 2; // now initiating an outgoing connection
    public static final int STATE_CONNECTED = 3;  // now connected to a remote device

    public static final int TYPE_INDEX = 0;
    public static final int TYPE_LINK = 1;
    public static final int TYPE_BATTERY = 2;
    public static final int TYPE_BUZZER = 3;
    public static final int TYPE_LED = 4;
    public static final int TYPE_AHRS = 5;
    public static final int TYPE_SONAR = 6;
    public static final int TYPE_TEMP = 7;
    public static final int TYPE_GAS = 8;
    public static final int TYPE_IR = 9;
    public static final int TYPE_KEY = 10;

    /**
     * Constructor. Prepares a new BluetoothChat session.
     * @param context  The UI Activity Context
     * @param handler  A Handler to send messages back to the UI Activity
     */
    public BluetoothService(Context context, Handler handler) {
        mAdapter = BluetoothAdapter.getDefaultAdapter();
        mState = STATE_NONE;
        mHandler = handler;
    }

    /**
     * Set the current state of the chat connection
     * @param state  An integer defining the current connection state
     */
    private synchronized void setState(int state) {
        if (D) Log.d(TAG, "setState() " + mState + " -> " + state);
        mState = state;

        // Give the new state to the Handler so the UI Activity can update
        mHandler.obtainMessage(NubrickActivity.MESSAGE_STATE_CHANGE, state, -1).sendToTarget();
    }

    /**
     * Return the current connection state. */
    public synchronized int getState() {
        return mState;
    }

    /**
     * Start the chat service. Specifically start AcceptThread to begin a
     * session in listening (server) mode. Called by the Activity onResume() */
    public synchronized void start() {
        if (D) Log.d(TAG, "start");

        // Cancel any thread attempting to make a connection
        if (mConnectThread != null) {mConnectThread.cancel(); mConnectThread = null;}

        // Cancel any thread currently running a connection
        if (mConnectedThread != null) {mConnectedThread.cancel(); mConnectedThread = null;}

        // Start the thread to listen on a BluetoothServerSocket

        if (mAcceptThread == null) {
            mAcceptThread = new AcceptThread();
            mAcceptThread.start();
        }
        setState(STATE_LISTEN);
    }

    /**
     * Start the ConnectThread to initiate a connection to a remote device.
     * @param device  The BluetoothDevice to connect
     */
    public synchronized void connect(BluetoothDevice device) {
        if (D) Log.d(TAG, "connect to: " + device);

        // Cancel any thread attempting to make a connection
        if (mState == STATE_CONNECTING) {
            if (mConnectThread != null) {mConnectThread.cancel(); mConnectThread = null;}
        }

        // Cancel any thread currently running a connection
        if (mConnectedThread != null) {mConnectedThread.cancel(); mConnectedThread = null;}

        // Start the thread to connect with the given device
        mConnectThread = new ConnectThread(device);
        mConnectThread.start();
        setState(STATE_CONNECTING);
    }

    /**
     * Start the ConnectedThread to begin managing a Bluetooth connection
     * @param socket  The BluetoothSocket on which the connection was made
     * @param device  The BluetoothDevice that has been connected
     */
    public synchronized void connected(BluetoothSocket socket, BluetoothDevice device) {
        if (D) Log.d(TAG, "connected");

        // Cancel the thread that completed the connection
        if (mConnectThread != null) {mConnectThread.cancel(); mConnectThread = null;}

        // Cancel any thread currently running a connection
        if (mConnectedThread != null) {mConnectedThread.cancel(); mConnectedThread = null;}

        // Cancel the accept thread because we only want to connect to one device
        if (mAcceptThread != null) {mAcceptThread.cancel(); mAcceptThread = null;}

        // Start the thread to manage the connection and perform transmissions
        mConnectedThread = new ConnectedThread(socket);
        mConnectedThread.start();

        // Send the name of the connected device back to the UI Activity
        Message msg = mHandler.obtainMessage(NubrickActivity.MESSAGE_DEVICE_NAME);
        Bundle bundle = new Bundle();
        bundle.putString(NubrickActivity.DEVICE_NAME, device.getName());
        msg.setData(bundle);
        mHandler.sendMessage(msg);

        setState(STATE_CONNECTED);
    }

    /**
     * Stop all threads
     */
    public synchronized void stop() {
        if (D) Log.d(TAG, "stop");
        if (mConnectThread != null) {mConnectThread.cancel(); mConnectThread = null;}
        if (mConnectedThread != null) {mConnectedThread.cancel(); mConnectedThread = null;}
        if (mAcceptThread != null) {mAcceptThread.cancel(); mAcceptThread = null;}
        setState(STATE_NONE);
    }

    /**
     * Write to the ConnectedThread in an unsynchronized manner
     * @param out The bytes to write
     * @see ConnectedThread#write(byte[])
     */
    public void write(byte[] out) {
        // Create temporary object
        ConnectedThread r;
        // Synchronize a copy of the ConnectedThread
        synchronized (this) {
            if (mState != STATE_CONNECTED) return;
            r = mConnectedThread;
        }
        // Perform the write unsynchronized
        r.write(out);
    }

    public void read(byte[] in,int inlen){
        ConnectedThread r;
        // Synchronize a copy of the ConnectedThread
        synchronized (this) {
            if (mState != STATE_CONNECTED) return;
            r = mConnectedThread;
        }
        r.read(in, inlen);
    }

    public void clear()
    {
        ConnectedThread r;
        // Synchronize a copy of the ConnectedThread
        synchronized (this) {
            if (mState != STATE_CONNECTED) return;
            r = mConnectedThread;
        }
        r.clear();
    }

    public void setReceiveType(int type)
    {
        mType = type;
        mTypechangeflag=true;
    }
    /**
     * Indicate that the connection attempt failed and notify the UI Activity.
     */
    private void connectionFailed() {
        setState(STATE_LISTEN);

        // Send a failure message back to the Activity
        Message msg = mHandler.obtainMessage(NubrickActivity.MESSAGE_TOAST);
        Bundle bundle = new Bundle();
        bundle.putString(NubrickActivity.TOAST, "Unable to connect device");
        msg.setData(bundle);
        mHandler.sendMessage(msg);
    }

    /**
     * Indicate that the connection was lost and notify the UI Activity.
     */
    private void connectionLost() {
        setState(STATE_LISTEN);

        // Send a failure message back to the Activity
        Message msg = mHandler.obtainMessage(NubrickActivity.MESSAGE_TOAST);
        Bundle bundle = new Bundle();
        bundle.putString(NubrickActivity.TOAST, "Device connection was lost");
        msg.setData(bundle);
        mHandler.sendMessage(msg);
    }

    /**
     * This thread runs while listening for incoming connections. It behaves
     * like a server-side client. It runs until a connection is accepted
     * (or until cancelled).
     */
    private class AcceptThread extends Thread {
        // The local server socket
        private final BluetoothServerSocket mmServerSocket;

        public AcceptThread() {
            BluetoothServerSocket tmp = null;

            // Create a new listening server socket
            try {
                tmp = mAdapter.listenUsingRfcommWithServiceRecord(NAME, MY_UUID);
            } catch (IOException e) {
                Log.e(TAG, "listen() failed", e);
            }
            mmServerSocket = tmp;
        }

        public void run() {
            if (D) Log.d(TAG, "BEGIN mAcceptThread" + this);
            setName("AcceptThread");
            BluetoothSocket socket = null;

            // Listen to the server socket if we're not connected
            while (mState != STATE_CONNECTED) {
                try {
                    // This is a blocking call and will only return on a
                    // successful connection or an exception
                    socket = mmServerSocket.accept();
                } catch (IOException e) {
                    Log.e(TAG, "accept() failed", e);
                    break;
                }

                // If a connection was accepted
                if (socket != null) {
                    synchronized (BluetoothService.this) {
                        switch (mState) {
                            case STATE_LISTEN:
                            case STATE_CONNECTING:
                                // Situation normal. Start the connected thread.
                                connected(socket, socket.getRemoteDevice());
                                break;
                            case STATE_NONE:
                            case STATE_CONNECTED:
                                // Either not ready or already connected. Terminate new socket.
                                try {
                                    socket.close();
                                } catch (IOException e) {
                                    Log.e(TAG, "Could not close unwanted socket", e);
                                }
                                break;
                        }
                    }
                }
            }
            if (D) Log.i(TAG, "END mAcceptThread");
        }

        public void cancel() {
            if (D) Log.d(TAG, "cancel " + this);
            try {
                mmServerSocket.close();
            } catch (IOException e) {
                Log.e(TAG, "close() of server failed", e);
            }
        }
    }


    /**
     * This thread runs while attempting to make an outgoing connection
     * with a device. It runs straight through; the connection either
     * succeeds or fails.
     */
    private class ConnectThread extends Thread {
        private final BluetoothSocket mmSocket;
        private final BluetoothDevice mmDevice;

        public ConnectThread(BluetoothDevice device) {
            mmDevice = device;
            BluetoothSocket tmp = null;

            // Get a BluetoothSocket for a connection with the
            // given BluetoothDevice
            try {
                tmp = device.createRfcommSocketToServiceRecord(MY_UUID);
            } catch (IOException e) {
                Log.e(TAG, "create() failed", e);
            }
            mmSocket = tmp;
        }

        public void run() {
            Log.i(TAG, "BEGIN mConnectThread");
            setName("ConnectThread");

            // Always cancel discovery because it will slow down a connection
            mAdapter.cancelDiscovery();

            // Make a connection to the BluetoothSocket
            try {
                // This is a blocking call and will only return on a
                // successful connection or an exception
                mmSocket.connect();
            } catch (IOException e) {
                connectionFailed();
                // Close the socket
                try {
                    mmSocket.close();
                } catch (IOException e2) {
                    Log.e(TAG, "unable to close() socket during connection failure", e2);
                }
                // Start the service over to restart listening mode
                BluetoothService.this.start();
                return;
            }

            // Reset the ConnectThread because we're done
            synchronized (BluetoothService.this) {
                mConnectThread = null;
            }

            // Start the connected thread
            connected(mmSocket, mmDevice);
        }

        public void cancel() {
            try {
                mmSocket.close();
            } catch (IOException e) {
                Log.e(TAG, "close() of connect socket failed", e);
            }
        }
    }

    /**
     * This thread runs during a connection with a remote device.
     * It handles all incoming and outgoing transmissions.
     */
    private class ConnectedThread extends Thread {
        private final BluetoothSocket mmSocket;
        private  InputStream mmInStream;
        private final OutputStream mmOutStream;

        public ConnectedThread(BluetoothSocket socket) {
            Log.d(TAG, "create ConnectedThread");
            mmSocket = socket;
            InputStream tmpIn = null;
            OutputStream tmpOut = null;

            // Get the BluetoothSocket input and output streams
            try {
                tmpIn = socket.getInputStream();
                tmpOut = socket.getOutputStream();
            } catch (IOException e) {
                Log.e(TAG, "temp sockets not created", e);
            }

            mmInStream = tmpIn;
            mmOutStream = tmpOut;
        }
/*
        public void run() {
            Log.i(TAG, "BEGIN mConnectedThread");

            byte[] buffer = new byte[1024];
            int bytes;

            // Keep listening to the InputStream while connected
            while (true) {

                try {
                    // Read from the InputStream
                    bytes = mmInStream.read(buffer);

                    // Send the obtained bytes to the UI Activity
                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                            .sendToTarget();
                } catch (IOException e) {
                    Log.e(TAG, "disconnected", e);
                    connectionLost();
                    break;
                }

            }

        }
        */
private class INDEX_REPORT {
    int indRepLen;                  //Index report length
    int devNum;                     //device prepared to ne used
    int devSta;                     //device connected state
    int indDatLen;                  //Index received data length
};
        INDEX_REPORT indexRep = new INDEX_REPORT();
        public void run() {
            Log.i(TAG, "BEGIN mConnectedThread");

            byte[] buffer = new byte[1024];
            byte[] head = new byte[2];
            int bytes=0;
            int count=0;
            int readstate=0;

            // Keep listening to the InputStream while connected
            while (true) {
                if(mTypechangeflag){
                    readstate=0;
                    mTypechangeflag=false;
                }

                try {
                    // Read from the InputStream
                    if(readstate==0) {
                        //head[1] = head[0];
                        head[0] = (byte) mmInStream.read();
                        //Log.d("####","head:"+String.valueOf(head[0]));
                        count++;
                        if(head[0]==0x55)
                        {
                            head[0] = (byte) mmInStream.read();
                            if(head[0]==0x55){
                                readstate = 1;
                                count=0;
                                //Log.d("####","head:"+"get");

                            }
                            count++;
                        }
                        //Log.d("####","head:"+String.valueOf(head[0])+":"+String.valueOf(head[1]));

                        if(count<=10 && readstate==0) {
                            continue;
                        }
                        else if(count>10&& readstate==0){ //not receive 257
                            buffer[0] = 'e';
                            buffer[1] = '1';
                            mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, 2, -1, buffer)
                                    .sendToTarget();
                            readstate = 0;
                            count=0;
                        }
                    }
                        switch (mType){
                            case TYPE_INDEX:
                                //bytes=8;
                                if(readstate==1) {
                                    bytes = read(buffer, 8);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate=2;
                                }else if(readstate==2) {
                                    bytes = read(buffer, 34);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                }
                                break;
                            case TYPE_LINK:
                                if(readstate==1) {
                                    bytes = read(buffer,8);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 2;
                                }else if(readstate==2) {
                                    bytes = read(buffer,6);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 3;
                                }else if(readstate==3){
                                   // bytes = read(buffer,6);
                                    mmInStream.read();
                                }
                                break;
                            case TYPE_BATTERY:
                                if(readstate==1){
                                    bytes = read(buffer,26);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 2;
                                }else if(readstate==2) {
                                    bytes = read(buffer,36);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 3;
                                }else if(readstate==3) {
                                    bytes = read(buffer, 9);
                                   // Log.d("bluetooth","readbuf0"+String.valueOf((buffer[0] | buffer[1] << 8)));
                                    //Log.d("bluetooth","readbuf1"+String.valueOf((buffer[9] | buffer[10] << 8)));
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 4;
                                }else if(readstate==4){
                                    bytes = read(buffer, 9);
                                    //Log.d("bluetooth","1readbuf0"+String.valueOf((buffer[0] | buffer[1] << 8)));
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    //mmInStream.read();
                                }
                                break;
                            case TYPE_BUZZER:
                                if(readstate==1){
                                    bytes = read(buffer,26);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 2;
                                }else if(readstate==2) {
                                    bytes = read(buffer,74);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 3;
                                }else if(readstate==3){
                                    bytes = read(buffer,15);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 4;
                                }else if(readstate==4){
                                    bytes = read(buffer,15);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    //mmInStream.read();
                                }
                                break;
                            case TYPE_LED:
                                if(readstate==1){
                                    bytes = read(buffer,26);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 2;
                                }else if(readstate==2) {
                                    bytes = read(buffer,74);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 3;
                                }else if(readstate==3){
                                    bytes = read(buffer,15);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 4;
                                }else if(readstate==4){
                                    bytes = read(buffer,15);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    //mmInStream.read();
                                }
                                break;
                            case TYPE_AHRS:
                                if(readstate==1){
                                    bytes = read(buffer,26);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 2;
                                }else if(readstate==2) {
                                    bytes = read(buffer,34);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 3;
                                }else if(readstate==3){
                                    bytes = read(buffer,8);
                                    //Log.d("bluetooth","readbuf:"+String.valueOf(buffer[0])+":"+String.valueOf(buffer[1])
                                   // +":"+String.valueOf(buffer[2])+":"+String.valueOf(buffer[3])+":"+String.valueOf(buffer[4])
                                    //        +":"+String.valueOf(buffer[5])+":"+String.valueOf(buffer[6])+":"+String.valueOf(buffer[7]));
                                    //Log.d("bluetooth","readbuf1"+String.valueOf((buffer[7] | buffer[8] << 8)));
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 4;
                                }else if(readstate==4){
                                   bytes = read(buffer,8);
                                    //Log.d("bluetooth","##readbuf:"+String.valueOf(buffer[0])+":"+String.valueOf(buffer[1])
                                    //        +":"+String.valueOf(buffer[2])+":"+String.valueOf(buffer[3])+":"+String.valueOf(buffer[4])
                                    //        +":"+String.valueOf(buffer[5])+":"+String.valueOf(buffer[6])+":"+String.valueOf(buffer[7]));
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    //mmInStream.read();
                                }
                                break;
                            case TYPE_SONAR:
                                if(readstate==1){
                                    bytes = read(buffer,26);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 2;
                                }else if(readstate==2) {
                                    bytes = read(buffer,36);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 3;
                                }else if(readstate==3){
                                    bytes = read(buffer,9);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 4;
                                }else if(readstate==4){
                                    bytes = read(buffer,9);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    //mmInStream.read();
                                }
                                break;
                            case TYPE_TEMP:
                                if(readstate==1){
                                    bytes = read(buffer,26);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 2;
                                }else if(readstate==2) {
                                    bytes = read(buffer,58);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 3;
                                }else if(readstate==3){
                                    bytes = read(buffer,14);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 4;
                                }else if(readstate==4){
                                    bytes = read(buffer,14);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    //mmInStream.read();
                                }
                                break;
                            case TYPE_GAS:
                                if(readstate==1){
                                    bytes = read(buffer,26);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 2;
                                }else if(readstate==2) {
                                    bytes = read(buffer,36);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 3;
                                }else if(readstate==3){
                                    bytes = read(buffer,9);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 4;
                                }else if(readstate==4){
                                    bytes = read(buffer,9);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    //mmInStream.read();
                                }
                                break;
                            case TYPE_IR:
                                if(readstate==1){
                                    bytes = read(buffer,26);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 2;
                                }else if(readstate==2) {
                                    bytes = read(buffer,58);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 3;
                                }else if(readstate==3){
                                    bytes = read(buffer,11);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 4;
                                }else if(readstate==4){
                                    bytes = read(buffer,11);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    //mmInStream.read();
                                }
                                break;
                            case TYPE_KEY:
                                if(readstate==1){
                                    bytes = read(buffer,26);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 2;
                                }else if(readstate==2) {
                                    bytes = read(buffer,22);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 3;
                                }else if(readstate==3){
                                    bytes = read(buffer,6);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    readstate = 4;
                                }else if(readstate==4){
                                    bytes = read(buffer,6);
                                    mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                            .sendToTarget();
                                    //mmInStream.read();
                                }
                                break;
                        }
                    /*
                        if(bytes!=0){
                            mHandler.obtainMessage(NubrickActivity.MESSAGE_READ, bytes, -1, buffer)
                                    .sendToTarget();
                        }
                    */

                    // Send the obtained bytes to the UI Activity

                } catch (IOException e) {
                    Log.e(TAG, "disconnected", e);
                    connectionLost();
                    break;
                }

            }

        }
        public  int read(byte[] buffer,int len)
        {
            int readCount = 0; //�Ѿ��ɹ���ȡ���ֽڵĸ���
            int ret;
            int zcont=0;
            while (readCount < len) {
                try {
                    ret =  mmInStream.read(buffer, readCount, len - readCount);
                    if (ret<0){
                        return 0;
                    }
                    else if(ret==0)
                    {
                        if(zcont++>50)
                            return readCount;
                    }else {
                        readCount += ret;
                    }
                } catch (IOException e) {
                    e.printStackTrace();
                    break;
                }
            }
            return readCount;
        }
        public void clear()
        {
            /*
            try {
                mmInStream.close();
                mmInStream=null;
            } catch (IOException e) {
                e.printStackTrace();
            }
            */
            try {
                while(mmInStream.available()!=0)
                {
                    mmInStream.read();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        /**
         * Write to the connected OutStream.
         * @param buffer  The bytes to write
         */
        public void write(byte[] buffer) {

            try {
                mmOutStream.write(buffer);

                // Share the sent message back to the UI Activity
                mHandler.obtainMessage(NubrickActivity.MESSAGE_WRITE, -1, -1, buffer)
                        .sendToTarget();
            } catch (IOException e) {
                Log.e(TAG, "Exception during write", e);
            }
        }

        public void cancel() {
            try {
                mmSocket.close();
            } catch (IOException e) {
                Log.e(TAG, "close() of connect socket failed", e);
            }
        }
    }
}
