/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/07/24 11:19 $
 * @brief    Sample code for System Control.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"

#define SIGNATURE       0x125ab234
#define FLAG_ADDR       0x20000FFC      /* last 4 bytes on end of SRAM */


void SYS_Init(void)
{
    /* Unlock protected registers */
    SYS_UnlockReg();

    CLK_SetCoreClock(FREQ_60MHZ);

    /* Enable USCI2 IP clock */
    CLK_EnableModuleClock(USCI2_MODULE);

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);
	
    /* Lock protected registers */
    SYS_LockReg();
}


__IO uint32_t u32PWDU_WakeFlag = 0;
__IO uint32_t u32WDT_Ticks = 0;
/*---------------------------------------------------------------------------------------------------------*/
/*  Brown Out Detector IRQ Handler                                                                         */
/*---------------------------------------------------------------------------------------------------------*/
void BOD_IRQHandler(void)
{
    SYS_CLEAR_BOD_INT_FLAG(); /* Clear BOD Interrupt Flag */
    printf("Brown Out is Detected\n");
}


/*---------------------------------------------------------------------------------------------------------*/
/*  Watchdog IRQ Handler                                                                                   */
/*---------------------------------------------------------------------------------------------------------*/
void WDT_IRQHandler(void)
{
    uint32_t wait_uart_stable = 2000;

    /* Clear WDT interrupt flag */
    WDT_CLEAR_TIMEOUT_INT_FLAG();

    /* Check WDT wake up flag */
    if(WDT_GET_TIMEOUT_WAKEUP_FLAG()) {
        /* Wait uart is stable */
        while(wait_uart_stable -- > 0);
        printf("\nWake up by WDT !!\n");
        /* Clear WDT wake up flag */
        WDT_CLEAR_TIMEOUT_WAKEUP_FLAG();
    } else {
        u32WDT_Ticks++;
        printf("WDT interrupt !!\n");
    }
}


/*---------------------------------------------------------------------------------------------------------*/
/*  Power Down Wake Up IRQ Handler                                                                         */
/*---------------------------------------------------------------------------------------------------------*/
void PWRWU_IRQHandler(void)
{
    printf("PDWU_IRQHandler running...\n");
    u32PWDU_WakeFlag = 1;
    CLK->PWRCTL |= CLK_PWRCTL_PDWKIF_Msk;    /* Clear Wake-up Interrupts */
}
	
int main()
{
    uint32_t u32data;
    uint32_t wait_uart_stable = 1000;
    /* Wait uart is stable (for system reset)*/
    while(wait_uart_stable -- > 0);
    
    /* In end of main function, program issued CPU reset and write-protection will be disabled. */
    if(SYS->REGLCTL == 1)
        SYS_LockReg();  /* enable write-protection function if it is disabled. */

    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);
		
    /*--- 1. Read PDID */
    printf("\n\nPDID 0x%08X\n", SYS_ReadPDID());    /* Display PDID */
    printf("CPU @ %dHz\n", SystemCoreClock);        /* Display System Core Clock */

    /*
     * This sample code will show some function about system manager controller and clock controller:
     * 1. Read PDID
     * 2. Get and clear reset source
     * 3. Setting about BOD
     * 4. System write-protection enable (lock) / disable (unlock)
     * 5. Power-down mode sleep and wake up by Watchdog timer
     * 6. CPU reset
     */
    printf("+-----------------------------------------+\n");
    printf("| NM1240 System Control Sample Code       |\n");
    printf("+-----------------------------------------+\n");

    /*--- 6-2. CPU reset */
    if (M32(FLAG_ADDR) == SIGNATURE)
    {
        printf("  CPU Reset success!\n");   /* SRAM data still be kept if just CPU reset. */
        M32(FLAG_ADDR) = 0;
        printf("  Press any key to continue ...\n");
        getchar();
    }

    /*--- 2. Get and clear reset source */
    /* Get reset source from last operation */
    u32data = SYS_GetResetSrc();
    printf("Get Reset Source = 0x%x\n", u32data);

    /* Clear reset source */
    printf("Clear Reset Source.\n");
    SYS_ClearResetSrc(u32data);

    u32data = SYS_GetResetSrc();
    printf("Get Reset Source again = 0x%x\n\n", u32data);

    /*--- 4. System write-protection enable (lock) / disable (unlock) */
    /* Unlock protected registers for Brown-Out Detector and power down settings */
    SYS_UnlockReg();

    /* Check if the write-protected registers are unlocked before BOD setting and CPU Reset */
    if (! SYS_IsRegLocked())
    {
        printf("Protected Address is Unlocked\n\n");
    }

    /*--- 3. Setting about BOD */
    /* Enable Brown-Out Detector and Low Voltage Reset function, and set Brown-Out Detector voltage 2.4V */
    SYS_EnableBOD(SYS_BODCTL_BOD_RST_EN, SYS_BODCTL_BOD_VL_2_4V);
    NVIC_EnableIRQ(BOD_IRQn);  /* Enable BOD IRQ */

    /*--- 5. Power-down mode sleep and wake up by Watchdog timer */
    /* Waiting for message send out */
    UUART_WAIT_TX_EMPTY(UUART2);
    CLK_EnableModuleClock(WDT_MODULE);  /* Enable WDT clock */
    /* WDT timeout every 2^16 WDT clock, disable system reset, enable wake up system */
    WDT_Open(WDT_TIMEOUT_2POW16, 0, FALSE, TRUE);
    WDT_EnableInt();                                /* Enable WDT interrupt */
    NVIC_EnableIRQ(WDT_IRQn);                       /* Enable WDT NVIC */

    /* Enable wake up interrupt source */
    CLK->PWRCTL |= CLK_PWRCTL_WAKEINT_EN;
    /* Enable IRQ request for PDWU interrupt */
    NVIC_EnableIRQ(PWRWU_IRQn);

    printf("u32PWDU_WakeFlag = %x\n",u32PWDU_WakeFlag);
    printf("Enter Power Down Mode and wait WDT interrupt >>>>>>>>>>>\n");
    /* clear software semaphore */
    u32PWDU_WakeFlag = 0;
    /* Waiting for message send out */
    UUART_WAIT_TX_EMPTY(UUART2);

    /* Let system enter to Power-down mode */
    CLK_PowerDown();

    printf("Waits for 3 times WDT interrupts.....\n");
    while (u32WDT_Ticks < 3);

    printf("<<<<<<<<<< Program resumes execution.\n");
    printf("u32PWDU_WakeFlag = %x\n\n",u32PWDU_WakeFlag);

    /*--- 6-1. CPU reset */
    /* Write a signature work to SRAM to check if it is reset by software */
    M32(FLAG_ADDR) = SIGNATURE;
    printf("\n\n  >>> Reset CPU <<<\n");

    /* Reset CPU */
    SYS_ResetCPU();
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
