/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/07/16 17:00 $
 * @brief    Sample code for System clock output to CLKO pin.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void SYS_Init(void)
{
    /* Unlock protected registers */
    SYS_UnlockReg();

    CLK_SetCoreClock(FREQ_60MHZ);

    /* Enable USCI2 IP clock */
    CLK_EnableModuleClock(USCI2_MODULE);

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Lock protected registers */
    SYS_LockReg();
}


void delay2s()
{
    CLK_SysTickDelay(200000);  /* delay 200ms */
    CLK_SysTickDelay(200000);  /* delay 200ms */
    CLK_SysTickDelay(200000);  /* delay 200ms */
    CLK_SysTickDelay(200000);  /* delay 200ms */
    CLK_SysTickDelay(200000);  /* delay 200ms */
    CLK_SysTickDelay(200000);  /* delay 200ms */
    CLK_SysTickDelay(200000);  /* delay 200ms */
    CLK_SysTickDelay(200000);  /* delay 200ms */
    CLK_SysTickDelay(200000);  /* delay 200ms */
    CLK_SysTickDelay(200000);  /* delay 200ms */
}


int main()
{
    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\n\nPDID 0x%08X\n", SYS_ReadPDID());    /* Display PDID */
    printf("CPU @ %dHz\n", SystemCoreClock);        /* Display System Core Clock */

    /*
     * This sample code will demonstrate how to output different clocks one after another 
     * to the same CLKO (PA0) pin.
     */
    printf("+-----------------------------------------+\n");
    printf("| NM1240 System Clock Output Sample Code  |\n");
    printf("+-----------------------------------------+\n");

    printf("Set GPA0 pin as CLKO output clock.\n");
    SYS->GPA_MFP = (SYS->GPA_MFP & ~SYS_GPA_MFP_PA0MFP_Msk) | SYS_GPA_MFP_PA0_CLKO;             /* GPA0 CLKO mode */
    PA->MODE = (PA->MODE & ~GPIO_MODE_MODE0_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE0_Pos);   /* GPA0 output mode */

    CLK_EnableModuleClock(CLKO_MODULE);

    printf("CLKO = HCLK / 1 = %dHz.\n", SystemCoreClock);
    CLK_EnableCKO(CLK_CLKO_SRC_HCLK, 0, 1);
    delay2s();

    printf("CLKO = HCLK / 2^(0+1) = %dHz.\n", SystemCoreClock/2);
    CLK_EnableCKO(CLK_CLKO_SRC_HCLK, 0, 0);
    delay2s();

    printf("CLKO = HCLK / 2^(2+1) = %dHz.\n", SystemCoreClock/8);
    CLK_EnableCKO(CLK_CLKO_SRC_HCLK, 2, 0);
    delay2s();

    SYS_UnlockReg();

    printf("CLKO = EXT / 1 = %luHz.\n", __EXT);
    CLK_EnableEXTCLK();
    CLK_EnableCKO(CLK_CLKO_SRC_EXT, 0, 1);
    delay2s();
    CLK_DisableEXTCLK();

    printf("CLKO = EXT / 2^(1+1) = %luHz.\n", __EXT/4);
    CLK_EnableEXTCLK();
    CLK_EnableCKO(CLK_CLKO_SRC_EXT, 1, 0);
    delay2s();
    CLK_DisableEXTCLK();

    SYS_LockReg();

    printf("=== THE END ===\n\n");
    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
