/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/07/13 13:27 $
 * @brief    Sample code for HDIV calculation feature.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void SYS_Init(void)
{
    /* Unlock protected registers */
    SYS_UnlockReg();

    CLK_SetCoreClock(FREQ_60MHZ);

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Lock protected registers */
    SYS_LockReg();

    /* Enable IP clock */
    CLK_EnableModuleClock(HDIV_MODULE);
    CLK_EnableModuleClock(USCI2_MODULE);
}


int main()
{
    int32_t x, y, q0, q1, q2, r0, r1, r2;
    int i;

    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\n\nPDID 0x%08X\n", SYS_ReadPDID());    /* Display PDID */
    printf("CPU @ %dHz\n", SystemCoreClock);        /* Display System Core Clock */

    /*
     * This sample code will demonstrate how to divide two signed integer by HDIV engine.
     */
    printf("+-------------------------------------------+\n");
    printf("|          NM1240 HDIV Sample Code          |\n");
    printf("+-------------------------------------------+\n");

    x = 123456;
    y = 21;
    for (i=0; i<5; i++)
    {
        q0 = HDIV0_Div(x, y);
        r0 = HDIV0_Mod(x, y);
        q1 = HDIV1_Div(x, y);
        r1 = HDIV1_Mod(x, y);
        q2 = HDIV2_Div(x, y);
        r2 = HDIV2_Mod(x, y);
		printf("Software calculation : %d / %d = %d ... %d\n", x, y, x / y, x % y);
        if (! HDIV_IS_DIVBYZERO0())
        {
            printf("HDIV0 calculation    : %d / %d = %d ... %d\n", x, y, q0, r0);
            
            if ((q0 == x / y) && (r0 == x % y))
                printf("==> HDIV0 Division PASS !!!\n");
            else
                printf("==> HDIV0 Division FAIL !!!\n");
        }
        else
        {
            printf("HDIV0 calculation : %d / %d, divide by 0 !!!\n", x, y);
            printf("==> Divide by zero checking PASS !!!\n");
        }
		
        if (! HDIV_IS_DIVBYZERO1())
        {
            printf("HDIV1 calculation    : %d / %d = %d ... %d\n", x, y, q1, r1);
            
            if ((q1 == x / y) && (r1 == x % y))
                printf("==> HDIV1 Division PASS !!!\n");
            else
                printf("==> HDIV1 Division FAIL !!!\n");
        }
        else
        {
            printf("HDIV1 calculation : %d / %d, divide by 0 !!!\n", x, y);
            printf("==> Divide by zero checking PASS !!!\n");
        }
		
        if (! HDIV_IS_DIVBYZERO2())
        {
            printf("HDIV2 calculation    : %d / %d = %d ... %d\n", x, y, q2, r2);
            
            if ((q2 == x / y) && (r2 == x % y))
                printf("==> HDIV2 Division PASS !!!\n");
            else
                printf("==> HDIV2 Division FAIL !!!\n");
        }
        else
        {
            printf("HDIV2 calculation : %d / %d, divide by 0 !!!\n", x, y);
            printf("==> Divide by zero checking PASS !!!\n");
        }
        y -= 7;
        printf("\n");
    }
		
    CLK_DisableModuleClock(HDIV_MODULE);
    printf("=== THE END ===\n\n");
    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
