/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 2 $
 * $Date: 2020/07/20 15:00 $
 * @brief    Sample code for GPIO Power-down Wake-up feature.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void SYS_Init(void)
{
    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Set Core Clock as 60MHz HIRC */
    CLK_SetCoreClock(FREQ_60MHZ);

    /* Enable USCI0 IP clock */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    GPIO_SetMode(PF, BIT1, GPIO_MODE_OUTPUT);
    GPIO_SetMode(PF, BIT2, GPIO_MODE_INPUT);

    /* Lock protected registers */
    SYS_LockReg();
}


/**
 * @brief       PortA/PortB/PortC/PortD IRQ
 *
 * @param       None
 *
 * @return      None
 *
 * @details     The PortA/PortB/PortC/PortD default IRQ, declared in startup_NM1240.s.
 */
void GPIO_IRQHandler(void)
{
    if(GPIO_GET_INT_FLAG(PB, BIT0))     /* To check if PB.0 interrupt occurred */
    {
        GPIO_CLR_INT_FLAG(PB, BIT0);    /* Clear PB.0 interrupt flag */
        /* printf("PB.0 INT occurred. \n"); */
    }
    else
    {    /* Un-expected interrupt. Just clear all PORTA, PORTB, PORTC, PORTD interrupts */
        GPIO_CLR_INT_FLAG(PA, GPIO_GET_INT_FLAG(PA, 0xFF));
        GPIO_CLR_INT_FLAG(PB, GPIO_GET_INT_FLAG(PB, 0xFF));
        GPIO_CLR_INT_FLAG(PC, GPIO_GET_INT_FLAG(PC, 0xFF));
        GPIO_CLR_INT_FLAG(PD, GPIO_GET_INT_FLAG(PD, 0xFE));
        GPIO_CLR_INT_FLAG(PE, GPIO_GET_INT_FLAG(PC, 0xFF));
        GPIO_CLR_INT_FLAG(PF, GPIO_GET_INT_FLAG(PD, 0x1F));
        /* printf("Un-expected interrupts. \n"); */
    }
}


int main()
{
    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\n\nPDID 0x%08X\n", SYS_ReadPDID());    /* Display PDID */
    printf("CPU @ %dHz\n", SystemCoreClock);        /* Display System Core Clock */

    /*
     * This sample code will demonstrate how to wake up system form Power-down mode
     * by GPIO interrupt.
     */
    printf("+--------------------------------------------+\n");
    printf("| NM1240 GPIO Power Down Wake Up Sample Code |\n");
    printf("+--------------------------------------------+\n");

    printf("Please keep PB.0 low and use rising edge to wake-up system ...\n");

    /* Config multiple function to GPIO mode for PB0 */
    SYS->GPB_MFP = (SYS->GPB_MFP & ~SYS_GPB_MFP_PB0MFP_Msk) | SYS_GPB_MFP_PB0_GPIO;
    GPIO_SetMode(PB, BIT0, GPIO_MODE_INPUT);
    NVIC_EnableIRQ(GP_IRQn);                    /* Enable GPIO NVIC */
    GPIO_EnableInt(PB, 0, GPIO_INT_RISING);     /* Enable PB0 interrupt by rising edge trigger */

    SYS_UnlockReg();
    /* Waiting for PB.0 rising-edge interrupt event */
    printf("Wait PB.0 to low\n");
    while(PB0 == 1);    /* wait PB.0 become low before get into power down mode */
    printf("Enter to Power-Down (rising-edge) ......  PB.0 = %d \n", PB0);
    UUART_WAIT_TX_EMPTY(UUART2);  /* To check if all the debug messages are finished */

    CLK_PowerDown();

    CLK_SysTickDelay(1000);
    printf("System waken-up done. PB.0 = %d\n\n", PB0);
    SYS_LockReg();
    printf("=== THE END ===\n\n");
    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
