/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 2 $
 * $Date: 2020/07/20 15:00 $
 * @brief    Sample code for GPIO I/O feature.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void SYS_Init(void)
{
    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Set Core Clock as 60MHz HIRC */
    CLK_SetCoreClock(FREQ_60MHZ);

    /* Enable USCI2 IP clock */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;

		/* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    GPIO_SetMode(PF, BIT1, GPIO_MODE_OUTPUT);
    GPIO_SetMode(PF, BIT2, GPIO_MODE_INPUT);

    /* Lock protected registers */
    SYS_LockReg();
}


/**
 * @brief       PortA/PortB/PortC/PortD IRQ
 *
 * @param       None
 *
 * @return      None
 *
 * @details     The PortA/PortB/PortC/PortD default IRQ, declared in startup_NM1240.s.
 */
uint32_t PAIntFlag=0, PBIntFlag=0, PCIntFlag=0, PDIntFlag=0, PEIntFlag=0, PFIntFlag=0;
void GPIO_IRQHandler(void)
{

    PAIntFlag = GPIO_GET_INT_FLAG(PA, 0xF);
    PBIntFlag = GPIO_GET_INT_FLAG(PB, 0xF);
    PCIntFlag = GPIO_GET_INT_FLAG(PC, 0xF);
    PDIntFlag = GPIO_GET_INT_FLAG(PD, 0xF);
    PEIntFlag = GPIO_GET_INT_FLAG(PE, 0xF);
    PFIntFlag = GPIO_GET_INT_FLAG(PF, 0xF);

    /* clear GPIO interrupt flag */
    GPIO_CLR_INT_FLAG(PA, PAIntFlag);
    GPIO_CLR_INT_FLAG(PB, PBIntFlag);
    GPIO_CLR_INT_FLAG(PC, PCIntFlag);
    GPIO_CLR_INT_FLAG(PD, PDIntFlag);
    GPIO_CLR_INT_FLAG(PE, PEIntFlag);
    GPIO_CLR_INT_FLAG(PF, PFIntFlag);

    /*
    printf("GPIO_IRQHandler() ...\n");
    printf("Interrupt Flag PA=0x%08X, PB=0x%08X, PC=0x%08X, PD=0x%08X\n",
        PAIntFlag, PBIntFlag, PCIntFlag, PDIntFlag);
    */
}


int main()
{
    int32_t i32Err;

    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\n\nPDID 0x%08X\n", SYS_ReadPDID());    /* Display PDID */
    printf("CPU @ %dHz\n", SystemCoreClock);        /* Display System Core Clock */

    /*
     * This sample code will use GPIO driver to control the GPIO pin direction and
     * the high/low state, and show how to use GPIO interrupts.
     */
    printf("+-----------------------------------------+\n");
    printf("| NM1240 GPIO Driver Sample Code          |\n");
    printf("+-----------------------------------------+\n");

    /*-----------------------------------------------------------------------------------------------------*/
    /* GPIO Basic Mode Test --- Use Pin Data Input/Output to control GPIO pin                              */
    /*-----------------------------------------------------------------------------------------------------*/
    printf("  >> Please connect PB.0 and PC.2 first << \n");
    printf("     Press any key to start test by using [Pin Data Input/Output Control] \n\n");
    getchar();

    /* Config multiple function to GPIO mode for PB0 and PC2 */
    SYS->GPB_MFP = (SYS->GPB_MFP & ~SYS_GPB_MFP_PB0MFP_Msk) | SYS_GPB_MFP_PB0_GPIO;
    SYS->GPC_MFP = (SYS->GPC_MFP & ~SYS_GPC_MFP_PC2MFP_Msk) | SYS_GPC_MFP_PC2_GPIO;

    GPIO_SetMode(PB, BIT0, GPIO_MODE_OUTPUT);
    GPIO_SetMode(PC, BIT2, GPIO_MODE_INPUT);

    GPIO_EnableInt(PC, 2, GPIO_INT_RISING);     /* Enable PC2 interrupt by rising edge trigger */
    NVIC_EnableIRQ(GP_IRQn);                    /* Enable GPIO NVIC */

    PAIntFlag = 0;
    PBIntFlag = 0;
    PCIntFlag = 0;
    PDIntFlag = 0;
    PEIntFlag = 0;
    PFIntFlag = 0;
    i32Err = 0;
    printf("  GPIO Output/Input test ...... \n");

    /* Use Pin Data Input/Output Control to pull specified I/O or get I/O pin status */
    PB0 = 0;                /* Output low */
    CLK_SysTickDelay(10);   /* wait for IO stable */
    if (PC2 != 0) {         /* check if the PB0 state is low */
        i32Err = 1;
    }

    PB0 = 1;                /* Output high */
    CLK_SysTickDelay(10);   /* wait for IO stable */
    if (PC2 != 1) {         /* check if the PB0 state is high */
        i32Err = 1;
    }

	 	 
    /* show the result */
    if ( i32Err ) {
        printf("  [FAIL] --- Please make sure PB.0 and PC.2 are connected. \n");
    } else {
        printf("  [OK] \n");
    }

    printf("  Check Interrupt Flag PA=0x%08X, PB=0x%08X, PC=0x%08X, PD=0x%08X, PE=0x%08X, PF=0x%08X\n",
        PAIntFlag, PBIntFlag, PCIntFlag, PDIntFlag, PEIntFlag, PFIntFlag);

    GPIO_SetMode(PB, BIT0, GPIO_MODE_INPUT);     /* Configure PB.0 to default Input mode */
    GPIO_SetMode(PC, BIT2, GPIO_MODE_INPUT);     /* Configure PC.2 to default Input mode */

    printf("=== THE END ===\n\n");
    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
