/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/08/07 14:23p $
 * @brief   NM1240 GDMA UART test
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"

volatile uint8_t g_u8EndFlagTX, g_u8EndFlagRX;

void Parameter_Init(void)
{
    g_u8EndFlagTX = 0;
    g_u8EndFlagRX = 0;
}

void GDMA0_IRQHandler (void)
{
    /* Clear TCIF */
    GDMA_CLR_TRANSFER_COMPLETE_FLAG(GDMA0);
    /* Set GDMA0_UART_RX Flag */
    g_u8EndFlagRX = 1;
}

void GDMA1_IRQHandler (void)
{
    /* Clear TCIF */
    GDMA_CLR_TRANSFER_COMPLETE_FLAG(GDMA1);
    /* Set GDMA1_UART_TX Flag */
    g_u8EndFlagTX = 1;
}

void SYS_Init(void)
{    
    /*---------------------------------------------------------------------------------------------------------*/
    /* Init System Clock                                                                                       */
    /*---------------------------------------------------------------------------------------------------------*/
    /* Enable HIRC clock */
    CLK_SetCoreClock(FREQ_60MHZ);

    /* Waiting for HIRC clock ready */
    CLK_WaitClockReady(CLK_STATUS_HIRCSTB_Msk);

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate PllClock, SystemCoreClock and CycylesPerUs automatically. */
    SystemCoreClockUpdate();
}

void UART2_Init(void)
{
    /* Enable USCI2 CLK */
    CLK_EnableModuleClock(USCI2_MODULE);
 
    /* Reset IP USCI2 */
    SYS_ResetModule(USCI2_RST);
 
    /* Open USCI2: 115200 */
    UUART_Open(UUART2, 115200);
 
    /* Set USCI_UART2 multi-function pins */
    SYS->GPF_MFP &= ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk);
    SYS->GPF_MFP |= ( SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);
 
    /* Set GPIO Mode */
    GPIO_SetMode(PF, BIT1, GPIO_MODE_OUTPUT);
    GPIO_SetMode(PF, BIT2, GPIO_MODE_INPUT);
}

int main()
{
    /*
        This sample code uses GDMA transfer and sets USCI_UART baud rate to 115200.
        Then, the user keys any 10 data.
        Check if the Enter data is equal to the Print data.
        PF1: UART2_TX
        PF2: UART2_RX
    */
    uint8_t Rx_Buffer[10], Tx_Buffer[10], i, c;
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS_Init();         /* Initial system clock       */
    UART2_Init();       /* Initial UART2              */
    SYS_LockReg();      /* Lock protected registers   */
    printf("\n\n");
    printf("+------------------------------------------------+\n");
    printf("|      NM1240 Sample Code for GDMA               |\n");
    printf("+------------------------------------------------+\n");
    printf("Initial NM1240 System Clock:\n");
    printf("CPU clock %dMHz\n",  CLK_GetCPUFreq()/1000000);
    printf("HCLK clock %dMHz\n", CLK_GetHCLKFreq()/1000000); 
    printf("EXT clock %dMHz (0MHz means no clock source)\n", CLK_GetEXTFreq()/1000000);
    printf("Reset status: 0x%08X\n", SYS->RSTSTS); 
    printf("\n");
    printf("+----------------------------------------------------------+\n");
    printf("|                    UART Configuration                    |\n");
    printf("+----------------------------------------------------------+\n");
    printf("| Transfer width : 8 bits                                  |\n");
    printf("| Baud Rate      : 115200                                  |\n");
    printf("+----------------------------------------------------------+\n");
    printf("|                    GDMA Configuration                    |\n");
    printf("+----------------------------------------------------------+\n");
    printf("| MODE           : USCI to Memory(UART)                    |\n");
    printf("| Transfer width : 8 BITS                                  |\n");
    printf("| Transfer count : Receive and Transmit 10 data            |\n");
    printf("| Burst mode     : DISABLE                                 |\n");
    printf("+----------------------------------------------------------+\n");

    /* Initial Parameter */
    Parameter_Init();

    /* Reset GDMA */
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS_ResetGDMA();
    SYS_LockReg();      /* Lock protected registers */
  
    /* Open GDMA0 for UART RX */
{
    GDMA_OPEN(GDMA0, GDMA_USCI_MODE, GDMA_TWS_8BITS, GDMA_BURST_DISABLE, SOURCE_ADDRESS_FIXED | Destination_ADDRESS_INC);
    
    /* Enable GDMA0 External Interrupt */
    NVIC_EnableIRQ(GDMA0_IRQn);
    
    /* Set transfer count */
    GDMA_SET_TRANSFER_COUNT(GDMA0, 10);
  
    /* Set destination base address */
    GDMA_SET_DST_BASE_ADDR(GDMA0, &Rx_Buffer[0]);
  
    /* Set source base address */
    GDMA_SET_SRC_BASE_ADDR(GDMA0, &UUART2->RXDAT);
}
    /* Open GDMA1 for UART TX */
{
    GDMA_OPEN(GDMA1, GDMA_USCI_MODE, GDMA_TWS_8BITS, GDMA_BURST_DISABLE, SOURCE_ADDRESS_INC | Destination_ADDRESS_FIXED);
    
    /* Enable GDMA1 External Interrupt */
    NVIC_EnableIRQ(GDMA1_IRQn);
    
    /* Set transfer count */
    GDMA_SET_TRANSFER_COUNT(GDMA1, 10);
    
    /* Set destination base address */
    GDMA_SET_DST_BASE_ADDR(GDMA1, &UUART2->TXDAT);
  
    /* Set source base address */
    GDMA_SET_SRC_BASE_ADDR(GDMA1, &Tx_Buffer[0]);
}
    /* Reset UART DMA CTL */
    UUART_ENABLE_DMA_CTL(UUART2, UUART_DMACTL_DMARST_Msk);
    /* Set UART DMA CTL */
    UUART_ENABLE_DMA_CTL(UUART2, UUART_DMACTL_DMAEN_Msk | UUART_DMACTL_RXDMAEN_Msk | UUART_DMACTL_TXDMAEN_Msk);  
  
    printf("UART receive and transfer data by GDMA\n");
    printf("Enter -> RXDAT -> Rx_Buffer -> Tx_Buffer -> TXDAT -> Print\n");
    printf("Check the input keys and output letters are the same or not?\n");
    printf("Press any 10 keys\n");
    printf("Enter:");
    
    /* Check UART RX is empty */
    while(!(UUART_IS_RX_EMPTY(UUART2)));
    
    /* Enable GDMA0 INT */
    GDMA_EnableINT(GDMA0);
    
    /* Enable GDMA0 */
    GDMA_ENABLE(GDMA0);

    for (i=0; i<10; i++){
        DEBUG_PORT->PROTSTS = (UUART_PROTSTS_RXENDIF_Msk);
        while (1) {
            if(DEBUG_PORT->PROTSTS & UUART_PROTSTS_RXENDIF_Msk) {
                DEBUG_PORT->PROTSTS = (UUART_PROTSTS_RXENDIF_Msk);
                c = DEBUG_PORT->RXDAT;
                break;
            }
        }
    printf("%c",c);
    }
    for (i=0; i<10; i++){
        Tx_Buffer[i] = Rx_Buffer[i];
    }
    printf("\nPrint:");
  
    /* Check the transmission of GDMA0_UART_RX is complete. */
    while(!g_u8EndFlagRX);
    g_u8EndFlagRX = 0;
    
    /* Disable GDMA0 INT */
    GDMA_DisableINT(GDMA0);
  
    /* Check UART TX is not full */
    while(UUART_IS_TX_FULL(UUART2));
  
    /* Enable GDMA1 INT */
    GDMA_EnableINT(GDMA1);
    
    /* Enable GDMA1 */
    GDMA_ENABLE(GDMA1);  

    /* Check the transmission of GDMA0_UART_RX is complete. */
    while(!g_u8EndFlagTX);
    g_u8EndFlagTX = 0;
    
    /* Disable GDMA1 INT */
    GDMA_DisableINT(GDMA1);
    
    printf("\nGDMA transfer finish!\n");
    while(1);  // end of while(1)
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
