/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/08/07 15:23p $
 * @brief   NM1240 GDMA SPI Slave test
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"

volatile uint8_t g_u8EndFlagTX, g_u8EndFlagRX;

void Parameter_Init(void)
{
    g_u8EndFlagTX = 0;
    g_u8EndFlagRX = 0;
}

void GDMA0_IRQHandler (void)
{
    /* Clear TCIF */
    GDMA_CLR_TRANSFER_COMPLETE_FLAG(GDMA0);
    
    /* Set GDMA0_I2C_RX Flag */
    g_u8EndFlagRX = 1;
}

void GDMA1_IRQHandler (void)
{
    /* Clear TCIF */
    GDMA_CLR_TRANSFER_COMPLETE_FLAG(GDMA1);
    
    /* Set GDMA1_I2C_TX Flag */
    g_u8EndFlagTX = 1;
}

void SYS_Init(void)
{    
    /*---------------------------------------------------------------------------------------------------------*/
    /* Init System Clock                                                                                       */
    /*---------------------------------------------------------------------------------------------------------*/
    /* Enable HIRC clock */
    CLK_SetCoreClock(FREQ_60MHZ);

    /* Waiting for HIRC clock ready */
    CLK_WaitClockReady(CLK_STATUS_HIRCSTB_Msk);

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate PllClock, SystemCoreClock and CycylesPerUs automatically. */
    SystemCoreClockUpdate();
}

void UART2_Init(void)
{
    /* Enable USCI2 CLK */
    CLK_EnableModuleClock(USCI2_MODULE);
  
    /* Reset IP USCI2 */
    SYS_ResetModule(USCI2_RST);
  
    /* Open USCI2: 115200 */
    UUART_Open(UUART2, 115200);
  
    /* Set USCI_SPI1 multi-function pins */
    SYS->GPF_MFP &= ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk);
    SYS->GPF_MFP |= ( SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);
  
    /* Set GPIO Mode */
    GPIO_SetMode(PF, BIT1, GPIO_MODE_OUTPUT);
    GPIO_SetMode(PF, BIT2, GPIO_MODE_INPUT);
}

void SPI1_Init(void)
{
    /* Enable USCI1 CLK */
    CLK_EnableModuleClock(USCI1_MODULE);
  
    /* Reset IP USCI1 */
    SYS_ResetModule(USCI1_RST);
  
    /* Open USCI1: Slave, Mode 0, 16 bits, 2MHz */
    USPI_Open(USPI1, USPI_SLAVE, USPI_MODE_0, 16, 2000000);
  
    /* Enable Auto SS */
    USPI_EnableAutoSS(USPI1, NULL, USPI_SS_ACTIVE_LOW);
  
    /* Reset SPI DMA CTL */
    USPI_ENABLE_DMA_CTL(USPI1, USPI_DMACTL_DMARST_Msk);
    /* Set SPI DMA CTL */
    USPI_ENABLE_DMA_CTL(USPI1, USPI_DMACTL_DMAEN_Msk | USPI_DMACTL_RXDMAEN_Msk | USPI_DMACTL_TXDMAEN_Msk);
  
    /* Set USCI_SPI1 multi-function pins */
    SYS->GPC_MFP &= ~(SYS_GPC_MFP_PC0MFP_Msk | SYS_GPC_MFP_PC1MFP_Msk | SYS_GPC_MFP_PC2MFP_Msk | SYS_GPC_MFP_PC3MFP_Msk);
    SYS->GPC_MFP |= (SYS_GPC_MFP_PC0_SPI1_CLK | SYS_GPC_MFP_PC1_SPI1_MISO | SYS_GPC_MFP_PC2_SPI1_MOSI | SYS_GPC_MFP_PC3_SPI1_SS);
    
    /* Set GPIO Mode */
    GPIO_SetMode(PC, BIT0, GPIO_MODE_INPUT);
    GPIO_SetMode(PC, BIT1, GPIO_MODE_OUTPUT);
    GPIO_SetMode(PC, BIT2, GPIO_MODE_INPUT);
    GPIO_SetMode(PC, BIT3, GPIO_MODE_INPUT);
}

int main()
{
    /*
        This sample code uses GDMA transfer and sets USCI_SPI bus clock to 2 MHz.
        Then, Master accesses Slave with 100 Bytes Write and 100 Bytes Read operations.
        Check if the read data is equal to the programmed data.
        PC0: SPI1_CLK
        PC1: SPI1_MISO
        PC2: SPI1_MOSI
        PC3: SPI1_SS
        PF1: UART2_TX
        PF2: UART2_RX
    */
    uint16_t Rx_Buffer[100], Tx_Buffer[100], i, error_cnt = 0;
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS_Init();         /* Initial system clock       */
    UART2_Init();       /* Initial UART2              */
    SPI1_Init();        /* Initial SPI2               */
    SYS_LockReg();      /* Lock protected registers   */
    printf("\n\n");
    printf("+------------------------------------------------+\n");
    printf("|      NM1240 Sample Code for GDMA               |\n");
    printf("+------------------------------------------------+\n");
    printf("Initial NM1240 System Clock:\n");
    printf("CPU clock %dMHz\n",  CLK_GetCPUFreq()/1000000);
    printf("HCLK clock %dMHz\n", CLK_GetHCLKFreq()/1000000);  
    printf("EXT clock %dMHz (0MHz means no clock source)\n", CLK_GetEXTFreq()/1000000);
    printf("Reset status: 0x%08X\n", SYS->RSTSTS);  
    printf("\n");
    printf("+----------------------------------------------------------+\n");
    printf("|                    SPI Configuration                     |\n");
    printf("+----------------------------------------------------------+\n");
    printf("| Mode           : Slave, Mode 0                           |\n");
    printf("| Transfer width : 16 bits                                 |\n");
    printf("| Baud Rate      : 2 MHz                                   |\n");
    printf("| Auto SS        : Enable                                  |\n");
    printf("+----------------------------------------------------------+\n");
    printf("|                    GDMA Configuration                    |\n");
    printf("+----------------------------------------------------------+\n");
    printf("| MODE           : USCI to Memory(SPI)                     |\n");
    printf("| Transfer width : 16BITS                                  |\n");
    printf("| Transfer count : 100                                     |\n");
    printf("| Burst mode     : DISABLE                                 |\n");
    printf("+----------------------------------------------------------+\n");
    printf("The slave is ready to transfer...\n");
    
    /* Set data of source array */
    for(i=0;i<100;i++) Tx_Buffer[i] = 0x5500 + i;
    /* Reset GDMA */
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS_ResetGDMA();
    SYS_LockReg();      /* Lock protected registers */
    
{
    /* Open GDMA0 for SPI RX */
    GDMA_OPEN(GDMA0, GDMA_USCI_MODE, GDMA_TWS_16BITS, GDMA_BURST_DISABLE, SOURCE_ADDRESS_FIXED | Destination_ADDRESS_INC);
    
    /* Set transfer count */
    GDMA_SET_TRANSFER_COUNT(GDMA0, 100);
    
    /* Set destination base address */
    GDMA_SET_DST_BASE_ADDR(GDMA0, &Rx_Buffer[0]);
    
    /* Set source base address */
    GDMA_SET_SRC_BASE_ADDR(GDMA0, &USPI1->RXDAT);

    /* Enable GDMA0 External Interrupt */
    NVIC_EnableIRQ(GDMA0_IRQn);
}
{
    /* Open GDMA1 for SPI TX */
    GDMA_OPEN(GDMA1, GDMA_USCI_MODE, GDMA_TWS_16BITS, GDMA_BURST_DISABLE, SOURCE_ADDRESS_INC | Destination_ADDRESS_FIXED);
    
    /* Set transfer count */
    GDMA_SET_TRANSFER_COUNT(GDMA1, 100);
    
    /* Set destination base address */
    GDMA_SET_DST_BASE_ADDR(GDMA1, &USPI1->TXDAT);
    
    /* Set source base address */
    GDMA_SET_SRC_BASE_ADDR(GDMA1, &Tx_Buffer[0]);

    /* Enable GDMA1 External Interrupt */
    NVIC_EnableIRQ(GDMA1_IRQn);
}
    /* Enable GDMA0 INT */
    GDMA_EnableINT(GDMA0);

    /* Enable GDMA0 */
    GDMA_ENABLE(GDMA0);

    /* Enable GDMA1 INT */
    GDMA_EnableINT(GDMA1);

    /* Enable GDMA1 */
    GDMA_ENABLE(GDMA1);    
    
    /* Check the transmission of GDMA0_I2C_RX is complete. */
    while(!g_u8EndFlagRX);
    g_u8EndFlagRX = 0;

    /* Disable GDMA0 INT */
    GDMA_DisableINT(GDMA0);
    
    /* Check the transmission of GDMA1_I2C_TX is complete. */
    while(!g_u8EndFlagTX);
    g_u8EndFlagTX = 0;
    
    /* Disable GDMA1 INT */
    GDMA_DisableINT(GDMA1);
    
    /* Print result */
    for(i=0;i<100;i++)
    {
        printf("\n(Tx_Buffer[%02d], Rx_Buffer[%02d]) = (0x%X, 0x%X)", i, i, Tx_Buffer[i], Rx_Buffer[i]);
        if (Rx_Buffer[i] != ((Tx_Buffer[i]&0xFF)|0xAA00))
        {
            printf(" - ERROR");
            error_cnt ++;
        }
    }
    if (error_cnt == 0){
        printf("\nGDMA memory to memory test: PASS, %d ERROR\n", error_cnt);
    }
    else{
        printf("\nGDMA memory to memory test: FAIL, %d ERROR\n", error_cnt);
    }
    printf("GDMA transfer finish!\n");
    while(1);  // end of while(1)
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
