/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/08/07 15:51p $
 * @brief   NM1240 GDMA memory to memory test
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"

void SYS_Init(void)
{    
    /*---------------------------------------------------------------------------------------------------------*/
    /* Init System Clock                                                                                       */
    /*---------------------------------------------------------------------------------------------------------*/
    /* Enable HIRC clock */
    CLK_SetCoreClock(FREQ_60MHZ);

    /* Waiting for HIRC clock ready */
    CLK_WaitClockReady(CLK_STATUS_HIRCSTB_Msk);

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate PllClock, SystemCoreClock and CycylesPerUs automatically. */
    SystemCoreClockUpdate();
}

void UART2_Init(void)
{
    /* Enable USCI2 CLK */
    CLK_EnableModuleClock(USCI2_MODULE);
  
    /* Reset IP USCI2 */
    SYS_ResetModule(USCI2_RST);
  
    /* Open USCI2: 115200 */
    UUART_Open(UUART2, 115200);
  
    /* Set USCI_SPI1 multi-function pins */
    SYS->GPF_MFP &= ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk);
    SYS->GPF_MFP |= ( SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);
  
    /* Set GPIO Mode */
    GPIO_SetMode(PF, BIT1, GPIO_MODE_OUTPUT);
    GPIO_SetMode(PF, BIT2, GPIO_MODE_INPUT);
}

int main()
{
    /*
        This sample code uses GDMA to transfer 40 data between SRAM.
        PF1: UART2_TX
        PF2: UART2_RX
    */
    uint32_t DST0[40], SRC0[40],DST1[40], SRC1[40], i, error_cnt = 0;
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS_Init();         /* Initial system clock       */
    UART2_Init();       /* Initial UART2              */
    SYS_LockReg();      /* Lock protected registers   */
    printf("\n\n");
    printf("+------------------------------------------------+\n");
    printf("|      NM1240 Sample Code for GDMA               |\n");
    printf("+------------------------------------------------+\n");
    printf("Initial NM1240 System Clock:\n");
    printf("CPU clock %dMHz\n",  CLK_GetCPUFreq()/1000000);
    printf("HCLK clock %dMHz\n", CLK_GetHCLKFreq()/1000000);  
    printf("EXT clock %dMHz (0MHz means no clock source)\n", CLK_GetEXTFreq()/1000000);
    printf("Reset status: 0x%08X\n", SYS->RSTSTS);
    printf("\n");
    printf("+----------------------------------------------------------+\n");
    printf("|                    GDMA Configuration                    |\n");
    printf("+----------------------------------------------------------+\n");
    printf("| MODE           : Memory to Memory(SOFTWARE)              |\n");
    printf("| Transfer width : 32BITS                                  |\n");
    printf("| Transfer count : 10                                      |\n");
    printf("| Burst mode     : ENABLE                                  |\n");
    printf("| SRC BASE ADDR  : Incremented                             |\n");
    printf("| DST BASE ADDR  : Incremented                             |\n");
    printf("+----------------------------------------------------------+\n");
    printf("Press any key to start\n");
    getchar();
    
    /* Set data of source array */
    for(i=0;i<40;i++) {
        SRC0[i] = 0x55550000 + i;
        SRC1[i] = 0xAAAA0000 + i;
    }
    
    /* Reset GDMA */
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS_ResetGDMA();
    SYS_LockReg();      /* Lock protected registers */
    
    /* Open GDMA0 */
    GDMA_OPEN(GDMA0, GDMA_SOFTWARE_MODE, GDMA_TWS_32BITS, GDMA_BURST_ENABLE, SOURCE_ADDRESS_INC | Destination_ADDRESS_INC);
        
    /* Set transfer count */
    GDMA_SET_TRANSFER_COUNT(GDMA0, 10);  //Transfer count must be divided 4 because burst mode is used.
    
    /* Set destination base address */
    GDMA_SET_DST_BASE_ADDR(GDMA0, &DST0[0]);
    
    /* Set source base address */
    GDMA_SET_SRC_BASE_ADDR(GDMA0, &SRC0[0]);
    
    /* Enable GDMA0 */
    GDMA_ENABLE(GDMA0);
    
    /* Open GDMA0 */
    GDMA_OPEN(GDMA1, GDMA_SOFTWARE_MODE, GDMA_TWS_32BITS, GDMA_BURST_ENABLE, SOURCE_ADDRESS_INC | Destination_ADDRESS_INC);
    
    /* Set transfer count */
    GDMA_SET_TRANSFER_COUNT(GDMA1, 10);  //Transfer count must be divided 4 because burst mode is used.
    
    /* Set destination base address */
    GDMA_SET_DST_BASE_ADDR(GDMA1, &DST1[0]);
    
    /* Set source base address */
    GDMA_SET_SRC_BASE_ADDR(GDMA1, &SRC1[0]);
    
    /* Enable GDMA0 */
    GDMA_ENABLE(GDMA1);
    
    /* Software request GDMA0 */
    GDMA_SOFTWARE_TRG(GDMA0);
    
    /* Software request GDMA1 */
    GDMA_SOFTWARE_TRG(GDMA1);
    
    /* Wait TCIF */
    while(!GDMA_GET_TRANSFER_COMPLETE_STATUS(GDMA0));
    
    /* Clear TCIF */
    GDMA_CLR_TRANSFER_COMPLETE_FLAG(GDMA0);
    
    /* Wait TCIF */
    while(!GDMA_GET_TRANSFER_COMPLETE_STATUS(GDMA1));
    
    /* Clear TCIF */
    GDMA_CLR_TRANSFER_COMPLETE_FLAG(GDMA1);
    
    /* Print result */
    for(i=0;i<40;i++)
    {
        printf("\n(SRC0[%02d], DST0[%02d]) = (0x%X, 0x%X)", i, i, SRC0[i], DST0[i]);
        if (DST0[i] != SRC0[i])
        {
            printf(" - ERROR");
            error_cnt ++;
        }
    }
    
    for(i=0;i<40;i++)
    {
        printf("\n(SRC1[%02d], DST1[%02d]) = (0x%X, 0x%X)", i, i, SRC1[i], DST1[i]);
        if (DST1[i] != SRC1[i])
        {
            printf(" - ERROR");
            error_cnt ++;
        }
    }
    
    if (error_cnt == 0){
        printf("\nGDMA memory to memory test: PASS, %d ERROR\n", error_cnt);
    }
    else{
        printf("\nGDMA memory to memory test: FAIL, %d ERROR\n", error_cnt);
    }
    while(1);  // end of while(1)
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
