/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/08/07 14:25p $
 * @brief   NM1240 GDMA I2C Master test
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"

volatile uint8_t g_u8EndFlagTX, g_u8EndFlagRX;

/* TX data structure */
typedef struct
{
    uint16_t Address;
    uint16_t data[100];
} I2C_TX_T;

/* RX data structure */
typedef struct
{
    uint16_t Address;
    uint16_t data[100];
} I2C_RX_T;

I2C_TX_T I2C_TX;
I2C_RX_T I2C_RX;

uint16_t* I2C_RX_ptr;

void Parameter_Init(void)
{
    g_u8EndFlagTX = 0;
    g_u8EndFlagRX = 0;
}

void GDMA0_IRQHandler (void)
{
    /* Disable UI2C TIMEROUT */
    UI2C_DisableTimeout(UI2C1);
    
    /* Clear TCIF */
    GDMA_CLR_TRANSFER_COMPLETE_FLAG(GDMA0);
    
    /* Set GDMA0_I2C_RX Flag */
    g_u8EndFlagRX = 1;
}

void GDMA1_IRQHandler (void)
{
    /* Disable UI2C TIMEROUT */
    UI2C_DisableTimeout(UI2C1);
    
    /* Clear TCIF */
    GDMA_CLR_TRANSFER_COMPLETE_FLAG(GDMA1);
    
    /* Set GDMA1_I2C_TX Flag */
    g_u8EndFlagTX = 1;
}

void USCI1_IRQHandler(void)
{
    volatile uint32_t u32Status;
    u32Status = UI2C_GET_PROT_STATUS(UI2C1);

    if(u32Status & UI2C_PROTIEN_ACKIEN_Msk){
        /* Clear USCI_I2C1 ACK Flag */
        UI2C_CLR_PROT_INT_FLAG(UI2C1, UI2C_PROTSTS_ACKIF_Msk);
    }
    else if(u32Status & UI2C_PROTIEN_NACKIEN_Msk){
        /* Clear USCI_I2C1 NACK Flag */
        UI2C_CLR_PROT_INT_FLAG(UI2C1, UI2C_PROTSTS_NACKIF_Msk);
    }
    else if (u32Status & UI2C_PROTSTS_TOIF_Msk) {
        /* Clear USCI_I2C1 Timeout Flag */
        UI2C_CLR_PROT_INT_FLAG(UI2C1, UI2C_PROTSTS_TOIF_Msk);
      
        /* Close I2C */
        UI2C_Close(UI2C1);
        printf("I2C time out!\n");
    }
}

void SYS_Init(void)
{    
    /*---------------------------------------------------------------------------------------------------------*/
    /* Init System Clock                                                                                       */
    /*---------------------------------------------------------------------------------------------------------*/
    /* Enable HIRC clock */
    CLK_SetCoreClock(FREQ_60MHZ);

    /* Waiting for HIRC clock ready */
    CLK_WaitClockReady(CLK_STATUS_HIRCSTB_Msk);

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate PllClock, SystemCoreClock and CycylesPerUs automatically. */
    SystemCoreClockUpdate();
}

void UART2_Init(void)
{
    /* Enable USCI2 CLK */
    CLK_EnableModuleClock(USCI2_MODULE);
  
    /* Reset IP USCI2 */
    SYS_ResetModule(USCI2_RST);
  
    /* Open USCI2: 115200 */
    UUART_Open(UUART2, 115200);
  
    /* Set USCI_SPI1 multi-function pins */
    SYS->GPF_MFP &= ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk);
    SYS->GPF_MFP |= ( SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);
  
    /* Set GPIO Mode */
    GPIO_SetMode(PF, BIT1, GPIO_MODE_OUTPUT);
    GPIO_SetMode(PF, BIT2, GPIO_MODE_INPUT);
}

void I2C1_Init(void)
{
    /* Enable USCI1 CLK */
    CLK_EnableModuleClock(USCI1_MODULE);
  
    /* Reset IP USCI1 */
    SYS_ResetModule(USCI1_RST);
  
    /* Open UI2C1: 1 MHz */
    UI2C_Open(UI2C1, 1000000);

    /* Asser ACK control. */
    UI2C_SET_CONTROL_REG(UI2C1, UI2C_CTL_AA);
  
    /* Enable TIMEROUT INT */
    UI2C_EnableInt(UI2C1, UI2C_TO_INT_MASK);
    
    /* Enable UI2C1 External Interrupt */
    NVIC_EnableIRQ(USCI1_IRQn);
  
    /* Set USCI_I2C1 multi-function pins */
    SYS->GPC_MFP &= ~(SYS_GPC_MFP_PC1MFP_Msk | SYS_GPC_MFP_PC2MFP_Msk);
    SYS->GPC_MFP |= (SYS_GPC_MFP_PC0_I2C1_SCL | SYS_GPC_MFP_PC2_I2C1_SDA);
}

void GDMA_I2C_MASTER_WRITE_DATA(uint16_t deviceaddress, uint16_t write_data_length, uint16_t* i2c_tx_ptr)
{
    /* Set device address, write command and add start control at the first transmitting data. */
    *i2c_tx_ptr = deviceaddress<<1 | 0x0 | UI2C_TXDAT_DMASTA_Msk;
  
    /* Add Stop control at the end of TX data */
    *(i2c_tx_ptr + write_data_length) |= UI2C_TXDAT_DMASTO_Msk;  
    
    /* Check I2C is idle */
    while(UI2C_GetStatus(UI2C1, UI2C_PROTSTS_ONBUSY_Msk));
    
    /* Reset UI2C DMA CTL */
    UI2C_ENABLE_DMA_CTL(UI2C1, UI2C_DMACTL_DMARST_Msk);
    /* Set UI2C DMA CTL */
    UI2C_ENABLE_DMA_CTL(UI2C1, UI2C_DMACTL_DMAEN_Msk | UI2C_DMACTL_TXDMAEN_Msk);
  
    /* Set transfer count */
    GDMA_SET_TRANSFER_COUNT(GDMA1, write_data_length + 1);
    
    /* Set destination base address */
    GDMA_SET_DST_BASE_ADDR(GDMA1, &UI2C1->TXDAT);
    
    /* Set source base address */
    GDMA_SET_SRC_BASE_ADDR(GDMA1, i2c_tx_ptr);
  
    /* Clear I2C NACK INT FLAG */
    UI2C_ClearIntFlag(UI2C1, UI2C_NACK_INT_MASK);
    
    /* Set timeout counter */
    UI2C_EnableTimeout(UI2C1, 20);
    
    /* Enable GDMA1 */
    GDMA_ENABLE(GDMA1);
}

void GDMA_I2C_MASTER_READ_DATA(uint16_t deviceaddress, uint16_t read_data_length, uint16_t* i2c_rx_ptr)
{      
    /* Set device address, read command and add start control at the first transmitting data */
    *i2c_rx_ptr = deviceaddress<<1 | 0x1 | UI2C_TXDAT_DMASTA_Msk;  

    /* Check I2C is idle */
    while(UI2C_GetStatus(UI2C1, UI2C_PROTSTS_ONBUSY_Msk));
  
    /* Reset UI2C DMA CTL */
    UI2C_ENABLE_DMA_CTL(UI2C1, UI2C_DMACTL_DMARST_Msk);
    /* Set UI2C DMA CTL */
    UI2C_ENABLE_DMA_CTL(UI2C1, UI2C_DMACTL_DMAEN_Msk | UI2C_DMACTL_TXDMAEN_Msk);
    
    /* Set GDMA1 transfer count */
    GDMA_SET_TRANSFER_COUNT(GDMA1, 1);
    
    /* Set GDMA1 destination base address */
    GDMA_SET_DST_BASE_ADDR(GDMA1, &UI2C1->TXDAT);
    
    /* Set GDMA1 source base address */
    GDMA_SET_SRC_BASE_ADDR(GDMA1, i2c_rx_ptr);
  
    /* Set GDMA0 transfer count */
    GDMA_SET_TRANSFER_COUNT(GDMA0, read_data_length);
    
    /* Point to the next data */
    i2c_rx_ptr = i2c_rx_ptr + 1;
    /* Set GDMA0 destination base address */
    GDMA_SET_DST_BASE_ADDR(GDMA0, i2c_rx_ptr);
    
    /* Set GDMA0 source base address */
    GDMA_SET_SRC_BASE_ADDR(GDMA0, &UI2C1->RXDAT);
    
    /* Clear I2C NACK INT FLAG */
    UI2C_ClearIntFlag(UI2C1, UI2C_NACK_INT_MASK);
    
    /* Enable GDMA1 */
    GDMA_ENABLE(GDMA1);
    
    /* Wait GDMA1 TCIF */
    while(!GDMA_GET_TRANSFER_COMPLETE_STATUS(GDMA1));
    
    /* Clear GDMA1 TCIF */
    GDMA_CLR_TRANSFER_COMPLETE_FLAG(GDMA1);
    
    /* Reset UI2C DMA CTL */
    UI2C_ENABLE_DMA_CTL(UI2C1, UI2C_DMACTL_DMARST_Msk);
    /* Set UI2C DMA CTL */
    UI2C_ENABLE_DMA_CTL(UI2C1, UI2C_DMACTL_DMAEN_Msk | UI2C_DMACTL_RXDMAEN_Msk | UI2C_DMACTL_NACKEN_Msk | UI2C_DMACTL_STOPEN_Msk);
    
    /* Enable TIMEROUT */
    UI2C_EnableTimeout(UI2C1, 20);
    
    /* Enable GDMA0 */
    GDMA_ENABLE(GDMA0);
}

int main()
{
    /*
        This sample code uses GDMA transfer and sets USCI_I2C bus clock to 1 MHz.
        Then, Master accesses Slave with 100 Bytes Write and 100 Bytes Read operations.
        Check if the read data is equal to the programmed data.
        PC0: I2C1_SCL
        PC2: I2C1_SDA
        PF1: UART2_TX
        PF2: UART2_RX
    */
    uint16_t i;
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS_Init();         /* Initial system clock       */
    UART2_Init();       /* Initial UART2              */
    I2C1_Init();        /* Initial I2C1               */
    SYS_LockReg();      /* Lock protected registers   */
    printf("\n\n");
    printf("+------------------------------------------------+\n");
    printf("|      NM1240 Sample Code for GDMA               |\n");
    printf("+------------------------------------------------+\n");
    printf("Initial NM1240 System Clock:\n");
    printf("CPU clock %dMHz\n",  CLK_GetCPUFreq()/1000000);
    printf("HCLK clock %dMHz\n", CLK_GetHCLKFreq()/1000000);  
    printf("EXT clock %dMHz (0MHz means no clock source)\n", CLK_GetEXTFreq()/1000000);
    printf("Reset status: 0x%08X\n", SYS->RSTSTS);  
    printf("\n");
    printf("+----------------------------------------------------------+\n");
    printf("|                    I2C Configuration                     |\n");
    printf("+----------------------------------------------------------+\n");
    printf("| Mode           : Master                                  |\n");
    printf("| Transfer width : 8 bits                                  |\n");
    printf("| Baud Rate      : 1 MHz                                   |\n");
    printf("+----------------------------------------------------------+\n");
    printf("|                    GDMA Configuration                    |\n");
    printf("+----------------------------------------------------------+\n");
    printf("| MODE           : USCI to Memory(I2C)                     |\n");
    printf("| Transfer width : 16BITS                                  |\n");
    printf("| Data           : Master write and read 100 data          |\n");
    printf("| Burst mode     : DISABLE                                 |\n");
    printf("+----------------------------------------------------------+\n");
    printf("Before starting the data transfer, make sure the slave device is ready.\n");
    printf("Press any key to start\n");
    getchar();

    /* Initial Parameter */
    Parameter_Init();
    
    /* Reset GDMA */
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS_ResetGDMA();
    SYS_LockReg();      /* Lock protected registers */
    
    /* TX data array */
    for(i=0;i<100;i++)I2C_TX.data[i] = i;
    
    /* Open GDMA0 for I2C RX */
    GDMA_OPEN(GDMA0, GDMA_USCI_MODE, GDMA_TWS_16BITS, GDMA_BURST_DISABLE, SOURCE_ADDRESS_FIXED | Destination_ADDRESS_INC);
    /* Enable GDMA0 External Interrupt */
    NVIC_EnableIRQ(GDMA0_IRQn);
    
    /* Open GDMA1 for I2C TX */
    GDMA_OPEN(GDMA1, GDMA_USCI_MODE, GDMA_TWS_16BITS, GDMA_BURST_DISABLE, SOURCE_ADDRESS_INC | Destination_ADDRESS_FIXED);
    /* Enable GDMA1 External Interrupt */
    NVIC_EnableIRQ(GDMA1_IRQn);

{
    /* Enable GDMA1 INT */
    GDMA_EnableINT(GDMA1);

    /* GDMA_I2C_MASTER_WRITE_DATA(device_address, write_data_length, i2c_tx_ptr) */
    GDMA_I2C_MASTER_WRITE_DATA(0x15, 100, &I2C_TX.Address);
    
    /* Check the transmission of GDMA1_I2C_TX is complete. */
    while(!g_u8EndFlagTX);
    g_u8EndFlagTX = 0;
    
    /* Disable GDMA1 INT */
    GDMA_DisableINT(GDMA1);
}
{
    /* Enable GDMA0 INT */
    GDMA_EnableINT(GDMA0);
    
    /* GDMA_I2C_MASTER_READ_DATA(device_address, read_data_length, i2c_rx_ptr) */
    GDMA_I2C_MASTER_READ_DATA(0x15, 100, &I2C_RX.Address);
    
    /* Check the transmission of GDMA0_I2C_RX is complete. */
    while(!g_u8EndFlagRX);
    g_u8EndFlagRX = 0;

    /* Disable GDMA0 INT */
    GDMA_DisableINT(GDMA0);
}

    /* Print result */
    for(i=0;i<100;i++)
    {
      printf("(I2C_TX.data[%02d], I2C_RX.data[%02d]) = (0x%02X, 0x%02X)\n", i, i, I2C_TX.data[i], I2C_RX.data[i]);
    }
    printf("GDMA transfer finish!\n");
    while(1);  // end of while(1)
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
