/******************************************************************************
 * @file     LDROM_main.c
 * @version  V1.00
 * $Revision: 2 $
 * $Date: 2020/07/20 11:28 $
 * @brief    This sample code includes LDROM image (fmc_ld_iap) 
 *           and APROM image (fmc_ap_main).
 *           It shows how to branch between APROM and LDROM. To run 
 *           this sample code, the boot mode must be "Boot from APROM 
 *           with IAP".
 *
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/   
#include <stdio.h>
#include "NM1240.h"
#include "fmc.h"

typedef void (FUNC_PTR)(void);

void SYS_Init(void)
{
   /* Unlock protected registers */
    SYS_UnlockReg();
	
	/* Set Core Clock as 60MHz HIRC */
   CLK_SetCoreClock(FREQ_60MHZ);
	
	  /* Enable USCI0 IP clock */
		CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;	
	
    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    GPIO_SetMode(PF, BIT1, GPIO_MODE_OUTPUT);
    GPIO_SetMode(PF, BIT2, GPIO_MODE_INPUT);

    /* Lock protected registers */
    SYS_LockReg();	
	}


#ifdef __ARMCC_VERSION
void __set_SP(uint32_t _sp)
{
		__ASM(
    "MSR MSP, r0 \n"
    "BX lr \n"			
			);
}
#endif

uint32_t UUART_Open(UUART_T* uart, uint32_t u32baudrate)
{
		int32_t multiple; 
	
		uart->CTL = 0x02;	
	
		multiple = 60000000 / u32baudrate;	/* Fsys_clk = 2 * _HXT*/
		multiple = multiple /5;						/* Set DSCNT = 4 --> divide by (4+1) */
		uart-> BRGEN = (multiple << 16) | (0x04 << 10);
	
		uart-> DATIN0 = 0x10;							/* input falling edge activate */
	
		uart->LINECTL = 0x08<<8 | 0x01;		/* 8-N-1 bit, LSB first */
	
		/* Uart Protocol setting */ 
		uart->PROTCTL = UUART_PROTCTL_PROTEN_Msk;
	
    return 0;
}
void SendChar_ToUART(int ch)
{

    while (UUART2->BUFSTS & UUART_BUFSTS_TXFULL_Msk);
    UUART2->TXDAT = ch;

    if (ch == '\n')
    {
				while (UUART2->BUFSTS & UUART_BUFSTS_TXFULL_Msk);
				UUART2->TXDAT = '\r';
    }
}


void print_msg(char *str)
{
	for ( ; *str ; str++)
        SendChar_ToUART(*str);
}

int main()
{
    FUNC_PTR    *func;

    SYS_Init();

    /* Unlock protected registers */
    SYS_UnlockReg();
	
    /* Init USCI UART2 to 115200-8n1 for print message */
		UUART_Open(UUART2, 115200);	
	
    /* Enable FMC ISP function */
    FMC_Open();
    
    print_msg("\n\n");
    print_msg("NM1240 FMC IAP Sample Code [LDROM code]\n");
		
    print_msg("\n\nPress any key to branch to APROM...\n");		
		
		while(UUART2->BUFSTS & UUART_BUFSTS_RXEMPTY_Msk)   /* Check RX empty => failed */
	  ;
    print_msg("\n\nChange VECMAP and branch to APROM...\n");
	
		while((UUART2->BUFSTS & UUART_BUFSTS_TXEMPTY_Msk) == 0)   /* Wait Tx empty */
		;

    func = (FUNC_PTR *) FMC_Read(FMC_APROM_BASE+4);
    __set_SP( FMC_Read(FMC_APROM_BASE) );

    /*  NOTE!
     *     Before change VECMAP, user MUST disable all interrupts.
     */
    FMC_SetVectorPageAddr(FMC_APROM_BASE);
    
    func();
    
    while (1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
