/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 2 $
 * $Date: 2020/07/19 14:30 $
 * @brief    Show FMC CRC32 calculating capability.
 *
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>

#include "NM1240.h"
#include "fmc.h"

void SYS_Init(void)
{
    /* Unlock protected registers */
    SYS_UnlockReg();
    
    /* Set Core Clock as 60MHz HIRC */
    CLK_SetCoreClock(FREQ_60MHZ);

    /* Enable USCI0 IP clock */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;   
		
		 /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
		GPIO_SetMode(PF, BIT1, GPIO_MODE_OUTPUT);
    GPIO_SetMode(PF, BIT2, GPIO_MODE_INPUT);
    
    /* Lock protected registers */
    SYS_LockReg();  
}

int main()
{
    int         ret;                   /* return value */
    uint32_t    u32Data, u32ChkSum;    /* temporary data */

    /* Unlock protected registers */
    SYS_UnlockReg();

    SYS_Init();

    /* Lock protected registers */
    SYS_LockReg();
    
    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200); 
    
    /*---------------------------------------------------------------------------------------------------------*/
    /* SAMPLE CODE                                                                                             */
    /*---------------------------------------------------------------------------------------------------------*/

    printf("+--------------------------------------------+\n");
    printf("|        NM1240 FMC_CRC32 Sample Code        |\n");
    printf("+--------------------------------------------+\n");

    SYS_UnlockReg();                   /* Unlock protected registers */

    FMC_Open();                        /* Enable FMC ISP function */

    u32Data = FMC_ReadCID();           /* Read company ID. Should be 0xDA. */
    printf("  Company ID ............................ [0x%08x]\n", u32Data);

    u32Data = FMC_ReadPID();           /* Read product ID. */
    printf("  Product ID ............................ [0x%08x]\n", u32Data);

    /* Read User Configuration CONFIG0 */
    printf("  User Config 0 ......................... [0x%08x]\n", FMC_Read(FMC_CONFIG_BASE));
    /* Read User Configuration CONFIG1 */
    printf("  User Config 1 ......................... [0x%08x]\n", FMC_Read(FMC_CONFIG_BASE+4));

    /* Read Data Flash base address */
    printf("  Data Flash Base Address ............... [0x%08x]\n", FMC_ReadDataFlashBaseAddr());

    printf("\nLDROM (0x%lx ~ 0x%lx) CRC32 checksum =>  ", FMC_LDROM_BASE, FMC_LDROM_BASE+FMC_LDROM_SIZE);

    /* 
     *  Request FMC hardware to run CRC32 calculation on flash range from FMC_LDROM_BASE and
     *  length is FMC_LDROM_SIZE. The CRC32 calculation result will be put in u32ChkSum.
     */
    ret = FMC_GetCRC32Sum(FMC_LDROM_BASE, FMC_LDROM_SIZE, &u32ChkSum);
    if (ret < 0)  {
        printf("Failed on calculating LDROM CRC32 checksum!\n");
        goto lexit;                    /* failed */
    }
    printf("0x%x\n", u32ChkSum);       /* print out LDROM CRC32 check sum value */

    printf("\nSPROM (0x%lx ~ 0x%lx) CRC32 checksum =>  ", FMC_SPROM0_BASE, FMC_SPROM0_BASE+FMC_SPROM_SIZE);
    
    /* 
     *  Request FMC hardware to run CRC32 calculation on flash range from FMC_SPROM_BASE and
     *  length is FMC_SPROM_SIZE. The CRC32 calculation result will be put in u32ChkSum.
     */
    ret = FMC_GetCRC32Sum(FMC_SPROM0_BASE, FMC_SPROM_SIZE, &u32ChkSum);
    if (ret < 0)  {
        printf("Failed on calculating SPROM CRC32 checksum!\n");
        goto lexit;     
    }
    printf("0x%x\n", u32ChkSum);       /* print out SPROM CRC32 check sum value */

    printf("\nAPROM (0x%lx ~ 0x%lx) CRC32 checksum =>  ", FMC_APROM_BASE, FMC_APROM_END);

    /* 
     *  Request FMC hardware to run CRC32 calculation on flash range from FMC_APROM_BASE and
     *  length is FMC_APROM_END. The CRC32 calculation result will be put in u32ChkSum.
     */
    ret = FMC_GetCRC32Sum(FMC_APROM_BASE, FMC_APROM_END - FMC_APROM_BASE, &u32ChkSum);
    if (ret < 0)  {
        printf("Failed on calculating APROM CRC32 checksum!\n");
        goto lexit;     
    }
    printf("0x%x\n", u32ChkSum);       /* print out APROM CRC32 check sum value */

    printf("\nFMC CRC32 checksum test done.\n");

lexit:
    FMC_Close();                       /* Disable FMC ISP function */
    SYS_LockReg();                     /* Lock protected registers */

    while (1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
