/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/07/24 11:48 $
 * @brief    Demonstrate the EPWM double buffer feature.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


uint32_t duty30, duty60;
void EPWM_IRQHandler(void)
{
    static int toggle = 0;  /* First two already fill into EPWM, so start from 30% */

    /* Update EPWM channel 0 duty */
    if(toggle == 0) {
        EPWM_SET_CMR(EPWM, 0, duty30);
    } else {
        EPWM_SET_CMR(EPWM, 0, duty60);
    }
    toggle ^= 1;
    /* Clear channel 0 period interrupt flag */
    EPWM_ClearPeriodIntFlag(EPWM, 0);
}

void SYS_Init(void)
{
    /* Unlock protected registers */
    SYS_UnlockReg();
    
    /* Enable 60MHz HIRC */
    CLK_SetCoreClock(FREQ_60MHZ);

    /* Enable IP clock */
    CLK_EnableModuleClock(USCI2_MODULE);
    CLK_EnableModuleClock(EPWM_MODULE);

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    GPIO_SetMode(PF, BIT1, GPIO_MODE_OUTPUT);
    GPIO_SetMode(PF, BIT2, GPIO_MODE_INPUT);

    /* Set GPA multi-function pins for EPWM Channel0 */
    SYS->GPA_MFP = (SYS->GPA_MFP & (~SYS_GPA_MFP_PA0MFP_Msk));
    SYS->GPA_MFP |= SYS_GPA_MFP_PA0_EPWM_CH0;

    /* Set GPA0 as output mode */
    GPIO_SetMode(PA, BIT0, GPIO_MODE_OUTPUT);

    /* Lock protected registers */
    SYS_LockReg();
}


int main()
{
    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\nThis sample code will output EPWM channel 0 to with different\n");
    printf("frequency and duty. The CMP load new value in period interrupt \n");
    printf("and it will take effect from next cycle\n");
    
    /* EPWM channel 0 wave form of this sample changed between 30% and 60% duty ratio */
    EPWM_ConfigOutputChannel(EPWM, 0, 1000, 30);

    /* Save 30% duty setting */
    duty30 = EPWM->CMPDAT[0];
    /* Calculate 60% duty setting. */
    duty60 = duty30 * 2;

    /* Enable output of all PWM channel 0 */
    EPWM_EnableOutput(EPWM, 1);

    /* Enable EPWM channel 0 period interrupt */
    EPWM_EnablePeriodInt(EPWM, 0, 0);
    NVIC_EnableIRQ(EPWM_IRQn);

    /* Start */
    EPWM_Start(EPWM, 0x1);

    /* Fill second duty setting immediately after EPWM start */
    EPWM_SET_CMR(EPWM, 0, duty60);

    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
