/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/07/24 11:46 $
 * @brief    Demonstrate the EPWM dead-zone feature.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void EPWM_IRQHandler(void)
{
    static uint32_t cnt;
    static uint32_t out;

    /* Channel 0 frequency is 100Hz, every 1 second enter this IRQ handler 100 times. */
    if(++cnt == 100) {
        if(out)
        {
            SYS->GPA_MFP |= (SYS_GPA_MFP_PA0_EPWM_CH0 |\
                                SYS_GPA_MFP_PA1_EPWM_CH1 |\
                                SYS_GPA_MFP_PA2_EPWM_CH2 |\
                                SYS_GPA_MFP_PA3_EPWM_CH3 |\
                                SYS_GPA_MFP_PA4_EPWM_CH4 |\
                                SYS_GPA_MFP_PA5_EPWM_CH5);
        }
        else
        {
            SYS->GPA_MFP &= ~(SYS_GPA_MFP_PA0MFP_Msk |\
                                SYS_GPA_MFP_PA1MFP_Msk |\
                                SYS_GPA_MFP_PA2MFP_Msk |\
                                SYS_GPA_MFP_PA3MFP_Msk |\
                                SYS_GPA_MFP_PA4MFP_Msk |\
                                SYS_GPA_MFP_PA5MFP_Msk);
        }
            EPWM_DisableOutput(EPWM, 0x3F);
        
        out ^= 1;
        cnt = 0;
    }

    /* Get channel 0 period interrupt flag */
    EPWM_GetDutyIntFlag(EPWM, 0);    
    
    /* Clear channel 0 period interrupt flag */
    EPWM_ClearDutyIntFlag(EPWM, 0);
    
    /* Clear channel 0 period interrupt flag */
    EPWM_ClearPeriodIntFlag(EPWM, 0);
}

void SYS_Init(void)
{
    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Enable 60MHz HIRC */
    CLK_SetCoreClock(FREQ_60MHZ);

    /* Enable IP clock */
    CLK_EnableModuleClock(USCI2_MODULE);
    CLK_EnableModuleClock(EPWM_MODULE);

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    GPIO_SetMode(PF, BIT1, GPIO_MODE_OUTPUT);
    GPIO_SetMode(PF, BIT2, GPIO_MODE_INPUT);

    /* Set GPA multi-function pins for EPWM Channel0~5 */
    SYS->GPA_MFP = (SYS->GPA_MFP & (~SYS_GPA_MFP_PA0MFP_Msk));
    SYS->GPA_MFP |= SYS_GPA_MFP_PA0_EPWM_CH0;
    SYS->GPA_MFP = (SYS->GPA_MFP & (~SYS_GPA_MFP_PA1MFP_Msk));
    SYS->GPA_MFP |= SYS_GPA_MFP_PA1_EPWM_CH1;
    SYS->GPA_MFP = (SYS->GPA_MFP & (~SYS_GPA_MFP_PA2MFP_Msk));
    SYS->GPA_MFP |= SYS_GPA_MFP_PA2_EPWM_CH2;
    SYS->GPA_MFP = (SYS->GPA_MFP & (~SYS_GPA_MFP_PA3MFP_Msk));
    SYS->GPA_MFP |= SYS_GPA_MFP_PA3_EPWM_CH3;
    SYS->GPA_MFP = (SYS->GPA_MFP & (~SYS_GPA_MFP_PA4MFP_Msk));
    SYS->GPA_MFP |= SYS_GPA_MFP_PA4_EPWM_CH4;
    SYS->GPA_MFP = (SYS->GPA_MFP & (~SYS_GPA_MFP_PA5MFP_Msk));
    SYS->GPA_MFP |= SYS_GPA_MFP_PA5_EPWM_CH5;

    /* Set GPA0~5 as output mode */
    GPIO_SetMode(PA, 0x3F, GPIO_MODE_OUTPUT);

    /* Lock protected registers */
    SYS_LockReg();
}

int main()
{
    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\nThis sample code will output EPWM channel to with different\n");
    printf("duty, enable dead zone function of all EPWM pairs.\n");
    printf("And also enable/disable EPWM output every 1 second.\n");
    
    /* EPWM0 frequency is 100Hz, duty 30% */
    EPWM_ConfigOutputChannel(EPWM, 0, 100, 30);
    EPWM_EnableDeadZone(EPWM, 0, 150);

    /* EPWM2 frequency is 100Hz, duty 50% */
    EPWM_ConfigOutputChannel(EPWM, 2, 100, 50);
    EPWM_EnableDeadZone(EPWM, 2, 200);

    /* EPWM4 frequency is 100Hz, duty 70% */
    EPWM_ConfigOutputChannel(EPWM, 4, 100, 70);
    EPWM_EnableDeadZone(EPWM, 4, 100);

    /* Enable complementary mode */
    EPWM_ENABLE_COMPLEMENTARY_MODE(EPWM);

    /* Enable output of all EPWM channels */
    EPWM_EnableOutput(EPWM, 0x3F);

    /* Enable EPWM channel 0 period interrupt, use channel 0 to measure time. */
    EPWM_EnableDutyInt(EPWM, 0, EPWM_DUTY_INT_UP_COUNT_MATCH_CMP); 
    EPWM_EnablePeriodInt(EPWM, 0, 0); 
    
    NVIC_EnableIRQ(EPWM_IRQn);

    /* Start */
    EPWM_Start(EPWM, 0x3F);

    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
