/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/07/24 11:44 $
 * @brief    Demonstrate the BDC PWM control sample.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"

void HBridge_Driver(uint32_t Duty, uint8_t State);

void EPWM_IRQHandler(void)
{
    /* Clear channel 0 period interrupt flag */
    EPWM_ClearPeriodIntFlag(EPWM, 0);
}

void SYS_Init(void)
{
    /* Unlock protected registers */
    SYS_UnlockReg();
    
    /* Enable 60MHz HIRC */
    CLK_SetCoreClock(FREQ_60MHZ);

    /* Enable IP clock */
    CLK_EnableModuleClock(USCI2_MODULE);
    CLK_EnableModuleClock(EPWM_MODULE);

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    GPIO_SetMode(PF, BIT1, GPIO_MODE_OUTPUT);
    GPIO_SetMode(PF, BIT2, GPIO_MODE_INPUT);

    /* Set GPA multi-function pins for EPWM Channel0 */
    SYS->GPA_MFP = (SYS->GPA_MFP & ~(SYS_GPA_MFP_PA0MFP_Msk|SYS_GPA_MFP_PA1MFP_Msk|SYS_GPA_MFP_PA2MFP_Msk|SYS_GPA_MFP_PA3MFP_Msk));
    SYS->GPA_MFP |= (SYS_GPA_MFP_PA0_EPWM_CH0|SYS_GPA_MFP_PA1_EPWM_CH1|SYS_GPA_MFP_PA2_EPWM_CH2|SYS_GPA_MFP_PA3_EPWM_CH3);

    /* Set GPA0 as output mode */
    GPIO_SetMode(PA, BIT0, GPIO_MODE_OUTPUT);

    /* Lock protected registers */
    SYS_LockReg();
}

int main()
{
    uint8_t c;
    
    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\nThis sample code will output EPWM driving mode of BDC with 4 case.\n");
    printf("The 4 modes are Coast, Forward, Backward and Brake.\n");
    
    /* 777EPWM channel 0 ~ 4 wave form of this sample with 50% duty and 1000Hz. \
     * If mode is center aligned mode, u32Frequency is need multiply by 2. */
    EPWM_ConfigOutputChannel(EPWM, 0, 2*1000, 50);
    EPWM_ConfigOutputChannel(EPWM, 1, 2*1000, 50);
    EPWM_ConfigOutputChannel(EPWM, 2, 2*1000, 50);
    EPWM_ConfigOutputChannel(EPWM, 3, 2*1000, 50);
        
    /* Set EPWM type to center aligned mode */
    EPWM_SET_ALIGNED_TYPE(EPWM, 0, EPWM_CENTER_ALIGNED);
    
    /* Set EPWM complementary mode */
    EPWM_ENABLE_COMPLEMENTARY_MODE(EPWM);
    
    /* Set EPWM dead zone */
    EPWM_EnableDeadZone(EPWM, 0, 50);
    EPWM_EnableDeadZone(EPWM, 2, 50);
    
    /* Disable phase change trigger */
    EPWM_PHCHG_TRGSEL(EPWM, EPWM_TRGSEL_DISABLE);

    /* Default PA0 ~ PA3 output low */
    EPWM_MASK_OUTPUT(EPWM, 0x3F, 0x00);
            
    /* Enable EPWM period interrupt */
    EPWM_EnablePeriodInt(EPWM, NULL, NULL);
    
    /* Interrupt enable */
    NVIC_EnableIRQ(EPWM_IRQn);
    NVIC_SetPriority(EPWM_IRQn, 0);

    /* Start */
    EPWM_Start(EPWM, 0x0F);
    
    printf("Press key to select BCD mode.\n");
    printf(" 0: Coast\n");
    printf(" 1: Forward\n");
    printf(" 2: Backward\n");
    printf(" 3: Brake\n");
    
    while(1) {
        c = getchar();       
        c = c - '0';
        
        if(c < 4) {
            printf("Current select: %d\r",c);
            HBridge_Driver(50, c);  
        }
    }
}

#define EPWM_SET_DUTY(epwm, u32ChannelNum, duty) { (epwm)->CMPDAT[(u32ChannelNum)] = duty * ((epwm)->PERIOD + 1) / 100; }

void HBridge_Driver(uint32_t Duty, uint8_t State)
{
    switch (State)
    {
        /* Coast mode, PA0 ~ PA3 = low */
        case 0:
            EPWM_MASK_OUTPUT(EPWM, 0x3F, 0x00);
            EPWM_SET_DUTY(EPWM, 0, 0);
            EPWM_SET_DUTY(EPWM, 1, 0);
            EPWM_SET_DUTY(EPWM, 2, 0);
            EPWM_SET_DUTY(EPWM, 3, 0);
            break;
        
        /* Forward mode, PA0 = EPWM0, PA1 = EPWM1, PA2 = low, PA3 = high */
        case 1:
            EPWM_MASK_OUTPUT(EPWM, 0x0C, 0x08);        
            EPWM_SET_DUTY(EPWM, 0, Duty);
            EPWM_SET_DUTY(EPWM, 1, Duty);
            break;
        
        /* Backward mode, PA0 = low, PA1 = high, PA2 = EPWM2, PA3 = EPWM3 */
        case 2:
            EPWM_MASK_OUTPUT(EPWM, 0x03, 0x02);
            EPWM_SET_DUTY(EPWM, 2, Duty);
            EPWM_SET_DUTY(EPWM, 3, Duty);
            break;
        
        /* Brake mode, PA0 and PA2 = low, PA1 and PA3 = high */
        case 3:
            EPWM_MASK_OUTPUT(EPWM, 0x0F, 0x0A);
            EPWM_SET_DUTY(EPWM, 0, 0);
            EPWM_SET_DUTY(EPWM, 1, 0);
            EPWM_SET_DUTY(EPWM, 2, 0);
            EPWM_SET_DUTY(EPWM, 3, 0);
            break;
        
        default:
            break;
    }
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
