/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/07/24 11:32 $
 * @brief    Demonstrate the BPWM dead-zone feature.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void BPWM0_IRQHandler(void)
{
    static uint32_t cnt;
    static uint32_t out;

    /* Channel 0 frequency is 100Hz, every 1 second enter this IRQ handler 100 times. */
    if(++cnt == 100) {
        if(out)
        {
            SYS->GPC_MFP |= SYS_GPC_MFP_PC0_BPWM_CH0;
            SYS->GPB_MFP |= SYS_GPB_MFP_PB2_BPWM_CH1;        
        }
        else
        {
            SYS->GPC_MFP = (SYS->GPC_MFP & (~SYS_GPC_MFP_PC0MFP_Msk));
            SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB2MFP_Msk));           
        }
        out ^= 1;
        cnt = 0;
    }
          
    /* Get channel 0 period interrupt flag */
    BPWM_GetDutyIntFlag(BPWM, 0);
    
    /* Clear channel 0 period interrupt flag */
    BPWM_ClearDutyIntFlag(BPWM, 0);
}

void SYS_Init(void)
{
    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Enable 60MHz HIRC */
    CLK_SetCoreClock(FREQ_60MHZ);

    /* Enable IP clock */
    CLK_EnableModuleClock(USCI2_MODULE);
    CLK_EnableModuleClock(BPWM_MODULE);

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    GPIO_SetMode(PF, BIT1, GPIO_MODE_OUTPUT);
    GPIO_SetMode(PF, BIT2, GPIO_MODE_INPUT);

    /* Set GPC0 multi-function pins for BPWM Channel0 */
    SYS->GPC_MFP = (SYS->GPC_MFP & (~SYS_GPC_MFP_PC0MFP_Msk));
    SYS->GPC_MFP |= SYS_GPC_MFP_PC0_BPWM_CH0;
    /* Set GPB2 multi-function pins for BPWM Channel1 */
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB2MFP_Msk));
    SYS->GPB_MFP |= SYS_GPB_MFP_PB2_BPWM_CH1;

    /* Set GPC0, GPB2 as output mode */
    GPIO_SetMode(PC, BIT0, GPIO_MODE_OUTPUT);
    GPIO_SetMode(PB, BIT2, GPIO_MODE_OUTPUT);

    /* Lock protected registers */
    SYS_LockReg();
}


int main()
{
    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\nThis sample code will output BPWM channel 0 to with different\n");
    printf("frequency and duty, enable dead zone function of all BPWM pairs.\n");
    printf("And also enable/disable BPWM output every 1 second.\n");
    
    /* BPWM0 frequency is 100Hz, duty 30% */
    BPWM_ConfigOutputChannel(BPWM, 0, 100, 30);
    BPWM_EnableDeadZone(BPWM, 0, 100);

    /* Enable BPWM channel 0 period interrupt, use channel 0 to measure time. */
    BPWM_EnableDutyInt(BPWM, 0, 0);
    NVIC_EnableIRQ(BPWM0_IRQn);

    /* Start */
    BPWM_Start(BPWM, 0x3);
    
    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
