/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 0 $
 * $Date: 2020/07/16  $
 * @brief     Sample code for ADC Timer Trigger feature.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void SYS_Init(void)
{
    /* Unlock protected registers */
    SYS_UnlockReg();
    
    /* Enable 48/60MHz HIRC */
    CLK_SetCoreClock(FREQ_60MHZ);

    /* Enable USCI2 IP clock */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Enable IP clock */
    CLK_EnableModuleClock(ADC_MODULE);
    SYS_ResetModule(ADC_RST);
    CLK_EnableModuleClock(TMR0_MODULE);

    /* Select IP clock source */

    /*---------------------------------------------------------------------------------------------------------*/
    /* Init I/O Multi-function                                                                                 */
    /*---------------------------------------------------------------------------------------------------------*/

    /* Lock protected registers */
    SYS_LockReg();
}

uint32_t  reg_adc0_0,reg_adc0_1, reg_adc1_0,reg_adc1_1;
void ADC0_IRQHandler(void)
{
    if(ADC_GET_INT_FLAG(ADC, ADC_STATUS_ADC0IF_Msk))
    {
        ADC_CLR_INT_FLAG(ADC, ADC_STATUS_ADC0IF_Msk);   /* clear ADC0 interrupt flag */
		reg_adc0_0 = ADC->VALSTS&BIT0;
		reg_adc0_1 = ADC->VALSTS&BIT1;
		reg_adc1_0 = ADC->VALSTS&BIT16;
		reg_adc1_1 = ADC->VALSTS&BIT17;		
       
		/*Read ADC Data and Status bit*/
		printf("Get ADC0 data0 = %4d (valid = %s)\n",ADC0_GET_CONV_DATA(ADC0_ADC0_P0_DAT),(reg_adc0_0)? "Yes" : "No");
		printf("Get ADC0 data1 = %4d (valid = %s)\n",ADC0_GET_CONV_DATA(ADC0_ADC0_P1_DAT),(reg_adc0_1)? "Yes" : "No");       
		printf("Get ADC1 data0 = %4d (valid = %s)\n",ADC1_GET_CONV_DATA(ADC1_ADC1_P0_DAT),(reg_adc1_0)? "Yes" : "No"); 
		printf("Get ADC1 data1 = %4d (valid = %s)\n",ADC1_GET_CONV_DATA(ADC1_ADC1_P1_DAT),(reg_adc1_1)? "Yes" : "No");		
    }
}


void TMR0_IRQHandler(void)
{
    /* clear timer interrupt flag */
    TIMER_ClearIntFlag(TIMER0);
    printf("Timer0 expired !!\n");
}


int main()
{
    char ch;

    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\n\nPDID 0x%08X\n", SYS_ReadPDID());    /* Display PDID */
    printf("CPU @ %dHz\n", SystemCoreClock);        /* Display System Core Clock */

    /*
     * This sample code will configure Timer0 to trigger ADC0_CH0 periodically
     * and print conversion results.
     */
    printf("+-------------------------------------------+\n");
    printf("| NM1240 ADC for Timer Trigger Sample Code |\n");
    printf("+-------------------------------------------+\n");

    /* Enable ADC clock */
    CLK_EnableModuleClock(ADC_MODULE);


    /* Reset IP */
    SYS_ResetModule(ADC_RST);

	/*ADC0CS_ADC0_P0 = PB0*/
    printf("Set GPB0 pin as ADC0_CH0(ADC0_P0) input pin.\n");
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB0MFP_Msk)) | SYS_GPB_MFP_PB0_ADC0_P0;
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE0_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE0_Pos);
    PB->DINOFF |= GPIO_DINOFF_DINOFF0_Msk;
  
	/*ADC0CS_ADC0_P1 = PB1*/
    printf("Set GPB1 pin as ADC0_CH1(ADC0_P1) input pin.\n");
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB1MFP_Msk)) | SYS_GPB_MFP_PB1_ADC0_P1;
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE1_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE1_Pos);
    PB->DINOFF |= GPIO_DINOFF_DINOFF1_Msk;	
	
	/*ADC1CS_ADC1_P0 = PB4*/
    printf("Set GPB4 pin as ADC1_CH0(ADC1_P0) input pin.\n");
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB4MFP_Msk)) | SYS_GPB_MFP_PB4_ADC1_P0;
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE4_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE4_Pos);
    PB->DINOFF |= GPIO_DINOFF_DINOFF4_Msk;

	/*ADC1CS_ADC1_P1 = PD2*/
    printf("Set GPD2 pin as ADC1_CH1(ADC1_P1) input pin.\n");
    SYS->GPD_MFP = (SYS->GPD_MFP & (~SYS_GPD_MFP_PD2MFP_Msk)) | SYS_GPD_MFP_PD2_ADC1_P1;
    PD->MODE = (PD->MODE & ~GPIO_MODE_MODE2_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE2_Pos);
    PD->DINOFF |= GPIO_DINOFF_DINOFF2_Msk;

    /*--- Initial Timer to trigger ADC */
    printf("Set Timer0 to periodic 1Hz. It will trigger ADC 1 times per second.\n");
    TIMER_Open(TIMER0, TIMER_PERIODIC_MODE, 1);
    TIMER_EnableInt(TIMER0);
    NVIC_EnableIRQ(TMR0_IRQn);

    /*--- Initial ADC */
    /* Enable ADC */
    ADC_Open(ADC, NULL);

    /* Configure ADC0: ADC0_CH0,ADC0_CH1, Timer0 trigger */
    ADC_ADC0ConfigChannel(ADC, ADC0CS_ADC0_P0|ADC0CS_ADC0_P1, ADC0_TIMER0_TRIGGER);
    /* Configure ADC1: ADC1_CH0,ADC1_CH1, Timer0 trigger */
    ADC_ADC1ConfigChannel(ADC, ADC1CS_ADC1_P0|ADC1CS_ADC1_P1, ADC1_TIMER0_TRIGGER);	
	
    /* Configure ADC conversion mode to Simultaneous Simple Mode */
    ADC_SET_SIMULTANEOUS_MODE(ADC);
		ADC_SetTriggerDelayTime(ADC,ADC_ADC1,100);
    /* Configure ADC sample time to 6 ADC clocks */
    ADC_SetSampleCnt(ADC, NULL, 5);

    ADC_ENABLE_INT(ADC, ADC_CTL_ADC0IEN_Msk);
    NVIC_EnableIRQ(ADC0_IRQn);

    /* Begin to do ADC conversion. */
    ch = 0;
    TIMER_Start(TIMER0);
    while (ch != 'q')
    {
        printf("Press 'q' to quit.\n");
        ch = getchar();
    }

    NVIC_DisableIRQ(TMR0_IRQn);
    TIMER_Close(TIMER0);
    CLK_DisableModuleClock(TMR0_MODULE);

    NVIC_DisableIRQ(ADC0_IRQn);
    ADC_Close(ADC);
    CLK_DisableModuleClock(ADC_MODULE);
    printf("=== THE END ===\n\n");
    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
