/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 0 $
 * $Date: 2020/07/16 14:50 $
 * @brief    Sample code for ADC Simultaneous Simple mode.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void SYS_Init(void)
{
    /* Unlock protected registers */
    SYS_UnlockReg();
    
    /* Enable 48/60MHz HIRC */
    CLK_SetCoreClock(FREQ_60MHZ);

    /* Enable USCI2 IP clock */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Enable IP clock */
    CLK_EnableModuleClock(ADC_MODULE);
    SYS_ResetModule(ADC_RST);

    /* Select IP clock source */

    /*---------------------------------------------------------------------------------------------------------*/
    /* Init I/O Multi-function                                                                                 */
    /*---------------------------------------------------------------------------------------------------------*/

    /* Lock protected registers */
    SYS_LockReg();
}


int main()
{
    char ch;
	uint32_t reg_adc0_1,reg_adc0_3, reg_adc1_3,reg_adc1_7;
    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\n\nPDID 0x%08X\n", SYS_ReadPDID());    /* Display PDID */
    printf("CPU @ %dHz\n", SystemCoreClock);        /* Display System Core Clock */

    /*
     * This sample code will convert ADC0 channel 0 and ADC1 channel 0 in Independent Simple mode
     * and print conversion results.
     */
    printf("+--------------------------------------------------+\n");
    printf("| NM1240 ADC Simultaneous Simple Mode Sample Code |\n");
    printf("+--------------------------------------------------+\n");

    /* Enable ADC clock */
    CLK_EnableModuleClock(ADC_MODULE);

 
    /* Reset IP */
    SYS_ResetModule(ADC_RST);

	/*ADC0CS_ADC0_P1 = PB1*/
    printf("Set GPB1 pin as ADC0_CH1(ADC0_P1) input pin.\n");
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB1MFP_Msk)) | SYS_GPB_MFP_PB1_ADC0_P1;
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE1_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE1_Pos);
    PB->DINOFF |= GPIO_DINOFF_DINOFF1_Msk;
	
	/*ADC0CS_ADC0_P3 = PC0*/
    printf("Set GPC0 pin as ADC0_CH3(ADC0_P3) input pin.\n");
    SYS->GPC_MFP = (SYS->GPC_MFP & (~SYS_GPC_MFP_PC0MFP_Msk)) | SYS_GPC_MFP_PC0_ADC0_P3;
    PC->MODE = (PC->MODE & ~GPIO_MODE_MODE0_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE0_Pos);
    PC->DINOFF |= GPIO_DINOFF_DINOFF1_Msk;	
	
	/*ADC1CS_ADC0_P0 = PB0*/
    printf("Set GPB0 pin as ADC1_CH3(ADC1CS_ADC0_P0) input pin.\n");
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB0MFP_Msk)) | SYS_GPB_MFP_PB0_ADC0_P0;
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE0_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE0_Pos);
    PB->DINOFF |= GPIO_DINOFF_DINOFF0_Msk;	
	

	
    /* Configure ADC0: ADC0_CH1,ADC0_CH3, software trigger */
    ADC_ADC0ConfigChannel(ADC, ADC0CS_ADC0_P1 | ADC0CS_ADC0_P3, ADC0_SOFTWARE_TRIGGER);	
	
    /* Configure ADC1: ADC1_CH3,ADC1_CH7, software trigger */
    ADC_ADC1ConfigChannel(ADC, ADC1CS_ADC0_P0 | ADC1CS_VSS, ADC1_SOFTWARE_TRIGGER);	
	
		ADC_SET_SIMULTANEOUS_MODE(ADC);

    /* Configure ADC sample time to 6 ADC clocks */
    ADC_SetSampleCnt(ADC, NULL, 5);
    /* Enable ADC */
    ADC_Open(ADC, NULL);
    /* Begin to do ADC conversion. */
    ch = 0;
    ADC_CLR_INT_FLAG(ADC, ADC_STATUS_ADC0IF_Msk | ADC_STATUS_ADC1IF_Msk);
    while (ch != 'q')
    {
        ADC_START_CONV(ADC, ADC_CTL_ADC0SWTRG_Msk);              	/* software trigger ADC0 */
        while (!ADC_GET_INT_FLAG(ADC, ADC_STATUS_ADC0IF_Msk));    /* Simultaneous  Mode: wait ADC0 completed here for both ADC0 and ADC1 conversion. */
        ADC_CLR_INT_FLAG(ADC, ADC_STATUS_ADC0IF_Msk);
				/* MUST read and keep valid flag in DAT register before any others read it. */
        reg_adc0_1 = ADC_Get_Data_Valid_Flag(ADC,ADC_VALSTS_ADC0VALID_CH1);
				reg_adc0_3 = ADC_Get_Data_Valid_Flag(ADC,ADC_VALSTS_ADC0VALID_CH3);
				reg_adc1_3 = ADC_Get_Data_Valid_Flag(ADC,ADC_VALSTS_ADC1VALID_CH3);
				reg_adc1_7 = ADC_Get_Data_Valid_Flag(ADC,ADC_VALSTS_ADC1VALID_CH7);
        printf("Get ADC0_data1 = %4d (valid = %s)\n", ADC0_GET_CONV_DATA(ADC0_ADC0_P1_DAT), (reg_adc0_1) ? "Yes" : "No");
        printf("Get ADC0_data3 = %4d (valid = %s)\n", ADC0_GET_CONV_DATA(ADC0_ADC0_P3_DAT), (reg_adc0_3) ? "Yes" : "No");
        printf("Get ADC1_data3 = %4d (valid = %s)\n", ADC1_GET_CONV_DATA(ADC1_ADC0_P0_DAT), (reg_adc1_3) ? "Yes" : "No");
        printf("Get ADC1_data7 = %4d (valid = %s)\n", ADC1_GET_CONV_DATA(ADC1_VSS_DAT), (reg_adc1_7) ? "Yes" : "No");

        printf("Press any key for next ADC conversion ... Press 'q' to quit.\n");
        ch = getchar();
    }

    ADC_Close(ADC);
    CLK_DisableModuleClock(ADC_MODULE);
    printf("=== THE END ===\n\n");
    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
