/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision:  $
 * $Date: 2020/07/16  $
 * @brief    Sample code for ADC Independent Simple mode with Temperature Sensor input.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void SYS_Init(void)
{
    /* Unlock protected registers */
    SYS_UnlockReg();
    
    /* Enable 48/60MHz HIRC */
    CLK_SetCoreClock(FREQ_60MHZ);

    /* Enable USCI2 IP clock */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Enable IP clock */
    CLK_EnableModuleClock(ADC_MODULE);
    SYS_ResetModule(ADC_RST);

    /* Select IP clock source */

    /*---------------------------------------------------------------------------------------------------------*/
    /* Init I/O Multi-function                                                                                 */
    /*---------------------------------------------------------------------------------------------------------*/

    /* Lock protected registers */
    SYS_LockReg();
}


int main()
{
    char ch;

    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\n\nPDID 0x%08X\n", SYS_ReadPDID());    /* Display PDID */
    printf("CPU @ %dHz\n", SystemCoreClock);        /* Display System Core Clock */

    /*
     * This sample code will convert ADC1 Temperature Sensor in Independent Simple mode
     * and print conversion results.
     */
    printf("+------------------------------------------------+\n");
    printf("| NM1240 ADC for Temperature Sensor Sample Code |\n");
    printf("+------------------------------------------------+\n");

    /* Enable ADC clock */
    CLK_EnableModuleClock(ADC_MODULE);

    /* Reset IP */
    SYS_ResetModule(ADC_RST);

    /* Enable ADC */
    ADC_Open(ADC, NULL);

    /* Configure ADC1: ADC1CS_TEMP_SNR(Temperature Sensor), software trigger */
    ADC_ADC1ConfigChannel(ADC, ADC1CS_TEMP_SENSOR, ADC1_SOFTWARE_TRIGGER);
    SYS->IVSCTL |= SYS_IVSCTL_VTEMPEN_Msk;  /* Enable Temperature Sensor */

    /* Configure ADC conversion mode to Independent Simple Mode */
    ADC_SET_INDEPENDENT_MODE(ADC);

    /* Configure ADC sample time to 6 ADC clocks */
    ADC_SetSampleCnt(ADC, NULL, 5);

    /* Begin to do ADC conversion. */
    ch = 0;
    ADC_CLR_INT_FLAG(ADC, ADC_STATUS_ADC1IF_Msk);
    while (ch != 'q')
    {
        ADC_START_CONV(ADC, ADC_CTL_ADC1SWTRG_Msk);              /* software trigger ADC1 */
        while (!ADC_GET_INT_FLAG(ADC, ADC_STATUS_ADC1IF_Msk));    /* wait ADC1 completed by polling */
        ADC_CLR_INT_FLAG(ADC, ADC_STATUS_ADC1IF_Msk);
        printf("Get ADC1 data6 = %d\n", ADC1_GET_CONV_DATA(ADC1_TEMP_SENSOR_DAT));

        printf("Press any key for next ADC conversion ... Press 'q' to quit.\n");
        ch = getchar();
    }

    ADC_Close(ADC);
    CLK_DisableModuleClock(ADC_MODULE);
    printf("=== THE END ===\n\n");
    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
