/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision:  $
 * $Date: 2020/07/15 14:37 $
 * @brief    Sample code for ADC sample value from ADC0_CH0 and output to DAC0.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void SYS_Init(void)
{
    /* Unlock protected registers */
    SYS_UnlockReg();
    
    /* Enable 48/60MHz HIRC */
    CLK_SetCoreClock(FREQ_60MHZ);
    
    /* Enable USCI2 IP clock */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Enable IP clock */
    CLK_EnableModuleClock(ADC_MODULE);
    SYS_ResetModule(ADC_RST);

    /* Select IP clock source */

    /*---------------------------------------------------------------------------------------------------------*/
    /* Init I/O Multi-function                                                                                 */
    /*---------------------------------------------------------------------------------------------------------*/

    /* Lock protected registers */
    SYS_LockReg();
}


int main()
{
    char ch;
    uint32_t u32_dat;

    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\n\nPDID 0x%08X\n", SYS_ReadPDID());    /* Display PDID */
    printf("CPU @ %dHz\n", SystemCoreClock);        /* Display System Core Clock */

    /*
     * This sample code will convert ADC0 channel 0 and ADC1 channel 0 in Independent 2SH mode
     * and print conversion results.
     */
    printf("+----------------------------------------------+\n");
    printf("| NM1240 ADC to DAC Sample Code                |\n");
    printf("+----------------------------------------------+\n");

    /* Enable ADC clock */
    CLK_EnableModuleClock(ADC_MODULE);


    /* Reset IP */
    SYS_ResetModule(ADC_RST);
    
    /* Enable ACMP clock */
    CLK_EnableModuleClock(ACMP_MODULE);

    /* Reset IP */
    SYS_ResetModule(ACMP_RST);

		/*ADC0CS_ADC0_P0 = PB0*/
    printf("Set GPB0 pin as ADC0_CH0(ADC0_P0) input pin.\n");
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB0MFP_Msk)) | SYS_GPB_MFP_PB0_ADC0_P0;
    GPIO_SetMode(PB, BIT0, GPIO_MODE_INPUT);
    GPIO_DISABLE_DIGITAL_PATH(PB, BIT0);
	
		/*ADC1CS_ADC1_P0 = PB4*/
    printf("Set GPB4 pin as ADC1_CH0(ADC1_P0) input pin.\n");
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB4MFP_Msk)) | SYS_GPB_MFP_PB4_ADC1_P0;
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE4_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE4_Pos);
    PB->DINOFF |= GPIO_DINOFF_DINOFF4_Msk;
	
    /* Set GPB5 to DAC0 */
    printf("Set GPB5 pin as DAC0 output pin.\n");
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB5MFP_Msk)) | SYS_GPB_MFP_PB5_DAC0;
    GPIO_SetMode(PB, BIT5, GPIO_MODE_OUTPUT);
    GPIO_DISABLE_DIGITAL_PATH(PB, BIT5);

    /* Enable ADC */
    ADC_Open(ADC, NULL);

    /* Configure ADC0: ADC0_CH0, software trigger */
    ADC_ADC0ConfigChannel(ADC, ADC0CS_ADC0_P0, ADC0_SOFTWARE_TRIGGER);

    /* Configure ADC1: ADC1_CH0, software trigger */
    ADC_ADC1ConfigChannel(ADC, ADC1CS_ADC1_P0, ADC1_SOFTWARE_TRIGGER);
	
	
    /* Configure ADC conversion mode to Independent Sample Mode */
    ADC_SET_INDEPENDENT_MODE(ADC);

    /* Configure ADC sample time to 6 ADC clocks */
    ADC_SetSampleCnt(ADC, NULL, 5);

    /* Begin to do ADC conversion. */
    ADC_CLR_INT_FLAG(ADC, ADC_STATUS_ADC0IF_Msk);
    
    printf("Start to convert ADC0_CH0(PB.0) value to DAC0(PB.5) output with refresh rate 0.5 sec.\n");
    while (1)
    {
        CLK_SysTickDelay(200000);
        CLK_SysTickDelay(200000);
        /* Software trigger ADC0 */
        ADC_START_CONV(ADC, ADC_CTL_ADC0SWTRG_Msk);    
				CLK_SysTickDelay(100000);
				ADC_START_CONV(ADC, ADC_CTL_ADC1SWTRG_Msk);   
		
        /* Wait ADC0 completed */        
        while (!ADC_GET_INT_FLAG(ADC, ADC_STATUS_ADC0IF_Msk)); 
        ADC_CLR_INT_FLAG(ADC, ADC_STATUS_ADC0IF_Msk);
        
        /* Show ADC data */
        u32_dat = ADC0_GET_CONV_DATA(ADC0_ADC0_P0_DAT);
        printf("ADC1 current value = %d mv    \r", u32_dat*5000/4096);
				printf("Get ADC0 data0 = %d\n",ADC0_GET_CONV_DATA(ADC0_ADC0_P0_DAT));
        printf("Get ADC1 data0 = %d\n",ADC1_GET_CONV_DATA(ADC1_ADC1_P0_DAT));        
        /* Output ADC0_CH0 current value to DAC0 */
        ACMP_DAC0_SEL(ACMP, u32_dat);
        
        if((DEBUG_PORT->BUFSTS & UUART_BUFSTS_RXEMPTY_Msk) == 0)  {
            ch = (DEBUG_PORT->RXDAT);
            if((ch == 'Q') || (ch == 'q')) {
                break;
            }
        }
    }

    ADC_Close(ADC);
    CLK_DisableModuleClock(ADC_MODULE);
    printf("\n=== THE END ===\n\n");
    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
