/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/07/16 16:44 $
 * @brief    Demonstrate analog comparator (ACMP) comparison by comparing
 *           ACMP0_P0 input and VBG voltage and shows the result on UART console
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"

void ACMP_IRQHandler(void)
{
    static uint32_t u32Cnt = 0;

    /* Clear ACMP 0 interrupt flag */
    ACMP_CLR_INT_FLAG(ACMP, 0);
    
    /* Check Comparator 0 Output Status */
    if(ACMP_GET_OUTPUT(ACMP, 0))
        printf("ACMP0_P0 voltage > Band-gap voltage (%d)\n", u32Cnt);
    else
        printf("ACMP0_P0 voltage <= Band-gap voltage (%d)\n", u32Cnt);

    u32Cnt++;
}

void SYS_Init(void)
{
    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Enable 60MHz HIRC */
    CLK_SetCoreClock(FREQ_60MHZ);
    
    /* Enable IP clock */
    CLK_EnableModuleClock(USCI2_MODULE);
    CLK_EnableModuleClock(ACMP_MODULE);

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    GPIO_SetMode(PF, BIT1, GPIO_MODE_OUTPUT);
    GPIO_SetMode(PF, BIT2, GPIO_MODE_INPUT);

    /* Set GPB0 multi-function pin for ACMP0 positive input pin */
    SYS->GPB_MFP = SYS_GPB_MFP_PB0_ACMP0_P0;

    /* Set GPB0 as input mode */
    GPIO_SetMode(PB, BIT0, GPIO_MODE_INPUT);

    /* Disable digital input path of analog pin ACMP0_P to prevent leakage */
    GPIO_DISABLE_DIGITAL_PATH(PB, (1 << 0));

    /* Lock protected registers */
    SYS_LockReg();
}

int main()
{
    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\n\n");
    printf("+---------------------------------------+\n");
    printf("|         ACMP Sample Code              |\n");
    printf("+---------------------------------------+\n");

    printf("\nThis sample code demonstrates ACMP0 function. Using ACMP0_P0 (PB.0) as ACMP0\n");
    printf("positive input and using internal band-gap voltage as the negative input\n");

    /* Configure ACMP0. Enable ACMP0 and select internal reference voltage as negative input. */
    ACMP_Open(ACMP, 0, ACMP_CTL_NEGSEL_VBG, ACMP_CTL_HYSTERESIS_DISABLE);
    
    /* Set ACMP0 rising edge and falling edge as interrupt condition. */
    ACMP_SELECT_INT_COND(ACMP, 0, ACMP_CTL_INTPOL_RF);
    
    /* Enable ACMP0 interrupt function */
    ACMP_ENABLE_INT(ACMP, 0);

    /* Enable ACMP01 interrupt */
    NVIC_EnableIRQ(ACMP_IRQn);
    
    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
