/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/07/13 10:26 $
 * @brief
 *           Implement USCI_SPI1 Master loop back transfer.
 *           This sample code needs to connect USCI_SPI1_MISO pin and USCI_SPI1_MOSI pin together.
 *           It will compare the received data with transmitted data.
 * @note
 * Copyright (C) 2018 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


#define TEST_COUNT             64

uint32_t g_au32SourceData[TEST_COUNT];
uint32_t g_au32DestinationData[TEST_COUNT];

void SYS_Init(void)
{
    int32_t i32TimeOutCnt = 2160000;

    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Enable 60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for 60MHz clock ready */
    while((CLK->STATUS & CLK_STATUS_HIRCSTB_Msk) != CLK_STATUS_HIRCSTB_Msk) {
        if(i32TimeOutCnt-- <= 0) break;
    }

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Enable IP clock */
    CLK->APBCLK = CLK->APBCLK | (CLK_APBCLK_USCI2CKEN_Msk | CLK_APBCLK_USCI1CKEN_Msk);

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Set USCI_SPI1 multi-function pins */
    SYS->GPC_MFP &= ~(SYS_GPC_MFP_PC0MFP_Msk | SYS_GPC_MFP_PC1MFP_Msk | SYS_GPC_MFP_PC2MFP_Msk | SYS_GPC_MFP_PC3MFP_Msk);
    SYS->GPC_MFP |= (SYS_GPC_MFP_PC0_SPI1_CLK | SYS_GPC_MFP_PC1_SPI1_MISO | SYS_GPC_MFP_PC2_SPI1_MOSI | SYS_GPC_MFP_PC3_SPI1_SS);

    /* Set GPC0,2,3 as output mode and GPC1 as Input mode */
    PC->MODE = (PC->MODE & ~(GPIO_MODE_MODE0_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE0_Pos);
    PC->MODE = (PC->MODE & ~(GPIO_MODE_MODE1_Msk));
    PC->MODE = (PC->MODE & ~(GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE2_Pos);
    PC->MODE = (PC->MODE & ~(GPIO_MODE_MODE3_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE3_Pos);

    /* Lock protected registers */
    SYS_LockReg();
}

void USCI_UART_Init(void)
{
	  /*---------------------------------------------------------------------------------------------------------*/
    /* Init USCI                                                                                               */
    /*---------------------------------------------------------------------------------------------------------*/
    /* Reset USCI2 */
    SYS->IPRST1 |=  SYS_IPRST1_USCI2RST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_USCI2RST_Msk;

    /* Configure USCI2 as UART mode */
    UUART2->CTL = (2 << UUART_CTL_FUNMODE_Pos);                                 /* Set UART function mode */
    UUART2->LINECTL = UUART_WORD_LEN_8 | UUART_LINECTL_LSB_Msk;                 /* Set UART line configuration */
    UUART2->DATIN0 = (2 << UUART_DATIN0_EDGEDET_Pos);                           /* Set falling edge detection */
    UUART2->BRGEN = (25 << UUART_BRGEN_CLKDIV_Pos) | (9 << UUART_BRGEN_DSCNT_Pos) | (1 << UUART_BRGEN_PDSCNT_Pos); /* Set UART baud rate as 115200bps */
    UUART2->PROTCTL |= UUART_PROTCTL_PROTEN_Msk;                                /* Enable UART protocol */
}

void USCI_SPI_Init(void)
{
    /*---------------------------------------------------------------------------------------------------------*/
    /* Init USCI_SPI                                                                                           */
    /*---------------------------------------------------------------------------------------------------------*/
    /* The USCI usage is exclusive */
    /* If user configure the USCI port as USPI function, that port cannot use UUART or UI2C function. */
    /* Select USCI_SPI1 protocol */
    USPI1->CTL &= ~USPI_CTL_FUNMODE_Msk;
    USPI1->CTL = 1 << USPI_CTL_FUNMODE_Pos;
    /* Configure USCI_SPI1 as a master, clock idle low, 16-bit transaction, drive output on falling clock edge and latch input on rising edge. */
    USPI1->PROTCTL |= USPI_MASTER | USPI_MODE_0;
    USPI1->LINECTL |= USPI_LINECTL_CTLOINV_Msk;
    /* Set USCI_SPI1 clock rate = f_PCLK / 2*(17+1) = 2MHz */
    USPI1->BRGEN = (USPI1->BRGEN & (~USPI_BRGEN_CLKDIV_Msk)) | (0x11 << USPI_BRGEN_CLKDIV_Pos);
    /* Enable USCI_SPI1 protocol */
    USPI1->PROTCTL |= USPI_PROTCTL_PROTEN_Msk;
}


int main()
{
    uint32_t u32DataCount, u32TestCount, u32Err;

    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    USCI_UART_Init();

    printf("\n\nPDID 0x%08X\n", (unsigned int)(SYS->PDID & SYS_PDID_PDID_Msk));    /* Display PDID */
    printf("CPU @ %dHz\n", SystemCoreClock);        /* Display System Core Clock */

    /* Init USCI_SPI */
    USCI_SPI_Init();

    printf("\n\n");
    printf("+------------------------------------------------------------------+\n");
    printf("|                   USCI_SPI Driver Sample Code                    |\n");
    printf("+------------------------------------------------------------------+\n");
    printf("\n");
    printf("\nThis sample code demonstrates USCI_SPI1 self loop back data transfer.\n");
    printf(" USCI_SPI1 configuration:\n");
    printf("     Master mode; data width 16 bits.\n");
    printf(" I/O connection:\n");
    printf("     PC.2 USCI_SPI1_MOSI <--> PC.1 USCI_SPI1_MISO \n");

    printf("\nUSCI_SPI1 Loopback test ");

    /* set the source data and clear the destination buffer */
    for(u32DataCount = 0; u32DataCount < TEST_COUNT; u32DataCount++) {
        g_au32SourceData[u32DataCount] = u32DataCount;
        g_au32DestinationData[u32DataCount] = 0;
    }

    u32Err = 0;
    for(u32TestCount = 0; u32TestCount < 0x1000; u32TestCount++) {
        /* set the source data and clear the destination buffer */
        for(u32DataCount = 0; u32DataCount < TEST_COUNT; u32DataCount++) {
            g_au32SourceData[u32DataCount]++;
            g_au32DestinationData[u32DataCount] = 0;
        }

        u32DataCount = 0;

        if((u32TestCount & 0x1FF) == 0) {
            putchar('.');
        }

        while(1) {
            /* Write to TX register */
            USPI1->TXDAT = g_au32SourceData[u32DataCount];
            /* Check USCI_SPI1 busy status */
            while(USPI1->PROTSTS & USPI_PROTSTS_BUSY_Msk);
            /* Read received data */
            g_au32DestinationData[u32DataCount] = USPI1->RXDAT;
            u32DataCount++;
            if(u32DataCount == TEST_COUNT)
                break;
        }

        /*  Check the received data */
        for(u32DataCount = 0; u32DataCount < TEST_COUNT; u32DataCount++) {
            if(g_au32DestinationData[u32DataCount] != g_au32SourceData[u32DataCount])
                u32Err = 1;
        }

        if(u32Err)
            break;
    }

    if(u32Err)
        printf(" [FAIL]\n\n");
    else
        printf(" [PASS]\n\n");

    /* Disable USCI_SPI1 function mode */
    USPI1->CTL &= ~USPI_CTL_FUNMODE_Msk;

    while(1);
}

/*** (C) COPYRIGHT 2018 Nuvoton Technology Corp. ***/
