/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/07/13 13:53 $
 * @brief    Show how to set USCI_I2C in Master mode and send data to Slave device.
 *           This sample code needs to work with USCI_I2C_Slave.
 * @note
 * Copyright (C) 2018 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


/*---------------------------------------------------------------------------------------------------------*/
/* Global variables                                                                                        */
/*---------------------------------------------------------------------------------------------------------*/
volatile uint8_t g_u8DeviceAddr;
volatile uint8_t g_au8TxData[3];
volatile uint8_t g_u8RxData;
volatile uint8_t g_u8DataLenM;
volatile uint8_t g_u8EndFlagM = 0;
volatile enum UI2C_MASTER_EVENT m_Event;

typedef void (*UI2C_FUNC)(uint32_t u32Status);

volatile static UI2C_FUNC s_UI2C1HandlerFn = NULL;

/*---------------------------------------------------------------------------------------------------------*/
/*  USCI_I2C1 IRQ Handler                                                                                  */
/*---------------------------------------------------------------------------------------------------------*/
void USCI1_IRQHandler(void)
{
    uint32_t u32Status;

    u32Status = UI2C_GET_PROT_STATUS(UI2C1);

    if (UI2C_GET_TIMEOUT_FLAG(UI2C1)) {
        /* Clear USCI_I2C1 Timeout Flag */
        /* UI2C_ClearTimeoutFlag(UI2C1); */
        UI2C1->PROTSTS = UI2C_PROTSTS_TOIF_Msk;
    } else {
        if (s_UI2C1HandlerFn != NULL)
            s_UI2C1HandlerFn(u32Status);
    }
}

/*---------------------------------------------------------------------------------------------------------*/
/*  USCI_I2C Rx Callback Function                                                                          */
/*---------------------------------------------------------------------------------------------------------*/
void USCI_I2C_MasterRx(uint32_t u32Status)
{
    if((u32Status & UI2C_PROTSTS_STARIF_Msk) == UI2C_PROTSTS_STARIF_Msk) {
        /* UI2C_CLR_PROT_INT_FLAG(UI2C1, UI2C_PROTSTS_STARIF_Msk); */ /* Clear START INT Flag */
        UI2C1->PROTSTS = UI2C_PROTSTS_STARIF_Msk;
        if(m_Event == MASTER_SEND_START) {
            /* UI2C_SET_DATA(UI2C1, (g_u8DeviceAddr << 1) | 0x00); */ /* Write SLA+W to Register TXDAT */
            UI2C1->TXDAT = (g_u8DeviceAddr << 1) | 0x00;
            m_Event = MASTER_SEND_ADDRESS;
        } else if(m_Event == MASTER_SEND_REPEAT_START) {
            /* UI2C_SET_DATA(UI2C1, (g_u8DeviceAddr << 1) | 0x01); */ /* Write SLA+R to Register TXDAT */
            UI2C1->TXDAT = (g_u8DeviceAddr << 1) | 0x01;
            m_Event = MASTER_SEND_H_RD_ADDRESS;
        }
        /* UI2C_SET_CONTROL_REG(UI2C1, UI2C_CTL_PTRG); */
        UI2C1->PROTCTL = (UI2C1->PROTCTL & ~0x2E) | (UI2C_CTL_PTRG);
    } else if((u32Status & UI2C_PROTSTS_ACKIF_Msk) == UI2C_PROTSTS_ACKIF_Msk) {
        /* UI2C_CLR_PROT_INT_FLAG(UI2C1, UI2C_PROTSTS_ACKIF_Msk); */  /* Clear ACK INT Flag */
        UI2C1->PROTSTS = UI2C_PROTSTS_ACKIF_Msk;
        if(m_Event == MASTER_SEND_ADDRESS) {
            /* UI2C_SET_DATA(UI2C1, g_au8TxData[g_u8DataLenM++]); */  /* SLA+W has been transmitted and write ADDRESS to Register TXDAT */
            UI2C1->TXDAT = g_au8TxData[g_u8DataLenM++];
            m_Event = MASTER_SEND_DATA;
            /* UI2C_SET_CONTROL_REG(UI2C1, UI2C_CTL_PTRG); */
            UI2C1->PROTCTL = (UI2C1->PROTCTL & ~0x2E) | (UI2C_CTL_PTRG);
        } else if(m_Event == MASTER_SEND_DATA) {
            if(g_u8DataLenM != 2) {
                /* UI2C_SET_DATA(UI2C1, g_au8TxData[g_u8DataLenM++]); */  /* ADDRESS has been transmitted and write DATA to Register TXDAT */
                UI2C1->TXDAT = g_au8TxData[g_u8DataLenM++];
                /* UI2C_SET_CONTROL_REG(UI2C1, UI2C_CTL_PTRG); */
                UI2C1->PROTCTL = (UI2C1->PROTCTL & ~0x2E) | (UI2C_CTL_PTRG);
            } else {
                m_Event = MASTER_SEND_REPEAT_START;
                /* UI2C_SET_CONTROL_REG(UI2C1, (UI2C_CTL_PTRG | UI2C_CTL_STA)); */    /* Send repeat START signal */
                UI2C1->PROTCTL = (UI2C1->PROTCTL & ~0x2E) | (UI2C_CTL_PTRG | UI2C_CTL_STA);
            }
        } else if(m_Event == MASTER_SEND_H_RD_ADDRESS) {
            m_Event = MASTER_READ_DATA;
            /* UI2C_SET_CONTROL_REG(UI2C1, UI2C_CTL_PTRG); */
            UI2C1->PROTCTL = (UI2C1->PROTCTL & ~0x2E) | (UI2C_CTL_PTRG);
        }
    } else if((u32Status & UI2C_PROTSTS_NACKIF_Msk) == UI2C_PROTSTS_NACKIF_Msk) {
        /* UI2C_CLR_PROT_INT_FLAG(UI2C1, UI2C_PROTSTS_NACKIF_Msk); */ /* Clear NACK INT Flag */
        UI2C1->PROTSTS = UI2C_PROTSTS_NACKIF_Msk;
        if(m_Event == MASTER_SEND_ADDRESS) {
            m_Event = MASTER_SEND_START;
            /* UI2C_SET_CONTROL_REG(UI2C1, (UI2C_CTL_PTRG | UI2C_CTL_STA)); */    /* Send START signal */
            UI2C1->PROTCTL = (UI2C1->PROTCTL & ~0x2E) | (UI2C_CTL_PTRG | UI2C_CTL_STA);
        } else if(m_Event == MASTER_READ_DATA) {
            g_u8RxData = (unsigned char) UI2C_GET_DATA(UI2C1) & 0xFF;
            g_u8EndFlagM = 1;
            m_Event = MASTER_STOP;
            /* UI2C_SET_CONTROL_REG(UI2C1, (UI2C_CTL_PTRG | UI2C_CTL_STO)); */    /* DATA has been received and send STOP signal */
            UI2C1->PROTCTL = (UI2C1->PROTCTL & ~0x2E) | (UI2C_CTL_PTRG | UI2C_CTL_STO);
        } else {
            /* TO DO */
            printf("Status 0x%x is NOT processed\n", u32Status);
        }
    }else if((u32Status & UI2C_PROTSTS_STORIF_Msk) == UI2C_PROTSTS_STORIF_Msk) {
		UI2C1->PROTSTS = UI2C_PROTSTS_STORIF_Msk;
		m_Event = MASTER_SEND_START;
	}
}

/*---------------------------------------------------------------------------------------------------------*/
/*  USCI_I2C Tx Callback Function                                                                          */
/*---------------------------------------------------------------------------------------------------------*/
void USCI_I2C_MasterTx(uint32_t u32Status)
{
    if((u32Status & UI2C_PROTSTS_STARIF_Msk) == UI2C_PROTSTS_STARIF_Msk) {
        /* UI2C_CLR_PROT_INT_FLAG(UI2C1, UI2C_PROTSTS_STARIF_Msk); */ /* Clear START INT Flag */
        UI2C1->PROTSTS = UI2C_PROTSTS_STARIF_Msk;
        /* UI2C_SET_DATA(UI2C1, (g_u8DeviceAddr << 1) | 0x00); */     /* Write SLA+W to Register TXDAT */
        UI2C1->TXDAT = (g_u8DeviceAddr << 1) | 0x00;
        m_Event = MASTER_SEND_ADDRESS;
        /* UI2C_SET_CONTROL_REG(UI2C1, UI2C_CTL_PTRG); */
        UI2C1->PROTCTL = (UI2C1->PROTCTL & ~0x2E) | (UI2C_CTL_PTRG);
    } else if((u32Status & UI2C_PROTSTS_ACKIF_Msk) == UI2C_PROTSTS_ACKIF_Msk) {
        /* UI2C_CLR_PROT_INT_FLAG(UI2C1, UI2C_PROTSTS_ACKIF_Msk); */  /* Clear ACK INT Flag */
        UI2C1->PROTSTS = UI2C_PROTSTS_ACKIF_Msk;
        if(m_Event == MASTER_SEND_ADDRESS) {
            /* UI2C_SET_DATA(UI2C1, g_au8TxData[g_u8DataLenM++]); */  /* SLA+W has been transmitted and write ADDRESS to Register TXDAT */
            UI2C1->TXDAT = g_au8TxData[g_u8DataLenM++];
            m_Event = MASTER_SEND_DATA;
            /* UI2C_SET_CONTROL_REG(UI2C1, UI2C_CTL_PTRG); */
            UI2C1->PROTCTL = (UI2C1->PROTCTL & ~0x2E) | (UI2C_CTL_PTRG);
        } else if(m_Event == MASTER_SEND_DATA) {
            if(g_u8DataLenM != 3) {
                /* UI2C_SET_DATA(UI2C1, g_au8TxData[g_u8DataLenM++]); */  /* ADDRESS has been transmitted and write DATA to Register TXDAT */
                UI2C1->TXDAT = g_au8TxData[g_u8DataLenM++];
                /* UI2C_SET_CONTROL_REG(UI2C1, UI2C_CTL_PTRG); */
                UI2C1->PROTCTL = (UI2C1->PROTCTL & ~0x2E) | (UI2C_CTL_PTRG);
            } else {
                g_u8EndFlagM = 1;
                m_Event = MASTER_STOP;
                /* UI2C_SET_CONTROL_REG(UI2C1, (UI2C_CTL_PTRG | UI2C_CTL_STO)); */        /* Send STOP signal */
                UI2C1->PROTCTL = (UI2C1->PROTCTL & ~0x2E) | (UI2C_CTL_PTRG | UI2C_CTL_STO);
            }
        }
    } else if((u32Status & UI2C_PROTSTS_NACKIF_Msk) == UI2C_PROTSTS_NACKIF_Msk) {
        /* UI2C_CLR_PROT_INT_FLAG(UI2C1, UI2C_PROTSTS_NACKIF_Msk); */ /* Clear NACK INT Flag */
        UI2C1->PROTSTS = UI2C_PROTSTS_NACKIF_Msk;
        g_u8EndFlagM = 0;
        if(m_Event == MASTER_SEND_ADDRESS) {     /* SLA+W has been transmitted and NACK has been received */
            m_Event = MASTER_SEND_START;
            /* UI2C_SET_CONTROL_REG(UI2C1, (UI2C_CTL_PTRG | UI2C_CTL_STA)); */        /* Send START signal */
            UI2C1->PROTCTL = (UI2C1->PROTCTL & ~0x2E) | (UI2C_CTL_PTRG | UI2C_CTL_STA);
        } else if(m_Event == MASTER_SEND_DATA) {    /* ADDRESS has been transmitted and NACK has been received */
            m_Event = MASTER_STOP;
            /* UI2C_SET_CONTROL_REG(UI2C1, (UI2C_CTL_PTRG | UI2C_CTL_STO)); */       /* Send STOP signal */
            UI2C1->PROTCTL = (UI2C1->PROTCTL & ~0x2E) | (UI2C_CTL_PTRG | UI2C_CTL_STO);
        } else {
            printf("Get Wrong NACK Event\n");
        }
    }else if((u32Status & UI2C_PROTSTS_STORIF_Msk) == UI2C_PROTSTS_STORIF_Msk) {
		UI2C1->PROTSTS = UI2C_PROTSTS_STORIF_Msk;
		m_Event = MASTER_SEND_START;
	}
}

void SYS_Init(void)
{
    int32_t i32TimeOutCnt = 2160000;

    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Enable 60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for 60MHz clock ready */
    while((CLK->STATUS & CLK_STATUS_HIRCSTB_Msk) != CLK_STATUS_HIRCSTB_Msk) {
        if(i32TimeOutCnt-- <= 0) break;
    }

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Enable IP clock */
    CLK->APBCLK = CLK->APBCLK | (CLK_APBCLK_USCI2CKEN_Msk | CLK_APBCLK_USCI1CKEN_Msk);

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Set GPC multi-function pins for USCI I2C1 GPC0(SCL) and GPC2(SDA) */
    SYS->GPC_MFP = SYS->GPC_MFP & ~(SYS_GPC_MFP_PC0MFP_Msk | SYS_GPC_MFP_PC2MFP_Msk) | (SYS_GPC_MFP_PC0_I2C1_SCL | SYS_GPC_MFP_PC2_I2C1_SDA);

    /* Lock protected registers */
    SYS_LockReg();
}

void UI2C1_Init(uint32_t u32ClkSpeed)
{
    uint32_t u32ClkDiv;
    uint32_t u32Pclk;

    SystemCoreClockUpdate();
    u32Pclk = SystemCoreClock;

    /* The USCI usage is exclusive */
    /* If user configure the USCI port as UI2C function, that port cannot use UUART or USPI function. */
    /* Open USCI_I2C1 and set clock to 100k */
    u32ClkDiv = (uint32_t) ((((((u32Pclk/2)*10)/(u32ClkSpeed))+5)/10)-1); /* Compute proper divider for USCI_I2C clock */

    /* Enable USCI_I2C protocol */
    UI2C1->CTL &= ~UI2C_CTL_FUNMODE_Msk;
    UI2C1->CTL = 4 << UI2C_CTL_FUNMODE_Pos;

    /* Data format configuration */
    /* 8 bit data length */
    UI2C1->LINECTL &= ~UI2C_LINECTL_DWIDTH_Msk;
    UI2C1->LINECTL |= 8 << UI2C_LINECTL_DWIDTH_Pos;

    /* MSB data format */
    UI2C1->LINECTL &= ~UI2C_LINECTL_LSB_Msk;

    /* Set USCI_I2C bus clock */
    UI2C1->BRGEN &= ~UI2C_BRGEN_CLKDIV_Msk;
    UI2C1->BRGEN |=  (u32ClkDiv << UI2C_BRGEN_CLKDIV_Pos);
    UI2C1->PROTCTL |=  UI2C_PROTCTL_PROTEN_Msk;

    UI2C_ENABLE_PROT_INT(UI2C1, (UI2C_PROTIEN_ACKIEN_Msk | UI2C_PROTIEN_NACKIEN_Msk | UI2C_PROTIEN_STORIEN_Msk | UI2C_PROTIEN_STARIEN_Msk));
    NVIC_EnableIRQ(USCI1_IRQn);
}

int32_t Read_Write_SLAVE(uint8_t slvaddr)
{
    uint32_t i;
    uint8_t u8Temp;

    g_u8DeviceAddr = slvaddr;

    for (i = 0; i < 0x100; i++) {
        g_au8TxData[0] = (uint8_t)((i & 0xFF00) >> 8);
        g_au8TxData[1] = (uint8_t)(i & 0x00FF);
        g_au8TxData[2] = (uint8_t)(g_au8TxData[1] + 3);

        g_u8DataLenM = 0;
        g_u8EndFlagM = 0;

        /* USCI_I2C function to write data to slave */
        s_UI2C1HandlerFn = (UI2C_FUNC)USCI_I2C_MasterTx;

        /* USCI_I2C as master sends START signal */
        UI2C_SET_CONTROL_REG(UI2C1, UI2C_CTL_STA);

        /* Wait USCI_I2C Tx Finish */
        while (g_u8EndFlagM == 0);
        g_u8EndFlagM = 0;

        /* USCI_I2C function to read data from slave */
        s_UI2C1HandlerFn = (UI2C_FUNC)USCI_I2C_MasterRx;

        g_u8DataLenM = 0;
        g_u8DeviceAddr = slvaddr;

        /* UI2C_SET_CONTROL_REG(UI2C1, UI2C_CTL_STA); */
        UI2C1->PROTCTL = (UI2C1->PROTCTL & ~0x2E) | (UI2C_CTL_STA);

        /* Wait USCI_I2C Rx Finish */
        while(g_u8EndFlagM == 0);
        g_u8EndFlagM = 0;

        /* Compare data */
        u8Temp = g_au8TxData[2];
        if (g_u8RxData != u8Temp) {
            printf("USCI_I2C Byte Write/Read Failed, Data 0x%x\n", g_u8RxData);
            return -1;
        }
    }
    printf("Master Access Slave (0x%X) Test OK\n", slvaddr);
    return 0;
}

void USCI_UART_Init(void)
{
	  /*---------------------------------------------------------------------------------------------------------*/
    /* Init USCI                                                                                               */
    /*---------------------------------------------------------------------------------------------------------*/
    /* Reset USCI2 */
    SYS->IPRST1 |=  SYS_IPRST1_USCI2RST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_USCI2RST_Msk;

    /* Configure USCI2 as UART mode */
    UUART2->CTL = (2 << UUART_CTL_FUNMODE_Pos);                                 /* Set UART function mode */
    UUART2->LINECTL = UUART_WORD_LEN_8 | UUART_LINECTL_LSB_Msk;                 /* Set UART line configuration */
    UUART2->DATIN0 = (2 << UUART_DATIN0_EDGEDET_Pos);                           /* Set falling edge detection */
    UUART2->BRGEN = (25 << UUART_BRGEN_CLKDIV_Pos) | (9 << UUART_BRGEN_DSCNT_Pos) | (1 << UUART_BRGEN_PDSCNT_Pos); /* Set UART baud rate as 115200bps */
    UUART2->PROTCTL |= UUART_PROTCTL_PROTEN_Msk;                                /* Enable UART protocol */
}


int main()
{
    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    USCI_UART_Init();

    /*
        This sample code sets USCI_I2C bus clock to 100kHz. Then, Master accesses Slave with Byte Write
        and Byte Read operations, and check if the read data is equal to the programmed data.
    */

    printf("+-------------------------------------------------------+\n");
    printf("|  USCI_I2C Driver Sample Code(Master) for access Slave |\n");
    printf("+-------------------------------------------------------+\n");

    /* Init USCI_I2C1 */
    UI2C1_Init(100000);

    /* Access Slave with no address mask */
    printf("\n");
    printf(" == No Mask Address ==\n");
    Read_Write_SLAVE(0x15);
    printf("SLAVE Address test OK.\n");

    /* Access Slave with address mask */
    printf("\n");
    printf(" == Mask Address ==\n");
    Read_Write_SLAVE(0x15 & ~0x01);
    printf("SLAVE Address Mask test OK.\n");

    while(1);
}

/*** (C) COPYRIGHT 2018 Nuvoton Technology Corp. ***/
