/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 3 $
 * $Date: 2020/07/17 15:17 $
 * @brief    Use timer to wake up system from Power-down mode periodically.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void TMR0_IRQHandler(void)
{
    /* Clear wake up flag */
    TIMER_ClearWakeupFlag(TIMER0);
    
    /* Clear interrupt flag */
    TIMER_ClearIntFlag(TIMER0);
}

void SYS_Init(void)
{
    int32_t i32TimeOutCnt = 2160000;

    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Enable 60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for HIRC clock ready */
    while((CLK->STATUS & CLK_STATUS_HIRCSTB_Msk) != CLK_STATUS_HIRCSTB_Msk) {
        if(i32TimeOutCnt-- <= 0) break;
    }

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Enable IP clock */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_TMR0CKEN_Msk;

    /* Select IP clock source */
    CLK->CLKSEL1 = CLK->CLKSEL1 & ~(CLK_CLKSEL1_TMR0SEL_Msk) | CLK_TMR0_SRC_LIRC;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

     /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Lock protected registers */
    SYS_LockReg();
}


int main()
{
    int i = 0;

    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    /*Initial Timer0 to periodic mode with 1Hz */
    TIMER0->CMP = __LIRC;
    TIMER0->CTL = (TIMER0->CTL & ~TIMER_CTL_PSC_Msk) | TIMER_PERIODIC_MODE | 0;
    
    /* Enable timer wake up system */
    TIMER0->CTL |= TIMER_CTL_WKEN_Msk;
    
    /* Enable Timer0 interrupt */
    TIMER0->CTL |= TIMER_CTL_INTEN_Msk;
    NVIC_EnableIRQ(TMR0_IRQn);
    
    /* Start Timer0 counting */
    TIMER0->CTL |= TIMER_CTL_CNTEN_Msk;
    
    /* Unlock protected registers */
    SYS_UnlockReg();
		
		/* Set PA0 as indicator*/
    GPIO_SetMode(PA, BIT0, GPIO_MODE_OUTPUT);
    
    while(1) {
        CLK_PowerDown();
        printf("Wake %d\n", i++);
				PA0 = ~PA0;

        /* Wait USCI UART buffer empty to get a cleaner console out */
        UUART2->PROTSTS = (UUART_PROTSTS_TXENDIF_Msk | UUART_PROTSTS_TXSTIF_Msk);
        while(!((UUART2->BUFSTS & UUART_BUFSTS_TXEMPTY_Msk) >> UUART_BUFSTS_TXEMPTY_Pos));
        if(UUART2->PROTSTS & UUART_PROTSTS_TXSTIF_Msk)
            while((UUART2->PROTSTS & UUART_PROTSTS_TXENDIF_Msk) != UUART_PROTSTS_TXENDIF_Msk);
    }
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
