/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 3 $
 * $Date: 2020/07/17 14:55 $
 * @brief    Use the ACMP0 positive input pin to demonstrate timer free counting mode
 *           function. And displays the measured input frequency to console
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void TMR0_IRQHandler(void)
{
    /* printf takes long time and affect the freq. calculation, we only print out once a while */
    static int cnt = 0;
    static uint32_t t0, t1;

    if(cnt == 0) {
        t0 = TIMER_GetCaptureData(TIMER0);
        cnt++;
    } else if(cnt == 1) {
        t1 = TIMER_GetCaptureData(TIMER0);
        cnt++;
        if(t0 > t1) {
            /* over run, drop this data and do nothing */
        } else {
            printf("Input frequency is %dHz\n", 1000000 / (t1 - t0));
        }
    } else {
        cnt = 0;
    }

    TIMER_ClearCaptureIntFlag(TIMER0);

}

void SYS_Init(void)
{
    int32_t i32TimeOutCnt = 2160000;

    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Enable 60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for HIRC clock ready */
    while((CLK->STATUS & CLK_STATUS_HIRCSTB_Msk) != CLK_STATUS_HIRCSTB_Msk) {
        if(i32TimeOutCnt-- <= 0) break;
    }

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Enable IP clock */
    CLK->APBCLK = CLK->APBCLK | (CLK_APBCLK_USCI2CKEN_Msk | CLK_APBCLK_TMR0CKEN_Msk | CLK_APBCLK_ACMPCKEN_Msk | CLK_APBCLK_CLKOCKEN_Msk);

    /* Select IP clock source */
    CLK->CLKSEL1 = CLK->CLKSEL1 | CLK_TMR0_SRC_HIRC;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Set GPB0 multi-function pin for ACMP0 positive input pin */
    SYS->GPB_MFP = SYS_GPB_MFP_PB0_ACMP0_P0;

    /* Set GPB0 as input mode */
    PB->MODE = PB->MODE & ~(GPIO_MODE_MODE0_Msk);

    /* Disable digital input path of analog pin ACMP0_P to prevent leakage */
    PB->DINOFF |= GPIO_DINOFF_DINOFF0_Msk;

    /* Set GPA0 multi-function pin for clock out */
    SYS->GPA_MFP = (SYS->GPA_MFP & ~SYS_GPA_MFP_PA0MFP_Msk) | (0x1<<SYS_GPA_MFP_PA0MFP_Pos); 
    
    /* Set GPA0 as output mode */
    PA->MODE = (PA->MODE & ~(GPIO_MODE_MODE0_Msk))|(GPIO_MODE_OUTPUT << GPIO_MODE_MODE0_Pos);
    
    /* Lock protected registers */
    SYS_LockReg();
}


int main()
{
    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\nThis sample code demonstrate timer free counting mode.\n");
    printf("Please connect input source or PA0 (0.915Khz) with ACMP0 positive input pin (PB.0).\n");
    printf("Press any key to continue\n");
    getchar();

    /* PA0 to outpout a 0.915KHz(60Mhz/2^16) waveform */  
    CLK_SetModuleClock(CLKO_MODULE, CLK_CLKO_SRC_HIRC, 0);
    CLK_EnableCKO(CLK_CLKO_SRC_HIRC, 0xF, 0);
    
    /* Update prescale to set proper resolution. */
    /* Timer 0 clock source is 60MHz, to set resolution to 1us, we need to */
    /* set clock divider to 60. e.g. set prescale to 60 - 1 = 59 */
    TIMER0->CTL = (TIMER0->CTL & ~TIMER_CTL_PSC_Msk) | 59;
    
    /* Set compare value as large as possible, so don't need to worry about counter overrun too frequently. */
    TIMER0->CMP = 0xFFFFFF;

    /* Configure Timer 0 free counting mode, capture TDR value on rising edge */
    TIMER0->EXTCTL = (TIMER0->EXTCTL & ~(TIMER_EXTCTL_CAPFUNCS_Msk | TIMER_EXTCTL_CAPEDGE_Msk)) |
                     TIMER_CAPTURE_FREE_COUNTING_MODE | TIMER_CAPTURE_RISING_EDGE | TIMER_EXTCTL_CAPEN_Msk;

    /* Configure ACMP0. Enable ACMP0 and select internal reference voltage as negative input. */
    ACMP->CTL[0] = (ACMP->CTL[0] & ~(ACMP_CTL_CPNSEL_Msk | ACMP_CTL_ACMPHYSEN_Msk)) |
                   (ACMP_CTL_NEGSEL_VBG | ACMP_CTL_HYSTERESIS_DISABLE | ACMP_CTL_ACMPEN_Msk);

    /* Set ACMP0 rising edge as interrupt condition. */
    ACMP->CTL[0] = (ACMP->CTL[0] & ~ACMP_CTL_EDGESEL_Msk) | ACMP_CTL_INTPOL_R;

    /* Start Timer 0 */
    TIMER0->CTL |= (TIMER_CTL_CNTEN_Msk | TIMER_PERIODIC_MODE);

    /* Enable timer interrupt */
    TIMER0->EXTCTL |= TIMER_EXTCTL_CAPIEN_Msk;
    
    NVIC_EnableIRQ(TMR0_IRQn);

    while(1);
}

/*** (C) COPYRIGHT 2018 Nuvoton Technology Corp. ***/
