/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 3 $
 * $Date: 2020/07/17 14:33 $
 * @brief    Use pin PB.3 to demonstrates timer event counter function
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void TMR0_IRQHandler(void)
{
    printf("Count 1000 falling events! Test complete\n");
    TIMER_ClearIntFlag(TIMER0);

}

void SYS_Init(void)
{
    int32_t i32TimeOutCnt = 2160000;

    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Enable 72MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for HIRC clock ready */
    while((CLK->STATUS & CLK_STATUS_HIRCSTB_Msk) != CLK_STATUS_HIRCSTB_Msk) {
        if(i32TimeOutCnt-- <= 0) break;
    }

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Enable IP clock */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_TMR0CKEN_Msk;

    /* Select IP clock source */
    CLK->CLKSEL1 = CLK->CLKSEL1 | CLK_TMR0_SRC_HIRC;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Set Timer event counting pin */
    SYS->GPB_MFP = SYS_GPB_MFP_PB3_T0;

    /* Set GPB3 as Input mode */
    PB->MODE = PB->MODE & ~(GPIO_MODE_MODE3_Msk);

    /* Lock protected registers */
    SYS_LockReg();
}


int main()
{
    int i;

    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\nThis sample code use TM0_CNT_OUT(PB.3) to count PB.2 input event\n");
    printf("Please connect PB.3 to PB.2, press any key to continue\n");
    getchar();

    PB->DOUT |= 1 << GPIO_DOUT_DOUT2_Pos;                     /* Set init state to high */
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE2_Msk) | (0x1 << GPIO_MODE_MODE2_Pos);  /* Set to output mode */

    /* Reset Timer 0 counter value */
    TIMER0->CTL = 0x0;

    /* Give prescale and compare value to what we need in event counter mode. */
    TIMER_SET_PRESCALE_VALUE(TIMER0, 0);
    TIMER0->CMP = 1000;
    TIMER0->CTL = TIMER0->CTL | TIMER_ONESHOT_MODE;

    /* Counter increase on falling edge */
    TIMER0->EXTCTL = (TIMER0->EXTCTL & ~TIMER_EXTCTL_CNTPHASE_Msk) | TIMER_COUNTER_FALLING_EDGE;
    TIMER0->CTL |= TIMER_CTL_EXTCNTEN_Msk;

    /* Start Timer 0 and enable timer interrupt */
    TIMER0->CTL |= (TIMER_CTL_CNTEN_Msk | TIMER_CTL_INTEN_Msk);
    NVIC_EnableIRQ(TMR0_IRQn);

    for(i = 0; i < 1000; i++) {
        PB->DOUT &= ~(1 << GPIO_DOUT_DOUT2_Pos); /* low */
        PB->DOUT |= (1 << GPIO_DOUT_DOUT2_Pos);  /* high */
    }

    while(1);
}

/*** (C) COPYRIGHT 2018 Nuvoton Technology Corp. ***/
