/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 2 $
 * $Date: 2020/07/17 14:44 $
 * @brief    Use the EPWM0 to demonstrate continuous capture mode 
 *           function. And displays the measured input period to console.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"

uint32_t ccap_isr_flag, ccap_dat[4];
void CCAP_IRQHandler(void)
{
    ccap_isr_flag = TIMERAC->CCAPCTL & (TIMER_CAP_FLAG_R1F | TIMER_CAP_FLAG_F1F | \
                                            TIMER_CAP_FLAG_R2F | TIMER_CAP_FLAG_F2F);
    
    ccap_dat[0] = TIMERAC->CCAP[TIMER_CAP_DAT_R1F];
    ccap_dat[1] = TIMERAC->CCAP[TIMER_CAP_DAT_F1F];
    ccap_dat[2] = TIMERAC->CCAP[TIMER_CAP_DAT_R2F];
    ccap_dat[3] = TIMERAC->CCAP[TIMER_CAP_DAT_F2F];
    
    TIMERAC->CCAPCTL |= TIMER_CAP_FLAG_R1F | TIMER_CAP_FLAG_F1F | \
                          TIMER_CAP_FLAG_R2F | TIMER_CAP_FLAG_F2F;
}

void SYS_Init(void)
{
    int32_t i32TimeOutCnt = 2160000;

    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Enable 60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for HIRC clock ready */
    while((CLK->STATUS & CLK_STATUS_HIRCSTB_Msk) != CLK_STATUS_HIRCSTB_Msk) {
        if(i32TimeOutCnt-- <= 0) break;
    }

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Enable IP clock */
    CLK->APBCLK = CLK->APBCLK | (CLK_APBCLK_USCI2CKEN_Msk | CLK_APBCLK_TMR0CKEN_Msk);

    /* Select IP clock source */
    CLK->CLKSEL1 = CLK->CLKSEL1 | CLK_TMR0_SRC_HIRC;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Set GPC2 multi-function pin for CCAP */
    SYS->GPC_MFP = SYS->GPC_MFP & ~(SYS_GPC_MFP_PC2MFP_Msk) | SYS_GPC_MFP_PC2_CCAP_P1;
    
    /* Set GPC2 as input mode */
    PC->MODE = (PC->MODE & ~GPIO_MODE_MODE2_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE2_Pos);
    
    /* Set GPA0 multi-function pin for EPWM0 */
    SYS->GPA_MFP = (SYS->GPA_MFP & (~SYS_GPA_MFP_PA0MFP_Msk)) | SYS_GPA_MFP_PA0_EPWM_CH0;
    
    /* Lock protected registers */
    SYS_LockReg();
}

int main()
{
    uint8_t c;
    uint32_t ccap_type;
    
    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\nThis sample code demonstrate timer continuous capture mode.\n");
    printf("Please connect capture source or EPWM CH0 (PA.0) to CCAP_P1 (PC.2).\n");      
    
    printf("\nEPWM channel 0 (PA.0) output frequence is 100Hz and duty is 70%%.\n");
    printf("CCAP timer is select TIMER0, counter unit is 1us (HIRC = 60MHz).\n");
    
    /* Initial EPWM0, frequence 100Hz and duty 70% */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_EPWMCKEN_Msk;
    PA->MODE = (PA->MODE & ~(GPIO_MODE_MODE0_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE0_Pos);

    EPWM->CTL |= EPWM_CTL_CNTMODE_Msk;
    EPWM->CLKDIV = 0x04;
		EPWM->PERIOD = 0x927B;
    EPWM->CMPDAT[0] = 0x668A;
    EPWM->CTL |= EPWM_CTL_CNTEN0_Msk;
    
    /* Initial timer, prescal is 60, CCAP unit is _HIRC/60 (1us). */
    TIMER0->CTL = TIMER_PERIODIC_MODE | (59 << TIMER_CTL_PSC_Pos);
    TIMER0->CMP = 0xFFFFF;
    
    /* Select CCAP interrupt trigger type */
    printf("\nSelect CCAP interrupt trigger type\
            \n  1: Rising edge 1 and Falling edge 1\
            \n  2: Rising edge 1, Falling edge 1 and Rising edge 2\
            \n  3: Rising edge 1, Falling edge 1, Rising edge 2 and Falling edge 2\n");
    c = getchar();  
    
    if(c == '1') 
        ccap_type = TIMER_CCAP_INT_R1_F1;
    else if(c == '2') 
        ccap_type = TIMER_CCAP_INT_R2_F1;
    else 
        ccap_type = TIMER_CCAP_INT_R2_F2;

    printf("  %x\n", ccap_type >> TIMER_CCAPCTL_CCAPIEN_Pos);
            
    /* Initial CCAP, use Timer0 as capture timer and capture from CCAP_P1 */
    TIMERAC->CCAPCTL = (TIMER_CCAP_CNT_TIMER0 | TIMER_CCAP_CH_P1) | ccap_type;
    
    printf("Press any key to start capture.\n");
    getchar();
    
    /* CCAP start */
    TIMERAC->CCAPCTL |= TIMER_CCAPCTL_CCAPEN_Msk;
    
    /* Enable CCAP interrupt */
    NVIC_EnableIRQ(CCAP_IRQn);
    
    /* Start capture timer */
    TIMER0->CTL |= TIMER_CTL_CNTEN_Msk;

    /* Show capture info in ISR */
    while(1)
    {
        if(ccap_isr_flag)
        {       
            printf("\nCapture finish!\n");
            
            if(ccap_isr_flag & TIMER_CAP_FLAG_F1F)
                printf("  1'st rising edge to 1'st falling edge: %d us\n", \
                        (ccap_dat[1] > ccap_dat[0])?(ccap_dat[1] - ccap_dat[0]):(TIMER0->CMP + ccap_dat[1] - ccap_dat[0]));    
            
            if(ccap_isr_flag & TIMER_CAP_FLAG_R2F)
                printf("  1'st falling edge to 2'nd fising edge: %d us\n", \
                        (ccap_dat[2] > ccap_dat[1])?(ccap_dat[2] - ccap_dat[1]):(TIMER0->CMP + ccap_dat[2] - ccap_dat[1]));  
            
            if(ccap_isr_flag & TIMER_CAP_FLAG_F2F)
                printf("  2'nd rising edge to 2'nd falling edge: %d us\n", \
                        (ccap_dat[3] > ccap_dat[2])?(ccap_dat[3] - ccap_dat[2]):(TIMER0->CMP + ccap_dat[3] - ccap_dat[2])); 
            
            
            ccap_isr_flag = 0;
            printf("\nPress any key to start next capture.\n");
            getchar();                            
                
            /* Select CCAP interrupt trigger type */
            printf("\nSelect CCAP interrupt trigger type\
                    \n  1: Rising edge 1 and Falling edge 1\
                    \n  2: Rising edge 1, Falling edge 1 and Rising edge 2\
                    \n  3: Rising edge 1, Falling edge 1, Rising edge 2 and Falling edge 2\n");
            c = getchar();  
            
            if(c == '1') 
                ccap_type = TIMER_CCAP_INT_R1_F1;
            else if(c == '2') 
                ccap_type = TIMER_CCAP_INT_R2_F1;
            else 
                ccap_type = TIMER_CCAP_INT_R2_F2;

            printf("  %x\n", ccap_type >> TIMER_CCAPCTL_CCAPIEN_Pos);
            
            TIMERAC->CCAPCTL = ((TIMERAC->CCAPCTL & ~(TIMER_CCAPCTL_CCAPIEN_Msk)) | ccap_type) | TIMER_CCAPCTL_CCAPEN_Msk;
        }
    }

}

/*** (C) COPYRIGHT 2018 Nuvoton Technology Corp. ***/
