/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/07/16 13:36 $
 * @brief    Sample code for HDIV calculation feature.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void SYS_Init(void)
{
    int32_t i32TimeOutCnt = 2160000;

    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Enable 60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for 60MHz clock ready */
    while((CLK->STATUS & CLK_STATUS_HIRCSTB_Msk) != CLK_STATUS_HIRCSTB_Msk) {
        if(i32TimeOutCnt-- <= 0) break;
    }

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Lock protected registers */
    SYS_LockReg();

    /* Enable IP clock */
    /* CLK_EnableModuleClock(HDIV_MODULE); */
    CLK->AHBCLK |= CLK_AHBCLK_HDIVCKEN_Msk;
    /* Enable USCI2 IP clock */
    CLK->APBCLK |= CLK_APBCLK_USCI2CKEN_Msk;

}


void USCI_UART_Init(void)
{
    /*---------------------------------------------------------------------------------------------------------*/
    /* Init USCI                                                                                               */
    /*---------------------------------------------------------------------------------------------------------*/
    /* Reset USCI2 */
    SYS->IPRST1 |=  SYS_IPRST1_USCI2RST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_USCI2RST_Msk;

    /* Configure USCI2 as UART mode */
    UUART2->CTL = (2 << UUART_CTL_FUNMODE_Pos);                                 /* Set UART function mode */
    UUART2->LINECTL = UUART_WORD_LEN_8 | UUART_LINECTL_LSB_Msk;                 /* Set UART line configuration */
    UUART2->DATIN0 = (2 << UUART_DATIN0_EDGEDET_Pos);                           /* Set falling edge detection */
    UUART2->BRGEN = (25 << UUART_BRGEN_CLKDIV_Pos) | (9 << UUART_BRGEN_DSCNT_Pos) | (1 << UUART_BRGEN_PDSCNT_Pos); /* Set UART baud rate as 115200bps */
    UUART2->PROTCTL |= UUART_PROTCTL_PROTEN_Msk;                                /* Enable UART protocol */
}


int main()
{
    int32_t x, y, q0, q1, q2, r0, r1, r2;
    int i;

    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    USCI_UART_Init();

    /* printf("\n\nPDID 0x%08X\n", SYS_ReadPDID()); */    /* Display PDID */
    printf("\n\nPDID 0x%08X\n", (unsigned int)(SYS->PDID & SYS_PDID_PDID_Msk)); /* Display PDID */

    printf("CPU @ %dHz\n", SystemCoreClock);        /* Display System Core Clock */

    /*
     * This sample code will demonstrate how to divide two signed integer by HDIV engine.
     */
    printf("+-----------------------------------------+\n");
    printf("| NM1240 HDIV Sample Code                 |\n");
    printf("+-----------------------------------------+\n");

    x = 123456;
    y = 21;
    for (i=0; i<5; i++)
    {
        /* q0 = HDIV0_Div(x, y); */
        HDIV0->DIVIDEND = x;
        HDIV0->DIVISOR = y;      /* HDIV begin to calculate here. */
        q0 = HDIV0->QUOTIENT;
        /* r0 = HDIV0_Mod(x, y); */
        // HDIV0->DIVIDEND = x;
        // HDIV0->DIVISOR = y;      /* HDIV begin to calculate here. */
        r0 = HDIV0->REM;

        /* q1 = HDIV1_Div(x, y); */
        HDIV1->DIVIDEND = x;
        HDIV1->DIVISOR = y;      /* HDIV begin to calculate here. */
        q1 = HDIV1->QUOTIENT;
        /* r1 = HDIV1_Mod(x, y); */
        // HDIV1->DIVIDEND = x;
        // HDIV1->DIVISOR = y;      /* HDIV begin to calculate here. */
        r1 = HDIV1->REM;
			
        /* q2 = HDIV2_Div(x, y); */
        HDIV2->DIVIDEND = x;
        HDIV2->DIVISOR = y;      /* HDIV begin to calculate here. */
        q2 = HDIV2->QUOTIENT;
        /* r2 = HDIV2_Mod(x, y); */
        // HDIV2->DIVIDEND = x;
        // HDIV2->DIVISOR = y;      /* HDIV begin to calculate here. */
        r2 = HDIV2->REM;
		
		printf("Software calculation : %d / %d = %d ... %d\n", x, y, x / y, x % y);	
		
        /* if (! HDIV_IS_DIVBYZERO0()) */
        if (! ((HDIV_STS->STATUS & HDIV_STATUS_DIVBYZERO0_Msk) >> HDIV_STATUS_DIVBYZERO0_Pos))
        {
            printf("HDIV0 calculation    : %d / %d = %d ... %d\n", x, y, q0, r0);
            
            if ((q0 == x / y) && (r0 == x % y))
                printf("==> HDIV0 Division PASS !!!\n");
            else
                printf("==> HDIV0 Division FAIL !!!\n");
        }
        else
        {
            printf("HDIV0 calculation : %d / %d, divide by 0 !!!\n", x, y);
            printf("==> Divide by zero checking PASS !!!\n");
        }
		
		/* if (! HDIV_IS_DIVBYZERO1()) */
        if (! ((HDIV_STS->STATUS & HDIV_STATUS_DIVBYZERO1_Msk) >> HDIV_STATUS_DIVBYZERO1_Pos))
        {
            printf("HDIV1 calculation    : %d / %d = %d ... %d\n", x, y, q1, r1);
			
            if ((q1 == x / y) && (r1 == x % y))
                printf("==> HDIV1 Division PASS !!!\n");
            else
                printf("==> HDIV1 Division FAIL !!!\n");
        }
        else
        {
            printf("HDIV1 calculation : %d / %d, divide by 0 !!!\n", x, y);
            printf("==> Divide by zero checking PASS !!!\n");
        }

		/* if (! HDIV_IS_DIVBYZERO2()) */
        if (! ((HDIV_STS->STATUS & HDIV_STATUS_DIVBYZERO2_Msk) >> HDIV_STATUS_DIVBYZERO2_Pos))
        {
            printf("HDIV2 calculation    : %d / %d = %d ... %d\n", x, y, q2, r2);
			
            if ((q2 == x / y) && (r2 == x % y))
                printf("==> HDIV2 Division PASS !!!\n");
            else
                printf("==> HDIV2 Division FAIL !!!\n");
        }
        else
        {
            printf("HDIV2 calculation : %d / %d, divide by 0 !!!\n", x, y);
            printf("==> Divide by zero checking PASS !!!\n");
        }
		
        y -= 7;
        printf("\n");
    }

    /* CLK_DisableModuleClock(HDIV_MODULE); */
    CLK->AHBCLK &= ~CLK_AHBCLK_HDIVCKEN_Msk;
    
    printf("=== THE END ===\n\n");
    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
