/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 2 $
 * $Date: 2018/10/05 15:00 $
 * @brief    Sample code for GPIO Power-down Wake-up feature.
 * @note
 * Copyright (C) 2018 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


/**
 *    @brief        Open and set USCI_UART function
 *
 *    @param[in]    uuart           The pointer of the specified USCI_UART module.
 *    @param[in]    u32baudrate     The baud rate of USCI_UART module.
 *
 *    @return       Real baud rate of USCI_UART module.
 *
 *    @details      This function use to enable USCI_UART function and set baud-rate.
 */
uint32_t UUART_Open(UUART_T* uuart, uint32_t u32baudrate)
{
    uint32_t u32PCLKFreq, u32PDSCnt, u32DSCnt, u32ClkDiv;
    uint32_t u32Tmp, u32Tmp2, u32Min, u32MinClkDiv, u32MinDSCnt;

    uint32_t u32Div;

    /* Get PCLK frequency */
    u32PCLKFreq = CLK_GetPCLKFreq();

    u32Div = u32PCLKFreq / u32baudrate;
    u32Tmp = (u32PCLKFreq / u32Div) - u32baudrate;
    u32Tmp2 = u32baudrate - (u32PCLKFreq / (u32Div+1));

    if(u32Tmp >= u32Tmp2) u32Div = u32Div + 1;

    u32Tmp = 0x400 * 0x10;
    for(u32PDSCnt = 1; u32PDSCnt <= 0x04; u32PDSCnt++) {
        if(u32Div <= (u32Tmp * u32PDSCnt)) break;
    }

    if(u32PDSCnt > 0x4) u32PDSCnt = 0x4;

    u32Div = u32Div / u32PDSCnt;

    /* Find best solution */
    u32Min = (uint32_t) - 1;
    u32MinDSCnt = 0;
    u32MinClkDiv = 0;

    u32Tmp = 0;

    for(u32DSCnt = 6; u32DSCnt <= 0x10; u32DSCnt++) { /* DSCNT could be 0x5~0xF */

        u32ClkDiv = u32Div / u32DSCnt;

        if(u32ClkDiv > 0x400) {
            u32ClkDiv = 0x400;
            u32Tmp = u32Div - (u32ClkDiv * u32DSCnt);
            u32Tmp2 = u32Tmp + 1;
        } else {
            u32Tmp = u32Div - (u32ClkDiv * u32DSCnt);
            u32Tmp2 = ((u32ClkDiv+1) * u32DSCnt) - u32Div;
        }

        if(u32Tmp >= u32Tmp2) {
            u32ClkDiv = u32ClkDiv + 1;
        } else u32Tmp2 = u32Tmp;

        if(u32Tmp2 < u32Min) {
            u32Min = u32Tmp2;
            u32MinDSCnt = u32DSCnt;
            u32MinClkDiv = u32ClkDiv;

            /* Break when get good results */
            if(u32Min == 0) {
                break;
            }
        }
    }

    /* Enable USCI_UART protocol */
    uuart->CTL &= ~UUART_CTL_FUNMODE_Msk;
    uuart->CTL = 2 << UUART_CTL_FUNMODE_Pos;

    /* Set USCI_UART line configuration */
    uuart->LINECTL = UUART_WORD_LEN_8 | UUART_LINECTL_LSB_Msk;
    uuart->DATIN0 = (2 << UUART_DATIN0_EDGEDET_Pos);  /* Set falling edge detection */

    /* Set USCI_UART baud rate */
    uuart->BRGEN = ((u32MinClkDiv-1) << UUART_BRGEN_CLKDIV_Pos) |
                  ((u32MinDSCnt-1) << UUART_BRGEN_DSCNT_Pos) |
                  ((u32PDSCnt-1) << UUART_BRGEN_PDSCNT_Pos);

    uuart->PROTCTL |= UUART_PROTCTL_PROTEN_Msk;

    return (u32PCLKFreq/u32PDSCnt/u32MinDSCnt/u32MinClkDiv);
}



void SYS_Init(void)
{
    /* Unlock protected registers */
    UNLOCKREG();

     /* Enable 60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for 60MHz clock ready */
    CLK_WaitClockReady(CLK_STATUS_HIRCSTB_Msk);

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Enable USCI0 IP clock */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Lock protected registers */
    UNLOCKREG();
}


/**
 * @brief       PortA/PortB/PortC/PortD IRQ
 *
 * @param       None
 *
 * @return      None
 *
 * @details     The PortA/PortB/PortC/PortD default IRQ, declared in startup_NM1240.s.
 */
void GPIO_IRQHandler(void)
{
    /* if(GPIO_GET_INT_FLAG(PB, BIT0)) */     /* To check if PB.0 interrupt occurred */
    if(PB->INTSRC & BIT0)
    {
        /* GPIO_CLR_INT_FLAG(PB, BIT0); */    /* Clear PB.0 interrupt flag */
        PB->INTSRC = BIT0;
        /* printf("PB.0 INT occurred. \n"); */
    }
    else
    {    /* Un-expected interrupt. Just clear all PORTA, PORTB, PORTC, PORTD interrupts */
        /* GPIO_CLR_INT_FLAG(PA, GPIO_GET_INT_FLAG(PA, 0x3F)); */
        PA->INTSRC = (PA->INTSRC & 0xFF);

        /* GPIO_CLR_INT_FLAG(PB, GPIO_GET_INT_FLAG(PB, 0x1F)); */
        PB->INTSRC = (PB->INTSRC & 0xFF);

        /* GPIO_CLR_INT_FLAG(PC, GPIO_GET_INT_FLAG(PC, 0x1F)); */
        PC->INTSRC = (PC->INTSRC & 0xFF);

        /* GPIO_CLR_INT_FLAG(PD, GPIO_GET_INT_FLAG(PD, 0x7F)); */
        PD->INTSRC = (PD->INTSRC & 0xFE);

        /* GPIO_CLR_INT_FLAG(PE, GPIO_GET_INT_FLAG(PE, 0x1F)); */
        PE->INTSRC = (PE->INTSRC & 0xFF);

        /* GPIO_CLR_INT_FLAG(PF, GPIO_GET_INT_FLAG(PF, 0x1F)); */
        PF->INTSRC = (PF->INTSRC & 0x1F);

        /* printf("Un-expected interrupts. \n"); */
    }
}


int main()
{
    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    /* printf("\n\nPDID 0x%08X\n", SYS_ReadPDID()); */    /* Display PDID */
    printf("\n\nPDID 0x%08X\n", (unsigned int)(SYS->PDID & SYS_PDID_PDID_Msk)); /* Display PDID */

    printf("CPU @ %dHz\n", SystemCoreClock);        /* Display System Core Clock */

    /*
     * This sample code will demonstrate how to wake up system form Power-down mode
     * by GPIO interrupt.
     */
    printf("+--------------------------------------------+\n");
    printf("| NM1240 GPIO Power Down Wake Up Sample Code |\n");
    printf("+--------------------------------------------+\n");

    printf("Please keep PB.0 low and use rising edge to wake-up system ...\n");

    /* Config multiple function to GPIO mode for PB0 */
    SYS->GPB_MFP = (SYS->GPB_MFP & ~SYS_GPB_MFP_PB0MFP_Msk) | SYS_GPB_MFP_PB0_GPIO;

    /* GPIO_SetMode(PB, BIT0, GPIO_PMD_INPUT); */
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE0_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE0_Pos);

    NVIC_EnableIRQ(GP_IRQn);                    /* Enable GPIO NVIC */

    /* GPIO_EnableInt(PB, 0, GPIO_INT_RISING); */     /* Enable PB0 interrupt by rising edge trigger */
    PB->INTEN = (PB->INTEN & ~(GPIO_INT_BOTH_EDGE << 0)) | (GPIO_INT_RISING << 0);
    PB->INTTYPE = (PB->INTTYPE & ~(BIT0 << 0)) | (GPIO_INTTYPE_EDGE << 0);

    /* Unlock protected registers */
    UNLOCKREG();

    /* Waiting for PB.0 rising-edge interrupt event */
    printf("Wait PB.0 to low\n");
    while(PB0 == 1);    /* wait PB.0 become low before get into power down mode */
    printf("Enter to Power-Down (rising-edge) ......  PB.0 = %d \n", PB0);

    /* UUART_WAIT_TX_EMPTY(UUART2); */  /* To check if all the debug messages are finished */
    //while(!((UUART2->BUFSTS & UUART_BUFSTS_TXEMPTY_Msk) >> UUART_BUFSTS_TXEMPTY_Pos))
	 while(!((((UUART2)->BUFSTS) & UUART_BUFSTS_TXEMPTY_Msk) >> UUART_BUFSTS_TXEMPTY_Pos)) {;}

    /* Enter Power-down mode */
    CLK_PowerDown();

	 CLK_SysTickDelay(1000);
    printf("System waken-up done. PB.0 = %d\n\n", PB0);
    LOCKREG();
    printf("=== THE END ===\n\n");
    while(1);
}

/*** (C) COPYRIGHT 2018 Nuvoton Technology Corp. ***/
