/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/07/24 10:49 $
 * @brief    Sample code for GPIO I/O feature.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


/**
  * @brief      This function check selected clock source status
  * @param[in]  u32ClkMask is selected clock source. Including
  *             - \ref CLK_STATUS_CLKSFAIL_Msk
  *             - \ref CLK_STATUS_HIRCSTB_Msk
  *             - \ref CLK_STATUS_LIRCSTB_Msk
  *             - \ref CLK_STATUS_XTLSTB_Msk
  * @return     0  clock is not stable
  *             1  clock is stable
  *
  * @details    To wait for clock ready by specified CLKSTATUS bit or timeout (~5ms)
  */
uint32_t CLK_WaitClockReady(uint32_t u32ClkMask)
{
    int32_t i32TimeOutCnt=2160000;

    while((CLK->STATUS & u32ClkMask) != u32ClkMask) {
        if(i32TimeOutCnt-- <= 0)
            return 0;
    }
    return 1;
}

/**
  * @brief      This function get PCLK frequency. The frequency unit is Hz.
  * @param      None
  * @return     PCLK frequency
  * @details    This function get PCLK frequency. The frequency unit is Hz.
  *             The PCLK is always same to HCLK in NM1240.
  */
uint32_t CLK_GetPCLKFreq(void)
{
    SystemCoreClockUpdate();
    return SystemCoreClock;
}



/**
 *    @brief        Open and set USCI_UART function
 *
 *    @param[in]    uuart           The pointer of the specified USCI_UART module.
 *    @param[in]    u32baudrate     The baud rate of USCI_UART module.
 *
 *    @return       Real baud rate of USCI_UART module.
 *
 *    @details      This function use to enable USCI_UART function and set baud-rate.
 */
uint32_t UUART_Open(UUART_T* uuart, uint32_t u32baudrate)
{
    uint32_t u32PCLKFreq, u32PDSCnt, u32DSCnt, u32ClkDiv;
    uint32_t u32Tmp, u32Tmp2, u32Min, u32MinClkDiv, u32MinDSCnt;

    uint32_t u32Div;

    /* Get PCLK frequency */
    u32PCLKFreq = CLK_GetPCLKFreq();

    u32Div = u32PCLKFreq / u32baudrate;
    u32Tmp = (u32PCLKFreq / u32Div) - u32baudrate;
    u32Tmp2 = u32baudrate - (u32PCLKFreq / (u32Div+1));

    if(u32Tmp >= u32Tmp2) u32Div = u32Div + 1;

    u32Tmp = 0x400 * 0x10;
    for(u32PDSCnt = 1; u32PDSCnt <= 0x04; u32PDSCnt++) {
        if(u32Div <= (u32Tmp * u32PDSCnt)) break;
    }

    if(u32PDSCnt > 0x4) u32PDSCnt = 0x4;

    u32Div = u32Div / u32PDSCnt;

    /* Find best solution */
    u32Min = (uint32_t) - 1;
    u32MinDSCnt = 0;
    u32MinClkDiv = 0;

    u32Tmp = 0;

    for(u32DSCnt = 6; u32DSCnt <= 0x10; u32DSCnt++) { /* DSCNT could be 0x5~0xF */

        u32ClkDiv = u32Div / u32DSCnt;

        if(u32ClkDiv > 0x400) {
            u32ClkDiv = 0x400;
            u32Tmp = u32Div - (u32ClkDiv * u32DSCnt);
            u32Tmp2 = u32Tmp + 1;
        } else {
            u32Tmp = u32Div - (u32ClkDiv * u32DSCnt);
            u32Tmp2 = ((u32ClkDiv+1) * u32DSCnt) - u32Div;
        }

        if(u32Tmp >= u32Tmp2) {
            u32ClkDiv = u32ClkDiv + 1;
        } else u32Tmp2 = u32Tmp;

        if(u32Tmp2 < u32Min) {
            u32Min = u32Tmp2;
            u32MinDSCnt = u32DSCnt;
            u32MinClkDiv = u32ClkDiv;

            /* Break when get good results */
            if(u32Min == 0) {
                break;
            }
        }
    }

    /* Enable USCI_UART protocol */
    uuart->CTL &= ~UUART_CTL_FUNMODE_Msk;
    uuart->CTL = 2 << UUART_CTL_FUNMODE_Pos;

    /* Set USCI_UART line configuration */
    uuart->LINECTL = UUART_WORD_LEN_8 | UUART_LINECTL_LSB_Msk;
    uuart->DATIN0 = (2 << UUART_DATIN0_EDGEDET_Pos);  /* Set falling edge detection */

    /* Set USCI_UART baud rate */
    uuart->BRGEN = ((u32MinClkDiv-1) << UUART_BRGEN_CLKDIV_Pos) |
                  ((u32MinDSCnt-1) << UUART_BRGEN_DSCNT_Pos) |
                  ((u32PDSCnt-1) << UUART_BRGEN_PDSCNT_Pos);

    uuart->PROTCTL |= UUART_PROTCTL_PROTEN_Msk;

    return (u32PCLKFreq/u32PDSCnt/u32MinDSCnt/u32MinClkDiv);
}


/**
  * @brief      This function execute delay function.
  * @param[in]  us      Delay time. The Max value is 2^24 / CPU Clock(MHz). Ex:
  *                     50MHz => 335544us, 48MHz => 349525us, 28MHz => 699050us ...
  * @return     None
  * @details    Use the SysTick to generate the delay time and the UNIT is in us.
  *             The SysTick clock source is from HCLK, i.e the same as system core clock.
  */
void CLK_SysTickDelay(uint32_t us)
{
    uint32_t delay_tick;

    delay_tick = us * CyclesPerUs;
    if (delay_tick > SysTick_LOAD_RELOAD_Msk)   /* SysTick_LOAD_RELOAD_Msk is 24 bits for NM1240 */
    {
        printf("ERROR: CLK_SysTickDelay(): the delay tick (%d) cannot > %lu !\n", us, SysTick_LOAD_RELOAD_Msk/CyclesPerUs);
        return;
    }
    SysTick->LOAD = delay_tick;
    SysTick->VAL  = (0x00);
    SysTick->CTRL = SysTick_CTRL_CLKSOURCE_Msk | SysTick_CTRL_ENABLE_Msk;

    /* Waiting for down-count to zero */
    while((SysTick->CTRL & SysTick_CTRL_COUNTFLAG_Msk) == 0);
    SysTick->CTRL = 0;
} 



void SYS_Init(void)
{
    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Enable 60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for 60MHz clock ready */
    CLK_WaitClockReady(CLK_STATUS_HIRCSTB_Msk);

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Enable USCI0 IP clock */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();
    
    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);


    /* Lock protected registers */
    SYS_LockReg();
}


/**
 * @brief       PortA/PortB/PortC/PortD IRQ
 *
 * @param       None
 *
 * @return      None
 *
 * @details     The PortA/PortB/PortC/PortD default IRQ, declared in startup_NM1240.s.
 */
uint32_t PAIntFlag=0, PBIntFlag=0, PCIntFlag=0, PDIntFlag=0, PEIntFlag=0, PFIntFlag=0;
void GPIO_IRQHandler(void)
{
    /* PAIntFlag = GPIO_GET_INT_FLAG(PA, 0xF); */
    PAIntFlag = PA->INTSRC & 0xF;

    /* PBIntFlag = GPIO_GET_INT_FLAG(PB, 0xF); */
    PBIntFlag = PB->INTSRC & 0xF;

    /* PCIntFlag = GPIO_GET_INT_FLAG(PC, 0xF); */
    PCIntFlag = PC->INTSRC & 0xF;

    /* PDIntFlag = GPIO_GET_INT_FLAG(PD, 0xF); */
    PDIntFlag = PD->INTSRC & 0xF;

    /* PEIntFlag = GPIO_GET_INT_FLAG(PE, 0xF); */
    PEIntFlag = PE->INTSRC & 0xF;

    /* PFIntFlag = GPIO_GET_INT_FLAG(PF, 0xF); */
    PFIntFlag = PF->INTSRC & 0xF;

    /* clear GPIO interrupt flag */
    /* GPIO_CLR_INT_FLAG(PA, PAIntFlag); */
    PA->INTSRC = PAIntFlag;
    
    /* GPIO_CLR_INT_FLAG(PB, PBIntFlag); */
    PB->INTSRC = PBIntFlag;

    /* GPIO_CLR_INT_FLAG(PC, PCIntFlag); */
    PC->INTSRC = PCIntFlag;

    /* GPIO_CLR_INT_FLAG(PD, PDIntFlag); */
    PD->INTSRC = PDIntFlag;

    /* GPIO_CLR_INT_FLAG(PE, PEIntFlag); */
    PE->INTSRC = PEIntFlag;

    /* GPIO_CLR_INT_FLAG(PF, PFIntFlag); */
    PF->INTSRC = PFIntFlag;

    /*
    printf("GPIO_IRQHandler() ...\n");
    printf("Interrupt Flag PA=0x%08X, PB=0x%08X, PC=0x%08X, PD=0x%08X\n",
        PAIntFlag, PBIntFlag, PCIntFlag, PDIntFlag);
    */
}


int main()
{
    int32_t i32Err;

    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    /* printf("\n\nPDID 0x%08X\n", SYS_ReadPDID()); */    /* Display PDID */
    printf("\n\nPDID 0x%08X\n", (unsigned int)(SYS->PDID & SYS_PDID_PDID_Msk)); /* Display PDID */

    printf("CPU @ %dHz\n", SystemCoreClock);        /* Display System Core Clock */

    /*
     * This sample code will use GPIO driver to control the GPIO pin direction and
     * the high/low state, and show how to use GPIO interrupts.
     */
    printf("+-----------------------------------------+\n");
    printf("| NM1240 GPIO Driver Sample Code          |\n");
    printf("+-----------------------------------------+\n");

    /*-----------------------------------------------------------------------------------------------------*/
    /* GPIO Basic Mode Test --- Use Pin Data Input/Output to control GPIO pin                              */
    /*-----------------------------------------------------------------------------------------------------*/
    printf("  >> Please connect PB.0 and PC.2 first << \n");
    printf("     Press any key to start test by using [Pin Data Input/Output Control] \n\n");
    getchar();

    /* Config multiple function to GPIO mode for PB0 and PC2 */
    SYS->GPB_MFP = (SYS->GPB_MFP & ~SYS_GPB_MFP_PB0MFP_Msk) | SYS_GPB_MFP_PB0_GPIO;
    SYS->GPC_MFP = (SYS->GPC_MFP & ~SYS_GPC_MFP_PC2MFP_Msk) | SYS_GPC_MFP_PC2_GPIO;

    /* GPIO_SetMode(PB, BIT0, GPIO_PMD_OUTPUT); */
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE0_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE0_Pos);

    /* GPIO_SetMode(PC, BIT2, GPIO_PMD_INPUT); */
    PC->MODE = (PC->MODE & ~GPIO_MODE_MODE2_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE2_Pos);

    /* GPIO_EnableInt(PC, 2, GPIO_INT_RISING); */     /* Enable PC2 interrupt by rising edge trigger */
    PC->INTEN = (PC->INTEN & ~(GPIO_INT_BOTH_EDGE << 2)) | (GPIO_INT_RISING << 2);
    PC->INTTYPE = (PC->INTTYPE & ~(BIT0 << 2)) | (GPIO_INTTYPE_EDGE << 2);

    NVIC_EnableIRQ(GP_IRQn);                    /* Enable GPIO NVIC */

    PAIntFlag = 0;
    PBIntFlag = 0;
    PCIntFlag = 0;
    PDIntFlag = 0;
    PEIntFlag = 0;
    PFIntFlag = 0;
    i32Err = 0;
    printf("  GPIO Output/Input test ...... \n");

    /* Use Pin Data Input/Output Control to pull specified I/O or get I/O pin status */
    PB0 = 0;                /* Output low */
    CLK_SysTickDelay(10);   /* wait for IO stable */
    if (PC2 != 0) {         /* check if the PB0 state is low */
        i32Err = 1;
    }

    PB0 = 1;                /* Output high */
    CLK_SysTickDelay(10);   /* wait for IO stable */
    if (PC2 != 1) {         /* check if the PB0 state is high */
        i32Err = 1;
    }

    /* show the result */
    if ( i32Err ) {
        printf("  [FAIL] --- Please make sure PB.0 and PC.2 are connected. \n");
    } else {
        printf("  [OK] \n");
    }

    printf("  Check Interrupt Flag PA=0x%08X, PB=0x%08X, PC=0x%08X, PD=0x%08X, PE=0x%08X, PF=0x%08X\n",
        PAIntFlag, PBIntFlag, PCIntFlag, PDIntFlag, PEIntFlag, PFIntFlag);

    /* GPIO_SetMode(PB, BIT0, GPIO_PMD_INPUT); */     /* Configure PB.0 to default Input mode */
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE0_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE0_Pos);

    /* GPIO_SetMode(PC, BIT2, GPIO_PMD_INPUT); */     /* Configure PC.2 to default Input mode */
    PC->MODE = (PC->MODE & ~GPIO_MODE_MODE2_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE2_Pos);

    printf("=== THE END ===\n\n");
    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
