/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/08/07 14:21p $
 * @brief   NM1240 GDMA UART test
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"

volatile uint8_t g_u8EndFlagTX, g_u8EndFlagRX;

void Parameter_Init(void)
{
    g_u8EndFlagTX = 0;
    g_u8EndFlagRX = 0;
}

void GDMA0_IRQHandler (void)
{
    /* Clear TCIF */
    GDMA0->CTL &= ~GDMA_CTL_TCIF_Msk;
    /* Set GDMA0_UART_RX Flag */
    g_u8EndFlagRX = 1;
}

void GDMA1_IRQHandler (void)
{
    /* Clear TCIF */
    GDMA1->CTL &= ~GDMA_CTL_TCIF_Msk;
    /* Set GDMA1_UART_TX Flag */
    g_u8EndFlagTX = 1;
}

void SYS_Init(void)
{    
    /*---------------------------------------------------------------------------------------------------------*/
    /* Init System Clock                                                                                       */
    /*---------------------------------------------------------------------------------------------------------*/
    int32_t i32TimeOutCnt = 2160000;
    
    /* Enable 60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for HIRC clock ready */
    while((CLK->STATUS & CLK_STATUS_HIRCSTB_Msk) != CLK_STATUS_HIRCSTB_Msk) {
        if(i32TimeOutCnt-- <= 0) break;
    }

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate PllClock, SystemCoreClock and CycylesPerUs automatically. */
    SystemCoreClockUpdate();
}

void UART2_Init(void)
{
    /* Enable USCI2 CLK */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;
    
    /* Reset IP USCI2 */
    SYS->IPRST1 |= SYS_IPRST1_USCI2RST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_USCI2RST_Msk;
    
    /* Open USCI2: 115200 */
    /* Enable USCI_UART protocol */
    UUART2->CTL &= ~UUART_CTL_FUNMODE_Msk;
    UUART2->CTL = 2 << UUART_CTL_FUNMODE_Pos;
    
    /* Set USCI_UART line configuration */
    UUART2->LINECTL = UUART_WORD_LEN_8 | UUART_LINECTL_LSB_Msk;
    UUART2->DATIN0 = (2 << UUART_DATIN0_EDGEDET_Pos);  /* Set falling edge detection */
    
    /* Set USCI_UART baud rate */
    UUART2->BRGEN = (86 << UUART_BRGEN_CLKDIV_Pos) |
                    (5 << UUART_BRGEN_DSCNT_Pos) | // DSCNT could be 0x5~0xF
                    (0 << UUART_BRGEN_PDSCNT_Pos);
    
    UUART2->PROTCTL |= UUART_PROTCTL_PROTEN_Msk;
    
    /* Set USCI_UART2 multi-function pins */
    SYS->GPF_MFP &= ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk);
    SYS->GPF_MFP |= ( SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);
    
    /* Set GPIO Mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);
}

int main()
{
    /*
        This sample code uses GDMA transfer and sets USCI_UART baud rate to 115200.
        Then, the user press any 10 keys.
        Check if the Enter data is equal to the Print data.
        PF1: UART2_TX
        PF2: UART2_RX
    */
    uint8_t Rx_Buffer[10], Tx_Buffer[10], i, c;
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS_Init();         /* Initial system clock       */
    UART2_Init();       /* Initial UART2              */
    SYS_LockReg();      /* Lock protected registers   */
    printf("\n\n");
    printf("+------------------------------------------------+\n");
    printf("|      NM1240 Sample Code for GDMA               |\n");
    printf("+------------------------------------------------+\n");
    printf("Initial NM1240 System Clock:\n");
    printf("CPU clock %dMHz\n",  SystemCoreClock/1000000);
    printf("HCLK clock %dMHz\n", SystemCoreClock/1000000);  
    printf("EXT clock %luMHz (0MHz means no clock source)\n", ( ((CLK->PWRCTL & CLK_PWRCTL_EXTCLKEN_Msk) >> CLK_PWRCTL_EXTCLKEN_Pos) == 0x03 )? __EXT/1000000 : 0/1000000);
    printf("Reset status: 0x%08X\n", SYS->RSTSTS);    
    printf("\n");
    printf("+----------------------------------------------------------+\n");
    printf("|                    UART Configuration                    |\n");
    printf("+----------------------------------------------------------+\n");
    printf("| Transfer width : 8 bits                                  |\n");
    printf("| Baud Rate      : 115200                                  |\n");
    printf("+----------------------------------------------------------+\n");
    printf("|                    GDMA Configuration                    |\n");
    printf("+----------------------------------------------------------+\n");
    printf("| MODE           : USCI to Memory(UART)                    |\n");
    printf("| Transfer width : 8 BITS                                  |\n");
    printf("| Transfer count : Receive and Transmit 10 data            |\n");
    printf("| Burst mode     : DISABLE                                 |\n");
    printf("+----------------------------------------------------------+\n");
    
    /* Initial Parameter */
    Parameter_Init();
    
    /* Reset GDMA */
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS->IPRST0 |= SYS_IPRST0_GDMARST_Msk;
    SYS_LockReg();      /* Lock protected registers */
    
    /* Open GDMA0 for UART RX */
{
    /* Enable GDMA controller clock */
    CLK->AHBCLK |= CLK_AHBCLK_GDMACKEN_Msk;
    
    /* Select transfer width */
    GDMA0->CTL = (GDMA0->CTL & ~(GDMA_CTL_TWS_Msk | GDMA_CTL_BME_Msk)) | (GDMA_TWS_8BITS << GDMA_CTL_TWS_Pos);
    
    /* Enable GDMA and set mode */
    GDMA0->CTL = (GDMA0->CTL & ~(GDMA_CTL_GDMAMS_Msk)) | (GDMA_USCI_MODE << GDMA_CTL_GDMAMS_Pos);
      
    /* Set address direction or fixed */
    GDMA0->CTL = (GDMA0->CTL & ~(0xF << GDMA_CTL_DADIR_Pos)) | ((SOURCE_ADDRESS_FIXED | Destination_ADDRESS_INC) << GDMA_CTL_DADIR_Pos);
    
    /* Enable GDMA0 External Interrupt */
    NVIC_EnableIRQ(GDMA0_IRQn);
    
    /* Set transfer count */
    GDMA0->TCNT = 10;
    
    /* Set destination base address */
    GDMA0->DSTB = (uint32_t)&Rx_Buffer[0];
    
    /* Set source base address */
    GDMA0->SRCB = (uint32_t)&UUART2->RXDAT;
}
    /* Open GDMA1 for UART TX */
{
    /* Select transfer width */
    GDMA1->CTL = (GDMA1->CTL & ~(GDMA_CTL_TWS_Msk | GDMA_CTL_BME_Msk)) | (GDMA_TWS_8BITS << GDMA_CTL_TWS_Pos);
    
    /* Enable GDMA and set mode */
    GDMA1->CTL = (GDMA1->CTL & ~(GDMA_CTL_GDMAMS_Msk)) | (GDMA_USCI_MODE << GDMA_CTL_GDMAMS_Pos);
      
    /* Set address direction or fixed */
    GDMA1->CTL = (GDMA1->CTL & ~(0xF << GDMA_CTL_DADIR_Pos)) |
                 ((SOURCE_ADDRESS_INC | Destination_ADDRESS_FIXED) << GDMA_CTL_DADIR_Pos);
    
    /* Enable GDMA1 External Interrupt */
    NVIC_EnableIRQ(GDMA1_IRQn);
    
    /* Set transfer count */
    GDMA1->TCNT = 10;
    
    /* Set destination base address */
    GDMA1->DSTB = (uint32_t)&UUART2->TXDAT;
    
    /* Set source base address */
    GDMA1->SRCB = (uint32_t)&Tx_Buffer[0];
}    
    /* Reset UART DMA CTL */
    UUART2->DMACTL |= UUART_DMACTL_DMARST_Msk;
    /* Set UART DMA CTL */
    UUART2->DMACTL |= UUART_DMACTL_DMAEN_Msk | UUART_DMACTL_RXDMAEN_Msk | UUART_DMACTL_TXDMAEN_Msk;    
    
    printf("UART receive and transfer data by GDMA\n");
    printf("Enter -> RXDAT -> Rx_Buffer -> Tx_Buffer -> TXDAT -> Print\n");
    printf("Check the input keys and output letters are the same or not?\n");
    printf("Press any 10 keys\n");
    printf("Enter:");
    
    /* Check UART RX is empty */
    while(!(UUART2->BUFSTS & UUART_BUFSTS_RXEMPTY_Msk));
    
    /* Enable GDMA0 INT */
    GDMA0->CTL |= GDMA_CTL_GIEN_Msk;
    
    /* Enable GDMA0 */
    GDMA_ENABLE(GDMA0);

    for (i=0; i<10; i++){
        DEBUG_PORT->PROTSTS = (UUART_PROTSTS_RXENDIF_Msk);
        while (1) {
            if(DEBUG_PORT->PROTSTS & UUART_PROTSTS_RXENDIF_Msk) {
                DEBUG_PORT->PROTSTS = (UUART_PROTSTS_RXENDIF_Msk);
                c = DEBUG_PORT->RXDAT;
                break;
            }
        }
        printf("%c",c);
    }
    for (i=0; i<10; i++){
        Tx_Buffer[i] = Rx_Buffer[i];
    }
    printf("\nPrint:");
    
    /* Check the transmission of GDMA0_UART_RX is complete. */
    while(!g_u8EndFlagRX);
    g_u8EndFlagRX = 0;
    
    /* Disable GDMA0 INT */
    GDMA0->CTL &= ~GDMA_CTL_GIEN_Msk;
    
    /* Check UART TX is not full */
    while(UUART2->BUFSTS & UUART_BUFSTS_TXFULL_Msk);
    
    /* Enable GDMA1 INT */
    GDMA1->CTL |= GDMA_CTL_GIEN_Msk;
    
    /* Enable GDMA1 */
    GDMA_ENABLE(GDMA1);

    /* Check the transmission of GDMA0_UART_RX is complete. */
    while(!g_u8EndFlagTX);
    g_u8EndFlagTX = 0;
    
    /* Disable GDMA1 INT */
    GDMA1->CTL &= ~GDMA_CTL_GIEN_Msk;
    
    printf("\nGDMA transfer finish!\n");
    while(1);  // end of while(1)
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
