/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/08/07 15:47p $
 * @brief   NM1240 GDMA SPI Slave test
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"

volatile uint8_t g_u8EndFlagTX, g_u8EndFlagRX;

void Parameter_Init(void)
{
    g_u8EndFlagTX = 0;
    g_u8EndFlagRX = 0;
}

void GDMA0_IRQHandler (void)
{
    /* Clear TCIF */
    GDMA0->CTL &= ~GDMA_CTL_TCIF_Msk;
    
    /* Set GDMA0_I2C_RX Flag */
    g_u8EndFlagRX = 1;
}

void GDMA1_IRQHandler (void)
{
    /* Clear TCIF */
    GDMA1->CTL &= ~GDMA_CTL_TCIF_Msk;
    
    /* Set GDMA1_I2C_TX Flag */
    g_u8EndFlagTX = 1;
}

void SYS_Init(void)
{    
    /*---------------------------------------------------------------------------------------------------------*/
    /* Init System Clock                                                                                       */
    /*---------------------------------------------------------------------------------------------------------*/
    int32_t i32TimeOutCnt = 2160000;
  
    /* Enable 60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for HIRC clock ready */
    while((CLK->STATUS & CLK_STATUS_HIRCSTB_Msk) != CLK_STATUS_HIRCSTB_Msk) {
        if(i32TimeOutCnt-- <= 0) break;
    }

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate PllClock, SystemCoreClock and CycylesPerUs automatically. */
    SystemCoreClockUpdate();
}

void UART2_Init(void)
{
    /* Enable USCI2 CLK */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;
  
    /* Reset IP USCI2 */
    SYS->IPRST1 |= SYS_IPRST1_USCI2RST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_USCI2RST_Msk;
  
    /* Open USCI2: 115200 */
    /* Enable USCI_UART protocol */
    UUART2->CTL &= ~UUART_CTL_FUNMODE_Msk;
    UUART2->CTL = 2 << UUART_CTL_FUNMODE_Pos;
  
    /* Set USCI_UART line configuration */
    UUART2->LINECTL = UUART_WORD_LEN_8 | UUART_LINECTL_LSB_Msk;
    UUART2->DATIN0 = (2 << UUART_DATIN0_EDGEDET_Pos);  /* Set falling edge detection */

    /* Set USCI_UART baud rate */
    UUART2->BRGEN = (86 << UUART_BRGEN_CLKDIV_Pos) |
                    (5 << UUART_BRGEN_DSCNT_Pos) | // DSCNT could be 0x5~0xF
                    (0 << UUART_BRGEN_PDSCNT_Pos);
    
    /* Enable UART */
    UUART2->PROTCTL |= UUART_PROTCTL_PROTEN_Msk;
  
    /* Set USCI_UART2 multi-function pins */
    SYS->GPF_MFP &= ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk);
    SYS->GPF_MFP |= ( SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);
  
    /* Set GPIO Mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);
}

void SPI1_Init(void)
{
    /* Enable USCI1 CLK */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI1CKEN_Msk;
  
    /* Reset IP USCI1 */
    SYS->IPRST1 |= SYS_IPRST1_USCI1RST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_USCI1RST_Msk;
  
    /* Open USCI1: Slave, Mode 0, 16 bits, 2MHz */
  
    /* Enable USCI_SPI protocol */
    USPI1->CTL &= ~USPI_CTL_FUNMODE_Msk;
    USPI1->CTL = 1 << USPI_CTL_FUNMODE_Pos;
  
    /* Data word length configuration */
    USPI1->LINECTL &= ~USPI_LINECTL_DWIDTH_Msk;
    USPI1->LINECTL |= (0 << USPI_LINECTL_DWIDTH_Pos);
  
    /* MSB first data format */
    USPI1->LINECTL &= ~USPI_LINECTL_LSB_Msk;
  
    /* Set slave selection signal active low */  
    USPI1->CTLIN0 |= USPI_CTLIN0_ININV_Msk;
  
    /* Set operating mode and transfer timing */
    //-- Set Mode=0, Clk=Low in idle(at initial), Rising-edge to Tx data, Falling-edge to Rx data
    //-- Set SPI as Slave
    USPI1->PROTCTL &= ~USPI_PROTCTL_SCLKMODE_Msk;
    USPI1->PROTCTL |= USPI_PROTCTL_SLAVE_Msk | (0x0 << USPI_PROTCTL_SCLKMODE_Pos);
  
    /* Set USCI_SPI bus clock */
    USPI1->BRGEN &= ~USPI_BRGEN_CLKDIV_Msk;
    USPI1->BRGEN |=  (14 << USPI_BRGEN_CLKDIV_Pos);
    
    /* Enable SPI */
    USPI1->PROTCTL |=  USPI_PROTCTL_PROTEN_Msk;
  
    /* Enable Auto SS */
    USPI1->LINECTL |= USPI_LINECTL_CTLOINV_Msk;
    USPI1->PROTCTL |= USPI_PROTCTL_AUTOSS_Msk;
  
    /* Reset SPI DMA CTL */
    USPI1->DMACTL |= USPI_DMACTL_DMARST_Msk;
    /* Set SPI DMA CTL */
    USPI1->DMACTL |= USPI_DMACTL_DMAEN_Msk | USPI_DMACTL_RXDMAEN_Msk | USPI_DMACTL_TXDMAEN_Msk;
  
    /* Set USCI_SPI1 multi-function pins */
    SYS->GPC_MFP &= ~(SYS_GPC_MFP_PC0MFP_Msk | SYS_GPC_MFP_PC1MFP_Msk | SYS_GPC_MFP_PC2MFP_Msk | SYS_GPC_MFP_PC3MFP_Msk);
    SYS->GPC_MFP |= (SYS_GPC_MFP_PC0_SPI1_CLK | SYS_GPC_MFP_PC1_SPI1_MISO | SYS_GPC_MFP_PC2_SPI1_MOSI | SYS_GPC_MFP_PC3_SPI1_SS);
    
    /* Set GPIO Mode */
    PC->MODE = (PC->MODE & ~(GPIO_MODE_MODE0_Msk | GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk | GPIO_MODE_MODE3_Msk)) |
               (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);
}

int main()
{
    /*
        This sample code uses GDMA transfer and sets USCI_SPI bus clock to 2 MHz.
        Then, Master accesses Slave with 100 Bytes Write and 100 Bytes Read operations.
        Check if the read data is equal to the programmed data.
        PC0: SPI1_CLK
        PC1: SPI1_MISO
        PC2: SPI1_MOSI
        PC3: SPI1_SS
        PF1: UART2_TX
        PF2: UART2_RX
    */
    uint16_t Rx_Buffer[100], Tx_Buffer[100], i, error_cnt = 0;
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS_Init();         /* Initial system clock       */
    UART2_Init();       /* Initial UART2              */
    SPI1_Init();        /* Initial SPI2               */
    SYS_LockReg();      /* Lock protected registers   */
    printf("\n\n");
    printf("+------------------------------------------------+\n");
    printf("|      NM1240 Sample Code for GDMA               |\n");
    printf("+------------------------------------------------+\n");
    printf("Initial NM1240 System Clock:\n");
    printf("CPU clock %dMHz\n",  SystemCoreClock/1000000);
    printf("HCLK clock %dMHz\n", SystemCoreClock/1000000);  
    printf("EXT clock %luMHz (0MHz means no clock source)\n", ( ((CLK->PWRCTL & CLK_PWRCTL_EXTCLKEN_Msk) >> CLK_PWRCTL_EXTCLKEN_Pos) == 0x03 )? __EXT/1000000 : 0/1000000);
    printf("Reset status: 0x%08X\n", SYS->RSTSTS);  
    printf("\n");
    printf("+----------------------------------------------------------+\n");
    printf("|                    SPI Configuration                     |\n");
    printf("+----------------------------------------------------------+\n");
    printf("| Mode           : Slave, Mode 0                           |\n");
    printf("| Transfer width : 16 bits                                 |\n");
    printf("| Baud Rate      : 2 MHz                                   |\n");
    printf("| Auto SS        : Enable                                  |\n");
    printf("+----------------------------------------------------------+\n");
    printf("|                    GDMA Configuration                    |\n");
    printf("+----------------------------------------------------------+\n");
    printf("| MODE           : USCI to Memory(SPI)                     |\n");
    printf("| Transfer width : 16BITS                                  |\n");
    printf("| Transfer count : 100                                     |\n");
    printf("| Burst mode     : DISABLE                                 |\n");
    printf("+----------------------------------------------------------+\n");
    printf("The slave is ready to transfer...\n");
    
    /* Initial Parameter */
    Parameter_Init();
    
    /* Set data of source array */
    for(i=0;i<100;i++) Tx_Buffer[i] = 0x5500 + i;
    /* Reset GDMA */
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS->IPRST0 |= SYS_IPRST0_GDMARST_Msk;
    SYS_LockReg();      /* Lock protected registers */
    
    /* Open GDMA0 for SPI RX */
{
    /* Enable GDMA controller clock */
    CLK->AHBCLK |= CLK_AHBCLK_GDMACKEN_Msk;
    
    /* Select transfer width */
    GDMA0->CTL = (GDMA0->CTL & ~(GDMA_CTL_TWS_Msk | GDMA_CTL_BME_Msk)) | (1 << GDMA_CTL_TWS_Pos);
    
    /* Enable GDMA and set mode */
    GDMA0->CTL = (GDMA0->CTL & ~(GDMA_CTL_GDMAMS_Msk)) | (1 << GDMA_CTL_GDMAMS_Pos);
      
    /* Set address direction or fixed */
    GDMA0->CTL = (GDMA0->CTL & ~(0xF << GDMA_CTL_DADIR_Pos)) | (8 << GDMA_CTL_DADIR_Pos);
    
    /* Set transfer count */
    GDMA0->TCNT = 100;
    
    /* Set destination base address */
    GDMA0->DSTB = (uint32_t)&Rx_Buffer[0];
    
    /* Set source base address */
    GDMA0->SRCB = (uint32_t)&USPI1->RXDAT;
    
    /* Enable GDMA0 External Interrupt */
    NVIC_EnableIRQ(GDMA0_IRQn);
}
    /* Open GDMA1 for SPI TX */   
{    
    /* Select transfer width */
    GDMA1->CTL = (GDMA1->CTL & ~(GDMA_CTL_TWS_Msk | GDMA_CTL_BME_Msk)) | (1 << GDMA_CTL_TWS_Pos);
    
    /* Enable GDMA and set mode */
    GDMA1->CTL = (GDMA1->CTL & ~(GDMA_CTL_GDMAMS_Msk)) | (1 << GDMA_CTL_GDMAMS_Pos);
      
    /* Set address direction or fixed */
    GDMA1->CTL = (GDMA1->CTL & ~(0xF << GDMA_CTL_DADIR_Pos)) | (4 << GDMA_CTL_DADIR_Pos);
    
    /* Set transfer count */
    GDMA1->TCNT = 100;
    
    /* Set destination base address */
    GDMA1->DSTB = (uint32_t)&USPI1->TXDAT;
    
    /* Set source base address */
    GDMA1->SRCB = (uint32_t)&Tx_Buffer[0];
    
    /* Enable GDMA1 External Interrupt */
    NVIC_EnableIRQ(GDMA1_IRQn);
}
    /* Enable GDMA0 INT */
    GDMA0->CTL |= GDMA_CTL_GIEN_Msk;
    
    /* Enable GDMA0 */
    GDMA0->CTL |= GDMA_CTL_GDMAEN_Msk;
    
    /* Enable GDMA1 INT */
    GDMA1->CTL |= GDMA_CTL_GIEN_Msk;
    
    /* Enable GDMA1 */
    GDMA1->CTL |= GDMA_CTL_GDMAEN_Msk;
    
    /* Check the transmission of GDMA0_I2C_RX is complete. */
    while(!g_u8EndFlagRX);
    g_u8EndFlagRX = 0;
    
    /* Disable GDMA0 INT */
    GDMA0->CTL &= ~GDMA_CTL_GIEN_Msk;
    
    /* Check the transmission of GDMA1_I2C_TX is complete. */
    while(!g_u8EndFlagTX);
    g_u8EndFlagTX = 0;
    
    /* Disable GDMA1 INT */
    GDMA1->CTL &= ~GDMA_CTL_GIEN_Msk;
    
    /* Print result */
    for(i=0;i<100;i++)
    {
        printf("\n(Tx_Buffer[%02d], Rx_Buffer[%02d]) = (0x%X, 0x%X)", i, i, Tx_Buffer[i], Rx_Buffer[i]);
        if (Rx_Buffer[i] != ((Tx_Buffer[i]&0xFF)|0xAA00))
        {
            printf(" - ERROR");
            error_cnt ++;
        }
    }
    if (error_cnt == 0){
        printf("\nGDMA memory to memory test: PASS, %d ERROR\n", error_cnt);
    }
    else{
        printf("\nGDMA memory to memory test: FAIL, %d ERROR\n", error_cnt);
    }
    printf("GDMA transfer finish!\n");
    while(1);  // end of while(1)
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
