/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/08/07 15:51p $
 * @brief   NM1240 GDMA memory to memory test
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"

void SYS_Init(void)
{    
    /*---------------------------------------------------------------------------------------------------------*/
    /* Init System Clock                                                                                       */
    /*---------------------------------------------------------------------------------------------------------*/
    int32_t i32TimeOutCnt = 2160000;
  
    /* Enable 60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for HIRC clock ready */
    while((CLK->STATUS & CLK_STATUS_HIRCSTB_Msk) != CLK_STATUS_HIRCSTB_Msk) {
        if(i32TimeOutCnt-- <= 0) break;
    }

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate PllClock, SystemCoreClock and CycylesPerUs automatically. */
    SystemCoreClockUpdate();
}

void UART2_Init(void)
{
    /* Enable USCI2 CLK */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;
  
    /* Reset IP USCI2 */
    SYS->IPRST1 |= SYS_IPRST1_USCI2RST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_USCI2RST_Msk;
  
    /* Open USCI2: 115200 */
    /* Enable USCI_UART protocol */
    UUART2->CTL &= ~UUART_CTL_FUNMODE_Msk;
    UUART2->CTL = 2 << UUART_CTL_FUNMODE_Pos;
  
    /* Set USCI_UART line configuration */
    UUART2->LINECTL = UUART_WORD_LEN_8 | UUART_LINECTL_LSB_Msk;
    UUART2->DATIN0 = (2 << UUART_DATIN0_EDGEDET_Pos);  /* Set falling edge detection */

    /* Set USCI_UART baud rate */
    UUART2->BRGEN = (86 << UUART_BRGEN_CLKDIV_Pos) |
                    (5 << UUART_BRGEN_DSCNT_Pos) | // DSCNT could be 0x5~0xF
                    (0 << UUART_BRGEN_PDSCNT_Pos);
                  
    UUART2->PROTCTL |= UUART_PROTCTL_PROTEN_Msk;
  
    /* Set USCI_UART2 multi-function pins */
    SYS->GPF_MFP &= ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk);
    SYS->GPF_MFP |= ( SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);
  
    /* Set GPIO Mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);
}

int main()
{
    /*
        This sample code uses GDMA to transfer 40 data between SRAM.
        PF1: UART2_TX
        PF2: UART2_RX
    */
    uint32_t DST0[40], SRC0[40],DST1[40], SRC1[40], i, error_cnt = 0;
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS_Init();         /* Initial system clock       */
    UART2_Init();       /* Initial UART2              */
    SYS_LockReg();      /* Lock protected registers   */
    printf("\n\n");
    printf("+------------------------------------------------+\n");
    printf("|      NM1240 Sample Code for GDMA               |\n");
    printf("+------------------------------------------------+\n");
    printf("Initial NM1240 System Clock:\n");
    printf("CPU clock %dMHz\n",  SystemCoreClock/1000000);
    printf("HCLK clock %dMHz\n", SystemCoreClock/1000000);  
    printf("EXT clock %luMHz (0MHz means no clock source)\n", ( ((CLK->PWRCTL & CLK_PWRCTL_EXTCLKEN_Msk) >> CLK_PWRCTL_EXTCLKEN_Pos) == 0x03 )? __EXT/1000000 : 0/1000000);
    printf("Reset status: 0x%08X\n", SYS->RSTSTS);  
    printf("\n");
    printf("+----------------------------------------------------------+\n");
    printf("|                    GDMA Configuration                    |\n");
    printf("+----------------------------------------------------------+\n");
    printf("| MODE           : Memory to Memory(SOFTWARE)              |\n");
    printf("| Transfer width : 32BITS                                  |\n");
    printf("| Transfer count : 10                                      |\n");
    printf("| Burst mode     : ENABLE                                  |\n");
    printf("| SRC BASE ADDR  : Incremented                             |\n");
    printf("| DST BASE ADDR  : Incremented                             |\n");
    printf("+----------------------------------------------------------+\n");
    printf("Press any key to start\n");
    getchar();
    
    /* Set data of source array */
    for(i=0;i<40;i++) {
        SRC0[i] = 0x55550000 + i;
        SRC1[i] = 0xAAAA0000 + i;
    }
    
    /* Reset GDMA */
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS->IPRST0 |= SYS_IPRST0_GDMARST_Msk;
    SYS_LockReg();      /* Lock protected registers */
    
    /* Open GDMA0 */
{    
    /* Enable GDMA controller clock */
    CLK->AHBCLK |= CLK_AHBCLK_GDMACKEN_Msk;
    
    /* Select transfer width */
    GDMA0->CTL = (GDMA0->CTL & ~(GDMA_CTL_TWS_Msk | GDMA_CTL_BME_Msk)) | (GDMA_TWS_32BITS << GDMA_CTL_TWS_Pos) | GDMA_CTL_BME_Msk;
    
    /* Enable GDMA and set mode */
    GDMA0->CTL = (GDMA0->CTL & ~(GDMA_CTL_GDMAMS_Msk)) | (GDMA_SOFTWARE_MODE << GDMA_CTL_GDMAMS_Pos);
      
    /* Set address direction or fixed */
    GDMA0->CTL = (GDMA0->CTL & ~(0xF << GDMA_CTL_DADIR_Pos)) | ((SOURCE_ADDRESS_INC | Destination_ADDRESS_INC) << GDMA_CTL_DADIR_Pos);
    
    /* Set transfer count */
    GDMA0->TCNT = 10;  //Transfer count must be divided 4 because burst mode is used.
    
    /* Set destination base address */
    GDMA0->DSTB = (uint32_t)&DST0[0];
    
    /* Set source base address */
    GDMA0->SRCB = (uint32_t)&SRC0[0];
    
    /* Enable GDMA0 */
    GDMA0->CTL |= GDMA_CTL_GDMAEN_Msk;
}
    /* Open GDMA1 */
{
    /* Select transfer width */
    GDMA1->CTL = (GDMA1->CTL & ~(GDMA_CTL_TWS_Msk | GDMA_CTL_BME_Msk)) | (GDMA_TWS_32BITS << GDMA_CTL_TWS_Pos) | GDMA_CTL_BME_Msk;
    
    /* Enable GDMA and set mode */
    GDMA1->CTL = (GDMA1->CTL & ~(GDMA_CTL_GDMAMS_Msk)) | (GDMA_SOFTWARE_MODE << GDMA_CTL_GDMAMS_Pos);
      
    /* Set address direction or fixed */
    GDMA1->CTL = (GDMA1->CTL & ~(0xF << GDMA_CTL_DADIR_Pos)) | ((SOURCE_ADDRESS_INC | Destination_ADDRESS_INC) << GDMA_CTL_DADIR_Pos);    
    
    
    /* Set transfer count */
    GDMA1->TCNT = 10;  //Transfer count must be divided 4 because burst mode is used.
    
    /* Set destination base address */
    GDMA1->DSTB = (uint32_t)&DST1[0];
    
    /* Set source base address */
    GDMA1->SRCB = (uint32_t)&SRC1[0];
    
    /* Enable GDMA1 */
    GDMA1->CTL |= GDMA_CTL_GDMAEN_Msk;
}

    /* Software request GDMA0 */
    GDMA0->CTL |= GDMA_CTL_SOFTTRG_Msk;
    
    /* Software request GDMA1 */
    GDMA1->CTL |= GDMA_CTL_SOFTTRG_Msk;
    
    /* Wait TCIF */
    while(!(GDMA0->CTL & GDMA_CTL_TCIF_Msk));
    
    /* Clear TCIF */
    GDMA0->CTL &= ~GDMA_CTL_TCIF_Msk;
    
    /* Wait TCIF */
    while(!(GDMA1->CTL & GDMA_CTL_TCIF_Msk));
    
    /* Clear TCIF */
    GDMA1->CTL &= ~GDMA_CTL_TCIF_Msk;
    
    /* Print result */
    for(i=0;i<40;i++)
    {
        printf("\n(SRC0[%02d], DST0[%02d]) = (0x%X, 0x%X)", i, i, SRC0[i], DST0[i]);
        if (DST0[i] != SRC0[i])
        {
            printf(" - ERROR");
            error_cnt ++;
        }
    }
    
    for(i=0;i<40;i++)
    {
        printf("\n(SRC1[%02d], DST1[%02d]) = (0x%X, 0x%X)", i, i, SRC1[i], DST1[i]);
        if (DST1[i] != SRC1[i])
        {
            printf(" - ERROR");
            error_cnt ++;
        }
    }
    
    if (error_cnt == 0){
        printf("\nGDMA memory to memory test: PASS, %d ERROR\n", error_cnt);
    }
    else{
        printf("\nGDMA memory to memory test: FAIL, %d ERROR\n", error_cnt);
    }
    while(1);  // end of while(1)
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
