/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/08/07 14:36p $
 * @brief   NM1240 GDMA I2C SLAVE test
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"

volatile uint8_t g_u8EndFlagTX, g_u8EndFlagRX;

/* TX data structure */
typedef struct
{
    uint16_t Address;
    uint16_t data[100];
} I2C_TX_T;

/* RX data structure */
typedef struct
{
    uint16_t Address;
    uint16_t data[100];
} I2C_RX_T;

I2C_TX_T I2C_TX;
I2C_RX_T I2C_RX;

uint16_t* I2C_RX_ptr;

void Parameter_Init(void)
{
    g_u8EndFlagTX = 0;
    g_u8EndFlagRX = 0;
}

void GDMA0_IRQHandler (void)
{
    /* Clear TCIF */
    GDMA0->CTL &= ~GDMA_CTL_TCIF_Msk;
    
    /* Set GDMA0_I2C_RX Flag */
    g_u8EndFlagRX = 1;
}

void GDMA1_IRQHandler (void)
{
    /* Clear TCIF */
    GDMA1->CTL &= ~GDMA_CTL_TCIF_Msk;
    
    /* Set GDMA1_I2C_TX Flag */
    g_u8EndFlagTX = 1;
}

void USCI1_IRQHandler(void)
{
    volatile uint32_t u32Status;
    u32Status = UI2C1->PROTSTS;

    if(u32Status & UI2C_PROTSTS_ACKIF_Msk){ 
        /* Clear USCI_I2C1 ACK Flag */
        UI2C1->PROTSTS = UI2C_PROTSTS_ACKIF_Msk;
    }
    else if(u32Status & UI2C_PROTSTS_NACKIF_Msk){
        /* Clear USCI_I2C1 NACK Flag */
        UI2C1->PROTSTS = UI2C_PROTSTS_NACKIF_Msk;
    }
    else if (u32Status & UI2C_PROTSTS_TOIF_Msk) {
        /* Clear USCI_I2C1 Timeout Flag */
        UI2C1->PROTSTS = UI2C_PROTSTS_TOIF_Msk;

        /* Close I2C */
        UI2C1->CTL &= ~UI2C_CTL_FUNMODE_Msk;
        printf("I2C time out!\n");
    }
}

void SYS_Init(void)
{    
    /*---------------------------------------------------------------------------------------------------------*/
    /* Init System Clock                                                                                       */
    /*---------------------------------------------------------------------------------------------------------*/
    int32_t i32TimeOutCnt = 2160000;
  
    /* Enable 60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for HIRC clock ready */
    while((CLK->STATUS & CLK_STATUS_HIRCSTB_Msk) != CLK_STATUS_HIRCSTB_Msk) {
        if(i32TimeOutCnt-- <= 0) break;
    }

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate PllClock, SystemCoreClock and CycylesPerUs automatically. */
    SystemCoreClockUpdate();
}

void UART2_Init(void)
{
    /* Enable USCI2 CLK */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;
  
    /* Reset IP USCI2 */
    SYS->IPRST1 |= SYS_IPRST1_USCI2RST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_USCI2RST_Msk;
  
    /* Open USCI2: 115200 */
    /* Enable USCI_UART protocol */
    UUART2->CTL &= ~UUART_CTL_FUNMODE_Msk;
    UUART2->CTL = 2 << UUART_CTL_FUNMODE_Pos;
  
    /* Set USCI_UART line configuration */
    UUART2->LINECTL = UUART_WORD_LEN_8 | UUART_LINECTL_LSB_Msk;
    UUART2->DATIN0 = (2 << UUART_DATIN0_EDGEDET_Pos);  /* Set falling edge detection */

    /* Set USCI_UART baud rate */
    UUART2->BRGEN = (86 << UUART_BRGEN_CLKDIV_Pos) |
                    (5 << UUART_BRGEN_DSCNT_Pos) | // DSCNT could be 0x5~0xF
                    (0 << UUART_BRGEN_PDSCNT_Pos);
                  
    UUART2->PROTCTL |= UUART_PROTCTL_PROTEN_Msk;
  
    /* Set USCI_UART2 multi-function pins */
    SYS->GPF_MFP &= ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk);
    SYS->GPF_MFP |= ( SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);
  
    /* Set GPIO Mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);
}

void I2C1_Init(void)
{
    /* Enable USCI1 CLK */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI1CKEN_Msk;
  
    /* Reset IP USCI1 */
    SYS->IPRST1 |= SYS_IPRST1_USCI1RST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_USCI1RST_Msk;
  
    /* Open UI2C1: 1 MHz */
    /* Enable USCI_I2C protocol */
    UI2C1->CTL &= ~UI2C_CTL_FUNMODE_Msk;
    UI2C1->CTL = 4 << UI2C_CTL_FUNMODE_Pos;

    /* Data format configuration */
    /* 8 bit data length */
    UI2C1->LINECTL &= ~UI2C_LINECTL_DWIDTH_Msk;
    UI2C1->LINECTL |= 8 << UI2C_LINECTL_DWIDTH_Pos;

    /* MSB data format */
    UI2C1->LINECTL &= ~UI2C_LINECTL_LSB_Msk;

    /* Set USCI_I2C bus clock */
    UI2C1->BRGEN &= ~UI2C_BRGEN_CLKDIV_Msk;
    UI2C1->BRGEN |=  (29 << UI2C_BRGEN_CLKDIV_Pos);
    
    /* Enable I2C */
    UI2C1->PROTCTL |=  UI2C_PROTCTL_PROTEN_Msk;

    /* Asser ACK control. */
    UI2C1->PROTCTL |= UI2C_PROTCTL_AA_Msk;

    /* Enable TIMEROUT INT */
    UI2C1->PROTIEN |= UI2C_PROTIEN_TOIEN_Msk;
    
    /* Enable UI2C1 External Interrupt */
    NVIC_EnableIRQ(USCI1_IRQn);
  
    /* Set USCI_I2C1 multi-function pins */
    SYS->GPC_MFP &= ~(SYS_GPC_MFP_PC1MFP_Msk | SYS_GPC_MFP_PC2MFP_Msk);
    SYS->GPC_MFP |= (SYS_GPC_MFP_PC0_I2C1_SCL | SYS_GPC_MFP_PC2_I2C1_SDA);
}

void GDMA_I2C_SLAVE_WRITE_DATA(uint16_t deviceaddress, uint16_t write_data_length, uint16_t* i2c_tx_ptr)
{
    /* Set device address and disable general call function */
    UI2C1->DEVADDR0  = deviceaddress;
    UI2C1->PROTCTL  = (UI2C1->PROTCTL & ~UI2C_PROTCTL_GCFUNC_Msk);

    /* Check I2C RX is not full */
    while(UI2C1->BUFSTS & UI2C_BUFSTS_RXFULL_Msk);
    
    /* Reset UI2C DMA CTL*/
    UI2C1->DMACTL |= UI2C_DMACTL_DMARST_Msk;
    /* Set UI2C DMA CTL */
    UI2C1->DMACTL |= UI2C_DMACTL_DMAEN_Msk | UI2C_DMACTL_RXDMAEN_Msk;
  
    /* Set GDMA0 transfer count */
    GDMA0->TCNT = 1;
    
    /* Set GDMA0 destination base address */
    GDMA0->DSTB = (uint32_t)i2c_tx_ptr;
    
    /* Set GDMA0 source base address */
    GDMA0->SRCB = (uint32_t)&UI2C1->RXDAT;

    /* Set GDMA1 transfer count */
    GDMA1->TCNT = write_data_length;
    
    /* Point to the next data */
    i2c_tx_ptr = i2c_tx_ptr + 1;
    /* Set destination base address */
    GDMA1->DSTB = (uint32_t)&UI2C1->TXDAT;
    
    /* Set source base address */
    GDMA1->SRCB = (uint32_t)i2c_tx_ptr;
  
    /* Clear I2C NACK INT FLAG */
    UI2C1->PROTSTS = UI2C_PROTSTS_NACKIF_Msk;
    
    /* Enable GDMA0 */
    GDMA0->CTL |= GDMA_CTL_GDMAEN_Msk;
    
    /* Wait GDMA0 TCIF */
    while(!(GDMA0->CTL & GDMA_CTL_TCIF_Msk));
    
    /* Clear GDMA0 TCIF */
    GDMA0->CTL &= ~GDMA_CTL_TCIF_Msk;
    
    /* Reset UI2C DMA CTL */
    UI2C1->DMACTL |= UI2C_DMACTL_DMARST_Msk;
    /* Set UI2C DMA CTL */
    UI2C1->DMACTL |= UI2C_DMACTL_DMAEN_Msk | UI2C_DMACTL_TXDMAEN_Msk;

    /* Enable GDMA1 */
    GDMA1->CTL |= GDMA_CTL_GDMAEN_Msk;
}

void GDMA_I2C_SLAVE_READ_DATA(uint16_t deviceaddress, uint16_t read_data_length, uint16_t* i2c_rx_ptr)
{      
    /* Set device address and disable general call function */
    UI2C1->DEVADDR0  = deviceaddress;
    UI2C1->PROTCTL  = (UI2C1->PROTCTL & ~UI2C_PROTCTL_GCFUNC_Msk);
  
    /* Check I2C RX is not full */
    while(UI2C1->BUFSTS & UI2C_BUFSTS_RXFULL_Msk);
    
    /* Reset UI2C DMA CTL */
    UI2C1->DMACTL |= UI2C_DMACTL_DMARST_Msk;
    /* Set UI2C DMA CTL */
    UI2C1->DMACTL |= UI2C_DMACTL_DMAEN_Msk | UI2C_DMACTL_RXDMAEN_Msk | UI2C_DMACTL_NACKEN_Msk;
    
    /* Set GDMA0 transfer count */
    GDMA0->TCNT = read_data_length + 1;
    
    /* Set GDMA0 destination base address */
    GDMA0->DSTB = (uint32_t)i2c_rx_ptr;
    
    /* Set GDMA0 source base address */
    GDMA0->SRCB = (uint32_t)&UI2C1->RXDAT;
    
    /* Clear I2C NACK INT FLAG */
    UI2C1->PROTSTS = UI2C_PROTSTS_NACKIF_Msk;
    
    /* Enable GDMA0 */
    GDMA0->CTL |= GDMA_CTL_GDMAEN_Msk;
}

int main()
{
    /*
        This sample code uses GDMA transfer and sets USCI_I2C bus clock to 1 MHz.
        Then, Master accesses Slave with 100 Bytes Write and 100 Bytes Read operations.
        Check if the read data is equal to the programmed data.
        PC0: I2C1_SCL
        PC2: I2C1_SDA
        PF1: UART2_TX
        PF2: UART2_RX
    */
    uint16_t i;
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS_Init();         /* Initial system clock       */
    UART2_Init();       /* Initial UART2              */
    I2C1_Init();        /* Initial I2C1               */
    SYS_LockReg();      /* Lock protected registers   */
    printf("\n\n");
    printf("+------------------------------------------------+\n");
    printf("|      NM1240 Sample Code for GDMA               |\n");
    printf("+------------------------------------------------+\n");
    printf("Initial NM1240 System Clock:\n");
    printf("CPU clock %dMHz\n",  SystemCoreClock/1000000);
    printf("HCLK clock %dMHz\n", SystemCoreClock/1000000);
    printf("EXT clock %luMHz (0MHz means no clock source)\n", ( ((CLK->PWRCTL & CLK_PWRCTL_EXTCLKEN_Msk) >> CLK_PWRCTL_EXTCLKEN_Pos) == 0x03 )? __EXT/1000000 : 0/1000000);
    printf("Reset status: 0x%08X\n", SYS->RSTSTS);  
    printf("\n");
    printf("+----------------------------------------------------------+\n");
    printf("|                    I2C Configuration                     |\n");
    printf("+----------------------------------------------------------+\n");
    printf("| Mode           : Slave                                   |\n");
    printf("| Transfer width : 8 bits                                  |\n");
    printf("| Baud Rate      : 1 MHz                                   |\n");
    printf("+----------------------------------------------------------+\n");
    printf("|                    GDMA Configuration                    |\n");
    printf("+----------------------------------------------------------+\n");
    printf("| MODE           : USCI to Memory(I2C)                     |\n");
    printf("| Transfer width : 16BITS                                  |\n");
    printf("| Data           : Slave read and write 100 data           |\n");
    printf("| Burst mode     : DISABLE                                 |\n");
    printf("+----------------------------------------------------------+\n");
    printf("The Slave is ready to transfer...\n");

    /* Initial Parameter */
    Parameter_Init();
    
    /* Reset GDMA */
    SYS_UnlockReg();    /* Unlock protected registers */
    SYS->IPRST0 |= SYS_IPRST0_GDMARST_Msk;
    SYS_LockReg();      /* Lock protected registers */
    
    /* TX data array */
    for(i=0;i<100;i++)I2C_TX.data[i] = 99 - i;
    
    /* Open GDMA0 for I2C RX */
{
    /* Enable GDMA controller clock */
    CLK->AHBCLK |= CLK_AHBCLK_GDMACKEN_Msk;
    
    /* Select transfer width */
    GDMA0->CTL = (GDMA0->CTL & ~(GDMA_CTL_TWS_Msk | GDMA_CTL_BME_Msk)) | (GDMA_TWS_16BITS << GDMA_CTL_TWS_Pos);
    
    /* Enable GDMA and set mode */
    GDMA0->CTL = (GDMA0->CTL & ~(GDMA_CTL_GDMAMS_Msk)) | (GDMA_USCI_MODE << GDMA_CTL_GDMAMS_Pos);
      
    /* Set address direction or fixed */
    GDMA0->CTL = (GDMA0->CTL & ~(0xF << GDMA_CTL_DADIR_Pos)) | ((SOURCE_ADDRESS_FIXED | Destination_ADDRESS_INC) << GDMA_CTL_DADIR_Pos);
    
    /* Enable GDMA0 External Interrupt */
    NVIC_EnableIRQ(GDMA0_IRQn);
}
    /* Open GDMA1 for I2C TX */
{    
    /* Select transfer width */
    GDMA1->CTL = (GDMA1->CTL & ~(GDMA_CTL_TWS_Msk | GDMA_CTL_BME_Msk)) | (GDMA_TWS_16BITS << GDMA_CTL_TWS_Pos);
    
    /* Enable GDMA and set mode */
    GDMA1->CTL = (GDMA1->CTL & ~(GDMA_CTL_GDMAMS_Msk)) | (GDMA_USCI_MODE << GDMA_CTL_GDMAMS_Pos);
      
    /* Set address direction or fixed */
    GDMA1->CTL = (GDMA1->CTL & ~(0xF << GDMA_CTL_DADIR_Pos)) | ((SOURCE_ADDRESS_INC | Destination_ADDRESS_FIXED) << GDMA_CTL_DADIR_Pos);
    
    /* Enable GDMA1 External Interrupt */
    NVIC_EnableIRQ(GDMA1_IRQn);
}
{
    /* Enable GDMA0 INT */
    GDMA0->CTL |= GDMA_CTL_GIEN_Msk;
    
    /* GDMA_I2C_SLAVE_READ_DATA(device_address, read_data_length, i2c_rx_ptr) */
    GDMA_I2C_SLAVE_READ_DATA(0x15, 100, &I2C_RX.Address);
    
    /* Check the transmission of GDMA0_I2C_RX is complete. */
    while(!g_u8EndFlagRX);
    g_u8EndFlagRX = 0;
    
    /* Disable GDMA0 INT */
    GDMA0->CTL &= ~GDMA_CTL_GIEN_Msk;
}
{
    /* Enable GDMA1 INT */
    GDMA1->CTL |= GDMA_CTL_GIEN_Msk;
    
    /* GDMA_I2C_SLAVE_WRITE_DATA(device_address, write_data_length, i2c_tx_ptr) */
    GDMA_I2C_SLAVE_WRITE_DATA(0x15, 100, &I2C_TX.Address);
    
    /* Check the transmission of GDMA1_I2C_TX is complete. */
    while(!g_u8EndFlagTX);
    g_u8EndFlagTX = 0;
    
    /* Disable GDMA1 INT */
    GDMA1->CTL &= ~GDMA_CTL_GIEN_Msk;
}

    /* Print result */
    for(i=0;i<100;i++)
    {
      printf("(I2C_TX.data[%02d], I2C_RX.data[%02d]) = (0x%02X, 0x%02X)\n", i, i, I2C_TX.data[i], I2C_RX.data[i]);
    }
    printf("GDMA transfer finish!\n");
    while(1);  // end of while(1)
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
