/******************************************************************************
 * @file     LDROM_main.c
 * @version  V1.00
 * $Revision: 2 $
 * $Date: 2020/07/20 15:30 $
 * @brief    This sample code includes LDROM image (fmc_ld_iap) 
 *           and APROM image (fmc_ap_main).
 *           It shows how to branch between APROM and LDROM. To run 
 *           this sample code, the boot mode must be "Boot from APROM 
 *           with IAP".
 *
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/   
#include <stdio.h>
#include "NM1240.h"
#include "fmc.h"

typedef void (FUNC_PTR)(void);

void SYS_Init(void)
{
   /* Unlock protected registers */
    SYS_UnlockReg();
	
	 /* Enable 60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk; 	
		
    /* Waiting for 60MHz clock ready */
    CLK_WaitClockReady(CLK_STATUS_HIRCSTB_Msk);	
	
		/* HCLK Clock source from HIRC */
		CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;
	
	  /* Enable USCI1 IP clock */
		CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;	

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();
	
		//* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);
    /* Lock protected registers */
    SYS_LockReg();	
	}


#ifdef __ARMCC_VERSION
void __set_SP(uint32_t _sp)
{
		__ASM(
    "MSR MSP, r0 \n"
    "BX lr \n"			
			);
}
#endif

uint32_t UUART_Open(UUART_T* uart, uint32_t u32baudrate)
{
		int32_t multiple; 
	
		uart->CTL = 0x02;	
		
		multiple = 60000000 / u32baudrate;	/* Fsys_clk = 2 * _HXT*/
		multiple = multiple /5;						/* Set DSCNT = 4 --> divide by (4+1) */
		uart-> BRGEN = (multiple << 16) | (0x04 << 10);
	
		uart-> DATIN0 = 0x10;							/* input falling edge activate */
	
		uart->LINECTL = 0x08<<8 | 0x01;		/* 8-N-1 bit, LSB first */
	
		/* Uart Protocol setting */ 
		uart->PROTCTL = UUART_PROTCTL_PROTEN_Msk;
	
    return 0;
}
void SendChar_ToUART(int ch)
{

    while (UUART2->BUFSTS & UUART_BUFSTS_TXFULL_Msk);
    UUART2->TXDAT = ch;

    if (ch == '\n')
    {
				while (UUART2->BUFSTS & UUART_BUFSTS_TXFULL_Msk);
				UUART2->TXDAT = '\r';
    }
}


void print_msg(char *str)
{
	for ( ; *str ; str++)
        SendChar_ToUART(*str);
}

int main()
{
    FUNC_PTR    *func;

    SYS_Init();

    /* Unlock protected registers */
    SYS_UnlockReg();
	
    /* Init USCI UART2 to 115200-8n1 for print message */
		UUART_Open(UUART2, 115200);	
	
    /* Enable FMC ISP function */
    /* FMC_Open() */
    FMC->ISPCTL |=  FMC_ISPCTL_ISPEN_Msk;
	
    print_msg("\n\n");
    print_msg("NM1240 FMC IAP Sample Code [LDROM code]\n");
		
    print_msg("\n\nPress any key to branch to APROM...\n");		
		
		while(UUART2->BUFSTS & UUART_BUFSTS_RXEMPTY_Msk)   /* Check RX empty => failed */
		;
    print_msg("\n\nChange VECMAP and branch to APROM...\n");
	
		while((UUART2->BUFSTS & UUART_BUFSTS_TXEMPTY_Msk) == 0)   /* Wait Tx empty */
    ;
    /* func = (FUNC_PTR *)FMC_Read(FMC_APROM_BASE + 4) */
    FMC->ISPCMD = FMC_ISPCMD_READ;
    FMC->ISPADDR = (FMC_APROM_BASE + 4);
    FMC->ISPTRG = FMC_ISPTRG_ISPGO_Msk;     
    while (FMC->ISPTRG & FMC_ISPTRG_ISPGO_Msk) ;
    func = (FUNC_PTR *)FMC->ISPDAT;						
						
    /*__set_SP(FMC_Read(FMC_APROM_BASE)) */
    FMC->ISPCMD = FMC_ISPCMD_READ;
    FMC->ISPADDR = FMC_APROM_BASE;
    FMC->ISPTRG = FMC_ISPTRG_ISPGO_Msk;     
    while (FMC->ISPTRG & FMC_ISPTRG_ISPGO_Msk) ;
    __set_SP(FMC->ISPDAT); 	

    /*  NOTE!
     *     Before change VECMAP, user MUST disable all interrupts.
     */
    /* FMC_SetVectorPageAddr(FMC_APROM_BASE) */
    FMC->ISPCMD = FMC_ISPCMD_VECMAP;
    FMC->ISPADDR = FMC_APROM_BASE;
    FMC->ISPTRG = FMC_ISPTRG_ISPGO_Msk; 
    while (FMC->ISPTRG & FMC_ISPTRG_ISPGO_Msk) ;

    func();
    
    while (1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
