/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/07/24 10:54 $
 * @brief    Demonstrate the EPWM dead-zone feature.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void EPWM_IRQHandler(void)
{
    static uint32_t cnt;
    static uint32_t out;

    /* Channel 0 frequency is 100Hz, every 1 second enter this IRQ handler 100 times. */
    if(++cnt == 100) {
        if(out) {
            SYS->GPA_MFP |= (SYS_GPA_MFP_PA0_EPWM_CH0 |\
                                SYS_GPA_MFP_PA1_EPWM_CH1 |\
                                SYS_GPA_MFP_PA2_EPWM_CH2 |\
                                SYS_GPA_MFP_PA3_EPWM_CH3 |\
                                SYS_GPA_MFP_PA4_EPWM_CH4 |\
                                SYS_GPA_MFP_PA5_EPWM_CH5);
        }
        else {
            SYS->GPA_MFP &= ~(SYS_GPA_MFP_PA0MFP_Msk |\
                                SYS_GPA_MFP_PA1MFP_Msk |\
                                SYS_GPA_MFP_PA2MFP_Msk |\
                                SYS_GPA_MFP_PA3MFP_Msk |\
                                SYS_GPA_MFP_PA4MFP_Msk |\
                                SYS_GPA_MFP_PA5MFP_Msk);
        }
        out ^= 1;
        cnt = 0;
    }
    /* Clear channel 0 period interrupt flag */
    EPWM->INTSTS = (((EPWM_INTSTS_CMPUIF0_Msk) << 0) | EPWM_INTSTS_PIF_Msk);
}

void SYS_Init(void)
{
    int32_t i32TimeOutCnt = 2160000;

    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Enable 60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for HIRC clock ready */
    while((CLK->STATUS & CLK_STATUS_HIRCSTB_Msk) != CLK_STATUS_HIRCSTB_Msk) {
        if(i32TimeOutCnt-- <= 0) break;
    }

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Enable IP clock */
    CLK->APBCLK = CLK->APBCLK | (CLK_APBCLK_USCI2CKEN_Msk | CLK_APBCLK_EPWMCKEN_Msk);

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Set GPA multi-function pins for EPWM Channel0~5 */
    SYS->GPA_MFP = (SYS->GPA_MFP & (~SYS_GPA_MFP_PA0MFP_Msk));
    SYS->GPA_MFP |= SYS_GPA_MFP_PA0_EPWM_CH0;
    SYS->GPA_MFP = (SYS->GPA_MFP & (~SYS_GPA_MFP_PA1MFP_Msk));
    SYS->GPA_MFP |= SYS_GPA_MFP_PA1_EPWM_CH1;
    SYS->GPA_MFP = (SYS->GPA_MFP & (~SYS_GPA_MFP_PA2MFP_Msk));
    SYS->GPA_MFP |= SYS_GPA_MFP_PA2_EPWM_CH2;
    SYS->GPA_MFP = (SYS->GPA_MFP & (~SYS_GPA_MFP_PA3MFP_Msk));
    SYS->GPA_MFP |= SYS_GPA_MFP_PA3_EPWM_CH3;
    SYS->GPA_MFP = (SYS->GPA_MFP & (~SYS_GPA_MFP_PA4MFP_Msk));
    SYS->GPA_MFP |= SYS_GPA_MFP_PA4_EPWM_CH4;
    SYS->GPA_MFP = (SYS->GPA_MFP & (~SYS_GPA_MFP_PA5MFP_Msk));
    SYS->GPA_MFP |= SYS_GPA_MFP_PA5_EPWM_CH5;

    /* Set GPA0~5 as output mode */
    PA->MODE = (PA->MODE & ~GPIO_MODE_MODE0_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE0_Pos);
    PA->MODE = (PA->MODE & ~GPIO_MODE_MODE1_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);
    PA->MODE = (PA->MODE & ~GPIO_MODE_MODE2_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE2_Pos);
    PA->MODE = (PA->MODE & ~GPIO_MODE_MODE3_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE3_Pos);
    PA->MODE = (PA->MODE & ~GPIO_MODE_MODE4_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE4_Pos);
    PA->MODE = (PA->MODE & ~GPIO_MODE_MODE5_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE5_Pos);

    /* Lock protected registers */
    SYS_LockReg();
}


int main()
{
    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\nThis sample code will output EPWM channel to with different\n");
    printf("duty, enable dead zone function of all EPWM pairs.\n");
    printf("And also enable/disable EPWM output every 1 second.\n");
    /* EPWM0 frequency is 100Hz, duty 30% */
    EPWM->CLKDIV = EPWM_CLK_DIV_16;
    EPWM->CTL = (EPWM->CTL & ~EPWM_CTL_CNTTYPE_Msk) | (EPWM_CTL_CNTMODE_Msk);
    EPWM->CMPDAT[0] = 13500;
    EPWM->PERIOD = 44999;
    
    /* set duration */
    EPWM->DTCTL = EPWM->DTCTL & ~(EPWM_DTCTL_DTCNT01_Msk) | 150;
    
    /* enable dead zone */
    EPWM->CTL |= EPWM_CTL_DTCNT01_Msk;

    /* EPWM2 frequency is 100Hz, duty 50% */
    EPWM->CMPDAT[2] = 22500;
    
    /* set duration */
    EPWM->DTCTL = EPWM->DTCTL & ~(EPWM_DTCTL_DTCNT23_Msk) | (200 << EPWM_DTCTL_DTCNT23_Pos);
    
    /* enable dead zone */
    EPWM->CTL |= EPWM_CTL_DTCNT23_Msk;

    /* EPWM4 frequency is 100Hz, duty 70% */
    EPWM->CMPDAT[4] = 31500;
    
    /* set duration */
    EPWM->DTCTL = EPWM->DTCTL & ~(EPWM_DTCTL_DTCNT45_Msk) | (100 << EPWM_DTCTL_DTCNT45_Pos);
    
    /* enable dead zone */
    EPWM->CTL |= EPWM_CTL_DTCNT45_Msk;

    /* Enable complementary mode */
    EPWM->CTL = (EPWM->CTL & ~EPWM_CTL_MODE_Msk) |(1UL << EPWM_CTL_MODE_Pos);

    /* Enable output of all EPWM channels */
    PA->MODE = (PA->MODE & ~GPIO_MODE_MODE0_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE0_Pos);
    PA->MODE = (PA->MODE & ~GPIO_MODE_MODE1_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);
    PA->MODE = (PA->MODE & ~GPIO_MODE_MODE2_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE2_Pos);
    PA->MODE = (PA->MODE & ~GPIO_MODE_MODE3_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE3_Pos);
    PA->MODE = (PA->MODE & ~GPIO_MODE_MODE4_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE4_Pos);
    PA->MODE = (PA->MODE & ~GPIO_MODE_MODE5_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE5_Pos);

    /* Enable EPWM channel 0 period interrupt, use channel 0 to measure time. */
    EPWM->INTEN |= ((EPWM_DUTY_INT_UP_COUNT_MATCH_CMP << 0) | EPWM_INTEN_PIEN_Msk);
    NVIC_EnableIRQ(EPWM_IRQn);

    /* Start */
    EPWM->CTL |= (EPWM_CTL_CNTEN0_Msk | EPWM_CTL_CNTEN1_Msk | EPWM_CTL_CNTEN2_Msk | \
                    EPWM_CTL_CNTEN3_Msk | EPWM_CTL_CNTEN4_Msk | EPWM_CTL_CNTEN5_Msk);

    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
