/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/07/24 10:33 $
 * @brief    Demonstrate the BDC PWM control sample.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"

void HBridge_Driver(uint32_t Duty, uint8_t State);

void EPWM_IRQHandler(void)
{
    /* Clear channel 0 period interrupt flag */
    EPWM->INTSTS = EPWM_INTSTS_PIF_Msk;
}

void SYS_Init(void)
{
    int32_t i32TimeOutCnt = 2160000;

    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Enable 60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for HIRC clock ready */
    while((CLK->STATUS & CLK_STATUS_HIRCSTB_Msk) != CLK_STATUS_HIRCSTB_Msk) {
        if(i32TimeOutCnt-- <= 0) break;
    }

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Enable IP clock */
    CLK->APBCLK = CLK->APBCLK | (CLK_APBCLK_USCI2CKEN_Msk | CLK_APBCLK_EPWMCKEN_Msk);

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Set GPA multi-function pins for EPWM Channel0 */
    SYS->GPA_MFP = (SYS->GPA_MFP & ~(SYS_GPA_MFP_PA0MFP_Msk|SYS_GPA_MFP_PA1MFP_Msk|SYS_GPA_MFP_PA2MFP_Msk|SYS_GPA_MFP_PA3MFP_Msk));
    SYS->GPA_MFP |= (SYS_GPA_MFP_PA0_EPWM_CH0|SYS_GPA_MFP_PA1_EPWM_CH1|SYS_GPA_MFP_PA2_EPWM_CH2|SYS_GPA_MFP_PA3_EPWM_CH3);

    /* Set GPA0~5 as output mode */
    PA->MODE = (PA->MODE & ~GPIO_MODE_MODE0_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE0_Pos);
    PA->MODE = (PA->MODE & ~GPIO_MODE_MODE1_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);
    PA->MODE = (PA->MODE & ~GPIO_MODE_MODE2_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE2_Pos);
    PA->MODE = (PA->MODE & ~GPIO_MODE_MODE3_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE3_Pos);

    /* Lock protected registers */
    SYS_LockReg();
}

int main()
{
    uint8_t c;
    uint16_t u16CNR;
    
    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\nThis sample code will output EPWM driving mode of BDC with 4 case.\n");
    printf("The 4 modes are Coast, Forward, Backward and Brake.\n");
        
    /* EPWM channel 0 ~ 4 wave form of this sample with 50% duty and 1000Hz */
    u16CNR = (SystemCoreClock / (1000))/2;
    EPWM->CLKDIV = 0;
    EPWM->PERIOD = u16CNR - 1;
    EPWM->CMPDAT[0] = 50 * (u16CNR) / 100;
    EPWM->CMPDAT[1] = 50 * (u16CNR) / 100;
    EPWM->CMPDAT[2] = 50 * (u16CNR) / 100;
    EPWM->CMPDAT[3] = 50 * (u16CNR) / 100;
    
    /* Set EPWM center aligned */
    EPWM->CTL = (EPWM->CTL & ~EPWM_CTL_CNTTYPE_Msk) | (EPWM_CENTER_ALIGNED);

    /* Set EPWM auto-reload mode */
    EPWM->CTL |= EPWM_CTL_CNTMODE_Msk;
    
    /* Set EPWM complementary mode */
    EPWM->CTL = (EPWM->CTL & ~EPWM_CTL_MODE_Msk) |(1UL << EPWM_CTL_MODE_Pos) | EPWM_CTL_CNTMODE_Msk;

    /* Default PA0 ~ PA3 output low */
    EPWM->PHCHG = (EPWM->PHCHG & ~((0x3F << EPWM_PHCHG_MSKEN0_Pos) | (0x3F << EPWM_PHCHG_MSKDAT0_Pos)) | \
                    ((0x3F << EPWM_PHCHG_MSKEN0_Pos) | (0 << EPWM_PHCHG_MSKDAT0_Pos)));

    /* Enable EPWM period interrupt */
    EPWM->INTEN |= (EPWM_INTEN_PIEN_Msk);
    
    /* Interrupt enable */
    NVIC_EnableIRQ(EPWM_IRQn);
    NVIC_SetPriority(EPWM_IRQn, 0);

    /* Start */
    EPWM->CTL |= 0x0F;
    
    printf("Press key to select BCD mode.\n");
    printf(" 0: Coast\n");
    printf(" 1: Forward\n");
    printf(" 2: Backward\n");
    printf(" 3: Brake\n");
    
    while(1) {
        c = getchar();       
        c = c - '0';
        
        if(c < 4) {
            printf("Current select: %d\r",c);
            HBridge_Driver(50, c);  
        }
    }
}

#define EPWM_SET_DUTY(epwm, u32ChannelNum, duty) { (epwm)->CMPDAT[(u32ChannelNum)] = duty * ((epwm)->PERIOD + 1) / 100; }

void HBridge_Driver(uint32_t Duty, uint8_t State)
{
    switch (State)
    {
        /* Coast mode, PA0 ~ PA3 = low */
        case 0:
            EPWM_MASK_OUTPUT(EPWM, 0x3F, 0x00);
            EPWM_SET_DUTY(EPWM, 0, 0);
            EPWM_SET_DUTY(EPWM, 1, 0);
            EPWM_SET_DUTY(EPWM, 2, 0);
            EPWM_SET_DUTY(EPWM, 3, 0);
            break;
        
        /* Forward mode, PA0 = EPWM0, PA1 = EPWM1, PA2 = low, PA3 = high */
        case 1:
            EPWM_MASK_OUTPUT(EPWM, 0x0C, 0x08);        
            EPWM_SET_DUTY(EPWM, 0, Duty);
            EPWM_SET_DUTY(EPWM, 1, Duty);
            break;
        
        /* Backward mode, PA0 = low, PA1 = high, PA2 = EPWM2, PA3 = EPWM3 */
        case 2:
            EPWM_MASK_OUTPUT(EPWM, 0x03, 0x02);
            EPWM_SET_DUTY(EPWM, 2, Duty);
            EPWM_SET_DUTY(EPWM, 3, Duty);
            break;
        
        /* Brake mode, PA0 and PA2 = low, PA1 and PA3 = high */
        case 3:
            EPWM_MASK_OUTPUT(EPWM, 0x0F, 0x0A);
            EPWM_SET_DUTY(EPWM, 0, 0);
            EPWM_SET_DUTY(EPWM, 1, 0);
            EPWM_SET_DUTY(EPWM, 2, 0);
            EPWM_SET_DUTY(EPWM, 3, 0);
            break;
        
        default:
            break;
    }
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
