/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision:  $
 * $Date: 2020/07/14  $
 * @brief    Sample code for ADC Timer Trigger feature.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void SYS_Init(void)
{
    int32_t i32TimeOutCnt = 2160000;

    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Enable 48/60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for HIRC clock ready */
    while((CLK->STATUS & CLK_STATUS_HIRCSTB_Msk) != CLK_STATUS_HIRCSTB_Msk) {
        if(i32TimeOutCnt-- <= 0) break;
    }

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Enable USCI2 IP clock */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Enable IP clock */
    CLK->APBCLK |= CLK_APBCLK_ADCCKEN_Msk;

    /* Reset Module */
    SYS->IPRST1 |= SYS_IPRST1_ADCRST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_ADCRST_Msk;

    /* Enable Module Clock(TMR0_MODULE); */
    CLK->APBCLK |= CLK_APBCLK_TMR0CKEN_Msk;

    /* Select IP clock source */

    /*---------------------------------------------------------------------------------------------------------*/
    /* Init I/O Multi-function                                                                                 */
    /*---------------------------------------------------------------------------------------------------------*/

    /* Lock protected registers */
    SYS_LockReg();
}

uint32_t  reg_adc0_0,reg_adc0_1, reg_adc1_0,reg_adc1_1;
void ADC0_IRQHandler(void)
{
    if(ADC->STATUS & ADC_STATUS_ADC0IF_Msk)
    {
        /* clear ADC0 interrupt flag */
        ADC->STATUS = ADC_STATUS_ADC0IF_Msk;  
    
		reg_adc0_0 = ADC->VALSTS&BIT0;
		reg_adc0_1 = ADC->VALSTS&BIT1;
		reg_adc1_0 = ADC->VALSTS&BIT16;
		reg_adc1_1 = ADC->VALSTS&BIT17;

		/*Read ADC Status bit*/
		printf("Get ADC0 data0 = %4d (valid = %s)\n",ADC->ADC0_DAT0,(reg_adc0_0)? "Yes" : "No");
		printf("Get ADC0 data1 = %4d (valid = %s)\n",ADC->ADC0_DAT1,(reg_adc0_1)? "Yes" : "No");       
		printf("Get ADC1 data0 = %4d (valid = %s)\n",ADC->ADC1_DAT0,(reg_adc1_0)? "Yes" : "No"); 
		printf("Get ADC1 data1 = %4d (valid = %s)\n",ADC->ADC1_DAT1,(reg_adc1_1)? "Yes" : "No");		
    }
}

void TMR0_IRQHandler(void)
{
    /* clear timer interrupt flag */
	TIMER0->INTSTS = TIMER_INTSTS_TIF_Msk;
    printf("Timer0 expired !!\n");
}

int main()
{
    char ch;

    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    /* printf("\n\nPDID 0x%08X\n", SYS_ReadPDID()); */    /* Display PDID */
    printf("\n\nPDID 0x%08X\n", (unsigned int)(SYS->PDID & SYS_PDID_PDID_Msk)); /* Display PDID */

    printf("CPU @ %dHz\n", SystemCoreClock);        /* Display System Core Clock */

    /*
     * This sample code will configure Timer0 to trigger ADC0_CH0,ADC0_CH1,ADC1_CH0,ADC1_CH1
	 *periodically and print conversion results.
     * 
     */
    printf("+-------------------------------------------+\n");
    printf("| NM1240 ADC for Timer Trigger Sample Code |\n");
    printf("+-------------------------------------------+\n");

    /* Enable ADC clock */
    CLK->APBCLK |= CLK_APBCLK_ADCCKEN_Msk;
    /* Reset IP */
    SYS->IPRST1 |= SYS_IPRST1_ADCRST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_ADCRST_Msk;

	/*Clear ADC Channel select*/
	ADC->CS_CTL &= ~ADC_CS_CTL_ADC0CSEN_Msk;
	/*ADC0 Channel select Set ADC0_CH0,ADC0_CH1,ADC0_CH0=ADC0CS_ADC0_P0,ADC0_CH1=ADC0CS_ADC0_P1 */
	ADC->CS_CTL |= ADC0CS_ADC0_P0|ADC0CS_ADC0_P1;	
	/*ADC1 Channel select Set ADC1_CH0,ADC1_CH1,ADC0_CH0=ADC1CS_ADC0_P0,ADC0_CH1=ADC1CS_ADC0_P1 */
	ADC->CS_CTL |= ADC1CS_ADC1_P0|ADC1CS_ADC1_P1;
	
	/*ADC0CS_ADC0_P0 = PB0*/
    printf("Set GPB0 pin as ADC0_CH0(ADC0_P0) input pin.\n");
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB0MFP_Msk)) | SYS_GPB_MFP_PB0_ADC0_P0;
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE0_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE0_Pos);
    PB->DINOFF |= GPIO_DINOFF_DINOFF0_Msk;

	/*ADC0CS_ADC0_P1 = PB1*/
    printf("Set GPB1 pin as ADC0_CH1(ADC0_P1) input pin.\n");
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB1MFP_Msk)) | SYS_GPB_MFP_PB1_ADC0_P1;
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE1_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE1_Pos);
    PB->DINOFF |= GPIO_DINOFF_DINOFF1_Msk;	
	
	/*ADC1CS_ADC1_P0 = PB4*/
    printf("Set GPB4 pin as ADC1_CH0(ADC1_P0) input pin.\n");
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB4MFP_Msk)) | SYS_GPB_MFP_PB4_ADC1_P0;
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE4_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE4_Pos);
    PB->DINOFF |= GPIO_DINOFF_DINOFF4_Msk;

	/*ADC1CS_ADC1_P1 = PD2*/
    printf("Set GPD2 pin as ADC1_CH1(ADC1_P1) input pin.\n");
    SYS->GPD_MFP = (SYS->GPD_MFP & (~SYS_GPD_MFP_PD2MFP_Msk)) | SYS_GPD_MFP_PD2_ADC1_P1;
    PD->MODE = (PD->MODE & ~GPIO_MODE_MODE2_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE2_Pos);
    PD->DINOFF |= GPIO_DINOFF_DINOFF2_Msk;

    /*--- Initial Timer to trigger ADC */
    printf("Set Timer0 to periodic 1Hz. It will trigger ADC 1 times per second.\n");
    TIMER_Open(TIMER0, TIMER_PERIODIC_MODE, 1);
    TIMER0->CTL |= TIMER_CTL_INTEN_Msk;
    NVIC_EnableIRQ(TMR0_IRQn);


    /* Enable ADC */
    ADC->CTL |= ADC_CTL_ADCEN_Msk;


    ADC->TRGSOR = (ADC->TRGSOR & ~(ADC_TRGSOR_ADC1TRGSOR_Msk | ADC_TRGSOR_ADC0TRGSOR_Msk | ADC_TRGSOR_ADC0STADCSEL_Msk | ADC_TRGSOR_ADC0PWMTRGSEL_Msk))
                 | ADC0_TIMER0_TRIGGER;
    ADC->CTL |= ADC_CTL_ADC0HWTRGEN_Msk;

    /* Configure ADC conversion mode to Simultaneous Simple Mode */
    ADC->CTL = (ADC->CTL & ~ADC_CTL_ADCMODE_Msk) | (0x03 << ADC_CTL_ADCMODE_Pos);
	
    /* Configure ADC sample time to 6 ADC clocks */
    ADC->SMPCNT = (ADC->SMPCNT & (~ADC_SMPCNT_ADCSMPCNT_Msk)) | ((5 & 0xF) << ADC_SMPCNT_ADCSMPCNT_Pos);

    ADC->CTL |= ADC_CTL_ADC0IEN_Msk;

    NVIC_EnableIRQ(ADC0_IRQn);

    /* Begin to do ADC conversion. */
    ch = 0;

    /* TIMER_Start */
    TIMER0->CTL |= TIMER_CTL_CNTEN_Msk;

    while (ch != 'q')
    {
        printf("Press 'q' to quit.\n");
        ch = getchar();
    }

    NVIC_DisableIRQ(TMR0_IRQn);

    /* TIMER Close */
    TIMER0->CTL = 0;
    TIMER0->EXTCTL = 0;

    CLK->APBCLK &= ~CLK_APBCLK_TMR0CKEN_Msk;

    NVIC_DisableIRQ(ADC0_IRQn);

    /* ADC Close */
    ADC->CS_CTL = (ADC0CS_VSS)       /* Switching to channel Vss to save power */
				| (ADC1CS_VSS);       /* Switching to channel Vss to save power */

    /* Disable ADC */
    ADC->CTL &= ~ADC_CTL_ADCEN_Msk;    
    SYS->IPRST1 |= SYS_IPRST1_ADCRST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_ADCRST_Msk;

    CLK->APBCLK &= ~CLK_APBCLK_ADCCKEN_Msk;

    printf("=== THE END ===\n\n");
    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
