/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/07/14  $
 * @brief    Sample code for ADC Simultaneous mode.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void SYS_Init(void)
{
    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Enable 48/60MHz HIRC */
    CLK->PWRCTL = CLK->PWRCTL | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for HIRC clock ready */
    CLK_WaitClockReady(CLK_STATUS_HIRCSTB_Msk);

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Enable USCI2 IP clock */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Enable IP clock */
    CLK->APBCLK |= CLK_APBCLK_ADCCKEN_Msk;

    /* Reset Module */
    SYS->IPRST1 |= SYS_IPRST1_ADCRST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_ADCRST_Msk;


    /* Lock protected registers */
    SYS_LockReg();
}

int main()
{
    char ch;
    uint32_t  reg_adc0_1,reg_adc0_3, reg_adc1_3,reg_adc1_7;

    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    /* printf("\n\nPDID 0x%08X\n", SYS_ReadPDID()); */    /* Display PDID */
    printf("\n\nPDID 0x%08X\n", (unsigned int)(SYS->PDID & SYS_PDID_PDID_Msk)); /* Display PDID */

    printf("CPU @ %dHz\n", SystemCoreClock);        /* Display System Core Clock */

    /*
     * This sample code will convert ADC0 channel 1, channel 3, and ADC1 channel 3, channel 7 in
     * ADC Simultaneous mode and print conversion results.
     */
    printf("+---------------------------------------------------------+\n");
    printf("| NM1240 ADC Simultaneous  Mode Sample Code |\n");
    printf("+---------------------------------------------------------+\n");

    /* Enable ADC clock */
    CLK->APBCLK |= CLK_APBCLK_ADCCKEN_Msk;
    /* Reset IP */
    SYS->IPRST1 |= SYS_IPRST1_ADCRST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_ADCRST_Msk;

	/*Clear ADC Channel select*/
	ADC->CS_CTL &= ~ADC_CS_CTL_ADC0CSEN_Msk;
	
	/*Setting ADC0_CH 1,3 ADC0_CH1=ADC0CS_ADC0_P1,ADC0_CH3=ADC0CS_ADC0_P3*/
	ADC->CS_CTL |= ADC0CS_ADC0_P1|ADC0CS_ADC0_P3 ;
	/*Setting ADC1_CH 3,7, ADC1_CH3=ADC1CS_ADC0_P0,ADC1_CH7=ADC1CS_VSS*/	
	ADC->CS_CTL |= ADC1CS_ADC0_P0 | ADC1CS_VSS;
	
	/*ADC0CS_ADC0_P1 = PB1*/
    printf("Set GPB1 pin as ADC0_CH1(ADC0_P1) input pin.\n");
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB1MFP_Msk)) | SYS_GPB_MFP_PB1_ADC0_P1;
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE1_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE1_Pos);
    PB->DINOFF |= GPIO_DINOFF_DINOFF1_Msk;
	
	/*ADC0CS_ADC0_P3 = PC0*/
    printf("Set GPC0 pin as ADC0_CH3(ADC0_P3) input pin.\n");
    SYS->GPC_MFP = (SYS->GPC_MFP & (~SYS_GPC_MFP_PC0MFP_Msk)) | SYS_GPC_MFP_PC0_ADC0_P3;
    PC->MODE = (PC->MODE & ~GPIO_MODE_MODE0_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE0_Pos);
    PC->DINOFF |= GPIO_DINOFF_DINOFF1_Msk;	
	
	/*ADC1CS_ADC0_P0 = PB0*/
    printf("Set GPB0 pin as ADC1_CH3(ADC1CS_ADC0_P0) input pin.\n");
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB0MFP_Msk)) | SYS_GPB_MFP_PB0_ADC0_P0;
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE0_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE0_Pos);
    PB->DINOFF |= GPIO_DINOFF_DINOFF0_Msk;

    printf(" ADC1_CH7 ADC1CS_VSS input pin.\n");	
	
    /* Enable ADC */
    ADC->CTL |= ADC_CTL_ADCEN_Msk;
	/* Disable ADC0 ADC1 Hardware trigger*/
    ADC->CTL &= ~ADC_CTL_ADC0HWTRGEN_Msk;
    ADC->CTL &= ~ADC_CTL_ADC1HWTRGEN_Msk;

    /* Configure ADC conversion mode to Simultaneous  Mode . */
    ADC->CTL = (ADC->CTL & ~ADC_CTL_ADCMODE_Msk) | (0X3 << ADC_CTL_ADCMODE_Pos);

    ADC->CTL |= ADC_CTL_ADCMODE_Msk;
    /* Configure ADC sample time to 5 ADC clocks */
    ADC->SMPCNT = (ADC->SMPCNT & (~ADC_SMPCNT_ADCSMPCNT_Msk)) | ((5 & 0xF) << ADC_SMPCNT_ADCSMPCNT_Pos);

  
    ch = 0;
    ADC->STATUS = (ADC_STATUS_ADC0IF_Msk|ADC_STATUS_ADC1IF_Msk);

    while (ch != 'q')
    {
        /* software trigger ADC0 */
        ADC->CTL |= ADC_CTL_ADC0SWTRG_Msk;

        /*MUST wait here. */
        while (!(ADC->STATUS & ADC_STATUS_ADC0IF_Msk));

        ADC->STATUS = ADC_STATUS_ADC0IF_Msk;

        /* MUST read and keep valid flag in DAT register before any others read it. */       
        reg_adc0_1 = ADC->VALSTS&BIT1;
		reg_adc0_3 = ADC->VALSTS&BIT3;
        reg_adc1_3 = ADC->VALSTS&BIT19;
		reg_adc1_7 = ADC->VALSTS&BIT23;

		/*Read ADC Status bit*/
        printf("Get ADC0 data1 = %4d (valid = %s)\n",ADC->ADC0_DAT1,(reg_adc0_1)? "Yes" : "No");
        printf("Get ADC0 data3 = %4d (valid = %s)\n",ADC->ADC0_DAT3,(reg_adc0_3)? "Yes" : "No");       
        printf("Get ADC1 data3 = %4d (valid = %s)\n",ADC->ADC1_DAT3,(reg_adc1_3)? "Yes" : "No"); 
		printf("Get ADC1 data7 = %4d (valid = %s)\n",ADC->ADC1_DAT7,(reg_adc1_7)? "Yes" : "No");
        printf("Press any key for next ADC conversion ... Press 'q' to quit.\n");
        ch = getchar();
    }

    /* ADC Close */
    ADC->CS_CTL = (ADC0CS_VSS)       /* Switching to channel Vss to save power */
				| (ADC1CS_VSS);       /* Switching to channel Vss to save power */
    /* Disable ADC */
    ADC->CTL &= ~ADC_CTL_ADCEN_Msk;
    SYS->IPRST1 |= SYS_IPRST1_ADCRST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_ADCRST_Msk;

    CLK->APBCLK &= ~CLK_APBCLK_ADCCKEN_Msk;

    printf("=== THE END ===\n\n");    
    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
