/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/07/14  $
 * @brief    Sample code for ADC Independent Simple mode.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void SYS_Init(void)
{
    int32_t i32TimeOutCnt = 2160000;

    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Enable 48/60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for HIRC clock ready */
    while((CLK->STATUS & CLK_STATUS_HIRCSTB_Msk) != CLK_STATUS_HIRCSTB_Msk) {
        if(i32TimeOutCnt-- <= 0) break;
    }

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Enable USCI2 IP clock */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Enable IP clock */
    CLK->APBCLK |= CLK_APBCLK_ADCCKEN_Msk;

    /* Reset Module. */
    SYS->IPRST1 |= SYS_IPRST1_ADCRST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_ADCRST_Msk;

    /* Select IP clock source */

    /*---------------------------------------------------------------------------------------------------------*/
    /* Init I/O Multi-function                                                                                 */
    /*---------------------------------------------------------------------------------------------------------*/

    /* Lock protected registers */
    SYS_LockReg();
}


int main()
{
    char ch;
	int tmp1;
    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    /* printf("\n\nPDID 0x%08X\n", SYS_ReadPDID()); */    /* Display PDID */
    printf("\n\nPDID 0x%08X\n", (unsigned int)(SYS->PDID & SYS_PDID_PDID_Msk)); /* Display PDID */

    printf("CPU @ %dHz\n", SystemCoreClock);        /* Display System Core Clock */

    /*
     * This sample code will convert ADC0 channel 0 and ADC1 channel 0 in Independent Simple mode
     * and print conversion results.
     */
    printf("+-------------------------------------------------+\n");
    printf("| NM1240 ADC Independent Simple Mode Sample Code |\n");
    printf("+-------------------------------------------------+\n");

    /* Enable ADC clock */
    CLK->APBCLK |= CLK_APBCLK_ADCCKEN_Msk;
    /* Reset IP */
    SYS->IPRST1 |= SYS_IPRST1_ADCRST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_ADCRST_Msk;

	/*Clear ADC Channel select*/
	ADC->CS_CTL &= ~ADC_CS_CTL_ADC0CSEN_Msk;
	/*ADC0 Channel select Set ADC0_CH 0,3,5, ADC0_CH0=ADC0CS_ADC0_P0,
	ADC0_CH3=ADC0CS_ADC0_P3,ADC0_CH5=ADC0CS_ADC0_P8*/
	ADC->CS_CTL |= ADC0CS_ADC0_P0|ADC0CS_ADC0_P3 ;
	/*ADC1 Channel select Set ADC1_CH 1,7, ADC1_CH1=ADC1CS_ADC1_P1,ADC1_CH7=ADC1CS_VSS*/
	ADC->CS_CTL |= ADC1CS_ADC1_P1 | ADC1CS_VSS;	
	
    printf("Set GPB0 pin as ADC0 CH0 input pin.\n");
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB0MFP_Msk)) | SYS_GPB_MFP_PB0_ADC0_P0;
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE0_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE0_Pos);
    PB->DINOFF |= GPIO_DINOFF_DINOFF0_Msk;
	
    printf("Set GPC0 pin as ADC0 CH3 input pin.\n");
    SYS->GPC_MFP = (SYS->GPC_MFP & (~SYS_GPC_MFP_PC0MFP_Msk)) | SYS_GPC_MFP_PC0_ADC0_P3;
    PC->MODE = (PC->MODE & ~GPIO_MODE_MODE0_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE0_Pos);
    PC->DINOFF |= GPIO_DINOFF_DINOFF0_Msk;	
	


    printf("Set GPD2 pin as ADC1 CH1 input pin.\n");
    SYS->GPD_MFP = (SYS->GPD_MFP & (~SYS_GPD_MFP_PD2MFP_Msk)) | SYS_GPD_MFP_PD2_ADC1_P1;
    PD->MODE = (PD->MODE & ~GPIO_MODE_MODE2_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE2_Pos);
    PD->DINOFF |= GPIO_DINOFF_DINOFF2_Msk;
	printf(" ADC1_CH7 ADC1CS_VSS input pin.\n");
    /* Enable ADC */
    ADC->CTL |= ADC_CTL_ADCEN_Msk;

    ADC->CTL &= ~ADC_CTL_ADC0HWTRGEN_Msk;

    ADC->CTL &= ~ADC_CTL_ADC1HWTRGEN_Msk;

    /* Configure ADC conversion mode to Independent Simple Mode */
    ADC->CTL = (ADC->CTL & ~ADC_CTL_ADCMODE_Msk) | (0x0 << ADC_CTL_ADCMODE_Pos);

    /* Configure ADC sample time to 6 ADC clocks */
    ADC->SMPCNT = (ADC->SMPCNT & (~ADC_SMPCNT_ADCSMPCNT_Msk)) | ((5 & 0xF) << ADC_SMPCNT_ADCSMPCNT_Pos);

    /* Begin to do ADC conversion. */
    ch = 0;    
    ADC->STATUS = (ADC_STATUS_ADC0IF_Msk|ADC_STATUS_ADC1IF_Msk);

    while (ch != 'q')
    {
        /* software trigger ADC0 */
        ADC->CTL |= ADC_CTL_ADC0SWTRG_Msk;

        /* Independent 2SH Mode: MUST wait ADC0 completed here. */
        while (!(ADC->STATUS & ADC_STATUS_ADC0IF_Msk));
        ADC->STATUS = ADC_STATUS_ADC0IF_Msk;
		
        printf("Get ADC0 data0 = %d\n",ADC->ADC0_DAT0 );
        printf("Get ADC0 data3 = %d\n",ADC->ADC0_DAT3 );

        /* software trigger ADC1 */
        ADC->CTL |= ADC_CTL_ADC1SWTRG_Msk;
 
        /* wait ADC1 completed by polling */
        while (!(ADC->STATUS & ADC_STATUS_ADC1IF_Msk));
        ADC->STATUS = ADC_STATUS_ADC1IF_Msk;
		
        printf("Get ADC1 data1 = %d\n",ADC->ADC1_DAT1 );
        printf("Get ADC1 data7 = %d\n",ADC->ADC1_DAT7 );    

        printf("Press any key for next ADC conversion ... Press 'q' to quit.\n");
        ch = getchar();
    }

    /* ADC Close. */
    ADC->CS_CTL = (ADC0CS_VSS)       /* Switching to channel Vss to save power */
				| (ADC1CS_VSS);       /* Switching to channel Vss to save power */
    /* Disable ADC */
    ADC->CTL &= ~ADC_CTL_ADCEN_Msk;
    SYS->IPRST1 |= SYS_IPRST1_ADCRST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_ADCRST_Msk;

    CLK->APBCLK &= ~CLK_APBCLK_ADCCKEN_Msk;

    printf("=== THE END ===\n\n");
    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
