/******************************************************************************
 * @file     main.c
 * @version  V1.00
 * $Revision: 1 $
 * $Date: 2020/07/14  $
 * @brief    Sample code for ADC sample value from ADC0_CH0 and output to DAC0.
 * @note
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include <stdio.h>
#include "NM1240.h"


void SYS_Init(void)
{
    int32_t i32TimeOutCnt = 2160000;

    /* Unlock protected registers */
    SYS_UnlockReg();

    /* Enable 48/60MHz HIRC */
    CLK->PWRCTL |= CLK_PWRCTL_HIRC_SEL_Msk | CLK_PWRCTL_HIRCEN_Msk;

    /* Waiting for HIRC clock ready */
    while((CLK->STATUS & CLK_STATUS_HIRCSTB_Msk) != CLK_STATUS_HIRCSTB_Msk) {
        if(i32TimeOutCnt-- <= 0) break;
    }

    /* HCLK Clock source from HIRC */
    CLK->CLKSEL0 = CLK->CLKSEL0 | CLK_HCLK_SRC_HIRC;

    /* Enable USCI2 IP clock */
    CLK->APBCLK = CLK->APBCLK | CLK_APBCLK_USCI2CKEN_Msk;

    /* Update System Core Clock */
    /* User can use SystemCoreClockUpdate() to calculate SystemCoreClock and cyclesPerUs automatically. */
    SystemCoreClockUpdate();

    /* USCI-Uart2-GPF1(TX) + GPF2(RX) */
    /* Set GPF multi-function pins for USCI UART2 GPF1(TX) and GPF2(RX) */
    SYS->GPF_MFP = (SYS->GPF_MFP & ~(SYS_GPF_MFP_PF1MFP_Msk | SYS_GPF_MFP_PF2MFP_Msk)) | (SYS_GPF_MFP_PF1_UART2_TXD | SYS_GPF_MFP_PF2_UART2_RXD);

    /* Set GPF1 as output mode and GPF2 as Input mode */
    PF->MODE = (PF->MODE & ~(GPIO_MODE_MODE1_Msk | GPIO_MODE_MODE2_Msk)) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE1_Pos);

    /* Enable IP clock */
    CLK->APBCLK |= CLK_APBCLK_ADCCKEN_Msk;

    /* Reset Module */
    SYS->IPRST1 |= SYS_IPRST1_ADCRST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_ADCRST_Msk;

    /* Select IP clock source */

    /*---------------------------------------------------------------------------------------------------------*/
    /* Init I/O Multi-function                                                                                 */
    /*---------------------------------------------------------------------------------------------------------*/

    /* Lock protected registers */
    SYS_LockReg();
}


int main()
{
    char ch;
    uint32_t u32_dat;

    SYS_Init();

    /* Init USCI UART2 to 115200-8n1 for print message */
    UUART_Open(UUART2, 115200);

    printf("\n\nPDID 0x%08X\n", SYS_ReadPDID());    /* Display PDID */
    printf("CPU @ %dHz\n", SystemCoreClock);        /* Display System Core Clock */

    /*
     * This sample code will convert ADC0 channel 0 and ADC1 channel 0 in Independent 2SH mode
     * and print conversion results.
     */
    printf("+----------------------------------------------+\n");
    printf("| NM1240 ADC to DAC Sample Code                |\n");
    printf("+----------------------------------------------+\n");

    /* Enable ADC clock */
    CLK->APBCLK |= CLK_APBCLK_ADCCKEN_Msk;
    /* Reset IP */
    SYS->IPRST1 |= SYS_IPRST1_ADCRST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_ADCRST_Msk;
    
    /* Enable ACMP clock */
    CLK->APBCLK |= CLK_APBCLK_ACMPCKEN_Msk;

    /* Reset IP */
    SYS->IPRST1 |= SYS_IPRST1_ACMPRST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_ACMPRST_Msk;

	/*Clear ADC Channel select*/
	ADC->CS_CTL &= ~ADC_CS_CTL_ADC0CSEN_Msk;
	/*ADC0 Channel select Set ADC0_CH0,ADC0_CH0=ADC0CS_ADC0_P0*/
	ADC->CS_CTL |= ADC0CS_ADC0_P0;	
	/*ADC1 Channel select Set ADC1_CH0,ADC1_CH0=ADC1CS_ADC0_P0*/
	ADC->CS_CTL |= ADC1CS_ADC1_P0;	
	
	
	/*ADC0CS_ADC0_P0 = PB0*/
    printf("Set GPB0 pin as ADC0_CH0(ADC0_P0) input pin.\n");
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB0MFP_Msk)) | SYS_GPB_MFP_PB0_ADC0_P0;
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE0_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE0_Pos);
    PB->DINOFF |= GPIO_DINOFF_DINOFF0_Msk;	
	
	/*ADC1CS_ADC1_P0 = PB4*/
    printf("Set GPB4 pin as ADC1_CH0(ADC1_P0) input pin.\n");
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB4MFP_Msk)) | SYS_GPB_MFP_PB4_ADC1_P0;
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE4_Msk) | (GPIO_MODE_INPUT << GPIO_MODE_MODE4_Pos);
    PB->DINOFF |= GPIO_DINOFF_DINOFF4_Msk;

    /* Set GPB5 to DAC0 */
    printf("Set GPB5 pin as DAC0 output pin.\n");
    SYS->GPB_MFP = (SYS->GPB_MFP & (~SYS_GPB_MFP_PB5MFP_Msk)) | SYS_GPB_MFP_PB5_DAC0;
    PB->MODE = (PB->MODE & ~GPIO_MODE_MODE5_Msk) | (GPIO_MODE_OUTPUT << GPIO_MODE_MODE5_Pos);
    PB->DINOFF |= GPIO_DINOFF_DINOFF5_Msk;

    /* Enable ADC */
    ADC->CTL |= ADC_CTL_ADCEN_Msk;

    ADC->CTL &= ~ADC_CTL_ADC0HWTRGEN_Msk;

    /* Configure ADC conversion mode to Independent Sample Mode */
    ADC->CTL = (ADC->CTL & ~ADC_CTL_ADCMODE_Msk) | (0 << ADC_CTL_ADCMODE_Pos);

    /* Configure ADC sample time to 5 clocks */
    ADC->SMPCNT = (ADC->SMPCNT & (~ADC_SMPCNT_ADCSMPCNT_Msk)) | ((5 & 0xF) << ADC_SMPCNT_ADCSMPCNT_Pos);

    /* Begin to do ADC conversion. */
    ADC->STATUS = ADC_STATUS_ADC0IF_Msk;

    printf("Start to convert ADC0_CH0(PB.0) value to DAC0(PB.5) output with refresh rate 0.5 sec.\n");
    while (ch != 'q')
    {
        CLK_SysTickDelay(200000);
        CLK_SysTickDelay(200000);
       
                  
        /* Software trigger ADC0 */        
        ADC->CTL |= ADC_CTL_ADC0SWTRG_Msk;
        CLK_SysTickDelay(100000);
		ADC->CTL |= ADC_CTL_ADC1SWTRG_Msk; 
        /* Wait ADC0 completed */
        while (!(ADC->STATUS & ADC_STATUS_ADC0IF_Msk)); 
        
        ADC->STATUS = ADC_STATUS_ADC0IF_Msk;
        
        /* Show ADC data */
        u32_dat = ADC->ADC0_DAT0 ;
        printf("ADC0 current value = %d mv    \r", u32_dat*5000/4096);
		printf("Get ADC0 data0 = %d\n",ADC->ADC0_DAT0);
        printf("Get ADC1 data0 = %d\n",ADC->ADC1_DAT0);
        /* Output ADC0_CH0 current value to DAC0 */
        ACMP->DACVAL = (ACMP->DACVAL & ~ACMP_DACVAL_DAC0_Msk) | ((u32_dat)<<ACMP_DACVAL_DAC0_Pos);
        
        if((DEBUG_PORT->BUFSTS & UUART_BUFSTS_RXEMPTY_Msk) == 0)  {
            ch = (DEBUG_PORT->RXDAT);
            if((ch == 'Q') || (ch == 'q')) {
                break;
            }
        }
		ch = getchar();
    }

    ADC->CS_CTL = (ADC0CS_VSS)       /* Switching to channel Vss to save power */
				| (ADC1CS_VSS);       /* Switching to channel Vss to save power */
    /* Disable ADC */
    ADC->CTL &= ~ADC_CTL_ADCEN_Msk;
    ADC->CTL &= ~ADC_CTL_ADCEN_Msk;
    SYS->IPRST1 |= SYS_IPRST1_ADCRST_Msk;
    SYS->IPRST1 &= ~SYS_IPRST1_ADCRST_Msk;

    CLK->APBCLK &= ~CLK_APBCLK_ADCCKEN_Msk;
    
    printf("\n=== THE END ===\n\n");
    
    while(1);
}

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
